-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure down_wf_aggregate
  (Node         : in     STree.SyntaxNode;
   Scope        : in     Dictionary.Scopes;
   Next_Node    :    out STree.SyntaxNode;
   EStack       : in out ExpStack.ExpStackType;
   HeapParam    : in out Lists.List_Heap;
   IsAnnotation : in     Boolean) is
   QUAL_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.qualified_expression,
                              True  => SPSymbols.annotation_qualified_expression);

   HasOthersPart            : Boolean;
   AssociationType          : Typ_Agg_Association_Type;
   NameExp                  : Exp_Record;
   OthersNode               : STree.SyntaxNode;
   Ptr                      : Lists.List;
   UnknownOrIndiscreteFound : Boolean;
   IndexTypeMark            : Dictionary.Symbol;
   ErrorFound               : Boolean := False;

   --------------------------------------------------------------------

   procedure CreateAggregateStackEntry
     (IndexTypeSymbol : in Dictionary.Symbol;
      AssociationType : in Typ_Agg_Association_Type;
      HasOthersPart   : in Boolean;
      Scope           : in Dictionary.Scopes)
   -- this procedure discriminates between the cases listed in S.P0468.53.11
   -- and sets up the stack entry accordingly
   --
   -- preconditions to entry to this procedure:
   --     aggregate is an array aggregate
   --
   -- NB aggregate may be a lone others clause
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out AggregateStack.State;
   --# derives AggregateStack.State from *,
   --#                                   AssociationType,
   --#                                   Dictionary.Dict,
   --#                                   HasOthersPart,
   --#                                   IndexTypeSymbol,
   --#                                   LexTokenManager.State,
   --#                                   Scope;
   is
      CompleteCheckRangeFrom  : Integer;
      CompleteCheckRangeTo    : Integer;
      CompleteCheckRangeState : CompleteCheck.TypRangeState;
      TypeLowerBound          : Typ_Type_Bound;
      TypeUpperBound          : Typ_Type_Bound;
      CompleteRec             : CompleteCheck.T;
      CheckCompleteness       : Boolean;
      WarnNoOthers            : Boolean;
      CheckOverlap            : Boolean;
      SignalOutOfRange        : Boolean;
   begin
      -- if index type unknown or not discrete then cannot do much at all
      if Dictionary.IsUnknownTypeMark (IndexTypeSymbol) or not Dictionary.IsDiscreteTypeMark (IndexTypeSymbol, Scope) then
         TypeLowerBound         := Unknown_Type_Bound;
         TypeUpperBound         := Unknown_Type_Bound;
         CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
         CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
         --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
         --   so the value is ignored, giving a flow error
         --# accept Flow, 10, CompleteCheckRangeState, "Expected ineffective assignment to CompleteCheckRangeState";
         CompleteCheck.Init (CompleteRec, -- expect flow error
                             CompleteCheckRangeFrom, CompleteCheckRangeTo, CompleteCheckRangeState);
         --# end accept;
         CheckCompleteness := False;
         WarnNoOthers      := False;
         SignalOutOfRange  := False;
         if Dictionary.IsUnknownTypeMark (IndexTypeSymbol) and AssociationType = Aggregate_Is_Named then
            CheckOverlap := True;
         else
            CheckOverlap := False;
         end if;
      else
         -- get bounds from dictionary
         Get_Type_Bounds (Type_Symbol => IndexTypeSymbol,
                          Lower_Bound => TypeLowerBound,
                          Upper_Bound => TypeUpperBound);

         if not (TypeLowerBound.Is_Defined and TypeUpperBound.Is_Defined) then
            -- one or other bound is unknown to the dictionary
            -- set flags accordingly
            CheckCompleteness := False;
            WarnNoOthers      := True;
            if AssociationType = Aggregate_Is_Positional then
               CheckOverlap     := False;
               SignalOutOfRange := False;
            else
               CheckOverlap     := True;
               SignalOutOfRange := True;
            end if;

            -- set up range for completeness checker
            -- if both bounds unknown use symmetric range
            if (not TypeLowerBound.Is_Defined) and (not TypeUpperBound.Is_Defined) then
               CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
               CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
               -- otherwise use range extending from known bound
            elsif TypeLowerBound.Is_Defined then
               CompleteCheckRangeFrom := TypeLowerBound.Value;
               CompleteCheckRangeTo   := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
            else  -- TypeUpperBound.IsDefined
               CompleteCheckRangeTo   := TypeUpperBound.Value;
               CompleteCheckRangeFrom := (CompleteCheckRangeTo - ExaminerConstants.CompleteCheckSize) + 1;
            end if;
            --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
            --   so the value is ignored, giving a flow error
            --# accept Flow, 10, CompleteCheckRangeState, "Expected ineffective assignment to CompleteCheckRangeState";
            CompleteCheck.Init (CompleteRec, -- expect flow error
                                CompleteCheckRangeFrom, CompleteCheckRangeTo, CompleteCheckRangeState);
            --# end accept;

         else
            -- both bounds known to dictionary
            -- set up completeness checker
            CompleteCheck.Init (CompleteRec, TypeLowerBound.Value, TypeUpperBound.Value, CompleteCheckRangeState);

            -- for positional association, the question of whether the
            -- type is too big for the completeness checker is irrelevant
            if AssociationType = Aggregate_Is_Positional then
               CheckCompleteness := True;
               WarnNoOthers      := False;
               CheckOverlap      := False;
               SignalOutOfRange  := False;
            else
               -- set flags according to whether range fits in completeness checker
               if CompleteCheckRangeState = CompleteCheck.RangeDoesFit then
                  CheckCompleteness := True;
                  WarnNoOthers      := False;
                  CheckOverlap      := True;
                  SignalOutOfRange  := False;
               else
                  CheckCompleteness := False;
                  WarnNoOthers      := True;
                  CheckOverlap      := True;
                  SignalOutOfRange  := True;
               end if;
            end if;
         end if;
      end if;

      AggregateStack.Push
        (IndexTypeSymbol,
         TypeLowerBound,
         TypeUpperBound,
         Typ_Agg_Flags'(Check_Completeness        => CheckCompleteness,
                        Warn_No_Others            => WarnNoOthers,
                        Check_Overlap             => CheckOverlap,
                        Signal_Out_Of_Range       => SignalOutOfRange,
                        Out_Of_Range_Seen         => False,
                        More_Entries_Than_Natural => False,
                        Has_Others_Part           => HasOthersPart,
                        Association_Type          => AssociationType),
         0,
         CompleteRec);

   end CreateAggregateStackEntry;

   --------------------------------------------------------------------
begin
   -- code to determine association type enhanced to detect the
   -- occurrence of a 'lone' others clause, and moved here so that
   -- the information is available to anonymous aggregates
   -- code determining presence of others part moved here for same reason
   case Syntax_Node_Type (Node => Child_Node (Child_Node (Node))) is
      when SPSymbols.positional_association | SPSymbols.annotation_positional_association =>
         AssociationType := Aggregate_Is_Positional;
      when SPSymbols.named_association | SPSymbols.annotation_named_association =>
         AssociationType := Aggregate_Is_Named;
      when others =>
         AssociationType := Aggregate_Is_Lone_Others;
   end case;

   --# assert True; -- for RTC generation

   OthersNode := Child_Node (Child_Node (Child_Node (Node)));

   -- PNA observation 9/3/02: I can't see how this first if part can ever be true given the current
   -- grammar.  The else if part may be.  Expression should be aggregate_or_expression.
   if Syntax_Node_Type (Node => OthersNode) = SPSymbols.expression
     or else Syntax_Node_Type (Node => OthersNode) = SPSymbols.annotation_expression then
      HasOthersPart := True;
   else
      OthersNode := Next_Sibling (OthersNode);

      if Syntax_Node_Type (Node => OthersNode) = SPSymbols.aggregate_or_expression or
        Syntax_Node_Type (Node => OthersNode) = SPSymbols.annotation_aggregate_or_expression then
         HasOthersPart := True;
      else
         HasOthersPart := False;
      end if;
   end if;

   --# assert True; -- for RTC generation

   HasOthersPart := HasOthersPart or (AssociationType = Aggregate_Is_Lone_Others);

   if Syntax_Node_Type (Node => Parent_Node (Current_Node => Node)) = QUAL_LOOKUP (IsAnnotation) then --this is a top level, not
                                                                                                      --embedded, aggregate
      ExpStack.Pop (NameExp, EStack);

      case NameExp.Sort is
         when Is_Type_Mark =>
            NameExp.Is_Constant := True;
            if Dictionary.IsArrayTypeMark (NameExp.Type_Symbol, Scope) then
               if Dictionary.IsUnconstrainedArrayType (NameExp.Type_Symbol) and not IsAnnotation then
                  -- Qualified aggregates of unconstrained array types only permitted in
                  -- annotation context.
                  -- So you are allowed to say, for example, "post X = T'(others => 0)" for a
                  -- subprogram that initializes an unconstrained array or "post X /= Y" where
                  -- both X and Y are unconstrained array types.
                  ErrorFound := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 39,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Parent_Node (Current_Node => Node)),
                     Id_Str    => LexTokenManager.Null_String);
               end if;

               --# assert True; -- for RTC generation

               NameExp.Param_Count := 1;   --used to record depth of dimension reached
               CreateAggregateStackEntry
                 (Dictionary.GetArrayIndex (NameExp.Type_Symbol, 1),
                  AssociationType,
                  HasOthersPart,
                  Scope);
               -- check types of all array dimensions, and warn if checking
               -- may be incomplete because any of the index types is unknown
               -- or indiscrete
               UnknownOrIndiscreteFound := False;

               -- NameExp.TypeSymbol here might denote an array subtype.  For subsequent
               -- checking of the aggregate, we need the first constrained subtype, so...
               NameExp.Type_Symbol := Dictionary.GetFirstConstrainedSubtype (NameExp.Type_Symbol);

               for I in Positive range 1 .. Dictionary.GetNumberOfDimensions (NameExp.Type_Symbol) loop

                  --# assert True; -- for RTC generation

                  IndexTypeMark := Dictionary.GetArrayIndex (NameExp.Type_Symbol, I);
                  if Dictionary.IsUnknownTypeMark (IndexTypeMark)
                    or else (not Dictionary.IsDiscreteTypeMark (IndexTypeMark, Scope)) then
                     UnknownOrIndiscreteFound := True;
                  end if;
               end loop;

               --# assert True; -- for RTC generation

               if UnknownOrIndiscreteFound then
                  ErrorFound := True;
                  ErrorHandler.Semantic_Warning
                    (Err_Num  => 307,
                     Position => Node_Position (Node => Parent_Node (Current_Node => Node)),
                     Id_Str   => LexTokenManager.Null_String);
               end if;
               NameExp.Errors_In_Expression := NameExp.Errors_In_Expression or ErrorFound;
               ExpStack.Push (NameExp, EStack);
               Next_Node := Child_Node (Current_Node => Node);

            elsif Dictionary.IsRecordTypeMark (NameExp.Type_Symbol, Scope) then
               if HasOthersPart then
                  ExpStack.Push (UnknownTypeRecord, EStack);
                  Next_Node := STree.NullNode;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 53,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => OthersNode),
                     Id_Str    => LexTokenManager.Null_String);

               elsif Dictionary.TypeIsExtendedTagged (NameExp.Type_Symbol)
                 and then Dictionary.ExtendedTaggedHasPrivateAncestors (NameExp.Type_Symbol, Scope) then
                  ExpStack.Push (UnknownTypeRecord, EStack);
                  Next_Node := STree.NullNode;
                  ErrorHandler.Semantic_Error_Sym
                    (Err_Num   => 833,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Parent_Node (Current_Node => Node)),
                     Sym       => NameExp.Type_Symbol,
                     Scope     => Scope);

               else -- OK, not illegal tagged record and has no others clause

                  -- NameExp.TypeSymbol here might denote a record subtype.  For subsequent
                  -- checking of the aggregate, we need the root record type, so...
                  NameExp.Type_Symbol := Dictionary.GetRootType (NameExp.Type_Symbol);

                  if AssociationType = Aggregate_Is_Named then
                     CreateNameList (Ptr, HeapParam);
                     NameExp.Param_List := Ptr;
                     ExpStack.Push (NameExp, EStack);
                     Next_Node := Child_Node (Current_Node => Node);
                  else --positional association
                     NameExp.Param_Count := 0;
                     ExpStack.Push (NameExp, EStack);
                     Next_Node := Child_Node (Current_Node => Node);
                  end if;
               end if;
            else --not a record or array
               ExpStack.Push (UnknownTypeRecord, EStack);
               ErrorHandler.Semantic_Error
                 (Err_Num   => 33,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Null_String);
               Next_Node := STree.NullNode;
            end if;

         when Is_Unknown =>
            --illegal name prefix but we can continue walk to check internal
            --validity of any expressions that follow.
            ExpStack.Push (UnknownTypeRecord, EStack);
            Next_Node := Child_Node (Current_Node => Node);

         when others =>
            ExpStack.Push (UnknownTypeRecord, EStack);
            ErrorHandler.Semantic_Error
              (Err_Num   => 95,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Parent_Node (Current_Node => Node)),
               Id_Str    => LexTokenManager.Null_String);
            Next_Node := Child_Node (Current_Node => Node);
      end case;

   else --it is an embedded aggregate of a multi-dim array
      ExpStack.Pop (NameExp, EStack);
      --increase depth of dimension count
      NameExp.Param_Count := NameExp.Param_Count + 1;
      CreateAggregateStackEntry
        (Dictionary.GetArrayIndex (NameExp.Type_Symbol, NameExp.Param_Count),
         AssociationType,
         HasOthersPart,
         Scope);
      ExpStack.Push (NameExp, EStack);
      Next_Node := Child_Node (Current_Node => Node);
   end if;
end down_wf_aggregate;
