-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure CreateFullSubProgDependency
  (Node        : in STree.SyntaxNode;
   SubprogSym  : in Dictionary.Symbol;
   Abstraction : in Dictionary.Abstractions) is
   ImportsInRelation, ExportsInRelation : SeqAlgebra.Seq;

   -------------------------------

   procedure AddToFullDependency (Sym  : in Dictionary.Symbol;
                                  Mode : in Dictionary.Modes)
   --# global in     ExportsInRelation;
   --#        in     ImportsInRelation;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    ExportsInRelation,
   --#                                    ImportsInRelation,
   --#                                    Mode,
   --#                                    Sym,
   --#                                    TheHeap;
   is

      procedure AddExport (Sym : in Dictionary.Symbol)
      --# global in     ExportsInRelation;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    ExportsInRelation,
      --#                                    Sym,
      --#                                    TheHeap;
      is
      begin --AddExport
         SeqAlgebra.AddMember (TheHeap, ExportsInRelation, Natural (Dictionary.SymbolRef (Sym)));
      end AddExport;

      ---------------

      procedure AddImport (Sym : in Dictionary.Symbol)
      --# global in     ImportsInRelation;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    ImportsInRelation,
      --#                                    Sym,
      --#                                    TheHeap;
      is
      begin --AddImport
         SeqAlgebra.AddMember (TheHeap, ImportsInRelation, Natural (Dictionary.SymbolRef (Sym)));
      end AddImport;

      ---------------

   begin --AddToFullDependency
      case Mode is
         when Dictionary.InMode =>
            AddImport (Sym);
         when Dictionary.OutMode =>
            AddExport (Sym);
         when Dictionary.InOutMode =>
            AddImport (Sym);
            AddExport (Sym);
         when Dictionary.DefaultMode =>
            AddImport (Sym); --can't occur for global
         when Dictionary.InvalidMode => --can't ocur
            null;
      end case;
   end AddToFullDependency;

   --------------------------------------------------

   procedure AddParametersToImportExportLists (SubprogSym : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     ExportsInRelation;
   --#        in     ImportsInRelation;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExportsInRelation,
   --#                                    ImportsInRelation,
   --#                                    SubprogSym,
   --#                                    TheHeap;
   is
      It  : Dictionary.Iterator;
      Sym : Dictionary.Symbol;

   begin --AddParametersToImportExportLists
      It := Dictionary.FirstSubprogramParameter (SubprogSym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         AddToFullDependency (Sym, Dictionary.GetSubprogramParameterMode (Sym));

         It := Dictionary.NextSymbol (It);
      end loop;
   end AddParametersToImportExportLists;

   --------------------------------------------------

   procedure AddGlobalsToImportExportLists (Abstraction : in Dictionary.Abstractions;
                                            SubprogSym  : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     ExportsInRelation;
   --#        in     ImportsInRelation;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Abstraction,
   --#                                    Dictionary.Dict,
   --#                                    ExportsInRelation,
   --#                                    ImportsInRelation,
   --#                                    SubprogSym,
   --#                                    TheHeap;
   is
      It  : Dictionary.Iterator;
      Sym : Dictionary.Symbol;

   begin --AddGlobalsToImportExportLists
      It := Dictionary.FirstGlobalVariable (Abstraction, SubprogSym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         AddToFullDependency (Sym, Dictionary.GetGlobalMode (Abstraction, SubprogSym, Sym));

         It := Dictionary.NextSymbol (It);
      end loop;
   end AddGlobalsToImportExportLists;

   --------------------------------------------------

   procedure AddExportsToDictionary
     (TheHeap     : in Heap.HeapRecord;
      Node        : in STree.SyntaxNode;
      SubprogSym  : in Dictionary.Symbol;
      Abstraction : in Dictionary.Abstractions)
   --# global in     ExportsInRelation;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ExportsInRelation,
   --#                                SubprogSym,
   --#                                TheHeap &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dictionary.Dict,
   --#                                ExportsInRelation,
   --#                                LexTokenManager.State,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym,
   --#                                TheHeap;
   is
      Member : SeqAlgebra.MemberOfSeq;

   begin --AddExportsToDictionary
      Member := SeqAlgebra.FirstMember (TheHeap, ExportsInRelation);
      while not SeqAlgebra.IsNullMember (Member) loop
         Dictionary.AddExport
           (Abstraction     => Abstraction,
            TheProcedure    => SubprogSym,
            TheExport       => Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                      M        => Member))),
            ExportReference => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                    End_Position   => Node_Position (Node => Node)),
            Annotation      => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                    End_Position   => Node_Position (Node => Node)));

         Member := SeqAlgebra.NextMember (TheHeap, Member);
      end loop;
   end AddExportsToDictionary;

   procedure PossiblyAddNullExport (ExportsInRelation : in SeqAlgebra.Seq;
                                    ImportsInRelation : in SeqAlgebra.Seq)
   --# global in     Abstraction;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in     SubprogSym;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Dictionary.Dict       from *,
   --#                                    Abstraction,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    ImportsInRelation,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    SubprogSym,
   --#                                    TheHeap &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Abstraction,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    ImportsInRelation,
   --#                                    LexTokenManager.State,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    SubprogSym,
   --#                                    TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Abstraction,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    ExportsInRelation,
   --#                                    ImportsInRelation,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    SubprogSym,
   --#                                    TheHeap;
   is
      Import                            : SeqAlgebra.MemberOfSeq;
      ExternalVariableFoundInImportList : Boolean := False;
   begin -- PossiblyAddNullExport
         -- check whether any of the imports are external variables
      Import := SeqAlgebra.FirstMember (TheHeap, ImportsInRelation);
      while not SeqAlgebra.IsNullMember (Import) loop
         if Dictionary.GetOwnVariableOrConstituentMode
           (Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                      M        => Import)))) /=
           Dictionary.DefaultMode then
            ExternalVariableFoundInImportList := True;
            exit;
         end if;
         Import := SeqAlgebra.NextMember (TheHeap, Import);
      end loop;
      -- if NONE of the imports are external variables then we need to create a suitable
      -- export by adding the NullVariable to the export set.
      if not ExternalVariableFoundInImportList then
         -- We must first make it a global of the subprogram
         Dictionary.AddGlobalVariable
           (Abstraction       => Abstraction,
            Subprogram        => SubprogSym,
            Variable          => Dictionary.GetNullVariable,
            Mode              => Dictionary.OutMode,
            PrefixNeeded      => False,
            Comp_Unit         => ContextManager.Ops.Current_Unit,
            VariableReference => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                      End_Position   => Node_Position (Node => Node)));
         -- and then we can add it to the export set.  Call to CreateFullDerives will do the rest
         SeqAlgebra.AddMember (TheHeap, ExportsInRelation, Natural (Dictionary.SymbolRef (Dictionary.GetNullVariable)));

      end if;
   end PossiblyAddNullExport;

begin -- CreateFullDependencyForSubprogram
      -- Use parameter and global modes to create set of imports and set of exports
   SeqAlgebra.CreateSeq (TheHeap, ImportsInRelation);
   SeqAlgebra.CreateSeq (TheHeap, ExportsInRelation);
   AddParametersToImportExportLists (SubprogSym);
   AddGlobalsToImportExportLists (Abstraction, SubprogSym);
   -- If the above calls to populate the set of imports and exports results in an
   -- empty sequence of exports but a non-empty sequence of imports then we need
   -- to do a further check.
   -- Nothing needs to be done if the imports include external (stream) variables
   -- because AddDerivesStreamEffects below will  make any imported streams into
   -- an export as well (in order to model stream side-effects.
   -- We only need to do something if all the imports are "ordinary"
   -- variables.
   -- In that case only we make the set of exports into {NullVariable}.
   if SeqAlgebra.IsEmptySeq (TheHeap, ExportsInRelation) and then not SeqAlgebra.IsEmptySeq (TheHeap, ImportsInRelation) then
      PossiblyAddNullExport (ExportsInRelation, ImportsInRelation);
   end if;
   AddExportsToDictionary (TheHeap, Node, SubprogSym, Abstraction);
   Dictionary.AddDependencyRelation
     (Abstraction,
      SubprogSym,
      Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                           End_Position   => Node_Position (Node => Node)));
   -- create "closure" dependency relation, all exports from all imports
   CreateFullDependency
     (Node        => Node,
      SubprogSym  => SubprogSym,
      Abstraction => Abstraction,
      ImportList  => ImportsInRelation,
      ExportList  => ExportsInRelation);

   SeqAlgebra.DisposeOfSeq (TheHeap, ImportsInRelation);
   SeqAlgebra.DisposeOfSeq (TheHeap, ExportsInRelation);

   AddDerivesStreamEffects (Node        => Node,
                            Subprog_Sym => SubprogSym,
                            Abstraction => Abstraction);

   if Abstraction = Dictionary.IsRefined then
      CheckDerivesConsistency (SubprogSym, Node_Position (Node => Node));
   end if;
end CreateFullSubProgDependency;
