-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure AssignmentCheck
  (Position   : in     LexTokenManager.Token_Position;
   Scope      : in     Dictionary.Scopes;
   TargetType : in     Dictionary.Symbol;
   ExpResult  : in out Exp_Record)
-- This procedure checks assignment compatibility given a target type and
-- a record from the expression stack.  It checks type mismatches, use of
-- unqualified string literals, attempts to assign unconstrained objects
-- and assignement of arrays where the bounds do not match.  If any of these
-- checks fail the stack record is changed to the UnknownTypeRecord.
-- Scalars are also checked for Constraint_Error.  If this check fails, the
-- value field of the stack record is changed to Maths.NoValue.
is

   ExpValue : Maths.Value;
   StoreRep : LexTokenManager.Lex_String;

   procedure RaiseError (ErrNum : in Natural;
                         Ref    : in Natural)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Position;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#           out ExpResult;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrNum,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         Ref,
   --#                                         SPARK_IO.File_Sys &
   --#         ExpResult                  from Dictionary.Dict;
   is
   begin
      ExpResult := UnknownTypeRecord;
      ErrorHandler.Semantic_Error
        (Err_Num   => ErrNum,
         Reference => Ref,
         Position  => Position,
         Id_Str    => LexTokenManager.Null_String);
   end RaiseError;

   ----------------

begin --AssignmentCheck
   if Dictionary.IsUnknownTypeMark (TargetType) or else Dictionary.IsUnknownTypeMark (ExpResult.Type_Symbol) then
      null;

   else
      if Dictionary.IsPredefinedStringType (ExpResult.Type_Symbol) then
         -- string literal or parameter
         -- check for type compatibility before other checks
         if not Dictionary.IsPredefinedStringType (Dictionary.GetRootType (TargetType)) then
            ErrorHandler.Semantic_Error_Sym2
              (Err_Num   => 107,
               Reference => 8,
               Position  => Position,
               Sym       => ExpResult.Type_Symbol,
               Sym2      => TargetType,
               Scope     => Scope);
         else
            if ExpResult.Range_RHS = Maths.NoValue then
               -- parameter
               -- can't assign a string parameter
               RaiseError (39, 7);

               -- if its a string literal its ok if the length is right
            elsif not Dictionary.IsPredefinedStringType (TargetType) then
               Maths.StorageRep (ExpResult.Range_RHS, StoreRep);
               if LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => Dictionary.GetScalarAttributeValue
                    (False,
                     LexTokenManager.Last_Token,
                     Dictionary.CurrentSymbol (Dictionary.FirstArrayIndex (TargetType))),
                  Lex_Str2 => StoreRep) /=
                 LexTokenManager.Str_Eq then
                  RaiseError (402, ErrorHandler.No_Reference); --constraint error
               end if;
            end if;
         end if;

      elsif Dictionary.IsUnconstrainedArrayType (ExpResult.Type_Symbol) then
         RaiseError (39, 7);

      elsif not Dictionary.CompatibleTypes (Scope, TargetType, ExpResult.Type_Symbol) then
         ErrorHandler.Semantic_Error_Sym2
           (Err_Num   => 107,
            Reference => 8,
            Position  => Position,
            Sym       => ExpResult.Type_Symbol,
            Sym2      => TargetType,
            Scope     => Scope);
         ExpResult := UnknownTypeRecord;

      elsif ExpResult.Is_ARange then
         RaiseError (91, ErrorHandler.No_Reference);

      elsif Illegal_Unconstrained (TargetType, ExpResult.Type_Symbol) then
         RaiseError (418, 5);

      else --if there is no error we can check fo constraint_error
         ConstraintCheck (ExpResult.Value, ExpValue, False, -- not in annotation expression
                          TargetType, Position);
         ExpResult.Value := ExpValue;
      end if;
   end if;
end AssignmentCheck;
