-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
procedure SearchForInheritedOperations
  (Name             : in     LexTokenManager.Lex_String;
   Scope            : in     Scopes;
   Prefix           : in     Symbol;
   Context          : in     Contexts;
   OpSym            :    out Symbol;
   KindOfOp         :    out KindsOfOp;
   ActualTaggedType :    out Symbol) is
   CurrentPackage   : Symbol;
   CallingPackage   : Symbol;
   PossibleOpSym    : Symbol    := NullSymbol;
   PossibleKindOfOp : KindsOfOp := NotASubprogram;

   function OperationCanBeInherited (TheOpSym : Symbol) return Boolean
   --# global in CallingPackage;
   --#        in CurrentPackage;
   --#        in Dict;
   is
      It          : Iterator;
      CurrentType : Symbol;
      Valid       : Boolean;

      function IsLocallyDeclared (TheType : Symbol) return Boolean
      --# global in CurrentPackage;
      --#        in Dict;
      is
      begin
         return GetRegion (GetScope (TheType)) = CurrentPackage;
      end IsLocallyDeclared;

   begin
      Valid := False;
      -- a subprogram is suitable for inheritance if it has a parameter
      -- of a tagged type declared in the same package and which the caller extends
      It := FirstSubprogramParameter (TheOpSym);
      while not IsNullIterator (It) loop
         CurrentType := GetType (CurrentSymbol (It));
         if TypeIsTagged (CurrentType)
           and then IsLocallyDeclared (CurrentType)
           and then IsAnExtensionOf (CurrentType, GetPackageExtendedType (CallingPackage)) then
            Valid := True;
            exit;
         end if;
         It := NextSymbol (It);
      end loop;
      return Valid;
   end OperationCanBeInherited;

begin -- SearchForInheritedOperations
   ActualTaggedType := NullSymbol; -- default

   -- this procedure will only be called when a normal search for an
   -- operation using LookUpItem or LookUpSelectedItem has failed.  We may
   -- be in some local scope so the first step is to get to the enclosing
   -- library package of the scope we start in if there is no prefix or
   -- the prefix package if there is one
   if Prefix = NullSymbol then
      CurrentPackage := GetLibraryPackage (Scope);
   else
      CurrentPackage := Prefix;
   end if;
   CallingPackage := CurrentPackage;
   -- now we can chain up the package "Extends" pointers looking for the
   -- required operation
   loop
      CurrentPackage := RawDict.GetPackageExtends (CurrentPackage);
      exit when CurrentPackage = NullSymbol; -- no more inherited packs

      -- Prior to release 7.1, a potentially inheritable operation must have
      -- been declared in the visible part of its package so LookupImmediateScope
      -- was a good choice for seeing if such an operation exists
      -- PossibleOpSym := LookupImmediateScope (Name,
      --                                        VisibleScope (CurrentPackage),
      --                                        Context);

      -- After release 7.1 the operation might be in the private part so we use
      -- LookUpSelectedItem instead; this makes operations correctly visible
      -- depending on whether we are looking from a child package or not.
      PossibleOpSym := LookupSelectedItem (CurrentPackage, Name, Scope, Context);

      if IsProcedure (PossibleOpSym) then
         PossibleKindOfOp := AProcedure;
         exit;
      end if;

      if IsFunction (PossibleOpSym) then
         PossibleKindOfOp := AFunction;
         exit;
      end if;

      if PossibleOpSym /= NullSymbol then -- something else found
         PossibleKindOfOp := NotASubprogram;
         exit;
      end if;
   end loop;

   -- At this point we have either found something and PossibleKindOfOp will say
   -- what it is or we have failed and PossibleOpSym is NullSymbol (and PossibleKindOfOp
   -- is NotASubprogram.  In any case a result of NotASubprogam is a failure and no further
   -- checks are required.
   if PossibleKindOfOp = NotASubprogram then
      KindOfOp := NotASubprogram;
      OpSym    := NullSymbol;
   else
      -- some kind of subprogram found so we need to check whether it has a parameter of
      -- a locally-declared tagged type
      if OperationCanBeInherited (PossibleOpSym) then
         OpSym            := PossibleOpSym;
         KindOfOp         := PossibleKindOfOp;
         ActualTaggedType := GetPackageExtendedType (CallingPackage);
      else -- not a suitable subprog to inherit
         KindOfOp := NotASubprogram;
         OpSym    := NullSymbol;
      end if;
   end if;
end SearchForInheritedOperations;
