/* Somaplayer - Copyright (C) 2003-4 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#ifdef ENABLE_STREAMING

#include "../../player.h"
#include "../../audio.h"
#include "../../file.h"
#ifdef ENABLE_LAME
#include "../lame/common.h"
#endif
#ifdef ENABLE_OGG
#include "../vorbis/common.h"
#endif
#include "../../other.h"
#include "../../util.h"
#include "streaming.h"

int
streaming_connect (audio * output)
{
  audio_data_streaming *data = (audio_data_streaming *) output->data;

#ifdef ENABLE_OPENSSL
  SSL *ssl;
  SSL_CTX *ctx;
#endif

  struct sockaddr_in sock;
  struct hostent *hp;

  msg (_("Connect to streaming server."));

  if (data->connect == HTTPS)
    {
#ifdef ENABLE_OPENSSL
      char *tmp;

      SSLeay_add_ssl_algorithms ();
      ctx = SSL_CTX_new (SSLv23_client_method ());
      data->bio = BIO_new_ssl_connect (ctx);
      BIO_get_ssl (data->bio, &ssl);

      if (!ssl)
	{
	  msg_error (_("Ssl connection error."));
	  return 1;
	}

      SSL_set_mode (ssl, SSL_MODE_AUTO_RETRY);

      if (!
	  (tmp =
	   (char *) malloc ((strlen (data->server) + 7) * sizeof (char))))
	fatal (_("Error: memory."));

      sprintf (tmp, "%s:%d", data->server, data->port);

      if (BIO_set_conn_hostname (data->bio, tmp) < 0)
	{
	  msg_error (_("Ssl connection error."));
	  free (tmp);
	  return 1;
	}

      if (BIO_do_connect (data->bio) < 0)
	{
	  msg_error (_("Ssl connection error."));
	  free (tmp);
	  return 1;
	}

      if (BIO_do_handshake (data->bio) < 0)
	{
	  msg_error (_("Ssl connection error."));
	  free (tmp);
	  return 1;
	}

      free (tmp);

#else
      msg_error (_("No support for ssl connection. Compile with SSL."));
      return 1;
#endif
    }

  else
    {

      if ((data->fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 1;
	}

      if (!(hp = gethostbyname (data->server)))
	{
	  msg_error (_("Error: unknow host."));
	  return 1;
	}

      memset ((void *) &sock, 0, sizeof (sock));
      sock.sin_family = AF_INET;
      sock.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;
      sock.sin_port = htons (data->port);

      if (connect (data->fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
	{
	  msg_error (_("Connect error."));
	  return 1;
	}

    }

  return 0;
}

void
streaming_show (audio * output)
{

  audio_data_streaming *data = (audio_data_streaming *) output->data;

  if (!data)
    return;

  switch (data->type)
    {

#ifdef ENABLE_LAME
    case OUT_LAME:
      if (data->type == OUT_LAME)
	streaming_lame_show (output);

      break;
#endif

#ifdef ENABLE_OGG
    case OUT_OGG:
      if (data->type == OUT_OGG)
	streaming_vorbis_show (output);

      break;
#endif
    }
}

void
streaming_quit (audio * output)
{

  audio_data_streaming *data = (audio_data_streaming *) output->data;

  if (!data)
    fatal (_("Internal error."));

  switch (data->type)
    {
#ifdef ENABLE_LAME
    case OUT_LAME:
      if (data->lame)
	{
	  lame_quit (data->lame);
	  data->lame = NULL;
	}
      break;
#endif

#ifdef ENABLE_OGG
    case OUT_OGG:
      if (data->vorbis)
	{
	  vorbis_quit (data->vorbis);
	  data->vorbis = NULL;
	}
      break;
#endif
    }

#ifdef ENABLE_OPENSSL
  if (data->connect == HTTPS)
    {
      if (data->bio != NULL)
	{
	  BIO_ssl_shutdown (data->bio);
	  BIO_free_all (data->bio);
	}
    }
  else
#endif
  if (data->fd != 0)
    close (data->fd);

  if (data->dumpfile)
    fclose (data->dumpfile);
}

int
streaming_write (audio * output, int channels, int bitrate, void *buffer, size_t length)
{
  audio_data_streaming *data = (audio_data_streaming *) output->data;

  static unsigned char *output_buf = NULL;
  int output_len = 0;
  int ret;

  if (!data)
    fatal (_("Internal error."));

  switch (data->type)
    {
#ifdef ENABLE_LAME
    case OUT_LAME:
      lame_write (data->lame, channels, bitrate, buffer, length, &output_buf,
		  &output_len);
      break;
#endif

#ifdef ENABLE_OGG
    case OUT_OGG:
      vorbis_write (data->vorbis, channels, bitrate, buffer, length, output,
		    streaming_socket_write);
      break;
#endif
    }

  if (output_len < 0)
    {
      msg (_("Error in encoding."));
      return -1;
    }

  if (output_len == 0)
    return 0;

  ret = streaming_socket_write (output, output_buf, output_len);

  if (data->dumpfile)
    {
      if (fwrite (output_buf, 1, output_len, data->dumpfile) != output_len)
	{
	  msg_error (_("Streaming output can't write the dumpfile local."));

	  fclose (data->dumpfile);

	  data->dumpfilelocal = NULL;
	  data->dumpfile = 0;
	}
    }

  free (output_buf);

  return ret;
}

audio_data_streaming *
streaming_audio_device (audio * output)
{

  audio_data_streaming *streaming;
  int ch = 0;
  int len;
  struct stat st;

  if (!
      (streaming =
       (audio_data_streaming *) malloc (sizeof (audio_data_streaming))))
    fatal (_("Error: memory."));

  if (!lstat (output->audio_dev, &st))
    {
      if (streaming_read_config (output->audio_dev, streaming))
	fatal (_("Streaming output error."));

    }
  else
    {

      streaming->type = 0;

#ifdef ENABLE_LAME
      streaming->type = OUT_LAME;
#endif

#ifdef ENABLE_OGG
      if (streaming->type != OUT_LAME)
	streaming->type = OUT_OGG;
#endif


#ifdef ENABLE_LAME
      streaming->lame = NULL;
#endif

#ifdef ENABLE_OGG
      streaming->vorbis = NULL;
#endif

      streaming->rate = 22050;
      streaming->quality = 8;
      streaming->bitrate = 24;
      streaming->channels = OUT_MONO;

      streaming->lowpass = 0;
      streaming->highpass = 0;

      streaming->password = NULL;
      streaming->server = NULL;
      streaming->port = 0;
      streaming->mount = NULL;

      streaming->genre = NULL;
      streaming->name = NULL;
      streaming->description = NULL;
      streaming->url = NULL;

      streaming->public = 1;

      streaming->aim = NULL;
      streaming->icq = NULL;
      streaming->irc = NULL;

      streaming->dumpfile = NULL;
      streaming->dumpfileremote = NULL;
      streaming->dumpfilelocal = NULL;

      streaming->connect = HTTP;
#ifdef ENABLE_OPENSSL
      streaming->bio = NULL;
#endif      
      streaming->fd = 0;

      len = strlen (output->audio_dev);

      while (ch < len)
	{

	  switch (*(output->audio_dev + ch))
	    {

	    case 'a':
	      if (!strncmp (output->audio_dev + ch, "aim=", 4))
		{
		  ch += 4;
		  streaming->aim =
		    audio_parse_str (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'b':
	      if (!strncmp (output->audio_dev + ch, "bitrate=", 6))
		{
		  ch += 8;
		  streaming->bitrate =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'c':
	      if (!strncmp (output->audio_dev + ch, "channels=", 9))
		{
		  int a;

		  ch += 9;
		  a = audio_parse_int (output->audio_dev, &ch, len);

		  switch (a)
		    {
		    case 1:
		      streaming->channels = OUT_MONO;
		      break;
		    case 2:
		      streaming->channels = OUT_STEREO;
		      break;
		    default:
		      msg_error (_("Error in audiodevice parameter mode: %s"),
				 output->audio_dev);
		      free (streaming);
		      return NULL;
		    }
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'd':

	      if (!strncmp (output->audio_dev + ch, "description=", 12))
		{
		  ch += 12;
		  streaming->description =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		if (!strncmp (output->audio_dev + ch, "dumpfilelocal=", 14))
		{
		  ch += 14;
		  streaming->dumpfilelocal =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		if (!strncmp (output->audio_dev + ch, "dumpfileremote=", 15))
		{
		  ch += 15;
		  streaming->dumpfileremote =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'g':

	      if (!strncmp (output->audio_dev + ch, "genre=", 6))
		{
		  ch += 6;
		  streaming->genre =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'h':

	      if (!strncmp (output->audio_dev + ch, "highpass=", 9))
		{
		  ch += 9;
		  streaming->highpass =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'i':

	      if (!strncmp (output->audio_dev + ch, "irc=", 4))
		{
		  ch += 4;
		  streaming->irc =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else if (!strncmp (output->audio_dev + ch, "icq=", 4))
		{
		  ch += 4;
		  streaming->icq =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'l':

	      if (!strncmp (output->audio_dev + ch, "lowpass=", 8))
		{
		  ch += 8;
		  streaming->lowpass =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'm':

	      if (!strncmp (output->audio_dev + ch, "mount=", 6))
		{
		  ch += 6;
		  streaming->mount =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'n':

	      if (!strncmp (output->audio_dev + ch, "name=", 5))
		{
		  ch += 5;
		  streaming->name =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'p':

	      if (!strncmp (output->audio_dev + ch, "port=", 5))
		{
		  ch += 5;
		  streaming->port =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else if (!strncmp (output->audio_dev + ch, "public=", 7))
		{
		  ch += 7;
		  streaming->public =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else if (!strncmp (output->audio_dev + ch, "password=", 9))
		{
		  ch += 9;
		  streaming->password =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'q':
	      if (!strncmp (output->audio_dev + ch, "quality=", 5))
		{
		  ch += 8;
		  streaming->quality =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'r':

	      if (!strncmp (output->audio_dev + ch, "rate=", 5))
		{
		  ch += 5;
		  streaming->rate =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 's':

	      if (!strncmp (output->audio_dev + ch, "server=", 7))
		{
		  char *server;
		  char *f;
		  int i = 0;

		  ch += 7;

		  server =
		    audio_parse_str (output->audio_dev, &ch, len);

		  if (!strncmp (server, "http://", 7))
		    {
		      streaming->connect = HTTP;
		      i = 7;
		    }
		  else if (!strncmp (server, "https://", 8))
		    {
		      streaming->connect = HTTPS;
		      i = 8;
		    }
		  else
		    {
		      streaming->connect = HTTP;
		      i = 0;
		    }

		  if (!(streaming->server = strdup (server + i)))
		    fatal (_("Error: memory."));

		  f = streaming->server;

		  while (*f++)
		    {
		      if (*f == ':')
			{
			  *f = 0;
			  streaming->port = atoi (++f);
			}
		      else if (*f == '/')
			{
			  *f = 0;
			  streaming->mount = ++f;
			}

		    }
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 't':

	      if (!strncmp (output->audio_dev + ch, "type=", 5))
		{
		  char *tmp;

		  ch += 5;
		  tmp = audio_parse_str (output->audio_dev, &ch, len);

		  if (!strcmp (tmp, "m") || !strcmp (tmp, "M"))
		    streaming->type = OUT_LAME;
		  else if (!strcmp (tmp, "o") || !strcmp (tmp, "O"))
		    streaming->type = OUT_OGG;
		  else
		    {
		      msg_error (_("Error in audiodevice parameter: %s"),
				 output->audio_dev);
		      free (streaming);
		      return NULL;
		    }

		  free (tmp);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    case 'u':

	      if (!strncmp (output->audio_dev + ch, "url=", 4))
		{
		  ch += 4;
		  streaming->url =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (streaming);
		  return NULL;
		}

	      break;

	    default:
	      msg_error (_("Error in audiodevice parameter: %s"),
			 output->audio_dev);
	      free (streaming);
	      return NULL;
	    }
	  ch++;
	}
    }

#ifndef ENABLE_LAME
  if (streaming->type == OUT_LAME)
    {
      msg_error (_("No support for lame encode. Compile with it."));
      free (streaming);
      return NULL;
    }
#endif

#ifndef ENABLE_OGG
  if (streaming->type == OUT_OGG)
    {
      msg_error (_("No support for ogg/vorbis encode. Compile with it."));
      free (streaming);
      return NULL;
    }
#endif

  if (streaming->type == 0)
    {
      msg_error (_("No encoding type. Compile with lame or ogg/vorbis."));
      free (streaming);
      return NULL;
    }

  if (!streaming->server)
    {
      msg_error (_("No server in streaming config."));
      free (streaming);
      return NULL;
    }

  if (!streaming->port)
    {
      msg_error (_("No port in streaming config."));
      free (streaming);
      return NULL;
    }

  if (!streaming->password)
    {
      msg_error (_("No password in streaming config."));
      free (streaming);
      return NULL;
    }

  if (!streaming->mount)
    {
      msg_error (_("No mount in streaming config."));
      free (streaming);
      return NULL;
    }

  switch (output->audio_type)
    {

#ifdef ENABLE_ICECAST
    case USE_ICECAST:
      if (streaming->aim)
	msg (_("Icecast no support for aim."));
      if (streaming->icq)
	msg (_("Icecast no support for icq."));
      if (streaming->irc)
	msg (_("Icecast no support for irc."));
      break;
#endif

#ifdef ENABLE_ICECAST2
    case USE_ICECAST2:
      if (streaming->dumpfileremote)
	msg (_("Icecast2 no support for dumpfileremote."));
      if (streaming->aim)
	msg (_("Icecast2 no support for aim."));
      if (streaming->icq)
	msg (_("Icecast2 no support for icq."));
      if (streaming->irc)
	msg (_("Icecast2 no support for irc."));
      break;
#endif

#ifdef ENABLE_SHOUTCAST
    case USE_SHOUTCAST:
      if (streaming->dumpfileremote)
	msg (_("Shoutcast no support for dumpfileremote."));
      if (streaming->description)
	msg (_("Shoutcast no support for description."));
      break;
#endif

    }

  return streaming;
}

int
streaming_socket_write (audio * ao, void *output, size_t ret)
{
  audio_data_streaming *data = (audio_data_streaming *) ao->data;

  if (!data)
    fatal (_("Internal error."));

  if (!ret)
    return 0;

#ifdef ENABLE_OPENSSL
  if (data->connect == HTTPS)
    return BIO_write (data->bio, output, ret);
#endif

  return write (data->fd, output, ret);
}

int
streaming_socket_read (audio * ao, char *output, int ret)
{
  audio_data_streaming *data = (audio_data_streaming *) ao->data;

#ifdef ENABLE_OPENSSL
  if (data->connect == HTTPS)
    return BIO_read (data->bio, output, ret);
  else
#endif
    return read (data->fd, output, ret);
}

#endif
