/* Somaplayer - Copyright (C) 2003-4 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "fconfig.h"
#include "audio.h"
#include "volume.h"
#include "other.h"
#include "buffer.h"

/* Read the config file ~/.somaplayer with the confuse lib. */
void
config_read (void)
{
  char *home;
  char *file;
  int a;
  cfg_t *cfg;

  cfg_opt_t opts[] = {
    CFG_STR ("buffer", NULL, CFGF_NONE),
    CFG_INT ("volume", 100, CFGF_NONE),
    CFG_BOOL ("random", cfg_false, CFGF_NONE),
    CFG_BOOL ("repeat", cfg_false, CFGF_NONE),
    CFG_BOOL ("noverbose", cfg_false, CFGF_NONE),
    CFG_BOOL ("nobuffer", cfg_false, CFGF_NONE),
#ifdef ENABLE_GTK
    CFG_INT ("x", -1, CFGF_NONE),
    CFG_INT ("y", -1, CFGF_NONE),
#endif
    CFG_STR ("audio", NULL, CFGF_NONE),
    CFG_END ()
  };

  if (!(home = getenv ("HOME")))
    {
      msg (_("I can't know your home. Set the enviroment!"));
      return;
    }

  a = strlen (home) + strlen ("/.somaplayer") + 1;
  if (!(file = (char *) malloc (a * sizeof (char))))
    fatal (_("Error: memory."));

  snprintf (file, a, "%s/.somaplayer", home);

  cfg = cfg_init (opts, CFGF_NOCASE);

  switch (cfg_parse (cfg, file))
    {
    case CFG_FILE_ERROR:
      msg (_("No config file: %s"), file);
      free (file);
      return;
    case CFG_PARSE_ERROR:
      msg (_("Error of parsing file: %s"), file);
      unlink (file);
      free (file);
      return;
    }

  msg (_("Read config file: %s"), file);

  free (file);

  if (cfg_getstr (cfg, "buffer"))
    play->realsize = buffer_size (cfg_getstr (cfg, "buffer"));

  if (play->realsize < SIZE_BUFFER)
    play->realsize = SIZE_BUFFER * 2;


  play->volume = cfg_getint (cfg, "volume");
  if (play->volume < VOLUME_MIN)
    play->volume = VOLUME_MIN;
  if (play->volume > VOLUME_MAX)
    play->volume = VOLUME_MAX;


  play->random = cfg_getbool (cfg, "random");
  play->repeat = cfg_getbool (cfg, "repeat");
  play->noverbose = cfg_getbool (cfg, "noverbose");
  play->nobuffer = cfg_getbool (cfg, "nobuffer");

  if (cfg_getstr (cfg, "audio"))
    play->config_audio = strdup (cfg_getstr (cfg, "audio"));
  else
    play->config_audio = NULL;

#ifdef ENABLE_GTK
  play->size_x = cfg_getint (cfg, "x");
  play->size_y = cfg_getint (cfg, "y");

  if (!play->size_x)
    play->size_x = -1;
  if (!play->size_y)
    play->size_y = -1;
#endif

  cfg_free (cfg);
}

/* Write! */
void
config_write (void)
{
  char *home;
  char *file;
  FILE *fl;
  audio *output = NULL;
  char *o = NULL;

  home = getenv ("HOME");
  if (!
      (file =
       (char *) malloc ((strlen (home) + strlen ("/.somaplayer") + 1) *
			sizeof (char))))
    fatal (_("Error: memory."));

  sprintf (file, "%s/.somaplayer", home);

  if (!(fl = fopen (file, "wb")))
    {
      msg_error (_("Error: open config file %s"), file);
      return;
    }

  free (file);
  free (home);

  fprintf (fl, "buffer = %d\n", play->realsize);
  fprintf (fl, "volume = %d\n", play->volume);
  fprintf (fl, "random = %s\n", play->random ? "true" : "false");
  fprintf (fl, "repeat = %s\n", play->repeat ? "true" : "false");
  fprintf (fl, "noverbose = %s\n", play->noverbose ? "true" : "false");
  fprintf (fl, "nobuffer = %s\n", play->nobuffer ? "true" : "false");

#ifdef ENABLE_AO
  output = play->output;
  while (output)
    {
      switch (output->audio_type)
	{
	case USE_ESD:
	  o = "esd";
	  break;
	case USE_ARTS:
	  o = "arts";
	  break;
	case USE_SUN:
	  o = "sun";
	  break;
	case USE_OSS:
	  o = "oss";
	  break;
	case USE_ALSA09:
	  o = "alsa09";
	  break;
	case USE_ALSA:
	  o = "alsa";
	  break;
	}

      if (o)
	break;

      output = output->next;
    }
#endif

  if (o)
    fprintf (fl, "audio = %s\n", o);

  fclose (fl);
}

/* EOF */
