/*
 * ----------------------------------------------------------------------
 * Emulation of Uzebox Videoport
 * (C) 2009 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state: not implemented 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------------------------
 */


#include <stdint.h>
#include <stdio.h>
#include "avr8_io.h"
#include "avr8_cpu.h"
#include "sgstring.h"
#include "signode.h"
#include "cycletimer.h"
#include "fbdisplay.h"

#define REG_PIN(base)   ((base) + 0x00)
#define REG_DDR(base)   ((base) + 0x01)
#define REG_PORT(base)  ((base) + 0x02)

#define DISPLAY_WIDTH 268 
#define DISPLAY_HEIGHT 224

typedef struct Uze_VidPort {
	
	SigNode *hsync;
	SigTrace *hsyncTrace;
        uint8_t reg_portx;  /* The portx register     */
        uint8_t reg_ddrx;   /* The direction register */
        uint8_t reg_pin;
	FbDisplay *display;
	uint32_t display_width;
	uint32_t display_height;
	uint8_t *fbuffer;
	/* Video state machine */
	uint8_t *line;
	uint32_t current_linebuf;
	uint32_t pixel_nr;
	uint32_t current_line;
	int x_repeat;
	int y_repeat;
	uint32_t real_line_length;
	//uint32_t current_fbidx;
} Uze_VidPort;


static uint8_t
pin_read(void *clientData,uint32_t address)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
        return port->reg_pin;
}

static void
pin_write(void *clientData,uint8_t value,uint32_t address)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
        port->reg_portx ^= value;
        //update_port_status(port,value);
        return;
}

static uint8_t
ddr_read(void *clientData,uint32_t address)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
        return port->reg_ddrx;
}

static void
ddr_write(void *clientData,uint8_t value,uint32_t address)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
        //uint8_t diff = value ^ port->reg_ddrx;
        port->reg_ddrx = value;
        //update_port_status(port,diff);
        return;
}

static uint8_t
port_read(void *clientData,uint32_t address)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
        return port->reg_portx;
}

static void
port_write(void *clientData,uint8_t value,uint32_t address)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
	if(port->pixel_nr + 1 < port->display_width) {
		port->line[port->pixel_nr++] = value;
		if(port->x_repeat) {
			port->line[port->pixel_nr++] = value;
		}	
	} 
        return;
}

static void hsync(SigNode *node,int value,void *clientData)
{
        Uze_VidPort *port = (Uze_VidPort *) clientData;
	static uint64_t last;
	uint32_t diff = CycleCounter_Get() - last;
	if(value == SIG_LOW) {
		if(port->current_line < port->display_height) {
			if(port->pixel_nr) {
				port->current_line++;
			}
		}
		if(port->current_line == 80) {
			port->real_line_length = port->pixel_nr;
			if(port->pixel_nr < (port->display_width >> 1)) {
				port->x_repeat = 1;
			} else if(port->pixel_nr >= port->display_width) {
				port->x_repeat = 0;
			}
			/*fprintf(stderr,"Port pixel %d\n",port->pixel_nr);*/
		}
		if((diff < 1000)) {
			if(port->current_line > 100) {
				uint32_t lines_empty;
				uint32_t pixels_empty;
				FbUpdateRequest fbudrq;
				lines_empty = (port->display_height - port->current_line);
				pixels_empty = (port->display_width - port->real_line_length);

				fbudrq.offset = ((lines_empty + 1) >> 1) * port->display_width
				 	+ ((pixels_empty + 1) >> 1);

				fbudrq.count = port->display_width * port->current_line;
				fbudrq.fbdata = port->fbuffer;
				FbDisplay_UpdateRequest(port->display,&fbudrq);
			}
			port->current_line = 0;
		}
		if(port->current_line < port->display_height) {
			port->line = port->fbuffer + port->current_line * port->display_width; 
		}
		port->pixel_nr = 0; 
	}	
	last = CycleCounter_Get();
}

static void
set_fbformat(Uze_VidPort *port) 
{
	FbFormat fbf;
	fbf.red_bits = 3;
        fbf.red_shift = 0;
        fbf.green_bits = 3;
        fbf.green_shift = 3;
        fbf.blue_bits = 2;
        fbf.blue_shift = 6;
        fbf.bits_per_pixel = 8;
        fbf.depth = 8;
	FbDisplay_SetFbFormat(port->display,&fbf);
}

void
Uze_VidPortNew(const char *name,uint16_t base,FbDisplay *display)
{
        Uze_VidPort *port = sg_new(Uze_VidPort);
	if(!display) {
		fprintf(stderr,"Uzebox Video requires a valid display\n");
		exit(1);
	}
        AVR8_RegisterIOHandler(REG_PIN(base),pin_read,pin_write,port);
        AVR8_RegisterIOHandler(REG_DDR(base),ddr_read,ddr_write,port);
        AVR8_RegisterIOHandler(REG_PORT(base),port_read,port_write,port);
	port->hsync = SigNode_New("%s.hsync",name);
	if(!port->hsync) {
		fprintf(stderr,"UzeVid: Can not create hsync signal line\n");
		exit(1);
	}
	port->display_width =  FbDisplay_Width(display);
	port->display_height =  FbDisplay_Height(display);
	port->fbuffer = sg_calloc(port->display_width * port->display_height);
	port->hsyncTrace = SigNode_Trace(port->hsync,hsync,port);
	port->display = display;
	port->line = port->fbuffer;
	port->x_repeat = port->y_repeat = 0;
	set_fbformat(port);
	fprintf(stderr,"Created Uzebox ATMega644 Video IO port \"%s\" width %d, height %d\n",
		name,port->display_width,port->display_height);
}

