/*
 * --------------------------------------------------------
 * Emulation of Samsung NAND Flash 
 *
 * (C) 2007 Jochen Karrer 
 *   Author: Jochen Karrer
 *
 * State: Nothing is implemented
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include "sgstring.h"
#include "nand.h"
#include "configfile.h"

typedef struct FlashDescription {
	char *type;
	int addrbytes;
	uint64_t size;
	int planes;
} FlashDescription;

static FlashDescription flash_types[] = {
	{ 
		.type = "K9F1208",
		.addrbytes = 4,
		.size = 528 * 32 * 4096,
		.planes = 4,
	},
};
/*
 * CMD sequences
 *	Read 1 		0x00/0x01	  (A/B Area 0-255,256-511)
 *	Read 2  	0x50		  (C Area 512-527)
 *	Reset		0xff
 *	Page PGM True	0x80	0x10
 * 	Page PGM Dummy	0x80	0x11
 *	Copyback PGM True  0x00 0x8a 0x10 
 *	Copyback PGM Dummy 0x03 0x8a 0x11
 *	Block Erase	0x60 	0xd0
 *	Multi Plane Block Erase 60--60 	0xd0
 *	Read Status		0x70
 *	Read Multi Plane Status	(0x71)
 * 
 */

#define STATE_IDLE		(0)
#define STATE_PGM_PAGE		(1)
#define STATE_COPYBACK_PGM	(2)
#define STATE_BLOCK_ERASE	(3)
#define STATE_MP_BLOCK_ERASE	(4)
static void
NandFlash_Access(NandFlashDevice *nf,NandFlashCmd *nfc) 
{
	int i;
	for(i=0;i<nfc->cmdlen;i++) {
		int cmd = nfc->cmd[i];
		switch(cmd) {
			case 0x00:
				/* sustained */
				nf->page_pointer = 0;
				break;

			case 0x01:
				/* only valid for one command */
				nf->page_pointer = 256;
				break;

			case 0x50:
				/* sustained */
				nf->page_pointer = 512;
				break;

			case 0x10:
				if(nf->state == STATE_PGM_PAGE) {
					// do_program
					nf->state = STATE_IDLE;
				} else if (nf->state == STATE_COPYBACK_PGM) {
					// do copyback
					nf->state = STATE_IDLE;
				}
				break;

			case 0x11:
				if(nf->state == STATE_PGM_PAGE) {
					nf->state = STATE_IDLE;
				} else if (nf->state == STATE_COPYBACK_PGM) {
					nf->state = STATE_IDLE;
				}
				break;

			case 0x70:	/* Read status */
			case 0x71:	/* Read multp. status */
				break;

			case 0x80:	/* Page PGM */
				nf->state = STATE_PGM_PAGE;
				break;

		}
	}
}

NandFlashDevice *
NandFlash_New(const char *name) 
{
	int i,nr_types;
	char *type,*imagedir,*filename;
	FlashDescription *fldescr;
	NandFlashDevice *nfd;
	type = Config_ReadVar(name,"type");	
	if(!type) {
		fprintf(stderr,"No type for Nand Flash \"%s\". Creating nothing\n",name);
		return NULL;
	}
	nr_types = sizeof(flash_types) / sizeof(FlashDescription);
	for(i=0;i < nr_types;i++) {
		fldescr = &flash_types[i];
		if(!strcmp(type,fldescr->type)) {
			break;
		}
	}
	if(i == nr_types) {
		fprintf(stderr,"Flash type \"%s\" not found\n",type);
		exit(1);
	}
	imagedir = Config_ReadVar("global","imagedir");
	if(!imagedir) {
		fprintf(stderr,"No directory for NAND Flash diskimage given\n");
		return NULL;
	}
	filename = alloca(strlen(imagedir) + strlen(name) + 50);
	sprintf(filename,"%s/%s.img",imagedir,name);
	nfd = sg_calloc(NandFlashDevice);
	nfd->disk_image = DiskImage_Open(filename,fldescr->size,DI_RDWR | DI_CREAT_FF | DI_SPARSE); 
	nfd->type = sg_strdup(type);
	nfd->busySig = SigNode_New("%s.busy",name);
	return nfd;
}
