/*
 * -----------------------------------------------------------------------
 * LPC-Flash 
 *      Emulation of the internal flash of Philips LPC  series
 *
 *
 * (C) 2005 Jochen Karrer
 *    Author: Jochen Karrer
 *
 * Status: Reading from flash is implemented 
 *	Mirror mapping, writing and state machine are missing
 *
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------------------------
 */

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>

#include <lpcflash.h>
#include <configfile.h>
#include <cycletimer.h>
#include <bus.h>
#include <signode.h>
#include <diskimage.h>


typedef struct LPCFlash {
	BusDevice bdev;
	DiskImage *disk_image;
	uint8_t *host_mem;
	int size;
} LPCFlash;

static void
flash_write(void *clientData,uint32_t value, uint32_t mem_addr,int rqlen) 
{
	fprintf(stderr,"Writing to lpcflash not implemented addr %08x, value %08x\n",mem_addr,value);
}

static void
Flash_Map(void *module_owner,uint32_t base,uint32_t mapsize,uint32_t flags) 
{
        LPCFlash *lflash = module_owner;
        uint8_t *host_mem=lflash->host_mem;
        if(1) {
                flags &= MEM_FLAG_READABLE;
                Mem_MapRange(base,host_mem,lflash->size,mapsize,flags);
        } else {
        	//IOH_NewRegion(base,mapsize,flash_read,flash_write,HOST_BYTEORDER,lflash);
	}
        IOH_NewRegion(base,mapsize,NULL,flash_write,HOST_BYTEORDER,lflash);
}

static void
Flash_UnMap(void *module_owner,uint32_t base,uint32_t mapsize) 
{
        Mem_UnMapRange(base,mapsize);
        IOH_DeleteRegion(base,mapsize);
}

static uint32_t
parse_memsize (char *str)
{
        uint32_t size;
        char c;
        if(sscanf(str,"%d",&size)!=1) {
                return 0;
        }
        if(sscanf(str,"%d%c",&size,&c)==1) {
                return size;
        }
        switch(tolower(c)) {
                case 'm':
                        return size*1024*1024;
                case 'k':
                        return size*1024;
        }
        return 0;
}

/*
 * -----------------------------------------------------------------
 * Create new lpc flash with diskimage
 * -----------------------------------------------------------------
 */
BusDevice *
LPCFlash_New(const char *flash_name) 
{
	LPCFlash *lflash = malloc(sizeof(LPCFlash));
	char *imagedir;
	char *sizestr;
	uint32_t flash_size;
	if(!lflash) {
		fprintf(stderr,"out of memory for LPCflash\n");
		exit(8);
	}
	memset(lflash,0,sizeof(LPCFlash));	
	imagedir = Config_ReadVar("global","imagedir");
	sizestr=Config_ReadVar(flash_name,"size");
        if(sizestr) {
                flash_size=parse_memsize(sizestr);
                if(flash_size==0) {
                        fprintf(stderr,"LPC flash bank \"%s\" not present\n",flash_name);
                        return NULL;
                }
	} else {
		fprintf(stderr,"size for LPC-flash \"%s\" not configured\n",flash_name);
		exit(1);
	}
	lflash->size = flash_size;
        if(imagedir) {
                char *mapfile = alloca(strlen(imagedir) + strlen(flash_name)+20);
                sprintf(mapfile,"%s/%s.img",imagedir,flash_name);
                lflash->disk_image = DiskImage_Open(mapfile,lflash->size,DI_RDWR | DI_CREAT_FF);
                if(!lflash->disk_image) {
                        fprintf(stderr,"Open disk image failed\n");
                        exit(42);
                }
                lflash->host_mem=DiskImage_Mmap(lflash->disk_image);
        } else {
                lflash->host_mem=malloc(lflash->size);
                if(!lflash->host_mem) {
                        fprintf(stderr,"Out of memory\n");
                        exit(637);
                }
                memset(lflash->host_mem,0xff,lflash->size);
        }
	lflash->bdev.first_mapping=NULL;
        lflash->bdev.Map=Flash_Map;
        lflash->bdev.UnMap=Flash_UnMap;
        lflash->bdev.owner=lflash;
        lflash->bdev.hw_flags=MEM_FLAG_READABLE;
	fprintf(stderr,"Created LPC Flash with size %d\n",flash_size);
        return &lflash->bdev;
}
