/*
 * --------------------------------------------------------------------------
 * Interface between Serial chip emulator and real terminal
 * 
 * (C) 2006 Jochen Karrer 
 *
 * State: not implemented
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------------------------
 */

#include <stdint.h>
#include <termios.h>


struct UartPort;

typedef void UartSinkProc(void *dev);
typedef void UartSrcProc(void *dev);
typedef void UartStatChgProc(void *dev);


typedef struct UartOps {
        void            (*set_mctrl)(struct UartPort *, uint32_t mctrl);
        uint32_t    	(*get_mctrl)(struct UartPort *);
        void            (*stop_tx)(struct UartPort *);
        void            (*start_tx)(struct UartPort *);
        //void            (*send_xchar)(struct UartPort *, char ch);
        void            (*start_rx)(struct UartPort *);
        void            (*stop_rx)(struct UartPort *);
       // void            (*enable_ms)(struct uart_port *);
        void            (*break_ctl)(struct UartPort *, int ctl);
        void            (*set_termios)(struct UartPort *, const struct termios *new);
        void            (*get_termios)(struct UartPort *, struct termios *new);
	int		(*write)(struct UartPort *port,const uint8_t *buf,int count);
	int		(*read)(struct UartPort *port,uint8_t *buf,int count);
} UartOps;

typedef struct UartPort {
	UartOps *ops;
	void *backend; /* stdio, real uart, tcp-socket ... */
	void *owner;
	UartSinkProc *sinkProc;
	UartSrcProc *srcProc;
	UartStatChgProc *statProc;
} UartPort;

static inline void 
Uart_SetMctrl(UartPort *port,uint32_t mctrl) 
{
	if(port->ops && port->ops->set_mctrl) {
		port->ops->set_mctrl(port,mctrl);
	}
}

static inline uint32_t 
Uart_GetMctrl(UartPort *port) 
{
	if(port->ops && port->ops->get_mctrl) {
		return port->ops->get_mctrl(port);
	} else {
		return 0;
	}
}

static inline void 
Uart_StartTx(UartPort *port) 
{
	//fprintf(stderr,"Uart StartTx\n");
	if(port->ops && port->ops->start_tx) {
		port->ops->start_tx(port);
	}
}

static inline void 
Uart_StopTx(UartPort *port)
{
	if(port->ops && port->ops->stop_tx) {
		port->ops->stop_tx(port);
	}
}

static inline void 
Uart_StartRx(UartPort *port) 
{
	if(port->ops && port->ops->start_rx) {
		port->ops->start_rx(port);
	}
}

static inline void 
Uart_StopRx(UartPort *port) 
{
	if(port->ops && port->ops->stop_rx) {
		port->ops->stop_rx(port);
	}
}
static inline void 
Uart_BreakCtl(UartPort *port,int ctl) 
{
	if(port->ops &&  port->ops->break_ctl) {
		port->ops->break_ctl(port,ctl);
	}
}

static inline void 
Uart_SetTermios(UartPort *port,const struct termios *termios) 
{
	if(port->ops && port->ops->set_termios) {
		port->ops->set_termios(port,termios);
	}
}
static inline void 
Uart_GetTermios(UartPort *port,struct termios *termios)
{
	if(port->ops && port->ops->get_termios) {
		port->ops->get_termios(port,termios);
	}
}
static inline int 
Uart_Read(UartPort *port,uint8_t *buf,int len)
{
	if(port->ops && port->ops->read) {
		return port->ops->read(port,buf,len);
	} else {
		return -1;
	}
}
static inline int 
Uart_Write(UartPort *port,const uint8_t *buf,int len)
{
	if(port->ops && port->ops->write) {
		return port->ops->write(port,buf,len);
	} else {
		return -1;
	}
}

UartPort *
Uart_New(const char *uart_name,UartSinkProc *sinkproc,UartSrcProc *srcproc,UartStatChgProc *statproc,void *owner);

