/*
 * ----------------------------------------------------------------------
 * Emulation of National semiconductor PC16550D Universal asynchronous
 * Receiver/Transmitter with Fifo
 * (C) 2005 Jochen Karrer 
 *   Author: Jochen Karrer
 *
 * state: never used, totally untested
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <errno.h>
#include <stdint.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <termios.h>
#include <sys/ioctl.h>

#include "bus.h"
#include "fio.h"
#include "signode.h"
#include "pc16550.h"

#define RBR(pc)	(0) 	/* Read  */
#define THR(pc)	(0) 	/* Write */
#define IER(pc)	(1)
#define		IER_ERBFI	(1<<0)
#define		IER_ETBEI	(1<<1)
#define		IER_ELSI	(1<<2)
#define		IER_EDSSI	(1<<3)
#define IIR(pc)	(2)	/* Read */
#define		IIR_NPENDING	(1<<0)
#define		IIR_INTID_MASK	(7<<1)
#define		IIR_INTID_SHIFT (1)
#define			IIR_TYPE_RLS 	(3)
#define			IIR_TYPE_RDA 	(2)
#define			IIR_TYPE_CTI 	(6)
#define			IIR_TYPE_THRE 	(1)
#define			IIR_TYPE_MST	(0)
#define FCR(pc)	(2)	/* Write */

#define LCR(pc)	(3)
#define		LCR_BITS_MASK (3)
#define		LCR_5BITS (0)
#define		LCR_6BITS (1)
#define		LCR_7BITS (2)
#define 	LCR_8BITS (3)
#define		LCR_STB  (4)
#define 	LCR_PEN	 (8)
#define 	LCR_EPS	 (0x10)
#define		LCR_SPAR (0x20)
#define		LCR_SBRK (0x40)
#define		LCR_DLAB (0x80)
#define	MCR(pc)	(4)
#define		MCR_DTR		(1<<0)
#define		MCR_RTS		(1<<1)
#define		MCR_OUT1	(1<<2)
#define		MCR_OUT2	(1<<3)
#define		MCR_LOOP	(1<<4)
#define LSR(pc)	(5)
#define		LSR_DR	(1)
#define 	LSR_OE  (1<<1)	
#define		LSR_PE  (1<<2)
#define 	LSR_FE	(1<<3)
#define		LSR_BI	(1<<4)
#define		LSR_THRE (1<<5)
#define		LSR_TEMT (1<<6)
#define		LSR_ERR	(1<<7)

#define MSR(pc)	(6)
#define		MSR_DCTS	(1<<0)
#define		MSR_DDSR	(1<<1)
#define		MSR_TERI	(1<<2)
#define		MSR_DDCD	(1<<3)
#define		MSR_CTS		(1<<4)
#define		MSR_DSR		(1<<5)
#define		MSR_RI		(1<<6)
#define		MSR_DCD		(1<<7)
#define SCR(pc)	(7)
#define DLL(pc)	(0)	/* DLAB=1 */
#define DLM(pc)	(1)	/* DLAB=1 */

#define DLAB(pc) (LCR(pc) & (1<<7))

#define RX_FIFO_SIZE (16)
#define RX_FIFO_MASK (RX_FIFO_SIZE-1)
#define RX_FIFO_COUNT(ser) (((RX_FIFO_SIZE + (ser)->rxfifo_wp - (ser)->rxfifo_rp)) & RX_FIFO_MASK)
#define RX_FIFO_ROOM(ser) (RX_FIFO_SIZE - RX_FIFO_COUNT(ser) - 1)

#define TX_FIFO_SIZE (16)
#define TX_FIFO_MASK (TX_FIFO_SIZE-1)
#define TX_FIFO_COUNT(ser) ((((ser)->txfifo_wp-(ser)->txfifo_rp)+TX_FIFO_SIZE)&TX_FIFO_MASK)
#define TX_FIFO_ROOM(ser) (TX_FIFO_SIZE-TX_FIFO_COUNT(ser)-1)


typedef struct PC16550 {
	BusDevice bdev;	
	int register_shift;

	int baudrate;
	uint8_t lcr;
	uint8_t mcr;
	uint8_t lsr;
	uint8_t msr;
	uint8_t dll,dlm;
	uint8_t ier;
	uint8_t iir;
	uint8_t scr;

	uint8_t rx_fifo[RX_FIFO_SIZE];
        int rxfifo_wp;
        int rxfifo_rp;

        uint8_t tx_fifo[TX_FIFO_SIZE];
        unsigned int txfifo_wp;
        unsigned int txfifo_rp;

	int fd;
	FIO_FileHandler input_fh;
        FIO_FileHandler output_fh;
        int ifh_is_active;
        int ofh_is_active;

	int interrupt_posted;
	SigNode *irqNode;
} PC16550;

static inline void enable_rx(PC16550 *pc);
static inline void disable_rx(PC16550 *pc);
static inline void enable_tx(PC16550 *pc);
static inline void disable_tx(PC16550 *pc);

static void
serial_close(PC16550 *pc) {
        if(pc->fd<0) {
                return;
        }
        disable_rx(pc);
        disable_tx(pc);
        close(pc->fd);
        pc->fd = -1;
        return;
}


static void
update_interrupts(PC16550 *pc) 
{
	/* do Priority encoding */
	int interrupt;
	uint8_t ier = pc->ier;
	uint8_t int_id = 0;
	if((ier & IER_ELSI) && 
	   (pc->lsr & (LSR_OE | LSR_PE | LSR_FE | LSR_BI)) ) {

		int_id = IIR_TYPE_RLS;	
		interrupt = 1;

	} else if((ier & IER_ERBFI) && (pc->lsr & LSR_DR)) { 
		// Fifo trigger level should be checked here also

		int_id = IIR_TYPE_RDA;
		interrupt = 1;

	/* 
	 * Character Timeout indication (IIR_TYPE_CTI) ommited here because 
         * trigger level IRQ mode is  not implemented 
         */
	} else if((ier & IER_ETBEI) && (pc->lsr & LSR_THRE)) {

		int_id = IIR_TYPE_THRE;
		interrupt = 1;

	} else if((ier & IER_EDSSI) &&
		pc->msr & (MSR_DCTS | MSR_TERI | MSR_DDSR | MSR_DDCD))	{
		int_id = IIR_TYPE_MST;
		interrupt = 1;
		
	} else {
		int_id = 0;
		interrupt = 0;
	}
	pc->iir = (pc->iir & ~IIR_INTID_MASK) || (int_id << IIR_INTID_SHIFT);
	if(interrupt) {
		pc->iir &= ~IIR_NPENDING;
		if(!pc->interrupt_posted) {
			SigNode_Set(pc->irqNode,SIG_LOW);
                        pc->interrupt_posted=1;
		}
	} else {
		pc->iir |= IIR_NPENDING;
		if(pc->interrupt_posted) {
			SigNode_Set(pc->irqNode,SIG_HIGH);
                        pc->interrupt_posted=0;
		}
	}
	
}

/*
 * --------------------------------------------------------
 * Filehandler for TX-Fifo
 *      Write chars from TX-Fifo to file as long
 *      as TX-Fifo is not empty and write would not block
 * --------------------------------------------------------
 */
static int
serial_output(void *cd,int mask) {
        PC16550 *pc=cd;
        int fill;
        while(pc->txfifo_rp!=pc->txfifo_wp) {
                int count,len;
                fill=TX_FIFO_COUNT(pc);
                len=fill;
                if((pc->txfifo_rp+fill)>TX_FIFO_SIZE) {
                        len = TX_FIFO_SIZE - pc->txfifo_rp;
                }
                count=write(pc->fd,&pc->tx_fifo[pc->txfifo_rp],len);
                if(count>0) {
                        pc->txfifo_rp = (pc->txfifo_rp+count)%TX_FIFO_SIZE;
                } else if (count==0) { // EOF
                        fprintf(stderr,"EOF on output\n");
                        serial_close(pc);
                        break;
                } else if ((count<0) && (errno==EAGAIN)) {
                        break;
                } else {
                        fprintf(stderr,"Error on output\n");
                        serial_close(pc);
                        break;
                }
        }
        if(TX_FIFO_COUNT(pc)==0) {
		pc->lsr |= LSR_THRE | LSR_TEMT;
                update_interrupts(pc);
        } else if(TX_FIFO_ROOM(pc)>0) {
		pc->lsr |= LSR_THRE;
                update_interrupts(pc);
        } 
        return 0;
}

/*
 * ------------------------------------
 * Put one byte to the rxfifo
 * ------------------------------------
 */
static inline int
serial_rx_char(PC16550 *pc,uint8_t c) {
        int room = RX_FIFO_ROOM(pc);
        if(room<1) {
                return -1;
        }
        pc->rx_fifo[pc->rxfifo_wp]=c;
        pc->rxfifo_wp=(pc->rxfifo_wp+1) & RX_FIFO_MASK;
        if(room==1) {
                disable_rx(pc);
                return 0;
        }
        return 1;
}


static int
serial_input(void *cd,int mask) {
        PC16550 *pc = cd;
        int fifocount;
        if(pc->fd<0) {
                fprintf(stderr,"Serial input with illegal fd %d\n",pc->fd);
                return -1;
        }
        while(1) {
                char c;
                int count=read(pc->fd,&c,1);
                if(count==1) {
                        serial_rx_char(pc,c);
                        //fprintf(stdout,"Console got %c\n",c);
                } else if (count==0) { // EOF
                        fprintf(stderr,"EOF reading from serial\n");
                        serial_close(pc);
                        break;
                } else if ((count<0) && (errno==EAGAIN)) {
                        break;
                } else {
                        fprintf(stderr,"Error on input\n");
                        serial_close(pc);
                        break;
                }
        }
        fifocount = RX_FIFO_COUNT(pc);
        if(fifocount) {
		pc->lsr |= LSR_DR;
                update_interrupts(pc);
        }
        return 0;
}

/*
 * ------------------------------------------------------
 * Flow control for receiver
 * ------------------------------------------------------
 */
static inline void
enable_rx(PC16550 *pc) {
        if((pc->fd >= 0) && !(pc->ifh_is_active)) {
                FIO_AddFileHandler(&pc->input_fh,pc->fd,FIO_READABLE,serial_input,pc);
        	pc->ifh_is_active=1;
        }
}

static inline void
disable_rx(PC16550 *pc) {
        if(pc->ifh_is_active) {
                FIO_RemoveFileHandler(&pc->input_fh);
        }
        pc->ifh_is_active=0;
}
static inline void
enable_tx(PC16550 *pc) {
        if((pc->fd>=0) && !(pc->ofh_is_active)) {
                FIO_AddFileHandler(&pc->output_fh,pc->fd,FIO_WRITABLE,serial_output,pc);
        	pc->ofh_is_active=1;
        }
}
static inline void
disable_tx(PC16550 *pc) {
        if(pc->ofh_is_active) {
                FIO_RemoveFileHandler(&pc->output_fh);
        }
        pc->ofh_is_active=0;
}

static void
update_serconfig(PC16550 *pc)
{
        int baudrate;
        uint32_t clk_rate=0;
        int divisor;
        tcflag_t bits;
        tcflag_t parodd;
        tcflag_t parenb;
        tcflag_t crtscts;
        struct termios termios;
        if(pc->fd<0) {
                return;
        }
        if(!isatty(pc->fd)) {
                return;
        }
	/* Does 16550 really have no automatic rtscts handshaking ? */
	crtscts=0; 
        if(pc->lcr & LCR_EPS) {
                parodd=0;
        } else {
                parodd=PARODD;
        }
        if(pc->lcr & LCR_EPS) {
                parenb=PARENB;
        } else {
                parenb=0;
        }
        switch(pc->lcr & LCR_BITS_MASK) {
                case LCR_5BITS:
                        bits=CS5; break;
                case LCR_6BITS:
                        bits=CS6; break;
                case LCR_7BITS:
                        bits=CS7; break;
                case LCR_8BITS:
                        bits=CS8; break;
                /* Can not be reached */
                default:
                        bits=CS8; break;
        }
        clk_rate = 1.843200;
        divisor=pc->dll+256*pc->dlm;
	if(divisor) {
        	baudrate=clk_rate/divisor;
	} else {
		baudrate = 0;
	}
	if((baudrate > 440000) && (baudrate < 480000)) {
                pc->baudrate=B460800;
        } else if((baudrate > 220000) && (baudrate < 240000)) {
                pc->baudrate=B230400;
        } else if((baudrate > 110000) && (baudrate < 120000)) {
                pc->baudrate=B115200;
        } else if((baudrate > 55000) && (baudrate < 60000)) {
                pc->baudrate=B57600;
        } else if((baudrate > 36571 ) && (baudrate < 40320)) {
                pc->baudrate=B38400;
        } else if((baudrate > 18285 ) && (baudrate < 20160)) {
                pc->baudrate=B19200;
        } else if((baudrate > 9142 ) && (baudrate < 10080)) {
                pc->baudrate=B9600;
        } else if((baudrate > 4571 ) && (baudrate < 5040)) {
                pc->baudrate=B4800;
        } else if((baudrate > 2285 ) && (baudrate < 2520)) {
                pc->baudrate=B2400;
        } else if((baudrate > 1890 ) && (baudrate < 1714)) {
                pc->baudrate=B1800;
        } else if((baudrate > 1142 ) && (baudrate < 1260)) {
                pc->baudrate=B1200;
        } else if((baudrate > 570 ) && (baudrate < 630)) {
                pc->baudrate=B600;
        } else if((baudrate > 285 ) && (baudrate < 315)) {
                pc->baudrate=B300;
        } else if((baudrate > 190 ) && (baudrate < 210)) {
                pc->baudrate=B200;
        } else if((baudrate > 142 ) && (baudrate < 158)) {
                pc->baudrate=B150;
        } else if((baudrate > 128 ) && (baudrate < 141)) {
                pc->baudrate=B134;
        } else if((baudrate > 105 ) && (baudrate < 116)) {
                pc->baudrate=B110;
        } else if((baudrate > 71 ) && (baudrate < 79)) {
                pc->baudrate=B75;
        } else if((baudrate > 47 ) && (baudrate < 53)) {
                pc->baudrate=B50;
        } else {
                fprintf(stderr,"PC16550-Serial emulator: Can not handle Baudrate %d\n",baudrate);
                pc->baudrate=B0;
        }
        
        if(tcgetattr(pc->fd,&termios)<0) {
                fprintf(stderr,"Can not  get terminal attributes\n");
                return;
        }
        if(pc->fd) {
                cfmakeraw(&termios);
        } else {
                termios.c_lflag &= ~(ISIG);
        }
        if(cfsetispeed ( &termios, pc->baudrate)<0) {
                fprintf(stderr,"Can not change Baudrate\n");
        }
        termios.c_cflag &= ~(CSIZE|PARENB|PARODD|CRTSCTS);
        termios.c_cflag |= bits | parenb | parodd | crtscts;

        /* Shit: TCSADRAIN would be better but tcsettattr seems to block */
        if(tcsetattr(pc->fd,TCSANOW,&termios)<0) {
                fprintf(stderr,"Can not  set terminal attributes\n");
                return;
        }
        return;

}

/*
 * --------------------------------------------
 * Write one character to the TX-Fifo
 * --------------------------------------------
 */
static inline void
thr_write(PC16550 *pc,uint8_t c) {
        int room;
	uint8_t old_lsr;
        pc->tx_fifo[pc->txfifo_wp] = c;
        pc->txfifo_wp = (pc->txfifo_wp+1) % TX_FIFO_SIZE;
	old_lsr=pc->lsr;	
        pc->lsr &= ~LSR_TEMT;
        room=TX_FIFO_ROOM(pc);
        if(room>0) {
		pc->lsr |= LSR_THRE;
        } else {
		pc->lsr &= ~LSR_THRE;
	}
	if(pc->lsr != old_lsr) {
		update_interrupts(pc);
	}
        enable_tx(pc);
}

/*
 * ----------------------------------------
 * Fetch a byte from RX-Fifo
 * ----------------------------------------
 */
static inline uint8_t
rbr_read(PC16550 *pc)  
{
        uint8_t data=0;
	if(RX_FIFO_COUNT(pc)>0) {
		data = pc->rx_fifo[pc->rxfifo_rp];
        	pc->rxfifo_rp = (pc->rxfifo_rp+1) & RX_FIFO_MASK;
	}
	if(RX_FIFO_COUNT(pc) == 0) {
		pc->lsr &= ~LSR_DR;
		update_interrupts(pc);
	}
	return data;
}

/* 
 * Divisor latch 
 */
static inline void 
dll_write(PC16550 *pc,uint8_t value)  
{
	pc->dll=value;
	update_serconfig(pc);
}
static inline void 
dlm_write(PC16550 *pc,uint8_t value)  
{
	pc->dlm=value;
	update_serconfig(pc);

}


static uint32_t
reg0_read(void *clientData,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	if(DLAB(pc)) {
		return pc->dll; /* Divisor latch last significant */
	} else {
		return rbr_read(pc);
	}
}

static void
reg0_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	if(DLAB(pc)) {
		dll_write(pc,value);
	} else {
		thr_write(pc,value);
	}
}
/*
 * ---------------------------------------------------------------------
 * Interrupt Enable register 
 *	Bit 0: 	Receive Data Interrupt and Timeout	
 *	Bit 1:  Transmit hold register Empty interrupt enable
 *	Bit 2:  Receive Line status interrupt enable
 *	Bit 3: 	Modem status change interrupt enable
 *	Bit 4-7: 0
 * ---------------------------------------------------------------------
 */
static uint32_t
ier_read(PC16550 *pc) 
{
        return pc->ier;
}

static void
ier_write(PC16550 *pc,uint8_t value) 
{
	pc->ier = value;
	update_interrupts(pc);
        return;
}
static uint32_t
reg1_read(void *clientData,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	if(DLAB(pc)) {
		return pc->dlm; /* Divisor latch most significant */
	} else {
		return ier_read(pc);
	}	
        return 0;
}

static void
reg1_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{	
	PC16550 *pc = clientData;
	if(DLAB(pc)) {
		dlm_write(pc,value);
	} else {
		ier_write(pc,value);
	}
        return;
}
/*
 * -------------------------------------------------------------
 * Interrupt Identification Register (Read only) 
 *	Bit 0:	Summary Interrupt
 *	Bit 1,2 Identify highest priority pending Interrupt
 *	Bit 3: 	Timeout interrupt in 16450 mode
 *	Bit 4,5 always 0
 *	Bit 6,7 set when fcr0 = 1
 * -------------------------------------------------------------
 */
static uint32_t
iir_read(void *clientData,uint32_t address,int rqlen) 
{
        return 0;
}

/* 
 * ----------------------------------------------------------------------------------
 * Fifo Control Register (Write only)
 *	Bit 0:	enable RCVR and XMIT Fifos, resetting clears them
 *      Bit 1:	Clear RX-Fifo and reset counters
 * 	Bit 2:	Reset XMIT-Fifo and reset counters
 *	Bit 3:  Change RXRDY and TXRDY Pins from mode 0 to mode 1
 * 	Bit 4,5: Reserved
 * 	Bit 6,7: Trigger level for RX-Fifo Interrupt {00,01,10,11} 1,4,8,14 Bytes
 * ---------------------------------------------------------------------------------
 */
static void
fcr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
        return;
}

/* 
 * --------------------------------------------------------------------------
 * Line Control Register
 * 	Bits 0 and 1: 00: 5Bit 01: 6Bit 10: 7Bit 11: 8Bit
 *	Bit  2:	   0 = 1 Stop Bit 1 = 1.5-2 Stop Bits
 *      Bit  3: PE 1 = Parity Enable
 * 	Bit  4: EPS 1 = Even Parity
 *	Bit  5: Stick parity
 *	Bit  6: Break control
 *	Bit  7: DLAB Divisor Latch Access Bit
 * ---------------------------------------------------------------------------
 */
static uint32_t
lcr_read(void *clientData,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
        return pc->lcr;
}

static void
lcr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	pc->lcr = value;
	update_serconfig(pc);
        return;
}
/*
 * ------------------------------------------------------------------------------
 * Modem Control Register
 *	Bit 0: Controls the #DTR output
 *	Bit 1: Controls the #RTS output 
 *	Bit 2: Controls the #OUT1 signal (Auxilliary)
 *	Bit 3: Controls the #OUT2 signal (Auxilliary)
 *	Bit 4: Local Loopback feature	
 *	Bit 5-7: always 0
 * ------------------------------------------------------------------------------
 */
static uint32_t
mcr_read(void *clientData,uint32_t address,int rqlen) 
{
        return 0;
}

static void
mcr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	uint8_t diff = pc->mcr ^ value;
	if(diff & (MCR_DTR | MCR_RTS)) {
		unsigned int clear=0;
                unsigned int set=0;
                if(!(value & MCR_DTR)) {
                        set |= TIOCM_DTR;
                } else {
                        clear |= TIOCM_DTR;
                }
                if(!(value & MCR_RTS)) {
                        set |= TIOCM_RTS;
                } else {
                        clear |= TIOCM_RTS;
                }
                if(isatty(pc->fd) && clear)
                        ioctl(pc->fd,TIOCMBIC,clear);
                if(isatty(pc->fd) && set)
                        ioctl(pc->fd,TIOCMSET,set);

	}
        return;
}
/*
 * -------------------------------------------------------------
 * Line Status Register
 * 	Bit 0: DR Data Ready indicator 
 *	Bit 1: OE Overrun Error indicator
 *	Bit 2: PE Parity Error indicator
 * 	Bit 3: FE Framing Error indicator
 *	Bit 4: BI Break Interrupt indicator
 *      Bit 5: THRE Transmitter Holding Register Empty
 *	Bit 6: TEMT Transmitter Empty indicator 
 *	Bit 7: Error Summary (not in 16450 mode)
 * -------------------------------------------------------------
 */
static uint32_t
lsr_read(void *clientData,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	return pc->lsr;
}

static void
lsr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
        return;
}
/*
 * -----------------------------------------------------------------
 * Modem Status register
 *	Bit 0: Delta CTS indicator
 *	Bit 1: Delta DSR indicator
 *	Bit 2: TERI Trailing edge of Ring Indicator (low to high)
 *	Bit 3: DCD Delta carrier detect
 *	Bit 4: Complement of RTS input
 *	Bit 5: Complement of DSR input
 * 	Bit 6: Complement of RI input
 * 	Bit 7: Complement of DCD intput
 * -----------------------------------------------------------------
 */
static uint32_t
msr_read(void *clientData,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	uint8_t msr;
	unsigned int host_status;
	msr = pc->msr;	
	if(isatty(pc->fd) && (ioctl(pc->fd,TIOCMGET,&host_status)>=0)) {
                msr=msr & ~(MSR_DCD | MSR_RI | MSR_DSR | MSR_CTS);
                if(!(host_status & TIOCM_CAR)) {
                        msr |= MSR_DCD;
                }
                if(!(host_status & TIOCM_RNG)) {
                        msr |= MSR_RI;
                }
                if(!(host_status & TIOCM_DSR)) {
                        msr |= MSR_DSR;
                }
                if(!(host_status & TIOCM_CTS)) {
                        msr |= MSR_CTS;
                }
        }
	if((pc->msr ^ msr) & MSR_DCD) {
                msr |= MSR_DDCD;
	}
	if((pc->msr ^ msr) & MSR_RI) {
                msr |= MSR_TERI;
	}
	if((pc->msr ^ msr) & MSR_DSR) {
                msr |= MSR_DDSR;
	}
	if((pc->msr ^ msr) & MSR_CTS) {
                msr |= MSR_DCTS;
	}
	pc->msr = msr & ~(MSR_DCTS | MSR_DDSR | MSR_TERI | MSR_DDCD);
	if((pc->ier & IER_EDSSI) && (pc->msr != msr)) {
		update_interrupts(pc);
	}
        return msr;
}

static void
msr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
        return;
}
/*
 * ------------------------------------------------
 * Scratchpad register
 * ------------------------------------------------
 */
static uint32_t
scr_read(void *clientData,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
        return pc->scr;
}

static void
scr_write(void *clientData,uint32_t value,uint32_t address,int rqlen) 
{
	PC16550 *pc = clientData;
	pc->scr = value;
        return;
}

#define ABS_ADDR(pc,base,reg) ((base) + ((reg)<<pc->register_shift))

static void
PC16550_Map(void *owner,uint32_t base,uint32_t mask,uint32_t flags) 
{
	PC16550 *pc=owner;
        IOH_New8(ABS_ADDR(pc,base,RBR(pc)),reg0_read,reg0_write,pc);
        IOH_New8(ABS_ADDR(pc,base,IER(pc)),reg1_read,reg1_write,pc);
        IOH_New8(ABS_ADDR(pc,base,IIR(pc)),iir_read,fcr_write,pc);
	IOH_New8(ABS_ADDR(pc,base,LCR(pc)),lcr_read,lcr_write,pc);
	IOH_New8(ABS_ADDR(pc,base,MCR(pc)),mcr_read,mcr_write,pc);
	IOH_New8(ABS_ADDR(pc,base,LSR(pc)),lsr_read,lsr_write,pc);
	IOH_New8(ABS_ADDR(pc,base,MSR(pc)),msr_read,msr_write,pc);
	IOH_New8(ABS_ADDR(pc,base,SCR(pc)),scr_read,scr_write,pc);

}

static void
PC16550_UnMap(void *owner,uint32_t base,uint32_t mask) 
{
	PC16550 *pc = owner;
        int i;
	for(i=0;i<8;i++) {
		IOH_Delete8(ABS_ADDR(pc,base,i));
	}
}

/*
 * ---------------------------------------------------
 * PC16550_New
 * 	Create a new National semicondcutor PC16650
 *      register_shift defines the space between
 *	to registers of 16550 in powers of two
 * ---------------------------------------------------
 */
BusDevice *
PC16550_New(char *devname,int register_shift) {
	char *nodename = alloca(strlen(devname)+30);
        PC16550 *pc = malloc(sizeof(PC16550));
        if(!pc) {
                fprintf(stderr,"Out of memory \n");
                exit(0);
        }
        memset(pc,0,sizeof(PC16550));
	pc->fd=-1;
	pc->register_shift = register_shift;
        pc->bdev.first_mapping=NULL;
        pc->bdev.Map=PC16550_Map;
        pc->bdev.UnMap=PC16550_UnMap;
        pc->bdev.owner=pc;
        pc->bdev.hw_flags=MEM_FLAG_WRITABLE|MEM_FLAG_READABLE;
	sprintf(nodename,"%s.irq",devname);
	pc->irqNode = SigNode_New(nodename);	
	SigNode_Set(pc->irqNode,SIG_HIGH); /* No request on startup */	
	pc->interrupt_posted=0;
	fprintf(stderr,"Created 16550 UART \"%s\"\n",devname);
        return &pc->bdev;
}
 
