/* Snownews - A lightweight console RSS newsreader
 * 
 * Copyright 2003-2004 Oliver Feiler <kiza@kcore.de>
 * http://kiza.kcore.de/software/snownews/
 *
 * main.c
 *
 * Please read the file README.patching before changing any code in this file!
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
#include <ncurses.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <signal.h>

#include "config.h"
#include "interface.h"
#include "ui-support.h"
#include "netio.h"
#include "main.h"
#include "xmlparse.h"
#include "updatecheck.h"
#include "categories.h"
#include "filters.h"
#include "setup.h"

extern struct feed *first_bak;		/* For use with the signal handler. */
extern char *browser;

/* Deinit ncurses and quit. */
void MainQuit (char * func, char * error) {
	if (error == NULL) {
		/* Only save settings if we didn't exit on error. */
		WriteCache();
	}
	clear();
	refresh();
	endwin();		/* Make sure no ncurses function is called after this point! */

	if (error == NULL) {		
		printf (_("Bye.\n\n"));
		
		/* Do this after everything else. If it doesn't work or hang
		   user can ctrl-c without interfering with the program operation
		   (like writing caches). */
		AutoVersionCheck();
		
		exit(0);
	} else {
		printf (_("Aborting program execution!\nAn internal error occured. Snownews has quit, no changes has been saved!\n"));
		printf (_("This shouldn't happen, please submit a bugreport to kiza@kcore.de, tell me what you where doing and include the output below in your mail.\n"));
		printf ("----\n");
		/* Please don't localize! I don't want to receive Japanese error messages. Thanks. :) */
		printf ("While executing: %s\n", func);
		printf ("Error as reported by the system: %s\n\n", error);
		exit(1);
	}
}

/* Signal handler function. */
void MainSignalHandler (int sig) {
	if (sig == SIGINT) {
		/* If there is a first_bak!=NULL a filter is set. Reset first_ptr
		   so the correct list gets written on the disk when exisiting via SIGINT. */
		if (first_bak != NULL)
			first_ptr = first_bak;
		MainQuit(NULL, NULL);
	}
}

/* http://foo.bar/address.rdf -> http:__foo.bar_address.rdf */
void Hashify (char * url) {
	int i;
	for (i = 0; i <= strlen(url); i++) {
		if (url[i] == '/') {
			url[i] = '_';
		}
		/* Cygwin doesn't seem to like anything besides a-z0-9 in filenames.
		   Zap'em! */
#ifdef __CYGWIN__
		if (((i < 48) && (i > 57)) ||
			((i < 65) && (i > 90)) ||
			((i < 97) && (i > 122)) ||
			(i != 0)) {
			url[i] = '_';
		}
#endif
	}
}



/* Update given feed from server.
 * Reload XML document and replace in memory cur_ptr->feed with it.
 */
int UpdateFeed (struct feed * cur_ptr) {
	char *tmpname;
	char *freeme;

	if (cur_ptr == NULL)
		return 1;
	
	/* Smart feeds are generated in the fly. */
	if (cur_ptr->smartfeed == 1)
		return 0;
		
	/* If current feed is an exec URL, run that command, otherwise fetch
	   resource from network. */
	if (cur_ptr->execurl) {
		FilterExecURL (cur_ptr);
	} else {
		/* Need to work on a copy of ->feedurl, because DownloadFeed()
		   changes the pointer. */
		tmpname = strdup (cur_ptr->feedurl);
		freeme = tmpname;		/* Need to make a copy, otherwise we cannot free
								   all RAM. */
	
		free (cur_ptr->feed);

		cur_ptr->feed = DownloadFeed (tmpname, cur_ptr, 0);

		free (freeme);

		/* Set title and link structure to something.
		To the feedurl in this case so the program show something
		as placeholder instead of crash. */
		if (cur_ptr->title == NULL)
			cur_ptr->title = strdup (cur_ptr->feedurl);
		if (cur_ptr->link == NULL)
			cur_ptr->link = strdup (cur_ptr->feedurl);

		/* If download function returns a NULL pointer return from here. */
		if (cur_ptr->feed == NULL) {
			return 1;
		}
	}
	
	/* Feed downloaded content through the defined filter. */
	if (cur_ptr->perfeedfilter != NULL)
		FilterPipe (cur_ptr);
	
	/* A NULL pointer will crash a strlen in xmlparse. Abort here, since
	   parsing an empty feed is pointless anyway. */
	if (cur_ptr->feed == NULL)
		return 1;
	
	if ((DeXML (cur_ptr)) != 0) {
		UIStatus (_("Invalid XML! Cannot parse this feed!"), 2);
		/* Activate feed problem flag. */
		cur_ptr->problem = 1;
		return 1;
	}
	
	/* We don't need these anymore. Free the raw XML to save some memory. */
	if (cur_ptr->feed != NULL) {
		free (cur_ptr->feed);
		cur_ptr->feed = NULL;
	}
		
	return 0;
}
int UpdateAllFeeds (void) {
	struct feed *cur_ptr;
	
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		if ((UpdateFeed (cur_ptr)) != 0)
			continue;
	}
	return 0;
}


/* Load feed from disk. And call UpdateFeed if neccessary. */
int LoadFeed (struct feed * cur_ptr) {
	int len = 0;				/* Internal usage for realloc. */
	int retval;
	char filebuf[4096];			/* File I/O block buffer. */
	char file[512];
 	char hashme[512];			/* Hashed filename. */
 	char tmp[1024];
	FILE *cache;
	
	/* Smart feeds are generated in the fly. */
	if (cur_ptr->smartfeed == 1)
		return 0;
	
	strncpy (hashme, cur_ptr->feedurl, sizeof(hashme));
	Hashify (hashme);
	snprintf (file, sizeof(file), "%s/.snownews/cache/%s", getenv("HOME"), hashme);
	cache = fopen (file, "r");
	
	if (cache == NULL) {
		snprintf (tmp, sizeof(tmp), _("Cache for %s is toast. Reloading from server..."), cur_ptr->feedurl);
		UIStatus (tmp, 0);
		
		if ((UpdateFeed (cur_ptr)) != 0)
			return 1;
		return 0;
	}

	/* Read complete cachefile. */
	while (!feof(cache)) {
		/* Use binary read, UTF-8 data!
		   It's about 4 times faster than the old code, too. */
		retval = fread (filebuf, 1, sizeof(filebuf), cache);
		if (retval == 0)
			break;
		cur_ptr->feed = realloc (cur_ptr->feed, len+retval + 1);
		memcpy (cur_ptr->feed+len, filebuf, retval);
		len += retval;
		if (retval != 4096)
			break;
	}
	
	cur_ptr->contentlength = len;
	
	if (cur_ptr->feed == NULL)
		return 1;
		
	cur_ptr->feed[len] = '\0';
	
	fclose (cache);
	
	/* After loading DeXMLize the mess. */
	if ((DeXML (cur_ptr)) != 0) {
		/* If loading the feed from the disk fails, try downloading from the net. */
		if ((UpdateFeed (cur_ptr)) != 0) {
			/* And if that fails as well, just continue without this feed. */
			snprintf (tmp, sizeof(tmp), _("Could not load %s!"), cur_ptr->feedurl);
			UIStatus (tmp, 2);
		}
	}
	
	free (cur_ptr->feed);
	cur_ptr->feed = NULL;
	
	return 0;
}
int LoadAllFeeds (int numfeeds) {
	float numobjperfeed;				/* Number of "progress bar" objects to draw per feed. */
	int count = 1;
	int numobjects;
	int i;
	int len;
	struct feed *cur_ptr;
	
	len = strlen (_("Loading cache ["));
	
	numobjperfeed = (COLS-len-2)/(double) numfeeds;
	
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		UIStatus (_("Loading cache ["), 0);
		/* Progress bar */
		numobjects = (count * numobjperfeed) - 2;
		if (numobjects < 1)
			numobjects = 1;
		attron (WA_REVERSE);
		for (i = 1; i <= numobjects; i++) {
			mvaddch (LINES-1, len+i, '=');
		}
		mvaddch (LINES-1, COLS-3, ']');
		refresh();
		attroff(WA_REVERSE);
		
		if ((LoadFeed (cur_ptr)) != 0) {
			continue;
		}
		count++;
	}
	return 0;
}

/* Write in memory structures to disk cache.
 * Usually called before program exit.
 */
void WriteCache (void) {
	char file[512];				/* File locations. */
	char hashme[512];			/* Cache file name. */
	char readstatus[2];
	char syscall[512];
	FILE *configfile;
	FILE *cache;
	struct feed *cur_ptr;
	struct newsitem *item;
	struct stat filetest;
	struct feedcategories *category;
	char *encoded;
	float numobjperfeed;				/* Number of "progress bar" objects to draw per feed. */
	int count = 1;
	int numobjects;
	int i;
	int numfeeds = 0;
	int len;
	
	UIStatus (_("Saving settings ["), 0);
	len = strlen (_("Saving settings ["));
	
	snprintf (file, sizeof(file), "%s/.snownews/browser", getenv("HOME"));
	configfile = fopen (file, "w+");
	if (configfile == NULL) {
		MainQuit (_("Save settings (browser)"), strerror(errno));
	}
	fputs (browser, configfile);
	fclose (configfile);
	
	snprintf (file, sizeof(file), "%s/.snownews/urls", getenv("HOME"));
	
	/* Make a backup of urls. This approach is really broken! */
	if ((stat (file, &filetest)) != -1) {
		if ((filetest.st_mode & S_IFREG) == S_IFREG) {
			snprintf (syscall, sizeof(file), "cp -f %s/.snownews/urls %s/.snownews/urls.bak", getenv("HOME"), getenv("HOME"));
			system (syscall);
		}
	}
	
	configfile = fopen (file, "w+");
	if (configfile == NULL) {
		MainQuit (_("Save settings (urls)"), strerror(errno));
	}
	
	/* Unoptimized vs. KISS */
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr)
		numfeeds++;
	
	numobjperfeed = (COLS-len-2)/(double) numfeeds;
	
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		/* Progress bar */
		numobjects = (count * numobjperfeed) - 2;
		if (numobjects < 1)
			numobjects = 1;
		attron (WA_REVERSE);
		for (i = 1; i <= numobjects; i++) {
			mvaddch (LINES-1, len+i, '=');
		}
		mvaddch (LINES-1, COLS-3, ']');
		refresh();
		attroff(WA_REVERSE);
		count++;
		
		fputs (cur_ptr->feedurl, configfile);
		
		fputc ('|', configfile);
		if (cur_ptr->override != NULL)
			fputs (cur_ptr->title, configfile);
		
		fputc ('|', configfile);
		if (cur_ptr->feedcategories != NULL) {
			for (category = cur_ptr->feedcategories; category != NULL; category = category->next_ptr) {
				fputs (category->name, configfile);
				/* Only add a colon of we run the loop again! */
				if (category->next_ptr != NULL)
					fputc (',', configfile);
			}
		}
		
		fputc ('|', configfile);
		if (cur_ptr->perfeedfilter != NULL)
			fputs (cur_ptr->perfeedfilter, configfile);
			
		fputc ('\n', configfile);		/* Add newline character. */
		
		/* Discard smart feeds from cache. */
		if (cur_ptr->smartfeed)
			continue;
		
		/* 
		 * Write cache.
		 */
		strncpy (hashme, cur_ptr->feedurl, sizeof(hashme));
		Hashify (hashme);
		snprintf (file, sizeof(file), "%s/.snownews/cache/%s", getenv("HOME"), hashme);
		cache = fopen (file, "w+");

		if (cache == NULL)
			MainQuit (_("Writing cache file"), strerror(errno));
		
		fputs ("<?xml version=\"1.0\" ?>\n\n<rdf:RDF\n  xmlns:rdf=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\"\n  xmlns=\"http://purl.org/rss/1.0/\">\n\n", cache);
		
		if (cur_ptr->lastmodified != NULL) {
			fputs ("<lastmodified>", cache);
			fputs (cur_ptr->lastmodified, cache);
			fputs ("</lastmodified>\n", cache);
		}
				
		fputs ("<channel rdf:about=\"", cache);
		
		encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->feedurl);
		fputs (encoded, cache);
		free (encoded);
		
		fputs ("\">\n<title>", cache);
		if (cur_ptr->original != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->original);
			fputs (encoded, cache);
			free (encoded);
		} else if (cur_ptr->title != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->title);
			fputs (encoded, cache);
			free (encoded);
		}
		fputs ("</title>\n<link>", cache);
		if (cur_ptr->link != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->link);
			fputs (encoded, cache);
			free (encoded);
		}
		fputs ("</link>\n<description>", cache);
		if (cur_ptr->description != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->description);
			fputs (encoded, cache);
			free (encoded);
		}
		fputs ("</description>\n</channel>\n\n", cache);
		
		for (item = cur_ptr->items; item != NULL; item = item->next_ptr) {
			fputs ("<item rdf:about=\"", cache);
			
			if (item->data->link != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->data->link);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("\">\n<title>", cache);
			if (item->data->title != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->data->title);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("</title>\n<link>", cache);
			if (item->data->link != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->data->link);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("</link>\n<description>", cache);
			if (item->data->description != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->data->description);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("</description>\n<readstatus>", cache);
			snprintf (readstatus, sizeof(readstatus), "%d", item->data->readstatus);
			fputs (readstatus, cache);			
			fputs ("</readstatus>\n</item>\n\n", cache);
		}
		
		fputs ("</rdf:RDF>", cache);
		fclose (cache);
	}
	fclose (configfile);
	
	return;
}


int main (int argc, char *argv[]) {
	int autoupdate = 0;		/* Automatically update feeds on app start... or not if set to 0. */
	int numfeeds;			/* Number of feeds loaded from Config(). */
		
#ifdef LOCALEPATH
  setlocale (LC_ALL, "");
  bindtextdomain ("snownews", LOCALEPATH);
  textdomain ("snownews");
#endif

	/* Kiza's really sucky argument checker.
	   Hey, it worked for Tornado. */
	if (argc > 1) {
		if (strcmp(argv[1], "--version") == 0 ||
		    strcmp(argv[1], "-V") == 0) {
			printf (_("Snownews version %s\n\n"), VERSION);
			return 0;
		} else if (strcmp(argv[1], "--update") == 0 ||
			strcmp(argv[1], "-u") == 0) {
			autoupdate = 1;
		} else if (strcmp(argv[1], "--disable-versioncheck") == 0) {
			printf ("The option \"--disable-versioncheck\" has been obsoleted.\n");
			printf ("Please see the manpage for details.\n");
			printf ("(To deactivate check, put -1 into ~/.snownews/updatecheck)\n");
			exit(1);
		} else {
			printf (_("Snownews version %s\n\n"), VERSION);
			printf (_("usage: snownews [-huV] [--help|--update|--version]\n\n"));
			printf (_("\t--update|-u\tAutomatically update every feed.\n"));
			printf (_("\t--help|-h\tPrint this help message.\n"));
			printf (_("\t--version|-V\tPrint version number and exit.\n"));
			printf (_("\t--disable-versioncheck (obsoleted)\n"));
			return 0;
		}
	}
	
	
	/* Install SIGINT signal handler. */
	signal (SIGINT, MainSignalHandler);
	
	InitCurses();
	
	/* Check if configfiles exist and create/read them. */
	numfeeds = Config();
		
	LoadAllFeeds (numfeeds);

	if (autoupdate)
		UpdateAllFeeds();
	
	/* Init the pRNG. See about.c for usages of rand() ;) */
	srand(time(0));
	
	/* Give control to main program loop. */
	UIMainInterface();

	/* We really shouldn't be here at all... ah well. */
	MainQuit(NULL, NULL);
	return 0;
}
