;;; Snd tests
;;;
;;; test 0: constants                            [330]
;;; test 1: defaults                             [862]
;;; test 2: headers                              [1029]
;;; test 3: variables                            [1327]
;;; test 4: sndlib                               [1710]
;;; test 5: overall checks                       [3506]
;;; test 6: vcts                                 [10735]
;;; test 7: colors                               [10947]
;;; test 8: clm                                  [11443]
;;; test 9: mixes                                [16920]
;;; test 10: marks                               [19916]
;;; test 11: dialogs                             [20614]
;;; test 12: sound file extensions etc           [20924]
;;; test 13: menus, edit lists, hooks, etc       [21338]
;;; test 14: all functions                       [22596]
;;; test 15: chan-local vars                     [23643]
;;; test 16: regularized funcs                   [24902]
;;; test 17: graphics                            [29276]
;;; test 18: enved                               [29350]
;;; test 19: save and restore                    [29370]
;;; test 20: transforms                          [29966]
;;; test 21: new stuff                           [31022]
;;; test 22: run                                 [31766]
;;; test 23: with-sound                          [36773]
;;; test 24: Snd user-interface                  [37408]
;;; test 25: X/Xt/Xm/Xpm                         [40577]
;;; test 26: Glib/gdk/gtk                        [45096]
;;; test 27: openGL                              [48025]
;;; test 28: errors                              [48129]
;;; end stats:                                   [50022]

;;; how to send ourselves a drop?  (button2 on menu is only the first half -- how to force 2nd?)
;;; need all html example code in autotests


(use-modules (ice-9 format) (ice-9 debug) (ice-9 optargs) (ice-9 popen))

(define (snd-display . args)
  (let ((str (if (null? (cdr args))
		 (car args)
		 (apply format #f args))))
    (newline) 
    (display str)
    (if (not (provided? 'snd-nogui))
	(begin
	  (snd-print #\newline)
	  (snd-print str)))))

(define tests 1)
(if (not (defined? 'snd-test)) (define snd-test -1))
(if (defined? 'disable-play) (disable-play))
(define keep-going #f)
(define full-test (< snd-test 0))
(define total-tests 28)
(if (not (defined? 'with-exit)) (define with-exit (< snd-test 0)))

(set! (with-background-processes) #f)
(define all-args #f) ; huge arg testing
(define with-big-file #t)
(set! (show-backtrace) #t)

(define debugging-device-channels 2)

(define pi 3.141592653589793)
(define max-optimization 6)

(define home-dir "/home/bil")
(if (file-exists? "/export/home/bil/cl/oboe.snd")
    (set! home-dir "/export/home/bil")
    (if (file-exists? "/Users/bill/cl/oboe.snd")
	(set! home-dir "/Users/bill")
	(if (file-exists? "/users/b/bil/cl/oboe.snd")
	    (set! home-dir "/users/b/bil"))))
(system (string-append "cp " home-dir "/.snd " home-dir "/dot-snd"))
(define cwd (string-append (getcwd) "/"))
(define sf-dir "/sf1")
(define sample-reader-tests 300)
(define original-save-dir (or (save-dir) "/zap/snd"))
(define original-temp-dir (or (temp-dir) "/zap/tmp"))
(debug-set! stack 0)
(debug-enable 'debug 'backtrace)
(read-enable 'positions)
(define (irandom n) (if (= n 0) 0 (inexact->exact (random n))))
(add-hook! bad-header-hook (lambda (n) #t))

(define with-gui (or (provided? 'snd-gtk)
		     (provided? 'snd-motif)))

(if (file-exists? "optimizer.log")
    (delete-file "optimizer.log"))
(define optimizer-log (open-output-file "optimizer.log"))
(reset-hook! optimization-hook)
(add-hook! optimization-hook 
	   (lambda (msg)
	     (display msg optimizer-log)
	     (newline optimizer-log)))

(define sf-dir1 (string-append home-dir sf-dir "/"))
(if (not (file-exists? (string-append sf-dir1 "alaw.wav")))
    (begin
      (set! sf-dir "/sf")
      (set! sf-dir1 (string-append home-dir sf-dir "/"))
      (if (not (file-exists? (string-append sf-dir1 "alaw.wav")))
	  (begin
	    (snd-display ";;;can't find sf directory!")
	    (set! sf-dir1 #f)))))
(set! sf-dir sf-dir1)

(if (not (string=? (getcwd) (string-append home-dir "/cl")))
    (for-each
     (lambda (file)
       (if (not (file-exists? file))
	   (begin
	     (display (format #f "copying ~A~%" file))
	     (copy-file (string-append home-dir "/cl/" file) (string-append (getcwd) "/" file)))))
     (list "4.aiff" "2.snd" "obtest.snd" "oboe.snd" "pistol.snd" "1a.snd" "now.snd" "fyow.snd"
	   "storm.snd" "z.snd" "1.snd" "cardinal.snd" "now.snd.scm" "2a.snd" "4a.snd" "zero.snd"
	   "loop.scm" "cmn-glyphs.lisp" "bullet.xpm" "mb.snd")))

(define (real-time) (exact->inexact (/ (get-internal-real-time) internal-time-units-per-second)))
(define (hundred n) (inexact->exact (round (* 100 n))))
(define times '())
(defmacro time (a) 
  `(let ((start (real-time))) 
     ,a 
     (let ((val (hundred (- (real-time) start))))
       (set! times (cons (list ',a val) times)))))

;(show-ptree 1)
(define original-prompt (listener-prompt))
(show-listener)
(set! (window-x) 600)
(set! (window-y) 10)

(define test14-file #f)
(define fneq (lambda (a b) (> (abs (- a b)) .001)))
(define ffneq (lambda (a b) (> (abs (- a b)) .01)))
(define fffneq (lambda (a b) (> (abs (- a b)) .1)))
(define (fneqerr val true-val err) (> (abs (- val true-val)) err))
(define (cneq a b)
  (or (> (abs (- (real-part a) (real-part b))) .001)
      (> (abs (- (imag-part a) (imag-part b))) .001)))

(define feql
  (lambda (a b)
    (if (null? a)
	(null? b)
	(if (null? b)
	    #f
	    (if (fneq (car a) (car b))
		#f
		(feql (cdr a) (cdr b)))))))
(define ffeql
  (lambda (a b)
    (if (list? a)
	(if (null? a)
	    (null? b)
	    (if (null? b)
		#f
		(if (fffneq (car a) (car b))
		    #f
		    (ffeql (cdr a) (cdr b)))))
	#f)))
(define fieql
  (lambda (a b)
    (if (null? a)
	(null? b)
	(if (null? b)
	    #f
	    (if (integer? (car a))
		(if (not (= (car a) (car b)))
		    #f
		    (if (fneq (car a) (car b))
			#f))
		(fieql (cdr a) (cdr b)))))))
(define fveql 
  (lambda (a b i)
    (if (null? b)
	#t
	(if (fneq (car b) (vct-ref a i))
	    #f
	    (fveql a (cdr b) (+ i 1))))))

(define (vequal v0 v1)
  (<= (vct-peak (vct-subtract! (vct-copy v0) v1)) .001))

(define (vvequal v0 v1)
  (<= (vct-peak (vct-subtract! (vct-copy v0) v1)) .00002))

(define (string-=? a b)
  (or (string=? a b)
      (let* ((alen (string-length a))
	     (blen (string-length b))
	     (j 0))
	(call-with-current-continuation
	 (lambda (return)
	   (do ((i 0 (1+ i)))
	       ((= i alen) (= j blen))
	     (let ((ac (string-ref a i))
		   (bc (string-ref b j)))
	       (if (char=? ac bc)
		   (set! j (1+ j))
		   (if (not (and (char=? ac #\-)	
				 (<= i (- alen 7))
				 (string=? (substring a i (+ i 6)) "-0.000")))
		       (if (and (char=? bc #\-)
				(<= j (- blen 7))
				(string=? (substring b j (+ j 6)) "-0.000"))
			   (begin
			     (set! j (1+ j))
			     (if (not (char=? ac (string-ref b j)))
				 (return #f)
				 (set! j (1+ j))))
			   (return #f)))))))))))

(define ran-state (seed->random-state (current-time)))
(define my-random
  (lambda (n)
    (if (= n 0) 
	0 ;sigh...
	(random n ran-state))))

;(define rs (lambda (n) (< (my-random 1.0) n)))
(define rs (lambda (n) #t))

(define timings (make-vector (+ total-tests 1) 0))

(snd-display ";;~A" (snd-version))
(define test-number -1)
(if (not (defined? 'before-test-hook)) (define before-test-hook (make-hook 1)))
(if (not (defined? 'after-test-hook)) (define after-test-hook (make-hook 1)))
(add-hook! before-test-hook (lambda (n)
			      (set! test-number n)
			      (if (and (> n 0) (number? (vector-ref timings (- n 1))))
				  (vector-set! timings (- n 1) (hundred (- (real-time) (vector-ref timings (- n 1))))))
			      (vector-set! timings n (real-time))
			      (snd-display ";test ~D" n)
			      (gc)
			      (set! (show-backtrace) #f)
					;(snd-display (gc-stats))
					;(if (file-exists? "memlog")
					;	 (system (format #f "cp memlog memlog.~D" (1- n))))
					;(if (defined? 'mem-report) (mem-report))
			      ))

;(add-hook! after-test-hook (lambda (n) (snd-display ";...~D" n)))
(add-hook! after-test-hook
	   (lambda (n)
	     (gc)
	     (if (not (null? (sounds))) 
		 (snd-display ";test ~D: open sounds: ~A" n (map short-file-name (sounds))))))


(define overall-start-time (real-time))
(snd-display ";~%~A~%" (strftime "%d-%b %H:%M %Z" (localtime (current-time))))

(define (log-mem tst) 
  (if (> tests 1) (snd-display ";test ~D " (1+ tst)))
  (if (and (> tests 50) (= (modulo tst 10) 0))  (if (defined? 'mem-report) (mem-report))))

(defmacro without-errors (func)
  `(catch #t 
	  (lambda ()
	    ,func)
	  (lambda args 
	    (car args))))

(if (not (provided? 'snd-hooks.scm)) (load "hooks.scm"))
(if (not (provided? 'snd-ws.scm)) (load "ws.scm"))

(define (reset-almost-all-hooks)
  (with-local-hook optimization-hook '() reset-all-hooks))

(define (list-p val)
  (and (list? val)
       (not (null? val))))

(define (arity-ok func args)
  (let ((arity (procedure-property func 'arity)))
    (and (list-p arity)
	 (>= args (car arity))
	 (or (caddr arity)
	     (<= args (+ (car arity) (cadr arity)))))))

(define (set-arity-ok func args)
  (let ((arity (if (procedure-with-setter? func)
		   (procedure-property (setter func) 'arity)
		   (procedure-property func 'arity))))
    (and (list-p arity)
	 (>= args (car arity))
	 (or (caddr arity)
	     (<= args (+ (car arity) (cadr arity)))))))

(if (and (> (length (script-args)) 0)
	 (> (script-arg) 0))
    (let ((arg (script-arg))
	  (args (script-args)))
      (if (not (string=? (list-ref args (1- arg)) "-l")) 
	  (snd-display ";script-args[~A]: ~A (~A)?" (1- arg) (list-ref args (1- arg)) args))
      (if (not (string=? (list-ref args arg) "snd-test")) 
	  (snd-display ";script-args[~A]: ~A (~A)?" arg (list-ref args arg) args))
      (if (> (length args) (1+ arg))
	  (begin
	    ;; test-number tests
	    (set! snd-test (string->number (list-ref args (1+ arg))))
	    (set! full-test (< snd-test 0))
	    (set! with-exit #t)
	    (set! (script-arg) (1+ arg))
	    (if (> (length (script-args)) (+ arg 2))
		(begin
		  (set! tests (string->number (list-ref args (+ arg 2))))
		  (set! (script-arg) (+ arg 2))))))))

(if (and (provided? 'snd-motif)
	 (provided? 'xm))
    (begin
      (if (not (provided? 'snd-event.scm)) (load "event.scm"))
      (if (not (provided? 'snd-snd-motif.scm)) (load "snd-motif.scm"))
      (if (not (provided? 'snd-popup.scm)) (load "popup.scm")))
    (if (and (provided? 'snd-gtk)
	     (provided? 'xg))
	(begin
	  (if (not (provided? 'snd-snd-gtk.scm)) (load "snd-gtk.scm"))
	  (if (not (provided? 'snd-gtk-popup.scm)) (load "gtk-popup.scm")))))

;(define widvardpy (make-variable-display "do-loop" "i*2" 'graph))


;;; ---------------- test 0: constants ----------------

(if (or full-test (= snd-test 0) (and keep-going (<= snd-test 0)))
    (letrec ((test-constants 
	      (lambda (lst)
		(if (not (null? lst))
		    (begin
		      (if (not (= (cadr lst) (caddr lst)))
			  (snd-display ";~A /= ~A (~A)~%"
				       (car lst) (cadr lst) (caddr lst)))
		      (test-constants (cdddr lst)))))))
      (run-hook before-test-hook 0)
      (if (or (not (null? (sounds)))
	      (not (null? (mixes)))
	      (not (null? (marks)))
	      (not (null? (regions))))
	  (snd-display ";start up: ~A ~A ~A ~A" (sounds) (mixes) (marks) (regions)))
      (test-constants
       (list
	'enved-amplitude enved-amplitude 0 
	'autocorrelation autocorrelation 3
	'bartlett-window bartlett-window 4 
	'blackman2-window blackman2-window 6 
	'blackman3-window blackman3-window 7 
	'blackman4-window blackman4-window 8
	'cauchy-window cauchy-window 12 
	'channels-combined channels-combined 1 
	'channels-separate channels-separate 0 
	'channels-superimposed channels-superimposed 2
	'connes-window connes-window 18
	'cursor-in-middle cursor-in-middle 3
	'cursor-in-view cursor-in-view 0 
	'cursor-on-left cursor-on-left 1 
	'cursor-on-right cursor-on-right 2 
	'dolph-chebyshev-window dolph-chebyshev-window 16
	'exponential-window exponential-window 9 
	'zoom-focus-active zoom-focus-active 2
	'zoom-focus-left zoom-focus-left 0
	'zoom-focus-middle zoom-focus-middle 3
	'zoom-focus-right zoom-focus-right 1 
	'fourier-transform fourier-transform 0 
	'gaussian-window gaussian-window 14 
	'graph-dots graph-dots 1
	'graph-dots-and-lines graph-dots-and-lines 3 
	'graph-filled graph-filled 2 
	'graph-lines graph-lines 0 
	'graph-lollipops graph-lollipops 4
	'hadamard-transform hadamard-transform 5
	'haar-transform haar-transform 6
	'hamming-window hamming-window 5
	'hann-window hann-window 1
	'hann-poisson-window hann-poisson-window 17
	'kaiser-window kaiser-window 11 
	'keyboard-no-action keyboard-no-action 4
	'cepstrum cepstrum 4
	'graph-once graph-once 0 
	'parzen-window parzen-window 3
	'poisson-window poisson-window 13
	'rectangular-window rectangular-window 0 
	'riemann-window riemann-window 10 
	'graph-as-sonogram graph-as-sonogram 1
	'graph-as-spectrogram graph-as-spectrogram 2 
        'graph-once graph-once 0
        'graph-as-wavogram graph-as-wavogram 3
	'enved-spectrum enved-spectrum 1
	'speed-control-as-float speed-control-as-float 0 
	'speed-control-as-ratio speed-control-as-ratio 1 
	'speed-control-as-semitone speed-control-as-semitone 2 
	'enved-srate enved-srate 2 
	'tukey-window tukey-window 15 
	'walsh-transform walsh-transform 2
	'wavelet-transform wavelet-transform 1
	'welch-window welch-window 2 
	'cursor-cross cursor-cross 0
	'cursor-line cursor-line 1
	'dont-normalize dont-normalize 0
	'envelope-linear envelope-linear 0
	'envelope-exponential envelope-exponential 1
	'normalize-by-channel normalize-by-channel 1
	'normalize-by-sound normalize-by-sound 2
	'normalize-globally normalize-globally 3
	'x-axis-in-samples x-axis-in-samples 1 
	'x-axis-in-beats x-axis-in-beats 3
	'x-axis-in-seconds x-axis-in-seconds 0 
	'x-axis-as-percentage x-axis-as-percentage 2
	'enved-add-point enved-add-point 0
	'enved-delete-point enved-delete-point 1
	'enved-move-point enved-move-point 2
	'time-graph time-graph 0
	'transform-graph transform-graph 1
	'lisp-graph lisp-graph 2
	'copy-context copy-context 0
	'cursor-context cursor-context 3
	'selection-context selection-context 2
	'mark-context mark-context 4
	'show-no-axes show-no-axes 0
	'show-all-axes show-all-axes 1
	'show-x-axis show-x-axis 2
	'show-all-axes-unlabelled show-all-axes-unlabelled 3
	'show-x-axis-unlabelled show-x-axis-unlabelled 4
	
	;; sndlib constants
	'mus-unsupported mus-unsupported 0
	'mus-next mus-next 1
	'mus-aifc mus-aifc 2
	'mus-riff mus-riff 3
	'mus-nist mus-nist 5
	'mus-raw mus-raw 11
	'mus-ircam mus-ircam 14
	'mus-aiff mus-aiff 48
	'mus-bicsf mus-bicsf 4
	'mus-voc mus-voc 9
	'mus-svx mus-svx 8
	'mus-soundfont mus-soundfont 25

	'mus-interp-none mus-interp-none 0
	'mus-interp-linear mus-interp-linear 1
	'mus-interp-sinusoidal mus-interp-sinusoidal 2
	'mus-interp-all-pass mus-interp-all-pass 3
	'mus-interp-lagrange mus-interp-lagrange 4
	'mus-interp-bezier mus-interp-bezier 5
	'mus-interp-hermite mus-interp-hermite 6

	'mus-unknown mus-unknown 0
	'mus-bshort mus-bshort 1
	'mus-lshort mus-lshort 10
	'mus-mulaw mus-mulaw 2
	'mus-alaw mus-alaw 6
	'mus-byte mus-byte 3
	'mus-ubyte mus-ubyte 7
	'mus-bfloat mus-bfloat 4
	'mus-lfloat mus-lfloat 12
	'mus-bint mus-bint 5
	'mus-lint mus-lint 11
	'mus-bintn mus-bintn 17
	'mus-lintn mus-lintn 18
	'mus-b24int mus-b24int 8
	'mus-l24int mus-l24int 16
	'mus-bdouble mus-bdouble 9
	'mus-ldouble mus-ldouble 13
	'mus-ubshort mus-ubshort 14
	'mus-ulshort mus-ulshort 15
	'mus-bfloat-unscaled mus-bfloat-unscaled 19
	'mus-lfloat-unscaled mus-lfloat-unscaled 20
	'mus-bdouble-unscaled mus-bdouble-unscaled 21
	'mus-ldouble-unscaled mus-ldouble-unscaled 22
	'mus-audio-default mus-audio-default 0
	'mus-audio-duplex-default mus-audio-duplex-default 1
	'mus-audio-line-out mus-audio-line-out 4
	'mus-audio-line-in mus-audio-line-in 5
	'mus-audio-microphone mus-audio-microphone 6
	'mus-audio-speakers mus-audio-speakers 7
	'mus-audio-dac-out mus-audio-dac-out 10
	'mus-audio-adat-in mus-audio-adat-in 2
	'mus-audio-aes-in mus-audio-aes-in 3
	'mus-audio-digital-in mus-audio-digital-in 8
	'mus-audio-digital-out mus-audio-digital-out 9
	'mus-audio-adat-out mus-audio-adat-out 11
	'mus-audio-aes-out mus-audio-aes-out 12
	'mus-audio-dac-filter mus-audio-dac-filter 13
	'mus-audio-mixer mus-audio-mixer 14
	'mus-audio-line1 mus-audio-line1 15
	'mus-audio-line2 mus-audio-line2 16
	'mus-audio-line3 mus-audio-line3 17
	'mus-audio-aux-input mus-audio-aux-input 18
	'mus-audio-cd mus-audio-cd 19
	'mus-audio-aux-output mus-audio-aux-output 20
	'mus-audio-spdif-in mus-audio-spdif-in 21
	'mus-audio-spdif-out mus-audio-spdif-out 22
	'mus-audio-amp mus-audio-amp 23
	'mus-audio-srate mus-audio-srate 24
	'mus-audio-channel mus-audio-channel 25
	'mus-audio-format mus-audio-format 26
	'mus-audio-port mus-audio-port 37
	'mus-audio-imix mus-audio-imix 27
	'mus-audio-igain mus-audio-igain 28
	'mus-audio-reclev mus-audio-reclev 29
	'mus-audio-pcm mus-audio-pcm 30
	'mus-audio-pcm2 mus-audio-pcm2 31
	'mus-audio-ogain mus-audio-ogain 32
	'mus-audio-line mus-audio-line 33
	'mus-audio-synth mus-audio-synth 34
	'mus-audio-bass mus-audio-bass 35
	'mus-audio-treble mus-audio-treble 36
	'mus-audio-direction mus-audio-direction 39
	'mus-audio-samples-per-channel mus-audio-samples-per-channel 38
	))
      
      (set! (region-graph-style) (region-graph-style))
      (if (not (equal? (region-graph-style) graph-lines))
	  (snd-display ";region-graph-style set def: ~A" (region-graph-style)))
      (set! (ask-before-overwrite) (ask-before-overwrite)) 
      (if (not (equal? (ask-before-overwrite) #f)) 
	  (snd-display ";ask-before-overwrite set def: ~A" (ask-before-overwrite)))
      (set! (audio-output-device) (audio-output-device))
      (if (not (equal? (audio-output-device)  0)) 
	  (snd-display ";audio-output-device set def: ~A" (audio-output-device)))
      (set! (auto-resize) (auto-resize))
      (if (not (equal? (auto-resize)  #t )) 
	  (snd-display ";auto-resize set def: ~A" (auto-resize)))
      (set! (auto-update) (auto-update))
      (if (not (equal? (auto-update)  #f)) 
	  (snd-display ";auto-update set def: ~A" (auto-update)))
      (set! (channel-style) (channel-style))
      (if (not (equal? (channel-style)  0 )) 
	  (snd-display ";channel-style set def: ~A" (channel-style)))
      (set! (color-cutoff) (color-cutoff))
      (if (fneq (color-cutoff)  0.003 )
	  (snd-display ";color-cutoff set def: ~A" (color-cutoff)))
      (set! (color-inverted) (color-inverted))
      (if (not (equal? (color-inverted)  #t)) 
	  (snd-display ";color-inverted set def: ~A" (color-inverted)))
      (set! (color-scale) (color-scale))
      (if (fneq (color-scale)  1.0 )
	  (snd-display ";color-scale set def: ~A" (color-scale)))
      (set! (auto-update-interval) (auto-update-interval))
      (if (fneq (auto-update-interval)  60.0 )
	  (snd-display ";auto-update-interval set def: ~A" (auto-update-interval)))
      (set! (cursor-update-interval) (cursor-update-interval))
      (if (fneq (cursor-update-interval)  0.05 )
	  (snd-display ";cursor-update-interval set def: ~A" (cursor-update-interval)))
      (set! (cursor-location-offset) (cursor-location-offset))
      (if (not (= (cursor-location-offset)  0))
	  (snd-display ";cursor-location-offset set def: ~A" (cursor-location-offset)))
      (set! (dac-combines-channels) (dac-combines-channels))
      (if (not (equal? (dac-combines-channels)  #t)) 
	  (snd-display ";dac-combines-channels set def: ~A" (dac-combines-channels)))
      (set! (dac-size) (dac-size))
      (if (not (equal? (dac-size)  256 )) 
	  (snd-display ";dac-size set def: ~A" (dac-size)))
      (set! (minibuffer-history-length) (minibuffer-history-length))
      (if (not (equal? (minibuffer-history-length)  8)) 
	  (snd-display ";minibuffer-history-length set def: ~A" (minibuffer-history-length)))
      (set! (data-clipped) (data-clipped))
      (if (not (equal? (data-clipped)  #f )) 
	  (snd-display ";data-clipped set def: ~A" (data-clipped)))
      (set! (default-output-chans) (default-output-chans))
      (if (not (equal? (default-output-chans)  1 )) 
	  (snd-display ";default-output-chans set def: ~A" (default-output-chans)))
      (set! (default-output-format) (default-output-format))
      (if (not (equal? (default-output-format) mus-bshort)) 
	  (snd-display ";default-output-format set def: ~A" (default-output-format)))
      (set! (default-output-srate) (default-output-srate))
      (if (not (equal? (default-output-srate)  22050 )) 
	  (snd-display ";default-output-srate set def: ~A" (default-output-srate)))
      (set! (default-output-type) (default-output-type))
      (if (not (equal? (default-output-type)  mus-next)) 
	  (snd-display ";default-output-type set def: ~A" (default-output-type)))
      (set! (dot-size) (dot-size))
      (if (not (equal? (dot-size)  1 )) 
	  (snd-display ";dot-size set def: ~A" (dot-size)))
      (set! (cursor-size) (cursor-size))
      (if (not (equal? (cursor-size)  15 )) 
	  (snd-display ";cursor-size set def: ~A" (cursor-size)))
      (set! (cursor-style) (cursor-style))
      (if (not (equal? (cursor-style)  cursor-cross )) 
	  (snd-display ";cursor-style set def: ~A" (cursor-style)))
      (set! (emacs-style-save-as) (emacs-style-save-as))
      (if (not (equal? (emacs-style-save-as)  #f)) 
	  (snd-display ";emacs-style-save-as set def: ~A" (emacs-style-save-as)))
      (set! (enved-base) (enved-base))
      (if (fneq (enved-base)  1.0 )
	  (snd-display ";enved-base set def: ~A" (enved-base)))
      (set! (enved-clip?) (enved-clip?))
      (if (not (equal? (enved-clip?)  #f )) 
	  (snd-display ";enved-clip? set def: ~A" (enved-clip?)))
      (set! (enved-filter) (enved-filter))
      (if (not (equal? (enved-filter) #t)) 
	  (snd-display ";enved-filter set def: ~A" (enved-filter)))
      (set! (enved-filter-order) (enved-filter-order))
      (if (not (equal? (enved-filter-order)  40)) 
	  (snd-display ";enved-filter-order set def: ~A" (enved-filter-order)))
      (set! (enved-in-dB) (enved-in-dB))
      (if (not (equal? (enved-in-dB)  #f )) 
	  (snd-display ";enved-in-dB set def: ~A" (enved-in-dB)))
      (set! (enved-style) (enved-style))
      (if (not (equal? (enved-style)  envelope-linear )) 
	  (snd-display ";enved-style set def: ~A" (enved-style)))
      (set! (enved-power) (enved-power))
      (if (fneq (enved-power)  3.0)
	  (snd-display ";enved-power set def: ~A" (enved-power)))
      (set! (enved-target) (enved-target))
      (if (not (equal? (enved-target)  0 )) 
	  (snd-display ";enved-target set def: ~A" (enved-target)))
      (set! (enved-wave?) (enved-wave?))
      (if (not (equal? (enved-wave?)  #f )) 
	  (snd-display ";enved-wave? set def: ~A" (enved-wave?)))
      (if with-gui
	  (begin
	    (set! (enved-envelope) (enved-envelope))
	    (if (not (equal? (enved-envelope)  '())) 
		(snd-display ";enved-envelope set def: ~A" (enved-envelope)))))
      (set! (eps-file) (eps-file))
      (if (not (equal? (eps-file)  "snd.eps" )) 
	  (snd-display ";eps-file set def: ~A" (eps-file)))
      (set! (eps-bottom-margin) (eps-bottom-margin))
      (if (fneq (eps-bottom-margin)  0.0)
	  (snd-display ";eps-bottom-margin set def: ~A" (eps-bottom-margin)))
      (set! (eps-left-margin) (eps-left-margin))
      (if (fneq (eps-left-margin)  0.0)
	  (snd-display ";eps-left-margin set def: ~A" (eps-left-margin)))
      (set! (eps-size) (eps-size))
      (if (fneq (eps-size)  1.0)
	  (snd-display ";eps-size set def: ~A" (eps-size)))
      (set! (fft-window-beta) (fft-window-beta))
      (if (fneq (fft-window-beta)  0.0 )
	  (snd-display ";fft-window-beta set def: ~A" (fft-window-beta)))
      (set! (fft-log-frequency) (fft-log-frequency))
      (if (not (equal? (fft-log-frequency)  #f )) 
	  (snd-display ";fft-log-frequency set def: ~A" (fft-log-frequency)))
      (set! (fft-log-magnitude) (fft-log-magnitude))
      (if (not (equal? (fft-log-magnitude)  #f )) 
	  (snd-display ";fft-log-magnitude set def: ~A" (fft-log-magnitude)))
      (set! (transform-size) (transform-size))
      (if (not (equal? (transform-size)  512 )) 
	  (snd-display ";transform-size set def: ~A" (transform-size)))
      (set! (transform-graph-type) (transform-graph-type))
      (if (not (equal? (transform-graph-type) 0))
	  (snd-display ";transform-graph-type set def: ~A" (transform-graph-type)))
      (set! (fft-window) (fft-window))
      (if (not (equal? (fft-window)  6 )) 
	  (snd-display ";fft-window set def: ~A" (fft-window)))
      (set! (graph-cursor) (graph-cursor))
      (if (not (equal? (graph-cursor)  34)) 
	  (snd-display ";graph-cursor set def: ~A" (graph-cursor)))
      (set! (graph-style) (graph-style))
      (if (not (equal? (graph-style)  graph-lines )) 
	  (snd-display ";graph-style set def: ~A" (graph-style)))
      (set! (graphs-horizontal) (graphs-horizontal))
      (if (not (equal? (graphs-horizontal)  #t)) 
	  (snd-display ";graphs-horizontal set def: ~A" (graphs-horizontal)))
      (set! (html-dir) (html-dir))
      (if (not (equal? (html-dir)  ".")) 
	  (snd-display ";html-dir set def: ~A" (html-dir)))
      (set! (html-program) (html-program))
      (if (not (equal? (html-program)  "mozilla")) 
	  (snd-display ";html-program set def: ~A" (html-program)))
      (set! (just-sounds) (just-sounds))
      (if (not (equal? (just-sounds)  #f)) 
	  (snd-display ";just-sounds set def: ~A" (just-sounds)))
      (set! (listener-prompt) (listener-prompt))
      (if (not (equal? (listener-prompt)  ">" )) 
	  (snd-display ";listener-prompt set def: ~A" (listener-prompt)))
      (set! (max-transform-peaks) (max-transform-peaks))
      (if (not (equal? (max-transform-peaks)  100)) 
	  (snd-display ";max-transform-peaks set def: ~A" (max-transform-peaks)))
      (set! (max-transform-peaks) -123)
      (if (not (equal? (max-transform-peaks) 100)) 
	  (snd-display ";max-transform-peaks set -123: ~A" (max-transform-peaks)))
      (set! (max-regions) (max-regions))
      (if (not (equal? (max-regions)  16 )) 
	  (snd-display ";max-regions set def: ~A" (max-regions)))
      (set! (min-dB) (min-dB))
      (if (fneq (min-dB)  -60.0 )
	  (snd-display ";min-dB set def: ~A" (min-dB)))
      (set! (selection-creates-region) (selection-creates-region))
      (if (not (equal? (selection-creates-region)  #t )) 
	  (snd-display ";selection-creates-region set def: ~A" (selection-creates-region)))
      (set! (transform-normalization) (transform-normalization))
      (if (not (equal? (transform-normalization)  normalize-by-channel)) 
	  (snd-display ";transform-normalization set def: ~A" (transform-normalization)))
      (set! (previous-files-sort) (previous-files-sort))
      (if (not (equal? (previous-files-sort)  0 )) 
	  (snd-display ";previous-files-sort set def: ~A" (previous-files-sort)))
      (set! (print-length) (print-length))
      (if (not (equal? (print-length)  12 )) 
	  (snd-display ";print-length set def: ~A" (print-length)))
      (set! (recorder-autoload) (recorder-autoload))
      (if (not (equal? (recorder-autoload)  #f)) 
	  (snd-display ";recorder-autoload set def: ~A" (recorder-autoload)))
      (set! (recorder-buffer-size) (recorder-buffer-size))
      (if (not (equal? (recorder-buffer-size)  4096 )) 
	  (snd-display ";recorder-buffer-size set def: ~A" (recorder-buffer-size)))
      (set! (recorder-file) (recorder-file))
      (if (not (equal? (recorder-file)  #f )) 
	  (snd-display ";recorder-file set def: ~A" (recorder-file)))
      (set! (recorder-max-duration) (recorder-max-duration))
      (if (fneq (recorder-max-duration)  1000000.0)
	  (snd-display ";recorder-max-duration set def: ~A" (recorder-max-duration)))
      (set! (recorder-out-chans) (recorder-out-chans))
      (if (not (equal? (recorder-out-chans)  2 )) 
	  (snd-display ";recorder-out-chans set def: ~A" (recorder-out-chans)))
      (set! (recorder-srate) (recorder-srate))
      (if (not (equal? (recorder-srate)  22050 )) 
	  (snd-display ";recorder-srate set def: ~A" (recorder-srate)))
      (set! (recorder-trigger) (recorder-trigger))
      (if (fneq (recorder-trigger)  0.0)
	  (snd-display ";recorder-trigger set def: ~A" (recorder-trigger)))
      (set! (save-state-file) (save-state-file))
      (if (not (equal? (save-state-file)  "saved-snd.scm" )) 
	  (snd-display ";save-state-file set def: ~A" (save-state-file)))
      (set! (show-axes) (show-axes))
      (if (not (equal? (show-axes)  1)) 
	  (snd-display ";show-axes set def: ~A" (show-axes)))
      (set! (show-transform-peaks) (show-transform-peaks))
      (if (not (equal? (show-transform-peaks)  #f )) 
	  (snd-display ";show-transform-peaks set def: ~A" (show-transform-peaks)))
      (set! (show-indices) (show-indices))
      (if (not (equal? (show-indices)  #f)) 
	  (snd-display ";show-indices set def: ~A" (show-indices)))
      (set! (show-backtrace) (show-backtrace))
      (if (not (equal? (show-backtrace)  #f)) 
	  (snd-display ";show-backtrace set def: ~A" (show-backtrace)))
      (set! (show-marks) (show-marks))
      (if (not (equal? (show-marks)  #t )) 
	  (snd-display ";show-marks set def: ~A" (show-marks)))
      (set! (show-mix-waveforms) (show-mix-waveforms))
      (if (not (equal? (show-mix-waveforms)  #t)) 
	  (snd-display ";show-mix-waveforms set def: ~A" (show-mix-waveforms)))
      (set! (show-selection-transform) (show-selection-transform))
      (if (not (equal? (show-selection-transform)  #f )) 
	  (snd-display ";show-selection-transform set def: ~A" (show-selection-transform)))
      (set! (show-y-zero) (show-y-zero))
      (if (not (equal? (show-y-zero)  #f )) 
	  (snd-display ";show-y-zero set def: ~A" (show-y-zero)))
      (set! (show-grid) (show-grid))
      (if (not (equal? (show-grid)  #f )) 
	  (snd-display ";show-grid set def: ~A" (show-grid)))
      (set! (sinc-width) (sinc-width))
      (if (not (equal? (sinc-width)  10 )) 
	  (snd-display ";sinc-width set def: ~A" (sinc-width)))
      (set! (spectro-cutoff) (spectro-cutoff))
      (if (fneq (spectro-cutoff)  1.0)
	  (snd-display ";spectro-cutoff set def: ~A" (spectro-cutoff)))
      (set! (spectro-hop) (spectro-hop))
      (if (not (equal? (spectro-hop)  4 )) 
	  (snd-display ";spectro-hop set def: ~A" (spectro-hop)))
      (set! (spectro-start) (spectro-start))
      (if (fneq (spectro-start)  0.0 )
	  (snd-display ";spectro-start set def: ~A" (spectro-start)))
      (set! (spectro-x-angle) (spectro-x-angle))
      (if (fneq (spectro-x-angle)  (if (provided? 'gl) 300.0 90.0))
	  (snd-display ";spectro-x-angle set def: ~A" (spectro-x-angle)))
      (set! (spectro-x-scale) (spectro-x-scale))
      (if (fneq (spectro-x-scale) (if (provided? 'gl) 1.5 1.0))
	  (snd-display ";spectro-x-scale set def: ~A" (spectro-x-scale)))
      (set! (spectro-y-angle) (spectro-y-angle))
      (if (fneq (spectro-y-angle) (if (provided? 'gl) 320.0 0.0))
	  (snd-display ";spectro-y-angle set def: ~A" (spectro-y-angle)))
      (set! (spectro-y-scale) (spectro-y-scale))
      (if (fneq (spectro-y-scale)  1.0 )
	  (snd-display ";spectro-y-scale set def: ~A" (spectro-y-scale)))
      (set! (spectro-z-angle) (spectro-z-angle))
      (if (fneq (spectro-z-angle) (if (provided? 'gl) 0.0 358.0))
	  (snd-display ";spectro-z-angle set def: ~A" (spectro-z-angle)))
      (set! (spectro-z-scale) (spectro-z-scale))
      (if (fneq (spectro-z-scale) (if (provided? 'gl) 1.0 0.1))
	  (snd-display ";spectro-z-scale set def: ~A" (spectro-z-scale)))
      (set! (temp-dir) (temp-dir))
      (if (not (equal? (temp-dir)  #f )) 
	  (snd-display ";temp-dir set def: ~A" (temp-dir)))
      (set! (ladspa-dir) (ladspa-dir))
      (if (not (equal? (ladspa-dir)  #f )) 
	  (snd-display ";ladspa-dir set def: ~A" (ladspa-dir)))
      (set! (tiny-font) (tiny-font))
      (if (and (not (equal? (tiny-font) "6x12"))
	       (not (equal? (tiny-font) "Monospace 8")))
	  (snd-display ";tiny-font set def: ~A" (tiny-font)))
      (set! (transform-type) (transform-type))
      (if (not (equal? (transform-type)  0 )) 
	  (snd-display ";transform-type set def: ~A" (transform-type)))
      (set! (trap-segfault) (trap-segfault))
      (if (not (equal? (trap-segfault)  #f)) 
	  (snd-display ";trap-segfault set def: ~A" (trap-segfault)))
      (set! (optimization) (optimization))
      (if (not (equal? (optimization) 0)) 
	  (snd-display ";optimization set def: ~A" (optimization)))
      (set! (verbose-cursor) (verbose-cursor))
      (if (not (equal? (verbose-cursor)  #f)) 
	  (snd-display ";verbose-cursor set def: ~A" (verbose-cursor)))
      (set! (vu-font) (vu-font))
      (if (not (equal? (vu-font)  #f )) 
	  (snd-display ";vu-font set def: ~A" (vu-font)))
      (set! (vu-font-size) (vu-font-size))
      (if (fneq (vu-font-size)  1.0 )
	  (snd-display ";vu-font-size set def: ~A" (vu-font-size)))
      (set! (vu-size) (vu-size))
      (if (fneq (vu-size)  1.0 )
	  (snd-display ";vu-size set def: ~A" (vu-size)))
      (set! (wavelet-type) (wavelet-type))
      (if (not (equal? (wavelet-type)  0 )) 
	  (snd-display ";wavelet-type set def: ~A" (wavelet-type)))
      (set! (time-graph-type) (time-graph-type))
      (if (not (equal? (time-graph-type)  graph-once)) 
	  (snd-display ";time-graph-type set def: ~A" (time-graph-type)))
      (set! (wavo-hop) (wavo-hop))
      (if (not (equal? (wavo-hop)  3 )) 
	  (snd-display ";wavo-hop set def: ~A" (wavo-hop)))
      (set! (wavo-trace) (wavo-trace))
      (if (not (equal? (wavo-trace)  64 )) 
	  (snd-display ";wavo-trace set def: ~A" (wavo-trace)))
      (set! (x-axis-style) (x-axis-style))
      (if (not (equal? (x-axis-style)  0 )) 
	  (snd-display ";x-axis-style set def: ~A" (x-axis-style)))
      (set! (beats-per-minute) (beats-per-minute))
      (if (fneq (beats-per-minute)  60.0 )
	  (snd-display ";beats-per-minute set def: ~A" (beats-per-minute)))
      (set! (zero-pad) (zero-pad))
      (if (not (equal? (zero-pad)  0)) 
	  (snd-display ";zero-pad set def: ~A" (zero-pad)))
      (set! (zero-pad) -123)
      (if (not (equal? (zero-pad)  0)) 
	  (snd-display ";zero-pad set -123: ~A" (zero-pad)))
      (if (not (equal? (zero-pad #t #t) '()))
	  (snd-display ";zero-pad #t: ~A" (zero-pad #t #t)))
      (set! (zoom-focus-style) (zoom-focus-style))
      (if (not (equal? (zoom-focus-style)  2 )) 
	  (snd-display ";zoom-focus-style set def: ~A" (zoom-focus-style)))
      (set! (mix-waveform-height) (mix-waveform-height))
      (if (not (equal? (mix-waveform-height)  20 )) 
	  (snd-display ";mix-waveform-height set def: ~A" (mix-waveform-height)))
      (set! (mix-tag-width) (mix-tag-width))
      (if (not (equal? (mix-tag-width)  6)) 
	  (snd-display ";mix-tag-width set def: ~A" (mix-tag-width)))
      (set! (mix-tag-height) (mix-tag-height))
      (if (not (equal? (mix-tag-height)  14)) 
	  (snd-display ";mix-tag-height set def: ~A" (mix-tag-height)))
      (set! (audio-output-device) (audio-output-device))
      (if (not (equal? (audio-output-device)  0 )) 
	  (snd-display ";audio-output-device set def: ~A" (audio-output-device)))
      
      (if (not (provided? 'snd-gtk))
	  (for-each
	   (lambda (func name)
	     (let ((val (func)))
	       (set! (func) "8x123")
	       (if (or (not (string? (func)))
		       (not (string=? val (func))))
		   (snd-display ";set ~A to bogus value: ~A ~A" name val (func)))))
	   (list axis-label-font axis-numbers-font tiny-font peaks-font bold-peaks-font)
	   (list 'axis-label-font 'axis-numbers-font 'tiny-font 'peaks-font 'bold-peaks-font)))
      (run-hook after-test-hook 0)
      ))


;;; ---------------- test 1: default values ----------------
(if (or full-test (= snd-test 1) (and keep-going (<= snd-test 1)))
    (letrec ((test-defaults
	      (lambda (lst)
		(if (not (null? lst))
		    (begin
		      (if (not (equal? (cadr lst)  (caddr lst)))
			  (if (and (number? (caddr lst))
				   (inexact? (caddr lst)))
			      (if (fneq (cadr lst) (caddr lst))
				  (snd-display ";~A /= ~A (~A)" (car lst) (caddr lst) (cadr lst)))
			      (snd-display ";~A /= ~A (~A)" (car lst) (caddr lst) (cadr lst))))
		      (test-defaults (cdddr lst)))))))
      (run-hook before-test-hook 1)
      (for-each close-sound (sounds)) ; in case others opened elsewhere
      (test-defaults
       (list
	'amp-control (without-errors (amp-control)) 'no-such-sound
	'amp-control-bounds (cadr (amp-control-bounds)) 8.0
	'ask-before-overwrite (ask-before-overwrite) #f 
	'audio-output-device (audio-output-device) 0
	'auto-resize (auto-resize) #t 
	'auto-update (auto-update) #f
	'channel-style (channel-style) 0 
	'color-cutoff (color-cutoff) 0.003 
	'color-inverted (color-inverted) #t
	'color-scale (color-scale) 1.0 
	'colormap (colormap) (if (provided? 'gl) 2 0)
	'contrast-control (without-errors (contrast-control)) 'no-such-sound
	'contrast-control-bounds (cadr (contrast-control-bounds)) 10.0
	'contrast-control-amp (contrast-control-amp) 1.0
	'contrast-control? (without-errors (contrast-control?)) 'no-such-sound
	'auto-update-interval (auto-update-interval) 60.0 
	'cursor-update-interval (cursor-update-interval) 0.05
	'cursor-location-offset (cursor-location-offset) 0
	'cursor-follows-play (cursor-follows-play) #f
	'cursor-size (cursor-size) 15
	'cursor-style (cursor-style) cursor-cross
	'dac-combines-channels (dac-combines-channels) #t
	'emacs-style-save-as (emacs-style-save-as) #f
	'dac-size (dac-size) 256 
	'minibuffer-history-length (minibuffer-history-length) 8
	'data-clipped (data-clipped) #f 
	'default-output-chans (default-output-chans) 1 
	'default-output-format (default-output-format) mus-bshort
	'default-output-srate (default-output-srate) 22050 
	'default-output-type (default-output-type) mus-next
	'dot-size (dot-size) 1 
	'enved-base (enved-base) 1.0 
	'enved-clip? (enved-clip?) #f 
	'enved-filter-order (enved-filter-order) 40
	'enved-filter (enved-filter) #t
	'enved-in-dB (enved-in-dB) #f 
	'enved-style (enved-style) envelope-linear
	'enved-power (enved-power) 3.0
	'enved-target (enved-target) 0 
	'enved-wave? (enved-wave?) #f 
	'enved-envelope (enved-envelope) '()
	'eps-file (eps-file) "snd.eps" 
	'eps-bottom-margin (eps-bottom-margin) 0.0
	'eps-left-margin (eps-left-margin) 0.0
	'eps-size (eps-size) 1.0
	'expand-control (without-errors (expand-control)) 'no-such-sound
	'expand-control-bounds (cadr (expand-control-bounds)) 20.0
	'expand-control-hop (expand-control-hop) 0.05
	'expand-control-jitter (expand-control-jitter) 0.1
	'expand-control-length (expand-control-length) 0.15
	'expand-control-ramp (expand-control-ramp) 0.4
	'expand-control? (without-errors (expand-control?)) 'no-such-sound
	'fft-window-beta (fft-window-beta) 0.0 
	'fft-log-frequency (fft-log-frequency) #f 
	'fft-log-magnitude (fft-log-magnitude) #f 
	'transform-size (transform-size) 512
	'transform-graph-type (transform-graph-type) 0
	'fft-window (fft-window) 6 
	'transform-graph? (without-errors (transform-graph?)) 'no-such-sound
	'filter-control-in-dB (filter-control-in-dB) #f
	'filter-control-coeffs (without-errors (filter-control-coeffs)) 'no-such-sound
	'filter-control-envelope (without-errors (filter-control-envelope)) 'no-such-sound
	'filter-control-in-hz (filter-control-in-hz) #f
	'filter-control-order (filter-control-order) 20
	'filter-control? (without-errors (filter-control?)) 'no-such-sound
	'graph-cursor (graph-cursor) 34
	'graph-style (graph-style) graph-lines
	'html-dir (html-dir) "."
	'html-program (html-program) "mozilla"
	'lisp-graph? (without-errors (lisp-graph?)) 'no-such-sound
	'graphs-horizontal (graphs-horizontal) #t
	'just-sounds (just-sounds) #f
	'listener-prompt (listener-prompt) ">" 
	'max-transform-peaks (max-transform-peaks) 100
	'max-regions (max-regions) 16 
	'min-dB (min-dB) -60.0 
	'selection-creates-region (selection-creates-region) #t 
	'transform-normalization (transform-normalization) normalize-by-channel
	'previous-files-sort (previous-files-sort) 0 
	'print-length (print-length) 12 
	'read-only (without-errors (read-only)) 'no-such-sound
	'recorder-autoload (recorder-autoload) #f
	'recorder-buffer-size (recorder-buffer-size) 4096 
	'recorder-file (recorder-file) #f 
	'recorder-max-duration (recorder-max-duration) 1000000.0
	'recorder-out-chans (recorder-out-chans) 2 
	'recorder-srate (recorder-srate) 22050 
	'recorder-trigger (recorder-trigger) 0.0
	'region-graph-style (region-graph-style) graph-lines
	'reverb-control-feedback (reverb-control-feedback) 1.09
	'reverb-control-length (without-errors (reverb-control-length)) 'no-such-sound
	'reverb-control-length-bounds (cadr (reverb-control-length-bounds)) 5.0
	'reverb-control-lowpass (reverb-control-lowpass) 0.7
	'reverb-control-scale (without-errors (reverb-control-scale)) 'no-such-sound
	'reverb-control-scale-bounds (cadr (reverb-control-scale-bounds)) 4.0
	'reverb-control? (without-errors (reverb-control?)) 'no-such-sound
	'save-state-file (save-state-file) "saved-snd.scm" 
	'show-axes (show-axes) 1
	'show-transform-peaks (show-transform-peaks) #f 
	'show-indices (show-indices) #f
	'show-backtrace (show-backtrace) #f
	'show-marks (show-marks) #t 
	'show-mix-waveforms (show-mix-waveforms) #t
	'show-selection-transform (show-selection-transform) #f 
	'show-y-zero (show-y-zero) #f 
	'show-grid (show-grid) #f 
	'show-controls (show-controls) #f
	'sinc-width (sinc-width) 10 
	'spectro-cutoff (spectro-cutoff) 1.0
	'spectro-hop (spectro-hop) 4 
	'spectro-start (spectro-start) 0.0 
	'spectro-x-angle (spectro-x-angle) (if (provided? 'gl) 300.0 90.0)
	'spectro-x-scale (spectro-x-scale) (if (provided? 'gl) 1.5 1.0)
	'spectro-y-angle (spectro-y-angle) (if (provided? 'gl) 320.0 0.0)
	'spectro-y-scale (spectro-y-scale) 1.0 
	'spectro-z-angle (spectro-z-angle) (if (provided? 'gl) 0.0 358.0)
	'spectro-z-scale (spectro-z-scale) (if (provided? 'gl) 1.0 0.1)
	'speed-control (without-errors (speed-control)) 'no-such-sound
	'speed-control-bounds (cadr (speed-control-bounds)) 20.0
	'sync (without-errors (sync)) 'no-such-sound
	'temp-dir (temp-dir) #f 
	'ladspa-dir (ladspa-dir) #f 
	'tiny-font (tiny-font) "6x12"
	'transform-type (transform-type) 0 
	'optimization (optimization) 0
	'verbose-cursor (verbose-cursor) #f
	'vu-font (vu-font) #f 
	'vu-font-size (vu-font-size) 1.0 
	'vu-size (vu-size) 1.0 
	'wavelet-type (wavelet-type) 0 
	'time-graph? (without-errors (time-graph?)) 'no-such-sound
	'time-graph-type (time-graph-type) graph-once
	'wavo-hop (wavo-hop) 3 
	'wavo-trace (wavo-trace) 64 
	'x-axis-style (x-axis-style) 0 
	'beats-per-minute (beats-per-minute) 60.0
	'zero-pad (zero-pad) 0
	'zoom-focus-style (zoom-focus-style) 2 
	'mix-waveform-height (mix-waveform-height) 20 
	'mix-tag-width (mix-tag-width) 6
	'mix-tag-height (mix-tag-height) 14
	'with-mix-tags (with-mix-tags) #t
	'with-relative-panes (with-relative-panes) #t
	'audio-output-device (audio-output-device) 0 
	))
      (if (not snd-remember-paths) (snd-display ";snd-remember-paths?"))
      (if memo-sound (snd-display ";memo-sound: ~A" memo-sound))
      (run-hook after-test-hook 1)
      ))


;;; ---------------- test 2: headers ----------------
(if (or full-test (= snd-test 2) (and keep-going (<= snd-test 2)))
    (if (string? sf-dir)
	(letrec ((test-headers
		  (lambda (base-files)
		    (if (not (null? base-files))
			(let ((testf (car base-files)))
			  (let ((file (string-append sf-dir (list-ref testf 0))))
			    (if (file-exists? file)
				(begin
				  (if (not (equal? (mus-sound-chans file) (list-ref testf 1)))
				      (snd-display ";~A: chans ~A /= ~A" 
						   (list-ref testf 0) 
						   (mus-sound-chans file) 
						   (list-ref testf 1)))
				  (if (not (equal? (mus-sound-srate file) (list-ref testf 2)))
				      (snd-display ";~A: srate ~A /= ~A" 
						   (list-ref testf 0) 
						   (mus-sound-srate file) 
						   (list-ref testf 2)))
				  (if (fneq (mus-sound-duration file) (list-ref testf 3))
				      (snd-display ";~A: duration ~A /= ~A" 
						   (list-ref testf 0)
						   (mus-sound-duration file) 
						   (list-ref testf 3)))
				  (if (and (not (= (mus-sound-data-format file) mus-unknown))
					   (not (= (mus-sound-header-type file) 27)) ; bogus header on test case (comdisco)
					   (< (+ (mus-sound-length file) 1)
					      (* (mus-sound-datum-size file) (mus-sound-duration file)
						 (mus-sound-srate file) (mus-sound-chans file))))
				      (snd-display ";mus-sound-length ~A: ~A (~A)" file
						   (mus-sound-length file)
						   (* (mus-sound-duration file) (mus-sound-srate file) 
						      (mus-sound-chans file) (mus-sound-datum-size file))))
				  (if (fneq (/ (mus-sound-frames file) (mus-sound-srate file)) (mus-sound-duration file))
				      (snd-display ";mus-sound-frames ~A: ~A (~A ~A)" file
						   (mus-sound-frames file)
						   (mus-sound-duration file)
						   (/ (mus-sound-frames file) (mus-sound-srate file))))
				  (if (> (abs (- (mus-sound-frames file) (/ (mus-sound-samples file) (mus-sound-chans file)))) 1)
				      (snd-display ";mus-sound-samples ~A: ~A ~A" file
						   (mus-sound-samples file)
						   (* (mus-sound-frames file) (mus-sound-chans file))))
				  (if (not (equal? (mus-header-type-name (mus-sound-header-type file)) (list-ref testf 4)))
				      (snd-display ";~A: type ~A /= ~A" 
						   (list-ref testf 0) 
						   (mus-header-type-name (mus-sound-header-type file))
						   (list-ref testf 4)))
				  (if (not (equal? (mus-data-format-name (mus-sound-data-format file)) (list-ref testf 5)))
				      (snd-display ";~A: type ~A /= ~A"
						   (list-ref testf 0) 
						   (mus-data-format-name (mus-sound-data-format file)) 
						   (list-ref testf 5)))
				  (let ((lst (mus-sound-loop-info file)))
				    (if (> (length testf) 6)
					(begin
					  (if (not (equal? (car lst) (list-ref testf 6))) 
					      (snd-display ";~A: loop start: ~A" (car lst) (list-ref testf 6)))
					  (if (not (equal? (cadr lst) (list-ref testf 7))) 
					      (snd-display ";~A: loop end: ~A" (cadr lst) (list-ref testf 7))))
					(if (not (null? lst))
					    (snd-display ";~A thinks it has loop info: ~A" file lst)))))
				(snd-display ";~A missing?" file))
			    (test-headers (cdr base-files))))))))
	  (run-hook before-test-hook 2)
	  ;; need to make sure raw defaults are consistent with following tests
	  (let ((ind (open-raw-sound :file (string-append sf-dir "addf8.nh") :channels 2 :srate 44100 :data-format mus-bshort)))
	    (if (sound? ind) (close-sound ind)))
	  (catch #t
		 (lambda ()
		   (if (not (= (mus-sound-header-type (string-append sf-dir "midi60.mid")) -1)) (snd-display ";midi60?")))
		 (lambda args args))
	  (test-headers
	   (list
	    (list "8svx-8.snd" 1 22050 1.88766443729401 "SVX8" "signed byte (8 bits)")
	    (list "Fnonull.aif" 1 8000 0.00112499995157123 "AIFC" "mulaw (8 bits)")
	    (list "Pmiscck.aif" 1 8000 0.00112499995157123 "AIFC" "mulaw (8 bits)")
	    (list "Pmiscck.wav" 1 8000 0.00112499995157123 "RIFF" "mulaw (8 bits)")
;	    (list "Pnossnd.aif" 1 8000 0.0 "AIFC" "mulaw (8 bits)")
	    (list "Poffset.aif" 1 8000 0.00112499995157123 "AIFC" "mulaw (8 bits)")
	    (list "Porder.aif" 1 8000 0.00112499995157123 "AIFC" "mulaw (8 bits)")
	    (list "Ptjunk.aif" 1 8000 0.00112499995157123 "AIFC" "mulaw (8 bits)")
	    (list "Ptjunk.wav" 1 8000 0.00112499995157123 "RIFF" "mulaw (8 bits)")
	    (list "SINE24-S.WAV" 2 44100 2.0 "RIFF" "little endian int (24 bits)")
	    (list "a1.asf" 1 16000 3.73656249046326 "asf" "unknown")
	    (list "a2.asf" 1 8000 4.63062477111816 "asf" "unknown")
	    (list "addf8.afsp" 1 8000 2.9760000705719 "Sun" "big endian short (16 bits)")
	    (list "addf8.d" 1 8000 2.9760000705719 "SPPACK" "big endian short (16 bits)")
	    (list "addf8.dwd" 1 8000 2.9760000705719 "DiamondWare" "little endian short (16 bits)")
	    (list "addf8.nh" 2 44100 0.269931972026825 "raw (no header)" "big endian short (16 bits)")
	    (list "addf8.sd" 1 8000 2.9760000705719 "ESPS" "big endian short (16 bits)")
	    (list "addf8.sf_mipseb" 1 8000 2.9760000705719 "IRCAM" "big endian short (16 bits)")
	    (list "addf8.sf_sun" 1 8000 2.9760000705719 "IRCAM" "big endian short (16 bits)")
	    (list "addf8.sf_vax_b" 1 8000 2.9760000705719 "IRCAM" "big endian short (16 bits)")
	    (list "addf8.wav" 1 8000 2.9760000705719 "RIFF" "little endian short (16 bits)")
	    (list "aebass.krz" 1 44100 3.0 "Kurzweil 2000" "big endian short (16 bits)")
	    (list "aiff-16.snd" 2 44100 0.746666669845581 "AIFF" "big endian short (16 bits)")
	    (list "aiff-8.snd" 2 44100 0.746666669845581 "AIFF" "signed byte (8 bits)")
	    (list "alaw.aifc" 1 44100 0.0367800444364548 "AIFC" "alaw (8 bits)")
	    (list "alaw.wav" 1 11025 8.70666694641113 "RIFF" "alaw (8 bits)")
	    (list "astor_basia.mp2" 2 44100 1.02179133892059 "raw (no header)" "big endian short (16 bits)")
	    (list "c.asf" 1 8000 21.3681259155273 "asf" "unknown")
	    (list "ce-c3.w02" 1 33000 3.88848495483398 "TX-16W" "unknown")
	    (list "ce-c4.w03" 1 33000 2.91618180274963 "TX-16W" "unknown")
	    (list "ce-d2.w01" 1 33000 3.46439385414124 "TX-16W" "unknown")
	    (list "clbonef.wav" 1 22050 2.57832193374634 "RIFF" "little endian float (32 bits)")
	    (list "cranker.krz" 1 44100 3.48267579078674 "Kurzweil 2000" "big endian short (16 bits)")
	    (list "d40130.aif" 1 10000 0.100000001490116 "AIFF" "big endian short (16 bits)")
	    (list "d40130.au" 1 10000 0.100000001490116 "Sun" "big endian short (16 bits)")
	    (list "d40130.dsf" 1 8000 0.125 "Delusion" "little endian short (16 bits)")
	    (list "d40130.fsm" 1 8000 0.125249996781349 "Farandole" "little endian short (16 bits)")
	    (list "d40130.iff" 1 10000 0.100000001490116 "SVX8" "signed byte (8 bits)")
	    (list "d40130.pat" 1 10000 0.100000001490116 "Gravis Ultrasound patch" "little endian short (16 bits)")
	    (list "d40130.sds" 1 10000 0.100000001490116 "MIDI sample dump" "unknown")
	    (list "d40130.sdx" 1 10000 0.100000001490116 "Sample dump" "unsigned little endian short (16 bits)")
	    (list "d40130.sf" 1 10000 0.100000001490116 "IRCAM" "little endian short (16 bits)")
	    (list "d40130.smp" 1 8000 0.125 "SMP" "little endian short (16 bits)")
	    (list "d40130.sou" 1 8000 0.125 "SBStudioII" "little endian short (16 bits)")
	    (list "d40130.st3" 1 8000 0.125 "Digiplayer ST3" "unsigned little endian short (16 bits)")
	    (list "d40130.uwf" 1 8000 0.125249996781349 "Ultratracker" "little endian short (16 bits)")
	    (list "d40130.voc" 1 10000 0.100100003182888 "VOC" "unsigned byte (8 bits)")
	    (list "d40130.w00" 1 16000 0.0625 "TX-16W" "unknown")
	    (list "d40130.wav" 1 10000 0.100000001490116 "RIFF" "little endian short (16 bits)")
	    (list "d43.wav" 1 10000 0.100000001490116 "RIFF" "little endian short (16 bits)")
	    (list "digit0v0.aiff" 1 8000 0.560000002384186 "AIFC" "big endian short (16 bits)")
	    (list "esps-16.snd" 1 8000 3.09737491607666 "ESPS" "big endian short (16 bits)")
	    (list "forest.aiff" 2 44100 3.907143 "AIFF" "big endian short (16 bits)" 24981 144332)
	    (list "g721.au" 1 11025 4.35328817367554 "Sun" "unknown")
	    (list "g722.aifc" 1 44100 0.0184353739023209 "AIFC" "unknown")
	    (list "gong.wve" 1 8000 3.96799993515015 "PSION" "alaw (8 bits)")
	    (list "gsm610.wav" 1 11025 1.7687075138092 "RIFF" "unknown")
	    (list "inrs-16.snd" 1 8000 2.46399998664856 "INRS" "little endian short (16 bits)")
	    (list "kirk.wve" 1 8000 1.40799999237061 "PSION" "alaw (8 bits)")
	    (list "loop.aiff" 1 44100 0.0367120169103146 "AIFC" "big endian short (16 bits)" 12 23)
	    (list "m.asf" 1 8000 64.9646224975586 "asf" "unknown")
	    (list "mary-sun4.sig" 1 8000 4.47612476348877 "Comdisco SPW signal" "big endian double (64 bits)")
	    (list "mocksong.wav" 1 11025 7.86956930160522 "RIFF" "little endian short (16 bits)")
	    (list "mono24.wav" 1 22050 1.98997735977173 "RIFF" "little endian int (24 bits)")
	    (list "msadpcm.wav" 1 11025 4.43501138687134 "RIFF" "unknown")
	    (list "n8.snd" 1 44100 0.0367800444364548 "Sun" "signed byte (8 bits)")
	    (list "nasahal.aif" 1 11025 9.89841270446777 "AIFF" "signed byte (8 bits)")
	    (list "nasahal.avi" 1 11025 10.4327440261841 "AVI" "little endian short (16 bits)")
	    (list "nasahal.dig" 1 11025 9.89841270446777 "Sound Designer 1" "big endian short (16 bits)")
	    (list "nasahal.ivc" 2 44100 0.449002265930176 "raw (no header)" "big endian short (16 bits)")
	    (list "nasahal.pat" 1 11025 3.95410442352295 "Gravis Ultrasound patch" "unsigned byte (8 bits)")
	    (list "nasahal.snd" 1 11025 9.89841270446777 "SNDT" "unsigned byte (8 bits)")
	    (list "nasahal.svx" 1 11025 9.89841270446777 "SVX8" "signed byte (8 bits)")
	    (list "nasahal.v8" 1 8000 13.6412496566772 "Covox V8" "unsigned byte (8 bits)")
	    (list "nasahal.voc" 1 11025 9.89941024780273 "VOC" "unsigned byte (8 bits)")
	    (list "nasahal.vox" 2 44100 0.224444448947906 "raw (no header)" "big endian short (16 bits)")
	    (list "nasahal8.wav" 1 11025 9.89841270446777 "RIFF" "unsigned byte (8 bits)")
	    (list "nasahalad.smp" 1 11025 4.94920635223389 "Goldwave sample" "little endian short (16 bits)")
	    (list "next-16.snd" 1 22050 1.00004529953003 "Sun" "big endian short (16 bits)")
	    (list "next-8.snd" 1 22050 0.226757362484932 "Sun" "signed byte (8 bits)")
	    (list "next-dbl.snd" 1 22050 0.226757362484932 "Sun" "big endian double (64 bits)")
	    (list "oboe.ldbl" 1 22050 2.30512475967407 "RIFF" "little endian double (64 bits)")
	    (list "next-flt.snd" 1 22050 0.226757362484932 "Sun" "big endian float (32 bits)")
	    (list "aifc-float.snd" 1 22050 0.226757362484932 "AIFC" "big endian float (32 bits)")
	    (list "next-mulaw.snd" 1 8012 2.03295063972473 "Sun" "mulaw (8 bits)")
	    (list "next24.snd" 1 44100 0.0367800444364548 "Sun" "big endian int (24 bits)")
	    (list "nist-01.wav" 1 16000 2.26912498474121 "NIST" "little endian short (16 bits)")
	    (list "nist-10.wav" 1 16000 2.26912498474121 "NIST" "big endian short (16 bits)")
	    (list "nist-16.snd" 1 16000 1.02400004863739 "NIST" "big endian short (16 bits)")
	    (list "nist-shortpack.wav" 1 16000 4.53824996948242 "NIST" "unknown")
	    (list "none.aifc" 1 44100 0.0367800444364548 "AIFC" "big endian short (16 bits)")
	    (list "nylon2.wav" 2 22050 1.14376413822174 "RIFF" "unknown")
	    (list "o2.adf" 1 44100 0.036780 "CSRE adf" "little endian short (16 bits)")
	    (list "o2.avr" 1 44100 0.0183900222182274 "AVR" "big endian short (16 bits)")
	    (list "o2.bicsf" 1 44100 0.0367800444364548 "IRCAM" "big endian short (16 bits)")
	    (list "o2.mpeg1" 2 44100 0.00709750549867749 "raw (no header)" "big endian short (16 bits)")
	    (list "o2.sd2" 2 44100 0.0183900222182274 "raw (no header)" "big endian short (16 bits)")
	    (list "o2.sf2" 1 44100 0.0367800444364548 "SoundFont" "little endian short (16 bits)")
	    (list "o2.smp" 1 8000 0.202749997377396 "SMP" "little endian short (16 bits)")
	    (list "o2.voc" 1 44100 0.0368934236466885 "VOC" "little endian short (16 bits)")
	    (list "o2.wave" 1 44100 0.0367800444364548 "RIFF" "little endian short (16 bits)")
	    (list "o2_12bit.aiff" 1 44100 0.0367800444364548 "AIFF" "big endian short (16 bits)")
	    (list "o2_18bit.aiff" 1 44100 0.0367800444364548 "AIFF" "big endian int (24 bits)")
	    (list "o2_711u.wave" 1 44100 0.0367800444364548 "RIFF" "mulaw (8 bits)")
	    (list "o2_722.snd" 1 44100 0.0183900222182274 "Sun" "unknown")
	    (list "o2_726.aiff" 1 8000 0.0367499999701977 "AIFC" "unknown")
	    (list "o2_726.snd" 1 44100 0.0230158735066652 "Sun" "unknown")
	    (list "o2_728.aiff" 1 8000 0.0367499999701977 "AIFC" "unknown")
	    (list "o2_8.iff" 1 44100 0.0367800444364548 "SVX8" "signed byte (8 bits)")
	    (list "o2_8.voc" 1 44100 0.0370294786989689 "VOC" "unsigned byte (8 bits)")
	    (list "o2_dvi.wave" 1 44100 0.0232199542224407 "RIFF" "unknown")
	    (list "o2_float.bicsf" 1 44100 0.0367800444364548 "IRCAM" "big endian float (32 bits)")
	    (list "o2_gsm.aiff" 1 8000 0.0367499999701977 "AIFC" "unknown")
	    (list "o2_u8.avr" 1 44100 0.0367800444364548 "AVR" "unsigned byte (8 bits)")
	    (list "o2_u8.wave" 1 44100 0.0367800444364548 "RIFF" "unsigned byte (8 bits)")
	    (list "o28.mpc" 1 44100 0.036780 "AKAI 4" "little endian short (16 bits)")
	    (list "oboe.g721" 1 22050 1.15287983417511 "Sun" "unknown")
	    (list "oboe.g723_24" 1 22050 0.864761888980865 "Sun" "unknown")
	    (list "oboe.g723_40" 1 22050 1.44126987457275 "Sun" "unknown")
	    (list "oboe.kts" 1 22050 2.305125 "Korg" "big endian short (16 bits)")
	    (list "oboe.its" 1 22050 2.305125 "Impulse Tracker" "little endian short (16 bits)")
	    (list "oboe.sf2" 1 22050 2.30512475967407 "SoundFont" "little endian short (16 bits)")
	    (list "oboe.paf" 1 22050 2.305125 "Ensoniq Paris" "big endian short (16 bits)")
	    (list "oboe.pf1" 1 22050 2.305125 "Ensoniq Paris" "little endian short (16 bits)")
	    (list "oboe.smp" 1 22050 2.305125 "snack SMP" "little endian short (16 bits)")
	    (list "oboe.nsp" 1 22050 2.305125 "CSL" "little endian short (16 bits)")
	    (list "oboe.nvf" 1 8000 6.353500 "Creative NVF" "unknown")
	    (list "oboe.wfp" 1 22050 2.305125 "Turtle Beach" "little endian short (16 bits)")
	    (list "oki.snd" 2 44100 0.0041950112208724 "raw (no header)" "big endian short (16 bits)")
	    (list "oki.wav" 1 44100 0.016780 "RIFF" "unknown")
	    (list "orv-dvi-adpcm.wav" 1 44100 1.92725622653961 "RIFF" "unknown")
	    (list "riff-16.snd" 1 22050 1.88766443729401 "RIFF" "little endian short (16 bits)")
	    (list "riff-8-u.snd" 1 11025 0.506848096847534 "RIFF" "unsigned byte (8 bits)")
	    (list "rooster.wve" 1 8000 2.04800009727478 "PSION" "alaw (8 bits)")
	    (list "sd1-16.snd" 1 44100 0.400544226169586 "Sound Designer 1" "big endian short (16 bits)")
;	    (list "segfault.snd" 16777216 576061440 1.24986669902682e-7 "Sun" "unknown")
	    (list "sf-16.snd" 1 22050 1.88766443729401 "IRCAM" "big endian short (16 bits)")
	    (list "si654.adc" 1 16000 6.71362495422363 "ADC/OGI" "big endian short (16 bits)")
	    (list "smp-16.snd" 1 8000 5.2028751373291 "SMP" "little endian short (16 bits)")
	    (list "sound.pat" 1 8000 1.95050001144409 "Gravis Ultrasound patch" "unsigned little endian short (16 bits)")
	    (list "sound.sap" 1 8000 1.95050001144409 "Goldwave sample" "little endian short (16 bits)")
	    (list "sound.sds" 1 8000 1.95050001144409 "MIDI sample dump" "unknown")
	    (list "sound.sfr" 1 8000 1.95050001144409 "SRFS" "little endian short (16 bits)")
	    (list "sound.v8" 1 8000 1.95050001144409 "Covox V8" "unsigned byte (8 bits)")
	    (list "sound.vox" 2 44100 0.044217687100172 "raw (no header)" "big endian short (16 bits)")
	    (list "step.omf" 1 11025 8.70666694641113 "OMF" "signed byte (8 bits)")
	    (list "step.qt" 1 11025 8.70630359649658 "Quicktime" "unsigned byte (8 bits)")
	    (list "sun-16-afsp.snd" 1 8000 2.9760000705719 "Sun" "big endian short (16 bits)")
	    (list "sun-mulaw.snd" 1 8000 4.61950016021729 "Sun" "mulaw (8 bits)")
	    (list "sw1038t_short.wav" 2 8000 6.0 "NIST" "mulaw (8 bits)")
	    (list "swirl.pat" 1 22050 1.0619500875473 "Gravis Ultrasound patch" "unsigned little endian short (16 bits)")
	    (list "sy85.snd" 1 8000 5.05600023269653 "Sy-85" "big endian short (16 bits)")
	    (list "sy99.snd" 1 8000 4.54400014877319 "Sy-99" "big endian short (16 bits)")
	    (list "telephone.wav" 1 16000 2.27881240844727 "NIST" "little endian short (16 bits)")
	    (list "trumps22.adp" 1 22050 3.092880 "RIFF" "unknown")
	    (list "truspech.wav" 1 8000 1.1599999666214 "RIFF" "unknown")
	    (list "ulaw.aifc" 1 44100 0.0367800444364548 "AIFC" "mulaw (8 bits)")
	    (list "voc-8-u.snd" 1 8000 1.49937498569489 "VOC" "unsigned byte (8 bits)")
	    (list "o28.voc" 1 44100 0.036893 "VOC" "little endian short (16 bits)")
	    (list "voxware.wav" 1 8000 0.324000000953674 "RIFF" "unknown")
	    (list "wd.w00" 1 8000 0.202749997377396 "Sy-99" "big endian short (16 bits)")
	    (list "wd1.smp" 1 8000 0.202749997377396 "SMP" "little endian short (16 bits)")
	    (list "wd1.wav" 1 44100 0.0367800444364548 "RIFF" "little endian short (16 bits)")
	    (list "wheel.mat" 2 44100 0.145646259188652 "raw (no header)" "big endian short (16 bits)")
	    (list "b8.pvf" 1 44100 0.036803 "Portable Voice Format" "signed byte (8 bits)")
	    (list "b16.pvf" 1 44100 0.036803 "Portable Voice Format" "big endian short (16 bits)")
	    (list "b32.pvf" 1 44100 0.036803 "Portable Voice Format" "big endian int (32 bits)")
	    (list "water.voc" 2 32000 42.3463897705078 "VOC" "little endian short (16 bits)")
	    (list "wood.dsf" 1 8000 0.202749997377396 "Delusion" "little endian short (16 bits)")
	    (list "wood.dvi" 1 22100 0.0278733037412167 "RIFF" "unknown")
	    (list "wood.dwd" 1 22100 0.0733936652541161 "DiamondWare" "signed byte (8 bits)")
	    (list "wood.fsm" 1 8000 0.202999994158745 "Farandole" "little endian short (16 bits)")
	    (list "wood.mad" 1 22100 0.0372398197650909 "RIFF" "unknown")
	    (list "wood.maud" 1 44100 0.0183900222182274 "MAUD" "big endian short (16 bits)")
	    (list "wood.pat" 1 22100 0.0733936652541161 "Gravis Ultrasound patch" "little endian short (16 bits)")
	    (list "wood.riff" 1 44100 0.0367800444364548 "RIFF" "little endian short (16 bits)")
	    (list "wood.rifx" 1 44100 0.0367800444364548 "RIFF" "big endian short (16 bits)")
	    (list "wood.sds" 1 22100 0.0733936652541161 "MIDI sample dump" "unknown")
	    (list "wood.sdx" 1 22100 0.0733936652541161 "Sample dump" "unsigned little endian short (16 bits)")
	    (list "wood.sf" 1 44100 0.0367800444364548 "IRCAM" "big endian short (16 bits)")
	    (list "wood.sndr" 2 44100 0.0092290248721838 "raw (no header)" "big endian short (16 bits)")
	    (list "wood.sndt" 1 44100 0.0367800444364548 "SNDT" "unsigned byte (8 bits)")
	    (list "wood.st3" 1 8000 0.202749997377396 "Digiplayer ST3" "unsigned little endian short (16 bits)")
	    (list "wood.uwf" 1 8000 0.202999994158745 "Ultratracker" "little endian short (16 bits)")
	    (list "wood.w00" 1 16000 0.101374998688698 "TX-16W" "unknown")
	    (list "wood12.aiff" 1 44100 0.0367800444364548 "AIFF" "big endian short (16 bits)")
	    (list "wood16.dwd" 2 44100 0.0367800444364548 "DiamondWare" "little endian short (16 bits)")
	    (list "wood16.wav" 2 44100 0.0367800444364548 "RIFF" "little endian short (16 bits)")
	    (list "wood16.nsp" 2 44100 0.0367800444364548 "CSL" "little endian short (16 bits)")
	    (list "wood16.smp" 2 44100 0.0367800444364548 "snack SMP" "little endian short (16 bits)")
	    (list "wood24.aiff" 1 44100 0.0367800444364548 "AIFF" "big endian int (24 bits)")
	    (list "woodblock.aiff" 1 44100 0.0367800444364548 "AIFF" "big endian short (16 bits)")
	    (list "woodflt.snd" 1 44100 0.0367800444364548 "Sun" "big endian float (32 bits)")
	    (list "RealDrums.sf2" 1 44100 6.39725637435913 "SoundFont" "little endian short (16 bits)")
	    (list "32bit.sf" 1 44100 4.6 "IRCAM" "little endian float (32 bits, unscaled)")
	    (list "PCM_48_8bit_m.w64" 1 48000 0.375 "SoundForge" "unsigned byte (8 bits)")
	    (list "oboe.sf6" 1 22050 2.305125 "SoundForge" "little endian short (16 bits)")
	    (list "addf8.24we" 1 8000 2.976000 "RIFF" "little endian int (24 bits)")
	    (list "hybrid.snd" 1 44100 4.600000 "BICSF" "big endian float (32 bits)")
	    (list "litmanna.sf" 1 44100 .533 "IRCAM" "little endian short (16 bits)")
	    (list "M1F1-float64C-AFsp.aif" 2 8000 2.9366 "AIFC" "big endian double (64 bits)")
	    (list "MacBoing.wav" 1 11127 0.696 "RIFF" "unsigned byte (8 bits)")
	    (list "t15.aiff" 2 44100 135.00 "AIFC" "little endian short (16 bits)")
	    (list "zulu_a4.w11" 1 33000 1.21987879276276 "TX-16W" "unknown" 23342 40042)))

	  (for-each (lambda (in-name real-name)
		      (if (not (string=? (mus-expand-filename in-name) real-name))
			  (snd-display ";mus-expand-filename ~A -> ~A" in-name (mus-expand-filename in-name)))
		      (if (file-exists? "/home/bil/./sf1/o2.voc")
			  (let ((ind (open-sound in-name)))
			    (if (not (sound? ind))
				(snd-display ";can't open ~A" in-name)
				(begin
				  (if (not (string=? (file-name ind) real-name))
				      (snd-display ";expand file name ~A: ~A" in-name (file-name ind)))
				  (close-sound ind))))))
		    (list "/home/bil/./sf1/o2.voc" "~/./sf1/o2.voc" "~/cl/../sf1/o2.voc" "/home/bil/cl/../sf1/o2.voc")
		    (list "/home/bil/sf1/o2.voc" "/home/bil/sf1/o2.voc" "/home/bil/sf1/o2.voc" "/home/bil/sf1/o2.voc"))
	  (run-hook after-test-hook 2)
	  ))
    )


(if (not (provided? 'snd-snd6.scm)) (load "snd6.scm"))

;;; ---------------- test 3: can variables be set/reset ----------------
(if (or full-test (= snd-test 3) (and keep-going (<= snd-test 3)))
    (let ((ind #f))
      (run-hook before-test-hook 3)
      (set! ind (open-sound "oboe.snd"))
      (if (and (file-exists? "funcs.cl") 
	       (not (defined? 'swellf)))
	  (load "funcs.cl"))
      (let ((td (temp-dir)))
	(catch #t
	       (lambda ()
		 (set! (temp-dir) (string-append home-dir "/test"))
		 (if (not (string=? (temp-dir) (string-append home-dir "/test")))
		     (snd-display ";set temp-dir: ~A?" (temp-dir))))
	       (lambda args args))
	(if td 
	    (set! (temp-dir) td)
	    (set! (temp-dir) "")))
      (if (fneq (sample 1000) 0.0328) (snd-display ";sample: ~A?" (sample 1000)))
      (if (or (not (hook? output-name-hook)) (not (hook-empty? output-name-hook)))
	  (snd-display ";output-name-hook: ~A?" output-name-hook))
      (if (or (not (hook? output-comment-hook)) (not (hook-empty? output-comment-hook)))
	  (snd-display ";output-comment-hook: ~A?" output-comment-hook))
      (if (or (not (hook? help-hook)) (not (hook-empty? help-hook)))
	  (snd-display ";help-hook: ~A?" help-hook))
      (if (or (not (hook? mark-drag-hook)) (not (hook-empty? mark-drag-hook)))
	  (snd-display ";mark-drag-hook: ~A?" mark-drag-hook))
      (if (or (not (hook? mark-drag-triangle-hook)) (not (hook-empty? mark-drag-triangle-hook)))
	  (snd-display ";mark-drag-triangle-hook: ~A?" mark-drag-triangle-hook))
      (if (or (not (hook? mix-drag-hook)) (not (hook-empty? mix-drag-hook)))
	  (snd-display ";mix-drag-hook: ~A?" mix-drag-hook))
      (if (or (not (hook? mouse-drag-hook)) (not (hook-empty? mouse-drag-hook)))
	  (snd-display ";mouse-drag-hook: ~A?" mouse-drag-hook))
      (if (or (not (hook? mouse-click-hook)) (not (hook-empty? mouse-click-hook)))
	  (snd-display ";mouse-click-hook: ~A?" mouse-click-hook))
      (if (or (not (hook? mouse-press-hook)) (not (hook-empty? mouse-press-hook)))
	  (snd-display ";mouse-press-hook: ~A?" mouse-press-hook))
      (if (or (not (hook? start-playing-hook)) (not (hook-empty? start-playing-hook)))
	  (snd-display ";start-playing-hook: ~A?" start-playing-hook))
      (if (or (not (hook? start-playing-selection-hook)) (not (hook-empty? start-playing-selection-hook)))
	  (snd-display ";start-playing-selection-hook: ~A?" start-playing-selection-hook))
      (if (or (not (hook? stop-playing-hook)) (not (hook-empty? stop-playing-hook)))
	  (snd-display ";stop-playing-hook: ~A?" stop-playing-hook))
      (if (or (not (hook? key-press-hook)) (not (hook-empty? key-press-hook)))
	  (snd-display ";key-press-hook: ~A?" key-press-hook))
      (if (or (not (hook? snd-error-hook)) (not (hook-empty? snd-error-hook)))
	  (snd-display ";snd-error-hook: ~A?" snd-error-hook))
      (if (or (not (hook? snd-warning-hook)) (not (hook-empty? snd-warning-hook)))
	  (snd-display ";snd-warning-hook: ~A?" snd-warning-hook))
      (if (or (not (hook? name-click-hook)) (not (hook-empty? name-click-hook)))
	  (snd-display ";name-click-hook: ~A?" name-click-hook))
      (if (or (not (hook? after-apply-hook)) (not (hook-empty? after-apply-hook)))
	  (snd-display ";after-apply-hook: ~A?" after-apply-hook))
      (if (or (not (hook? enved-hook)) (not (hook-empty? enved-hook)))
	  (snd-display ";enved-hook: ~A?" enved-hook))
      (if (or (not (hook? mouse-enter-label-hook)) (not (hook-empty? mouse-enter-label-hook)))
	  (snd-display ";mouse-enter-label-hook: ~A?" mouse-enter-label-hook))
      (if (or (not (hook? mouse-enter-graph-hook)) (not (hook-empty? mouse-enter-graph-hook)))
	  (snd-display ";mouse-enter-graph-hook: ~A?" mouse-enter-graph-hook))
      (if (or (not (hook? mouse-enter-listener-hook)) (not (hook-empty? mouse-enter-listener-hook)))
	  (snd-display ";mouse-enter-listener-hook: ~A?" mouse-enter-listener-hook))
      (if (or (not (hook? mouse-leave-label-hook)) (not (hook-empty? mouse-leave-label-hook)))
	  (snd-display ";mouse-leave-label-hook: ~A?" mouse-leave-label-hook))
      (if (or (not (hook? mouse-leave-graph-hook)) (not (hook-empty? mouse-leave-graph-hook)))
	  (snd-display ";mouse-leave-graph-hook: ~A?" mouse-leave-graph-hook))
      (if (or (not (hook? mouse-leave-listener-hook)) (not (hook-empty? mouse-leave-listener-hook)))
	  (snd-display ";mouse-leave-listener-hook: ~A?" mouse-leave-listener-hook))
      (if (or (not (hook? window-property-changed-hook)) (not (hook-empty? window-property-changed-hook)))
	  (snd-display ";window-property-changed-hook: ~A?" window-property-changed-hook))
      (if (or (not (hook? initial-graph-hook)) (not (hook-empty? initial-graph-hook)))
	  (snd-display ";initial-graph-hook: ~A?" initial-graph-hook))
      (if (or (not (hook? after-graph-hook)) (not (hook-empty? after-graph-hook)))
	  (snd-display ";after-graph-hook: ~A?" after-graph-hook))
      (if (or (not (hook? graph-hook)) (not (hook-empty? graph-hook)))
	  (snd-display ";graph-hook: ~A?" graph-hook))
      
      (set! (show-controls) #t)
      (if with-gui
	  (begin
	    (let ((wid (enved-dialog) ))
	      (if (not (equal? wid (list-ref (dialog-widgets) 2)))
		  (snd-display ";enved-dialog -> ~A ~A" wid (list-ref (dialog-widgets) 2))))
	    (if (not (list-ref (dialog-widgets) 2)) (snd-display ";enved-dialog?"))
	    (set! (enved-envelope) '(0.0 0.0 1.0 1.0 2.0 0.0))
	    (if (not (equal? (enved-envelope) (list 0.0 0.0 1.0 1.0 2.0 0.0)))
		(snd-display ";set enved-envelope: ~A?" (enved-envelope)))
	    (set! (enved-envelope) (enved-envelope))
	    (if (not (equal? (enved-envelope) (list 0.0 0.0 1.0 1.0 2.0 0.0)))
		(snd-display ";set enved-envelope to self: ~A?" (enved-envelope)))
	    (let ((wid (orientation-dialog)))
	      (if (not (list-ref (dialog-widgets) 1)) (snd-display ";orientation-dialog?"))
	      (if (not (equal? wid (list-ref (dialog-widgets) 1)))
		  (snd-display ";orientation-dialog -> ~A ~A" wid (list-ref (dialog-widgets) 1))))))
      
      (letrec ((test-vars
		(lambda (lst)
		  (if (not (null? lst))
		      (let* ((name (list-ref (car lst) 0))
			     (getfnc (list-ref (car lst) 1))
			     (setfnc (lambda (val) (set! (getfnc) val)))
			     (initval (list-ref (car lst) 2))
			     (newval (list-ref (car lst) 3)))
			
			(setfnc newval)
			(let ((nowval (getfnc)))
			  (if (and (not (equal? newval nowval))
				   (or (not (list? newval))
				       (not (feql newval nowval))))
			      (if (and (number? newval) (inexact? newval))
				  (if (> (abs (- newval nowval)) .01)
				      (snd-display ";~A /= ~A (~A)" name newval nowval))
				  (snd-display ";~A /= ~A (~A)" name newval nowval)))
			  (setfnc initval)
			  (set! (getfnc) newval)
			  (let ((nowval (getfnc)))
			    (if (and (not (equal? newval nowval))
				     (or (not (list? newval))
					 (not (feql newval nowval))))
				(if (and (number? newval) (inexact? newval))
				    (if (> (abs (- newval nowval)) .01)
					(snd-display ";set! ~A /= ~A (~A)" name newval nowval))
				    (snd-display ";set! ~A /= ~A (~A)" name newval nowval)))
			    (setfnc initval))
			  (test-vars (cdr lst))))))))
	(test-vars 
	 (list
	  (list 'amp-control amp-control 1.0 0.5)
	  (list 'amp-control-bounds amp-control-bounds (list 0.0 8.0) (list 1.0 5.0))
	  (list 'ask-before-overwrite ask-before-overwrite #f #t)
	  (list 'audio-input-device audio-input-device 0 1)
	  (list 'audio-output-device audio-output-device 0 1)
	  (list 'auto-resize auto-resize #t #f)
	  (list 'auto-update auto-update #f #t)
	  (list 'channel-style channel-style 0 1)
	  (list 'colormap colormap 2 0)
	  (list 'color-cutoff color-cutoff 0.003 0.01)
	  (list 'color-inverted color-inverted #t #f)
	  (list 'color-scale color-scale 1.0 0.5)
	  (list 'contrast-control contrast-control 0.0 0.5)
	  (list 'contrast-control-bounds contrast-control-bounds (list 0.0 10.0) (list 1.0 5.0))
	  (list 'contrast-control-amp contrast-control-amp 1.0 0.5)
	  (list 'contrast-control? contrast-control? #f #t)
	  (list 'auto-update-interval auto-update-interval 60.0 120.0)
	  (list 'cursor-update-interval cursor-update-interval 0.05 0.10)
	  (list 'cursor-location-offset cursor-location-offset 0 32768)
	  (list 'cursor-follows-play cursor-follows-play #f #t)
	  (list 'cursor-size cursor-size 15 30)
	  (list 'cursor-style cursor-style cursor-cross cursor-line)
	  (list 'dac-combines-channels dac-combines-channels #t #f)
	  (list 'dac-size dac-size 256 512)
	  (list 'minibuffer-history-length minibuffer-history-length 8 16)
	  (list 'data-clipped data-clipped #f #t)
	  (list 'default-output-chans default-output-chans 1 2)
	  (list 'default-output-format default-output-format 1 1)
	  (list 'default-output-srate default-output-srate 22050 44100)
	  (list 'default-output-type default-output-type mus-next mus-aifc)
	  (list 'dot-size dot-size 1 4)
	  (list 'enved-base enved-base 1.0  1.5)
	  (list 'enved-clip? enved-clip? #f #t)
	  (list 'enved-in-dB enved-in-dB #f #t)
	  (list 'enved-style enved-style envelope-linear envelope-exponential)
	  (list 'enved-power enved-power 3.0 3.5)
	  (list 'enved-target enved-target 0 1)
	  (list 'enved-wave? enved-wave? #f #t)
	  (list 'eps-file eps-file "snd.eps" "snd-1.eps")
	  (list 'eps-left-margin eps-left-margin 0.0 72.0)
	  (list 'eps-size eps-size 1.0 2.0)
	  (list 'eps-bottom-margin eps-bottom-margin 0.0 36.0)
	  (list 'expand-control expand-control 1.0 2.0)
	  (list 'expand-control-bounds expand-control-bounds (list .001 20.0) (list 1.0 2.0))
	  (list 'expand-control-hop expand-control-hop 0.05 0.1)
	  (list 'expand-control-jitter expand-control-jitter 0.1 0.2)
	  (list 'expand-control-length expand-control-length 0.15 0.2)
	  (list 'expand-control-ramp expand-control-ramp 0.4 0.2)
	  (list 'expand-control? expand-control? #f #t)
	  (list 'fft-window-beta fft-window-beta 0.0  0.5)
	  (list 'fft-log-frequency fft-log-frequency #f #t)
	  (list 'fft-log-magnitude fft-log-magnitude #f #t)
	  (list 'transform-size transform-size 512 1024)
	  (list 'transform-graph-type transform-graph-type 0 1)
	  (list 'fft-window fft-window 6 5)
	  (list 'transform-graph? transform-graph? #f #t)
	  (list 'filter-control-in-dB filter-control-in-dB #f #t)
	  (list 'filter-control-envelope filter-control-envelope (list 0.0 1.0 1.0 1.0) (list 0.0 1.0 1.0 0.0))
	  (list 'enved-filter enved-filter #t #f)
	  (list 'enved-filter-order enved-filter-order 40 20)
	  (list 'filter-control-in-hz filter-control-in-hz #f #t)
	  (list 'filter-control-order filter-control-order 20 40)
	  (list 'filter-control? filter-control? #f #t)
	  (list 'graph-cursor graph-cursor 34 33)
	  (list 'graph-style graph-style 0 1)
	  (list 'just-sounds just-sounds #f #t)
	  (list 'listener-prompt listener-prompt ">" ":")
	  (list 'max-transform-peaks max-transform-peaks 100 10)
	  (list 'max-regions max-regions 16 6)
	  (list 'min-dB min-dB -60.0 -90.0)
	  (list 'mix-waveform-height mix-waveform-height 20 40)
	  (list 'mix-tag-height mix-tag-height 14 20)
	  (list 'mix-tag-width mix-tag-width 6 20)
	  (list 'selection-creates-region selection-creates-region #t #f)
	  (list 'transform-normalization transform-normalization normalize-by-channel dont-normalize)
	  (list 'previous-files-sort previous-files-sort 0 1)
	  (list 'print-length print-length 12 16)
	  (list 'recorder-autoload recorder-autoload #f #t)
	  (list 'recorder-out-chans recorder-out-chans 2 1)
	  (list 'recorder-buffer-size recorder-buffer-size 4096 256)
	  (list 'recorder-max-duration recorder-max-duration 1000000.0 1000.0)
	  (list 'recorder-trigger recorder-trigger 0.0 0.1)
	  (list 'region-graph-style region-graph-style graph-lines graph-lollipops)
	  (list 'reverb-control-decay reverb-control-decay 1.0 2.0)
	  (list 'reverb-control-feedback reverb-control-feedback 1.09 1.6)
	  (list 'reverb-control-length reverb-control-length 1.0 2.0)
	  (list 'reverb-control-length-bounds reverb-control-length-bounds (list 0.0 5.0) (list 1.0 2.0))
	  (list 'reverb-control-lowpass reverb-control-lowpass 0.7 0.9)
	  (list 'reverb-control-scale reverb-control-scale 0.0 0.2)
	  (list 'reverb-control-scale-bounds reverb-control-scale-bounds (list 0.0 4.0) (list 0.0 0.2))
	  (list 'reverb-control? reverb-control? #f #t)
	  (list 'show-axes show-axes 1 0)
	  (list 'show-transform-peaks show-transform-peaks #f #t)
	  (list 'show-indices show-indices #f #t)
	  (list 'show-backtrace show-backtrace #f #t)
	  (list 'show-marks show-marks #t #f)
	  (list 'show-mix-waveforms show-mix-waveforms #t #f)
	  (list 'show-selection-transform show-selection-transform #f #t)
	  (list 'show-y-zero show-y-zero #f #t)
	  (list 'show-grid show-grid #f #t)
	  (list 'sinc-width sinc-width 10 40)
	  (list 'spectro-cutoff spectro-cutoff 1.0 0.7)
	  (list 'spectro-hop spectro-hop 4 10)
	  (list 'spectro-start spectro-start 0.0 0.1)
	  (list 'spectro-x-angle spectro-x-angle (if (provided? 'gl) 300.0 90.0) 60.0)
	  (list 'spectro-x-scale spectro-x-scale (if (provided? 'gl) 1.5 1.0) 2.0)
	  (list 'spectro-y-angle spectro-y-angle (if (provided? 'gl) 320.0 0.0) 60.0)
	  (list 'spectro-y-scale spectro-y-scale 1.0 2.0)
	  (list 'spectro-z-angle spectro-z-angle (if (provided? 'gl) 0.0 358.0) 60.0)
	  (list 'spectro-z-scale spectro-z-scale (if (provided? 'gl) 1.0 0.1) 0.2)
	  (list 'speed-control speed-control 1.0 0.5)
	  (list 'speed-control-bounds speed-control-bounds (list 0.05 20.0) (list 1.0 5.0))
	  (list 'speed-control-style speed-control-style 0 1)
	  (list 'speed-control-tones speed-control-tones 12 18)
	  (list 'sync sync 0 1)
	  (list 'tiny-font tiny-font "6x12" "9x15")
	  (list 'transform-type transform-type 0 1)
	  (list 'verbose-cursor verbose-cursor #f #t)
	  (list 'vu-size vu-size 1.0 2.0)
	  (list 'vu-font-size vu-font-size 1.0 2.0)
	  (list 'wavelet-type wavelet-type 0 1)
	  (list 'time-graph? time-graph? #f #t)
	  (list 'time-graph-type time-graph-type graph-once graph-as-wavogram)
	  (list 'wavo-hop wavo-hop 3 6)
	  (list 'wavo-trace wavo-trace 64 128)
	  (list 'with-mix-tags with-mix-tags #t #f)
	  (list 'with-relative-panes with-relative-panes #t #f)
	  (list 'with-gl with-gl (provided? 'gl) #f)
	  (list 'x-axis-style x-axis-style 0 1)
	  (list 'beats-per-minute beats-per-minute 30.0 120.0)
	  (list 'zero-pad zero-pad 0 1)
	  (list 'zoom-focus-style zoom-focus-style 2 1))))
      
      (letrec ((test-bad-args
		(lambda (lst)
		  (if (not (null? lst))
		      (let* ((name (list-ref (car lst) 0))
			     (getfnc (list-ref (car lst) 1))
			     (setfnc (lambda (val) (set! (getfnc) val)))
			     (initval (list-ref (car lst) 2))
			     (newvals (list-ref (car lst) 3)))
			(for-each
			 (lambda (n)
			   (catch #t 
				  (lambda ()
				    (setfnc n))
				  (lambda args (car args)))
			   (let ((nowval (getfnc)))
			     (if (equal? n nowval)
				 (snd-display ";~A = ~A (~A)" name n initval))
			     (setfnc initval)))
			 newvals)
			(test-bad-args (cdr lst)))))))
	(test-bad-args
	 (list
	  (list 'amp-control amp-control 1.0 '(-1.0 123.123))
	  (list 'amp-control-bounds amp-control-bounds (list 0.0 8.0) (list #f (list 0.0) (list 1.0 0.0) 2.0))
	  (list 'channel-style channel-style 0 '(32 -1 1.0))
	  (list 'colormap colormap 2 '(321 -123))
	  (list 'color-cutoff color-cutoff 0.003 '(-1.0 123.123))
	  (list 'color-scale color-scale 1.0 '(-32.0 2000.0))
	  (list 'contrast-control contrast-control 0.0 '(-123.123 123.123))
	  (list 'contrast-control-bounds contrast-control-bounds (list 0.0 10.0) (list #f (list 0.0) (list 1.0 0.0) 2.0))
	  (list 'cursor-size cursor-size 15 '(1.123 -2.5))
	  (list 'dac-size dac-size 256 '(-1 0 -123))
	  (list 'dot-size dot-size 1 '(0 -1 -123))
	  (list 'enved-target enved-target 0 '(123 -321))
	  (list 'expand-control expand-control 1.0 '(-1.0 0.0))
	  (list 'expand-control-bounds expand-control-bounds (list 0.001 20.0) (list #f (list 0.0) (list 1.0 0.0) 2.0))
	  (list 'expand-control-hop expand-control-hop 0.05 '(-1.0))
	  (list 'expand-control-length expand-control-length 0.15 '(-1.0 0.0))
	  (list 'expand-control-ramp expand-control-ramp 0.4 '(-1.0 1.0 123.123))
	  (list 'fft-window-beta fft-window-beta 0.0  '(-1.0 123.123))
	  (list 'transform-size transform-size 512 '(-1 0))
	  (list 'zero-pad zero-pad 0 '(-1 -123))
	  (list 'cursor-style cursor-style cursor-cross '(-1))
	  (list 'cursor-style cursor-style cursor-line '(2 123))
	  (list 'transform-graph-type transform-graph-type 0 '(-1 123))
	  (list 'fft-window fft-window 6 '(-1 123))
	  (list 'enved-filter-order enved-filter-order 40 '(-1 0))
	  (list 'filter-control-order filter-control-order 20 '(-10 -1 0))
	  (list 'max-transform-peaks max-transform-peaks 100 '(-1))
	  (list 'max-regions max-regions 16 '(-1 -123))
	  (list 'previous-files-sort previous-files-sort 0 '(-1 123))
	  (list 'reverb-control-length reverb-control-length 1.0 '(-1.0))
	  (list 'show-axes show-axes 1 '(-1 123))
	  (list 'sinc-width sinc-width 10 '(-10))
	  (list 'spectro-cutoff spectro-cutoff 1.0 '(-1.0))
	  (list 'spectro-hop spectro-hop 4 '(-10 -1 0))
	  (list 'spectro-start spectro-start 0.0 '(-1.0))
	  (list 'speed-control speed-control 1.0 '(0.0))
	  (list 'speed-control-bounds speed-control-bounds (list 0.05 20.0) (list #f (list 0.0) (list 1.0 0.0) 2.0))
	  (list 'speed-control-style speed-control-style 0 '(-1 10))
	  (list 'transform-type transform-type 0 '(-1 123))
	  (list 'wavelet-type wavelet-type 0 '(-1 123))
	  (list 'wavo-hop wavo-hop 1 '(0 -123))
	  (list 'wavo-trace wavo-trace 1 '(0 -123))
	  (list 'x-axis-style x-axis-style 0 '(-1 123))
	  (list 'zoom-focus-style zoom-focus-style 2 '(-1 123)))))
      
      (set! (window-width) 300)
      (set! (window-height) 300)
      (if (not (equal? (window-width) 300))
	  (snd-display ";window width: ~A /= 300?" (window-width)))
      (if (not (equal? (window-height) 300))
	  (snd-display ";window height: ~A /= 300?" (window-height)))
      (set! (window-x) 123)
      (set! (window-y) 321)
      (if (not (equal? (window-x) 123))
	  (snd-display ";window x: ~A /= 123?" (window-x)))
      (if (not (equal? (window-y) 321))
	  (snd-display ";window y: ~A /= 321?" (window-y)))
      (set! (window-y) 10) ; get it back out of harm's way
      (set! (vu-font) "8x15")
      (set! (color-scale) 100.0)
      (if (fneq (color-scale) 100.0) (snd-display ";color-scale to 100: ~A" (color-scale)))
      
      (if (procedure? (search-procedure))
	  (snd-display ";global search procedure: ~A?" (search-procedure)))
      (set! (search-procedure) (lambda (y) (> y .1)))
      (if (not (procedure? (search-procedure)))
	  (snd-display ";set global search procedure: ~A?" (search-procedure)))
      (if (not ((search-procedure) .2))
	  (snd-display ";search > .1 .2"))
      (if ((search-procedure) .02)
	  (snd-display ";search > .1 .02"))
      (set! (search-procedure) (lambda (y) (< y 0.0)))
      (if ((search-procedure) .02)
	  (snd-display ";search < 0.0 .02"))
      (set! (search-procedure) #f)
      (if (procedure? (search-procedure))
	  (snd-display ";global search procedure after reset: ~A?" (search-procedure)))
      (set! (search-procedure) (lambda (y) (> y .1)))
      (if (not (procedure? (search-procedure)))
	  (snd-display ";set global search procedure: ~A?" (search-procedure)))
      
      (set! (enved-filter-order) 5)
      (if (not (= (enved-filter-order) 6)) (snd-display ";set enved-filter-order 5: ~A" (enved-filter-order)))
      (if with-gui
	  (begin
	    (set! (enved-envelope) 'zero_to_one) ; funcs.cl above
	    (if (not (feql (enved-envelope) zero_to_one)) (snd-display ";set symbol enved-envelope: ~A ~A" (enved-envelope) zero_to_one))
	    (set! (enved-envelope) "mod_down")
	    (if (not (feql (enved-envelope) mod_down)) (snd-display ";set string enved-envelope: ~A ~A" (enved-envelope) mod_down))))

      (close-sound ind) 
      (dismiss-all-dialogs)
      
      (if (provided? 'snd-debug)
	  (begin
	    (snd-stdin-test "(set! (enved-filter-order) 12)")
	    (if (not (= (enved-filter-order) 12)) (snd-display ";set enved-filter-order 12 via stdin: ~A" (enved-filter-order)))
	    (snd-stdin-test "(set! (enved-filter")
	    (snd-stdin-test "-order) 10)")
	    (if (not (= (enved-filter-order) 10)) (snd-display ";set enved-filter-order 12 via 2 stdin: ~A" (enved-filter-order)))))
      (run-hook after-test-hook 3)
      ))

;;; ---------------- test 4: sndlib tests ----------------

(define play-sound-1
  (lambda (file)
    (let* ((sound-fd (mus-sound-open-input file))
           (chans (mus-sound-chans file))
           (frames (mus-sound-frames file))
           (bufsize 256)
           (data (make-sound-data chans bufsize))
           (bytes (* bufsize chans 2)))
      (mus-sound-read sound-fd 0 (1- bufsize) chans data)
      (catch #t
	     (lambda ()
	       (let ((audio-fd (mus-audio-open-output mus-audio-default (mus-sound-srate file) chans mus-lshort bytes)))
		 (if (= audio-fd -1)
		     (set! audio-fd (mus-audio-open-output mus-audio-default (mus-sound-srate file) chans mus-bshort bytes)))
		 (if (= audio-fd -1)
		     (snd-display ";can't play ~A" file)
		     (begin
		       (do ((i 0 (+ i bufsize)))
			   ((>= i frames))
			 (mus-audio-write audio-fd data bufsize)
			 (mus-sound-read sound-fd 0 (1- bufsize) chans data))
		       (mus-audio-close audio-fd)))))
	     (lambda args "can't open audio"))
      (mus-sound-close-input sound-fd))))


(define (frame->byte file frame)
  (+ (mus-sound-data-location file)
     (* (mus-sound-chans file)
	(mus-sound-datum-size file)
	frame)))

(if (not (provided? 'snd-snd5.scm)) (load "snd5.scm"))
(if (not (provided? 'snd-mix.scm)) (load "mix.scm"))
(if (not (provided? 'snd-env.scm)) (load "env.scm"))

(define (show-input-1 . arg)
  ;; from rtio.scm
  (define (card+device card device)
    (logior (ash card 16) device))
  (let* ((our-short (if (little-endian?) mus-lshort mus-bshort))
	 (our-srate 22050)
	 (our-dac-buffer-size-in-bytes 512)
	 (our-dac-buffer-size-in-shorts 256)
	 (our-chans 1)
	 (our-chan 0)
	 (our-default-card-number 0)
	 (in-sys (if (not (null? arg)) 
		     (car arg) 
		     our-default-card-number))
	 (in-port (catch 'mus-error
			 (lambda ()
			   (mus-audio-open-input 
			    (card+device in-sys mus-audio-default) 
			    our-srate our-chans our-short our-dac-buffer-size-in-bytes))
			 (lambda args -1)))
	 (data (make-sound-data our-chans our-dac-buffer-size-in-shorts))
    	 (vobj (make-vct our-dac-buffer-size-in-shorts)))
    (if (= in-port -1)
	(snd-display ";can't open audio input port!")
	(begin
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (mus-audio-read in-port data our-dac-buffer-size-in-shorts)
	    (graph (sound-data->vct data our-chan vobj)))
	  (mus-audio-close in-port)))))

(if (or full-test (= snd-test 4) (and keep-going (<= snd-test 4)))
    (begin
      (run-hook before-test-hook 4)
      (do ((clmtest 0 (1+ clmtest))) ((= clmtest tests)) 
	(log-mem clmtest)
	(clear-listener)
	(let ((chns (mus-sound-chans "oboe.snd"))
	      (dl (mus-sound-data-location "oboe.snd"))
	      (fr (mus-sound-frames "oboe.snd"))
	      (smps (mus-sound-samples "oboe.snd"))
	      (len (mus-sound-length "oboe.snd"))
	      (size (mus-sound-datum-size "oboe.snd"))
	      (com (mus-sound-comment "oboe.snd"))
	      (sr (mus-sound-srate "oboe.snd"))
	      (m1 (mus-sound-maxamp-exists? "oboe.snd"))
	      (mal (mus-sound-maxamp "oboe.snd"))
	      (mz (mus-sound-maxamp "z.snd"))
	      (bytes (mus-bytes-per-sample (mus-sound-data-format "oboe.snd")))
	      (sys (mus-audio-systems)))
	  (if (or (not (= (car mz) 0))
		  (fneq (cadr mz) 0.0))
	      (snd-display ";mus-sound-maxamp z.snd: ~A" mz))
	  (let ((formats (list mus-bshort mus-lshort mus-mulaw mus-alaw mus-byte mus-ubyte mus-bfloat mus-lfloat
			       mus-bint mus-lint mus-bintn mus-lintn mus-b24int mus-l24int mus-bdouble mus-ldouble
			       mus-ubshort mus-ulshort mus-bdouble-unscaled mus-ldouble-unscaled mus-bfloat-unscaled 
			       mus-lfloat-unscaled))
		(sizes (list 2 2 1 1 1 1 4 4 
			     4 4 4 4 3 3 8 8
			     2 2 8 8 4
			     4)))
	    (for-each
	     (lambda (frm siz)
	       (if (not (= (mus-bytes-per-sample frm) siz))
		   (snd-display ";mus-bytes-per-sample ~A: ~A" (mus-data-format-name frm) siz)))
	     formats
	     sizes))
	  (if (provided? 'snd-debug)
	      (for-each
	       (lambda (format type name)
		 (if (not (string=? name (mus-header-original-format-name format type)))
		     (snd-display ";original name: ~A ~A" name (mus-header-original-format-name format type))))
	       (list 29 13 17 10 8 23 20
		     (logior (ash (char->integer #\M) 24) (ash (char->integer #\A) 16) (ash (char->integer #\C) 8) (char->integer #\!))
		     2 #x10 #x13 #x16 #x19 #x22 #x25 #x28 #x32
		     #x35 #x38 #x3b #x40 #x50 #x55 #x61 #x64 #x67
		     #x71 #x74 #x77 #x80 #x83 #x86 #x91 #x98 #x101
		     #x111 #x200 #x203 #x220 #x680 #x1002 #x1100 #x1500)
	       (list mus-next mus-next mus-next mus-next mus-next mus-next mus-next
		     mus-aifc
		     mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff
		     mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff
		     mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff 
		     mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff mus-riff)
	       (list "delat_mulaw_8" "dsp_data_24" "mulaw_squelch" "dsp_core" "indirect" "adpcm_g721" "compressed_emphasized"
		     "!CAM"
		     "ADPCM" "OKI_ADPCM" "Sierra_ADPCM" "DIGIFIX" "HP cu codec" "DSPGroup_TrueSpeech" "APTX" "lrc" "MSN audio codec"
		     "DIGIREAL" "NMS_VBXADPCM" "Rockwell adpcm" "G721_ADPCM" "MPEG" "Mpeg layer 3" "ESS Tech pcm" "G726" "DSAT display"
		     "voxware ac10 " "voxware rt24" "voxware vr12 " "softsound" "G729A" "DF GSM610" "sbc24" "philips lpcbb" "Irat"
		     "vivo G723" "Creative_ADPCM" "Creative fastspeech 10" "quarterdeck" "VME vmpcm" "OLICELP" "LH codec" "Soundspace musicompression")))
	  (mus-sound-report-cache "hiho.tmp")
	  (if (defined? 'read-line)
	      (let ((p (open-input-file "hiho.tmp")))
		(if (not p)
		    (snd-display ";mus-sound-report-cache->hiho.tmp failed?")
		    (let ((line (read-line p)))
		      (if (or (not (string? line))
			      (not (string=? "sound table:")))
			  (snd-display ";print-cache 1: ~A?" line))
		      (close-port p)
		      (delete-file "hiho.tmp")))))
	  (if (< (string-length (mus-audio-report)) 10)
	      (snd-display ";mus-audio-report: ~A" (mus-audio-report)))
	  (if (and (not (= sys 1)) (not (= sys 2))) (snd-display ";mus-audio-systems: ~A?" sys))
	  (if (not (= chns 1)) (snd-display ";oboe: mus-sound-chans ~D?" chns))
	  (if (not (= dl 28)) (snd-display ";oboe: mus-sound-data-location ~D?" dl))
	  (if (not (= fr 50828)) (snd-display ";oboe: mus-sound-frames ~D?" fr))
	  (if (not (= smps 50828)) (snd-display ";oboe: mus-sound-samples ~D?" smps))
	  (if (not (= len (+ 28 (* 2 50828)))) (snd-display ";oboe: mus-sound-length ~D?" len))
	  (if (not (= size 2)) (snd-display ";oboe: mus-sound-datum-size ~D?" size))
	  (if (not (= bytes 2)) (snd-display ";oboe: sound-bytes ~D?" bytes))
	  (if (not (= sr 22050)) (snd-display ";oboe: mus-sound-srate ~D?" sr))
	  (if (and m1 (= clmtest 0)) (snd-display ";oboe: mus-sound-maxamp-exists before maxamp: ~A" m1))
	  (if (not (mus-sound-maxamp-exists? "oboe.snd")) 
	      (snd-display ";oboe: mus-sound-maxamp-exists after maxamp: ~A" (mus-sound-maxamp-exists? "oboe.snd")))
	  (mus-audio-set-oss-buffers 4 12)
	  
	  (let ((str (strftime "%d-%b %H:%M %Z" (localtime (mus-sound-write-date "oboe.snd")))))
	    (if (not (string=? str "19-Oct 09:46 PDT"))
		(snd-display ";mus-sound-write-date oboe.snd: ~A?" str)))
	  (let ((str (strftime "%d-%b %H:%M %Z" (localtime (mus-sound-write-date "pistol.snd")))))
	    (if (not (string-=? str "19-Oct 09:46 PDT"))
		(snd-display ";mus-sound-write-date pistol.snd: ~A?" str)))
	  
	  (let ((index (open-sound "oboe.snd"))
		(long-file-name (let ((name "test"))
				  (do ((i 0 (1+ i)))
				      ((= i 33)) ; 40 is about the limit in Linux (256 char limit here from OS, not Snd)
				    (set! name (string-append name "-test")))
				  (string-append name ".snd"))))
	    (save-sound-as long-file-name index)
	    (close-sound index)
	    (set! index (open-sound long-file-name))
	    (if (not (sound? index)) (snd-display ";can't find test...snd"))
	    (if (or (not (>= (string-length (file-name index)) (string-length long-file-name)))
		    (not (>= (string-length (short-file-name index)) (string-length long-file-name))))
		(snd-display ";file-name lengths: ~A ~A ~A"
			     (string-length (file-name index))
			     (string-length (short-file-name index))
			     (string-length long-file-name)))
	    (close-sound index)
	    (mus-sound-forget long-file-name)
	    (delete-file long-file-name))

	  (let* ((fsnd (string-append sf-dir "forest.aiff")))
	    (if (file-exists? fsnd)
		(begin
		  (system (format #f "cp ~A fmv.snd" fsnd))
		  (let ((index (open-sound "fmv.snd")))
		    (if (not (equal? (sound-loop-info index) (mus-sound-loop-info fsnd)))
			(snd-display ";loop-info: ~A ~A" (sound-loop-info index) (mus-sound-loop-info fsnd)))
		    (set! (sound-loop-info index) (list 12000 14000 1 2 3 4))
		    (if (not (equal? (sound-loop-info index) (list 12000 14000 1 2 3 4 1 1)))
			(snd-display ";set loop-info: ~A" (sound-loop-info index)))
		    (save-sound-as "fmv1.snd" index mus-aifc)
		    (close-sound index)
		    (if (not (equal? (mus-sound-loop-info "fmv1.snd") (list 12000 14000 1 2 3 4 1 1)))
			(snd-display ";saved loop-info: ~A" (mus-sound-loop-info "fmv1.snd"))))))
	    (let ((index (open-sound "oboe.snd")))
	      (save-sound-as "fmv.snd" index mus-aifc)
	      (close-sound index))
	    (let ((index (open-sound "fmv.snd")))
	      (if (not (equal? (sound-loop-info index) '()))
		  (snd-display ";null loop-info: ~A" (sound-loop-info index)))
	      (set! (sound-loop-info index) (list 1200 1400 4 3 2 1))
	      (if (not (equal? (sound-loop-info index) (list 1200 1400 4 3 2 1 1 1)))
		  (snd-display ";set null loop-info: ~A" (sound-loop-info index)))
	      (save-sound-as "fmv1.snd" :sound index :header-type mus-aifc)
	      (close-sound index)
	      (if (not (equal? (mus-sound-loop-info "fmv1.snd") (list 1200 1400 4 3 2 1 1 1)))
		  (snd-display ";saved null loop-info: ~A" (mus-sound-loop-info "fmv1.snd"))))
	    (let ((index (open-sound "fmv.snd")))
	      (set! (sound-loop-info) (list 1200 1400 4 3 2 1 1 0))
	      (if (not (equal? (sound-loop-info index) (list 1200 1400 0 0 2 1 1 0)))
		  (snd-display ";set null loop-info (no mode1): ~A" (sound-loop-info index)))
	      (save-sound-as "fmv1.snd" index mus-aifc)
	      (close-sound index)
	      (if (not (equal? (mus-sound-loop-info "fmv1.snd") (list 1200 1400 0 0 2 1 1 0)))
		  (snd-display ";saved null loop-info (no mode1): ~A" (mus-sound-loop-info "fmv1.snd")))))
	  
	  (if (mus-audio-sun-outputs 1 2 3) (snd-display ";mus-audio-sun-outputs: ~A" (mus-audio-sun-outputs 1 2 3)))
	  
	  (if com (snd-display ";oboe: mus-sound-comment ~A?" com))
	  (let ((fsnd (string-append sf-dir "nasahal8.wav")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com 
							      (string-append "ICRD: 1997-02-22" 
									     (string #\newline)
									     "IENG: Paul R. Roger"
									     (string #\newline)
									     "ISFT: Sound Forge 4.0"
									     (string #\newline)))))
		      (snd-display ";mus-sound-comment \"nasahal8.wav\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "8svx-8.snd")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "File created by Sound Exchange  ")))
		      (snd-display ";mus-sound-comment \"8svx-8.snd\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "sun-16-afsp.snd")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "AFspdate:1981/02/11 23:03:34 UTC")))
		      (snd-display ";mus-sound-comment \"sun-16-afsp.snd\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "smp-16.snd")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "Converted using Sox.                                        ")))
		      (snd-display ";mus-sound-comment \"smp-16.snd\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "d40130.au")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "1994 Jesus Villena")))
		      (snd-display ";mus-sound-comment \"d40130.au\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "wood.maud")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "file written by SOX MAUD-export ")))
		      (snd-display ";mus-sound-comment \"wood.maud\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "addf8.sf_mipseb")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) 
			  (not (string-=? com "date=\"Feb 11 18:03:34 1981\" info=\"Original recorded at 20 kHz, 15-bit D/A, digitally filtered and resampled\" speaker=\"AMK female\" text=\"Add the sum to the product of these three.\" ")))
		      (snd-display ";mus-sound-comment \"addf8.sf_mipseb\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "mary-sun4.sig")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com (string-append "MARY HAD A LITTLE LAMB" (string #\newline)))))
		      (snd-display ";mus-sound-comment \"mary-sun4.sig\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "nasahal.pat")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "This patch saved with Sound Forge 3.0.")))
		      (snd-display ";mus-sound-comment \"nasahal.pat\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "next-16.snd")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) 
			  (not (string-=? com ";Written on Mon 1-Jul-91 at 12:10 PDT  at localhost (NeXT) using Allegro CL and clm of 25-June-91")))
		      (snd-display ";mus-sound-comment \"next-16.snd\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "wood16.nsp")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "Created by Snack   ")))
		      (snd-display ";mus-sound-comment \"wood16.nsp\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "wood.sdx")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "1994 Jesus Villena")))
		      (snd-display ";mus-sound-comment \"wood.sdx\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "clmcom.aif")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com "this is a comment")))
		      (snd-display ";mus-sound-comment \"clmcom.aif\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "anno.aif")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) (not (string-=? com (string-append "1994 Jesus Villena" (string #\newline)))))
		      (snd-display ";mus-sound-comment \"anno.aif\") -> ~A?" com)))))
	  (let ((fsnd (string-append sf-dir "telephone.wav")))
	    (if (file-exists? fsnd)
		(begin
		  (set! com (mus-sound-comment fsnd))
		  (if (or (not (string? com)) 
			  (not (string-=? com (string-append "sample_byte_format -s2 01"
							     (string #\newline)
							     "channel_count -i 1"
							     (string #\newline)
							     "sample_count -i 36461"
							     (string #\newline)
							     "sample_rate -i 16000"
							     (string #\newline)
							     "sample_n_bytes -i 2"
							     (string #\newline)
							     "sample_sig_bits -i 16"
							     (string #\newline)))))
		      (snd-display ";mus-sound-comment \"telephone.wav\") -> ~A?" com)))))

	  (if (not (string? (mus-sound-comment (string-append sf-dir "traffic.aiff"))))
	      (snd-display ";mus-sound-comment traffic: ~A" (mus-sound-comment (string-append sf-dir "traffic.aiff"))))
	  
	  (if (fneq (cadr mal) .14724) (snd-display ";oboe: mus-sound-maxamp ~F?" (cadr mal)))
	  (if (not (= (car mal) 24971)) (snd-display ";oboe: mus-sound-maxamp at ~D?" (car mal)))
	  (set! (mus-sound-maxamp "oboe.snd") (list 1234 .5))
	  (set! mal (mus-sound-maxamp "oboe.snd"))
	  (if (fneq (cadr mal) .5) (snd-display ";oboe: set! mus-sound-maxamp ~F?" (cadr mal)))
	  (if (not (= (car mal) 1234)) (snd-display ";oboe: set! mus-sound-maxamp at ~D?" (car mal)))
	  (set! mal (mus-sound-maxamp "4.aiff"))
	  (if (not (feql mal (list 810071 0.245 810071 0.490 810071 0.735 810071 0.980)))
	      (snd-display ";mus-sound-maxamp 4.aiff: ~A?" mal))
	  (set! (mus-sound-maxamp "4.aiff") (list 12345 .5 54321 .2 0 .1 9999 .01))
	  (set! mal (mus-sound-maxamp "4.aiff"))
	  (if (not (feql mal (list 12345 .5 54321 .2 0 .1 9999 .01)))
	      (snd-display ";set! mus-sound-maxamp 4.aiff: ~A?" mal))
	  (let ((var (catch #t (lambda () (set! (mus-sound-maxamp "oboe.snd") (list 1234))) (lambda args args))))
	    (if (not (eq? (car var) 'wrong-type-arg))
		(snd-display ";set! mus-sound-maxamp bad arg: ~A" var)))
	  (if (and (not (= (mus-sound-type-specifier "oboe.snd") #x646e732e))  ;little endian reader
		   (not (= (mus-sound-type-specifier "oboe.snd") #x2e736e64))) ;big endian reader
	      (snd-display ";oboe: mus-sound-type-specifier: ~X?" (mus-sound-type-specifier "oboe.snd")))
	  (if (not (string-=? (strftime "%d-%b-%Y %H:%M" (localtime (file-write-date "oboe.snd"))) "19-Oct-1998 09:46"))
	      (snd-display ";oboe: file-write-date: ~A?" (strftime "%d-%b-%Y %H:%M" (localtime (file-write-date "oboe.snd")))))
	  (play-sound-1 "oboe.snd")
	  
	  (let ((lasth (do ((i 1 (1+ i)))
			   ((string-=? (mus-header-type-name i) "unsupported") i))))
	    (if (< lasth 50) (snd-display ";header-type[~A] = ~A" lasth (mus-header-type-name lasth))))
	  (let ((lasth (do ((i 1 (1+ i)))
			   ((string-=? (mus-data-format-name i) "unknown") i))))
	    (if (< lasth 10) (snd-display ";data-format[~A] = ~A" lasth (mus-data-format-name lasth))))
	  
	  (set! (transform-normalization) dont-normalize)
	  (if (not (= (transform-normalization) dont-normalize))
	      (snd-display ";set-transform-normalization none -> ~A" (transform-normalization)))
	  (set! (transform-normalization) normalize-globally)
	  (if (not (= (transform-normalization) normalize-globally))
	      (snd-display ";set-transform-normalization globally -> ~A" (transform-normalization)))
	  (set! (transform-normalization) normalize-by-channel)
	  (if (not (= (transform-normalization) normalize-by-channel))
	      (snd-display ";set-transform-normalization channel -> ~A" (transform-normalization)))
	  
	  (let ((len 100))
	    (for-each
	     (lambda (type allowed-diff)
	       (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1))
		     (v (make-vct len))
		     (maxdiff 0.0)
		     (maxpos #f))
		 (vct-set! v 0 0.999)
		 (vct-set! v 1 -1.0)
		 (vct-set! v 2 .1)
		 (vct-set! v 3 -.1)
		 (vct-set! v 4 .01)
		 (vct-set! v 5 -.01)
		 (vct-set! v 4 .001)
		 (vct-set! v 5 -.001)
		 (vct-set! v 6 0.0)
		 (do ((i 7 (1+ i)))
		     ((= i len))
		   (vct-set! v i (- 1.0 (random 2.0))))
		 (vct->channel v 0 len ind 0)
		 (save-sound-as "test1.snd" ind mus-next :data-format type)
		 (close-sound ind)
		 (set! ind (open-sound "test1.snd"))
		 (let ((v1 (channel->vct 0 len ind 0)))
		   (do ((i 0 (1+ i)))
		       ((= i len))
		     (let ((diff (abs (- (vct-ref v i) (vct-ref v1 i)))))
		       (if (> diff maxdiff)
			   (begin
			     (set! maxdiff diff)
			     (set! maxpos i)))))
		   (if (> maxdiff allowed-diff)
		       (snd-display ";~A: ~A at ~A (~A ~A)" (mus-data-format-name type) maxdiff maxpos (vct-ref v maxpos) (vct-ref v1 maxpos)))
		   (close-sound ind))))
	     (list mus-bshort   mus-lshort   mus-mulaw   mus-alaw   mus-byte  
		   mus-lfloat   mus-bint     mus-lint    mus-b24int mus-l24int
		   mus-ubshort  mus-ulshort  mus-ubyte   mus-bfloat mus-bdouble 
		   mus-ldouble)
	     (list (expt 2 -15) (expt 2 -15) 0.02 0.02 (expt 2 -7)
		   (expt 2 -23) (expt 2 -23) (expt 2 -23) (expt 2 -23) (expt 2 -23) ; assuming sndlib bits=24 here (if int)
		   (expt 2 -15) (expt 2 -15) (expt 2 -7) (expt 2 -23) (expt 2 -23)
		   (expt 2 -23))))
	  
	  (let* ((ob (view-sound "oboe.snd"))
		 (samp (sample 1000 ob))
		 (old-comment (mus-sound-comment "oboe.snd"))
		 (str (string-append "written " 
				     (strftime "%a %d-%b-%Y %H:%M %Z" 
					       (localtime (current-time))))))
	    (set! (comment ob) str)
	    (save-sound-as "test.snd" ob mus-aifc mus-bdouble)
	    (set! (filter-control-in-hz) #t)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug))
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (provided? 'snd-debug)
		  (begin
		    (if (snd-sound-pointer 12345) (snd-display ";snd-sound-pointer #f: ~A" (snd-sound-pointer 12345)))
		    (if (not (number? (snd-sound-pointer ab))) (snd-display ";snd-sound-pointer ~A: ~A" ab (snd-sound-pointer ab)))))
	      (if (not (= (header-type ab) mus-aifc)) 
		  (snd-display ";save-as aifc -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-aifc)) 
		  (snd-display ";saved-as aifc -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";aifc[1000] = ~A?" (sample 1000 ab)))
	      (if (not (string-=? (mus-sound-comment "test.snd") str))
		  (snd-display ";output-comment: ~A ~A" (mus-sound-comment "test.snd") str))
	      (if (or (not (string? (comment ab)))
		      (not (string-=? (comment ab) str)))
		  (snd-display ";output-comment (comment): ~A ~A" (comment ab) str))
	      (close-sound ab))
	    (if (not (equal? old-comment (mus-sound-comment "oboe.snd")))
		(snd-display ";set-comment overwrote current ~A ~A" old-comment (mus-sound-comment "oboe.snd")))
	    (set! (filter-control-in-hz) #f)
	    (save-sound-as "test.snd" ob mus-raw)
	    (let ((ab (open-raw-sound "test.snd" 1 22050 mus-bshort)))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-raw)) 
		  (snd-display ";save-as raw -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-raw)) 
		  (snd-display ";saved-as raw -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";raw[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-nist mus-bint)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-nist)) 
		  (snd-display ";save-as nist -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-nist)) 
		  (snd-display ";saved-as nist -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-bint)) 
		  (snd-display ";save-as int -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-bint)) 
		  (snd-display ";saved-as int -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";nist[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (reset-hook! output-comment-hook)
	    (add-hook! output-comment-hook
		       (lambda (str) 
			 (string-append str " [written by me]")))
	    (save-sound-as :file "test.snd" :sound ob :header-type mus-riff :data-format mus-lfloat)
	    (reset-hook! output-comment-hook)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-riff)) 
		  (snd-display ";save-as riff -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-riff)) 
		  (snd-display ";saved-as riff -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-lfloat)) 
		  (snd-display ";save-as float -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-lfloat)) 
		  (snd-display ";saved-as float -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";riff[1000] = ~A?" (sample 1000 ab)))
	      (if (or (not (string? (comment ab)))
		      (not (string-=? (comment ab) 
				     (string-append "written " 
						    (strftime "%a %d-%b-%Y %H:%M %Z" (localtime (current-time)))
						    " [written by me]"))))
		  (snd-display ";output-comment-hook: ~A~%(~A)" (comment ab) (mus-sound-comment "test.snd")))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-aiff mus-b24int)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-aiff)) 
		  (snd-display ";save-as aiff -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-aiff)) 
		  (snd-display ";saved-as aiff -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-b24int))
		  (snd-display ";save-as 24-bit -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-b24int))
		  (snd-display ";saved-as 24-bit -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";aiff[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-ircam mus-mulaw)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-ircam)) 
		  (snd-display ";save-as ircam -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-ircam)) 
		  (snd-display ";saved-as ircam -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-mulaw))
		  (snd-display ";save-as mulaw -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-mulaw))
		  (snd-display ";saved-as mulaw -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";ircam[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-next mus-alaw)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-next)) 
		  (snd-display ";save-as next -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-next)) 
		  (snd-display ";saved-as next -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-alaw)) 
		  (snd-display ";save-as alaw -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-alaw)) 
		  (snd-display ";saved-as alaw -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";next (alaw)[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-next mus-bdouble)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug))
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-next)) 
		  (snd-display ";save-as dbl next -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (data-format ab) mus-bdouble)) 
		  (snd-display ";save-as dbl -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";next (dbl)[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-next mus-bshort)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug))
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-next)) 
		  (snd-display ";save-as next -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-next)) 
		  (snd-display ";saved-as next -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-bshort)) 
		  (snd-display ";save-as short -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-bshort)) 
		  (snd-display ";saved-as short -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";next (short)[1000] = ~A?" (sample 1000 ab)))
	      (set! (y-bounds ab 0) (list -3.0 3.0))
	      (set! (data-format ab) mus-lshort)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd"))) ; these set!'s can change the index via update-sound
	      (if (not (= (data-format ab) mus-lshort)) (snd-display ";set data-format: ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (equal? (y-bounds ab 0) (list -3.0 3.0))) (snd-display ";set data format y-bounds: ~A?" (y-bounds ab 0)))
	      (set! (y-bounds ab 0) (list 2.0))
	      (if (not (equal? (y-bounds ab 0) (list -2.0 2.0))) (snd-display ";set data format y-bounds 1: ~A?" (y-bounds ab 0)))
	      (set! (y-bounds ab 0) (list -2.0))
	      (if (not (equal? (y-bounds ab 0) (list -2.0 2.0))) (snd-display ";set data format y-bounds -2: ~A?" (y-bounds ab 0)))
	      (set! (header-type ab) mus-aifc)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (header-type ab) mus-aifc)) (snd-display ";set header-type: ~A?" (mus-header-type-name (header-type ab))))
	      (set! (channels ab) 3)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (channels ab) 3)) (snd-display ";set chans: ~A?" (channels ab)))
	      (set! (data-location ab) 1234)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (data-location ab) 1234)) (snd-display ";set data-location: ~A?" (data-location ab)))
	      (let ((old-size (data-size ab)))
		(set! (data-size ab) 1234)
		(if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
		(if (not (= (data-size ab) 1234)) (snd-display ";set data-size: ~A?" (data-size ab)))
		(set! (data-size ab) old-size))
	      (set! (srate ab) 12345)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (srate ab) 12345)) (snd-display ";set srate: ~A?" (srate ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-next mus-bfloat)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (if (not (= (header-type ab) mus-next)) 
		  (snd-display ";save-as next -> ~A?" (mus-header-type-name (header-type ab))))
	      (if (not (= (mus-sound-header-type "test.snd") mus-next)) 
		  (snd-display ";saved-as next -> ~A?" (mus-header-type-name (mus-sound-header-type "test.snd"))))
	      (if (not (= (data-format ab) mus-bfloat)) 
		  (snd-display ";save-as float -> ~A?" (mus-data-format-name (data-format ab))))
	      (if (not (= (mus-sound-data-format "test.snd") mus-bfloat)) 
		  (snd-display ";saved-as float -> ~A?" (mus-data-format-name (mus-sound-data-format "test.snd"))))
	      (if (fneq (sample 1000 ab) samp) (snd-display ";next (float)[1000] = ~A?" (sample 1000 ab)))
	      (close-sound ab))
	    (save-sound-as "test.snd" ob mus-next mus-bshort)
	    (close-sound ob)
	    (let ((ab (open-sound "test.snd")))
	      (if (and (provided? 'xm) (provided? 'snd-debug)) 
		  (XtCallCallbacks (cadr (sound-widgets ab)) XmNactivateCallback (snd-sound-pointer ab)))
	      (set! (data-format) mus-lshort)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (data-format) mus-lshort)) (snd-display ";set data-format: ~A?" (mus-data-format-name (data-format))))
	      (set! (header-type) mus-aifc)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (header-type) mus-aifc)) (snd-display ";set header-type: ~A?" (mus-header-type-name (header-type))))
	      (set! (channels) 3)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (channels) 3)) (snd-display ";set chans: ~A?" (channels)))
	      (set! (data-location) 1234)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (data-location) 1234)) (snd-display ";set data-location: ~A?" (data-location)))
	      (set! (srate) 12345)
	      (if (not (= ab (find-sound "test.snd"))) (set! ab (find-sound "test.snd")))
	      (if (not (= (srate) 12345)) (snd-display ";set srate: ~A?" (srate)))
	      (close-sound ab)))
	  (let ((fsnd (string-append sf-dir "t15.aiff")))
	    (if (file-exists? fsnd)
		(let ((ind (open-sound fsnd)))
		  (if (or (fneq (sample 132300 ind 0) .148)
			  (fneq (sample 132300 ind 1) .126))
		      (snd-display ";aifc sowt trouble: ~A ~A" (sample 132300 ind 0) (sample 132300 ind 1)))
		  (close-sound ind))))
	  (let ((fsnd (string-append sf-dir "M1F1-float64C-AFsp.aif")))
	    (if (file-exists? fsnd)
		(let ((ind (open-sound fsnd)))
		  (if (or (fneq (sample 8000 ind 0) -0.024)
			  (fneq (sample 8000 ind 1) 0.021))
		      (snd-display ";aifc fl64 trouble: ~A ~A" (sample 8000 ind 0) (sample 8000 ind 1)))
		  (close-sound ind))))
	  
	  (for-each (lambda (n vals)
		      (let ((val (catch #t (lambda () 
					     (list (mus-sound-chans n)
						   (mus-sound-srate n)
						   (mus-sound-frames n)))
					(lambda args (car args)))))
			(if (and (not (equal? val vals))
				 (not (eq? val 'mus-error)))
			    (snd-display ";~A: ~A ~A" n val vals))))
		    (list (string-append sf-dir "bad_chans.snd")
			  (string-append sf-dir "bad_srate.snd")
			  (string-append sf-dir "bad_data_format.snd")
			  (string-append sf-dir "bad_chans.aifc")
			  (string-append sf-dir "bad_srate.aifc")
			  (string-append sf-dir "bad_length.aifc")
			  (string-append sf-dir "bad_chans.riff")
			  (string-append sf-dir "bad_srate.riff")
			  (string-append sf-dir "bad_chans.nist")
			  (string-append sf-dir "bad_srate.nist")
			  (string-append sf-dir "bad_length.nist"))
		    (list (list 0 22050 0)
			  (list 1 0 0)
			  (list 1 22050 4411)
			  (list 0 22050 0)
			  (list 1 0 0)
			  (list 1 22050 -10)
			  (list 0 22050 0)
			  (list 1 0 0)
			  (list 0 22050 0)
			  (list 1 0 0)
			  (list 1 22050 -10)))
	  
	  (let ((ind (open-sound (string-append "/usr/include/sys/" home-dir "/cl/oboe.snd"))))
	    (if (or (not (sound? ind))
		    (not (string=? (short-file-name ind) "oboe.snd")))
		(snd-display ";open-sound with slashes: ~A ~A" ind (and (sound? ind) (short-file-name ind))))
	    (add-hook! bad-header-hook (lambda (n) #t))
	    (for-each (lambda (n)
			(begin
			  (catch #t (lambda () 
				      (insert-sound n))
				 (lambda args (car args)))
			  (catch #t (lambda () 
				      (convolve-with n))
				 (lambda args (car args)))
			  (catch #t (lambda () 
				      (mix n))
				 (lambda args (car args)))
			  (catch #t (lambda () 
				      (let ((ind (open-sound n)))
					(if (and (number? ind)
						 (sound? ind))
					    (close-sound ind))))
				 (lambda args (car args)))))
		      (list "/home/bil/sf1/bad_chans.snd"
			    "/home/bil/sf1/bad_srate.snd"
			    "/home/bil/sf1/bad_chans.aifc"
			    "/home/bil/sf1/bad_srate.aifc"
			    "/home/bil/sf1/bad_length.aifc"
			    "/home/bil/sf1/bad_chans.riff"
			    "/home/bil/sf1/bad_srate.riff"
			    "/home/bil/sf1/bad_chans.nist"
			    "/home/bil/sf1/bad_location.nist"
			    "/home/bil/sf1/bad_field.nist"
			    "/home/bil/sf1/bad_srate.nist"
			    "/home/bil/sf1/bad_length.nist"))
	    (close-sound ind))
	  
	  (let* ((ob (open-sound (string-append "~/baddy/" home-dir "/cl/oboe.snd")))
		 (sd (samples->sound-data))
		 (mx (sound-data-maxamp sd)))
	    (if (not (= (sound-data-length sd) 50828)) (snd-display ";oboe->sd: len ~A?" (sound-data-length sd)))
	    (if (fneq (sound-data-ref sd 0 1000) .0328369) (snd-display ";oboe->sd[1000]: ~A?" (sound-data-ref sd 0 1000)))
	    (if (not (= (length mx) 1)) (snd-display ";sound-data-maxamp oboe.snd: ~A?" (sound-data-maxamp sd)))
	    (if (not (= (maxamp ob 0) (car mx))) (snd-display ";sound-data-maxamp oboe.snd: ~A ~A?" (sound-data-maxamp sd) (maxamp ob 0)))
	    
	    (let ((var (catch #t (lambda () (set! (selected-channel) 1)) (lambda args args))))
	      (if (not (eq? (car var) 'no-such-channel))
		  (snd-display ";set selected-channel bad chan: ~A" var)))
	    (let ((var (catch #t (lambda () (set! (selected-channel 123456) 1)) (lambda args args))))
	      (if (not (eq? (car var) 'no-such-sound))
		  (snd-display ";set selected-channel bad snd: ~A" var)))
	    (let ((var (catch #t (lambda () (sound-data-ref sd 2 1000)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";sound-data-ref bad chan: ~A" var)))
	    (let ((var (catch #t (lambda () (sound-data-ref sd -1 1000)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";sound-data-ref bad chan -1: ~A" var)))
	    (let ((var (catch #t (lambda () (sound-data-ref sd 0 -1)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";sound-data-ref bad frame: ~A" var)))
	    (let ((var (catch #t (lambda () (sound-data-ref sd 0 10000000)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";sound-data-ref bad frame high: ~A" var)))
	    (let ((var (catch #t (lambda () (sound-data-set! sd 2 1000 1)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";sound-data-set! bad chan: ~A" var)))
	    (let ((var (catch #t (lambda () (sound-data-set! sd 0 10000000 1)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";sound-data-set! bad frame: ~A" var)))
	    (let* ((v (make-vct 3))
		   (var (catch #t (lambda () (vct->sound-data v sd 2)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct->sound-data-set! bad chan: ~A" var)))
	    (close-sound ob))
	  (if (selected-sound)
	      (snd-display ";selected-sound ~A ~A" (selected-sound) (sounds)))
	  
	  (let* ((vals (make-vector 32))
		 (err (mus-audio-mixer-read mus-audio-microphone mus-audio-amp 0 vals)))
	    (if (= err -1) 
		(snd-display ";mus-audio-mixer-read?")
		(begin
		  (mus-audio-save)
		  (let ((old-val (vector-ref vals 0)))
		    (vector-set! vals 0 .5)
		    (set! err (mus-audio-mixer-write mus-audio-microphone mus-audio-amp 0 vals))
		    (if (= err -1) 
			(snd-display ";mus-audio-mixer-write?"))
		    (clear-audio-inputs)
		    (mus-audio-restore)
		    (mus-audio-mixer-read mus-audio-microphone mus-audio-amp 0 vals)
		    (if (fneq (vector-ref vals 0) old-val) (snd-display ";mus-audio-restore: ~A ~A?" old-val (vector-ref vals 0))))))
	    (for-each 
	     (lambda (field)
	       (for-each
		(lambda (device)
		  (if (not (= (mus-audio-mixer-read device field 0 vals) -1))
		      (mus-audio-mixer-write device field 0 vals)))
		(list mus-audio-default mus-audio-duplex-default mus-audio-line-out mus-audio-line-in mus-audio-microphone
		      mus-audio-speakers mus-audio-dac-out mus-audio-adat-in mus-audio-aes-in mus-audio-digital-in
		      mus-audio-digital-out mus-audio-adat-out mus-audio-aes-out mus-audio-dac-filter mus-audio-mixer
		      mus-audio-line1 mus-audio-line2 mus-audio-line3 mus-audio-aux-input mus-audio-cd mus-audio-aux-output
		      mus-audio-spdif-in mus-audio-spdif-out)))
	     (list mus-audio-amp mus-audio-srate mus-audio-channel mus-audio-format mus-audio-port mus-audio-imix
		   mus-audio-igain mus-audio-reclev mus-audio-pcm mus-audio-pcm2 mus-audio-ogain mus-audio-line
		   mus-audio-line1 mus-audio-line2 mus-audio-line3 mus-audio-cd
		   mus-audio-synth mus-audio-bass mus-audio-treble mus-audio-direction mus-audio-samples-per-channel))
	    )
	  
	  (if (file-exists? (string-append (or sf-dir "") "a.sf2"))
	      (let ((fil (open-sound (string-append (or sf-dir "") "a.sf2"))))
		(if fil
		    (let ((loops (and fil (soundfont-info))))
		      (if (or (null? loops)
			      (not (= (caddar loops) 65390))
			      (not (= (cadadr loops) 65490)))
			  (snd-display ";soundfont-info: ~A?" loops))
		      (close-sound fil)))))
	  
	  (if (file-exists? "fmv5.snd") (delete-file "fmv5.snd"))
	  (let ((fd (mus-sound-open-output "fmv5.snd" 22050 1 mus-bshort mus-aiff "no comment"))
		(sdata (make-sound-data 1 100)))
	    (do ((i 0 (1+ i)))
		((= i 100))
	      (sound-data-set! sdata 0 i (* i .01)))
	    (if (not (string-=? "#<sound-data: 1 chan, 100 frames>" (format #f "~A" sdata)))
		(snd-display ";print sound-data: ~A?" (format #f "~A" sdata)))
	    (let ((edat sdata)
		  (edat1 (make-sound-data 1 100))
		  (edat2 (make-sound-data 2 100)))
	      (if (not (eq? sdata edat)) (snd-display ";sound-data not eq? ~A ~A" sdata edat))
	      (if (not (equal? sdata edat)) (snd-display ";sound-data not equal? ~A ~A" sdata edat))
	      (if (equal? sdata edat1) (snd-display ";sound-data 1 equal? ~A ~A" sdata edat1))
	      (if (equal? edat2 edat1) (snd-display ";sound-data 2 equal? ~A ~A" edat2 edat1))
	      (do ((i 0 (1+ i)))
		  ((= i 100))
		(set! (sound-data-ref edat1 0 i) (sound-data-ref sdata 0 i)))
	      (if (not (equal? sdata edat1)) (snd-display ";sound-data 3 not equal? ~A ~A" sdata edat1)))
	    (let ((v0 (make-vct 100))
		  (v1 (make-vct 3)))
	      (sound-data->vct sdata 0 v0) 
	      (if (fneq (vct-ref v0 10) .1) (snd-display ";sound-data->vct: ~A?" v0))
	      (sound-data->vct sdata 0 v1) 
	      (if (fneq (vct-ref v1 1) .01) (snd-display ";sound-data->(small)vct: ~A?" v1))
	      (vct->sound-data v0 sdata 0) 
	      (if (fneq (sound-data-ref sdata 0 10) .1) (snd-display ";vct->sound-data: ~A?" (sound-data-ref sdata 0 10)))
	      (let ((var (catch #t (lambda () (sound-data->vct sdata 2 v0)) (lambda args args))))
		(if (not (eq? (car var) 'out-of-range))
		    (snd-display ";sound-data->vct bad chan: ~A" var)))
	      (let ((var (catch #t (lambda () (mus-audio-write 1 (make-sound-data 3 3) 123)) (lambda args args))))
		(if (not (eq? (car var) 'out-of-range))
		    (snd-display ";mus-audio-write bad frames: ~A" var))))
	    
	    (let ((v0 (make-vct 10))
		  (sdata2 (make-sound-data 2 10)))
	      (do ((i 0 (1+ i)))
		  ((= i 10))
		(sound-data-set! sdata2 0 i 0.1)
		(sound-data-set! sdata2 1 i 0.2))
	      (sound-data->vct sdata2 0 v0) 
	      (if (fneq (vct-ref v0 1) .1) (snd-display ";sound-data->vct[1]: ~A?" v0))
	      (sound-data->vct sdata2 1 v0) 
	      (if (fneq (vct-ref v0 1) .2) (snd-display ";sound-data->vct[2]: ~A?" v0))
	      (vct->sound-data v0 sdata2 0) 
	      (if (fneq (sound-data-ref sdata2 0 1) .2) 
		  (snd-display ";vct->sound-data[2]: ~A?" (sound-data-ref sdata2 0 1)))
	      (vct-fill! v0 .3)
	      (vct->sound-data v0 sdata2 1) 
	      (if (fneq (sound-data-ref sdata2 1 1) .3) 
		  (snd-display ";vct->sound-data[3]: ~A?" (sound-data-ref sdata2 1 1))))
	    (mus-sound-write fd 0 99 1 sdata)
	    (mus-sound-close-output fd 200)
	    (set! fd (mus-sound-reopen-output "fmv5.snd" 1 mus-bshort mus-aiff (mus-sound-data-location "fmv5.snd")))
	    (mus-sound-close-output fd 200)
	    (set! fd (mus-sound-open-input "fmv5.snd"))
	    (mus-sound-read fd 0 99 1 sdata)
	    (if (fneq (sound-data-ref sdata 0 10) .1) (snd-display ";mus-sound-write: ~A?" (sound-data-ref sdata 0 10)))
	    (let ((pos (mus-sound-seek-frame fd 20)))
	      (if (not (= pos (ftell fd))) 
		  (snd-display ";1 mus-sound-seek-frame: ~A ~A?" pos (ftell fd)))
	      (if (not (= pos (frame->byte "fmv5.snd" 20)))
		  (snd-display ";2 mus-sound-seek-frame(2): ~A ~A?" pos (frame->byte "fmv5.snd" 20))))
	    (mus-sound-read fd 0 10 1 sdata)
	    (if (fneq (sound-data-ref sdata 0 0) .2) (snd-display ";2 mus-sound-seek: ~A?" (sound-data-ref sdata 0 0)))
	    (mus-sound-close-input fd))
	  
	  (let ((var (catch #t (lambda () (mus-sound-open-output "fmv.snd" 22050 -1 mus-bshort mus-aiff "no comment")) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";mus-sound-open-output bad chans: ~A" var)))
	  (let ((var (catch #t (lambda () (mus-sound-open-output "fmv.snd" 22050 1 -1 mus-aiff "no comment")) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";mus-sound-open-output bad format: ~A" var)))
	  (let ((var (catch #t (lambda () (mus-sound-open-output "fmv.snd" 22050 1 mus-bshort -1 "no comment")) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";mus-sound-open-output bad type: ~A" var)))
	  
	  (let ((var (catch #t (lambda () (mus-sound-reopen-output "fmv.snd" -1 mus-bshort mus-aiff #f)) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";mus-sound-reopen-output bad chans: ~A" var)))
	  (let ((var (catch #t (lambda () (mus-sound-reopen-output "fmv.snd" 1 -1 mus-aiff #f)) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";mus-sound-reopen-output bad format: ~A" var)))
	  (let ((var (catch #t (lambda () (mus-sound-reopen-output "fmv.snd" 1 mus-bshort -1 #f)) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";mus-sound-reopen-output bad type: ~A" var)))
	  
	  (for-each
	   (lambda (proc name)
	     (let ((var (catch #t (lambda () (proc mus-audio-default 22050 -1 mus-lshort 512)) (lambda args args))))
	       (if (not (eq? (car var) 'out-of-range))
		   (snd-display ";~A bad chans: ~A" name var)))
	     (let ((var (catch #t (lambda () (proc mus-audio-default 22050 1 -1 512)) (lambda args args))))
	       (if (not (eq? (car var) 'out-of-range))
		   (snd-display ";~A bad format: ~A" name var)))
	     (let ((var (catch #t (lambda () (proc -1 22050 1 mus-lshort 512)) (lambda args args))))
	       (if (not (eq? (car var) 'out-of-range))
		   (snd-display ";~A bad device: ~A" name var)))
	     (let ((var (catch #t (lambda () (proc mus-audio-default -22050 1 mus-lshort 512)) (lambda args args))))
	       (if (not (eq? (car var) 'out-of-range))
		   (snd-display ";~A bad srate: ~A" name var)))
	     (let ((var (catch #t (lambda () (proc mus-audio-default 22050 1 mus-lshort -512)) (lambda args args))))
	       (if (not (eq? (car var) 'out-of-range))
		   (snd-display ";~A bad size: ~A" name var))))
	   (list mus-audio-open-output mus-audio-open-input)
	   (list "mus-audio-open-output" "mus-audio-open-input"))
	  
	  (for-each
	   (lambda (file)
	     (let ((tag (catch #t
			       (lambda () (open-sound (string-append sf-dir file)))
			       (lambda args args))))
	       (if (not (eq? (car tag) 'mus-error))
		   (snd-display ";open-sound ~A: ~A" file tag))))
	   (list "trunc.snd" "trunc.aiff" "trunc.wav" "trunc.sf" "trunc.voc" "trunc.nist" "bad.wav" 
		 "trunc1.aiff" "badform.aiff"))
	  (add-hook! open-raw-sound-hook (lambda (file choice) (list 1 22050 mus-bshort)))
	  (let ((ind (open-sound (string-append sf-dir "empty.snd"))))
	    (if (or (not (= (data-format ind) mus-bshort))
		    (not (= (chans ind) 1))
		    (not (= (srate ind) 22050))
		    (not (= (data-location ind) 0))
		    (not (= (frames ind) 0)))
		(snd-display ";open raw: ~A ~A ~A ~A ~A" (data-format ind) (chans ind) (srate ind) (data-location ind) (frames ind)))
	    (reset-hook! open-raw-sound-hook)
	    (close-sound ind))
	  
	  (let ((vals (make-vector 32)))
	    (for-each 
	     (lambda (proc name)
	       (let ((var (catch #t (lambda () (proc -1 mus-audio-amp 0 vals)) (lambda args args))))
		 (if (not (eq? (car var) 'out-of-range))
		     (snd-display ";~A bad device: ~A" name var)))
	       (let ((var (catch #t (lambda () (proc mus-audio-microphone -1 0 vals)) (lambda args args))))
		 (if (not (eq? (car var) 'out-of-range))
		     (snd-display ";~A bad field: ~A" name var))))
	     (list mus-audio-mixer-read mus-audio-mixer-write)
	     (list "mus-audio-mixer-read" "mus-audio-mixer-write")))
	  (mus-audio-mixer-write mus-audio-microphone mus-audio-amp 0 (make-vector 0))
	  
	  (let* ((ind (open-sound (string-append "/usr/local/" (getcwd) "/2.snd"))) ; check the "//" path reset case
		 (sd1 (samples->sound-data 12000 10 ind 0))
		 (vc1 (sound-data->vct sd1))
		 (vc2 (samples->vct 12000 10 ind 0))
		 (sd2 (vct->sound-data vc2)))
	    (if (not (equal? vc1 vc2)) (snd-display ";samples->sound-data->vct: ~A ~A" vc1 vc2))
	    (if (not (equal? sd1 sd2)) (snd-display ";sound-data->vct->sound-data: ~A ~A" sd1 sd2))
	    (scale-by 2.0 ind 0)
	    (set! sd1 (samples->sound-data 12000 10 ind 0 #f 0))
	    (set! vc1 (sound-data->vct sd1))
	    (set! vc2 (samples->vct 12000 10 ind 0 #f 0))
	    (set! sd2 (vct->sound-data vc2))
	    (if (not (equal? vc1 vc2)) (snd-display ";edpos samples->sound-data->vct: ~A ~A" vc1 vc2))
	    (if (not (equal? sd1 sd2)) (snd-display ";edpos sound-data->vct->sound-data: ~A ~A" sd1 sd2))
	    (set! sd1 (samples->sound-data 12000 10 ind 1))
	    (set! vc1 (sound-data->vct sd1))
	    (set! vc2 (samples->vct 12000 10 ind 1))
	    (set! sd2 (vct->sound-data vc2))
	    (if (not (equal? vc1 vc2)) (snd-display ";1 samples->sound-data->vct: ~A ~A" vc1 vc2))
	    (if (not (equal? sd1 sd2)) (snd-display ";1 sound-data->vct->sound-data: ~A ~A" sd1 sd2))
	    (scale-by 2.0 ind 1)
	    (set! sd1 (samples->sound-data 12000 10 ind 1))
	    (set! vc1 (sound-data->vct sd1))
	    (set! vc2 (samples->vct 12000 10 ind 1))
	    (set! sd2 (vct->sound-data vc2))
	    (if (not (equal? vc1 vc2)) (snd-display ";1 scaled samples->sound-data->vct: ~A ~A" vc1 vc2))
	    (if (not (equal? sd1 sd2)) (snd-display ";1 scaled sound-data->vct->sound-data: ~A ~A" sd1 sd2))
	    (close-sound ind))
	  
	  (for-each 
	   (lambda (chans)
	     (for-each 
	      (lambda (df-ht)
		(let ((samps (if (= chans 1) 100000
				 (if (= chans 2) 50000
				     1000))))
		  (if (file-exists? "fmv5.snd") (delete-file "fmv5.snd"))
		  (let ((fd (mus-sound-open-output "fmv5.snd" 22050 chans (car df-ht) (cadr df-ht) "no comment"))
			(sdata (make-sound-data chans samps))
			(ndata (make-sound-data chans samps)))
		    (run 
		     (lambda ()
		       (do ((k 0 (1+ k)))
			   ((= k chans))
			 (do ((i 0 (1+ i)))
			     ((= i samps))
			   (sound-data-set! sdata k i (- (random 2.0) 1.0))))))
		    (mus-sound-write fd 0 (- samps 1) chans sdata)
		    (mus-sound-close-output fd (* samps chans (mus-bytes-per-sample (car df-ht))))
		    (set! fd (mus-sound-open-input "fmv5.snd"))
		    (mus-sound-read fd 0 (- samps 1) chans ndata)
		    (let ((pos (mus-sound-seek-frame fd 100)))
		      (if (not (= pos (ftell fd))) 
			  (snd-display ";mus-sound-seek-frame[~A]: chans ~A ~A (~A ~A)?" 
				       pos chans (ftell fd) (mus-header-type-name (cadr df-ht)) (mus-data-format-name (car df-ht))))
		      (if (not (= pos (frame->byte "fmv5.snd" 100))) 
			  (snd-display ";mus-sound-seek-frame(100): ~A ~A (~A ~A ~A)?" 
				       pos (frame->byte "fmv5.snd" 100) chans (mus-header-type-name (cadr df-ht)) (mus-data-format-name (car df-ht)))))
		    (mus-sound-close-input fd)
		    (catch #t
			   (lambda ()
			     (run
			      (lambda ()
				(do ((k 0 (1+ k)))
				    ((= k chans))
				  (do ((i 0 (1+ i)))
				      ((= i samps))
				    (if (fneq (sound-data-ref sdata k i) (sound-data-ref ndata k i))
					(throw 'read-write-error)))))))
			   (lambda args 
			     (begin 
			       (snd-display ";read-write trouble: ~A ~A"
					    (mus-data-format-name (car df-ht))
					    (mus-header-type-name (cadr df-ht)))
			       (car args)))))))
	      (list (list mus-bshort mus-next)
		    (list mus-bfloat mus-aifc)
		    (list mus-lshort mus-aifc)
		    (list mus-lfloat mus-riff)
		    (list mus-lshort mus-nist)
		    (list mus-bint mus-aiff)
		    (list mus-lint mus-next)
		    (list mus-bintn mus-next)
		    (list mus-lintn mus-next)
		    (list mus-b24int mus-aifc)
		    (list mus-l24int mus-riff)
		    (list mus-bfloat mus-ircam)
		    (list mus-bfloat-unscaled mus-next)
		    (list mus-lfloat-unscaled mus-next)
		    (list mus-bdouble-unscaled mus-next)
		    (list mus-ldouble-unscaled mus-next)
		    (list mus-bdouble mus-next)
		    (list mus-ldouble mus-next)
		    (list mus-ulshort mus-next)
		    (list mus-ubshort mus-next))))
	   (list 1 2 4 8))
	  
	  (let ((ind (open-sound (string-append "/usr//usr/include/" home-dir "/cl/oboe.snd"))))
	    (show-input-1)
	    (close-sound ind))
	  
	  (let ((fd (mus-sound-open-output "fmv.snd" 22050 1 mus-bshort mus-next "no comment"))
		(sdata (make-sound-data 1 10)))
	    (define (sound-data-channel->list sd chan)
	      (let ((ls '()))
		(do ((i (1- (sound-data-length sd)) (1- i)))
		    ((< i 0) ls)
		  (set! ls (cons (sound-data-ref sd chan i) ls)))))
	    (define (sound-data->list sd)
	      (let ((lst '()))
		(do ((i (1- (sound-data-chans sd)) (1- i)))
		    ((< i 0) lst)
		  (set! lst (cons (sound-data-channel->list sd i) lst)))))
	    (sound-data-set! sdata 0 1 .1)
	    (mus-sound-write fd 0 9 1 sdata)
	    (mus-sound-close-output fd 20)
	    (set! fd (mus-sound-open-input "fmv.snd"))
	    (mus-sound-read fd 0 9 1 sdata)
	    (if (or (fneq (sound-data-ref sdata 0 0) 0.0)
		    (fneq (sound-data-ref sdata 0 1) 0.1)
		    (fneq (sound-data-ref sdata 0 2) 0.0)
		    (fneq (sound-data-ref sdata 0 6) 0.0))
		(snd-display ";read/write: ~A?" (sound-data->list sdata)))
	    (mus-sound-close-input fd)  
	    (set! fd (mus-sound-reopen-output "fmv.snd" 1 mus-bshort mus-next (mus-sound-data-location "fmv.snd")))
	    (mus-sound-seek-frame fd 0)
	    (sound-data-set! sdata 0 2 .1)
	    (sound-data-set! sdata 0 3 .1)
	    (mus-sound-write fd 0 9 1 sdata)
	    (mus-sound-close-output fd 20)
	    (set! fd (mus-sound-open-input "fmv.snd"))
	    (mus-sound-read fd 0 9 1 sdata)
	    (if (or (fneq (sound-data-ref sdata 0 0) 0.0)
		    (fneq (sound-data-ref sdata 0 1) 0.1)
		    (fneq (sound-data-ref sdata 0 2) 0.1)
		    (fneq (sound-data-ref sdata 0 3) 0.1)
		    (fneq (sound-data-ref sdata 0 6) 0.0))
		(snd-display ";re-read/write: ~A?" (sound-data->list sdata)))
	    (mus-sound-close-input fd)
	    
	    ;; check data-clipped choices
	    (let ((ind (view-sound "oboe.snd")))
	      (set! (data-clipped) #f)
	      (map-channel (lambda (y) (* y 10.0)) 0 (frames) ind 0)
	      (save-sound-as "test.snd" ind mus-next mus-bfloat)
	      (undo 1 ind 0)
	      (let ((ind1 (open-sound "test.snd")))
		(if (fneq (maxamp ind1 0) (* 10 (maxamp ind 0)))
		    (snd-display ";clipping 0: ~A ~A" (maxamp ind1 0) (maxamp ind 0)))
		(close-sound ind1))
	      (delete-file "test.snd")
	      (set! (data-clipped) #t)
	      (map-channel (lambda (y) (* y 10.0)) 0 (frames) ind 0)
	      (save-sound-as "test.snd" ind mus-next mus-bfloat)
	      (undo 1 ind 0)
	      (let ((ind1 (open-sound "test.snd")))
		(if (fneq (maxamp ind1 0) 1.0)
		    (snd-display ";clipping 1: ~A ~A" (maxamp ind1 0) (maxamp ind 0)))
		(close-sound ind1))
	      (delete-file "test.snd")
	      (set! (data-clipped) #f)
	      (let ((mx (maxamp ind)))
		(map-channel (lambda (y) (+ y (- 1.001 mx))) 0 (frames) ind 0)
		(save-sound-as "test.snd" ind mus-next mus-bshort)
		(let* ((ind1 (open-sound "test.snd"))
		       (baddy (scan-channel (lambda (y) (< y 0.0)))))
		  (if (not (list-p baddy))
		      (snd-display ";clipping 2: ~A" baddy))
		  (close-sound ind1))
		(delete-file "test.snd")
		(set! (data-clipped) #t)
		(save-sound-as "test.snd" ind mus-next mus-bshort)
		(let* ((ind1 (open-sound "test.snd"))
		       (baddy (scan-channel (lambda (y) (< y 0.0)))))
		  (if (list-p baddy)
		      (snd-display ";clipping 3: ~A" baddy))
		  (close-sound ind1))
		(delete-file "test.snd")
		(set! (data-clipped) #f))
	      (close-sound ind))
	    
	    (delete-file "fmv.snd")
	    (let ((com "this is a comment which we'll repeat enough times to trigger an internal loop"))
	      (do ((i 0 (1+ i)))
		  ((= i 3))
		(set! com (string-append com com)))
	      (set! fd (mus-sound-open-output "fmv.snd" 22050 4 mus-lshort mus-riff com)))
	    (set! sdata (make-sound-data 4 10))
	    (do ((i 0 (1+ i)))
		((= i 4))
	      (sound-data-set! sdata i 1 .1))
	    (mus-sound-write fd 0 9 4 sdata)
	    (mus-sound-close-output fd 80)
	    (set! fd (mus-sound-open-input "fmv.snd"))
	    (mus-sound-read fd 0 9 4 sdata)
	    (do ((i 0 (1+ i)))
		((= i 4))
	      (if (or (fneq (sound-data-ref sdata i 0) 0.0)
		      (fneq (sound-data-ref sdata i 1) 0.1)
		      (fneq (sound-data-ref sdata i 2) 0.0)
		      (fneq (sound-data-ref sdata i 6) 0.0))
		  (snd-display ";read/write[~A]: ~A?" i (sound-data-channel->list sdata i))))
	    (mus-sound-close-input fd)  
	    (set! fd (mus-sound-reopen-output "fmv.snd" 4 mus-lshort mus-riff (mus-sound-data-location "fmv.snd")))
	    (mus-sound-seek-frame fd 0)
	    (do ((i 0 (1+ i)))
		((= i 4))
	      (sound-data-set! sdata i 2 .1)
	      (sound-data-set! sdata i 3 .1))
	    (mus-sound-write fd 0 9 4 sdata)
	    (mus-sound-close-output fd 80)
	    (set! fd (mus-sound-open-input "fmv.snd"))
	    (mus-sound-read fd 0 9 4 sdata)
	    (do ((i 0 (1+ i)))
		((= i 4))
	      (if (or (fneq (sound-data-ref sdata i 0) 0.0)
		      (fneq (sound-data-ref sdata i 1) 0.1)
		      (fneq (sound-data-ref sdata i 2) 0.1)
		      (fneq (sound-data-ref sdata i 3) 0.1)
		      (fneq (sound-data-ref sdata i 6) 0.0))
		  (snd-display ";re-read/write[~A]: ~A?" i (sound-data-channel->list sdata i))))
	    (mus-sound-close-input fd))
	  
	  (if (file-exists? (string-append sf-dir "32bit.sf"))
	      (let ((ind (open-sound (string-append sf-dir "32bit.sf"))))
		(if (fneq (maxamp ind 0) .228) (snd-display ";32bit max: ~A" (maxamp ind 0)))
		(close-sound ind)))
	  
	  (let ((test-data (lambda (file beg dur data)
			     (catch #t
				    (lambda ()
				      (let* ((ind (open-sound file))
					     (ndata (samples->vct beg dur ind 0)))
					(if (not (vequal data ndata))
					    (snd-display ";~A: ~A != ~A" file data ndata))
					(close-sound ind)))
				    (lambda args args)))))
	    (test-data (string-append sf-dir "next-dbl.snd") 10 10 (vct 0.475 0.491 0.499 0.499 0.492 0.476 0.453 0.423 0.387 0.344))
	    (test-data (string-append sf-dir "oboe.ldbl") 1000 10 (vct 0.033 0.035 0.034 0.031 0.026 0.020 0.013 0.009 0.005 0.004))
	    
	    (test-data (string-append sf-dir "next-flt.snd") 10 10 (vct 0.475 0.491 0.499 0.499 0.492 0.476 0.453 0.423 0.387 0.344))
	    (test-data (string-append sf-dir "clbonef.wav") 1000 10 (vct 0.111 0.101 0.070 0.032 -0.014 -0.060 -0.085 -0.108 -0.129 -0.152))
	    
	    (test-data (string-append sf-dir "next-8.snd") 10 10 (vct 0.898 0.945 0.977 0.992 0.992 0.977 0.945 0.906 0.844 0.773))
	    (test-data (string-append sf-dir "o2_u8.wave") 1000 10 (vct -0.164 -0.219 -0.258 -0.242 -0.180 -0.102 -0.047 0.000 0.039 0.055))
	    
	    (test-data (string-append sf-dir "next-16.snd") 1000 10 (vct -0.026 -0.022 -0.024 -0.030 -0.041 -0.048 -0.050 -0.055 -0.048 -0.033))
	    (test-data (string-append sf-dir "o2.wave") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    
	    (test-data (string-append sf-dir "o2_18bit.aiff") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    (test-data (string-append sf-dir "o2_12bit.aiff") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    
	    (test-data (string-append sf-dir "next24.snd") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    (test-data (string-append sf-dir "mono24.wav") 1000 10 (vct 0.005 0.010 0.016 0.008 -0.007 -0.018 -0.025 -0.021 -0.005 0.001))
	    
	    (test-data (string-append sf-dir "o2_711u.wave") 1000 10 (vct -0.164 -0.219 -0.254 -0.242 -0.172 -0.103 -0.042 0.005 0.042 0.060))
	    (test-data (string-append sf-dir "alaw.wav") 1000 10 (vct -0.024 -0.048 -0.024 0.000 0.008 0.008 0.000 -0.040 -0.064 -0.024))
	    
	    ;; it is not a bug if these don't match if MUS_SAMPLE_BITS is not 24
	    (test-data (string-append sf-dir "b32.pvf") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    (test-data (string-append sf-dir "b32.wave") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    (test-data (string-append sf-dir "b32.snd") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    (test-data (string-append sf-dir "32bit.sf") 1000 10 (vct 0.016 0.014 0.013 0.011 0.010 0.010 0.010 0.010 0.012 0.014))
	    
	    (test-data (string-append sf-dir "nist-shortpack.wav") 10000 10 (vct 0.021 0.018 0.014 0.009 0.004 -0.001 -0.004 -0.006 -0.007 -0.008))
	    (test-data (string-append sf-dir "wood.sds") 1000 10 (vct -0.160 -0.216 -0.254 -0.239 -0.175 -0.102 -0.042 0.005 0.041 0.059))
	    (test-data (string-append sf-dir "oboe.g721") 1000 10 (vct -0.037 -0.040 -0.040 -0.041 -0.042 -0.038 -0.028 -0.015 -0.005 0.002))
	    (test-data (string-append sf-dir "oboe.g723_40") 1000 10 (vct -0.037 -0.040 -0.041 -0.041 -0.041 -0.038 -0.028 -0.015 -0.005 0.003))
	    (test-data (string-append sf-dir "mus10.snd") 10000 10 (vct 0.004 0.001 0.005 0.009 0.017 0.015 0.008 0.011 0.009 0.012))
	    (test-data (string-append sf-dir "ieee-text-16.snd") 1000 10 (vct -0.052 -0.056 -0.069 -0.077 -0.065 -0.049 -0.054 -0.062 -0.066 -0.074))
	    (test-data (string-append sf-dir "hcom-16.snd") 10000 10 (vct 0.000 0.000 0.000 0.008 0.000 -0.016 -0.016 -0.016 -0.008 0.000))
	    (test-data (string-append sf-dir "ce-c3.w02") 1000 10 (vct 0.581 0.598 0.596 0.577 0.552 0.530 0.508 0.479 0.449 0.425))
	    (test-data (string-append sf-dir "nasahal.avi") 20000 10 (vct 0.390 0.120 -0.399 -0.131 0.464 0.189 -0.458 -0.150 0.593 0.439))
	    (test-data (string-append sf-dir "oki.wav") 100 10 (vct 0.396 0.564 0.677 0.779 0.761 0.540 0.209 -0.100 -0.301 -0.265))
	    
	    (test-data (string-append sf-dir "trumps22.adp") 5000 10 (vct 0.267 0.278 0.309 0.360 0.383 0.414 0.464 0.475 0.486 0.495))
	    )
	  
	  (let ((errs (list "no error" "no frequency method" "no phase method" "null gen arg to method" "no length method"
			    "no free method" "no describe method" "no data method" "no scaler method"
			    "memory allocation failed" "unstable two pole error"
			    "can't open file" "no sample input" "no sample output"
			    "no such channel" "no file name provided" "no location method" "no channel method"
			    "no such fft window" "unsupported data format" "header read failed"
			    "unsupported header type" "file descriptors not initialized" "not a sound file" "file closed" "write error"
			    "bogus free" "buffer overflow" "buffer underflow" "file overflow"
			    "header write failed" "can't open temp file" "interrupted" "bad envelope"
			    "audio channels not available" "audio srate not available" "audio format not available"
			    "no audio input available" "audio configuration not available" 
			    "no audio lines available" "audio write error" "audio size not available" "audio device not available"
			    "can't close audio" "can't open audio" "audio read error" "audio amp not available"
			    "can't write audio" "can't read audio" "no audio read permission" 
			    "can't close file" "arg out of range"
			    "midi open error" "midi read error" "midi write error" "midi close error" "midi init error" "midi misc error"
			    "no channels method" "no hop method" "no width method" "no file-name method" "no ramp method" "no run method"
			    "no increment method" "no inspect method" "no offset method"
			    "no xcoeff method" "no ycoeff method" "no xcoeffs method" "no ycoeffs method" "can't translate"
			    )))
	    (call-with-current-continuation
	     (lambda (quit)
	       (do ((i 0 (1+ i)))
		   ((= i 72))
		 (if (not (string-=? (list-ref errs i) (mus-error-to-string i)))
		     (begin
		       (snd-display ";mus-error-to-string ~D: ~A ~A" i (list-ref errs i) (mus-error-to-string i))
		       (quit #f)))))))
	  (let ((new-id (mus-make-error "hiho all messed up")))
	    (if (not (string=? (mus-error-to-string new-id) "hiho all messed up"))
		(snd-display ";mus-make-error :~A ~A" new-id (mus-error-to-string new-id))))
	  
	  (let ((cur-srate (mus-sound-srate "oboe.snd"))
		(cur-chans (mus-sound-chans "oboe.snd"))
		(cur-format (mus-sound-data-format "oboe.snd"))
		(cur-type (mus-sound-header-type "oboe.snd"))
		(cur-loc (mus-sound-data-location "oboe.snd"))
		(cur-samps (mus-sound-samples "oboe.snd")))
	    (set! (mus-sound-srate "oboe.snd") (* cur-srate 2))
	    (if (not (= (* cur-srate 2) (mus-sound-srate "oboe.snd"))) 
		(snd-display ";set mus-sound-srate: ~A -> ~A" cur-srate (mus-sound-srate "oboe.snd")))
	    (set! (mus-sound-samples "oboe.snd") (* cur-samps 2))
	    (if (not (= (* cur-samps 2) (mus-sound-samples "oboe.snd"))) 
		(snd-display ";set mus-sound-samples: ~A -> ~A" cur-samps (mus-sound-samples "oboe.snd")))
	    (set! (mus-sound-chans "oboe.snd") (* cur-chans 2))
	    (if (not (= (* cur-chans 2) (mus-sound-chans "oboe.snd"))) 
		(snd-display ";set mus-sound-chans: ~A -> ~A" cur-chans (mus-sound-chans "oboe.snd")))
	    (set! (mus-sound-data-location "oboe.snd") (* cur-loc 2))
	    (if (not (= (* cur-loc 2) (mus-sound-data-location "oboe.snd"))) 
		(snd-display ";set mus-sound-data-location: ~A -> ~A" cur-loc (mus-sound-data-location "oboe.snd")))
	    (set! (mus-sound-header-type "oboe.snd") mus-nist)
	    (if (not (= mus-nist (mus-sound-header-type "oboe.snd"))) 
		(snd-display ";set mus-sound-header-type: ~A -> ~A" cur-type (mus-sound-header-type "oboe.snd")))
	    (set! (mus-sound-data-format "oboe.snd") mus-lintn)
	    (if (not (= mus-lintn (mus-sound-data-format "oboe.snd"))) 
		(snd-display ";set mus-sound-data-format: ~A -> ~A" cur-format (mus-sound-data-format "oboe.snd")))
	    (set! (mus-sound-srate "oboe.snd") cur-srate)
	    (set! (mus-sound-samples "oboe.snd") cur-samps)
	    (set! (mus-sound-chans "oboe.snd") cur-chans)
	    (set! (mus-sound-data-location "oboe.snd") cur-loc)
	    (set! (mus-sound-header-type "oboe.snd") cur-type)
	    (set! (mus-sound-data-format "oboe.snd") cur-format))
	  
	  (let ((ind (mus-sound-open-input "oboe.snd")))
	    (set! (mus-file-data-clipped ind) #t)
	    (if (not (mus-file-data-clipped ind)) (snd-display ";file data-clipped?"))
	    (set! (mus-file-prescaler ind) 2.0)
	    (if (fneq (mus-file-prescaler ind) 2.0) (snd-display ";prescaler: ~A" (mus-file-prescaler ind)))
	    (mus-sound-close-input ind))
	  
;	  (with-sound (:output "/home/bil/zap/sounds/bigger.snd" :srate 44100 :play #f)
;	    (do ((i 0 (1+ i))) ((= i 72000))
;	      (fm-violin i .1 440 (+ .01 (* (/ i 72000.0) .9)))))

	  (if with-big-file
	      (let ((big-file-name "/home/bil/zap/sounds/bigger.snd"))
		(if (file-exists? big-file-name)
		    (begin
		      (if (not (= (mus-sound-samples big-file-name) 3175160310))
			  (snd-display ";bigger samples: ~A" (mus-sound-samples big-file-name)))
		      (if (not (= (mus-sound-frames big-file-name) 3175160310))
			  (snd-display ";bigger frames: ~A" (mus-sound-frames big-file-name)))
		      (if (not (= (mus-sound-length big-file-name) 6350320648))
			  (snd-display ";bigger bytes: ~A" (mus-sound-length big-file-name)))
		      (if (fneq (mus-sound-duration big-file-name) 71999.1015)
			  (snd-display ";bigger dur: ~A" (mus-sound-duration big-file-name)))
		      (let ((ind (open-sound big-file-name)))
			(if (not (= (frames ind) 3175160310)) (snd-display ";bigger frames: ~A" (frames ind)))
			(if (not (= (frames ind 0 0) 3175160310)) (snd-display ";bigger edpos-frames: ~A" (frames ind)))
			(let ((m1 (add-mark (* 44100 50000) ind)))
			  (if (not (= (mark-sample m1) (* 44100 50000))) (snd-display ";bigger mark at: ~A" (mark-sample m1)))
			  (set! (mark-sample m1) (* 44100 66000))
			  (if (not (= (mark-sample m1) (* 44100 66000))) (snd-display ";bigger mark to: ~A" (mark-sample m1))))
			(let ((mx (mix-sound "oboe.snd" (* 44100 60000))))
			  (if (not (= (mix-position mx) (* 44100 60000))) (snd-display ";bigger mix at: ~A" (mix-position mx)))
			  (set! (mix-position mx) (* 44100 61000))
			  (if (not (= (mix-position mx) (* 44100 61000))) (snd-display ";bigger mix to: ~A" (mix-position mx)))
			  (undo 2))
			(let ((old-select (selection-creates-region)))
			  (set! (selection-creates-region) #f)
			  (select-all ind)
			  (if (not (= (selection-frames) (frames ind))) (snd-display ";bigger select all: ~A ~A" (selection-frames) (frames)))
			  (set! (selection-position) (* 44100 50000))
			  (if (not (= (selection-position) (* 44100 50000))) (snd-display ";bigger select pos: ~A" (selection-position)))
			  (set! (selection-position) 0)
			  (set! (selection-frames) (* 44100 65000))
			  (if (not (= (selection-frames) (* 44100 65000))) (snd-display ";bigger select len: ~A" (selection-frames)))
			  (set! (selection-creates-region) old-select))
			(set! (cursor ind) (* 44100 50000))
			(if (not (= (cursor ind) (* 44100 50000))) (snd-display ";bigger cursor: ~A" (cursor ind)))
			(let ((val (backward-sample)))
			  (if (not (= (cursor ind) (1- (* 44100 50000)))) (snd-display ";backup bigger cursor: ~A" (cursor ind)))
			  (if (not (= val (1- (* 44100 50000)))) (snd-display ";backup rtn bigger cursor: ~A" (cursor ind))))
			(let ((val (forward-sample)))
			  (if (not (= (cursor ind) (* 44100 50000))) (snd-display ";up bigger cursor: ~A" (cursor ind)))
			  (if (not (= val (* 44100 50000))) (snd-display ";up rtn bigger cursor: ~A" (cursor ind))))
			(let ((m1 (add-mark (* 44123 51234) ind)))
			  (if (not (= (mark-sample m1) (* 44123 51234))) (snd-display ";bigger mark at: ~A" (mark-sample m1)))
			  (let ((mid (find-mark (* 44123 51234))))
			    (if (or (not (number? mid)) (not (= mid m1))) (snd-display ";bigger mark seach: ~A ~A" mid m1))))
			(let ((mx (mix-sound "oboe.snd" (* 44123 61234))))
			  (let ((mxd (find-mix (* 44123 61234))))
			    (if (or (not (number? mxd)) (not (= mxd mx))) (snd-display ";bigger find-mix ~A ~A" mxd mx))))
			(set! (cursor ind) (* 44123 51234))
			(if (not (= (cursor ind) (* 44123 51234))) (snd-display ";bigger cursor 123: ~A" (cursor ind)))
			(close-sound ind))
		      ))))

	  (let ((ind (new-sound "tmp.snd" mus-riff mus-l24int 22050 1 :size 100000)))
	    (let ((x -0.5) 
		  (incr (/ 1.0 (frames)))) 
	      (map-channel (lambda (n) 
			     (let ((val x)) 
			       (set! x (+ x incr)) 
			       val))))
	    (save-sound)
	    (close-sound ind)
	    (set! ind (open-sound "tmp.snd"))
	    (let ((reg (select-all)))
	      (save-selection "tmp1.snd" mus-next mus-l24int)
	      (let ((ind1 (open-sound "tmp1.snd")))
		(let* ((x -0.5) 
		       (incr (/ 1.0 (frames))) 
		       (err (scan-channel 
			     (lambda (n) 
			       (let ((val x)) 
				 (set! x (+ x incr)) 
				 (fneq val n)))
			     0 100000 ind1)))
		  (if err (snd-display ";l24 (next) selection not saved correctly? ~A" err)))
		(close-sound ind1))
	      (save-selection "tmp1.snd" mus-aifc mus-l24int)
	      (let ((ind1 (open-sound "tmp1.snd")))
		(let* ((x -0.5) 
		       (incr (/ 1.0 (frames))) 
		       (err (scan-channel 
			     (lambda (n) 
			       (let ((val x)) 
				 (set! x (+ x incr)) 
				 (fneq val n)))
			     0 100000 ind1)))
		  (if err (snd-display ";l24 (aifc) selection not saved correctly? ~A" err)))
		(close-sound ind1))
	      (save-region reg "tmp1.snd" mus-next mus-l24int)
	      (let ((ind1 (open-sound "tmp1.snd")))
		(let* ((x -0.5) 
		       (incr (/ 1.0 (frames))) 
		       (err (scan-channel 
			     (lambda (n) 
			       (let ((val x)) 
				 (set! x (+ x incr)) 
				 (fneq val n)))
			     0 100000 ind1)))
		  (if err (snd-display ";l24 (next) region not saved correctly? ~A" err)))
		(close-sound ind1))
	      (delete-file "tmp1.snd")
	      (close-sound ind)
	      (delete-file "tmp.snd")))
	  (let ((ind (new-sound "tmp.snd" mus-next mus-bfloat 22050 1 :size 10 :comment #f)))
	    (map-channel (lambda (y) 1.0))
	    (env-channel '(0 0 .1 .1 .2 .2 .3 .3 .4 .4 .5 .5 .6 .6 .7 .7 .8 .8 .9  .9))
	    (if (not (vequal (channel->vct) (vct 0.000 0.100 0.200 0.300 0.400 0.500 0.600 0.700 0.800 0.900)))
		(snd-display ";ramp env by .1: ~A" (channel->vct)))
	    (close-sound ind))
	  ))

      (reset-hook! open-raw-sound-hook)
      (add-hook! open-raw-sound-hook (lambda (a b) #t))
      (reset-hook! bad-header-hook)
      (add-hook! bad-header-hook (lambda (n) #t))
      (let* ((magic-words (list ".snd" "FORM" "AIFF" "AIFC" "COMM" "COMT" "INFO" "INST" "inst" "MARK" "SSND"
				"FVER" "NONE" "ULAW" "ulaw" "ima4" "raw " "sowt" "in32" "in24" "ni23" "fl32"
				"FL32" "fl64" "twos" "ALAW" "alaw" "APPL" "CLM " "RIFF" "RIFX" "WAVE" "fmt "
				"data" "fact" "clm " "NIST" "8SVX" "16SV" "Crea" "tive" "SOUN" "D SA" "MPLE"
				"BODY" "VHDR" "CHAN" "ANNO" "NAME" "2BIT" "HCOM" "FSSD" "%//\n" "%---" "ALaw"
				"Soun" "MAUD" "MHDR" "MDAT" "mdat" "MThd" "sfbk" "sdta" "shdr" "pdta"
				"LIST" "GF1P" "ATCH" "$SIG" "NAL_" "GOLD" " SAM" "SRFS" "Diam" "ondW" "CSRE"
				"SND " "SNIN" "SNDT" "DDSF" "FSMu" "UWFD" "LM89" "SY80" "SY85" "SCRS" "DSPL"
				"AVI " "strf" "movi" "PRAM" " paf" "fap " "DS16" "HEDR" "HDR8" "SDA_" "SDAB"
				"SD_B" "NOTE" "file" "=sam" "SU7M" "SU7R" "PVF1" "PVF2" "AUTH" "riff" "TWIN"
				"IMPS" "SMP1" "Maui" "SDIF" "NVF "))
	     (len (length magic-words))
	     (ctr 0))
	(for-each
	 (lambda (magic)
	   (if (file-exists? "test.snd")
	       (delete-file "test.snd"))
	   (mus-sound-forget "test.snd")
	   ;; try random garbage
	   (with-output-to-file "test.snd"
	     (lambda ()
	       (display magic)
	       (do ((i 0 (1+ i)))
		   ((= i 128))
		 (write (random 1.0)))))
	   (let ((tag (catch #t
			     (lambda ()
			       (open-sound "test.snd"))
			     (lambda args (car args)))))
	     (if (and (number? tag)
		      (sound? tag))
		 (snd-display ";open-sound garbage ~A: ~A?" magic tag)))
	   (delete-file "test.snd")
	   (mus-sound-forget "test.snd")
	   ;; try plausible garbage
	   (with-output-to-file "test.snd"
	     (lambda ()
	       (display magic)
	       (do ((i 0 (1+ i)))
		   ((= i 128))
		 (write (random 128)))))
	   (let ((tag (catch #t
			     (lambda ()
			       (open-sound "test.snd"))
			     (lambda args (car args)))))
	     (if (and (number? tag)
		      (sound? tag))
		 (snd-display ";open-sound plausible garbage ~A: ~A?" magic tag)))
	   (delete-file "test.snd")
	   (mus-sound-forget "test.snd")
	   ;; write very plausible garbage
	   (with-output-to-file "test.snd"
	     (lambda ()
	       (display magic)
	       (do ((i 1 (1+ i)))
		   ((= i 12))
		 (if (< (+ ctr i) len)
		     (display (list-ref magic-words (+ ctr i)))
		     (display (list-ref magic-words i))))))
	   (let ((tag (catch #t
			     (lambda ()
			       (open-sound "test.snd"))
			     (lambda args (car args)))))
	     (if (and (number? tag)
		      (sound? tag))
		 (snd-display ";open-sound very plausible garbage ~A: ~A?" magic tag)))
	   (set! ctr (1+ ctr)))
	 magic-words))
      (if (file-exists? "test.snd") (delete-file "test.snd"))
      (mus-sound-forget "test.snd")

      (with-output-to-file "test.snd"
	(lambda ()
	  (display ".snd")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\034) ; location
	  (write-char #\000) (write-char #\001) (write-char #\215) (write-char #\030) ; nominal size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\022) ; format
	  (write-char #\000) (write-char #\000) (write-char #\126) (write-char #\042) ; srate
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\001) ; chans
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; comment
	  (write-char #\000) (write-char #\001) ; samp 1
	  ))
      (if (not (= (mus-sound-data-format "test.snd") mus-bshort))
	  (snd-display ";next 18: ~A" (mus-sound-data-format "test.snd")))
      (delete-file "test.snd")
      (mus-sound-forget "test.snd")
      (with-output-to-file "test.snd"
	(lambda ()
	  (display ".snd")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\004) ; location
	  (write-char #\000) (write-char #\001) (write-char #\215) (write-char #\030) ; nominal size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\022) ; format
	  (write-char #\000) (write-char #\000) (write-char #\126) (write-char #\042) ; srate
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\001) ; chans
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; comment
	  (write-char #\000) (write-char #\001) ; samp 1
	  ))
      (let ((tag (catch #t
			(lambda ()
			  (open-sound "test.snd"))
			(lambda args (car args)))))
	(if (and (number? tag)
		 (sound? tag))
	    (begin
	      (snd-display ";open-sound next bad location ~A: ~A?" (data-location tag) tag)
	      (close-sound tag))))
      (delete-file "test.snd")
      (mus-sound-forget "test.snd")
      (with-output-to-file "test.snd"
	(lambda ()
	  (display ".snd")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\034) ; location
	  (write-char #\000) (write-char #\001) (write-char #\215) (write-char #\030) ; nominal size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\122) ; format
	  (write-char #\000) (write-char #\000) (write-char #\126) (write-char #\042) ; srate
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\001) ; chans
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; comment
	  (write-char #\000) (write-char #\001) ; samp 1
	  ))
      (let ((tag (catch #t
			(lambda ()
			  (open-sound "test.snd"))
			(lambda args (car args)))))
	(if (and (number? tag)
		 (sound? tag))
	    (begin
	      (snd-display ";open-sound next bad format ~A: ~A?" (data-format tag) tag)
	      (close-sound tag))))
      (delete-file "test.snd")
      (mus-sound-forget "test.snd")

      (letrec ((make-aifc-file 
		(lambda (frames auth-lo bits)
		  (with-output-to-file "test.aif"
		    (lambda ()
		      (display "FORM")
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\146) ; len
		      (display "AIFCFVER")
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\004) ; version chunk size
		      (write-char #\242) (write-char #\200) (write-char #\121) (write-char #\100) ; version  
		      (display "COMM")
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\046) ; COMM chunk size
		      (write-char #\000) (write-char #\001) ; 1 chan
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char frames) ; frames
		      (write-char #\000) (write-char bits) ; bits
		      (write-char #\100) (write-char #\016) (write-char #\254) (write-char #\104) (write-char #\000) 
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ;
		      ;; srate as 80-bit float (sheesh)
		      (display "NONE") ; compression
		      (write-char #\016) ; pascal string len
		      (display "not compressed")
		      (write-char #\000)
		      (display "AUTH")
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char auth-lo) ; AUTH chunk size
		      (display "bil")
		      (write-char #\000)
		      (display "SSND")
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
		      (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
		      (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
		      )))))
	(if (file-exists? "test.aif") (delete-file "test.aif"))
	(mus-sound-forget "test.aif")
	;;correct (make-aifc-file #\002 #\004 #\020)
	(make-aifc-file #\102 #\004 #\020)
	(let ((ind (open-sound "test.aif")))
	  (if (not (= (frames ind) 2)) (snd-display ";bad frames in header: ~A" (frames ind)))
	  (close-sound ind))
	(delete-file "test.aif")
	(mus-sound-forget "test.aif")
	(make-aifc-file #\002 #\150 #\020)
	(let ((tag (catch #t
			  (lambda ()
			    (open-sound "test.aif"))
			  (lambda args (car args)))))
	  (if (and (number? tag)
		   (sound? tag))
	      (begin
		(snd-display ";open-sound aifc no ssnd chunk ~A: ~A?" (data-location tag) tag)
		(close-sound tag))))
	(delete-file "test.aif")
	(mus-sound-forget "test.aif")
	(make-aifc-file #\002 #\000 #\020)
	(let ((tag (catch #t
			  (lambda ()
			    (open-sound "test.aif"))
			  (lambda args (car args)))))
	  (if (and (number? tag)
		   (sound? tag))
	      (begin
		(snd-display ";open-sound aifc 0-len auth chunk ~A: ~A?" (data-location tag) tag)
		(close-sound tag))))
	(delete-file "test.aif")
	(mus-sound-forget "test.aif")
	(make-aifc-file #\002 #\150 #\120)
	(let ((tag (catch #t
			  (lambda ()
			    (open-sound "test.aif"))
			  (lambda args (car args)))))
	  (if (and (number? tag)
		   (sound? tag))
	      (begin
		(snd-display ";open-sound bits 80 ~A: ~A?" (data-format tag) tag)
		(close-sound tag))))
	(delete-file "test.aif")
	(mus-sound-forget "test.aif"))
      (with-output-to-file "test.aif"
	(lambda ()
	  (display "FORM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\176) ; len
	  (display "AIFCFVER")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\004) ; version chunk size
	  (write-char #\242) (write-char #\200) (write-char #\121) (write-char #\100) ; version  
	  (display "COMM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\046) ; COMM chunk size
	  (write-char #\000) (write-char #\001) ; 1 chan
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\002) ; frames
	  (write-char #\000) (write-char #\020) ; bits
	  (write-char #\100) (write-char #\016) (write-char #\254) (write-char #\104) (write-char #\000) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; srate as 80-bit float (sheesh)
	  (display "NONE") ; compression
	  (write-char #\016) ; pascal string len
	  (display "not compressed")
	  (write-char #\000)
	  (display "AUTH")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\004) ; AUTH chunk size
	  (display "bil")
	  (write-char #\000)
	  (display "ANNO")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\004) ; AUTH chunk size
	  (display "cat")
	  (write-char #\000)
	  (display "NAME")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\004) ; AUTH chunk size
	  (display "dog")
	  (write-char #\000)
	  (display "SSND")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
	  (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
	  ))
      (if (not (= (string-length (mus-sound-comment "test.aif")) 15))
	  (snd-display ";aifc 3 aux comments: ~A?" (mus-sound-comment "test.aif")))
      (delete-file "test.aif")
      (mus-sound-forget "test.aif")
      (with-output-to-file "test.aif"
	(lambda ()
	  (display "FORM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\142) ; len
	  (display "AIFC")
	  (display "SSND")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
	  (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
	  (display "COMM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\046) ; COMM chunk size
	  (write-char #\000) (write-char #\001) ; 1 chan
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\002) ; frames
	  (write-char #\000) (write-char #\020) ; bits
	  (write-char #\100) (write-char #\016) (write-char #\254) (write-char #\104) (write-char #\000) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; srate as 80-bit float (sheesh)
	  (display "NONE") ; compression
	  (write-char #\016) ; pascal string len
	  (display "not compressed")
	  (write-char #\000)
	  (display "COMT")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) 
	  (display "bil")
	  (write-char #\000)
	  ))
      (if (not (string=? (substring (mus-sound-comment "test.aif") 0 3) "bil"))
	  (snd-display ";aifc trailing comt comment: ~A?" (mus-sound-comment "test.aif")))
      (if (not (= (mus-sound-frames "test.aif") 2))
	  (snd-display ";aifc trailing comt frames: ~A?" (mus-sound-frames "test.aif")))
      (let ((ind (open-sound "test.aif")))
	(if (or (fneq (sample 0) 0.00198)
		(fneq (sample 1) 0.00195)
		(fneq (sample 2) 0.0)
		(fneq (sample 3) 0.0))
	    (snd-display ";aifc trailing comt samps: ~A ~A ~A ~A" (sample 0) (sample 1) (sample 2) (sample 3)))
	(close-sound ind))
      (delete-file "test.aif")
      (mus-sound-forget "test.aif")
      (with-output-to-file "test.aif"
	(lambda ()
	  (display "FORM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\142) ; len
	  (display "AIFC")
	  (display "SSND")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
	  (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
	  (display "COMM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\046) ; COMM chunk size
	  (write-char #\000) (write-char #\001) ; 1 chan
	  (write-char #\000) (write-char #\000) (write-char #\100) (write-char #\102) ; frames
	  (write-char #\000) (write-char #\020) ; bits
	  (write-char #\100) (write-char #\016) (write-char #\254) (write-char #\104) (write-char #\000) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; srate as 80-bit float (sheesh)
	  (display "NONE") ; compression
	  (write-char #\016) ; pascal string len
	  (display "not compressed")
	  (write-char #\000)
	  (display "COMT")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) 
	  (display "bil")
	  (write-char #\000)
	  ))
      (if (not (string=? (substring (mus-sound-comment "test.aif") 0 3) "bil"))
	  (snd-display ";aifc trailing comt comment: ~A?" (mus-sound-comment "test.aif")))
      (if (not (= (mus-sound-frames "test.aif") 2))
	  (snd-display ";aifc trailing comt (bogus) frames: ~A?" (mus-sound-frames "test.aif")))
      (let ((ind (open-sound "test.aif")))
	(if (or (fneq (sample 0) 0.00198)
		(fneq (sample 1) 0.00195)
		(fneq (sample 2) 0.0)
		(fneq (sample 3) 0.0))
	    (snd-display ";aifc trailing comt samps (bogus frame setting): ~A ~A ~A ~A" (sample 0) (sample 1) (sample 2) (sample 3)))
	(close-sound ind))
      (delete-file "test.aif")
      (mus-sound-forget "test.aif")
      (with-output-to-file "test.aif"
	(lambda ()
	  (display "FORM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\142) ; len
	  (display "AIFC")
	  (display "SSND")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
	  (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
	  (display "COMM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\046) ; COMM chunk size
	  (write-char #\000) (write-char #\001) ; 1 chan
	  (write-char #\000) (write-char #\000) (write-char #\100) (write-char #\102) ; frames
	  (write-char #\000) (write-char #\020) ; bits
	  (write-char #\100) (write-char #\016) (write-char #\254) (write-char #\104) (write-char #\000) 
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; srate as 80-bit float (sheesh)
	  (display "NONE") ; compression
	  (write-char #\016) ; pascal string len
	  (display "not compressed")
	  (write-char #\000)
	  (display "SSND")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
	  (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
	  ))
      (let ((tag (catch #t
			(lambda ()
			  (open-sound "test.aif"))
			(lambda args (car args)))))
	(if (and (number? tag)
		 (sound? tag))
	    (begin
	      (snd-display ";open-sound aifc 2 ssnd chunks ~A: ~A?" (data-location tag) tag)
	      (close-sound tag))))
      (delete-file "test.aif")
      (mus-sound-forget "test.aif")
      
      (with-output-to-file "test.aif"
	(lambda ()
	  (display "FORM")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\040) ; len
	  (display "AIFC")
	  (display "SSND")
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\014) ; SSND chunk size
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; SSND data loc
	  (write-char #\000) (write-char #\000) (write-char #\000) (write-char #\000) ; block size?
	  (write-char #\000) (write-char #\101) (write-char #\000) (write-char #\100) ; two samples
	  ))
      (let ((tag (catch #t
			(lambda ()
			  (open-sound "test.aif"))
			(lambda args (car args)))))
	(if (not (eq? tag 'mus-error))
	    (begin
	      (snd-display ";open-sound aifc no comm chunk ~A?" tag)
	      (if (and (number? tag)
		       (sound? tag))
		  (close-sound tag)))))
      (delete-file "test.aif")
      (mus-sound-forget "test.aif")

      (reset-hook! bad-header-hook)
      (reset-hook! open-raw-sound-hook)
      (if (not (null? (sounds))) (map close-sound (sounds)))
      
      (run-hook after-test-hook 4)
      ))

;;; ---------------- test 5: simple overall checks ----------------

(if (not (provided? 'snd-snd4.scm)) (load "snd4.scm")) ; needed for various scan/map extensions
(if (not (provided? 'snd-snd7.scm)) (load "snd7.scm")) ; forward-graph
(define a-ctr 0)

(define (test-edpos test-func func-name change-thunk ind1)
  (let ((fr1 (test-func ind1 0))
	(fr2 (test-func ind1 0 0))
	(fr3 (test-func ind1 0 current-edit-position))
	(fr4 (test-func ind1 0 (lambda (snd chn) 0))))
    (if (not (and (= fr1 fr2)
		  (= fr1 fr3)
		  (= fr1 fr4)))
	(snd-display ";initial ~A: ~A ~A ~A ~A?" func-name fr1 fr2 fr3 fr4))
    (change-thunk)
    (let ((fr5 (test-func ind1 0))
	  (fr6 (test-func ind1 0 1))
	  (fr7 (test-func ind1 0 current-edit-position))
	  (fr8 (test-func ind1 0 (lambda (snd chn) (edit-position snd chn)))))
      (if (not (and (= fr5 fr6)
		    (= fr5 fr7)
		    (= fr5 fr8)))
	  (snd-display ";~A (edpos 1): ~A ~A ~A ~A?" func-name fr5 fr6 fr7 fr8))
      (set! fr5 (test-func ind1 0 0))
      (set! fr6 (test-func ind1 0 (lambda (snd chn) 0)))
      (if (not (and (= fr1 fr5)
		    (= fr1 fr6)))
	  (snd-display ";~A (edpos -1): ~A ~A ~A?" func-name fr1 fr5 fr6))))
  (revert-sound ind1))

(define (test-edpos-1 test-func func-name ind1)
  (let ((v0 (samples->vct 12000 10 ind1 0)))
    (test-func ind1 0)
    (let ((v1 (samples->vct 12000 10 ind1 0)))
      (if (vequal v0 v1)
	  (snd-display ";~A (0) no change! ~A ~A" func-name v0 v1))
      (test-func ind1 0)
      (let ((v2 (samples->vct 12000 10 ind1 0)))
	(if (not (vequal v1 v2))
	    (snd-display ";~A (1) ~A ~A" func-name v1 v2))
	(test-func ind1 (lambda (snd chn) 0))
	(set! v2 (samples->vct 12000 10 ind1 0))
	(if (not (vequal v1 v2))
	    (snd-display ";~A (2) ~A ~A" func-name v1 v2)))))
  (revert-sound ind1))

(define (vfequal v0 v1)
  (define (dequal ctr len)
    (if (= ctr len)
	#t
	(and (< (abs (- (vct-ref v0 ctr) (vct-ref v1 ctr))) .01)
	     (dequal (1+ ctr) len))))
  (let ((len (vct-length v0)))
    (and (= len (vct-length v1))
	 (dequal 0 len))))

(define (test-orig func0 func1 func-name ind1)
  (let ((v0 (samples->vct 12000 10 ind1 0)))
    (func0 ind1)
    (let ((v1 (samples->vct 12000 10 ind1 0)))
      (if (vfequal v0 v1)
	  (snd-display ";~A (orig: 0) no change! ~A ~A" func-name v0 v1))
      (func1 ind1)
      (let ((v2 (samples->vct 12000 10 ind1 0)))
	(if (not (vfequal v0 v2))
	    (snd-display ";~A (orig: 1) ~A ~A" func-name v0 v2))))
    (revert-sound ind1)))

(if (not (provided? 'snd-extensions.scm)) (load "extensions.scm"))
(if (not (provided? 'snd-examp.scm)) (load "examp.scm"))
(if (not (provided? 'snd-dsp.scm)) (load "dsp.scm"))
(if (not (provided? 'snd-pvoc.scm)) (load "pvoc.scm"))
(if (not (provided? 'snd-edit-menu.scm)) (load "edit-menu.scm"))
(define g-init-val 0)

(define* (make-bandpass-2 flo1 fhi1 flo2 fhi2 #:optional (len 30))
  (let* ((f1 (make-bandpass flo1 fhi1 len))
	 (f2 (make-bandpass flo2 fhi2 len)))
    (vct-add! (mus-xcoeffs f1) (mus-xcoeffs f2))
    f1))

(define* (cosine-channel-via-ptree #:optional (beg 0) (dur #f) (snd #f) (chn #f) (edpos #f))
  ;; vct: angle increment
  (ptree-channel
   (lambda (y data forward)
     (declare (y real) (data vct) (forward boolean))
     (let* ((angle (vct-ref data 0))
	    (incr (vct-ref data 1))
	    (val (* y (cos angle))))
       (if forward
	   (vct-set! data 0 (+ angle incr))
	   (vct-set! data 0 (- angle incr)))
       val))
   beg dur snd chn edpos #t
   (lambda (frag-beg frag-dur)
     (let ((incr (/ pi frag-dur)))
       (vct (+ (* -0.5 pi) (* frag-beg incr))
	    incr)))))

(define* (cosine-channel #:optional (beg 0) (dur #f) (snd #f) (chn #f) (edpos #f))
  (let ((old-opt (optimization)))
    (set! (optimization) 0)
    (map-channel
     (let* ((samps (or dur (frames snd chn)))
	    (incr (/ pi samps))
	    (angle (* -0.5 pi)))
       (lambda (y)
	 (let ((val (* y (cos angle))))
	   (set! angle (+ angle incr))
	   val)))
     beg dur snd chn edpos)
    (set! (optimization) old-opt)))

(define* (xen-channel proc #:optional (beg 0) (dur #f) (snd #f) (chn #f) (edpos #f) (env-too #f) (init-func #f))
  (ptree-channel proc beg dur snd chn edpos env-too init-func #f))

(define (check-maxamp ind val name)
  (if (fneq (maxamp ind 0) val) (snd-display ";maxamp amp-env ~A: ~A should be ~A" name (maxamp ind) val))
  (let ((pos (find (lambda (y) (>= (abs y) (- val .001))))))
    (if (not pos) (snd-display ";actual maxamp ~A vals not right" name))
    (let ((mx 0.0))
      (scan-chan (lambda (y) 
		   (if (> (abs y) mx) 
		       (set! mx (abs y))) 
		   #f))
      (if (fneq mx val) (snd-display ";actual ~A max: ~A (correct: ~A)" name mx val)))))

(define (check-env-vals name gen)
  (let ((ctr -1))
    (scan-chan (lambda (y)
		 (let ((val (env gen)))
		   (set! ctr (1+ ctr))
		   (if (fneq val y)
		       (begin
			 (display (format #f "~%;check-env-vals ~A at ~D: ~A ~A" name ctr val y))
			 #t)
		       #f))))))

(define (our-x->position ind x) 
  (let ((ax (axis-info ind 0)))
    (list
     (+ (list-ref ax 10) 
	(/ (* (- x (list-ref ax 2))
	      (- (list-ref ax 12) (list-ref ax 10)))
	   (- (list-ref ax 4) (list-ref ax 2))))
     (x->position x ind))))

(define (region-to-vct r c len)
  (let* ((rs (make-region-sample-reader 0 r c))
	 (v (make-vct len)))
    (do ((i 0 (1+ i)))
	((= i len) v)
      (vct-set! v i (next-sample rs)))))

(define (region2vct r c len)
  (region-samples->vct 0 len r c))

(define old-opt-val (optimization))

(if (or full-test (= snd-test 5) (and keep-going (<= snd-test 5)))
    (begin
      (run-hook before-test-hook 5)
      
      (set! (optimization) max-optimization) ; these trees assume optimization is on
      
      (if (dac-is-running) (snd-display ";dac is running??"))
      (do ((clmtest 0 (1+ clmtest))) ((= clmtest tests)) 
	(log-mem clmtest)

      ;; basic edit tree cases
      (let ((ind (new-sound "test.snd")))
	(if (not (string-=? (display-edits) (string-append "
EDITS: 0

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)
")))
	    (snd-display ";new 0: ~A" (display-edits)))
	(insert-samples 10 10 (make-vct 10))
	(if (not (string-=? (display-edits) (string-append "
EDITS: 2

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)

 (silence 1 9) ; (insert-extend) [1:3]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, cp->sounds[-1][0:8, 0.000])
   (at 10, end_mark)

 (insert 10 10) ; insert-samples [2:4]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, cp->sounds[-1][0:8, 0.000])
   (at 10, cp->sounds[1][0:9, 1.000]) [buf: 10] 
   (at 20, end_mark)
")))
	    (snd-display ";new 1: ~A" (display-edits)))
	(undo)
	(insert-samples 0 10 (make-vct 10))
	(if (not (string-=? (display-edits) (string-append "
EDITS: 2

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)

 (silence 1 9) ; (insert-extend) [1:3]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, cp->sounds[-1][0:8, 0.000])
   (at 10, end_mark)

 (insert 0 10) ; insert-samples [2:4]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 10] 
   (at 10, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 11, cp->sounds[-1][0:8, 0.000])
   (at 20, end_mark)
")))
	    (snd-display ";new 2: ~A" (display-edits)))
	(undo 2)
	(insert-samples 0 10 (make-vct 10))
	(if (not (string-=? (display-edits) (string-append "
EDITS: 1

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)

 (insert 0 10) ; insert-samples [1:3]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 10] 
   (at 10, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 11, end_mark)
")))
	    (snd-display ";new 3: ~A" (display-edits)))
	(undo)
	(set! (sample 0) .5)
	(if (not (string-=? (display-edits) (string-append "
EDITS: 1

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)

 (set 0 1) ; set! sample [1:2]:
   (at 0, cp->sounds[1][0:0, 1.000]) [buf: 1] 
   (at 1, end_mark)
")))
	    (snd-display ";new 4: ~A" (display-edits)))      
	(undo)
	(set! (samples 0 10) (make-vct 10))
	(if (not (string-=? (display-edits) (string-append "
EDITS: 1

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)

 (set 0 10) ; set-samples [1:2]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
")))
	    (snd-display ";new 5: ~A" (display-edits)))
	(delete-samples 3 4)
	(if (not (string-=? (display-edits ind 0 2) "
 (delete 3 4) ; delete-samples [2:3]:
   (at 0, cp->sounds[1][0:2, 1.000]) [buf: 10] 
   (at 3, cp->sounds[1][7:9, 1.000]) [buf: 10] 
   (at 6, end_mark)
"))
	    (snd-display ";new 6: ~A" (display-edits ind 0 2)))
	(set! (samples 1 4) (make-vct 4))
	(if (not (string-=? (display-edits ind 0 3) "
 (set 1 4) ; set-samples [3:4]:
   (at 0, cp->sounds[1][0:0, 1.000]) [buf: 10] 
   (at 1, cp->sounds[2][0:3, 1.000]) [buf: 4] 
   (at 5, cp->sounds[1][9:9, 1.000]) [buf: 10] 
   (at 6, end_mark)
"))
	    (snd-display ";new 7: ~A" (display-edits ind 0 3)))
	(undo 2)
	(insert-samples 2 3 (make-vct 3))
	(insert-samples 2 1 (make-vct 1))
	(insert-samples 4 1 (make-vct 1))
	(insert-samples 15 1 (make-vct 1))
	(if (not (string-=? (display-edits) (string-append "
EDITS: 5

 (begin) [0:2]:
   (at 0, cp->sounds[0][0:0, 0.000]) [file: " cwd "test.snd[0]]
   (at 1, end_mark)

 (set 0 10) ; set-samples [1:2]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 10] 
   (at 10, end_mark)

 (insert 2 3) ; insert-samples [2:4]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[2][0:2, 1.000]) [buf: 3] 
   (at 5, cp->sounds[1][2:9, 1.000]) [buf: 10] 
   (at 13, end_mark)

 (insert 2 1) ; insert-samples [3:5]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[3][0:0, 1.000]) [buf: 1] 
   (at 3, cp->sounds[2][0:2, 1.000]) [buf: 3] 
   (at 6, cp->sounds[1][2:9, 1.000]) [buf: 10] 
   (at 14, end_mark)

 (insert 4 1) ; insert-samples [4:7]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[3][0:0, 1.000]) [buf: 1] 
   (at 3, cp->sounds[2][0:0, 1.000]) [buf: 3] 
   (at 4, cp->sounds[4][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[2][1:2, 1.000]) [buf: 3] 
   (at 7, cp->sounds[1][2:9, 1.000]) [buf: 10] 
   (at 15, end_mark)

 (insert 15 1) ; insert-samples [5:8]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[3][0:0, 1.000]) [buf: 1] 
   (at 3, cp->sounds[2][0:0, 1.000]) [buf: 3] 
   (at 4, cp->sounds[4][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[2][1:2, 1.000]) [buf: 3] 
   (at 7, cp->sounds[1][2:9, 1.000]) [buf: 10] 
   (at 15, cp->sounds[5][0:0, 1.000]) [buf: 1] 
   (at 16, end_mark)
")))
	    (snd-display ";new 8: ~A" (display-edits)))
	(delete-samples 2 1)
	(if (not (string-=? (display-edits ind 0 6) "
 (delete 2 1) ; delete-samples [6:7]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[2][0:0, 1.000]) [buf: 3] 
   (at 3, cp->sounds[4][0:0, 1.000]) [buf: 1] 
   (at 4, cp->sounds[2][1:2, 1.000]) [buf: 3] 
   (at 6, cp->sounds[1][2:9, 1.000]) [buf: 10] 
   (at 14, cp->sounds[5][0:0, 1.000]) [buf: 1] 
   (at 15, end_mark)
"))
	    (snd-display ";new 9: ~A" (display-edits ind 0 6)))
	(delete-samples 0 5)
	(if (not (string-=? (display-edits ind 0 7) "
 (delete 0 5) ; delete-samples [7:4]:
   (at 0, cp->sounds[2][2:2, 1.000]) [buf: 3] 
   (at 1, cp->sounds[1][2:9, 1.000]) [buf: 10] 
   (at 9, cp->sounds[5][0:0, 1.000]) [buf: 1] 
   (at 10, end_mark)
"))
	    (snd-display ";new 10: ~A" (display-edits ind 0 7)))
	(delete-samples 6 4)
	(if (not (string-=? (display-edits ind 0 8) "
 (delete 6 4) ; delete-samples [8:3]:
   (at 0, cp->sounds[2][2:2, 1.000]) [buf: 3] 
   (at 1, cp->sounds[1][2:6, 1.000]) [buf: 10] 
   (at 6, end_mark)
"))
	    (snd-display ";new 11: ~A" (display-edits ind 0 8)))
	(delete-samples 0 1)
	(if (not (string-=? (display-edits ind 0 9) "
 (delete 0 1) ; delete-samples [9:2]:
   (at 0, cp->sounds[1][2:6, 1.000]) [buf: 10] 
   (at 5, end_mark)
"))
	    (snd-display ";new 12: ~A" (display-edits ind 0 9)))
	
	(delete-samples 0 5)
	(if (not (string-=? (display-edits ind 0 10) "
 (delete 0 5) ; delete-samples [10:1]:
   (at 0, end_mark)
"))
	    (snd-display ";new 13: ~A" (display-edits ind 0 10)))
	(delete-samples 0 10)
	(if (not (= (edit-position) 10))
	    (snd-display ";no-op delete deleted something! ~A" (display-edits)))
	(insert-samples 0 3 (make-vct 3))
	(if (not (string-=? (display-edits ind 0 11) "
 (insert 0 3) ; insert-samples [11:2]:
   (at 0, cp->sounds[6][0:2, 1.000]) [buf: 3] 
   (at 3, end_mark)
"))
	    (snd-display ";new 14: ~A" (display-edits ind 0 11)))
	(delete-samples 2 1)
	(if (not (string-=? (display-edits ind 0 12) "
 (delete 2 1) ; delete-samples [12:2]:
   (at 0, cp->sounds[6][0:1, 1.000]) [buf: 3] 
   (at 2, end_mark)
"))
	    (snd-display ";new 15: ~A" (display-edits ind 0 12)))
	(set! (sample 0) .5)
	(if (not (string-=? (display-edits ind 0 13) "
 (set 0 1) ; set! sample [13:3]:
   (at 0, cp->sounds[7][0:0, 1.000]) [buf: 1] 
   (at 1, cp->sounds[6][1:1, 1.000]) [buf: 3] 
   (at 2, end_mark)
"))
	    (snd-display ";new 16: ~A" (display-edits ind 0 13)))
	(set! (sample 1) .5)
	(if (not (string-=? (display-edits ind 0 14) "
 (set 1 1) ; set! sample [14:3]:
   (at 0, cp->sounds[7][0:0, 1.000]) [buf: 1] 
   (at 1, cp->sounds[8][0:0, 1.000]) [buf: 1] 
   (at 2, end_mark)
"))
	    (snd-display ";new 17: ~A" (display-edits ind 0 14)))  
	(map-channel (lambda (y) 1.0) 0 10)
	(if (not (string-=? (display-edits ind 0 15) "
 (set 0 10) ; map-channel [15:2]:
   (at 0, cp->sounds[9][0:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";new 18: ~A" (display-edits ind 0 15)))  
	(insert-samples 0 10 (make-vct 10))
	(if (not (string-=? (display-edits ind 0 16) "
 (insert 0 10) ; insert-samples [16:3]:
   (at 0, cp->sounds[10][0:9, 1.000]) [buf: 10] 
   (at 10, cp->sounds[9][0:9, 1.000]) [buf: 10] 
   (at 20, end_mark)
"))
	    (snd-display ";new 19: ~A" (display-edits ind 0 16)))
	(set! (samples 2 3) (make-vct 3))
	(if (not (string-=? (display-edits ind 0 17) "
 (set 2 3) ; set-samples [17:5]:
   (at 0, cp->sounds[10][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[11][0:2, 1.000]) [buf: 3] 
   (at 5, cp->sounds[10][5:9, 1.000]) [buf: 10] 
   (at 10, cp->sounds[9][0:9, 1.000]) [buf: 10] 
   (at 20, end_mark)
"))
	    (snd-display ";new 20: ~A" (display-edits ind 0 17)))
	(set! (samples 0 12) (make-vct 12))
	(if (not (string-=? (display-edits ind 0 18) "
 (set 0 12) ; set-samples [18:3]:
   (at 0, cp->sounds[12][0:11, 1.000]) [buf: 12] 
   (at 12, cp->sounds[9][2:9, 1.000]) [buf: 10] 
   (at 20, end_mark)
"))
	    (snd-display ";new 21: ~A" (display-edits ind 0 18)))
	(set! (samples 30 10) (make-vct 10))
	(if (not (string-=? (display-edits ind 0 19) "
 (silence 20 11) ; (change-extend) [19:4]:
   (at 0, cp->sounds[12][0:11, 1.000]) [buf: 12] 
   (at 12, cp->sounds[9][2:9, 1.000]) [buf: 10] 
   (at 20, cp->sounds[-1][0:10, 0.000])
   (at 31, end_mark)
"))
	    (snd-display ";new 22 silence extension: ~A" (display-edits ind 0 19)))
	(if (not (string-=? (display-edits ind 0 20) "
 (set 30 10) ; set-samples [20:5]:
   (at 0, cp->sounds[12][0:11, 1.000]) [buf: 12] 
   (at 12, cp->sounds[9][2:9, 1.000]) [buf: 10] 
   (at 20, cp->sounds[-1][0:9, 0.000])
   (at 30, cp->sounds[13][0:9, 1.000]) [buf: 10] 
   (at 40, end_mark)
"))
	    (snd-display ";new 21: ~A" (display-edits ind 0 20)))
	(close-sound ind))
      
      ;; scale/ramp
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 10)
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 2) "
 (scale 0 10) ; scale-channel 0.500 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 0.500]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 0: ~A" (display-edits ind 0 2)))
	(undo)
	(scale-channel 0.5 0 3)
	(if (not (string-=? (display-edits ind 0 2) "
 (scale 0 3) ; scale-channel 0.500 0 3 [2:3]:
   (at 0, cp->sounds[1][0:2, 0.500]) [buf: 10] 
   (at 3, cp->sounds[1][3:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 1: ~A" (display-edits ind 0 2)))
	(undo)
	(scale-channel 0.5 5 5)
	(if (not (string-=? (display-edits ind 0 2) "
 (scale 5 5) ; scale-channel 0.500 5 5 [2:3]:
   (at 0, cp->sounds[1][0:4, 1.000]) [buf: 10] 
   (at 5, cp->sounds[1][5:9, 0.500]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 2: ~A" (display-edits ind 0 2)))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 2) "
 (scale 2 4) ; scale-channel 0.500 2 4 [2:4]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:5, 0.500]) [buf: 10] 
   (at 6, cp->sounds[1][6:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 2a: ~A" (display-edits ind 0 2)))
	(undo)
	(scale-channel 0.5 10 10)
	(if (not (= (edit-position) 1))
	    (snd-display ";scale beyond end edited? ~A" (display-edits)))
	(scale-channel 0.5 100 10)
	(if (not (= (edit-position) 1))
	    (snd-display ";scale way beyond end edited? ~A" (display-edits)))
	(scale-channel 0.5 5 10)
	(if (not (string-=? (display-edits ind 0 2) "
 (scale 5 5) ; scale-channel 0.500 5 5 [2:3]:
   (at 0, cp->sounds[1][0:4, 1.000]) [buf: 10] 
   (at 5, cp->sounds[1][5:9, 0.500]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 3: ~A" (display-edits ind 0 2)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string-=? (display-edits ind 0 2) "
 (set 4 1) ; set! sample [2:4]:
   (at 0, cp->sounds[1][0:3, 1.000]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 4: ~A" (display-edits ind 0 2)))
	(scale-channel 0.5 0 4)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 0 4) ; scale-channel 0.500 0 4 [3:4]:
   (at 0, cp->sounds[1][0:3, 0.500]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 5: ~A" (display-edits ind 0 3)))
	(scale-channel 0.5 4 1)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 4 1) ; scale-channel 0.500 4 1 [4:4]:
   (at 0, cp->sounds[1][0:3, 0.500]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 0.500]) [buf: 1] 
   (at 5, cp->sounds[1][5:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 6: ~A" (display-edits ind 0 4)))
	(scale-channel 0.5 0 7)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 7) ; scale-channel 0.500 0 7 [5:5]:
   (at 0, cp->sounds[1][0:3, 0.250]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 0.250]) [buf: 1] 
   (at 5, cp->sounds[1][5:6, 0.500]) [buf: 10] 
   (at 7, cp->sounds[1][7:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 7: ~A" (display-edits ind 0 5)))
	(scale-channel 0.5 1 4)
	(if (not (string-=? (display-edits ind 0 6) "
 (scale 1 4) ; scale-channel 0.500 1 4 [6:6]:
   (at 0, cp->sounds[1][0:0, 0.250]) [buf: 10] 
   (at 1, cp->sounds[1][1:3, 0.125]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 0.125]) [buf: 1] 
   (at 5, cp->sounds[1][5:6, 0.500]) [buf: 10] 
   (at 7, cp->sounds[1][7:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 8: ~A" (display-edits ind 0 6)))
	(undo 4)
	(scale-channel 0.5 1 8)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 1 8) ; scale-channel 0.500 1 8 [3:6]:
   (at 0, cp->sounds[1][0:0, 1.000]) [buf: 10] 
   (at 1, cp->sounds[1][1:3, 0.500]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 0.500]) [buf: 1] 
   (at 5, cp->sounds[1][5:8, 0.500]) [buf: 10] 
   (at 9, cp->sounds[1][9:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";scl 9: ~A" (display-edits ind 0 3)))
	(undo 2)
	
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 0 10) ; ramp-channel 0.000 1.000 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 1.000, [1]-0.000 -> 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 0: ~A" (display-edits ind 0 2)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 0 10) ; scale-channel 0.500 0 10 [3:2]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]-0.000 -> 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 1: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 0 5) ; scale-channel 0.500 0 5 [3:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.444]) [buf: 10] 
   (at 5, cp->sounds[1][5:9, 1.000, [1]0.556 -> 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 2: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 2 4) ; scale-channel 0.500 2 4 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.111]) [buf: 10] 
   (at 2, cp->sounds[1][2:5, 0.500, [1]0.222 -> 0.556]) [buf: 10] 
   (at 6, cp->sounds[1][6:9, 1.000, [1]0.667 -> 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 3: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 5 5)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 5 5) ; scale-channel 0.500 5 5 [3:3]:
   (at 0, cp->sounds[1][0:4, 1.000, [1]-0.000 -> 0.444]) [buf: 10] 
   (at 5, cp->sounds[1][5:9, 0.500, [1]0.556 -> 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 4: ~A" (display-edits ind 0 3)))
	(undo 2)
	(ramp-channel .2 .6 2 6)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 2 6) ; ramp-channel 0.200 0.600 2 6 [2:4]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:7, 1.000, [1]0.200 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 5: ~A" (display-edits ind 0 2)))
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 0 5) ; scale-channel 0.500 0 5 [3:5]:
   (at 0, cp->sounds[1][0:1, 0.500]) [buf: 10] 
   (at 2, cp->sounds[1][2:4, 0.500, [1]0.200 -> 0.360]) [buf: 10] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.440 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 6: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 2 6)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 2 6) ; scale-channel 0.500 2 6 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:7, 0.500, [1]0.200 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 7: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 5 4)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 5 4) ; scale-channel 0.500 5 4 [3:6]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:4, 1.000, [1]0.200 -> 0.360]) [buf: 10] 
   (at 5, cp->sounds[1][5:7, 0.500, [1]0.440 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:8, 0.500]) [buf: 10] 
   (at 9, cp->sounds[1][9:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 8: ~A" (display-edits ind 0 3)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string-=? (display-edits ind 0 3) "
 (set 4 1) ; set! sample [3:6]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.440 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 9: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 4 1)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 4 1) ; scale-channel 0.500 4 1 [3:6]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][4:4, 0.500, [1]0.360 -> 0.360]) [buf: 10] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.440 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 10: ~A" (display-edits ind 0 3)))
	(undo)
	(delete-sample 4)
	(if (not (string-=? (display-edits ind 0 3) "
 (delete 4 1) ; delete-sample [3:5]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][5:7, 1.000, [1]0.440 -> 0.600]) [buf: 10] 
   (at 7, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 9, end_mark)
"))
	    (snd-display ";ramp 11: ~A" (display-edits ind 0 3)))
	(undo)
	(delete-samples 4 2)
	(if (not (string-=? (display-edits ind 0 3) "
 (delete 4 2) ; delete-samples [3:5]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][6:7, 1.000, [1]0.520 -> 0.600]) [buf: 10] 
   (at 6, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 8, end_mark)
"))
	    (snd-display ";ramp 12: ~A" (display-edits ind 0 3)))
	(undo)
	(delete-samples 4 3)
	(if (not (string-=? (display-edits ind 0 3) "
 (delete 4 3) ; delete-samples [3:5]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][7:7, 1.000, [1]0.600 -> 0.600]) [buf: 10] 
   (at 5, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 7, end_mark)
"))
	    (snd-display ";ramp 13: ~A" (display-edits ind 0 3)))
	(undo)
	(delete-samples 4 4)
	(if (not (string-=? (display-edits ind 0 3) "
 (delete 4 4) ; delete-samples [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 6, end_mark)
"))
	    (snd-display ";ramp 14: ~A" (display-edits ind 0 3)))
	(undo)
	(delete-samples 4 5)
	(if (not (string-=? (display-edits ind 0 3) "
 (delete 4 5) ; delete-samples [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][9:9, 1.000]) [buf: 10] 
   (at 5, end_mark)
"))
	    (snd-display ";ramp 15: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-channel 0.5 4 2)
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 4 2) ; scale-channel 0.500 4 2 [3:6]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[1][4:5, 0.500, [1]0.360 -> 0.440]) [buf: 10] 
   (at 6, cp->sounds[1][6:7, 1.000, [1]0.520 -> 0.600]) [buf: 10] 
   (at 8, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";ramp 16: ~A" (display-edits ind 0 3)))
	(undo)
	(pad-channel 4 1)
	(if (not (string-=? (display-edits ind 0 3) "
 (silence 4 1) ; pad-channel from 4 for 1 [3:6]:
   (at 0, cp->sounds[1][0:1, 1.000]) [buf: 10] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280]) [buf: 10] 
   (at 4, cp->sounds[-1][0:0, 0.000])
   (at 5, cp->sounds[1][4:7, 1.000, [1]0.360 -> 0.600]) [buf: 10] 
   (at 9, cp->sounds[1][8:9, 1.000]) [buf: 10] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp 17: ~A" (display-edits ind 0 3)))
	(close-sound ind))
      
      ;; xramp
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 10)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 0 10) ; xramp-channel 0.000 1.000 32.000 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";xramp 1: ~A" (display-edits ind 0 2)))
	(undo)
	(xramp-channel 0.0 1.0 0.325)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 0 10) ; xramp-channel 0.000 1.000 0.325 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 1.000, [4]-0.000 -> -1.124, off: 1.481, scl: -1.481]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";xramp 2: ~A" (display-edits ind 0 2)))
	(undo)
	(xramp-channel 0.0 1.0 0.0)
	(if (not (string-=? (display-edits ind 0 2) (string-append "
 (scale 0 10) ; scale-channel 0.000 0 10 [2:2]:
   (at 0, cp->sounds[0][0:9, 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 10, end_mark)
")))
	    (snd-display ";xramp 3: ~A" (display-edits ind 0 2)))
	(undo)
	(xramp-channel 0.0 1.0 1.0)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 0 10) ; ramp-channel 0.000 1.000 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 1.000, [1]-0.000 -> 1.000]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";xramp 4: ~A" (display-edits ind 0 2)))
	(undo)
	(xramp-channel 0.5 1.5 32.0)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 0 10) ; xramp-channel 0.500 1.500 32.000 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 1.000, [4]0.000 -> 3.466, off: 0.468, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";xramp 5: ~A" (display-edits ind 0 2)))
	(if (or (fneq (maxamp) 1.5) (fneq (sample 0) 0.5))
	    (snd-display ";xramp 5 vals: ~A ~A" (maxamp) (sample 0)))
	(undo)
	(xramp-channel -0.5 1.5 32.0)
	(if (not (string-=? (display-edits ind 0 2) "
 (ramp 0 10) ; xramp-channel -0.500 1.500 32.000 0 10 [2:2]:
   (at 0, cp->sounds[1][0:9, 1.000, [4]0.000 -> 3.466, off: -0.565, scl: 0.065]) [buf: 10] 
   (at 10, end_mark)
"))
	    (snd-display ";xramp 6: ~A" (display-edits ind 0 2)))
	(if (or (fneq (maxamp) 1.5) (fneq (sample 0) -0.5))
	    (snd-display ";xramp 6 vals: ~A ~A" (maxamp) (sample 0)))
	(undo)
	(xramp-channel 0.0 1.0 32.0)
	(let ((vals (channel->vct))
	      (ctr 0))
	  (scale-channel 0.5)
	  (if (not (string-=? (display-edits ind 0 3) "
 (scale 0 10) ; scale-channel 0.500 0 10 [3:2]:
   (at 0, cp->sounds[1][0:9, 0.500, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	      (snd-display ";xramp 7: ~A" (display-edits ind 0 3)))
	  (set! ctr 0)
	  (let ((baddy (scan-chan (lambda (y) (if (fneq y (* 0.5 (vct-ref vals ctr))) #t (begin (set! ctr (1+ ctr)) #f))))))
	    (if baddy (snd-display ";trouble in xramp 7: ~A" baddy)))
	  (undo)
	  (delete-sample 0)
	  (if (not (string-=? (display-edits ind 0 3) "
 (delete 0 1) ; delete-sample [3:2]:
   (at 0, cp->sounds[1][1:9, 1.000, [4]0.385 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 9, end_mark)
"))
	      (snd-display ";xramp 8: ~A" (display-edits ind 0 3)))
	  (set! ctr 1)
	  (let ((baddy (scan-chan (lambda (y) (if (fneq y (vct-ref vals ctr)) #t (begin (set! ctr (1+ ctr)) #f))))))
	    (if baddy (snd-display ";trouble in xramp 8: ~A" baddy)))
	  (undo)
	  (delete-samples 0 2)
	  (if (not (string-=? (display-edits ind 0 3) "
 (delete 0 2) ; delete-samples [3:2]:
   (at 0, cp->sounds[1][2:9, 1.000, [4]0.770 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 8, end_mark)
"))
	      (snd-display ";xramp 9: ~A" (display-edits ind 0 3)))
	  (set! ctr 2)
	  (let ((baddy (scan-chan (lambda (y) (if (fneq y (vct-ref vals ctr)) #t (begin (set! ctr (1+ ctr)) #f))))))
	    (if baddy (snd-display ";trouble in xramp 9: ~A" baddy)))
	  (undo)
	  (delete-sample 0)
	  (delete-sample 0)
	  (if (not (string-=? (display-edits ind 0 4) "
 (delete 0 1) ; delete-sample [4:2]:
   (at 0, cp->sounds[1][2:9, 1.000, [4]0.770 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 8, end_mark)
"))
	      (snd-display ";xramp 10: ~A" (display-edits ind 0 4)))
	  (undo 2)
	  (delete-sample 4)
	  (if (not (string-=? (display-edits ind 0 3) "
 (delete 4 1) ; delete-sample [3:3]:
   (at 0, cp->sounds[1][0:3, 1.000, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[1][5:9, 1.000, [4]1.925 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 9, end_mark)
"))
	      (snd-display ";xramp 11: ~A" (display-edits ind 0 3)))
	  (undo)
	  (delete-samples 4 2)
	  (if (not (string-=? (display-edits ind 0 3) "
 (delete 4 2) ; delete-samples [3:3]:
   (at 0, cp->sounds[1][0:3, 1.000, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[1][6:9, 1.000, [4]2.310 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 8, end_mark)
"))
	      (snd-display ";xramp 12: ~A" (display-edits ind 0 3)))
	  (undo)
	  (scale-channel 0.5 4 2)
	  (if (not (string-=? (display-edits ind 0 3) "
 (scale 4 2) ; scale-channel 0.500 4 2 [3:4]:
   (at 0, cp->sounds[1][0:3, 1.000, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[1][4:5, 0.500, [4]1.540 -> 1.925, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 6, cp->sounds[1][6:9, 1.000, [4]2.310 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	      (snd-display ";xramp 13: ~A" (display-edits ind 0 3)))
	  (set! ctr 0)
	  (let ((baddy (scan-chan (lambda (y)
				    (if (or (and (> ctr 5) (fneq y (vct-ref vals ctr)))
					    (and (< ctr 4) (fneq y (vct-ref vals ctr)))
					    (and (or (= ctr 4) (= ctr 5)) (fneq y (* 0.5 (vct-ref vals ctr)))))
					#t
					(begin (set! ctr (1+ ctr)) #f))))))
	    (if baddy (snd-display ";trouble in xramp 8: ~A" baddy)))
	  (undo)
	  (scale-channel 0.5 0 2)
	  (if (not (string-=? (display-edits ind 0 3) "
 (scale 0 2) ; scale-channel 0.500 0 2 [3:3]:
   (at 0, cp->sounds[1][0:1, 0.500, [4]0.000 -> 0.385, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 2, cp->sounds[1][2:9, 1.000, [4]0.770 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	      (snd-display ";xramp 14: ~A" (display-edits ind 0 3)))
	  (undo)
	  (pad-channel 4 2)
	  (if (not (string-=? (display-edits ind 0 3) "
 (silence 4 2) ; pad-channel from 4 for 2 [3:4]:
   (at 0, cp->sounds[1][0:3, 1.000, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[-1][0:1, 0.000])
   (at 6, cp->sounds[1][4:9, 1.000, [4]1.540 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 12, end_mark)
"))
	      (snd-display ";xramp 15: ~A" (display-edits ind 0 3)))
	  (undo)
	  (set! (sample 4) 1.0)
	  (if (not (string-=? (display-edits ind 0 3) "
 (set 4 1) ; set! sample [3:4]:
   (at 0, cp->sounds[1][0:3, 1.000, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:9, 1.000, [4]1.925 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	      (snd-display ";xramp 16: ~A" (display-edits ind 0 3)))
	  (undo)
	  (set! (samples 4 2) (make-vct 2))
	  (if (not (string-=? (display-edits ind 0 3) "
 (set 4 2) ; set-samples [3:4]:
   (at 0, cp->sounds[1][0:3, 1.000, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[2][0:1, 1.000]) [buf: 2] 
   (at 6, cp->sounds[1][6:9, 1.000, [4]2.310 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	      (snd-display ";xramp 17: ~A" (display-edits ind 0 3)))
	  (undo)
	  (scale-channel 0.5)
	  (set! (samples 4 2) (make-vct 2))
	  (if (not (string-=? (display-edits ind 0 4) "
 (set 4 2) ; set-samples [4:4]:
   (at 0, cp->sounds[1][0:3, 0.500, [4]0.000 -> 1.155, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 4, cp->sounds[2][0:1, 1.000]) [buf: 2] 
   (at 6, cp->sounds[1][6:9, 0.500, [4]2.310 -> 3.466, off: -0.032, scl: 0.032]) [buf: 10] 
   (at 10, end_mark)
"))
	      (snd-display ";xramp 18: ~A" (display-edits ind 0 4)))
	  )
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 100)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(ramp-channel 0.0 1.0)
	(if (not (string=? (display-edits ind 0 12) "
 (ramp 0 100) ; ramp-channel 0.000 1.000 0 100 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp 1: ~A" (display-edits ind 0 12)))
	(if (fneq (maxamp) 0.5) (snd-display ";multi ramp 1 maxamp: ~A" (maxamp)))
	(undo)
	(ramp-channel 0.1 1.0 10 90)
	(if (not (string=? (display-edits ind 0 12) "
 (ramp 10 90) ; ramp-channel 0.100 1.000 10 90 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.100 -> 0.191]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.201 -> 0.292]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.302 -> 0.393]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.403 -> 0.494]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.504 -> 0.596]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp 2: ~A" (display-edits ind 0 12)))
	(if (fneq (maxamp) 0.5) (snd-display ";multi ramp 2 maxamp: ~A" (maxamp)))
	(undo)
	(ramp-channel 0.0 0.9 0 90)
	(if (not (string=? (display-edits ind 0 12) "
 (ramp 0 90) ; ramp-channel 0.000 0.900 0 90 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.496]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.506 -> 0.597]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.607 -> 0.698]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.708 -> 0.799]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.809 -> 0.900]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp 3: ~A" (display-edits ind 0 12)))
	(if (fneq (maxamp) 0.5) (snd-display ";multi ramp 3 maxamp: ~A" (maxamp)))
	(if (fneq (sample 89) 0.45) (snd-display ";multi ramp 3 sample 89: ~A" (sample 89)))
	(if (fneq (sample 90) 0.5) (snd-display ";multi ramp 3 sample 90: ~A" (sample 90)))
	(undo)
	(ramp-channel 0.1 0.9 10 80)
	(if (not (string=? (display-edits ind 0 12) "
 (ramp 10 80) ; ramp-channel 0.100 0.900 10 80 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.100 -> 0.191]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.201 -> 0.292]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.708 -> 0.799]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.809 -> 0.900]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp 4: ~A" (display-edits ind 0 12)))
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(ramp-channel 0.0 1.0)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(if (not (string=? (display-edits ind 0 12) "
 (scale 90 10) ; scale-channel 0.500 90 10 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp 5: ~A" (display-edits ind 0 12)))
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	
	;; ptree+scale
	(map-channel (lambda (y) 1.0) 0 100)
	(cosine-channel 0 100)
	(let ((map-data (channel->vct)))
	  (undo)
	  (cosine-channel-via-ptree 0 100)
	  (let ((tree-data (channel->vct)))
	    (if (not (vequal map-data tree-data))
		(snd-display ";map and ptree cosine disagree: ~A ~A" map-data tree-data)))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (if (not (string=? (display-edits ind 0 12 #f) "
 (scale 90 10) ; scale-channel 0.500 90 10 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500, loc: 0, pos: 0, scl: 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, loc: 0, pos: 10, scl: 1.000]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, loc: 0, pos: 20, scl: 1.000]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, loc: 0, pos: 30, scl: 1.000]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, loc: 0, pos: 40, scl: 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, loc: 0, pos: 50, scl: 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, loc: 0, pos: 60, scl: 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, loc: 0, pos: 70, scl: 1.000]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, loc: 0, pos: 80, scl: 1.000]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, loc: 0, pos: 90, scl: 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 0: ~A" (display-edits ind 0 12 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and ptree->scl cosine disagree: ~A ~A" map-data scl-data)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (cosine-channel-via-ptree 0 100)
	  (if (not (string=? (display-edits ind 0 12 #f) "
 (ptree[0] 0 100) ; ptree 0 0 100 [12:11]:
   (at 0, cp->sounds[1][0:9, 1.000, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, loc: 0, pos: 80, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 1.000, loc: 0, pos: 90, scl: 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 1: ~A" (display-edits ind 0 12 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and scl->ptree cosine disagree: ~A ~A" map-data scl-data)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (cosine-channel 10 90)
	  (set! map-data (channel->vct))
	  
	  (undo)
	  (cosine-channel-via-ptree 10 90)
	  (let ((tree-data (channel->vct)))
	    (if (not (vequal map-data tree-data))
		(snd-display ";map and ptree 10:90 cosine disagree: ~A ~A" map-data tree-data)))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (if (not (string-=? (display-edits ind 0 12 #f) "
 (scale 90 10) ; scale-channel 0.500 90 10 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, loc: 0, pos: 0, scl: 1.000]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, loc: 0, pos: 10, scl: 1.000]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, loc: 0, pos: 20, scl: 1.000]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, loc: 0, pos: 30, scl: 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, loc: 0, pos: 40, scl: 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, loc: 0, pos: 50, scl: 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, loc: 0, pos: 60, scl: 1.000]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, loc: 0, pos: 70, scl: 1.000]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, loc: 0, pos: 80, scl: 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 2: ~A" (display-edits ind 0 12 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and scl->ptree 10:90 cosine disagree: ~A ~A" map-data scl-data)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (cosine-channel 10 80)
	  (set! map-data (channel->vct))
	  (undo)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (cosine-channel-via-ptree 10 80)
	  (let ((tree-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data tree-data))
		(snd-display ";map and ptree 10:80 cosine disagree: ~A ~A" map-data tree-data)))
	  (if (not (string-=? (display-edits ind 0 12 #f) "
 (ptree[0] 10 80) ; ptree 0 10 80 [12:11]:
   (at 0, cp->sounds[1][0:9, 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 3: ~A" (display-edits ind 0 12 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and scl->ptree 10:80 cosine disagree: ~A ~A" map-data scl-data)))
	  
	  ;; ptree + ramp
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (env-channel '(0 0 1 1))
	  (cosine-channel 0 100)
	  (set! map-data (channel->vct))
	  (undo)
	  (cosine-channel-via-ptree 0 100)
	  (let ((tree-data (channel->vct)))
	    (if (not (vequal map-data tree-data))
		(snd-display ";ptree->ramp cosine disagree: ~A ~A" map-data tree-data)))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (if (not (string-=? (display-edits ind 0 13 #f) "
 (scale 90 10) ; scale-channel 0.500 90 10 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, loc: 0, pos: 0, scl: 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, loc: 0, pos: 10, scl: 1.000]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, loc: 0, pos: 20, scl: 1.000]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, loc: 0, pos: 30, scl: 1.000]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, loc: 0, pos: 40, scl: 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, loc: 0, pos: 50, scl: 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, loc: 0, pos: 60, scl: 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, loc: 0, pos: 70, scl: 1.000]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, loc: 0, pos: 80, scl: 1.000]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, loc: 0, pos: 90, scl: 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 4: ~A" (display-edits ind 0 13 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and (scl) ptree->ramp cosine disagree: ~A ~A" map-data scl-data)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (env-channel '(0 0 1 1))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (cosine-channel-via-ptree 0 100)
	  (if (not (string-=? (display-edits ind 0 13 #f) "
 (ptree[0] 0 100) ; ptree 0 0 100 [13:11]:
   (at 0, cp->sounds[1][0:9, 1.000, [1]0.000 -> 0.091, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, [1]0.101 -> 0.192, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, [1]0.202 -> 0.293, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, [1]0.303 -> 0.394, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, [1]0.404 -> 0.495, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]0.505 -> 0.596, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, [1]0.606 -> 0.697, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, [1]0.707 -> 0.798, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, [1]0.808 -> 0.899, loc: 0, pos: 80, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 1.000, [1]0.909 -> 1.000, loc: 0, pos: 90, scl: 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 5: ~A" (display-edits ind 0 13 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 2.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and ptree->ramp (scl) cosine disagree: ~A ~A" map-data scl-data)))
	  
	  (scale-channel 0.5)
	  (if (not (string-=? (display-edits ind 0 14 #f) "
 (scale 0 100) ; scale-channel 0.500 0 100 [14:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, loc: 0, pos: 80, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, loc: 0, pos: 90, scl: 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 5 + scl: ~A" (display-edits ind 0 14 #f)))
	  (let ((scl-data (vct-scale! (channel->vct) 4.0)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";map and ptree->ramp (scl twice) cosine disagree: ~A ~A" map-data scl-data)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (env-channel '(0 0 1 1))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (cosine-channel-via-ptree 10 80)
	  (set! map-data (channel->vct))
	  (if (not (string-=? (display-edits ind 0 13 #f) "
 (ptree[0] 10 80) ; ptree 0 10 80 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, [1]0.101 -> 0.192, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, [1]0.202 -> 0.293, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, [1]0.303 -> 0.394, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, [1]0.404 -> 0.495, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]0.505 -> 0.596, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, [1]0.606 -> 0.697, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, [1]0.707 -> 0.798, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, [1]0.808 -> 0.899, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 6: ~A" (display-edits ind 0 13 #f)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (env-channel '(0 0 1 1))
	  (cosine-channel-via-ptree 10 80)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (let ((scl-data (channel->vct)))
	    (if (not (vequal map-data scl-data))
		(snd-display ";ptree+ramp order matters? ~A ~A" map-data scl-data)))
	  
	  (revert-sound)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (env-channel '(0 0 1 1))
	  (cosine-channel-via-ptree 15 70)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (if (not (string-=? (display-edits ind 0 13 #f) "
 (scale 90 10) ; scale-channel 0.500 90 10 [13:13]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091]) [buf: 100] 
   (at 10, cp->sounds[1][10:14, 0.500, [1]0.101 -> 0.141]) [buf: 100] 
   (at 15, cp->sounds[1][15:19, 0.500, [1]0.152 -> 0.192, loc: 0, pos: 0, scl: 1.000]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, loc: 0, pos: 5, scl: 1.000]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, loc: 0, pos: 15, scl: 1.000]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, loc: 0, pos: 25, scl: 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, loc: 0, pos: 35, scl: 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, loc: 0, pos: 45, scl: 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, loc: 0, pos: 55, scl: 1.000]) [buf: 100] 
   (at 80, cp->sounds[1][80:84, 0.500, [1]0.808 -> 0.848, loc: 0, pos: 65, scl: 1.000]) [buf: 100] 
   (at 85, cp->sounds[1][85:89, 0.500, [1]0.859 -> 0.899]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";multi-tree 7: ~A" (display-edits ind 0 13 #f)))
	  (close-sound ind)))
      
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 100)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.75 (* i 10) 10))
	(ptree-channel (lambda (y data forward)
			 (declare (y real) (data vct) (forward boolean))
			 (* y (vct-ref data 0)))
		       0 (frames) ind 0 #f #f
		       (lambda (pos dur)
			 (vct 0.5)))
	(ptree-channel (lambda (y data forward)
			 (declare (y real) (data vct) (forward boolean))
			 (* y (vct-ref data 0)))
		       20 45 ind 0 #f #f
		       (lambda (pos dur)
			 (vct 0.25)))
	(let ((data (channel->vct))
	      (orig (make-vct 100 1.0)))
	  (vct-scale! orig 0.75) ; scale-channel
	  (vct-scale! orig 0.5)  ; ptree-channel
	  (do ((i 20 (1+ i)))
	      ((= i 65))
	    (vct-set! orig i (* (vct-ref orig i) .25)))
	  (if (not (vvequal orig data))
	      (snd-display ";p2 pos test data: ~A" data))
	  (if (not (string-=? (display-edits ind 0 13 #f) "
 (ptree[1] 20 45) ; ptree 1 20 45 [13:12]:
   (at 0, cp->sounds[1][0:9, 1.000, loc: 0, pos: 0, scl: 0.750]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, loc: 0, pos: 10, scl: 0.750]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 20, scl: 0.750]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, loc2: 1, pos2: 10, scl2: 1.000, loc: 0, pos: 30, scl: 0.750]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, loc2: 1, pos2: 20, scl2: 1.000, loc: 0, pos: 40, scl: 0.750]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, loc2: 1, pos2: 30, scl2: 1.000, loc: 0, pos: 50, scl: 0.750]) [buf: 100] 
   (at 60, cp->sounds[1][60:64, 1.000, loc2: 1, pos2: 40, scl2: 1.000, loc: 0, pos: 60, scl: 0.750]) [buf: 100] 
   (at 65, cp->sounds[1][65:69, 1.000, loc: 0, pos: 65, scl: 0.750]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, loc: 0, pos: 70, scl: 0.750]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, loc: 0, pos: 80, scl: 0.750]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 1.000, loc: 0, pos: 90, scl: 0.750]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";p2 pos multi: ~A" (display-edits ind 0 13 #f)))
	  (close-sound ind)))
      
      (let ((ind (new-sound "test.snd"))
	    (map-data #f))
	
	;; ptree + xramp
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(env-sound '(0 0 1 1) 0 100 32.0)
	(cosine-channel 0 100)
	(set! map-data (channel->vct))
	(undo)
	(cosine-channel-via-ptree 0 100)
	(let ((tree-data (channel->vct)))
	  (if (not (vequal map-data tree-data))
	      (snd-display ";ptree->xramp cosine disagree: ~A ~A" map-data tree-data)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(if (not (string-=? (display-edits ind 0 13 #f) "
 (scale 90 10) ; scale-channel 0.500 90 10 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [4]0.000 -> 0.315, off: -0.032, scl: 0.032, loc: 0, pos: 0, scl: 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [4]0.350 -> 0.665, off: -0.032, scl: 0.032, loc: 0, pos: 10, scl: 1.000]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [4]0.700 -> 1.015, off: -0.032, scl: 0.032, loc: 0, pos: 20, scl: 1.000]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [4]1.050 -> 1.365, off: -0.032, scl: 0.032, loc: 0, pos: 30, scl: 1.000]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [4]1.400 -> 1.715, off: -0.032, scl: 0.032, loc: 0, pos: 40, scl: 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [4]1.750 -> 2.065, off: -0.032, scl: 0.032, loc: 0, pos: 50, scl: 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [4]2.100 -> 2.416, off: -0.032, scl: 0.032, loc: 0, pos: 60, scl: 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [4]2.451 -> 2.766, off: -0.032, scl: 0.032, loc: 0, pos: 70, scl: 1.000]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [4]2.801 -> 3.116, off: -0.032, scl: 0.032, loc: 0, pos: 80, scl: 1.000]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [4]3.151 -> 3.466, off: -0.032, scl: 0.032, loc: 0, pos: 90, scl: 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-tree 9: ~A" (display-edits ind 0 13 #f)))
	(let ((scl-data (vct-scale! (channel->vct) 2.0)))
	  (if (not (vequal map-data scl-data))
	      (snd-display ";map and (scl) ptree->xramp cosine disagree: ~A ~A" map-data scl-data)))
	
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(env-sound '(0 0 1 1) 0 100 32.0)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(cosine-channel-via-ptree 0 100)
	(if (not (string-=? (display-edits ind 0 13 #f) "
 (ptree[0] 0 100) ; ptree 0 0 100 [13:11]:
   (at 0, cp->sounds[1][0:9, 1.000, [4]0.000 -> 0.315, off: -0.032, scl: 0.032, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, [4]0.350 -> 0.665, off: -0.032, scl: 0.032, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, [4]0.700 -> 1.015, off: -0.032, scl: 0.032, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, [4]1.050 -> 1.365, off: -0.032, scl: 0.032, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, [4]1.400 -> 1.715, off: -0.032, scl: 0.032, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [4]1.750 -> 2.065, off: -0.032, scl: 0.032, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, [4]2.100 -> 2.416, off: -0.032, scl: 0.032, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, [4]2.451 -> 2.766, off: -0.032, scl: 0.032, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, [4]2.801 -> 3.116, off: -0.032, scl: 0.032, loc: 0, pos: 80, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 1.000, [4]3.151 -> 3.466, off: -0.032, scl: 0.032, loc: 0, pos: 90, scl: 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-tree 10: ~A" (display-edits ind 0 13 #f)))
	(let ((scl-data (vct-scale! (channel->vct) 2.0)))
	  (if (not (vequal map-data scl-data))
	      (snd-display ";map and ptree->xramp (scl) cosine disagree: ~A ~A" map-data scl-data)))
	
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(env-sound '(0 0 1 1) 0 100 32.0)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(cosine-channel-via-ptree 10 80)
	(set! map-data (channel->vct))
	(if (not (string-=? (display-edits ind 0 13 #f) "
 (ptree[0] 10 80) ; ptree 0 10 80 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [4]0.000 -> 0.315, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, [4]0.350 -> 0.665, off: -0.032, scl: 0.032, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, [4]0.700 -> 1.015, off: -0.032, scl: 0.032, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, [4]1.050 -> 1.365, off: -0.032, scl: 0.032, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, [4]1.400 -> 1.715, off: -0.032, scl: 0.032, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [4]1.750 -> 2.065, off: -0.032, scl: 0.032, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, [4]2.100 -> 2.416, off: -0.032, scl: 0.032, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, [4]2.451 -> 2.766, off: -0.032, scl: 0.032, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, [4]2.801 -> 3.116, off: -0.032, scl: 0.032, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [4]3.151 -> 3.466, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-tree 11: ~A" (display-edits ind 0 13 #f)))
	
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(env-sound '(0 0 1 1) 0 100 32.0)
	(cosine-channel-via-ptree 10 80)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(let ((scl-data (channel->vct)))
	  (if (not (vequal map-data scl-data))
	      (snd-display ";ptree+xramp order matters? ~A ~A" map-data scl-data)))
	
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(env-sound '(0 0 1 1) 0 100 32.0)
	(cosine-channel-via-ptree 15 70)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(if (not (string-=? (display-edits ind 0 13 #f) "
 (scale 90 10) ; scale-channel 0.500 90 10 [13:13]:
   (at 0, cp->sounds[1][0:9, 0.500, [4]0.000 -> 0.315, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 10, cp->sounds[1][10:14, 0.500, [4]0.350 -> 0.490, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 15, cp->sounds[1][15:19, 0.500, [4]0.525 -> 0.665, off: -0.032, scl: 0.032, loc: 0, pos: 0, scl: 1.000]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [4]0.700 -> 1.015, off: -0.032, scl: 0.032, loc: 0, pos: 5, scl: 1.000]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [4]1.050 -> 1.365, off: -0.032, scl: 0.032, loc: 0, pos: 15, scl: 1.000]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [4]1.400 -> 1.715, off: -0.032, scl: 0.032, loc: 0, pos: 25, scl: 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [4]1.750 -> 2.065, off: -0.032, scl: 0.032, loc: 0, pos: 35, scl: 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [4]2.100 -> 2.416, off: -0.032, scl: 0.032, loc: 0, pos: 45, scl: 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [4]2.451 -> 2.766, off: -0.032, scl: 0.032, loc: 0, pos: 55, scl: 1.000]) [buf: 100] 
   (at 80, cp->sounds[1][80:84, 0.500, [4]2.801 -> 2.941, off: -0.032, scl: 0.032, loc: 0, pos: 65, scl: 1.000]) [buf: 100] 
   (at 85, cp->sounds[1][85:89, 0.500, [4]2.976 -> 3.116, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [4]3.151 -> 3.466, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-tree 12: ~A" (display-edits ind 0 13 #f)))
	
	(close-sound ind))
      
      ;; ptree2
      (let ((ind (new-sound "test.snd"))
	    (case1 #f)
	    (case2 #f))
	(map-chan (lambda (y) 1.0) 0 10)
	(ptree-channel (lambda (y) (* y 0.5)))
	(if (not (vequal (channel->vct) (make-vct 11 0.5)))
	    (snd-display ";ptree2 0: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 2) "
 (ptree[0] 0 11) ; ptree 0 0 11 [2:2]:
   (at 0, cp->sounds[1][0:10, 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) (* y 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2 3: ~A" (display-edits ind 0 2)))
	(ptree-channel (lambda (y) (* y 1.5)))
	(if (not (vequal (channel->vct) (make-vct 11 0.75)))
	    (snd-display ";ptree2 1: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 3) "
 (ptree[1] 0 11) ; ptree 1 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) (* y 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2 4: ~A" (display-edits ind 0 3)))
	(undo)
	(scale-by 1.5)
	(if (not (vequal (channel->vct) (make-vct 11 0.75)))
	    (snd-display ";ptree2 5: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 3) "
 (scale 0 11) ; scale-channel 1.500 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.500, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) (* y 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2 6: ~A" (display-edits ind 0 3)))
	(ptree-channel (lambda (y) (* y 0.1)))
	(if (not (vequal (channel->vct) (make-vct 11 0.075)))
	    (snd-display ";ptree2 7: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 4) "
 (ptree[1] 0 11) ; ptree 1 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, loc2: 1, pos2: 0, scl2: 1.500, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) (* y 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2 8: ~A" (display-edits ind 0 4)))
	(undo 3)
	(scale-by 0.5)
	(ptree-channel (lambda (y) (* y 1.5)))
	(scale-by 2.0)
	(ptree-channel (lambda (y) (* y 0.1)))
	(scale-by 3.0)
	(if (not (vequal (channel->vct) (make-vct 11 0.45)))
	    (snd-display ";ptree2 9: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 6) "
 (scale 0 11) ; scale-channel 3.000 0 11 [6:2]:
   (at 0, cp->sounds[1][0:10, 3.000, loc2: 1, pos2: 0, scl2: 2.000, loc: 0, pos: 0, scl: 0.500, code: (lambda (y) (* y 1.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2 10: ~A" (display-edits ind 0 6)))
	(undo 2)
	(ptree-channel (lambda (y) (* y 0.1)) 2 4)
	(scale-by 3.0)
	(if (not (vequal (channel->vct) (vct 4.500 4.500 0.450 0.450 0.450 0.450 4.500 4.500 4.500 4.500 4.500)))
	    (snd-display ";ptree2 11: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 6) "
 (scale 0 11) ; scale-channel 3.000 0 11 [6:4]:
   (at 0, cp->sounds[1][0:1, 6.000, loc: 0, pos: 0, scl: 0.500, code: (lambda (y) (* y 1.5))]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 3.000, loc2: 1, pos2: 0, scl2: 2.000, loc: 0, pos: 2, scl: 0.500, code: (lambda (y) (* y 1.5))]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 6.000, loc: 0, pos: 6, scl: 0.500, code: (lambda (y) (* y 1.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2 12: ~A" (display-edits ind 0 6)))
	(close-sound ind))
      
      ;; ptree2-zero
      (let ((ind (new-sound "test.snd"))
	    (case1 #f)
	    (case2 #f))
	(map-chan (lambda (y) 1.0) 0 10)
	(scale-by 0.0)
	(ptree-channel (lambda (y) (+ y 0.5)))
	(if (not (vequal (channel->vct) (make-vct 11 0.5)))
	    (snd-display ";ptree2-zero 0: ~A" (channel->vct)))
	(ptree-channel (lambda (y) (+ y 0.25)))
	(if (not (vequal (channel->vct) (make-vct 11 0.75)))
	    (snd-display ";ptree2-zero 1: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 4) (string-append "
 (ptree[1] 0 11) ; ptree 1 0 11 [4:2]:
   (at 0, cp->sounds[0][0:10, 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 0.000, code: (lambda (y) (+ y 0.5))]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	    (snd-display ";ptree2-zero 2: ~A" (display-edits ind 0 4)))
	(undo 3)
	(scale-channel 0.0 2 4)
	(ptree-channel (lambda (y) (+ y 0.5)))
	(ptree-channel (lambda (y) (+ y 0.25)))
	(if (not (vequal (channel->vct) (vct 1.750 1.750 0.750 0.750 0.750 0.750 1.750 1.750 1.750 1.750 1.750)))
	    (snd-display ";ptree2-zero 3: ~A" (channel->vct)))
	
	;; ptree2-ramp
	(revert-sound)
	(map-chan (lambda (y) 1.0) 0 10)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y) (* y .5)))
	(ptree-channel (lambda (y) (+ y .25)))
	(if (not (vequal (channel->vct) (vct 0.250 0.300 0.350 0.400 0.450 0.500 0.550 0.600 0.650 0.700 0.750)))
	    (snd-display ";ptree2-ramp 1: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 4) "
 (ptree[1] 0 11) ; ptree 1 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) (* y 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2-ramp 2: ~A" (display-edits ind 0 4)))
	(scale-by 0.5)
	(if (not (vequal (channel->vct) (vct 0.125 0.150 0.175 0.200 0.225 0.250 0.275 0.300 0.325 0.350 0.375)))
	    (snd-display ";ptree2-ramp 3: ~A" (channel->vct)))
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 11) ; scale-channel 0.500 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) (* y 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree2-ramp 4: ~A" (display-edits ind 0 5)))
	
	;; ptree+ramp3
	(revert-sound)
	(map-chan (lambda (y) 1.0) 0 10)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y) (* y .5)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp3: ~A" (display-edits ind 0 5 #f)))
	(undo 1)
	(ptree-channel (lambda (y data forward)
			 (declare (y real) (data vct) (forward boolean))
			 (* y (vct-ref data 0)))
		       0 (frames) ind 0 #f #t
		       (lambda (pos dur)
			 (vct 0.5)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptreec-ramp3: ~A" (display-edits ind 0 5 #f)))
	(undo 1)
	(xen-channel (lambda (y data forward) (* y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.25)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (xen[0] 0 11) ; xen 0 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xen-ramp3: ~A" (display-edits ind 0 5 #f)))
	(undo 4)
	(xramp-channel 0.0 1.0 32.0)
	(xramp-channel 0.0 1.0 32.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [3]0.000 -> 3.466, [4]0.000 -> 3.466, off: -0.032, scl: 0.032, off2: -0.032, scl2: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp2: ~A" (display-edits ind 0 4)))
	(close-sound ind))
      
      ;; ramp2
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 0: ~A" (display-edits ind 0 3)))
	(if (not (vequal (channel->vct) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 1.000)))
	    (snd-display ";ramp2 (1): ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 11) ; scale-channel 0.500 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 1: ~A" (display-edits ind 0 4)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.400, [2]-0.000 -> 0.400]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [1]0.500 -> 1.000, [2]0.500 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 2: ~A" (display-edits ind 0 4)))
	(if (not (vequal (channel->vct) (vct 0.000 0.005 0.020 0.045 0.080 0.250 0.360 0.490 0.640 0.810 1.000)))
	    (snd-display ";ramp2 (2): ~A" (channel->vct)))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 2 4) ; scale-channel 0.500 2 4 [4:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [2]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 0.500, [1]0.200 -> 0.500, [2]0.200 -> 0.500]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [1]0.600 -> 1.000, [2]0.600 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 3: ~A" (display-edits ind 0 4)))
	(undo 2)
	(ramp-channel 0.75 0.25)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; ramp-channel 0.750 0.250 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]0.750 -> 0.250]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 4: ~A" (display-edits ind 0 3)))
	(undo)
	(ramp-channel .2 .6 2 6)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 2 6) ; ramp-channel 0.200 0.600 2 6 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:7, 1.000, [1]0.200 -> 0.700, [2]0.200 -> 0.600]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 5: ~A" (display-edits ind 0 3)))
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:5]:
   (at 0, cp->sounds[1][0:1, 0.500, [1]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 0.500, [1]0.200 -> 0.400, [2]0.200 -> 0.360]) [buf: 11] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [2]0.440 -> 0.600]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 6: ~A" (display-edits ind 0 4)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string-=? (display-edits ind 0 4) "
 (set 4 1) ; set! sample [4:6]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.300, [2]0.200 -> 0.280]) [buf: 11] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [2]0.440 -> 0.600]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 7: ~A" (display-edits ind 0 4)))
	(undo 3)
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 100)
	
	;; multi-ramp2
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(ramp-channel 0.0 1.0)
	(ramp-channel 1.0 0.0)
	(if (not (string-=? (display-edits ind 0 13) "
 (ramp 0 100) ; ramp-channel 1.000 0.000 0 100 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, [2]1.000 -> 0.909]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, [2]0.899 -> 0.808]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, [2]0.798 -> 0.707]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, [2]0.697 -> 0.606]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, [2]0.596 -> 0.505]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, [2]0.495 -> 0.404]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, [2]0.394 -> 0.303]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, [2]0.293 -> 0.202]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, [2]0.192 -> 0.101]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, [2]0.091 -> -0.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp2 1: ~A" (display-edits ind 0 13)))
	(undo 12)
	(ramp-channel 0.0 1.0 10 20)
	(ramp-channel 0.0 1.0 50 10)
	(ramp-channel 0.0 1.0 25 10)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 25 10) ; ramp-channel 0.000 1.000 25 10 [4:8]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:24, 1.000, [1]-0.000 -> 0.737]) [buf: 100] 
   (at 25, cp->sounds[1][25:29, 1.000, [1]0.789 -> 1.000, [2]-0.000 -> 0.444]) [buf: 100] 
   (at 30, cp->sounds[1][30:34, 1.000, [1]0.556 -> 1.000]) [buf: 100] 
   (at 35, cp->sounds[1][35:49, 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]-0.000 -> 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:99, 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp2 2: ~A" (display-edits ind 0 4)))
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	;; ramp ptree cases
	(for-each
	 (lambda (func func-zero name)
	   (func)
	   (ramp-channel 0 1)
	   (if (not (string-=? (display-edits ind 0 3 #f) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	       (snd-display ";~A 1: ~A" name (display-edits ind 0 3 #f)))
	   (if (not (vequal (channel->vct) (vct 0.000 0.050 0.100 0.150 0.200 0.250 0.300 0.350 0.400 0.450 0.500)))
	       (snd-display ";~A 1: ~A" name (channel->vct)))
	   (ramp-channel 0 1)
	   (if (not (string-=? (display-edits ind 0 4 #f) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	       (snd-display ";~A 2: ~A" name (display-edits ind 0 4 #f)))
	   (if (not (vequal (channel->vct) (vct 0.000 0.005 0.020 0.045 0.080 0.125 0.180 0.245 0.320 0.405 0.500)))
	       (snd-display ";~A 2: ~A" name (channel->vct)))
	   (ramp-channel 0 1)
	   (if (not (string-=? (display-edits ind 0 5 #f) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	       (snd-display ";~A 3: ~A" name (display-edits ind 0 5 #f)))
	   (if (not (vequal (channel->vct) (vct 0.000 0.000 0.004 0.013 0.032 0.062 0.108 0.171 0.256 0.364 0.500)))
	       (snd-display ";~A 3: ~A" name (channel->vct)))
	   
	   (undo 4)
	   (scale-by 0.0)
	   (func-zero)
	   (ramp-channel 0 1)
	   (if (not (string-=? (display-edits ind 0 4 #f) (string-append "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[0][0:10, 1.000, [1]-0.000 -> 1.000, loc: 0, pos: 0, scl: 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	       (snd-display ";~A-zero 1: ~A" name (display-edits ind 0 4 #f)))
	   (if (not (vequal (channel->vct) (vct 0.000 0.050 0.100 0.150 0.200 0.250 0.300 0.350 0.400 0.450 0.500)))
	       (snd-display ";~A-zero 1: ~A" name (channel->vct)))
	   (ramp-channel 0 1)
	   (if (not (string-=? (display-edits ind 0 5 #f) (string-append "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [5:2]:
   (at 0, cp->sounds[0][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, loc: 0, pos: 0, scl: 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	       (snd-display ";~A-zero 2: ~A" name (display-edits ind 0 5 #f)))
	   (if (not (vequal (channel->vct) (vct 0.000 0.005 0.020 0.045 0.080 0.125 0.180 0.245 0.320 0.405 0.500)))
	       (snd-display ";~A-zero 2: ~A" name (channel->vct)))
	   (ramp-channel 0 1)
	   (if (not (string-=? (display-edits ind 0 6 #f) (string-append "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [6:2]:
   (at 0, cp->sounds[0][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	       (snd-display ";~A-zero 3: ~A" name (display-edits ind 0 6 #f)))
	   (if (not (vequal (channel->vct) (vct 0.000 0.000 0.004 0.013 0.032 0.062 0.108 0.171 0.256 0.364 0.500)))
	       (snd-display ";~A-zero 3: ~A" name (channel->vct)))
	   (undo 5)
	   )
	 (list 
	  (lambda () (ptree-channel (lambda (y) (* y 0.5))))
	  (lambda () (xen-channel (lambda (y data forward) (* y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.5))))
	  (lambda () (ptree-channel
		      (lambda (y data forward)
			(declare (y real) (data vct) (forward boolean))
			(* y (vct-ref data 0)))
		      0 (frames) ind 0 #f #f (lambda (p d) (vct 0.5)))))
	 (list 
	  (lambda () (ptree-channel (lambda (y) (+ y 0.5))))
	  (lambda () (xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.5))))
	  (lambda () (ptree-channel
		      (lambda (y data forward)
			(declare (y real) (data vct) (forward boolean))
			(+ y (vct-ref data 0)))
		      0 (frames) ind 0 #f #f (lambda (p d) (vct 0.5)))))
	 (list "ramp-ptree" "ramp-xen" "ramp-ptreec"))
	
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	;; xramp ptree cases
	(for-each
	 (lambda (func func-zero name twice)
	   (func)
	   (xramp-channel 0 1 32)
	   (if (not (vequal (channel->vct) (vct 0.000 0.007 0.016 0.029 0.048 0.075 0.113 0.166 0.242 0.349 0.500)))
	       (snd-display ";~A 1: ~A" name (channel->vct)))
	   (if twice
	       (begin
		 (xramp-channel 0 1 32)
		 (if (not (vequal (channel->vct) (vct 0.000 0.000 0.001 0.002 0.005 0.011 0.025 0.055 0.117 0.243 0.500)))
		     (snd-display ";~A 2: ~A" name (channel->vct)))
		 (undo 1)))
	   (undo 2)
	   (scale-by 0.0)
	   (func-zero)
	   (xramp-channel 0 1 32)
	   (if (not (vequal (channel->vct) (vct 0.000 0.007 0.016 0.029 0.048 0.075 0.113 0.166 0.242 0.349 0.500)))
	       (snd-display ";~A-zero 1: ~A" name (channel->vct)))
	   (if twice
	       (begin
		 (xramp-channel 0 1 32)
		 (if (not (vequal (channel->vct) (vct 0.000 0.000 0.001 0.002 0.005 0.011 0.025 0.055 0.117 0.243 0.500)))
		     (snd-display ";~A-zero 2: ~A" name (channel->vct)))
		 (undo 1)))
	   (undo 3))
	 (list 
	  (lambda () (ptree-channel (lambda (y) (* y 0.5))))
	  (lambda () (xen-channel (lambda (y data forward) (* y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.5))))
	  (lambda () (ptree-channel
		      (lambda (y data forward)
			(declare (y real) (data vct) (forward boolean))
			(* y (vct-ref data 0)))
		      0 (frames) ind 0 #f #f (lambda (p d) (vct 0.5)))))
	 (list 
	  (lambda () (ptree-channel (lambda (y) (+ y 0.5))))
	  (lambda () (xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.5))))
	  (lambda () (ptree-channel
		      (lambda (y data forward)
			(declare (y real) (data vct) (forward boolean))
			(+ y (vct-ref data 0)))
		      0 (frames) ind 0 #f #f (lambda (p d) (vct 0.5)))))
	 (list "xramp-ptree" "xramp-xen" "xramp-ptreec")
	 (list #t #f #t))
	
	(close-sound ind))
      
      ;; ramp-xramp, xramp-ramp
      (let ((ind (new-sound "test.snd"))
	    (case1 #f)
	    (case2 #f))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(ramp-channel 0.0 1.0)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 0: ~A" (display-edits ind 0 3)))
	(set! case1 (channel->vct))
	(if (not (vequal case1 (vct 0.000 0.001 0.006 0.018 0.039 0.075 0.135 0.233 0.387 0.628 1.000)))
	    (snd-display ";ramp-xramp (1): ~A" case1))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 11) ; scale-channel 0.500 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 1: ~A" (display-edits ind 0 4)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.400, [4]0.000 -> 1.386, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [1]0.500 -> 1.000, [4]1.733 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 2: ~A" (display-edits ind 0 4)))
	(set! case2 (channel->vct))
	(if (not (vequal case2 (vct 0.000 0.001 0.003 0.009 0.019 0.075 0.135 0.233 0.387 0.628 1.000)))
	    (snd-display ";ramp-xramp (2): ~A" case2))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 2 4) ; scale-channel 0.500 2 4 [4:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 0.500, [1]0.200 -> 0.500, [4]0.693 -> 1.733, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [1]0.600 -> 1.000, [4]2.079 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 3: ~A" (display-edits ind 0 4)))
	(undo 2)
	(xramp-channel 0.75 0.25 32.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; xramp-channel 0.750 0.250 32.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]3.466 -> 0.000, off: 0.234, scl: 0.016]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 4: ~A" (display-edits ind 0 3)))
	(undo)
	(xramp-channel .2 .6 3.0 2 6)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 2 6) ; xramp-channel 0.200 0.600 3.000 2 6 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:7, 1.000, [1]0.200 -> 0.700, [4]-0.000 -> 1.099, off: -0.000, scl: 0.200]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 5: ~A" (display-edits ind 0 3)))
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:5]:
   (at 0, cp->sounds[1][0:1, 0.500, [1]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 0.500, [1]0.200 -> 0.400, [4]-0.000 -> 0.439, off: -0.000, scl: 0.200]) [buf: 11] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [4]0.659 -> 1.099, off: -0.000, scl: 0.200]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 6: ~A" (display-edits ind 0 4)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string-=? (display-edits ind 0 4) "
 (set 4 1) ; set! sample [4:6]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.300, [4]-0.000 -> 0.220, off: -0.000, scl: 0.200]) [buf: 11] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [4]0.659 -> 1.099, off: -0.000, scl: 0.200]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp-xramp 7: ~A" (display-edits ind 0 4)))
	(revert-sound)
	(map-chan (lambda (y) 1.0) 0 10)
	
	(xramp-channel 0.0 1.0 32.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 0: ~A" (display-edits ind 0 3)))
	(if (not (vequal case1 (channel->vct)))
	    (snd-display ";xramp-ramp (1): ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 11) ; scale-channel 0.500 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 1: ~A" (display-edits ind 0 4)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.400, [4]0.000 -> 1.386, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [1]0.500 -> 1.000, [4]1.733 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 2: ~A" (display-edits ind 0 4)))
	(if (not (vequal case2 (channel->vct)))
	    (snd-display ";xramp-ramp (2): ~A" (channel->vct)))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 4) "
 (scale 2 4) ; scale-channel 0.500 2 4 [4:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 0.500, [1]0.200 -> 0.500, [4]0.693 -> 1.733, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [1]0.600 -> 1.000, [4]2.079 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 3: ~A" (display-edits ind 0 4)))
	(undo 2)
	(ramp-channel 0.75 0.25)
	(if (not (string=? (display-edits ind 0 3) "
 (ramp 0 11) ; ramp-channel 0.750 0.250 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]0.750 -> 0.250, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 4: ~A" (display-edits ind 0 3)))
	(undo)
	(ramp-channel .2 .6 2 6)
	(if (not (string=? (display-edits ind 0 3) "
 (ramp 2 6) ; ramp-channel 0.200 0.600 2 6 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:7, 1.000, [1]0.200 -> 0.600, [4]0.693 -> 2.426, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [4]2.773 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 5: ~A" (display-edits ind 0 3)))
	(scale-channel 0.5 0 5)
	(if (not (string=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:5]:
   (at 0, cp->sounds[1][0:1, 0.500, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 0.500, [1]0.200 -> 0.360, [4]0.693 -> 1.386, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.440 -> 0.600, [4]1.733 -> 2.426, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [4]2.773 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 6: ~A" (display-edits ind 0 4)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string=? (display-edits ind 0 4) "
 (set 4 1) ; set! sample [4:6]:
   (at 0, cp->sounds[1][0:1, 1.000, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.280, [4]0.693 -> 1.040, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.440 -> 0.600, [4]1.733 -> 2.426, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [4]2.773 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp-ramp 7: ~A" (display-edits ind 0 4)))
	(close-sound ind))
      
      ;; ramp2+xramp
      (let ((ind (new-sound "test.snd"))
	    (case1 #f)
	    (case2 #f))
	(map-chan (lambda (y) 1.0) 0 10)
	(xramp-channel 0.0 1.0 32.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 0: ~A" (display-edits ind 0 4)))
	(set! case1 (channel->vct))
	(if (not (vequal case1 (vct 0.000 0.000 0.001 0.005 0.015 0.038 0.081 0.163 0.310 0.565 1.000)))
	    (snd-display ";ramp2+xramp (1): ~A" case1))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 11) ; scale-channel 0.500 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 1: ~A" (display-edits ind 0 5)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 5) ; scale-channel 0.500 0 5 [5:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.400, [2]-0.000 -> 0.400, [4]0.000 -> 1.386, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [1]0.500 -> 1.000, [2]0.500 -> 1.000, [4]1.733 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 2: ~A" (display-edits ind 0 5)))
	(set! case2 (channel->vct))
	(if (not (vequal case2 (vct 0.000 0.000 0.001 0.003 0.008 0.038 0.081 0.163 0.310 0.565 1.000)))
	    (snd-display ";ramp2+xramp (2): ~A" case2))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 2 4) ; scale-channel 0.500 2 4 [5:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [2]-0.000 -> 0.100, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 0.500, [1]0.200 -> 0.500, [2]0.200 -> 0.500, [4]0.693 -> 1.733, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [1]0.600 -> 1.000, [2]0.600 -> 1.000, [4]2.079 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 3: ~A" (display-edits ind 0 5)))
	(undo 2)
	(ramp-channel 0.75 0.25)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; ramp-channel 0.750 0.250 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]0.750 -> 0.250, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 4: ~A" (display-edits ind 0 4)))
	(undo)
	(ramp-channel .2 .6 2 6)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 2 6) ; ramp-channel 0.200 0.600 2 6 [4:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:7, 1.000, [1]0.200 -> 0.700, [2]0.200 -> 0.600, [4]0.693 -> 2.426, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000, [4]2.773 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 5: ~A" (display-edits ind 0 4)))
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 5) ; scale-channel 0.500 0 5 [5:5]:
   (at 0, cp->sounds[1][0:1, 0.500, [1]-0.000 -> 0.100, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 0.500, [1]0.200 -> 0.400, [2]0.200 -> 0.360, [4]0.693 -> 1.386, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [2]0.440 -> 0.600, [4]1.733 -> 2.426, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000, [4]2.773 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 6: ~A" (display-edits ind 0 5)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string-=? (display-edits ind 0 5) "
 (set 4 1) ; set! sample [5:6]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [4]0.000 -> 0.347, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.300, [2]0.200 -> 0.280, [4]0.693 -> 1.040, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [2]0.440 -> 0.600, [4]1.733 -> 2.426, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000, [4]2.773 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2+xramp 7: ~A" (display-edits ind 0 5)))
	(revert-sound)
	(map-chan (lambda (y) 1.0) 0 10)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp+ramp2 0: ~A" (display-edits ind 0 4)))
	(if (not (vequal case1 (channel->vct)))
	    (snd-display ";xramp+ramp2 (1): ~A" (channel->vct)))
	
	(revert-sound ind)
	(map-channel (lambda (y) 1.0) 0 100)
	(scale-channel 0.75)
	(ramp-channel .5 1)
	(ptree-channel (lambda (y) (* y (/ 1.0 0.75))))
	(scale-channel 2.0)
	(ramp-channel 1 .5)
	(ptree-channel (lambda (y) (* y .25)))
	(scale-channel 4.0)
	(ramp-channel 0 1)
	(if (fneq (maxamp) 1.0)
	    (snd-display ";rprpr max: ~A" (maxamp)))
	
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd"))
	    (case3 #f))
	(map-channel (lambda (y) 1.0) 0 100)
	(scale-channel 0.5)
	(xramp-channel 1.0 0.0 32.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(set! case3 (channel->vct))
	(undo 4)
	
	;; multi-ramp2+xramp
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(xramp-channel 1.0 0.0 32.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 14) "
 (ramp 0 100) ; ramp-channel 0.000 1.000 0 100 [14:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, [2]0.000 -> 0.091, [4]3.466 -> 3.151, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, [2]0.101 -> 0.192, [4]3.116 -> 2.801, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, [2]0.202 -> 0.293, [4]2.766 -> 2.451, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, [2]0.303 -> 0.394, [4]2.416 -> 2.100, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, [2]0.404 -> 0.495, [4]2.065 -> 1.750, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, [2]0.505 -> 0.596, [4]1.715 -> 1.400, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, [2]0.606 -> 0.697, [4]1.365 -> 1.050, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, [2]0.707 -> 0.798, [4]1.015 -> 0.700, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, [2]0.808 -> 0.899, [4]0.665 -> 0.350, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, [2]0.909 -> 1.000, [4]0.315 -> -0.000, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp2+xramp 1: ~A" (display-edits ind 0 14)))
	(if (not (vequal case3 (channel->vct)))
	    (snd-display ";multi-ramp2+xramp: ~A" (channel->vct)))
	(revert-sound)
	(map-channel (lambda (y) 1.0) 0 100)
	(xramp-channel 1.0 0.0 32.0)
	
	(ramp-channel 0.0 1.0 10 20)
	(ramp-channel 0.0 1.0 50 10)
	(ramp-channel 0.0 1.0 25 10)
	(if (not (string-=? (display-edits ind 0 5) "
 (ramp 25 10) ; ramp-channel 0.000 1.000 25 10 [5:8]:
   (at 0, cp->sounds[1][0:9, 1.000, [4]3.466 -> 3.151, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 10, cp->sounds[1][10:24, 1.000, [1]-0.000 -> 0.737, [4]3.116 -> 2.626, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 25, cp->sounds[1][25:29, 1.000, [1]0.789 -> 1.000, [2]-0.000 -> 0.444, [4]2.591 -> 2.451, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 30, cp->sounds[1][30:34, 1.000, [1]0.556 -> 1.000, [4]2.416 -> 2.275, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 35, cp->sounds[1][35:49, 1.000, [4]2.240 -> 1.750, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]-0.000 -> 1.000, [4]1.715 -> 1.400, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 60, cp->sounds[1][60:99, 1.000, [4]1.365 -> 0.000, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp2+xramp 2: ~A" (display-edits ind 0 5)))
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd"))
	    (case3 #f))
	(map-channel (lambda (y) 1.0) 0 100)
	(scale-channel 0.5)
	(ramp-channel 0.0 1.0)
	(xramp-channel 1.0 0.0 32.0)
	(set! case3 (channel->vct))
	(undo 3)
	
	;; multi-ramp-xramp
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(ramp-channel 0.0 1.0)
	(xramp-channel 1.0 0.0 32.0)
	(if (not (string-=? (display-edits ind 0 13) "
 (ramp 0 100) ; xramp-channel 1.000 0.000 32.000 0 100 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, [4]3.466 -> 3.151, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, [4]3.116 -> 2.801, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, [4]2.766 -> 2.451, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, [4]2.416 -> 2.100, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, [4]2.065 -> 1.750, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, [4]1.715 -> 1.400, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, [4]1.365 -> 1.050, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, [4]1.015 -> 0.700, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, [4]0.665 -> 0.350, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, [4]0.315 -> -0.000, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp-xramp 1: ~A" (display-edits ind 0 13)))
	(if (not (vequal case3 (channel->vct)))
	    (snd-display ";multi-ramp-xramp: ~A" (channel->vct)))
	(undo 12)
	(xramp-channel 0.0 1.0 3.0 10 20)
	(xramp-channel 0.0 1.0 3.0 50 10)
	(xramp-channel 0.0 1.0 3.0 25 10)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 25 10) ; xramp-channel 0.000 1.000 3.000 25 10 [4:8]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:24, 1.000, [4]-0.000 -> 0.810, off: -0.500, scl: 0.500]) [buf: 100] 
   (at 25, cp->sounds[1][25:29, 1.000, [3]0.000 -> 0.488, [4]0.867 -> 1.099, off: -0.500, scl: 0.500, off2: -0.500, scl2: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:34, 1.000, [4]0.610 -> 1.099, off: -0.500, scl: 0.500]) [buf: 100] 
   (at 35, cp->sounds[1][35:49, 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [4]0.000 -> 1.099, off: -0.500, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:99, 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp-xramp 2: ~A" (display-edits ind 0 4)))
	(revert-sound)
	
	(map-channel (lambda (y) 1.0) 0 100)
	;; multi-xramp-ramp
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(xramp-channel 1.0 0.0 32.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 13) "
 (ramp 0 100) ; ramp-channel 0.000 1.000 0 100 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, [4]3.466 -> 3.151, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, [4]3.116 -> 2.801, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, [4]2.766 -> 2.451, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, [4]2.416 -> 2.100, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, [4]2.065 -> 1.750, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, [4]1.715 -> 1.400, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, [4]1.365 -> 1.050, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, [4]1.015 -> 0.700, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, [4]0.665 -> 0.350, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, [4]0.315 -> -0.000, off: -0.032, scl: 0.032]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-xramp-ramp 1: ~A" (display-edits ind 0 13)))
	(if (not (vequal case3 (channel->vct)))
	    (snd-display ";case3 xramp-ramp: ~A" (channel->vct)))
	(undo 12)
	(ramp-channel 0.0 1.0 10 20)
	(ramp-channel 0.0 1.0 50 10)
	(ramp-channel 0.0 1.0 25 10)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 25 10) ; ramp-channel 0.000 1.000 25 10 [4:8]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:24, 1.000, [1]-0.000 -> 0.737]) [buf: 100] 
   (at 25, cp->sounds[1][25:29, 1.000, [1]0.789 -> 1.000, [2]-0.000 -> 0.444]) [buf: 100] 
   (at 30, cp->sounds[1][30:34, 1.000, [1]0.556 -> 1.000]) [buf: 100] 
   (at 35, cp->sounds[1][35:49, 1.000]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]-0.000 -> 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:99, 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-xramp-ramp 2: ~A" (display-edits ind 0 4)))
	(close-sound ind))
      
      ;; xramp2
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(xramp-channel 0.0 1.0 2.0)
	(xramp-channel 0.0 1.0 2.0)
	(if (not (string=? (display-edits ind 0 3) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 2.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [3]0.000 -> 0.693, [4]0.000 -> 0.693, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 0: ~A" (display-edits ind 0 3)))
	(if (not (vequal (channel->vct) (vct 0.000 0.005 0.022 0.053 0.102 0.172 0.266 0.390 0.549 0.750 1.000)))
	    (snd-display ";xramp2 (1): ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string=? (display-edits ind 0 4) "
 (scale 0 11) ; scale-channel 0.500 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [3]0.000 -> 0.693, [4]0.000 -> 0.693, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 1: ~A" (display-edits ind 0 4)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [3]0.000 -> 0.277, [4]0.000 -> 0.277, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [3]0.347 -> 0.693, [4]0.347 -> 0.693, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 2: ~A" (display-edits ind 0 4)))
	(if (not (vequal (channel->vct) (vct 0.000 0.003 0.011 0.027 0.051 0.172 0.266 0.390 0.549 0.750 1.000)))
	    (snd-display ";xramp2 (2): ~A" (channel->vct)))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string=? (display-edits ind 0 4) "
 (scale 2 4) ; scale-channel 0.500 2 4 [4:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [3]0.000 -> 0.069, [4]0.000 -> 0.069, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 0.500, [3]0.139 -> 0.347, [4]0.139 -> 0.347, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [3]0.416 -> 0.693, [4]0.416 -> 0.693, off: -1.000, scl: 1.000, off2: -1.000, scl2: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 3: ~A" (display-edits ind 0 4)))
	(undo 2)
	(xramp-channel 0.75 0.25 0.3)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; xramp-channel 0.750 0.250 0.300 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [3]-1.204 -> -0.000, [4]0.000 -> 0.693, off: -1.000, scl: 1.000, off2: 0.964, scl2: -0.714]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 4: ~A" (display-edits ind 0 3)))
	(undo)
	(xramp-channel .2 .6 32.0 2 6)
	(if (not (string=? (display-edits ind 0 3) "
 (ramp 2 6) ; xramp-channel 0.200 0.600 32.000 2 6 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [4]0.000 -> 0.069, off: -1.000, scl: 1.000]) [buf: 11] 
   (at 2, cp->sounds[1][2:7, 1.000, [3]0.000 -> 3.466, [4]0.139 -> 0.485, off: -1.000, scl: 1.000, off2: 0.187, scl2: 0.013]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [4]0.555 -> 0.693, off: -1.000, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 5: ~A" (display-edits ind 0 3)))
	(scale-channel 0.5 0 5)
	(if (not (string=? (display-edits ind 0 4) "
 (scale 0 5) ; scale-channel 0.500 0 5 [4:5]:
   (at 0, cp->sounds[1][0:1, 0.500, [4]0.000 -> 0.069, off: -1.000, scl: 1.000]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 0.500, [3]0.000 -> 1.386, [4]0.139 -> 0.277, off: -1.000, scl: 1.000, off2: 0.187, scl2: 0.013]) [buf: 11] 
   (at 5, cp->sounds[1][5:7, 1.000, [3]2.079 -> 3.466, [4]0.347 -> 0.485, off: -1.000, scl: 1.000, off2: 0.187, scl2: 0.013]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [4]0.555 -> 0.693, off: -1.000, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 6: ~A" (display-edits ind 0 4)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string=? (display-edits ind 0 4) "
 (set 4 1) ; set! sample [4:6]:
   (at 0, cp->sounds[1][0:1, 1.000, [4]0.000 -> 0.069, off: -1.000, scl: 1.000]) [buf: 11] 
   (at 2, cp->sounds[1][2:3, 1.000, [3]0.000 -> 0.693, [4]0.139 -> 0.208, off: -1.000, scl: 1.000, off2: 0.187, scl2: 0.013]) [buf: 11] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [3]2.079 -> 3.466, [4]0.347 -> 0.485, off: -1.000, scl: 1.000, off2: 0.187, scl2: 0.013]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [4]0.555 -> 0.693, off: -1.000, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp2 7: ~A" (display-edits ind 0 4)))
	(undo 3)
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 100)
	
	;; multi-xramp2
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(xramp-channel 0.0 1.0 3.0)
	(xramp-channel 1.0 0.0 0.3)
	(if (not (string-=? (display-edits ind 0 13) "
 (ramp 0 100) ; xramp-channel 1.000 0.000 0.300 0 100 [13:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [3]-1.204 -> -1.095, [4]-0.000 -> 0.100, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [3]-1.082 -> -0.973, [4]0.111 -> 0.211, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [3]-0.961 -> -0.851, [4]0.222 -> 0.322, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [3]-0.839 -> -0.730, [4]0.333 -> 0.433, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [3]-0.718 -> -0.608, [4]0.444 -> 0.544, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [3]-0.596 -> -0.486, [4]0.555 -> 0.655, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [3]-0.474 -> -0.365, [4]0.666 -> 0.766, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [3]-0.353 -> -0.243, [4]0.777 -> 0.877, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [3]-0.231 -> -0.122, [4]0.888 -> 0.988, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [3]-0.109 -> -0.000, [4]0.999 -> 1.099, off: -0.500, scl: 0.500, off2: 1.429, scl2: -1.429]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-xramp2 1: ~A" (display-edits ind 0 13)))
	(close-sound ind))
      
      ;; ptree+ramp2
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y) y))
	(if (not (string-=? (display-edits ind 0 4 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp2 0: ~A" (display-edits ind 0 4 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 1.000)))
	    (snd-display ";ptree-ramp2 (1): ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (scale 0 11) ; scale-channel 0.500 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp2 1: ~A" (display-edits ind 0 5 #f)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (scale 0 5) ; scale-channel 0.500 0 5 [5:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.400, [2]-0.000 -> 0.400, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [1]0.500 -> 1.000, [2]0.500 -> 1.000, loc: 0, pos: 5, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp2 2: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.005 0.020 0.045 0.080 0.250 0.360 0.490 0.640 0.810 1.000)))
	    (snd-display ";ptree-ramp2 (2): ~A" (channel->vct)))
	(undo 4)
	
	(scale-channel .5)
	(env-channel '(0 0 1 1 2 0))
	(ramp-channel 0 1 2 3)
	(ptree-channel (lambda (y) y))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [5:5]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.200, loc: 0, pos: 0, scl: 0.500]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 1.000, [1]0.400 -> 0.800, [2]0.000 -> 1.000, loc: 0, pos: 2, scl: 0.500]) [buf: 11] 
   (at 5, cp->sounds[1][5:5, 1.000, [1]1.000 -> 1.000, loc: 0, pos: 5, scl: 0.500]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [1]0.800 -> 0.000, loc: 0, pos: 6, scl: 0.500]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp2 4: ~A" (display-edits ind 0 5 #f)))
	
	(undo 4)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y data forward)
			 (declare (y real) (data vct) (forward boolean))
			 (* y (vct-ref data 0)))
		       0 (frames) ind 0 #f #t
		       (lambda (pos dur)
			 (vct 0.5)))
	(if (not (string-=? (display-edits ind 0 4 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp2 5: ~A" (display-edits ind 0 4 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.005 0.020 0.045 0.080 0.125 0.180 0.245 0.320 0.405 0.500)))
	    (snd-display ";ptree+closure+ramp2: ~A" (channel->vct)))
	
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 100)
	
	;; multi-ramp2
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(ramp-channel 0.0 1.0)
	(ramp-channel 1.0 0.0)
	(ptree-channel (lambda (y) y))
	(if (not (string-=? (display-edits ind 0 14 #f) "
 (ptree[0] 0 100) ; ptree 0 0 100 [14:11]:
   (at 0, cp->sounds[1][0:9, 1.000, [1]0.000 -> 0.091, [2]1.000 -> 0.909, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, [1]0.101 -> 0.192, [2]0.899 -> 0.808, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, [1]0.202 -> 0.293, [2]0.798 -> 0.707, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, [1]0.303 -> 0.394, [2]0.697 -> 0.606, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, [1]0.404 -> 0.495, [2]0.596 -> 0.505, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]0.505 -> 0.596, [2]0.495 -> 0.404, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, [1]0.606 -> 0.697, [2]0.394 -> 0.303, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, [1]0.707 -> 0.798, [2]0.293 -> 0.202, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, [1]0.808 -> 0.899, [2]0.192 -> 0.101, loc: 0, pos: 80, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 1.000, [1]0.909 -> 1.000, [2]0.091 -> -0.000, loc: 0, pos: 90, scl: 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";ptree multi-ramp2 1: ~A" (display-edits ind 0 14 #f)))
	(undo 12)
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y) (+ y .1)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp3 0: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.100 0.101 0.108 0.127 0.164 0.225 0.316 0.443 0.612 0.829 1.100)))
	    (snd-display ";ptree-ramp3 1: ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 6 #f) "
 (scale 0 11) ; scale-channel 0.500 0 11 [6:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp3 2: ~A" (display-edits ind 0 6 #f)))
	
	(undo 5)
	(ramp-channel 0 1)
	(ramp-channel 0 1 5 5)
	(ramp-channel 0 1 7 3)
	(ptree-channel (lambda (y) (+ y .1)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (ptree[0] 0 11) ; ptree 0 0 11 [5:5]:
   (at 0, cp->sounds[1][0:4, 1.000, [1]-0.000 -> 0.400, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 5, cp->sounds[1][5:6, 1.000, [1]0.500 -> 0.600, [2]0.000 -> 0.250, loc: 0, pos: 5, scl: 1.000]) [buf: 11] 
   (at 7, cp->sounds[1][7:9, 1.000, [1]0.700 -> 0.900, [2]0.500 -> 1.000, [3]0.000 -> 1.000, loc: 0, pos: 7, scl: 1.000]) [buf: 11] 
   (at 10, cp->sounds[1][10:10, 1.000, [1]1.000 -> 1.000, loc: 0, pos: 10, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree-ramp3 3: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.100 0.200 0.300 0.400 0.500 0.100 0.250 0.100 0.400 1.000 1.100)))
	    (snd-display ";ptree-ramp3 4: ~A" (channel->vct)))
	(close-sound ind))
      
      ;; ramp3
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 0: ~A" (display-edits ind 0 4)))
	(if (not (vequal (channel->vct) (vct 0.000 0.001 0.008 0.027 0.064 0.125 0.216 0.343 0.512 0.729 1.000)))
	    (snd-display ";ramp3 (1): ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 11) ; scale-channel 0.500 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 1: ~A" (display-edits ind 0 5)))
	(undo)
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 5) ; scale-channel 0.500 0 5 [5:3]:
   (at 0, cp->sounds[1][0:4, 0.500, [1]-0.000 -> 0.400, [2]-0.000 -> 0.400, [3]-0.000 -> 0.400]) [buf: 11] 
   (at 5, cp->sounds[1][5:10, 1.000, [1]0.500 -> 1.000, [2]0.500 -> 1.000, [3]0.500 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 2: ~A" (display-edits ind 0 5)))
	(if (not (vequal (channel->vct) (vct 0.000 0.001 0.004 0.014 0.032 0.125 0.216 0.343 0.512 0.729 1.000)))
	    (snd-display ";ramp3 (2): ~A" (channel->vct)))
	(undo)
	(scale-channel 0.5 2 4)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 2 4) ; scale-channel 0.500 2 4 [5:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [2]-0.000 -> 0.100, [3]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:5, 0.500, [1]0.200 -> 0.500, [2]0.200 -> 0.500, [3]0.200 -> 0.500]) [buf: 11] 
   (at 6, cp->sounds[1][6:10, 1.000, [1]0.600 -> 1.000, [2]0.600 -> 1.000, [3]0.600 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 3: ~A" (display-edits ind 0 5)))
	(undo 2)
	(ramp-channel 0.75 0.25)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; ramp-channel 0.750 0.250 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]0.750 -> 0.250]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 4: ~A" (display-edits ind 0 4)))
	(undo)
	(ramp-channel .2 .6 2 6)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 2 6) ; ramp-channel 0.200 0.600 2 6 [4:4]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [2]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:7, 1.000, [1]0.200 -> 0.700, [2]0.200 -> 0.700, [3]0.200 -> 0.600]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000, [2]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 5: ~A" (display-edits ind 0 4)))
	(scale-channel 0.5 0 5)
	(if (not (string-=? (display-edits ind 0 5) "
 (scale 0 5) ; scale-channel 0.500 0 5 [5:5]:
   (at 0, cp->sounds[1][0:1, 0.500, [1]-0.000 -> 0.100, [2]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:4, 0.500, [1]0.200 -> 0.400, [2]0.200 -> 0.400, [3]0.200 -> 0.360]) [buf: 11] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [2]0.500 -> 0.700, [3]0.440 -> 0.600]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000, [2]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 6: ~A" (display-edits ind 0 5)))
	(undo)
	(set! (sample 4) .5)
	(if (not (string-=? (display-edits ind 0 5) "
 (set 4 1) ; set! sample [5:6]:
   (at 0, cp->sounds[1][0:1, 1.000, [1]-0.000 -> 0.100, [2]-0.000 -> 0.100]) [buf: 11] 
   (at 2, cp->sounds[1][2:3, 1.000, [1]0.200 -> 0.300, [2]0.200 -> 0.300, [3]0.200 -> 0.280]) [buf: 11] 
   (at 4, cp->sounds[2][0:0, 1.000]) [buf: 1] 
   (at 5, cp->sounds[1][5:7, 1.000, [1]0.500 -> 0.700, [2]0.500 -> 0.700, [3]0.440 -> 0.600]) [buf: 11] 
   (at 8, cp->sounds[1][8:10, 1.000, [1]0.800 -> 1.000, [2]0.800 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3 7: ~A" (display-edits ind 0 5)))
	(undo 3)
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-channel (lambda (y) 1.0) 0 100)
	
	;; multi-ramp3
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (scale-channel 0.5 (* i 10) 10))
	(ramp-channel 0.0 1.0)
	(ramp-channel 1.0 -0.5)
	(ramp-channel -0.5 1.5)
	(if (not (string=? (display-edits ind 0 14) "
 (ramp 0 100) ; ramp-channel -0.500 1.500 0 100 [14:11]:
   (at 0, cp->sounds[1][0:9, 0.500, [1]0.000 -> 0.091, [2]1.000 -> 0.864, [3]-0.500 -> -0.318]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 0.500, [1]0.101 -> 0.192, [2]0.848 -> 0.712, [3]-0.298 -> -0.116]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 0.500, [1]0.202 -> 0.293, [2]0.697 -> 0.561, [3]-0.096 -> 0.086]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 0.500, [1]0.303 -> 0.394, [2]0.545 -> 0.409, [3]0.106 -> 0.288]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 0.500, [1]0.404 -> 0.495, [2]0.394 -> 0.258, [3]0.308 -> 0.490]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 0.500, [1]0.505 -> 0.596, [2]0.242 -> 0.106, [3]0.510 -> 0.692]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 0.500, [1]0.606 -> 0.697, [2]0.091 -> -0.045, [3]0.712 -> 0.894]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 0.500, [1]0.707 -> 0.798, [2]-0.061 -> -0.197, [3]0.914 -> 1.096]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 0.500, [1]0.808 -> 0.899, [2]-0.212 -> -0.348, [3]1.116 -> 1.298]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 0.500, [1]0.909 -> 1.000, [2]-0.364 -> -0.500, [3]1.318 -> 1.500]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp3 1: ~A" (display-edits ind 0 14)))
	(undo 13)
	(ramp-channel 0.0 1.0 10 30)
	(ramp-channel 0.0 1.0 50 20)
	(ramp-channel 0.0 1.0 20 15)
	(ramp-channel 0.0 1.0 30 30)
	(if (not (string-=? (display-edits ind 0 5) "
 (ramp 30 30) ; ramp-channel 0.000 1.000 30 30 [5:10]:
   (at 0, cp->sounds[1][0:9, 1.000]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, [1]0.000 -> 0.310]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, [1]0.345 -> 0.655, [2]-0.000 -> 0.643]) [buf: 100] 
   (at 30, cp->sounds[1][30:34, 1.000, [1]0.690 -> 0.828, [2]0.714 -> 1.000, [3]0.000 -> 0.138]) [buf: 100] 
   (at 35, cp->sounds[1][35:39, 1.000, [1]0.862 -> 1.000, [2]0.172 -> 0.310]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, [1]0.345 -> 0.655]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, [1]-0.000 -> 0.474, [2]0.690 -> 1.000]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, [1]0.526 -> 1.000]) [buf: 100] 
   (at 70, cp->sounds[1][70:99, 1.000]) [buf: 100] 
   (at 100, end_mark)
"))
	    (snd-display ";multi-ramp3 2: ~A" (display-edits ind 0 5)))
	
	;; xen
	(let ((ind (new-sound "test.snd")))
	  (map-chan (lambda (y) 1.0) 0 10)
	  (xen-channel (lambda (y data forward)
			 (+ y (list-ref data 0)))
		       0 (frames) ind 0 #f #f
		       (lambda (pos dur)
			 (list 0.5)))
	  (if (not (string=? (display-edits ind 0 2) "
 (xen[0] 0 11) ; xen 0 0 11 [2:2]:
   (at 0, cp->sounds[1][0:10, 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y data forward) (+ y (list-ref data 0))), init: (lambda (pos dur) (list 0.5))]) [buf: 11] 
   (at 11, end_mark)
"))
	      (snd-display ";xen 0: ~A" (display-edits ind 0 2)))
	  (if (not (vequal (channel->vct) (make-vct 11 1.5)))
	      (snd-display ";xen 0: ~A" (channel->vct)))
	  (undo)
	  (ramp-channel 0.0 1.0)
	  (xen-channel (lambda (y data forward)
			 (let ((val (* 0.5 (+ y (list-ref data 0)))))
			   (list-set! data 0 y)
			   val))
		       0 (frames) ind 0 #f #f
		       (let ((edpos (edit-position ind 0)))
			 (lambda (pos dur)
			   (list (if (= pos 0) 0.0
				     (sample (1- pos) ind 0 edpos))))))
	  (if (not (string-=? (display-edits ind 0 2 #f) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [2:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))	
	      (snd-display ";xen 1: ~A" (display-edits ind 0 2 #f)))
	  (if (not (vequal (channel->vct) (vct 0.000 0.050 0.150 0.250 0.350 0.450 0.550 0.650 0.750 0.850 0.950)))
	      (snd-display ";xen 1: ~A" (channel->vct)))
	  (undo 2)
	  
	  (ramp-channel 0.0 1.0)
	  (ramp-channel 1.0 0.0)
	  (let ((outer .5))
	    (xen-channel (lambda (y data forward)
			   (data y))
			 0 (frames) ind 0 #f #f
			 (lambda (pos dur)
			   (lambda (y)
			     (* outer y)))))
	  (if (not (string-=? (display-edits ind 0 4 #f) "
 (xen[0] 0 11) ; xen 0 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]1.000 -> -0.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	      (snd-display ";xen 2: ~A" (display-edits ind 0 4 #f)))
	  (if (not (vequal (channel->vct) (vct 0.000 0.045 0.080 0.105 0.120 0.125 0.120 0.105 0.080 0.045 0.000)))
	      (snd-display ";xen 2: ~A" (channel->vct)))
	  (undo 3)
	  
	  (xramp-channel 0.0 1.0 3.0)
	  (xen-channel (lambda (y data forward)
			 (let ((val (* y (vector-ref data 0))))
			   (vector-set! data 0 (+ (vector-ref data 0) (if forward .1 -.1)))
			   val))
		       0 (frames) ind 0 #f #f
		       (lambda (pos dur)
			 (let ((val (make-vector 1)))
			   (vector-set! val 0 (* pos .1))
			   val)))
	  (if (not (string-=? (display-edits ind 0 3 #f) "
 (xen[0] 0 11) ; xen 0 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [4]-0.000 -> 1.099, off: -0.500, scl: 0.500, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	      (snd-display ";xen 3: ~A" (display-edits ind 0 3 #f)))
	  (if (not (vequal (channel->vct) (vct 0.000 0.006 0.025 0.059 0.110 0.183 0.280 0.405 0.563 0.760 1.000)))
	      (snd-display ";xen 3: ~A" (channel->vct)))
	  
	  (undo 2)
	  (pad-channel 2 5)
	  (xen-channel (lambda (y data forward) (+ y .1)))
	  (if (not (string=? (display-edits ind 0 3 #f) "
 (xen[0] 0 16) ; xen 0 0 16 [3:4]:
   (at 0, cp->sounds[1][0:1, 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 2, cp->sounds[-1][0:4, 1.000, loc: 0, pos: 2, scl: 0.000])
   (at 7, cp->sounds[1][2:10, 1.000, loc: 0, pos: 7, scl: 1.000]) [buf: 11] 
   (at 16, end_mark)
"))
	      (snd-display ";xen 5: ~A" (display-edits ind 0 3 #f)))
	  (if (not (vequal (channel->vct 0 10) (vct 1.100 1.100 0.100 0.100 0.100 0.100 0.100 1.100 1.100 1.100)))
	      (snd-display ";xen 5: ~A" (channel->vct)))
	  
	  (revert-sound ind)
	  (map-channel (lambda (y) 1.0) 0 100)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (scale-channel 0.5 (* i 10) 10))
	  (xen-channel (lambda (y data forward)
			 (let ((val (* y (vector-ref data 0))))
			   (vector-set! data 0 (+ (vector-ref data 0) (if forward .01 -.01)))
			   val))
		       0 (frames) ind 0 #f #f
		       (lambda (pos dur)
			 (let ((val (make-vector 1)))
			   (vector-set! val 0 (* pos .01))
			   val)))
	  (if (not (string=? (display-edits ind 0 12 #f) "
 (xen[0] 0 100) ; xen 0 0 100 [12:11]:
   (at 0, cp->sounds[1][0:9, 1.000, loc: 0, pos: 0, scl: 0.500]) [buf: 100] 
   (at 10, cp->sounds[1][10:19, 1.000, loc: 0, pos: 10, scl: 0.500]) [buf: 100] 
   (at 20, cp->sounds[1][20:29, 1.000, loc: 0, pos: 20, scl: 0.500]) [buf: 100] 
   (at 30, cp->sounds[1][30:39, 1.000, loc: 0, pos: 30, scl: 0.500]) [buf: 100] 
   (at 40, cp->sounds[1][40:49, 1.000, loc: 0, pos: 40, scl: 0.500]) [buf: 100] 
   (at 50, cp->sounds[1][50:59, 1.000, loc: 0, pos: 50, scl: 0.500]) [buf: 100] 
   (at 60, cp->sounds[1][60:69, 1.000, loc: 0, pos: 60, scl: 0.500]) [buf: 100] 
   (at 70, cp->sounds[1][70:79, 1.000, loc: 0, pos: 70, scl: 0.500]) [buf: 100] 
   (at 80, cp->sounds[1][80:89, 1.000, loc: 0, pos: 80, scl: 0.500]) [buf: 100] 
   (at 90, cp->sounds[1][90:99, 1.000, loc: 0, pos: 90, scl: 0.500]) [buf: 100] 
   (at 100, end_mark)
"))
	      (snd-display ";xen 4: ~A" (display-edits ind 0 12 #f)))
	  (if (not (vequal (channel->vct 0 10) (vct 0.000 0.005 0.010 0.015 0.020 0.025 0.030 0.035 0.040 0.045)))
	      (snd-display ";xen 4: ~A" (channel->vct)))
	  (close-sound ind))
	
	(let ((vals (channel->vct)))
	  (undo 4)
	  (ptree-channel (lambda (y) y))
	  (ramp-channel 0.0 1.0 10 30)
	  (ptree-channel (lambda (y) y))
	  (ramp-channel 0.0 1.0 50 20)
	  (ptree-channel (lambda (y) y))
	  (ramp-channel 0.0 1.0 20 15)
	  (ptree-channel (lambda (y) y))
	  (ramp-channel 0.0 1.0 30 30)
	  (if (not (vequal vals (channel->vct)))
	      (snd-display ";ramp3 opt vs unopt: ~A ~A" vals (channel->vct))))
	
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(xen-channel (lambda (y data forward)
		       (+ y (list-ref data 0)))
		     0 (frames) ind 0 #f #f
		     (lambda (pos dur)
		       (list 0.1)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (xen[0] 0 11) ; xen 0 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xen-ramp3 0: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.100 0.101 0.108 0.127 0.164 0.225 0.316 0.443 0.612 0.829 1.100)))
	    (snd-display ";xen-ramp3 1: ~A" (channel->vct)))
	(scale-channel 0.5)
	(if (not (string-=? (display-edits ind 0 6 #f) "
 (scale 0 11) ; scale-channel 0.500 0 11 [6:2]:
   (at 0, cp->sounds[1][0:10, 0.500, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xen-ramp3 2: ~A" (display-edits ind 0 6 #f)))
	
	(undo 5)
	(ramp-channel 0 1)
	(ramp-channel 0 1 5 5)
	(ramp-channel 0 1 7 3)
	(xen-channel (lambda (y data forward)
		       (+ y (list-ref data 0)))
		     0 (frames) ind 0 #f #f
		     (lambda (pos dur)
		       (list 0.1)))
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (xen[0] 0 11) ; xen 0 0 11 [5:5]:
   (at 0, cp->sounds[1][0:4, 1.000, [1]-0.000 -> 0.400, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 5, cp->sounds[1][5:6, 1.000, [1]0.500 -> 0.600, [2]0.000 -> 0.250, loc: 0, pos: 5, scl: 1.000]) [buf: 11] 
   (at 7, cp->sounds[1][7:9, 1.000, [1]0.700 -> 0.900, [2]0.500 -> 1.000, [3]0.000 -> 1.000, loc: 0, pos: 7, scl: 1.000]) [buf: 11] 
   (at 10, cp->sounds[1][10:10, 1.000, [1]1.000 -> 1.000, loc: 0, pos: 10, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xen-ramp3 3: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.100 0.200 0.300 0.400 0.500 0.100 0.250 0.100 0.400 1.000 1.100)))
	    (snd-display ";xen-ramp3 4: ~A" (channel->vct)))
	(close-sound ind))
      
      ;; various cases not optimized, presumably
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	;; ramp+xramp (now optimized)
	(ramp-channel 0.0 1.0)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp+xramp: ~A" (display-edits ind 0 3)))
	(undo 2)
	
	;; xramp+xramp -- this one now optimized
	(xramp-channel 0.0 1.0 0.32)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string=? (display-edits ind 0 3) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [3]0.000 -> 3.466, [4]0.000 -> -1.139, off: 1.471, scl: -1.471, off2: -0.032, scl2: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp+xramp: ~A" (display-edits ind 0 3)))
	(undo 2)
	
	;; xramp+xramp+xramp
	(xramp-channel 0.0 1.0 0.32)
	(xramp-channel 0.0 1.0 32.0)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string=? (display-edits ind 0 4) "
 (set 0 11) ; xramp-channel [4:2]:
   (at 0, cp->sounds[2][0:10, 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp+xramp+xramp: ~A" (display-edits ind 0 4)))
	(undo 3)
	
	;; xramp+xramp+ramp (now optimized)
	(xramp-channel 0.0 1.0 0.32)
	(xramp-channel 0.0 1.0 32.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [3]0.000 -> 3.466, [4]0.000 -> -1.139, off: 1.471, scl: -1.471, off2: -0.032, scl2: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp+xramp+ramp: ~A" (display-edits ind 0 4)))
	(undo 3)
	
	;; xramp+ramp (now optimized)
	(xramp-channel 0.0 1.0 32.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp+ramp: ~A" (display-edits ind 0 3)))
	(undo 2)
	
	;; ramp+ramp+xramp
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string-=? (display-edits ind 0 4) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp+ramp+xramp: ~A" (display-edits ind 0 4)))
	(undo 3)
	
	;; ramp+ramp+ramp+ramp
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 5) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, [4]-0.000 -> 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp+ramp+ramp+ramp: ~A" (display-edits ind 0 5)))
	(undo 4)
	
	;; ramp+ramp+ramp+xramp
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(xramp-channel 0.0 1.0 32.0)
	(if (not (string-=? (display-edits ind 0 5) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp+ramp+ramp+xramp: ~A" (display-edits ind 0 5)))
	(undo 4)
	
	;; ptree+ramp (now optimized)
	(ptree-channel (lambda (y) y))
	(ramp-channel 0.0 1.0)
	(if (not (string-=? (display-edits ind 0 3) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) y)]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree+ramp: ~A" (display-edits ind 0 3)))
	(undo 2)
	
	;; ramp+xramp+ptree (now optimized)
	(ramp-channel 0.0 1.0)
	(xramp-channel 0.0 1.0 32.0)
	(ptree-channel (lambda (y) y))
	(if (not (string-=? (display-edits ind 0 4) "
 (ptree[0] 0 11) ; ptree 0 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) y)]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp+xramp+ptree: ~A" (display-edits ind 0 4)))
	(undo 3)
	
	;; xramp+ramp+ptree (now optimized)
	(xramp-channel 0.0 1.0 32.0)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y) y))
	(if (not (string-=? (display-edits ind 0 4) "
 (ptree[0] 0 11) ; ptree 0 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) y)]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp+ramp+ptree: ~A" (display-edits ind 0 4)))
	(undo 3)
	
	;; ramp3+ptree (now optimized)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ptree-channel (lambda (y) y))
	(if (not (string-=? (display-edits ind 0 5) "
 (ptree[0] 0 11) ; ptree 0 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, [3]-0.000 -> 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) y)]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp3+ptree: ~A" (display-edits ind 0 5)))
	(undo 4)
	
	;; ptree+ptree (now optimized)
	(ptree-channel (lambda (y) y))
	(ptree-channel (lambda (y) y))
	(if (not (string=? (display-edits ind 0 3) "
 (ptree[1] 0 11) ; ptree 1 0 11 [3:2]:
   (at 0, cp->sounds[1][0:10, 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000, code: (lambda (y) y)]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ptree+ptree: ~A" (display-edits ind 0 3)))
	(undo 2)
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd")))
	(map-chan (lambda (y) 1.0) 0 10)
	
	;; ramp ptree2 cases
	(ptree-channel (lambda (y) (* y 0.5)))
	(ptree-channel (lambda (y) (* y 2.0)))
	(ramp-channel 0 1)
	(if (not (string-=? (display-edits ind 0 4 #f) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp ptree2: ~A" (display-edits ind 0 4 #f)))
	(if (not (vequal (channel->vct) (vct 0.0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0)))
	    (snd-display ";ramp ptree2: ~A" (channel->vct)))
	(ramp-channel 0 1)
	(if (not (string-=? (display-edits ind 0 5 #f) "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [5:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [1]-0.000 -> 1.000, [2]-0.000 -> 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";ramp2 ptree2: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 1.000)))
	    (snd-display ";ramp2 ptree2: ~A" (channel->vct)))
	(undo 2)
	(xramp-channel 0 1 32)
	(if (not (string=? (display-edits ind 0 4 #f) "
 (ramp 0 11) ; xramp-channel 0.000 1.000 32.000 0 11 [4:2]:
   (at 0, cp->sounds[1][0:10, 1.000, [4]0.000 -> 3.466, off: -0.032, scl: 0.032, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 1.000]) [buf: 11] 
   (at 11, end_mark)
"))
	    (snd-display ";xramp ptree2: ~A" (display-edits ind 0 4 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.013 0.032 0.059 0.097 0.150 0.226 0.333 0.484 0.698 1.000)))
	    (snd-display ";xramp ptree2: ~A" (channel->vct)))
	
	(undo 3)
	(scale-channel 0.0)
	
	(ptree-channel (lambda (y) (+ y 0.5)))
	(ptree-channel (lambda (y) (* y 2.0)))
	(ramp-channel 0 1)
	(if (not (string=? (display-edits ind 0 4 #f) (string-append "
 (ptree[1] 0 11) ; ptree 1 0 11 [4:2]:
   (at 0, cp->sounds[0][0:10, 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	    (snd-display ";ramp ptree2 zero: ~A" (display-edits ind 0 4 #f)))
	(if (not (vequal (channel->vct) (vct 0.0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0)))
	    (snd-display ";ramp ptree2 zero: ~A" (channel->vct)))
	(ramp-channel 0 1)
	(if (not (string-=? (display-edits ind 0 5 #f) (string-append "
 (ramp 0 11) ; ramp-channel 0.000 1.000 0 11 [5:2]:
   (at 0, cp->sounds[0][0:10, 1.000, [1]-0.000 -> 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	    (snd-display ";ramp2 ptree2 zero: ~A" (display-edits ind 0 5 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 1.000)))
	    (snd-display ";ramp2 ptree2 zero: ~A" (channel->vct)))
	(undo 2)
	(xramp-channel 0 1 32)
	(if (not (string=? (display-edits ind 0 4 #f) (string-append "
 (ptree[1] 0 11) ; ptree 1 0 11 [4:2]:
   (at 0, cp->sounds[0][0:10, 1.000, loc2: 1, pos2: 0, scl2: 1.000, loc: 0, pos: 0, scl: 0.000]) [file: " (getcwd) "/test.snd[0]]
   (at 11, end_mark)
")))
	    (snd-display ";xramp ptree2 zero: ~A" (display-edits ind 0 4 #f)))
	(if (not (vequal (channel->vct) (vct 0.000 0.013 0.032 0.059 0.097 0.150 0.226 0.333 0.484 0.698 1.000)))
	    (snd-display ";xramp ptree2 zero: ~A" (channel->vct)))
	
	(close-sound ind))
      
      (let ((ind (new-sound :file "test.snd" :header-type mus-next :data-format mus-bfloat :srate 22050 :channels 1 :comment "xen tests" :size 10)))
	(do ((ramps 1 (1+ ramps)))
	    ((= ramps 5))
	  (revert-sound)
	  (map-chan (lambda (y) 1.0))
	  (do ((i 0 (1+ i)))
	      ((= i ramps))
	    (ramp-channel 0.0 1.0))
	  (let ((old-vals (channel->vct)))
	    (xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.25)))
	    (let ((new-vals (channel->vct)))
	      (if (not (vequal (vct-offset! old-vals .25) new-vals))
		  (snd-display ";xen-ramp~A: i ~A ~A" old-vals new-vals)))))
	(do ((ramps 1 (1+ ramps)))
	    ((= ramps 5))
	  (revert-sound)
	  (xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.75)))
	  (do ((i 0 (1+ i)))
	      ((= i ramps))
	    (ramp-channel 0.0 1.0))
	  (let ((old-vals (channel->vct)))
	    (revert-sound)
	    (map-chan (lambda (y) 0.5))
	    (xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.25)))
	    (do ((i 0 (1+ i)))
		((= i ramps))
	      (ramp-channel 0.0 1.0))
	    (let ((new-vals (channel->vct)))
	      (if (not (vequal old-vals new-vals))
		  (snd-display ";ramp~A-xen(zero): i ~A ~A" old-vals new-vals)))))
	(revert-sound)
	(map-chan (lambda (y) 1.0))
	(ramp-channel 0.0 1.0)
	(xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.25)))	    
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(ramp-channel 0.0 1.0)
	(let ((old-vals (channel->vct)))
	  (revert-sound)
	  (map-chan (lambda (y) 1.0))
	  (ramp-channel 0.0 1.0)
	  (ptree-channel (lambda (y) (+ y .25)))
	  (ramp-channel 0.0 1.0)
	  (ramp-channel 0.0 1.0)
	  (ramp-channel 0.0 1.0)
	  (let ((new-vals (channel->vct)))
	    (let ((new-vals (channel->vct)))
	      (if (not (vequal old-vals new-vals))
		  (snd-display ";ramp3-xen-ramp: ~A ~A" old-vals new-vals)))))
	(close-sound ind))

      (let ((data (make-vct 101 1.0))
	    (rto1-data (make-vct 101))
	    (xto1-data (make-vct 101))
	    (cos-data (make-vct 101))
	    (ind (new-sound "test.snd")))
	;; test-ops.scm for 7 and 8 cases (40 min per branch)
	
	(define (set-to-1) (map-chan (lambda (y) 1.0) 0 100))
	(define (cset-to-1 dat) (do ((i 0 (1+ i))) ((= i 101)) (vct-set! dat i 1.0)))
	(define (ramp-to-1) (ramp-channel 0.0 1.0))
	(define (cramp-to-1 dat) (vct-multiply! dat rto1-data))
	(define (scale-by-half) (scale-channel 0.5))
	(define (cscale-by-half dat) (vct-scale! dat 0.5000))
	(define (scale-by-two) (scale-channel 2.0 30 40))
	(define (cscale-by-two dat) (do ((i 30 (1+ i))) ((= i 70)) (vct-set! dat i (* (vct-ref dat i) 2.0))))
	(define (xramp-to-1) (xramp-channel 0.0 1.0 32.0))
	(define (cxramp-to-1 dat) (vct-multiply! dat xto1-data))
	(define (scale-mid) (scale-channel 0.125 30 30))
	(define (cscale-mid dat) (do ((i 30 (1+ i))) ((= i 60)) (vct-set! dat i (* (vct-ref dat i) 0.125))))
	(define (on-air) (scale-channel 0.0 10 30))
	(define (con-air dat) (do ((i 10 (1+ i))) ((= i 40)) (vct-set! dat i 0.0)))
	(define (ptree) (ptree-channel (lambda (y) (* y 0.75)) 20 20))
	(define (cptree dat) (do ((i 20 (1+ i))) ((= i 40)) (vct-set! dat i (* (vct-ref dat i) .75))))
	(define (ptreec) (cosine-channel-via-ptree))
	(define (cptreec dat) (vct-multiply! dat cos-data))
	(define (xen) (xen-channel (lambda (y data forward) (* y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.25))))
	(define (cxen dat) (vct-scale! dat 0.25))
	(define (rev-channel->vct)
	  (let* ((l (vct-length data))
		 (r (make-sample-reader (1- l) ind 0 -1))
		 (d (make-vct l)))
	    (do ((i (1- l) (1- i)))
		((< i 0))
	      (vct-set! d i (r)))
	    d))
	(define (ptreec1)
	  (ptree-channel (lambda (y data forward)
			   (declare (y real) (data vct) (forward boolean))
			   (* y (vct-ref data 0)))
			 10 50 ind 0 #f #f
			 (lambda (pos dur)
			   (vct 0.625))))
	(define (cptreec1 dat) (do ((i 10 (1+ i))) ((= i 60)) (vct-set! dat i (* (vct-ref dat i) 0.625))))
	
	(let ((xe (make-env '(0 0 1 1) :end 100 :base 32.0)))
	  (do ((i 0 (1+ i))
	       (incr (/ pi 101.0))
	       (ang (* -0.5 pi)))
	      ((= i 101))
	    (vct-set! rto1-data i (* i .01))
	    (vct-set! xto1-data i (env xe))
	    (vct-set! cos-data i (cos ang))
	    (set! ang (+ ang incr))))
	
	(set! (squelch-update ind) #t)
	
	;; 0 case
	(set-to-1)
	(if (not (vvequal data (channel->vct)))
	    (snd-display ";0 case! ~A" (channel->vct)))
	(if (not (vvequal data (rev-channel->vct)))
	    (snd-display ";0 case rev! ~A" (rev-channel->vct)))
	
	;; 1 case
	(for-each
	 (lambda (func check)
	   (revert-sound)
	   (set-to-1)
	   (cset-to-1 data)
	   (func)
	   (check data)
	   (if (not (vvequal data (channel->vct)))
	       (snd-display ";1 case: ~A ~A" (procedure-name func) (channel->vct)))
	   (if (not (vvequal data (rev-channel->vct)))
	       (snd-display ";1 rev case: ~A ~A" (procedure-name func) (rev-channel->vct))))
	 (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	 (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen))
	
	;; 2 case
	(for-each
	 (lambda (func check)
	   (for-each
	    (lambda (func1 check1)
	      (revert-sound)
	      (set-to-1)
	      (cset-to-1 data)
	      (func)
	      (check data)
	      (func1)
	      (check1 data)
	      (if (not (vvequal data (channel->vct)))
		  (snd-display ";2 case: ~A(~A): ~A" (procedure-name func1) (procedure-name func) (channel->vct)))
	      (if (not (vvequal data (rev-channel->vct)))
		  (snd-display ";2 rev case: ~A(~A): ~A" (procedure-name func1) (procedure-name func) (rev-channel->vct))))
	    (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	    (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
	 (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	 (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen))
	
	;; 3 case
	(for-each
	 (lambda (func check)
	   (for-each
	    (lambda (func1 check1)
	      (for-each
	       (lambda (func2 check2)
		 (revert-sound)
		 (set-to-1)
		 (cset-to-1 data)
		 (func)
		 (check data)
		 (func1)
		 (check1 data)
		 (func2)
		 (check2 data)
		 (if (not (vvequal data (channel->vct)))
		     (snd-display ";3 case: ~A(~A(~A)): ~A" (procedure-name func2) (procedure-name func1) (procedure-name func) (channel->vct)))
		 (if (not (vvequal data (rev-channel->vct)))
		     (snd-display ";3 rev case: ~A(~A(~A)): ~A" (procedure-name func2) (procedure-name func1) (procedure-name func) (rev-channel->vct))))
	       (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	       (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
	    (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	    (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
	 (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	 (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen))
	
	(if all-args
	    (begin
	      ;; 4 case
	      (for-each
	       (lambda (func check)
		 (for-each
		  (lambda (func1 check1)
		    (for-each
		     (lambda (func2 check2)
		       (for-each
			(lambda (func3 check3)
			  (revert-sound)
			  (set-to-1)
			  (cset-to-1 data)
			  (func)
			  (check data)
			  (func1)
			  (check1 data)
			  (func2)
			  (check2 data)
			  (func3)
			  (check3 data)
			  (if (not (vvequal data (channel->vct)))
			      (snd-display ";4 case: ~A(~A(~A(~A))): ~A" 
					   (procedure-name func3) (procedure-name func2) (procedure-name func1) (procedure-name func) 
					   (channel->vct)))
			  (if (not (vvequal data (rev-channel->vct)))
			      (snd-display ";4 rev case: ~A(~A(~A(~A))): ~A" 
					   (procedure-name func3) (procedure-name func2) (procedure-name func1) (procedure-name func) 
					   (rev-channel->vct))))
			(list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
			(list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
		     (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
		     (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
		  (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
		  (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
	       (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
	       (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen))
	      
	      ;; 5 case
	      (for-each
	       (lambda (func check)
		 (for-each
		  (lambda (func1 check1)
		    (for-each
		     (lambda (func2 check2)
		       (for-each
			(lambda (func3 check3)
			  (for-each
			   (lambda (func4 check4)
			     (revert-sound)
			     (set-to-1)
			     (cset-to-1 data)
			     (func)
			     (check data)
			     (func1)
			     (check1 data)
			     (func2)
			     (check2 data)
			     (func3)
			     (check3 data)
			     (func4)
			     (check4 data)
			     (if (not (vvequal data (channel->vct)))
				 (snd-display ";5 case: ~A(~A(~A(~A(~A)))): ~A" 
					      (procedure-name func4) (procedure-name func3) (procedure-name func2) 
					      (procedure-name func1) (procedure-name func) 
					      (channel->vct))))
			   (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
			   (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
			(list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
			(list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
		     (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
		     (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
		  (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1)
		  (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1)))
	       (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1)
	       (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1))
	      
	      ;; 6 case
	      (for-each
	       (lambda (func check)
		 (for-each
		  (lambda (func1 check1)
		    (for-each
		     (lambda (func2 check2)
		       (for-each
			(lambda (func3 check3)
			  (for-each
			   (lambda (func4 check4)
			     (for-each
			      (lambda (func5 check5)
				(revert-sound)
				(set-to-1)
				(cset-to-1 data)
				(func)
				(check data)
				(func1)
				(check1 data)
				(func2)
				(check2 data)
				(func3)
				(check3 data)
				(func4)
				(check4 data)
				(func5)
				(check5 data)
				(if (not (vvequal data (channel->vct)))
				    (snd-display ";6 case: ~A(~A(~A(~A(~A(~A))))): ~A" 
						 (procedure-name func5) (procedure-name func4) (procedure-name func3) 
						 (procedure-name func2) (procedure-name func1) (procedure-name func) 
						 (channel->vct))))
			      (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
			      (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
			   (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
			   (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
			(list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1 xen)
			(list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1 cxen)))
		     (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air ptree ptreec ptreec1)
		     (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air cptree cptreec cptreec1)))
		  (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air)
		  (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air)))
	       (list scale-by-two ramp-to-1 xramp-to-1 scale-by-half scale-mid on-air)
	       (list cscale-by-two cramp-to-1 cxramp-to-1 cscale-by-half cscale-mid con-air))
	      
	      ))
	(close-sound ind))
      
      (set! (optimization) old-opt-val)
      
      (let ((ind (open-sound "oboe.snd")))
	(set! (cursor) 1000)
	(delete-sample 321)
	(if (not (= (cursor) 999)) (snd-display ";delete-sample before cursor: ~A" (cursor)))
	(if (not (= (cursor ind 0 0) 1000)) (snd-display ";delete-sample before cursor (0): ~A" (cursor ind 0 0)))
	(undo)
	(if (not (= (cursor) 1000)) (snd-display ";delete-sample after cursor undo: ~A" (cursor)))
	(undo -1)
	(if (not (= (cursor) 999)) (snd-display ";delete-sample before cursor redo: ~A" (cursor)))
	(redo -1)
	(delete-sample 1321)
	(if (not (= (cursor) 1000)) (snd-display ";delete-sample after cursor: ~A" (cursor)))
	(undo)
	(delete-samples 0 100)
	(if (not (= (cursor) 900)) (snd-display ";delete-samples before cursor: ~A" (cursor)))
	(undo)
	(delete-samples 1100 100)
	(if (not (= (cursor) 1000)) (snd-display ";delete-samples after cursor: ~A" (cursor)))
	(undo)
	(insert-samples 100 100 (make-vct 100))
	(if (not (= (cursor) 1100)) (snd-display ";insert-samples before cursor: ~A" (cursor)))
	(undo)
	(insert-samples 1100 100 (make-vct 100))
	(if (not (= (cursor) 1000)) (snd-display ";insert-samples after cursor: ~A" (cursor)))
	(undo)
	(set! (samples 0 100) (make-vct 100))
	(if (not (= (cursor) 1000)) (snd-display ";set-samples cursor: ~A" (cursor)))
	(set! (show-axes ind 0) show-x-axis-unlabelled)
	(update-time-graph)
	(set! (show-axes ind 0) show-all-axes-unlabelled)
	(update-time-graph)
	(close-sound ind))
      
      (let* ((index (open-sound "oboe.snd"))
	     (bnds (x-bounds index))
	     (xp (x-position-slider))
	     (yp (y-position-slider))
	     (xz (x-zoom-slider))
	     (yz (y-zoom-slider)))
	(if (not (string=? (snd-completion " open-so") " open-sound"))
	    (snd-display ";completion: ~A" (snd-completion " open-so")))
	(if (not (string=? (snd-completion " open-sound") " open-sound"))
	    (snd-display ";completion: ~A" (snd-completion " open-so")))
	(if (not (string=? (snd-completion " zoom-focus-r") " zoom-focus-right"))
	    (snd-display ";completion: ~A" (snd-completion " zoom-focus-r")))
	(play-and-wait "oboe.snd")
	(play-and-wait "oboe.snd" 12000)
	(play-and-wait "oboe.snd" 12000 15000)
	(play-and-wait 0 #f #f #f #f (1- (edit-position)))
	(let ((old-speed (speed-control index))
	      (old-style (speed-control-style))
	      (old-open (show-controls index)))
	  (set! (show-controls index) #t)
	  (set! (speed-control index) -2.0)
	  (play-and-wait 12345 index)
	  (set! (speed-control-style) speed-control-as-semitone)
	  (set! (speed-control index) 0.5)
	  (set! (speed-control-style) speed-control-as-ratio)
	  (set! (speed-control index) 0.25)
	  (set! (speed-control index) old-speed)
	  (set! (speed-control-style) old-style)
	  (set! (show-controls index) old-open))
	(bomb index #t)
	(let ((k (disk-kspace "oboe.snd")))
	  (if (or (not (number? k))
		  (<= k 0))
	      (snd-display ";disk-kspace = ~A" (disk-kspace "oboe.snd")))
	  (set! k (disk-kspace "/baddy/hiho"))
	  (if (not (= k -1))
	      (snd-display ";disk-kspace of bogus file = ~A" (disk-kspace "/baddy/hiho"))))
	(if (not (= (transform-samples-size) 0)) (snd-display ";transform-samples-size ~A?" (transform-samples-size)))
	(set! (transform-graph?) #t)
	(let ((pk (fft-peak index 0 1.0)))
	  (if (not pk) (snd-display ";fft-peak? ")))
	(set! (time-graph?) #t)
	(if (not (string=? (x-axis-label) "time")) (snd-display ";def time x-axis-label: ~A" (x-axis-label)))
	(set! (x-axis-label index 0 time-graph) "no time")
	(if (not (string=? (x-axis-label) "no time")) (snd-display ";time x-axis-label: ~A" (x-axis-label index 0 time-graph)))
	(graph '(0 0 1 1 2 0) "lisp")
	(update-lisp-graph)
	(if (not (string=? (x-axis-label index 0 lisp-graph) "lisp")) (snd-display ";def lisp x-axis-label: ~A" (x-axis-label index 0 lisp-graph)))
	(set! (x-axis-label index 0 lisp-graph) "no lisp")
	(if (not (string=? (x-axis-label index 0 lisp-graph) "no lisp")) (snd-display ";lisp x-axis-label: ~A" (x-axis-label index 0 lisp-graph)))
	(graph-data (make-vct 4))
	(update-lisp-graph)
	(graph '#(0 0 1 1 2 0))
	(do ((i 0 (1+ i))) 
	    ((= i 32)) 
	  (graph '#(0 1 2)) 
	  (graph (list '#(0 1 2) '#(3 2 1) '#(1 2 3)))
	  (graph (list '#(0 1 2) '#(3 2 1))))
	(set! (x-bounds) (list 0.0 0.01))
	(let ((data (make-graph-data)))
	  (if (vct? data)
	      (let ((mid (inexact->exact (round (* .5 (vct-length data))))))
		(if (not (= (vct-length data) (1+ (- (right-sample) (left-sample)))))
		    (snd-display ";make-graph-data bounds: ~A ~A -> ~A" (left-sample) (right-sample) (vct-length data)))
		(if (fneq (vct-ref data mid)
			  (sample (+ (left-sample) mid)))
		    (snd-display ";make-graph-data[~D]: ~A ~A" mid (vct-ref data mid) (sample (+ (left-sample) mid)))))))
	(let ((data (make-graph-data index 0 0 100 199)))
	  (if (vct? data)
	      (begin
		(if (not (= (vct-length data) 100))
		    (snd-display ";make-graph-data 100:199: ~A" (vct-length data)))
		(if (fneq (vct-ref data 50) (sample 50))
		    (snd-display ";make-graph-data 50: ~A ~A" (vct-ref data 50) (sample 50))))))
	(set! (x-bounds) (list 0.0 0.1))
	(update-transform-graph)
	(if (not (string=? (x-axis-label index 0 transform-graph) "frequency")) (snd-display ";def fft x-axis-label: ~A" (x-axis-label index 0 transform-graph)))
	(set! (x-axis-label index 0 transform-graph) "fourier")
	(if (not (string=? (x-axis-label index 0 transform-graph) "fourier")) (snd-display ";fft x-axis-label: ~A" (x-axis-label index 0 transform-graph)))
	(if (and (number? (transform-samples-size))
		 (= (transform-samples-size) 0))
	    (snd-display ";transform-graph? transform-samples-size ~A?" (transform-samples-size)))
	(update-transform-graph)
	(peaks "tmp.peaks")
	(if (defined? 'read-line)
	    (let ((p (open-input-file "tmp.peaks")))
	      (if (not p)
		  (snd-display ";peaks->tmp.peaks failed?")
		  (let ((line (read-line p)))
		    (if (or (not (string? line))
			    (not (string=? "Snd: fft peaks" (substring line 0 14))))
			(snd-display ";peaks 1: ~A?" line))
		    (set! line (read-line p))
		    (set! line (read-line p))
		    (if (or (not (string? line))
			    (not (string=? "oboe.snd, fft 512 points beginning at sample 0 (0.000 secs), blackman2-window" line)))
			(snd-display ";peaks 2: ~A?" line))
		    (set! line (read-line p))
		    (set! line (read-line p))
		    (if (or (not (string? line))
			    (and (not (string=? "  86.132812  1.00000" line))
				 (not (string=? "  0.000000  1.00000" line)))) ; fht/fft disagreement about 0/1 (groan)
			(snd-display ";peaks 3: ~A?" line))
		    (close-port p)))))
	(delete-file "tmp.peaks")
	(peaks)
	(if (and (provided? 'xm) 
		 (or (not (list-ref (dialog-widgets) 20))
		     (not (XtIsManaged (list-ref (dialog-widgets) 20)))))
	    (snd-display ";peaks but no help?"))
	(dismiss-all-dialogs)
	(let* ((num-transforms 7)
	       (num-transform-graph-types 3))
	  (set! (transform-graph? index 0) #t)
	  (set! (transform-size index 0) 64)
	  (do ((i 0 (1+ i)))
	      ((= i num-transforms))
	    (set! (transform-type) i)
	    (do ((j 0 (1+ j)))
		((= j num-transform-graph-types))
	      (set! (transform-graph-type index 0) j)
	      (update-transform-graph index 0))))
	(set! (transform-type) fourier-transform)
	
	(if (read-only index) (snd-display ";read-only open-sound: ~A?" (read-only index)))
	(set! (read-only index) #t)
	(if (not (read-only index)) (snd-display ";set-read-only: ~A?" (read-only index)))
	(bind-key (char->integer #\a) 0 (lambda () (set! a-ctr 3)))
	(key (char->integer #\a) 0) 
	(if (not (= a-ctr 3)) (snd-display ";bind-key: ~A?" a-ctr))
	(let ((str (with-output-to-string (lambda () (display (key-binding (char->integer #\a) 0))))))
	  (if (not (string=? str "#<procedure #f (() (set! a-ctr 3))>"))
	      (snd-display ";key-binding: ~A?" str)))
	(unbind-key (char->integer #\a) 0)
	(set! a-ctr 0)
	(key (char->integer #\a) 0) 
	(do ((i 0 (1+ i)))
	    ((= i 5))
	  (let ((psf (eps-file)))
	    (if (and psf (string? psf))
		(begin
		  (if (file-exists? psf) (delete-file psf))
		  (set! (graph-style) i)
		  (graph->ps)
		  (if (not (file-exists? psf)) 
		      (snd-display ";graph->ps: ~A?" psf)
		      (delete-file psf))))))
	(let ((err (catch 'cannot-print 
			  (lambda () 
			    (graph->ps "/bad/bad.eps"))
			  (lambda args 12345))))
	  (if (not (= err 12345)) (snd-display ";graph->ps err: ~A?" err)))
	(let ((n2 (or (open-sound "2.snd") (open-sound "4.aiff"))))
	  (set! (transform-graph? n2) #t)
	  (set! (channel-style n2) channels-superimposed)
	  (if (not (= (channel-style n2) channels-superimposed)) (snd-display ";channel-style->~D: ~A?" channels-superimposed (channel-style n2)))
	  (graph->ps "aaa.eps")
	  (set! (channel-style n2) channels-combined)
	  (if (not (= (channel-style n2) channels-combined)) (snd-display ";channel-style->~D: ~A?" channels-combined (channel-style n2)))
	  (graph->ps "aaa.eps")
	  (set! (channel-style n2) channels-separate)
	  (if (not (= (channel-style n2) channels-separate)) (snd-display ";channel-style->~D: ~A?" channels-separate (channel-style n2)))
	  (graph->ps "aaa.eps")
	  (close-sound n2))
	(if (= (channels index) 1)
	    (begin
	      (set! (channel-style index) channels-superimposed)
	      (if (not (= (channel-style index) channels-separate)) (snd-display ";channel-style[0]->~D: ~A?" channels-separate (channel-style index)))))
	(set! (sync index) 32)
	(if (not (= (sync index) 32)) (snd-display ";sync->32: ~A?" (sync index)))
	(set! (sync index) 0)
	(set! (channel-sync index 0) 12)
	(if (not (= (channel-sync index 0) 12)) (snd-display ";sync-chn->12: ~A?" (channel-sync index 0)))
	(set! (channel-sync index 0) 0)
	(if (not (= a-ctr 0)) (snd-display ";unbind-key: ~A?" a-ctr))
	(if (fneq xp 0.0) (snd-display ";x-position-slider: ~A?" xp))
	(if (fneq yp 0.0) (snd-display ";y-position-slider: ~A?" yp))
	(if (fneq xz 0.04338) (snd-display ";x-zoom-slider: ~A?" xz))
	(if (fneq yz 1.0) (snd-display ";y-zoom-slider: ~A?" yz))
	(if (or (fneq (car bnds) 0.0) (fneq (cadr bnds) 0.1)) (snd-display ";x-bounds: ~A?" bnds))
	(if (not (= (find-sound "oboe.snd") index)) (snd-display ";oboe: index ~D /= ~D?" (find-sound "oboe.snd") index))
	(if (not (sound? index)) (snd-display ";oboe: ~D not ok?" index))
	(if (not (= (chans index) 1)) (snd-display ";oboe: chans ~D?" (chans index)))
	(if (not (= (channels index) 1)) (snd-display ";oboe: channels ~D?" (channels index)))
	(if (not (= (frames index) 50828)) (snd-display ";oboe: frames ~D?" (frames index)))
	(if (not (= (srate index) 22050)) (snd-display ";oboe: srate ~D?" (srate index)))
	(if (not (= (data-location index) 28)) (snd-display ";oboe: location ~D?" (data-location index)))
	(if (not (= (data-size index) (* 50828 2))) (snd-display ";oboe: size ~D?" (data-size index)))
	(if (not (= (data-format index) mus-bshort)) (snd-display ";oboe: format ~A?" (data-format index)))
	(if (fneq (maxamp index) .14724) (snd-display ";oboe: maxamp ~F?" (maxamp index)))
	(if (comment index) (snd-display ";oboe: comment ~A?" (comment index)))
	(if (not (= (string-length "asdf") 4)) (snd-display ";string-length: ~A?" (string-length "asdf")))
	(if (not (string=? (short-file-name index) "oboe.snd")) (snd-display ";oboe short name: ~S?" (short-file-name index)))
	(let ((matches (count-matches (lambda (a) (> a .125)))))
	  (if (not (= matches 1313)) (snd-display ";count-matches: ~A?" matches)))
	(let ((spot (find (lambda (a) (> a .13)))))
	  (if (or (null? spot) (not (= (cadr spot) 8862))) (snd-display ";find: ~A?" spot)))
	(set! (right-sample) 3000) 
	(let ((samp (right-sample)))
	  (if (> (abs (- samp 3000)) 1) (snd-display ";right-sample: ~A?" samp)))
	(set! (left-sample) 1000) 
	(let ((samp (left-sample)))
	  (if (> (abs (- samp 1000)) 1) (snd-display ";left-sample: ~A?" samp)))
	(let ((eds (edits)))
	  (if (or (not (= (car eds) 0)) (not (= (cadr eds) 0)))
	      (snd-display ";edits: ~A?" eds))
	  (if (not (= (edit-position) (car eds)))
	      (snd-display ";edit-position: ~A ~A?" (edit-position) eds)))
	(play-and-wait 0 index 0)
	
	(bomb index #f)
	(select-all index 0) 
	(let ((r0 (car (regions))))
	  (if (not (selection?)) (snd-display ";selection?"))
	  (if (not (region? r0)) (snd-display ";region?"))
	  (if (not (= (selection-chans) 1)) (snd-display ";selection-chans(1): ~A" (selection-chans)))
	  (if (not (= (selection-srate) (srate index))) (snd-display ";selection-srate: ~A ~A" (selection-srate) (srate index)))
	  (if (fneq (region-maxamp r0) (maxamp index)) (snd-display ";region-maxamp (1): ~A?" (region-maxamp r0)))
	  (if (fneq (selection-maxamp index 0) (maxamp index)) (snd-display ";selection-maxamp (1): ~A?" (selection-maxamp index 0)))
	  (save-region r0 "temp.dat")
	  (if (file-exists? "temp.dat")
	      (delete-file "temp.dat")
	      (snd-display ";save-region file disappeared?"))
	  (play-region r0 #t) ;needs to be #t here or it never gets run
	  (if (not (= (length (regions)) 2)) (snd-display ";regions: ~A?" (regions)))
	  (if (not (selection-member? index)) (snd-display ";selection-member?: ~A" (selection-member? index)))
	  (if (not (= (region-srate r0) 22050)) (snd-display ";region-srate: ~A?" (region-srate r0)))
	  (if (not (= (region-chans r0) 1)) (snd-display ";region-chans: ~A?" (region-chans r0)))
	  (if (not (= (region-frames r0) 50828)) (snd-display ";region-frames: ~A?" (region-frames r0)))
	  (if (not (= (selection-frames) 50828)) (snd-display ";selection-frames: ~A?" (selection-frames 0)))
	  (if (not (= (selection-position) 0)) (snd-display ";selection-position: ~A?" (selection-position)))
	  (if (not (= (region-position r0 0) 0)) (snd-display ";region-position: ~A?" (region-position r0 0)))
	  (if (fneq (region-maxamp r0) (maxamp index)) (snd-display ";region-maxamp: ~A?" (region-maxamp r0)))
	  (if (fneq (selection-maxamp index 0) (maxamp index)) (snd-display ";selection-maxamp: ~A?" (selection-maxamp index 0)))
	  (let ((samps1 (samples->vct 0 50827 index 0))
		(samps2 (region-samples->vct 0 50828 r0 0))
		(vr (make-sample-reader 0 index 0 1)))
	    (if (not (sample-reader? vr)) (snd-display ";~A not sample-reader?" vr))
	    (if (not (= (sample-reader-position vr) 0)) (snd-display ";initial sample-reader-position: ~A" (sample-reader-position vr)))
	    (if (not (equal? (sample-reader-home vr) (list index 0))) 
		(snd-display ";sample-reader-home: ~A ~A?" (sample-reader-home vr) (list index 0)))
	    (if (sample-reader-at-end? vr) (snd-display ";~A init at end?" vr))
	    (let ((err (catch #t
			      (lambda ()
				(region-samples->vct -1 1233 r0))
			      (lambda args (car args)))))
	      (if (not (eq? err 'no-such-sample)) (snd-display ";region->vct -1: ~A" err)))
	    (let ((err (catch #t
			      (lambda ()
				(region-samples->vct 12345678 1 r0))
			      (lambda args (car args)))))
	      ;; should this return 'no-such-sample?
	      (if err (snd-display ";region->vct 12345678: ~A" err)))
	    (let ((reader-string (format #f "~A" vr)))
	      (if (not (string=? reader-string "#<sample-reader: oboe.snd[0: 0] from 0, at 0>"))
		  (snd-display ";sample reader actually got: [~S]" reader-string)))
	    (let ((evr vr))
	      (if (not (equal? evr vr)) (snd-display ";sample-reader equal? ~A ~A" vr evr)))
	    (catch 'break
		   (lambda ()
		     (do ((i 0 (1+ i)))
			 ((= i 50827))
		       (if (not (= (if (odd? i) (next-sample vr) (read-sample vr)) (vct-ref samps1 i) (vct-ref samps2 i)))
			   (begin
			     (snd-display ";readers disagree at ~D" i)
			     (throw 'break)))))
		   (lambda args (car args)))
	    (free-sample-reader vr)))
	(let ((var (catch #t (lambda () (make-sample-reader 0 index -1)) (lambda args args))))
	  (if (not (eq? (car var) 'no-such-channel))
	      (snd-display ";make-sample-reader bad chan (-1): ~A" var)))
	(let ((var (catch #t (lambda () (make-sample-reader 0 index 1)) (lambda args args))))
	  (if (not (eq? (car var) 'no-such-channel))
	      (snd-display ";make-sample-reader bad chan (1): ~A, ~A" var index)))
	(let ((fd (make-sample-reader 0)))
	  (if (mix-sample-reader? fd) (snd-display ";sample-reader: mix ~A" fd))
	  (if (region-sample-reader? fd) (snd-display ";sample-reader: region ~A" fd))
	  (if (track-sample-reader? fd) (snd-display ";sample-reader: track ~A" fd))
	  (if (not (sample-reader? fd)) (snd-display ";sample-reader: normal ~A" fd))
	  (if (not (= (sample-reader-position fd) 0)) (snd-display ";sample-reader: position: ~A" fd))
	  (free-sample-reader fd)
	  (let ((str (format #f "~A" fd)))
	    (if (not (string=? (substring str (- (string-length str) 16)) "at eof or freed>"))
		(snd-display ";freed sample-reader: ~A [~A]?" str (substring str (- (string-length str) 16))))))
	(let* ((reg (car (regions)))
	       (chns (region-chans reg))
	       (var (catch #t (lambda () (make-region-sample-reader 0 reg (+ chns 1))) (lambda args args))))
	  (if (not (eq? (car var) 'no-such-channel))
	      (snd-display ";make-region-sample-reader bad chan (2): ~A ~A" var (regions))))
	
	(revert-sound index)
	(insert-sample 100 .5 index) 
	(let ((var (catch #t (lambda () (insert-sound "oboe.snd" 0 1)) (lambda args args))))
	  (if (not (eq? (car var) 'no-such-channel))
	      (snd-display ";insert-sound bad chan (1): ~A" var)))
	(let ((var (catch #t (lambda () (insert-sample -12 1.0)) (lambda args args))))
	  (if (not (eq? (car var) 'no-such-sample))
	      (snd-display ";insert-sample bad pos: ~A" var)))
	(set! (show-axes index 0) show-no-axes)
	(update-transform-graph index) 
	(update-time-graph index) 
	(if (or (fneq (sample 100) .5)
		(not (= (frames index) 50829)))
	    (snd-display ";insert-sample: ~A ~A?" (sample 100) (frames index)))
	(let ((v0 (make-vector 3))
	      (v1 (make-vct 3)))
	  (vct-fill! v1 .75)
	  (do ((i 0 (1+ i))) ((= i 3)) (vector-set! v0 i .25))
	  (insert-samples 200 3 v0 index) 
	  (insert-samples 300 3 v1 index) 
	  (if (or (fneq (sample 201) .25)
		  (fneq (sample 301) .75)
		  (not (= (frames index) 50835)))
	      (snd-display ";insert-samples: ~A ~A ~A?" (sample 201) (sample 301) (frames index))))
	(save-sound-as "hiho.snd" index mus-next mus-bshort :srate 22050)
	(let ((nindex (view-sound "hiho.snd")))
	  (if (fneq (sample 101 nindex) (sample 101 index))
	      (snd-display ";save-sound-as: ~A ~A?" (sample 101 nindex) (sample 101 index)))
	  (if (not (read-only nindex)) (snd-display ";read-only view-sound: ~A?" (read-only nindex)))
	  
	  (set! (speed-control-style nindex) speed-control-as-semitone)
	  (if (not (= (speed-control-style nindex) speed-control-as-semitone))
	      (snd-display ";speed-control-style set semi: ~A" (speed-control-style nindex)))
	  (set! (speed-control-tones nindex) -8)
	  (if (not (= (speed-control-tones nindex) 12))
	      (snd-display ";speed-control-tones -8: ~A" (speed-control-tones nindex)))
	  (set! (speed-control-tones nindex) 18)
	  (if (not (= (speed-control-tones nindex) 18))
	      (snd-display ";speed-control-tones 18: ~A" (speed-control-tones nindex)))
	  (graph->ps "aaa.eps")
	  (close-sound nindex))
	(revert-sound index)
	(set! (sample 50 index) .5) 
	(if (fneq (sample 50) .5) (snd-display ";set-sample: ~A?" (sample 50)))
	(let ((v0 (make-vector 3)))
	  (do ((i 0 (1+ i))) ((= i 3)) (vector-set! v0 i .25))
	  (set! (samples 60 3 index) v0) 
	  (if (or (fneq (sample 60) .25) (fneq (sample 61) .25))
	      (snd-display ";set-samples: ~A ~A ~A?" (sample 60) (sample 61) (sample 62))))
	(set! (samples 10 3 index) (list 0.1 0.2 0.3))
	(if (not (vequal (samples->vct 10 3 index) (vct 0.1 0.2 0.3)))
	    (snd-display ";set-samples via list: ~A" (samples->vct 10 3 index)))
	(revert-sound index)
	(save-sound-as "temporary.snd" index)
	(set! (samples 100000 20000 index) "temporary.snd")
	(if (not (vequal (samples->vct 110000 10) (samples->vct 10000 10)))
	    (snd-display ";set samples to self: ~A ~A" (samples->vct 110000 10) (samples->vct 10000 10)))
	(revert-sound index)
	(delete-sample 100 index) 
	(if (not (file-exists? "temporary.snd"))
	    (snd-display ";set-samples temp deleted?"))
	(delete-file "temporary.snd")
	(if (not (= (frames index) 50827)) (snd-display ";delete-sample: ~A?" (frames index)))
	(delete-samples 0 100 index) 
	(if (not (= (frames index) 50727)) (snd-display ";delete-samples: ~A?" (frames index)))
	(revert-sound index)
	(let ((maxa (maxamp index)))
	  (scale-to .5 index) 
	  (let ((newmaxa (maxamp index)))
	    (if (fneq newmaxa .5) (snd-display ";scale-to: ~A?" newmaxa))
	    (undo 1 index) 
	    (scale-by 2.0 index) 
	    (set! newmaxa (maxamp index))
	    (if (fneq newmaxa (* 2.0 maxa)) (snd-display ";scale-by: ~A?" newmaxa))
	    (revert-sound index)
	    (scale-by -1 index)
	    (mix "oboe.snd")
	    (if (fneq (maxamp index 0) 0.0) (snd-display ";invert+mix->~A" (maxamp)))
	    (revert-sound index)
	    (select-all index) 
	    (if (not (= (length (regions)) 3)) (snd-display ";regions(2): ~A?" (regions)))
	    (scale-selection-to .5) 
	    (set! newmaxa (maxamp index))
	    (if (fneq newmaxa .5) (snd-display ";scale-selection-to: ~A?" newmaxa))
	    (revert-sound index)
	    (select-all index) 
	    (scale-selection-by 2.0) 
	    (set! newmaxa (maxamp index))
	    (if (fneq newmaxa (* 2.0 maxa)) (snd-display ";scale-selection-by: ~A?" newmaxa))
	    (revert-sound index)
	    (select-all index) 
	    (let ((rread (make-region-sample-reader 0 (car (regions))))
		  (sread (make-sample-reader 0 index))
		  (rvect (region-samples->vct 0 100 (car (regions))))
		  (svect (samples 0 100 index)))
	      (if (fneq (vct-ref rvect 1) (region-sample 1 (car (regions))))
		  (snd-display ";region-sample: ~A ~A?" (region-sample 1 (car (regions))) (vct-ref rvect 1)))
	      (do ((i 0 (1+ i)))
		  ((= i 100))
		(let ((rval (next-sample rread))
		      (sval (next-sample sread)))
		  (if (fneq rval sval) (snd-display ";sample-read: ~A ~A?" rval sval))
		  (if (fneq rval (vct-ref rvect i)) (snd-display ";region-samples: ~A ~A?" rval (vct-ref rvect i)))
		  (if (fneq sval (vct-ref svect i)) (snd-display ";samples: ~A ~A?" sval (vct-ref svect i)))))
	      (free-sample-reader rread) 
	      (let ((val0 (next-sample sread)))
		(if (sample-reader-at-end? sread) (snd-display ";premature end?"))
		(previous-sample sread)
		(let ((val1 (previous-sample sread)))
		  (if (fneq val0 val1) (snd-display ";previous-sample: ~A ~A?" val0 val1))))
	      (free-sample-reader sread))))
	(revert-sound index)
	(let ((s100 (sample 100))
	      (s40 (sample 40))
	      (len (frames))
	      (addlen (mus-sound-frames "fyow.snd")))
	  (set! (cursor-style) cursor-line)
	  (set! (cursor-size) 25)
	  (set! (cursor index) 50) 
	  (if (not (= (cursor-style) cursor-line))
	      (snd-display ";cursor-style: ~A? " (cursor-style)))
	  (if (not (= (cursor-size) 25))
	      (snd-display ";cursor-size: ~A? " (cursor-size)))
	  (set! (cursor-style) cursor-cross)
	  (set! (cursor-size) 15)
	  (set! (cursor index 0) 30) 
	  (set! (cursor-style) cursor-line)
	  (set! (cursor index 0) 20) 
	  (set! (cursor-style index 0)
		(lambda (snd chn ax)
		  (let* ((point (cursor-position))
			 (x (car point))
			 (y (cadr point))
			 (size (inexact->exact (floor (/ (cursor-size) 2)))))
		    (draw-line (- x size) (- y size) (+ x size) (+ y size) snd chn cursor-context)    
		    (draw-line (- x size) (+ y size) (+ x size) (- y size) snd chn cursor-context))))
	  (if (not (procedure? (cursor-style index 0))) (snd-display ";set cursor-style to proc: ~A" (cursor-style index 0)))
	  (set! (cursor index) 50)
	  (insert-sound "fyow.snd" (cursor) 0 index 0) 
	  (if (or (fneq (sample 40) s40) (not (fneq (sample 100) s100)) (fneq (sample 100) 0.001831))
	      (snd-display ";insert-sound: ~A?" (sample 100)))
	  (if (not (= (frames) (+ len addlen))) (snd-display ";insert-sound len: ~A?" (frames)))
	  (save-sound-as "not-temporary.snd")
	  (insert-samples 0 100 "not-temporary.snd")
	  (set! (cursor index 0 0) (- (frames index 0 0) 2))
	  (revert-sound)
	  (if (not (= (cursor index 0) (- (frames index 0) 2)))
	      (snd-display ";set edpos cursor: ~A ~A ~A" (cursor) (cursor index 0 0) (- (frames index 0 0) 2)))
	  (if (not (file-exists? "not-temporary.snd"))
	      (snd-display ";insert-samples deleted its file?")
	      (delete-file "not-temporary.snd"))
	  (let ((id (make-region 0 99)))
	    (insert-region 60 id index) 
	    (if (not (= (frames) (+ len 100))) (snd-display ";insert-region len: ~A?" (frames)))
	    (if (fneq (sample 100) s40) (snd-display ";insert-region: ~A ~A?" (sample 100) s40))
	    (let ((var (catch #t (lambda () (insert-region 0 (+ 1000 (apply max (regions))))) (lambda args args))))
	      (if (not (eq? (car var) 'no-such-region))
		  (snd-display ";insert-region bad id: ~A" var)))
	    (save-region id "fmv.snd")
	    (if (not (= (mus-sound-header-type "fmv.snd") mus-next))
		(snd-display ";save-region header: ~A?" (mus-header-type-name (mus-sound-header-type "fmv.snd"))))
	    (if (not (= (mus-sound-data-format "fmv.snd") mus-out-format))
		(snd-display ";save-region format: ~A?" (mus-data-format-name (mus-sound-data-format "fmv.snd"))))
	    (if (not (= (mus-sound-srate "fmv.snd") (region-srate id)))
		(snd-display ";save-region srate: ~A (~A)" (mus-sound-srate "fmv.snd") (region-srate id)))
	    (if (not (= (mus-sound-chans "fmv.snd") (region-chans id)))
		(snd-display ";save-region chans: ~A (~A)" (mus-sound-chans "fmv.snd") (region-chans id)))
	    (if (not (= (mus-sound-frames "fmv.snd") (region-frames id)))
		(snd-display ";save-region length: ~A (~A)" (mus-sound-frames "fmv.snd") (region-frames id)))
	    (if (not (= (region-position id 0) 0))
		(snd-display ";save-region position: ~A" (region-position id 0)))
	    (delete-file "fmv.snd")
	    (save-region id "fmv.snd" mus-riff mus-lshort "this is a comment")
	    (if (not (= (mus-sound-header-type "fmv.snd") mus-riff))
		(snd-display ";save-region riff header: ~A?" (mus-header-type-name (mus-sound-header-type "fmv.snd"))))
	    (if (not (= (mus-sound-data-format "fmv.snd") mus-lshort))
		(snd-display ";save-region lshort format: ~A?" (mus-data-format-name (mus-sound-data-format "fmv.snd"))))
	    (if (not (= (mus-sound-frames "fmv.snd") (region-frames id)))
		(snd-display ";save-region length: ~A (~A)" (mus-sound-frames "fmv.snd") (region-frames id)))
	    (if (not (string=? (mus-sound-comment "fmv.snd") "this is a comment"))
		(snd-display ";save-region comment: ~A" (mus-sound-comment "fmv.snd")))
	    (delete-file "fmv.snd")
	    (save-region id :file "fmv.snd" :header-type mus-riff :data-format mus-lshort :comment "this is a comment")
	    (if (not (= (mus-sound-header-type "fmv.snd") mus-riff))
		(snd-display ";save-region opt riff header: ~A?" (mus-header-type-name (mus-sound-header-type "fmv.snd"))))
	    (if (not (= (mus-sound-data-format "fmv.snd") mus-lshort))
		(snd-display ";save-region opt lshort format: ~A?" (mus-data-format-name (mus-sound-data-format "fmv.snd"))))
	    (if (not (= (mus-sound-frames "fmv.snd") (region-frames id)))
		(snd-display ";save-region opt length: ~A (~A)" (mus-sound-frames "fmv.snd") (region-frames id)))
	    (if (not (string=? (mus-sound-comment "fmv.snd") "this is a comment"))
		(snd-display ";save-region opt comment: ~A" (mus-sound-comment "fmv.snd")))
	    (delete-file "fmv.snd")
	    (save-region id :comment "this is a comment" :file "fmv.snd" :data-format mus-lshort :header-type mus-riff)
	    (if (not (= (mus-sound-header-type "fmv.snd") mus-riff))
		(snd-display ";save-region opt1 riff header: ~A?" (mus-header-type-name (mus-sound-header-type "fmv.snd"))))
	    (if (not (= (mus-sound-data-format "fmv.snd") mus-lshort))
		(snd-display ";save-region opt1 lshort format: ~A?" (mus-data-format-name (mus-sound-data-format "fmv.snd"))))
	    (if (not (= (mus-sound-frames "fmv.snd") (region-frames id)))
		(snd-display ";save-region opt1 length: ~A (~A)" (mus-sound-frames "fmv.snd") (region-frames id)))
	    (if (not (string=? (mus-sound-comment "fmv.snd") "this is a comment"))
		(snd-display ";save-region opt1 comment: ~A" (mus-sound-comment "fmv.snd")))
	    (delete-file "fmv.snd")
	    (save-region id "fmv.snd" :data-format mus-bshort)
	    (if (not (= (mus-sound-header-type "fmv.snd") mus-next))
		(snd-display ";save-region opt2 next header: ~A?" (mus-header-type-name (mus-sound-header-type "fmv.snd"))))
	    (if (not (= (mus-sound-data-format "fmv.snd") mus-bshort))
		(snd-display ";save-region opt2 bshort format: ~A?" (mus-data-format-name (mus-sound-data-format "fmv.snd"))))
	    (delete-file "fmv.snd")
	    ))
	(close-sound index)
	(let ((var (catch #t (lambda () (new-sound "hi.snd" 0 1 100 0)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";new-sound bad chan: ~A" var)))
	(set! index (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "unequal lens"))
	(insert-silence 0 1000 index 1)
	(if (or (not (= (frames index 0) 1))
		(not (= (frames index 1) 1001)))
	    (snd-display ";silence 1: ~A ~A" (frames index 0) (frames index 1)))
	(save-sound index)
	(if (or (not (= (frames index 0) 1001))
		(not (= (frames index 1) 1001)))
	    (snd-display ";saved silence 1: ~A ~A" (frames index 0) (frames index 1)))
	(if (not (= (mus-sound-frames "fmv.snd") 1001))
	    (snd-display ";saved framers silence 1: ~A" (mus-sound-frames "fmv.snd")))
	(let ((v0 (samples->vct 0 1000 index 0))
	      (v1 (samples->vct 0 1000 index 1)))
	  (if (fneq (vct-peak v0) 0.0)
	      (snd-display ";auto-pad 0: ~A" (vct-peak v0)))
	  (if (fneq (vct-peak v1) 0.0)
	      (snd-display ";silence 0: ~A" (vct-peak v1))))
	(close-sound index)
	(delete-file "fmv.snd")
	
	(set! index (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "unequal lens"))
	(pad-channel 0 1000 index 1)
	(if (or (not (= (frames index 0) 1))
		(not (= (frames index 1) 1001)))
	    (snd-display ";pad-channel 1: ~A ~A" (frames index 0) (frames index 1)))
	(let ((v0 (samples->vct 0 1000 index 0))
	      (v1 (samples->vct 0 1000 index 1)))
	  (if (fneq (vct-peak v0) 0.0)
	      (snd-display ";pad 0: ~A" (vct-peak v0)))
	  (if (fneq (vct-peak v1) 0.0)
	      (snd-display ";pad 1: ~A" (vct-peak v1))))
	(map-channel (lambda (n) 1.0) 0 2 index 0)
	(map-channel (lambda (n) 1.0) 0 1002 index 1)
	(pad-channel 0 1000 index 0 1)
	(if (not (= (frames index 1) 1002))
	    (snd-display ";pad-channel ed 1: ~A ~A" (frames index 0) (frames index 1)))
	(close-sound index)
	(delete-file "fmv.snd")
	
	(set! index (new-sound "fmv.snd" mus-ircam mus-bshort 22050 1 "this is a comment"))
	(let ((v0 (make-vct 128)))
	  (vct-set! v0 64 .5)
	  (vct-set! v0 127 .5)
	  (vct->samples 0 128 v0 index 0)
	  (make-selection 0 126) 
	  (smooth-selection) 
	  (set! v0 (samples->vct 0 128 index 0 v0))
	  (if (or (fneq (sample 127) .5) (fneq (sample 120) .4962) (fneq (sample 32) 0.07431) (fneq (sample 64) 0.25308))
	      (snd-display ";smooth-selection: ~A?" v0))
	  (revert-sound index)
	  (vct-fill! v0 0.0)
	  (vct-set! v0 10 .5)
					;(vct->samples 0 128 v0 index 0) 
	  (vct->samples v0)
	  (select-all) 
	  (set! (sinc-width) 40)
	  (src-selection 0.5) 
	  (set! v0 (samples->vct 0 128 index 0 v0))
	  (if (or (fneq (sample 20) .5) (fneq (sample 30) 0.0) (fneq (sample 17) -.1057) )
	      (snd-display ";src-selection: ~A?" v0))
	  (revert-sound index)
	  (vct-fill! v0 0.0)
	  (vct-set! v0 10 .5)
					;(vct->samples 0 128 v0 index 0) 
	  (vct->samples 0 v0)
	  (select-all) 
	  (filter-selection '(0 0 .1 1 1 0) 40) 
	  (set! v0 (samples->vct 0 128 index 0 v0)) 
	  (if (or (fneq (sample 29) .1945) (fneq (sample 39) -.0137) (fneq (sample 24) -0.01986))
	      (snd-display ";filter-selection: ~A?" v0))
	  (revert-sound index)
	  (vct-fill! v0 1.0)
	  (vct->samples 0 128 v0 index 0) 
	  (select-all) 
	  (filter-selection (make-one-zero :a0 .5 :a1 0.0))
	  (set! v0 (samples->vct 0 128 index 0 v0)) 
	  (if (or (fneq (sample 29) .5) (fneq (sample 39) .5) (fneq (sample 24) 0.5))
	      (snd-display ";filter-selection one-zero: ~A?" v0))
	  (revert-sound index)
	  (vct-fill! v0 1.0)
	  (vct->samples 0 128 v0 index 0) 
	  (if (file-exists? "fmv5.snd") (delete-file "fmv5.snd"))
	  (select-all) 
	  (env-selection '(0 0 1 1 2 0) 1.0) 
	  (set! v0 (samples->vct 0 128 index 0 v0)) 
	  (if (or (fneq (sample 64) 1.0) (fneq (sample 20) .3125) (fneq (sample 119) 0.127))
	      (snd-display ";env-selection: ~A ~A ~A ~A?" (sample 64) (sample 20) (sample 119) v0))
	  (save-selection "fmv5.snd" mus-next mus-bint 22050 "") ;1.0->-1.0 if short
	  (revert-sound index)
	  (let ((tag (catch #t (lambda () (file->array "/baddy/hiho" 0 0 128 v0)) (lambda args (car args)))))
	    (if (not (eq? tag 'no-such-file)) (snd-display ";file->array w/o file: ~A" tag)))
	  (let ((tag (catch #t (lambda () (file->array "fmv5.snd" 123 0 128 v0)) (lambda args (car args)))))
	    (if (not (eq? tag 'no-such-channel)) (snd-display ";file->array w/o channel: ~A" tag)))
	  (file->array "fmv5.snd" 0 0 128 v0) 
	  (if (or (fneq (vct-ref v0 64) 1.0) (fneq (vct-ref v0 20) .3125) (fneq (vct-ref v0 119) 0.127))
	      (snd-display ";save-selection: ~A ~A ~A ~A?" (vct-ref v0 64) (vct-ref v0 20) (vct-ref v0 119) v0))
	  (if (not (= (mus-sound-header-type "fmv5.snd") mus-next))
	      (snd-display ";save-selection type: ~A?" (mus-header-type-name (mus-sound-header-type "fmv5.snd"))))
	  (if (not (= (mus-sound-data-format "fmv5.snd") mus-bint))
	      (snd-display ";save-selection format: ~A?" (mus-data-format-name (mus-sound-data-format "fmv5.snd"))))
	  (if (not (= (mus-sound-srate "fmv5.snd") 22050))
	      (snd-display ";save-selection srate: ~A?" (mus-sound-srate "fmv5.snd")))
	  (vct-fill! v0 0.0)
	  (vct-set! v0 100 .5)
	  (vct-set! v0 2 -.5)
	  (vct->samples 0 128 v0 index 0) 
	  (select-all) 
	  (without-errors (reverse-selection)) 
	  (save-selection "fmv4.snd" mus-riff mus-lfloat 44100 "this is a comment")
	  (set! v0 (samples->vct 0 128 index 0 v0)) 
	  (if (or (fneq (sample 27) 0.5) (fneq (sample 125) -.5))
	      (snd-display ";reverse-selection: ~A?" v0))
	  (file->array "fmv4.snd" 0 0 128 v0) 
	  (if (or (fneq (sample 27) 0.5) (fneq (sample 125) -.5))
	      (snd-display ";save reverse-selection: ~A?" v0))
	  (if (not (= (mus-sound-header-type "fmv4.snd") mus-riff))
	      (snd-display ";save-selection type 1: ~A?" (mus-header-type-name (mus-sound-header-type "fmv4.snd"))))
	  (if (not (= (mus-sound-data-format "fmv4.snd") mus-lfloat))
	      (snd-display ";save-selection format 1: ~A?" (mus-data-format-name (mus-sound-data-format "fmv4.snd"))))
	  (if (not (= (mus-sound-srate "fmv4.snd") 44100))
	      (snd-display ";save-selection srate 1: ~A?" (mus-sound-srate "fmv4.snd")))
	  (if (not (string=? (mus-sound-comment "fmv4.snd") "this is a comment"))
	      (snd-display ";save-selection comment: ~A?" (mus-sound-comment "fmv4.snd")))
	  (delete-file "fmv4.snd")
	  (save-selection :file "fmv4.snd" :header-type mus-riff :data-format mus-lfloat :srate 44100 :comment "this is a comment")
	  (if (not (= (mus-sound-header-type "fmv4.snd") mus-riff))
	      (snd-display ";save-selection opt type 1: ~A?" (mus-header-type-name (mus-sound-header-type "fmv4.snd"))))
	  (if (not (= (mus-sound-data-format "fmv4.snd") mus-lfloat))
	      (snd-display ";save-selection opt format 1: ~A?" (mus-data-format-name (mus-sound-data-format "fmv4.snd"))))
	  (if (not (= (mus-sound-srate "fmv4.snd") 44100))
	      (snd-display ";save-selection opt srate 1: ~A?" (mus-sound-srate "fmv4.snd")))
	  (if (not (string=? (mus-sound-comment "fmv4.snd") "this is a comment"))
	      (snd-display ";save-selection opt comment: ~A?" (mus-sound-comment "fmv4.snd")))
	  (delete-file "fmv4.snd")
	  (save-selection :file "fmv4.snd" :data-format mus-bfloat :channel 0)
	  (if (not (= (mus-sound-header-type "fmv4.snd") mus-next))
	      (snd-display ";save-selection opt1 type 1: ~A?" (mus-header-type-name (mus-sound-header-type "fmv4.snd"))))
	  (if (not (= (mus-sound-data-format "fmv4.snd") mus-bfloat))
	      (snd-display ";save-selection opt1 format 1: ~A?" (mus-data-format-name (mus-sound-data-format "fmv4.snd"))))
	  (if (not (= (mus-sound-chans "fmv4.snd") 1))
	      (snd-display ";save-selection opt1 chans: ~A?" (mus-sound-chans "fmv4.snd")))
	  (delete-file "fmv4.snd")
	  (revert-sound index)
	  (vct-fill! v0 0.0)
	  (vct-set! v0 2 1.0)
	  (let ((v1 (make-vct 256)))
	    (do ((i 0 (1+ i)))
		((= i 128))
	      (vct-set! v1 i (vct-ref v0 i)))
	    (vct->samples 0 128 v1 index 0))
	  (select-all) 
	  (convolve-selection-with "fmv5.snd" .5) 
	  (set! v0 (samples->vct 0 128 index 0 v0))
	  (if (fneq (sample 66) -.5) (snd-display ";convolve-selection-with: ~A ~A ~A?" (vct-ref v0 66) (sample 66) v0))
	  (close-sound index))
	(let* ((obind (open-sound "oboe.snd"))
	       (vol (maxamp obind))
	       (dur (frames)))
	  (set! (amp-control obind) 2.0)
	  (if (fffneq (amp-control obind) 2.0) (snd-display ";set amp-control ~A" (amp-control obind)))
	  (reset-controls obind)
	  (if (ffneq (amp-control obind) 1.0) (snd-display ";reset amp-control ~A" (amp-control obind)))
	  (set! (amp-control-bounds obind) (list 0.0 4.0))
	  (if (not (equal? (amp-control-bounds obind) (list 0.0 4.0))) (snd-display ";amp-control-bounds: ~A" (amp-control-bounds)))
	  (set! (amp-control obind) 2.0)
	  (if (eq? (without-errors (apply-controls obind)) 'no-such-sound) (snd-display ";apply-controls can't find oboe.snd?"))
	  (let ((newamp (maxamp obind)))
	    (if (> (abs (- (* 2.0 vol) newamp)) .05) (snd-display ";apply amp: ~A -> ~A?" vol newamp))
	    (set! (amp-control-bounds obind) (list 0.0 8.0))
	    (set! (speed-control-bounds obind) (list 1.0 5.0))
	    (if (not (equal? (speed-control-bounds obind) (list 1.0 5.0))) (snd-display ";speed-control-bounds: ~A" (speed-control-bounds)))
	    (set! (speed-control obind) 0.5)
	    (set! (speed-control-bounds obind) (list .05 20.0))
	    (add-mark 1234)
	    (apply-controls obind)
	    (let ((newdur (frames obind)))
	      (set! (speed-control obind) 1.0)
	      (if (not (< (- newdur (* 2.0 dur)) 256)) (snd-display ";apply speed: ~A -> ~A?" dur newdur))
	      ;; within 256 which is apply's buffer size (it always flushes full buffers) 
	      (set! (contrast-control? obind) #t)
	      (set! (contrast-control-bounds obind) (list 0.5 2.5))
	      (if (not (equal? (contrast-control-bounds obind) (list 0.5 2.5))) (snd-display ";contrast-control-bounds: ~A" (contrast-control-bounds)))
	      (set! (contrast-control obind) 1.0)
	      (apply-controls obind)
	      (set! (contrast-control-bounds obind) (list 0.0 10.0))
	      (if (not (equal? (contrast-control-bounds obind) (list 0.0 10.0))) (snd-display ";contrast-control-bounds (2): ~A" (contrast-control-bounds)))
	      (let ((secamp (maxamp obind))
		    (secdur (frames obind)))
		(if (fneq secamp .989) (snd-display ";apply contrast: ~A?" secamp))
		(if (not (= secdur newdur)) (snd-display ";apply contrast length: ~A -> ~A?" newdur secdur))
		(undo 3 obind)
		(set! (reverb-control? obind) #t)
		(set! (reverb-control-scale-bounds obind) (list 0.0 1.0))
		(if (not (equal? (reverb-control-scale-bounds obind) (list 0.0 1.0))) 
		    (snd-display ";reverb-control-scale-bounds: ~A" (reverb-control-scale-bounds)))
		(set! (reverb-control-length-bounds obind) (list 0.0 2.0))
		(if (not (equal? (reverb-control-length-bounds obind) (list 0.0 2.0))) 
		    (snd-display ";reverb-control-length-bounds: ~A" (reverb-control-length-bounds)))
		(set! (reverb-control-scale obind) .2)
		(apply-controls obind)
		(let ((revamp (maxamp obind))
		      (revdur (frames obind)))
		  (if (ffneq revamp .214) (snd-display ";apply reverb scale: ~A?" revamp))
		  (if (not (< (- revdur (+ 50828 (inexact->exact (round (* (reverb-control-decay) 22050))))) 256)) 
		      (snd-display ";apply reverb length: ~A?" revdur))
		  (undo 1 obind)
		  (set! (expand-control? obind) #t)
		  (set! (expand-control-bounds obind) (list 1.0 3.0))
		  (if (not (equal? (expand-control-bounds obind) (list 1.0 3.0))) (snd-display ";expand-control-bounds: ~A" (expand-control-bounds)))
		  (set! (expand-control obind) 1.5)
		  (apply-controls obind)
		  (let ((expamp (maxamp obind))
			(expdur (frames obind)))
		    (if (> (abs (- expamp .152)) .01) (snd-display ";apply expand-control scale: ~A?" expamp))
		    (if (not (> expdur (* 1.25 50828))) (snd-display ";apply expand-control length: ~A?" expdur))
		    (set! (expand-control-bounds obind) (list 0.001 20.0))
		    (undo 1 obind)
		    (set! (filter-control? obind) #t)
		    (set! (filter-control-order obind) 40)
		    (set! (filter-control-envelope obind) '(0 0 1 .5 1 0))
		    (apply-controls obind)
		    (let ((fltamp (maxamp obind))
			  (fltdur (frames obind)))
		      (if (> (abs (- fltamp .01)) .005) (snd-display ";apply filter scale: ~A?" fltamp))
		      (if (> (- fltdur (+ 40 50828)) 256) (snd-display ";apply filter length: ~A?" fltdur))
		      (undo 1 obind)))))))
	  (revert-sound obind)
	  (make-selection 1000 1000)
	  (scale-selection-to .1)
	  (scale-selection-by 2.0)
	  (make-selection 2000 2001)
	  (scale-selection-by 2.0)
	  (scale-selection-to .5)
	  (make-selection 1000 2001)
	  (scale-selection-to .5)
	  (scale-selection-by .5)
	  (make-selection 2000 2000)
	  (scale-selection-by 2.0)
	  (scale-selection-to .5)
	  (make-selection 1000 1001)
	  (scale-selection-to .1)
	  (scale-selection-by 2.0)
	  (make-selection 999 2002)
	  (scale-selection-to 1.0)
	  (scale-selection-by .5)
	  (let ((tree (edit-tree))
		(true-tree '((0 0 0 998 1.0 0.0 0.0 0) 
			     (999 0 999 999 0.999969720840454 0.0 0.0 0) 
			     (1000 0 1000 1000 6.09052181243896 0.0 0.0 0) 
			     (1001 0 1001 1001 0.999969720840454 0.0 0.0 0) 
			     (1002 0 1002 1999 0.499984979629517 0.0 0.0 0) 
			     (2000 0 2000 2000 7.54652404785156 0.0 0.0 0) 
			     (2001 0 2001 2001 3.7732629776001 0.0 0.0 0) 
			     (2002 0 2002 2002 0.999969720840454 0.0 0.0 0) 
			     (2003 0 2003 50827 1.0 0.0 0.0 0) 
			     (50828 -2 0 0 0.0 0.0 0.0 0))))
	    (if (not (= (length tree) (length true-tree)))
		(snd-display ";edit trees are not same length: ~A ~A?" (length tree) (length true-tree))
		(let ((len (length tree)))
		  (do ((i 0 (1+ i)))
		      ((= i len))
		    (let ((branch (list-ref tree i))
			  (true-branch (list-ref true-tree i)))
		      (if (or (not (= (car branch) (car true-branch)))
			      (not (= (cadr branch) (cadr true-branch)))
			      (not (= (caddr branch) (caddr true-branch)))
			      (not (= (cadddr branch) (cadddr true-branch)))
			      (fneq (list-ref branch 4) (list-ref true-branch 4)))
			  (snd-display ";edit trees disagree at ~D: ~A ~A" i branch true-branch)))))))
	  (insert-silence 1001 8)
	  (insert-silence 900 50)
	  (insert-silence 2005 1)
	  (insert-silence 999 2)
	  (let ((tree (edit-tree))
		(true-tree '((0 0 0 899 1.0 0.0 0.0 0) 
			     (900 -1 0 49 0.0 0.0 0.0 0) 
			     (950 0 900 948 1.0 0.0 0.0 0) 
			     (999 -1 0 1 0.0 0.0 0.0 0) 
			     (1001 0 949 998 1.0 0.0 0.0 0) 
			     (1051 0 999 999 0.999969720840454 0.0 0.0 0) 
			     (1052 0 1000 1000 6.09052181243896 0.0 0.0 0) 
			     (1053 -1 0 7 0.0 0.0 0.0 0) 
			     (1061 0 1001 1001 0.999969720840454 0.0 0.0 0)
			     (1062 0 1002 1946 0.499984979629517 0.0 0.0 0) 
			     (2007 -1 0 0 0.0 0.0 0.0 0) 
			     (2008 0 1947 1999 0.499984979629517 0.0 0.0 0) 
			     (2061 0 2000 2000 7.54652404785156 0.0 0.0 0) 
			     (2062 0 2001 2001 3.7732629776001 0.0 0.0 0) 
			     (2063 0 2002 2002 0.999969720840454 0.0 0.0 0) 
			     (2064 0 2003 50827 1.0 0.0 0.0 0) 
			     (50889 -2 0 0 0.0 0.0 0.0 0))))
	    (if (not (= (length tree) (length true-tree)))
		(snd-display ";silenced edit trees are not same length: ~A ~A?" (length tree) (length true-tree))
		(let ((len (length tree)))
		  (do ((i 0 (1+ i)))
		      ((= i len))
		    (let ((branch (list-ref tree i))
			  (true-branch (list-ref true-tree i)))
		      (if (or (not (= (car branch) (car true-branch)))
			      (not (= (cadr branch) (cadr true-branch)))
			      (not (= (caddr branch) (caddr true-branch)))
			      (not (= (cadddr branch) (cadddr true-branch)))
			      (fneq (list-ref branch 4) (list-ref true-branch 4)))
			  (snd-display ";silenced edit trees disagree at ~D: ~A ~A" i branch true-branch)))))))
	  (if (or (fneq (sample 998) -.03)
		  (fneq (sample 999) 0.0)
		  (fneq (sample 1000) 0.0)
		  (fneq (sample 1001) -.03))
	      (snd-display (format ";insert-silence [999 for 2]: ~A ~A ~A ~A?" (sample 998) (sample 999) (sample 1000) (sample 1001) )))
	  (if (or (fneq (sample 2006) -.033)
		  (fneq (sample 2007) 0.0)
		  (fneq (sample 2008) -.033))
	      (snd-display (format ";insert-silence [2007 for 1]: ~A ~A ~A?" (sample 2006) (sample 2007) (sample 2008))))
	  (revert-sound obind)
	  (add-mark 1200 obind 0)
	  (let ((mark-num (length (marks obind 0))))
	    (scale-by 2.0 obind 0)
	    (let ((mark-now (length (marks obind 0))))
	      (if (not (= mark-num mark-now))
		  (snd-display ";mark lost after scaling?"))
	      (set! (selection-position) 0)
	      (set! (selection-frames) 100)
	      (scale-selection-to .5)
	      (set! mark-now (length (marks obind 0)))
	      (if (not (= mark-num mark-now))
		  (snd-display ";mark lost after selection scaling?")))
	    (let ((m1 (add-mark 1000)))
	      (set! (cursor obind 0) 100)
	      (key (char->integer #\u) 4 obind)
	      (key (char->integer #\1) 0 obind)
	      (key (char->integer #\0) 0 obind)
	      (key (char->integer #\0) 0 obind)
	      (key (char->integer #\o) 4 obind)
	      (if (not (= (mark-sample m1) 1100))
		  (snd-display ";mark after zeros: ~D (1100)? " (mark-sample m1)))))
	  (revert-sound obind)
	  (let ((frs (frames obind)))
	    (make-region 0 999 obind 0)
	    (if (not (selection?)) (snd-display ";make-region but no selection? ~A" (selection?)))
	    (delete-selection)
	    (if (not (= (frames obind) (- frs 1000)))
		(snd-display ";delete-selection: ~A?" (frames obind)))
	    (let ((val (sample 0 obind 0)))
	      (undo)
	      (if (fneq (sample 1000) val)
		  (snd-display ";delete-selection val: ~A ~A" val (sample 1000)))
	      (insert-selection)
	      (let ((var (catch #t (lambda () (insert-selection 0 obind 123)) (lambda args args))))
		(if (not (eq? (car var) 'no-such-channel))
		    (snd-display ";insert-selection bad chan: ~A" var)))
	      (let ((var (catch #t (lambda () (mix-selection 0 obind 123)) (lambda args args))))
		(if (not (eq? (car var) 'no-such-channel))
		    (snd-display ";mix-selection bad chan: ~A" var)))
	      (if (not (= (frames obind) (+ frs 1000)))
		  (snd-display ";insert-selection: ~A?" (frames obind)))
	      (if (fneq (sample 2000) val)
		  (snd-display ";insert-selection val: ~A ~A" val (sample 2000)))
	      (set! val (sample 900))
	      (mix-selection)
	      (if (fneq (sample 900) (* 2 val))
		  (snd-display ";mix-selection val: ~A ~A" (* 2 val) (sample 900)))
	      (if (not (= (frames obind) (+ frs 1000)))
		  (snd-display ";mix-selection: ~A?" (frames obind)))))
	  (close-sound obind))
	
	(let* ((ind (open-sound "2.snd"))
	       (apply-to-sound 0)
	       (apply-to-channel 1)
	       (apply-to-selection 2)
	       (len (frames ind)))
	  (set! (sync ind) 1)
	  (set! (speed-control ind) .5)
	  (apply-controls ind apply-to-sound) ; temp 1
	  (if (> (abs (- (frames) (* 2 len))) 256)
	      (snd-display ";apply srate .5: ~A ~A" (frames) (* 2 len)))
	  (make-selection 0 (frames))
	  (set! (speed-control ind) .5)
	  (apply-controls ind apply-to-selection) ; temp 2
	  (if (> (abs (- (frames) (* 4 len))) 256)
	      (snd-display ";apply srate .5 to selection: ~A ~A" (frames) (* 4 len)))
	  (env-sound '(0 0 1 1) 0 (frames) 32.0) ; temp 3
	  (let ((reg (select-all))) ; make multi-channel region
	    (insert-region 0 reg) ; temp 4
	    (insert-selection 0))  ; temp 5
	  (revert-sound ind)
	  (set! (speed-control) .5)
	  (set! (sync ind) 0)
	  (set! (selected-channel ind) 1)
	  (apply-controls ind apply-to-channel)
	  (if (> (abs (- (frames ind 1) (* 2 len))) 256)
	      (snd-display ";apply srate .5 to chan 1: ~A ~A" (frames ind 1) (* 2 len)))
	  (if (not (= (frames ind 0) len))
	      (snd-display ";apply srate .5 but chan 0: ~A ~A" (frames ind 0) len))
	  (set! (speed-control ind) .5)
	  (apply-controls ind apply-to-sound 1000)
	  (make-selection 2000 4000)
	  (set! (speed-control ind) .5)
	  (apply-controls ind apply-to-selection)
	  (set! (selected-channel ind) #f)
	  (if (selected-channel ind) (snd-display ";selected-channel #f: ~A" (selected-channel ind)))
	  (close-sound ind))
	
	(let* ((ind1 (open-sound "oboe.snd"))
	       (mx1 (maxamp ind1 0))
	       (ind2 (open-sound "2.snd"))
	       (mx20 (maxamp ind2 0))
	       (mx21 (maxamp ind2 1)))
	  (select-sound ind1)
	  (scale-sound-by 2.0)
	  (let ((nmx (maxamp ind1 0)))
	    (if (fneq (* 2 mx1) nmx) (snd-display ";scale-sound-by 2.0: ~A ~A?" mx1 nmx))
	    (if (not (equal? (edit-fragment 1 ind1 0) (list "scale-channel 2.000 0 50828" "scale" 0 50828)))
		(snd-display ";scale-sound-by: ~A?" (edit-fragment 1 ind1 0))))
	  (scale-sound-to 0.5)
	  (let ((nmx (maxamp ind1 0)))
	    (if (fneq nmx 0.5) (snd-display ";scale-sound-to 0.5: ~A?" nmx))
	    (if (not (equal? (edit-fragment 2 ind1 0) (list "scale-channel 1.698 0 50828" "scale" 0 50828)))
		(snd-display ";scale-sound-to: ~A?" (edit-fragment 2 ind1 0))))
	  (scale-sound-by 0.0 0 1000 ind1 0)
	  (let ((nmx (maxamp ind1 0)))
	    (if (fneq 0.5 nmx) (snd-display ";scale-sound-by 0.0: ~A ~A?" mx1 nmx))
	    (if (not (equal? (edit-fragment 3 ind1 0) (list "scale-channel 0.000 0 1000" "scale" 0 1000)))
		(snd-display ";scale-sound-by 0.0: ~A?" (edit-fragment 3 ind1 0))))
	  (let* ((v (samples->vct 0 1000 ind1 0))
		 (pk (vct-peak v)))
	    (if (fneq pk 0.0) (snd-display ";scale-sound-by 0.0 [0:1000]: ~A?" pk)))
	  (revert-sound ind1)
	  (let ((oldv (samples->vct 12000 10 ind1 0)))
	    (scale-sound-by 2.0 12000 10 ind1 0)
	    (let ((newv (samples->vct 12000 10 ind1 0)))
	      (do ((i 0 (1+ i)))
		  ((= i 10))
		(if (fneq (* 2.0 (vct-ref oldv i)) (vct-ref newv i))
		    (snd-display ";scale ~D: ~A ~A?" i (vct-ref oldv i) (vct-ref newv i)))))
	    (if (not (equal? (edit-fragment 1 ind1 0) (list "scale-channel 2.000 12000 10" "scale" 12000 10)))
		(snd-display ";scale-sound-by 2.0 [12000:10]: ~A?" (edit-fragment 1 ind1 0))))
	  (revert-sound ind1)
	  (select-sound ind2)
	  (scale-sound-by 2.0)
	  (let ((nmx (maxamp ind2 0)))
	    (if (fneq (* 2 mx20) nmx) (snd-display ";2:0 scale-sound-by 2.0: ~A ~A?" mx20 nmx)))
	  (let ((nmx (maxamp ind2 1)))
	    (if (fneq (* 2 mx21) nmx) (snd-display ";2:1 scale-sound-by 2.0: ~A ~A?" mx21 nmx)))
	  (scale-sound-to 0.5)
	  (let ((nmx (max (maxamp ind2 0) (maxamp ind2 1))))
	    (if (fneq nmx 0.5) (snd-display ";2 scale-sound-to 0.5: ~A (~A)?" nmx (maxamp ind2))))
	  (scale-sound-by 0.0 0 1000 ind2 1)
	  (if (not (equal? (edit-fragment 3 ind2 1) (list "scale-channel 0.000 0 1000" "scale" 0 1000)))
	      (snd-display ";2:1 scale-sound-by 0.0: ~A?" (edit-fragment 3 ind2 1)))
	  (let* ((v (samples->vct 0 1000 ind2 1))
		 (pk (vct-peak v)))
	    (if (fneq pk 0.0) (snd-display ";2:1 scale-sound-by 0.0 [0:1000]: ~A?" pk)))
	  (revert-sound ind2)
	  (let ((oldv (samples->vct 12000 10 ind2 0)))
	    (scale-sound-by 2.0 12000 10 ind2 0)
	    (let ((newv (samples->vct 12000 10 ind2 0)))
	      (do ((i 0 (1+ i)))
		  ((= i 10))
		(if (fneq (* 2.0 (vct-ref oldv i)) (vct-ref newv i))
		    (snd-display ";2 scale ~D: ~A ~A?" i (vct-ref oldv i) (vct-ref newv i))))))
	  (revert-sound ind2)
	  (set! (sync ind2) 3)
	  (set! (sync ind1) 3)
	  (scale-sound-by 2.0)
	  (let ((nmx (maxamp ind1 0)))
	    (if (fneq mx1 nmx) (snd-display ";sync scale-sound-by 2.0: ~A ~A?" mx1 nmx)))
	  (let ((nmx (maxamp ind2 0)))
	    (if (fneq (* 2 mx20) nmx) (snd-display ";2:0 sync scale-sound-by 2.0: ~A ~A?" mx20 nmx)))
	  (let ((nmx (maxamp ind2 1)))
	    (if (fneq (* 2 mx21) nmx) (snd-display ";2:1 sync scale-sound-by 2.0: ~A ~A?" mx21 nmx)))
	  (scale-sound-to 1.0 20000 40000 ind2 1)
	  (let ((nmx (maxamp ind1 0)))
	    (if (fneq mx1 nmx) (snd-display ";sync scale-sound-to 1.0: ~A ~A?" mx1 nmx)))
	  (let ((nmx (maxamp ind2 0)))
	    (if (fneq (* 2 mx20) nmx) (snd-display ";2:0 sync scale-sound-to 1.0: ~A ~A?" mx20 nmx)))
	  (let ((nmx (maxamp ind2 1)))
	    (if (fneq nmx 1.0) (snd-display ";2:1 sync scale-sound-to 1.0: ~A?" nmx)))
	  
	  (close-sound ind1)
	  (close-sound ind2))
	
	(let* ((ind (open-sound "now.snd"))
	       (cur-amp (amp-control ind)))
	  (if (not (= now-snd-index ind)) (snd-display ";memo-sound: ~A ~A ~A" memo-sound ind now-snd-index))
	  (set! (amp-control ind) .5)
	  (if (ffneq (amp-control ind) .5) (snd-display ";amp-control (.5): ~A?" (amp-control ind)))
	  (set! (amp-control ind 0) .25)
	  (if (ffneq (amp-control ind) .5) (snd-display ";amp-control after local set (.5): ~A?" (amp-control ind)))
	  (if (ffneq (amp-control ind 0) .25) (snd-display ";amp-control 0 (.25): ~A?" (amp-control ind 0)))
	  (set! (amp-control ind) 1.0)
	  (if (ffneq (amp-control ind) 1.0) (snd-display ";amp-control (1.0): ~A?" (amp-control ind)))
	  (if (ffneq (amp-control ind 0) .25) (snd-display ";amp-control 0 after set (.25): ~A?" (amp-control ind 0)))
	  (set! (transform-graph-type ind 0) graph-as-sonogram)
	  (if (not (= (transform-samples-size ind 0) 0)) 
	      (snd-display ";transform-samples-size: ~A" (transform-samples-size ind 0)))
	  (if (transform-sample 0 0 ind 0) (snd-display ";transform-sample (empty): ~A" (transform-sample 0 0 ind 0)))
	  (if (transform-samples->vct ind 0) (snd-display ";transform-samples->vct (empty): ~A" (transform-samples->vct ind 0)))
	  (close-sound ind)
	  (set! ind (open-sound "4.aiff"))
	  (if (ffneq (amp-control ind) 1.0) (snd-display ";amp-control upon open (1.0): ~A?" (amp-control ind)))
	  (if (ffneq (amp-control ind 2) 1.0) (snd-display ";amp-control 2 upon open (1.0): ~A?" (amp-control ind 2)))
	  (set! (amp-control ind) .5)
	  (if (ffneq (amp-control ind 2) .5) (snd-display ";amp-control 2 after global set (.5): ~A?" (amp-control ind 2)))
	  (set! (amp-control ind 2) .25)
	  (if (ffneq (amp-control ind 2) .25) (snd-display ";amp-control 2 (.25): ~A?" (amp-control ind 2)))
	  (if (ffneq (amp-control ind 1) .5) (snd-display ";amp-control 1 after local set (.5): ~A?" (amp-control ind 1)))
	  (let ((before-ran #f)
		(after-ran #f))
	    (reset-hook! after-apply-hook)
	    (add-hook! after-apply-hook (lambda (snd) (set! after-ran snd)))
	    (apply-controls ind)
	    (if (not (= ind after-ran)) (snd-display ";after-apply-hook: ~A?" after-ran))
	    (reset-hook! after-apply-hook))
	  (revert-sound ind)
	  (set! (sync ind) 1)
	  (scale-to '#(.1 .2))
	  (let ((mx (maxamp ind #t)))
	    (if (or (fneq (list-ref mx 0) .1)
		    (fneq (list-ref mx 1) .2)
		    (fneq (list-ref mx 2) .2)
		    (fneq (list-ref mx 3) .2))
		(snd-display ";scale to with vector: ~A" mx)))
	  (set! (filter-control-envelope ind) '(0 0 1 1))
	  (if (not (feql '(0.0 0.0 1.0 1.0) (filter-control-envelope ind))) 
	      (snd-display ";set filter-control-envelope: ~A?" (filter-control-envelope ind)))
	  (set! (filter-control-order ind) 20)
	  (if (not (vequal (filter-control-coeffs ind)
			   (vct -0.007 0.010 -0.025 0.029 -0.050 0.055 -0.096 0.109 -0.268 0.241 
				0.241 -0.268 0.109 -0.096 0.055 -0.050 0.029 -0.025 0.010 -0.007)))
	      (snd-display ";highpass coeffs: ~A" (filter-control-coeffs ind)))
	  (set! (filter-control-envelope ind) (filter-control-envelope ind))
	  (if (not (feql '(0.0 0.0 1.0 1.0) (filter-control-envelope ind))) 
	      (snd-display ";set filter-control-envelope to self: ~A?" (filter-control-envelope ind)))
	  (set! (filter-control-envelope ind) '(0 1 1 0))
	  (if (not (vequal (filter-control-coeffs ind)
			   (vct 0.003 0.002 0.004 0.002 0.007 0.003 0.014 0.012 0.059 0.394 
				0.394 0.059 0.012 0.014 0.003 0.007 0.002 0.004 0.002 0.003)))
	      (snd-display ";lowpass coeffs: ~A" (filter-control-coeffs ind)))
	  (close-sound ind))
	
	(let* ((obind (open-sound "4.aiff"))
	       (amps (maxamp obind #t)))
	  (if (< (window-width) 600) 
	      (set! (window-width) 600))
	  (if (< (window-height) 600)
	      (set! (window-height) 600))
	  (set! (x-bounds obind 0) (list 0.0 0.1))
	  (set! (show-axes obind 0) show-x-axis)
	  (update-time-graph)
	  (set! (amp-control obind) 0.1)
	  (select-channel 2)
	  (if (eq? (without-errors (apply-controls obind 1)) 'no-such-sound) (snd-display ";apply-controls can't find 4.aiff?"))
	  (let ((newamps (maxamp obind #t)))
	    (if (or (fneq (car amps) (car newamps))
		    (fneq (cadr amps) (cadr newamps))
		    (> (abs (- (* 0.1 (caddr amps)) (caddr newamps))) .05)
		    (fneq (cadddr amps) (cadddr newamps)))
		(snd-display ";apply amps:~%  ~A ->~%  ~A?" amps newamps))
	    (undo 1 obind 2)
	    (set! (amp-control obind) 0.1)
	    (make-region 0 (frames obind) obind 1)
	    (without-errors (apply-controls obind 2))
	    (set! newamps (maxamp obind #t))
	    (if (or (fneq (car amps) (car newamps))
		    (> (abs (- (* 0.1 (cadr amps)) (cadr newamps))) .05)
		    (fneq (caddr amps) (caddr newamps))
		    (fneq (cadddr amps) (cadddr newamps)))
		(snd-display ";apply selection amp:~%  ~A ->~%  ~A?" amps newamps))
	    (if with-gui
		(let* ((axinfo (axis-info obind 0 time-graph))
		       (losamp (car axinfo))
		       (hisamp (cadr axinfo))
		       (x0 (list-ref axinfo 2))
		       (y0 (list-ref axinfo 3))
		       (x1 (list-ref axinfo 4))
		       (y1 (list-ref axinfo 5))
		       (xpos (+ x0 (* .5 (- x1 x0))))
		       (ypos (+ y0 (* .75 (- y1 y0)))))
		  (define (cp-x x) (inexact->exact (floor (+ (list-ref axinfo 10) 
						      (* (- x x0) (/ (- (list-ref axinfo 12) (list-ref axinfo 10)) 
								     (- x1 x0)))))))
		  (define (cp-y y) (inexact->exact (floor (+ (list-ref axinfo 13) 
						      (* (- y1 y) (/ (- (list-ref axinfo 11) (list-ref axinfo 13)) 
								     (- y1 y0)))))))
		  (select-channel 0)
		  (set! (cursor obind) 100)
		  (let ((xy (cursor-position obind)))
		    (if (fneq (position->x (car xy)) (/ (cursor obind) (srate obind)))
			(snd-display ";cursor-position: ~A ~A ~A?" (car xy) (position->x (car xy)) (/ (cursor obind) (srate obind)))))
		  (if (fneq (position->x (x->position xpos)) xpos)
		      (snd-display ";x<->position: ~A ~A?" (position->x (x->position xpos)) xpos))
		  (if (> (abs (- (position->y (y->position ypos)) ypos)) .5)
		      (snd-display ";y<->position: ~A ~A?" (position->y (y->position ypos)) ypos))
		  (if (not (= losamp (left-sample obind 0)))
		      (snd-display ";axis-info[0 losamp]: ~A ~A?" losamp (left-sample obind 0)))
		  (if (not (= hisamp (right-sample obind 0)))
		      (snd-display ";axis-info[1 hisamp]: ~A ~A?" hisamp (right-sample obind 0)))
		  (if (fneq (list-ref axinfo 6) 0.0)
		      (snd-display ";axis-info[6 xmin]: ~A?" (list-ref axinfo 6)))
		  (if (fneq (list-ref axinfo 7) -1.0)
		      (snd-display ";axis-info[7 ymin]: ~A?" (list-ref axinfo 7)))
		  (if (fneq (list-ref axinfo 9) 1.0)
		      (snd-display ";axis-info[9 ymax]: ~A?" (list-ref axinfo 9)))
		  (if (> (abs (apply - (our-x->position obind x0))) 1) 
		      (snd-display ";x0->position: ~A?" (our-x->position obind x0)))
		  (if (> (abs (apply - (our-x->position obind x1))) 1) 
		      (snd-display ";x1->position: ~A?" (our-x->position obind x1)))
		  (if (> (abs (apply - (our-x->position obind (* 0.5 (+ x0 x1))))) 1)
		      (snd-display ";xmid->position: ~A?" (our-x->position obind (* 0.5 (+ x0 x1)))))
		  (if (not full-test)
		      (begin
			(if (> (abs (- (x->position xpos) (cp-x xpos))) 1)
			    (snd-display ";cp-x .5: ~A ~A?" (x->position xpos) (cp-x xpos)))
			(if (> (abs (- (y->position ypos) (cp-y ypos))) 1)
			    (snd-display ";cp-y .75: ~A ~A?" (y->position ypos) (cp-y ypos)))
			(do ((i 0 (1+ i)))
			    ((= i 10))
			  (let ((xpos (+ x0 (my-random (- x1 x0))))
				(ypos (+ y0 (my-random (- y1 y0)))))
			    (if (> (abs (- (x->position xpos) (cp-x xpos))) 1)
				(snd-display ";cp-x[~A] ~A: ~A ~A?" i xpos (x->position xpos) (cp-x xpos)))
			    (if (> (abs (- (y->position ypos) (cp-y ypos))) 1)
				(snd-display ";cp-y[~A] ~A: ~A ~A?" i ypos (y->position ypos) (cp-y ypos)))
			    (if (fneq (position->x (cp-x xpos)) xpos)
				(snd-display ";x->position cp-x ~A ~A" xpos (position->x (cp-x xpos))))
			    (if (fffneq (position->y (cp-y ypos)) ypos)
				(snd-display ";y->position cp-y ~A ~A" ypos (position->y (cp-y ypos))))))))
		  (set! (left-sample obind 0) 1234)
		  (if (not (= 1234 (car (axis-info obind 0))))
		      (snd-display ";axis-info[0 losamp at 1234]: ~A ~A?" (car (axis-info obind 0)) (left-sample obind 0)))
		  (set! axinfo (axis-info obind 0))
		  (set! x0 (list-ref axinfo 2))
		  (set! x1 (list-ref axinfo 4))
		  (if (> (abs (apply - (our-x->position obind x0))) 1) 
		      (snd-display ";x0a->position: ~A?" (our-x->position obind x0)))
		  (if (> (abs (apply - (our-x->position obind x1))) 1) 
		      (snd-display ";x1a->position: ~A?" (our-x->position obind x1)))
		  (if (> (abs (apply - (our-x->position obind (* 0.5 (+ x0 x1))))) 1)
		      (snd-display ";xmida->position: ~A?" (our-x->position obind (* 0.5 (+ x0 x1)))))
		  (set! (y-bounds obind 0) (list -2.0 3.0))
		  (if (fneq (list-ref (axis-info obind 0) 7) -2.0)
		      (snd-display ";axis-info[7 ymin -2.0]: ~A?" (list-ref (axis-info obind 0) 7)))
		  (if (fneq (list-ref (axis-info obind 0) 9) 3.0)
		      (snd-display ";axis-info[9 ymax 3.0]: ~A?" (list-ref (axis-info obind 0) 9)))
		  
		  ))
	    (close-sound obind)))
	
	(let ((ind1 (open-sound "oboe.snd")))
	  (test-orig (lambda (snd) (src-sound 2.0 1.0 ind1)) (lambda (snd) (src-sound 0.5 1.0 ind1)) 'src-sound ind1)
	  (test-orig (lambda (snd) (src-channel 2.0)) (lambda (snd) (src-channel 0.5)) 'src-channel ind1)
	  (test-orig (lambda (snd) (scale-by 2.0 ind1)) (lambda (snd) (scale-by 0.5 ind1)) 'scale-by ind1)
	  (test-orig (lambda (snd) (scale-sound-by 2.0 ind1)) (lambda (snd) (scale-sound-by 0.5 ind1)) 'scale-sound-by ind1)
	  (test-orig (lambda (snd) (scale-channel 2.0)) (lambda (snd) (scale-channel 0.5)) 'scale-channel ind1)
	  (test-orig (lambda (snd) (reverse-sound ind1)) (lambda (snd) (reverse-sound ind1)) 'reverse-sound ind1)
	  (test-orig (lambda (snd) (reverse-channel)) (lambda (snd) (reverse-channel)) 'reverse-channel ind1)
	  (test-orig (lambda (snd) (env-sound '(0 1.0 1 2.0) ind1)) (lambda (snd) (env-sound '(0 1.0 1 0.5) ind1)) 'env-sound ind1)
	  (test-orig (lambda (snd) (env-sound '(0 1.0 1 2.0 2 1.0) ind1)) (lambda (snd) (env-sound '(0 1.0 1 0.5 2 1.0) ind1)) 'env-sound ind1)
	  (test-orig (lambda (snd) (env-channel (make-env :envelope '(0 1.0 1 2.0) :end (frames))))
		     (lambda (snd) (env-channel (make-env :envelope '(0 1.0 1 0.5) :end (frames)))) 'env-channel ind1)
	  (test-orig (lambda (snd) (env-channel '(0 1.0 1 2.0)))
		     (lambda (snd) (env-channel '(0 1.0 1 0.5))) 'env-channel ind1)
	  (test-orig (lambda (snd) (env-channel (make-env :envelope '(0 2 1 2 2 0.5 3 0.5) :base 0 :end (frames))))
		     (lambda (snd) (env-channel (make-env :envelope '(0 0.5 1 0.5 2 2 3 2) :base 0 :end (frames)))) 'env-channel ind1)
	  (test-orig (lambda (snd) (map-channel (lambda (n) (* n 2)))) (lambda (snd) (map-channel (lambda (n) (* n 0.5)))) 'map-channel ind1)
	  (test-orig (lambda (snd) (map-channel (lambda (n) (* n 2)) 1234)) (lambda (snd) (map-channel (lambda (n) (* n 0.5)) 1234)) 'map-channel ind1)
	  (test-orig (lambda (snd) (map-channel (lambda (n) (* n 2)) 12005 10)) (lambda (snd) (map-channel (lambda (n) (* n 0.5)) 12005 10)) 'map-channel ind1)
	  (test-orig (lambda (snd) (map-channel (let ((vect (make-vector 1 0.0))) (lambda (y) (vector-set! vect 0 (* y 2)) vect))))
		     (lambda (snd) (map-channel (lambda (y) (list (* y 0.5))))) 'map-channel ind1)
	  (test-orig (lambda (snd) (map-channel 
				    (let ((vect (make-vector 2 0.0))) 
				      (lambda (y) 
					(vector-set! vect 0 (* y 2))
					(vector-set! vect 1 (* y 2))
					vect))))
		     (lambda (snd) (map-channel
				    (let ((outp #f))
				      (lambda (y) 
					(if outp
					    (set! outp #f)
					    (set! outp (* y 0.5)))
					outp))))
		     'map-channel ind1)
	  (test-orig (lambda (snd) (map-chan (lambda (n) (* n 2)))) (lambda (snd) (map-chan (lambda (n) (* n 0.5)))) 'map-chan ind1)
	  (test-orig (lambda (snd) (pad-channel 1000 2000 ind1)) (lambda (snd) (delete-samples 1000 2000 ind1)) 'pad-channel ind1)
	  (test-orig (lambda (snd) (clm-channel (make-one-zero :a0 2.0 :a1 0.0)))
		     (lambda (snd) (clm-channel (make-one-zero :a0 0.5 :a1 0.0))) 'clm-channel ind1)
	  (test-orig (lambda (snd) (clm-channel (make-one-pole :a0 2.0 :b1 0.0)))
		     (lambda (snd) (clm-channel (make-one-pole :a0 0.5 :b1 0.0))) 'clm-channel ind1)
	  (test-orig (lambda (snd) (filter-sound (make-one-zero :a0 2.0 :a1 0.0) 0 ind1 0)) 
		     (lambda (snd) (filter-sound (make-one-zero :a0 0.5 :a1 0.0)) 0 ind1 0) 'filter-sound ind1)
	  
	  (let ((var (catch #t (lambda () (src-sound '(0 0 1 1))) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";src-sound env at 0: ~A" var)))
	  (let ((var (catch #t (lambda () (src-sound '(0 1 1 -1))) (lambda args args))))
	    (if (not (eq? (car var) 'out-of-range))
		(snd-display ";src-sound env through 0: ~A" var)))
	  
	  (scale-to 1.0 ind1)
	  (let ((v0 (make-vct 10))
		(v1 (samples->vct 12000 10 ind1 0)))
	    (vct-set! v0 0 1.0)
	    (array->file "fmv3.snd" v0 10 22050 1)
	    (copy-file "oboe.snd" "fmv4.snd")
	    (convolve-with "fmv3.snd" 1.0 ind1)
	    (convolve-files "fmv4.snd" "fmv3.snd" 1.0 "fmv5.snd")
	    (let ((v2 (samples->vct 12000 10 ind1 0)))
	      (if (not (vfequal v1 v2))
		  (snd-display ";~A (orig: 0) ~A ~A" 'convolve-with v1 v2))
	      (file->array "fmv5.snd" 0 12000 10 v2)
	      (if (not (vfequal v1 v2))
		  (snd-display ";convolve-files: (orig: 0) ~A ~A" v1 v2)))
	    (delete-file "fmv3.snd")
	    (delete-file "fmv5.snd"))
	  (convolve-files "2.snd" "oboe.snd" 0.5 "fmv5.snd")
	  (if (fneq (cadr (mus-sound-maxamp "fmv5.snd")) 0.5) (snd-display ";convolve-files stereo: ~A" (mus-sound-maxamp "fmv5.snd")))
	  (delete-file "fmv5.snd")
	  (scale-to .25 ind1)
	  (set! (y-bounds ind1) '())
	  (if (not (equal? (y-bounds ind1) (list -.25 .25)))
	      (snd-display ";y-bounds '(): ~A?" (y-bounds ind1)))
	  (revert-sound ind1)
	  
	  (scale-to 1.0 ind1)
	  (let ((v0 (make-vct 10))
		(v1 (samples->vct 12000 10 ind1 0)))
	    (vct-set! v0 5 1.0)
	    (array->file "fmv3.snd" v0 10 22050 1)
	    (convolve-with "fmv3.snd" 1.0 ind1)
	    (convolve-files "fmv4.snd" "fmv3.snd" 1.0 "fmv5.snd")
	    (let ((v2 (samples->vct 12005 10 ind1 0)))
	      (if (not (vfequal v1 v2))
		  (snd-display ";~A (orig: 2) ~A ~A" 'convolve-with v1 v2))
	      (file->array "fmv5.snd" 0 12005 10 v2)
	      (if (not (vfequal v1 v2))
		  (snd-display ";convolve-files: (orig: 2) ~A ~A" v1 v2)))
	    (delete-file "fmv3.snd")
	    (delete-file "fmv4.snd")
	    (delete-file "fmv5.snd"))
	  
	  (revert-sound ind1)
	  (let ((old-val (selection-creates-region))
		(old-regions (regions)))
	    (set! (selection-creates-region) #f)
	    (select-all ind1)
	    (set! (selection-creates-region) old-val)
	    (if (not (equal? old-regions (regions)))
		(snd-display ";selection-create-region: ~A -> ~A?" old-regions (regions))))
	  (convolve-selection-with "pistol.snd" (maxamp))
	  (let ((data (samples->vct 12000 10 ind1 0)))
	    (convolve-with "pistol.snd" (maxamp ind1 0 0) ind1 0 0)
	    (let ((new-data (samples->vct 12000 10 ind1 0)))
	      (if (not (vfequal data new-data))
		  (snd-display ";convolve-selection-with: ~A ~A?" data new-data))))
	  (revert-sound ind1)
	  (make-selection 1000 2000 ind1)
	  (let ((ma (maxamp ind1)))
	    (convolve-selection-with "pistol.snd" ma)
	    (if (fneq (maxamp ind1) ma) (snd-display ";convolve-selection-with 1000: ~A ~A?" ma (maxamp ind1))))
	  (make-selection 1000 2000 ind1)
	  (let ((id (make-region)))
	    (if (not (region? id))
		(snd-display ";make-region argless: ~A" id))
	    (if (not (= (region-frames id 0) (selection-frames)))
		(snd-display ";region/selection-frames: ~A ~A (~A)?" (region-frames id 0) (selection-frames) (region-frames id)))
	    (if (fneq (region-sample 0 id) (sample 1000 ind1))
		(snd-display ";region-sample from make-region: ~A ~A?" (region-sample 0 id) (sample 1000 ind1))))
	  (close-sound ind1))
	(let* ((ind (open-sound "2.snd"))
	       (reg (make-region 0 100 ind #t)))
	  (if (not (= (region-chans reg) 2))
	      (snd-display ";make-region chan #t: ~A" (region-chans reg)))
	  (close-sound ind))
	
	(let ((ind1 (open-sound "2.snd")))
	  (let ((v0 (samples->vct 12000 10 ind1 0))
		(v1 (samples->vct 12000 10 ind1 1)))
	    (swap-channels ind1)
	    (let ((v2 (samples->vct 12000 10 ind1 0))
		  (v3 (samples->vct 12000 10 ind1 1)))
	      (if (or (vequal v0 v2)
		      (vequal v1 v3))
		  (snd-display ";swap-channels 0: no change! ~A ~A ~A ~A" v0 v2 v1 v3)))
	    (swap-channels ind1)
	    (let ((v2 (samples->vct 12000 10 ind1 0))
		  (v3 (samples->vct 12000 10 ind1 1)))
	      (if (or (not (vequal v0 v2))
		      (not (vequal v1 v3)))
		  (snd-display ";swap-channels 1: ~A ~A ~A ~A" v0 v2 v1 v3)))
	    ;; as long as we're here...
	    (set! (cursor ind1 0) 100)
	    (set! (cursor ind1 1) 200)
	    (if (or (not (= (cursor ind1 0) 100)) 
		    (not (= (cursor ind1 1) 200)))
		(snd-display ";cursor: ~A ~A?" (cursor ind1 0) (cursor ind1 1)))
	    (forward-sample 10 ind1 0)
	    (forward-sample -10 ind1 1)
	    (if (or (not (= (cursor ind1 0) 110)) 
		    (not (= (cursor ind1 1) 190)))
		(snd-display ";cursor (1): ~A ~A?" (cursor ind1 0) (cursor ind1 1)))
	    (backward-sample -10 ind1 0)
	    (backward-sample 10 ind1 1)
	    (if (or (not (= (cursor ind1 0) 120)) 
		    (not (= (cursor ind1 1) 180)))
		(snd-display ";cursor (2): ~A ~A?" (cursor ind1 0) (cursor ind1 1)))
	    (set! (sync ind1) 1)
	    (scale-by (list .5 .25) ind1)
	    (scale-by '#(2.0 4.0) ind1)
	    (scale-by #f ind1)
	    (revert-sound ind1)
	    (let ((amps (maxamp ind1 #t)))
	      (swap-channels ind1 0 ind1)
	      (let ((newamps (maxamp ind1 #t)))
		(if (or (fneq (car amps) (cadr newamps))
			(fneq (cadr amps) (car newamps)))
		    (snd-display ";swap-channels with cp def: ~A ~A" amps newamps)))
	      (swap-channels ind1 1)
	      (let ((newamps (maxamp ind1 #t)))
		(if (or (fneq (car amps) (car newamps))
			(fneq (cadr amps) (cadr newamps)))
		    (snd-display ";swap-channels with cp def 0: ~A ~A" amps newamps))))
	    (close-sound ind1)))
	
	(let ((ind1 (open-sound "oboe.snd"))
	      (ind2 (open-sound "2.snd")))
	  (let ((ups1 (count-matches (lambda (n) (> n .1)) 0 ind1 0))
		(ups2 (let ((count 0))
			(scan-chan (lambda (n)
				     (if (> n .1)
					 (set! count (+ count 1)))
				     #f)
				   0 (frames ind1) ind1 0)
			count)))
	    (if (not (= ups1 ups2))
		(snd-display ";scan-chan: ~A ~A?" ups1 ups2))
	    (set! ups1 (count-matches (lambda (n) (> n .03)) 0 ind2 0))
	    (set! ups2 (count-matches (lambda (n) (> n .03)) 0 ind2 1))
	    (let ((ups3 (let ((count 0))
			  (scan-chan (lambda (n)
				       (if (> n .03)
					   (set! count (+ count 1)))
				       #f)
				     0 (frames ind2) ind2 0)
			  count))
		  (ups4 (let ((count 0))
			  (scan-chan (lambda (n)
				       (if (> n .03)
					   (set! count (+ count 1)))
				       #f)
				     0 (frames ind2) ind2 1)
			  count)))
	      (if (not (= ups1 ups3))
		  (snd-display ";2[0] scan-chan: ~A ~A?" ups1 ups3))
	      (if (not (= ups2 ups4))
		  (snd-display ";2[1] scan-chan: ~A ~A?" ups2 ups4)))
	    
	    (set! (sync ind2) #t)
	    (let ((total
		   (let ((count 0)) 
		     (scan-chans (lambda (n) 
				   (if (> n .03) 
				       (set! count (+ count 1))) 
				   #f))
		     count)))
	      (if (not (= total (+ ups1 ups2)))
		  (snd-display ";scan-chans: ~A ~A?" total (+ ups1 ups2))))
	    (set! (sync ind2) #f)
	    (let ((total
		   (let ((count 0)) 
		     (scan-sound-chans (lambda (n) 
					 (if (> n .03) 
					     (set! count (+ count 1))) 
					 #f)
				       0 (frames ind2) ind2)
		     count)))
	      (if (not (= total (+ ups1 ups2)))
		  (snd-display ";scan-sound-chans: ~A ~A?" total (+ ups1 ups2))))
	    (set! (sync ind2) #f)
	    (let ((total
		   (let ((count 0)) 
		     (scan-across-all-chans (lambda (nd len) 
					      (do ((i 0 (1+ i)))
						  ((= i len) #f) 
						(if (> (vector-ref nd i) .03) 
						    (set! count (+ count 1))))))
		     count))
		  (ups3 (count-matches (lambda (n) (> n .03)) 0 ind1 0)))
	      (if (not (= total (+ ups1 ups2 ups3)))
		  (snd-display ";scan-across-all-chans: ~A ~A?" total (+ ups1 ups2 ups3))))
	    (let ((total
		   (let ((count 0)) 
		     (scan-all-chans (lambda (n) 
				       (if (> n .03)
					   (set! count (+ count 1)))
				       #f))
		     count))
		  (ups3 (count-matches (lambda (n) (> n .03)) 0 ind1 0)))
	      (if (not (= total (+ ups1 ups2 ups3)))
		  (snd-display ";scan-all-chans: ~A ~A?" total (+ ups1 ups2 ups3)))))
	  
	  (select-sound ind1)
	  (forward-graph)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 0)))
	      (snd-display ";forward from ~A 0 to ~A ~A?" ind1 (selected-sound) (selected-channel)))
	  (forward-graph)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 1)))
	      (snd-display ";forward from ~A 0 to ~A ~A?" ind2 (selected-sound) (selected-channel)))
	  (forward-graph 1)
	  (if (or (not (= (selected-sound) ind1))
		  (not (= (selected-channel) 0)))
	      (snd-display ";forward from ~A 1 to ~A ~A?" ind2 (selected-sound) (selected-channel)))
	  (forward-graph 2)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 1)))
	      (snd-display ";forward from ~A 0 to ~A ~A?" ind1 (selected-sound) (selected-channel)))
	  (forward-graph 0)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 1)))
	      (snd-display ";forward 0 from ~A 1 to ~A ~A?" ind2 (selected-sound) (selected-channel)))
	  (backward-graph 2)
	  (if (or (not (= (selected-sound) ind1))
		  (not (= (selected-channel) 0)))
	      (snd-display ";backward 2 from ~A 1 to ~A ~A?" ind2 (selected-sound) (selected-channel)))
	  (backward-graph)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 1)))
	      (snd-display ";backward 2 from ~A 0 to ~A ~A?" ind1 (selected-sound) (selected-channel)))
	  (forward-graph -1)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 0)))
	      (snd-display ";forward -1 from ~A 1 to ~A ~A?" ind1 (selected-sound) (selected-channel)))
	  (backward-graph -1)
	  (if (or (not (= (selected-sound) ind2))
		  (not (= (selected-channel) 1)))
	      (snd-display ";backward -1 from ~A 0 to ~A ~A?" ind1 (selected-sound) (selected-channel)))
	  
	  (close-sound ind1)
	  (close-sound ind2))
	
	(let ((ind1 (open-sound "oboe.snd"))
	      (ind2 (open-sound "2.snd")))
	  (let ((ups1 (maxamp ind1 0))
		(ups2 (maxamp ind2 #t)))
	    (map-chan (lambda (n)
			(* n 2.0))
		      0 (frames ind1) "times 2" ind1 0)
	    (map-sound-chans (lambda (n)
			       (* n 2.0))
			     0 (frames ind2) "times 2" ind2)
	    (let ((ups3 (maxamp ind1 0))
		  (ups4 (maxamp ind2 #t)))
	      (if (fneq ups3 (* ups1 2.0))
		  (snd-display ";map-chan: ~A ~A?" ups3 (* ups1 2.0)))
	      (if (or (fneq (car ups4) (* (car ups2) 2.0))
		      (fneq (cadr ups4) (* (cadr ups2) 2.0)))
		  (snd-display ";map-sound-chans: ~A ~A?" (map (lambda (n) (* 2 n)) ups2) ups4)))
	    
	    (set! (sync ind2) #t)
	    (set! (sync ind1) #t)
	    (map-chans (lambda (n) (* n 0.5)))
	    (let ((ups3 (maxamp ind1 0))
		  (ups4 (maxamp ind2 #t)))
	      (if (fneq ups3 ups1)
		  (snd-display ";map-chans: ~A ~A?" ups3 ups1))
	      (if (or (fneq (car ups4) (car ups2))
		      (fneq (cadr ups4) (cadr ups2)))
		  (snd-display ";map-chans: ~A ~A?" ups2 ups4)))
	    (set! (sync ind1) #f)
	    
	    (let ((len-err #f))
	      (map-across-all-chans (lambda (data len)
				      (if (not (= len 3))
					  (set! len-err len)
					  (begin
					    (vector-set! data 0 (* (vector-ref data 0) 4.0))
					    (let ((chan0-sample (vector-ref data 1)))
					      (vector-set! data 1 (vector-ref data 2))
					      (vector-set! data 2 chan0-sample))))
				      data))
	      
	      (if (number? len-err)
		  (snd-display ";map-across-all-chans len: ~A?" len-err))
	      (let ((ups3 (maxamp ind1 0))
		    (ups4 (maxamp ind2 #t)))
		(if (fneq ups3 (* 4 ups1))
		    (snd-display ";map-across-all-chans 1: ~A ~A?" ups3 ups1))
		(if (or (fneq (car ups4) (cadr ups2))
			(fneq (car ups4) (cadr ups2)))
		    (snd-display ";map-across-all-chans 2: ~A ~A?" ups2 ups4))))
	    (revert-sound ind1)
	    (revert-sound ind2)
	    (map-all-chans (lambda (n) (* n 4.0)))
	    
	    (let ((ups3 (maxamp ind1 0))
		  (ups4 (maxamp ind2 0))
		  (ups5 (maxamp ind1 0 0))
		  (ups6 (maxamp ind2 0 0)))
	      (if (fneq ups3 (* 4 ups5))
		  (snd-display ";map-all-chans: ~A ~A?" ups3 ups5))
	      (if (fneq ups4 (* 4 ups6))
		  (snd-display ";map-all-chans(2): ~A ~A?" ups4 ups6)))
	    
	    (close-sound ind1)
	    (close-sound ind2)))
	
	(let* ((ind1 (open-sound "oboe.snd"))
	       (len (frames ind1))
	       (ctr 0))
	  (map-chan (lambda (n)
		      (if (= ctr 1) (set! ctr 0) (set! ctr 1))
		      (if (= ctr 0)
			  (* n 2.0)
			  #f))
		    0 (frames ind1) "cut 2" ind1 0)
	  (if (> (frames ind1) (+ (* len 2) 1))
	      (snd-display ";map-chan cut: ~A ~A?" len (frames ind1)))
	  (revert-sound ind1)
	  (set! ctr 0)
	  (map-chan (lambda (n)
		      (set! ctr (1+ ctr))
		      (if (> ctr 3)
			  #t
			  n))
		    0 (frames ind1) "cut none" ind1 0)
	  (if (> ctr 4)
	      (snd-display ";map-chan no-edit count: ~A?" ctr))
	  (revert-sound ind1)
	  (let ((v1 (make-vct 2)))
	    (map-chan (lambda (n)
			(vct-set! v1 0 n)
			(vct-set! v1 1 (* n 3))
			v1)
		      0 (frames ind1) "cut 2" ind1 0))
	  (if (> (abs (- (frames ind1) (* len 2))) 3)
	      (snd-display ";map-chan double: ~A ~A?" len (frames ind1)))
	  (close-sound ind1))
	(let* ((ind1 (open-sound "oboe.snd")))
	  (test-edpos maxamp 'maxamp (lambda () (scale-by 2.0 ind1 0)) ind1)
	  (test-edpos frames 'frames (lambda () (src-sound 2.0 1.0 ind1 0)) ind1)
	  (test-edpos 
	   (lambda* (#:optional (snd 0) (chn 0) (edpos current-edit-position)) (count-matches (lambda (n1) (> n1 .1)) 0 snd chn edpos)) 
	   'count-matches
	   (lambda () (scale-by 2.0 ind1 0)) 
	   ind1)
	  (test-edpos 
	   (lambda* (#:optional (snd 0) (chn 0) (edpos current-edit-position)) (cadr (find (lambda (n2) (> n2 .1)) 0 snd chn edpos)))
	   'find
	   (lambda () (delete-samples 0 100 ind1 0))
	   ind1)
	  (test-edpos 
	   (lambda* (#:optional (snd 0) (chn 0) (edpos current-edit-position)) 
		    (let ((samp 0)) 
		      (scan-chan (lambda (n3) 
				   (if (> n3 .1) 
				       samp 
				       (begin 
					 (set! samp (1+ samp)) 
					 #f)))
				 0 (frames snd chn) snd chn edpos)
		      samp))
	   'scan-chan
	   (lambda () (delete-samples 0 100 ind1 0))
	   ind1)
	  
	  (src-sound 2.0 1.0 ind1 0)
	  (play-and-wait 0 ind1 0 #f #f 0)
	  (play-and-wait 0 ind1 0 #f #f 1)
	  (play-and-wait 0 ind1 0 #f #f (lambda (snd chn) (edit-position snd chn)))
	  (undo 1 ind1 0)
	  (play-and-wait 0 ind1 0 #f #f 1)
	  
	  (delete-samples 0 10000 ind1 0)
	  (save-sound-as "fmv.snd" ind1 :edit-position 0)
	  (save-sound-as "fmv1.snd" ind1 :edit-position (lambda (snd chn) 1))
	  (let ((var (catch #t (lambda () (save-sound-as "fmv2.snd" ind1 :channel 1234)) (lambda args args))))
	    (if (not (eq? (car var) 'no-such-channel))
		(snd-display ";save-sound-as bad chan: ~A" var)))
	  (if (not (= (mus-sound-frames "fmv.snd") (frames ind1 0 0)))
	      (snd-display ";save-sound-as (edpos): ~A ~A?" (mus-sound-frames "fmv.snd") (frames ind1 0 0)))
	  (if (not (= (mus-sound-frames "fmv1.snd") (frames ind1 0 1)))
	      (snd-display ";save-sound-as (edpos 1): ~A ~A?" (mus-sound-frames "fmv.snd") (frames ind1 0 1)))
	  (if (= (mus-sound-frames "fmv.snd") (frames ind1 0 1))
	      (snd-display ";save-sound-as (edpos 1)(2): ~A ~A?" (mus-sound-frames "fmv.snd") (frames ind1 0 1)))
	  (let ((ind2 (open-sound "fmv.snd"))
		(ind3 (open-sound "fmv1.snd")))
	    (if (not (vequal (samples->vct 12000 10 ind1 0 #f 0) (samples->vct 12000 10 ind2 0)))
		(snd-display ";save-sound-as (edpos 3): ~A ~A?" (samples->vct 12000 10 ind1 0 #f 0) (samples->vct 12000 10 ind2 0)))
	    (if (not (vequal (samples->vct 12000 10 ind1 0 #f 1) (samples->vct 12000 10 ind3 0)))
		(snd-display ";save-sound-as (edpos 4): ~A ~A?" (samples->vct 12000 10 ind1 0 #f 1) (samples->vct 12000 10 ind3 0)))
	    (if (vequal (samples->vct 12000 10 ind2 0) (samples->vct 12000 10 ind3 0))
		(snd-display ";save-sound-as (edpos 5): ~A ~A?" (samples->vct 12000 10 ind2 0) (samples->vct 12000 10 ind3 0)))
	    (select-sound ind3)
	    (set! (comment) "hiho")
	    (if (not (string=? (comment) "hiho")) (snd-display ";set! comment no index: ~A" (comment)))
	    (close-sound ind2)
	    (close-sound ind3))
	  (delete-file "fmv.snd")
	  (delete-file "fmv1.snd")
	  
	  (test-edpos-1 (lambda (snd pos) (reverse-sound snd 0 pos)) 'reverse-sound ind1)
	  (test-edpos-1 (lambda (snd pos) (env-sound '(0 0 1 1 2 0) 0 20000 1.0 snd 0 pos)) 'env-sound ind1)
	  (test-edpos-1 (lambda (snd pos) (src-sound 0.5 1.0 snd 0 pos)) 'src-sound ind1)
	  (test-edpos-1 (lambda (snd pos) (filter-sound (make-fir-filter 6 (list->vct '(.1 .2 .3 .3 .2 .1))) 6 snd 0 pos)) 'filter-sound ind1)
	  (test-edpos-1 (lambda (snd pos) (convolve-with "pistol.snd" .5 snd 0 pos)) 'convolve-with ind1)
	  
	  (let ((ind (new-sound "fmv.snd"))
		(v (make-vct 2000))
		(ctr 0))
	    (vct-map! v (lambda ()
			  (let ((val (sin (* ctr 2.0 (/ pi 10.0)))))
			    (set! ctr (1+ ctr))
			    val)))
	    (vct->channel v 0 2000 ind 0)
	    (filter-sound '(0 0 .09 0 .1 1 .11 0 1 0) 1024)
	    (if (> (maxamp) .025) (snd-display ";filter-sound maxamp 1: ~A" (maxamp)))
	    (undo)
	    (filter-sound '(0 0 .19 0 .2 1 .21 0 1 0) 1024)  
	    (if (< (maxamp) .9) (snd-display ";filter-sound maxamp 2: ~A" (maxamp)))
	    (undo)
	    (filter-sound '(0 0 .29 0 .3 1 .31 0 1 0) 1024)  
	    (if (> (maxamp) .02) (snd-display ";filter-sound maxamp 3: ~A" (maxamp)))
	    (close-sound ind))
	  
	  (if (and (provided? 'xm) (provided? 'snd-debug))
	      (XtCallCallbacks (menu-option "Files") XmNactivateCallback (snd-global-state))
	      (view-files-dialog))
	  (set! (previous-files-sort-procedure)
		(lambda (lst)
		  (sort lst 
			(lambda (a b)
			  (> (mus-sound-duration a) (mus-sound-duration b))))))
	  (if (not (procedure? (previous-files-sort-procedure)))
	      (snd-display ";previous-files-sort-procedure: ~A" (previous-files-sort-procedure)))
	  (set! (previous-files-sort) 5)
	  (close-sound ind1)
	  (let ((val (catch #t
			    (lambda ()
			      (set! (previous-files-sort-procedure) (lambda (a b c) #f)))
			    (lambda args (car args)))))
	    (if (not (eq? val 'bad-arity))
		(snd-display ";previous-files-sort-procedure arity error: ~A" val)))
	  (do ((i 0 (1+ i)))
	      ((= i 5))
	    (set! (previous-files-sort) i))
	  (set! (previous-files-sort) 1)
	  (dismiss-all-dialogs)
	  )
	
	(with-output-to-file "sndtst" 
	  (lambda ()
	    (display (string-append "#!" home-dir "/cl/snd -b
!#
 (use-modules (ice-9 format))
 (if (= (length (script-args)) 2) ;i.e. (\"-l\" \"script\")
   (display \"usage: script file-name...\n\")
   (do ((arg (+ (script-arg) 1) (1+ arg)))
       ((= arg (length (script-args))))
     (let ((name (list-ref (script-args) arg)))
       (display (format #f \"~A: ~A~%\" name (mus-sound-comment name))))))
 (exit)
"))))
	
	(system "chmod 777 sndtst")
	(let ((val (shell "sndtst fyow.snd")))
	  (if (not (string=? val "fyow.snd: ;Written on Tue 11-May-93 at 15:55 PDT by me at localhost (NeXT) using Allegro CL and clm of 11-May-93
"))
	      (snd-display ";script: ~A?" val)
	      (delete-file "sndtst")))
	
	(with-output-to-file "sndtst" 
	  (lambda ()
	    (display (string-append "#!" home-dir "/cl/snd -b
!#
 (open-sound \"fmv.snd\")
 (scale-by 2.0)
 (save-sound)
 (exit)
"))))
	(system "chmod 777 sndtst")
	(if (file-exists? "fmv.snd") (delete-file "fmv.snd"))
	(mus-sound-prune)
	(system "cp oboe.snd fmv.snd")
	(sleep 1) ; force dates to be different
	(let* ((ind (open-sound "fmv.snd"))
	       (samps (samples->vct 5000 10))
	       (date (mus-sound-write-date "fmv.snd"))
	       (s-in #f)
	       (s-out #f))
	  (if (not (= (frames ind) (mus-sound-frames "oboe.snd")))
	      (snd-display ";cp oboe.snd fmv.snd: ~A ~A" (frames ind) (mus-sound-frames "oboe.snd")))
	  (reset-hook! update-hook)
	  (add-hook! update-hook (lambda (snd)
				   (set! s-in snd)
				   (lambda (newsnd)
				     (set! s-out newsnd))))
	  (scale-by 3.0)
	  (catch #t
		 (lambda ()
		   (add-mark 101 ind)
		   (add-mark 202 ind))
		 (lambda args (snd-display ";got error from add-mark: ~A" args)))
	  (system "sndtst") 
	  (if (= (mus-sound-write-date "fmv.snd") date)
	      (snd-display ";script didn't overwrite fmv.snd?"))
	  (set! (sync ind) 123)
	  (let ((nind (update-sound ind)))
	    (if (not (equal? (edits nind) (list 0 0)))
		(snd-display ";update-sound edits: ~A?" (edits nind)))
	    (if (not (equal? (map mark-sample (marks nind 0)) (list 101 202)))
		(snd-display ";update-sound marks: ~A" (map mark-sample (marks nind 0))))
	    (if (not (= (sync nind) 123)) (snd-display ";update-sound sync: ~A" (sync nind)))
	    (let ((nsamps (samples->vct 5000 10)))
	      (if (not (vequal samps (vct-scale! nsamps 0.5)))
		  (snd-display ";udpate-sound amps: ~A ~A?" samps nsamps)))
	    (if (or (not (number? s-in)) (not (= ind s-in))) (snd-display ";update-hook init: ~A ~A" ind s-in))
	    (if (or (not (number? s-out)) (not (= nind s-in))) (snd-display ";update-hook done: ~A ~A" nind s-out)))
	  (reset-hook! update-hook)
	  (close-sound ind)
	  (let ((ind (open-sound "fmv.snd")))
	    (add-hook! update-hook (lambda (snd) (lambda () (snd-error "oops"))))
	    (system "sndtst")
	    (let ((tag (catch #t (lambda ()
				   (update-sound ind))
			      (lambda args args))))
	      (if (not (eq? (car tag) 'bad-arity))
		  (snd-display ";bad update-hook result: ~A" tag))))
	  (close-sound (find-sound "fmv.snd"))
	  (delete-file "fmv.snd")
	  (delete-file "sndtst"))
	
	(let* ((ind (open-sound "oboe.snd"))
	       (mx (maxamp ind 0))
	       (e0 (channel-amp-envs ind 0)))
	  
	  (define (peak-env-equal? name index e diff)
	    (let* ((reader (make-sample-reader 0 index 0))
		   (e-size (vct-length (car e)))
		   (samps-per-bin (inexact->exact (ceiling (/ (frames index) e-size))))
		   (mins (car e))
		   (maxs (cadr e))
		   (max-diff 0.0))
	      (call-with-current-continuation
	       (lambda (return)
		 (do ((e-bin 0)
		      (samp 0 (1+ samp))
		      (mx -10.0)
		      (mn 10.0))
		     ((= e-bin e-size) #t)
		   (if (>= samp (inexact->exact (floor samps-per-bin)))
		       (let ((mxdiff (abs (- mx (vct-ref maxs e-bin))))
			     (mndiff (abs (- mn (vct-ref mins e-bin)))))
			 (if (> mxdiff max-diff)
			     (set! max-diff mxdiff))
			 (if (> mndiff max-diff)
			     (set! max-diff mndiff))
			 (if (or (> mxdiff diff)
				 (> mndiff diff))
			     (begin
			       (snd-display ";~A: peak-env-equal? [bin ~D of ~D]: (~,4F to ~,4F), diff: ~,5F" 
					    name
					    e-bin e-size
					    mn mx
					    (max mxdiff mndiff))
			       (return #f)))
			 (set! samp 0)
			 (set! mx -10.0)
			 (set! mn 10.0)
			 (set! e-bin (+ e-bin 1))))
		   (let ((val (next-sample reader)))
		     (if (< val mn)
			 (set! mn val))
		     (if (> val mx)
			 (set! mx val))))))
;	    (snd-display ";~A max-diff: ~A" name max-diff)
	      ))
	  
	  (if (null? e0)
	      (snd-display ";no amp env data")
	      (let ((mx1 (vct-peak (car e0)))
		    (mx2 (vct-peak (cadr e0))))
		(if (fneq mx (max mx1 mx2))
		    (snd-display ";amp env max: ~A ~A ~A" mx mx1 mx2))
		(peak-env-equal? "straight peak" ind e0 .0001)
		(scale-by 3.0)
		(let* ((e1 (channel-amp-envs ind 0 1))
		       (mx3 (vct-peak (car e1)))
		       (mx4 (vct-peak (cadr e1))))
		  (if (or (fneq (* 3.0 mx1) mx3)
			  (fneq (* 3.0 mx2) mx4))
		      (snd-display ";3.0 amp env max: ~A ~A ~A ~A" mx1 mx2 mx3 mx4))
		  (peak-env-equal? "scaled peak" ind e1 .0001))
		(if (fneq (maxamp ind 0) (* 3 mx)) 
		    (snd-display ";maxamp after scale: ~A ~A" mx (maxamp ind 0)))
		(undo)
		(set! (selection-member? #t) #f)
		(set! (selection-member? ind 0) #t)
		(set! (selection-position ind 0) 20000)
		(set! (selection-frames ind 0) 12000)
		(scale-selection-by 3.0)
		(let* ((e1 (channel-amp-envs ind 0 1))
		       (mx3 (vct-peak (car e1)))
		       (mx4 (vct-peak (cadr e1))))
		  (if (or (fneq (* 3.0 mx1) mx3)
			  (fneq (* 3.0 mx2) mx4))
		      (snd-display ";selection 3.0 amp env max: ~A ~A ~A ~A" mx1 mx2 mx3 mx4))
		  (if (fneq (maxamp ind 0) (* 3 mx)) 
		      (snd-display ";maxamp after selection scale: ~A ~A" mx (maxamp ind 0)))
		  (peak-env-equal? "selection peak" ind e1 .0001))
		(map-chan abs ind 0)
		(let* ((e1 (channel-amp-envs ind 0 2))
		       (mx3 (vct-peak (car e1)))
		       (mx4 (vct-peak (cadr e1))))
		  (if (fneq (* 3.0 mx2) mx4)
		      (snd-display ";abs selection 3.0 amp env max: ~A ~A ~A ~A" mx1 mx2 mx3 mx4))
		  (if (fneq (maxamp ind 0) (* 3 mx)) 
		      (snd-display ";maxamp after abs selection scale: ~A ~A" mx (maxamp ind 0)))
		  (if (ffneq mx3 0.03)
		      (snd-display ";abs max: ~A ~A" mx3 mx4))
		  (peak-env-equal? "map-chan peak" ind e1 .0001))
		(delete-samples 10000 5000)
		(let* ((e1 (channel-amp-envs ind 0))
		       (mx3 (vct-peak (car e1)))
		       (mx4 (vct-peak (cadr e1))))
		  (if (fneq (* 3.0 mx2) mx4)
		      (snd-display ";abs selection 3.0 amp env max: ~A ~A ~A ~A" mx1 mx2 mx3 mx4))
		  (if (fneq (maxamp ind 0) (* 3 mx)) 
		      (snd-display ";maxamp after abs selection scale: ~A ~A" mx (maxamp ind 0)))
		  (if (ffneq mx3 0.03)
		      (snd-display ";abs max: ~A ~A" mx3 mx4))
		  (peak-env-equal? "delete peak" ind e1 .0001))
		(scale-selection-by -.333)
		(let* ((e1 (channel-amp-envs ind 0 4))
		       (mx3 (vct-peak (car e1)))
		       (mx4 (vct-peak (cadr e1))))
		  (if (fneq (maxamp ind 0) mx)
		      (snd-display ";maxamp after minus abs selection scale: ~A ~A" mx (maxamp ind 0)))
		  (if (fneq (maxamp ind 0) mx3)
		      (snd-display ";mx3 maxamp after minus abs selection scale: ~A ~A" mx mx3))
		  (peak-env-equal? "scale-selection peak" ind e1 .0001))
		
		(revert-sound ind)
		(ramp-channel 0.0 1.0)
		(peak-env-equal? "ramp-channel peak" ind (channel-amp-envs ind 0 1) .001)
		(undo)
		(env-channel '(0 0 1 1 2 0))
		(peak-env-equal? "env-channel peak" ind (channel-amp-envs ind 0 1) .002)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) :scaler 0.5 :end (1- (frames))))
		(peak-env-equal? "scaled env-channel peak" ind (channel-amp-envs ind 0 1) .002)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) 0.5 :end (1- (frames))))
		(peak-env-equal? "scaled nokey env-channel peak" ind (channel-amp-envs ind 0 1) .001)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) :scaler 0.5 :offset 0.5 :end (1- (frames))))
		(peak-env-equal? "scaled and offset env-channel peak" ind (channel-amp-envs ind 0 1) .001)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 .5 3 0) :base 0.0 :end (1- (frames))))
		(peak-env-equal? "env-channel base 0.0 peak" ind (channel-amp-envs ind 0 1) .001)
		(undo)
		(xramp-channel 0.0 1.0 32.0)
		(peak-env-equal? "xramp 32.0 peak" ind (channel-amp-envs ind 0 1) .008)
		(undo)
		(xramp-channel 0.0 1.0 .032)
		(peak-env-equal? "xramp .032 peak" ind (channel-amp-envs ind 0 1) .004)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 .5 3 0) :base 10.0 :end (1- (frames))))
		(peak-env-equal? "env-channel base 10.0 peak" ind (channel-amp-envs ind 0 1) .003)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) :base .10 :end (1- (frames))))
		(peak-env-equal? "env-channel base .1 peak" ind (channel-amp-envs ind 0 1) .003)
		(undo)
		(ptree-channel (lambda (y) (* y 2)) 0 (frames) ind 0 #f #t)
		(peak-env-equal? "ptree-channel peak" ind (channel-amp-envs ind 0 1) .0001)
		(undo)
		(ramp-channel 0.0 1.0)
		(ptree-channel (lambda (y) (* y 2)) 0 (frames) ind 0 #f #t)
		(peak-env-equal? "ptree+ramp peak" ind (channel-amp-envs ind 0 2) .01)
		(undo 2)
		(xramp-channel 0.0 1.0 3.0)
		(ptree-channel (lambda (y) (* y 2)) 0 (frames) ind 0 #f #t)
		(peak-env-equal? "ptree+xramp peak" ind (channel-amp-envs ind 0 2) .004)
		(undo 2)
		(ptree-channel (lambda (y data forward)
				 (declare (y real) (data vct) (forward boolean))
				 (* y (vct-ref data 0)))
			       0 (frames) ind 0 #f #t
			       (lambda (pos dur)
				 (vct 0.5)))
		(peak-env-equal? "ptree+closure peak" ind (channel-amp-envs ind 0 1) .009)
		(undo)
		(ramp-channel 0.0 1.0)
		(ptree-channel (lambda (y data forward)
				 (declare (y real) (data vct) (forward boolean))
				 (* y (vct-ref data 0)))
			       0 (frames) ind 0 #f #t
			       (lambda (pos dur)
				 (vct 0.5)))
		(peak-env-equal? "ptree+ramp+closure peak" ind (channel-amp-envs ind 0 2) .01)
		(undo 2)
		(xramp-channel 0.0 1.0 3.0)
		(ptree-channel (lambda (y data forward)
				 (declare (y real) (data vct) (forward boolean))
				 (* y (vct-ref data 0)))
			       0 (frames) ind 0 #f #t
			       (lambda (pos dur)
				 (vct 0.5)))
		(peak-env-equal? "ptree+xramp+closure peak" ind (channel-amp-envs ind 0 2) .001)
		(undo 2)
		(insert-samples 1000 5000 (make-vct 5000 .5))
		(peak-env-equal? "insert-samples peak" ind (channel-amp-envs ind 0 1) .0001)
		(undo)
		(set! (samples 500 100) (make-vct 100 .1))
		(peak-env-equal? "set-samples peak" ind (channel-amp-envs ind 0) .0001)
		(undo)
		
		(revert-sound ind)
		(ramp-channel 0.0 1.0)
		(ramp-channel 1.0 0.0)
		(peak-env-equal? "ramp2 peak" ind (channel-amp-envs ind 0 2) .002)
		
		(revert-sound ind)
		(env-channel '(0 0 1 1))
		(env-channel '(0 0 1 1 2 0))
		(peak-env-equal? "env ramp2 peak" ind (channel-amp-envs ind 0 2) .002)
		
		(revert-sound ind)
		(env-channel '(0 0 1 1))
		(env-channel '(0 0 1 1 2 0))
		(ptree-channel (lambda (y) (* y 2.0)))
		(peak-env-equal? "ptree-ramp20 peak" ind (channel-amp-envs ind 0 3) .0001)
		
		(revert-sound ind)
		(ramp-channel 0.0 1.0 12000 5000)
		(peak-env-equal? "ramp-channel peak" ind (channel-amp-envs ind 0 1) .002)
		(undo)
		(env-channel '(0 0 1 1 2 0) 12000 5000)
		(peak-env-equal? "env-channel peak" ind (channel-amp-envs ind 0 1) .003)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 4999) 12000 5000)
		(peak-env-equal? "scaled env-channel peak" ind (channel-amp-envs ind 0 1) .004)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) 0.5 :end 4999) 12000 5000)
		(peak-env-equal? "scaled nokey env-channel peak" ind (channel-amp-envs ind 0 1) .004)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 0) :scaler 0.5 :offset 0.5 :end 4999) 12000 5000)
		(peak-env-equal? "scaled and offset env-channel peak" ind (channel-amp-envs ind 0 1) .002)
		(undo)
		(xramp-channel 0.0 1.0 32.0 2000 1000)
		(peak-env-equal? "xramp 32.0 peak (1)" ind (channel-amp-envs ind 0 1) .009)
		(undo)
		(xramp-channel 0.0 1.0 .032 2000 1000)
		(peak-env-equal? "xramp .032 peak (1)" ind (channel-amp-envs ind 0 1) .009)
		(undo)
		(env-channel (make-env '(0 0 1 1 2 .5 3 0) :base 10.0 :end 4999) 12000 5000)
		(peak-env-equal? "env-channel base 10.0 peak" ind (channel-amp-envs ind 0 1) .1)
		;; this can be way off because the envelope is not very closely sampled in this case
		(undo)
		(ptree-channel (lambda (y) (* y 2)) 2000 1000 ind 0 #f #t)
		(peak-env-equal? "ptree-channel peak" ind (channel-amp-envs ind 0 1) .0001)
		(undo)
		(ramp-channel 0.0 1.0)
		(ptree-channel (lambda (y) (* y 2)) 2000 1000 ind 0 #f #t)
		(peak-env-equal? "ptree+ramp peak" ind (channel-amp-envs ind 0 2) .001)
		(undo 2)
		(xramp-channel 0.0 1.0 3.0)
		(ptree-channel (lambda (y) (* y 2)) 2000 1000 ind 0 #f #t)
		(peak-env-equal? "ptree+xramp peak" ind (channel-amp-envs ind 0 2) .001)
		(undo 2)
		(ptree-channel (lambda (y data forward)
				 (declare (y real) (data vct) (forward boolean))
				 (* y (vct-ref data 0)))
			       2000 1000 ind 0 #f #t
			       (lambda (pos dur)
				 (vct 0.5)))
		(peak-env-equal? "ptree+closure peak" ind (channel-amp-envs ind 0 1) .0001)
		(undo)
		(ramp-channel 0.0 1.0)
		(ptree-channel (lambda (y data forward)
				 (declare (y real) (data vct) (forward boolean))
				 (* y (vct-ref data 0)))
			       2000 1000 ind 0 #f #t
			       (lambda (pos dur)
				 (vct 0.5)))
		(peak-env-equal? "ptree+ramp+closure peak" ind (channel-amp-envs ind 0 2) .001)
		
		(revert-sound ind)
		(ramp-channel 0.0 1.0)
		(ramp-channel 1.0 0.0 2000 1000)
		(peak-env-equal? "ramp2 peak" ind (channel-amp-envs ind 0 2) .002)
		
		(revert-sound ind)
		(env-channel '(0 0 1 1))
		(env-channel '(0 0 1 1 2 0) 2000 1000)
		(peak-env-equal? "env ramp2 peak" ind (channel-amp-envs ind 0 2) .002)
		
		(revert-sound ind)
		(env-channel '(0 0 1 1))
		(env-channel '(0 0 1 1 2 0))
		(ptree-channel (lambda (y) (* y 2.0)) 2000 1000)
		(peak-env-equal? "ptree-ramp21 peak" ind (channel-amp-envs ind 0 3) .002)
		
		(revert-sound ind)
		(env-channel '(0 0 1 1))
		(env-channel '(0 0 1 1 2 0))
		(env-channel '(0 0 1 1) 12000 5000)
		(peak-env-equal? "ptree-ramp3 peak" ind (channel-amp-envs ind 0 3) .01)
		
		(revert-sound ind)
		(xen-channel (lambda (y data forward)
			       (* y (list-ref data 0)))
			     0 (frames) ind 0 #f #t
			     (lambda (pos dur)
			       (list 0.5)))
		(peak-env-equal? "xen peak" ind (channel-amp-envs ind 0 1) .0001)
		
		(revert-sound ind)
		(xen-channel (lambda (y data forward)
			       (* y (list-ref data 0)))
			     2000 1000 ind 0 #f #t
			     (lambda (pos dur)
			       (list 0.5)))
		(peak-env-equal? "xen peak selection" ind (channel-amp-envs ind 0 1) .0001)
		
		))
	  (close-sound ind))
	
	(let ((ind (new-sound "test.snd")))
	  (map-chan (lambda (y) 1.0) 0 50000)
	  (ramp-channel 0.5 1.0 1000 4000)
	  (let* ((peaks (channel-amp-envs ind 0))
		 (mx (cadr peaks))
		 (mn (car peaks)))
	    (call-with-current-continuation
	     (lambda (break)
	       (if (not (continuation? break)) (snd-display ";not a continuation: ~A" break))
	       (if (continuation? abs) (snd-display ";abs is a continuation?"))
	       (if (continuation? open-sound) (snd-display ";open-sound is a continuation?"))
	       (if (continuation? 32) (snd-display ";32 is a continuation?"))
	       (if (continuation? (let ((hi 1)) (lambda () hi))) (snd-display ";closure is a continuation?"))
	       (do ((i 0 (1+ i)))
		   ((= i (- (vct-length mn) 4)))
		 (if (< (vct-ref mn i) 0.5) (begin (snd-display ";peak min: ~A ~A" (vct-ref mn i) i) (break #f)))
		 (if (< (vct-ref mx i) 0.5) (begin (snd-display ";peak max: ~A ~A" (vct-ref mx i) i) (break #f)))))))
	  (undo 2)
	  (map-chan (lambda (y) -1.0) 0 50000)
	  (ramp-channel 0.5 1.0 1000 4000)
	  (let* ((peaks (channel-amp-envs ind 0))
		 (mx (cadr peaks))
		 (mn (car peaks)))
	    (call-with-current-continuation
	     (lambda (break)
	       (do ((i 0 (1+ i)))
		   ((= i (- (vct-length mn) 4)))
		 (if (> (vct-ref mn i) -0.5) (begin (snd-display ";1 peak min: ~A ~A" (vct-ref mn i) i) (break #f)))
		 (if (> (vct-ref mx i) -0.5) (begin (snd-display ";1 peak max: ~A ~A" (vct-ref mx i) i) (break #f)))))))
	  (close-sound ind))
	
	(let ((index (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "channel tests")))
	  (define (test-channel-func func val-func init-val)
	    (let* ((len (frames index))
		   (chns (chans index))
		   (val #f))
	      (set! g-init-val init-val)
	      (do ((k 0 (1+ k)))
		  ((= k 2))
		(set! val (val-func len))
		(set! (sync index) k)
		(do ((i 0 (1+ i)))
		    ((= i chns))
		  (map-channel (lambda (n) 0.0) 0 len index i)
		  (if (scan-channel (lambda (n) (> (abs n) .001)) 0 len index i)
		      (snd-display ";init scan: ~A?" (scan-channel (lambda (n) (> (abs n) 0.001))))))
		;; now it's cleared
		(do ((i 0 (1+ i)))
		    ((= i chns))
		  (map-channel (lambda (n) g-init-val) 0 len index i)
		  (func 0 len index i)
		  (do ((j 0 (1+ j)))
		      ((= j chns))
		    (let ((vi (channel->vct 0 len index j)))
		      (if (= j i)
			  (if (not (vequal vi val))
			      (snd-display ";chan func: ~A ~A" vi val))
			  (if (scan-channel (lambda (n) (> (abs n) .001)) 0 len index j)
			      (snd-display ";chan func leaks? ~A ~A: ~A" i j (scan-channel (lambda (n) (> (abs n) 0.001)) 0 len index j))))))
		  (map-channel (lambda (n) 0.0) 0 len index i))
		(do ((i 0 (1+ i)))
		    ((= i chns))
		  (map-channel (lambda (n) g-init-val) 0 len index i)
		  (let ((ed (edit-position index i)))
		    (map-channel (lambda (n) (+ g-init-val 1.0)) 0 len index i)
		    (func 0 len index i ed)
		    (do ((j 0 (1+ j)))
			((= j chns))
		      (let ((vi (channel->vct 0 len index j)))
			(if (= j i)
			    (if (not (vequal vi val))
				(snd-display ";ed chan func: ~A ~A" vi val))
			    (if (scan-channel (lambda (n) (> (abs n) 0.001)) 0 len index j)
				(snd-display ";ed chan func leaks? ~A ~A ~A: ~A" i j ed (scan-channel (lambda (n) (> (abs n) 0.001)) 0 len index j))))))
		    (map-channel (lambda (n) 0.0) 0 len index i)))
		(let* ((beg (inexact->exact (floor (/ len 3))))
		       (dur beg)
		       (nv (val-func dur)))
		  (vct-fill! val 0.0)
		  (do ((i beg (1+ i))
		       (j 0 (1+ j)))
		      ((= j dur))
		    (vct-set! val i (vct-ref nv j)))
		  (do ((i 0 (1+ i)))
		      ((= i chns))
		    (map-channel (lambda (n) g-init-val) beg dur index i)
		    (func beg dur index i)
		    (add-mark beg index i)
		    (do ((j 0 (1+ j)))
			((= j chns))
		      (let ((vi (channel->vct 0 len index j)))
			(if (= j i)
			    (if (not (vequal vi val))
				(snd-display ";chan func n: ~A ~A" vi val))
			    (if (scan-channel (lambda (n) (> (abs n) 0.001)) 0 len index j)
				(snd-display ";dur chan func leaks? ~A ~A: ~A" i j (scan-channel (lambda (n) (> (abs n) 0.001)) 0 len index j))))))
		    (map-channel (lambda (n) 0.0) 0 len index i))))))
	  
	  (insert-silence 0 10 index 0)
	  (insert-silence 0 10 index 1)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (clm-channel (make-env :envelope '(0 0 1 1) :end (1- dur)) beg dur index chan edpos))
			     (lambda (dur)
			       (let ((e (make-env :envelope '(0 0 1 1) :end (1- dur)))
				     (v (make-vct dur)))
				 (do ((i 0 (1+ i)))
				     ((= i dur))
				   (vct-set! v i (env e)))
				 v))
			     0.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (clm-channel (make-oscil :frequency 0.0 :initial-phase (/ pi 2)) beg dur index chan edpos))
			     (lambda (dur)
			       (let ((v (make-vct dur)))
				 (vct-fill! v 1.0)
				 v))
			     0.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (scale-channel 0.5 beg dur index chan edpos))
			     (lambda (dur)
			       (let ((v (make-vct dur)))
				 (vct-fill! v 0.5)
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (env-channel (make-env :envelope '(0 0 1 1) :end (1- dur)) beg dur index chan edpos))
			     (lambda (dur)
			       (let ((e (make-env :envelope '(0 0 1 1) :end (1- dur)))
				     (v (make-vct dur)))
				 (do ((i 0 (1+ i)))
				     ((= i dur))
				   (vct-set! v i (env e)))
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (env-channel '(0 0 1 1) beg dur index chan edpos))
			     (lambda (dur)
			       (let ((e (make-env :envelope '(0 0 1 1) :end (1- dur)))
				     (v (make-vct dur)))
				 (do ((i 0 (1+ i)))
				     ((= i dur))
				   (vct-set! v i (env e)))
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (let ((v (make-vct dur)))
					(vct-fill! v -1.0)
					(vct->channel v beg dur index chan)))
			     (lambda (dur)
			       (let ((v (make-vct dur)))
				 (vct-fill! v -1.0)
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (delete-samples beg dur index chan edpos)
				      (pad-channel beg dur index chan edpos))
			     (lambda (dur)
			       (make-vct dur))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (let ((v (make-vct dur)))
					(vct-fill! v -1.0)
					(delete-samples beg dur index chan edpos)
					(insert-samples beg dur v index chan edpos)))
			     (lambda (dur)
			       (let ((v (make-vct dur)))
				 (vct-fill! v -1.0)
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (let ((v (make-vct dur)))
					(vct-fill! v -1.0)
					(set! (samples beg dur index chan #f "test-channel" 0 edpos) v)))
			     (lambda (dur)
			       (let ((v (make-vct dur)))
				 (vct-fill! v -1.0)
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (env-channel (make-env :envelope '(0 0 1 1) :end (1- dur)) beg dur index chan edpos)
				      (reverse-channel beg dur index chan))
			     (lambda (dur)
			       (let ((e (make-env :envelope '(0 1 1 0) :end (1- dur)))
				     (v (make-vct dur)))
				 (do ((i 0 (1+ i)))
				     ((= i dur))
				   (vct-set! v i (env e)))
				 v))
			     1.0)
	  
	  (test-channel-func (lambda* (beg dur index chan #:optional edpos)
				      (env-channel (make-env :envelope '(0 0 1 1) :end (1- dur)) beg dur index chan edpos)
				      (set! (sample (+ beg dur) index chan) 1.0)
				      (smooth-channel beg dur index chan)
				      (if (not (= beg 0))
					  (set! (sample (+ beg dur) index chan) 0.0)))
			     (lambda (dur)
			       (let ((v (make-vct dur)))
				 (do ((i 0 (1+ i)))
				     ((= i dur))
				   (vct-set! v i (+ 0.5 (* 0.5 (cos (+ pi (/ (* pi i) dur)))))))
				 v))
			     1.0)
	  (if (not (equal? (edits index) (list 276 0)))
	      (snd-display ";channel edits: ~A" (edits index)))
	  (let ((old-max (maxamp index #t))
		(regdata (map (lambda (n)
				(region-samples->vct 0 10 n))
			      (regions)))
		(old-pos0 (edit-position index 0))
		(old-pos1 (edit-position index 1))
		(old-reglen (map region-frames (regions)))
		(s61-files '()))
	    (add-hook! save-state-hook
		       (lambda (file)
			 (set! s61-files (cons file s61-files))
			 #f))
	    (save-state "s61.scm")
	    (close-sound index)
	    (for-each forget-region (regions))
	    (load "s61.scm")
	    (if (not (equal? old-reglen (map region-frames (regions))))
		(snd-display ";region-frames after save: ~A ~A" old-reglen (map region-frames (regions))))
	    (for-each (lambda (n data)
			(if (not (vequal data (region-samples->vct 0 10 n)))
			    (snd-display ";region after save ~A: ~A ~A" n data (region-samples->vct 0 10 n))))
		      (regions)
		      regdata)
	    (set! index (find-sound "fmv.snd"))
	    (if (not (equal? (maxamp index #t) old-max))
		(snd-display ";maxes: ~A ~A" (maxamp index #t) old-max))
	    (if (not (equal? (edits index) (list 276 0))) ; extend adds 2
		(snd-display ";saved channel edits: ~A" (edits index)))
	    
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (let ((pos (random (car (edits index)))))
		(scale-channel (random 2.0) (random 5) (random 5) index 0 pos)
		(set! (edit-position index) (inexact->exact (floor (* (car (edits index)) .7))))))
	    
	    (close-sound index)
	    (for-each
	     (lambda (n)
	       (forget-region n))
	     (regions))
	    (for-each
	     (lambda (file)
	       (if (file-exists? file) 
		   (delete-file file)))
	     s61-files)
	    (delete-file "s61.scm")
	    ))
	
	(let ((index (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "channel tests"))
	      (v (make-vct 10))
	      (sw (sinc-width)))
	  (set! (sinc-width) 10)
	  (vct-set! v 0 1.0)
	  (vct->channel v 0 10 index 0)
	  (src-channel 0.5 0 10 index 0)
	  (let ((v (make-vct 10))
		(s (make-src :srate 0.5
			     :input (let ((val 1.0))
				      (lambda (dir)
					(let ((rtn val))
					  (set! val 0.0)
					  rtn))))))
	    (vct-set! v 0 (src s))
	    (do ((i 1 (1+ i)))
		((= i 10))
	      (vct-set! v i (src s)))
	    (if (not (vequal v (channel->vct 0 10 index 0)))
		(snd-display ";src-channel: ~A ~A" v (channel->vct 0 10 index 0)))
	    (if (not (vequal (make-vct 10) (channel->vct 0 10 index 1)))
		(snd-display ";src-channel leaks: ~A" (channel->vct 0 10 index 1)))
	    (let ((tag (catch #t (lambda () (src s 1.0 (lambda (a b) a))) (lambda args (car args)))))
	      (if (not (eq? tag 'bad-arity)) 
		  (snd-display ";src bad func: ~A" tag))))
	  (revert-sound index)
	  (vct->channel v 0 10 index 1)
	  (vct->channel v 10 10 index 1)
	  (src-channel (make-env :envelope '(1 1 2 2) :end 20) 0 20 index 1)
	  (if (not (vequal (channel->vct 0 10 index 1) (vct 1.000 -0.000 -0.048 0.068 -0.059 0.022 0.030 -0.100 0.273 0.606)))
	      (snd-display ";src-channel env: ~A" (channel->vct 0 10 index 1)))
	  (if (not (vequal (make-vct 10) (channel->vct 0 10 index 0)))
	      (snd-display ";src-channel env leaks: ~A" (channel->vct 0 10 index 0)))
	  (set! (sinc-width) sw)
	  (close-sound index))
	
	(let* ((ind (open-sound "oboe.snd"))
	       (rid0 (make-region 2000 2020 ind 0))
	       (rid0-data (region2vct rid0 0 20)))
	  (scale-sound-by 2.0)
	  (play-region rid0 #t)
	  (let ((nv (region2vct rid0 0 20)))
	    (if (not (vequal rid0-data nv)) (snd-display ";deferred region after scaling:~%  ~A~%  ~A" rid0-data nv)))
	  (let ((nv (region-to-vct rid0 0 20)))
	    (if (not (vequal rid0-data nv)) (snd-display ";deferred region after scaling (rs):~%  ~A~%  ~A" rid0-data nv)))
	  (undo)
	  (scale-by 4.0)
	  (play-region rid0 #t)
	  (let ((nv (region2vct rid0 0 20)))
	    (if (not (vequal rid0-data nv)) (snd-display ";file region after scaling:~%  ~A~%  ~A" rid0-data nv)))
	  (let ((nv (region-to-vct rid0 0 20)))
	    (if (not (vequal rid0-data nv)) (snd-display ";file region after scaling (rs):~%  ~A~%  ~A" rid0-data nv)))
	  (let* ((rid1 (make-region 2000 2020 ind 0))
		 (rid1-data (region2vct rid1 0 20)))
	    (scale-to .5)
	    (let ((nv (region2vct rid1 0 20)))
	      (if (not (vequal rid1-data nv)) (snd-display ";deferred region after scale-to:~%  ~A~%  ~A" rid1-data nv)))
	    (close-sound ind)
	    (play-region rid0 #t)
	    (play-region rid1 #t)
	    (let ((nv (region2vct rid1 0 20)))
	      (if (not (vequal rid1-data nv)) (snd-display ";deferred region after close:~%  ~A~%  ~A" rid1-data nv)))
	    (let ((nv (region2vct rid0 0 20)))
	      (if (not (vequal rid0-data nv)) (snd-display ";file region after close:~%  ~A~%  ~A" rid0-data nv))))
	  
	  (for-each
	   (lambda (s1 l1 s2 l2)
	     (set! ind (open-sound "2.snd"))
	     (set! (selection-member? #t) #f)
	     (set! (selection-member? ind 0) #t)
	     (set! (selection-position ind 0) s1)
	     (set! (selection-frames ind 0) l1)
	     (set! (selection-member? ind 1) #t)
	     (set! (selection-position ind 1) s2)
	     (set! (selection-frames ind 1) l2)
	     (let* ((rid2 (make-region))
		    (rid20-data (region2vct rid2 0 l1))
		    (rid21-data (region2vct rid2 1 l2)))
	       (if (not (= (region-chans rid2) 2)) (snd-display ";region-chans of sync'd sound: ~A?" (region-chans rid2)))
	       (swap-channels ind 0 ind 1)
	       (let ((nv (region2vct rid2 0 l1)))
		 (if (not (vequal rid20-data nv)) (snd-display ";deferred region after scaling (20):~%  ~A~%  ~A" rid20-data nv)))
	       (let ((nv (region-to-vct rid2 0 l1)))
		 (if (not (vequal rid20-data nv)) (snd-display ";deferred region after scaling (20 rs):~%  ~A~%  ~A" rid20-data nv)))
	       (let ((nv (region2vct rid2 1 l2)))
		 (if (not (vequal rid21-data nv)) (snd-display ";deferred region after scaling (21):~%  ~A~%  ~A" rid21-data nv)))
	       (let ((nv (region-to-vct rid2 1 l2)))
		 (if (not (vequal rid21-data nv)) (snd-display ";deferred region after scaling (21 rs):~%  ~A~%  ~A" rid21-data nv)))
	       (close-sound ind)
	       (let ((nv (region2vct rid2 0 l1)))
		 (if (not (vequal rid20-data nv)) (snd-display ";deferred region after scaling (20):~%  ~A~%  ~A" rid20-data nv)))
	       (let ((nv (region-to-vct rid2 0 l1)))
		 (if (not (vequal rid20-data nv)) (snd-display ";deferred region after scaling (20 rs):~%  ~A~%  ~A" rid20-data nv)))
	       (let ((nv (region2vct rid2 1 l2)))
		 (if (not (vequal rid21-data nv)) (snd-display ";deferred region after scaling (21):~%  ~A~%  ~A" rid21-data nv)))
	       (let ((nv (region-to-vct rid2 1 l2)))
		 (if (not (vequal rid21-data nv)) (snd-display ";deferred region after scaling (21 rs):~%  ~A~%  ~A" rid21-data nv)))
	       ))
	   (list 2000 2000 2000 0 2000 0 2000)
	   (list 20 10 20 20 20 10 20)
	   (list 2000 2000 2000 2000 0 2000 0)
	   (list 20 20 10 20 20 20 10))
	  
	  (let ((ind (open-sound "obtest.snd")))
	    (set! (read-only ind) #t)
	    (delete-samples 0 1000 ind 0)
	    (let ((val (save-sound ind)))
	      (if val (snd-display ";save-sound read-only: ~A" val)))
	    (if (not (equal? (edits ind) (list 1 0))) (snd-display ";read-only ignored? ~A" (edits ind)))
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (not (string=? str "can't write obtest.snd (it is read-only)"))
		      (snd-display ";read-only report-in-minibuffer: ~A?" str))
		  (set! str (widget-text (list-ref (sound-widgets ind) 4))) ; listener
		  (if (not (string? str))
		      (snd-display ";widget-text of listener?: ~A?" str))))
	    (set! (read-only ind) #f)
	    (revert-sound ind)
	    (save-sound ind)
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (not (string=? str "(no changes need to be saved)"))
		      (snd-display ";save unneeded report-in-minibuffer: ~A?" str))))
	    (key (char->integer #\j) 4)
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (not (string=? str "no marks"))
		      (snd-display ";C-j w/o marks: ~A?" str))))
	    (key (char->integer #\-) 4)
	    (key (char->integer #\j) 4)
	    (key (char->integer #\j) 4)
	    (key (char->integer #\x) 4)
	    (key (char->integer #\c) 0)
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (widget-text (cadr (main-widgets))) 
		      (snd-display ";widget-text of non-text widget: ~A" (widget-text (cadr (main-widget)))))
		  (set! (widget-text (list-ref (channel-widgets ind 0) 2)) "F")
		  (if (not (string=? (widget-text (list-ref (channel-widgets ind 0) 2)) "F"))
		      (snd-display ";set button label to F: ~A" (widget-text (list-ref (channel-widgets ind 0) 2)) "F"))
		  (if (not (string=? str "no marks"))
		      (snd-display ";C-x c w/o marks: ~A?" str))))
	    (add-mark 123)
	    (key (char->integer #\u) 4)
	    (key (char->integer #\6) 4)
	    (key (char->integer #\j) 4)
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (not (string=? str "no such mark"))
		      (snd-display ";C-u 6 C-j: ~A?" str))))
	    (key (char->integer #\u) 4)
	    (key (char->integer #\6) 4)
	    (key (char->integer #\x) 4)
	    (key (char->integer #\c) 0)
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (not (string=? str "no such mark"))
		      (snd-display ";C-u 6 C-x c: ~A?" str))))
	    (close-sound ind))
	  
	  (let ((ind (view-sound "obtest.snd")))
	    (delete-samples 0 1000 ind 0)
	    (save-sound ind)
	    (if (not (equal? (edits ind) (list 1 0))) (snd-display ";view read-only ignored? ~A" (edits ind)))
	    (if with-gui
		(let ((str (widget-text (list-ref (sound-widgets ind) 3))))
		  (if (not (string=? str "can't write obtest.snd (it is read-only)"))
		      (snd-display ";view read-only report-in-minibuffer: ~A?" str))))
	    (close-sound ind))
	  
	  (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1)))
	    (insert-silence 0 150000)
	    (map-channel (lambda (y) 0.5))
	    (env-sound '(0 0 1 1 2 0))
	    (fp 1.0 0.3 20)
	    (let ((old-curse (cursor-follows-play)))
	      (set! (cursor-follows-play) #t)
	      (play-and-wait)
	      (set! (cursor-follows-play) old-curse))
	    (close-sound ind))
	  (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1)))
	    (for-each
	     (lambda (dur)
	       (insert-silence 0 dur)
	       (map-channel (lambda (y) 1.0))
	       (env-sound '(0 0 1 1 2 0))
	       (let ((reader (make-sample-reader (1- (frames)) ind 0 -1)))
		 (if (not (= (sample-reader-position reader) (1- (frames)))) (snd-display ";sample-reader-position: ~A" (sample-reader-position reader)))
		 (map-channel (lambda (y) (read-sample reader))))
	       (scan-channel (let ((pos 0)
				   (e (make-env '(0 0 1 1 2 0) :end dur)))
			       (lambda (y)
				 (let ((val (env e)))
				   (if (fneq val y) 
				       (begin
					 (display (format #f "~%;trouble in reverse read at ~D ~A ~A" pos val y))
					 #t)
				       (begin
					 (set! pos (1+ pos))
					 #f))))))
	       (revert-sound))
	     (list 150 1500 150000))
	    (close-sound ind))
	  (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1)))
	    (insert-silence 0 1000)
	    (map-channel (lambda (y) 1.0))
	    (env-sound '(0 0 1 1 2 0))
	    (scale-channel 0.0 100 200)
	    (let ((reader (make-sample-reader (1- (frames)) ind 0 -1)))
	      (map-channel (lambda (y) (read-sample reader))))
	    (scan-channel (let ((pos 0)
				(e (make-env '(0 0 1 1 2 0) :end 1000)))
			    (lambda (y)
			      (let ((val (env e)))
				(if (or (and (or (> pos 900) (<= pos 700))
					     (fneq val y))
					(and (> pos 700) (<= pos 900)
					     (fneq y 0.0)))
				    (begin
				      (display (format #f "~%;trouble in reverse read 2 at ~D ~A ~A" pos val y))
				      #t)
				    (begin
				      (set! pos (1+ pos))
				      #f))))))
	    (close-sound ind))
	  (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1)))
	    (insert-silence 0 150000)
	    (map-channel (lambda (y) 1.0))
	    (let ((edpos (edit-position)))
	      (do ((i 0 (1+ i)))
		  ((= i 7))
		(if (= i 5)
		    (scale-channel 0.5 1000 12345))
		(env-sound '(0 0 1 1 2.5 0 3 1 4 0))
		(if (= i 1)
		    (delete-samples 50 100)
		    (if (= i 2)
			(insert-samples 300 100 (vct-fill! (make-vct 100) 0.5))
			(if (= i 3)
			    (scale-channel 0.0 1000 1000)
			    (if (= i 4)
				(vct->channel (vct-fill! (make-vct 100) .5) 500 100)
				(if (= i 6)
				    (env-sound '(0 1 1 0) 10000 2000))))))
		(let ((reader (make-sample-reader (1- (frames)) ind 0 -1)))
		  (map-channel (lambda (y) (read-sample reader))))
		(let ((reader (make-sample-reader (1- (frames)) ind 0 -1)))
		  (map-channel (lambda (y) (read-sample reader))))
		(scan-channel (let ((old-reader (make-sample-reader 0 ind 0 1 (- (edit-position ind 0) 2)))
				    (pos 0))
				(lambda (y)
				  (let ((val (read-sample old-reader)))
				    (if (fneq y val)
					(begin
					  (display (format #f "~%;trouble in reverse (~D) read at ~D ~A ~A" i pos val y))
					  #t)
					(begin
					  (set! pos (1+ pos))
					  #f))))))
		(set! (edit-position ind 0) edpos)))
	    (close-sound ind))
	  (let ((reader #f)
		(last-proc #f))
	    (define (scan-again)
	      (if (sample-reader-at-end? reader)
		  #f
		  (let ((val (last-proc (reader))))
		    (if val 
			(list val (1- (sample-reader-position reader)))
			(scan-again)))))
	    (define* (my-scan-chan #:optional proc)
	      (if proc 
		  (begin
		    (set! last-proc proc)
		    (set! reader (make-sample-reader 0))))
	      (scan-again))
	    (let ((ind (open-sound "oboe.snd"))
		  (val #f))
	      (let ((samp (sample 1000)))
		(set! (cursor ind 0) 1000)
		(if (fneq (sample) samp)
		    (snd-display ";sample no args: ~A ~A" (sample) samp)))
	      (set! val (my-scan-chan (lambda (y) (> y .1))))
	      (if (not (equal? val (list #t 4423)))
		  (snd-display ";my-scan-chan: ~A" val))
	      (set! val (scan-again))
	      (if (not (equal? val (list #t 4463)))
		  (snd-display ";scan-again: ~A" val))
	      (let ((val (find (lambda (y)
				 (let ((val (find (lambda (y) (> y .1)))))
				   val)))))
		(if (not (equal? val (list (list #t 4423) 0)))
		    (snd-display ";find twice: ~A" val)))
	      (let ((val (find (lambda (y)
				 (count-matches (lambda (y) (> y .1)))))))
		(if (not (equal? val (list 2851 0)))
		    (snd-display ";find+count: ~A" val)))
	      (set! (cursor) 1000)
	      (set! (sample) .5)
	      (if (fneq (sample 1000) .5)
		  (snd-display ";set sample no arg: ~A ~A" (sample 1000) (sample 0)))
	      (close-sound ind)))
	  
	  ;; edit-menu.scm tests
	  (let ((ind (view-sound "oboe.snd")))
	    (make-selection 1000 1999 ind 0)
	    (let ((newsnd (selection->new)))
	      (if (not (sound? newsnd)) (snd-display ";selection->new -> ~A" newsnd))
	      (if (not (= (frames newsnd 0) 1000)) (snd-display ";selection->new frames: ~A" (frames newsnd 0)))
	      (if (not (equal? (edits ind 0) (list 0 0))) (snd-display ";selection->new edited original? ~A" (edits ind 0)))
	      (let ((newfile (file-name newsnd)))
		(close-sound newsnd)
		(delete-file newfile)
		(mus-sound-forget newfile)))
	    (make-selection 1000 1999 ind 0)
	    (let ((newsnd (cut-selection->new)))
	      (if (not (sound? newsnd)) (snd-display ";cut-selection->new -> ~A" newsnd))
	      (if (not (= (frames newsnd 0) 1000)) (snd-display ";cut-selection->new frames: ~A" (frames newsnd 0)))
	      (if (not (equal? (edits ind 0) (list 1 0))) (snd-display ";cut-selection->new did not edit original? ~A" (edits ind 0)))
	      (if (not (= (frames ind 0) (- (frames ind 0 0) 1000))) 
		  (snd-display ";cut-selection->new cut: ~A ~A" (frames ind 0) (- (frames ind 0 0) 1000)))
	      (undo 1 ind 0)
	      (let ((newfile (file-name newsnd)))
		(close-sound newsnd)
		(delete-file newfile)
		(mus-sound-forget newfile)))
	    (make-selection 1000 1999 ind 0)
	    (append-selection)
	    (if (not (= (frames ind 0) (+ (frames ind 0 0) 1000)))
		(snd-display ";append-selection: ~A ~A" (frames ind 0) (frames ind 0 0)))
	    (append-sound "oboe.snd")
	    (if (not (= (frames ind 0) (+ (* 2 (frames ind 0 0)) 1000)))
		(snd-display ";append-sound: ~A ~A" (frames ind 0) (frames ind 0 0)))
	    (revert-sound ind)
	    (let ((m1 (add-mark 1000))
		  (m2 (add-mark 12000)))
	      (trim-front)
	      (if (not (equal? (edits ind 0) (list 1 0))) (snd-display ";time-front did not edit original? ~A" (edits ind 0)))
	      (if (not (= (frames ind 0) (- (frames ind 0 0) 1000))) 
		  (snd-display ";trim-front: ~A ~A" (frames ind 0) (- (frames ind 0 0) 1000)))
	      (if (not (= (mark-sample m2) 11000)) (snd-display ";trim-front m2: ~A" (mark-sample m2)))
	      (undo 1 ind 0)
	      (trim-back)
	      (if (not (equal? (edits ind 0) (list 1 0))) (snd-display ";time-back did not edit original? ~A" (edits ind 0)))
	      (if (not (= (frames ind 0) 12001)) (snd-display ";trim-back: ~A" (frames ind 0)))
	      (if (not (= (mark-sample m1) 1000)) (snd-display ";trim-back m1: ~A" (mark-sample m1)))
	      (undo 1 ind 0)
	      (add-mark 22000)
	      (crop)
	      (if (not (equal? (edits ind 0) (list 1 0))) (snd-display ";crop did not edit original? ~A" (edits ind 0)))
	      (if (not (= (frames ind 0) 21001)) (snd-display ";crop: ~A" (frames ind 0)))
	      (undo 1 ind 0)
	      (close-sound ind)))
	  
	  (let ((ind (new-sound "test.snd")))
	    (map-chan (lambda (y) 1.0) 0 1000)
	    (env-channel (make-env '(0 1 1 1) :scaler .5 :end 1000))
	    (check-maxamp ind .5 "simple scaler")
	    (check-env-vals "simple scaler" (make-env '(0 1 1 1) :scaler .5 :end 1000))
	    (if (= (edit-position) 2)
		(undo)
		(snd-display ";env+scl was no-op"))
	    (env-channel (make-env '(0 1 1 1) :offset .5 :end 1000))
	    (check-maxamp ind 1.5 "simple offset")
	    (check-env-vals "simple offset" (make-env '(0 1 1 1) :offset .5 :end 1000))
	    (if (= (edit-position) 2)
		(undo)
		(snd-display ";env+offset was no-op"))
	    (env-channel (make-env '(0 0 1 1 2 0) :offset .5 :scaler 2.0 :end 1000))
	    (check-maxamp ind 2.5 "off+scl")
	    (check-env-vals "off+scl" (make-env '(0 0 1 1 2 0) :offset .5 :scaler 2.0 :end 1000))
	    (undo)
	    (env-channel (make-env '(0 -1 1 0 2 -1) :offset .5 :scaler 2.0 :end 1000))
	    (check-maxamp ind 1.5 "off+scl #2") ; 1.5 because abs max
	    (let ((mx -12.0))
	      (scan-chan (lambda (y) 
			   (if (> y mx) 
			       (set! mx y))
			   #f))
	      (if (fneq mx 0.5) (snd-display ";non abs max: ~A (correct: 0.5)" mx)))
	    (check-env-vals "off+scl #2" (make-env '(0 -1 1 0 2 -1) :offset .5 :scaler 2.0 :end 1000))
	    (undo)
	    (env-sound '(0 .5 1 .75 2 .25) 0 (frames) 32.0)
	    (check-maxamp ind 0.75 "xramp")
	    (check-env-vals "xramp" (make-env '(0 .5 1 .75 2 .25) :base 32.0 :end 1000))
	    
	    (close-sound ind))
	  
	  (let ((hlb (make-hilbert-transform 8))
		(data (make-vct 20)))
	    (do ((i 0 (1+ i)))
		((= i 20))
	      (vct-set! data i (hilbert-transform hlb (if (= i 0) 1.0 0.0))))
	    (if (not (vequal data (vct 0.0 -0.010 0.0 -0.046 0.0 -0.152 0.0 -0.614 0.0 0.614 0.0 0.152 0.0 0.046 0.0 0.010 0.0 0.0 0.0 0.0)))
		(snd-display ";hilbert-transform impulse response: ~A" data)))

	  (let ((ind (new-sound "test.snd")))
	    (pad-channel 0 1000)
	    (set! (sample 100) 1.0)
	    (let ((h (make-hilbert-transform 100)))
	      (map-channel (lambda (y) (hilbert-transform h y)))
	      (map-channel (lambda (y) (hilbert-transform h y)))
	      (map-channel (lambda (y) (hilbert-transform h y)))
	      (map-channel (lambda (y) (hilbert-transform h y)))
	      ;; now ideally we'd be back to an impulse
	      (if (> (abs (- (sample 500) .98)) .01)
		  (snd-display ";hilbert impulse: ~A" (sample 500)))
	      (set! (sample 500) 0.0)
	      (if (> (maxamp ind 0) .02)
		  (snd-display ";hilbert sidelobes: ~A" (maxamp ind 0)))
	      (revert-sound))
	    (pad-channel 0 1000)
	    (set! (sample 100) 1.0)
	    (let ((lo (make-lowpass (* .1 pi) 20))
		  (hi (make-highpass (* .1 pi) 20)))
	      (map-channel (lambda (y) (+ (lowpass lo y) (highpass hi y))))
	      (if (fneq (sample 120) 1.0)
		  (snd-display ";lowpass+highpass impulse: ~A" (sample 120)))
	      (set! (sample 120) 0.0)
	      (if (fneq (maxamp ind 0) 0.0)
		  (snd-display ";lowpass+highpass sidelobes: ~A" (maxamp ind 0))))
	    (undo 2)
	    (let ((lo (make-bandpass (* .1 pi) (* .2 pi) 20))
		  (hi (make-bandstop (* .1 pi) (* .2 pi) 20)))
	      (map-channel (lambda (y) (+ (bandpass lo y) (bandstop hi y))))
	      (if (fneq (sample 120) 1.0)
		  (snd-display ";bandpass+bandstop impulse: ~A" (sample 120)))
	      (set! (sample 120) 0.0)
	      (if (fneq (maxamp ind 0) 0.0)
		  (snd-display ";bandpass+bandstop sidelobes: ~A" (maxamp ind 0))))
	    (close-sound ind))
	  
	  (let ((ind (new-sound "test.snd")))
	    (map-channel (lambda (y) (- 1.0 (random 2.0))) 0 10000)
	    (let ((f2 (make-bandpass-2 (* .12 pi) (* .15 pi) (* .22 pi) (* .25 pi) 100)))
	      (map-channel (lambda (y) (fir-filter f2 y)))
	      (let ((data (channel->vct)))
		(undo)
		(let* ((f1 (make-bandpass (* .12 pi) (* .15 pi) 100))
		       (f2 (make-bandpass (* .22 pi) (* .25 pi) 100)))
		  (map-channel (lambda (y) (+ (fir-filter f1 y) (fir-filter f2 y))))
		  (let ((data1 (channel->vct)))
		    (vct-subtract! data data1)
		    (if (> (vct-peak data) .00001)
			(snd-display ";fir-filter 2: ~A" (vct-peak data))))
		  (undo))))
	    (close-sound ind))
	  
	  (let ((ind (new-sound  "test.snd" mus-next mus-bfloat 22050 1 "ramp re-order tests" 100)))
	    (map-channel (lambda (y) 1.0))
	    (for-each
	     (lambda (lst)
	       (let ((name (car lst))
		     (try-scale (cadr lst))
		     (f1 (caddr lst))
		     (f2 (cadddr lst))
		     (edpos (edit-position ind 0)))
		 (f1)
		 (let ((v1 (channel->vct 0 100 ind 0)))
		   (set! (edit-position ind 0) edpos)
		   (f2)
		   (let ((v2 (channel->vct 0 100 ind 0)))
		     (if (not (vequal v1 v2))
			 (snd-display ";env reordering test ~A: ~A ~A" name v1 v2))
		     (set! (edit-position ind 0) edpos)))
		 (if try-scale
		     (begin
		       (scale-by 2.0)
		       (f1)
		       (let ((v1 (channel->vct 0 100 ind 0)))
			 (set! (edit-position ind 0) edpos)
			 (f2)
			 (scale-by 2.0)
			 (let ((v2 (channel->vct 0 100 ind 0)))
			   (if (not (vequal v1 v2))
			       (snd-display ";scaled (2) env reordering test ~A: ~A ~A" name v1 v2))
			   (set! (edit-position ind 0) edpos)))
		       (f1)
		       (scale-by .5)
		       (let ((v1 (channel->vct 0 100 ind 0)))
			 (set! (edit-position ind 0) edpos)
			 (scale-by .5)
			 (f2)
			 (let ((v2 (channel->vct 0 100 ind 0)))
			   (if (not (vequal v1 v2))
			       (snd-display ";scaled (.5) env reordering test ~A: ~A ~A" name v1 v2))
			   (set! (edit-position ind 0) edpos)))))))
	     
	     (list (list "ramp-xramp" #t
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1) 0 100 2.0))
			 (lambda ()
			   (env-sound '(0 0 1 1) 0 100 2.0)
			   (env-sound '(0 0 1 1 2 0))))
		   (list "ramp2-xramp (1)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1 3 0))
			   (env-sound '(0 0 1 1) 0 100 2.0))
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1) 0 100 2.0)
			   (env-sound '(0 0 1 1 3 0))))
		   (list "ramp2-xramp (2)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 3 0) 0 100 2.0))
			 (lambda ()
			   (env-sound '(0 0 1 1 3 0) 0 100 2.0)
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1))))
		   (list "xramp2-ramp (1)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 3 0) 0 100 3.0))
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1 3 0) 0 100 3.0)
			   (env-sound '(0 0 1 1))))
		   (list "xramp2-ramp (2)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1 3 0))
			   (env-sound '(0 0 1 1) 0 100 3.0))
			 (lambda ()
			   (env-sound '(0 0 1 1 3 0))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1) 0 100 3.0)))
		   (list "ptree-ramp" #t
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (ptree-channel (lambda (y) (* y 2.0))))
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1))))
		   (list "ptree-xramp" #t
			 (lambda ()
			   (env-sound '(0 0 1 1) 0 100 2.0)
			   (ptree-channel (lambda (y) (* y 2.0))))
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1) 0 100 2.0)))
		   (list "ptree-ramp2 (1)" #t
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0)))
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1 2 0))))
		   (list "ptree-ramp2 (2)" #t
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0)))
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0))
			   (ptree-channel (lambda (y) (* y 2.0)))))
		   (list "xramp-ptree-ramp (1)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1)))
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1))))
		   (list "xramp-ptree-ramp (2)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0))
			 (lambda ()
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1))
			   (ptree-channel (lambda (y) (* y 2.0)))))
		   (list "xramp-ptree-ramp (3)" #t
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0))
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (ptree-channel (lambda (y) (* y 2.0)))))
		   (list "xramp-ptree-ramp-zero" #f
			 (lambda ()
			   (scale-by 0.0)
			   (ptree-channel (lambda (y) 1.0))
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0))
			 (lambda ()
			   (scale-by 0.0)
			   (ptree-channel (lambda (y) 1.0))
			   (env-sound '(0 0 1 1 2 0) 0 100 2.0)
			   (env-sound '(0 0 1 1))))
		   (list "ptree-ramp3 (1)" #t
			 (lambda ()
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1 3 0)))
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1 3 0))))
		   (list "ptree-ramp3 (2)" #t
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0))
			   (ptree-channel (lambda (y) (* y 2.0)))
			   (env-sound '(0 0 1 1 3 0)))
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1 3 0))
			   (ptree-channel (lambda (y) (* y 2.0)))))
		   (list "ramp4" #t
			 (lambda ()
			   (env-sound '(0 0 1 1))
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1 3 0))
			   (env-sound '(0 0 1 1 4 0)))
			 (lambda ()
			   (env-sound '(0 0 1 1 4 0))
			   (env-sound '(0 0 1 1 2 0))
			   (env-sound '(0 0 1 1 3 0))
			   (env-sound '(0 0 1 1))))
		   
		   ))
	    (close-sound ind))

	  (let ((ind (new-sound  "test.snd" mus-next mus-bfloat 22050 1 "ramp re-order tests" 100))
		(oldopt (optimization)))
	    
	    (define (reversed-channel->vct start dur snd chn)
	      (let* ((len (frames snd chn))
		     (data (make-vct dur))
		     (sf (make-sample-reader (+ start dur -1) snd chn -1)))
		(do ((i (+ start dur -1) (1- i)))
		    ((< i start))
		  (vct-set! data i (read-sample sf)))
		data))
	    
	    (do ((k 0 (1+ k)))
		((= k 2))
	      (revert-sound ind)
	      (if (= k 0)
		  (map-channel (lambda (y) 1.0))
		  (begin
		    (scale-by 0.0)
		    (ptree-channel (lambda (y) 1.0))))
	      (let* ((ramp1 (lambda () (env-sound '(0 0 1 1))))
		     (ramp2 (lambda () (env-sound '(0 0 1 1 2 0))))
		     (ramp3 (lambda () (env-sound '(0 1 1 0 2 0 3 1))))
		     (ramp4 (lambda () (env-sound '(0 1 1 0 2 1))))
		     (xramp1 (lambda () (env-sound '(0 0 1 1) 0 100 2.0)))
		     (xramp2 (lambda () (env-sound '(0 1 1 0 2 0 3 1) 0 100 .6)))
		     (ptree1 (lambda () (ptree-channel (lambda (y) (+ y .2)))))
		     (ptree2 (lambda () (ptree-channel (lambda (y) (- y .1)))))
		     (map1 (lambda () (map-channel (lambda (y) (+ y .2)))))
		     (map2 (lambda () (map-channel (lambda (y) (- y .1)))))
		     (ops (list ramp1 ramp2 ramp3 xramp1 xramp2 ptree1 ptree2 ramp4))
		     (op-names (list "ramp1" "ramp2" "ramp3" "xramp1" "xramp2" "ptree1" "ptree2" "ramp4"))
		     (op (lambda (which) ((list-ref ops which))))
		     (op-name (lambda (which) (list-ref op-names which)))
		     (vals1 #f)
		     (vals2 #f)
		     (scl 1.0)
		     (scalers '())
		     (edpos (edit-position ind 0)))
		
		(let ((op1 (list 0 3 5))
		      (op2 (list 1 4 6)))
		  (for-each
		   (lambda (first)
		     (for-each 
		      (lambda (second)
			(if (and (not (= first second))
				 (let ((val (+ k
					       (if (> first 4) 1 0)
					       (if (> second 4) 1 0))))
				   (not (> val 2))))
			    (begin
			      (set! scalers '())
			      (set! scl (+ .5 (random 1.0)))
			      (scale-by scl)
			      (set! scalers (cons scl scalers))
			      (op first)
			      (set! scl (+ .5 (random 1.0)))
			      (scale-by scl)
			      (set! scalers (cons scl scalers))
			      (op second)
			      (set! scl (+ .5 (random 1.0)))
			      (scale-by scl)
			      (set! scalers (cons scl scalers))
			      (set! vals1 (channel->vct 0 100 ind 0))
			      (let ((rvals (reversed-channel->vct 0 100 ind 0)))
				(if (not (vequal rvals vals1))
				    (snd-display ";virtual op reversed tests: ~A(~A~A) * ~A: ~A ~A => ~A"
						 (op-name second) (op-name first)
						 (if (= k 1) "(ptree_zero)" "")
						 scalers vals1 rvals
						 (vct-peak (vct-subtract! (vct-copy vals1) rvals)))))
			      (set! (edit-position ind 0) edpos)
			      (set! (optimization) 0)
			      (set! scalers (reverse scalers))
			      (scale-by (car scalers)) (set! scalers (cdr scalers))
			      (if (= first 5) (map1) (if (= first 6) (map2) (op first)))
			      (scale-by (car scalers)) (set! scalers (cdr scalers))
			      (if (= second 5) (map1) (if (= second 6) (map2) (op second)))
			      (scale-by (car scalers)) (set! scalers (cdr scalers))
			      (set! (optimization) oldopt)
			      (set! vals2 (channel->vct 0 100 ind 0))
			      (if (not (vequal vals1 vals2)) 
				  (snd-display ";virtual op tests: ~A(~A~A) * ~A: ~A ~A => ~A at ~A"
					       (op-name second) (op-name first)
					       (if (= k 1) "(ptree_zero)" "")
					       scalers vals1 vals2
					       (vct-peak (vct-subtract! (vct-copy vals1) vals2))
					       (let* ((pks (vct-subtract! (vct-copy vals1) vals2))
						      (pk (abs (vct-ref pks 0)))
						      (loc 0))
						 (do ((i 1 (1+ i)))
						     ((= i 100) (list loc (vct-ref pks loc) (vct-ref vals1 loc) (vct-ref vals2 loc)))
						   (if (> (abs (vct-ref pks i)) pk)
						       (begin
							 (set! loc i)
							 (set! pk (abs (vct-ref pks i)))))))
					       ))
			      (let ((rvals (reversed-channel->vct 0 100 ind 0)))
				(if (not (vequal rvals vals2))
				    (snd-display ";virtual op reversed tests (2): ~A(~A~A) * ~A: ~A ~A => ~A"
						 (op-name second) (op-name first)
						 (if (= k 1) "(ptree_zero)" "")
						 scalers vals2 rvals
						 (vct-peak (vct-subtract! (vct-copy vals2) rvals)))))
			      (set! (edit-position ind 0) edpos))))
		      op2))
		   op1))
		(let ((op1 (list 0 3 5))
		      (op2 (list 1 4 6))
		      (op3 (list 2 3 4 5 6)))
		  (for-each
		   (lambda (first)
		     (for-each 
		      (lambda (second)
			(if (not (= first second))
			    (for-each
			     (lambda (third)
			       (if (and (not (= first third))
					(not (= second third))
					(let ((val (+ k
						      (if (or (= first 5) (= first 6)) 1 0)
						      (if (or (= second 5) (= second 6)) 1 0)
						      (if (or (= third 5) (= third 6)) 1 0))))
					  (not (> val 2))))
				   (begin
				     (set! scalers '())
				     (set! scl (+ .5 (random 1.0)))
				     (scale-by scl)
				     (set! scalers (cons scl scalers))
				     (op first)
				     (set! scl (+ .5 (random 1.0)))
				     (scale-by scl)
				     (set! scalers (cons scl scalers))
				     (op second)
				     (set! scl (+ .5 (random 1.0)))
				     (scale-by scl)
				     (set! scalers (cons scl scalers))
				     (op third)
				     (set! scl (+ .5 (random 1.0)))
				     (scale-by scl)
				     (set! scalers (cons scl scalers))
				     (set! vals1 (channel->vct 0 100 ind 0))
				     (let ((rvals (reversed-channel->vct 0 100 ind 0)))
				       (if (not (vequal rvals vals1))
					   (snd-display ";virtual op reversed tests: ~A(~A(~A~A)) * ~A: ~A ~A => ~A"
							(op-name third) (op-name second) (op-name first)
							(if (= k 1) "(ptree_zero)" "")
							scalers vals1 rvals
							(vct-peak (vct-subtract! (vct-copy vals1) rvals)))))
				     (set! (edit-position ind 0) edpos)
				     (set! (optimization) 0)
				     (set! scalers (reverse scalers))
				     (scale-by (car scalers)) (set! scalers (cdr scalers))
				     (if (= first 5) (map1) (if (= first 6) (map2) (op first)))
				     (scale-by (car scalers)) (set! scalers (cdr scalers))
				     (if (= second 5) (map1) (if (= second 6) (map2) (op second)))
				     (scale-by (car scalers)) (set! scalers (cdr scalers))
				     (if (= third 5) (map1) (if (= third 6) (map2) (op third)))
				     (scale-by (car scalers)) (set! scalers (cdr scalers))
				     (set! (optimization) oldopt)
				     (set! vals2 (channel->vct 0 100 ind 0))
				     (if (not (vequal vals1 vals2)) 
					 (snd-display ";virtual op tests: ~A * ~A(~A(~A~A)): ~A ~A => ~A at ~A"
						      scalers (op-name third) (op-name second) (op-name first) 
						      (if (= k 1) "(ptree_zero)" "")
						      vals1 vals2
						      (vct-peak (vct-subtract! (vct-copy vals1) vals2))
						      (let* ((pks (vct-subtract! (vct-copy vals1) vals2))
							     (pk (abs (vct-ref pks 0)))
							     (loc 0))
							(do ((i 1 (1+ i)))
							    ((= i 100) (list loc (vct-ref pks loc) (vct-ref vals1 loc) (vct-ref vals2 loc)))
							  (if (> (abs (vct-ref pks i)) pk)
							      (begin
								(set! loc i)
								(set! pk (abs (vct-ref pks i)))))))
						      ))
				     (let ((rvals (reversed-channel->vct 0 100 ind 0)))
				       (if (not (vequal rvals vals2))
					   (snd-display ";virtual op reversed tests (2): ~A(~A(~A~A)) * ~A: ~A ~A => ~A"
							(op-name third) (op-name second) (op-name first)
							(if (= k 1) "(ptree_zero)" "")
							scalers vals2 rvals
							(vct-peak (vct-subtract! (vct-copy vals2) rvals)))))
				     (set! (edit-position ind 0) edpos))))
			     op3)))
		      op2))
		   op1))
		(let ((op1 (list 0 3 5))
		      (op2 (list 0 1 3 4 5 6))
		      (op3 (list 0 1 2 3 4 5 6))
		      (op4 (list 0 1 2 3 4 5 6)))
		  (for-each
		   (lambda (first)
		     (for-each 
		      (lambda (second)
			(if (not (= first second))
			    (for-each
			     (lambda (third)
			       (if (and (not (= first third))
					(not (= second third)))
				   (for-each
				    (lambda (fourth)
				      (if (and (not (= first fourth))
					       (not (= second fourth))
					       (not (= third fourth))
					       (let ((val (+ k
							     (if (or (= first 5) (= first 6)) 1 0)
							     (if (or (= second 5) (= second 6)) 1 0)
							     (if (or (= third 5) (= third 6)) 1 0)
							     (if (or (= fourth 5) (= fourth 6)) 1 0))))
						 (not (> val 2))))
					  (begin
					    (set! scalers '())
					    (set! scl (+ .5 (random 1.0)))
					    (scale-by scl)
					    (set! scalers (cons scl scalers))
					    (op first)
					    (set! scl (+ .5 (random 1.0)))
					    (scale-by scl)
					    (set! scalers (cons scl scalers))
					    (op second)
					    (set! scl (+ .5 (random 1.0)))
					    (scale-by scl)
					    (set! scalers (cons scl scalers))
					    (op third)
					    (set! scl (+ .5 (random 1.0)))
					    (scale-by scl)
					    (set! scalers (cons scl scalers))
					    (op fourth)
					    (set! scl (+ .5 (random 1.0)))
					    (scale-by scl)
					    (set! scalers (cons scl scalers))
					    (set! vals1 (channel->vct 0 100 ind 0))
					    (let ((rvals (reversed-channel->vct 0 100 ind 0)))
					      (if (not (vequal rvals vals1))
						  (snd-display ";virtual op reversed tests: ~A(~A(~A(~A~A))) * ~A: ~A ~A => ~A"
							       (op-name fourth) (op-name third) (op-name second) (op-name first)
							       (if (= k 1) "(ptree_zero)" "")
							       scalers vals1 rvals
							       (vct-peak (vct-subtract! (vct-copy vals1) rvals)))))
					    (set! (edit-position ind 0) edpos)
					    (set! (optimization) 0)
					    (set! scalers (reverse scalers))
					    (scale-by (car scalers)) (set! scalers (cdr scalers))
					    (if (= first 5) (map1) (if (= first 6) (map2) (op first)))
					    (scale-by (car scalers)) (set! scalers (cdr scalers))
					    (if (= second 5) (map1) (if (= second 6) (map2) (op second)))
					    (scale-by (car scalers)) (set! scalers (cdr scalers))
					    (if (= third 5) (map1) (if (= third 6) (map2) (op third)))
					    (scale-by (car scalers)) (set! scalers (cdr scalers))
					    (if (= fourth 5) (map1) (if (= fourth 6) (map2) (op fourth)))
					    (scale-by (car scalers)) (set! scalers (cdr scalers))
					    (set! (optimization) oldopt)
					    (set! vals2 (channel->vct 0 100 ind 0))
					    (if (not (vequal vals1 vals2)) 
						(snd-display ";virtual op tests: ~A * ~A(~A(~A(~A~A))): ~A ~A => ~A at ~A"
							     scalers
							     (op-name fourth) (op-name third) (op-name second) (op-name first)
							     (if (= k 1) "(ptree_zero)" "") 
							     vals1 vals2
							     (vct-peak (vct-subtract! (vct-copy vals1) vals2))
							     (let* ((pks (vct-subtract! (vct-copy vals1) vals2))
								    (pk (abs (vct-ref pks 0)))
								    (loc 0))
							       (do ((i 1 (1+ i)))
								   ((= i 100) (list loc (vct-ref pks loc) (vct-ref vals1 loc) (vct-ref vals2 loc)))
								 (if (> (abs (vct-ref pks i)) pk)
								     (begin
								       (set! loc i)
								       (set! pk (abs (vct-ref pks i)))))))
							     ))
					    (let ((rvals (reversed-channel->vct 0 100 ind 0)))
					      (if (not (vequal rvals vals2))
						  (snd-display ";virtual op reversed tests (2): ~A(~A(~A(~A~A))) * ~A: ~A ~A => ~A"
							       (op-name fourth) (op-name third) (op-name second) (op-name first)
							       (if (= k 1) "(ptree_zero)" "")
							       scalers vals2 rvals
							       (vct-peak (vct-subtract! (vct-copy vals2) rvals)))))
					    (set! (edit-position ind 0) edpos))))
				    op4)))
			     op3)))
		      op2))
		   op1))
		(if all-args
		    (let ((op1 (list 0 3 5))
			  (op2 (list 0 1 3 4 5 6))
			  (op3 (list 0 1 2 3 4 5 6))
			  (op4 (list 0 1 2 3 4 5 6 7))
			  (op5 (list 7 3 5)))
		      (for-each
		       (lambda (first)
			 (for-each 
			  (lambda (second)
			    (if (not (= first second))
				(for-each
				 (lambda (third)
				   (if (and (not (= first third))
					    (not (= second third)))
				       (for-each
					(lambda (fourth)
					  (if (and (not (= first fourth))
						   (not (= second fourth))
						   (not (= third fourth)))
					      (for-each
					       (lambda (fifth)
						 (if (and (not (= first fifth))
							  (not (= second fifth))
							  (not (= third fifth))
							  (not (= fourth fifth))
							  (let ((val (+ k
									(if (or (= first 5) (= first 6)) 1 0)
									(if (or (= second 5) (= second 6)) 1 0)
									(if (or (= third 5) (= third 6)) 1 0)
									(if (or (= fourth 5) (= fourth 6)) 1 0)
									(if (or (= fifth 5) (= fifth 6)) 1 0))))
							    (not (> val 2))))
						     (begin
						       (set! scalers '())
						       (set! scl (+ .5 (random 1.0)))
						       (scale-by scl)
						       (set! scalers (cons scl scalers))
						       (op first)
						       (set! scl (+ .5 (random 1.0)))
						       (scale-by scl)
						       (set! scalers (cons scl scalers))
						       (op second)
						       (set! scl (+ .5 (random 1.0)))
						       (scale-by scl)
						       (set! scalers (cons scl scalers))
						       (op third)
						       (set! scl (+ .5 (random 1.0)))
						       (scale-by scl)
						       (set! scalers (cons scl scalers))
						       (op fourth)
						       (set! scl (+ .5 (random 1.0)))
						       (scale-by scl)
						       (set! scalers (cons scl scalers))
						       (op fifth)
						       (set! scl (+ .5 (random 1.0)))
						       (scale-by scl)
						       (set! scalers (cons scl scalers))
						       (set! vals1 (channel->vct 0 100 ind 0))
						       (let ((rvals (reversed-channel->vct 0 100 ind 0)))
							 (if (not (vequal rvals vals1))
							     (snd-display ";virtual op reversed tests: ~A(~A(~A(~A(~A~A)))) * ~A: ~A ~A => ~A"
									  (op-name fifth) (op-name fourth) (op-name third) (op-name second) (op-name first)
									  (if (= k 1) "(ptree_zero)" "")
									  scalers vals1 rvals
									  (vct-peak (vct-subtract! (vct-copy vals1) rvals)))))
						       (set! (edit-position ind 0) edpos)
						       (set! (optimization) 0)
						       (set! scalers (reverse scalers))
						       (scale-by (car scalers)) (set! scalers (cdr scalers))
						       (if (= first 5) (map1) (if (= first 6) (map2) (op first)))
						       (scale-by (car scalers)) (set! scalers (cdr scalers))
						       (if (= second 5) (map1) (if (= second 6) (map2) (op second)))
						       (scale-by (car scalers)) (set! scalers (cdr scalers))
						       (if (= third 5) (map1) (if (= third 6) (map2) (op third)))
						       (scale-by (car scalers)) (set! scalers (cdr scalers))
						       (if (= fourth 5) (map1) (if (= fourth 6) (map2) (op fourth)))
						       (scale-by (car scalers)) (set! scalers (cdr scalers))
						       (if (= fifth 5) (map1) (if (= fifth 6) (map2) (op fifth)))
						       (scale-by (car scalers)) (set! scalers (cdr scalers))
						       (set! (optimization) oldopt)
						       (set! vals2 (channel->vct 0 100 ind 0))
						       (if (not (vequal vals1 vals2)) 
							   (snd-display ";virtual op tests: ~A * ~A(~A(~A(~A(~A~A)))): ~A ~A => ~A at ~A"
									scalers
									(op-name fifth) (op-name fourth) (op-name third) (op-name second) (op-name first)
									(if (= k 1) "(ptree_zero)" "") 
									vals1 vals2
									(vct-peak (vct-subtract! (vct-copy vals1) vals2))
									(let* ((pks (vct-subtract! (vct-copy vals1) vals2))
									       (pk (abs (vct-ref pks 0)))
									       (loc 0))
									  (do ((i 1 (1+ i)))
									      ((= i 100) (list loc (vct-ref pks loc) (vct-ref vals1 loc) (vct-ref vals2 loc)))
									    (if (> (abs (vct-ref pks i)) pk)
										(begin
										  (set! loc i)
										  (set! pk (abs (vct-ref pks i)))))))
									))
						       (let ((rvals (reversed-channel->vct 0 100 ind 0)))
							 (if (not (vequal rvals vals2))
							     (snd-display ";virtual op reversed tests (2): ~A(~A(~A(~A(~A~A)))) * ~A: ~A ~A => ~A"
									  (op-name fifth) (op-name fourth) (op-name third) (op-name second) (op-name first)
									  (if (= k 1) "(ptree_zero)" "")
									  scalers vals2 rvals
									  (vct-peak (vct-subtract! (vct-copy vals2) rvals)))))
						       (set! (edit-position ind 0) edpos))))
					       op5)))
					op4)))
				 op3)))
			  op2))
		       op1)))
		
		(if all-args
		    (let ((op1 (list 0 3 5))
			  (op2 (list 0 1 3 4 5 6))
			  (op3 (list 0 1 2 3 4 5 6))
			  (op4 (list 0 1 2 3 4 5 6))
			  (op5 (list 7 3 5))
			  (op6 (list 0 1 2 3 4 5 6 7)))
		      (for-each
		       (lambda (first)
			 (for-each 
			  (lambda (second)
			    (if (not (= first second))
				(for-each
				 (lambda (third)
				   (if (and (not (= first third))
					    (not (= second third)))
				       (for-each
					(lambda (fourth)
					  (if (and (not (= first fourth))
						   (not (= second fourth))
						   (not (= third fourth)))
					      (for-each
					       (lambda (fifth)
						 (if (and (not (= first fifth))
							  (not (= second fifth))
							  (not (= third fifth))
							  (not (= fourth fifth)))
						     (for-each
						      (lambda (sixth)
							(if (and (not (= first sixth))
								 (not (= second sixth))
								 (not (= third sixth))
								 (not (= fourth sixth))
								 (not (= fifth sixth))
								 (let ((val (+ k
									       (if (or (= first 5) (= first 6)) 1 0)
									       (if (or (= second 5) (= second 6)) 1 0)
									       (if (or (= third 5) (= third 6)) 1 0)
									       (if (or (= fourth 5) (= fourth 6)) 1 0)
									       (if (or (= fifth 5) (= fifth 6)) 1 0)
									       (if (or (= sixth 5) (= sixth 6)) 1 0))))
								   (not (> val 2)))
								 (let ((val (+ (if (member first (list 0 1 2 3 4 7)) 1 0)
									       (if (member second (list 0 1 2 3 4 7)) 1 0)
									       (if (member third (list 0 1 2 3 4 7)) 1 0)
									       (if (member fourth (list 0 1 2 3 4 7)) 1 0)
									       (if (member fifth (list 0 1 2 3 4 7)) 1 0)
									       (if (member sixth (list 0 1 2 3 4 7)) 1 0))))
								   (not (> val 4))))
							    (begin
							      (set! scalers '())
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (op first)
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (op second)
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (op third)
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (op fourth)
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (op fifth)
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (op sixth)
							      (set! scl (+ .5 (random 1.0)))
							      (scale-by scl)
							      (set! scalers (cons scl scalers))
							      (set! vals1 (channel->vct 0 100 ind 0))
							      (let ((rvals (reversed-channel->vct 0 100 ind 0)))
								(if (not (vequal rvals vals1))
								    (snd-display ";virtual op reversed tests: ~A(~A(~A(~A(~A(~A~A))))) * ~A: ~A ~A => ~A"
										 (op-name sixth) (op-name fifth) (op-name fourth) 
										 (op-name third) (op-name second) (op-name first)
										 (if (= k 1) "(ptree_zero)" "")
										 scalers vals1 rvals
										 (vct-peak (vct-subtract! (vct-copy vals1) rvals)))))
							      (set! (edit-position ind 0) edpos)
							      (set! (optimization) 0)
							      (set! scalers (reverse scalers))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (if (= first 5) (map1) (if (= first 6) (map2) (op first)))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (if (= second 5) (map1) (if (= second 6) (map2) (op second)))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (if (= third 5) (map1) (if (= third 6) (map2) (op third)))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (if (= fourth 5) (map1) (if (= fourth 6) (map2) (op fourth)))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (if (= fifth 5) (map1) (if (= fifth 6) (map2) (op fifth)))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (if (= sixth 5) (map1) (if (= sixth 6) (map2) (op sixth)))
							      (scale-by (car scalers)) (set! scalers (cdr scalers))
							      (set! (optimization) oldopt)
							      (set! vals2 (channel->vct 0 100 ind 0))
							      (if (not (vequal vals1 vals2)) 
								  (snd-display ";virtual op tests: ~A * ~A(~A(~A(~A(~A(~A~A))))): ~A ~A => ~A at ~A"
									       scalers
									       (op-name sixth) (op-name fifth) (op-name fourth) 
									       (op-name third) (op-name second) (op-name first)
									       (if (= k 1) "(ptree_zero)" "") 
									       vals1 vals2
									       (vct-peak (vct-subtract! (vct-copy vals1) vals2))
									       (let* ((pks (vct-subtract! (vct-copy vals1) vals2))
										      (pk (abs (vct-ref pks 0)))
										      (loc 0))
										 (do ((i 1 (1+ i)))
										     ((= i 100) (list loc (vct-ref pks loc) (vct-ref vals1 loc) (vct-ref vals2 loc)))
										   (if (> (abs (vct-ref pks i)) pk)
										       (begin
											 (set! loc i)
											 (set! pk (abs (vct-ref pks i)))))))
									       ))
							      (let ((rvals (reversed-channel->vct 0 100 ind 0)))
								(if (not (vequal rvals vals2))
								    (snd-display ";virtual op reversed tests (2): ~A(~A(~A(~A(~A(~A~A))))) * ~A: ~A ~A => ~A"
										 (op-name sixth) (op-name fifth) (op-name fourth) 
										 (op-name third) (op-name second) (op-name first)
										 (if (= k 1) "(ptree_zero)" "")
										 scalers vals2 rvals
										 (vct-peak (vct-subtract! (vct-copy vals2) rvals)))))
							      (set! (edit-position ind 0) edpos))))
						      op6)))
					       op5)))
					op4)))
				 op3)))
			  op2))
		       op1)))))
	    (close-sound ind))
	  
	  
	  (let* ((ind (open-sound "oboe.snd"))
		 (mx (maxamp ind 0)))
	    (as-one-edit
	     (lambda ()
	       (ptree-channel (lambda (y) (* y 2)))
	       (env-sound '(0 0 1 1))))
	    (if (not (= (edit-position ind 0) 1)) (snd-display ";as-one-edit env+ptree pos: ~A" (edit-position ind 0)))
	    (if (fneq (maxamp ind 0) .1825) (snd-display ";as-one-edit env+ptree max: ~A" (maxamp ind 0)))
	    (undo)
	    (as-one-edit
	     (lambda ()
	       (ptree-channel (lambda (y) (* y 2)))
	       (ptree-channel (lambda (y) (* y 2)))))
	    (if (not (= (edit-position ind 0) 1)) (snd-display ";as-one-edit ptree+ptree pos: ~A" (edit-position ind 0)))
	    (if (fneq (maxamp ind 0) (* 4 mx)) (snd-display ";as-one-edit ptree+ptree max: ~A ~A" (maxamp ind 0) (* 4 mx)))
	    (close-sound ind))
	  (let ((ind (new-sound  "test.snd" mus-next mus-bfloat 22050 1 "ptree tests" 10)))
	    ;; offset-channel
	    (offset-channel .1)
	    (if (not (vequal (channel->vct 0 10) (make-vct 10 .1)))
		(snd-display ";offset-channel (.1): ~A" (channel->vct 0 10)))
	    (offset-channel -.2 5 5)
	    (if (not (vequal (channel->vct 0 10) (vct .1 .1 .1 .1 .1 -.1 -.1 -.1 -.1 -.1)))
		(snd-display ";offset-channel (-.1): ~A" (channel->vct 0 10)))
	    (undo)
	    (offset-channel .9 0 10 ind 0)
	    (if (not (vequal (channel->vct 0 10) (make-vct 10 1.0)))
		(snd-display ";offset-channel (1): ~A" (channel->vct 0 10)))
	    ;; sine-env and sine-ramp...
	    (revert-sound ind)
	    (map-channel (lambda (y) 1.0))
	    (sine-ramp 0.0 1.0)
	    (if (not (vequal (channel->vct) (vct 0.000 0.024 0.095 0.206 0.345 0.500 0.655 0.794 0.905 0.976)))
		(snd-display ";sine-ramp 0 1: ~A" (channel->vct)))
	    (revert-sound ind)
	    (offset-channel 1.0)
	    (sine-ramp 1.0 0.0)
	    (if (not (vequal (channel->vct) (vct 1.000 0.976 0.905 0.794 0.655 0.500 0.345 0.206 0.095 0.024)))
		(snd-display ";sine-ramp 1 0: ~A" (channel->vct)))
	    (if (> (optimization) 0)
		(if (not (string=? (edit-fragment-type-name (list-ref (car (edit-tree)) 7)) "ed_ptree2_zero"))
		    (snd-display ";sine-ramp tree op: ~A ~A" (edit-fragment-type-name (list-ref (car (edit-tree)) 7)) (edit-tree))))
	    (revert-sound ind)
	    (as-one-edit
	     (lambda ()
	       (map-channel (lambda (y) 1.0))
	       (env-channel '(0 0 1 1))
	       (xen-channel (lambda (y data forward) (+ y (car data))) 0 (frames) ind 0 #f #f (lambda (p d) (list 0.25)))))
	    (if (not (vequal (channel->vct) (vct 0.250 0.361 0.472 0.583 0.694 0.806 0.917 1.028 1.139 1.250)))
		(snd-display ";as-one-edit xen: ~A" (channel->vct)))
	    (if (not (string=? (edit-fragment-type-name (list-ref (car (edit-tree)) 7)) "ed_xen_ramp"))
		(snd-display ";as-one-edit xen tree op: ~A ~A" (edit-fragment-type-name (list-ref (car (edit-tree)) 7)) (edit-tree)))
	    (close-sound ind)
	    (set! ind (new-sound  "test.snd" mus-next mus-bfloat 22050 1 "sine-env tests" 100))
	    (map-channel (lambda (y) 1.0))
	    (sine-env-channel '(0 0 1 1 2 -.5 3 1))
	    (if (or (not (vequal (channel->vct 20 10) (vct 0.664 0.708 0.750 0.790 0.827 0.862 0.893 0.921 0.944 0.964)))
		    (not (vequal (channel->vct 60 10) (vct -0.381 -0.417 -0.446 -0.470 -0.486 -0.497 -0.500 -0.497 -0.486 -0.470))))
		(snd-display ";sine-env-channel 0: ~A ~A" (channel->vct 20 10) (channel->vct 60 10)))
	    (if (not (= (edit-position ind 0) 2)) (snd-display ";as-one-edit sine-env-channel: ~A" (edit-position ind 0)))
	    (revert-sound ind)
	    (offset-channel -1.0)
	    (sine-env-channel '(0 0 1 1 2 1 3 0) 40 20)
	    (if (or (not (vequal (channel->vct 40 20) (vct -0.000 -0.050 -0.188 -0.389 -0.611 -0.812 -0.950 -1.000 -1.000 -1.000
							   -1.000 -1.000 -1.000 -1.000 -1.000 -0.950 -0.812 -0.611 -0.389 -0.188)))
		    (not (vequal (channel->vct 30 10) (make-vct 10 -1.0))))
		(snd-display ";off+sine-env: ~A ~A" (channel->vct 40 20) (channel->vct 30 10)))
	    (revert-sound ind)
	    (let ((tag (catch #t
			      (lambda ()
				(xen-channel (lambda (y) (* y 2))))
			      (lambda args (car args)))))
	      (if (not (eq? tag 'bad-arity))
		  (snd-display ";xen-channel with 1-arg func: ~A ~A" tag (optimization))))
	    (ptree-channel (lambda (y d f) (* y 2)) 0 (frames) ind 0 #f #f (lambda (p d) (vct 1.0)))
	    (revert-sound ind)
	    (scale-by 0.0)
	    (dither-channel)
	    (let ((mx (maxamp)))
	      (if (or (< mx .00003) (> mx .0001))
		  (snd-display ";dithering: ~A" mx)))
	    (revert-sound ind)
	    (map-channel (ring-mod 10 (list 0 0 1 (hz->radians 100))))
	    (map-channel (osc-formants .99 '(400 800 1200) '(400 800 1200) '(4 2 3)))
	    (map-channel (zecho .5 .75 6 10.0))
	    (map-channel (flecho .5 .9))
	    (filtered-env '(0 0 1 1 2 0))
	    (map-channel (formant-filter .99 2400))
	    (map-channel (comb-filter .8 32))
	    (map-channel (zcomb .8 32 '(0 0 1 10)))
	    (map-channel (notch-filter .8 32))
	    (let ((ind1 (open-sound "now.snd")))
	      (select-sound ind1)
	      (if (fneq (maxamp) .309) (snd-display ";squelch-vowels init: ~A" (maxamp)))
	      (squelch-vowels)
	      (if (fneq (maxamp) .047) (snd-display ";squelch-vowels maxamp: ~A" (maxamp)))
	      (select-sound ind)
	      (map-channel (cross-synthesis ind1 .5 128 6.0))
	      (revert-sound ind1)
	      (fft-edit 40 8000)
	      (fft-squelch .1)
	      (close-sound ind)
	      (revert-sound ind1)
	      (scramble-channel .01)
	      (revert-sound ind1)
	      (close-sound ind1)))

	  (let ((ind (new-sound  "test.snd" mus-next mus-bfloat 22050 1 "special env tests" 100)))
	    (map-channel (lambda (y) 1.0))
	    
	    (blackman4-ramp 0.0 1.0)
	    (let ((vals (channel->vct)))
	      (undo)
	      (blackman4-env-channel '(0 0 1 1))
	      (let ((new-vals (channel->vct)))
		(if (not (vequal vals new-vals))
		    (snd-display ";blackman4-env-channel/ramp: ~A ~A" vals new-vals))
		(undo)
		(blackman4-ramp 0.0 1.0 0 50)
		(set! vals (channel->vct))
		(undo)
		(blackman4-env-channel '(0 0 1 1 2 1))
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";blackman4-env-channel/ramp 1: ~A ~A" vals new-vals))
		(undo)
		(blackman4-env-channel '(0 0 1 1 2 -.5 3 0))
		(if (not (vequal (channel->vct 60 10) (vct -0.109 -0.217 -0.313 -0.392 -0.451 -0.488 -0.499 -0.499 -0.499 -0.499)))
		    (snd-display ";blackman4 to -.5: ~A" (channel->vct 60 10)))
		(undo)
		
		(ramp-squared 0.0 1.0)
		(set! vals (channel->vct))
		(undo)
		(env-squared-channel '(0 0 1 1))
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-squared/ramp: ~A ~A" vals new-vals))
		(undo)
		(ramp-squared 0.0 1.0 #t 0 50)
		(set! vals (channel->vct))
		(undo)
		(env-squared-channel '(0 0 1 1 2 1))
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-squared/ramp 1: ~A ~A" vals new-vals))
		(undo)
		(env-squared-channel '(0 0 1 1 2 -.5 3 0))
		(if (not (vequal (channel->vct 60 10) (vct -0.450 -0.466 -0.478 -0.488 -0.494 -0.499 -0.500 -0.500 -0.498 -0.496)))
		    (snd-display ";env-squared to -.5: ~A" (channel->vct 60 10)))
		(undo)
		(env-squared-channel '(0 0 1 1 2 -.5 3 0) #f)
		(if (not (vequal (channel->vct 60 10) (vct -0.004 -0.080 -0.158 -0.240 -0.324 -0.410 -0.500 -0.500 -0.498 -0.496)))
		    (snd-display ";env-squared unsymmetric to -.5: ~A" (channel->vct 60 10)))
		(undo)
		
		(ramp-squared 0.0 1.0)
		(set! vals (channel->vct))
		(undo)
		(env-expt-channel '(0 0 1 1) 2)
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-expt2/ramp: ~A ~A" vals new-vals))
		(undo)
		(env-squared-channel '(0 0 1 1 2 -.5 3 0))
		(set! vals (channel->vct))
		(undo)
		(env-expt-channel '(0 0 1 1 2 -.5 3 0) 2.0)
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-expt2/env-squared: ~A ~A" vals new-vals))
		(undo)
		(env-squared-channel '(0 0 1 1 2 -.5 3 0) #f)
		(set! vals (channel->vct))
		(undo)
		(env-expt-channel '(0 0 1 1 2 -.5 3 0) 2.0 #f)
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-expt2/env-squared unsymmetric: ~A ~A" vals new-vals))
		(undo)

		(ramp-expt 0.0 1.0 32.0)
		(set! vals (channel->vct))
		(undo)
		(env-expt-channel '(0 0 1 1) 32.0)
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-expt/ramp 32: ~A ~A" vals new-vals))
		(undo)
		(ramp-expt 0.0 1.0 32.0 #f 0 50)
		(set! vals (channel->vct))
		(undo)
		(env-expt-channel '(0 0 1 1 2 1) 32.0)
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-expt/ramp 1 32: ~A ~A" vals new-vals))
		(undo)
		(ramp-expt 0.0 1.0 .1)
		(set! vals (channel->vct))
		(undo)
		(env-expt-channel '(0 0 1 1) .1)
		(set! new-vals (channel->vct))
		(if (not (vequal vals new-vals))
		    (snd-display ";env-expt/ramp .1: ~A ~A" vals new-vals))
		(undo)

		(env-expt-channel '(0 0 1 1 2 -.5 3 0) 12.0)
		(if (not (vequal (channel->vct 30 10) (vct 0.319 0.472 0.691 1.000 0.537 0.208 -0.022 -0.182 -0.291 -0.365)))
		    (snd-display ";env-expt to -.5 12.0: ~A" (channel->vct 30 10)))
		(undo)
		(env-expt-channel '(0 0 1 1 2 -.5 3 0) 12.0 #f)
		(if (not (vequal (channel->vct 30 10) (vct 0.319 0.472 0.691 1.000 1.000 1.000 1.000 1.000 1.000 1.000)))
		    (snd-display ";env-expt to -.5 12.0 unsymmetric: ~A" (channel->vct 30 10)))
		(undo)
		(close-sound ind))))

	  (let ((ind (new-sound  "test.snd" mus-next mus-bfloat 22050 1 "ramp re-order tests" 101)))
	    (offset-channel 1.0)
	    (env-sound '(0 0 1 1))
	    (contrast-channel 1.0)
	    (let ((reader (make-sample-reader 0)))
	      (call-with-current-continuation
	       (lambda (break)
		 (do ((i 0 (1+ i))
		      (val 0.0 (+ val .01)))
		     ((= i 100))
		   (let ((y (reader))
			 (ny (sin (+ (* val 0.5 pi) (* 1.0 (sin (* val 2.0 pi)))))))
		     (if (fneq y ny)
			 (break (snd-display ";contrast-channel: ~A ~A ~A" val y ny))))))))
	    (undo)
	    (compand-channel)
	    (let ((reader (make-sample-reader 0)))
	      (call-with-current-continuation
	       (lambda (break)
		 (do ((i 0 (1+ i))
		      (val 0.0 (+ val .01)))
		     ((= i 100))
		   (let ((y (reader))
			 (ny (array-interp compand-table (+ 8.0 (* 8.0 val)) 17)))
		     (if (fneq y ny)
			 (break (snd-display ";compand-channel: ~A ~A ~A" val y ny))))))))
	    (undo 2)
	    (ring-modulate-channel 1000)
	    (let ((reader (make-sample-reader 0))
		  (incr (/ (* 2 pi 1000) (srate))))
	      (call-with-current-continuation
	       (lambda (break)
		 (do ((i 0 (1+ i))
		      (val 0.0 (+ val incr)))
		     ((= i 100))
		   (let ((y (reader))
			 (ny (sin val)))
		     (if (fneq y ny)
			 (break (snd-display ";ring-modulate-channel: ~A ~A ~A" val y ny))))))))
	    (undo)
	    (env-sound '(0 0 1 1))
	    (smooth-channel-via-ptree)
	    (let ((reader (make-sample-reader 0))
		  (incr (/ pi 101)))
	      (call-with-current-continuation
	       (lambda (break)
		 (do ((i 0 (1+ i))
		      (val pi (+ val incr)))
		     ((= i 100))
		   (let ((y (reader))
			 (ny (+ 0.5 (* 0.5 (cos val)))))
		     (if (fneq y ny)
			 (break (snd-display ";smooth-channel-via-ptree: ~A ~A ~A" val y ny))))))))
	    (undo 2)
	    (env-channel '(0 1 1 0 2 1) 10 11)
	    (if (not (vequal (channel->vct 0 30) (vct 1 1 1 1 1 1 1 1 1 1 1.000 0.800 0.600 0.400 0.200 0.000 0.200 0.400 0.600 0.800 1 1 1 1 1 1 1 1 1 1)))
		(snd-display ";env+ptree: ~A ~A ~A" (channel->vct 0 10) (channel->vct 10 10) (channel->vct 20 10)))
	    (close-sound ind))

	  ))
      )
      (run-hook after-test-hook 5)
      ))


;;; ---------------- test 6 vcts ----------------

(if (or full-test (= snd-test 6) (and keep-going (<= snd-test 6)))
    (begin 
      (run-hook before-test-hook 6)
      (do ((clmtest 0 (1+ clmtest))) ((= clmtest tests)) 
	(log-mem clmtest)
	(let ((v0 (make-vct 10))
	      (v1 (make-vct 10))
	      (vlst (make-vct 3)))
	  (if (not (vct? v0)) (snd-display ";v0 isn't a vct?!?"))
	  (if (equal? v0 10) (snd-display ";v0 is 10!?"))
	  (if (vct? 10) (snd-display ";10 is a vct?"))
	  (if (not (= (vct-length v0) 10)) (snd-display ";v0 length = ~D?" (vct-length v0)))
	  (vct-fill! v0 1.0)
	  (vct-fill! v1 0.5)
	  (if (equal? v0 v1) (snd-display ";vct equal? ~A ~A" v0 v1))
	  (if (eq? v0 v1) (snd-display ";vct eq? ~A ~A" v0 v1))
	  (let ((v2 v1)
		(v3 (make-vct 10))
		(v4 (make-vct 3)))
	    (if (not (eq? v1 v2)) (snd-display ";vct not eq? ~A ~A" v1 v2))
	    (vct-fill! v3 0.5) 
	    (if (not (equal? v3 v1)) (snd-display ";vct not equal? ~A ~A" v3 v1))
	    (if (equal? v4 v1) (snd-display ";len diff vct equal? ~A ~A" v4 v1))
	    (set! (vct-ref v3 0) 1.0)
	    (if (fneq (vct-ref v3 0) 1.0) (snd-display ";set! vct-ref: ~A" (vct-ref v3 0))))
	  (vct-set! vlst 1 .1)
	  (if (not (feql (vct->list vlst) (list 0.0 0.1 0.0))) (snd-display ";vct->list: ~A?" (vct->list vlst)))
	  (let* ((vect '#(0.0 1.0 2.0 3.0))
		 (v123 (vct 0.0 1.0 2.0 3.0))
		 (v2 (vector->vct vect))
		 (v3 v2)
		 (str (format #f "~A" v2))
		 (str1 (format #f "~A" (make-vct 32))))
	    (if (not (eq? #f (vector->vct (make-vector 0)))) (snd-display ";vector->vct empty vect: ~A" (vector->vct (make-vector 0))))
	    (if (not (string=? str "#<vct[len=4]: 0.000 1.000 2.000 3.000>"))
		(snd-display ";vct print: ~%  ~A~%  ~A?" str v2))
	    (if (and (= (print-length) 12)
		     (not (string=? str1 "#<vct[len=32]: 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 ...>")))
		(snd-display ";vct(32) print: ~%  ~A~%" str1))
	    (if (not (vequal v123 v2)) (snd-display ";vector->vct: ~A" v2))
	    (if (not (equal? (vct->vector v123) vect)) (snd-display ";vct->vector: ~A ~A" vect (vct->vector v123)))
	    (if (not (equal? v3 v2)) (snd-display ";vct=? ~A ~A?" v2 v3))
	    (if (not (= (vct-length v2) 4)) (snd-display ";vector->vct length: ~A?" (vct-length v2)))
	    (if (fneq (vct-ref v2 2) 2.0) (snd-display ";vector->vct: ~A?" v2))
	    (vct-move! v2 0 2)
	    (if (fneq (vct-ref v2 0) 2.0) (snd-display ";vct-move!: ~A?" v2)))
	  (let ((v2 (make-vct 4)))
	    (do ((i 0 (1+ i)))
		((= i 4))
	      (vct-set! v2 i i))
	    (vct-move! v2 3 2 #t)
	    (if (or (fneq (vct-ref v2 3) 2.0) (fneq (vct-ref v2 2) 1.0))
		(snd-display ";vct-move! back: ~A?" v2)))
	  
	  (let ((v0 (make-vct 3)))
	    (let ((var (catch #t (lambda () (vct-ref v0 10)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-ref high index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-ref v0 -1)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-ref low index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-set! v0 10 1.0)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-set! high index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-set! v0 -1 1.0)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-set! low index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-move! v0 10 0 #t)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-move! high index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-move! v0 0 10 #t)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-move! high 2 index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-move! v0 -10 0 #f)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-move! back high index: ~A" var)))
	    (let ((var (catch #t (lambda () (vct-move! v0 0 -10 #f)) (lambda args args))))
	      (if (not (eq? (car var) 'out-of-range))
		  (snd-display ";vct-move! back high 2 index: ~A" var))))
	  
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (if (fneq (vct-ref v0 i) 1.0) (snd-display ";fill v0[~D] = ~F?" i (vct-ref v0 i)))
	    (if (fneq (vct-ref v1 i) 0.5) (snd-display ";preset v1[~D] = ~F?" i (vct-ref v1 i))))
	  (vct-add! v0 v1)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (if (fneq (vct-ref v0 i) 1.5) (snd-display ";add v0[~D] = ~F?" i (vct-ref v0 i))))
	  (vct-subtract! v0 v1)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (if (fneq (vct-ref v0 i) 1.0) (snd-display ";subtract v0[~D] = ~F?" i (vct-ref v0 i))))
	  (let ((v2 (vct-copy v0)))
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v2 i) 1.0) (snd-display ";copy v0[~D] = ~F?" i (vct-ref v2 i))))
	    (vct-scale! v2 5.0)
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v2 i) 5.0) (snd-display ";scale v2[~D] = ~F?" i (vct-ref v2 i))))
	    (vct-offset! v0 -1.0)
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v0 i) 0.0) (snd-display ";offset v0[~D] = ~F?" i (vct-ref v0 i))))
	    (vct-multiply! v2 v1)
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v2 i) 2.5) (snd-display ";multiply v2[~D] = ~F?" i (vct-ref v2 i))))
	    (if (fneq (vct-peak v2) 2.5) (snd-display ";v2's peak is ~F?" (vct-peak v2)))
	    (vct-set! v2 5 123.0)
	    (if (fneq (vct-peak v2) 123.0) (snd-display ";v2's set peak is ~F?" (vct-peak v2)))
	    (let ((vn (make-vct 32))
		  (vb (make-vct 64))
		  (vs (make-vct 3))
		  (vss (make-vct 1)))
	      (do ((i 0 (1+ i)))
		  ((= i 32))
		(vct-set! vn i i))
	      (let ((vnew (vct-subseq vn 3)))
		(if (fneq (vct-ref vnew 0) 3.0) (snd-display ";vct-subseq[3:] ~A?" (vct-ref vnew 0)))
		(if (not (= (vct-length vnew) 29)) (snd-display ";vct-subseq[3:] length: ~A?" (vct-length vnew))))
	      (let ((vnew (vct-subseq vn 3 8)))
		(if (fneq (vct-ref vnew 0) 3.0) (snd-display ";vct-subseq[3:8] ~A?" (vct-ref vnew 0)))
		(if (not (= (vct-length vnew) 6)) (snd-display ";vct-subseq[3:8] length: ~A?" (vct-length vnew))))
	      (vct-subseq vn 3 3 vs)
	      (if (or (fneq (vct-ref vs 0) 3.0)
		      (fneq (vct-ref vs 1) 0.0)
		      (fneq (vct-ref vs 2) 0.0))
		  (snd-display ";vct-subseq[3:3->vs] ~A?" vs))
	      (vct-subseq vn 0 32 vs)
	      (if (not (= (vct-length vs) 3)) (snd-display ";vct-subseq[0:32->vs] length: ~A?" (vct-length vs)))
	      (vct-subseq vn 2 3 vss)
	      (if (fneq (vct-ref vss 0) 2.0) (snd-display ";vct-subseq[2:3->vss] ~A?" (vct-ref vss 0)))
	      (vct-set! vb 8 123.0)
	      (vct-subseq vn 1 8 vb)
	      (if (fneq (vct-ref vb 0) 1.0) (snd-display ";vct-subseq[1:8->vb] ~A?" (vct-ref vb 0)))
	      (if (fneq (vct-ref vb 8) 123.0) (snd-display ";vct-subseq[1:8->vb][8] ~A?" (vct-ref vb 8))))
	    (vct-do! v0 (lambda (i) (* i .5)))
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v0 i) (* i .5)) (snd-display ";do v0[~D] = ~F?" i (vct-ref v0 i))))
	    (vct-map! v0 (lambda () 1.0))
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v0 i) 1.0) (snd-display ";map v0[~D] = ~F?" i (vct-ref v0 i))))
	    (vcts-do! v0 v1 v2 (lambda (num i) (list i 0.0 (* i 2))))
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v0 i) i) (snd-display ";dos v0[~D] = ~F?" i (vct-ref v0 i)))
	      (if (fneq (vct-ref v1 i) 0.0) (snd-display ";dos v1[~D] = ~F?" i (vct-ref v1 i)))
	      (if (fneq (vct-ref v2 i) (* i 2)) (snd-display ";dos v2[~D] = ~F?" i (vct-ref v2 i))))
	    (vcts-map! v0 v1 v2 (lambda (num) (list 0.0 1.0 2.0)))
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref v0 i) 0.0) (snd-display ";maps v0[~D] = ~F?" i (vct-ref v0 i)))
	      (if (fneq (vct-ref v1 i) 1.0) (snd-display ";maps v1[~D] = ~F?" i (vct-ref v1 i)))
	      (if (fneq (vct-ref v2 i) 2.0) (snd-display ";maps v2[~D] = ~F?" i (vct-ref v2 i))))
	    (let ((fd (open-sound-file)))
	      (close-sound-file fd 0)
	      (let ((frs (mus-sound-frames (if (little-endian?) "test.wav" "test.snd"))))
		(if (not (= frs 0))
		    (snd-display ";open-sound-file defaults: ~A" frs))))
	    (let ((fd (open-sound-file "hiho.snd" 1 :srate 22050 :comment "hiho is from snd-test")))
	      (vct->sound-file fd v2 10)
	      (close-sound-file fd 10)
	      (let ((var (catch #t (lambda () (vct->sound-file -1 v2 1)) (lambda args args))))
		(if (not (eq? (car var) 'out-of-range))
		    (snd-display ";vct->sound-file bad fd: ~A" var)))
	      (let ((v3 (make-vct 40)))
		(file->array "hiho.snd" 0 0 10 v3)
		(if (fneq (vct-ref v3 5) (vct-ref v2 5))
		    (snd-display ";vct->sound-file: ~A ~A?" v2 v3)))))
	  (if (fneq ((vct 1.0 2.0 3.0) 1) 2.0)
	      (snd-display ";(vct...) = ~A?" ((vct 1.0 2.0 3.0) 1)))
	  (let ((v1 (vct 1 2 3 4)))
	    (if (fneq (v1 1) 2.0)
		(snd-display ";(v1 1) = ~A?" (v1 1))))
	  (let ((nind (open-sound "oboe.snd")))
	    (set! (speed-control nind) .5)
	    (play-and-wait)
	    (apply-controls)
	    (close-sound nind))
	  (let ((v1 (make-vct 32)))
	    (vct-map! v1
		      (lambda ()
			(let ((v2 (make-vct 3)))
			  (vct-map! v2 (lambda () .1))
			  (vct-ref v2 0))))
	    (if (fneq (vct-ref v1 12) .1) (snd-display ";vct-map! twice: ~A" (vct-ref v1 12))))
	  (let ((hi (make-vct 3)))
	    (let ((tag (catch #t
			      (lambda () (vct-subseq hi 1 0))
			      (lambda args (car args)))))
	      (if (not (eq? tag 'out-of-range))
		  (snd-display ";vct-subseq 1 0: ~A" tag)))
	    (if (vct) (snd-display ";(vct) -> ~A" (vct)))
	    (let ((tag (catch #t (lambda () (make-vct 0)) (lambda args (car args)))))
	      (if (not (eq? tag 'out-of-range)) (snd-display ";make-vct 0 -> ~A" tag)))
	    (let ((ho (make-vct 3)))
	      (vct-add! hi ho 4)))
	  (let ((v0 (make-vct 5 .1))
		(v1 (make-vct 6 .2)))
	    (vct-add! v0 v1 2)
	    (if (not (vequal v0 (vct .1 .1 .3 .3 .3)))
		(snd-display ";vct-add + offset: ~A" v0)))
	  ))
      (run-hook after-test-hook 6)
      ))


;;; ---------------- test 7: colors ----------------

(define colormap-error-max 0.0)
(define cfneq (lambda (a b) (> (abs (- a b)) colormap-error-max)))
(define old-colormap-size (colormap-size))

(if (and (or full-test (= snd-test 7) (and keep-going (<= snd-test 7)))
	 (or (provided? 'snd-gtk)
	     (provided? 'snd-motif)))
    (letrec ((test-color
	      (lambda (lst)
		(if (not (null? lst))
		    (let* ((name (list-ref (car lst) 0))
			   (getfnc (list-ref (car lst) 1))
			   (setfnc (lambda (val) (set! (getfnc) val)))
			   (initval (list-ref (car lst) 2)))
		      (if (not (color? initval)) (snd-display ";~A not color?" initval))
		      ;; we'll get warnings here if the cell chosen didn't exactly match the one requested -- not a bug
		      ;; (if (not (equal? (getfnc) initval))
		      ;;	  (snd-display ";~A /= ~A (~A)?" name initval (getfnc)))
		      (setfnc beige)
		      (if (not (equal? (getfnc) beige))
			  (snd-display ";set-~A /= beige (~A)?" name (getfnc)))
		      (setfnc initval)
		      (test-color (cdr lst)))))))
      (run-hook before-test-hook 7)
      (if (not (provided? 'snd-rgb.scm)) (load "rgb.scm"))
      (let* ((c1 (catch 'no-such-color
			(lambda () (make-color 0 0 1))
			(lambda args #f)))
	     (c2 c1)
	     (c3 (catch 'no-such-color
			(lambda () (make-color 0 0 1))
			(lambda args #f))))
	(if (not (equal? c1 c2)) (snd-display ";color equal? ~A ~A?" c1 c2))
	(if (not (eq? c1 c2)) (snd-display ";color eq? ~A ~A?" c1 c2))
	(if (provided? 'snd-motif) (if (not (equal? c1 c3)) (snd-display ";diff color equal? ~A ~A?" c1 c3)))
	(if (eq? c1 c3) (snd-display ";diff color eq? ~A ~A?" c1 c3))
	(if (not (equal? (color->list c1) (list 0.0 0.0 1.0)))
	    (snd-display ";color->list: ~A ~A?" c1 (color->list c1))))
      (do ((i 0 (1+ i))) ((> i flag-colormap))
	(let ((val (colormap-ref i 0))
	      (true-val (list-ref (list '(0.0 0.0 0.0) '(0.0 0.0 0.0) '(0.0 0.0 0.0) '(0.0 1.0 1.0)
					'(0.0 0.0 7.01915007248035e-4) '(0.0 0.0 0.0) '(0.0 0.0 0.0)
					'(0.0 0.0 0.49999) '(1.0 0.0 0.0) '(1.0 0.0 0.0) '(0.0 0.0 1.0)
					'(1.0 0.0 1.0) '(0.0 0.500007629510948 0.4) '(1.0 0.0 0.0)
					'(1.0 0.0 0.0) '(0.0 0.0 0.0))
				  i)))
	  (if (not (feql val true-val))
	      (snd-display ";colormap-ref ~A: ~A (~A)" i val true-val))))
      (catch 'no-such-color
	     (lambda () 
	       (test-color
		(list
		 (list 'basic-color basic-color ivory2)
		 (list 'cursor-color cursor-color red)
		 (list 'data-color data-color black)
		 (list 'enved-waveform-color enved-waveform-color blue)
		 (list 'filter-control-waveform-color filter-control-waveform-color blue)
		 (list 'graph-color graph-color white)
		 (list 'highlight-color highlight-color ivory1)
		 (list 'listener-color listener-color alice-blue)
		 (list 'listener-text-color listener-text-color black)
		 (list 'mark-color mark-color red)
		 (list 'mix-color mix-color dark-gray)
		 (list 'position-color position-color ivory3)
		 (list 'pushed-button-color pushed-button-color lightsteelblue1)
		 (list 'sash-color sash-color light-green)
		 (list 'selected-data-color selected-data-color black)
		 (list 'selected-graph-color selected-graph-color white)
		 (list 'selection-color selection-color lightsteelblue1)
		 (list 'text-focus-color text-focus-color white)
		 (list 'zoom-color zoom-color ivory4)
		       
		 (list 'quit-button-color quit-button-color indian-red)
		 (list 'help-button-color help-button-color lightsteelblue2)
		 (list 'reset-button-color reset-button-color goldenrod1)
		 (list 'doit-button-color doit-button-color palegreen2)
		 (list 'doit-again-button-color doit-again-button-color darkolivegreen1)
		 ))

	       (let ((ind (open-sound "oboe.snd")))
		 (set! (selected-data-color) light-green)
		 (set! (data-color) blue)
		 (set! (selected-graph-color) black)
		 (let ((red (make-color 1.0 0.0 0.0)))
		   (set! (foreground-color ind 0 cursor-context) red)
		   (let ((col (foreground-color ind 0 cursor-context)))
		     (if (not (equal? col red))
			 (snd-display ";set foreground cursor color: ~A ~A" col red)))
		   (set! (foreground-color) blue)
		   (let ((col (foreground-color)))
		     (if (not (equal? col blue))
			 (snd-display ";set foreground-color: ~A ~A" col black)))
		   (set! (foreground-color ind) black)
		   (let ((col (foreground-color ind)))
		     (if (not (equal? col black))
			 (snd-display ";set foreground-color with ind: ~A ~A" col black))))
		 (set! (selected-graph-color) (make-color 0.96 0.96 0.86))
		 (set! (data-color) black)
		 (set! (selected-data-color) blue)
		 (set! (graph-color) white)
		 (close-sound ind)))
	     (lambda args args))

      (for-each 
       (lambda (n err)
	 (set! (colormap-size) n)
	 (set! colormap-error-max err)
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r (if (< x 3/4)
			   (* 7/8 x)
			   (- (* 11/8 x) 3/8)))
		    (g (if (< x 3/8)
			   (* 7/8 x)
			   (if (< x 3/4)
			       (- (* 29/24 x) 1/8)
			       (+ (* 7/8 x) 1/8))))
		    (b (if (< x 3/8)
			   (* 29/24 x)
			   (+ (* 7/8 x) 1/8)))
		    (rgb (colormap-ref bone-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";bone ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r (if (< x 4/5)
			   (* 5/4 x)
			   1.0))
		    (g (* 4/5 x))
		    (b (* 1/2 x))
		    (rgb (colormap-ref copper-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";copper ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r 0.0)
		    (g x)
		    (b (- 1.0 (/ g 2.0)))
		    (rgb (colormap-ref winter-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";winter ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r 1.0)
		    (g x)
		    (b 0.0)
		    (rgb (colormap-ref autumn-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";autumn ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r x)
		    (g (- 1.0 r))
		    (b 1.0)	     
		    (rgb (colormap-ref cool-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";cool ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r (if (< x 3/8)
			   (* 8/3 x)
			   1.0))
		    (g (if (< x 3/8)
			   0.0
			   (if (< x 3/4)
			       (- (* 8/3 x) 1.0)
			       1.0)))
		    (b (if (< x 3/4)
			   0.0
			   (- (* 4 x) 3)))
		    (rgb (colormap-ref hot-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";hot ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r (if (< x 3/8)
			   0.0
			   (if (< x 5/8)
			       (- (* 4 x) 3/2)
			       (if (< x 7/8)
				   1.0
				   (+ (* -4 x) 9/2)))))
		    (g (if (< x 1/8)
			   0.0
			   (if (< x 3/8)
			       (- (* 4 x) 1/2)
			       (if (< x 5/8)
				   1.0
				   (if (< x 7/8)
				       (+ (* -4 x) 7/2)
				       0.0)))))
		    (b (if (< x 1/8)
			   (+ (* 4 x) 1/2)
			   (if (< x 3/8)
			       1.0
			       (if (< x 5/8)
				   (+ (* -4 x) 5/2)
				   0.0))))
		    (rgb (colormap-ref jet-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";jet ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r (if (< x 3/8)
			   (* 14/9 x)
			   (+ (* 2/3 x) 1/3)))
		    (g (if (< x 3/8)
			   (* 2/3 x)
			   (if (< x 3/4)
			       (- (* 14/9 x) 1/3)
			       (+ (* 2/3 x) 1/3))))			
		    (b (if (< x 3/4)
			   (* 2/3 x)
			   (- (* 2 x) 1)))
		    (rgb (colormap-ref pink-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";pink ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r 1.0)
		    (g x)
		    (b (- 1.0 g))
		    (rgb (colormap-ref spring-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";spring ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r x)
		    (g x)
		    (b x)
		    (rgb (colormap-ref gray-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";gray ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r 0.0)
		    (g 0.0)
		    (b 0.0)
		    (rgb (colormap-ref black-and-white-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";black-and-white ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r x)
		    (g (+ 0.5 (/ r 2)))
		    (b 0.4)
		    (rgb (colormap-ref summer-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";summer ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (r (if (< x 2/5)
			   1.0
			   (if (< x 3/5)
			       (+ (* -5 x) 3)
			       (if (< x 4/5)
				   0.0
				   (- (* 10/3 x) 8/3)))))
		    (g (if (< x 2/5)
			   (* 5/2 x)
			   (if (< x 3/5)
			       1.0
			       (if (< x 4/5)
				   (+ (* -5 x) 4)
				   0.0))))
		    (b (if (< x 3/5)
			   0.0
			   (if (< x 4/5)
			       (- (* 5 x) 3)
			       1.0)))
		    (rgb (colormap-ref rainbow-colormap x))
		    (r1 (list-ref rgb 0))
		    (g1 (list-ref rgb 1))
		    (b1 (list-ref rgb 2)))
	       (if (and (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		   (snd-display ";rainbow ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1))))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (rgb (colormap-ref prism-colormap x)))
	       (if (and (< x (- 1.0 (/ 1.0 n)))
			(not (feql rgb '(1 0 0)))
			(not (feql rgb '(1 0.5 0)))
			(not (feql rgb '(1 1 0)))
			(not (feql rgb '(0 1 0)))
			(not (feql rgb '(0 0 1)))
			(not (feql rgb '(.6667 0 1))))
		   (snd-display ";prism ~A" rgb)))))
	 
	 (let ((x 0.0))
	   (do ((i 0 (1+ i))) ((= i 10))
	     (let* ((x (random 1.0))
		    (rgb (colormap-ref flag-colormap x)))
	       (if (and (< x (- 1.0 (/ 1.0 n)))
			(not (feql rgb '(1 0 0)))
			(not (feql rgb '(1 1 1)))
			(not (feql rgb '(0 0 1)))
			(not (feql rgb '(0 0 0))))
		   (snd-display ";flag: ~A" rgb)))))
	 )
       (list 512 64)
       (list 0.005 0.04))
      
      (let ((ind (add-colormap "white" (lambda (size) (list (make-vct size 1.0) (make-vct size 1.0) (make-vct size 1.0))))))
	(if (not (feql (colormap-ref ind 0.5) '(1.0 1.0 1.0)))
	    (snd-display ";white colormap: ~A" (colormap-ref ind 0.5)))
	(let ((tag (catch #t (lambda () (set! (colormap) ind)) (lambda args args))))
	  (if (or (eq? tag 'no-such-colormap)
		  (not (= (colormap) ind)))
	      (snd-display ";colormap white: ~A ~A ~A" tag ind (colormap))))
	(if (not (string=? (colormap-name ind) "white"))
	    (snd-display ";white colormap name: ~A" (colormap-name ind))))
      
      (let ((tag (catch #t (lambda () (delete-colormap 1234)) (lambda args (car args)))))
	(if (not (eq? tag 'no-such-colormap))
	    (snd-display ";delete-colormap 1234: ~A" tag)))
      (let ((tag (catch #t (lambda () (colormap-ref 1234 0.5)) (lambda args (car args)))))
	(if (not (eq? tag 'no-such-colormap))
	    (snd-display ";colormap-ref 1234: ~A" tag)))
      (let ((tag (catch #t (lambda () (colormap-ref -1 0.5)) (lambda args (car args)))))
	(if (not (eq? tag 'no-such-colormap))
	    (snd-display ";colormap-ref -1: ~A" tag)))
      (let ((tag (catch #t (lambda () (set! (colormap) 1234)) (lambda args (car args)))))
	(if (not (eq? tag 'no-such-colormap))
	    (snd-display "; set colormap 1234: ~A" tag)))
      (let ((tag (catch #t (lambda () (set! (colormap) -1)) (lambda args (car args)))))
	(if (not (eq? tag 'no-such-colormap))
	    (snd-display "; set colormap -1: ~A" tag)))
      (let ((tag (catch #t (lambda () (colormap-ref copper-colormap 2.0)) (lambda args (car args)))))
	(if (not (eq? tag 'out-of-range))
	    (snd-display ";colormap-ref 2.0: ~A" tag)))
      
      (set! (colormap-size) old-colormap-size)
      (if (not (= (colormap-size) old-colormap-size))
	  (snd-display ";set colormap-size: ~A ~A" (colormap-size) old-colormap-size))
      
      (if (not (string=? (colormap-name black-and-white-colormap) "black-and-white"))
	  (snd-display ";black-and-white: ~A" (colormap-name black-and-white-colormap)))
      (if (not (string=? (colormap-name gray-colormap) "gray"))
	  (snd-display ";gray: ~A" (colormap-name gray-colormap)))
      (if (not (string=? (colormap-name rainbow-colormap) "rainbow"))
	  (snd-display ";rainbow: ~A" (colormap-name rainbow-colormap)))
      
      (let ((purple-cmap (add-colormap "purple" 
				       (lambda (size) 
					 (let ((r (make-vct size))
					       (g (make-vct size))
					       (b (make-vct size))
					       (incr (exact->inexact (/ 256.0 size)))
					       (er (list 0 60 60 116 128 252 192 252 256 60))
					       (eg (list 0 0  64 0   128 252 192 252 256 0))
					       (eb (list 0 80        128 252 192 0   256 80)))
					   (do ((i 0 (1+ i))
						(x 0.0 (+ x incr)))
					       ((= i size))
					     (vct-set! r i (exact->inexact (/ (envelope-interp x er) 256.0)))
					     (vct-set! g i (exact->inexact (/ (envelope-interp x eg) 256.0)))
					     (vct-set! b i (exact->inexact (/ (envelope-interp x eb) 256.0))))
					   (list r g b)))))
	    (sin-cmap (add-colormap "sin" 
				    (lambda (size) 
				      (let ((r (make-vct size))
					    (g (make-vct size))
					    (b (make-vct size))
					    (incr (exact->inexact (/ (* 2 3.14159) size))))
					(do ((i 0 (1+ i))
					     (x 0.0 (+ x incr)))
					    ((= i size))
					  (vct-set! r i (abs (sin (* 1.5 x))))
					  (vct-set! g i (abs (sin (* 3.5 x))))
					  (vct-set! b i (abs (sin (* 2.5 x)))))
					(list r g b)))))
	    (another-sin-cmap (add-colormap "another-sin" 
					    (lambda (size) 
					      (let ((r (make-vct size))
						    (g (make-vct size))
						    (b (make-vct size))
						    (incr (exact->inexact (/ (* 2 3.14159) size))))
						(do ((i 0 (1+ i))
						     (x 0.0 (+ x incr)))
						    ((= i size))
						  (vct-set! r i (abs (sin (* 2.5 x))))
						  (vct-set! g i (abs (sin (* 3.5 x))))
						  (vct-set! b i (abs (sin (* 4.5 x)))))
					   (list r g b))))))
	(delete-colormap pink-colormap)
	(let ((tag (catch #t (lambda () (set! (colormap) pink-colormap)) (lambda args args))))
	  (if (or (not (eq? (car tag) 'no-such-colormap))
		  (= (colormap) pink-colormap))
	      (snd-display ";delete pink colormap: ~A ~A ~A" tag pink-colormap (colormap))))

	(for-each
	 (lambda (n)
	   (set! (colormap-size) n)
	   (let ((x 0.0))
	     (do ((i 0 (1+ i))) ((= i 10))
	       (let* ((x (random 1.0))
		      (r (if (< x 4/5) (* 5/4 x) 1.0))
		      (g (* 4/5 x))
		      (b (* 1/2 x))
		      (rgb (colormap-ref copper-colormap x))
		      (r1 (list-ref rgb 0))
		      (g1 (list-ref rgb 1))
		      (b1 (list-ref rgb 2)))
		 (if (and (> n 2) (< x (- 1.0 (/ 1.0 n))) (or (cfneq r r1) (cfneq g g1) (cfneq b b1)))
		     (snd-display ";copper size reset ~A: ~,3F (~,3F): ~{~,3F ~} ~{~,3F ~}" 
				  n x (max (abs (- r r1)) (abs (- g g1)) (abs (- b b1))) (list r g b) (list r1 g1 b1)))))))
	 (list 1024 256 2 512))
	(set! (colormap-size) 512))


      (run-hook after-test-hook 7)
      ))


(reset-hook! graph-hook)
(clear-sincs)

;;; ---------------- test 8: clm ----------------

(if (not (provided? 'snd-moog.scm)) (load "moog.scm"))

(define (jc-reverb decay-dur low-pass volume amp-env)
  "(jc-reverb decay-dur low-pass volume amp-env) is the old Chowning reverberator: (jc-reverb 2.0 #f .1 #f)"
  (let* ((allpass1 (make-all-pass -0.700 0.700 1051))
	 (allpass2 (make-all-pass -0.700 0.700  337))
	 (allpass3 (make-all-pass -0.700 0.700  113))
	 (comb1 (make-comb 0.742 4799))
	 (comb2 (make-comb 0.733 4999))
	 (comb3 (make-comb 0.715 5399))
	 (comb4 (make-comb 0.697 5801))
	 (outdel1 (make-delay (round (* .013 (srate)))))
	 (dur (+ decay-dur (/ (frames) (srate))))
	 (envA (if amp-env (make-env :envelope amp-env :scaler volume :duration dur) #f))
	 (len (round (* dur (srate)))))
    (map-chan
     (let ((comb-sum 0.0)
	   (comb-sum-1 0.0)
	   (comb-sum-2 0.0)
	   (all-sums 0.0)
	   (delA 0.0)
	   (delB 0.0))
       (lambda (inval)
	 (let ((allpass-sum (all-pass allpass3 (all-pass allpass2 (all-pass allpass1 inval)))))
	   (set! comb-sum-2 comb-sum-1)
	   (set! comb-sum-1 comb-sum)
	   (set! comb-sum 
		 (+ (comb comb1 allpass-sum)
		    (comb comb2 allpass-sum)
		    (comb comb3 allpass-sum)
		    (comb comb4 allpass-sum)))
	   (if low-pass
	       (set! all-sums (+ (* .25 (+ comb-sum comb-sum-2)) (* .5 comb-sum-1)))
	       (set! all-sums comb-sum))
	   (+ inval
	      (if envA
		  (* (env envA) (delay outdel1 all-sums))
		  (* volume (delay outdel1 all-sums)))))))
     0 (round (* dur (srate))))))

;;; -------- scissor-tailed flycatcher
;;;
;;; mix a scissor-tailed flycatcher call into the current sound
;;; see bird.scm for lots more birds

(if (not (provided? 'snd-bird.scm)) (load "bird.scm"))

(define (scissor begin-time)
  "(scissor beg) is the scissor-tailed flycatcher"
  (let ((scissorf '(0 0  40 1  60 1  100 0)))
    (bigbird begin-time 0.05 1800 1800 .2 
	     scissorf 
	     '(0 0  25 1  75 1  100 0) 
	     '(1 .5  2 1  3 .5  4 .1  5 .01))))


;;; -------- fm-violin
;;;
;;; here we're using the keyword stuff in guile/ice-9/optargs.scm
;;; CLM version is v.ins, C version is in sndlib.html
;;; a version treating the entire violin as a generator is in fmv.scm.

(define fm-violin 
  (lambda* (startime dur frequency amplitude #:key
		     (fm-index 1.0)
		     (amp-env '(0 0  25 1  75 1  100 0))
		     (periodic-vibrato-rate 5.0) 
		     (random-vibrato-rate 16.0)
		     (periodic-vibrato-amplitude 0.0025) 
		     (random-vibrato-amplitude 0.005)
		     (noise-amount 0.0) 
		     (noise-freq 1000.0)
		     (ind-noise-freq 10.0) 
		     (ind-noise-amount 0.0)
		     (amp-noise-freq 20.0) 
		     (amp-noise-amount 0.0)
		     (gliss-env '(0 0  100 0)) 
		     (glissando-amount 0.0) 
		     (fm1-env '(0 1  25 .4  75 .6  100 0))  
		     (fm2-env '(0 1  25 .4  75 .6  100 0)) 
		     (fm3-env '(0 1  25 .4  75 .6  100 0))
		     (fm1-rat 1.0) 
		     (fm2-rat 3.0)	 
		     (fm3-rat 4.0)                    
		     (fm1-index #f) 
		     (fm2-index #f) 
		     (fm3-index #f)
		     (base 1.0)
		     (reverb-amount 0.01)
		     (degree #f) (distance 1.0) (degrees #f)
		     #:allow-other-keys)
	   (let* ((beg (floor (* startime (srate))))
		  (len (floor (* dur (srate))))
		  (end (+ beg len))
		  (frq-scl (hz->radians frequency))
		  (modulate (not (zero? fm-index)))
		  (maxdev (* frq-scl fm-index))
		  (logfreq (log frequency))
		  (sqrtfreq (sqrt frequency))
		  (index1 (or fm1-index (min pi (* maxdev (/ 5.0 logfreq)))))
		  (index2 (or fm2-index (min pi (* maxdev 3.0 (/ (- 8.5 logfreq) (+ 3.0 (* frequency .001)))))))
		  (index3 (or fm3-index (min pi (* maxdev (/ 4.0 sqrtfreq)))))
		  (easy-case (and (zero? noise-amount)
				  (equal? fm1-env fm2-env)
				  (equal? fm1-env fm3-env)
				  (= fm1-rat (floor fm1-rat))
				  (= fm2-rat (floor fm2-rat))
				  (= fm3-rat (floor fm3-rat))))
		  (coeffs (and easy-case modulate
			       (partials->polynomial
				(list fm1-rat index1
				      (floor (/ fm2-rat fm1-rat)) index2
				      (floor (/ fm3-rat fm1-rat)) index3))))
		  (norm (or (and easy-case modulate 1.0) index1))
		  (carrier (make-oscil frequency))
		  (fmosc1  (and modulate (make-oscil (* fm1-rat frequency))))
		  (fmosc2  (and modulate (or easy-case (make-oscil (* fm2-rat frequency)))))
		  (fmosc3  (and modulate (or easy-case (make-oscil (* fm3-rat frequency)))))
		  (ampf  (make-env amp-env :scaler amplitude :base base :duration dur))
		  (indf1 (and modulate (make-env fm1-env norm :duration dur)))
		  (indf2 (and modulate (or easy-case (make-env fm2-env index2 :duration dur))))
		  (indf3 (and modulate (or easy-case (make-env fm3-env index3 :duration dur))))
		  (frqf (make-env gliss-env (* glissando-amount frq-scl) :duration dur))
		  (pervib (make-triangle-wave periodic-vibrato-rate (* periodic-vibrato-amplitude frq-scl)))
		  (ranvib (make-rand-interp random-vibrato-rate (* random-vibrato-amplitude frq-scl)))
		  (fm-noi (and (not (= 0.0 noise-amount))
			       (make-rand noise-freq (* pi noise-amount))))
		  (ind-noi (and (not (= 0.0 ind-noise-amount)) 
				(not (= 0.0 ind-noise-freq))
				(make-rand-interp ind-noise-freq ind-noise-amount)))
		  (amp-noi (and (not (= 0.0 amp-noise-amount)) 
				(not (= 0.0 amp-noise-freq))
				(make-rand-interp amp-noise-freq amp-noise-amount)))
		  (vib 0.0) 
		  (modulation 0.0)
		  (loc (make-locsig :channels (channels) :degree (or degree degrees (random 90.0)) :reverb reverb-amount :distance distance))
		  (fuzz 0.0)
		  (ind-fuzz 1.0)
		  (amp-fuzz 1.0)
		  (out-data (make-vct len)))
	     (vct-map! out-data
		       (lambda ()
			 (if (not (= 0.0 noise-amount))
			     (set! fuzz (rand fm-noi)))
			 (set! vib (+ (env frqf) (triangle-wave pervib) (rand-interp ranvib)))
			 (if ind-noi (set! ind-fuzz (+ 1.0 (rand-interp ind-noi))))
			 (if amp-noi (set! amp-fuzz (+ 1.0 (rand-interp amp-noi))))
			 (if modulate
			     (if easy-case
				 (set! modulation
				       (* (env indf1) 
					  (polynomial coeffs (oscil fmosc1 vib)))) ;(* vib fm1-rat)??
				 (set! modulation
				       (+ (* (env indf1) (oscil fmosc1 (+ (* fm1-rat vib) fuzz)))
					  (* (env indf2) (oscil fmosc2 (+ (* fm2-rat vib) fuzz)))
					  (* (env indf3) (oscil fmosc3 (+ (* fm3-rat vib) fuzz)))))))
			 (* (env ampf) amp-fuzz
			    (oscil carrier (+ vib (* ind-fuzz modulation))))))
	     (if (= (channels) 2)
		 (let ((bsamps (vct-copy out-data)))
		   (mix-vct (vct-scale! bsamps (locsig-ref loc 1)) beg #f 1 #f)
		   (mix-vct (vct-scale! out-data (locsig-ref loc 0)) beg #f 0 #f))
		 (mix-vct out-data beg #f 0 #f)))))

; (fm-violin 0 1 440 .1 :fm-index 2.0)
(define (fltit)
  "(fltit) returns a time-varying filter: (map-chan (fltit))"
  (let* ((coeffs (list .1 .2 .3 .4 .4 .3 .2 .1))
	 (flt (make-fir-filter 8 (list->vct coeffs)))
	 (es (make-vector 8)))
    (do ((i 0 (1+ i)))
	((= i 8))
      (vector-set! es i (make-env (list 0 (list-ref coeffs i) 1 0) :end 100)))
    (vector-set! es 5 (make-env '(0 .4 1 1) :duration 1.0))
    (lambda (x)
      (let ((val (fir-filter flt x))
	    (xcof (mus-xcoeffs flt)))
	(do ((i 0 (1+ i)))
	    ((= i 8))
	  (vct-set! xcof i (env (vector-ref es i))))
	val))))

(define (freq-sweep dur)
  (let ((phase 0.0)
	(freq 0.0)
	(incr (/ pi (* dur 1.05 (mus-srate)))))
    (map-channel 
     (lambda (y)
       (let ((val (sin phase))) 
	 (set! phase (+ phase freq)) 
	 (set! freq (+ freq incr))
	 (* .5 val))))))

(define* (make-ssb-am-1 freq-1 #:optional (order 40))
  (let* ((freq freq-1)
	 (carrier-freq (abs freq-1))
	 (cos-carrier (make-oscil carrier-freq (* .5 pi)))
	 (sin-carrier (make-oscil carrier-freq))
	 (dly (make-delay order))
	 (hlb (make-hilbert-transform order)))
    (lambda (y)
      (let ((ccos (oscil cos-carrier))
	    (csin (oscil sin-carrier))
	    (yh (hilbert-transform hlb y))
	    (yd (delay dly y)))
	(if (> freq 0.0)
	    (- (* ccos yd) ; shift up
	       (* csin yh))
	    (+ (* ccos yd) ; shift down
	       (* csin yh)))))))

(define (ssb-am-1 gen y) (gen y))

(define (rough-spectrum ind)
  (let ((r (make-sample-reader 0 ind 0))
	(spect (make-vct 10))
	(mx 0.0))
    (do ((i 0 (1+ i)))
	((= i 10))
      (let ((sum 0.0))
	(do ((j 0 (1+ j)))
	    ((= j 1000))
	  (let ((val (r)))
	    (set! sum (+ sum (* val val)))))
	(if (> sum mx) (set! mx sum))
	(vct-set! spect i sum)))
    (vct-scale! spect (/ 1.0 mx))))

(define* (print-and-check gen name desc insp #:optional insp2)
  (if (not (string=? (mus-name gen) name))
      (snd-display ";mus-name ~A: ~A?" name (mus-name gen)))
  (if (not (string=? (mus-describe gen) desc))
      (snd-display ";mus-describe ~A: ~A?" (mus-name gen) (mus-describe gen)))
  (if (and (not (string=? (mus-inspect gen) insp))
	   (or (not insp2)
	       (not (string=? (mus-inspect gen) insp2))))
      (snd-display ";mus-inspect ~A: ~A?" (mus-name gen) (mus-inspect gen)))
  (let ((egen gen))
    (if (not (equal? egen gen))
	(snd-display ";equal? ~A: ~A?" gen egen))))

(define (test-gen-equal g0 g1 g2)
  ;; g0 = g1 at start != g2
  (let ((g3 g0)
	(gad (make-frame 2)))
    (if (not (eq? g0 g3))
	(snd-display ";let ~A eq? ~A ~A" (mus-name g0) g0 g3))
    (if (eq? g0 g1)
	(snd-display ";arg ~A eq? ~A ~A" (mus-name g0) g0 g1))
    (if (not (equal? g0 g1))
	(snd-display ";~A equal? ~A ~A" (mus-name g0) g0 g1))
    (if (equal? g0 g2)
	(snd-display ";~A not equal? ~A ~A" (mus-name g0) g0 g2))
    (if (equal? g0 gad)
	(snd-display ";~A equal frame? ~A ~A" (mus-name g0) g0 gad))
    (g0)
    (g3)
    (g3)
    (if (not (eq? g0 g3))
	(snd-display ";run let ~A eq? ~A ~A" (mus-name g0) g0 g3))
    (if (eq? g0 g1)
	(snd-display ";arg ~A eq? ~A ~A" (mus-name g0) g0 g1))
    (if (equal? g0 g1)
	(snd-display ";run ~A equal? ~A ~A" (mus-name g0) g0 g1))
    (if (equal? g0 g2)
	(snd-display ";run ~A not equal? ~A ~A" (mus-name g0) g0 g2))))

(define (fm-test gen)
  (set! (mus-frequency gen) 0.0)
  (set! (mus-phase gen) 0.0)
  (gen 0.0)
  (if (fneq (mus-phase gen) 0.0) (snd-display ";~A phase(0): ~A" gen (mus-phase gen)))
  (gen 1.0)
  (if (fneq (mus-phase gen) 1.0) (snd-display ";~A phase(1): ~A" gen (mus-phase gen)))
  (gen 0.0)
  (if (fneq (mus-phase gen) 1.0) (snd-display ";~A phase(1, 0): ~A" gen (mus-phase gen)))
  (set! (mus-frequency gen) (radians->hz 2.0))
  (gen 0.0)
  (if (fneq (mus-phase gen) 3.0) (snd-display ";~A phase(1, 2): ~A ~A" gen (mus-phase gen) (mus-frequency gen)))
  (gen 1.0)
  (if (fneq (mus-phase gen) 6.0) (snd-display ";~A phase(3, 2, 1): ~A ~A" gen (mus-phase gen) (mus-frequency gen)))
  (do ((i 0 (1+ i))) ((= i 10)) (gen 10.0))
  (if (fneq (mus-phase gen) (+ 26 (- 100 (* 2 pi 20)))) (snd-display ";~A phase (over): ~A ~A" gen (mus-phase gen) (mus-frequency gen))))

(if (or full-test (= snd-test 8) (and keep-going (<= snd-test 8)))
    (do ((clmtest 0 (1+ clmtest))) ((= clmtest tests))
      (if (= clmtest 0) (run-hook before-test-hook 8))
      (log-mem clmtest)
      (set! (mus-srate) 22050)
      (let ((samps (seconds->samples 1.0))
	    (secs (samples->seconds 22050)))
	(if (not (= samps 22050)) (snd-display ";seconds->samples: ~A" samps))
	(if (fneq secs 1.0) (snd-display ";samples->seconds: ~A" secs)))
      (if (not (= (mus-file-buffer-size) 8192)) (snd-display ";mus-file-buffer-size: ~D?" (mus-file-buffer-size)))
      (let ((var (catch #t (lambda () (set! (mus-file-buffer-size) #f)) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";mus-file-buffer-size bad size: ~A" var)))
      (set! (mus-file-buffer-size) 128)
      (if (not (= (mus-file-buffer-size) 128)) (snd-display ";mus-file-buffer-size: ~D?" (mus-file-buffer-size)))
      (set! (mus-file-buffer-size) 8192)
      
      (if (not (= (mus-array-print-length) 8)) (snd-display ";mus-array-print-length: ~D?" (mus-array-print-length)))
      (set! (mus-array-print-length) 32)
      (if (not (= (mus-array-print-length) 32)) (snd-display ";set mus-array-print-length: ~D?" (mus-array-print-length)))
      (set! (mus-array-print-length) 8)
      (if (fneq (mus-srate) 22050.0) (snd-display ";mus-srate: ~F?" (mus-srate)))
      (if (fneq (hz->radians 1.0) 2.84951704088598e-4) (snd-display ";hz->radians: ~F?" (hz->radians 1.0)))
      (if (fneq (radians->hz 2.84951704088598e-4) 1.0) (snd-display ";radians->hz: ~F?" (radians->hz 2.84951704088598e-4)))
      (if (fneq (radians->degrees 1.0) 57.2957801818848) (snd-display ";radians->degrees: ~F?" (radians->degrees 1.0)))
      (if (fneq (degrees->radians 57.2957801818848) 1.0) (snd-display ";degrees->radians: ~F?" (degrees->radians 57.2957801818848)))
      (if (fneq (linear->db .25) -12.0411996841431) (snd-display ";linear->db: ~F?" (linear->db .25)))
      (if (fneq (db->linear -12.0411996841431) .25) (snd-display ";db->linear: ~F?" (db->linear -12.0411996841431)))
      (if (fneq (hz->radians 1.0) (in-hz 1.0)) (snd-display ";in-hz: ~F? " (in-hz 1.0)))
      (if (fneq (ring-modulate .4 .5) .2) (snd-display ";ring-modulate: ~F?" (ring-modulate .4 .5)))
      (if (fneq (amplitude-modulate 1.0 .5 .4) .7) (snd-display ";amplitude-modulate: ~F?" (amplitude-modulate 1.0 .5 .4)))
      (if (fneq (contrast-enhancement 0.1 0.75) (sin (+ (* 0.1 (/ pi 2)) (* .75 (sin (* 0.1 2.0 pi))))))
	  (snd-display ";contrast-enhancement: ~F (0.562925306221587)" (contrast-enhancement 0.1 0.75)))
      (let ((lv0 (partials->polynomial '(1 1 2 1) 1))
	    (lv1 (partials->polynomial '(1 1 2 1) 0))
	    (lv2 (partials->polynomial '(1 1 2 1 3 1 5 1) 1))
	    (lv3 (partials->polynomial '(1 1 2 1 3 1 5 1) 0))
	    (lv4 (partials->polynomial '(1 1 2 .5 3 .1 6 .01) 1))
	    (lv5 (partials->polynomial '(1 1 2 .5 3 .1 6 .01) 0))
	    (lv6 (partials->polynomial '(1 9 2 3 3 5 4 7 5 1))) ; MLB
	    (lv7 (partials->polynomial '(7 1)))
	    (lv7a (partials->polynomial '(7 1) 1)) ; first kind (default)
	    (lv8 (partials->polynomial '(7 1) 0)) ; second kind (confusing parameter!)
	    )
	(if (not (fveql lv0 '(-1.000 1.000 2.000) 0)) (snd-display ";partials->polynomial(1): ~A?" lv0))
	(if (not (fveql lv1 '(1.000 2.000 0.0) 0)) (snd-display ";partials->polynomial(2): ~A?" lv1))
	(if (not (fveql lv2 '(-1.000 3.000 2.000 -16.000 0.000 16.000) 0)) (snd-display ";partials->polynomial(3): ~A?" lv2))
	(if (not (fveql lv3 '(1.000 2.000 -8.000 0.000 16.000 0.000) 0)) (snd-display ";partials->polynomial(4): ~A?" lv3))
	(if (not (fveql lv4 '(-0.510 0.700 1.180 0.400 -0.480 0.000 0.320) 0)) (snd-display ";partials->polynomial(5): ~A?" lv4))
	(if (not (fveql lv5 '(0.900 1.060 0.400 -0.320 0.000 0.320 0.000) 0)) (snd-display ";partials->polynomial(6): ~A?" lv5))
	(if (not (vequal lv6 (vct 4.000 -1.000 -50.000 0.000 56.000 16.000))) (snd-display ";partials->polynomial(7): ~A?" lv6))
	(if (not (vequal lv7 (vct 0.000 -7.000 0.000 56.000 0.000 -112.000 0.000 64.000))) (snd-display ";partials->polynomial(8): ~A?" lv7))
	(if (not (vequal lv8 (vct -1.000 0.000 24.000 0.000 -80.000 0.000 64.000 0.000))) (snd-display ";partials->polynomial(9): ~A?" lv8))
	(if (not (vequal lv7 lv7a)) (snd-display ";partials->polynomial kind=1? ~A ~A" lv7 lv7a))
	)
      
      ;; check phase-quadrature cancellations
      (let ((cos-coeffs (partials->polynomial '(1 1 2 1) 1))
	    (sin-coeffs (partials->polynomial '(1 1 2 1) 0)))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 440.0) 22050.0))))
	    ((= i 1100))
	  (let* ((x (cos a))
		 (y (sin a))
		 (cax (polynomial cos-coeffs x))
		 (sax (polynomial sin-coeffs x))
		 (upper (- (* (cos (* 2 a)) cax) (* (sin (* 2 a)) (* y sax))))
		 (lower (+ (* (cos (* 2 a)) cax) (* (sin (* 2 a)) (* y sax))))
		 (upper2 (+ (cos (* a 3)) (cos (* a 4))))
		 (lower2 (+ 1.0 (cos a))))
	    (if (or (fneq upper upper2)
		    (fneq lower lower2))
		(snd-display ";~A ~A, ~A ~A" upper upper2 lower lower2)))))

      (let* ((amps (list->vct '(1.0)))
	     (phases (list->vct '(0.0)))
	     (val (sine-bank amps phases)))
	(if (fneq val 0.0) (snd-display ";sine-bank: ~A 0.0?" val))
	(vct-set! phases 0 (/ pi 2))
	(set! val (sine-bank amps phases))
	(if (fneq val 1.0) (snd-display ";sine-bank: ~A 1.0?" val))
	(set! amps (list->vct '(0.5 0.25 1.0)))
	(set! phases (list->vct '(1.0 0.5 2.0)))
	(set! val (sine-bank amps phases))
	(if (fneq val 1.44989) (snd-display ";sine-bank: ~A 1.449?" val)))
      
      (let ((rdat (make-vct 16))
	    (idat (make-vct 16))
	    (vdat (make-vct 16)))
	(vct-set! rdat 0 1.0)
	(vct-set! vdat 0 1.0)
	(let ((v0 (spectrum rdat idat (make-fft-window rectangular-window 16) 1)) ;rectangular here to avoid clobbering 0-th data point
	      (v1 (snd-spectrum vdat rectangular-window 16 #t)))
	  (do ((i 0 (1+ i)))
	      ((= i 8)) ;should all be 1.0 (impulse in)
	    (if (fneq (vct-ref v0 i) (vct-ref v1 i))
		(snd-display ";spectra not equal: ~A ~A" v0 v1))))
	(let ((v0 (spectrum rdat idat (make-fft-window rectangular-window 17) 1)) ;rectangular here to avoid clobbering 0-th data point
	      (v1 (snd-spectrum vdat rectangular-window 16 #t)))
	  (do ((i 0 (1+ i)))
	      ((= i 8)) ;should all be 1.0 (impulse in)
	    (if (fneq (vct-ref v0 i) (vct-ref v1 i))
		(snd-display ";spectra not equal: ~A ~A" v0 v1))))
	(let ((var (catch #t (lambda () (spectrum rdat idat #f -1)) (lambda args args))))
	  (if (or (vct? var) 
		  (not (eq? (car var) 'out-of-range)))
	      (snd-display ";spectrum bad type: ~A" var))))
      
      (let ((rdat (make-vct 16))
	    (idat (make-vct 16))
	    (xdat (make-vct 16))
	    (ydat (make-vct 16))
	    (rvec (make-vector 16 0.0))
	    (ivec (make-vector 16 0.0)))
	(vct-set! rdat 0 1.0)
	(vct-set! idat 1 1.0)
	(vct-set! xdat 0 1.0)
	(vct-set! ydat 1 1.0)
	(vector-set! rvec 0 1.0)
	(vector-set! ivec 1 1.0)
	(let ((v0 (convolution rdat idat 8))
	      (vt (vct-convolve! rvec ivec)) ; oops... (should vectors be supported at all?)
	      (v1 (vct-convolve! xdat ydat)))
	  (if (not (vector? vt)) (snd-display ";convolve with vector returns: ~A" vt))
	  (if (or (fneq (vct-ref v0 0) 0.0) (fneq (vct-ref v0 1) 1.0)) (snd-display ";vct convolution: ~A?" v0))
	  (if (or (fneq (vector-ref vt 0) 0.0) (fneq (vector-ref vt 1) 1.0)) (snd-display ";vector convolution: ~A?" v0))
	  (if (or (fneq (vct-ref v1 0) 0.0) (fneq (vct-ref v1 1) 1.0)) (snd-display ";vct-convolve!: ~A?" v1))
	  (do ((i 0 (1+ i)))
	      ((= i 8)) 
	    (if (or (fneq (vct-ref v0 i) (vct-ref v1 i))
		    (fneq (vct-ref v0 i) (vector-ref vt i)))
		(snd-display ";convolutions not equal: ~A ~A ~A" v0 v1 vt))))
	(let ((var (catch #t (lambda () (convolution rdat idat -1)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";convolution bad len: ~A" var)))
	(convolution rdat idat 20)
	(set! idat (make-vct 8))
	(convolution rdat idat 20))
      
      (let ((rdat (make-vct 16))
	    (idat (make-vct 16))
	    (xdat (make-vct 16))
	    (ydat (make-vct 16)))
	(vct-set! rdat 3 1.0)
	(vct-set! xdat 3 1.0)
	(fft rdat idat 1)
	(mus-fft xdat ydat 16 1)
	(if (fneq (vct-ref rdat 0) (vct-ref xdat 0)) (snd-display ";ffts: ~A ~A?" rdat xdat))
	(fft rdat idat -1)
	(mus-fft xdat ydat 17 -1) ; mistake is deliberate
	(do ((i 0 (1+ i)))
	    ((= i 16))
	  (if (or (and (= i 3) (or (fneq (vct-ref rdat i) 16.0) (fneq (vct-ref xdat i) 16.0)))
		  (and (not (= i 3)) (or (fneq (vct-ref rdat i) 0.0) (fneq (vct-ref xdat i) 0.0))))
	      (snd-display ";fft real[~D]: ~A ~A?" i (vct-ref rdat i) (vct-ref xdat i)))
	  (if (or (fneq (vct-ref idat i) 0.0) (fneq (vct-ref ydat i) 0.0))
	      (snd-display ";fft imag[~D]: ~A ~A?" i (vct-ref idat i) (vct-ref ydat i))))
	(let ((var (catch #t (lambda () (mus-fft xdat ydat -1 0)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";mus-fft bad len: ~A" var))))
      
      (let ((rdat (make-vct 20))
	    (idat (make-vct 19)))
	(vct-set! rdat 3 1.0)
	(mus-fft rdat idat)
	(convolution rdat idat)
	(spectrum rdat idat #f))
      
      (let ((rdat (make-vector 16))
	    (idat (make-vector 16)))
	(do ((i 0 (1+ i)))
	    ((= i 16))
	  (vector-set! rdat i 0.0)
	  (vector-set! idat i 0.0))
	(vector-set! rdat 3 1.0)
	(fft rdat idat 1)
	(fft rdat idat -1)
	(if (or (fneq (vector-ref rdat 3) 16.0)
		(fneq (vector-ref rdat 4) 0.0))
	    (snd-display ";vector fft real[3 or 4]: ~A ~A?" (vector-ref rdat 3) (vector-ref rdat 4))))
      
      (let ((v0 (make-vct 10))
	    (v1 (make-vct 10)))
	(vct-fill! v0 1.0)
	(multiply-arrays v0 v1 1)
	(if (not (vequal v0 (vct 0.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0 1.0)))
	    (snd-display ";multiply-arrays[0]: ~A?" v0))
	(multiply-arrays v0 v1 100)
	(if (fneq (vct-peak v0) 0.0)
	    (snd-display ";multiply-arrays[100]: ~A?" v0))
	(vct-fill! v0 1.0)
	(vct-fill! v1 0.5)
	(multiply-arrays v0 v1)
	(if (fneq (vct-ref v0 0) 0.5) (snd-display ";multiple-arrays: ~F?" (vct-ref v0 0)))
	(let ((sum (dot-product v0 v1)))
	  (if (fneq sum 2.5) (snd-display ";dot-product: ~F?" sum)))
	(clear-array v0)
	(if (fneq (vct-ref v0 3) 0.0) (snd-display ";clear-array: ~A?" v0))
	(vct-fill! v0 1.0)
	(vct-fill! v1 0.5)
	(let ((v2 (rectangular->polar v0 v1)))
	  (if (fneq (vct-ref v2 0) 1.118) (snd-display ";rectangular->polar: ~A?" v2)))
	(vct-set! v0 0 1.0)
	(vct-set! v1 0 1.0)
	(rectangular->polar v0 v1)
	(if (or (fneq (vct-ref v0 0) (sqrt 2.0))
		(fneq (vct-ref v1 0) (- (atan 1.0 1.0)))) ;(tan (atan 1.0 1.0)) -> 1.0 
	    (snd-display ";rectangular->polar (~A ~A): ~A ~A?" (sqrt 2.0) (- (atan 1.0 1.0)) (vct-ref v0 0) (vct-ref v1 0)))
	(polar->rectangular v0 v1)
	(if (or (fneq (vct-ref v0 0) 1.0)
		(fneq (vct-ref v1 0) 1.0))
	    (snd-display ";polar->rectangular (1 1): ~A ~A?" (vct-ref v0 0) (vct-ref v1 0)))
	
	(let ((v0 (make-vct 1))
	      (v1 (make-vct 1))
	      (v (make-vct 1))
	      (val .123))
	  (vct-set! v0 0 1.0)
	  (vct-set! v1 0 1.0)
	  (vct-map! v (lambda ()
			(rectangular->polar v0 v1)
			(set! val (vct-ref v0 0))
			(polar->rectangular v0 v1)
			(vct-ref v1 0)))
	  (if (fneq (vct-ref v 0) 1.0) (snd-display ";run r->p not inverted: ~A" v))
	  (if (fneq val (sqrt 2.0)) (snd-display ";r->p: ~A" val)))
	
	(let* ((ind (open-sound "oboe.snd"))
	       (rl (samples->vct 1200 512))
	       (im (make-vct 512)))
	  (fft rl im 512)
	  (let ((rl-copy (vct-copy rl))
		(im-copy (vct-copy im)))
	    (rectangular->polar rl im)
	    (polar->rectangular rl im)
	    (do ((i 0 (1+ i)))
		((= i 512))
	      (if (or (fneq (vct-ref rl i) (vct-ref rl-copy i))
		      (fneq (vct-ref im i) (vct-ref im-copy i)))
		  (snd-display ";polar->rectangular[~D]: ~A ~A ~A ~A" 
			       i 
			       (vct-ref rl i) (vct-ref rl-copy i)
			       (vct-ref im i) (vct-ref im-copy i)))))
	  (close-sound ind)))
      
      (let* ((vals (make-vct 1 1.0))
	     (v1 (edot-product 0.0 vals)))
	(if (fneq v1 1.0) ; exp 0.0 * 1.0
	    (snd-display ";edot 1.0: ~A" v1))
	(vct-set! vals 0 0.0)
	(set! v1 (edot-product 0.0 vals))
	(if (fneq v1 0.0) ; exp 0.0 * 0.0
	    (snd-display ";edot 0.0: ~A" v1))
	(set! vals (make-vector 1 1.0))
	(set! v1 (edot-product 0.0 vals))  
	(if (fneq v1 1.0) ; exp 0.0 * 1.0
	    (snd-display ";edot 1.0: ~A" v1))
	(vector-set! vals 0 0.0+i)
	(set! v1 (edot-product 0.0 vals))
	(if (cneq v1 0.0+i)
	    (snd-display ";edot i: ~A" v1))
	(set! vals (make-vct 4 1.0))
	(set! v1 (edot-product (* 0.25 2 pi) vals))
	(let ((v2 (+ (exp (* 0 2 pi))
		     (exp (* 0.25 2 pi))
		     (exp (* 0.5 2 pi))
		     (exp (* 0.75 2 pi)))))
	  (if (fneq v1 v2) (snd-display ";edot 4: ~A ~A" v1 v2)))
	(set! vals (make-vector 4 0.0))  
	(do ((i 0 (1+ i)))
	    ((= i 4))
	  (vector-set! vals i (+ i 1.0)))
	(set! v1 (edot-product (* 0.25 2 pi 0.0-i) vals))
	(let ((v2 (+ (* 1 (exp (* 0 2 pi 0.0-i)))
		     (* 2 (exp (* 0.25 2 pi 0.0-i)))
		     (* 3 (exp (* 0.5 2 pi 0.0-i)))
		     (* 4 (exp (* 0.75 2 pi 0.0-i))))))
	  (if (cneq v1 v2) (snd-display ";edot 4 -i: ~A ~A" v1 v2)))
	(do ((i 0 (1+ i)))
	    ((= i 4))
	  (vector-set! vals i (+ i 1.0+i)))
	(set! v1 (edot-product (* 0.25 2 pi 0.0-i) vals))
	(let ((v2 (+ (* 1+i (exp (* 0 2 pi 0.0-i)))
		     (* 2+i (exp (* 0.25 2 pi 0.0-i)))
		     (* 3+i (exp (* 0.5 2 pi 0.0-i)))
		     (* 4+i (exp (* 0.75 2 pi 0.0-i))))))
	  (if (cneq v1 v2) (snd-display ";edot 4 -i * i: ~A ~A" v1 v2))))

      (let ((v0 (make-vct 3)))
	(vct-set! v0 0 1.0)
	(vct-set! v0 1 0.5)
	(vct-set! v0 2 0.1)
	(if (or (fneq (polynomial v0 0.0) 1.0)
		(fneq (polynomial v0 1.0) 1.6)
		(fneq (polynomial v0 2.0) 2.4))
	    (snd-display ";polynomial: ~A ~A ~A?"
			 (polynomial v0 0.0)
			 (polynomial v0 1.0)
			 (polynomial v0 2.0))))
      
      (let ((v0 (make-vct 10)))
	(do ((i 0 (1+ i))) ((= i 10))
	  (vct-set! v0 i i))
	(if (fneq (array-interp v0 3.5) 3.5) (snd-display ";array-interp: ~F?" (array-interp v0 3.5)))
	(if (fneq (array-interp v0 13.5) 3.5) (snd-display ";array-interp(13.5): ~F?" (array-interp v0 13.5)))
	(if (fneq (array-interp v0 -6.5) 3.5) (snd-display ";array-interp(-6.5): ~F?" (array-interp v0 -6.5)))
	(if (fneq (array-interp v0 103.6) 3.6) (snd-display ";array-interp(103.5): ~F?" (array-interp v0 103.6)))
	(if (fneq (array-interp v0 -106.6) 3.4) (snd-display ";array-interp(-106.6): ~F?" (array-interp v0 -106.6)))
	(if (fneq (array-interp v0 -0.5) 4.5) (snd-display ";array-interp(-0.5): ~F?" (array-interp v0 -0.5)))
	;; interpolating between 9 and 0 here (confusing...)
	(if (fneq (array-interp v0 -0.9) 8.1) (snd-display ";array-interp(-0.9): ~F?" (array-interp v0 -0.9)))
	(if (fneq (array-interp v0 -0.1) 0.9) (snd-display ";array-interp(-0.1): ~F?" (array-interp v0 -0.1)))
	(if (fneq (array-interp v0 9.1) 8.1) (snd-display ";array-interp(9.1): ~F?" (array-interp v0 9.1)))
	(if (fneq (array-interp v0 9.9) 0.9) (snd-display ";array-interp(9.9): ~F?" (array-interp v0 9.9)))
	(if (fneq (array-interp v0 10.1) 0.1) (snd-display ";array-interp(10.1): ~F?" (array-interp v0 10.1)))
	(let ((var (catch #t (lambda () (array-interp v0 1 -10)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";array-interp bad index: ~A" var))))
      
      (let ((gen (make-delay 3))
	    (gen2 (make-delay 3))
	    (gen1 (make-delay 4 :initial-contents '(1.0 0.5 0.25 0.0)))
	    (gen3 (make-delay 4 :initial-contents (vct 1.0 0.5 0.25 0.0)))
	    (v0 (make-vct 10))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "delay" 
			 "delay: line[3, step]: [0.000 0.000 0.000]"
			 "dly line[3,3 at 0,0 (external)]: [0.000 0.000 0.000], xscl: 0.000000, yscl: 0.000000, type: step")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (delay gen i)))
	(vct-map! v1 (let ((i 0)) (lambda () (let ((val (if (delay? gen2) (delay gen2 i) -1.0))) (set! i (1+ i)) val))))
	(if (not (vequal v1 v0)) (snd-display ";map delay: ~A ~A" v0 v1))
	(if (not (delay? gen)) (snd-display ";~A not delay?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";delay length: ~D?" (mus-length gen)))
	(if (or (fneq (vct-ref v0 1) 0.0) (fneq (vct-ref v0 4) 1.0) (fneq (vct-ref v0 8) 5.0))
	    (snd-display ";delay output: ~A" v0))
	(if (or (fneq (delay gen1) 1.0) 
		(fneq (delay gen1) 0.5)
		(fneq (delay gen1) 0.25)
		(fneq (delay gen1) 0.0)
		(fneq (delay gen1) 0.0))
	    (snd-display ";delay with list initial-contents confused"))
	(if (or (fneq (delay gen3) 1.0) 
		(fneq (delay gen3) 0.5)
		(fneq (delay gen3) 0.25)
		(fneq (delay gen3) 0.0)
		(fneq (delay gen3) 0.0))
	    (snd-display ";delay with vct initial-contents confused"))
	(let ((var (catch #t (lambda () (make-delay :size #f)) (lambda args args))))
	  (if (not (eq? (car var) 'wrong-type-arg))
	      (snd-display ";make-delay bad size #f: ~A" var)))
	(let ((var (catch #t (lambda () (make-delay 3 :initial-element (make-oscil))) (lambda args args))))
	  (if (not (eq? (car var) 'wrong-type-arg))
	      (snd-display ";make-delay bad initial element: ~A" var)))
	(let ((var (catch #t (lambda () (make-delay -3)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";make-delay bad size: ~A" var))))
      
      (test-gen-equal (let ((d1 (make-delay 3))) (delay d1 1.0) d1) 
		      (let ((d2 (make-delay 3))) (delay d2 1.0) d2)
		      (let ((d3 (make-delay 4))) (delay d3 1.0) d3))
      (test-gen-equal (make-delay 3 :initial-element 1.0) (make-delay 3 :initial-element 1.0) (make-delay 3 :initial-element 0.5))
      (test-gen-equal (make-delay 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-delay 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-delay 3 :initial-contents '(1.0 1.0 1.0)))
      
      (let ((gen (make-delay 5)))
	(delay gen 1.0)
	(delay gen 0.0)
	(delay gen 0.5)
	(let ((data (vct-copy (mus-data gen))))
	  (vct-set! (mus-data gen) 0 0.3)
	  (if (fneq (vct-ref (mus-data gen) 0) 0.3)
	      (snd-display ";delay data 0: ~A" (vct-ref (mus-data gen) 0)))
	  (vct-set! data 0 .75)
	  (set! (mus-data gen) data)
	  (if (fneq (vct-ref (mus-data gen) 0) 0.75)
	      (snd-display ";delay set data 0: ~A" (vct-ref (mus-data gen) 0)))
	  (delay gen 0.0)
	  (delay gen 0.0)
	  (let ((val (delay gen 0.0)))
	    (if (fneq val 0.75)
		(snd-display ";set delay data: ~A ~A" val (mus-data gen)))))
	(if (mus-data (make-oscil))
	    (snd-display ";mus-data osc: ~A" (mus-data (make-oscil)))))
      
      (let* ((del (make-delay 5 :max-size 8)))
	(delay del 1.0)
	(do ((i 0 (1+ i))) ((= i 4)) (delay del 0.0))
	(let ((v0 (make-vct 5)))
	  (do ((i 0 (1+ i)))
	      ((= i 5))
	    (vct-set! v0 i (delay del 0.0 0.4)))
	  (if (not (vequal v0 (vct 0.600 0.400 0.000 0.000 0.000)))
	      (snd-display ";zdelay: ~A" v0))
	  (delay del 1.0)
	  (delay del 0.0 0.4)
	  (if (not (string=? (mus-describe del) "delay: line[5,8, linear]: [0.000 0.000 0.000 1.000 0.000]"))
	      (snd-display ";describe zdelay: ~A" (mus-describe del)))
	  (if (not (string=? (mus-inspect del) 
			     "dly line[5,8 at 4,7 (external)]: [0.000 0.000 0.000 1.000 0.000], xscl: 0.000000, yscl: 0.000000, type: linear"))
	      (snd-display ";inspect zdelay: ~A" (mus-inspect del)))))
      (let ((tag (catch #t (lambda () 
			     (let ((gen (make-oscil)))
			       (tap gen)))
			(lambda args (car args)))))
	(if (not (eq? tag 'wrong-type-arg))
	    (snd-display ";tap of oscil: ~A" tag)))
      
      (let ((dly (make-delay 3))
	    (flt (make-one-zero .5 .4))
	    (v (make-vct 20))
	    (inval 1.0))
	(vct-map! v (lambda ()
		      (let ((res (delay dly (+ inval (* (one-zero flt (tap dly)) .6)))))
			(set! inval 0.0)
			res)))
	(if (not (vequal v (vct 0.0 0.0 0.0 1.0 0.0 0.0 0.300 0.240 0.0 0.090 0.144 0.058 0.027 0.065 0.052 0.022 0.026 0.031 0.019 0.013)))
	    (snd-display ";tap with low pass: ~A" v)))
      
      (let ((dly (make-delay 3))
	    (v (make-vct 20))
	    (inval 1.0))
	(vct-map! v (lambda ()
		      (let ((res (delay dly (+ inval (tap dly)))))
			(set! inval 0.0)
			res)))
	(if (not (vequal v (vct 0.0 0.0 0.0 1.0 0.0 0.0 1.0 0.0 0.0 1.0 0.0 0.0 1.0 0.0 0.0 1.0 0.0 0.0 1.0 0.0)))
	    (snd-display ";simple tap: ~A" v)))
      
      (let ((dly (make-delay 6))
	    (v (make-vct 20))
	    (inval 1.0))
	(vct-map! v (lambda ()
		      (let ((res (delay dly (+ inval (tap dly -2.0)))))
			(set! inval 0.0)
			res)))
	(set! (print-length) (max 20 (print-length)))
	(if (not (vequal v (vct 0.0 0.0 0.0 0.0 0.0 0.0 1.0 0.0 0.0 0.0 1.0 0.0 0.0 0.0 1.0 0.0 0.0 0.0 1.0 0.0)))
	    (snd-display ";tap back 2: ~A" v)))
      
      (let ((dly (make-delay 3))
	    (flt (make-one-zero .5 .4))
	    (v (make-vct 20))
	    (inval 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 20))
	  (vct-set! v i (let ((res (delay dly (+ inval (* (one-zero flt (tap dly)) .6)))))
			  (set! inval 0.0)
			  res)))
	(if (not (vequal v (vct 0.0 0.0 0.0 1.0 0.0 0.0 0.300 0.240 0.0 0.090 0.144 0.058 0.027 0.065 0.052 0.022 0.026 0.031 0.019 0.013)))
	    (snd-display ";tap with low pass: ~A" v)))
      
      (let ((d1 (make-delay 4))
	    (d2 (make-delay 4 :max-size 5 :type mus-interp-linear))
	    (d3 (make-delay 4 :max-size 5 :type mus-interp-all-pass))
	    (d4 (make-delay 4 :max-size 5 :type mus-interp-none))
	    (d5 (make-delay 4 :max-size 4 :type mus-interp-lagrange))
	    (d6 (make-delay 4 :max-size 4 :type mus-interp-hermite))
	    (d7 (make-delay 4 :max-size 4 :type mus-interp-linear))
	    (v1 (make-vct 20))
	    (v2 (make-vct 20))
	    (v3 (make-vct 20))
	    (v4 (make-vct 20))
	    (v5 (make-vct 20))
	    (v6 (make-vct 20))
	    (v7 (make-vct 20)))
	(if (not (= (mus-interp-type d1) mus-interp-none)) (snd-display ";d1 interp type: ~A" (mus-interp-type d1)))
	(if (not (= (mus-interp-type d2) mus-interp-linear)) (snd-display ";d2 interp type: ~A" (mus-interp-type d2)))
	(if (not (= (mus-interp-type d3) mus-interp-all-pass)) (snd-display ";d3 interp type: ~A" (mus-interp-type d3)))
	(if (not (= (mus-interp-type d4) mus-interp-none)) (snd-display ";d4 interp type: ~A" (mus-interp-type d4)))
	(if (not (= (mus-interp-type d5) mus-interp-lagrange)) (snd-display ";d5 interp type: ~A" (mus-interp-type d5)))
	(if (not (= (mus-interp-type d6) mus-interp-hermite)) (snd-display ";d6 interp type: ~A" (mus-interp-type d6)))
	(if (not (= (mus-interp-type d7) mus-interp-linear)) (snd-display ";d7 interp type: ~A" (mus-interp-type d7)))
	(vct-set! v1 0 (delay d1 1.0))
	(vct-set! v2 0 (delay d2 1.0))
	(vct-set! v3 0 (delay d3 1.0))
	(vct-set! v4 0 (delay d4 1.0))
	(vct-set! v5 0 (delay d5 1.0))
	(vct-set! v6 0 (delay d6 1.0))
	(vct-set! v7 0 (delay d7 1.0))
	(delay-tick d1 0.0)
	(delay-tick d2 0.0)
	(delay-tick d3 0.0)
	(delay-tick d4 0.0)
	(delay-tick d5 0.0)
	(delay-tick d6 0.0)
	(delay-tick d7 0.0)
	(do ((i 1 (1+ i))
	     (j -0.2 (- j 0.2)))
	    ((= i 20))
	  (vct-set! v1 i (tap d1 j))
	  (vct-set! v2 i (tap d2 j))
	  (vct-set! v3 i (tap d3 j))
	  (vct-set! v4 i (tap d4 j))
	  (vct-set! v5 i (tap d5 j))
	  (vct-set! v6 i (tap d6 j))
	  (vct-set! v7 i (tap d7 j)))
	(set! (print-length) (max 20 (print-length)))
	(if (and (not (vequal v1 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0)))
		 (not (vequal v1 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0))))
	    (snd-display ";delay interp none (1): ~A" v1))
	(if (not (vequal v2 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.200 0.400 0.600 0.800 1.0 0.800 0.600 0.400 0.200 0.0 0.0 0.0 0.0 0.0)))
	    (snd-display ";delay interp linear (2): ~A" v2))
	(if (not (vequal v3 (vct 0.0 0.0 0.0 0.0 0.0 1.0 0.0 0.429 0.143 0.095 0.905 0.397 0.830 0.793 0.912 -0.912 0.608 -0.261 0.065 -0.007)))
	    (snd-display ";delay interp all-pass (3): ~A" v3))
	(if (and (not (vequal v4 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 1.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0)))
		 (not (vequal v4 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 1.0 1.0 1.0 1.0 0.0 0.0 0.0 0.0 0.0))))
	    (snd-display ";delay interp none (4): ~A" v4))
	(if (not (vequal v5 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.120 0.280 0.480 0.720 1.000 0.960 0.840 0.640 0.360 0.000 -0.080 -0.120 -0.120 -0.080)))
	    (snd-display ";delay interp lagrange (5): ~A" v5))
	(if (not (vequal v6 (vct 0.0 -0.016 -0.048 -0.072 -0.064 0.0 0.168 0.424 0.696 0.912 1.0 0.912 0.696 0.424 0.168 0.0 -0.064 -0.072 -0.048 -0.016)))
	    (snd-display ";delay interp hermite (6): ~A" v6))
	(if (not (vequal v7 (vct 0.0 0.0 0.0 0.0 0.0 0.0 0.200 0.400 0.600 0.800 1.0 0.800 0.600 0.400 0.200 0.0 0.0 0.0 0.0 0.0)))
	    (snd-display ";delay interp linear (7): ~A" v7)))

      (let ((gen (make-all-pass .4 .6 3))
	    (v0 (make-vct 10))
	    (gen1 (make-all-pass .4 .6 3))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "all-pass"
			 "all-pass: feedback: 0.400, feedforward: 0.600, line[3, step]:[0.000 0.000 0.000]"
			 "dly line[3,3 at 0,0 (external)]: [0.000 0.000 0.000], xscl: 0.600000, yscl: 0.400000, type: step")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (all-pass gen 1.0)))
	(vct-map! v1 (lambda () (if (all-pass? gen1) (all-pass gen1 1.0) -1.0)))
	(if (not (vequal v1 v0)) (snd-display ";map all-pass: ~A ~A" v0 v1))
	(if (not (all-pass? gen)) (snd-display ";~A not all-pass?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";all-pass length: ~D?" (mus-length gen)))
	(if (not (= (mus-order gen) 3)) (snd-display ";all-pass order: ~D?" (mus-order gen)))
	(if (fneq (mus-feedback gen) .4) (snd-display ";all-pass feedback: ~F?" (mus-feedback gen)))
	(if (fneq (mus-feedforward gen) .6) (snd-display ";all-pass feedforward: ~F?" (mus-feedforward gen)))
	(if (or (fneq (vct-ref v0 1) 0.6) (fneq (vct-ref v0 4) 1.84) (fneq (vct-ref v0 8) 2.336))
	    (snd-display ";all-pass output: ~A" v0))
	(set! (mus-feedback gen) 0.5) 
	(if (fneq (mus-feedback gen) .5) (snd-display ";all-pass set-feedback: ~F?" (mus-feedback gen)))
	(set! (mus-feedforward gen) 0.5) 
	(if (fneq (mus-feedforward gen) .5) (snd-display ";all-pass set-feedforward: ~F?" (mus-feedforward gen))))
      
      (test-gen-equal (let ((d1 (make-all-pass 0.7 0.5 3))) (all-pass d1 1.0) d1)
		      (let ((d2 (make-all-pass 0.7 0.5 3))) (all-pass d2 1.0) d2) 
		      (let ((d3 (make-all-pass 0.7 0.5 4))) (all-pass d3 1.0) d3))
      (test-gen-equal (make-all-pass 0.7 0.5 3 :initial-element 1.0) 
		      (make-all-pass 0.7 0.5 3 :initial-element 1.0) 
		      (make-all-pass 0.7 0.5 3 :initial-element 0.5))
      (test-gen-equal (make-all-pass 0.7 0.5 3 :initial-element 1.0) 
		      (make-all-pass 0.7 0.5 3 :initial-element 1.0) 
		      (make-all-pass 0.5 0.5 3 :initial-element 1.0))
      (test-gen-equal (make-all-pass 0.7 0.5 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-all-pass 0.7 0.5 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-all-pass 0.7 0.5 3 :initial-contents '(1.0 1.0 1.0)))
      (let ((err (catch #t (lambda () (make-all-pass :feedback .2 :feedforward .1 :size -1)) (lambda args args))))
	(if (or (not (eq? (car err) 'out-of-range))
		(not (string=? (cadr err) "make-all-pass"))
		(not (string=? (caddr err) "size ~A < 0?"))
		(not (= (car (cadddr err)) -1)))
	    (snd-display ";make-all-pass bad size error message: ~A" err)))
      
      (let ((gen (make-average 4))
	    (v0 (make-vct 10))
	    (gen1 (make-average 4))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "average"
			 "average: 0.000, line[4]:[0.000 0.000 0.000 0.000]"
			 "dly line[4,4 at 0,0 (external)]: [0.000 0.000 0.000 0.000], xscl: 0.000000, yscl: 0.250000, type: step")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (average gen 1.0)))
	(vct-map! v1 (lambda () (if (average? gen1) (average gen1 1.0) -1.0)))
	(if (not (vequal v1 v0)) (snd-display ";map average: ~A ~A" v0 v1))
	(if (not (average? gen)) (snd-display ";~A not average?" gen))
	(if (not (= (mus-length gen) 4)) (snd-display ";average length: ~D?" (mus-length gen)))
	(if (not (= (mus-order gen) 4)) (snd-display ";average order: ~D?" (mus-order gen)))
	(if (or (fneq (vct-ref v0 1) 0.5) (fneq (vct-ref v0 4) 1.0) (fneq (vct-ref v0 8) 1.0))
	    (snd-display ";average output: ~A" v0)))

      (let* ((gen (make-average 8))
	     (val (average gen)))
	(if (fneq val 0.0) (snd-display ";empty average: ~A" val))
	(set! val (average gen 1.0))
	(if (fneq val 0.125) (snd-display ";average 1: ~A" val))
	(set! val (average gen 1.0))
	(if (fneq val 0.25) (snd-display ";average 2: ~A" val))
	(set! val (average gen 0.5))
	(if (fneq val 0.3125) (snd-display ";average 2: ~A" val))
	(do ((i 0 (1+ i))) ((= i 5)) (set! val (average gen 0.0))) 
	(if (fneq val 0.3125) (snd-display ";average 6: ~A" val))
	(set! val (average gen 0.0))
	(if (fneq val 0.1875) (snd-display ";average 7: ~A" val))
	(set! val (average gen 0.0))
	(if (fneq val 0.0625) (snd-display ";average 8: ~A" val))
	(set! val (average gen 0.0))
	(if (fneq val 0.0) (snd-display ";average 9: ~A" val))
	)
      (let* ((gen (make-average 10 :initial-element .5))
	     (val (average gen 0.5)))
	(if (fneq val 0.5) (snd-display ";average initial-element: ~A" val)))
      (let* ((gen (make-average 3 :initial-contents '(1.0 1.0 1.0)))
	     (val (average gen 1.0)))
	(if (fneq val 1.0) (snd-display ";average initial-contents: ~A" val)))
      
      (test-gen-equal (let ((d1 (make-average 3 :initial-contents '(0.7 0.5 3)))) (average d1 1.0) d1)
		      (let ((d2 (make-average 3 :initial-contents (vct 0.7 0.5 3)))) (average d2 1.0) d2) 
		      (let ((d3 (make-average 4 :initial-contents '(0.7 0.5 0.1 4)))) (average d3 1.0) d3))
      (test-gen-equal (make-average 3 :initial-element 1.0) 
		      (make-average 3 :initial-element 1.0) 
		      (make-average 3 :initial-element 0.5))
      (test-gen-equal (make-average 3 :initial-element 1.0) 
		      (make-average 3 :initial-element 1.0) 
		      (make-average 4 :initial-element 1.0))
      (test-gen-equal (make-average 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-average 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-average 3 :initial-contents '(1.0 1.0 1.0)))
      (let ((err (catch #t (lambda () (make-average :size -1)) (lambda args args))))
	(if (or (not (eq? (car err) 'out-of-range))
		(not (string=? (cadr err) "make-average"))
		(not (string=? (caddr err) "size ~A < 0?"))
		(not (= (car (cadddr err)) -1)))
	    (snd-display ";make-average bad size error message: ~A" err)))

      (let ((gen (make-comb .4 3))
	    (v0 (make-vct 10))
	    (gen1 (make-comb .4 3))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "comb"
			 "comb: scaler: 0.400, line[3, step]: [0.000 0.000 0.000]"
			 "dly line[3,3 at 0,0 (external)]: [0.000 0.000 0.000], xscl: 0.000000, yscl: 0.400000, type: step")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (comb gen 1.0)))
	(vct-map! v1 (lambda () (if (comb? gen1) (comb gen1 1.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map comb: ~A ~A" v0 v1))
	(if (not (comb? gen)) (snd-display ";~A not comb?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";comb length: ~D?" (mus-length gen)))
	(if (not (= (mus-order gen) 3)) (snd-display ";comb order: ~D?" (mus-order gen)))
	(if (fneq (mus-feedback gen) .4) (snd-display ";comb feedback: ~F?" (mus-feedback gen)))
	(if (or (fneq (vct-ref v0 1) 0.0) (fneq (vct-ref v0 4) 1.0) (fneq (vct-ref v0 8) 1.4))
	    (snd-display ";comb output: ~A" v0)))
      
      (test-gen-equal (let ((d1 (make-comb 0.7 3))) (comb d1 1.0) d1) 
		      (let ((d2 (make-comb 0.7 3))) (comb d2 1.0) d2) 
		      (let ((d3 (make-comb 0.7 4))) (comb d3 1.0) d3))
      (test-gen-equal (make-comb 0.7 3 :initial-element 1.0) 
		      (make-comb 0.7 3 :initial-element 1.0) 
		      (make-comb 0.7 3 :initial-element 0.5))
      (test-gen-equal (make-comb 0.7 3 :initial-element 1.0) 
		      (make-comb 0.7 3 :initial-element 1.0) 
		      (make-comb 0.5 3 :initial-element 1.0))
      (test-gen-equal (make-comb 0.7 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-comb 0.7 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-comb 0.7 3 :initial-contents '(1.0 1.0 1.0)))
      
      (let* ((del (make-comb 0.0 5 :max-size 8)))
	(comb del 1.0)
	(do ((i 0 (1+ i))) ((= i 4)) (comb del 0.0))
	(let ((v0 (make-vct 5)))
	  (do ((i 0 (1+ i)))
	      ((= i 5))
	    (vct-set! v0 i (comb del 0.0 0.4)))
	  (if (not (vequal v0 (vct 0.600 0.400 0.000 0.000 0.000))) ; this is assuming interpolation in the delay...
	      (snd-display ";zcomb: ~A" v0))
	  (comb del 1.0)
	  (comb del 0.0 0.4)
	  (if (not (string=? (mus-describe del) "comb: scaler: 0.000, line[5,8, linear]: [0.000 0.000 0.000 1.000 0.000]"))
	      (snd-display ";describe zcomb: ~A" (mus-describe del)))
	  (if (not (string=? (mus-inspect del) 
			     "dly line[5,8 at 4,7 (external)]: [0.000 0.000 0.000 1.000 0.000], xscl: 0.000000, yscl: 0.000000, type: linear"))
	      (snd-display ";inspect zcomb: ~A" (mus-inspect del))))
	(set! (mus-feedback del) 1.0)
	(if (fneq (mus-feedback del) 1.0)
	    (snd-display ";comb feedback set: ~A" (mus-feedback del))))
      
      (let ((gen (make-notch .4 3))
	    (v0 (make-vct 10))
	    (gen1 (make-notch .4 3))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "notch"
			 "notch: scaler: 0.400, line[3, step]: [0.000 0.000 0.000]"
			 "dly line[3,3 at 0,0 (external)]: [0.000 0.000 0.000], xscl: 0.400000, yscl: 0.000000, type: step")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (notch gen 1.0)))
	(vct-map! v1 (lambda () (if (notch? gen1) (notch gen1 1.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map notch: ~A ~A" v0 v1))
	(if (not (notch? gen)) (snd-display ";~A not notch?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";notch length: ~D?" (mus-length gen)))
	(if (not (= (mus-order gen) 3)) (snd-display ";notch order: ~D?" (mus-order gen)))
	(if (fneq (mus-feedforward gen) .4) (snd-display ";notch feedforward: ~F?" (mus-feedforward gen)))
	(if (or (fneq (vct-ref v0 1) 0.4) (fneq (vct-ref v0 4) 1.4) (fneq (vct-ref v0 8) 1.4))
	    (snd-display ";notch output: ~A" v0))
	(set! (mus-feedforward gen) 1.0)
	(if (fneq (mus-feedforward gen) 1.0)
	    (snd-display ";notch feedforward set: ~A" (mus-feedforward gen))))
      
      (test-gen-equal (let ((d1 (make-notch 0.7 3))) (notch d1 1.0) d1)
		      (let ((d2 (make-notch 0.7 3))) (notch d2 1.0) d2)
		      (let ((d3 (make-notch 0.7 4))) (notch d3 1.0) d3))
      (test-gen-equal (make-notch 0.7 3 :initial-element 1.0) 
		      (make-notch 0.7 3 :initial-element 1.0) 
		      (make-notch 0.7 3 :initial-element 0.5))
      (test-gen-equal (make-notch 0.7 3 :initial-element 1.0) 
		      (make-notch 0.7 3 :initial-element 1.0) 
		      (make-notch 0.5 3 :initial-element 1.0))
      (test-gen-equal (make-notch 0.7 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-notch 0.7 3 :initial-contents '(1.0 0.0 0.0)) 
		      (make-notch 0.7 3 :initial-contents '(1.0 1.0 1.0)))
      
      ;; make sure all-pass is the same as comb/notch given the appropriate feedback/forward settings
      
      (let ((gen (make-comb 0.5 5))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (comb gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.500)))
	    (snd-display ";comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (all-pass gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.500)))
	    (snd-display ";all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (notch gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (all-pass gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";all-pass (5 .5 0): ~A" v0)))
      
      ;; make sure zall-pass is the same as zcomb/znotch given the appropriate feedback/forward and "pm" settings
      
      (let ((gen (make-comb 0.5 5 :max-size 20))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (comb gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.500)))
	    (snd-display ";1comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5 :max-size 20))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (all-pass gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.500)))
	    (snd-display ";1all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5 :max-size 20))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (notch gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";1notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5 :max-size 20))
	    (v0 (make-vct 11))
	    (in1 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 11))
	  (vct-set! v0 i (all-pass gen in1))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";1all-pass (5 .5 0): ~A" v0)))
      
      ;; now actually use the size difference
      
      (let ((gen (make-comb 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (comb gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.800 0.400 0.000 0.000 0.000 0.000 0.000 0.160 0.360 0.200 0.040 0.000 0.000 0.000)))
	    (snd-display ";2comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.800 0.400 0.000 0.000 0.000 0.000 0.000 0.160 0.360 0.200 0.040 0.000 0.000 0.000)))
	    (snd-display ";2all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (notch gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 0.000 0.800 0.400 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";2notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 0.000 0.800 0.400 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";2all-pass (5 .5 0): ~A" v0)))
      
      (let ((gen (make-comb 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (comb gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.800 0.000 0.000 0.160 0.160 0.000 0.080 0.064 0.016 0.035 0.013 0.018 0.007 0.007 0.003 0.002)))
	    (snd-display ";3comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.800 0.000 0.000 0.160 0.160 0.000 0.080 0.064 0.016 0.035 0.013 0.018 0.007 0.007 0.003 0.002)))
	    (snd-display ";3all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (notch gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.800 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";3notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .2)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.800 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";3all-pass (5 .5 0): ~A" v0)))
      
      (let ((gen (make-comb 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .01)))
	    ((= i 20))
	  (vct-set! v0 i (comb gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 0.950 0.060 0.000 0.000 0.000 0.428 0.079 0.004 0.000 0.000 0.182 0.067 0.008 0.000 0.000)))
	    (snd-display ";4comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .01)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 0.950 0.060 0.000 0.000 0.000 0.428 0.079 0.004 0.000 0.000 0.182 0.067 0.008 0.000 0.000)))
	    (snd-display ";4all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .01)))
	    ((= i 20))
	  (vct-set! v0 i (notch gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 0.950 0.060 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";4notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5 :max-size 20))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .01)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 0.950 0.060 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";4all-pass (5 .5 0): ~A" v0)))
      
      ;; now run off either end of the delay line "by accident"
      
      (let ((gen (make-comb 0.5 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (comb gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.500 1.000 0.250 0.125 0.094 0.062 0.055 0.047 0.039 0.031 0.029)))
	    (snd-display ";5comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.500 1.000 0.250 0.125 0.094 0.062 0.055 0.047 0.039 0.031 0.029)))
	    (snd-display ";5all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (notch gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.500 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";5notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (+ angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.500 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";5all-pass (5 .5 0): ~A" v0)))
      
      
      (let ((gen (make-comb 0.5 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (comb gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.500 0.000 0.125 0.000 0.031 0.016 0.004 1.000 0.000 0.250 0.031 0.000 0.012 0.002 0.250 0.125 0.008)))
	    (snd-display ";6comb (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.5 0.0 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.000 0.000 0.000 0.500 0.000 0.125 0.000 0.031 0.016 0.004 1.000 0.000 0.250 0.031 0.000 0.012 0.002 0.250 0.125 0.008)))
	    (snd-display ";6all-pass (5 0 .5): ~A" v0)))
      
      (let ((gen (make-notch 0.5 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (notch gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.500 0.000 0.000 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";6notch (5 .5): ~A" v0)))
      
      (let ((gen (make-all-pass 0.0 0.5 5 :max-size 10))
	    (v0 (make-vct 20))
	    (in1 1.0))
	(do ((i 0 (1+ i))
	     (angle 0.0 (- angle .5)))
	    ((= i 20))
	  (vct-set! v0 i (all-pass gen in1 angle))
	  (set! in1 0.0))
	(if (not (vequal v0 (vct 0.500 0.000 0.000 0.500 0.000 0.000 0.000 0.000 0.000 0.000 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	    (snd-display ";6all-pass (5 .5 0): ~A" v0)))
      
      
      
      (let ((gen (make-one-pole .4 .7))
	    (v0 (make-vct 10))
	    (gen1 (make-one-pole .4 .7))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "one-pole"
			 "one-pole: a0: 0.400, b1: 0.700, y1: 0.000"
			 "smpflt a0: 0.400000, a1: 0.000000, a2: 0.000000, b1: 0.700000, b2: 0.000000, x1: 0.000000, x2: 0.000000, y1: 0.000000, y2: 0.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (one-pole gen 1.0)))
	(vct-map! v1 (lambda () (if (one-pole? gen) (one-pole gen1 1.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map one-pole: ~A ~A" v0 v1))
	(if (not (one-pole? gen)) (snd-display ";~A not one-pole?" gen))
	(if (not (= (mus-order gen) 1)) (snd-display ";one-pole order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) .4) (snd-display ";one-pole a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-b1 gen) .7) (snd-display ";one-pole b1: ~F?" (mus-b1 gen)))
	(if (or (fneq (vct-ref v0 1) 0.120) (fneq (vct-ref v0 4) 0.275) (fneq (vct-ref v0 8) 0.245))
	    (snd-display ";one-pole output: ~A" v0))
	(if (fneq (mus-ycoeff gen 1) .7) (snd-display ";1p ycoeff 1 .7: ~A" gen))
	(set! (mus-ycoeff gen 1) .1)
	(if (fneq (mus-ycoeff gen 1) .1) (snd-display ";1p set ycoeff 1 .1: ~A" gen))
	(if (fneq (mus-xcoeff gen 0) .4) (snd-display ";1p xcoeff 0 .4: ~A" gen))
	(set! (mus-xcoeff gen 0) .3)
	(if (fneq (mus-xcoeff gen 0) .3) (snd-display ";1p set xcoeff 0 .3: ~A" gen)))

      
      (let ((gen (make-one-zero .4 .7))
	    (v0 (make-vct 10))
	    (gen1 (make-one-zero .4 .7))
	    (v1 (make-vct 10)))
	(print-and-check gen
			 "one-zero"
			 "one-zero: a0: 0.400, a1: 0.700, x1: 0.000"
			 "smpflt a0: 0.400000, a1: 0.700000, a2: 0.000000, b1: 0.000000, b2: 0.000000, x1: 0.000000, x2: 0.000000, y1: 0.000000, y2: 0.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (one-zero gen 1.0)))
	(vct-map! v1 (lambda () (if (one-zero? gen) (one-zero gen1 1.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map one-zero: ~A ~A" v0 v1))
	(if (not (one-zero? gen)) (snd-display ";~A not one-zero?" gen))
	(if (not (= (mus-order gen) 1)) (snd-display ";one-zero order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) .4) (snd-display ";one-zero a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-a1 gen) .7) (snd-display ";one-zero a1: ~F?" (mus-a1 gen)))
	(if (fneq (vct-ref v0 1) 1.1) (snd-display ";one-zero output: ~A" v0))
	(if (fneq (mus-xcoeff gen 0) .4) (snd-display ";1z xcoeff 0 .4: ~A" gen))
	(set! (mus-xcoeff gen 0) .1)
	(if (fneq (mus-xcoeff gen 0) .1) (snd-display ";1z set xcoeff 0 .1: ~A" gen)))
      
      (let ((gen (make-two-zero .4 .7 .3))
	    (v0 (make-vct 10))
	    (gen1 (make-two-zero .4 .7 .3))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "two-zero"
			 "two-zero: a0: 0.400, a1: 0.700, a2: 0.300, x1: 0.000, x2: 0.000"
			 "smpflt a0: 0.400000, a1: 0.700000, a2: 0.300000, b1: 0.000000, b2: 0.000000, x1: 0.000000, x2: 0.000000, y1: 0.000000, y2: 0.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (two-zero gen 1.0)))
	(vct-map! v1 (lambda () (if (two-zero? gen1) (two-zero gen1 1.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map two-zero: ~A ~A" v0 v1))
	(if (not (two-zero? gen)) (snd-display ";~A not two-zero?" gen))
	(if (not (= (mus-order gen) 2)) (snd-display ";two-zero order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) .4) (snd-display ";two-zero a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-a1 gen) .7) (snd-display ";two-zero a1: ~F?" (mus-a1 gen)))
	(if (fneq (mus-a2 gen) .3) (snd-display ";two-zero a2: ~F?" (mus-a2 gen)))
	(if (or (fneq (vct-ref v0 1) 1.1) (fneq (vct-ref v0 8) 1.4)) (snd-display ";two-zero output: ~A" v0))
	(if (fneq (mus-xcoeff gen 0) .4) (snd-display ";2z xcoeff 0 .4: ~A" gen))
	(set! (mus-xcoeff gen 0) .1)
	(if (fneq (mus-xcoeff gen 0) .1) (snd-display ";2z set xcoeff 0 .1: ~A" gen)))
      
      (let ((gen (make-two-zero .4 .7 .3)))
	(let ((val (gen 1.0)))
	  (if (fneq val .4) (snd-display ";2zero->0.4: ~A" val))
	  (set! val (gen 0.5))
	  (if (fneq val .9) (snd-display ";2zero->0.9: ~A" val))
	  (set! val (gen 1.0))
	  (if (fneq val 1.05) (snd-display ";2zero->1.05: ~A" val))))
      
      (let ((gen (make-two-pole .4 .7 .3))
	    (v0 (make-vct 10))
	    (gen1 (make-two-pole .4 .7 .3))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "two-pole"
			 "two-pole: a0: 0.400, b1: 0.700, b2: 0.300, y1: 0.000, y2: 0.000"
			 "smpflt a0: 0.400000, a1: 0.000000, a2: 0.000000, b1: 0.700000, b2: 0.300000, x1: 0.000000, x2: 0.000000, y1: 0.000000, y2: 0.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (two-pole gen 1.0)))
	(vct-map! v1 (lambda () (if (two-pole? gen1) (two-pole gen1 1.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map two-pole: ~A ~A" v0 v1))
	(if (not (two-pole? gen)) (snd-display ";~A not two-pole?" gen))
	(if (not (= (mus-order gen) 2)) (snd-display ";two-pole order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) .4) (snd-display ";two-pole a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-b1 gen) .7) (snd-display ";two-pole b1: ~F?" (mus-b1 gen)))
	(if (fneq (mus-b2 gen) .3) (snd-display ";two-pole b2: ~F?" (mus-b2 gen)))
	(if (or (fneq (vct-ref v0 1) 0.12) (fneq (vct-ref v0 8) 0.201)) (snd-display ";two-pole output: ~A" v0))
	(if (fneq (mus-ycoeff gen 1) .7) (snd-display ";2p ycoeff 1 .7: ~A" gen))
	(set! (mus-ycoeff gen 1) .1)
	(if (fneq (mus-ycoeff gen 1) .1) (snd-display ";2p set ycoeff 1 .1: ~A" gen))
	(if (fneq (mus-xcoeff gen 0) .4) (snd-display ";2p xcoeff 0 .4: ~A" gen))
	(set! (mus-xcoeff gen 0) .3)
	(if (fneq (mus-xcoeff gen 0) .3) (snd-display ";2p set xcoeff 0 .3: ~A" gen)))
      
      (let ((gen (make-two-pole .4 .7 .3)))
	(let ((val (gen 1.0)))
	  (if (fneq val .4) (snd-display ";a0->out 2pole: ~A" val))
	  (set! val (gen 0.5))
	  (if (fneq val -.08) (snd-display ";a0->out 2pole (-0.08): ~A" val))
	  (set! val (gen 1.0))
	  (if (fneq val 0.336) (snd-display ";a0->out 2pole (0.336): ~A" val))))
      
      (let ((var (catch #t (lambda () (make-two-pole :b1 3.0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-two-pole bad b1: ~A" var)))
      (let ((var (catch #t (lambda () (make-two-pole :b2 2.0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-two-pole bad b2: ~A" var)))
      (let ((var (catch #t (lambda () (make-two-pole :b2 2.0 :b1)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-two-pole bad keys: ~A" var)))
      (let ((var (catch #t (lambda () (make-two-pole :b2 2.0 3.0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-two-pole bad args: ~A" var)))
      
      (let ((gen (make-oscil 440.0))
	    (gen1 (make-oscil 440.0))
	    (gen2 (make-oscil 440.0))
	    (v0 (make-vct 10))
	    (v1 (make-vct 10))
	    (v2 (make-vct 10)))
	(print-and-check gen 
			 "oscil"
			 "oscil freq: 440.000Hz, phase: 0.000"
			 "osc freq: 0.125379, phase: 0.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (oscil gen 0.0))
	  (vct-set! v1 i (mus-apply gen1 0.0 0.0)))
	(vct-map! v2 (lambda () (if (oscil? gen2) (oscil gen2 0.0) -1.0)))
	(if (not (vequal v0 v2)) (snd-display ";map oscil: ~A ~A" v0 v2))
	(if (not (oscil? gen)) (snd-display ";~A not oscil?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";oscil phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";oscil frequency: ~F?" (mus-frequency gen)))
	(if (not (= (mus-cosines gen) 1)) (snd-display ";oscil cosines: ~D?" (mus-cosines gen)))
	(if (or (fneq (vct-ref v0 1) 0.125) (fneq (vct-ref v0 8) 0.843)) (snd-display ";oscil output: ~A" v0))
	(set! (mus-phase gen) 0.0)
	(if (fneq (mus-phase gen) 0.0) (snd-display ";oscil set-phase: ~F?" (mus-phase gen)))
	(set! (mus-frequency gen) 100.0)
	(if (fneq (mus-frequency gen) 100.0) (snd-display ";oscil set-frequency: ~F?" (mus-frequency gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (if (fneq (vct-ref v0 i) (vct-ref v1 i))
	      (snd-display ";mus-apply oscil at ~D: ~A ~A?" i (vct-ref v0 i) (vct-ref v1 i))))
	(if (fneq (mus-apply) 0.0)
	    (snd-display ";(mus-apply): ~A" (mus-apply))))
      
      (let ((o1 (make-oscil (+ (mus-srate) 100)))
	    (o2 (make-oscil 100)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (let ((val1 (o1))
		(val2 (o2)))
	    (if (fneq val1 val2) (snd-display ";~D: o1: ~A, o2: ~A" i val1 val2)))))
      
      (let ((gen1 (make-oscil 100.0))
	    (gen2 (make-oscil -100.0))
	    (mx 0.0))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (set! mx (max mx (abs (+ (gen1) (gen2))))))
	(if (fneq mx 0.0)
	    (snd-display ";oscil +-: ~A" mx)))

      (let ((gen1 (make-oscil 100.0 (* pi 0.5)))
	    (gen2 (make-oscil -100.0 (* pi 0.5)))
	    (mx 0.0))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (set! mx (max mx (abs (- (gen1) (gen2))))))
	(if (fneq mx 0.0)
	    (snd-display ";cosil +-: ~A" mx)))

      (fm-test (make-oscil))
      (fm-test (make-sine-summation))
      (fm-test (make-square-wave))
      (fm-test (make-triangle-wave))
      (fm-test (make-sum-of-cosines))
      (fm-test (make-sum-of-sines))
      (fm-test (make-sawtooth-wave))
      (fm-test (make-rand))
      (fm-test (make-rand-interp))
      (fm-test (make-pulse-train))
      
      (let ((gen (make-oscil 440.0))
	    (gen1 (make-oscil 440.0)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (let ((oval (oscil gen .1))
		(mval (mus-run gen1 .1)))
	    (if (fneq oval mval)
		(snd-display ";mus-run ~A but oscil ~A?" mval oval)))))
      
      (let ((gen (make-oscil 440.0))
	    (gen1 (make-oscil 440.0))
	    (gen2 (make-oscil 440.0))
	    (gen3 (make-oscil 440.0))
	    (fm-index (hz->radians 440.0))
	    (v0 (make-vct 10))
	    (v1 (make-vct 10)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (oscil gen (* fm-index (oscil gen1 0.0))))
	  (vct-set! v1 i (mus-apply gen2 (* fm-index (mus-apply gen3 0.0 0.0)) 0.0)))
	(if (or (fneq (vct-ref v0 1) 0.125) (fneq (vct-ref v0 6) 0.830) (fneq (vct-ref v0 8) 0.987))
	    (snd-display ";oscil fm output: ~A" v0))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (if (fneq (vct-ref v0 i) (vct-ref v1 i))
	      (snd-display ";mus-apply fm oscil at ~D: ~A ~A?" i (vct-ref v0 i) (vct-ref v1 i)))))
      
      (test-gen-equal (make-oscil 440.0) (make-oscil 440.0) (make-oscil 100.0))
      (test-gen-equal (make-oscil 440.0) (make-oscil 440.0) (make-oscil 440.0 1.0))
      
      (let ((gen (make-oscil 440.0))
	    (gen1 (make-oscil 440.0))
	    (pm-index 2.0)
	    (v0 (make-vct 10)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (gen 0.0 (* pm-index (gen1 0.0)))))
	(if (or (fneq (vct-ref v0 1) 0.367) (fneq (vct-ref v0 6) 0.854) (fneq (vct-ref v0 8) 0.437))
	    (snd-display ";oscil pm output: ~A" v0)))
      
      (let ((gen (make-oscil 440.0)))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((val1 (sin (mus-phase gen)))
		 (val2 (gen 0.0)))
	    (if (fneq val1 val2)
		(snd-display ";oscil: ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-oscil 440.0 :initial-phase (* pi 0.5))))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 440.0) 22050.0))))
	    ((= i 900))
	  (let* ((val1 (cos a))
		 (val2 (gen 0.0)))
	    (if (fneq val1 val2)
		(snd-display ";oscil (cos): ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-oscil 0.0))
	    (gen1 (make-oscil 40.0)))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 40.0) 22050.0))))
	    ((= i 1100))
	  (let* ((val1 (sin (sin a)))
		 (val2 (oscil gen 0.0 (oscil gen1 0.0))))
	    (if (fneq val1 val2)
		(snd-display ";oscil pm: ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-oscil 0.0))
	    (gen1 (make-oscil 40.0))
	    (a1 0.0))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 40.0) 22050.0))))
	    ((= i 100))
	  (let* ((fm (sin a))
		 (val1 (sin a1))
		 (val2 (oscil gen (oscil gen1 0.0))))
	    (set! a1 (+ a1 fm))
	    (if (fneq val1 val2)
		(snd-display ";oscil fm: ~A: ~A ~A" i val1 val2)))))

      (let ((var (catch #t (lambda () (mus-location (make-oscil))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";mus-location bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (set! (mus-location (make-oscil)) 0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";set mus-location bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (mus-scaler (make-oscil))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";mus-scaler bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (set! (mus-scaler (make-oscil)) 0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";set mus-scaler bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (mus-length (make-oscil))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";mus-length bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (set! (mus-length (make-oscil)) 0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";set mus-length bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (mus-frequency (make-one-pole))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";mus-frequency bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (set! (mus-frequency (make-one-pole)) 0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";set mus-frequency bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (mus-scaler (make-delay 3))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";mus-scaler bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (set! (mus-scaler (make-delay 3)) 0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";set mus-scaler bad gen: ~A" var)))
      (let ((var (catch #t (lambda () (make-delay (* 1024 1024 40))) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-delay huge line: ~A" var)))
      (let ((var (catch #t (lambda () (make-delay 32 :max-size (* 1024 1024 40))) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-delay huge line: ~A" var)))
      
      (let ((amps (make-vector 3))
	    (oscils (make-vector 3))
	    (fms (make-vector 3))
	    (results (make-vector 10)))
	(do ((i 0 (1+ i))) ((= i 3))
	  (vector-set! amps i (* (+ i 1) .1))
	  (vector-set! oscils i (make-oscil :frequency (* (+ i 1) 220.0)))
	  (vector-set! fms i (* i .05)))
	(do ((i 0 (1+ i))) ((= i 10))
	  (vector-set! results i (oscil-bank amps oscils fms)))
	(if (or (fneq (vector-ref results 1) 0.12639)
		(fneq (vector-ref results 5) 0.48203)
		(fneq (vector-ref results 9) 0.41001))
	    (snd-display ";oscil-bank: ~A?" results)))
      
      (let ((amps (make-vector 3))
	    (oscils (make-vector 3))
	    (fms (make-vector 3))
	    (results (make-vector 10)))
	(do ((i 0 (1+ i))) ((= i 3))
	  (vector-set! amps i (* (+ i 1) .1))
	  (vector-set! oscils i (make-oscil :frequency (* (+ i 1) 220.0)))
	  (vector-set! fms i (* i .05)))
	(do ((i 0 (1+ i))) ((= i 10))
	  (vector-set! results i (mus-bank oscils amps fms)))
	(if (or (fneq (vector-ref results 1) 0.12639)
		(fneq (vector-ref results 5) 0.48203)
		(fneq (vector-ref results 9) 0.41001))
	    (snd-display ";mus-bank: ~A?" results))
	(do ((i 0 (1+ i))) ((= i 10))
	  (vector-set! results i (mus-bank oscils amps fms (lambda (i) 0.0)))))
      (let ((amps (make-vct 3))
	    (oscils (make-vector 3))
	    (fms (make-vct 3))
	    (results (make-vct 10)))
	(do ((i 0 (1+ i))) ((= i 3))
	  (vct-set! amps i (* (+ i 1) .1))
	  (vector-set! oscils i (make-oscil :frequency (* (+ i 1) 220.0)))
	  (vct-set! fms i (* i .05)))
	(vct-map! results (lambda () (mus-bank oscils amps fms)))
	(if (or (fneq (vct-ref results 1) 0.12639)
		(fneq (vct-ref results 5) 0.48203)
		(fneq (vct-ref results 9) 0.41001))
	    (snd-display ";mus-bank (2 run): ~A?" results)))
      (let ((amps (make-vct 3))
	    (oscils (make-vector 3))
	    (fms (make-vct 3))
	    (fm1s (make-vct 3))
	    (results (make-vct 10)))
	(do ((i 0 (1+ i))) ((= i 3))
	  (vct-set! amps i (* (+ i 1) .1))
	  (vector-set! oscils i (make-oscil :frequency (* (+ i 1) 220.0)))
	  (vct-set! fms i (* i .05)))
	(vct-map! results (lambda () (mus-bank oscils amps fms fm1s)))
	(if (or (fneq (vct-ref results 1) 0.12639)
		(fneq (vct-ref results 5) 0.48203)
		(fneq (vct-ref results 9) 0.41001))
	    (snd-display ";mus-bank (3 run): ~A?" results)))
      (let ((amps (make-vct 3))
	    (oscs (make-vector 3))
	    (amps1 (make-vct 3))
	    (oscs1 (make-vector 3))
	    (v1 (make-vct 10)))
	(do ((i 0 (1+ i)))
	    ((= i 3))
	  (vct-set! amps i (* (1+ i) .1))
	  (vct-set! amps1 i (* (1+ i) .1))
	  (vector-set! oscs i (make-oscil :frequency (* (1+ i) 100.0)))
	  (vector-set! oscs1 i (make-oscil :frequency (* (1+ i) 100.0))))
	(vct-map! v1 (lambda () (mus-bank oscs amps)))
	(call-with-current-continuation
	 (lambda (break)
	   (do ((i 0 (1+ i)))
	       ((= i 10))
	     (let ((val (+ (* (vct-ref amps1 0) (oscil (vector-ref oscs1 0)))
			   (* (vct-ref amps1 1) (oscil (vector-ref oscs1 1)))
			   (* (vct-ref amps1 2) (oscil (vector-ref oscs1 2))))))
	       (if (fneq (vct-ref v1 i) val)
		   (begin
		     (snd-display ";opt'd mus-bank ~A: ~A ~A (~A ~A)" i val (vct-ref v1 i) oscs amps)
		     (break))))))))
      
      (let ((gen (make-buffer 3)))
	(if (not (buffer-empty? gen)) (snd-display ";new buf not buffer-empty: ~A?" gen))
	(sample->buffer gen 1.0)
	(sample->buffer gen 0.5)
	(sample->buffer gen 0.25)
	
	(print-and-check gen 
			 "buffer"
			 "buffer: length: 3, loc: 0, fill: 3.000"
			 "rblk buf[3 (external)]: [1.000 0.500 0.250], loc: 0, fill_time: 3.000000, empty: true")
	(if (not (buffer-full? gen)) (snd-display ";buffer-full: ~A?" gen))
	(if (not (buffer? gen)) (snd-display ";~A not buffer?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";buffer length: ~D?" (mus-length gen)))
	(if (not (= (mus-increment gen) 3)) (snd-display ";buffer fill-time: ~D?" (mus-increment gen)))
	(if (or (fneq (buffer->sample gen) 1.0) (fneq (buffer->sample gen) 0.5) (fneq (buffer->sample gen) 0.25))
	    (snd-display (format "buffer output?")))
	(if (not (buffer-empty? gen)) (snd-display ";emptied buf not buffer-empty: ~A?" gen))
	(let ((fr0 (make-frame 2 .1 .2))
	      (fr1 (make-frame 2 0.0 0.0)))
	  (frame->buffer gen fr0)
	  (set! fr1 (buffer->frame gen fr1))
	  (if (not (equal? fr0 fr1)) (snd-display ";frame->buffer: ~A ~A?" fr0 fr1)))
	(set! (mus-data gen) (make-vct 2))
	(set! (mus-length gen) 2)
	(if (not (= (mus-length gen) 2)) (snd-display ";set buffer length: ~D?" (mus-length gen))))
	
      (let ((gen (make-buffer 6))
	    (fr1 (make-frame 2 .1 .2))
	    (fr2 (make-frame 2 .3 .4))
	    (fr3 (make-frame 2 .5 .6))
	    (fr4 (make-frame 2 .7 .8)))
	(frame->buffer gen fr1)
	
	(print-and-check gen 
			 "buffer"
			 "buffer: length: 6, loc: 0, fill: 2.000"
			 "rblk buf[6 (external)]: [0.100 0.200 0.000 0.000 0.000 0.000], loc: 0, fill_time: 2.000000, empty: true")
	(frame->buffer gen fr2)
	(frame->buffer gen fr3)
	
	(print-and-check gen 
			 "buffer"
			 "buffer: length: 6, loc: 0, fill: 6.000"
			 "rblk buf[6 (external)]: [0.100 0.200 0.300 0.400 0.500 0.600], loc: 0, fill_time: 6.000000, empty: true")
	(buffer->frame gen fr2)
	(if (not (equal? fr2 fr1)) (snd-display ";buffer->frame: ~A ~A?" fr1 fr2))
	
	(print-and-check gen 
			 "buffer"
			 "buffer: length: 6, loc: 2, fill: 6.000"
			 "rblk buf[6 (external)]: [0.100 0.200 0.300 0.400 0.500 0.600], loc: 2, fill_time: 6.000000, empty: true")
	(let ((f (buffer->frame gen)))
	  (if (not (= (mus-channels f) 1)) (snd-display ";buffer->frame default: ~A?" f))
	  (if (fneq (frame-ref f 0) .3) (snd-display ";buffer->frame: ~A?" f))
	  (buffer->frame gen fr1)
	  (if (not (equal? fr1 (make-frame 2 .4 .5))) (snd-display ";buffer->frame offset: ~A?" fr1))
	  (frame->buffer gen fr4)
	  
	  (print-and-check gen 
			   "buffer"
			   "buffer: length: 6, loc: 0, fill: 3.000"
			   "rblk buf[6 (external)]: [0.600 0.700 0.800 0.000 0.000 0.000], loc: 0, fill_time: 3.000000, empty: true")))
      
      (test-gen-equal (make-buffer 3) (make-buffer 3) (make-buffer 4))
      (let ((gen (make-buffer 3))
	    (gen1 (make-buffer 3))
	    (gen2 (make-buffer 3)))
	(sample->buffer gen 1.0)
	(sample->buffer gen 0.5)
	(sample->buffer gen1 1.0)
	(sample->buffer gen1 0.5)
	(sample->buffer gen2 1.0)
	(sample->buffer gen2 0.5)
	(sample->buffer gen2 0.25)
	(test-gen-equal gen gen1 gen2))
      
      (let ((gen (make-sum-of-cosines 10 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-sum-of-cosines 10 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "sum-of-cosines"
			 "sum-of-cosines freq: 440.000Hz, phase: 0.000, cosines: 10"
			 "cosp freq: 0.125379, phase: 0.000000, cosines: 10, scaler: 0.100000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (sum-of-cosines gen 0.0)))
	(vct-map! v1 (lambda () (if (sum-of-cosines? gen1) (sum-of-cosines gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map sum-of-cosines: ~A ~A" v0 v1))
	(if (not (sum-of-cosines? gen)) (snd-display ";~A not sum-of-cosines?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";sum-of-cosines phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";sum-of-cosines frequency: ~F?" (mus-frequency gen)))
	(if (fneq (mus-scaler gen) .1) (snd-display ";sum-of-cosines scaler: ~F?" (mus-scaler gen)))
	(if (not (= (mus-cosines gen) 10)) (snd-display ";sum-of-cosines cosines: ~D?" (mus-cosines gen)))
	(if (not (= (mus-length gen) 10)) (snd-display ";sum-of-cosines length: ~D?" (mus-length gen)))
	(if (or (fneq (vct-ref v0 1) 0.722) (fneq (vct-ref v0 8) -0.143)) (snd-display ";sum-of-cosines output: ~A" v0))
	(set! (mus-scaler gen) .5) (if (fneq (mus-scaler gen) 0.5) (snd-display ";sum-of-cosines set-scaler: ~F?" (mus-scaler gen)))
	(set! (mus-cosines gen) 5) (if (not (= (mus-cosines gen) 5)) (snd-display ";set sum-of-cosines cosines: ~D?" (mus-cosines gen)))
	(if (fneq (mus-scaler gen) .2) (snd-display ";set cosines->scaler: ~A" (mus-scaler gen))))
      
      (test-gen-equal (make-sum-of-cosines 3 440.0) (make-sum-of-cosines 3 440.0) (make-sum-of-cosines 5 440.0))
      (test-gen-equal (make-sum-of-cosines 3 440.0) (make-sum-of-cosines 3 440.0) (make-sum-of-cosines 3 440.0 1.0))
      (test-gen-equal (make-sum-of-cosines 3 440.0) (make-sum-of-cosines 3 440.0) (make-sum-of-cosines 3 400.0))
      
      (let ((gen (make-sum-of-cosines 10)))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((den (sin (* (mus-phase gen) 0.5)))
		 (val1 (if (= 0.0 den)
			   1.0
			   (min 1.0 (* (mus-scaler gen)
				       (- (/ (sin (* (mus-phase gen)
						     (+ (mus-cosines gen) 0.5)))
					     (* 2.0 den))
					  0.5)))))
		 (val2 (gen 0.0)))
	    (if (> (abs (- val1 val2)) .002)
		(snd-display ";sum-of-cosines: ~A: ~A ~A" i val1 val2)))))

      (let ((gen1 (make-sum-of-cosines 10 100.0))
	    (gen2 (make-sum-of-cosines 10 -100.0))
	    (mx 0.0))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (set! mx (max mx (abs (- (gen1) (gen2))))))
	(if (fneq mx 0.0)
	    (snd-display ";sum-of-cosines +-: ~A" mx)))

      (let ((gen (make-sum-of-sines 10 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-sum-of-sines 10 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "sum-of-sines"
			 "sum-of-sines freq: 440.000Hz, phase: 0.000, sines: 10"
			 "cosp freq: 0.125379, phase: 0.000000, cosines: 10, scaler: 0.131579")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (sum-of-sines gen 0.0)))
	(vct-map! v1 (lambda () (if (sum-of-sines? gen1) (sum-of-sines gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map sum-of-sines: ~A ~A" v0 v1))
	(if (not (sum-of-sines? gen)) (snd-display ";~A not sum-of-sines?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";sum-of-sines phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";sum-of-sines frequency: ~F?" (mus-frequency gen)))
	(if (fneq (mus-scaler gen) .1315) (snd-display ";sum-of-sines scaler: ~F?" (mus-scaler gen)))
	(if (not (= (mus-cosines gen) 10)) (snd-display ";sum-of-sines cosines: ~D?" (mus-cosines gen)))
	(if (not (= (mus-length gen) 10)) (snd-display ";sum-of-sines length: ~D?" (mus-length gen)))
	(if (or (fneq (vct-ref v0 1) 0.784) (fneq (vct-ref v0 8) 0.181)) (snd-display ";sum-of-sines output: ~A" v0))
	(set! (mus-scaler gen) .5) (if (fneq (mus-scaler gen) 0.5) (snd-display ";sum-of-sines set-scaler: ~F?" (mus-scaler gen)))
	(set! (mus-cosines gen) 5) (if (not (= (mus-cosines gen) 5)) (snd-display ";set sum-of-sines cosines: ~D?" (mus-cosines gen)))
	(if (fneq (mus-scaler gen) .2525) (snd-display ";set sines->scaler: ~A" (mus-scaler gen))))
      
      (test-gen-equal (make-sum-of-sines 3 440.0) (make-sum-of-sines 3 440.0) (make-sum-of-sines 5 440.0))
      (test-gen-equal (make-sum-of-sines 3 440.0) (make-sum-of-sines 3 440.0) (make-sum-of-sines 3 440.0 1.0))
      (test-gen-equal (make-sum-of-sines 3 440.0) (make-sum-of-sines 3 440.0) (make-sum-of-sines 3 400.0))
      
      (let ((gen (make-sum-of-sines 5)))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((val1 (* (sum-of-n-sines (mus-phase gen) 5) (mus-scaler gen)))
		 (val2 (gen 0.0)))
	    (if (fneq val1 val2)
		(snd-display ";sum-of-sines: ~A ~A" val1 val2)))))

      (let ((gen1 (make-sum-of-sines 10 100.0))
	    (gen2 (make-sum-of-sines 10 -100.0))
	    (mx 0.0))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (set! mx (max mx (abs (+ (gen1) (gen2))))))
	(if (fneq mx 0.0)
	    (snd-display ";sum-of-sines +-: ~A" mx)))

      (let ((gen (make-sine-summation 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-sine-summation 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "sine-summation"
			 "sine-summation: frequency: 440.000, phase: 0.000, n: 1, a: 0.500, ratio: 1.000"
			 "sss freq: 0.125379, phase: 0.000000, a: 0.500000, b: 1.000000, an: 0.250000, a2: 1.250000, n: 1")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (sine-summation gen 0.0)))
	(vct-map! v1 (lambda () (if (sine-summation? gen1) (sine-summation gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map sine-summation: ~A ~A" v0 v1))
	(if (not (sine-summation? gen)) (snd-display ";~A not sine-summation?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";sine-summation phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";sine-summation frequency: ~F?" (mus-frequency gen)))
	(if (or (fneq (vct-ref v0 1) 0.249) (fneq (vct-ref v0 8) 1.296)) (snd-display ";sine-summation output: ~A" v0))
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";mus-scaler (a) sine-summation: ~A" (mus-scaler gen)))
	(set! (mus-scaler gen) 0.75)
	(if (fneq (mus-scaler gen) 0.75) (snd-display ";mus-scaler (set a) sine-summation: ~A" (mus-scaler gen))))
      
      (test-gen-equal (make-sine-summation 440.0) (make-sine-summation 440.0) (make-sine-summation 100.0))
      (test-gen-equal (make-sine-summation 440.0) (make-sine-summation 440.0) (make-sine-summation 440.0 1.0))
      (test-gen-equal (make-sine-summation 440.0) (make-sine-summation 440.0) (make-sine-summation 440.0 0.0 3))
      
      (let ((gen1 (make-sine-summation 1000 0 1 0.0 1))
	    (gen2 (make-oscil 1000))
	    (gen3 (make-sine-summation 1000 0 1 0.5 2))
	    (gen4 (make-oscil 1000))
	    (gen5 (make-oscil 3000))
	    (v0 (make-vct 10)))
	
	(call-with-current-continuation
	 (lambda (give-up)
	   (do ((i 0 (1+ i)))
	       ((= i 100))
	     (let ((ss (sine-summation gen1 0.0))
		   (os (oscil gen2 0.0))
		   (ss1 (sine-summation gen3 0.0))
		   (os1 (+ (oscil gen4 0.0) 
			   (* 0.5 (oscil gen5 0.0)))))
	       (if (ffneq ss os)
		   (begin
		     (snd-display ";sine-summation 1: ~A: os: ~A ss: ~A" i os ss)
		     (give-up)))
	       (if (ffneq ss1 os1)
		   (begin
		     (snd-display ";sine-summation 2: ~A: os1: ~A ss1: ~A" i os1 ss1)
		     (give-up)))))))
	(set! gen1 (make-sine-summation 440.0 0.0 0.0))
	(sine-summation gen1)
	(let ((val (sine-summation gen1)))
	  (if (fneq val 0.125050170279874) (snd-display ";sine-summation n=0: ~A" val))))
      
      (let ((ind (new-sound "test.snd" mus-next mus-bfloat)))
	(pad-channel 0 1000)
	(let ((gen (make-cosine-summation 100.0)))
	  (map-channel (lambda (y) (* .2 (cosine-summation gen 0.5))))
	  (let ((vals (channel->vct 280 10)))
	    (if (not (vequal vals (vct 0.229 0.224 0.218 0.211 0.203 0.195 0.187 0.178 0.169 0.160)))
		(snd-display ";cosine-summation: ~A" vals)))
	  (undo))
	(let ((angle 0.0)) 
	  (map-channel (lambda (y) (let ((val (sum-of-n-sines angle 3))) (set! angle (+ angle .1)) (* .1 val))))
	  (let ((vals (channel->vct 260 10)))
	    (if (not (vequal vals (vct 0.226 0.200 0.166 0.129 0.091 0.056 0.025 0.001 -0.015 -0.023)))
		(snd-display ";sum-of-n-sines: ~A" vals)))
	  (undo))
	(let ((angle 0.0)) 
	  (map-channel (lambda (y) (let ((val (sum-of-n-odd-sines angle 3))) (set! angle (+ angle .1)) (* .1 val))))
	  (let ((vals (channel->vct 260 10)))
	    (if (not (vequal vals (vct 0.035 0.007 0.000 0.014 0.039 0.069 0.091 0.100 0.092 0.070)))
		(snd-display ";sum-of-n-odd-sines: ~A" vals)))
	  (undo))
	(let ((angle 0.0)) 
	  (map-channel (lambda (y) (let ((val (sum-of-n-odd-cosines angle 3))) (set! angle (+ angle .1)) (* .1 val))))
	  (let ((vals (channel->vct 250 10)))
	    (if (not (vequal vals (vct 0.270 0.298 0.292 0.253 0.189 0.112 0.037 -0.024 -0.061 -0.072)))
		(snd-display ";sum-of-n-odd-cosines: ~A" vals)))
	  (undo))
	(close-sound ind))
      
      (let ((gen (make-asymmetric-fm 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-asymmetric-fm 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "asymmetric-fm"
			 "asymmetric-fm freq: 440.000Hz, phase: 0.000, ratio: 1.000, r: 1.000"
			 "asyfm r: 1.000000, freq: 0.125379, phase: 0.000000, ratio: 1.000000, cosr: 0.000000, sinr: 1.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (asymmetric-fm gen 0.0)))
	(vct-map! v1 (lambda () (if (asymmetric-fm? gen1) (asymmetric-fm gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map asymmetric-fm: ~A ~A" v0 v1))
	(if (not (asymmetric-fm? gen)) (snd-display ";~A not asymmetric-fm?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";asymmetric-fm phase: ~F?" (mus-phase gen)))
	(set! (mus-phase gen) 1.0)
	(if (fneq (mus-phase gen) 1.0) (snd-display ";set! asymmetric-fm phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";asymmetric-fm frequency: ~F?" (mus-frequency gen)))
	(set! (mus-frequency gen) 100.0)
	(if (fneq (mus-frequency gen) 100.0) (snd-display ";set! asymmetric-fm frequency: ~F?" (mus-frequency gen)))
	(if (or (fneq (vct-ref v0 2) 0.248) (fneq (vct-ref v0 8) .843)) (snd-display ";asymmetric-fm output: ~A" v0))
	(if (fneq (mus-scaler gen) 1.0) (snd-display ";mus-scaler (r) asymmetric-fm: ~A" (mus-scaler gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";mus-scaler (set r) asymmetric-fm: ~A" (mus-scaler gen))))
      
      (test-gen-equal (make-asymmetric-fm 440.0) (make-asymmetric-fm 440.0) (make-asymmetric-fm 100.0))
      (test-gen-equal (make-asymmetric-fm 440.0) (make-asymmetric-fm 440.0) (make-asymmetric-fm 440.0 1.0))
      (test-gen-equal (make-asymmetric-fm 440.0) (make-asymmetric-fm 440.0) (make-asymmetric-fm 440.0 0.0 3))
      
      (let ((gen1 (make-asymmetric-fm 1000 0 1.0 0.1))
	    (gen2 (make-oscil 1000)))
	(call-with-current-continuation
	 (lambda (give-up)
	   (do ((i 0 (1+ i)))
	       ((= i 100))
	     (let ((ss (asymmetric-fm gen1 0.0 0.0))
		   (os (oscil gen2 0.0)))
	       (if (fneq ss os)
		   (begin
		     (snd-display ";asymmetric-fm 1: ~A: os: ~A ss: ~A" i os ss)
		     (give-up))))))))
      
      (let ((vct0 (make-vct 2048))
	    (vct1 (make-vct 2048))
	    (gen3 (make-asymmetric-fm 1000 0 1.0 0.2))
	    (gen4 (make-oscil 1000))
	    (gen5 (make-oscil 200))
	    (fm1 (in-hz (* 1.0 .2 1000)))) ; make notions of "index" match
	(do ((i 0 (1+ i)))
	    ((= i 2048))
	  (vct-set! vct0 i (asymmetric-fm gen3 1.0 0.0))
	  (vct-set! vct1 i (oscil gen4 (* fm1 (oscil gen5)))))
	(let* ((spectr1 (snd-spectrum vct0 rectangular-window 2048 #t))
	       (spectr2 (snd-spectrum vct1 rectangular-window 2048 #t)))
	  (call-with-current-continuation
	   (lambda (give-up)
	     (do ((i 1 (1+ i)))
		 ((= i 512))
	       (if (ffneq (vct-ref spectr1 i) (vct-ref spectr2 i))
		   (begin
		     (snd-display ";asymmetric-fm 2: ~A: ~A ~A" (* i (/ 22050 2048)) (vct-ref spectr1 i) (vct-ref spectr2 i))
		     (give-up))))))))
      
      (let ((gen (make-asymmetric-fm 40.0 0.0 1.0 0.1))
	    (gen1 (make-asyfm :frequency 40.0 :ratio .1 :index 2.0)))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 40.0) 22050.0))))
	    ((= i 1100))
	  (let* ((val1 (asymmetric-fm gen 2.0)) ; 1.0=index
		 (val3 (asyfm-J gen1 0.0))
		 (r 1.0)
		 (ratio 0.1)
		 (index 2.0)
		 (freq (hz->radians 40.0))
		 (phase a)
		 (cr (* 0.5 (- r (/ 1.0 r))))
		 (sr (* 0.5 (+ r (/ 1.0 r))))
		 (th a)
		 (mth (* ratio th))
		 (val2 (* (exp (* index cr (cos mth)))
			  (sin (+ th (* index sr (sin mth)))))))
	    (if (or (fneq val1 val2)
		    (fneq val1 val3))
		(snd-display ";asyfm by hand: ~A: ~A ~A" i val1 val2 val3)))))

      (let ((vct0 (make-vct 2048))
	    (vct1 (make-vct 2048))
	    (gen3 (make-asymmetric-fm 1000 0 2.0 0.1))
	    (gen4 (make-asymmetric-fm 1000 0 0.5 0.1)))
	(do ((i 0 (1+ i)))
	    ((= i 2048))
	  (vct-set! vct0 i (asymmetric-fm gen3 2.0 0.0))
	  (vct-set! vct1 i (asymmetric-fm gen4 2.0 0.0)))
	(let* ((spectr1 (snd-spectrum vct0 rectangular-window 2048 #t))
	       (spectr2 (snd-spectrum vct1 rectangular-window 2048 #t))
	       (s1-loc 0)
	       (s2-loc 0))
	  (do ((i 1 (1+ i)))
	      ((= i 256))
	    (if (< (abs (- 1.0 (vct-ref spectr1 i))) .01) (set! s1-loc i))
	    (if (< (abs (- 1.0 (vct-ref spectr2 i))) .01) (set! s2-loc i)))
	  (if (> s2-loc s1-loc) (snd-display ";asymmetric-fm peaks: ~A ~A" s1-loc s2-loc))
	  (let ((center (* (/ 22050 2048) (* .5 (+ s1-loc s2-loc)))))
	    (if (> (abs (- 1000 center)) 60) (snd-display ";asymmetric-fm center: ~A" center)))
	  (set! (mus-scaler gen3) 0.5)
	  (do ((i 0 (1+ i)))
	      ((= i 2048))
	    (vct-set! vct0 i (asymmetric-fm gen3 2.0 0.0)))
	  (set! spectr1 (snd-spectrum vct0 rectangular-window 2048 #t))
	  (do ((i 1 (1+ i)))
	      ((= i 256))
	    (if (< (abs (- 1.0 (vct-ref spectr1 i))) .01) (set! s1-loc i)))
	  (if (not (= s2-loc s1-loc)) (snd-print (format #f "asymmetric-fm set r peaks: ~A ~A" s1-loc s2-loc)))
	  (do ((i 0 (1+ i)))
	      ((= i 2048))
	    (vct-set! vct0 i (asymmetric-fm gen3 2.0 0.0)))
	  (snd-spectrum vct0 rectangular-window 2048 #t 0.0 #t)
	  (do ((i 1 (1+ i)))
	      ((= i 256))
	    (if (< (abs (- 1.0 (vct-ref vct0 i))) .01) (set! s1-loc i)))
	  (if (not (= s2-loc s1-loc)) (snd-print (format #f "asymmetric-fm set r in place peaks: ~A ~A" s1-loc s2-loc)))))
      
      (let ((gen (make-fir-filter 3 (list->vct '(.5 .25 .125))))
	    (v0 (make-vct 10))
	    (gen1 (make-fir-filter 3 (list->vct '(.5 .25 .125))))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "fir-filter"
			 "fir-filter: order: 3"
			 "flt order: 3, state (local): [0.000 0.000 0.000], x: [0.500 0.250 0.125], y: nil")
	(vct-set! v0 0 (fir-filter gen 1.0))
	(do ((i 1 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (fir-filter gen 0.0)))
	(vct-map! v1 (let ((inp 1.0))
		       (lambda () 
			 (let ((val (if (fir-filter? gen1) (fir-filter gen1 inp) -1.0)))
			   (set! inp 0.0)
			   val))))
	(if (not (vequal v0 v1)) (snd-display ";map fir-filter: ~A ~A" v0 v1))
	(if (not (fir-filter? gen)) (snd-display ";~A not fir-filter?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";fir-filter length: ~D?" (mus-length gen)))
	(if (or (fneq (vct-ref v0 1) 0.25) (fneq (vct-ref v0 2) .125)) (snd-display ";fir-filter output: ~A" v0))
	(let ((data (mus-xcoeffs gen)))
	  (if (fneq (vct-ref data 1) .25) (snd-display ";fir-filter xcoeffs: ~A?" data)))
	(let ((tag (catch #t (lambda () (mus-xcoeff gen 123)) (lambda args (car args)))))
	  (if (not (eq? tag 'mus-error))
	      (snd-display ";xcoeff 123: ~A" tag)))
	(let ((tag (catch #t (lambda () (mus-ycoeff gen 123)) (lambda args (car args)))))
	  (if (not (eq? tag 'mus-error))
	      (snd-display ";fir ycoeff 123: ~A" tag))))
      
      (test-gen-equal (let ((f1 (make-fir-filter 3 (list->vct '(.5 .25 .125))) )) (fir-filter f1 1.0) f1)
		      (let ((f2 (make-fir-filter 3 (list->vct '(.5 .25 .125))) )) (fir-filter f2 1.0) f2)
		      (let ((f3 (make-fir-filter 3 (list->vct '(.75 .25 .125))))) (fir-filter f3 1.0) f3))
      (test-gen-equal (let ((f1 (make-fir-filter 3 (list->vct '(.5 .25 .125))) )) (fir-filter f1 1.0) f1)
		      (let ((f2 (make-fir-filter 3 (list->vct '(.5 .25 .125))) )) (fir-filter f2 1.0) f2)
		      (let ((f3 (make-fir-filter 2 (list->vct '(.5 .25))))) (fir-filter f3 1.0) f3))
      
      (let* ((coeffs (list .1 .2 .3 .4 .4 .3 .2 .1))
	     (flt (make-fir-filter 8 (list->vct coeffs)))
	     (es (make-vector 8)))
	(do ((i 0 (1+ i)))
	    ((= i 8))
	  (vector-set! es i (make-env (list 0 (list-ref coeffs i) 1 0) :end 101)))
	(vector-set! es 5 (make-env '(0 .4 1 1) :end 101))
	(let ((data (make-vct 100)))
	  (do ((k 0 (1+ k)))
	      ((= k 100))
	    (let ((val (fir-filter flt (if (= (modulo k 12) 0) 1.0 0.0)))
		  (xcof (mus-xcoeffs flt)))
	      (do ((i 0 (1+ i)))
		  ((= i 8))
		(vct-set! xcof i (env (vector-ref es i))))
	      (vct-set! data k val)))
	  (if (or (fneq (vct-ref data 1) .2)
		  (fneq (vct-ref data 10) 0.0)
		  (fneq (vct-ref data 18) 0.166)
		  (fneq (vct-ref data 89) 0.923))
	      (snd-display ";filter xcoeffs: ~A?" data))))
      
      (let ((gen (make-iir-filter 3 (list->vct '(.5 .25 .125))))
	    (v0 (make-vct 10))
	    (gen1 (make-iir-filter 3 (list->vct '(.5 .25 .125))))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "iir-filter"
			 "iir-filter: order: 3"
			 "flt order: 3, state (local): [0.000 0.000 0.000], x: nil, y: [0.500 0.250 0.125]")
	(vct-set! v0 0 (iir-filter gen 1.0))
	(do ((i 1 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (iir-filter gen 0.0)))
	(vct-map! v1 (let ((inp 1.0))
		       (lambda ()
			 (let ((val (if (iir-filter? gen1) (iir-filter gen1 inp) -1.0)))
			   (set! inp 0.0)
			   val))))
	(if (not (vequal v0 v1)) (snd-display ";map iir-filter: ~A ~A" v0 v1))
	(if (not (iir-filter? gen)) (snd-display ";~A not iir-filter?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";iir-filter length: ~D?" (mus-length gen)))
	(if (or (fneq (vct-ref v0 1) -0.25) (fneq (vct-ref v0 2) -.062)) (snd-display ";iir-filter output: ~A" v0))
	(let ((data (mus-ycoeffs gen)))
	  (if (fneq (vct-ref data 1) .25) (snd-display ";iir-filter ycoeffs: ~A?" data)))
	(let ((tag (catch #t (lambda () (mus-ycoeff gen 123)) (lambda args (car args)))))
	  (if (not (eq? tag 'mus-error))
	      (snd-display ";ycoeff 123: ~A" tag)))
	(let ((tag (catch #t (lambda () (mus-xcoeff gen 123)) (lambda args (car args)))))
	  (if (not (eq? tag 'mus-error))
	      (snd-display ";iir xcoeff 123: ~A" tag))))
      
      (test-gen-equal (let ((f1 (make-iir-filter 3 (list->vct '(.5 .25 .125))))) (iir-filter f1 1.0) f1)
		      (let ((f2 (make-iir-filter 3 (list->vct '(.5 .25 .125))) )) (iir-filter f2 1.0) f2)
		      (let ((f3 (make-iir-filter 3 (list->vct '(.75 .25 .125))))) (iir-filter f3 1.0) f3))
      (test-gen-equal (let ((f1 (make-iir-filter 3 (list->vct '(.5 .25 .125))) )) (iir-filter f1 1.0) f1)
		      (let ((f2 (make-iir-filter 3 (list->vct '(.5 .25 .125))) )) (iir-filter f2 1.0) f2)
		      (let ((f3 (make-iir-filter 2 (list->vct '(.5 .25))))) (iir-filter f3 1.0) f3))
      
      (let ((gen (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .25 .125))))
	    (v0 (make-vct 10))
	    (gen1 (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .25 .125))))
	    (v1 (make-vct 10))
	    (gen2 (make-biquad .1 .2 .3 .4 .5)))
	(print-and-check gen 
			 "filter"
			 "filter: order: 3"
			 "flt order: 3, state (local): [0.000 0.000 0.000], x: [0.500 0.250 0.125], y: [0.500 0.250 0.125]")
	(vct-set! v0 0 (filter gen 1.0))
	(do ((i 1 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (filter gen 0.0)))
	(vct-map! v1 (let ((inp 1.0))
		       (lambda () 
			 (let ((val (if (filter? gen1) (filter gen1 inp) -1.0)))
			   (set! inp 0.0)
			   val))))
	(if (not (vequal v0 v1)) (snd-display ";map filter: ~A ~A" v0 v1))
	(if (not (filter? gen)) (snd-display ";~A not filter?" gen))
	(if (not (= (mus-length gen) 3)) (snd-display ";filter length: ~D?" (mus-length gen)))
	(if (or (fneq (vct-ref v0 1) 0.125) (fneq (vct-ref v0 2) .031)) (snd-display ";filter output: ~A" v0))
	(if (not (filter? gen2)) (snd-display ";make-biquad: ~A" gen2))
	(let ((xs (mus-xcoeffs gen))
	      (ys (mus-ycoeffs gen)))
	  (if (or (not (equal? xs (list->vct '(.5 .25 .125))))
		  (not (equal? xs ys)))
	      (snd-display ";mus-xcoeffs: ~A ~A?" xs ys))))
      
      (let ((var (catch #t (lambda () (make-filter :order 2 :xcoeffs (vct 1.0 0.5) :ycoeffs (vct 2.0 1.0 0.5))) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-filter bad coeffs: ~A" var)))
      (let ((var (catch #t (lambda () (make-filter :order 0 :xcoeffs (vct 1.0 0.5))) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-filter bad order: ~A" var)))
      (let ((var (catch #t (lambda () (make-fir-filter :order 22 :xcoeffs (vct 1.0 0.5))) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-fir-filter bad coeffs: ~A" var)))
      (let ((var (catch #t (lambda () (make-iir-filter :order 22 :ycoeffs (vct 1.0 0.5))) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-iir-filter bad coeffs: ~A" var)))
      (let ((var (catch #t (lambda () (make-fir-filter -1)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-fir-filter bad order: ~A" var)))
      (let ((var (make-filter :order 2 :ycoeffs (vct 1.0 0.5))))
	(if (not (iir-filter? var))
	    (snd-display ";make-filter with only y: ~A" var)))
      
      (test-gen-equal (let ((f1 (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .25 .125))))) (filter f1 1.0) f1)
		      (let ((f2 (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .25 .125))))) (filter f2 1.0) f2)
		      (let ((f3 (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .5 .5))))) (filter f3 1.0) f3))
      (test-gen-equal (let ((f1 (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .25 .125))))) (filter f1 1.0) f1)
		      (let ((f2 (make-filter 3 (list->vct '(.5 .25 .125)) (list->vct '(.5 .25 .125))))) (filter f2 1.0) f2)
		      (let ((f3 (make-filter 3 (list->vct '(.5 .5 .125)) (list->vct '(.5 .25 .0625))))) (filter f3 1.0) f3))
      
      (let ((fr (make-fir-filter 6 (vct 0 1 2 3 4 5))))
	(if (not (= (mus-length fr) 6)) (snd-display ";filter-length: ~A" (mus-length fr)))
	(set! (mus-length fr) 3)
	(if (not (= (mus-length fr) 3)) (snd-display ";set filter-length: ~A" (mus-length fr))))

      (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050)))
	(pad-channel 0 10000)
	(freq-sweep .45)
	(let ((sp (rough-spectrum ind)))
	  (if (and (not (vequal sp (vct 0.962 0.998 0.998 0.998 0.998 0.999 0.999 0.998 0.997 1.000)))
		   (not (vequal sp (vct 0.963 0.999 0.999 0.999 0.999 0.999 1.000 1.000 0.998 0.997))))
	      (snd-display ";initial rough spectrum: ~A" sp)))
	
	(let ((b (make-butter-high-pass 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.915 -0.162 -0.146 -0.131 -0.117 -0.103 -0.090 -0.078 -0.066 -0.056)))
	      (snd-display ";butter high: ~A" v))
	  (set! b (make-butter-high-pass 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.150 0.833 0.980 0.994 0.997 0.998 0.999 0.998 0.997 1.000)))
		     (not (vequal sp (vct 0.150 0.833 0.981 0.995 0.998 0.999 1.000 1.000 0.998 0.997))))
		(snd-display ";hp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-low-pass 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.004 0.014 0.026 0.035 0.043 0.049 0.053 0.055 0.057 0.057)))
	      (snd-display ";butter low: ~A" v))
	  (set! b (make-butter-low-pass 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 1.000 0.212 0.024 0.005 0.001 0.000 0.000 0.000 0.000 0.000)))
		(snd-display ";lp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-band-pass 440.0 50.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.007 0.014 0.013 0.013 0.012 0.011 0.009 0.008 0.007 0.005)))
	      (snd-display ";butter bandpass: ~A" v))
	  (set! b (make-butter-band-pass 1000.0 500.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 0.888 1.000 0.144 0.056 0.027 0.014 0.008 0.004 0.002 0.000)))
		(snd-display ";bp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-band-reject 440.0 50.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.993 -0.014 -0.013 -0.013 -0.012 -0.011 -0.009 -0.008 -0.007 -0.005)))
	      (snd-display ";butter bandstop: ~A" v))
	  (set! b (make-butter-band-reject 1000.0 500.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.662 0.687 0.953 0.980 0.989 0.994 0.997 0.997 0.997 1.000)))
		     (not (vequal sp (vct 0.664 0.689 0.955 0.982 0.992 0.996 0.999 1.000 0.999 0.998))))
		(snd-display ";bs rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((v (spectrum->coeffs 10 (vct 0 1.0 0 0 0 0 0 0 1.0 0)))
	      (v1 (make-fir-coeffs 10 (vct 0 1.0 0 0 0 0 0 0 1.0 0))))
	  (if (not (vequal v (vct -0.190 -0.118 0.000 0.118 0.190 0.190 0.118 0.000 -0.118 -0.190)))
	      (snd-display ";spectrum->coeffs: ~A" v))
	  (if (not (vequal v v1))
	      (snd-display ";spectrum->coeffs v make-fir-coeffs: ~A ~A" v v1)))

	(let ((notched-spectr (make-vct 20)))
	  (vct-set! notched-spectr 2 1.0) 
	  (let ((v (spectrum->coeffs 20 notched-spectr))
		(v1 (make-fir-coeffs 20 notched-spectr)))
	    (if (not (vequal v (vct 0.095 0.059 -0.000 -0.059 -0.095 -0.095 -0.059 0.000 0.059 0.095 
				    0.095 0.059 0.000 -0.059 -0.095 -0.095 -0.059 -0.000 0.059 0.095)))
		(snd-display ";spectrum->coeffs (notch): ~A" v))
	    (if (not (vequal v v1))
		(snd-display ";spectrum->coeffs v(2) make-fir-coeffs: ~A ~A" v v1))
	    (let ((flt (make-fir-filter 20 v)))
	      (map-channel (lambda (y) (fir-filter flt y)))))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 0.007 0.493 1.000 0.068 0.030 0.019 0.014 0.011 0.009 0.009)))
		(snd-display ";sp->coeff rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((rspect (make-vct 20)))
	  (do ((i 0 (1+ i)))
	      ((= i 20))
	    (vct-set! rspect i (random 1.0)))
	  (let ((v (spectrum->coeffs 20 rspect))
		(v1 (make-fir-coeffs 20 rspect)))
	    (if (not (vequal v v1))
		(snd-display ";spectrum->coeffs v(3) make-fir-coeffs: ~A ~A" v v1))))
	
	(let ((b (make-highpass (hz->radians 1000.0) 10))
	      (v (make-vct 20))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (fir-filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct -0.001 -0.002 -0.005 -0.011 -0.021 -0.034 -0.049 -0.065 -0.078 -0.087 
				  0.909 -0.087 -0.078 -0.065 -0.049 -0.034 -0.021 -0.011 -0.005 -0.002)))
	      (snd-display ";dsp.scm high: ~A" v))
	  (set! b (make-highpass (hz->radians 1000.0) 20))
	  (map-channel (lambda (y) (fir-filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.053 0.774 0.998 0.997 0.997 0.996 0.996 0.996 0.997 1.000)))
		     (not (vequal sp (vct 0.053 0.776 1.000 0.998 0.998 0.998 0.998 0.998 0.998 1.000))))
		(snd-display ";dsp hp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-lowpass (hz->radians 1000.0) 10))
	      (v (make-vct 20))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (fir-filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.001 0.002 0.005 0.011 0.021 0.034 0.049 0.065 0.078 0.087 0.091 0.087 0.078 0.065
				  0.049 0.034 0.021 0.011 0.005 0.002)))
	      (snd-display ";dsp.scm low: ~A" v))
	  (set! b (make-lowpass (hz->radians 1000.0) 20))
	  (map-channel (lambda (y) (fir-filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 1.000 0.054 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
		(snd-display ";dsp lp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-bandpass (hz->radians 1500.0) (hz->radians 2000.0) 10))
	      (v (make-vct 20))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (fir-filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.001 -0.001 -0.005 -0.011 -0.017 -0.019 -0.013 0.003 0.022 0.039 0.045
				  0.039 0.022 0.003 -0.013 -0.019 -0.017 -0.011 -0.005 -0.001)))
	      (snd-display ";dsp.scm bp: ~A" v))
	  (set! b (make-bandpass (hz->radians 1500.0) (hz->radians 2000.0) 20))
	  (map-channel (lambda (y) (fir-filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 0.010 1.000 0.154 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
		(snd-display ";dsp bp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-bandstop (hz->radians 1500.0) (hz->radians 2000.0) 10))
	      (v (make-vct 20))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (fir-filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct -0.001 0.001 0.005 0.011 0.017 0.019 0.013 -0.003 -0.022 -0.039 0.955
				  -0.039 -0.022 -0.003 0.013 0.019 0.017 0.011 0.005 0.001)))
	      (snd-display ";dsp.scm bs: ~A" v))
	  (set! b (make-bandstop (hz->radians 1500.0) (hz->radians 2000.0) 20))
	  (map-channel (lambda (y) (fir-filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.904 0.425 0.821 0.998 0.997 0.996 0.996 0.996 0.997 1.000)))
		     (not (vequal sp (vct 0.906 0.425 0.822 1.000 0.999 0.998 0.998 0.998 0.998 1.000))))
		(snd-display ";dsp bs rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-differentiator 10))
	      (v (make-vct 20))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (fir-filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct -0.008 0.011 -0.021 0.039 -0.066 0.108 -0.171 0.270 -0.456 0.977
				  0.000 -0.977 0.456 -0.270 0.171 -0.108 0.066 -0.039 0.021 -0.011)))
	      (snd-display ";dsp.scm df: ~A" v))
	  (set! b (make-differentiator 20))
	  (map-channel (lambda (y) (fir-filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 0.004 0.027 0.075 0.147 0.242 0.362 0.506 0.674 0.864 1.000)))
		(snd-display ";dsp df rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-iir-high-pass-1 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.941 -0.111 -0.098 -0.086 -0.076 -0.067 -0.059 -0.052 -0.046 -0.041)))
	      (snd-display ";iir-1 high: ~A" v))
	  (set! b (make-iir-high-pass-1 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.228 0.706 0.879 0.940 0.967 0.982 0.990 0.994 0.996 1.000)))
		     (not (vequal sp (vct 0.229 0.709 0.883 0.944 0.971 0.986 0.994 0.999 1.000 1.000))))
		(snd-display ";iir-1 hp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-iir-low-pass-1 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.059 0.111 0.098 0.086 0.076 0.067 0.059 0.052 0.046 0.041)))
	      (snd-display ";iir-1 low: ~A" v))
	  (set! b (make-iir-low-pass-1 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 1.000 0.402 0.164 0.080 0.043 0.023 0.013 0.006 0.003 0.001)))
		(snd-display ";iir-1 lp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-iir-high-pass-2 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.915 -0.162 -0.146 -0.131 -0.117 -0.103 -0.090 -0.078 -0.066 -0.056)))
	      (snd-display ";iir-2 high: ~A" v))
	  (set! b (make-iir-high-pass-2 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.150 0.833 0.980 0.994 0.997 0.998 0.999 0.998 0.997 1.000)))
		     (not (vequal sp (vct 0.150 0.833 0.981 0.995 0.998 0.999 1.000 1.000 0.998 0.997))))
		(snd-display ";iir-2 hp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-iir-low-pass-2 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (butter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.004 0.014 0.026 0.035 0.043 0.049 0.053 0.055 0.057 0.057)))
	      (snd-display ";iir-2 low: ~A" v))
	  (set! b (make-iir-low-pass-2 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 1.000 0.212 0.024 0.005 0.001 0.000 0.000 0.000 0.000 0.000)))
		(snd-display ";iir-2 lp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-iir-band-pass-2 440.0 490.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.007 0.014 0.013 0.013 0.012 0.010 0.009 0.008 0.006 0.004)))
	      (snd-display ";iir bp-2 bandpass: ~A" v))
	  (set! b (make-iir-band-pass-2 1000.0 1500.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (not (vequal sp (vct 0.239 1.000 0.117 0.041 0.019 0.010 0.005 0.003 0.001 0.000)))
		(snd-display ";iir bp-2 rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-iir-band-stop-2 440.0 500.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.992 -0.017 -0.016 -0.015 -0.014 -0.012 -0.011 -0.009 -0.007 -0.005)))
	      (snd-display ";iir-2 bandstop: ~A" v))
	  (set! b (make-iir-band-stop-2 1000.0 1500.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.836 0.525 0.943 0.979 0.989 0.994 0.997 0.997 0.997 1.000)))
		     (not (vequal sp (vct 0.838 0.527 0.945 0.981 0.991 0.996 0.999 1.000 0.999 0.998))))
		(snd-display ";iir bs-2 rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-hp 4 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (filter b inv))) (set! inv 0.0) val)))
	  (if (and (not (vequal v (vct 0.725 -0.466 -0.315 -0.196 -0.104 -0.036 0.014 0.047 0.0685 0.0775)))
		   (not (vequal v (vct 0.725 -0.466 -0.315 -0.196 -0.104 -0.035 0.015 0.049 0.070 0.081))))
	      (snd-display ";butter hp: ~A" v))
	  (set! b (make-butter-hp 4 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.0505 0.982 1.000 1.000 0.998 0.998 0.999 0.998 0.996 0.999)))
		     (not (vequal sp (vct 0.051 0.982 1.000 1.000 0.998 0.998 0.998 0.999 0.997 0.995)))
		     (not (vequal sp (vct 0.051 0.991 1.000 1.000 0.998 0.998 0.999 0.999 0.997 0.995))))
		(snd-display ";bhp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-lp 4 440.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000))) ;; ???
	      (snd-display ";butter lp: ~A" v))
	  (set! b (make-butter-lp 4 1000.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 1.000 0.035 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
		     (not (vequal sp (vct 1.000 0.038 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000))))
		(snd-display ";blp rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-bp 4 440.0 500.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (filter b inv))) (set! inv 0.0) val)))
	  (if (not (vequal v (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
	      (snd-display ";butter bp: ~A" v))
	  (set! b (make-butter-bp 4 1000.0 1500.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.026 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
		     (not (vequal sp (vct  0.022 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000)))
		     (not (vequal sp (vct 0.042 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000))))
		(snd-display ";bp 4 rough spectrum: ~A" sp)))
	  (undo))
	
	(let ((b (make-butter-bs 4 440.0 500.0))
	      (v (make-vct 10))
	      (inv 1.0))
	  (vct-map! v (lambda () (let ((val (filter b inv))) (set! inv 0.0) val)))
	  (if (and (not (vequal v (vct 0.978 -0.043 -0.041 -0.038 -0.035 -0.031 -0.026 -0.0225 -0.015 -0.0085)))
		   (not (vequal v (vct 0.978 -0.043 -0.041 -0.038 -0.035 -0.031 -0.027 -0.022 -0.017 -0.011))))
	      (snd-display ";butter bs: ~A" v))
	  (set! b (make-butter-bs 4 1000.0 1500.0))
	  (map-channel (lambda (y) (filter b y)))
	  (let ((sp (rough-spectrum ind)))
	    (if (and (not (vequal sp (vct 0.798 0.657 1.000 0.997 0.996 0.997 0.997 0.996 0.995 0.998)))
		     (not (vequal sp (vct 0.795 0.668 1.000 0.997 0.996 0.997 0.997 0.997 0.995 0.994)))
		     (not (vequal sp (vct 0.801 0.698 1.000 0.997 0.996 0.997 0.997 0.997 0.995 0.994)))
		     (not (vequal sp (vct 0.884 0.586 1.000 0.996 0.996 0.997 0.997 0.997 0.995 0.994)))
		     (not (vequal sp (vct 0.793 0.667 1.000 0.997 0.996 0.997 0.997 0.997 0.995 0.994))))
		(snd-display ";bs 4 rough spectrum: ~A" sp)))
	  (undo))
	
	(close-sound ind))
      
      (let ((gen (make-sawtooth-wave 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-sawtooth-wave 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "sawtooth-wave"
			 "sawtooth-wave freq: 440.000Hz, phase: 3.142, amp: 1.000"
			 "sw current_value: 0.000000, freq: 0.125379, phase: 3.141593, base: 0.318310")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (sawtooth-wave gen 0.0)))
	(vct-map! v1 (lambda () (if (sawtooth-wave? gen1) (sawtooth-wave gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map sawtooth: ~A ~A" v0 v1))
	(if (not (sawtooth-wave? gen)) (snd-display ";~A not sawtooth-wave?" gen))
	(if (fneq (mus-phase gen) 4.39538) (snd-display ";sawtooth-wave phase: ~F?" (mus-phase gen))) ;starts at pi
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";sawtooth-wave frequency: ~F?" (mus-frequency gen)))
	(set! (mus-frequency gen) 100.0)
	(if (fneq (mus-frequency gen) 100.0) (snd-display ";set! sawtooth-wave frequency: ~F?" (mus-frequency gen)))
	(if (fneq (mus-scaler gen) 1.0) (snd-display ";sawtooth-wave scaler: ~F?" (mus-scaler gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";set! sawtooth-wave scaler: ~F?" (mus-scaler gen)))
	(if (or (fneq (vct-ref v0 1) 0.04) (fneq (vct-ref v0 8) .319)) (snd-display ";sawtooth-wave output: ~A" v0)))
      
      (test-gen-equal (make-sawtooth-wave 440.0) (make-sawtooth-wave 440.0) (make-sawtooth-wave 120.0))
      (test-gen-equal (make-sawtooth-wave 440.0) (make-sawtooth-wave 440.0) (make-sawtooth-wave 440.0 1.0 1.0))
      (test-gen-equal (make-sawtooth-wave 440.0) (make-sawtooth-wave 440.0) (make-sawtooth-wave 440.0 0.5))
      
      (let ((gen1 (make-sawtooth-wave 100.0))
	    (gen2 (make-sawtooth-wave -100.0))
	    (mx 0.0))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (set! mx (max mx (abs (+ (gen1) (gen2))))))
	(if (fneq mx 0.0)
	    (snd-display ";sawtooth +-: ~A" mx)))

      (let ((gen (make-square-wave 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-square-wave 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "square-wave"
			 "square-wave freq: 440.000Hz, phase: 0.000, amp: 1.000"
			 "sw current_value: 1.000000, freq: 0.125379, phase: 0.000000, base: 1.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (square-wave gen 0.0)))
	(let ((w 1.0))
	  (vct-map! v1 (lambda () 
			 (set! w (mus-width gen1))
			 (if (square-wave? gen1) (square-wave gen1 0.0) -1.0)))
	  (if (fneq w 0.5) (snd-display ";mus-width opt: ~A" w)))
	(if (not (vequal v0 v1)) (snd-display ";map square-wave: ~A ~A" v0 v1))
	(if (not (square-wave? gen)) (snd-display ";~A not square-wave?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";square-wave phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";square-wave frequency: ~F?" (mus-frequency gen)))
	(if (fneq (mus-scaler gen) 1.0) (snd-display ";square-wave scaler: ~F?" (mus-scaler gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";set! square-wave scaler: ~F?" (mus-scaler gen)))
	(if (fneq (mus-width gen) 0.5) (snd-display ";square-wave width: ~A" (mus-width gen)))
	(set! (mus-width gen) 0.75)
	(if (fneq (mus-width gen) 0.75) (snd-display ";set! square-wave width: ~A" (mus-width gen)))
	(if (or (fneq (vct-ref v0 1) 1.0) (fneq (vct-ref v0 8) 1.0)) (snd-display ";square-wave output: ~A" v0)))
      
      (test-gen-equal (make-square-wave 440.0) (make-square-wave 440.0) (make-square-wave 120.0))
      (test-gen-equal (make-square-wave 440.0) (make-square-wave 440.0) (make-square-wave 440.0 1.0 1.0))
      (test-gen-equal (make-square-wave 440.0) (make-square-wave 440.0) (make-square-wave 440.0 0.5))
      
      (let ((old-srate (mus-srate)))
	(set! (mus-srate) 500.0)
	(let ((gen (make-square-wave 100.0 -0.5 (* pi 0.5)))
	      (v0 (make-vct 20)))
	  (do ((i 0 (1+ i)))
	      ((= i 20))
	    (vct-set! v0 i (gen)))
	  (if (not (vequal v0 (vct -0.5 -0.5 0.0 0.0 -0.5 -0.5 -0.5 0.0 0.0 -0.5 -0.5 -0.5 0.0 0.0 -0.5 -0.5 -0.5 0.0 0.0 -0.5)))
	      (snd-display ";square-wave -.5: ~A " v0)))
	(set! (mus-srate) old-srate))

      (let ((gen (make-triangle-wave 440.0))
	    (gen1 (make-triangle-wave 440.0 1.0 pi))
	    (v0 (make-vct 10))
	    (gen2 (make-triangle-wave 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "triangle-wave"
			 "triangle-wave freq: 440.000Hz, phase: 0.000, amp: 1.000"
			 "sw current_value: 0.000000, freq: 0.125379, phase: 0.000000, base: 0.636620")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (triangle-wave gen 0.0)))
	(vct-map! v1 (lambda () (if (triangle-wave? gen2) (triangle-wave gen2 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map triangle-wave: ~A ~A" v0 v1))
	(if (not (triangle-wave? gen)) (snd-display ";~A not triangle-wave?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";triangle-wave phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-phase gen1) pi) (snd-display ";init triangle-wave phase: ~F?" (mus-phase gen1)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";triangle-wave frequency: ~F?" (mus-frequency gen)))
	(if (fneq (mus-scaler gen) 1.0) (snd-display ";triangle-wave scaler: ~F?" (mus-scaler gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";set! triangle-wave scaler: ~F?" (mus-scaler gen)))
	(if (or (fneq (vct-ref v0 1) 0.080) (fneq (vct-ref v0 8) 0.639)) (snd-display ";triangle-wave output: ~A" v0)))
      
      (let ((gen1 (make-triangle-wave 100.0))
	    (gen2 (make-triangle-wave -100.0))
	    (mx 0.0))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (set! mx (max mx (abs (+ (gen1) (gen2))))))
	(if (fneq mx 0.0)
	    (snd-display ";triangle +-: ~A" mx)))

      (test-gen-equal (make-triangle-wave 440.0) (make-triangle-wave 440.0) (make-triangle-wave 120.0))
      (test-gen-equal (make-triangle-wave 440.0) (make-triangle-wave 440.0) (make-triangle-wave 440.0 1.0 1.0))
      (test-gen-equal (make-triangle-wave 440.0) (make-triangle-wave 440.0) (make-triangle-wave 440.0 0.5))
      
      (let ((gen (make-pulse-train 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-pulse-train 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "pulse-train"
			 "pulse-train freq: 440.000Hz, phase: 0.000, amp: 1.000"
			 "sw current_value: 0.000000, freq: 0.125379, phase: 6.283185, base: 1.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (pulse-train gen 0.0)))
	(vct-map! v1 (lambda () (if (pulse-train? gen1) (pulse-train gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map pulse-train: ~A ~A" v0 v1))
	(if (not (pulse-train? gen)) (snd-display ";~A not pulse-train?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";pulse-train phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";pulse-train frequency: ~F?" (mus-frequency gen)))
	(if (fneq (mus-scaler gen) 1.0) (snd-display ";pulse-train scaler: ~F?" (mus-scaler gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";set! pulse-train scaler: ~F?" (mus-scaler gen)))
	(if (or (fneq (vct-ref v0 0) 1.0) (fneq (vct-ref v0 8) 0.0)) (snd-display ";pulse-train output: ~A" v0)))
      
      (test-gen-equal (make-pulse-train 440.0) (make-pulse-train 440.0) (make-pulse-train 120.0))
      (test-gen-equal (make-pulse-train 440.0) (make-pulse-train 440.0) (make-pulse-train 440.0 1.0 1.0))
      (test-gen-equal (make-pulse-train 440.0) (make-pulse-train 440.0) (make-pulse-train 440.0 0.5))

      (let ((old-srate (mus-srate)))
	(set! (mus-srate) 500.0)
	(let ((gen (make-pulse-train 100.0 -0.5 (* pi 0.5)))
	      (v0 (make-vct 20)))
	  (do ((i 0 (1+ i)))
	      ((= i 20))
	    (vct-set! v0 i (gen)))
	  (if (not (vequal v0 (vct  0.0 0.0 0.0 0.0 -0.5 0.0 0.0 0.0 0.0 -0.5 0.0 0.0 0.0 0.0 -0.5 0.0 0.0 0.0 0.0 -0.5)))
	      (snd-display ";pulse-train -.5: ~A " v0)))
	(set! (mus-srate) old-srate))

      
      (let ((gen (make-ppolar .1 1200.0))
	    (v0 (make-vct 10)))
	(vct-set! v0 0 (two-pole gen 1.0))
	(do ((i 1 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (two-pole gen 0.0)))
	(if (not (two-pole? gen)) (snd-display ";~A not ppolar?" gen))
	(if (not (= (mus-order gen) 2)) (snd-display ";ppolar order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) 1.0) (snd-display ";ppolar a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-b1 gen) -.188) (snd-display ";ppolar b1: ~F?" (mus-b1 gen)))
	(if (fneq (mus-b2 gen) .01) (snd-display ";ppolar b2: ~F?" (mus-b2 gen)))
	(if (or (fneq (vct-ref v0 0) 1.0) (fneq (vct-ref v0 1) .188)) (snd-display ";ppolar output: ~A" v0)))
      
      (test-gen-equal (let ((z1 (make-ppolar .1 600.0))) (two-pole z1 1.0) z1)
		      (let ((z2 (make-ppolar .1 600.0))) (two-pole z2 1.0) z2)
		      (let ((z3 (make-ppolar .1 1200.0))) (two-pole z3 1.0) z3))
      (test-gen-equal (let ((z1 (make-ppolar .1 600.0))) (two-pole z1 1.0) z1)
		      (let ((z2 (make-ppolar .1 600.0))) (two-pole z2 1.0) z2)
		      (let ((z3 (make-ppolar .2 1200.0))) (two-pole z3 1.0) z3))
      (test-gen-equal (let ((z1 (make-ppolar .1 600.0))) (two-pole z1 1.0) z1)
		      (let ((z2 (make-ppolar .1 600.0))) (two-pole z2 1.0) z2)
		      (let ((z3 (make-ppolar .1 600.0))) (two-pole z3 0.5) z3))
      
      (let ((gen (make-zpolar :radius .1 :frequency 1200.0))
	    (v0 (make-vct 10)))
	(vct-set! v0 0 (two-zero gen 1.0))
	(do ((i 1 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (two-zero gen 0.0)))
	(if (not (two-zero? gen)) (snd-display ";~A not zpolar?" gen))
	(if (not (= (mus-order gen) 2)) (snd-display ";zpolar order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) 1.0) (snd-display ";zpolar a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-a1 gen) -.188) (snd-display ";zpolar a1: ~F?" (mus-a1 gen)))
	(if (fneq (mus-a2 gen) .01) (snd-display ";zpolar a2: ~F?" (mus-a2 gen)))
	(if (or (fneq (vct-ref v0 0) 1.0) (fneq (vct-ref v0 1) -.188)) (snd-display ";zpolar output: ~A" v0)))
      
      (test-gen-equal (let ((z1 (make-zpolar .1 600.0))) (two-zero z1 1.0) z1)
		      (let ((z2 (make-zpolar .1 600.0))) (two-zero z2 1.0) z2)
		      (let ((z3 (make-zpolar .1 1200.0))) (two-zero z3 1.0) z3))
      (test-gen-equal (let ((z1 (make-zpolar .1 600.0))) (two-zero z1 1.0) z1)
		      (let ((z2 (make-zpolar .1 600.0))) (two-zero z2 1.0) z2)
		      (let ((z3 (make-zpolar .2 1200.0))) (two-zero z3 1.0) z3))
      (test-gen-equal (let ((z1 (make-zpolar .1 600.0))) (two-zero z1 1.0) z1)
		      (let ((z2 (make-zpolar .1 600.0))) (two-zero z2 1.0) z2)
		      (let ((z3 (make-zpolar .1 600.0))) (two-zero z3 0.5) z3))
      
      (let ((gen (make-formant .9 1200.0 1.0))
	    (v0 (make-vct 10))
	    (gen1 (make-formant .9 1200.0 1.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "formant"
			 "formant: radius: 0.900, frequency: 1200.000, (gain: 1.000)"
			 "smpflt a0: 0.063710, a1: 1.000000, a2: -0.900000, b1: -1.695789, b2: 0.810000, x1: 0.000000, x2: 0.000000, y1: 0.000000, y2: 0.000000")
	(vct-set! v0 0 (formant gen 1.0))
	(do ((i 1 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (formant gen 0.0)))
	(vct-map! v1 (let ((inp 1.0))
		       (lambda () 
			 (let ((val (if (formant? gen1) (formant gen1 inp) -1.0)))
			   (set! inp 0.0)
			   val))))
	(if (not (vequal v0 v1)) (snd-display ";map formant: ~A ~A" v0 v1))
	(if (not (formant? gen)) (snd-display ";~A not formant?" gen))
	(if (not (= (mus-order gen) 2)) (snd-display ";formant order: ~D?" (mus-order gen)))
	(if (fneq (mus-a0 gen) 0.06371) (snd-display ";formant a0: ~F?" (mus-a0 gen)))
	(if (fneq (mus-a1 gen) 1.0) (snd-display ";formant a1: ~F?" (mus-a1 gen)))
	(if (fneq (mus-a2 gen) -0.9) (snd-display ";formant a2: ~F?" (mus-a2 gen)))
	(if (fneq (mus-b1 gen) -1.6957893) (snd-display ";formant b1: ~F?" (mus-b1 gen)))
	(if (fneq (mus-b2 gen) 0.81) (snd-display ";formant b2: ~F?" (mus-b2 gen)))
	(if (fneq (mus-formant-radius gen) .9) (snd-display ";formant radius: ~F?" (mus-formant-radius gen)))
	(if (fneq (mus-frequency gen) 1200.0) (snd-display ";formant frequency: ~F?" (mus-frequency gen)))
	(if (or (fneq (vct-ref v0 0) .064) (fneq (vct-ref v0 1) .108)) (snd-display ";formant output: ~A" v0))

	(if (fneq (mus-ycoeff gen 2) .81) (snd-display ";frm ycoeff 2 .81: ~A" (mus-ycoeff gen 2)))
	(set! (mus-ycoeff gen 2) .1)
	(if (fneq (mus-ycoeff gen 2) .1) (snd-display ";frm set ycoeff 2 .1: ~A" (mus-ycoeff gen 2)))
	(if (fneq (mus-xcoeff gen 2) -0.9) (snd-display ";frm xcoeff 2 -0.9: ~A" (mus-xcoeff gen 2)))
	(set! (mus-xcoeff gen 2) .3)
	(if (fneq (mus-xcoeff gen 2) .3) (snd-display ";frm set xcoeff 2 .3: ~A" (mus-xcoeff gen 2)))

	(set! (mus-a0 gen) .5) (if (fneq (mus-a0 gen) 0.5) (snd-display ";formant set-a0: ~F?" (mus-a0 gen)))
	(set! (mus-a1 gen) .5) (if (fneq (mus-a1 gen) 0.5) (snd-display ";formant set-a1: ~F?" (mus-a1 gen)))
	(set! (mus-a2 gen) .5) (if (fneq (mus-a2 gen) 0.5) (snd-display ";formant set-a2: ~F?" (mus-a2 gen)))
	(set! (mus-b1 gen) .5) (if (fneq (mus-b1 gen) 0.5) (snd-display ";formant set-b1: ~F?" (mus-b1 gen)))
	(set! (mus-b2 gen) .5) (if (fneq (mus-b2 gen) 0.5) (snd-display ";formant set-b2: ~F?" (mus-b2 gen)))
	(set! (mus-formant-radius gen) .01) 
	(if (fneq (mus-formant-radius gen) 0.01) (snd-display ";formant set-radius: ~F?" (mus-formant-radius gen))))

      
      (test-gen-equal (let ((f1 (make-formant .9 1200.0 1.0))) (formant f1 1.0) f1)
		      (let ((f2 (make-formant .9 1200.0 1.0))) (formant f2 1.0) f2)
		      (let ((f3 (make-formant .9 600.0 1.0))) (formant f3 1.0) f3))
      (test-gen-equal (let ((f1 (make-formant .9 1200.0 1.0))) (formant f1 1.0) f1)
		      (let ((f2 (make-formant .9 1200.0 1.0))) (formant f2 1.0) f2)
		      (let ((f3 (make-formant .99 1200.0 1.0))) (formant f3 1.0) f3))
      (test-gen-equal (let ((f1 (make-formant .9 1200.0 1.0))) (formant f1 1.0) f1)
		      (let ((f2 (make-formant .9 1200.0 1.0))) (formant f2 1.0) f2)
		      (let ((f3 (make-formant .9 1200.0 0.5))) (formant f3 1.0) f3))
      
      (let ((frm (make-formant .1 440.0)))
	(mus-set-formant-radius-and-frequency frm 2.0 100.0)
	(if (fneq (mus-formant-radius frm) 2.0) (snd-display ";set-formant-radius-etc: ~A" (mus-formant-radius frm)))
	(if (fneq (mus-frequency frm) 100.0) (snd-display ";set-formant-radius-etc (frq): ~A" (mus-frequency frm))))
      
      (let ((fs (make-vector 1))
	    (f0 (make-formant .1 1000.0))
	    (amps (make-vector 1 1.0))
	    (val 1.0)
	    (v0 (make-vct 10))
	    (v1 (make-vct 10)))
	(vector-set! fs 0 (make-formant .1 1000.0))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (formant f0 val))
	  (vct-set! v1 i (formant-bank amps fs val))
	  (set! val 0.0))
	(if (not (vequal v0 v1)) (snd-display ";formant bank: ~A ~A" v0 v1)))
      
      (let ((fs (make-vector 2))
	    (f0 (make-formant .1 1000.0))
	    (f1 (make-formant .2 100.0))
	    (amps (make-vector 2 1.0))
	    (val 1.0)
	    (v0 (make-vct 10))
	    (v1 (make-vct 10)))
	(vector-set! fs 0 (make-formant .1 1000.0))
	(vector-set! fs 1 (make-formant .2 100.0))
	(vector-set! amps 0 0.5)
	(vector-set! amps 1 0.25)
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (+ (* 0.5 (formant f0 val)) (* 0.25 (formant f1 val))))
	  (vct-set! v1 i (formant-bank amps fs val))
	  (set! val 0.0))
	(if (not (vequal v0 v1)) (snd-display ";formant bank 1: ~A ~A" v0 v1)))
      
      (let ((fs (make-vector 2))
	    (amps (make-vct 2 1.0))
	    (val 1.0)
	    (v (make-vct 5)))
	(vector-set! fs 0 (make-formant .1 1000.0))
	(vector-set! fs 1 (make-formant .2 100.0))
	(vct-set! amps 0 0.5)
	(vct-set! amps 1 0.25)
	(vct-map! v (lambda () (let ((res (formant-bank amps fs val))) (set! val 0.0) res)))
	(if (not (vequal v (vct 0.146 0.029 -0.011 -0.003 -0.000))) (snd-display ";run formant-bank: ~A" v)))
      
      (let ((ob (open-sound "oboe.snd")))
	(define (poltergeist frek amp R gain frek-env R-env)
	  ;; test courtesy of Anders Vinjar
	  (let ((filt (make-formant R frek gain))
		(fe (make-env :envelope frek-env :end (frames) :offset frek))
		(re (make-env :envelope R-env :end (frames) :offset R)))
	    (lambda (y)
	      (let ((outval (formant filt (* amp y))))
		(mus-set-formant-radius-and-frequency filt (env re) (env fe))
		outval))))
	(map-chan (poltergeist 300 0.1 0.0 30.0 '(0 100 1 4000.0) '(0 0.99 1 .9)))  ;; should sound like "whyieee?"
	(play-and-wait 0 ob)
	(close-sound ob))
      
      (let ((gen (make-mixer 2 .5 .25 .125 1.0))
	    (fr0 (make-frame 2 1.0 1.0))
	    (fr1 (make-frame 2 0.0 0.0)))
	(print-and-check gen 
			 "mixer"
			 "mixer: chans: 2, vals: [(0.500 0.250) (0.125 1.000)]"
			 "mixer: chans: 2, vals: [(0.500 0.250) (0.125 1.000)]")
	(print-and-check fr0 
			 "frame"
			 "frame[2]: [1.000 1.000]"
			 "frame[2]: [1.000 1.000]")
	(if (not (frame? fr0)) (snd-display ";~A not a frame?" fr0))
	(if (not (mixer? gen)) (snd-display ";~A not a mixer?" gen))
	(if (equal? fr0 fr1) (snd-display ";frame=? ~A ~A?" fr0 fr1))
	(if (not (= (mus-channels fr0) 2)) (snd-display ";frame channels: ~D?" (mus-channels fr0)))
	(if (not (= (mus-length fr1) 2)) (snd-display ";frame length: ~D?" (mus-length fr0)))
	(if (not (= (mus-channels gen) 2)) (snd-display ";mixer channels: ~D?" (mus-channels gen)))
	(frame->frame gen fr0 fr1)
	(if (or (fneq (frame-ref fr0 0) 1.0)
		(fneq (frame-ref fr1 1) 1.25)
		(fneq (mixer-ref gen 0 0) .5))
	    (snd-display ";fr0: ~A" fr0))
	(frame-set! fr1 0 1.0)
	(set! (frame-ref fr1 1) 1.0)
	(let ((fr3 (frame+ fr0 fr1))
	      (fr4 (frame* fr0 fr1))
	      (fr5 (sample->frame fr1 .5)))
	  (if (or (fneq (frame-ref fr3 0) 2.0)
		  (fneq (frame-ref fr4 0) 1.0))
	      (snd-display ";fr+*: ~A ~A" fr3 fr4))
	  (if (fneq (frame-ref fr5 0) .5) 
	      (snd-display ";sample->frame: ~A?" (frame-ref fr5 0))))
	(let ((fr3 (make-frame 2))
	      (fr4 (make-frame 4)))
	  (frame-set! fr3 0 1.0)
	  (set! (frame-ref fr4 0) 0.5)
	  (frame-set! fr4 2 1.0)
	  (if (not (feql (frame->list (frame+ fr3 fr4)) (list 1.5 0.0)))
	      (snd-display ";frame+ unequal chans: ~A?" (frame+ fr3 fr4))))
	(let ((fr3 (make-frame 2))
	      (fr4 (make-frame 4)))
	  (frame-set! fr3 0 1.0)
	  (frame-set! fr4 0 0.5)
	  (frame-set! fr4 2 1.0)
	  (if (not (feql (frame->list (frame* fr3 fr4)) (list 0.5 0.0)))
	      (snd-display ";frame* unequal chans: ~A?" (frame* fr3 fr4))))
	(let* ((mx1 (make-mixer 2 1.0 0.0 0.0 1.0))
	       (mx2 (mixer* gen mx1))
	       (fr4 (make-frame 2 1.0 1.0))
	       (fr5 (make-frame 2 1.0 1.0))
	       (val (frame->sample mx1 fr1)))
	  (if (fneq val 1.0) (snd-display ";frame->sample: ~A?" val))
	  (if (fneq (frame->sample fr5 fr4) 2.0) (snd-display ";frame->sample ~A" (frame->sample fr5 fr4)))
	  (if (not (equal? (frame->list fr1) (list 1.0 1.0))) (snd-display ";frame->list: ~A?" (frame->list fr1)))
	  (if (or (fneq (mixer-ref mx2 0 1) .25) (fneq (mixer-ref mx2 1 0) .125)) (snd-display ";mixer*: ~A?" mx2))
	  (if (not (equal? mx2 gen)) (snd-display ";mixer=? ~A ~A?" gen mx2))
	  (if (equal? mx2 mx1) (snd-display ";mixer/=? ~A ~A?" mx1 mx2))
	  ;; mus-data doesn't apply from scheme level here
					;(if (not (vct? (mus-data fr4))) (snd-display ";mus-data frame: ~A" (mus-data fr4)))
					;(if (not (vct? (mus-data mx1))) (snd-display ";mus-data mixer: ~A" (mus-data mx1)))
	  (mixer-set! mx2 0 0 2.0)
	  (if (fneq (mixer-ref mx2 0 0) 2.0) (snd-display ";mixer-set: ~A?" mx2))
	  (set! fr0 (sample->frame mx2 1.0))
	  (if (or (fneq (frame-ref fr0 0) 2.0) (fneq (frame-ref fr0 1) .25)) (snd-display ";sample->frame: ~A?" fr0))
	  (let ((frout (make-frame 2)))
	    (sample->frame mx2 1.0 frout)
	    (if (not (equal? frout fr0)) (snd-display ";sample->frame via frout: ~A ~A?" frout fr0)))))
      
      (let ((var (catch #t (lambda () (make-mixer 2 0.0 0.0 0.0 0.0 0.0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-mixer extra args: ~A" var)))
      (let ((var (catch #t (lambda () (let ((fr1 (make-frame 2 1.0 0.0))) (frame->sample (make-oscil) fr1))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";frame->sample bad arg: ~A" var)))
      (let* ((hi (make-mixer 1 1))
	     (tag (catch #t (lambda () (mixer-set! hi 1 1 1.0)) (lambda args (car args)))))
	(if (not (eq? tag 'mus-error)) (snd-display ";mixer-set! 1 1 of 0: ~A (~A)" tag hi)))
      (let* ((hi (make-mixer 1 1))
	     (tag (catch #t (lambda () (set! (mixer-ref hi 1 1) 1.0)) (lambda args (car args)))))
	(if (not (eq? tag 'mus-error)) (snd-display ";set! mixer-ref 1 1 of 0: ~A (~A)" tag hi)))
      (let* ((hi (make-mixer 1))
	     (tag (catch #t (lambda () (mixer-set! hi 1 0 1.0)) (lambda args (car args)))))
	(if (not (eq? tag 'mus-error)) (snd-display ";mixer-set! 1 0 of 0: ~A (~A)" tag hi)))
      (let* ((hi (make-mixer 1))
	     (tag (catch #t (lambda () (mixer-set! hi 0 1 1.0)) (lambda args (car args)))))
	(if (not (eq? tag 'mus-error)) (snd-display ";mixer-set! 0 1 of 0: ~A (~A)" tag hi)))
      (let* ((hi (make-frame 1))
	     (tag (catch #t (lambda () (frame-set! hi 1 1.0)) (lambda args (car args)))))
	(if (not (eq? tag 'mus-error)) (snd-display ";frame-set! 1 of 0: ~A (~A)" tag hi)))
      (let* ((hi (make-frame 1))
	     (tag (catch #t (lambda () (set! (frame-ref hi 1) 1.0)) (lambda args (car args)))))
	(if (not (eq? tag 'mus-error)) (snd-display ";set! frame-ref 1 of 0: ~A (~A)" tag hi)))
      (let* ((tag (catch #t (lambda () (make-frame 0)) (lambda args (car args)))))
	(if (not (eq? tag 'out-of-range)) (snd-display ";make-frame 0: ~A" tag)))
      (let* ((tag (catch #t (lambda () (make-mixer 0)) (lambda args (car args)))))
	(if (not (eq? tag 'out-of-range)) (snd-display ";make-mixer 0: ~A" tag)))
      
      (let ((fr1 (make-frame 1 1))
	    (fr2 (make-frame 2 1 2))
	    (fr4 (make-frame 4 1 2 3 4)) 
	    (fr8 (make-frame 8 1 2 3 4 5 6 7 8))
	    (mx1 (make-mixer 1 5))
	    (mx1id (make-mixer 1 1))
	    (mx2 (make-mixer 2 1 2 3 4))
	    (mx2id (make-mixer 2 1 0 0 1))
	    (mx4 (make-mixer 4))
	    (mx4id (make-mixer 4))
	    (mx8 (make-mixer 8))
	    (mx8id (make-mixer 8)))
	(do ((i 0 (1+ i)))
	    ((= i 4))
	  (mixer-set! mx4id i i 1)
	  (set! (mixer-ref mx4 0 i) 1))
	(do ((i 0 (1+ i)))
	    ((= i 8))
	  (mixer-set! mx8id i i 1)
	  (mixer-set! mx8 i 0 1))
	(if (not (equal? (frame->frame mx1id fr1) (make-frame 1 1))) (snd-display ";frame->frame 1 id: ~A?"            (frame->frame mx1id fr1)))
	(if (not (equal? (frame->frame mx1 fr1) (make-frame 1 5))) (snd-display ";frame->frame 1: ~A?"                 (frame->frame mx1 fr1)))  
	(if (not (equal? (frame->frame mx2id fr1) (make-frame 2 1 0))) (snd-display ";frame->frame 2 1 id: ~A?"        (frame->frame mx2id fr1)))  
	(if (not (equal? (frame->frame mx2 fr1) (make-frame 2 1 2))) (snd-display ";frame->frame 2 1: ~A?"             (frame->frame mx2 fr1)))  
	(if (not (equal? (frame->frame mx4 fr1) (make-frame 4 1 1 1 1))) (snd-display ";frame->frame 4 1: ~A?"         (frame->frame mx4 fr1)))  
	(if (not (equal? (frame->frame mx8 fr1) (make-frame 8 1 0 0 0 0 0 0 0))) (snd-display ";frame->frame 8 1: ~A?" (frame->frame mx8 fr1))) 
	(if (not (equal? (frame->frame mx1 fr2) (make-frame 1 5))) (snd-display ";frame->frame 1 2: ~A?"               (frame->frame mx1 fr2)))   
	(if (not (equal? (frame->frame mx2id fr2) (make-frame 2 1 2))) (snd-display ";frame->frame 2id 2: ~A?"         (frame->frame mx2id fr2)))  
	(if (not (equal? (frame->frame mx2 fr2) (make-frame 2 7 10))) (snd-display ";frame->frame 2 2: ~A?"            (frame->frame mx2 fr2)))  
	(if (not (equal? (frame->frame mx4id fr2) (make-frame 4 1 2 0 0))) (snd-display ";frame->frame 4id 2: ~A?"     (frame->frame mx4id fr2)))  
	(if (not (equal? (frame->frame mx8id fr2) (make-frame 8 1 2 0 0 0 0 0 0))) (snd-display ";frame->frame 8id 2: ~A?" (frame->frame mx8id fr2)))  
	(if (not (equal? (frame->frame mx4 fr2) (make-frame 4 1 1 1 1))) (snd-display ";frame->frame 4 2: ~A?"         (frame->frame mx4 fr2)))  
	(if (not (equal? (frame->frame mx8 fr2) (make-frame 8 3 0 0 0 0 0 0 0))) (snd-display ";frame->frame 8 2: ~A?" (frame->frame mx8 fr2))) 
	(if (not (equal? (frame->frame mx1 fr4) (make-frame 1 5))) (snd-display ";frame->frame 1 4: ~A?"               (frame->frame mx1 fr4))) 
	(if (not (equal? (frame->frame mx1 fr8) (make-frame 1 5))) (snd-display ";frame->frame 1 8: ~A?"               (frame->frame mx1 fr8))) 
	(if (not (equal? (frame->frame mx8id fr2) (make-frame 8 1 2 0 0 0 0 0 0))) (snd-display ";frame->frame 8id 2: ~A?" (frame->frame mx8id fr2)))
	(if (not (equal? (frame->frame mx4id fr2) (make-frame 4 1 2 0 0))) (snd-display ";frame->frame 4id 2: ~A?"     (frame->frame mx4id fr2)))  
	(if (not (equal? (frame->frame mx8 fr4) (make-frame 8 10 0 0 0 0 0 0 0))) (snd-display ";frame->frame 8 4: ~A?" (frame->frame mx8 fr4))) 
	(if (not (equal? (frame->frame mx4 fr4) (make-frame 4 1 1 1 1))) (snd-display ";frame->frame 4 4: ~A?"         (frame->frame mx4 fr4))))
      
      (let ((fr1 (make-frame 2))
	    (fr2 (make-frame 2))
	    (mx1 (make-mixer 2))
	    (mx2 (make-mixer 2)))
	(frame-set! fr1 0 .1)
	(let ((fradd (frame+ fr1 fr1 fr2)))
	  (if (not (equal? fr2 fradd)) (snd-display ";frame+ with res frame: ~A ~A" fr2 fradd))
	  (if (not (equal? fr2 (make-frame 2 0.2 0.0))) (snd-display ";frame+ res: ~A" fr2))
	  (set! fradd (frame* fr1 fr1 fr2))
	  (if (not (equal? fr2 fradd)) (snd-display ";frame* with res frame: ~A ~A" fr2 fradd))
	  (if (or (fneq (frame-ref fr2 0) .01) (fneq (frame-ref fr2 1) 0.0)) (snd-display ";frame* res: ~A" fr2)))
	(set! (mixer-ref mx1 0 0) .1)
	(let ((mxadd (mixer* mx1 mx1 mx2)))
	  (if (not (equal? mx2 mxadd)) (snd-display ";mixer* with res frame: ~A ~A" mx2 mxadd))
	  (if (fneq (mixer-ref mx2 0 0) .01) (snd-display ";mixer* res: ~A" mx2))))
      
      (for-each 
       (lambda (chans)
	 (let ((m1 (make-mixer chans)))
	   (if (or (not (= (mus-channels m1) chans))
		   (not (= (mus-length m1) chans)))
	       (snd-display ";mixer ~A chans but: ~A ~A" chans (mus-channels m1) (mus-length m1)))
	   (do ((i 0 (1+ i)))
	       ((= i chans))
	     (do ((j 0 (1+ j)))
		 ((= j chans))
	       (mixer-set! m1 i j (+ (* i .01) (* j .1)))))
	   (do ((i 0 (1+ i)))
	       ((= i chans))
	     (do ((j 0 (1+ j)))
		 ((= j chans))
	       (if (fneq (mixer-ref m1 i j) (+ (* i .01) (* j .1)))
		   (snd-display ";mixer[~A ~A] = ~A (~A)?" i j (mixer-ref m1 i j) (+ (* i .01) (* j .1))))))
	   (let ((mempty (make-mixer chans))
		 (midentity (make-mixer chans))
		 (mpick (make-mixer chans)))
	     (do ((i 0 (1+ i)))
		 ((= i chans))
	       (mixer-set! midentity i i 1.0))
	     (mixer-set! mpick (1- chans) (1- chans) 1.0)
	     (let ((mzero (mixer* m1 mempty))
		   (msame (mixer* m1 midentity))
		   (mone (mixer* m1 mpick)))
	       (do ((i 0 (1+ i)))
		   ((= i chans))
		 (do ((j 0 (1+ j)))
		     ((= j chans))
		   (if (fneq (mixer-ref mzero i j) 0.0) (snd-display ";mzero ~A ~A = ~A?" i j (mixer-ref mzero i j)))
		   (if (fneq (mixer-ref m1 i j) (mixer-ref msame i j)) (snd-display ";msame ~A ~A?" (mixer-ref msame i j) (mixer-ref m1 i j)))
		   (if (and (fneq (mixer-ref mone i j) 0.0)
			    (not (= i (1- chans)))
			    (not (= j (1- chans))))
		       (snd-display ";mone ~A ~A = ~A?" i j (mixer-ref mone i j)))))))))
       (list 1 2 4 8))

      (let ((fr (make-frame 4)))
	(if (not (= (mus-length fr) 4)) (snd-display ";frame-length: ~A" (mus-length fr)))
	(set! (mus-length fr) 2)
	(if (not (= (mus-length fr) 2)) (snd-display ";set frame-length: ~A" (mus-length fr))))
      (let ((mx (make-mixer 4 4)))
	(let ((tag (catch #t (lambda () (set! (mus-length mx) 2)) (lambda args (car args)))))
	  (if (not (eq? tag 'mus-error)) (snd-display ";set mixer-length: ~A ~A" tag (mus-length mx)))))
      
      (let ((gen (make-fft-window hamming-window 16)))
	(if (not (vequal gen (vct 0.080 0.115 0.215 0.364 0.540 0.716 0.865 1.000 1.000 0.865 0.716 0.540 0.364 0.215 0.115 0.080)))
	    (snd-display ";hamming window: ~A" gen)))
      (let ((gen (make-fft-window rectangular-window 16)))
	(if (not (vequal gen (vct 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000)))
	    (snd-display ";rectangular window: ~A" gen)))
      (let ((gen (make-fft-window hann-window 16)))
	(if (not (vequal gen (vct 0.000 0.038 0.146 0.309 0.500 0.691 0.854 1.000 1.000 0.854 0.691 0.500 0.309 0.146 0.038 0.000)))
	    (snd-display ";hann window: ~A" gen)))
      (let ((gen (make-fft-window welch-window 16)))
	(if (not (vequal gen (vct 0.000 0.234 0.438 0.609 0.750 0.859 0.938 1.000 1.000 0.938 0.859 0.750 0.609 0.438 0.234 0.000)))
	    (snd-display ";welch window: ~A" gen)))
      (let ((gen (make-fft-window parzen-window 16)))
	(if (not (vequal gen (vct 0.000 0.125 0.250 0.375 0.500 0.625 0.750 1.000 1.000 0.750 0.625 0.500 0.375 0.250 0.125 0.000)))
	    (snd-display ";parzen window: ~A" gen)))
      (let ((gen (make-fft-window bartlett-window 16)))
	(if (not (vequal gen (vct 0.000 0.125 0.250 0.375 0.500 0.625 0.750 1.000 1.000 0.750 0.625 0.500 0.375 0.250 0.125 0.000)))
	    (snd-display ";bartlett window: ~A" gen)))
      (let ((gen (make-fft-window blackman2-window 16)))
	(if (not (vequal gen (vct 0.005 0.020 0.071 0.177 0.344 0.558 0.775 1.000 1.000 0.775 0.558 0.344 0.177 0.071 0.020 0.005)))
	    (snd-display ";blackman2 window: ~A" gen)))
      (let ((gen (make-fft-window blackman3-window 16)))
	(if (not (vequal gen (vct 0.000 0.003 0.022 0.083 0.217 0.435 0.696 1.000 1.000 0.696 0.435 0.217 0.083 0.022 0.003 0.000)))
	    (snd-display ";blackman3 window: ~A" gen)))
      (let ((gen (make-fft-window blackman4-window 16)))
	(if (not (vequal gen (vct 0.002 0.002 0.003 0.017 0.084 0.263 0.562 1.000 1.000 0.562 0.263 0.084 0.017 0.003 0.002 0.002)))
	    (snd-display ";blackman4 window: ~A" gen)))
      (let ((gen (make-fft-window exponential-window 16)))
	(if (not (vequal gen (vct 0.000 0.087 0.181 0.283 0.394 0.515 0.646 0.944 0.944 0.646 0.515 0.394 0.283 0.181 0.087 0.000)))
	    (snd-display ";exponential window: ~A" gen)))
      (let ((gen (make-fft-window riemann-window 16)))
	(if (not (vequal gen (vct 0.000 0.139 0.300 0.471 0.637 0.784 0.900 1.000 1.000 0.900 0.784 0.637 0.471 0.300 0.139 0.000)))
	    (snd-display ";riemann window: ~A" gen)))
      (let ((gen (make-fft-window kaiser-window 16 2.5)))
	(if (not (vequal gen (vct 0.304 0.426 0.550 0.670 0.779 0.871 0.941 1.000 1.000 0.941 0.871 0.779 0.670 0.550 0.426 0.304)))
	    (snd-display ";kaiser window: ~A" gen)))
      (let ((gen (make-fft-window cauchy-window 16 2.5)))
	(if (not (vequal gen (vct 0.138 0.173 0.221 0.291 0.390 0.532 0.719 1.000 1.000 0.719 0.532 0.390 0.291 0.221 0.173 0.138)))
	    (snd-display ";cauchy window: ~A" gen)))
      (let ((gen (make-fft-window poisson-window 16 2.5)))
	(if (not (vequal gen (vct 0.082 0.112 0.153 0.210 0.287 0.392 0.535 1.000 1.000 0.535 0.392 0.287 0.210 0.153 0.112 0.082)))
	    (snd-display ";poisson window: ~A" gen)))
      (let ((gen (make-fft-window gaussian-window 16 1.0)))
	(if (not (vequal gen (vct 0.607 0.682 0.755 0.823 0.882 0.932 0.969 1.000 1.000 0.969 0.932 0.882 0.823 0.755 0.682 0.607)))
	    (snd-display ";gaussian window: ~A" gen)))
      (let ((gen (make-fft-window tukey-window 16)))
	(if (not (vequal gen (vct 0.000 0.038 0.146 0.309 0.500 0.691 0.854 1.000 1.000 0.854 0.691 0.500 0.309 0.146 0.038 0.000)))
	    (snd-display ";tukey window: ~A" gen)))
      (without-errors
       (let ((gen (make-fft-window dolph-chebyshev-window 16 1.0)))
	 (if (not (vequal gen (vct 1.000 0.274 0.334 0.393 0.446 0.491 0.525 0.546 0.553 0.546 0.525 0.491 0.446 0.393 0.334 0.274)))
	     (snd-display ";dolph-chebyshev window: ~A" gen))))
      
      (let ((v0 (make-vct 10))
	    (gen (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 10))
	    (v1 (make-vct 10))
	    (gen1 (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 10)))
	(print-and-check gen 
			 "env"
			 "env: linear, pass: 0 (dur: 11), index: 0, scaler: 0.5000, offset: 0.0000, data: [0.000 0.000 1.000 1.000 2.000 0.000]"
			 "seg scaler: 0.5000, offset: 0.0000, rate: 0.100000, current_value: 0.000000, base: 1.000000, offset: 0.000000, scaler: 0.500000, power: 0.000000, init_y: 0.000000, init_power: 0.000000, pass: 0, end: 10, style: 0, index: 0, size: 3, original_data[6]: [0.000 0.000 1.000 1.000 2.000 0.000], rates[3]: [0.100 -0.100 0.000], passes[3]: [5 10 100000000]")
	(if (not (env? gen)) (snd-display ";~A not env?" gen))
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";env scaler ~F?" (mus-scaler gen)))
	(if (fneq (mus-increment gen) 1.0) (snd-display ";env base (1.0): ~A?" (mus-increment gen)))
	(if (not (= (mus-length gen) 10)) (snd-display ";env length: ~A" (mus-length gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (env gen)))
	(let ((off 123.0))
	  (vct-map! v1 (lambda () 
			 (set! off (mus-offset gen1))
			 (if (env? gen1) (env gen1) -1.0)))
	  (if (fneq off 0.0) (snd-display ";mus-offset opt: ~A" off)))
	(if (not (vequal v0 v1)) (snd-display ";map env: ~A ~A" v0 v1))
	(if (or (fneq (vct-ref v0 0) 0.0) (fneq (vct-ref v0 1) .1) (fneq (vct-ref v0 6) .4))
	    (snd-display ";~A output: ~A" gen v0))
	(if (fneq (env-interp 1.6 gen) 0.2) (snd-display ";env-interp ~A at 1.6: ~F?" gen (env-interp 1.5 gen)))
	(set! gen (make-env :envelope '(0 1 1 0) :base 32.0 :end 10))
	(if (fneq (mus-increment gen) 32.0) (snd-display ";env base (32.0): ~A?" (mus-increment gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (env gen)))
	(if (or (fneq (vct-ref v0 0) 1.0) (fneq (vct-ref v0 1) .698) (fneq (vct-ref v0 8) .032))
	    (snd-display ";~A output: ~A" gen v0))
	(set! gen (make-env :envelope '(0 1 1 0) :base .0325 :end 10))
	(if (fneq (mus-increment gen) .0325) (snd-display ";env base (.0325): ~A?" (mus-increment gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (env gen)))
	(if (or (fneq (vct-ref v0 0) 1.0) (fneq (vct-ref v0 1) .986) (fneq (vct-ref v0 8) .513))
	    (snd-display ";~A output: ~A" gen v0))
	(set! gen (make-env :envelope '(0 1 1 .5 2 0) :base 0.0 :end 10 :offset 1.0))
	(if (fneq (mus-offset gen) 1.0) (snd-display ";mus-offset: ~A" (mus-offset gen)))
	(if (fneq (mus-increment gen) 0.0) (snd-display ";env base (0.0): ~A?" (mus-increment gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (if (= i 3)
	      (if (not (= (mus-location gen) 3))
		  (snd-display ";env location: ~A?" (mus-location gen))))
	  (vct-set! v0 i (env gen)))
	(if (or (fneq (vct-ref v0 0) 2.0) (fneq (vct-ref v0 6) 1.5) (fneq (vct-ref v0 8) 1.5))
	    (snd-display ";~A output: ~A" gen v0))
	(if (fneq (env-interp 1.5 gen) 1.5) (snd-display ";env-interp ~A at 1.5: ~F?" gen (env-interp 1.5 gen)))
	(set! (mus-location gen) 6)
	(if (not (= (mus-location gen) 6)) (snd-display ";set! mus-location ~A (6)?" (mus-location gen)))
	(let ((val (env gen)))
	  (if (fneq val 1.5) (snd-display ";set! mus-location 6 -> ~A (1.5)?" val)))
	(set! (mus-location gen) 0)
	(let ((val (env gen)))
	  (if (fneq val 2.0) (snd-display ";set! mus-location 0 -> ~A (2.0)?" val)))
	(let ((gen (make-env '(0 0 1 -1 2 0) :end 10)))
	  (do ((i 0 (1+ i)))
	      ((= i 5))
	    (let ((val (env gen)))
	      (if (fneq val (/ i -5.0)) (snd-display ";neg env: ~D ~A" i val))))
	  (do ((i 0 (1+ i)))
	      ((= i 5))
	    (let ((val (env gen)))
	      (if (fneq val (+ -1.0 (/ i 5.0))) (snd-display ";neg env: ~D ~A" (+ i 5) val)))))
	(let ((gen (make-env '(0 0 1 -1 2 0) :end 10 :base 0.5))
	      (v (vct 0.0 -0.14869 -0.31950 -0.51571 -0.74110 -1.0 -0.74110 -0.51571 -0.31950 -0.14869)))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (let ((val (env gen)))
	      (if (fneq val (vct-ref v i)) (snd-display ";neg exp env: ~D ~A" i val)))))
	
	(let ((e (make-env '(0 0 1 1) :end 9)))
	  (if (fneq (env-interp 1.0 e) 1.0) (snd-display ";env-interp 0011 at 1: ~A" (env-interp 1.0 e)))
	  (if (fneq (env-interp 2.0 e) 1.0) (snd-display ";env-interp 0011 at 2: ~A" (env-interp 2.0 e)))
	  (if (fneq (env-interp 0.0 e) 0.0) (snd-display ";env-interp 0011 at 0: ~A" (env-interp 0.0 e)))
	  (if (fneq (env-interp 0.444 e) 0.444) (snd-display ";env-interp 0011 at .444: ~A" (env-interp 0.45 e)))
	  (restart-env e)
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (let ((val (env e)))
	      (if (fneq val (* i .111111)) (snd-display ";ramp env over 10: ~A at ~A" val i)))))
	(let ((e (make-env '(0 0 .5 .5 1 1) :base 32 :end 9))
	      (v (vct 0.0 0.0243 0.0667 0.1412 0.2716 0.5000 0.5958 0.7090 0.8425 1.0)))
	  (do ((i 0 (1+ i))
	       (x 0.0 (+ x 0.11111)))
	      ((= i 10))
	    (let ((val (env-interp x e)))
	      (if (fneq val (vct-ref v i)) (snd-display ";(0 .5 1) env-interp over 10: ~A at ~A (~A)" val i (vct-ref v i))))))
	(let ((e (make-env '(0 -1.0 1 1) :base 32 :end 9))
	      (v (vct -1.0 -0.9697 -0.9252 -0.8597 -0.7635 -0.6221 -0.4142 -0.1088 0.34017 1.0)))
	  (do ((i 0 (1+ i))
	       (x 0.0 (+ x 0.11111)))
	      ((= i 10))
	    (let ((val (env-interp x e)))
	      (if (fneq val (vct-ref v i)) (snd-display ";(-1 1) env-interp over 10: ~A at ~A (~A)" val i (vct-ref v i))))))
	(let ((e (make-env '(0 -1.0 .5 .5 1 0) :base 32 :end 9))
	      (v (vct -1.0 -0.952 -0.855 -0.661 -0.274 0.5 0.356 0.226 0.107 0.0)))
	  (do ((i 0 (1+ i))
	       (x 0.0 (+ x 0.11111)))
	      ((= i 10))
	    (let ((val (env-interp x e)))
	      (if (fneq val (vct-ref v i)) (snd-display ";(-1 .5 0) env-interp over 10: ~A at ~A (~A)" val i (vct-ref v i))))))
	(let ((e (make-env '(0 0.0 .5 .5 1 -1.0) :base 32 :end 9))
	      (v (vct 0.0 0.085 0.177 0.276 0.384 0.5 -0.397 -0.775 -0.933 -1.0)))
	  (do ((i 0 (1+ i))
	       (x 0.0 (+ x 0.11111)))
	      ((= i 10))
	    (let ((val (env-interp x e)))
	      (if (fneq val (vct-ref v i)) (snd-display ";(0 .5 -1) env-interp over 10: ~A at ~A (~A)" val i (vct-ref v i))))))
	(let ((e (make-env '(0 0 1 1) :end 9 :base 4.0)))
	  (if (fneq (env-interp 1.0 e) 1.0) (snd-display ";env-interp 0011 4 at 1: ~A" (env-interp 1.0 e)))
	  (if (fneq (env-interp 0.0 e) 0.0) (snd-display ";env-interp 0011 4 at 0: ~A" (env-interp 0.0 e)))
	  (if (fneq (env-interp 0.45 e) 0.2839) (snd-display ";env-interp 0011 4 at .45: ~A" (env-interp 0.45 e))))
	(let ((e (make-env '(0 0 1 1) :end 9 :base 0.2)))
	  (if (fneq (env-interp 1.0 e) 1.0) (snd-display ";env-interp 0011 2 at 1: ~A" (env-interp 1.0 e)))
	  (if (fneq (env-interp 0.0 e) 0.0) (snd-display ";env-interp 0011 2 at 0: ~A" (env-interp 0.0 e)))
	  (if (fneq (env-interp 0.45 e) 0.6387) (snd-display ";env-interp 0011 2 at .45: ~A" (env-interp 0.45 e))))
	
	(let ((e1 (make-env '(0 0 1 1) :base 32.0 :end 10))
	      (v (vct 0.000 0.013 0.032 0.059 0.097 0.150 0.226 0.333 0.484 0.698 1.00)))
	  (do ((i 0 (1+ i)))
	      ((> i 10))
	    (let ((val (env e1)))
	      (if (fneq val (vct-ref v i))
		  (snd-display ";exp env direct (32.0): ~A ~A" val (vct-ref v i))))))
	
	(let ((e1 (make-env '(0 1 1 2) :base 32.0 :end 10))
	      (v (vct 1.000 1.013 1.032 1.059 1.097 1.150 1.226 1.333 1.484 1.698 2.00)))
	  (do ((i 0 (1+ i)))
	      ((> i 10))
	    (let ((val (env e1)))
	      (if (fneq val (vct-ref v i))
		  (snd-display ";exp env direct (32.0) offset: ~A ~A" val (vct-ref v i))))))
	(let ((e1 (make-env '(0 1 1 2) :base 32.0 :dur 11))
	      (v (vct 1.000 1.013 1.032 1.059 1.097 1.150 1.226 1.333 1.484 1.698 2.00)))
	  (do ((i 0 (1+ i)))
	      ((> i 10))
	    (let ((val (env e1)))
	      (if (fneq val (vct-ref v i))
		  (snd-display ";exp env direct (32.0) offset (and dur): ~A ~A" val (vct-ref v i))))))
	
	(let ((e1 (make-env '(0 0 1 1) :base 0.032 :end 10))
	      (v (vct 0.000 0.301 0.514 0.665 0.772 0.848 0.902 0.940 0.967 0.986 1.0)))
	  (do ((i 0 (1+ i)))
	      ((> i 10))
	    (let ((val (env e1)))
	      (if (fneq val (vct-ref v i))
		  (snd-display ";exp env direct (.032): ~A ~A" val (vct-ref v i))))))
	
	(let ((e1 (make-env '(0 0 1 1) :base .03125 :end 10))
	      (e2 (make-env '(0 0 1 1 2 0) :base 32.0 :end 10))
	      (e3 (make-env '(0 0 .1 1 2 0) :base 1.1 :end 100)))
	  (do ((i 0 (1+ i)))
	      ((= i 10))
	    (let ((lv1 (env-interp (* i .1) e1))
		  (lv2 (env e1))
		  (lv3 (env-interp (* i .2) e2))
		  (lv4 (env e2)))
	      (if (ffneq lv1 lv2) (snd-display ";env-interp[rmp ~F]: ~A (~A)?" (* .1 i) lv1 lv2))
	      (if (ffneq lv3 lv4) (snd-display ";env-interp[pyr ~F]: ~A (~A)?" (* .2 i) lv3 lv4))))
	  (do ((i 0 (1+ i)))
	      ((= i 100))
	    (let ((lv5 (env-interp (* i .02) e3))
		  (lv6 (env e3)))
	      (if (ffneq lv5 lv6) (snd-display ";env-interp[tri ~F]: ~A (~A)?" (* .02 i) lv5 lv6)))))
	
	(let ((e1 (make-env '(0 0 1 1 2 0) :end 9))
	      (lv1 (make-vct 11))
	      (lv2 (make-vct 11))
	      (lv3 (make-vct 11)))
	  (do ((i 0 (1+ i))) ((= i 11)) (vct-set! lv1 i (env e1)))
	  (do ((i 0 (1+ i))) ((= i 11)) (vct-set! lv2 i (env e1)))
	  (restart-env e1)
	  (do ((i 0 (1+ i))) ((= i 11)) (vct-set! lv3 i (env e1)))
	  (if (not (vequal lv1 lv3)) (snd-display ";restart-env: ~A ~A?" lv1 lv3))
	  (if (not (vequal lv2 (make-vct 11))) (snd-display ";restart-env 1: ~A?" lv2)))
	
	(set! gen (make-env '(0 0 1 1 2 0) :end 10))
	(do ((i 0 (1+ i))) ((= i 4)) (env gen))
	(let ((val (env gen)))
	  (if (fneq val .8) (snd-display ";env(5): ~A?" val))
	  (restart-env gen)
	  (do ((i 0 (1+ i))) ((= i 4)) (env gen))
	  (set! val (env gen))
	  (if (fneq val .8) (snd-display ";restart-env: ~A?" val))
	  (set! (mus-location gen) 6)
	  (let ((val (env gen)))
	    (if (fneq val 0.8) (snd-display ";set! mus-location 6 -> ~A (0.8)?" val)))))
      
      (let ((gen (make-env '(0 0 1 1) :base .032 :end 11)))
	(set! (mus-location gen) 5)
	(let ((val (env gen)))
	  (if (fneq val 0.817)
	      (snd-display ";set env location with base: ~A ~A" val gen))))
      (let ((gen (make-env '(0 0 1 1) :base .032 :dur 12)))
	(set! (mus-location gen) 5)
	(let ((val (env gen)))
	  (if (fneq val 0.817)
	      (snd-display ";set env location with base and dur: ~A ~A" val gen))))
      
      (test-gen-equal (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 9) (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 9) (make-env '(0 0 1 1 2 0) :scaler 0.25 :end 9))
      (test-gen-equal (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 9) (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 9) (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 10))
      (test-gen-equal (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 9) (make-env '(0 0 1 1 2 0) :scaler 0.5 :end 9) (make-env '(0 0 1 1 3 0) :scaler 0.5 :end 9))
      
      (let ((var (catch #t (lambda () (make-env :envelope '())) (lambda args args))))
	(if (not (eq? (car var) 'no-data))
	    (snd-display ";make-env null env: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :end 0)) (lambda args args))))
	(if (not (eq? (car var) 'no-data))
	    (snd-display ";make-env no env: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 0) :end -1)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-env bad end: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 0) :start -1)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-env bad start: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 0) :dur -1)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-env bad dur: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 0) :duration -1.0)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-env bad duration: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 0) :base -1.0)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-env bad base: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(1 1 0 0) :end 10)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-env bad env 1 1 0 0: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 1 -1 0) :end 10)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-env bad env 0 1 -1 0: ~A" var)))
      (let ((var (catch #t (lambda () (make-env :envelope '(0 1 1 0) :end 10 :dur 10)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-env bad end/dur: ~A" var)))
      
      (let ((gen (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1))))
	    (gen1 (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1) (make-vct 512))))
	    (gen2 (partials->wave '(1 1 2 1 3 1 4 1) #f #t))
	    (gen3 (make-table-lookup))
	    (v0 (make-vct 10))
	    (v1 (make-vct 10))
	    (gen4 (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1))))
	    (v2 (make-vct 10)))
	(print-and-check gen 
			 "table-lookup"
			 "table-lookup: freq: 440.000Hz, phase: 0.000, length: 512, interp: linear"
			 (mus-inspect gen))
	;; problem with mus-inspect here is that it includes the table pointer itself
	(if (not (= (mus-length gen) 512)) (snd-display ";table-lookup length: ~A?" (mus-length gen)))
	(if (not (= (mus-length gen3) 512)) (snd-display ";default table-lookup length: ~A?" (mus-length gen3)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (table-lookup gen 0.0))
	  (vct-set! v1 i (mus-apply gen1 0.0)))
	(vct-map! v2 (lambda () (if (table-lookup? gen4) (table-lookup gen4 0.0) -1.0)))
	(if (not (vequal v0 v2)) (snd-display ";map table-lookup: ~A ~A" v0 v2))
	(set! gen4 (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1))))
	(vct-map! v2 (lambda () (table-lookup gen4)))
	(if (not (vequal v0 v2)) (snd-display ";map table-lookup (no fm): ~A ~A" v0 v2))
	(if (not (table-lookup? gen)) (snd-display ";~A not table-lookup?" gen))
	(if (not (vct? (mus-data gen))) (snd-display ";mus-data table-lookup: ~A" (mus-data gen)))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";table-lookup phase: ~F?" (mus-phase gen)))
	(set! (mus-phase gen) 1.0)
	(if (fneq (mus-phase gen) 1.0) (snd-display ";set! table-lookup phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";table-lookup frequency: ~F?" (mus-frequency gen)))
	(set! (mus-frequency gen) 100.0)
	(if (fneq (mus-frequency gen) 100.0) (snd-display ";set! table-lookup frequency: ~F?" (mus-frequency gen)))
	(if (or (fneq (vct-ref v0 1) 0.373) (fneq (vct-ref v0 8) 1.75)) (snd-display ";table-lookup output: ~A" v0))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (if (fneq (vct-ref v0 i) (vct-ref v1 i))
	      (snd-display ";mus-apply table-lookup at ~D: ~A ~A?" i (vct-ref v0 i) (vct-ref v1 i))))
	(set! gen (make-table-lookup 440.0 :wave (phase-partials->wave (list 1 1 0 2 1 (* pi .5)))))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (table-lookup gen 0.0)))
	(if (or (fneq (vct-ref v0 1) 1.094) (fneq (vct-ref v0 8) .421)) (snd-display ";table-lookup phase output: ~A" v0))
	(if (or (fneq (vct-peak (partials->wave '(1 1 2 1))) 1.76035475730896)
		(fneq (vct-peak (partials->wave '(1 1 2 1) #f #t)) 1.0)
		(fneq (vct-peak (partials->wave '(1 1 2 1 3 1 4 1) #f #t)) 1.0))
	    (snd-display ";normalized partials?"))
	(set! (mus-data gen) (phase-partials->wave (list 1 1 0 2 1 (* pi .5)) #f #t)))
      
      (test-gen-equal (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1)))
		      (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1)))
		      (make-table-lookup 100.0 :wave (partials->wave '(1 1 2 1))))
      (test-gen-equal (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1)))
		      (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 1)))
		      (make-table-lookup 440.0 :wave (partials->wave '(1 1 2 .5))))
      
      (let ((hi (make-table-lookup :size 256)))
	(if (not (= (mus-length hi) 256)) (snd-display ";table-lookup set length: ~A?" (mus-length hi))))
      (let ((tag (catch #t (lambda () (make-table-lookup :size 0)) (lambda args (car args)))))
	(if (not (eq? tag 'out-of-range)) (snd-display ";table-lookup size 0: ~A" tag)))
      
      (let ((gen (make-table-lookup 440.0 :wave (partials->wave '(1 1)))))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 440.0) 22050.0))))
	    ((= i 1100))
	  (let* ((val1 (sin a))
		 (val2 (gen 0.0)))
	    (if (fneq val1 val2)
		(snd-display ";table lookup (1 1): ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-table-lookup 4.0 :wave (partials->wave '(1 1)))))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 4.0) 22050.0))))
	    ((= i 1100))
	  (let* ((val1 (sin a))
		 (val2 (gen 0.0)))
	    (if (fneq val1 val2)
		(snd-display ";table lookup (1 1) 4: ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-table-lookup 440.0 :wave (partials->wave '(1 .75 3 .25)))))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 440.0) 22050.0))))
	    ((= i 1100))
	  (let* ((val1 (+ (* .75 (sin a)) (* .25 (sin (* 3 a)))))
		 (val2 (gen 0.0)))
	    (if (fneq val1 val2)
		(snd-display ";table lookup (1 .75 3 .25): ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-table-lookup 0.0 :wave (partials->wave '(1 1))))
	    (gen1 (make-table-lookup 40.0 :wave (partials->wave '(1 1))))
	    (a1 0.0))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 40.0) 22050.0))))
	    ((= i 100))
	  (let* ((fm (sin a))
		 (val1 (sin a1))
		 (val2 (table-lookup gen (table-lookup gen1 0.0))))
	    (set! a1 (+ a1 fm))
	    (if (fneq val1 val2)
		(snd-display ";lookup/lookup fm: ~A: ~A ~A" i val1 val2)))))

      (for-each 
       (lambda (args)
	 (let ((type (car args))
	       (vals (cadr args)))
	   (let* ((tbl1 (make-table-lookup :frequency 0.0 :size 4 :type type)))
	     (vct-set! (mus-data tbl1) 1 1.0)
	     (let ((v (make-vct 10)))
	       (do ((i 0 (1+ i)))
		   ((= i 10))
		 (vct-set! v i (table-lookup tbl1 (/ (* 2 pi .2) 4))))
	       (if (not (vequal v vals))
		   (snd-display ";tbl interp ~A: ~A" type v))
	       (if (not (= (mus-interp-type tbl1) type)) (snd-display ";tbl interp-type (~A): ~A" type (mus-interp-type tbl1)))))))
       (list 
	(list mus-interp-none (vct 0.000 0.000 0.000 0.000 0.000 1.000 1.000 1.000 1.000 1.000))
;	(list mus-interp-none (vct 0.000 0.000 0.000 0.000 0.000 0.000 1.000 1.000 1.000 1.000))
	(list mus-interp-linear (vct 0.000 0.200 0.400 0.600 0.800 1.000 0.800 0.600 0.400 0.200))
	(list mus-interp-lagrange (vct 0.000 0.120 0.280 0.480 0.720 1.000 0.960 0.840 0.640 0.360))
	(list mus-interp-hermite (vct 0.000 0.168 0.424 0.696 0.912 1.000 0.912 0.696 0.424 0.168))))

      (let ((gen0 (make-waveshape 440.0 :wave (partials->waveshape '(1 1))))
	    (gen (make-waveshape 440.0 :size 512 :partials '(1 1)))
	    (v0 (make-vct 10))
	    (gen1 (make-waveshape 440.0 :wave (partials->waveshape '(1 1))))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "waveshape"
			 "waveshape freq: 440.000Hz, phase: 0.000, size: 512"
			 "ws osc freq: 0.125379, phase: 0.000000, offset: 255.500000, table[512 (external)]: [-1.000 -0.996 -0.992 -0.988 -0.984 -0.980 -0.977 -0.973...(0: -1.000, 511: 1.000)]")
	(if (not (= (mus-length gen) 512)) (snd-display ";waveshape length: ~A?" (mus-length gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (let ((val0 (waveshape gen0 1.0 0.0))
		(val (waveshape gen 1.0 0.0)))
	    (if (fneq val val0) (snd-display ";waveshape: ~A /= ~F?" val val0))
	    (vct-set! v0 i val)))
	(vct-map! v1 (lambda () (if (waveshape? gen1) (waveshape gen1 1.0 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map waveshape: ~A ~A" v0 v1))
	(set! gen1 (make-waveshape 440.0 :wave (partials->waveshape '(1 1))))
	(vct-map! v1 (lambda () (waveshape gen1 1.0)))
	(if (not (vequal v0 v1)) (snd-display ";1 map waveshape: ~A ~A" v0 v1))
	(if (not (waveshape? gen)) (snd-display ";~A not waveshape?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";waveshape phase: ~F?" (mus-phase gen)))
	(set! (mus-phase gen) 1.0)
	(if (fneq (mus-phase gen) 1.0) (snd-display ";set! waveshape phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";waveshape frequency: ~F?" (mus-frequency gen)))
	(set! (mus-frequency gen) 100.0)
	(if (fneq (mus-frequency gen) 100.0) (snd-display ";waveshape frequency: ~F?" (mus-frequency gen)))
	(if (not (vct? (mus-data gen))) (snd-display ";mus-data waveshape: ~A" (mus-data gen)))
	(if (or (fneq (vct-ref v0 1) 0.125) (fneq (vct-ref v0 8) .843)) (snd-display ";waveshape output: ~A" v0))
	(set! (mus-data gen0) (make-vct 32))
	(set! (mus-length gen0) 32)
	(if (not (= (mus-length gen0) 32)) (snd-display ";set mus-length waveshape: ~A" (mus-length gen0))))
      
      (test-gen-equal (make-waveshape 440.0 :partials '(1 1)) (make-waveshape 440.0 :partials '(1 1)) (make-waveshape 100.0 :partials '(1 1)))
      (test-gen-equal (make-waveshape 440.0 :partials '(1 1)) (make-waveshape 440.0 :partials '(1 1)) (make-waveshape 4400.0 :partials '(1 1 2 .5)))
      
      (let ((d11 (partials->waveshape '(1 1) 16)))
	(if (not (vequal d11 (vct -1.000 -0.867 -0.733 -0.600 -0.467 -0.333 -0.200 -0.067 0.067 0.200 0.333 0.467 0.600 0.733 0.867 1.000)))
	    (snd-display ";partials->waveshape 1 1: ~A" d11))
	(set! d11 (partials->waveshape '(2 1) 16))
	(if (not (vequal d11 (vct -1.000 -0.502 -0.076 0.280 0.564 0.778 0.920 0.991 0.991 0.920 0.778 0.564 0.280 -0.076 -0.502 -1.000)))
	    (snd-display ";partials->waveshape 2 1: ~A" d11)))

      (let ((gen (make-waveshape 440.0 :partials '(1 1))))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((a (mus-phase gen))
		 (val1 (sin a))
		 (val2 (gen 1.0 0.0)))
	  (if (fneq val1 val2)
	      (snd-display ";waveshaper (1 1) ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-waveshape 440.0 :partials '(2 1))))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((a (mus-phase gen))
		 (val1 (sin (+ (* pi 0.5) (* 2 a))))
		 (val2 (gen 1.0 0.0)))
	  (if (fneq val1 val2)
	      (snd-display ";waveshaper (2 1) ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-waveshape 440.0 :partials '(1 1 2 .5))))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((a (mus-phase gen))
		 (val1 (+ (* .6667 (sin a)) (* .3333 (sin (+ (* pi 0.5) (* 2 a))))))
		 (val2 (gen 1.0 0.0)))
	  (if (fneq val1 val2)
	      (snd-display ";waveshaper (1 1 2 .5) ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-waveshape 440.0 :partials '(1 1))))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let* ((a (mus-phase gen))
		 (val1 (* .5 (sin a)))
		 (val2 (gen 0.5 0.0)))
	  (if (fneq val1 val2)
	      (snd-display ";waveshaper (1 1) .5 index ~A: ~A ~A" i val1 val2)))))

      (let ((var (catch #t (lambda () (make-waveshape 440.0 :partials '(1 1) :size #f)) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-waveshape bad size: ~A" var)))
      (let ((var (catch #t (lambda () (make-waveshape 440.0 :wave 3.14)) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-waveshape bad wave: ~A" var)))
      (let ((var (catch #t (lambda () (make-waveshape 440.0 :size 0)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-waveshape bad size -1: ~A" var)))
      (let ((hi (make-waveshape :size 256)))
	(if (not (= (mus-length hi) 256)) (snd-display ";waveshape set length: ~A?" (mus-length hi))))
      
      (let ((gen (make-waveshape 0.0 :wave (partials->waveshape '(1 1))))
	    (gen1 (make-waveshape 40.0 :wave (partials->waveshape '(1 1))))
	    (a1 0.0))
	(do ((i 0 (1+ i))
	     (a 0.0 (+ a (/ (* 2 pi 40.0) 22050.0))))
	    ((= i 400))
	  (let* ((fm (sin a))
		 (val1 (sin a1))
		 (val2 (waveshape gen 1.0 (waveshape gen1 1.0))))
	    (set! a1 (+ a1 fm))
	    (if (> (abs (- val1 val2)) .002)
		(snd-display ";waveshape fm: ~A: ~A ~A" i val1 val2)))))

      (let ((gen (make-wave-train 440.0 0.0 (make-vct 20)))
	    (v0 (make-vct 10))
	    (gen1 (make-wave-train 440.0 0.0 (make-vct 20)))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "wave-train"
			 "wave-train freq: 440.000Hz, phase: 0.000, size: 20, interp: linear"
			 "wt freq: 440.000000, phase: 0.000000, wave[20 (external)]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], type: linear, b: rblk buf[20 (local)]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], loc: 0, fill_time: 0.000000, empty: true")
	(do ((i 0 (1+ i)))
	    ((= i 20))
	  (vct-set! (mus-data gen) i (* i .5))
	  (vct-set! (mus-data gen1) i (vct-ref (mus-data gen) i)))
	(if (not (= (vct-length (mus-data gen)) 20)) (snd-display ";wave-train data length: ~A?" (vct-length (mus-data gen))))
	(if (not (= (mus-length gen) 20)) (snd-display ";wave-train length: ~A?" (mus-length gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (wave-train gen 0.0)))
	(vct-map! v1 (lambda () (if (wave-train? gen1) (wave-train gen1) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map wave-train: ~A ~A" v0 v1))
	(if (not (wave-train? gen)) (snd-display ";~A not wave-train?" gen))
	(if (fneq (mus-phase gen) 0.0) (snd-display ";wave-train phase: ~F?" (mus-phase gen)))
	(set! (mus-phase gen) 1.0)
	(if (fneq (mus-phase gen) 1.0) (snd-display ";set wave-train phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";wave-train frequency: ~F?" (mus-frequency gen)))
	(set! (mus-frequency gen) 100.0)
	(if (fneq (mus-frequency gen) 100.0) (snd-display ";set wave-train freq: ~A" (mus-frequency gen)))
	(if (or (fneq (vct-ref v0 1) 0.5) (fneq (vct-ref v0 8) 4.0)) (snd-display ";wave-train output: ~A" v0))
	(if (not (vct? (mus-data gen))) (snd-display ";mus-data wave-train: ~A" (mus-data gen)))
	(set! (mus-data gen) (make-vct 3)))
      (set! (mus-data (make-oscil)) (make-vct 3))
      
      (test-gen-equal (make-wave-train 440.0 0.0 (make-vct 20)) (make-wave-train 440.0 0.0 (make-vct 20)) (make-wave-train 100.0 0.0 (make-vct 20)))
      (test-gen-equal (make-wave-train 440.0 0.0 (make-vct 20)) (make-wave-train 440.0 0.0 (make-vct 20)) (make-wave-train 440.0 1.0 (make-vct 20)))
      
      (let ((hi (make-wave-train :size 256)))
	(if (not (= (mus-length hi) 256)) (snd-display ";wave-train set length: ~A?" (mus-length hi)))
	(set! (mus-length hi) 128)
	(if (not (= (mus-length hi) 128)) (snd-display ";set wave-train set length: ~A?" (mus-length hi))))

      (for-each 
       (lambda (args)
	 (let ((type (car args))
	       (vals (cadr args)))
	   (let* ((tbl1 (make-wave-train :frequency 3000.0 :initial-phase (/ (* 2.0 pi .2) 4) :size 4 :type type)))
	     (vct-set! (mus-data tbl1) 1 1.0)
	     (let ((v (make-vct 10)))
	       (do ((i 0 (1+ i)))
		   ((= i 10))
		 (vct-set! v i (wave-train tbl1 0.0))) ;(wave-train tbl1 (/ (* 2 pi .2) 4))))
	       (if (not (vequal v vals))
		   (snd-display ";tbl interp ~A: ~A ~A" type v (mus-inspect tbl1)))
	       (if (not (= (mus-interp-type tbl1) type)) (snd-display ";tbl interp-type (~A): ~A" type (mus-interp-type tbl1)))))))
       (list 
	(list mus-interp-none (vct 0.000 1.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 1.000))
	(list mus-interp-linear (vct 0.200 0.800 0.000 0.000 0.000 0.000 0.000 0.000 0.200 0.800))
	(list mus-interp-lagrange (vct 0.120 0.960 -0.080 0.000 0.000 0.000 0.000 0.000 0.120 0.960))
	(list mus-interp-hermite (vct 0.168 0.912 -0.064 -0.016 0.000 0.000 0.000 0.000 0.168 0.912))))

      (let ((tag (catch #t (lambda () (make-wave-train :size 0)) (lambda args (car args)))))
	(if (not (eq? tag 'out-of-range)) (snd-display ";wave-train size 0: ~A" tag)))
      
      (let ((gen (make-readin "oboe.snd" 0 1490))
	    (v0 (make-vct 10))
	    (gen1 (make-readin "oboe.snd" 0 1490))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "readin"
			 "readin: oboe.snd[chan 0], loc: 1490, dir: 1"
			 "rdin chan: 0, dir: 1, loc: 1490, chans: 1, data_start: 0, data_end: -1, file_end: 50828, file_name: oboe.snd")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (readin gen)))
	(vct-map! v1 (lambda () 
		       (if (readin? gen1) 
			   (if (= (mus-channel gen1) 0) 
			       (readin gen1) 
			       1.0) 
			   (if (string=? (mus-file-name gen1) "oboe.snd")
			       -1.0
			       -1.0))))
	(if (not (vequal v0 v1)) (snd-display ";map readin: ~A ~A" v0 v1))
	(if (not (readin? gen)) (snd-display ";~A not readin?" gen))
	(if (not (mus-input? gen)) (snd-display ";~A not input?" gen))
	(if (not (= (mus-length gen) 50828)) (snd-display ";readin length: ~A?" (mus-length gen)))
	(if (not (= (mus-channel gen) 0)) (snd-display ";readin chan: ~A?" (mus-channel gen)))
	(if (not (string=? (mus-file-name gen) "oboe.snd")) (snd-display ";readin mus-file-name: ~A" (mus-file-name gen)))
	(if (or (fneq (vct-ref v0 1) -0.009) (fneq (vct-ref v0 7) .029)) (snd-display ";readin output: ~A" v0))
	(set! (mus-location gen) 1000)
	(if (not (= (mus-location gen) 1000)) (snd-display ";set! mus-location: ~A?" (mus-location gen)))
	(let ((val (readin gen)))
	  (if (fneq val 0.033) (snd-display ";set! mus-location readin: ~A?" val)))
	(set! (mus-increment gen) -1)
	(if (fneq (mus-increment gen) -1.0) (snd-display ";set increment readin: ~A" (mus-increment gen))))
      (let ((tag (catch #t (lambda () (make-readin "/baddy/hiho" 0 124)) (lambda args args))))
	(if (not (eq? (car tag) 'no-such-file)) (snd-display ";make-readin w/o file: ~A" tag)))
      (let ((tag (catch #t (lambda () (make-readin "oboe.snd" 123 124)) (lambda args args))))
	(if (not (eq? (car tag) 'out-of-range)) (snd-display ";make-readin with bad chan: ~A" tag)))
      
      (test-gen-equal (make-readin "oboe.snd" 0) (make-readin "oboe.snd" 0) (make-readin "oboe.snd" 0 1230))
      (test-gen-equal (make-readin "oboe.snd" 0) (make-readin "oboe.snd" 0) (make-readin "pistol.snd" 0))
      (test-gen-equal (make-readin "2.snd" 1) (make-readin "2.snd" 1) (make-readin "2.snd" 0))
      
      (let ((gen (make-readin "2.snd" 1))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "readin"
			 "readin: 2.snd[chan 1], loc: 0, dir: 1"
			 "rdin chan: 1, dir: 1, loc: 0, chans: 2, data_start: 0, data_end: -1, file_end: 22051, file_name: 2.snd")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (readin gen)))
	(if (not (= (mus-channel gen) 1)) (snd-display ";readin chan 1: ~A?" (mus-channel gen)))
	(if (or (fneq (vct-ref v0 1) 0.010) (fneq (vct-ref v0 7) -.006)) (snd-display ";readin 1 output: ~A" v0))
	(print-and-check gen 
			 "readin"
			 "readin: 2.snd[chan 1], loc: 10, dir: 1"
			 "rdin chan: 1, dir: 1, loc: 10, chans: 2, data_start: 0, data_end: 8191, file_end: 22051, file_name: 2.snd"))
      
      (let ((gen (make-file->sample "oboe.snd"))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "file->sample"
			 "file->sample: oboe.snd"
			 "rdin chan: 0, dir: 0, loc: 0, chans: 1, data_start: 0, data_end: -1, file_end: 50828, file_name: oboe.snd")
	(if (not (mus-input? gen)) (snd-display ";~A not input?" gen))
	(if (not (= (mus-length gen) 50828)) (snd-display ";file->sample length: ~A?" (mus-length gen)))
	(if (not (string=? (mus-file-name gen) "oboe.snd")) (snd-display ";file->sample mus-file-name: ~A" (mus-file-name gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (file->sample gen (+ 1490 i))))
	(if (not (file->sample? gen)) (snd-display ";~A not file->sample?" gen))
	(if (or (fneq (vct-ref v0 1) -0.009) (fneq (vct-ref v0 7) .029)) (snd-display ";file->sample output: ~A" v0))
	(if (fneq (mus-increment gen) 0.0) (snd-display ";file->sample increment: ~A" (mus-increment gen)))
	(set! (mus-increment gen) 1.0)
	(if (fneq (mus-increment gen) 1.0) (snd-display ";file->sample set increment: ~A" (mus-increment gen))))
      
      (let* ((ind (open-sound "oboe.snd"))
	     (gen (make-snd->sample ind))
	     (gen1 (make-snd->sample ind))
	     (v0 (make-vct 10)))
	(print-and-check gen 
			 "snd->sample"
			 "snd->sample: reading oboe.snd (1 chan) at 0:[no readers]"
			 "snd->sample: reading oboe.snd (1 chan) at 0:[no readers]")
	(if (not (equal? gen gen)) (snd-display ";snd->sample not eq? itself?"))
	(if (equal? gen gen1) (snd-display ";snd->sample eq? not itself?"))
	(if (not (mus-input? gen)) (snd-display ";snd->sample ~A not input?" gen))
	(if (not (= (frames ind) (mus-length gen))) (snd-display ";snd->sample len: ~A ~A" (frames ind) (mus-length gen)))
	(if (not (string=? (mus-file-name gen) (string-append cwd "oboe.snd")))
	    (snd-display ";snd->sample mus-file-name: ~A ~A" (mus-file-name gen) (string-append cwd "oboe.snd")))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (snd->sample gen (+ 1490 i))))
	(if (not (snd->sample? gen)) (snd-display ";~A not snd->sample?" gen))
	(if (or (fneq (vct-ref v0 1) -0.009) (fneq (vct-ref v0 7) .029)) (snd-display ";snd->sample output: ~A" v0))
	(if (not (= (mus-channels gen) 1)) (snd-display ";snd->sample channels: ~A" (mus-channels gen)))
	(if (not (= (mus-location gen) 1499)) (snd-display ";snd->sample location: ~A" (mus-location gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (ina (+ 1490 i) gen)))
	(if (or (fneq (vct-ref v0 1) -0.009) (fneq (vct-ref v0 7) .029)) (snd-display ";snd->sample ina output: ~A" v0))
	(close-sound ind))
      
      (let* ((ind (open-sound "2.snd"))
	     (gen (make-snd->sample ind))
	     (v0 (make-vct 10)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (snd->sample gen (+ 1490 i) 0))
	  (vct-set! v0 i (snd->sample gen (+ 1490 i) 1)))
	(print-and-check gen 
			 "snd->sample"
			 "snd->sample: reading 2.snd (2 chans) at 1499:[#<sample-reader: 2.snd[0: 0] from 1490, at 1500>, #<sample-reader: 2.snd[1: 0] from 1490, at 1500>]"
			 "snd->sample: reading 2.snd (2 chans) at 1499:[#<sample-reader: 2.snd[0: 0] from 1490, at 1500>, #<sample-reader: 2.snd[1: 0] from 1490, at 1500>]")
	(if (not (mus-input? gen)) (snd-display ";snd->sample ~A not input?" gen))
	(if (not (string=? (mus-file-name gen) (string-append cwd "2.snd")))
	    (snd-display ";snd->sample mus-file-name: ~A ~A" (mus-file-name gen) (string-append cwd "2.snd")))
	(if (not (snd->sample? gen)) (snd-display ";~A not snd->sample?" gen))
	(if (not (= (mus-channels gen) 2)) (snd-display ";snd->sample channels (2): ~A" (mus-channels gen)))
	(if (not (= (mus-location gen) 1499)) (snd-display ";snd->sample location (2): ~A" (mus-location gen)))
	(close-sound ind))
      
      (let* ((fgen (make-file->sample "oboe.snd"))
	     (gen (make-xen->sample (lambda (samp chan) (* 2.0 (file->sample fgen samp chan)))))
	     (gen1 (make-xen->sample (lambda (s c) 1.0)))
	     (gen2 gen)
	     (v0 (make-vct 10)))
	(print-and-check gen 
			 "xen->sample"
			 "xen->sample: #<procedure #f ((samp chan) (* 2.0 (file->sample fgen samp chan)))>"
			 "xen->sample: #<procedure #f ((samp chan) (* 2.0 (file->sample fgen samp chan)))>")
	(if (not (mus-input? gen)) (snd-display ";xen->sample ~A not input?" gen))
	(if (not (equal? gen gen)) (snd-display ";xen->sample not eq? itself?"))
	(if (equal? gen gen1) (snd-display ";xen->sample eq? ~A ~A" gen gen1))
	(if (not (equal? gen gen2)) (snd-display ";xen->sample not eq? ~A ~A" gen gen2))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (xen->sample gen (+ 1490 i) 0)))
	(if (not (xen->sample? gen)) (snd-display ";~A not xen->sample?" gen))
	(if (or (fneq (vct-ref v0 1) (* 2 -0.009)) (fneq (vct-ref v0 7) (* 2 .029))) (snd-display ";xen->sample output: ~A" v0))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (ina (+ 1490 i) gen)))
	(if (or (fneq (vct-ref v0 1) (* 2 -0.009)) (fneq (vct-ref v0 7) (* 2 .029))) (snd-display ";xen->sample ina output: ~A" v0)))
      
      (let ((gen (make-file->frame "oboe.snd"))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "file->frame"
			 "file->frame: oboe.snd"
			 "rdin chan: 0, dir: 0, loc: 0, chans: 1, data_start: 0, data_end: -1, file_end: 50828, file_name: oboe.snd")
	(if (not (mus-input? gen)) (snd-display ";~A not input?" gen))
	(if (not (= (mus-length gen) 50828)) (snd-display ";file->frame length: ~A?" (mus-length gen)))
	(if (not (string=? (mus-file-name gen) "oboe.snd")) (snd-display ";file->frame mus-file-name: ~A" (mus-file-name gen)))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (frame-ref (file->frame gen (+ 1490 i)) 0)))
	(if (not (file->frame? gen)) (snd-display ";~A not file->frame?" gen))
	(if (or (fneq (vct-ref v0 1) -0.009) (fneq (vct-ref v0 7) .029)) (snd-display ";file->frame output: ~A" v0)))
      
      (if (file-exists? "fmv.snd") (delete-file "fmv.snd"))
      (if (file-exists? "fmv1.snd") (delete-file "fmv1.snd"))
      (if (file-exists? "fmv2.snd") (delete-file "fmv2.snd"))
      (if (file-exists? "fmv3.snd") (delete-file "fmv3.snd"))
      (let ((gen (make-sample->file "fmv.snd" 2 mus-lshort mus-riff)))
	(print-and-check gen 
			 "sample->file"
			 "sample->file: fmv.snd"
			 "rdout chan: 0, loc: 0, file_name: fmv.snd, chans: 2, data_start: 0, data_end: 8191, out_end: 0")
	(if (not (mus-output? gen)) (snd-display ";~A not output?" gen))
	(if (not (sample->file? gen)) (snd-display ";~A not sample->file?" gen))
	(if (not (= (mus-length gen) 8192)) (snd-display ";sample->file length: ~A?" (mus-length gen)))
	(let ((genx gen))
	  (if (not (equal? genx gen)) (snd-display ";sample->file equal? ~A ~A" genx gen)))
	(if (not (string=? (mus-file-name gen) "fmv.snd")) (snd-display ";sample->file mus-file-name: ~A" (mus-file-name gen)))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (sample->file gen i 0 (* i .001))
	  (sample->file gen i 1 (* i .01)))
	(outa 50 .015 gen)
	(outb 50 .15 gen)
	(out-any 60 .015 0 gen)
	(out-any 60 .15 1 gen)
	(mus-close gen))
      (let* ((gen (make-file->sample "fmv.snd"))
	     (val0 (in-any 20 0 gen))
	     (val1 (in-any 20 1 gen))
	     (val2 (ina 30 gen))
	     (val3 (inb 30 gen))
	     (val4 (file->sample gen 40 0))
	     (val5 (file->sample gen 40 1))
	     (val6 (in-any 50 0 gen))
	     (val7 (in-any 50 1 gen))
	     (val8 (in-any 60 0 gen))
	     (val9 (in-any 60 1 gen)))
	(print-and-check gen 
			 "file->sample"
			 "file->sample: fmv.snd"
			 "rdin chan: 0, dir: 0, loc: 0, chans: 2, data_start: 20, data_end: 100, file_end: 100, file_name: fmv.snd")
	(if (not (= (mus-channels gen) 2)) (snd-display ";make-sample->file chans: ~A?" (mus-channels gen)))
	(if (not (mus-input? gen)) (snd-display ";~A not input?" gen))
	(if (or (fneq val0 .02) (fneq val1 .2)) (snd-display ";in-any: ~A ~A?" val0 val1))
	(if (or (fneq val2 .03) (fneq val3 .3)) (snd-display ";inab: ~A ~A?" val2 val3))
	(if (or (fneq val4 .04) (fneq val5 .4)) (snd-display ";sample->file: ~A ~A?" val4 val5))
	(if (or (fneq val6 .065) (fneq val7 .65)) (snd-display ";outab: ~A ~A?" val6 val7))
	(if (or (fneq val8 .075) (fneq val9 .75)) (snd-display ";out-any: ~A ~A?" val8 val9)))
      
      (let ((gen (make-sample->file "fmv.snd" 4 mus-lshort mus-riff)))
	(print-and-check gen 
			 "sample->file"
			 "sample->file: fmv.snd"
			 "rdout chan: 0, loc: 0, file_name: fmv.snd, chans: 4, data_start: 0, data_end: 8191, out_end: 0")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (outa i .1 gen)
	  (outb i .2 gen)
	  (outc i .3 gen)
	  (outd i .4 gen))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (outa i .01 gen)
	  (outb i .02 gen)
	  (outc i .03 gen)
	  (outd i .04 gen))
	(mus-close gen))
      (let* ((gen (make-file->sample "fmv.snd")))
	(print-and-check gen 
			 "file->sample"
			 "file->sample: fmv.snd"
			 "rdin chan: 0, dir: 0, loc: 0, chans: 4, data_start: 0, data_end: -1, file_end: 10, file_name: fmv.snd")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (if (or (fneq (ina i gen) .11)
		  (fneq (inb i gen) .22)
		  (fneq (in-any i 2 gen) .33)
		  (fneq (in-any i 3 gen) .44))
	      (snd-display ";4-chan out/in[~A]: ~A ~A ~A ~A?" i (ina i gen) (inb i gen) (in-any i 2 gen) (in-any i 3 gen)))))
      
      (let ((gen (make-sample->file "fmv.snd" 4 mus-lshort mus-riff)))
	(run
	 (lambda ()
	   (do ((i 0 (1+ i)))
	       ((= i 10))
	     (outa i .1 gen)
	     (outb i .2 gen)
	     (outc i .3 gen)
	     (outd i .4 gen))
	   (do ((i 0 (1+ i)))
	       ((= i 10))
	     (outa i .01 gen)
	     (outb i .02 gen)
	     (outc i .03 gen)
	     (outd i .04 gen))))
	(mus-close gen))
      (let* ((gen (make-file->sample "fmv.snd")))
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (if (or (fneq (ina i gen) .11)
		  (fneq (inb i gen) .22)
		  (fneq (in-any i 2 gen) .33)
		  (fneq (in-any i 3 gen) .44))
	      (snd-display ";4-chan out/in[~A]: ~A ~A ~A ~A?" i (ina i gen) (inb i gen) (in-any i 2 gen) (in-any i 3 gen)))))

      (let ((var (catch #t (lambda () (make-sample->file "fmv.snd" -1 mus-lshort mus-next)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-sample->file bad chans: ~A" var)))
      (let ((var (catch #t (lambda () (mus-location (make-oscil))) (lambda args args))))
	(if (or (not (list? var)) (not (eq? (car var) 'mus-error)))
	    (snd-display ";mus-location oscil: ~A" var)))
      (let ((var (catch #t (lambda () (make-sample->file "fmv.snd" 1 -1 mus-next)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-sample->file bad format: ~A" var)))
      (let ((var (catch #t (lambda () (make-sample->file "fmv.snd" 1 mus-lshort -1)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-sample->file bad type: ~A" var)))
      
      (let ((gen (make-frame->file "fmv1.snd" 2 mus-bshort mus-next)))
	(print-and-check gen 
			 "frame->file"
			 "frame->file: fmv1.snd"
			 "rdout chan: 0, loc: 0, file_name: fmv1.snd, chans: 2, data_start: 0, data_end: 8191, out_end: 0")
	(if (not (mus-output? gen)) (snd-display ";~A not output?" gen))
	(if (not (frame->file? gen)) (snd-display ";~A not frame->file?" gen))
	(if (not (= (mus-length gen) 8192)) (snd-display ";frame->file length: ~A?" (mus-length gen)))
	(if (not (string=? (mus-file-name gen) "fmv1.snd")) (snd-display ";frame->file mus-file-name: ~A" (mus-file-name gen)))
	(set! (mus-length gen) 4096)
	(if (not (= (mus-length gen) 4096)) (snd-display ";frame->file length (1): ~A?" (mus-length gen)))
	(set! (mus-length gen) 8192)
	(let ((fr0 (make-frame 2 0.0 0.0)))
	  (do ((i 0 (1+ i)))
	      ((= i 100))
	    (frame-set! fr0 0 (* i .001))
	    (frame-set! fr0 1 (* i .01))
	    (frame->file gen i fr0)))
	(mus-close gen))
      (let* ((gen (make-file->frame "fmv1.snd"))
	     (val4 (file->frame gen 40))
	     (frout (make-frame 2)))
	(if (or (fneq (frame-ref val4 0) .04) (fneq (frame-ref val4 1) .4))
	    (snd-display ";frame->file output: ~A?" val4))
	(file->frame gen 40 frout)
	(if (not (equal? frout val4))
	    (snd-display ";frame->file output via frame: ~A ~A?" frout val4)))
      
      (let ((gen (make-sample->file "fmv2.snd" 4 mus-bshort mus-aifc)))
	(print-and-check gen 
			 "sample->file"
			 "sample->file: fmv2.snd"
			 "rdout chan: 0, loc: 0, file_name: fmv2.snd, chans: 4, data_start: 0, data_end: 8191, out_end: 0")
	(if (not (mus-output? gen)) (snd-display ";~A not output?" gen))
	(if (not (sample->file? gen)) (snd-display ";~A not sample->file?" gen))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (sample->file gen i 0 (* i .001))
	  (sample->file gen i 1 (* i .01))
	  (sample->file gen i 2 (* i .002))
	  (sample->file gen i 3 (* i .003)))
	(outa 50 .015 gen)
	(outb 50 .15 gen)
	(outc 50 .02 gen)
	(outd 50 .03 gen)
	(out-any 60 .015 0 gen)
	(out-any 60 .15 1 gen)
	(out-any 60 .02 2 gen)
	(out-any 60 .03 3 gen)
	(mus-close gen))
      (let* ((gen (make-file->sample "fmv2.snd"))
	     (val0 (in-any 20 2 gen))
	     (val1 (in-any 20 3 gen))
	     (val2 (file->sample gen 50 2))
	     (val3 (file->sample gen 50 3))
	     (val4 (file->sample gen 60 2))
	     (val5 (file->sample gen 60 3)))
	(if (not (= (mus-channels gen) 4)) (snd-display ";make-file->sample (4) chans: ~A?" (mus-channels gen)))
	(if (not (= (mus-increment gen) 0.0)) (snd-display ";file->sample increment: ~A" (mus-increment gen))) ; dir never set in this case
	(if (or (fneq val0 .04) (fneq val1 .06)) (snd-display ";in-any(0, 4): ~A ~A?" val0 val1))
	(if (or (fneq val2 .12) (fneq val3 .18)) (snd-display ";file->sample(4): ~A ~A?" val2 val3))
	(if (or (fneq val4 .14) (fneq val5 .21)) (snd-display ";in-any(4, 4): ~A ~A?" val4 val5)))
      
      (let ((v0 (make-vct 1000))
	    (os (make-oscil 440.0)))
	(do ((i 0 (1+ i)))
	    ((= i 1000))
	  (vct-set! v0 i (* .1 (oscil os))))
	(array->file "fmv3.snd" v0 10000 22050 1) ; 10000 deliberate
	(let ((v1 (make-vct 1000)))
	  (file->array "fmv3.snd" 0 0 1000 v1)
	  (do ((i 0 (1+ i)))
	      ((= i 1000))
	    (if (fneq (vct-ref v0 i) (vct-ref v1 i)) 
		(snd-display ";array->file->array: ~A ~A ~A?" i (vct-ref v0 i) (vct-ref v1 i)))))
	
	(let ((var (catch #t (lambda () (array->file "fmv3.snd" v0 -1 1000 1)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";array->file bad samps: ~A" var)))
	(let ((var (catch #t (lambda () (array->file "/bad/baddy/fmv3.snd" v0 1 1000 1)) (lambda args args))))
	  (if (not (eq? (car var) 'mus-error))
	      (snd-display ";array->file bad file: ~A" var)))
	(let ((var (catch #t (lambda () (file->array "fmv3.snd" -1 0 -1 v0)) (lambda args args))))
	  (if (not (eq? (car var) 'out-of-range))
	      (snd-display ";file->array bad samps: ~A" var))))
      
      (let ((gen (make-rand 10000.0))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "rand"
			 "rand freq: 10000.000Hz, phase: 0.000, amp: 1.000"
			 "noi freq: 2.849517, phase: 0.000000, base: 1.000000, output: 0.000000, incr: 0.000000")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (rand gen 0.0)))
	(if (not (rand? gen)) (snd-display ";~A not rand?" gen))
	(if (fneq (mus-phase gen) 3.3624296) (snd-display ";rand phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 10000.0) (snd-display ";rand frequency: ~F?" (mus-frequency gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";set! mus-scaler rand: ~A" (mus-scaler gen)))
	(if (= (vct-ref v0 1) (vct-ref v0 8)) (snd-display ";rand output: ~A" v0)))
      
      (let ((gen (make-rand 10000.0 :envelope '(0 0 1 1)))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "rand"
			 "rand freq: 10000.000Hz, phase: 0.000, amp: 1.000, with distribution envelope"
			 "noi freq: 2.849517, phase: 0.000000, base: 1.000000, output: 0.000000, incr: 0.000000, envelope: [0.020 0.054 0.072 0.086 0.098 0.108 0.118 0.126...(0: 0.020, 511: 1.000)]")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (rand gen 0.0)))
	(if (not (rand? gen)) (snd-display ";(dist) ~A not rand?" gen))
	(if (fneq (mus-frequency gen) 10000.0) (snd-display ";(dist) rand frequency: ~F?" (mus-frequency gen)))
	(if (= (vct-ref v0 1) (vct-ref v0 8)) (snd-display ";(dist) rand output: ~A" v0))
	(if (or (not (vct? (mus-data gen)))
		(not (= (mus-length gen) (vct-length (mus-data gen))))
		(not (= (mus-length gen) 512)))
	    (snd-display ";(dist) rand data: ~A ~A" (mus-length gen) (mus-data gen))))
      
      (let ((gen1 (make-rand 10000.0 :envelope '(0 0 1 1)))
	    (gen2 (make-rand 10000.0 :envelope '(0 1 1 0)))
	    (up1 0)
	    (down1 0)
	    (bad1 0)
	    (up2 0)
	    (down2 0)
	    (bad2 0))
	(do ((i 0 (1+ i)))
	    ((= i 1000))
	  (let ((val1 (rand gen1))
		(val2 (rand gen2)))
	    (if (>= val1 .5)
		(set! up1 (1+ up1))
		(if (>= val1 0.0)
		    (set! down1 (1+ down1))
		    (set! bad1 (1+ bad1))))
	    (if (>= val2 .5)
		(set! up2 (1+ up2))
		(if (>= val2 0.0)
		    (set! down2 (1+ down2))
		    (set! bad2 (1+ bad2))))))
	(if (or (not (= bad1 0))
		(not (= bad2 0))
		(> (* 2.5 down1) up1)
		(> (* 2.5 up2) down2))
	    (snd-display "; rand dist: ~A ~A ~A, ~A ~A ~A" down1 up1 bad1 down2 up2 bad2)))

      (test-gen-equal (make-rand 1000) (make-rand 1000) (make-rand 500))
      (test-gen-equal (make-rand 1000) (make-rand 1000) (make-rand 1000 0.5))
      
      (let ((gen (make-rand-interp 4000.0))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "rand-interp"
			 (mus-describe gen)
			 (mus-inspect gen)) ; problem here is the random incr field
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (rand-interp gen 0.0)))
	(if (not (rand-interp? gen)) (snd-display ";~A not rand-interp?" gen))
	(if (fneq (mus-phase gen) 5.114882) (snd-display ";rand-interp phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 4000.0) (snd-display ";rand-interp frequency: ~F?" (mus-frequency gen)))
	(set! (mus-scaler gen) 0.5)
	(if (fneq (mus-scaler gen) 0.5) (snd-display ";set! mus-scaler rand-interp: ~A" (mus-scaler gen)))
	(if (= (vct-ref v0 1) (vct-ref v0 8)) (snd-display ";rand-interp output: ~A" v0)))
      
      (let ((gen (make-rand-interp 4000.0 :envelope '(-1 1 0 0 1 1)))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "rand-interp"
			 (mus-describe gen)
			 (mus-inspect gen)) ; problem here is the random incr field
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (rand-interp gen 0.0)))
	(if (not (rand-interp? gen)) (snd-display ";(dist) ~A not rand-interp?" gen))
	(if (= (vct-ref v0 1) (vct-ref v0 8)) (snd-display ";(dist) rand-interp output: ~A" v0))
	(if (or (not (vct? (mus-data gen)))
		(not (= (mus-length gen) (vct-length (mus-data gen))))
		(not (= (mus-length gen) 512)))
	    (snd-display ";(dist) rand-interp data: ~A ~A" (mus-length gen) (mus-data gen))))
      
      (let ((gen (make-rand 10000.0 1.0))
	    (gen1 (make-rand-interp 10000.0 1.0)))
	(do ((i 0 (1+ i)))
	    ((= i 1000))
	  (let* ((val1 (gen 0.0))
		 (val2 (gen1 0.0)))
	    (if (or (> val1 1.0)
		    (< val1 -1.0))
		(snd-display ";rand: ~A ~A" val1 gen))
	    (if (or (> val2 1.0)
		    (< val2 -1.0))
		(snd-display ";rand-interp: ~A ~A" val2 gen1)))))

      (let ((gen (make-rand 10000.0 :distribution (inverse-integrate '(0 0 1 1))))
	    (v0 (make-vct 10)))
	(print-and-check gen 
			 "rand"
			 "rand freq: 10000.000Hz, phase: 0.000, amp: 1.000, with distribution envelope"
			 "noi freq: 2.849517, phase: 0.000000, base: 1.000000, output: 0.000000, incr: 0.000000, envelope: [0.000 0.054 0.072 0.086 0.098 0.108 0.118 0.126...(0: 0.000, 511: 1.000)]")

	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (rand gen 0.0)))
	(if (not (rand? gen)) (snd-display ";(dist 2) ~A not rand?" gen))
	(if (fneq (mus-frequency gen) 10000.0) (snd-display ";(dist 2) rand frequency: ~F?" (mus-frequency gen)))
	(if (= (vct-ref v0 1) (vct-ref v0 8)) (snd-display ";(dist 2) rand output: ~A" v0))
	(if (or (not (vct? (mus-data gen)))
		(not (= (mus-length gen) (vct-length (mus-data gen))))
		(not (= (mus-length gen) 512)))
	    (snd-display ";(dist 2) rand data: ~A ~A" (mus-length gen) (mus-data gen))))
      
      (let ((gen1 (make-rand 10000.0 :distribution (inverse-integrate '(0 0 1 1))))
	    (gen2 (make-rand 10000.0 :distribution (inverse-integrate '(0 1 1 0))))
	    (up1 0)
	    (down1 0)
	    (bad1 0)
	    (up2 0)
	    (down2 0)
	    (bad2 0))
	(do ((i 0 (1+ i)))
	    ((= i 1000))
	  (let ((val1 (rand gen1))
		(val2 (rand gen2)))
	    (if (>= val1 .5)
		(set! up1 (1+ up1))
		(if (>= val1 0.0)
		    (set! down1 (1+ down1))
		    (set! bad1 (1+ bad1))))
	    (if (>= val2 .5)
		(set! up2 (1+ up2))
		(if (>= val2 0.0)
		    (set! down2 (1+ down2))
		    (set! bad2 (1+ bad2))))))
	(if (or (not (= bad1 0))
		(not (= bad2 0))
		(> (* 2.5 down1) up1)
		(> (* 2.5 up2) down2))
	    (snd-display "; rand dist 2: ~A ~A ~A, ~A ~A ~A" down1 up1 bad1 down2 up2 bad2)))

      (let ((v1 (inverse-integrate '(-1 1 1 1))))
	(if (fneq (vct-ref v1 4) -0.984)
	    (snd-display ";inverse-integrate -1 to 1 uniform: ~A" v1)))
      (let ((v1 (inverse-integrate '(0 1 1 1))))
	(if (fneq (vct-ref v1 4) .008)
	    (snd-display ";inverse-integrate 0 to 1 uniform: ~A" v1)))
      (let ((v1 (inverse-integrate '(0 1 1 0))))
	(if (fneq (vct-ref v1 4) .004)
	    (snd-display ";inverse-integrate 0 to 1 1 to 0: ~A" v1)))
      (let ((v1 (inverse-integrate '(0 0 .5 1 1 0))))
	(if (fneq (vct-ref v1 4) .073)
	    (snd-display ";inverse-integrate triangle: ~A" v1)))
      (let ((v1 (inverse-integrate (gaussian-envelope 1.0))))
	(if (fneq (vct-ref v1 4) -0.593)
	    (snd-display ";inverse-integrate gaussian: ~A" v1)))

      (let ((minp 1.0)
	    (maxp -1.0))
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let ((val1 (mus-random 1.0)))
	    (if (< val1 minp) (set! minp val1))
	    (if (> val1 maxp) (set! maxp val1))
	    (if (or (> val1 1.0)
		    (< val1 -1.0))
		(snd-display ";mus-random: ~A" val1))))
	(if (or (< maxp .9)
		(> minp -.9))
	    (snd-display ";mus-random: ~A ~A" minp maxp))
	(set! minp 12.0)
	(set! maxp -12.0)
	(do ((i 0 (1+ i)))
	    ((= i 1100))
	  (let ((val1 (mus-random 12.0)))
	    (if (< val1 minp) (set! minp val1))
	    (if (> val1 maxp) (set! maxp val1))
	    (if (or (> val1 12.0)
		    (< val1 -12.0))
		(snd-display ";mus-random (12): ~A" val1))))
	(if (or (< maxp 11.0)
		(> minp -11.0))
	    (snd-display ";mus-random (12): ~A ~A" minp maxp)))

      (let ((v (lambda (n) ; chi^2 or mus-random
		 (let ((hits (make-vector 10 0)))
		   (do ((i 0 (1+ i )))
		       ((= i n))
		     (let ((y (inexact->exact (floor (+ 5 (mus-random 5.0))))))
		       (vector-set! hits y (1+ (vector-ref hits y)))))
		   (let ((sum 0.0)
			 (p (/ n 10.0)))
		     (do ((i 0 (1+ i)))
			 ((= i 10) sum)
		       (let ((num (- (vector-ref hits i) p)))
			 (set! sum (+ sum (/ (* num num) p))))))))))
	
	;;:(v 10000)
	;;#(999 1017 1002 1024 1048 971 963 1000 980 996) 5.8
	;; if less than 4 complain
	
	(let ((vr (v 10000)))
	  (if (< vr 4.0)
	      (snd-display ";mus-random not so random? ~A (chi)" vr))))
      
      (let ((v1 (lambda (n)
		  (let ((hits (make-vector 10 0))
			(gen (make-rand 22050.0)))
		    (do ((i 0 (1+ i )))
			((= i n))
		      (let ((y (inexact->exact (floor (+ 5 (* 5 (rand gen 0.0)))))))
			(vector-set! hits y (1+ (vector-ref hits y)))))
		    (let ((sum 0.0)
			  (p (/ n 10.0)))
		      (do ((i 0 (1+ i)))
			  ((= i 10) sum)
			(let ((num (- (vector-ref hits i) p)))
			  (set! sum (+ sum (/ (* num num) p))))))))))
	
	;;:(v1 10000)
	;;#(979 1015 977 1008 954 1049 997 1020 1015 986) 6.606
	
	(let ((vr (v1 10000)))
	  (if (< vr 4.0)
	      (snd-display ";rand not so random? ~A (chi)" vr))))
      
      (let ((v2 (lambda (n) ; Kolmogorov-Smirnov
		  (let ((vals (make-vector n 0.0))
			(sn (sqrt n)))
		    (do ((i 1 (1+ i)))
			((= i n))
		      (vector-set! vals i (+ 0.5 (mus-random 0.5))))
		    (set! vals (sort vals <))
		    (let ((K+ 0.0)
			  (K- 0.0)
			  (incr (/ 1.0 n))
			  (y 0.0))
		      (do ((i 1 (1+ i))
			   (x incr (+ x incr)))
			  ((= i n))
			(let ((Kp (- x (vector-ref vals i)))
			      (Km (- (vector-ref vals i) y)))
			  (if (> Kp K+) (set! K+ Kp))
			  (if (> Km K-) (set! K- Km))
			  (set! y x)))
		      (list (* sn K+) (* sn K-)
			    (- .07089 (/ 0.15 sn)) 
			    (- .1601 (/ .014 sn))
			    (- .3793 (/ 0.15 sn))
			    (- .5887 (/ 0.15 sn))))))))
	
	;;:(v2 1000)
	;;(0.419489806081307 0.536508579184211 0.0661465835097474 0.159657281127576 0.374556583509747 0.583956583509747)
	;; if < .2 complain?
	
	(let* ((vr (v2 1000))
	       (kp (car vr))
	       (km (cadr vr))
	       (k (list-ref vr 3)))
	  (if (or (< kp k)
		  (< km k))
	      (snd-display ";mus-random not random? ~A (KS)" vr))))
      
      (let ((data (make-vct 65536)))
	(do ((i 0 (1+ i)))
	    ((= i 65536))
	  (vct-set! data i (mus-random 1.0)))
	(let* ((ndat (snd-spectrum data rectangular-window 65536 #t 0.0 #f #f))
	       (peak (vct-peak ndat))
	       (sum 0.0))
	  (if (> peak 1000.0)
	      (snd-display ";mus-random spectral peak: ~A" peak))
	  (do ((i 0 (1+ i)))
	      ((= i 32768))
	    (set! sum (+ sum (vct-ref ndat i))))
	  (if (> (/ sum 32768.0) 200.0)
	      (snd-display ";random average: ~A ~A" (/ sum 32768.0) (vct-ref ndat 0)))
	  (do ((i 0 (1+ i)))
	      ((= i 65536))
	    (vct-set! data i (mus-random 1.0)))
	  (autocorrelate data)
	  (vct-set! data 0 0.0)
	  (let ((pk (vct-peak data)))
	    (if (> pk 1000)
		(snd-display ";random autocorrelate peak: ~A" (vct-peak data)))
	    (set! sum 0.0)
	    (do ((i 0 (1+ i)))
		((= i 32768))
	      (set! sum (+ sum (abs (vct-ref data i)))))
	    (if (> (/ sum 32768.0) 200.0)
		(snd-display ";random autocorrelate average: ~A" (/ sum 32768.0))))))
      
      (set! (locsig-type) mus-interp-linear)
      (let* ((gen (make-locsig 30.0 :channels 2))
	     (gen1 (make-locsig 60.0 :channels 2))
	     (gen2 (make-locsig 60.0 :channels 4))
	     (gen200 (make-locsig 200.0 :channels 4))
	     (gen3 gen1)
	     (fr0 (locsig gen 0 1.0)))
	(print-and-check gen 
			 "locsig"
			 "locsig: chans 2, outn: [0.667 0.333], interp: linear"
			 "locs outn[2]: [0.667 0.333], revn[0]: nil, interp: linear")
	(if (not (locsig? gen)) (snd-display ";~A not locsig?" gen))
	(if (not (eq? gen1 gen3)) (snd-display ";locsig eq? ~A ~A" gen1 gen3))
	(if (not (equal? gen1 gen3)) (snd-display ";locsig equal? ~A ~A" gen1 gen3))
	(if (eq? gen1 gen2) (snd-display ";locsig 1 eq? ~A ~A" gen1 gen2))
	(if (equal? gen gen1) (snd-display ";locsig 2 equal? ~A ~A" gen gen1))
	(if (equal? gen gen2) (snd-display ";locsig 3 equal? ~A ~A" gen gen2))
	(if (or (fneq (frame-ref fr0 0) .667) (fneq (frame-ref fr0 1) .333)) (snd-display ";locsig output: ~A" fr0))
	(if (or (fneq (locsig-ref gen 0) .667) (fneq (locsig-ref gen 1) .333))
	    (snd-display ";locsig ref: ~F ~F?" (locsig-ref gen 0) (locsig-ref gen 1)))
	(if (not (vequal (mus-data gen) (vct 0.667 0.333)))
	    (snd-display ";locsig gen outn: ~A" (mus-data gen)))
	(if (not (vequal (mus-data gen1) (vct 0.333 0.667)))
	    (snd-display ";locsig gen2 outn: ~A" (mus-data gen1)))
	(if (not (vequal (mus-data gen2) (vct 0.333 0.667 0.000 0.000)))
	    (snd-display ";locsig gen2 outn: ~A" (mus-data gen2)))
	(if (not (vequal (mus-data gen200) (vct 0.000 0.000 0.778 0.222)))
	    (snd-display ";locsig gen200 outn: ~A" (mus-data gen200)))
	(set! (locsig-ref gen 0) .25)
	(if (not (vequal (mus-data gen) (vct 0.250 0.333)))
	    (snd-display ";locsig gen .25 outn: ~A" (mus-data gen)))
	(set! fr0 (locsig gen 0 1.0))
	(if (fneq (frame-ref fr0 0) .25) (snd-display ";set locsig-ref: ~F?" (frame-ref fr0 0)))
	(locsig-set! gen 0 .5)
	(if (not (vequal (mus-data gen) (vct 0.500 0.333)))
	    (snd-display ";locsig gen .5 outn: ~A" (mus-data gen)))
	(set! fr0 (locsig gen 0 1.0))
	(if (fneq (frame-ref fr0 0) .5) (snd-display ";locsig-set: ~F?" (frame-ref fr0 0)))
	(set! gen (make-locsig 120.0 2.0 .1 :channels 4))
	(if (not (vequal (mus-data gen) (vct 0.000 0.333 0.167 0.000)))
	    (snd-display ";locsig gen 120 outn: ~A" (mus-data gen)))
	(set! fr0 (locsig gen 0 1.0))
	(if (or (fneq (frame-ref fr0 1) .333) (fneq (frame-ref fr0 2) .167)) (snd-display ";locsig quad output: ~A" fr0))
	(set! gen (make-locsig 300.0 2.0 .1 :channels 4))
	(if (not (vequal (mus-data gen) (vct 0.167 0.000 0.000 0.333)))
	    (snd-display ";locsig gen 300 outn: ~A" (mus-data gen)))
	(set! fr0 (locsig gen 0 1.0))
	(if (or (fneq (frame-ref fr0 3) .333) (fneq (frame-ref fr0 0) .167)) (snd-display ";300 locsig quad output: ~A" fr0))
	(move-locsig gen1 90.0 1.0)
	(if (not (vequal (mus-data gen1) (vct 0.000 1.000)))
	    (snd-display ";locsig gen1 90 outn: ~A" (mus-data gen)))
	(if (or (fneq (locsig-ref gen1 0) 0.0) (fneq (locsig-ref gen1 1) 1.0)) (snd-display ";move-locsig 90 1: ~A" gen1))
	(move-locsig gen1 0.0 1.0)
	(if (not (vequal (mus-data gen1) (vct 1.000 0.000)))
	    (snd-display ";locsig gen1 0 outn: ~A" (mus-data gen)))
	(if (or (fneq (locsig-ref gen1 0) 1.0) (fneq (locsig-ref gen1 1) 0.0)) (snd-display ";move-locsig 0 1: ~A" gen1))
	(move-locsig gen1 45.0 1.0)
	(if (not (vequal (mus-data gen1) (vct 0.500 0.500)))
	    (snd-display ";locsig gen1 45 outn: ~A" (mus-data gen)))
	(if (or (fneq (locsig-ref gen1 0) 0.5) (fneq (locsig-ref gen1 1) 0.5)) (snd-display ";move-locsig 45 1: ~A" gen1))
	(move-locsig gen1 135.0 2.0)
	(if (not (vequal (mus-data gen1) (vct 0.000 0.500)))
	    (snd-display ";locsig gen1 135 outn: ~A" (mus-data gen)))
	(move-locsig gen1 -270.0 3.0)
	(if (not (vequal (mus-data gen1) (vct 0.000 0.333)))
	    (snd-display ";locsig gen1 -270 outn: ~A" (mus-data gen))))
      
      (for-each 
       (lambda (chans)
	 (let ((m1 (make-locsig :channels chans)))
	   (if (or (not (= (mus-channels m1) chans))
		   (not (= (mus-length m1) chans)))
	       (snd-display ";locsig ~A chans but: ~A ~A" chans (mus-channels m1) (mus-length m1)))
	   (do ((i 0 (1+ i)))
	       ((= i chans))
	     (locsig-set! m1 i (* i .1)))
	   (do ((i 0 (1+ i)))
	       ((= i chans))
	     (if (fneq (locsig-ref m1 i) (* i .1))
		 (snd-display ";locsig[~A] = ~A (~A)?" i (locsig-ref m1 i) (* i .1))))))
       (list 1 2 4 8))
      
      (let ((var (catch #t (lambda () (make-locsig :channels 0)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";make-locsig bad (0) chans: ~A" var)))
      (let ((var (catch #t (lambda () (make-locsig :channels -2)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-locsig bad (-2) chans: ~A" var)))
      (let ((var (catch #t (lambda () (make-locsig :output 1)) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-locsig bad output: ~A" var)))
      (let ((var (catch #t (lambda () (locsig-ref (make-locsig) 1)) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";locsig-ref bad chan: ~A" var)))
      (let ((var (catch #t (lambda () (make-locsig :revout 1)) (lambda args args))))
	(if (not (eq? (car var) 'wrong-type-arg))
	    (snd-display ";make-locsig bad revout: ~A" var)))
      (let ((var (catch #t (lambda () (let ((locs (make-locsig 200 :channels 2))) (locsig-ref locs -1))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";locsig-ref bad chan: ~A" var)))
      (let ((var (catch #t (lambda () (let ((locs (make-locsig))) (locsig-set! locs 2 .1))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";locsig-set! bad chan (2): ~A" var)))
      (let ((var (catch #t (lambda () (let ((locs (make-locsig :reverb .1))) (locsig-reverb-ref locs 2))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";locsig-reverb-ref bad reverb chan (2): ~A" var)))
      (let ((var (catch #t (lambda () (let ((locs (make-locsig :reverb .1))) (locsig-reverb-set! locs 2 .1))) (lambda args args))))
	(if (not (eq? (car var) 'mus-error))
	    (snd-display ";locsig-reverb-set! bad reverb chan (2): ~A" var)))
      
      (if (file-exists? "fmv4.snd") (delete-file "fmv4.snd"))
      (if (file-exists? "fmv4.reverb") (delete-file "fmv4.reverb"))
      (let* ((gen (make-frame->file "fmv4.snd" 2 mus-bshort mus-next))
	     (rev (make-frame->file "fmv4.reverb" 1 mus-bshort mus-next))
	     (lc (make-locsig 60.0 :reverb .1 :channels 2 :output gen :revout rev)))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (locsig lc i 1.0))
	(if (fneq (locsig-reverb-ref lc 0) .1) (snd-display ";locsig reverb ref: ~A?" (locsig-reverb-ref lc 0)))
	(set! (locsig-reverb-ref lc 0) .3)
	(if (fneq (locsig-reverb-ref lc 0) .3) (snd-display ";set locsig reverb ref: ~A?" (locsig-reverb-ref lc 0)))
	(locsig-reverb-set! lc 0 .2)
	(if (fneq (locsig-reverb-ref lc 0) .2) (snd-display ";locsig reverb set: ~A?" (locsig-reverb-ref lc 0)))
	(mus-close gen)
	(mus-close rev)
	(let ((v0 (make-vct 100))
	      (v1 (make-vct 100))
	      (v2 (make-vct 100)))
	  (file->array "fmv4.snd" 0 0 100 v0)
	  (file->array "fmv4.snd" 1 0 100 v1)
	  (file->array "fmv4.reverb" 0 0 100 v2)
	  (if (fneq (vct-ref v2 0) .1) (snd-display ";locsig reverb: ~A?" v2))
	  (if (fneq (* 2 (vct-ref v0 0)) (vct-ref v1 0)) (snd-display ";locsig direct: ~A ~A?" (vct-ref v0 0) (vct-ref v1 0)))))
      
      (let* ((gen (make-frame->file "fmv4.snd" 4 mus-bshort mus-next))
	     (rev (make-frame->file "fmv4.reverb" 4 mus-bshort mus-next))
	     (lc (make-locsig 60.0 :reverb .1 :channels 4 :distance 4.0 :output gen :revout rev)))
	(print-and-check lc
			 "locsig"
			 "locsig: chans 4, outn: [0.083 0.167 0.000 0.000], revn: [0.017 0.033 0.000 0.000], interp: linear"
			 "locs outn[4]: [0.083 0.167 0.000 0.000], revn[4]: [0.017 0.033 0.000 0.000], interp: linear")
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (locsig lc i 1.0))
	(do ((i 0 (1+ i)))
	    ((= i 4))
	  (locsig-reverb-set! lc i (* i .1))
	  (if (fneq (locsig-reverb-ref lc i) (* i .1))
	      (snd-display ";locsig reverb set![~A]: ~A?" i (locsig-reverb-ref lc i))))
	(print-and-check lc
			 "locsig"
			 "locsig: chans 4, outn: [0.083 0.167 0.000 0.000], revn: [0.000 0.100 0.200 0.300], interp: linear"
			 "locs outn[4]: [0.083 0.167 0.000 0.000], revn[4]: [0.000 0.100 0.200 0.300], interp: linear")
	(if (not (vct? (mus-data lc))) (snd-display ";out data locsig: ~A" (mus-data lc)))
	(if (not (vct? (mus-xcoeffs lc))) (snd-display ";rev data locsig: ~A" (mus-xcoeffs lc)))
	(let ((xcs (mus-xcoeffs lc)))
	  (if (fneq (mus-xcoeff lc 0) (vct-ref xcs 0)) (snd-display ";locsig xcoeff: ~A ~A" (mus-xcoeff lc 0) (vct-ref xcs 0)))
	  (if (fneq (mus-xcoeff lc 1) .1) (snd-display ";locsig xcoeff 1: ~A ~A (.1)" (mus-xcoeff lc 0) (vct-ref xcs 0))))
	(mus-close gen)
	(mus-close rev))
      
      (print-and-check (make-locsig 160 :channels 4)
		       "locsig"
		       "locsig: chans 4, outn: [0.000 0.222 0.778 0.000], interp: linear"
		       "locs outn[4]: [0.000 0.222 0.778 0.000], revn[0]: nil, interp: linear")
      (print-and-check (make-locsig -200 :channels 4)
		       "locsig"
		       "locsig: chans 4, outn: [0.000 0.222 0.778 0.000], interp: linear"
		       "locs outn[4]: [0.000 0.222 0.778 0.000], revn[0]: nil, interp: linear")
      (print-and-check (make-locsig 160 :channels 4 :distance .5)
		       "locsig"
		       "locsig: chans 4, outn: [0.000 0.222 0.778 0.000], interp: linear"
		       "locs outn[4]: [0.000 0.222 0.778 0.000], revn[0]: nil, interp: linear")
      (print-and-check (make-locsig 320 :channels 4)
		       "locsig"
		       "locsig: chans 4, outn: [0.556 0.000 0.000 0.444], interp: linear"
		       "locs outn[4]: [0.556 0.000 0.000 0.444], revn[0]: nil, interp: linear")
      (print-and-check (make-locsig -40 :channels 4)
		       "locsig"
		       "locsig: chans 4, outn: [0.556 0.000 0.000 0.444], interp: linear"
		       "locs outn[4]: [0.556 0.000 0.000 0.444], revn[0]: nil, interp: linear")
      (print-and-check (make-locsig 320 :channels 2)
		       "locsig"
		       "locsig: chans 2, outn: [0.000 1.000], interp: linear"
		       "locs outn[2]: [0.000 1.000], revn[0]: nil, interp: linear")
      (print-and-check (make-locsig -40 :channels 2)
		       "locsig"
		       "locsig: chans 2, outn: [0.000 1.000], interp: linear"
		       "locs outn[2]: [0.000 1.000], revn[0]: nil, interp: linear")
      (letrec ((locsig-data
		(lambda (gen)
		  (let* ((chans (mus-channels gen))
			 (dat (make-vct chans)))
		    (do ((i 0 (1+ i)))
			((= i chans))
		      (vct-set! dat i (locsig-ref gen i)))
		    dat))))
	(let ((gen (make-locsig -.1 :channels 8)))
	  (if (not (vequal (locsig-data gen) (vct 0.998 0.000 0.000 0.000 0.000 0.000 0.000 0.002)))
	      (snd-display ";locsig -.1(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig -359.9 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.998 0.002 0.000 0.000 0.000 0.000 0.000 0.000)))
	      (snd-display ";locsig -359.9(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig -359.9 :channels 4))
	  (if (not (vequal (locsig-data gen) (vct 0.999 0.001 0.000 0.000)))
	      (snd-display ";locsig -359.9(4): ~A" (locsig-data gen)))
	  (set! gen (make-locsig -360.1 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.998 0.000 0.000 0.000 0.000 0.000 0.000 0.002)))
	      (snd-display ";locsig -360.1(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig -700 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.556 0.444 0.000 0.000 0.000 0.000 0.000 0.000)))
	      (snd-display ";locsig -700(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig -700 :channels 2))
	  (if (not (vequal (locsig-data gen) (vct 0.778 0.222)))
	      (snd-display ";locsig -700(2): ~A" (locsig-data gen)))
	  (set! gen (make-locsig 20 :channels 2))
	  (if (not (vequal (locsig-data gen) (vct 0.778 0.222)))
	      (snd-display ";locsig 20(2): ~A" (locsig-data gen)))
	  (set! gen (make-locsig 123456.0 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.467 0.000 0.000 0.000 0.000 0.000 0.000 0.533)))
	      (snd-display ";locsig 123456(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig 336.0 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.467 0.000 0.000 0.000 0.000 0.000 0.000 0.533)))
	      (snd-display ";locsig 336(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig -123456.0 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.467 0.533 0.000 0.000 0.000 0.000 0.000 0.000)))
	      (snd-display ";locsig -123456(8): ~A" (locsig-data gen)))
	  (set! gen (make-locsig 24.0 :channels 8))
	  (if (not (vequal (locsig-data gen) (vct 0.467 0.533 0.000 0.000 0.000 0.000 0.000 0.000)))
	      (snd-display ";locsig 24(8): ~A" (locsig-data gen)))))
      
      (for-each 
       (lambda (rev-chans)
	 
	 (define (locsig-scalers chans degree type)
	   (define (fmod a b)
	     (let ((pos (inexact->exact (floor (/ a b)))))
	       (- a (* pos b))))
	   (if (= chans 1)
	       (vct 1.0)
	       (let* ((deg (if (= chans 2)
			       (max 0.0 (min 90.0 degree))
			       (fmod degree 360.0)))
		      (degs-per-chan (if (= chans 2)
					 90.0
					 (/ 360.0 chans)))
		      (pos (/ deg degs-per-chan))
		      (left (inexact->exact (floor pos)))
		      (right (modulo (+ left 1) chans))
		      (frac (- pos left))
		      (v (make-vct chans)))
		 (if (= type mus-interp-linear)
		     (begin
		       (vct-set! v left (- 1.0 frac))
		       (vct-set! v right frac))
		     (let* ((ldeg (* (/ pi 2) (- 0.5 frac)))
			    (norm (/ (sqrt 2.0) 2.0))
			    (c (cos ldeg))
			    (s (sin ldeg)))
		       (vct-set! v left (* norm (+ c s)))
		       (vct-set! v right (* norm (- c s)))))
		 v)))
	 
	 (if (file-exists? "test.reverb") (delete-file "test.reverb"))
	 (let ((revfile (if (> rev-chans 0)
			    (make-frame->file "test.reverb" rev-chans mus-bshort mus-next)
			    #f)))
	   (for-each
	    (lambda (type)
	      ;; global type def as well as local par override etc
	      
	      (set! (locsig-type) type)
	      (if (not (= (locsig-type) type)) (snd-display ";locsig-type: ~A ~A" type (locsig-type)))
	      
	      (call-with-current-continuation
	       (lambda (quit)
		 
		 (for-each 
		  (lambda (deg)
		    (let ((gen (make-locsig deg :channels 1 :revout revfile :reverb .1 :distance 2.0))
			  (revs (if revfile (locsig-scalers rev-chans deg type))))
		      (if (not (= (mus-channels gen) 1)) (snd-display ";locsig ~A: ~A" deg gen))
		      (if (fneq (locsig-ref gen 0) 0.5) (snd-display ";locsig scaler[~A] ~A: ~A" type deg (locsig-ref gen 0)))
		      (if revfile
			  (do ((i 0 (1+ i)))
			      ((= i rev-chans))
			    (if (fneq (locsig-reverb-ref gen i) (* (/ .1 (sqrt 2.0)) (vct-ref revs i)))
				(begin 
				  (snd-display ";mono locrev[~A] ~A at ~A: ~A ~A" 
					       type gen deg 
					       (locsig-reverb-ref gen i) 
					       (* (/ .1 (sqrt 2.0)) (vct-ref revs i)))
				  (quit)))))))
		  (list 0.0 45.0 90.0 1234.0))
		 
		 (for-each
		  (lambda (ltype)
		    (for-each 
		     (lambda (deg)
		       (let ((gen (make-locsig deg :channels 1 :type ltype)))
			 (if (not (= (mus-channels gen) 1)) (snd-display ";locsig ~A: ~A" deg gen))
			 (if (fneq (locsig-ref gen 0) 1.0) (snd-display ";locsig[~A] scaler ~A: ~A" ltype deg (locsig-ref gen 0)))))
		     (list 0.0 45.0 90.0 1234.0)))
		  (list mus-interp-linear mus-interp-sinusoidal))
		 
		 (for-each
		  (lambda (chans)
		    (for-each 
		     (lambda (deg)
		       (let ((gen (make-locsig deg :channels chans :revout revfile :reverb .1)))
			 (if (not (= (mus-channels gen) chans)) (begin (snd-display ";multi locsig ~A: ~A" deg gen) (quit)))
			 (let ((scalers (locsig-scalers chans deg type))
			       (revs (if revfile (locsig-scalers rev-chans deg type))))
			   (do ((i 0 (1+ i)))
			       ((= i chans))
			     (if (fneq (locsig-ref gen i) (vct-ref scalers i)) 
				 (begin 
				   (snd-display ";locsig[~A] ~A at ~A: ~A ~A" type gen deg (locsig-ref gen i) (vct-ref scalers i)) 
				   (quit))))
			   (if revfile
			       (do ((i 0 (1+ i)))
				   ((= i rev-chans))
				 (if (fneq (locsig-reverb-ref gen i) (* .1 (vct-ref revs i)))
				     (begin 
				       (snd-display ";locrev[~A] ~A at ~A: ~A ~A" 
						    type gen deg 
						    (locsig-reverb-ref gen i) 
						    (* .1 (vct-ref revs i)))
				       (quit))))))))
		     (list 0.0 45.0 90.0 120.0 180.0 275.0 315.0 300.0 15.0 1234.0)))
		  (list 2 3 4 5 8 12 16 24))
		 
		 (for-each
		  (lambda (chans)
		    (for-each 
		     (lambda (ltype)
		       (for-each
			(lambda (deg)
			  (let ((gen (make-locsig deg :channels chans :type ltype :revout revfile :reverb .1)))
			    (if (not (= (mus-channels gen) chans)) (begin (snd-display ";stereo locsig ~A: ~A" deg gen) (quit)))
			    (let ((scalers (locsig-scalers chans deg ltype))
				  (revs (if revfile (locsig-scalers rev-chans deg ltype))))
			      (do ((i 0 (1+ i)))
				  ((= i chans))
				(if (fneq (locsig-ref gen i) (vct-ref scalers i)) 
				    (begin
				      (snd-display ";locsig[~A] ~A at ~A: ~A ~A" ltype gen deg (locsig-ref gen i) (vct-ref scalers i))
				      (quit))))
			      (if revfile
				  (do ((i 0 (1+ i)))
				      ((= i rev-chans))
				    (if (fneq (locsig-reverb-ref gen i) (* .1 (vct-ref revs i)))
					(begin 
					  (snd-display ";locrev[~A] ~A at ~A: ~A ~A" 
						       type gen deg 
						       (locsig-reverb-ref gen i) 
						       (* .1 (vct-ref revs i)))
					  (quit))))))))
			(list 0.0 45.0 90.0 120.0 180.0 275.0 315.0 300.0 15.0 1234.0)))
		     (list mus-interp-linear mus-interp-sinusoidal)))
		  (list 2 3 4 5 8 12 16 24))
		 )))
	    (list mus-interp-linear mus-interp-sinusoidal))
	   (if revfile (mus-close revfile))))
       (list 0 1 2 4))
      
      (let ((gen (make-src :srate 2.0))
	    (v0 (make-vct 10))
	    (rd (make-readin "oboe.snd" 0 2000))
	    (gen1 (make-src :srate 2.0))
	    (gen2 (make-src :srate 0.0)) ; make sure this is allowed
	    (v1 (make-vct 10))
	    (rd1a (make-readin "oboe.snd" 0 2000)))
	(print-and-check gen 
			 "src"
			 "src: width: 10, x: 0.000, incr: 2.000, sinc table len: 10000"
			 "sr x: 0.000000, incr: 2.000000, width: 10, sinc table len: 10000, data[21]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)]")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (src gen 0.0 (lambda (dir) (readin rd)))))
	(vct-map! v1 (lambda () (if (src? gen1) (src gen1 0.0 (lambda (dir) (readin rd1a))))))
	(if (not (vequal v0 v1)) (snd-display ";run src: ~A ~A" v0 v1))
	(if (not (src? gen)) (snd-display ";~A not scr?" gen))
	(if (or (fneq (vct-ref v0 1) .001) (fneq (vct-ref v0 7) .021)) (snd-display ";src output: ~A" v0))
	(if (fneq (mus-increment gen) 2.0) (snd-display ";src increment: ~F?" (mus-increment gen)))
	(if (fneq (mus-increment gen2) 0.0) (snd-display ";src 0.0 increment: ~F?" (mus-increment gen2)))
	(if (fneq (mus-increment rd) 1.0) (snd-display ";readin increment: ~F?" (mus-increment rd)))
	(if (not (= (mus-length gen) 10)) (snd-display ";src length: ~A" (mus-length gen)))
	(let ((gold gen))
	  (set! gen (make-src (lambda (dir)
				0.0)))
	  (if (equal? gen gold) (snd-display ";src eqaul? ~A ~A" gen gold))))
      
      (let ((var (catch #t (lambda () (make-src :width -1)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-src bad width: ~A" var)))
      
      (let ((s1 (make-src (lambda (y) 1.0) 2.0)))
	(src s1 25.0) ; try to tickle segfault
	(src s1 25.0)
	(src s1 125.0)
	(src s1 -25.0)
	(src s1 -125.0))
      (do ((i 0 (1+ i)))
	  ((= i 10))
	(make-src (lambda (y) 1.0) 1.5 :width (+ 5 (* i 10))))
      (clear-sincs)
      
      (let ((gen (make-granulate :expansion 2.0))
	    (v0 (make-vct 1000))
	    (rd (make-readin "oboe.snd" 0 4000))
	    (gen1 (make-granulate :expansion 2.0))
	    (v1 (make-vct 1000))
	    (rd1b (make-readin "oboe.snd" 0 4000)))
	(print-and-check gen 
			 "granulate"
			 "granulate: expansion: 2.000 (551/1102), scaler: 0.600, length: 0.150 secs (3308 samps), ramp: 0.060"
			 "grn_info s20: 1102, s50: 441, rmp: 1323, amp: 0.600000, len: 3308, cur_out: 0, cur_in: 0, input_hop: 551, ctr: 0, output_hop: 1102, in_data_start: 5513, in_data[5513]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], data[4410]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)]"
			 "grn_info s20: 1102, s50: 441, rmp: 1323, amp: 0.600000, len: 3308, cur_out: 0, cur_in: 0, input_hop: 551, ctr: 0, output_hop: 1102, in_data_start: 5512, in_data[5512]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], data[4409]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)]")
	(do ((i 0 (1+ i)))
	    ((= i 1000))
	  (vct-set! v0 i (granulate gen (lambda (dir) (readin rd)))))
	(vct-map! v1 (lambda () (if (granulate? gen1) (granulate gen1 (lambda (dir) (readin rd1b))))))
	(let ((worst (abs (- (vct-peak v0) (vct-peak v1)))))
	  (if (> worst .01) (snd-display ";run granulate: ~A" worst)))
	(let ((genx gen1))
	  (if (not (equal? genx gen1))
	      (snd-display ";granulate equal? ~A ~A ~A" genx gen1 (equal? genx gen1))))
	(if (equal? gen gen1) (snd-display ";granulate equal? ~A ~A" gen gen1))
	(if (= (vct-peak v0) 0.0) (snd-display ";granulate output peak: ~F?" (vct-peak v0)))
	(if (not (granulate? gen)) (snd-display ";~A not granulate?" gen))
	(if (fneq (mus-increment gen) 2.0) (snd-display ";granulate increment: ~F?" (mus-increment gen)))
	(if (fneq (mus-scaler gen) 0.6) (snd-display ";granulate scaler: ~F?" (mus-scaler gen)))
	(if (ffneq (mus-frequency gen) 0.05) (snd-display ";granulate frequency: ~F?" (mus-frequency gen)))
	(if (not (= (mus-ramp gen) 1323)) (snd-display ";granulate ramp: ~F?" (mus-ramp gen)))
	(if (not (= (mus-length gen) 3308)) (snd-display ";granulate length: ~A?" (mus-length gen)))
	(if (not (= (mus-hop gen) 1102)) (snd-display ";granulate hop: ~A?" (mus-hop gen)))
	(set! (mus-hop gen) 1000) (if (not (= (mus-hop gen) 1000)) (snd-display ";granulate set-hop: ~A?" (mus-hop gen)))
	(set! (mus-ramp gen) 1000) (if (not (= (mus-ramp gen) 1000)) (snd-display ";granulate set-ramp: ~A?" (mus-ramp gen)))
	(set! (mus-length gen) 3000) (if (not (= (mus-length gen) 3000)) (snd-display ";granulate set-length: ~A?" (mus-length gen)))
	(set! (mus-increment gen) 3.0)
	(if (> (abs (- (mus-increment gen) 3.0)) .01) (snd-display ";granulate set-increment: ~F?" (mus-increment gen)))
	(set! (mus-frequency gen) .1)
	(if (fneq (mus-frequency gen) .1) (snd-display ";set granulate freq: ~A" (mus-frequency gen)))
	(let ((tag (catch #t (lambda () (granulate gen (lambda (a b) a))) (lambda args (car args)))))
	  (if (not (eq? tag 'bad-arity)) 
	      (snd-display ";granulate bad func: ~A" tag))))
      (let ((var (catch #t (lambda () (make-granulate :hop 35.0 :length 35.0)) (lambda args args))))
	(if (not (eq? (car var) 'out-of-range))
	    (snd-display ";make-granulate bad sizes: ~A" var)))
      
      (let ((ind (open-sound "oboe.snd")))
	(let ((grn (make-granulate :expansion 2.0
				   :edit (lambda (g)
					   (let ((grain (mus-data g))  ; current grain
						 (len (mus-length g))) ; current grain length
					     (do ((i 0 (1+ i)))
						 ((= i len) len)       ; grain length unchanged in this case
					       (vct-set! grain i (* 2 (vct-ref grain i))))))))
	      (rd (make-sample-reader 0)))
	  (map-channel (lambda (y) (granulate grn (lambda (dir) (rd))))))
	(let ((mx (maxamp)))
	  (undo)
	  (let ((grn (make-granulate :expansion 2.0
				     :edit (lambda (g)
					     (let ((grain (mus-data g))  ; current grain
						   (len (mus-length g))) ; current grain length
					       (do ((i 0 (1+ i)))
						   ((= i len) len)       ; grain length unchanged in this case
						 (vct-set! grain i (* 4 (vct-ref grain i))))))))
		(rd (make-sample-reader 0)))
	    (map-channel (lambda (y) (granulate grn (lambda (dir) (rd)))))))
	(close-sound ind))

      (let* ((v0 (make-vct 32))
	     (v1 (make-vct 256))
	     (v2 (make-vct 256))
	     (v01 (make-vct 32))
	     (v11 (make-vct 256))
	     (v21 (make-vct 256)))
	(do ((i 1 (1+ i)))
	    ((= i 16))
	  (vct-set! v0 i (/ 1.0 i))
	  (vct-set! v01 i (/ 1.0 i)))
	(vct-set! v1 0 1.0)
	(vct-set! v11 0 1.0)
	(let ((gen (make-convolve :filter v0))
	      (n -1)
	      (gen1 (make-convolve :filter v01))
	      (n1 -1))
	  (print-and-check gen 
			   "convolve"
			   "convolve: size: 64"
			   "conv fftsize: 64, fftsize2: 32, filtersize: 32, ctr: 32, rl1: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], rl2: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], buf: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], filter: [0.000 1.000 0.500 0.333 0.250 0.200 0.167 0.143...(0: 0.000, 1: 1.000)]")
	  (if (not (convolve? gen)) (snd-display ";~A not convolve?" gen))
	  (let ((genx gen1))
	    (if (not (equal? genx gen1)) (snd-display ";convolve equal?: ~A ~A ~A" genx gen1 (equal? genx gen1))))
	  (if (equal? gen gen1) (snd-display ";convolve equal? ~A ~A" gen gen1))
	  (if (not (= (mus-length gen) 64)) (snd-display ";convolve fft len: ~D?" (mus-length gen)))
	  (do ((i 0 (1+ i)))
	      ((= i 128))
	    (vct-set! v2 i (convolve gen (lambda (dir) (set! n (+ n 1)) (vct-ref v1 n)))))
	  (vct-map! v21 (lambda () (if (convolve? gen1) (convolve gen1 (lambda (dir) (set! n1 (+ n1 1)) (vct-ref v11 n1))))))
	  (if (not (vequal v2 v21)) (snd-display ";run gran: ~A ~A" v2 v21))
	  (if (or (fneq (vct-ref v2 0) 0.0)
		  (fneq (vct-ref v2 1) 1.0)
		  (fneq (vct-ref v2 4) 0.25)
		  (fneq (vct-ref v2 7) 0.143))
	      (snd-display ";convolve output: ~A?" v2))
	  (let ((tag (catch #t (lambda () (convolve gen (lambda (a b) a))) (lambda args (car args)))))
	    (if (not (eq? tag 'bad-arity)) 
		(snd-display ";convolve bad func: ~A" tag))))
	
	(convolve-files "oboe.snd" "fyow.snd" .5 "fmv.snd")
	(if (fneq (cadr (mus-sound-maxamp "fmv.snd")) .5) 
	    (snd-display ";convolve-files: ~A /= .5?" (cadr (mus-sound-maxamp "fmv.snd"))))
	(play-sound-1 "fmv.snd"))
      
      (let* ((fd (mus-sound-open-input "oboe.snd"))
	     (chans (mus-sound-chans "oboe.snd"))
	     (data (make-sound-data chans 2000)))
	(if (not (sound-data? data)) (snd-display ";~A not sound-data?" data))
	(if (not (= (sound-data-chans data) 1)) (snd-display ";sound-data chans: ~A?" (sound-data-chans data)))
	(if (not (= (sound-data-length data) 2000)) (snd-display ";sound-data length: ~A?" (sound-data-length data)))
	(mus-sound-read fd 0 1999 chans data)
	(let ((val (sound-data-ref data 0 1497)))
	  (mus-sound-close-input fd)
	  (if (fneq val 0.02893066) (snd-display ";mus-sound-read: ~F?" val))))
      
      (let ((ind (new-sound "fmv.snd")))
	(set! (sample 1) .1)
	(save-sound ind)
	(if (not (equal? (edits ind 0) (list 0 0)))
	    (snd-display ";weird: edits not cleared after save-sound?: ~A" (edits ind 0)))
	(close-sound ind)
	(set! ind (open-sound "fmv.snd"))
	(if (not (= (frames ind 0) 2))
	    (snd-display ";save-sound 2 samps: ~A?" (frames ind 0)))
	(if (or (fneq (sample 0) 0.0)
		(fneq (sample 1) 0.1))
	    (snd-display ";save-sound: ~A ~A?" (sample 0) (sample 1)))
	(do ((i 3 (1+ i)))
	    ((= i 6))
	  (set! (sample i) (* i .1))
	  (save-sound ind)
	  (if (not (equal? (edits ind 0) (list 0 0)))
	      (snd-display ";weird: edits not cleared after save-sound ~A?: ~A" i (edits ind 0)))
	  (close-sound ind)
	  (set! ind (open-sound "fmv.snd"))
	  (if (not (= (frames ind 0) (+ i 1)))
	      (snd-display ";save-sound ~A samps: ~A?" (+ i 1) (frames ind 0)))
	  (if (or (fneq (sample 0) 0.0)
		  (fneq (sample 1) 0.1)
		  (fneq (sample i) (* i 0.1)))
	      (snd-display ";save-sound ~A: ~A ~A ~A?" i (sample 0) (sample 1) (sample i))))
	(close-sound ind))
      
      (let ((ind (new-sound "test.snd" :srate 22050 :channels 1 :size 1000))
	    (gen (make-ssb-am 100.0)))
	(map-channel (lambda (y) (ssb-am gen 0.0)))
	(if (fneq (maxamp) 0.0) (snd-display ";ssb-am 0.0: ~A" (maxamp)))
	(let ((gen1 (make-oscil 220.0)))
	  (map-channel (lambda (y) (* 0.5 (oscil gen1))))
	  (set! gen (make-ssb-am 100.0 100))
	  (map-channel (lambda (y) (ssb-am gen y)))
	  (delete-samples 0 200)
	  (set! gen1 (make-oscil 320.0 :initial-phase (asin (* 2 (sample 0))))) ; depends on rising side
	  (map-channel (lambda (y) (- y (* 0.5 (oscil gen1)))))
	  (if (> (maxamp) .003) (snd-display ";ssb-am cancelled: ~A" (maxamp)))
	  (close-sound ind)))

      (let ((ind (new-sound "test.snd" :srate 22050 :channels 1 :size 1000))
	    (ctr 0))
	(map-channel (lambda (y) (let ((val (sin (/ (* 2 pi ctr) 50)))) (set! ctr (+ ctr 1)) val)))
	;; 441 Hz 
	(ssb-bank 441 882 1 100)
	(delete-samples 0 217)
	(let ((gen1 (make-oscil 882.0 :initial-phase (asin (sample 0)))))
	  (map-channel (lambda (y) (- y (oscil gen1))))
	  (if (> (maxamp) .04) (snd-display ";ssb-bank cancelled: ~A" (maxamp))))
	(close-sound ind))

      (if *output* 
	  (begin
	    (snd-display ";*output* ~A" *output*)
	    (set! *output* #f)))

      (let ((nind (new-sound "fmv.snd" mus-aifc mus-bshort 22050 1 "this is a comment")))
	(time (fm-violin 0 1 440 .1))
	(play-and-wait 0 nind)
	(save-sound nind)
	(if (not (sound? nind)) (snd-display ";save sound clobbered ~A?" nind))
	(let ((oboe-index (or (find-sound "oboe.snd") (open-sound "oboe.snd"))))
	  (if (= oboe-index nind) (snd-display ";find-sound found bogus case: ~A" oboe-index))
	  (cnvtest oboe-index nind .1) 
	  (select-sound nind)
	  (select-channel 0)
	  (if (not (= (selected-sound) nind)) (snd-display ";selected-sound: ~A?" (selected-sound)))
	  (if (not (= (selected-channel) 0)) (snd-display ";selected-channel: ~A?" (selected-channel)))
	  (jc-reverb 1.0 #f .1 #f) 
	  (play-and-wait 0 nind)
	  (voiced->unvoiced 1.0 256 2.0 2.0) 
	  (map-chan (fltit))
	  (close-sound oboe-index))
	(if (not (sound? nind)) (snd-display ";close sound clobbered ~A?" nind))
	(let ((fr (frames nind 0)))
	  (do ((k 0 (1+ k)))
	      ((= k 10))
	    (delete-samples 10 100 nind 0)
	    (save-sound nind)) ;flush out memory leaks here
	  (if (not (= (frames nind 0) (- fr 1000)))
	      (snd-display ";delete-samples: ~A ~A" fr (frames nind 0))))
	(revert-sound nind)
	(close-sound nind))
      (if (file-exists? "fmv.snd") (delete-file "fmv.snd"))
      
      (let ((nind (new-sound "fmv.snd")))
	(if (not (= (header-type nind) (default-output-type)))
	    (snd-display ";new-sound default header-type: ~A ~A?"
			 (mus-header-type-name (header-type nind))
			 (mus-header-type-name (default-output-type))))
	(if (not (= (data-format nind) (default-output-format)))
	    (snd-display ";new-sound default data-format: ~A ~A?"
			 (mus-data-format-name (data-format nind))
			 (mus-data-format-name (default-output-format))))
	(if (not (= (chans nind) (default-output-chans)))
	    (snd-display ";new-sound default chans: ~A ~A?" (chans nind) (default-output-chans)))
	(if (not (= (srate nind) (default-output-srate)))
	    (snd-display ";new-sound default srate: ~A ~A?" (srate nind) (default-output-srate)))
	(close-sound nind)
	(if (file-exists? "fmv.snd") (delete-file "fmv.snd")))
      (let ((nind (new-sound "fmv.snd" mus-nist mus-bshort 22050 1 "this is a comment")))
	(set! (sample 0 nind) 1.0) 
	(start-progress-report nind)
	(convolve-with "oboe.snd") 
	(progress-report .1 "hiho" 0 1 nind)
	(if (fneq (sample 1000) -0.22299) (snd-display ";convolve-with: ~A?" (sample 1000)))
	(progress-report .3 "hiho" 0 1 nind)
	(revert-sound nind)
	(progress-report .5 "hiho" 0 1 nind)
	(set! (sample 200) .0001) 
	(set! (sample 100) 1.0) 
	(progress-report .8 "hiho" 0 1 nind)
	(smooth-sound 0 100) 
	(finish-progress-report nind)
	(if (or (fneq (sample 50) .5) (fneq (sample 30) 0.20608) (fneq (sample 90) 0.9755))
	    (snd-display ";smooth: ~A ~A ~A?" (sample 50) (sample 30) (sample 90)))
	(undo) 
	(set! (sinc-width) 40) 
	(set! (sample 100) 0.5) 
	(if (fneq (sample 100) 0.5) (snd-display ";set-sample 100: ~A?" (sample 100)))
	(src-sound .1) 
	(if (or (fneq (sample 1000) 0.5) (fneq (sample 1024) 0.0625) (fneq (sample 1010) 0.0))
	    (snd-display ";src-sound: ~A ~A ~A?" (sample 1000) (sample 1024) (sample 1010)))
	(revert-sound)
	(close-sound nind))
      (let ((nind (new-sound "fmv.snd" mus-riff mus-lshort 22050 1 "this is a comment" 22050)))
	(if (not (= (frames nind) 22050)) (snd-display "; new-sound initial-length: ~A" (frames nind)))
	(mix "pistol.snd") 
	(map-chan (expsrc 2.0 nind)) 
	(play-and-wait 0 nind)
	(undo) 
	(let ((eds (edits)))
	  (if (or (not (= (car eds) 1)) (not (= (cadr eds) 1)))
	      (snd-display ";undo edits: ~A?" eds))
	  (if (not (= (edit-position) (car eds)))
	      (snd-display ";undo edit-position: ~A ~A?" (edit-position) eds)))
	(expsnd '(0 1 2 .4)) 
	(map-chan (comb-chord .95 100 .3)) 
	(map-chan (formants .99 900 .02 1800 .01 2700)) 
	(map-chan (moving-formant .99 '(0 1200 1 2400))) 
	(scale-to .3) 
	(play-and-wait 0)
	(let ((eds (edits)))
	  (if (or (not (= (car eds) 6)) (not (= (cadr eds) 0)))
	      (snd-display ";edits(6): ~A?" eds))
	  (if (not (= (edit-position) (car eds)))
	      (snd-display ";edit-position(6): ~A ~A?" (edit-position) eds)))
	(set! (edit-position) 1)
	(if (not (= (edit-position) 1))
	    (snd-display ";set edit-position(1): ~A?" (edit-position)))
	(set! (edit-position) 4)
	(if (not (= (edit-position) 4))
	    (snd-display ";set edit-position(4): ~A?" (edit-position)))
	(revert-sound nind)
	(mix "pistol.snd") 
	(map-chan (zecho .5 .75 6 10.0) 0 65000) 
	(map-chan (am 440)) 
	(add-mark 1200)
	(add-mark 2300)
	(key (char->integer #\x) 4)
	(key (char->integer #\c) 0) ; trigger mark-define-region
	(reverse-sound nind) 
	(play-and-wait 0 nind)
	(revert-sound nind)
	(let ((mid (mix-sound "pistol.snd" 0)))
	  (if (not (equal? (mix-home mid) (list (selected-sound) 0)))
	      (snd-display ";mix-sound mix-home: ~A (~A or ~A 0)" (mix-home mid) (selected-sound) nind)))
	(hello-dentist 40.0 .1) 
	(fp 1.0 .3 20) 
	(play-and-wait 0 nind)
	(revert-sound nind)
	(enveloped-mix "oboe.snd" 0 '(0 0 1 1 2 0)) 
	(pvoc :pitch 0.5 :time 1.0 :snd nind) 
	(play-and-wait 0 nind)
	(revert-sound nind)
	(close-sound nind))
      
      (let ((make-mix-output (lambda (name i)
			       (if (or (= i 0) (= i 1)) 
				   name
				   (continue-sample->file name))))
	    (make-mix-input (lambda (name i)
			      (if (or (= i 0) (= i 2))
				  name
				  (make-file->frame name)))))
	(define* (mus-mix-1 outf inf #:optional outloc frames inloc mixer envs)
	  (if envs
	      (mus-mix outf inf outloc frames inloc mixer envs)
	      (if mixer
		  (mus-mix outf inf outloc frames inloc mixer)
		  (if inloc
		      (mus-mix outf inf outloc frames inloc)
		      (if frames
			  (mus-mix outf inf outloc frames)
			  (if outloc
			      (mus-mix outf inf outloc)
			      (mus-mix outf inf))))))
	  (if (not (string? outf))
	      (mus-close outf)))

	(do ((k 0 (1+ k)))
	    ((= k 4))
	  (if (file-exists? "fmv.snd") (delete-file "fmv.snd"))
	  (if (file-exists? "fmv1.snd") (delete-file "fmv1.snd"))
	  (if (file-exists? "fmv2.snd") (delete-file "fmv2.snd"))
	  (if (file-exists? "fmv3.snd") (delete-file "fmv3.snd"))
	  (let ((v0 (make-vct 12)))
	    (vct-fill! v0 0.1)
	    (array->file "fmv1.snd" v0 12 22050 1)
	    (vct-fill! v0 0.2)
	    (array->file "fmv2.snd" v0 12 22050 2)
	    (vct-fill! v0 0.3)
	    (array->file "fmv3.snd" v0 12 22050 4)
	    (do ((i 0 (1+ i))) ((= i 12)) (vct-set! v0 i (* i .01)))
	    (array->file "fmv.snd" v0 12 22050 1)
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv1.snd" k))
	    (file->array "fmv.snd" 0 0 12 v0)
	    (call-with-current-continuation
	     (lambda (break)
	       (do ((i 0 (1+ i))) ((= i 12)) 
		 (if (fneq (vct-ref v0 i) (+ 0.1 (* i .01))) 
		     (break (snd-display ";~D mus-mix(1->1): ~A?" k v0))))))
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv2.snd" k) 3 9 0 (make-mixer 2 0.3 0.0 0.7 0.0))
	    (file->array "fmv.snd" 0 0 12 v0)
	    (if (or (fneq (vct-ref v0 0) .1) (fneq (vct-ref v0 3) .33) (fneq (vct-ref v0 9) .19)) (snd-display ";~D mus-mix(2->1): ~A?" k v0))
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv3.snd" k))
	    (file->array "fmv.snd" 0 0 12 v0)
	    (if (or (fneq (vct-ref v0 0) .4) (fneq (vct-ref v0 3) .33)) (snd-display ";~D mus-mix(4->1): ~A?" k v0))
	    (let ((e0 (make-env '(0 0 1 1) :end 10))
		  (vf (make-vector 1))
		  (vf1 (make-vector 1)))
	      (vector-set! vf 0 vf1)
	      (vector-set! vf1 0 e0)
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv1.snd" k) 0 12 0 (make-mixer 1 1.0) vf)
	      (file->array "fmv.snd" 0 0 12 v0)
	      (if (or (fneq (vct-ref v0 0) .4) (fneq (vct-ref v0 3) .360) (fneq (vct-ref v0 9) .28)) (snd-display ";~D mus-mix(env): ~A?" k v0))
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv2.snd" k) 0 12 0 (make-mixer 2 1.0 1.0 1.0 1.0) vf)) 
	    ;; clm2xen should protect us here
	    (let ((vf (make-vector 2))
		  (vf1 (make-vector 2))
		  (vf2 (make-vector 2)))
	      (vector-set! vf 0 vf1)
	      (vector-set! vf 1 vf2)
	      (vector-set! vf1 0 (make-env '(0 0 1 1) :end 9))
	      (vector-set! vf2 1 (make-env '(0 0 1 1) :end 9))
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv2.snd" k) 0 12 0 (make-mixer 2 1.0 1.0 1.0 1.0) vf)
	      (let ((tag (catch #t
				(lambda ()
				  (vector-set! vf 0 (make-oscil))
				  (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv2.snd" k) 0 12 0 (make-mixer 2 1.0 1.0 1.0 1.0) vf))
				(lambda args (car args)))))
		(if (not (eq? tag 'bad-type))
		    (snd-display ";~D mix w oscil-vect: ~A" k tag)))
	      (vector-set! vf 0 vf1)
	      (vector-set! vf 1 vf2)
	      (let ((tag (catch #t
				(lambda ()
				  (vector-set! vf1 0 (make-oscil))
				  (vector-set! vf2 1 (sqrt -1.0))
				  (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv2.snd" k) 0 12 0 (make-mixer 2 1.0 1.0 1.0 1.0) vf))
				(lambda args (car args)))))
		(if (not (eq? tag 'bad-type))
		    (snd-display ";~D mix w oscil-env: ~A" k tag))))
	    (delete-file "fmv.snd")
	    (do ((i 0 (1+ i))) ((= i 12)) (vct-set! v0 i (* i .01)))
	    (array->file "fmv.snd" v0 12 22050 4)
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv1.snd" k))
	    (file->array "fmv.snd" 0 0 3 v0) ; chan 0 start 0 len 3
	    (if (or (fneq (vct-ref v0 0) .1) (fneq (vct-ref v0 2) .18)) (snd-display ";~D mus-mix(1->4): ~A?" k v0))
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv2.snd" k)  0 3 0 (make-mixer 2 0.3 0.0 0.7 0.0))
	    (file->array "fmv.snd" 0 0 3 v0)
	    (if (or (fneq (vct-ref v0 0) .3) (fneq (vct-ref v0 2) .38)) (snd-display ";~D mus-mix(2->4): ~A?" k v0))
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "fmv3.snd" k) 0 2 0)
	    (file->array "fmv.snd" 0 0 3 v0)
	    (if (or (fneq (vct-ref v0 0) .6) (fneq (vct-ref v0 2) .38)) (snd-display ";~D mus-mix(4->4): ~A?" k v0)))
	  
	  (if (file-exists? "fmv.snd") (delete-file "fmv.snd"))
	  (let ((v0 (make-vct 12))
		(len (mus-sound-frames "oboe.snd")))
	    (array->file "fmv.snd" v0 12 22050 1)
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "oboe.snd" k))
	    (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "oboe.snd" k) 0 len 0 (make-mixer 1 0.5))
	    (let* ((egen (make-vector 1))
		   (outv (make-vector 1)))
	      (vector-set! outv 0 egen)
	      (vector-set! egen 0 (make-env :envelope '(0 0 1 1) :end len))
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "oboe.snd" k) 0 len 0 #f outv)
	      (vector-set! egen 0 (make-env :envelope '(0 1 1 0) :end len))
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "oboe.snd" k) 0 len 0 (make-mixer 1 1.0) outv))
	    (let ((ind-oboe (open-sound "oboe.snd"))
		  (ind-mix (open-sound "fmv.snd")))
	      (if (not (vequal (samples->vct 1000 10 ind-oboe)
			       (vct-scale! (samples->vct 1000 10 ind-mix) (/ 1.0 2.5))))
		  (snd-display ";~D mus-mix 1 chan: ~A ~A" k
			       (samples->vct 1000 10 ind-oboe)
			       (samples->vct 1000 10 ind-mix)))
	      (close-sound ind-oboe)
	      (close-sound ind-mix))
	    (delete-file "fmv.snd")
	    (let ((v0 (make-vct 12))
		  (len (mus-sound-frames "2.snd")))
	      (array->file "fmv.snd" v0 12 22050 2)
	      (if (not (= (mus-sound-chans "fmv.snd") 2))
		  (snd-display ";~D array->file chans? ~A" k (mus-sound-chans "fmv.snd")))
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "2.snd" k))
	      (mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "2.snd" k) 0 len 0 (make-mixer 2 0.5 0.0 0.0 0.5))
	      (let* ((egen0 (make-vector 2))
		     (egen1 (make-vector 2))
		     (outv (make-vector 2)))
		(vector-set! outv 0 egen0)
		(vector-set! outv 1 egen1)
		(vector-set! egen0 0 (make-env :envelope '(0 0 1 1) :end len))
		(vector-set! egen1 1 (make-env :envelope '(0 0 1 1) :end len))
		(mus-mix-1 (make-mix-output "fmv.snd" k) (make-mix-input "2.snd" k) 0 len 0 #f outv))
	      (let ((ind-mix (open-sound "fmv.snd")))
		(if (not (= (channels ind-mix) 2))
		    (snd-display ";~D fmv re-read chans? ~A ~A" k (mus-sound-chans "fmv.snd") (channels ind-mix)))
		(if (not (vequal (samples->vct 1000 10 ind-mix 0)
				 (vct 0.003 0.010 0.012 0.011 0.008 0.004 0.002 0.002 0.007 0.017)))
		    (snd-display ";~D mus-mix 2 chan (2.snd written: ~A): ~A ~A" k
				 (strftime "%d-%b %H:%M %Z" (localtime (mus-sound-write-date "2.snd")))
				 (samples->vct 1000 10 ind-mix 0)
				 (samples->vct 1000 10 ind-mix 1)))
		(close-sound ind-mix)
		(delete-file "fmv.snd"))))
	  ); end do loop
	); end let
	
      (let* ((gen (make-phase-vocoder #f 512 4 256 1.0 #f #f #f))
	     (val (phase-vocoder gen)))
	(if (fneq val 0.0) (snd-display ";simple no-in pv call: ~A" val)))

      (let* ((ind (open-sound "oboe.snd"))
	     (pi2 (* 2.0 pi))
	     (pv (make-phase-vocoder #f
				     512 4 128 1.0
				     #f ;no change to analysis
				     #f ;no change to edits
				     #f ;no change to synthesis
				     ))
	     (reader (make-sample-reader 0)))
	(if (not (phase-vocoder? pv)) (snd-display ";~A not phase-vocoder?" pv))
	(print-and-check pv 
			 "phase-vocoder"
			 "phase-vocoder: outctr: 128, interp: 128, filptr: 0, N: 512, D: 128, in_data: nil"
			 "pv_info outctr: 128, interp: 128, filptr: 0, N: 512, D: 128, in_data: nil, amps: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], freqs: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)]")
	(select-sound ind)
	(map-chan (lambda (val)
 		    (phase-vocoder pv (lambda (dir) 
					(next-sample reader)))))
	(vct-set! (pv-amp-increments pv) 0 .1)
	(if (fneq (vct-ref (pv-amp-increments pv) 0) .1)
	    (snd-display ";set pv-amp-increments: ~A?" (vct-ref (pv-amp-increments pv) 0)))
	(vct-set! (pv-amps pv) 0 .1)
	(if (fneq (vct-ref (pv-amps pv) 0) .1)
	    (snd-display ";set pv-amps: ~A?" (vct-ref (pv-amps pv) 0)))
	(vct-set! (pv-phases pv) 0 .1)
	(if (fneq (vct-ref (pv-phases pv) 0) .1)
	    (snd-display ";set pv-phases: ~A?" (vct-ref (pv-phases pv) 0)))
	(vct-set! (pv-phase-increments pv) 0 .1)
	(if (fneq (vct-ref (pv-phase-increments pv) 0) .1)
	    (snd-display ";set pv-phase-increments: ~A?" (vct-ref (pv-phase-increments pv) 0)))
	(vct-set! (pv-freqs pv) 0 .1)
	(if (fneq (vct-ref (pv-freqs pv) 0) .1)
	    (snd-display ";set pv-freqs: ~A?" (vct-ref (pv-freqs pv) 0)))
	(undo 1)
	(free-sample-reader reader)
	(let ((lastphases (make-vct 512)))
	  (set! pv (make-phase-vocoder #f
				       512 4 128 1.0
				       #f ;no change to analysis
				       (lambda (v)
					; new editing func changes pitch
					 (let* ((N (mus-length v)) ;mus-increment => interp, mus-data => in-data
						(D (mus-hop v))
						(freqs (pv-freqs v)))
					   (do ((k 0 (1+ k))
						(pscl (/ 1.0 D))
						(kscl (/ pi2 N)))
					       ((= k (inexact->exact (floor (/ N 2)))))
					     (let ((phasediff (- (vct-ref freqs k) (vct-ref lastphases k))))
					       (vct-set! lastphases k (vct-ref freqs k))
					       (if (> phasediff pi) (do () ((<= phasediff pi)) (set! phasediff (- phasediff pi2))))
					       (if (< phasediff (- pi)) (do () ((>= phasediff (- pi))) (set! phasediff (+ phasediff pi2))))
					       (vct-set! freqs k 
							 (* 0.5
							    (+ (* pscl phasediff)
							       (* k kscl))))))
					   #f))
				       #f ; no change to synthesis
				       ))
	  (set! reader (make-sample-reader 0))
	  (map-chan (lambda (val)
		      (phase-vocoder pv (lambda (dir) 
					  (next-sample reader))))))
	(undo 1)
	(free-sample-reader reader)
	(set! pv (make-phase-vocoder #f
				     512 4 (inexact->exact (* 128 2.0)) 1.0
				     #f ;no change to analysis
				     #f ;no change to edits
				     #f ;no change to synthesis
				     ))
	(set! reader (make-sample-reader 0))
	(let* ((len (inexact->exact (* 2.0 (frames ind))))
	       (data (make-vct len)))
	  (vct-map! data
		    (lambda ()
		      (phase-vocoder pv (lambda (dir) (next-sample reader)))))
	  (set! (samples 0 len) data))
	(undo 1)
	(free-sample-reader reader)
	
	(let ((incalls 0)
	      (outcalls 0))
	  (set! pv (make-phase-vocoder #f
				       512 4 (inexact->exact (* 128 2.0)) 1.0
				       (lambda (v infunc)
					 (set! incalls (+ incalls 1))
					 #t)
				       #f ;no change to edits
				       (lambda (v)
					 (set! outcalls (+ outcalls 1))
					 0.0)
				       ))
	  (set! reader (make-sample-reader 0))
	  (let* ((len (inexact->exact (* 2.0 (frames ind))))
		 (data (make-vct len)))
	    (vct-map! data
		      (lambda ()
			(phase-vocoder pv (lambda (dir) (next-sample reader)))))
	    (set! (samples 0 len) data))
	  (undo 1)
	  (free-sample-reader reader)
	  (if (or (= incalls 0)
		  (= outcalls 0))
	      (snd-display ";phase-vocoder incalls: ~A, outcalls: ~A" incalls outcalls))
	  (set! (pv-outctr pv) (pv-outctr pv))
	  (let ((tag (catch #t (lambda () (phase-vocoder pv (lambda (a b) a))) (lambda args (car args)))))
	    (if (not (eq? tag 'bad-arity)) 
		(snd-display ";phase-vocoder bad func: ~A" tag))))
	(let ((tag (catch #t 
			  (lambda () (make-phase-vocoder #f 512 4 256 1.0 (lambda (a b c) #f) #f #f)) 
			  (lambda args args))))
	  (if (not (eq? (car tag) 'bad-arity)) (snd-display ";make-phase-vocoder bad analyze func: ~A" tag)))
	(let ((tag (catch #t
			  (lambda () (make-phase-vocoder #f 512 4 256 1.0 (lambda (a b) 0.0) (lambda (a b c) #f) #f)) 
			  (lambda args args))))
	  (if (not (eq? (car tag) 'bad-arity)) (snd-display ";make-phase-vocoder bad edit func: ~A" tag)))
	(let ((tag (catch #t 
			  (lambda () (make-phase-vocoder #f 512 4 256 1.0 (lambda (a b) 0.0) (lambda (a) #f) (lambda (a b) 0)))
			  (lambda args args))))
	  (if (not (eq? (car tag) 'bad-arity)) (snd-display ";make-phase-vocoder bad synthesize func: ~A" tag)))
	(let ((geno (make-phase-vocoder (lambda (dir) 0.0))))
	  (let ((genx (make-phase-vocoder :input (lambda (dir) 0.0))))
	    (if (equal? geno genx) (snd-display ";phase-vocoder equal? ~A ~A" geno genx))
	    (if (fneq (mus-frequency genx) 1.0) (snd-display ";mus-frequency phase-vocoder: ~A" (mus-frequency genx)))
	    (set! (mus-frequency genx) 2.0)
	    (if (fneq (mus-frequency genx) 2.0) (snd-display ";set mus-frequency phase-vocoder: ~A" (mus-frequency genx)))
	    (if (fneq (mus-increment genx) 128) (snd-display ";mus-increment phase-vocoder: ~A" (mus-increment genx)))
	    (set! (mus-increment genx) 256)
	    (if (fneq (mus-increment genx) 256) (snd-display ";set mus-increment phase-vocoder: ~A" (mus-increment genx)))
	    (if (not (= (mus-hop genx) 128)) (snd-display ";phase vocoder hop: ~A" (mus-hop genx)))
	    (set! (mus-hop genx) 64)
	    (if (not (= (mus-hop genx) 64)) (snd-display ";set phase vocoder hop: ~A" (mus-hop genx)))
	    (if (not (= (mus-length genx) 512)) (snd-display ";phase vocoder length: ~A" (mus-length genx)))
	    (set! (mus-length genx) 64)
	    (if (not (= (mus-length genx) 64)) (snd-display ";set phase vocoder length: ~A" (mus-length genx)))
	    (let ((genxx genx))
	      (if (not (equal? genx genxx)) (snd-display ";phase-vocoder equal: ~A ~A" genxx genx)))))
	(close-sound ind))
      
      (let ((ind (open-sound "oboe.snd")))
	(let ((gen (make-moog-filter 500.0 .1)))
	  (if (fneq 500.0 (car gen)) (snd-display ";moog freq: ~A" (car gen)))
	  (if (fneq .1 (cadr gen)) (snd-display ";moog Q: ~A" (cadr gen)))
	  (if (not (vct? (list-ref gen 2))) (snd-display ";moog state: ~A" (list-ref gen 2)))
	  (if (fneq 0.0 (list-ref gen 3)) (snd-display ";moog A? ~A" (list-ref gen 3)))
	  (if (fneq -0.861 (list-ref gen 4)) (snd-display ";moog freqtable: ~A" (lisst-ref gen 4)))
	  (let ((vals (make-vct 20)))
	    (vct-set! vals 0 (moog-filter gen 1.0))
	    (do ((i 1 (1+ i)))
		((= i 20))
	      (vct-set! vals i (moog-filter gen 0.0)))
	    (if (not (vequal vals (vct 0.0    0.0    0.0025 0.0062 0.0120 0.0198 0.0292 0.0398 0.0510 0.0625
				       0.0739 0.0847 0.0946 0.1036 0.1113 0.1177 0.1228 0.1266 0.1290 0.1301)))
		(snd-display ";moog output: ~A" vals))))
	(close-sound ind))

      (let ((gen (make-ssb-am 440.0))
	    (v0 (make-vct 10))
	    (gen1 (make-ssb-am 440.0))
	    (v1 (make-vct 10)))
	(print-and-check gen 
			 "ssb-am"
			 "ssb-am: shift: up, sin/cos: 439.999975 Hz (0.000000 radians), order: 40"
			 "ssbam: shift_up: true, sin: osc freq: 0.125379, phase: 0.000000, cos: osc freq: 0.125379, phase: 1.570796, delay: dly line[40,40 at 0,0 (local)]: [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], xscl: 0.000000, yscl: 0.000000, type: step, hilbert: flt order: 80, state (local): [0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000...(0: 0.000, 0: 0.000)], x: [-0.000 -0.001 -0.000 -0.002 -0.000 -0.002 -0.000 -0.003...(39: -0.636, 41: 0.636)], y: nil")
	(do ((i 0 (1+ i)))
	    ((= i 10))
	  (vct-set! v0 i (ssb-am gen 0.0)))
	(vct-map! v1 (lambda () (if (ssb-am? gen1) (ssb-am gen1 0.0) -1.0)))
	(if (not (vequal v0 v1)) (snd-display ";map ssb-am: ~A ~A" v0 v1))
	(if (not (ssb-am? gen)) (snd-display ";~A not ssb-am?" gen))
	(if (fneq (mus-phase gen) 1.253787) (snd-display ";ssb-am phase: ~F?" (mus-phase gen)))
	(if (fneq (mus-frequency gen) 440.0) (snd-display ";ssb-am frequency: ~F?" (mus-frequency gen)))
	(if (not (= (mus-order gen) 40)) (snd-display ";ssb-am order: ~F?" (mus-order gen)))
	(if (not (= (mus-cosines gen) 1)) (snd-display ";ssb-am cosines: ~D?" (mus-cosines gen)))
	(if (not (= (mus-length gen) 40)) (snd-display ";ssb-am length: ~D?" (mus-length gen)))
	(if (not (= (mus-interp-type gen) mus-interp-none)) (snd-display ";ssb-am interp type: ~D?" (mus-interp-type gen)))
	(if (fneq (mus-xcoeff gen 0) 0.0) (snd-display ";ssb-am xcoeff 0: ~A" (mus-xcoeff gen 0)))
	(if (fneq (mus-xcoeff gen 1) -.001) (snd-display ";ssb-am xcoeff 1: ~A" (mus-xcoeff gen 1)))
	)
      
      (test-gen-equal (make-ssb-am 440.0) (make-ssb-am 440.0) (make-ssb-am 500.0))

      (let ((o1 (make-ssb-am 400.0))
	    (o2 (make-ssb-am-1 400.0)))
	(call-with-current-continuation
	 (lambda (break)
	   (do ((i 0 (1+ i)))
	       ((= i 100))
	     (let* ((inval (sin (* .1 i)))
		    (o1o (ssb-am o1 inval))
		    (o2o (ssb-am-1 o2 inval)))
	       (if (fneq o1o o2o)
		   (begin
		     (snd-display ";ssb-am (up): ~A ~A at ~A" o1o o2o i)
		     (break))))))))
      
      (let ((o1 (make-ssb-am -100.0))
	    (o2 (make-ssb-am-1 -100.0)))
	(call-with-current-continuation
	 (lambda (break)
	   (do ((i 0 (1+ i)))
	       ((= i 100))
	     (let* ((inval (random 1.0))
		    (o1o (ssb-am o1 inval))
		    (o2o (ssb-am-1 o2 inval)))
	       (if (fneq o1o o2o)
		   (begin
		     (snd-display ";ssb-am (down): ~A ~A at ~A" o1o o2o i)
		     (break))))))))
      
      (let ((o1 (make-ssb-am 1000.0 100))
	    (o2 (make-ssb-am-1 1000.0 100)))
	(call-with-current-continuation
	 (lambda (break)
	   (do ((i 0 (1+ i)))
	       ((= i 100))
	     (let* ((inval (random 1.0))
		    (o1o (ssb-am o1 inval))
		    (o2o (ssb-am-1 o2 inval)))
	       (if (fneq o1o o2o)
		   (begin
		     (snd-display ";ssb-am (down): ~A ~A at ~A" o1o o2o i)
		     (break))))))))
      
      (let ((make-procs (list
			 make-all-pass make-asymmetric-fm make-average
			 make-comb (lambda () (make-convolve :filter (vct 0 1 2))) make-delay (lambda () (make-env '(0 1 1 0)))
			 (lambda () (make-filter :xcoeffs (vct 0 1 2))) (lambda () (make-fir-filter :xcoeffs (vct 0 1 2))) 
			 make-formant (lambda () (make-frame 3)) make-granulate
			 (lambda () (make-iir-filter :xcoeffs (vct 0 1 2))) make-locsig (lambda () (make-mixer 3 3)) 
			 make-notch make-one-pole make-one-zero make-oscil make-ppolar
			 make-pulse-train make-rand make-rand-interp make-sawtooth-wave
			 make-sine-summation make-square-wave make-src make-sum-of-cosines make-sum-of-sines make-table-lookup make-triangle-wave
			 make-two-pole make-two-zero make-wave-train make-waveshape make-zpolar make-phase-vocoder make-ssb-am
			 (lambda () (make-filter :ycoeffs (vct 0 1 2)))
			 (lambda () (make-filter :xcoeffs (vct 1 2 3) :ycoeffs (vct 0 1 2)))))
	    (run-procs (list all-pass asymmetric-fm average
			     comb convolve delay env 
			     filter fir-filter formant (lambda (gen ind) (frame-ref gen ind)) granulate
			     iir-filter (lambda (gen a) (locsig gen 0 a)) (lambda (gen a) (mixer-ref gen a 0)) notch one-pole one-zero oscil two-pole
			     pulse-train rand rand-interp sawtooth-wave
			     sine-summation square-wave (lambda (gen a) (src gen 0.0 a)) sum-of-cosines sum-of-sines table-lookup triangle-wave
			     two-pole two-zero wave-train waveshape two-zero phase-vocoder ssb-am
			     filter filter))
	    (ques-procs (list all-pass? asymmetric-fm? average?
			      comb? convolve? delay? env? 
			      filter? fir-filter? formant? frame? granulate?
			      iir-filter? locsig? mixer? notch? one-pole? one-zero? oscil? two-pole?
			      pulse-train? rand? rand-interp? sawtooth-wave?
			      sine-summation? square-wave? src? sum-of-cosines? sum-of-sines? table-lookup? triangle-wave?
			      two-pole? two-zero? wave-train? waveshape? two-zero? phase-vocoder? ssb-am?
			      filter? filter?))
	    (func-names (list 'all-pass 'asymmetric-fm 'average
			      'comb 'convolve 'delay 'env 
			      'filter-x 'fir-filter 'formant 'frame 'granulate
			      'iir-filter 'locsig 'mixer 'notch 'one-pole 'one-zero 'oscil 'two-pole
			      'pulse-train 'rand 'rand-interp 'sawtooth-wave
			      'sine-summation 'square-wave 'src 'sum-of-cosines 'sum-of-sines 'table-lookup 'triangle-wave
			      'two-pole 'two-zero 'wave-train 'waveshape 'two-zero 'phase-vocoder 'ssb-am
			      'filter-y 'filter-xy))
	    (gen-args (list 0.0 0.0 1.0
			    0.0 (lambda (dir) 0.0) 0.0 #f
			    0.0 0.0 0.0 0 (lambda (dir) 0.0)
			    0.0 0.0 0 0.0 0.0 0.0 0.0 0.0
			    0.0 0.0 0.0 0.0
			    0.0 0.0 (lambda (dir) 0.0) 0.0 0.0 0.0 0.0
			    0.0 0.0 0.0 0.0 0.0 (lambda (dir) 0.0) 0.0
			    0.0 0.0))
	    (generic-procs (list mus-a0 mus-a1 mus-a2 mus-b1 mus-b2 mus-bank mus-channel mus-channels mus-cosines mus-data
				 mus-feedback mus-feedforward mus-formant-radius mus-frequency mus-hop mus-increment mus-length
				 mus-location mus-mix mus-order mus-phase mus-ramp mus-random mus-run mus-scaler mus-xcoeffs
				 mus-ycoeffs))
	    (generic-names (list 'mus-a0 'mus-a1 'mus-a2 'mus-b1 'mus-b2 'mus-bank 'mus-channel 
				 'mus-channels 'mus-cosines 'mus-data
				 'mus-feedback 'mus-feedforward 'mus-formant-radius 'mus-frequency 'mus-hop 'mus-increment 'mus-length
				 'mus-location 'mus-mix 'mus-order 'mus-phase 'mus-ramp 'mus-random 'mus-run 'mus-scaler 'mus-xcoeffs
				 'mus-ycoeffs)))
	(for-each
	 (lambda (make run ques arg name)
	   (let ((gen (make)))
	     (if (not (ques gen)) (snd-display ";~A: ~A -> ~A?" name make gen))
	     (let ((tag (catch #t (lambda () (if arg (run gen arg) (run gen))) (lambda args args))))
	       (if (and (not (number? tag)) 
			(not (frame? tag)))
		   (snd-display ";~A: ~A ~A ~A: ~A" name run gen arg tag)))
	     (for-each
	      (lambda (func genname)
		(let ((tag (catch #t (lambda () (func gen)) (lambda args (car args)))))
		  (if (and (not (symbol? tag))
			   (procedure-with-setter? func)
			   (or (not (eq? genname 'mus-data))
			       (vct? tag)))
		      (let ((tag1 (catch #t (lambda () (set! (func gen) tag)) (lambda args (car args)))))
			(if (and (symbol? tag1)
				 (not (eq? tag1 'mus-error)))
			    (snd-display ";~A set ~A ~A ~A -> ~A" name genname gen tag tag1))))))
	      generic-procs generic-names)))
	 make-procs run-procs ques-procs gen-args func-names))
      
      (run-hook after-test-hook 8)
      ))



;;; ---------------- test 9: mix ----------------

(define (track-end id) (+ (track-position id) (track-frames id) -1))
(define-envelope xrmx '(0 0 1 1) 32.0)
(define-envelope xrmx1 '(0 0 1 1) 0.03)
(define-envelope xrmx2 '(0 0 1 1) 0.5)

(if (or full-test (= snd-test 9) (and keep-going (<= snd-test 9)))
    (begin
	  (run-hook before-test-hook 9)
      (do ((test-ctr 0 (1+ test-ctr)))
	  ((= test-ctr tests))
	(let ((new-index (new-sound "hiho.wave" mus-next mus-bshort 22050 1)))
	  (log-mem test-ctr)
	  (select-sound new-index)
	  (if (find-mix 0 new-index 0) (snd-display ";found non-existent mix? ~A" (find-mix 0 new-index 0)))
	  (let ((mix-id (mix "pistol.snd" 100)))
	    (if (not (mix? mix-id)) (snd-display ";~A not mix?" mix-id))
	    (view-mixes-dialog)
	    (let ((pos (mix-position mix-id))
		  (len (mix-frames mix-id))
		  (loc (mix-locked? mix-id))
		  (inv (mix-inverted? mix-id))
		  (anc (mix-tag-position mix-id))
		  (spd (mix-speed mix-id))
		  (trk (mix-track mix-id))
		  (snd (car (mix-home mix-id)))
		  (chn (cadr (mix-home mix-id)))
		  (chns (mix-chans mix-id))
		  (amp (mix-amp mix-id 0))
		  (mr (make-mix-sample-reader mix-id)))
	      (if (not (mix-sample-reader? mr)) (snd-display ";~A not mix-sample-reader?" mr))
	      (if (track-sample-reader? mr) (snd-display ";mix sample-reader: track ~A" mr))
	      (if (region-sample-reader? mr) (snd-display ";mix sample-reader: region ~A" mr))
	      (if (sample-reader? mr) (snd-display ";mix sample-reader: normal ~A" mr))
	      (if (not (= (sample-reader-position mr) 0)) (snd-display ";mix sample reader position: ~A" (sample-reader-position mr)))
	      (if (sample-reader-at-end? mr) (snd-display ";mix sample reader at end? ~A" mr))
	      (if (not (equal? (sample-reader-home mr) mix-id))
		  (snd-display ";~A home: ~A" mr (sample-reader-home mr)))
	      (let ((reader-string (format #f "~A" mr)))
		(if (not (string=? (substring reader-string 0 22) "#<mix-sample-reader mi"))
		    (snd-display ";mix sample reader actually got: [~S]" (substring reader-string 0 22))))
	      (let ((var (catch #t (lambda () (mix-amp mix-id 1234)) (lambda args args))))
		(if (not (eq? (car var) 'no-such-channel))
		    (snd-display ";mix-amp bad chan: ~A" var)))
	      (let ((var (catch #t (lambda () (set! (mix-amp mix-id 1234) .1)) (lambda args args))))
		(if (not (eq? (car var) 'no-such-channel))
		    (snd-display ";set mix-amp bad chan: ~A" var)))
	      (let ((var (catch #t (lambda () (set! (mix-amp-env mix-id 1234) '(0 0 1 1))) (lambda args args))))
		(if (not (eq? (car var) 'no-such-channel))
		    (snd-display ";set mix-amp-env bad chan: ~A" var)))
	      (do ((i 0 (1+ i)))
		  ((= i 99))
		(let ((mx (if (odd? i) (read-mix-sample mr) (read-mix-sample mr)))
		      (sx (sample (+ 100 i))))
		  (if (fneq mx sx) (snd-display ";read-mix-sample: ~A ~A?" mx sx))))
	      (let ((mx (mr))
		    (sx (sample 199)))
		(if (fneq mx sx) (snd-display ";mix-sample 100: ~A ~A?" mx sx)))
	      (free-sample-reader mr)
	      (if (not (= pos 100)) (snd-display ";mix-position: ~A?" pos))
	      (if (not (= len 41623)) (snd-display ";mix-frames: ~A?" len))
	      (if loc (snd-display ";mix-locked?: ~A?" loc))
	      (if inv (snd-display ";mix-inverted?: ~A?" inv))
	      (if (not (= anc 0)) (snd-display ";mix-tag-position: ~A?" anc))
	      (if (not (= trk 0)) (snd-display ";mix-track: ~A?" trk))
	      (if (not (= snd new-index)) (snd-display ";s mix-home: ~A?" snd))		
	      (if (not (= chn 0)) (snd-display ";c mix-home: ~A?" chn))
	      (if (not (= chns 1)) (snd-display ";mix-chans: ~A?" chn))
	      (if (fneq amp 1.0) (snd-display ";mix-amp: ~A?" amp))
	      (if (fneq spd 1.0) (snd-display ";mix-speed: ~A?" spd))
	      (catch 'mus-error
		     (lambda () (play-mix mix-id))
		     (lambda args (snd-display ";can't play mix")))
	      (let ((tag (catch #t
				(lambda () (set! (mix-track mix-id) -1))
				(lambda args (car args)))))
		(if (not (eq? tag 'out-of-range)) (snd-display ";set mix-track -1: ~A (~A)" tag (mix-track mix-id))))
	      (set! (mix-position mix-id) 200) 
	      (set! (mix-amp mix-id 0) 0.5) 
	      (set! (mix-speed mix-id) 2.0) 
	      (let ((trk (make-track mix-id)))
		(let ((tag (catch #t
				  (lambda () (play-track 1231233))
				  (lambda args (car args)))))
		  (if (not (eq? tag 'no-such-track)) (snd-display ";play-track bad track: ~A" tag)))
		(let ((tag (catch #t
				  (lambda () (play-track 1231233 #t))
				  (lambda args (car args)))))
		  (if (not (eq? tag 'no-such-track)) (snd-display ";play-track bad track #t: ~A" tag)))
		(let ((tag (catch #t
				  (lambda () (play-track 1231233 0))
				  (lambda args (car args)))))
		  (if (not (eq? tag 'no-such-track)) (snd-display ";play-track bad track index: ~A" tag)))
		(play-track trk))
	      (set! (mix-tag-position mix-id) 30) 
	      (set! (mix-amp-env mix-id 0) '(0.0 0.0 1.0 1.0)) 
	      (let ((val (mix-amp-env mix-id 0)))
		(set! (mix-amp-env mix-id 0) (mix-amp-env mix-id 0))
		(if (not (feql (mix-amp-env mix-id 0) val)) 
		    (snd-display ";set mix-amp-env to self: ~A ~A" val (mix-amp-env mix-id 0))))
	      (set! (mix-tag-y mix-id) 20) 
	      (let ((pos (mix-position mix-id))
		    (spd (mix-speed mix-id))
		    (trk (mix-track mix-id))
		    (amp (mix-amp mix-id 0))
		    (my (mix-tag-y mix-id))
		    (anc (mix-tag-position mix-id)))
		(if (not (= pos 200)) (snd-display ";set-mix-position: ~A?" pos))
		(if (not (= my 20)) (snd-display ";set-mix-tag-y: ~A?" my))
		(if (not (track? trk)) (snd-display ";set-mix-track: ~A?" trk))
		(if (fneq amp 0.5) (snd-display ";set-mix-amp: ~A?" amp))
		(if (fneq spd 2.0) (snd-display ";set-mix-speed: ~A?" spd))
		(if (not (= anc 30)) (snd-display ";set-mix-tag-position: ~A?" anc))
		(if (not (equal? (mix-amp-env mix-id 0) '(0.0 0.0 1.0 1.0))) (snd-display ";set-mix-amp-env: ~A?" (mix-amp-env mix-id 0))))
	      ))
	  (mix-vct (make-vct 3 .1) 100)
	  (set! (cursor) 0)
	  (let ((nid (forward-mix)))
	    (if (or (not (mix? nid))
		    (not (= (cursor) (mix-position nid))))
		(snd-display ";forward-mix ~A ~A ~A?" nid (cursor) (and (mix? nid) (mix-position nid))))
	    (let ((nid1 (forward-mix 2)))
	      (if (or (not (mix? nid1))
		      (not (= (cursor) (mix-position nid1))))
		  (snd-display ";forward-mix(2) ~A ~A ~A ~A ~A?" nid nid1 (cursor) (and (mix? nid1) (mix-position nid1)) (mixes)))
	      (set! nid1 (backward-mix))
	      (if (or (not (mix? nid1))
		      (not (= (cursor) (mix-position nid1))))
		  (snd-display ";backward-mix(2) ~A ~A ~A?" nid1 (cursor) (and (mix? nid1) (mix-position nid1))))))
	  (let ((nid (find-mix 100)))
	    (if (or (not (mix? nid))
		    (not (= (mix-position nid) 100)))
		(snd-display ";find-mix(100): ~A ~A ~A?" nid (and (mix? nid) (mix-position nid)) (map mix-position (mixes new-index 0)))))
	  (let ((nid (find-mix 200)))
	    (if (or (not (mix? nid))
		    (not (= (mix-position nid) 200)))
		(snd-display ";find-mix(200): ~A ~A?" nid (and (mix? nid) (mix-position nid)))))
	  (let ((mix-id (mix "oboe.snd" 100)))
	    (set! (mix-waveform-height) 40)
	    (set! (mix-property :hiho mix-id) 123)
	    (if (not (= (mix-property :hiho mix-id) 123)) (snd-display ";mix-property: ~A" (mix-property mix-id)))
	    (if (mix-property :not-there mix-id) (snd-display ";mix-not-property: ~A" (mix-property :not-there mix-id)))
	    (update-time-graph)
	    (set! (mix-waveform-height) 20))
	  ;; now track tests (mix.scm)
	  (revert-sound new-index)
	  (let ((mix-ids (make-vector 6))
		(trk (make-track)))
	    (do ((i 0 (1+ i)))
		((= i 6))
	      (vector-set! mix-ids i (mix "oboe.snd" (* i 1000))))
	    (set! (mix-track (vector-ref mix-ids 0)) trk)
	    (set! (mix-track (vector-ref mix-ids 2)) trk)
	    (set! (mix-track (vector-ref mix-ids 4)) trk)
	    (if (not (= (track-position trk) 0)) (snd-display ";track-position: ~D?" (track-position trk)))
	    (let* ((mr (make-track-sample-reader trk))
		   (reader-string (format #f "~A" mr)))
	      (if (not (string=? (substring reader-string 0 24) "#<track-sample-reader tr"))
		  (snd-display ";track sample reader actually got: [~S]" (substring reader-string 0 24)))
	      (if (not (= (sample-reader-position mr) 0)) (snd-display ";track sample reader position: ~A" (sample-reader-position mr)))
	      (if (sample-reader-at-end? mr) (snd-display ";track sample reader at end? ~A" mr))
	      (if (not (equal? (sample-reader-home mr) (list trk 0))) 
		  (snd-display ";track ~A home: ~A" mr (sample-reader-home mr)))
	      (if (mix-sample-reader? mr) (snd-display ";track sample-reader: mix ~A" mr))
	      (if (region-sample-reader? mr) (snd-display ";track sample-reader: region ~A" mr))
	      (if (sample-reader? mr) (snd-display ";track sample-reader: normal ~A" mr))
	      (free-sample-reader mr))
	    (let ((curend (track-end trk))
		  (curframes (track-frames trk))
		  (curmixpos (map mix-position (track trk)))
		  (curmixframes (map mix-frames (track trk))))
	      (set! (track-position trk) 500)
	      (if (not (= (track-position trk) 500)) (snd-display ";set track-position: ~D?" (track-position trk)))
	      (if (not (= (mix-position (vector-ref mix-ids 0)) 500)) (snd-display ";track-position ~D = ~D?" 0 (mix-position (vector-ref mix-ids 0))))
	      (if (not (= (mix-position (vector-ref mix-ids 1)) 1000)) (snd-display ";track-position ~D = ~D?" 1 (mix-position (vector-ref mix-ids 1))))
	      (if (not (= (mix-position (vector-ref mix-ids 4)) 4500)) (snd-display ";track-position ~D = ~D?" 4 (mix-position (vector-ref mix-ids 4))))
	      (if (not (< (abs (- (track-end trk) (+ curend 500))) 2))
		  (snd-display ";track-end: ~D (cur+500: ~D, ~D, ~D + ~D -> ~D, ~A ~A ~A ~A ~A)?" 
			       (track-end trk) (+ curend 500) curframes
			       (track-position trk) (track-frames trk)
			       (+ (track-position trk) (track-frames trk) -1)
			       (track trk) (map mix-frames (track trk)) (map mix-position (track trk))
			       curmixpos curmixframes)))
	    (set! (track-amp trk) .5)
	    (if (fneq (track-amp trk) .5) (snd-display ";set track-amp ~F?" (track-amp trk)))
	    (set! (track-amp trk) (+ (track-amp trk) .25))
	    (if (fneq (track-amp trk) .75) (snd-display ";incf track-amp ~F?" (track-amp trk)))
	    (transpose-track trk 12)
	    (if (fneq (track-speed trk) 2.0)
		(snd-display ";transpose-track: ~F?" (track-speed trk)))
	    (retempo-track trk 2.0)
	    (if (not (= (track-frames trk) (/ (+ 4000 50828) 2)))
		(snd-display ";track-tempo: ~D (~D, ~A ~A)?" 
			     (track-frames trk) (/ (+ 4000 50828) 2)
			     (track trk) (map mix-frames (track trk))))
	    (catch 'no-such-color
		   (lambda () (set! (track-color trk) (make-color .8 .8 .8)))
		   (lambda args args))
	    (let ((trk2 (make-track)))
	      (set! (mix-track (vector-ref mix-ids 1)) trk2)
	      (set! (mix-track (vector-ref mix-ids 3)) trk2)
	      (catch 'no-such-color 
		     (lambda () (set! (track-color trk2) (make-color .2 .8 0)))
		     (lambda args args))
	      (let ((t2 (track->vct trk2))
		    (t3 (mix->vct (vector-ref mix-ids 5))))
		(if (or (fneq (vct-ref t2 1000) (vct-ref t3 1000))
			(fneq (vct-ref t3 1000) 0.0328369))
		    (snd-display ";track->vct: ~F, mix->vct: ~F (.0328369)?" (vct-ref t2 1000) (vct-ref t3 1000))))
	      (set! (track-amp-env trk) '(0 0 1 1))
	      (play-and-wait)))
	  (let ((v1 (envelope-interp 1.0 '(0 0 2.0 1.0)))
		(v2 (envelope-interp 1.0 '(0 0.0 1 1.0 2 0.0)))
		(v3 (envelope-interp 2.0 '(0 0.0 1 1.0)))
		(v4 (envelope-interp 0.0 '(1 .5 2 0))))
	    (if (fneq v1 0.5) (snd-display ";envelope-interp(1): ~F (0.5)?" v1))
	    (if (fneq v2 1.0) (snd-display ";envelope-interp(2): ~F (1.0)?" v2))
	    (if (fneq v3 1.0) (snd-display ";envelope-interp(3): ~F (1.0)?" v3))
	    (if (fneq v4 0.5) (snd-display ";envelope-interp(4): ~F (0.5)?" v4)))
	  (let ((v1 (envelope-interp 0.0 '(-1 0  0 1  1 -1)))
		(v2 (envelope-interp -0.5 '(-1 0  0 1  1 -1)))
		(v3 (envelope-interp -0.5 '(-1 -1  0 1  1 -1)))
		(v4 (envelope-interp -0.5 '(-1 -1  1 1)))
		(v5 (envelope-interp -1.5 '(-1 -1  1 1)))
		(v6 (envelope-interp 1.5 '(-1 -1  1 1))))
	    (if (fneq v1 1.0) (snd-display ";envelope-interp(1a): ~A" v1))
	    (if (fneq v2 0.5) (snd-display ";envelope-interp(2a): ~A" v2))
	    (if (fneq v3 0.0) (snd-display ";envelope-interp(3a): ~A" v3))
	    (if (fneq v4 -0.5) (snd-display ";envelope-interp(4a): ~A" v4))
	    (if (fneq v5 -1.0) (snd-display ";envelope-interp(5a): ~A" v5))
	    (if (fneq v6 1.0) (snd-display ";envelope-interp(6a): ~A" v6)))
	  (let ((v1 (multiply-envelopes '(0.0 0.0 2.0 0.5) '(0.0 0.0 1.0 2.0 2.0 1.0)))
		(v2 (window-envelope 1.0 3.0 '(0.0 0.0 5.0 1.0))))
	    (if (not (feql v1 (list 0.0 0.0 0.5 0.5 1.0 0.5))) (snd-display ";multiply-envelopes: ~A?" v1))
	    (if (not (feql v2 (list 1.0 0.2 3.0 0.6))) (snd-display ";window-envelope: ~A?" v2)))
	  (close-sound new-index))
	(let ((index (open-sound "pistol.snd"))
	      (data (samples->vct 0 100)))
	  (convolve-with "oboe.snd" #f)
	  (let ((scl (maxamp)))
	    (convolve-with "oboe.snd" scl index 0 0)
	    (if (fneq (maxamp) scl) 
		(snd-display ";convolve-with amps: ~A ~A" (maxamp) scl)
		(let ((preader (make-sample-reader 0 index 0 1 1))
		      (reader (make-sample-reader 0))
		      (len (frames)))
		  (call-with-current-continuation 
		   (lambda (break) 
		     (do ((i 0 (1+ i))) 
			 ((= i len))
		       (let ((val0 (preader))
			     (val1 (reader)))
			 (if (fneq val0 val1)
			     (begin
			       (snd-display ";convolve-with amps at: ~A: ~A ~A" i val0 val1)
			       (break))))))))))
	  (close-sound index)
	  (let ((reader (make-sample-reader 0 "pistol.snd")))
	    (do ((i 0 (1+ i)))
		((= i 10))
	      (if (fneq (vct-ref data i) (next-sample reader))
		  (snd-display ";external reader trouble")))
	    (free-sample-reader reader)))
	)
      (dismiss-all-dialogs)
      
      (let ((ind (new-sound "new.snd"))
	    (mxs (make-vector 10))
	    (trk33 (make-track)))
	(call-with-current-continuation
	 (lambda (quit)
	   (do ((i 0 (1+ i)))
	       ((= i 10))
	     (let ((v (make-vct 1)))
	       (vct-set! v 0 (* i .05))
	       (vector-set! mxs i (mix-vct v i ind 0))
	       (if (not (mix? (vector-ref mxs i)))
		   (quit (snd-display ";mix-vct at ~A failed? " i))
		   (set! (mix-track (vector-ref mxs i)) trk33))))
	   (let ((tr (make-track-sample-reader trk33))
		 (tr1 (make-track-sample-reader trk33 #t 5)))
	     (do ((i 0 (1+ i)))
		 ((= i 10))
	       (let ((val (if (odd? i) (read-track-sample tr) (read-track-sample tr))))
		 (if (fneq val (* i .05))
		     (begin
		       (close-sound ind)
		       (quit (snd-display ";read track at ~A: ~A?" i val))))))
	     (let ((val (read-track-sample tr1)))
	       (if (fneq val (* .05 5))
		   (snd-display ";track-sample-reader with beg: ~A ~A ~A" val (* 5 .05) tr1)))
	     (free-sample-reader tr)
	     (free-sample-reader tr1)
	     (save-sound ind)
	     (if (not (mix? (vector-ref mxs 0)))
		 (snd-display ";saved mixes not re-activated?"))
	     (close-sound ind)))))
      
      (let* ((ind (open-sound "oboe.snd"))
	     (open-readers (make-vector 100 #f))
	     (mix1 (mix-vct (vct 0.1 0.2 0.3) 120 ind 0 #t "origin!"))
	     (mix2 (mix-vct (vct 0.1 0.2 0.3) 1200 ind 0 #t))
	     (mix3 (mix-vct (vct 0.1 0.2 0.3) 12000 ind 0 #t))
	     (trk123 (make-track))
	     (reg1 (make-region 200 300 ind 0)))
	(set! (mix-track mix1) trk123)
	(set! (mix-track mix2) trk123)
	(set! (mix-track mix3) trk123)
	(do ((i 0 (1+ i)))
	    ((= i sample-reader-tests))
	  (let* ((cur (random 4))
		 (r (random 100)))
	    (if (= cur 0)
		(begin
		  (vector-set! open-readers r (make-sample-reader (random 30000) ind 0))
		  (if (not (sample-reader? (vector-ref open-readers r))) (snd-display ";sample-reader? ~A?" (vector-ref open-readers r)))
		  (next-sample (vector-ref open-readers r))
		  (if (not (equal? (list ind 0) (sample-reader-home (vector-ref open-readers r))))
		      (snd-display ";sample-reader-home ~A?" (sample-reader-home (vector-ref open-readers r)))))
		(if (= cur 1)
		    (begin
		      (vector-set! open-readers r (make-region-sample-reader (random 90) reg1))
		      (if (not (region-sample-reader? (vector-ref open-readers r))) (snd-display ";region-sample-reader? ~A?" (vector-ref open-readers r)))
		      (next-sample (vector-ref open-readers r)))
		    (if (= cur 2)
			(begin
			  (vector-set! open-readers r (make-mix-sample-reader mix1))
			  (if (not (mix-sample-reader? (vector-ref open-readers r))) (snd-display ";mix-sample-reader? ~A?" (vector-ref open-readers r)))
			  (let ((val (read-mix-sample (vector-ref open-readers r))))
			    (if (fneq val 0.1) (snd-display ";read-mix-sample: ~A" val))))
			(begin
			  (vector-set! open-readers r (make-track-sample-reader trk123))
			  (if (not (track-sample-reader? (vector-ref open-readers r))) (snd-display ";track-sample-reader? ~A?" (vector-ref open-readers r)))
			  (let ((val (read-track-sample (vector-ref open-readers r))))
			    (if (fneq val 0.1) (snd-display ";read-track-sample: ~A" val)))))))
	    (if (> (random 1.0) .25)
		(let ((rr (random 100)))
		  (if (vector-ref open-readers rr)
		      (if (sample-reader? (vector-ref open-readers rr)) (free-sample-reader (vector-ref open-readers rr))
			  (if (mix-sample-reader? (vector-ref open-readers rr)) (free-sample-reader (vector-ref open-readers rr))
			      (if (track-sample-reader? (vector-ref open-readers rr)) (free-sample-reader (vector-ref open-readers rr))))))
		  (vector-set! open-readers rr #f)))))
	(do ((i 0 (1+ i)))
	    ((= i 100))
	  (vector-set! open-readers i #f))
	(close-sound ind)
	(gc))
      
      (let ((id (open-sound "oboe.snd")))
	(make-selection 1000 2000 id 0)
	(let ((mix-id (mix-selection 3000 id 0)))
	  (set! (mix-amp mix-id 0) .5)
	  (if (fneq (mix-amp mix-id 0) .5)
	      (snd-display ";mix-amp .5: ~A" (mix-amp mix-id 0)))
	  (scale-by .5)
	  (if (not (mix-locked? mix-id))
	      (snd-display ";mix not locked? ~A" mix-id))
	  (let ((var (catch #t (lambda () (set! (mix-amp mix-id 0) 1.0)) (lambda args args))))
	    (if (not (eq? (car var) 'no-such-mix))
		(snd-display ";set locked mix amp: ~A" var)))
	  (let ((var (catch #t (lambda () (set! (mix-position mix-id) 10)) (lambda args args))))
	    (if (not (eq? (car var) 'no-such-mix))
		(snd-display ";set locked mix position: ~A" var)))
	  (let ((var (catch #t (lambda () (set! (mix-speed mix-id) 1.5)) (lambda args args))))
	    (if (not (eq? (car var) 'no-such-mix))
		(snd-display ";set locked mix speed: ~A" var)))
	  (let ((var (catch #t (lambda () (set! (mix-amp-env mix-id 0) '(0 0 1 1))) (lambda args args))))
	    (if (not (eq? (car var) 'no-such-mix))
		(snd-display ";set locked mix amp env: ~A" var)))
	  (undo)
	  (if (mix-locked? mix-id)
	      (snd-display ";undo locked mix: ~A" mix-id)
	      (begin
		(set! (mix-position mix-id) 10)
		(if (not (= (mix-position mix-id) 10))
		    (snd-display ";mix-position 10: ~A" (mix-position mix-id)))))
	  (close-sound id)))
      (set! (print-length) 30)
      (let ((index (new-sound "test.snd"))
	    (v1 (make-vct 1))
	    (v2 (make-vct 2))
	    (v3 (make-vct 3)))
	(vct-fill! v1 .1)
	(vct-fill! v2 .2)
	(vct-fill! v3 .3)
	(let ((id1 (map (lambda (start)
			  (mix-vct v1 start))
			(list 0 10 20)))
	      (id2 (map (lambda (start)
			  (mix-vct v2 start))
			(list 1 12 23)))
	      (id3 (map (lambda (start)
			  (mix-vct v3 start))
			(list 2 14 26)))
	      (trk1 (make-track)))
	  (if (not (vequal (channel->vct)
			   (vct .1 .2 .5 .3 .3  0 0  0 0 0  
				.1 0  .2 .2 .3 .3 .3 0 0 0
				.1 0  0  .2 .2  0 .3 .3 .3)))
	      (snd-display ";mix tests off to a bad start: ~A" (channel->vct)))
	  (if (not (vequal (mix->vct (car id2)) (vct .2 .2)))
	      (snd-display ";mix->vct of .2: ~A" (mix->vct (car id2))))
	  (set! (mix-track (car id1)) trk1)
	  (let ((tr1 trk1))
	    (if (not (equal? (track tr1) (list (car id1))))
		(snd-display ";1 track->~A ~A" (track tr1) (list (car id1))))
	    (if (not (= (track-position tr1) (mix-position (car id1))))
		(snd-display ";1 track-position ~A ~A (~A)" tr1 (track-position tr1) (mix-position (car id1))))
	    (if (not (= (track-frames tr1) (mix-frames (car id1))))
		(snd-display ";1 track-frames ~A frames: ~A (mix frames: ~A)" (track tr1) (track-frames tr1) (mix-frames (car id1))))
	    (if (not (= (track-end tr1) (+ (mix-position (car id1)) (mix-frames (car id1)) -1)))
		(snd-display ";1 track-end ~A ~A ~A" (track-end tr1) (mix-position (car id1)) (mix-frames (car id1))))
	    (if (fneq (track-amp tr1) (mix-amp (car id1) 0))
		(snd-display ";1 track-amp: ~A ~A" (track-amp tr1) (mix-amp (car id1) 0)))
	    (if (fneq (track-speed tr1) (mix-speed (car id1)))
		(snd-display ";1 track-speed: ~A ~A" (track-speed tr1) (mix-speed (car id1))))
	    (if (not (vequal (track->vct tr1) (mix->vct (car id1))))
		(snd-display ";1 track->vct ~A ~A" (track->vct tr1) (mix->vct (car id1))))
	    (set! (track-amp tr1) 0.0)
	    (if (not (vequal (channel->vct)
			     (vct 0 .2 .5 .3 .3  0 0  0 0 0  
				  .1 0  .2 .2 .3 .3 .3 0 0 0
				  .1 0  0  .2 .2  0 .3 .3 .3)))
		(snd-display ";first mix deleted: ~A" (channel->vct)))
	    (undo)
	    (if (fneq (mix-amp (car id1) 0) 1.0)
		(snd-display ";1 undo delete-track amp: ~A" (mix-amp (car id1) 0)))
	    (set! (track-amp tr1) 2.0)
	    (if (not (vequal (channel->vct)
			     (vct .2 .2 .5 .3 .3  0 0  0 0 0  
				  .1 0  .2 .2 .3 .3 .3 0 0 0
				  .1 0  0  .2 .2  0 .3 .3 .3)))
		(snd-display ";1 set track-amp: ~A" (channel->vct)))
	    (set! (track-position tr1) 8)
	    (if (not (= (track-position tr1) 8))
		(snd-display ";moved track 1: ~A" (track-position tr1)))
	    (if (not (vequal (channel->vct)
			     (vct 0 .2 .5 .3 .3  0 0  0 .2 0  
				  .1 0  .2 .2 .3 .3 .3 0 0 0
				  .1 0  0  .2 .2  0 .3 .3 .3)))
		(snd-display ";1 set track-position 8: ~A" (channel->vct)))
	    (reverse-track tr1)
	    (if (not (vequal (channel->vct)
			     (vct 0 .2 .5 .3 .3  0 0  0 .2 0  
				  .1 0  .2 .2 .3 .3 .3 0 0 0
				  .1 0  0  .2 .2  0 .3 .3 .3)))
		(snd-display ";1 reverse track: ~A" (channel->vct)))
	    
	    (let* ((trk2 (make-track (cadr id1) (cadr id2) (cadr id3)))
		   (tr2 trk2))
	      (if (not (= (track-position tr2) (mix-position (cadr id1))))
		  (snd-display ";2 track-position ~A ~A (~A)" tr2 (track-position tr2) (mix-position (cadr id1))))
	      (set! (track-amp tr2) 2.0)
	      (if (not (vequal (channel->vct)
			       (vct 0 .2 .5 .3 .3  0 0  0 .2 0  
				    .2 0  .4 .4 .6 .6 .6 0 0 0
				    .1 0  0  .2 .2  0 .3 .3 .3)))
		  (snd-display ";2 set track-amp: ~A" (channel->vct)))
	      (set! (track-position tr2) (- (track-position tr2) 1))
	      (revert-sound index)
	      
	      (set! id1 (map (lambda (start)
			       (mix-vct v1 start))
			     (list 0 10 20)))
	      (set! id2 (map (lambda (start)
			       (mix-vct v2 start))
			     (list 1 12 23)))
	      (set! id3 (map (lambda (start)
			       (mix-vct v3 start))
			     (list 2 14 26)))
	      (if (not (vequal (channel->vct)
			       (vct .1 .2 .5 .3 .3  0 0  0 0 0  
				    .1 0  .2 .2 .3 .3 .3 0 0 0
				    .1 0  0  .2 .2  0 .3 .3 .3)))
		  (snd-display ";mix tests 2nd start: ~A" (channel->vct)))
	      (set! tr1 (apply make-track id1))
	      (set! tr2 (apply make-track id3))
	      (let ((old-pos (map mix-position (track tr1))))
		(if (not (equal? old-pos (map mix-position id1)))
		    (snd-display ";old-pos: ~A ~A" old-pos (map mix-position id1)))
		(retempo-track tr1 2) ; tempo > 1 is faster
		(if (not (vequal (channel->vct)
				 (vct .1 .2 .5 .3 .3  .1 0  0 0 0  
				      .1 0  .2 .2 .3 .3 .3 0 0 0
				      0 0  0  .2 .2  0 .3 .3 .3)))
		    (snd-display ";3 track-tempo .5: ~A -> ~A, ~A" old-pos (map mix-position (track tr1)) (channel->vct))))
	      (set! (track-amp tr1) 0.0)
	      (if (not (vequal (channel->vct)
			       (vct 0 .2 .5 .3 .3  0 0  0 0 0  
				    0 0  .2 .2 .3 .3 .3 0 0 0
				    0 0  0  .2 .2  0 .3 .3 .3)))
		  (snd-display ";3 track-amp 0: ~A" (channel->vct)))
	      (delete-all-mixes)
	      ))
	  (close-sound index)
	  ))
      
      (let* ((ind (open-sound "2.snd"))
	     (md (mix "1a.snd" 1000 0 ind 1 #t)))
	(if (fneq (maxamp ind 1) .1665) (snd-display ";maxamp after mix into chan 2: ~A" (maxamp ind 1)))
	(set! (mix-amp md 0) 0.0)
	(if (or (not (equal? (edits ind 0) (list 0 0)))
		(not (equal? (edits ind 1) (list 2 0))))
	    (snd-display ";mix into chan2 zeroed: ~A ~A" (edits ind 0) (edits ind 1)))
	(if (fneq (maxamp ind 1) .066) (snd-display ";maxamp after mix zeroed into chan 2: ~A" (maxamp ind 1)))
	(set! (mix-amp md 0) 0.5)
	(if (fneq (maxamp ind 1) .116) (snd-display ";maxamp after mix 0.5 into chan 2: ~A" (maxamp ind 1)))
	(set! (mix-speed md) 2.0)
	(if (fneq (/ (mix-frames md) (mus-sound-frames "1a.snd")) 0.5)
	    (snd-display ";mix srate chan 2: ~A ~A" (mix-frames md) (mus-sound-frames "1a.snd")))
	(update-time-graph)
	(set! (mix-speed md) 0.5)
	(update-time-graph)
	(set! (mix-amp md 0) 1.0)
	(if (fneq (maxamp ind 1) .166) (snd-display ";non-sync mix-speed: ~A" (maxamp ind 1)))
	(set! (mix-amp-env md 0) '(0 0 1 1 2 0))
	(update-time-graph)
	(set! (mix-speed md) 1.0)
	(update-time-graph)
	(revert-sound ind)
	(set! (sync ind) 1)
	(let ((m0 (maxamp ind 0))
	      (m1 (maxamp ind 1))
	      (len (frames ind 0)))
	  (set! md (mix "2.snd")) ; should double both chans, no len change
	  (if (or (not (= (frames ind 0) len))
		  (fneq (maxamp ind 0) (* 2 m0))
		  (fneq (maxamp ind 1) (* 2 m1)))
	      (snd-display ";mix twice syncd: 0: ~A -> ~A, m1: ~A -> ~A, len: ~A -> ~A"
			   m0 (maxamp ind 0) m1 (maxamp ind 1) len (frames ind 0)))
	  (if (and (provided? 'snd-motif)
		   (provided? 'xm))
	      (begin
		(let ((wid (view-mixes-dialog)))
		  (if (not (equal? wid (list-ref (dialog-widgets) 16)))
		      (snd-display ";view-mixes-dialog -> ~A ~A" wid (list-ref (dialog-widgets) 16))))
		(let* ((mixd (list-ref (dialog-widgets) 16))
		       (spdscr (find-child mixd "mix-speed"))
		       (dragged #f))
		  (add-hook! mix-drag-hook (lambda (n) (set! dragged n)))
		  (XtCallCallbacks spdscr XmNvalueChangedCallback
				   (let ((cb (XmScrollBarCallbackStruct)))
				     (set! (.value cb) 650)
				     (set! (.event cb) (XEvent))
				     cb))
		  (let* ((xy (mix-tag-xy md))
			 (cwid (car (channel-widgets ind 0)))
			 (x (+ (car xy) 1))
			 (y (- (cadr xy) 2))
			 (pos (mix-position md)))
		    (focus-widget cwid)
		    (drag-event cwid 1 0 x y (+ x 50) y) (force-event)
		    (XtUnmanageChild mixd)
		    (reset-hook! mix-drag-hook)
		    ))))
	  (reset-hook! mix-release-hook)
	  (close-sound ind)))
      
      (let ((ind (new-sound "fmv.snd" mus-next mus-bshort 22050 1 "mix tests")))
	(insert-silence 0 20 ind)
	(let ((indout (new-sound "test.snd" mus-next mus-bshort 22050 1 "mix tests")))
	  (insert-silence 0 10 indout)
	  (set! (sample 2 indout 0) .5)
	  (set! (sample 5 indout 0) .25)
	  (save-sound indout)
	  (close-sound indout))
	(let ((tag (mix "test.snd")))
	  (let ((samps (channel->vct 0 20))
		(v (make-vct 20 0.0)))
	    (vct-set! v 2 .5)
	    (vct-set! v 5 .25)
	    (if (not (vequal samps v))
		(snd-display ";mix 1->1: ~A ~A" samps v)))
	  (if (not (mix? tag)) (snd-display ";mix 1->1 tag: ~A" tag))
	  (undo))
	(let ((tag (mix "test.snd" 5)))
	  (let ((samps (channel->vct 0 20))
		(v (make-vct 20 0.0)))
	    (vct-set! v 7 .5)
	    (vct-set! v 10 .25)
	    (if (not (vequal samps v))
		(snd-display ";mix 1->1 at 5: ~A ~A" samps v)))
	  (if (not (mix? tag)) (snd-display ";mix 1->1 at 5 tag: ~A" tag))
	  (undo))
	(let ((tag (mix "test.snd" 0 0 ind 0 #f)))
	  (let ((samps (channel->vct 0 20))
		(v (make-vct 20 0.0)))
	    (vct-set! v 2 .5)
	    (vct-set! v 5 .25)
	    (if (not (vequal samps v))
		(snd-display ";mix 1->1 at 0 #f: ~A ~A" samps v)))
	  (if (mix? tag) (snd-display ";mix 1->1 at 5 #f tag: ~A" tag))
	  (undo))
	(let ((indout (new-sound "test.snd" mus-next mus-bshort 22050 2 "mix tests")))
	  (insert-silence 0 10 indout 0)
	  (insert-silence 0 10 indout 1)
	  (set! (sample 2 indout 0) .5)
	  (set! (sample 5 indout 0) .25)
	  (set! (sample 2 indout 1) .95)
	  (set! (sample 5 indout 1) .125)
	  (save-sound indout)
	  (close-sound indout))
	(let ((tag (mix "test.snd" 0 1)))
	  (let ((samps (channel->vct 0 20))
		(v (make-vct 20 0.0)))
	    (vct-set! v 2 .95)
	    (vct-set! v 5 .125)
	    (if (not (vequal samps v))
		(snd-display ";mix 2->1: ~A ~A" samps v)))
	  (if (not (mix? tag)) (snd-display ";mix 2->1 tag: ~A" tag))
	  (undo))
	(let ((tag (mix "test.snd" 5 1)))
	  (let ((samps (channel->vct 0 20))
		(v (make-vct 20 0.0)))
	    (vct-set! v 7 .95)
	    (vct-set! v 10 .125)
	    (if (not (vequal samps v))
		(snd-display ";mix 2->1 at 5: ~A ~A" samps v)))
	  (if (not (mix? tag)) (snd-display ";mix 2->1 at 5 tag: ~A" tag))
	  (undo))
	(close-sound ind)
	(set! ind (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "mix tests"))
	(insert-silence 0 20 ind 0)
	(insert-silence 0 20 ind 1)
	(let ((tag (mix "test.snd")))
	  (let ((samps0 (channel->vct 0 20 ind 0))
		(samps1 (channel->vct 0 20 ind 1))
		(v (make-vct 20 0.0)))
	    (vct-set! v 2 .95) ; new version of mix auto-syncs in this case
	    (vct-set! v 5 .125)
	    (if (not (vequal samps1 v))
		(snd-display ";mix 1->1 (2): ~A ~A" samps1 v))
	    (vct-set! v 2 .5)
	    (vct-set! v 5 .25)
	    (if (not (vequal samps0 v))
		(snd-display ";mix 1->1 (3): ~A ~A" samps0 v)))
	  (if (not (mix? tag)) (snd-display ";mix 1->1 tag: ~A" tag))
	  (undo 1 ind 0)
	  (undo 1 ind 1))
	(let ((tag (mix "test.snd" 0 1 ind 1 #f))) ; samp:0, in-chan: 1
	  (let ((samps0 (channel->vct 0 20 ind 0))
		(samps1 (channel->vct 0 20 ind 1))
		(v (make-vct 20 0.0)))
	    (if (not (vequal samps0 v))
		(snd-display ";mix 1->1 (4): ~A ~A" samps0 v))
	    (vct-set! v 2 .95)
	    (vct-set! v 5 .125)
	    (if (not (vequal samps1 v))
		(snd-display ";mix 1->1 (5): ~A ~A" samps1 v)))
	  (if (mix? tag) (snd-display ";mix 1->1 tag (5): ~A" tag))
	  (undo 1 ind 1))
	(set! (sync ind) 1)
	(let ((tag (mix "test.snd")))
	  (let ((samps0 (channel->vct 0 20 ind 0))
		(samps1 (channel->vct 0 20 ind 1))
		(v (make-vct 20 0.0)))
	    (vct-set! v 2 .5)
	    (vct-set! v 5 .25)
	    (if (not (vequal samps0 v))
		(snd-display ";mix 1->1 (6): ~A ~A" samps0 v))
	    (vct-set! v 2 .95)
	    (vct-set! v 5 .125)
	    (if (not (vequal samps1 v))
		(snd-display ";mix 1->1 (7): ~A ~A" samps1 v)))
	  (undo))
	(set! (cursor ind) 5)
	(let ((tag (mix "test.snd")))
	  (let ((samps0 (channel->vct 0 20 ind 0))
		(samps1 (channel->vct 0 20 ind 1))
		(v (make-vct 20 0.0)))
	    (vct-set! v 7 .5)
	    (vct-set! v 10 .25)
	    (if (not (vequal samps0 v))
		(snd-display ";mix 1->1 (8): ~A ~A" samps0 v))
	    (vct-set! v 7 .95)
	    (vct-set! v 10 .125)
	    (if (not (vequal samps1 v))
		(snd-display ";mix 1->1 (9): ~A ~A" samps1 v)))
	  (undo))
	(close-sound ind))
      (delete-file "test.snd")
      (delete-file "fmv.snd")
      
      ;; check ripple_mixes
      (let* ((ind (open-sound "oboe.snd"))
	     (data (channel->vct 100 100))
	     (m1 (mix-vct data 321 ind 0 #t))
	     (m2 (mix-vct data 123 ind 0 #t)))
	(set! (mix-position m1) 500)
	(if (not (= (mix-position m1) 500)) (snd-display ";mix-position m1[0]: ~A" (mix-position m1)))
	(if (not (= (mix-position m2) 123)) (snd-display ";mix-position m2[0]: ~A" (mix-position m2)))
	(undo)
	(set! (mix-position m2) 500)
	(if (not (= (mix-position m2) 500)) (snd-display ";mix-position m2[1]: ~A" (mix-position m2)))
	(if (not (= (mix-position m1) 321)) (snd-display ";mix-position m1[1]: ~A" (mix-position m1)))
	(undo)
	(insert-silence 0 100)
	(if (not (= (mix-position m1) (+ 100 321))) (snd-display ";mix-position m1[2]: ~A" (mix-position m1)))
	(if (not (= (mix-position m2) (+ 100 123))) (snd-display ";mix-position m2[2]: ~A" (mix-position m2)))
	(delete-samples 0 50)
	(if (not (= (mix-position m1) (+ 50 321))) (snd-display ";mix-position m1[3]: ~A" (mix-position m1)))
	(if (not (= (mix-position m2) (+ 50 123))) (snd-display ";mix-position m2[3]: ~A" (mix-position m2)))
	(undo 2)
	(let ((ntrack (make-track)))
	  (set! (mix-track m1) ntrack)
	  (set! (mix-track m2) ntrack)
	  (reverse-track ntrack)
	  (if (not (= (mix-position m1) 123)) (snd-display ";mix-position m1[4]: ~A" (mix-position m1)))
	  (if (not (= (mix-position m2) 321)) (snd-display ";mix-position m1[4]: ~A" (mix-position m2))))
	(undo)
	(set! (mix-position m2) 500)
	(undo)
	(scale-channel 0.5 1000 100)
	(if (not (= (mix-position m2) 123)) (snd-display ";mix-position m2[5]: ~A" (mix-position m2)))
	(if (not (= (mix-position m1) 321)) (snd-display ";mix-position m1[5]: ~A" (mix-position m1)))
	(undo)
	(set! (mix-position m2) 500)
	(undo)
	(ptree-channel (lambda (y) (* y 0.5)) 2000 100)
	(if (not (= (mix-position m2) 123)) (snd-display ";mix-position m2[6]: ~A" (mix-position m2)))
	(if (not (= (mix-position m1) 321)) (snd-display ";mix-position m1[6]: ~A" (mix-position m1)))
	(undo)
	(set! (mix-position m2) 500)
	(undo-edit)
	(ramp-channel 0.0 1.0 3000 100)
	(catch #t
	       (lambda ()
		 (if (not (= (mix-position m2) 123)) (snd-display ";mix-position m2[7]: ~A" (mix-position m2)))
		 (if (not (= (mix-position m1) 321)) (snd-display ";mix-position m1[7]: ~A" (mix-position m1))))
	       (lambda args (snd-display ";mix-position trouble: ~A" args)))
	(close-sound ind))
      
      ;; check that current console is correct
      (let ((ind (open-sound "storm.snd")))
	(set! (x-bounds) (list 0 80.0))
	(make-selection 1000000 1050000)
	(let ((m1 (mix-selection 900000))
	      (m2 (mix-selection 400000)))
	  (as-one-edit (lambda () 
			 (set! (mix-position m1) 0) 
			 (set! (mix-position m2) 1)))
	  (if (or (not (= (mix-position m1) 0))
		  (not (= (mix-position m2) 1)))
	      (snd-display ";as-one-edit positions: ~A ~A" (mix-position m1) (mix-position m2)))
	  (undo-channel)
	  (if (or (not (= (mix-position m1) 900000))
		  (not (= (mix-position m2) 400000)))
	      (snd-display ";as-one-edit positions after undo: (~A): ~A (~A): ~A" m1 (mix-position m1) m2 (mix-position m2)))
	  (redo-channel)
	  (if (or (not (= (mix-position m1) 0))
		  (not (= (mix-position m2) 1)))
	      (snd-display ";as-one-edit positions after redo: ~A ~A" (mix-position m1) (mix-position m2)))
	  (close-sound ind)))
      
      (let ((ind (open-sound "2.snd")))
	(make-selection 0 10000 ind)
	(if (not (= (selection-chans) 2))
	    (snd-display ";stereo selection: ~A" (selection-chans)))
	(set! (sync ind) #t)
	(let ((md (mix-selection 500 ind)))
	  (if (not (= (mix-chans md) 2))
	      (snd-display ";mix-chans of stereo selection: ~A" (mix-chans md)))
	  (if (not (mix? (1+ md)))
	      (snd-display ";where is 2nd mix? ~A ~A" md (mixes)))
	  (if (not (= (edit-position ind 0) 1))
	      (snd-display ";edit-position 0 after stereo mix selection: ~A" (edit-position ind 0)))
	  (if (not (= (edit-position ind 1) 1))
	      (snd-display ";edit-position 1 after stereo mix selection: ~A" (edit-position ind 1)))
	  (set! (sync ind) #f)
	  (undo-edit 1 ind 0)
	  (delete-sample 25 ind 0)
	  (set! (mix-position (1+ md)) 750)
	  (if (not (= (edit-position ind 1) 2))
	      (snd-display ";edit-position 1 after stereo mix selection moved: ~A" (edit-position ind 2)))
	  (revert-sound ind)
	  (delete-sample 25 ind 1)
	  (if (or (mix? md) (mix? (1+ md)))
	      (snd-display ";undo mix stereo sel: ~A ~A" (mix? md) (mix? (1+ md))))
	  (close-sound ind)))
      
      (let ((ind (new-sound "test.snd"))
	    (v (make-vct 20)))
	(do ((i 0 (1+ i))) ((= i 20)) (vct-set! v i (* i .01)))
	(vct->channel v)
	(do ((i 0 (1+ i))) ((= i 20)) (vct-set! v i (* i -.01)))
	(let ((mx (mix-vct v 10)))
	  (let ((hi (make-mix-sample-reader mx))
		(ho (make-mix-sample-reader mx 5)))
	    (call-with-current-continuation
	     (lambda (break)
	       (do ((i 0 (1+ i)))
		   ((= i 10))
		 (let ((ho-val (ho))
		       (hi-val (hi)))
		   (if (fneq hi-val (* i -.01))
		       (begin
			 (snd-display ";mix-reader at ~A from 0: ~A" i hi-val)
			 (break)))
		   (if (fneq ho-val (* (+ i 5) -.01))
		       (begin
			 (snd-display ";mix-reader at ~A from 5: ~A" i ho-val)
			 (break)))))))))
	(revert-sound ind)
	(set! v (make-vct 21))
	(vct-fill! v 0.5)
	(vct->channel v)
	(let ((mx (mix-vct v 10)))
	  (set! (mix-amp-env mx 0) '(0 0 1 1))
	  (let ((hi (make-mix-sample-reader mx 0))
		(ho (make-mix-sample-reader mx 10)))
	    (call-with-current-continuation
	     (lambda (break)
	       (do ((i 0 (1+ i)))
		   ((= i 10))
		 (let ((ho-val (ho))
		       (hi-val (hi)))
		   (if (fneq hi-val (* i .025))
		       (begin
			 (snd-display ";mix-reader env'd at ~A from 0: ~A" i hi-val)
			 (break)))
		   (if (fneq ho-val (* (+ i 10) .025))
		       (begin
			 (snd-display ";mix-reader env'd at ~A from 10: ~A" i ho-val)
			 (break)))))))))
	(close-sound ind))

      (let* ((ind (open-sound "oboe.snd"))
	     (id (mix-vct (make-vct 10 .1))))
	(set! (mix-position id) 100)
	(if (or (not (= (mix-position id) 100))
		(not (= (edit-position ind 0) 2)))
	    (snd-display ";mix-position init: ~A ~A" (mix-position id) (edit-position ind 0)))
	(set! (mix-position id) 100)
	(if (or (not (= (mix-position id) (mix-position id)))
		(not (= (edit-position ind 0) 2)))
	    (snd-display ";mix-position 2 (no-op): ~A ~A" (mix-position id) (edit-position ind 0)))
	(set! (mix-amp id 0) 1.0)
	(if (or (fneq (mix-amp id 0) 1.0)
		(not (= (edit-position ind 0) 2)))
	    (snd-display ";mix-amp no-op: ~A ~A" (mix-amp id 0) (edit-position ind 0)))
	(set! (mix-amp id) 0.5)
	(if (or (fneq (mix-amp id 0) 0.5)
		(not (= (edit-position ind 0) 3)))
	    (snd-display ";mix-amp .5: ~A ~A" (mix-amp id 0) (edit-position ind 0)))
	(set! (mix-amp id 0) (mix-amp id 0))
	(if (or (fneq (mix-amp id 0) 0.5)
		(not (= (edit-position ind 0) 3)))
	    (snd-display ";mix-amp no-op: ~A ~A" (mix-amp id 0) (edit-position ind 0)))
	(set! (mix-speed id) 1.0)
	(if (or (fneq (mix-speed id) 1.0)
		(not (= (edit-position ind 0) 3)))
	    (snd-display ";mix-speed no-op: ~A ~A" (mix-speed id 0) (edit-position ind 0)))
	(set! (mix-speed id) .5)
	(if (or (fneq (mix-speed id) 0.5)
		(not (= (edit-position ind 0) 4)))
	    (snd-display ";mix-speed .5: ~A ~A" (mix-speed id 0) (edit-position ind 0)))
	(set! (mix-speed id) (mix-speed id))
	(if (or (fneq (mix-speed id) 0.5)
		(not (= (edit-position ind 0) 4)))
	    (snd-display ";mix-speed 2 no-op: ~A ~A" (mix-speed id 0) (edit-position ind 0)))
	(set! (mix-amp-env id 0) '(0 0 1 1))
	(if (not (= (edit-position ind 0) 5))
	    (snd-display ";mix-amp-env init: ~A ~A" (mix-amp-env id 0) (edit-position ind 0)))
	(set! (mix-amp-env id 0) '(0 0 1 1))
	(if (not (= (edit-position ind 0) 5))
	    (snd-display ";mix-amp-env no-op: ~A ~A" (mix-amp-env id 0) (edit-position ind 0)))
	(close-sound ind))

      (letrec ((test-mix-disconnect
		(lambda (name id0 chn0 id1 chn1)
		  (let ((amp0 (mix-amp id0 chn0))
			(amp1 (mix-amp id1 chn1))
			(env0 (mix-amp-env id0 chn0))
			(env1 (mix-amp-env id1 chn1)))
		    (set! (mix-amp id0 chn0) (* (mix-amp id0 chn0) .5))
		    (if (or (fneq (mix-amp id0 chn0) (* .5 amp0))
			    (fneq (mix-amp id1 chn1) amp1))
			(snd-display ";pan-mix disconnect amp ~A: ~A (~A) ~A (~A)" 
				     name (mix-amp id0 chn0) amp0 (mix-amp id1 chn1) amp1))
		    (set! (mix-amp-env id1 chn1) (list 0.0 (random 1.0) 1.0 (random 1.0)))
		    (if (or (not (equal? (mix-amp-env id0 chn0) env0))
			    (feql (mix-amp-env id1 chn1) (or env1 '())))
			(snd-display ";pan-mix disconnect amp-env ~A: ~A (~A) ~A (~A)" 
				     name (mix-amp-env id0 chn0) env0 (mix-amp-env id1 chn1) env1))
		    (if (not (= id0 id1))
			(let ((pos0 (mix-position id0))
			      (pos1 (mix-position id1))
			      (spd0 (mix-speed id0))
			      (spd1 (mix-speed id1)))
			  (set! (mix-position id0) (+ pos0 12))
			  (if (or (= (mix-position id0) pos0)
				  (not (= (mix-position id1) pos1)))
			      (snd-display ";pan-mix disconnect position ~A: ~A (~A) ~A (~A)" 
					   name (mix-position id0) pos0 (mix-position id1) pos1))
			  (set! (mix-speed id1) (* (mix-speed id1) 1.5))
			  (if (or (fneq (mix-speed id1) (* 1.5 spd1))
				  (fneq (mix-speed id0) spd0))
			      (snd-display ";pan-mix disconnect speed ~A: ~A (~A) ~A (~A)" 
					   name (mix-speed id0) spd0 (mix-speed id1) spd1)))))))
	       (find-next-child 
		(lambda (widget name ptr)
		  (call-with-current-continuation
		   (lambda (return)
		     (for-each-child
		      widget
		      (lambda (child)
			(if (and (string=? (XtName child) name)
				 (not (equal? child ptr)))
			    (return child))))
		     #f))))
	       (env-at-one
		(lambda (e)
		  (or (null? e)
		      (and (< (abs (- (cadr e) 1.0)) .001)
			   (env-at-one (cddr e)))))))
	
	(if (and (provided? 'snd-motif)
		 (provided? 'xm)
		 #t)
	    (let* ((ind (open-sound "oboe.snd"))
		   (mix1 (mix-vct (vct 0.1 0.2 0.3) 120 ind 0 #t "origin!"))
		   (mix2 (mix-vct (vct 0.1 0.2 0.3) 1200 ind 0 #t))
		   (mix3 (mix-vct (vct 0.1 0.2 0.3) 12000 ind 0 #t)))
	      (set! (mix-track mix1) (make-track))
	      (set! (mix-track mix3) (mix-track mix1))
	      (if (not (equal? (mixes ind 0) (list mix1 mix2 mix3)))
		  (snd-display ";mixes: ~A ~A" (mixes ind 0) (list mix1 mix2 mix3)))
	      (if (not (equal? (mixes ind #t) (list (list mix1 mix2 mix3))))
		  (snd-display ";mixes #t: ~A ~A" (mixes ind #t) (list (list mix1 mix2 mix3))))
	      (if (not (equal? (mixes) (list (list (list mix1 mix2 mix3)))))
		  (snd-display ";mixes all: ~A ~A" (mixes) (list (list (list mix1 mix2 mix3)))))
		
		(view-mixes-dialog)
		(set! (mix-dialog-mix) mix1)
		(let* ((mixd (list-ref (dialog-widgets) 16))
		       (nxt (find-child mixd "Next"))
		       (prev (find-child mixd "Previous"))
		       (tplay (find-child mixd "mix-track-play")))
		  (click-button tplay) (force-event)
		  (if (or (not (XtIsSensitive nxt))
			  (XtIsSensitive prev))
		      (snd-display ";mix-dialog next/previous: ~A ~A ~A ~A" nxt (XtIsSensitive nxt) prev (XtIsSensitive prev)))
		  (click-button nxt) (force-event)
		  (click-button nxt) (force-event)
		  (if (or (XtIsSensitive nxt)
			  (not (XtIsSensitive prev)))
		      (snd-display ";mix-dialog (1) next/previous: ~A ~A ~A ~A" nxt (XtIsSensitive nxt) prev (XtIsSensitive prev)))
		  (click-button prev) (force-event)
		  (click-button prev) (force-event))
		(dismiss-all-dialogs)
	      (close-sound ind)))
	)

      (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "copy-mix tests" 300)))
	
	(define (check-copied-mix original copy pos)
	  (if (not (mix? copy)) (snd-display ";copy-mix returns bad mix: ~A" copy))
	  (if (not (= (mix-track copy) 0)) (snd-display ";copy-mix set track: ~A ~A" (mix-track copy) (mix-track original)))
	  (if (not (= (mix-chans copy) (mix-chans original))) (snd-display ";copy-mix chans: ~A ~A" (mix-chans copy) (mix-chans original)))
	  (if (not (= (mix-tag-position copy) (mix-tag-position original))) 
	      (snd-display ";copy-mix anchor: ~A ~A" (mix-tag-position copy) (mix-tag-position original)))
	  (if (not (= (mix-frames copy) (mix-frames original))) (snd-display ";copy-mix frames: ~A ~A" (mix-frames copy) (mix-frames original)))
	  (if (not (= (mix-position copy) pos)) (snd-display ";copy-mix set position: ~A ~A ~A" (mix-position copy) pos (mix-position original)))   
	  (if (fneq (mix-speed copy) (mix-speed original)) (snd-display ";copy-mix speed: ~A ~A" (mix-speed copy) (mix-speed original)))
	  (if (fneq (mix-maxamp copy) (mix-maxamp original)) (snd-display ";copy-mix maxamp: ~A ~A" (mix-maxamp copy) (mix-maxamp original)))
	  (do ((i 0 (1+ i)))
	      ((= i (mix-chans copy)))
	    (if (fneq (mix-amp copy i) (mix-amp original i)) (snd-display ";copy-mix amp[~A]: ~A ~A" i (mix-amp copy i) (mix-amp original i)))
	    (let ((copy-amp-env (mix-amp-env copy i))
		  (original-amp-env (mix-amp-env original i)))
	      (if (and (list? copy-amp-env)
		       (list? original-amp-env))
		  (if (not (feql copy-amp-env original-amp-env)) 
		      (snd-display ";copy-mix amp-env[~A]: ~A ~A" i copy-amp-env original-amp-env))
		  (if (not (eq? copy-amp-env original-amp-env))
		      (snd-display ";copy-mix amp-env[~A]: ~A ~A" i copy-amp-env original-amp-env)))))
	  (if (not (equal? (mix-home copy) (mix-home original))) (snd-display ";copy-mix home: ~A ~A" (mix-home copy) (mix-home original))))
	
	(let* ((mix1 (mix-vct (make-vct 10 .5) 10))
	       (copy-mix1 (copy-mix mix1)))
	  (let ((old-color (mix-color mix1)))
	    (set! (mix-color mix1) (make-color 0 1 1))
	    (let ((new-color (mix-color mix1)))
	      (if (not (equal? new-color (make-color 0 1 1)))
		  (snd-display ";mix-color ~A ~A = ~A (~A)?" mix1 (make-color 0 1 1) new-color old-color))))
	  (check-copied-mix mix1 copy-mix1 10)
	  (set! (mix-amp mix1) 2.0)
	  (set! copy-mix1 (copy-mix mix1 20))
	  (check-copied-mix mix1 copy-mix1 20)    
	  (set! (mix-speed mix1) 2.0)
	  (set! copy-mix1 (copy-mix mix1 40))
	  (check-copied-mix mix1 copy-mix1 40)    
	  (set! (mix-amp-env mix1 0) '(0 0 1 1))
	  (set! copy-mix1 (copy-mix mix1 60))
	  (check-copied-mix mix1 copy-mix1 60)
	  (set! (mix-tag-position mix1) 2)
	  (set! copy-mix1 (copy-mix mix1 80))
	  (check-copied-mix mix1 copy-mix1 80)
	  (set! (mix-position mix1) 100)
	  (set! copy-mix1 (copy-mix mix1))
	  (check-copied-mix mix1 copy-mix1 100)
	  (set! mix1 (mix "2a.snd" 100 #t ind))
	  (set! (mix-amp mix1 0) 0.5)
	  (set! (mix-amp mix1 1) 0.75)
	  (set! (mix-amp-env mix1 0) '(0 1 1 0))
	  (set! copy-mix1 (copy-mix mix1 200))
	  (check-copied-mix mix1 copy-mix1 200)
	  (set! (mix-amp-env mix1 1) '(0 0 1 1))
	  (set! copy-mix1 (copy-mix mix1 300))
	  (check-copied-mix mix1 copy-mix1 300)
	  (let ((mix2 (copy-mix copy-mix1 500)))
	    (check-copied-mix copy-mix1 mix2 500))
	  (close-sound ind)
	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 4 "copy-mix tests" 300))
	  (set! mix1 (mix "2a.snd" 10 #t ind 2))
	  (set! copy-mix1 (copy-mix mix1 20))
	  (check-copied-mix mix1 copy-mix1 20)
	  (close-sound ind)))

      (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "copy-mix tests" 300))
	    (old-color (mix-color)))
	(set! (mix-color) (make-color 1 1 0))
	(let ((mix1 (mix-vct (make-vct 10 .5) 10)))
	  (if (or (not (equal? (mix-color) (make-color 1 1 0)))
		  (not (equal? (mix-color mix1) (make-color 1 1 0))))
	      (snd-display ";set mix-color: ~A ~A ~A ~A" (mix-color) (mix-color mix1) (make-color 1 1 0) old-color))
	  (set! (mix-color) old-color)
	  (save-mix mix1 "test.snd")
	  (let ((ind1 (open-sound "test.snd")))
	    (if (not (= (frames ind1) (mix-frames mix1))) (snd-display ";save-mix frames: ~A ~A" (mix-frames mix1) (frames ind1)))
	    (if (not (vequal (channel->vct 0 10) (mix->vct mix1)))
		(snd-display ";save-mix data: ~A ~A" (mix->vct mix1) (channel->vct 0 10 ind1)))
	    (close-sound ind1)))
	(close-sound ind))

      (let* ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "lock mix tests" 300))
	     (mix1 (mix-vct (make-vct 10 .5) 10)))
	(if (mix-locked? mix1) (snd-display ";make mix locked: ~A ~A" mix1 (mix-locked? mix1)))
	(delete-mix mix1)
	(if (fneq (maxamp ind 0) 0.0) (snd-display ";delete-mix maxamp: ~A" (maxamp ind 0)))
	(if (not (mix-locked? mix1)) (snd-display ";delete mix not locked: ~A ~A" mix1 (mix-locked? mix1)))
	(undo-channel 1 ind 0)
	(if (fneq (maxamp ind 0) 0.5) (snd-display ";undelete-mix maxamp: ~A" (maxamp ind 0)))
	(if (mix-locked? mix1) (snd-display ";undelete mix locked: ~A ~A" mix1 (mix-locked? mix1)))
	(redo-channel 1 ind 0)
	(if (fneq (maxamp ind 0) 0.0) (snd-display ";redelete-mix maxamp: ~A" (maxamp ind 0)))
	(if (not (mix-locked? mix1)) (snd-display ";redelete mix not locked: ~A ~A" mix1 (mix-locked? mix1)))
	(undo 2)
	(if (mix? mix1) (snd-display ";undo 2 kept mix?"))
	(if (fneq (maxamp ind 0) 0.0) (snd-display ";no delete-mix maxamp: ~A" (maxamp ind 0)))
	(let ((tag (catch 'no-such-mix
			  (lambda () (if (mix-locked? mix1) (snd-display ";no delete mix locked: ~A ~A" mix1 (mix-locked? mix1))))
			  (lambda args (car args)))))
	  (if (not (eq? tag 'no-such-mix)) (snd-display ";pending mix release accessible?")))
	(let ((tag (catch 'no-such-mix
			  (lambda () (set! (mix-locked? mix1) #t))
			  (lambda args (car args)))))
	  (if (not (eq? tag 'no-such-mix)) (snd-display ";pending mix release settable?")))
	(redo)
	(if (mix-locked? mix1) (snd-display ";mix un/locked: ~A ~A" mix1 (mix-locked? mix1)))
	(if (fneq (maxamp ind 0) 0.5) (snd-display ";reundelete-mix maxamp: ~A" (maxamp ind 0)))
	(let ((track1 (make-track mix1)))
	  (delete-track track1)
	  (if (fneq (maxamp ind 0) 0.0) (snd-display ";delete-track maxamp: ~A" (maxamp ind 0)))
	  (undo)
	  (if (fneq (maxamp ind 0) 0.5) (snd-display ";undelete-track maxamp: ~A" (maxamp ind 0)))
	  (redo)
	  (if (fneq (maxamp ind 0) 0.0) (snd-display ";redelete-track maxamp: ~A" (maxamp ind 0))))
	(revert-sound ind)
	(let* ((mix2 (mix-vct (make-vct 10 .5) 10))
	       (track2 (make-track mix2)))
	  (set! (track-amp-env track2) '(0 0 1 1))
	  (delete-track track2)
	  (if (fneq (maxamp ind 0) 0.0) (snd-display ";delete-track (amp-env) maxamp: ~A" (maxamp ind 0)))
	  (undo)
	  (let ((mix3 (mix-vct (make-vct 10 .5) 50)))
	    (set! (mix-track mix3) track2))
	  (delete-track track2)
	  (if (fneq (maxamp ind 0) 0.0) (snd-display ";redelete-track (amp-env) maxamp: ~A" (maxamp ind 0)))
	  (close-sound ind)))

      (let* ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "mix undo tests" 300))
	     (mix1 (mix-vct (make-vct 10 .1) 10))
	     (track1 (make-track))
	     (track2 (make-track)))
	(set! (mix-track mix1) track1)
	(if (not (= (mix-track mix1) track1)) (snd-display ";mix-track start ~A: ~A" track1 (mix-track mix1)))
	(set! (track-amp track2) 2.0)
	(set! (mix-amp mix1 0) 3.0)
	(if (fneq (maxamp ind 0) .3) (snd-display ";mix-track chain setup: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(set! (mix-track mix1) track2)
	(if (not (= (mix-track mix1) track2)) (snd-display ";mix-track set ~A: ~A" track2 (mix-track mix1)))
	(if (fneq (maxamp ind 0) .6) (snd-display ";mix-track chain set: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(undo)
	(if (not (= (mix-track mix1) track1)) (snd-display ";mix-track undo ~A: ~A" track1 (mix-track mix1)))
	(if (fneq (maxamp ind 0) .3) (snd-display ";mix-track chain undo setup: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))	
	(undo 2)
	(if (not (= (mix-track mix1) 0)) (snd-display ";mix-track undo2: ~A ~A ~A: ~A" track1 track2 (mix-track mix1) (edit-position ind 0)))
	(if (fneq (maxamp ind 0) .1) (snd-display ";mix-track chain undo: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(set! (mix-position mix1) 20)
	(if (fneq (maxamp ind 0) .1) (snd-display ";mix-track chain undo: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(let ((pos0 (mix-tag-position mix1))
	      (y0 (mix-tag-y mix1)))
	  (set! (mix-amp mix1) 1.0)
	  (set! (mix-tag-position mix1) 3)
	  (set! (mix-tag-y mix1) 6)
	  (if (not (= (mix-tag-position mix1) 3)) (snd-display ";mix-tag-position chain test 0: ~A" (mix-tag-position mix1)))
	  (if (not (= (mix-tag-y mix1) 6)) (snd-display ";mix-tag-y chain test 0: ~A" (mix-tag-y mix1)))
	  (undo)
	  (if (not (= (mix-tag-position mix1) pos0)) (snd-display ";mix-tag-position chain test 1: ~A" (mix-tag-position mix1)))
	  (if (not (= (mix-tag-y mix1) y0)) (snd-display ";mix-tag-y chain test 1: ~A" (mix-tag-y mix1)))
	  (close-sound ind)))

      (let* ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "copy-mix tests" 300))
	     (mix1 (mix-vct (make-vct 10 .1) 10))
	     (track1 (make-track))
	     (track2 (make-track))
	     (edpos 0))
	(set! edpos (edit-position ind 0))
	(set! (mix-track mix1) track1)
	(if (not (= (mix-track mix1) track1)) (snd-display ";mix-track start ~A: ~A" track1 (mix-track mix1)))
	(set! (track-amp track2) 2.0)
	(set! (mix-amp mix1 0) 3.0)
	(if (fneq (maxamp ind 0) .3) (snd-display ";mix-track chain setup: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(set! (mix-track mix1) track2)
	(if (not (= (mix-track mix1) track2)) (snd-display ";mix-track set ~A: ~A" track2 (mix-track mix1)))
	(if (fneq (maxamp ind 0) .6) (snd-display ";mix-track chain set: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(undo)
	(if (not (= (mix-track mix1) track1)) (snd-display ";mix-track undo ~A: ~A" track1 (mix-track mix1)))
	(if (fneq (maxamp ind 0) .3) (snd-display ";mix-track chain undo setup: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))	
	(set! (edit-position ind 0) edpos)
	(if (not (= (mix-track mix1) 0)) (snd-display ";mix-track undo2: ~A" (mix-track mix1)))
	(if (fneq (maxamp ind 0) .1) (snd-display ";mix-track chain undo: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(set! (mix-position mix1) 20)
	(if (fneq (maxamp ind 0) .1) (snd-display ";mix-track chain undo: ~A ~A" (maxamp ind 0) (mix-amp mix1 0)))
	(let ((pos0 (mix-tag-position mix1))
	      (y0 (mix-tag-y mix1)))
	  (set! (mix-amp mix1) 1.0)
	  (set! (mix-tag-position mix1) 3)
	  (set! (mix-tag-y mix1) 6)
	  (set! (mix-amp mix1 0) .1)
	  (if (not (= (mix-tag-position mix1) 3)) (snd-display ";mix-tag-position chain test 0: ~A" (mix-tag-position mix1)))
	  (if (not (= (mix-tag-y mix1) 6)) (snd-display ";mix-tag-y chain test 0: ~A" (mix-tag-y mix1)))
	  (undo)
	  (if (not (= (mix-tag-position mix1) pos0)) (snd-display ";mix-tag-position chain test 1: ~A ~A" pos0 (mix-tag-position mix1)))
	  (if (not (= (mix-tag-y mix1) y0)) (snd-display ";mix-tag-y chain test 1: ~A ~A" y0 (mix-tag-y mix1)))
	  (redo)
	  (if (not (= (mix-tag-position mix1) 3)) (snd-display ";mix-tag-position chain test 2: ~A" (mix-tag-position mix1)))
	  (if (not (= (mix-tag-y mix1) 6)) (snd-display ";mix-tag-y chain test 2: ~A" (mix-tag-y mix1)))
	  (close-sound ind)))

      (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "copy-track tests" 300)))
	
	(define (check-copied-track original copy pos)
	  (if (not (track? copy)) (snd-display ";copy-track returns bad track: ~A" copy))
	  (if (not (= (track-track copy) 0)) (snd-display ";copy-track set track: ~A ~A" (track-track copy) (track-track original)))
	  (if (not (= (track-chans copy) (track-chans original))) (snd-display ";copy-track chans: ~A ~A" (track-chans copy) (track-chans original)))
	  (if (not (= (track-frames copy) (track-frames original))) (snd-display ";copy-track frames: ~A ~A" (track-frames copy) (track-frames original)))
	  (if (not (= (track-position copy) pos)) (snd-display ";copy-track set position: ~A ~A ~A" (track-position copy) pos (track-position original)))   
	  (if (fneq (track-speed copy) (track-speed original)) (snd-display ";copy-track speed: ~A ~A" (track-speed copy) (track-speed original)))
	  (if (fneq (track-amp copy) (track-amp original)) (snd-display ";copy-track amp: ~A ~A" (track-amp copy) (track-amp original)))
	  (let ((copy-amp-env (track-amp-env copy))
		(original-amp-env (track-amp-env original)))
	    (if (and (list? copy-amp-env)
		     (list? original-amp-env))
		(if (not (feql copy-amp-env original-amp-env)) 
		    (snd-display ";copy-track amp-env: ~A ~A" copy-amp-env original-amp-env))
		(if (not (eq? copy-amp-env original-amp-env))
		    (snd-display ";copy-track amp-env: ~A ~A" copy-amp-env original-amp-env))))
	  (if (not (= (length (track copy)) (length (track original)))) 
	      (snd-display ";copy-track mix lists differ: ~A ~A" (track copy) (track original))
	      (for-each
	       (lambda (orig cop)
		 (if (= (mix-track orig) original)
		     (if (not (= (mix-track cop) copy))
			 (snd-display ";copy-track mix-tracks: ~A -> ~A" (mix-track orig) (mix-track cop)))))
	       (track original)
	       (track copy)))
	  (if (> (track-chans original) 0)
	      (let ((vcopy (track->vct copy))
		    (vorig (track->vct original)))
		(if (not (vequal vcopy vorig)) (snd-display ";copy-track data: ~A ~A" vcopy vorig)))))
	
	(let ((tag (catch #t (lambda () (copy-track 0)) (lambda args (car args)))))
	  (if (not (eq? tag 'no-such-track)) (snd-display ";copy-track 0: ~A" tag)))
	(let* ((mix1 (mix-vct (make-vct 10 1.0) 100))
	       (track0 (make-track))
	       (track1 (make-track mix1)))
	  (if (not (equal? (track track1) (list mix1))) (snd-display ";make-track for copy: ~A ~A" mix1 (track track1)))
	  (if (not (= track1 (mix-track mix1))) (snd-display ";make-track for copy mix: ~A ~A" (mix-track mix1) track1))
	  (let* ((copy-track0 (copy-track track0))
		 (edpos (edit-position ind 0))
		 (copy-track1 (copy-track track1)))
	    (if (not (equal? (track track1) (list mix1))) (snd-display ";copy-track clobbered original: ~A ~A" mix1 (track track1)))
	    (if (not (= track1 (mix-track mix1))) (snd-display ";copy-track clobbered original: ~A ~A" (mix-track mix1) track1))
	    (check-copied-track track0 copy-track0 -1)
	    (check-copied-track track1 copy-track1 100)
	    (if (not (= (edit-position ind 0) (1+ edpos)))
		(snd-display ";copy-track not atomic? ~A ~A" edpos (edit-position ind 0)))
	    (undo 2)))
	(let* ((mix1 (mix-vct (make-vct 10 1.0) 50))
	       (track1 (make-track mix1))
	       (track2 (copy-track track1 200)))
	  (check-copied-track track1 track2 200)
	  (undo)
	  (set! (track-amp track1) 2.0)
	  (set! (track-amp-env track1) '(0 0 1 1))
	  (set! (mix-amp mix1) 0.25)
	  (set! track2 (copy-track track1 200))
	  (check-copied-track track1 track2 200)
	  (revert-sound ind))
	(let* ((mix1 (mix-vct (make-vct 10 1.0) 50))
	       (mix2 (mix-vct (make-vct 10 1.0) 75))
	       (track1 (make-track mix1 mix2))
	       (track2 (copy-track track1 200)))
	  (check-copied-track track1 track2 200)
	  (let* ((mix3 (mix-vct (make-vct 10 1.0) 10))
		 (mix4 (mix-vct (make-vct 10 1.0) 20))
		 (track3 (make-track mix3 mix4)))
	    (set! (track-track track3) track1)
	    (set! track2 (copy-track track1 300))
	    (check-copied-track track1 track2 300))
	  )
	(close-sound ind)
	)
      
      (let ((old-with-mix-tags (with-mix-tags)))
	(set! (with-mix-tags) #t)
	(let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track tests" 1000)))
	  (let ((mix1 (mix-vct (make-vct 10 .4) 100)))
	    
	    ;; -------- empty and 1-mix tracks --------
	    (if (not (mix? mix1)) (snd-display ";can't even get track tests started!"))
	    (let ((track1 (make-track mix1)))
	      
	      (define (track-states-match? track-id state) ; state is (list amp speed track env color)
		(and (< (abs (- (track-amp track-id) (list-ref state 0))) .0001)
		     (< (abs (- (track-speed track-id) (list-ref state 1))) .0001)
		     (= (track-track track-id) (list-ref state 2))
		     (or (and (not (track-amp-env track-id))
			      (not (list-ref state 3)))
			 (feql (track-amp-env track-id) (list-ref state 3)))
		     (equal? (track-color track-id) (list-ref state 4))))
	      (define (track-state->list track-id)
		(list (track-amp track-id)
		      (track-speed track-id)
		      (track-track track-id)
		      (track-amp-env track-id)
		      (track-color track-id)))
	      
	      (if (not (track? track1)) (snd-display ";track? ~A" track1))
	      (let ((tag (catch #t
				(lambda () (set! (track-track track1) -1))
				(lambda args (car args)))))
		(if (not (eq? tag 'out-of-range)) (snd-display ";set track-track -1: ~A (~A)" tag (track-track track1))))
	      (if (not (= (track-chans track1) 1)) (snd-display ";track-chans 1 mix: ~A" (track-chans track1)))
	      (if (not (= (mix-track mix1) track1)) (snd-display ";make track didn't set track? ~A" (mix-track mix1)))
	      (if (not (equal? (track track1) (list mix1))) (snd-display ";track 1: ~A ~A" mix1 (track track1)))
	      (if (not (track-states-match? track1 '(1.0 1.0 0 () #f)))
		  (snd-display ";track states 1: ~A" (track-state->list track1)))
	      (if (not (= (edit-position ind 0) 2))
		  (snd-display ";tracked mix edit position: ~A ~A" (edit-position ind 0) (edit-tree ind 0)))
	      (if (fneq (maxamp ind 0) .4) (snd-display ";mixed maxamp .4: ~A" (maxamp ind 0)))
	      (if (not (= (track-position track1) 100)) (snd-display ";track position mix1: ~A" (track-position track1)))
	      (if (not (= (track-frames track1) 10)) (snd-display ";track frames mix1: ~A" (track-frames track1)))
	      (set! (track-amp track1) 2.0)
	      (if (not (track-states-match? track1 '(2.0 1.0 0 () #f)))
		  (snd-display ";track states 2: ~A" (track-state->list track1)))
	      (if (not (= (edit-position ind 0) 3))
		  (snd-display ";tracked mix amp-2 edit position: ~A ~A" (edit-position ind 0) (edit-tree ind 0)))
	      (if (fneq (maxamp ind 0) .8) (snd-display ";mixed maxamp .8: ~A" (maxamp ind 0)))
	      (undo 1)
	      (if (not (track-states-match? track1 '(1.0 1.0 0 () #f)))
		  (snd-display ";track states 3 (undo): ~A" (track-state->list track1)))
	      (if (not (= (edit-position ind 0) 2))
		  (snd-display ";tracked mix edit position (undo): ~A ~A" (edit-position ind 0) (edit-tree ind 0)))
	      (if (fneq (maxamp ind 0) .4) (snd-display ";mixed maxamp .4 (undo): ~A" (maxamp ind 0)))
	      (redo 1)
	      (if (not (track-states-match? track1 '(2.0 1.0 0 () #f)))
		  (snd-display ";track states 4 (redo): ~A" (track-state->list track1)))
	      (if (not (= (edit-position ind 0) 3))
		  (snd-display ";tracked mix amp-2 edit position (redo): ~A ~A" (edit-position ind 0) (edit-tree ind 0)))
	      (if (fneq (maxamp ind 0) .8) (snd-display ";mixed maxamp .8 (redo): ~A" (maxamp ind 0)))
	      (let ((mr (make-mix-sample-reader mix1))
		    (tr (make-track-sample-reader track1)))
		(call-with-current-continuation
		 (lambda (break)
		   (do ((i 0 (1+ i)))
		       ((= i (mix-frames mix1)))
		     (let ((ms (read-mix-sample mr))
			   (ts (read-track-sample tr)))
		       (if (or (fneq ms ts)
			       (fneq ms .8))
			   (begin
			     (snd-display ";mix+track readers: ~A ~A (.8)" ms ts)
			     (break))))))))
	      (for-each
	       (lambda (samp val)
		 (if (fneq (sample samp) val) (snd-display ";track read ~D: ~A (~A)" samp (sample samp) val)))
	       (list 99 105 111 199 207 211 299 306 311)
	       (list 0.0 .8 0.0 0.0 0.0 0.0 0.0 0.0 0.0))
	      (if (fneq (sample 99) 0.0) (snd-display ";track read 99: ~A" (sample 99)))
	      (if (fneq (sample 111) 0.0) (snd-display ";track read 111: ~A" (sample 111)))
	      (set! (track-position track1) 200)
	      (if (not (= (track-position track1) 200)) (snd-display ";track position mix1 200: ~A" (track-position track1)))
	      (if (not (= (track-frames track1) 10)) (snd-display ";track frames mix1 200: ~A" (track-frames track1)))
	      (if (not (= (mix-position mix1) 200)) (snd-display ";track position mix 1 200: ~A" (mix-position mix1)))
	      (if (not (= (mix-frames mix1) 10)) (snd-display ";track frames mix 1 200: ~A" (mix-frames mix1)))
	      (if (not (track-states-match? track1 '(2.0 1.0 0 () #f)))
		  (snd-display ";track states 5 (move): ~A" (track-state->list track1)))
	      (for-each
	       (lambda (samp val)
		 (if (fneq (sample samp) val) (snd-display ";track read ~D: ~A (~A)" samp (sample samp) val)))
	       (list 99 105 111 199 207 211 299 306 311)
	       (list 0.0 0.0 0.0 0.0 .8 0.0 0.0 0.0 0.0))
	      (undo 1)
	      (set! (mix-position mix1) 300)
	      (if (not (= (track-position track1) 300)) (snd-display ";track position mix1 300: ~A" (track-position track1)))
	      (if (not (= (track-frames track1) 10)) (snd-display ";track frames mix1 300: ~A" (track-frames track1)))
	      (if (not (= (mix-position mix1) 300)) (snd-display ";track position mix 1 300: ~A" (mix-position mix1)))
	      (if (not (= (mix-frames mix1) 10)) (snd-display ";track frames mix 1 300: ~A" (mix-frames mix1)))
	      (if (not (track-states-match? track1 '(2.0 1.0 0 () #f)))
		  (snd-display ";track states 6 (move mix): ~A" (track-state->list track1)))
	      (for-each
	       (lambda (samp val)
		 (if (fneq (sample samp) val) (snd-display ";track read ~D: ~A (~A)" samp (sample samp) val)))
	       (list 99 105 111 199 207 211 299 306 311)
	       (list 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.8 0.0))
	      (undo 1)
	      (let ((track2 (make-track))
		    (pos (edit-position ind)))
		(if (not (equal? (track track2) '())) (snd-display ";empty track: ~A ~A" mix1 (track track2)))
		(if (not (= (track-chans track2) 0)) (snd-display ";track-chans no mix: ~A" (track-chans track2)))
		(if (not (= (track-position track2) -1)) (snd-display ";empty track position: ~A" (track-position track2)))	  
		(if (not (= (track-frames track2) 0)) (snd-display ";empty track frames: ~A" (track-frames track2)))
		(if (not (track-states-match? track2 '(1.0 1.0 0 () #f)))
		    (snd-display ";empty track states 7: ~A" (track-state->list track2)))
		(let ((tag (catch #t (lambda () (set! (track-track track1) track1)) (lambda args (car args)))))
		  (if (not (eq? tag 'out-of-range)) (snd-display ";circular track: ~A" tag)))
		(set! (track-track track1) track2)
		(if (not (track-states-match? track1 (list 2.0 1.0 track2 '() #f)))
		    (snd-display ";track states 8 (track): ~A" (track-state->list track1)))
		(let ((tag (catch #t (lambda () (set! (track-track track2) track1)) (lambda args (car args)))))
		  (if (not (eq? tag 'out-of-range)) (snd-display ";circular track 2: ~A" tag)))
		(if (fneq (maxamp ind 0) .8) (snd-display ";track+track maxamp .8: ~A" (maxamp ind 0)))
		(set! (track-amp track2) 0.5)
		(if (not (track-states-match? track2 '(0.5 1.0 0 () #f)))
		    (snd-display ";empty track states 9 (amp): ~A" (track-state->list track2)))
		(if (fneq (maxamp ind 0) .4) (snd-display ";track+track maxamp .4: ~A" (maxamp ind 0)))
		(set! (track-amp track1) 0.5)
		(if (fneq (maxamp ind 0) .1) (snd-display ";track+track maxamp .1: ~A" (maxamp ind 0)))
		(set! (track-track track1) 0)
		(if (fneq (maxamp ind 0) .2) (snd-display ";track+track maxamp .2: ~A" (maxamp ind 0)))
		(set! (edit-position ind) pos)
		(if (not (track-states-match? track1 '(2.0 1.0 0 () #f)))
		    (snd-display ";track states 10 (edit-pos): ~A" (track-state->list track1)))
		(set! (track-speed track1) 0.5)
		(if (not (= (mix-frames mix1) 20)) (snd-display ";mix-frames with track-speed: ~A" (mix-frames mix1)))
		(if (not (= (track-frames track1) 20)) (snd-display ";track-frames with track-speed: ~A" (track-frames track1)))
		(if (not (track-states-match? track1 '(2.0 0.5 0 () #f)))
		    (snd-display ";track states 11 (speed): ~A" (track-state->list track1)))
		;; track2 amp at this point is 0.5
		(set! (mix-track mix1) 0)
		(if (not (equal? (track track1) '())) (snd-display ";empty track 1: ~A ~A" mix1 (track track1)))
		(if (not (= (mix-frames mix1) 10)) (snd-display ";mix-frames without track-speed: ~A" (mix-frames mix1)))
		(if (not (= (track-frames track1) 0)) (snd-display ";track-frames without track-speed: ~A" (track-frames track1)))
		(if (not (= (track-position track1) -1)) (snd-display ";empty track1 position: ~A" (track-position track1)))	  	  
		(if (fneq (maxamp ind 0) .4) (snd-display ";no track maxamp .4: ~A ~A" (maxamp ind 0)	(mix-amp mix1))) 
		(set! (mix-track mix1) track2)
		(if (not (equal? (track track2) (list mix1))) (snd-display ";track 2: ~A ~A" mix1 (track track2)))
		(if (not (equal? (track track1) '())) (snd-display ";empty track (set 2): ~A ~A ~A" mix1 (track track1) (track track2)))
		(if (not (= (track-position track2) 100)) (snd-display ";track 2 position mix1: ~A" (track-position track2)))
		(if (not (= (track-frames track2) 10)) (snd-display ";track 2 frames mix1: ~A" (track-frames track2)))
		(if (fneq (maxamp ind 0) .2) (snd-display ";track 2 maxamp .2: ~A ~A ~A" (maxamp ind 0) (mix-amp mix1) (track-amp track2)))
		(set! (mix-amp-env mix1 0) '(0 0 1 1))
		(let ((tv (track->vct track2))
		      (mv (mix->vct mix1)))
		  ;; should be the same: a ramp to .2
		  (if (or (not (= (vct-length mv) (vct-length tv)))
			  (not (vequal tv mv))
			  (not (vequal tv (vct 0.000 0.022 0.044 0.067 0.089 0.111 0.133 0.156 0.178 0.200))))
		      (snd-display ";amp env ramp track2 mix1: ~A ~A" tv mv))
		  (set! (track-amp-env track2) '(0 0 1 1))
		  (set! tv (track->vct track2))
		  (set! mv (mix->vct mix1))
		  (if (or (not (= (vct-length mv) (vct-length tv)))
			  (not (vequal tv mv))
			  (not (vequal tv (vct 0.000 0.002 0.008 0.018 0.032 0.050 0.072 0.098 0.128 0.162))))
		      ;; presumably this is a round-off error in that we have 10 or so segments and 10 samples 
		      (snd-display ";amp env 2 ramp track2 mix1: ~A ~A" tv mv))
		  (set! (mix-amp-env mix1 0) #f)
		  (set! tv (track->vct track2))
		  (set! mv (mix->vct mix1))
		  (if (or (not (= (vct-length mv) (vct-length tv)))
			  (not (vequal tv mv))
			  (not (vequal tv (vct 0.000 0.022 0.044 0.067 0.089 0.111 0.133 0.156 0.178 0.200))))
		      (snd-display ";amp env ramp track2: ~A ~A" tv mv))
		  (set! (track-speed track1) 1.0)
		  (set! (mix-track mix1) track1)
		  ;; that should undo the track2 env and apply the track1 amp
		  (set! tv (track->vct track1))
		  (set! mv (mix->vct mix1))
		  (if (or (not (= (vct-length mv) (vct-length tv)))
			  (not (vequal tv mv))
			  (not (vequal tv (make-vct 10 0.8))))
		      (snd-display ";no amp env track1 mix1: ~A ~A" tv mv)))
		(let ((color (make-color 0 1 0)))
		  (set! (track-color track1) color)
		  (if (not (equal? (track-color track1) color))
		      (snd-display ";track color green: ~A" (track-color track1)))
		  (if (not (track-states-match? track1 (list 2.0 1.0 0 '() color)))
		      (snd-display ";track 1 states 9 color: ~A" (track-state->list track1))))
		(free-track track2)
		(if (track? track2) (snd-display ";free-track track?"))
		(if (member track2 (tracks)) (snd-display ";free-track tracks: ~A in ~A" track2 (tracks))))
	      
	      (revert-sound ind)
	      ;; now track1 should be in its unedited state
	      (if (not (track-states-match? track1 '(1.0 1.0 0 () #f)))
		  (snd-display ";track states 12 after revert: ~A" (track-state->list track1)))
	      (if (not (member track1 (tracks))) (snd-display ";revert->tracks: ~A ~A" track1 (tracks)))
	      (if (not (equal? (track track1) '())) (snd-display ";revert past mix track1: ~A ~A ~A" (track track1) mix1 (mix? mix1)))
	      (let ((tag (catch #t (lambda () (track->vct track1)) (lambda args (car args)))))
		(if (not (eq? 'no-such-channel tag)) (snd-display ";track->vct empty track: ~A" tag)))
	      
	      ;; one sample mix
	      (set! mix1 (mix-vct (make-vct 1 .1) 50))
	      (set! (mix-track mix1) track1)
	      (if (not (= (track-position track1) 50)) (snd-display ";track 1 position mix-samp: ~A" (track-position track1)))
	      (if (not (= (track-frames track1) 1)) (snd-display ";track 1 frames mix-samp: ~A" (track-frames track1)))

	      (let ((start-state (track-state->list track1)))
		(set! (track-amp-env track1) '(0 1 1 0))
		(undo)
		(if (not (track-states-match? track1 start-state))
		    (snd-display ";undo after set amp-env: ~A ~A" start-state (track-state->list track1)))
		(redo)
		(if (not (feql (track-amp-env track1) '(0.0 1.0 1.0 0.0)))
		    (snd-display ";redo set track-amp-env: ~A" (track-amp-env track1)))
		(set! (track-amp-env track1) #f)
		(if (not (track-states-match? track1 start-state))
		    (snd-display ";redo/undo after set amp-env: ~A ~A" start-state (track-state->list track1))))

	      (let ((edpos (edit-position ind 0))
		    (state (track-state->list track1)))
		(as-one-edit
		 (lambda ()
		   (set! (track-amp track1) 4.0)
		   (set! (track-speed track1) 1.5)
		   (set! (track-amp-env track1) '(0 1 1 0))))
		(if (not (= edpos (1- (edit-position ind 0))))
		    (snd-display ";backup in as-one-edit: ~A ~A" edpos (edit-position ind 0)))
		(if (not (track-states-match? track1 '(4.0 1.5 0 (0 1 1 0) #f)))
		    (snd-display ";track states after as-one-edit: ~A" (track-state->list track1)))
		(undo)
		(if (not (track-states-match? track1 state))
		    (snd-display ";track states after undone as-one-edit: ~A ~A" state (track-state->list track1)))

		(redo)
		(if (not (track-states-match? track1 '(4.0 1.5 0 (0 1 1 0) #f)))
		    (snd-display ";track states after as-one-edit redo: ~A" (track-state->list track1)))
		(set! (track-amp-env track1) #f)
		(if (not (track-states-match? track1 '(4.0 1.5 0 () #f)))
		    (snd-display ";track states after amp env #f: ~A" (track-state->list track1)))
		(undo))

	      (revert-sound ind)
	      
	      ;; -------- multi mix tracks --------
	      (set! mix1 (mix-vct (make-vct 100 .1) 50))
	      (let ((mix2 (mix-vct (make-vct 100 .2) 250)))
		(set! (mix-track mix1) track1)
		(set! (mix-track mix2) track1)
		(if (not (= (track-chans track1) 1)) (snd-display ";track-chans mono mix: ~A" (track-chans track1)))
		(if (not (= (track-position track1) 50)) (snd-display ";track 1 position mix2/3: ~A" (track-position track1)))
		(if (not (= (track-frames track1) 300)) (snd-display ";track 1 frames mix2/3: ~A" (track-frames track1)))
		(if (not (equal? (track track1) (list mix1 mix2))) (snd-display ";track1 mix2/3 track: ~A ~A ~A" mix1 mix2 (track track1)))
		(if (not (track-states-match? track1 '(1.0 1.0 0 () #f)))
		    (snd-display ";track states 1 mix2/3: ~A" (track-state->list track1)))
		(if (fneq (maxamp) .2) (snd-display ";track1 mix2/3 maxamp: ~A" (maxamp)))
		(let ((edpos (edit-position ind 0)))
		  (set! (track-amp track1) 2.0)
		  (if (fneq (maxamp) .4) (snd-display ";track1 mix2/3 *2 maxamp: ~A" (maxamp)))
		  (if (or (fneq (sample 51) .2)
			  (fneq (sample 251) .4))
		      (snd-display ";track1 mix2/3 *2 samples: ~A ~A" (sample 51) (sample 251)))
		  (if (or (fneq (mix-amp mix1) 1.0)
			  (fneq (mix-amp mix2) 1.0))
		      (snd-display ";track1 mix2/3 mix amps: ~A ~A" (mix-amp mix1) (mix-amp mix2)))
		  (if (not (= (edit-position ind 0) (1+ edpos)))
		      (snd-display ";track amp set was not atomic: ~A ~A" edpos (edit-position ind 0)))
		  (undo)
		  (set! (track-position track1) 100)
		  (if (not (= (mix-position mix1) 100)) (snd-display ";set track position 100 mix1: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 300)) (snd-display ";set track position 100 mix2: ~A" (mix-position mix2)))
		  (if (not (= (track-position track1) 100)) (snd-display ";set track position 100 track1: ~A" (track-position track1)))
		  (if (not (= (edit-position ind 0) (1+ edpos)))
		      (snd-display ";track position set was not atomic: ~A ~A" edpos (edit-position ind 0)))
		  ;; now try to see if the remix was correct (kinda messy...)
		  (if (or (fneq (sample 51) 0.0)
			  (fneq (sample 99) 0.0)
			  (fneq (sample 251) 0.0)
			  (fneq (sample 299) 0.0))
		      (snd-display ";set track-position, bad cancel?: ~A ~A ~A ~A" (sample 51) (sample 99) (sample 251) (sample 299)))
		  (if (or (fneq (sample 100) 0.1)
			  (fneq (sample 199) 0.1)
			  (fneq (sample 301) 0.2)
			  (fneq (sample 399) 0.2))
		      (snd-display ";set track-position, bad remix after cancel?: ~A ~A ~A ~A" (sample 100) (sample 199) (sample 301) (sample 399)))
		  (undo)
		  (set! (track-speed track1) 0.5)
		  (if (or (not (= (mix-frames mix1) 200))
			  (not (= (mix-frames mix2) 200)))
		      (snd-display ";set track speed mix-frames: ~A ~A ~A" (mix-frames mix1) (mix-frames mix2) (track-frames track1)))
		  (if (not (= (edit-position ind 0) (1+ edpos)))
		      (snd-display ";track speed set was not atomic: ~A ~A" edpos (edit-position ind 0)))
		  (if (not (track-states-match? track1 '(1.0 0.5 0 () #f)))
		      (snd-display ";track states speed set mix2/3: ~A" (track-state->list track1)))
		  (undo)
		  (let ((old-track-color (track-color track1))
			(old-mix1-color (mix-color mix1))
			(old-mix2-color (mix-color mix2))
			(color (make-color 0 1 1)))
		    (set! (track-color track1) color)
		    (if (not (= (edit-position ind 0) edpos))
			(snd-display ";track color set was an edit?: ~A ~A" edpos (edit-position ind 0)))
		    (if (not (track-states-match? track1 (list 1.0 1.0 0 '() color)))
			(snd-display ";track states color set mix2/3: ~A" (track-state->list track1)))
		    (if (or (not (equal? (mix-color mix1) color))
			    (not (equal? (mix-color mix2) color))
			    (not (equal? (track-color track1) color)))
			(snd-display ";set track color mix2/3: ~A ~A ~A" (mix-color mix1) (mix-color mix2) (track-color track1)))
		    (set! (track-amp-env track1) '(0 0 .5 0 .51 1 1 1))
		    (if (not (feql (track-amp-env track1) '(0 0 .5 0 .51 1 1 1)))
			(snd-display ";set track-amp-env mix2/3: ~A" (track-amp-env track1)))
		    (if (or (fneq (sample 51) 0.0)
			    (fneq (sample 99) 0.0)
			    (fneq (sample 251) 0.2)
			    (fneq (sample 299) 0.2))
			(snd-display ";set track-position, bad remix after cancel?: ~A ~A ~A ~A" (sample 100) (sample 199) (sample 301) (sample 399)))
		    (if (not (= (edit-position ind 0) (1+ edpos)))
			(snd-display ";track amp env set was not atomic: ~A ~A" edpos (edit-position ind 0)))
		    (if (not (track-states-match? track1 (list 1.0 1.0 0 (list 0 0 .5 0 .51 1 1 1) color)))
			(snd-display ";track states amp-env set mix2/3: ~A" (track-state->list track1)))
		    (if (not (equal? (track-color track1) color)) (snd-display ";track color not stacked? ~A ~A" color (track-color track1)))
		    (set! (track-color track1) old-track-color)
		    (let* ((edpos (edit-position ind 0))
			   (track2a (make-track mix1 mix2)))
		      (if (not (= (1+ edpos) (edit-position ind 0))) 
			  (snd-display ";make-track not atomic? ~A ~A ~A" edpos (edit-position ind 0) (track track2a)))
		      (if (not (= (mix-track mix1) (mix-track mix2) track2a)) 
			  (snd-display ";make-track didn't reset track? ~A ~A ~A" (mix-track mix1) (mix-track mix2) track2a))
		      (if (not (equal? (track track1) '())) (snd-display ";make-track didn't cancel old track? ~A" (track track1)))
		      (set! edpos (edit-position ind 0))
		      (delete-track track2a)
		      (if (not (= (1+ edpos) (edit-position ind 0)))
			  (snd-display ";delete-track not atomic? ~A ~A" edpos (edit-position ind 0)))
		      (if (or (not (mix-locked? mix1)) (not (mix-locked? mix2)))
			  (snd-display ";delete-track didn't lock mixes? ~A ~A" (mix-locked? mix1) (mix-locked? mix2)))
		      (if (fneq (track-amp track2a) 0.0) (snd-display ";delete-track track amp: ~A" (track-amp track2a)))
		      (if (fneq (maxamp ind 0) 0.0) (snd-display ";delete-track maxamp: ~A" (maxamp ind 0)))
		      (catch 'no-such-mix
			     (lambda ()
			       (set! (mix-track mix1) 0))
			     (lambda args args))
		      (if (= (mix-track mix1) 0) (snd-display ";locked mix set track: ~A" (mix-track mix1)))
		      (let ((old-amp (mix-amp mix1 0)))
			(catch 'no-such-mix
			       (lambda ()
				 (set! (mix-amp mix1 0) 123.0))
			       (lambda args args))
			(if (fneq (mix-amp mix1 0) old-amp) (snd-display ";locked mix set amp: ~A ~A" old-amp (mix-amp mix1 0))))
		      (let ((old-speed (mix-speed mix1)))
			(catch 'no-such-mix
			       (lambda ()
				 (set! (mix-speed mix1) 123.0))
			       (lambda args args))
			(if (fneq (mix-speed mix1) old-speed) (snd-display ";locked mix set speed: ~A ~A" old-speed (mix-speed mix1))))
		      (let ((old-position (mix-position mix1)))
			(catch 'no-such-mix 
			       (lambda ()
				 (set! (mix-position mix1) 123))
			       (lambda args args))
			(if (not (= (mix-position mix1) old-position)) (snd-display ";locked mix set position: ~A ~A" old-position (mix-position mix1))))
		      (undo)
		      (if (fneq (maxamp ind 0) 0.2) (snd-display ";undo delete-track maxamp: ~A" (maxamp ind 0)))
		      (if (or (mix-locked? mix1) (mix-locked? mix2))
			  (snd-display ";undo delete-track didn't unlock mixes? ~A ~A" (mix-locked? mix1) (mix-locked? mix2)))
		      ))))
	      (revert-sound ind)))
	  (let* ((mix1 (mix-vct (make-vct 100 .2) 50))
		 (mix2 (mix-vct (make-vct 100 .2) 250))
		 (mix3 (mix-vct (make-vct 100 .2) 500))
		 (track3 (make-track mix1 mix2 mix3))
		 (edpos (edit-position ind 0)))
	    (lock-track track3)
	    (if (or (not (mix-locked? mix1))
		    (not (mix-locked? mix2))
		    (not (mix-locked? mix3)))
		(snd-display ";lock-track: ~A ~A ~A" (mix-locked? mix1) (mix-locked? mix2) (mix-locked? mix3)))
	    (if (not (equal? (track track3) '())) (snd-display ";locked track: ~A" (track track3)))
	    (if (not (= (edit-position ind 0) (1+ edpos))) (snd-display ";lock track not atomic?: ~A ~A" edpos (edit-position ind 0)))
	    (close-sound ind)
	    (if (not (equal? (track track3) '())) (snd-display ";close-sound unset track: ~A ~A" (track track3) (map mix? (track track3)))))
	  ;; stereo track
	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 2 "track tests" 1000))
	  (let* ((mixid (mix "2a.snd" 100 #t ind))
		 (trk (mix-track mixid))
		 (mixids (track trk)))
	    (if (not (> (mix-track mixid) 0)) (snd-display ";multimix 1: ~A ~A ~A" (mix-track mixid) trk mixids))
	    (if (not (apply = (map mix-track mixids))) (snd-display ";multimix 2: ~A" (map mix-track mixids)))
	    (if (not (= 1 (edit-position ind 0) (edit-position ind 1)))
		(snd-display ";multimix 3 edpos: ~A ~A" (edit-position ind 0) (edit-position ind 1)))
	    (if (or (not (equal? (edit-fragment 1 ind 0) (list "mix" "set" 100 4412)))
		    (not (equal? (edit-fragment 1 ind 1) (list "mix" "set" 100 4412))))
		(snd-display ";multimix 4 origin? ~A ~A" (edit-fragment 1 ind 0) (edit-fragment 1 ind 1)))
	    (if (not (= (track-chans trk) 2)) (snd-display ";track-chans stereo mix: ~A" (track-chans trk)))

	    ;; track properties
	    (set! (track-property :hiho trk) 123)
	    (if (not (= (track-property :hiho trk) 123)) (snd-display ";track-property: ~A" (track-property trk)))
	    (if (track-property :not-there trk) (snd-display ";track-not-property: ~A" (track-property :not-there trk)))
	    (set! (track-property :hi trk) "hi")
	    (if (not (string=? (track-property :hi trk) "hi")) (snd-display ";track(1)-property: ~A" (track-property :hi trk)))

	    ;; 2chan basic stuff
	    (let ((max0 (maxamp ind 0))
		  (max1 (maxamp ind 1)))
	      (set! (track-amp trk) 2.0)
	      (if (or (fneq (maxamp ind 0) (* 2 max0))
		      (fneq (maxamp ind 1) (* 2 max1)))
		  (snd-display ";2chn track-amp: ~A ~A -> ~A ~A" max0 max1 (maxamp ind 0) (maxamp ind 1)))
	      (if (not (= 2 (edit-position ind 0) (edit-position ind 1)))
		  (snd-display ";2chn amp edpos: ~A ~A" (edit-position ind 0) (edit-position ind 1)))
	      (undo 1 ind 0)
	      (undo 1 ind 1)
	      (set! (track-position trk) 500)
	      (if (not (= (track-position trk) 500)) (snd-display ";2chn track-position: ~A" (track-position trk)))
	      (if (or (not (apply = (map mix-position mixids)))
		      (not (= (mix-position (car mixids)) 500)))
		  (snd-display ";2chn track-position mixes: ~A" (map mix-position mixids)))
	      (if (not (= 2 (edit-position ind 0) (edit-position ind 1)))
		  (snd-display ";2chn position edpos: ~A ~A" (edit-position ind 0) (edit-position ind 1)))
	      (undo 1 ind 0)
	      (undo 1 ind 1)
	      (let ((old-frames (frames)))
		(set! (track-speed trk) .5)
		(if (not (= (frames ind 0) (frames ind 1) (+ 100 (* 2 (- old-frames 100))))) ; mix is at 100
		    (snd-display ";2chn speed: ~A ~A ~A" (frames ind 0) (frames ind 1) (* 2 old-frames)))
		(if (not (= 2 (edit-position ind 0) (edit-position ind 1)))
		    (snd-display ";2chn speed edpos: ~A ~A" (edit-position ind 0) (edit-position ind 1)))
		(undo 1 ind 0)
		(undo 1 ind 1))
	      (set! (track-amp-env trk) '(0 1 .1 2 .9 2 1 0))
	      (if (or (fneq (* 2 max0) (maxamp ind 0))
		      (fneq (* 2 max1) (maxamp ind 1)))
		  (snd-display ";2ch track-env: ~A ~A -> ~A ~A" max0 max1 (maxamp ind 0) (maxamp ind 1)))
	      (if (not (= 2 (edit-position ind 0) (edit-position ind 1)))
		  (snd-display ";2chn env edpos: ~A ~A" (edit-position ind 0) (edit-position ind 1)))
	      (undo 1 ind 0)
	      (undo 1 ind 1))
	    (close-sound ind)
	    (if (not (equal? (track trk) '())) (snd-display ";close-sound unset track (2): ~A ~A" (track trk) (map mix? (track trk)))))

	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track tests" 1000))      
	  (let* ((track4 (make-track))
		 (mixid (mix-vct (make-vct 10 .5) 100 ind 0 #t "snd-test" track4)))
	    (if (not (= (mix-track mixid) track4)) (snd-display ";mix-vct with track: ~A" (mix-track mixid)))
	    (if (not (= 1 (edit-position ind 0)))
		(snd-display ";mix-vct w/track not atomic?: ~A" (edit-position ind 0)))
	    (if (fneq (maxamp ind 0) 0.5) (snd-display ";mix-vct+track maxamp: ~A" (maxamp ind 0)))
	    (if (fneq (mix-maxamp mixid) 0.5) (snd-display ";mix-maxamp: ~A" (mix-maxamp mixid)))
	    (if (fneq (track-maxamp track4 0) 0.5) (snd-display ";track-maxamp: ~A" (track-maxamp track4 0)))
	    (undo)
	    (if (fneq (maxamp ind 0) 0.0) (snd-display ";mix-vct+track undo maxamp: ~A" (maxamp ind 0)))
	    (redo)
	    (if (not (= (mix-track mixid) track4)) (snd-display ";mix-vct with track (redo): ~A" (mix-track mixid)))
	    (if (fneq (maxamp ind 0) 0.5) (snd-display ";mix-vct+track redo maxamp: ~A" (maxamp ind 0)))
	    (let ((amix (mix "1a.snd" 200 0 ind 0 #t #f track4)))
	      (if (not (= (mix-track amix) track4)) (snd-display ";mix with track: ~A" (mix-track amix)))
	      (if (not (= 2 (edit-position ind 0)))
		  (snd-display ";mix w/track not atomic?: ~A" (edit-position ind 0)))
	      (if (fneq (maxamp ind 0) 0.5) (snd-display ";mix+track maxamp: ~A" (maxamp ind 0)))
	      (undo)
	      (if (fneq (maxamp ind 0) 0.5) (snd-display ";mix+track undo maxamp: ~A" (maxamp ind 0)))
	      (redo)
	      (if (not (= (mix-track amix) track4)) (snd-display ";mix with track (redo): ~A" (mix-track amix)))
	      (if (fneq (maxamp ind 0) 0.5) (snd-display ";mix+track redo maxamp: ~A" (maxamp ind 0)))
	      (if (not (equal? (track track4) (list mixid amix))) (snd-display ";mix-vct+mix+track: ~A" (track track4)))
	      (let ((rid (make-region 100 110 ind 0)))
		;; should be ok because with-mix-tags was turned on above
		(if (fneq (region-maxamp rid) 0.5) (snd-display ";region(mix) picked up wrong section? ~A" (region-maxamp rid)))
		(let ((rmix (mix-region 900 rid ind 0 track4)))
		  (if (not (= (mix-track rmix) track4)) (snd-display ";mix-region with track: ~A" (mix-track rmix)))
		  (if (not (= 3 (edit-position ind 0)))
		      (snd-display ";mix-region w/track not atomic?: ~A" (edit-position ind 0)))
		  (if (fneq (maxamp ind 0) 0.56) (snd-display ";mix-region+track maxamp: ~A" (maxamp ind 0)))
		  (undo)
		  (if (fneq (maxamp ind 0) 0.5) (snd-display ";mix-region+track undo maxamp: ~A" (maxamp ind 0)))
		  (redo)
		  (if (not (= (mix-track rmix) track4)) (snd-display ";mix-region with track (redo): ~A" (mix-track rmix)))
		  (if (fneq (maxamp ind 0) 0.56) (snd-display ";mix-region+track redo maxamp: ~A" (maxamp ind 0)))
		  (if (not (equal? (track track4) (list mixid amix rmix))) (snd-display ";mix-vct+mix-region+mix+track: ~A" (track track4)))
		  (make-selection 400 500 ind 0)
		  (let ((smix (mix-selection 4000 ind 0 track4)))
		    (if (not (= (mix-track smix) track4)) (snd-display ";mix-selection with track: ~A" (mix-track smix)))
		    (if (not (= 4 (edit-position ind 0)))
			(snd-display ";mix-selection w/track not atomic?: ~A" (edit-position ind 0)))
		    (if (fneq (maxamp ind 0) 0.56) (snd-display ";mix-selection+track maxamp: ~A" (maxamp ind 0)))
		    (undo)
		    (if (fneq (maxamp ind 0) 0.56) (snd-display ";mix-selection+track undo maxamp: ~A" (maxamp ind 0)))
		    (redo)
		    (if (not (= (mix-track smix) track4)) (snd-display ";mix-selection with track (redo): ~A" (mix-track smix)))
		    (if (fneq (maxamp ind 0) 0.56) (snd-display ";mix-selection+track redo maxamp: ~A" (maxamp ind 0)))
		    (if (not (equal? (track track4) (list mixid amix rmix smix))) (snd-display ";mix-vct+mix-selection+mix+track: ~A" (track track4)))
		    (set! (track-amp track4) .5)
		    (if (fneq (maxamp ind 0) 0.28) (snd-display ";mix-selection+track reset amp maxamp: ~A" (maxamp ind 0)))
		    (let ((bmix (mix-vct (make-vct 10 .75) 4300 ind 0 #t "snd-test" track4)))
		      (if (not (= (mix-track bmix) track4)) (snd-display ";mix-vct with track amp: ~A" (mix-track smix)))
		      (if (not (= 6 (edit-position ind 0)))
			  (snd-display ";mix-vct w/track amp not atomic?: ~A" (edit-position ind 0)))
		      (if (fneq (maxamp ind 0) 0.375) (snd-display ";mix-vct+track amp maxamp: ~A" (maxamp ind 0)))
		      (if (not (equal? (track track4) (list mixid amix rmix smix bmix))) 
			  (snd-display ";mix-vct+mix-selection+mix+track amp: ~A" (track track4))))
		    (delete-track track4)
		    (if (fneq (maxamp ind 0) 0.0) (snd-display ";maxamp delete-track4: ~A" (maxamp ind 0))))))))
	  (close-sound ind)
	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 2 "track tests" 1000))      
	  (let* ((trk (make-track))
		 (mix1 (mix-vct (make-vct 10 .5) 10 ind 0 #t "snd-test" trk))
		 (mix2 (mix-vct (make-vct 10 .25) 20 ind 1 #t "snd-test" trk))
		 (mix3 (mix-vct (make-vct 10 .3) 30 ind 0 #t "snd-test" trk))
		 (mix4 (mix-vct (make-vct 10 .1) 40 ind 1 #t "snd-test" trk)))
	    (if (not (= (track-chans trk) 2)) (snd-display ";track-chans 4 mix-vct: ~A ~A" (track-chans trk) (map mix-home (track trk))))
	    (if (fneq (maxamp ind 0) .5) (snd-display ";maxamp mix-vct 4 0: ~A" (maxamp ind 0)))
	    (if (fneq (maxamp ind 1) .25) (snd-display ";maxamp mix-vct 4 1: ~A" (maxamp ind 1)))
	    (if (not (= (track-position trk) 10)) (snd-display ";overall track-position: ~A" (track-position trk)))
	    (if (not (= (track-frames trk) 40)) (snd-display ";overall track-frames: ~A" (track-frames trk)))
	    (if (not (= (track-position trk 0) 10)) (snd-display ";chn0 track-position: ~A" (track-position trk 0)))
	    (if (not (= (track-position trk 1) 20)) (snd-display ";chn1 track-position: ~A" (track-position trk 1)))
	    (if (not (= (track-frames trk 0) 30)) (snd-display ";chn0 track-frames: ~A" (track-frames trk 0)))
	    (if (not (= (track-frames trk 1) 30)) (snd-display ";chn1 track-frames: ~A" (track-frames trk 1)))
	    (if (not (equal? (track trk 0) (list mix1 mix3))) (snd-display ";track + chan0: ~A ~A" (track trk 0) (track trk)))
	    (if (not (equal? (track trk 1) (list mix2 mix4))) (snd-display ";track + chan1: ~A ~A" (track trk 1) (track trk)))
	    (set! (track-position trk) 0)
	    (if (not (= (track-position trk) 0)) (snd-display ";overall track-position 0: ~A" (track-position trk)))
	    (if (not (= (track-frames trk) 40)) (snd-display ";overall track-frames 0: ~A" (track-frames trk)))
	    (if (not (= (track-position trk 0) 0)) (snd-display ";chn0 track-position 0: ~A" (track-position trk 0)))
	    (if (not (= (track-position trk 1) 10)) (snd-display ";chn1 track-position 0: ~A" (track-position trk 1)))
	    (if (not (= (track-frames trk 0) 30)) (snd-display ";chn0 track-frames 0: ~A" (track-frames trk 0)))
	    (if (not (= (track-frames trk 1) 30)) (snd-display ";chn1 track-frames 0: ~A" (track-frames trk 1)))
	    (if (not (= (mix-position mix3) 20)) (snd-display ";chn0 track-position mix3: ~A" (mix-position mix3)))
	    (if (not (= (mix-position mix4) 30)) (snd-display ";chn1 track-position mix4: ~A" (mix-position mix4)))
	    (set! (track-position trk 0) 20)
	    (if (not (= (track-position trk) 10)) (snd-display ";overall track-position 1: ~A" (track-position trk)))
	    (if (not (= (track-frames trk) 40)) (snd-display ";overall track-frames 1: ~A" (track-frames trk)))
	    (if (not (= (track-position trk 0) 20)) (snd-display ";chn0 track-position 1: ~A" (track-position trk 0)))
	    (if (not (= (track-position trk 1) 10)) (snd-display ";chn1 track-position 1: ~A" (track-position trk 1)))
	    (if (not (= (track-frames trk 0) 30)) (snd-display ";chn0 track-frames 1: ~A" (track-frames trk 0)))
	    (if (not (= (track-frames trk 1) 30)) (snd-display ";chn1 track-frames 1: ~A" (track-frames trk 1)))
	    (if (not (= (mix-position mix3) 40)) (snd-display ";chn0 track-position mix3 1: ~A" (mix-position mix3)))
	    (if (not (= (mix-position mix4) 30)) (snd-display ";chn1 track-position mix4 1: ~A" (mix-position mix4)))
	    (set! (track-position trk 1) 20)
	    (if (not (= (track-position trk) 20)) (snd-display ";overall track-position 2: ~A" (track-position trk)))
	    (if (not (= (track-frames trk) 30)) (snd-display ";overall track-frames 2: ~A" (track-frames trk)))
	    (if (not (= (track-position trk 0) 20)) (snd-display ";chn0 track-position 2: ~A" (track-position trk 0)))
	    (if (not (= (track-position trk 1) 20)) (snd-display ";chn1 track-position 2: ~A" (track-position trk 1)))
	    (if (not (= (track-frames trk 0) 30)) (snd-display ";chn0 track-frames 2: ~A" (track-frames trk 0)))
	    (if (not (= (track-frames trk 1) 30)) (snd-display ";chn1 track-frames 2: ~A" (track-frames trk 1)))
	    (if (not (= (mix-position mix3) 40)) (snd-display ";chn0 track-position mix3 2: ~A" (mix-position mix3)))
	    (if (not (= (mix-position mix4) 40)) (snd-display ";chn1 track-position mix4 2: ~A" (mix-position mix4)))
	    (close-sound ind))
	  
	  (let* ((ind0 (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track tests" 60))
		 (ind1 (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track tests" 60)))
	    (mix-vct (make-vct 3 .5) 10 ind0 0 #f)
	    (mix-vct (make-vct 3 .2) 40 ind0 0 #f)
	    (let* ((mix1 (mix-vct (make-vct 3 .5) 10 ind1 0 #t))
		   (mix2 (mix-vct (make-vct 3 .2) 40 ind1 0 #t))
		   (trk (make-track mix1 mix2)))
	      (filter-sound (vct .1 .2 .3 .3 .2 .1) 6 ind0 0)
	      (filter-track trk (list .1 .2 .3 .3 .2 .1))
	      (if (not (= (edit-position ind0 0) 3)) (snd-display ";filter-sound edpos: ~A" (edit-position ind0 0)))
	      (if (not (= (edit-position ind1 0) 4)) (snd-display ";filter-track edpos: ~A" (edit-position ind1 0)))
	      (if (or (fneq (maxamp ind0 0) .4)
		      (fneq (maxamp ind1 0) .4))
		  (snd-display ";filter-track maxamps: ~A ~A" (maxamp ind0 0) (maxamp ind 0)))
	      (if (not (vvequal (channel->vct 0 50 ind0 0) (channel->vct 0 50 ind1 0)))
		  (snd-display ";filters not the same: ~A ~A" (channel->vct 0 50 ind0 0) (channel->vct 0 50 ind1 0)))
	      (close-sound ind0)
	      (close-sound ind1)))

	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 4 "track->vct tests" 1000))      
	  (let* ((trk (make-track))
		 (mix1 (mix-vct (make-vct 10 .5) 10 ind 0 #t "snd-test" trk))
		 (mix2 (mix-vct (make-vct 10 .25) 20 ind 1 #t "snd-test" trk))
		 (mix3 (mix-vct (make-vct 10 .3) 30 ind 2 #t "snd-test" trk))
		 (mix4 (mix-vct (make-vct 10 .1) 40 ind 1 #t "snd-test" trk)))
	    (if (not (= (track-chans trk) 3)) (snd-display ";tracks chans arg: ~A" (track-chans trk)))
	    (let ((v (track->vct trk 0)))
	      (if (not (vvequal v (make-vct 10 .5))) (snd-display ";track->vct 0: ~A" v))
	      (set! v (track->vct trk 2))
	      (if (not (vvequal v (make-vct 10 .3))) (snd-display ";track->vct 2: ~A" v))
	      (set! v (track->vct trk 1))
	      (let ((v1 (make-vct 30 0.0)))
		(do ((i 0 (1+ i))
		     (j 20 (1+ j)))
		    ((= i 10))
		  (vct-set! v1 i .25)
		  (vct-set! v1 j .1))
		(if (not (vvequal v v1)) (snd-display ";track->vct 3: ~A" v))))
	    (let ((tag (catch #t (lambda () (track->vct trk 3)) (lambda args (car args)))))
	      (if (not (eq? tag 'no-such-channel)) (snd-display ";track->vct track 3: ~A" tag)))
	    (let ((tag (catch #t (lambda () (track->vct (+ trk 123))) (lambda args (car args)))))
	      (if (not (eq? tag 'no-such-track)) (snd-display ";track->vct untrack: ~A" tag)))
	    (save-track trk "test.snd")
	    (let ((ind0 (open-sound "test.snd")))
	      (if (not (= (chans ind0) 3)) (snd-display ";save-track chans: ~A" (chans ind0)))
	      (if (not (= (frames ind0) 40)) (snd-display ";save-track frames: ~A" (frames ind0)))
	      (if (not (feql (maxamp ind0 #t) (list .5 .25 .3))) (snd-display ";save-track maxamp: ~A" (maxamp ind0 #t)))
	      (let ((v (channel->vct 0 20 ind0 0))
		    (v1 (make-vct 20)))
		(do ((i 0 (1+ i))) ((= i 10)) (vct-set! v1 i .5))
		(if (not (vvequal v v1)) (snd-display ";save-track 0: ~A" v)))
	      (let ((tag (catch #t (lambda () (save-track trk "test.snd" 3)) (lambda args (car args)))))
		(if (not (eq? tag 'no-such-channel)) (snd-display ";save-track track 3: ~A" tag)))
	      (let ((tag (catch #t (lambda () (save-track (+ trk 123) "test.snd")) (lambda args (car args)))))
		(if (not (eq? tag 'no-such-track)) (snd-display ";save-track untrack: ~A" tag)))
	      (close-sound ind0))
	    (save-track trk "test.snd" 1)
	    (let ((ind0 (open-sound "test.snd")))
	      (if (not (= (chans ind0) 1)) (snd-display ";save-track chan1: ~A" (chans ind0)))
	      (if (not (= (frames ind0) 30)) (snd-display ";save-track chan1 frames: ~A" (frames ind0)))
	      (if (fneq (maxamp ind0 0) .25) (snd-display ";save-track maxamp chan1: ~A" (maxamp ind0 0)))
	      (let ((v (channel->vct 0 30 ind0 0))
		    (v1 (make-vct 30)))
		(do ((i 0 (1+ i))
		     (j 20 (1+ j)))
		    ((= i 10))
		  (vct-set! v1 i .25)
		  (vct-set! v1 j .1))
		(if (not (vvequal v v1)) (snd-display ";save-track chan1: ~A" v)))
	      (close-sound ind0))

	    (close-sound ind)
	    (let ((tag (catch #t (lambda () (save-track trk "test.snd")) (lambda args (car args)))))
	      (if (not (eq? tag 'no-such-channel)) (snd-display ";save-track empty track: ~A" tag)))
	    (let ((tag (catch #t (lambda () (save-track trk "test.snd" 1)) (lambda args (car args)))))
	      (if (not (eq? tag 'no-such-channel)) (snd-display ";save-track empty track (1): ~A" tag)))
	    (let ((tag (catch #t (lambda () (track->vct trk)) (lambda args (car args)))))
	      (if (not (eq? tag 'no-such-channel)) (snd-display ";track->vct empty track: ~A" tag)))
	    (let ((tag (catch #t (lambda () (track->vct trk 1)) (lambda args (car args)))))
	      (if (not (eq? tag 'no-such-channel)) (snd-display ";track->vct empty track (1): ~A" tag))))

	  (if (provided? 'snd-debug)
	      ;;window_env: env *e, off_t local_beg, off_t local_dur, off_t e_beg, off_t e_dur
	      ;;multiply_envs: env *e1, env *e2, Float maxx
	      (begin
		(if (not (feql (window-env '(0 0 1 1) 0 100 0 500) '(0.0 0.0 1.0 0.2))) 
		    (snd-display ";window-env 1: ~A" (window-env '(0 0 1 1) 0 100 0 500)))
		(if (not (feql (window-env '(0 0 1 1) 0 500 0 500) '(0.0 0.0 1.0 1.0))) 
		    (snd-display ";window-env 2: ~A" (window-env '(0 0 1 1) 0 500 0 500)))
		(if (not (feql (window-env '(0 0 1 1) 100 200 0 500) '(0.0 0.2 1.0 0.6)))
		    (snd-display ";window-env 3: ~A" (window-env '(0 0 1 1) 100 200 0 500)))
		(if (not (feql (window-env '(0 0 1 1 2 0) 100 300 0 500) '(0.0 0.4 0.5 1.0 1.0 0.4)))
		    (snd-display ";window-env 4: ~A" (window-env '(0 0 1 1 2 0) 100 300 0 500)))
		(if (not (feql (window-env '(0 0 1 1 2 0) 100 400 0 500) '(0.0 0.4 0.375 1.0 1.0 0.0)))
		    (snd-display ";window-env 4: ~A" (window-env '(0 0 1 1 2 0) 100 400 0 500)))

		(if (not (feql (invert-env '(0 0 1 1)) '(0 1 1 0)))
		    (snd-display ";invert-env ramp: ~A" (invert-env '(0 0 1 1))))
		(if (not (feql (invert-env '(0 0 1 1 2 0.5)) '(0 1 1 0 2 0.5)))
		    (snd-display ";invert-env ramp 1: ~A" (invert-env '(0 0 1 1 2 0.5))))

		(if (not (feql (multiply-envs '(0 0 1 1) '(0 0 1 1) .2) '(0.0 0.0 0.2 0.04 0.4 0.16 0.6 0.36 0.8 0.64 1.0 1.0)))
		    (snd-display ";multiply-envs: ~A" (multiply-envs '(0 0 1 1) '(0 0 1 1) .2)))
		(if (not (feql (multiply-envs '(0 0 1 1) '(0 0 .5 1 1 0) .2) '(0.0 0.0 0.2 0.08 0.4 0.32 0.5 0.5 0.7 0.42 0.9 0.18 1.0 0.0)))
		    (snd-display ";multiply-envs 1: ~A" (multiply-envs '(0 0 1 1) '(0 0 .5 1 1 0) .2)))))
	  
	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track amp-env tests" 300))
	  (let ((track0 (make-track)))
	    (set! (track-amp-env track0) '(0 0 1 1))
	    (set! (track-position track0) 123)
	    (set! (track-speed track0) .5)
	    (set! (track-speed track0) 1.0)
	    (let ((mix1 (mix-vct (make-vct 10 1.0) 10)))
	      (set! (mix-track mix1) track0)
	      (let ((tdata (track->vct track0))
		    (mdata (mix->vct mix1)))
		(if (not (vequal tdata mdata)) (snd-display ";1 mix track: ~A ~A" tdata mdata)))
	      (set! (mix-position mix1) 30)
	      (if (not (= (track-position track0) 30)) (snd-display ";mix-pos change track-pos: ~A ~A" (mix-position mix1) (track-position track0)))
	      ))
	  (close-sound ind)

	  (set! ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track amp-env tests" 300))      
	  (let* ((trk (make-track))
		 (mix1 (mix-vct (make-vct 100 .5) 0 ind 0 #t "snd-test" trk))
		 (mix2 (mix-vct (make-vct 100 .3) 200 ind 0 #t "snd-test" trk))
		 (edpos (edit-position ind 0)))

	    (define (veqerr v0 v1 err)
	      (let ((this-err (vct-peak (vct-subtract! (vct-copy v0) v1))))
		;(if (< (* 2 this-err) err) (snd-display ";veqerr: ~A ~A" this-err err))
		(<= this-err err)))

	    (let ((data (channel->vct 0 300 ind 0))
		  (vdata (make-vct 300))
		  (e (make-env '(0 0 1 1) :end 299)))
	      (do ((i 0 (1+ i))
		   (j 200 (1+ j)))
		  ((= i 100))
		(vct-set! vdata i .5)
		(vct-set! vdata j .3))
	      (if (not (veqerr data vdata .00001)) (snd-display ";track not yet ramped: ~A ~A" data vdata))
	      (set! (track-amp-env trk) '(0 0 1 1))
	      (do ((i 0 (1+ i))) ((= i 300)) (vct-set! vdata i (* (vct-ref vdata i) (env e))))
	      (set! data (channel->vct 0 300 ind 0))
	      (if (not (veqerr data vdata .002)) (snd-display ";track-amp-env ramped: ~A ~A" data vdata))
	      (if (not (= (edit-position ind 0) (1+ edpos))) (snd-display ";track-amp-env ramped not atomic? ~A ~A" edpos (edit-position ind 0)))
	      (set! (mix-amp-env mix1) '(0 1 1 0))
	      (if (not (= (edit-position ind 0) (+ edpos 2))) (snd-display ";mix-amp-env ramped not atomic? ~A ~A" edpos (edit-position ind 0)))
	      (set! data (channel->vct 0 300 ind 0))
	      (set! e (make-env '(0 1 1 0) :end 99))
	      (do ((i 0 (1+ i))) ((= i 100)) (vct-set! vdata i (* (vct-ref vdata i) (env e))))
	      (if (not (veqerr data vdata .001)) (snd-display ";track-amp-env + mix-amp-env ramped: ~A ~A" data vdata))
	      (set! (mix-amp mix1) 2.0)
	      (if (not (= (edit-position ind 0) (+ edpos 3))) (snd-display ";mix-amp ramped not atomic? ~A ~A" edpos (edit-position ind 0)))
	      (set! data (channel->vct 0 300 ind 0))
	      (do ((i 0 (1+ i))) ((= i 100)) (vct-set! vdata i (* 2 (vct-ref vdata i))))
	      (if (not (veqerr data vdata .001)) (snd-display ";track-amp-env + mix-amp-env + amp ramped: ~A ~A" data vdata))
	      (set! (track-position trk) 100)
	      (if (not (= (edit-position ind 0) (+ edpos 4))) (snd-display ";track-pos+amp-env ramped not atomic? ~A ~A" edpos (edit-position ind 0)))
	      (if (not (= (frames ind 0) 400)) (snd-display ";set track position extend file: ~A" (frames ind 0)))
	      (if (not (= (track-frames trk) 300)) (snd-display ";set track position extend file track: ~A" (track-frames trk)))
	      (set! data (channel->vct 100 300 ind 0))
	      (if (not (veqerr data vdata .001)) (snd-display ";track-amp-env + mix-amp-env + amp ramped + position: ~A ~A" data vdata))	      

	      (set! (mix-amp-env mix1) #f)
	      (set! (mix-amp mix2 0) (/ 10.0 3.0))
	      (set! e (make-env '(0 0 1 1) :end 299))
	      (do ((i 0 (1+ i))
		   (j 200 (1+ j)))
		  ((= i 100))
		(vct-set! vdata i 1.0)
		(vct-set! vdata j 1.0))
	      (do ((i 0 (1+ i)))
		  ((= i 300))
		(vct-set! vdata i (* (env e) (vct-ref vdata i))))
	      (set! data (channel->vct 100 300 ind 0))
	      (if (not (veqerr data vdata .003)) (snd-display ";track-amp-env to 1.0: ~A ~A" data vdata))	      
	      (set! (track-speed trk) 0.5)
	      (if (not (= (track-frames trk) 400)) (snd-display ";track frames after speed+amp-env: ~A" (track-frames trk)))
	      (if (not (= (frames ind 0) 500)) (snd-display ";set track speed extend file: ~A" (frames ind 0)))
	      (if (or (fneq (sample 100) 0.0)
		      (fneqerr (sample 200) 0.25 .0015)
		      (fneqerr (sample 400) 0.75 .0015))
		  (snd-display ";track-amp-env+speed0.5 samps: ~A ~A ~A" (sample 100) (sample 200) (sample 400)))
	      
	      (revert-sound ind)
	      (set! mix1 (mix-vct (make-vct 100 1.0) 0 ind 0 #t "snd-test" trk))
	      (set! mix2 (mix-vct (make-vct 100 1.0) 100 ind 0 #t "snd-test" trk))
	      (if (not (equal? (track trk) (list mix1 mix2))) (snd-display ";unset track upon revert? ~A ~A" (track trk) (list mix1 mix2)))
	      (set! (track-amp-env trk) '(0 0 1 1))
	      (if (or (fneq (sample 0) 0.0)
		      (fneq (sample 50) 0.252)
		      (fneq (sample 99) 0.5)
		      (fneq (sample 100) 0.5)
		      (fneq (sample 199) 1.0))
		  (snd-display ";mix-speed/position track + track-amp-env: ~A"
			       (map sample (list 0 50 99 100 199))))
	      (set! (mix-speed mix2) 0.25)
	      (if (or (fneq (sample 0) 0.0)
		      (fneq (sample 50) 0.101)
		      (fneq (sample 99) 0.2)
		      (fneq (sample 100) 0.2)
		      (fneq (sample 200) 0.4)
		      (fneqerr (sample 300) 0.6 .003)
		      (fneqerr (sample 400) 0.801 .01)
		      (fneqerr (sample 450) 0.900 .01))
		  (snd-display ";mix-speed lengthens track + track-amp-env: ~A"
			       (map sample (list 0 50 99 100 200 300 400 450))))
	      (undo)
	      (set! (mix-speed mix2) 2.0)
	      (if (or (fneq (sample 0) 0.0)
		      (fneq (sample 50) 0.337)
		      (fneq (sample 99) 0.667)
		      (fneqerr (sample 110) 0.730 .1) ; (try to) avoid end points that are sinc functions
		      (fneqerr (sample 135) 0.900 .1) ; da same
		      (fneq (sample 151) 0.0)
		      (fneq (sample 200) 0.0))
		  (snd-display ";mix-speed shortens track + track-amp-env: ~A"
			       (map sample (list 0 50 99 110 135 151 200))))
	      (undo)
	      (set! (mix-position mix2) 400)
	      (if (or (fneq (sample 0) 0.0)
		      (fneq (sample 50) 0.101)
		      (fneq (sample 99) 0.2)
		      (fneq (sample 100) 0.0)
		      (fneq (sample 200) 0.0)
		      (fneq (sample 300) 0.0)
		      (fneq (sample 400) 0.8)
		      (fneq (sample 450) 0.901))
		  (snd-display ";mix-position lengthens track + track-amp-env: ~A"
			       (map sample (list 0 50 99 100 200 300 400 450))))
	      (undo)
	      (set! (track-position trk) 300)
	      (set! (mix-position mix1) 0)
	      (if (or (fneq (sample 0) 0.0)
		      (fneq (sample 50) 0.101)
		      (fneq (sample 99) 0.2)
		      (fneq (sample 100) 0.0)
		      (fneq (sample 200) 0.0)
		      (fneq (sample 300) 0.0)
		      (fneq (sample 400) 0.8)
		      (fneq (sample 450) 0.901))
		  (snd-display ";mix-position (backwards) lengthens track + track-amp-env: ~A"
			       (map sample (list 0 50 99 100 200 300 400 450))))
	      (undo 2)
	      (set! (mix-position mix2) 50)
	      (if (or (fneq (sample 0) 0.0)   ; 1 is 0 to .67, 2 is .3 to 1
		      (fneq (sample 49) 0.33)
		      (fneq (sample 50) 0.67) ; both are at .3
		      (fneq (sample 99) 1.33) ; both .67
		      (fneq (sample 100) 0.67)
		      (fneq (sample 149) 1.0)
		      (fneq (sample 200) 0.0))
		  (snd-display ";mix-position 2 shortens track + track-amp-env: ~A"
			       (map sample (list 0 49 50 99 100 149 200))))
	      (undo)
	      (set! (mix-position mix1) 100)
	      (if (or (fneq (sample 0) 0.0)
		      (fneq (sample 50) 0.0)
		      (fneq (sample 99) 0.0)
		      (fneq (sample 100) 0.0)
		      (fneq (sample 150) 1.01)
		      (fneq (sample 199) 2.0)
		      (fneq (sample 200) 0.0))
		  (snd-display ";mix-position 1 shortens track + track-amp-env: ~A"
			       (map sample (list 0 50 99 100 150 199 200))))
	      (undo)

	      ))
	  (close-sound ind)

	  (let* ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 2 "multi-channel track position tests" 300))
		 (mix1 (mix-vct (make-vct 10 1.0) 0 ind 0))
		 (mix2 (mix-vct (make-vct 10 1.0) 200 ind 1))
		 (trk (make-track mix1 mix2)))
	    (if (not (= (track-chans trk) 2)) (snd-display ";2chan track-position tests off to bad start: ~A" (track-chans trk)))
	    (if (or (not (= (track-position trk 0) 0))
		    (not (= (track-position trk 1) 200))
		    (not (= (track-position trk) 0))
		    (not (= (mix-position mix1) 0))
		    (not (= (mix-position mix2) 200)))
		(snd-display ";2chan track-position pos: ~A ~A ~A ~A ~A" 
			     (track-position trk 0) (track-position trk 1) (track-position trk)
			     (mix-position mix1) (mix-position mix2)))
	    (set! (track-position trk 0) 25)
	    (if (or (not (= (track-position trk 0) 25))
		    (not (= (track-position trk 1) 200))
		    (not (= (track-position trk) 25))
		    (not (= (mix-position mix1) 25))
		    (not (= (mix-position mix2) 200)))
		(snd-display ";2chan track-position pos 2: ~A ~A ~A ~A ~A" 
			     (track-position trk 0) (track-position trk 1) (track-position trk)
			     (mix-position mix1) (mix-position mix2)))
	    (set! (track-position trk 1) 100)
	    (if (or (not (= (track-position trk 0) 25))
		    (not (= (track-position trk 1) 100))
		    (not (= (track-position trk) 25))
		    (not (= (mix-position mix1) 25))
		    (not (= (mix-position mix2) 100)))
		(snd-display ";2chan track-position pos 3: ~A ~A ~A ~A ~A" 
			     (track-position trk 0) (track-position trk 1) (track-position trk)
			     (mix-position mix1) (mix-position mix2)))
	    (set! (track-position trk 1) 10)
	    (if (or (not (= (track-position trk 0) 25))
		    (not (= (track-position trk 1) 10))
		    (not (= (track-position trk) 10))
		    (not (= (mix-position mix1) 25))
		    (not (= (mix-position mix2) 10)))
		(snd-display ";2chan track-position pos 4: ~A ~A ~A ~A ~A" 
			     (track-position trk 0) (track-position trk 1) (track-position trk)
			     (mix-position mix1) (mix-position mix2)))
	    (set! (mix-position mix1) 0)
	    (set! (mix-position mix2) 200)
	    (if (or (not (= (track-position trk 0) 0))
		    (not (= (track-position trk 1) 200))
		    (not (= (track-position trk) 0))
		    (not (= (mix-position mix1) 0))
		    (not (= (mix-position mix2) 200)))
		(snd-display ";2chan track-position pos: ~A ~A ~A ~A ~A" 
			     (track-position trk 0) (track-position trk 1) (track-position trk)
			     (mix-position mix1) (mix-position mix2)))
	    (set! (track-amp-env trk) '(0 0 1 1))
	    (if (or (not (vequal (track->vct trk 0) (vct 0.000 0.005 0.011 0.016 0.021 0.026 0.032 0.037 0.042 0.048)))
		    (not (vequal (track->vct trk 1) (vct 0.952 0.958 0.963 0.968 0.974 0.979 0.984 0.989 0.995 1.000))))
		(snd-display ";2chan track-pos amp-env: ~A ~A" (track->vct trk 0) (track->vct trk 1)))
	    (set! (track-position trk 0) 100)
	    (if (or (not (vequal (track->vct trk 0) (vct 0.000 0.010 0.020 0.030 0.040 0.051 0.061 0.071 0.081 0.091)))
		    (not (vequal (track->vct trk 1) (vct 0.909 0.919 0.929 0.939 0.949 0.960 0.970 0.980 0.990 1.000))))
		(snd-display ";2chan track-pos amp-env 2: ~A ~A" (track->vct trk 0) (track->vct trk 1)))
	    (set! (track-position trk 1) 100)
	    (if (or (not (vequal (track->vct trk 0) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000)))
		    (not (vequal (track->vct trk 1) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000))))
		(snd-display ";2chan track-pos amp-env 3: ~A ~A" (track->vct trk 0) (track->vct trk 1)))
	    (set! (track-position trk 1) 0)
	    (if (or (not (vequal (track->vct trk 1) (vct 0.000 0.010 0.020 0.030 0.040 0.051 0.061 0.071 0.081 0.091)))
		    (not (vequal (track->vct trk 0) (vct 0.909 0.919 0.929 0.939 0.949 0.960 0.970 0.980 0.990 1.000))))
		(snd-display ";2chan track-pos amp-env 4: ~A ~A" (track->vct trk 0) (track->vct trk 1)))
	    (let ((mix3 (mix-vct (make-vct 10 1.0) 200 ind 1)))
	      (set! (mix-track mix3) trk))
	    (if (or (not (vequal (track->vct trk 0) (vct 0.476 0.481 0.487 0.492 0.497 0.503 0.508 0.513 0.519 0.524)))
		    (not (vequal (channel->vct 0 10 ind 1) (vct 0.000 0.005 0.011 0.016 0.021 0.026 0.032 0.037 0.042 0.048)))
		    (not (vequal (channel->vct 200 10 ind 1) (vct 0.952 0.958 0.963 0.968 0.974 0.979 0.984 0.989 0.995 1.000))))
		(snd-display ";2chan track-pos amp-env 5: ~A ~A" (track->vct trk 0) (track->vct trk 1)))
	    (let ((edpos (edit-position ind 1)))
	      (set! (track-position trk 0) 50)
	      (if (not (= (edit-position ind 1) edpos)) (snd-display ";set track pos changed edpos: ~A ~A" edpos (edit-position ind 1)))
	      (if (or (not (vequal (track->vct trk 0) (vct 0.238 0.243 0.249 0.254 0.259 0.265 0.270 0.275 0.280 0.286)))
		      (not (vequal (channel->vct 0 10 ind 1) (vct 0.000 0.005 0.011 0.016 0.021 0.026 0.032 0.037 0.042 0.048)))
		      (not (vequal (channel->vct 200 10 ind 1) (vct 0.952 0.958 0.963 0.968 0.974 0.979 0.984 0.989 0.995 1.000))))
		  (snd-display ";2chan track-pos amp-env 6: ~A ~A" (track->vct trk 0) (track->vct trk 1))))
	    (close-sound ind))

	  (let* ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "multi-channel track position tests" 300))
		 (mix1 (mix-vct (make-vct 10 1.0) 0 ind 0))
		 (trk (make-track)))
	    (set! (track-amp-env trk) '(0 0 1 1))
	    (if (not (feql (track-amp-env trk) (list 0 0 1 1)))
		(snd-display ";empty track env: ~A" (track-amp-env trk)))
	    (set! (mix-track mix1) trk)
	    (if (not (vequal (track->vct trk) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000)))
		(snd-display ";track amp-env 1 mix: ~A" (track->vct trk)))
	    (set! (track-position trk) 10)
	    (if (not (vequal (channel->vct 0 20 ind 0) (vct 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000 0.000
							    0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000)))
		(snd-display ";move 1 mix track + env: ~A" (channel->vct 0 20 ind 0)))
	    (if (not (vequal (mix->vct mix1) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000)))
		(snd-display ";track amp-env 1 mix->vct: ~A" (mix->vct mix1)))
	    (set! (mix-track mix1) 0)
	    (if (not (vequal (mix->vct mix1) (vct 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000)))
		(snd-display ";untrack mix -> vct: ~A" (mix->vct mix1)))
	    (if (not (feql (track-amp-env trk) (list 0 0 1 1)))
		(snd-display ";newly empty track env: ~A" (track-amp-env trk)))
	    (set! (mix-speed mix1) 0.5)
	    (set! (mix-track mix1) trk)
	    (if (or (not (= (mix-frames mix1) 20))
		    (not (= (track-frames trk) 20)))
		(snd-display ";mix-speed for track frames: ~A ~A" (mix-frames mix1) (track-frames trk)))
	    (if (or (fneq (sample 30) 0.0) (fneq (sample 10) 0.0) (fneq (sample 20) 0.526))
		(snd-display ";mix-speed + track amp env: ~A" (track->vct trk)))
	    (set! (track-speed trk) 2.0)
	    (if (or (not (= (mix-frames mix1) 10))
		    (not (= (track-frames trk) 10)))
		(snd-display ";mix-speed (2) for track frames: ~A ~A" (mix-frames mix1) (track-frames trk)))
	    (if (not (vequal (track->vct trk) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000)))
		(snd-display ";mix+track speed + amp-env: ~A" (track->vct trk)))
	    (close-sound ind))

	  (let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track + track tests" 300)))
	    (let ((mix1 (mix-vct (make-vct 10 1.0) 20 ind 0))
		  (track1 (make-track))
		  (track2 (make-track))
		  (edpos (edit-position ind 0)))
	      ;; track->track+env
	      (set! (track-amp-env track2) '(0 0 1 1))
	      (set! (track-track track1) track2)
	      (set! (mix-track mix1) track1)
	      (if (not (equal? (track track1) (track track2) (list mix1)))
		  (snd-display ";embedded track: ~A ~A ~A" (track track1) (track track2) (list mix1)))
	      (if (not (= 20 (mix-position mix1) (track-position track1) (track-position track2)))
		  (snd-display ";embedded track pos: ~A ~A ~A" (mix-position mix1) (track-position track1) (track-position track2)))
	      (if (not (= 10 (mix-frames mix1) (track-frames track1) (track-frames track2)))
		  (snd-display ";embedded track dur: ~A ~A ~A" (mix-frames mix1) (track-frames track1) (track-frames track2)))
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000))))
		  (snd-display ";embedded track amp-env(t->e): ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (set! edpos (edit-position ind 0))
	      (set! (track-position track1) 50)
	      (if (not (= 50 (mix-position mix1) (track-position track1) (track-position track2)))
		  (snd-display ";embedded track set pos: ~A ~A ~A" (mix-position mix1) (track-position track1) (track-position track2)))
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000))))
		  (snd-display ";embedded track amp-env(t->e) set pos: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (set! (edit-position ind 0) edpos)
	      (set! (track-position track2) 50)
	      (if (not (= 50 (mix-position mix1) (track-position track1) (track-position track2)))
		  (snd-display ";embedded track set pos 1: ~A ~A ~A" (mix-position mix1) (track-position track1) (track-position track2)))
	      (set! (edit-position ind 0) edpos)
	      (set! (mix-position mix1) 50)
	      (if (not (= 50 (mix-position mix1) (track-position track1) (track-position track2)))
		  (snd-display ";embedded track set pos 2: ~A ~A ~A" (mix-position mix1) (track-position track1) (track-position track2)))
	      (set! (edit-position ind 0) edpos)
	      (set! (track-speed track2) 0.5)
	      (if (not (= 20 (mix-frames mix1) (track-frames track1) (track-frames track2)))
		  (snd-display ";embedded track set speed: ~A ~A ~A" (mix-frames mix1) (track-frames track1) (track-frames track2)))
	      (set! (edit-position ind 0) edpos)
	      (set! (track-speed track1) 0.5)
	      (if (not (= 20 (mix-frames mix1) (track-frames track1) (track-frames track2)))
		  (snd-display ";embedded track set speed 1: ~A ~A ~A" (mix-frames mix1) (track-frames track1) (track-frames track2)))
	      (set! (edit-position ind 0) edpos)
	      (set! (mix-speed mix1) 0.5)
	      (if (not (= 20 (mix-frames mix1) (track-frames track1) (track-frames track2)))
		  (snd-display ";embedded track set speed 2: ~A ~A ~A" (mix-frames mix1) (track-frames track1) (track-frames track2)))
	      (set! (edit-position ind 0) edpos)
	      (if (not (= 10 (mix-frames mix1) (track-frames track1) (track-frames track2)))
		  (snd-display ";embedded track undo set speed: ~A ~A ~A" (mix-frames mix1) (track-frames track1) (track-frames track2)))
	      (set! (track-amp track2) 0.5)
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct-scale! (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000) 0.5))))
		  (snd-display ";embedded track set amp(t->e): ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (undo)
	      (set! (track-amp track1) 0.5)
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct-scale! (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000) 0.5))))
		  (snd-display ";embedded track set amp(t->e) 1: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (undo)
	      (set! (mix-amp mix1) 0.5)
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct-scale! (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000) 0.5))))
		  (snd-display ";embedded track set amp(t->e) 2: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (undo)
	      (set! (track-amp-env track2) #f)
	      (set! (track-amp-env track1) '(0 0 1 1))
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000))))
		  (snd-display ";embedded track amp-env(t->e) 1: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (set! (mix-amp-env mix1 0) '(0 0 1 1))
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 1.0))))
		  (snd-display ";embedded track amp-env(t->e) 2: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (set! (track-amp-env track2) '(0 0 1 1))
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.001 0.008 0.027 0.064 0.125 0.216 0.343 0.512 0.729 1.0))))
		  (snd-display ";embedded track amp-env(t->e) 3: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (set! (mix-amp-env mix1 0) #f)
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 1.0))))
		  (snd-display ";embedded track amp-env(t->e) 4: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (set! (track-amp-env track1) #f)
	      (if (or (not (vequal (mix->vct mix1) (track->vct track1)))
		      (not (vequal (mix->vct mix1) (track->vct track2)))
		      (not (vequal (track->vct track2) (vct 0.000 0.111 0.222 0.333 0.444 0.556 0.667 0.778 0.889 1.000))))
		  (snd-display ";embedded track amp-env(t->e) 1: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
	      (if (not (= 10 (mix-frames mix1) (track-frames track1) (track-frames track2)))
		  (snd-display ";embedded track dur back out: ~A ~A ~A" (mix-frames mix1) (track-frames track1) (track-frames track2)))
		
	      ;; two mixes track(1)+track-amp-env
	      (let ((mix2 (mix-vct (make-vct 10 1.0) 30 ind 0)))
		(set! (mix-track mix2) track1)
		;; technically the end point is off by 1, so we get the .5's repeated, but
		;;   this seems cleaner to me than "the right thing" -- the user will be
		;;   expecting end point matches, and in any case a 10-sample mix is silly
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
								     0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000))))
		    (snd-display ";embedded track 2mix ampenv: ~A" (channel->vct 20 20 ind 0)))
		(set! (mix-track mix2) track2)
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
								     0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000))))
		    (snd-display ";embedded track 2mix ampenv (2): ~A" (channel->vct 20 20 ind 0)))
		(set! (mix-track mix2) track1)
		(if (not (equal? (track track1) (track track2) (list mix1 mix2)))
		    (snd-display ";embedded track 2mix: ~A ~A ~A" (track track1) (track track2) (list mix1 mix2)))
		(if (not (= 20 (track-position track1) (track-position track2)))
		    (snd-display ";embedded track pos 2mix: ~A ~A" (track-position track1) (track-position track2)))
		(if (not (= 20 (track-frames track1) (track-frames track2)))
		    (snd-display ";embedded track dur 2mix: ~A ~A" (track-frames track1) (track-frames track2)))
		(set! (track-position track1) 50)
		(if (not (= 50 (mix-position mix1) (track-position track1) (track-position track2)))
		    (snd-display ";embedded track set pos 2mix: ~A ~A ~A" (mix-position mix1) (track-position track1) (track-position track2)))
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 50 20 ind 0) (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
								     0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000))))
		    (snd-display ";embedded track amp-env(t->e) set pos 2mix: ~A ~A ~A" (track->vct track1) (track->vct track2) (mix->vct mix1)))
		(undo)
		(set! (track-position track2) 50)
		(if (not (= 50 (mix-position mix1) (track-position track1) (track-position track2)))
		    (snd-display ";embedded track set pos 2mix 2: ~A ~A ~A" (mix-position mix1) (track-position track1) (track-position track2)))
		(undo)
		(set! (track-speed track2) 0.5)
		(if (not (= 30 (track-frames track1) (track-frames track2)))
		    (snd-display ";embedded track set speed 2mix: ~A ~A" (track-frames track1) (track-frames track2)))
		(undo)
		(set! (track-speed track1) 0.5)
		(if (not (= 30 (track-frames track1) (track-frames track2)))
		    (snd-display ";embedded track set speed 1: ~A ~A" (track-frames track1) (track-frames track2)))
		(undo)
		(set! (track-amp track2) 0.5)
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct-scale! (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
										 0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000) 0.5))))
		    (snd-display ";embedded track amp(t->e) set pos 2mix 2: ~A" (channel->vct 20 20 ind 0)))
		(undo)
		(set! (track-amp track1) 0.5)
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct-scale! (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
										 0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000) 0.5))))
		    (snd-display ";embedded track amp(t->e) set pos 2mix 3: ~A" (channel->vct 20 20 ind 0)))
		(undo)
		(set! (mix-amp mix1) 0.5)
		(set! (mix-amp mix2) 0.5)
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct-scale! (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
										 0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000) 0.5))))
		    (snd-display ";embedded track amp(t->e) set pos 2mix 4: ~A" (channel->vct 20 20 ind 0)))
		(undo 2)
		(set! (track-amp-env track2) #f)
		(set! (track-amp-env track1) '(0 0 1 1))
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct 0.000 0.056 0.111 0.167 0.222 0.278 0.333 0.389 0.444 0.500 
								     0.500 0.556 0.611 0.667 0.722 0.778 0.833 0.889 0.944 1.000))))
		    (snd-display ";embedded track amp-env(t->e) 2mix 1: ~A" (channel->vct 20 20 ind 0)))
		(set! (track-amp-env track2) '(0 0 1 1))
		;; here we're getting the 1/10 resampling inaccuracies
		(if (or (not (vequal (track->vct track1) (track->vct track2)))
			(not (vequal (channel->vct 20 20 ind 0) (vct 0.000 0.002 0.010 0.022 0.040 0.062 0.090 0.122 0.160 0.202 
								     0.250 0.303 0.360 0.422 0.490 0.562 0.640 0.722 0.810 0.903))))
		    (snd-display ";embedded track amp-env(t->e) 2mix 3: ~A" (channel->vct 20 20 ind 0)))

		(let* ((edpos (edit-position ind 0))
		       (track3 (make-track mix1 mix2)))

		  (if (or (not (equal? (track track3) (list mix1 mix2)))
			  (not (equal? (track track1) '()))
			  (not (equal? (track track2) '())))
		      (snd-display ";make track overrides: ~A ~A ~A" (track track1) (track track2) (track track3)))
		  (if (not (vequal (channel->vct 20 20 ind 0) (make-vct 20 1.0)))
		      (snd-display ";make-track overrides vals: ~A" (channel->vct 20 20 ind 0)))
		  (set! (edit-position ind 0) edpos)
		  (let ((track4 (make-track mix2)))
		    (if (or (not (equal? (track track4) (list mix2)))
			    (not (equal? (track track1) (list mix1)))
			    (not (equal? (track track2) (list mix1)))
			    (not (equal? (track track3) '())))
		      (snd-display ";make track again overrides: ~A ~A ~A ~A" (track track1) (track track2) (track track3) (track track4)))
		    (if (not (vequal (channel->vct 20 20 ind 0) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 
								     1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000)))
			(snd-display ";make-track again overrides vals: ~A" (channel->vct 20 20 ind 0)))
		    (save-state "s61.scm")
		    (close-sound ind)
		    (load "s61.scm")
		    ;; this currently screws up when there's a track amp-env and a mix-track change resets the env bounds
		    (set! ind (find-sound "test.snd"))
		    (if (not (sound? ind))
			(snd-display ";can't restore test.snd: ~A?" (sounds))
			(if (not (vequal (channel->vct 20 20 ind 0) (vct 0.000 0.010 0.040 0.090 0.160 0.250 0.360 0.490 0.640 0.810 
								     1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000 1.000)))
			    (snd-display ";track save/restore: ~A" (channel->vct 20 20 ind 0))))
		    ))

	      (if (sound? ind) (close-sound ind)))))

	  ;; track-tempo tests
	  (let* ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 1 "track tests" 1000))
		 (trk (make-track))
		 (initial-edpos (edit-position ind 0)))
	    (if (fneq (track-tempo trk) 1.0) (snd-display ";initial track tempo: ~A" (track-tempo trk)))
	    (set! (track-tempo trk) 0.5)
	    (if (fneq (track-tempo trk) 0.5) (snd-display ";track-tempo set: ~A" (track-tempo trk)))
	    (if (not (= (edit-position ind 0) initial-edpos)) (snd-display ";no-op set track_tempo edits: ~A ~A" edpos (edit-position ind 0)))
	    (set! (track-tempo trk) 1.0)
	    (let ((mix0 (mix-vct (make-vct 10 .1) 100)))
	      (set! (mix-track mix0) trk)
	      (if (not (= (mix-position mix0) 100)) (snd-display ";track tempo initial mix pos: ~A" (mix-position mix0)))
	      (set! (track-tempo trk) 0.5)
	      (if (not (= (mix-position mix0) 100)) (snd-display ";track tempo mix pos: ~A" (mix-position mix0)))
	      (set! (track-tempo trk) 1.0)
	      (let ((mix1 (mix-vct (make-vct 10 .3) 300)))
		(set! (mix-track mix1) trk)
		(if (not (= (mix-position mix0) 100)) (snd-display ";track (2) tempo initial mix0 pos: ~A" (mix-position mix0)))
		(if (not (= (mix-position mix1) 300)) (snd-display ";track (2) tempo initial mix1 pos: ~A" (mix-position mix1)))
		(let ((edpos1 (edit-position ind 0)))
		  (set! (track-tempo trk) 0.5)
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track tempo (2) mix0 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 500)) (snd-display ";track tempo (2) mix1 pos: ~A" (mix-position mix1)))
		  (if (not (= (edit-position ind 0) (1+ edpos1))) (snd-display ";track tempo not atomic: ~A ~A" edpos1 (edit-position ind 0))))
		(set! (track-tempo trk) 1.0)      
		(if (not (= (mix-position mix0) 100)) (snd-display ";track (2) tempo back mix0 pos: ~A" (mix-position mix0)))
		(if (not (= (mix-position mix1) 300)) (snd-display ";track (2) tempo back mix1 pos: ~A" (mix-position mix1)))
		(set! (track-tempo trk) 2.0)
		(if (not (= (mix-position mix0) 100)) (snd-display ";track tempo (2) mix0 2 pos: ~A" (mix-position mix0)))
		(if (not (= (mix-position mix1) 200)) (snd-display ";track tempo (2) mix1 2 pos: ~A" (mix-position mix1)))
		(set! (track-tempo trk) 1.0)      
		(let ((mix2 (mix-vct (make-vct 10 .4) 400)))
		  (set! (mix-track mix2) trk)
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track (3) tempo initial mix0 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 300)) (snd-display ";track (3) tempo initial mix1 pos: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 400)) (snd-display ";track (3) tempo initial mix2 pos: ~A" (mix-position mix2)))
		  (set! (track-tempo trk) 0.5)
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track tempo (3) mix0 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 500)) (snd-display ";track tempo (3) mix1 pos: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 700)) (snd-display ";track tempo (3) mix2 pos: ~A" (mix-position mix2)))
		  (set! (track-tempo trk) 1.0)      
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track (3) tempo back mix0 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 300)) (snd-display ";track (3) tempo back mix1 pos: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 400)) (snd-display ";track (3) tempo back mix2 pos: ~A" (mix-position mix2)))
		  (set! (track-tempo trk) 2.0)
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track tempo (3) mix0 2 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 200)) (snd-display ";track tempo (3) mix1 2 pos: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 250)) (snd-display ";track tempo (3) mix2 2 pos: ~A" (mix-position mix2)))
		  
		  (set! (track-amp-env trk) '(0 0 1 1))
		  (set! (track-tempo trk) 1.0)
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track (4) tempo initial mix0 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 300)) (snd-display ";track (4) tempo initial mix1 pos: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 400)) (snd-display ";track (4) tempo initial mix2 pos: ~A" (mix-position mix2)))
		  (set! (track-amp-env trk) '(0 1 1 0))	
		  (set! (track-tempo trk) 2.0)
		  (if (not (= (mix-position mix0) 100)) (snd-display ";track tempo (4) mix0 2 pos: ~A" (mix-position mix0)))
		  (if (not (= (mix-position mix1) 200)) (snd-display ";track tempo (4) mix1 2 pos: ~A" (mix-position mix1)))
		  (if (not (= (mix-position mix2) 250)) (snd-display ";track tempo (4) mix2 2 pos: ~A" (mix-position mix2)))
		  (close-sound ind)))))

	  ;; pan-mix tests
	  (let* ((ind (new-sound "fmv.snd" mus-next mus-bshort 22050 1 "pan-mix tests"))
		 (id0 (pan-mix "1a.snd")))
	    (if (or (not (= (mix-track id0) 0))
		    (fneq (mix-amp id0 0) 1.0)
		    (not (equal? (mix-amp-env id0 0) '())))
		(snd-display ";pan-mix 1->1 all opt: ~A ~A ~A" (mix-track id0) (mix-amp id0 0) (mix-amp-env id0 0)))
	    (if (fneq (maxamp ind 0) (mix-maxamp id0)) (snd-display ";pan-mix 1->1 maxamps: ~A ~A" (maxamp ind 0) (mix-maxamp id0)))
	    (if (not (= (mix-position id0) 0)) (snd-display ";pan-mix 1->1 pos: ~A" (mix-position id0)))
	    (if (not (= (mix-chans id0) 1)) (snd-display ";pan-mix 1->1 mix chans: ~A" (mix-chans id0)))
	    (revert-sound ind)
	    
	    (set! id0 (pan-mix "1a.snd" 10000 '(0 0 1 1)))
	    (if (or (not (= (mix-track id0) 0))
		    (fneq (mix-amp id0 0) 1.0)
		    (not (feql (mix-amp-env id0 0) '(0 0 1 1))))
		(snd-display ";pan-mix 1->1 2: ~A ~A ~A" (mix-track id0) (mix-amp id0 0) (mix-amp-env id0 0)))
	    (if (not (= (mix-position id0) 10000)) (snd-display ";pan-mix 1->1 pos 2: ~A" (mix-position id0)))
	    (revert-sound ind)
	    
	    (set! id0 (pan-mix "1a.snd" 80000 .4))
	    (if (or (not (= (mix-track id0) 0))
		    (fneq (mix-amp id0 0) 0.4)
		    (not (feql (mix-amp-env id0 0) '())))
		(snd-display ";pan-mix 1->1 3: ~A ~A ~A" (mix-track id0) (mix-amp id0 0) (mix-amp-env id0 0)))
	    (if (not (= (mix-position id0) 80000)) (snd-display ";pan-mix 1->1 pos 2: ~A" (mix-position id0)))
	    (if (not (= (frames ind 0) (+ 80000 (mus-sound-frames "1a.snd")))) (snd-display ";pan-mix past end frames: ~A" (frames ind 0)))
	    (revert-sound ind)
	    
	    (set! id0 (pan-mix "2a.snd" 100))
	    (if (or (= (mix-track id0) 0)
		    (not (mix? (1+ id0)))
		    (not (= (mix-track id0) (mix-track (1+ id0)))))
		(snd-display ";pan-mix 2->1: ~A ~A ~A" (mix-track id0) (mix? (1+ id0)) (if (mix? (1+ id0)) (mix-track (1+ id0)) #f)))
	    (if (not (= (mix-chans id0) 2)) (snd-display ";pan-mix 2->1 mix chans: ~A" (mix-chans id0)))
	    (if (not (= (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0)) 100))
		(snd-display ";pan-mix 2->1 pos: ~A ~A ~A" (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0))))
	    (if (or (fneq (mix-maxamp id0) (maxamp ind 0))
		    (fneq (mix-maxamp (1+ id0)) 0.0))
		(snd-display ";pan-mix 2->1 maxamps: ~A ~A ~A" (maxamp ind 0) (mix-maxamp id0) (mix-maxamp (1+ id0))))
	    (if (not (equal? (track (mix-track id0)) (list id0 (1+ id0)))) 
		(snd-display ";pan-mix 2->1 track: ~A ~A" (track (mix-track id0)) id0))
	    (let ((max1 (maxamp ind 0))
		  (maxid0 (mix-maxamp id0))
		  (maxid1 (mix-maxamp (1+ id0))))
	      (set! (track-amp-env (mix-track id0)) '(0 0 0 0))
	      (if (or (fneq (mix-maxamp id0) 0.0)
		      (fneq (mix-maxamp (1+ id0)) max1))
		  (snd-display ";pan-mix 2->1 maxamps (reversed): ~A ~A ~A" (maxamp ind 0) (mix-maxamp id0) (mix-maxamp (1+ id0))))
	      (revert-sound ind)
	      
	      (let ((maxs (mus-sound-maxamp "2a.snd")))
		(set! id0 (pan-mix "2a.snd" 100 .4))
		;; .4 means .4 in chan 0, .6 in chan 1
		(let ((expected-max (max (* .4 (cadr maxs)) (* .6 (list-ref maxs 3)))))
		  (if (fneq (maxamp ind 0) expected-max) (snd-display ";pan-mix scaled: ~A ~A" (maxamp ind 0) maxs)))
		(if (not (= (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0)) 100))
		    (snd-display ";pan-mix 2->1 pos 2: ~A ~A ~A" (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0))))
		(if (or (fneq (mix-amp id0 0) 1.0) (fneq (mix-amp (1+ id0) 1) 1.0))
		    (snd-display ";pan-mix 2->1 mix amps: ~A ~A" (mix-amp id0 0) (mix-amp (1+ id0) 1)))
		(if (or (fneq (mix-amp id0 1) 0.0) (fneq (mix-amp (1+ id0) 0) 0.0))
		    (snd-display ";pan-mix 2->1 mix amps (off case): ~A ~A" (mix-amp id0 1) (mix-amp (1+ id0) 0)))
		(if (not (feql (track-amp-env (mix-track id0)) '(0 .4 1.0 .4)))
		    (snd-display ";pan-mix 2->1 .4 env: ~A" (track-amp-env (mix-track id0))))
		(revert-sound ind)))
	    
	    (set! id0 (pan-mix "2a.snd" 100 '(0 0 1 1)))
	    (if (not (= (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0)) 100))
		(snd-display ";pan-mix 2->1 pos 3: ~A ~A ~A" (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0))))
	    (if (or (fneq (mix-amp id0 0) 1.0) (fneq (mix-amp (1+ id0) 1) 1.0))
		(snd-display ";pan-mix 2->1 mix amps 3: ~A ~A" (mix-amp id0 0) (mix-amp (1+ id0) 1)))
	    (if (or (fneq (mix-amp id0 1) 0.0) (fneq (mix-amp (1+ id0) 0) 0.0))
		(snd-display ";pan-mix 2->1 mix amps (off case) 3: ~A ~A" (mix-amp id0 1) (mix-amp (1+ id0) 0)))
	    (if (not (feql (track-amp-env (mix-track id0)) '(0 0 1 1)))
		(snd-display ";pan-mix 2->1 ramp env: ~A" (track-amp-env (mix-track id0))))
	    (if (or (fneq (maxamp ind 0) .0372) 
		    (fneq (maxamp ind 0) (max (mix-maxamp id0) (mix-maxamp (1+ id0)))))
		(snd-display ";ramp pan-mix 2->1 maxamp: ~A ~A ~A" (maxamp ind 0) (mix-maxamp id0) (mix-maxamp (1+ id0))))
	    (revert-sound ind)
	    
	    (mus-sound-forget "4.aiff") ; force re-read of maxamps?
	    (set! id0 (pan-mix "4.aiff" 100))
	    (if (not (= (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0)) 100))
		(snd-display ";pan-mix 4->1 pos 4: ~A ~A ~A" (mix-position id0) (mix-position (1+ id0)) (track-position (mix-track id0))))
	    (if (or (fneq (mix-amp id0 0) 1.0) (fneq (mix-amp (1+ id0) 1) 1.0))
		(snd-display ";pan-mix 4->1 mix amps 4: ~A ~A" (mix-amp id0 0) (mix-amp (1+ id0) 1)))
	    (if (or (fneq (mix-amp id0 1) 0.0) (fneq (mix-amp (1+ id0) 0) 0.0))
		(snd-display ";pan-mix 4->1 mix amps (off case) 4: ~A ~A" (mix-amp id0 1) (mix-amp (1+ id0) 0)))
	    (if (not (feql (track-amp-env (mix-track id0)) '(0 1 1 1)))
		(snd-display ";pan-mix 4->1 ramp env 4: ~A" (track-amp-env (mix-track id0))))
	    (let ((maxs (mus-sound-maxamp "4.aiff")))
	      (if (fneq (maxamp ind 0) (cadr maxs))
		  (snd-display ";ramp pan-mix 4->1 maxamp 4: ~A ~A" (maxamp ind 0) maxs))
	      (set! (track-amp-env (mix-track id0)) '(0 0 1 0))
	      (if (fneq (maxamp ind 0) (list-ref maxs 3))
		  (snd-display ";ramp pan-mix 4->1 maxamp 4(2): ~A ~A" (maxamp ind 0) maxs)))
	    (close-sound ind))

	  (let* ((ind (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "pan-mix tests"))
		 (id0 (pan-mix "1a.snd"))
		 (id1 (1+ id0))
		 (trk (mix-track id0)))
	    (if (not (track? trk)) (snd-display ";pan-mix 1->2 track: ~A?" trk))
	    (if (not (= (mix-chans id0) 1)) (snd-display ";pan-mix 1->2 mix chans: ~A" (mix-chans id0)))
	    (if (or (fneq (mix-amp id0) 1.0) (fneq (mix-amp id1) 1.0)) 
		(snd-display ";pan-mix 1->2 amps: ~A ~A" (mix-amp id0) (mix-amp id1)))
	    (if (or (fneq (maxamp ind 0) (mix-maxamp id0))
		    (fneq (maxamp ind 1) 0.0))
		(snd-display ";pan-mix 1->2 maxamps: ~A ~A ~A" (maxamp ind 0) (mix-maxamp id0) (mix-maxamp id1)))
	    (if (not (equal? (track (mix-track id0)) (list id0 id1)))
		(snd-display ";pan-mix 1->2 track: ~A ~A" (track (mix-track id0)) id0))
	    (set! (track-amp-env (mix-track id0)) '(0 0 1 0))
	    (if (or (fneq (maxamp ind 1) (mix-maxamp id1))
		    (fneq (maxamp ind 0) 0.0))
		(snd-display ";pan-mix 1->2 maxamps reversed: ~A ~A ~A" (maxamp ind #t) (mix-maxamp id0) (mix-maxamp id1)))
	    (revert-sound ind)
	    
	    (set! id0 (pan-mix "2a.snd" 100))
	    (set! id1 (1+ id0))
	    (set! trk (mix-track id0))
	    (if (not (= (mix-chans id0) 2)) (snd-display ";pan-mix 2->2 mix chans: ~A" (mix-chans id0)))
	    (if (not (equal? (track trk) (list id0 id1)))
		(snd-display ";pan-mix 2->2 track: ~A ~A" (track trk) id0))
	    (if (or (fneq (mix-amp id0 0) 1.0) (fneq (mix-amp id0 1) 0.0)
		    (fneq (mix-amp id1 0) 0.0) (fneq (mix-amp id1 1) 1.0))
		(snd-display ";pan-mix 2->2 amps: ~A ~A" (mix-amp id0 0) (mix-amp id0 1) (mix-amp id1 0) (mix-amp id1 1)))
	    (if (or (fneq (maxamp ind 0) (mix-maxamp id0))
		    (fneq (maxamp ind 1) 0.0))
		(snd-display ";pan-mix 2->2 maxamps: ~A ~A ~A" (maxamp ind 0) (mix-maxamp id0) (mix-maxamp id1)))
	    (set! (track-amp-env (mix-track id0)) '(0 0 1 0))
	    (if (or (fneq (maxamp ind 1) (mix-maxamp id1))
		    (fneq (maxamp ind 0) 0.0))
		(snd-display ";pan-mix 2->2 maxamps reversed: ~A ~A ~A" (maxamp ind #t) (mix-maxamp id0) (mix-maxamp id1)))
	    (revert-sound ind)
	    
	    (set! id0 (pan-mix "2a.snd" 1000 .4))
	    (set! id1 (1+ id0))
	    (set! trk (mix-track id0))
	    (let ((maxs (mus-sound-maxamp "2a.snd")))
	      (if (or (fneq (maxamp ind 0) (* .4 (list-ref maxs 1)))
		      (fneq (maxamp ind 1) (* .6 (list-ref maxs 3))))
		  (snd-display ";pan-mix 2->2 .4: ~A ~A" maxs (maxamp ind #t)))
	      (if (not (feql (track-amp-env trk) (list 0 .4 1 .4)))
		  (snd-display ";pan-mix 2->2 .4 env: ~A" (track-amp-env trk)))
	      (if (not (equal? (track trk) (list id0 id1)))
		  (snd-display ";pan-mix 2->2 track .4: ~A ~A" (track trk) id0))
	      (revert-sound))
	    
	    (set! id0 (pan-mix "4.aiff"))
	    (let ((maxs (mus-sound-maxamp "4.aiff")))
	      (if (or (fneq (maxamp ind 0) (list-ref maxs 1))
		      (fneq (maxamp ind 1) 0.0))
		  (snd-display ";pan-mix 4->2 max: ~A ~A" maxs (maxamp ind #t)))
	      (if (not (= (mix-chans id0) 4)) (snd-display ";pan-mix 4->2 mix chans: ~A" (mix-chans id0)))    
	      (close-sound ind))
	    
	    (set! ind (new-sound "fmv.snd" mus-next mus-bshort 22050 4 "pan-mix tests"))
	    (set! id0 (pan-mix "1a.snd"))
	    (set! id1 (1+ id0))
	    (set! trk (mix-track id0))
	    (if (not (equal? (track trk) (list id0 id1)))
		(snd-display ";pan-mix 1->4 track: ~A ~A" (track trk) id0))
	    (close-sound ind)))

	(let* ((ind (new-sound "test.snd" mus-next mus-bshort 22050 1 "pan-mix-* tests"))
	       (id0 (pan-mix-vct (make-vct 100 .3))))
	  (if (or (not (= (mix-track id0) 0))
		  (fneq (mix-amp id0 0) 1.0)
		  (not (equal? (mix-amp-env id0 0) '())))
	      (snd-display ";pan-mix-vct 1->1 all opt: ~A ~A ~A" (mix-track id0) (mix-amp id0 0) (mix-amp-env id0 0)))
	  (if (or (fneq (maxamp ind 0) (mix-maxamp id0)) 
		  (fneq (maxamp ind 0) 0.3))
	      (snd-display ";pan-mix-vct 1->1 maxamps: ~A ~A" (maxamp ind 0) (mix-maxamp id0)))
	  (if (not (= (mix-position id0) 0)) (snd-display ";pan-mix-vct 1->1 pos: ~A" (mix-position id0)))
	  (if (not (= (mix-chans id0) 1)) (snd-display ";pan-mix-vct 1->1 mix chans: ~A" (mix-chans id0)))
	  (let ((ind1 (new-sound "fmv.snd" mus-next mus-bshort 22050 1 "pan-mix-* tests")))
	    (let ((reg (make-region 0 50 ind 0)))
	      (let ((id1 (pan-mix-region reg)))
		(if (or (fneq (maxamp ind1 0) (mix-maxamp id1)) 
			(fneq (maxamp ind1 0) 0.3))
		    (snd-display ";pan-mix-region 1->1 maxamps: ~A ~A" (maxamp ind1 0) (mix-maxamp id1)))
		(select-all)
		(revert-sound ind)
		(set! id0 (pan-mix-selection 0 1.0 ind 0))
		(if (or (fneq (maxamp ind 0) (mix-maxamp id0)) 
			(fneq (maxamp ind 0) 0.3))
		    (snd-display ";pan-mix-selection 1->1 maxamps: ~A ~A" (maxamp ind 0) (mix-maxamp id0)))
		(close-sound ind)
		(close-sound ind1)))))

	(let ((old-mix-tag (with-mix-tags)))
	  (dynamic-wind
	   (lambda () (set! (with-mix-tags) #f))
	   (lambda ()
	     (let* ((ind (new-sound "fmv.snd" mus-next mus-bshort 22050 1 "locked pan-mix tests"))
		    (id0 (pan-mix "1a.snd"))
		    (max1a (cadr (mus-sound-maxamp "1a.snd")))
		    (max2a (cadr (mus-sound-maxamp "2a.snd"))))
	       (if (or (not (mix? id0))
		       (not (mix-locked? id0)))
		   (snd-display ";no tag pan-mix: ~A ~A" id0 (mixes)))
	       (if (fneq (maxamp ind 0) max1a) (snd-display ";no-tag pan-mix 1->1 maxamps: ~A ~A" (maxamp ind 0) max1a))
	       (if (not (mix-locked? id0)) (snd-display ";no-tag pan-mix 1->1 not locked?"))
	       (revert-sound ind)
	       (set! id0 (pan-mix "2a.snd" 100))
	       (if (fneq (maxamp ind 0) max2a) (snd-display ";no-tag pan-mix 2->1 maxamps: ~A ~A" (maxamp ind 0) max2a))
	       (if (not (vequal (channel->vct 3000 10) (make-vct 10 0.0)))
		   (snd-display ";no-tag pan-mix 2->1 channel 2: ~A" (channel->vct 3000 10)))
	       (if (not (mix-locked? id0)) (snd-display ";no-tag pan-mix 2->1 not locked?"))
	       (close-sound ind)
	       (set! ind (new-sound "fmv.snd" mus-next mus-bshort 22050 2 "locked pan-mix tests"))
	       (set! id0 (pan-mix "1a.snd"))
	       (if (not (feql (maxamp ind #t) (list max1a 0.0)))
		   (snd-display ";no-tag pan-mix 1->2 maxamps: ~A ~A" (maxamp ind #t) max1a))
	       (if (not (mix-locked? id0)) (snd-display ";no-tag pan-mix 1->2 not locked?"))
	       (revert-sound ind)
	       (set! id0 (pan-mix "2a.snd" 100))
	       (if (not (feql (maxamp ind #t) (list max1a 0.0)))
		   (snd-display ";no-tag pan-mix 2->2 maxamps: ~A ~A" (maxamp ind #t) max1a))
	       (if (not (mix-locked? id0)) (snd-display ";no-tag pan-mix 2->2 not locked?"))
	       (close-sound ind)))
	   (lambda () (set! (with-mix-tags) old-mix-tag))))

	(let ((ind (new-sound "test.snd" mus-next mus-bfloat 22050 2 "copy sample-reader tests" 1000)))
	  (vct->channel (vct .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0) 101 10 ind 0)
	  (vct->channel (vct .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0) 201 10 ind 1)
	  (let ((rd1 (make-sample-reader 100 ind 0))
		(rd2 (make-sample-reader 200 ind 1)))
	    (rd1) (rd2) (rd1) (rd2)
	    (let ((rd11 (copy-sample-reader rd1))
		  (rd22 (copy-sample-reader rd2)))
	      (if (or (not (sample-reader? rd11)) (not (sample-reader? rd22)))
		  (snd-display ";copy-sample-reader (normal): ~A ~A" rd11 rd22))
	      (if (or (mix-sample-reader? rd11) (mix-sample-reader? rd22)
		      (track-sample-reader? rd11) (track-sample-reader? rd22)
		      (region-sample-reader? rd11) (region-sample-reader? rd22))
		  (snd-display ";copy sample-reader-p trouble: ~A ~A ~A ~A ~A ~A"
			       (mix-sample-reader? rd11) (mix-sample-reader? rd22)
			       (track-sample-reader? rd11) (track-sample-reader? rd22)
			       (region-sample-reader? rd11) (region-sample-reader? rd22)))
	      (if (or (not (equal? (sample-reader-home rd11) (list ind 0)))
		      (not (equal? (sample-reader-home rd22) (list ind 1))))
		  (snd-display ";copy normal reader home: ~A ~A" (sample-reader-home rd11) (sample-reader-home rd22)))
	      (if (or (sample-reader-at-end? rd11) (sample-reader-at-end? rd22))
		  (snd-display ";copy normal reader end?: ~A ~A" (sample-reader-at-end? rd11) (sample-reader-at-end? rd22)))
	      (if (or (not (= (sample-reader-position rd11) (sample-reader-position rd1) 102))
		      (not (= (sample-reader-position rd22) (sample-reader-position rd2) 202)))
		  (snd-display ";copy normal reader position: ~A ~A ~A ~A" 
			       (sample-reader-position rd11) (sample-reader-position rd1)
			       (sample-reader-position rd22) (sample-reader-position rd2)))
	      (call-with-current-continuation
	       (lambda (break)
		 (let ((v1 (vct .2 .3 .4 .5 .6 .7 .8 .9 1.0 0.0 0.0 0.0)))
		   (do ((i 0 (1+ i)))
		       ((= i 10))
		     (let ((rd1v (rd1))
			   (rd11v (rd11))
			   (rd2v (next-sample rd2))
			   (rd22v (read-sample rd22)))
		       (if (or (fneq rd1v rd11v)
			       (fneq rd1v (vct-ref v1 i))
			       (fneq rd2v rd22v)
			       (fneq rd2v (vct-ref v1 i)))
			   (begin
			     (snd-display ";copy sample reader vals at ~A: ~A ~A ~A ~A ~A"
					  i rd1v rd11v rd2v rd22v (vct-ref v1 i))
			     (break))))))))
	      (free-sample-reader rd1)
	      (free-sample-reader rd11)))
	  (let ((mx1m (mix-vct (vct .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0) 95 ind 0))
		(mx2m (mix-vct (vct .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0) 195 ind 1)))
	    (let ((mx1rd (make-mix-sample-reader mx1m 2))
		  (mx2rd (make-mix-sample-reader mx2m 4)))
	      (mx1rd)
	      (mx1rd)
	      (let ((val1 (mx1rd))
		    (val2 (mx2rd)))
		(if (or (fneq val1 val2)
			(fneq val2 .5))
		    (snd-display ";precopy-mix-sample-reader vals: ~A ~A" val1 val2))
		(let ((mx11rd (copy-sample-reader mx1rd))
		      (mx22rd (copy-sample-reader mx2rd)))
		  (if (or (not (mix-sample-reader? mx11rd)) (not (mix-sample-reader? mx22rd)))
		      (snd-display ";copy-sample-reader (mix): ~A ~A" mx11rd mx22rd))
		  (if (or (sample-reader? mx11rd) (sample-reader? mx22rd)
			  (track-sample-reader? mx11rd) (track-sample-reader? mx22rd)
			  (region-sample-reader? mx11rd) (region-sample-reader? mx22rd))
		      (snd-display ";copy mix sample-reader-p trouble: ~A ~A ~A ~A ~A ~A"
				   (sample-reader? mx11rd) (sample-reader? mx22rd)
				   (track-sample-reader? mx11rd) (track-sample-reader? mx22rd)
				   (region-sample-reader? mx11rd) (region-sample-reader? mx22rd)))
		  (if (or (not (equal? (sample-reader-home mx11rd) mx1m))
			  (not (equal? (sample-reader-home mx22rd) mx2m)))
		      (snd-display ";copy mix reader home: ~A ~A" (sample-reader-home mx11rd) (sample-reader-home mx22rd)))
		  (if (or (sample-reader-at-end? mx11rd) (sample-reader-at-end? mx22rd))
		      (snd-display ";copy mix reader end?: ~A ~A" (sample-reader-at-end? mx11rd) (sample-reader-at-end? mx22rd)))
		  (if (or (not (= (sample-reader-position mx11rd) (sample-reader-position mx1rd) 5))
			  (not (= (sample-reader-position mx22rd) (sample-reader-position mx2rd) 5)))
		      (snd-display ";copy mix reader position: ~A ~A ~A ~A" 
				   (sample-reader-position mx11rd) (sample-reader-position mx1rd)
				   (sample-reader-position mx22rd) (sample-reader-position mx2rd)))
		  (let ((tag (catch 'wrong-type-arg
				    (lambda () (next-sample mx11rd))
				    (lambda args (car args)))))
		    (if (not (eq? tag 'wrong-type-arg)) (snd-display ";next-sample of mix reader: ~A" tag)))
		  (call-with-current-continuation
		   (lambda (break)
		     (let ((v1 (vct .6 .7 .8 .9 1.0 0.0 0.0 0.0)))
		       (do ((i 0 (1+ i)))
			   ((= i 6))
			 (let ((mx1rdv (mx1rd))
			       (mx11rdv (mx11rd))
			       (mx2rdv (read-mix-sample mx2rd))
			       (mx22rdv (read-mix-sample mx22rd)))
			   (if (or (fneq mx1rdv mx11rdv)
				   (fneq mx1rdv (vct-ref v1 i))
				   (fneq mx2rdv mx22rdv)
				   (fneq mx2rdv (vct-ref v1 i)))
			       (begin
				 (snd-display ";copy mix sample reader vals at ~A: ~A ~A ~A ~A ~A"
					      i mx1rdv mx11rdv mx2rdv mx22rdv (vct-ref v1 i))
				 (break))))))))
		  (free-sample-reader mx1rd)
		  (free-sample-reader mx11rd))))
	    (let ((trk (make-track)))
	      (set! (mix-track mx1m) trk)
	      (set! (mix-track mx2m) trk)
	      (let ((mx1rd (make-track-sample-reader trk 0 2))
		    (mx2rd (make-track-sample-reader trk 1 4)))
		(mx1rd)
		(mx1rd)
		(let ((val1 (mx1rd))
		      (val2 (mx2rd)))
		  (if (or (fneq val1 val2)
			  (fneq val2 .5))
		      (snd-display ";precopy-track-sample-reader vals: ~A ~A" val1 val2))
		  (let ((mx11rd (copy-sample-reader mx1rd))
			(mx22rd (copy-sample-reader mx2rd)))
		    (if (or (not (track-sample-reader? mx11rd)) (not (track-sample-reader? mx22rd)))
			(snd-display ";copy-sample-reader (track): ~A ~A" mx11rd mx22rd))
		    (if (or (sample-reader? mx11rd) (sample-reader? mx22rd)
			    (mix-sample-reader? mx11rd) (mix-sample-reader? mx22rd)
			    (region-sample-reader? mx11rd) (region-sample-reader? mx22rd))
			(snd-display ";copy track sample-reader-p trouble: ~A ~A ~A ~A ~A ~A"
				     (sample-reader? mx11rd) (sample-reader? mx22rd)
				     (mix-sample-reader? mx11rd) (mix-sample-reader? mx22rd)
				     (region-sample-reader? mx11rd) (region-sample-reader? mx22rd)))
		    (if (or (not (equal? (sample-reader-home mx11rd) (list trk 0)))
			    (not (equal? (sample-reader-home mx22rd) (list trk 1))))
			(snd-display ";copy track reader home: ~A ~A" (sample-reader-home mx11rd) (sample-reader-home mx22rd)))
		    (if (or (sample-reader-at-end? mx11rd) (sample-reader-at-end? mx22rd))
			(snd-display ";copy track reader end?: ~A ~A" (sample-reader-at-end? mx11rd) (sample-reader-at-end? mx22rd)))
		    (if (or (not (= (sample-reader-position mx11rd) (sample-reader-position mx1rd) 5))
			    (not (= (sample-reader-position mx22rd) (sample-reader-position mx2rd) 5)))
			(snd-display ";copy track reader position: ~A ~A ~A ~A" 
				     (sample-reader-position mx11rd) (sample-reader-position mx1rd)
				     (sample-reader-position mx22rd) (sample-reader-position mx2rd)))
		    (let ((tag (catch 'wrong-type-arg
				      (lambda () (next-sample mx11rd))
				      (lambda args (car args)))))
		      (if (not (eq? tag 'wrong-type-arg)) (snd-display ";next-sample of track reader: ~A" tag)))
		    (call-with-current-continuation
		     (lambda (break)
		       (let ((v1 (vct .6 .7 .8 .9 1.0 0.0 0.0 0.0)))
			 (do ((i 0 (1+ i)))
			     ((= i 6))
			   (let ((mx1rdv (mx1rd))
				 (mx11rdv (mx11rd))
				 (mx2rdv (read-track-sample mx2rd))
				 (mx22rdv (read-track-sample mx22rd)))
			     (if (or (fneq mx1rdv mx11rdv)
				     (fneq mx1rdv (vct-ref v1 i))
				     (fneq mx2rdv mx22rdv)
				     (fneq mx2rdv (vct-ref v1 i)))
				 (begin
				   (snd-display ";copy track sample reader vals at ~A: ~A ~A ~A ~A ~A"
						i mx1rdv mx11rdv mx2rdv mx22rdv (vct-ref v1 i))
				   (break))))))))
		    (free-sample-reader mx1rd)
		    (free-sample-reader mx11rd))))))
	  (set! (sync ind) 1)
	  (let ((reg (make-region 90 220 ind #t)))
	    (if (not (= (region-frames reg) (1+ (- 220 90)))) (snd-display ";make-region frames: ~A" (region-frames reg)))
	    (if (not (= (region-chans reg) 2)) (snd-display ";make-region chans: ~A" (region-chans reg)))
	    (if (not (= (region-frames reg 0) (1+ (- 220 90)))) (snd-display ";make-region frames[0]: ~A" (region-frames reg 0)))
	    (if (not (= (region-frames reg 1) (1+ (- 220 90)))) (snd-display ";make-region frames[1]: ~A" (region-frames reg 1)))
	    (if (not (= (region-position reg 0) 90)) (snd-display ";make-region position[0]: ~A" (region-position reg 0)))
	    (if (not (= (region-position reg 1) 90)) (snd-display ";make-region position[1]: ~A" (region-position reg 1)))
	    (if (not (= (region-position reg) 90)) (snd-display ";make-region position[]: ~A" (region-position reg)))
	    
	    ;; beg = 0, chan 2 not highlighted
	    
	    (let ((rd1 (make-region-sample-reader 0 reg 0))
		  (rd2 (make-region-sample-reader 100 reg 1)))
	      (let ((rd11 (copy-sample-reader rd1))
		    (rd22 (copy-sample-reader rd2)))
		(if (or (not (region-sample-reader? rd11)) (not (region-sample-reader? rd22)))
		    (snd-display ";copy-sample-reader (region): ~A ~A" rd11 rd22))
		(if (or (mix-sample-reader? rd11) (mix-sample-reader? rd22)
			(track-sample-reader? rd11) (track-sample-reader? rd22)
			(sample-reader? rd11) (sample-reader? rd22))
		    (snd-display ";copy (region) sample-reader-p trouble: ~A ~A ~A ~A ~A ~A"
				 (mix-sample-reader? rd11) (mix-sample-reader? rd22)
				 (track-sample-reader? rd11) (track-sample-reader? rd22)
				 (sample-reader? rd11) (sample-reader? rd22)))
		(if (or (not (equal? (sample-reader-home rd11) (list reg 0)))
			(not (equal? (sample-reader-home rd22) (list reg 1))))
		    (snd-display ";copy region reader home: ~A ~A" (sample-reader-home rd11) (sample-reader-home rd22)))
		(if (or (sample-reader-at-end? rd11) (sample-reader-at-end? rd22))
		    (snd-display ";copy region reader end?: ~A ~A" (sample-reader-at-end? rd11) (sample-reader-at-end? rd22)))
		(if (or (not (= (sample-reader-position rd11) (sample-reader-position rd1) 0))
			(not (= (sample-reader-position rd22) (sample-reader-position rd2) 100)))
		    (snd-display ";copy region reader position: ~A ~A ~A ~A" 
				 (sample-reader-position rd11) (sample-reader-position rd1)
				 (sample-reader-position rd22) (sample-reader-position rd2)))
		(call-with-current-continuation
		 (lambda (break)
		   (let ((v1 (vct 0.000 0.000 0.000 0.000 0.000 0.100 0.200 0.300 0.400 0.500 0.600 0.800 1.000 1.200 1.400 0.500 0.600 0.700 0.800 0.900)))
		     (do ((i 0 (1+ i)))
			 ((= i 20))
		       (let ((rd1v (rd1))
			     (rd11v (rd11))
			     (rd2v (read-region-sample rd2))
			     (rd22v (read-region-sample rd22)))
			 (if (or (fneq rd1v rd11v)
				 (fneq rd1v (vct-ref v1 i))
				 (fneq rd2v rd22v)
				 (fneq rd2v (vct-ref v1 i)))
			     (begin
			       (snd-display ";copy region sample reader vals at ~A: ~A ~A ~A ~A ~A"
					    i rd1v rd11v rd2v rd22v (vct-ref v1 i))
			       (break))))))))
		(free-sample-reader rd1)
		(free-sample-reader rd11))))
	  (close-sound ind))

	  (let ((ind (new-sound "tmp.snd" mus-next mus-bfloat 22050 1 "mix with exp env tests" 1000)))
	    (let ((mx (mix-vct (make-vct 10 1.0) 100)))
	      (set! (mix-amp-env mx) xrmx)
	      (let ((data (channel->vct 100 10)))
		(if (not (vequal data (vct 0.000 0.015 0.037 0.070 0.118 0.189 0.293 0.446 0.670 1.000)))
		    (snd-display ";mix-vct + exp env: ~A" data)))
	      (revert-sound ind)
	      (set! mx (mix-vct (make-vct 100 1.0) 100))
	      (set! (mix-amp-env mx) xrmx)
	      (let ((trk (make-track mx)))
		(set! (track-amp-env trk) '(0 0 1 1))
		(let ((data (channel->vct 120 10)))
		  (if (not (vequal data (vct 0.006 0.008 0.009 0.010 0.011 0.012 0.013 0.014 0.015 0.017)))
		      (snd-display ";mix-vct + exp + track ramp (20): ~A" data)))
		(let ((data (channel->vct 190 10)))
		  (if (not (vequal data (vct 0.665 0.702 0.740 0.777 0.814 0.851 0.888 0.926 0.963 1.000)))
		      (snd-display ";mix-vct + exp + track ramp (90): ~A" data)))
		(set! (track-amp-env trk) xrmx)
		(let ((data (channel->vct 190 10)))
		  (if (not (vequal data (vct 0.538 0.589 0.641 0.692 0.743 0.795 0.846 0.897 0.949 1.000)))
		      (snd-display ";mix-vct + exp + track exp (90): ~A" data)))
		(set! (mix-amp-env mx) '(0 0 1 1))
		(let ((data (channel->vct 190 10)))
		  (if (not (vequal data (vct 0.665 0.702 0.740 0.777 0.814 0.851 0.888 0.926 0.963 1.000)))
		      (snd-display ";mix-vct + ramp + track exp (90): ~A" data)))
		(set! (mix-amp-env mx) #f)
		(let ((data (channel->vct 190 10)))
		  (if (not (vequal data (vct 0.721 0.748 0.776 0.804 0.834 0.865 0.897 0.930 0.964 1.000)))
		      (snd-display ";mix-vct + track exp (90): ~A" data)))
		(set! (track-amp-env trk) #f)
		(set! (mix-amp-env mx) xrmx)
		(let ((data (channel->vct 190 10)))
		  (if (not (vequal data (vct 0.721 0.748 0.776 0.804 0.834 0.865 0.897 0.930 0.964 1.000)))
		      (snd-display ";mix-vct exp + track null (90): ~A" data)))
		(revert-sound ind)
		(set! mx (mix-vct (make-vct 100 1.0) 100))
		(let* ((mx1r (mix-vct (make-vct 100 1.0) 200))
		       (trk1 (make-track mx mx1r)))
		  (set! (track-amp-env trk1) xrmx2)
		  (let ((data (channel->vct 190 10)))
		    (if (not (vequal data (vct 0.542 0.547 0.552 0.557 0.562 0.567 0.572 0.576 0.581 0.586)))
			(snd-display ";mix-vct track amp env xrmx2 190 10: ~A" data)))
		  (let ((data (channel->vct 200 10)))
		    (if (not (vequal data (vct 0.586 0.591 0.596 0.601 0.606 0.611 0.615 0.620 0.625 0.630)))
			(snd-display ";mix-vct track amp env xrmx2 200 10: ~A" data)))
		  (close-sound ind)))))
	
	  (let ((ind (new-sound "tmp.snd" mus-next mus-bfloat 22050 2 "stereo mix with exp env tests" 1000)))
	    (let* ((mx0t (mix-vct (make-vct 10 1.0) 100 ind 0))
		   (mx1t (mix-vct (make-vct 10 1.0) 100 ind 1))
		   (trk (make-track mx0t mx1t)))
	      (set! (mix-inverted? mx1t) #t)
	      (set! (track-amp-env trk) xrmx)
	      (let ((data (channel->vct 100 10 ind 0)))
		(if (not (vequal data (vct 0.000 0.015 0.037 0.070 0.118 0.189 0.293 0.446 0.670 1.000)))
		    (snd-display ";0 mix-vct + exp env: ~A" data)))
	      (let ((data (channel->vct 100 10 ind 1)))
		(if (not (vequal data (vct 1.000 0.985 0.963 0.930 0.882 0.811 0.707 0.554 0.330 0.000)))
		    (snd-display ";1 mix-vct + exp env: ~A" data)))
	      (close-sound ind)))

	(set! (with-mix-tags) old-with-mix-tags))

      (let ((old (tempo-control-bounds)))
	(if (or (fneq (car old) 0.0) (fneq (cadr old) 8.0))
	    (snd-display ";tempo-control-bounds defaults: ~A" old))
	(set! (tempo-control-bounds) (list 0.0 2.0))
	(set! old (tempo-control-bounds))
	(if (or (fneq (car old) 0.0) (fneq (cadr old) 2.0))
	    (snd-display ";tempo-control-bounds set (0.0 2.0): ~A" old))
	(set! (tempo-control-bounds) (list 0.0 8.0)))
      
      (let* ((ind (new-sound "test.snd" :size 10))
	     (mx1 (mix-vct (make-vct 2 .1) 0))
	     (mx2 (mix-vct (make-vct 2 .2) 2))
	     (mx3 (mix-vct (make-vct 2 .3) 4))
	     (mx4 (mix-vct (make-vct 2 .4) 6))
	     (mx5 (mix-vct (make-vct 2 .5) 8)))
	(if (not (vequal (channel->vct) (vct .1 .1 .2 .2 .3 .3 .4 .4 .5 .5)))
	    (snd-display ";delete-all-tracks init: ~A" (channel->vct)))
	(delete-all-tracks)
	(if (not (vequal (channel->vct) (vct .1 .1 .2 .2 .3 .3 .4 .4 .5 .5)))
	    (snd-display ";delete-all-tracks no-op: ~A" (channel->vct)))
	(let ((trk (make-track mx1 mx3))
	      (trk1 (make-track)))
	  (set! (mix-track mx4) trk1)
	  (delete-all-tracks)
	  (if (not (vequal (channel->vct) (vct 0 0 .2 .2 0 0 0 0 .5 .5)))
	      (snd-display ";delete-all-tracks: ~A" (channel->vct)))
	  (close-sound ind)))

      (run-hook after-test-hook 9)
      ))


(clear-sincs)

(define maxval 0.0)
(define data-max
  (lambda (beg end)
    (set! maxval 0.0)
    (apply for-each 
	   (lambda (snd chn)
	     (scan-chan (lambda (n)
			  (set! maxval (max maxval (abs n)))
			  #f)
			0 #f snd chn))
	   (all-chans))
    maxval))

(define data-max2
  (lambda (beg end snd)
    (set! maxval 0.0)
    (do ((i 0 (1+ i)))
	((= i (chans snd)) maxval)
      (scan-chan (lambda (n)
		   (set! maxval (max maxval (abs n)))
		   #f)
		 0 #f snd i))))

(define data-max1
  (lambda (beg end snd chn)
    (set! maxval 0.0)
    (scan-chan 
     (lambda (data)
       (let ((curval (abs data)))
	 (if (> curval maxval) (set! maxval curval))
	 #f))
     beg end snd chn)
    maxval))

(if (not (provided? 'snd-marks.scm)) (load "marks.scm"))

;;; ---------------- test 10: marks ----------------
(if (or full-test (= snd-test 10) (and keep-going (<= snd-test 10)))
    (begin 
      (run-hook before-test-hook 10)
      (do ((test-ctr 0 (1+ test-ctr)))
	  ((= test-ctr tests))
	(let ((ind0 (view-sound "oboe.snd"))
	      (ind1 (view-sound "pistol.snd"))
	      (v0 (make-vct 100))
	      (vc (make-vector 10)))
	  (log-mem test-ctr)
	  (vct-fill! v0 .1)
	  (vector-set! vc 0 (mix-vct v0 0 ind0))
	  (vector-set! vc 1 (mix-vct v0 1000 ind0))
	  (vector-set! vc 2 (mix-vct v0 2000 ind0))
	  (vector-set! vc 3 (mix-vct v0 3000 ind0))
	  (vector-set! vc 4 (mix-vct v0 4000 ind0))
	  (vector-set! vc 5 (mix-vct v0 0 ind1))
	  (vector-set! vc 6 (mix-vct v0 1000 ind1))
	  (vector-set! vc 7 (mix-vct v0 2000 ind1))
	  (vector-set! vc 8 (mix-vct v0 3000 ind1))
	  (vector-set! vc 9 (mix-vct v0 4000 ind1))
	  (let ((t0 (make-track (vector-ref vc 0) (vector-ref vc 3) (vector-ref vc 5)))
		(t1 (make-track (vector-ref vc 2) (vector-ref vc 6) (vector-ref vc 8))))
	    (set! (track-amp t0) .5)
	    (time (transpose-track t1 3))
	    (catch 'no-such-color
		   (lambda () (set! (track-color t1) (make-color 0 0 1)))
		   (lambda args args))
	    (let ((t0e (track-end t0)))
	      (set! (track-position t0) 1000)
	      (if (not (= (track-position t0) 1000)) (snd-display ";track-position: ~A?" (track-position t0)))
	      (if (not (= (track-end t0) (+ t0e 1000))) (snd-display ";track-end: ~A ~A?" t0e (track-end t0))))
	    (if (not (= (track-frames t0) 3100)) (snd-display ";track-frames: ~A?" (track-frames t0)))
	    (retempo-track t0 2.0)
	    (if with-gui
		(let ((col (color->list (track-color t1))))
		  (if (or (fneq (car col) 0.0) (fneq (cadr col) 0.0) (fneq (caddr col) 1.0))
		      (snd-display ";track-color: ~A?" col))))
	    (if (not (= (track-frames t0) 1600)) (snd-display ";track-tempo -> length: ~A?" (track-frames t0))))
	  (close-sound ind0)
	  (close-sound ind1)
	  (set! ind0 (new-sound "fmv.snd" mus-aifc mus-bshort 22050 1 "this is a comment"))
	  (let ((v0 (make-vector 10)))
	    (do ((i 0 (1+ i))) ((= i 10)) (vector-set! v0 i 1.0))
	    (insert-samples 0 10 v0 ind0) 
	    (time (env-sound '(0 0 1 1) 0 10 1.0 ind0))
	    (do ((i 0 (1+ i))) ((= i 10)) (if (fneq (sample i) (* i .1111)) (snd-display ";1 env-sound[~D]: ~A?" i (sample i))))
	    (undo) 
	    (env-sound (make-env '(0 0 1 1) :end 9) 0 10 1.0 ind0) 
	    (do ((i 0 (1+ i))) ((= i 10)) (if (fneq (sample i) (* i .1111)) (snd-display ";2 env-sound[~D]: ~A?" i (sample i))))
	    (undo) 
	    (env-sound '(0 0 .5 1 1 1) 0 10 0.0 ind0) 
	    (if (or (fneq (sample 3) 0.0) (fneq (sample 8) 1.0) )
		(snd-display ";env-sound stepped: ~A ~A?" (sample 3) (sample 8)))
	    (undo) 
	    (env-sound '(0 0 1 1) 0 10 32.0 ind0) 
	    (if (or (fneq (sample 3) 0.070) (fneq (sample 8) 0.67) )
		(snd-display ";env-sound exp: ~A ~A?" (sample 3) (sample 8)))
	    (undo) 
	    (env-sound (make-env '(0 0 1 1) :base 32.0 :end 9) 0 10 32.0 ind0) 
	    (if (or (fneq (sample 3) 0.070) (fneq (sample 8) 0.67) )
		(snd-display ";env-sound exp: ~A ~A?" (sample 3) (sample 8)))
	    (undo)
	    (env-sound '(0 2))
	    (do ((i 0 (1+ i))) ((= i 10)) (if (fneq (sample i) 2.0) (snd-display ";3 env-sound[~D]: ~A?" i (sample i))))
	    (undo)
	    (env-sound '(0 2) 2 4 1.0 ind0)
	    (if (or (fneq (sample 1) 1.0) (fneq (sample 2) 2.0) (fneq (sample 5) 2.0) (fneq (sample 8) 1.0))
		(snd-display ";3 env-sound exp: ~A ~A ~A ~A?" (sample 1) (sample 2) (sample 5) (sample 8)))
	    (undo) 
	    (do ((i 1 (1+ i))) ((= i 10)) (set! (sample i) 0.0))
	    (filter-sound '(0 1 1 0) 4)
	    (if (or (fneq (sample 1) 0.3678) (fneq (sample 2) .3678) (fneq (sample 3) .132) (fneq (sample 4) 0.0))
		(snd-display ";filter-sound env: ~A?" (samples 0 8)))
	    (undo)
	    (filter-sound '(0 1 1 0) 1024)
	    (undo)
	    (filter-sound (make-fir-filter 6 (list->vct '(.1 .2 .3 .3 .2 .1))))
	    (undo)
	    (filter-sound (make-delay 120))
	    (undo)
	    (filter-sound (make-formant .99 1200))
	    (undo)
	    (let ((vc0 (make-vct 4)))
	      (vct-set! vc0 0 .125) (vct-set! vc0 1 .25) (vct-set! vc0 2 .25) (vct-set! vc0 3 .125) 
	      (filter-sound vc0 4) 
	      (if (or (fneq (sample 0) 0.125) (fneq (sample 1) .25) (fneq (sample 2) .25) (fneq (sample 5) 0.0))
		  (snd-display ";filter-sound direct: ~A?" (samples 0 8)))
	      (revert-sound)))
	  (close-sound ind0)
	  
	  (set! ind0 (new-sound "fmv.snd" mus-aifc mus-bshort 22050 2 "this is a comment"))
	  (let ((v0 (make-vector 10))
		(ind1 (new-sound "fmv1.snd" mus-aifc mus-bshort 22050 1 "this is a comment")))
	    (set! (sync ind0) 123)
	    (set! (sync ind1) 123)
	    (do ((i 0 (1+ i))) ((= i 10)) (vector-set! v0 i 1.0))
	    (insert-samples 0 10 v0 ind0 0)
	    (insert-samples 0 10 v0 ind0 1)
	    (insert-samples 0 10 v0 ind1 0)
	    (env-sound '(0 0 1 1) 0 10 1.0 ind0)
	    (do ((i 0 (1+ i))) 
		((= i 10)) 
	      (if (fneq (sample i ind0 0) (* i .1111)) (snd-display ";ind0:0 1 env-sound[~D]: ~A?" i (sample i ind0 0)))
	      (if (fneq (sample i ind0 1) (* i .1111)) (snd-display ";ind0:1 1 env-sound[~D]: ~A?" i (sample i ind0 1)))
	      (if (fneq (sample i ind1 0) (* i .1111)) (snd-display ";ind1:0 1 env-sound[~D]: ~A?" i (sample i ind1 0))))
	    (undo) 
	    (env-sound (make-env '(0 0 1 1) :end 9) 0 10 1.0 ind0) 
	    (do ((i 0 (1+ i))) 
		((= i 10)) 
	      (if (fneq (sample i ind0 0) (* i .1111)) (snd-display ";ind0:0 2 env-sound[~D]: ~A?" i (sample i ind0 0)))
	      (if (fneq (sample i ind0 1) (* i .1111)) (snd-display ";ind0:1 2 env-sound[~D]: ~A?" i (sample i ind0 1)))
	      (if (fneq (sample i ind1 0) (* i .1111)) (snd-display ";ind1:0 2 env-sound[~D]: ~A?" i (sample i ind1 0))))
	    (undo) 
	    (env-sound '(0 0 .5 1 1 1) 0 10 0.0 ind0) 
	    (if (or (fneq (sample 3 ind0 0) 0.0) (fneq (sample 8 ind0 0) 1.0) ) 
		(snd-display ";ind0:0 env-sound stepped: ~A ~A?" (sample 3 ind0 0) (sample 8 ind0 0)))
	    (if (or (fneq (sample 3 ind0 1) 0.0) (fneq (sample 8 ind0 1) 1.0) ) 
		(snd-display ";ind0:1 env-sound stepped: ~A ~A?" (sample 3 ind0 1) (sample 8 ind0 1)))
	    (if (or (fneq (sample 3 ind1 0) 0.0) (fneq (sample 8 ind1 0) 1.0) ) 
		(snd-display ";ind1:0 env-sound stepped: ~A ~A?" (sample 3 ind1 0) (sample 8 ind1 0)))
	    (undo)
	    (revert-sound ind0)
	    (revert-sound ind1)
	    (insert-samples 0 10 v0 ind0 0)
	    (insert-samples 0 10 v0 ind0 1)
	    (insert-samples 0 10 v0 ind1 0)
	    (filter-sound (make-one-zero :a0 0.5 :a1 0.0) 0 ind0)
	    (do ((i 0 (1+ i))) 
		((= i 10)) 
	      (if (fneq (sample i ind0 0) 0.5) (snd-display ";ind0:0 1 filter-sound[~D]: ~A?" i (sample i ind0 0)))
	      (if (fneq (sample i ind0 1) 0.5) (snd-display ";ind0:1 1 filter-sound[~D]: ~A?" i (sample i ind0 1)))
	      (if (fneq (sample i ind1 0) 0.5) (snd-display ";ind1:0 1 filter-sound[~D]: ~A?" i (sample i ind1 0))))
	    
	    (close-sound ind1))
	  (close-sound ind0)
	  
	  (set! ind0 (new-sound "fmv.snd" mus-aifc mus-bshort 22050 1 "this is a comment"))
	  (let ((v0 (make-vct 10))
		(old5 (sample 5 ind0)))
	    (vct-fill! v0 0.1)
	    (insert-samples 10 10 v0 ind0)
	    (env-sound '(0 0 1 2) 10 10 1.0 ind0)
	    (do ((i 0 (1+ i))) ((= i 10)) 
	      (if (fneq (sample (+ i 10) ind0) (* i .0222)) (snd-display ";env-sound [~D]: ~A?" (+ i 10) (sample (+ i 10) ind0))))
	    (if (fneq (sample 5 ind0) old5) (snd-display ";env-sound 5: ~A ~A?" old5 (sample 5 ind0)))
	    (undo)
	    (env-sound '(0 0 1 2) 10 10 4.0 ind0)
	    (set! v0 (samples->vct 10 10))
	    (if (or (fneq (vct-ref v0 3) 0.039) (fneq (vct-ref v0 8) .162)) (snd-display ";env-sound 4: ~A" v0))
	    (undo)
	    (env-sound '(0 0 1 2) 10 10 .05 ind0)
	    (set! v0 (samples->vct 10 10))
	    (if (or (fneq (vct-ref v0 3) 0.133) (fneq (vct-ref v0 8) .196)) (snd-display ";env-sound 05: ~A" v0)))
	  
	  (close-sound ind0)
	  (set! ind0 (new-sound "fmv.snd" mus-aifc mus-bshort 22050 2 "this is a comment"))
	  (set! ind1 (new-sound "fmv1.snd" mus-next mus-bshort 22050 1 "this is a comment"))
	  (let ((v0 (make-vector 10)))
	    (do ((i 0 (1+ i))) ((= i 10)) (vector-set! v0 i 1.0))
	    (insert-samples 0 10 v0 ind0 0) 
	    (do ((i 0 (1+ i))) ((= i 10)) (vector-set! v0 i 0.1))
	    (insert-samples 0 10 v0 ind0 1) 
	    (do ((i 0 (1+ i))) ((= i 10)) (vector-set! v0 i 0.01))
	    (insert-samples 0 10 v0 ind1 0) 
	    (let ((val (data-max1 0 9 ind0 0)))
	      (if (fneq val 1.0) (snd-display ";scan-chan[0,0]: ~A?" val)))
	    (let ((val (data-max1 0 9 ind0 1)))
	      (if (fneq val 0.1) (snd-display ";scan-chan[0,1]: ~A?" val)))
	    (let ((val (data-max1 0 9 ind1 0)))
	      (if (fneq val 0.01) (snd-display ";scan-chan[1,0]: ~A?" val)))
	    (let ((val (data-max1 0 9 #f #f)))
	      (if (fneq val 0.01) (snd-display ";scan-chans: ~A?" val)))
	    (let ((val (data-max 0 9)))
	      (if (fneq val 1.0) (snd-display ";scan-all-chans: ~A?" val)))
	    (let ((val (data-max2 0 9 ind0)))
	      (if (fneq val 1.0) (snd-display ";scan-across-sound-chans: ~A?" val))))
	  (close-sound ind0)
	  (close-sound ind1)
	  
	  (set! ind0 (new-sound "fmv.snd" mus-aifc mus-bshort 22050 2 "this is a comment"))
	  (mix "oboe.snd")
	  (let ((m1 (add-mark 100)))
	    (delete-sample 10)
	    (let ((m2 (add-mark 200)))
	      (delete-sample 10)
	      (let ((m3 (add-mark 300)))
		(undo)
		(save-sound)
		(if (not (= (length (marks ind0 0)) 2))
		    (snd-display ";marks after save: ~A" (marks ind0 0)))
		(if (or (not (mark? m1))
			(not (= (mark-sample m1) 99)))
		    (snd-display ";save-sound mark1: ~A" (mark-sample m1)))
		(if (or (not (mark? m2))
			(not (= (mark-sample m2) 200)))
		    (snd-display ";save-sound mark2: ~A" (mark-sample m2)))
		(if (mark? m3) (snd-display ";save-sound mark3: ~A" m3)))))
	  (close-sound ind0)
	  
	  (let ((fd (open-sound "oboe.snd"))
		(m1 (add-mark 123))
		(sync-val (+ 1 (mark-sync-max))))
	    (if (not (mark? m1)) (snd-display ";mark?"))
	    (if (not (= (mark-sample m1) 123)) (snd-display ";add-mark: ~A? " (mark-sample m1)))
	    (set! (mark-property :hiho m1) 123)
	    (if (not (= (mark-property :hiho m1) 123)) (snd-display ";mark-property: ~A" (mark-property m1)))
	    (if (mark-property :not-there m1) (snd-display ";mark-not-property: ~A" (mark-property :not-there m1)))
	    (if (not (eq? (without-errors (mark-sample 12345678)) 'no-such-mark)) (snd-display ";mark-sample err: ~A?" (mark-sample 12345678)))
	    (if (not (eq? (without-errors (add-mark 123 123)) 'no-such-sound)) (snd-display ";add-mark err: ~A?" (add-mark 123 123)))
	    (let ((m2 (without-errors (add-mark 12345 fd 0))))
	      (if (eq? m2 'no-such-mark) (snd-display ";add-mark failed?"))
	      (if (not (= (mark-sample m2) 12345)) (snd-display ";add-mark 0 0: ~A?" (mark-sample m2)))
	      (if (not (= (mark-sync m2) 0)) (snd-display ";init mark-sync: ~A?" (mark-sync m2)))
	      (set! (mark-sync m2) sync-val)
	      (if (not (= (mark-sync m2) sync-val)) (snd-display ";set-mark-sync (~A): ~A?" sync-val (mark-sync m2)))
	      (let* ((syncs (syncd-marks sync-val))
		     (chans (marks fd 0))
		     (samps (map mark-sample chans)))
		(if (not (equal? syncs (list m2))) (snd-display ";syncd-marks: ~A?" syncs))
		(if (not (equal? chans (list m1 m2))) (snd-display ";marks: ~A?" chans))
		(if (not (equal? samps (list (mark-sample m1) (mark-sample m2)))) (snd-display ";map samps: ~A?" samps))
		(delete-samples 200 100 fd 0)
		(set! chans (marks fd))
		(set! samps (map mark-sample (car chans)))
		(if (not (equal? samps (list (mark-sample m1 0) (- (mark-sample m2 0) 100)))) (snd-display ";map samps: ~A?" samps))
		(let ((descr (describe-mark m2)))
		  (if (not (equal? descr (list (list 'mark m2 'sound fd "oboe.snd" 'channel 0) 12345 12245)))
		      (snd-display ";describe-mark: ~A?" descr)))
		(set! (mark-sync m1) (mark-sync m2))
		(move-syncd-marks sync-val 100)
		(set! chans (marks fd))
		(set! samps (map mark-sample (car chans)))
		(if (not (equal? samps (list (+ (mark-sample m1 0) 100) (mark-sample m2 0)))) (snd-display ";syncd move samps: ~A?" samps))
		(set! (cursor) 500)
		(backward-mark)
		(if (not (= (cursor) (mark-sample m1))) (snd-display ";backward-mark: ~A?" (cursor)))
		(forward-mark 1)
		(if (not (= (cursor) (mark-sample m2))) (snd-display ";forward-mark: ~A?" (cursor)))
		(set! (mark-sync m1) #t)
		(if (not (= (mark-sync m1) 1)) (snd-display ";mark-sync via bool: ~A" (mark-sync m1)))
		(delete-mark m1)
		(set! chans (marks fd 0))
		(if (not (equal? chans (list m2))) (snd-display ";delete-mark? ~A" chans))
		(undo)
		(set! chans (marks fd 0))
		(if (not (equal? chans (list m1 m2))) (snd-display ";delete-mark then undo? ~A" chans))
		(redo)
		(if (not (string=? (mark-name m2) "")) (snd-display ";init mark-name: ~A?" (mark-name m2)))
		(set! (mark-name m2) "hiho!")
		(if (not (string=? (mark-name m2) "hiho!")) (snd-display ";set-mark-name: ~A?" (mark-name m2)))
		(undo)
		(if (not (string=? (mark-name m2) "")) (snd-display ";undo mark-name: ~A?" (mark-name m2)))
		(redo)
		(if (not (string=? (mark-name m2) "hiho!")) (snd-display ";redo mark-name: ~A?" (mark-name m2)))
		(let ((m3 (find-mark "hiho!"))
		      (m4 (find-mark (mark-sample m2)))
		      (m5 (find-mark "not-a-mark"))
		      (m6 (find-mark 123456787))
		      (m7 (mark-name->id "hiho!")))
		  (if (not (eq? m2 m3 m4 m7)) (snd-display ";find-mark: ~A ~A ~A ~A?" m2 m3 m4 m7))
		  (if (not (eq? m5 m6 #f)) (snd-display ";find-not-a-mark: ~A ~A?" m5 m6))
		  (set! (mark-sample m2) 2000)
		  (set! m1 (add-mark 1000))
		  (set! m3 (add-mark 3000))
		  (set! m4 (add-mark 4000))
		  (insert-samples 2500 500 (make-vct 500) fd 0)
		  (set! samps (map mark-sample (marks fd 0)))
		  (if (not (equal? samps '(1000 2000 3500 4500))) (snd-display ";insert ripple: ~A?" samps))
		  (set! (mark-sample m3) 300)
		  (set! (cursor) 500)
		  (backward-mark)
		  (if (not (= (cursor) 300)) (snd-display ";sort marks: ~A?" (cursor)))
		  (if (not (equal? (mark-home m2) (list fd 0))) (snd-display ";mark-home: ~A?" (mark-home m2)))
		  (let ((sd (open-sound "4.aiff")))
		    (set! m3 (add-mark 1000 sd 2))
		    (set! m4 (add-mark 1000 sd 3))
		    (if (not (equal? (mark-home m3) (list sd 2))) (snd-display ";marks->sound 4: ~A?" (mark-home m3)))
		    (close-sound sd))
		  (let ((file (save-marks fd)))
		    (if (or (not file)
			    (not (string=? file (string-append cwd "oboe.marks"))))
			(snd-display ";save-marks -> ~A?" file)))
		  (let ((file (save-marks fd "hiho.marks")))
		    (if (or (not file)
			    (not (string=? file "hiho.marks")))
			(snd-display ";save-marks with arg -> ~A?" file))
		    (let ((val (system (format #f "diff hiho.marks ~A" (string-append cwd "oboe.marks")))))
		      (if (not (= val 0))
			  (snd-display ";save marks differs"))))
		  (close-sound fd)
		  (let ((s1 (open-sound "oboe.snd"))
			(s2 (open-sound "oboe.snd")))
		    (add-mark 123 s1 0)
		    (add-mark 321 s2 0)
		    (set! (verbose-cursor) #t)
		    (save-state "s61.scm")
		    (set! (verbose-cursor) #f)
		    (close-sound s1)
		    (close-sound s2))
		  (load "s61.scm")
		  (if (not (verbose-cursor)) (snd-display ";save-state verbose-cursor?"))
		  (let ((s1 (find-sound "oboe.snd" 0))
			(s2 (find-sound "oboe.snd" 1)))
		    (if (or (not (sound? s1)) (not (sound? s2)))
			(snd-display ";can't re-open sounds? ~A ~A" s1 s2)
			(let ((m1 (marks s1))
			      (m2 (marks s2)))
			  (if (or (not (= (length m1) 1))
				  (not (= (length m2) 1))
				  (not (= (length (car m1)) 1))
				  (not (= (length (car m2)) 1)))
			      (snd-display ";save-marks via save-state to: ~A ~A" m1 m2)
			      (let ((samp1 (mark-sample (caar m1)))
				    (samp2 (mark-sample (caar m2))))
				(if (or (not (= samp1 123))
					(not (= samp2 321)))
				    (snd-display ";save-marks via save-state posiitons: ~A ~A" samp1 samp2))))))
		    (if (sound? s1) (close-sound s1))
		    (if (sound? s2) (close-sound s2)))
		  (let ((fd (open-sound "pistol.snd")))
		    (let ((file (save-marks)))
		      (if file
			  (snd-display ";save-marks no marks -> ~A?" file)))
		    (close-sound fd))
		  (let ((fd (open-sound "oboe.snd")))
		    (load "oboe.marks")
		    (let ((mlst (marks fd 0)))
		      (if (not (= (length mlst) 2)) (snd-display ";restore-marks: ~A?" mlst))
		      (if (or (not (= (mark-sample (car mlst)) 123))
			      (not (= (mark-sample (cadr mlst)) 12345)))
			  (snd-display ";restored-marks: ~D ~D?" (mark-sample (car mlst)) (mark-sample (cadr mlst)))))
		    (close-sound fd))
		  (let ((fd (open-sound "oboe.snd")))
		    (let ((m1 (add-mark 1000)))
		      (let ((m2 (add-mark 2500)))
			(let ((m3 (add-mark (- (frames) 4000))))
			  (let ((ms (marks fd 0)))
			    (src-sound -.5)
			    (if (not (equal? (marks fd 0) (reverse (marks fd 0 0))))
				(snd-display ";src rev marks: ~A ~A" (marks fd 0) (reverse (marks fd 0 0))))
			    (let ((ms1 (map mark-sample (marks fd 0))))
			      (if (not (equal? ms1 (list 7998 96654 99654))) ; off-by-1 somewhere...
				  (snd-display ";src rev mark locs: ~A" ms1)))))))
		    (close-sound fd))
		  (let ((fd (open-sound "4.aiff")))
		    (let ((m1 (add-mark 1000 fd 0))
			  (m2 (add-mark 2000 fd 1))
			  (m3 (add-mark 3000 fd 2))
			  (m4 (add-mark 4000 fd 3)))
		      (if (= (length (marks)) 0) (snd-display ";marks (no args): ~A" (marks)))
		      (save-marks fd)
		      (close-sound fd)
		      (set! fd (open-sound "4.aiff"))
		      (load "4.marks")
		      (delete-file "4.marks")
		      (do ((i 0 (1+ i)))
			  ((= i 4))
			(let ((mlst (marks fd i)))
			  (if (not (= (length mlst) 1))
			      (snd-display ";save-marks[~A]: ~A?" i mlst))
			  (if (not (= (mark-sample (car mlst)) (* (+ i 1) 1000)))
			      (snd-display ";save-marks[~A] at ~A?" i (mark-sample (car mlst))))))
		      (close-sound fd)))
		  
		  ))))
	  
	  (let* ((ind (open-sound "pistol.snd"))
		 (samp1 1834)
		 (samp2 8345)
		 (m1 (add-mark samp1 ind 0))
		 (m2 (add-mark samp2)))
	    (src-sound -1)
	    (if (not (= (mark-sample m1) 39788))
		(snd-display ";src -1 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) 33277))
		(snd-display ";src -1 m2 -> ~A" (mark-sample m2)))
	    (undo)
	    (src-sound .5)
	    (if (not (= (mark-sample m1) (* 2 samp1)))
		(snd-display ";src .5 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) (* 2 samp2)))
		(snd-display ";src .5 m2 -> ~A" (mark-sample m2)))
	    (undo)
	    (delete-samples 1000 100)
	    (if (not (= (mark-sample m1) (- samp1 100)))
		(snd-display ";delete 100 m1 -> ~A" (mark-sample m1)))
	    (insert-silence 1000 100)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";insert 100 m1 -> ~A" (mark-sample m1)))
	    (revert-sound ind)
	    (delete-samples 2000 100)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";delete(2) 100 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) (- samp2 100)))
		(snd-display ";delete(2) 100 m2 -> ~A" (mark-sample m2)))
	    (insert-silence 2000 100)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";insert(2) 100 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) samp2))
		(snd-display ";insert(2) 100 m2 -> ~A" (mark-sample m2)))
	    (revert-sound ind)
	    (delete-samples 10000 100)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";delete(3) 100 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) samp2))
		(snd-display ";delete(3) 100 m2 -> ~A" (mark-sample m2)))
	    (insert-silence 10000 100)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";insert(3) 100 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) samp2))
		(snd-display ";insert(3) 100 m2 -> ~A" (mark-sample m2)))
	    (src-sound '(0 .5 1 .5 2 1))
	    (if (not (= (mark-sample m1) (* 2 samp1)))
		(snd-display ";src env .5 m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) (* 2 samp2)))
		(snd-display ";src env .5 m2 -> ~A" (mark-sample m2)))
	    (undo)
	    (reverse-sound)
	    (if (not (= (mark-sample m1) 39788))
		(snd-display ";reverse-sound m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) 33277))
		(snd-display ";reverse-sound m2 -> ~A" (mark-sample m2)))
	    (undo)
	    (src-sound '(0 -.5 1 -.5 2 -1))
	    (if (not (= (mark-sample m1) 68598))
		(snd-display ";src -env m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) 61160))
		(snd-display ";src -env m2 -> ~A" (mark-sample m2)))
	    (revert-sound ind)
	    (src-channel (make-env '(0 .5 1 1) :end 8000) 2000 10000)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";src-channel(1) m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) 11345))
		(snd-display ";src-channel(1) m2 -> ~A" (mark-sample m2)))
	    (undo)
	    (src-channel (make-env '(0 .5 1 1) :end 8000) 0 8000)
	    (if (not (= (mark-sample m1) 3303))
		(snd-display ";src-channel(2) m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) samp2))
		(snd-display ";src-channel(2) m2 -> ~A" (mark-sample m2)))
	    (undo)
	    (src-channel (make-env '(0 .5 1 1) :end 8000) 10000 8000)
	    (if (not (= (mark-sample m1) samp1))
		(snd-display ";src-channel(3) m1 -> ~A" (mark-sample m1)))
	    (if (not (= (mark-sample m2) samp2))
		(snd-display ";src-channel(3) m2 -> ~A" (mark-sample m2)))
	    (close-sound ind)
	    (set! ind (open-sound "2.snd"))
	    (set! (sync ind) #t)
	    (let ((m3 (add-mark 1000 ind 0))
		  (m4 (add-mark 8000 ind 1)))
	      (swap-channels)
	      (if (or (not (equal? (mark-home m3) (list ind 1)))
		      (not (equal? (mark-home m4) (list ind 0))))
		  (snd-display ";swapped mark homes: ~A ~A?" (mark-home m3) (mark-home m4)))
	      (if (or (not (= (mark-sample m3) 1000))
		      (not (= (mark-sample m4) 8000)))
		  (snd-display ";swapped mark samples: ~A ~A?" (mark-sample m3) (mark-sample m4)))
	      (close-sound ind))
	    (set! ind (open-sound "2.snd"))
	    (set! (sync ind) #t)
	    (let ((m3 (add-mark 1000 ind 0)))
	      (delete-samples 1000 10 ind 1)
	      (swap-channels)
	      (if (not (equal? (mark-home m3) (list ind 1)))
		  (snd-display ";edited swapped mark home: ~A?" (mark-home m3)))
	      (if (not (= (mark-sample m3) 1000))
		  (snd-display ";edited swapped mark sample: ~A" (mark-sample m3)))
	      (delete-marks))
	    (close-sound ind))
	  
	  (let* ((ind (open-sound "oboe.snd"))
		 (m1 (add-mark 123 ind 0))
		 (m2 (add-mark 234 ind 0)))
	    (define-selection-via-marks m1 m2)
	    (if (not (selection?))
		(snd-display ";define-selection-via-marks failed?")
		(let ((mc (selection-members)))
		  (if (not (equal? mc (list (list ind 0)))) (snd-display ";selection-members after mark definition: ~A (should be '((~A 0)))" mc ind))
		  (if (not (= (selection-position) 123)) (snd-display ";selection-position 123: ~A" (selection-position)))
		  (if (not (= (selection-frames) 112)) (snd-display ";selection-frames 112: ~A" (selection-frames)))))
	    (set! m1 (add-mark 1000 ind 0))
	    (set! m2 (add-mark 2000 ind 0))
	    (define-selection-via-marks m1 m2)
	    (if (not (selection?))
		(snd-display ";define-selection-via-marks repeat failed?")
		(let ((mc (selection-members)))
		  (if (not (equal? mc (list (list ind 0)))) (snd-display ";selection-members after 2nd mark definition: ~A (should be '((~A 0)))" mc ind))
		  (if (not (= (selection-position) 1000)) (snd-display ";selection-position 1000: ~A" (selection-position)))
		  (if (not (= (selection-frames) 1001)) (snd-display ";selection-frames 1001: ~A" (selection-frames)))))
	    (set! (selection-member? #t) #f)
	    (if (selection?) (snd-display ";can't clear selection via selection-member?"))
	    (set! (selection-member? ind 0) #t)
	    (set! (selection-position ind 0) 2000)
	    (set! (selection-frames ind 0) 1234)
	    (snap-marks)
	    (set! m1 (find-mark 2000 ind 0))
	    (if (not (mark? m1)) (snd-display ";snap-marks start: ~A" (map mark-sample (marks ind 0))))
	    (set! m2 (find-mark (+ 2000 1234)))
	    (if (not (mark? m2)) (snd-display ";snap-marks end: ~A" (map mark-sample (marks ind 0))))
	    (set! (selection-position ind 0) (+ (frames ind 0) 1123))
	    (if (not (= (selection-position ind 0) (1- (frames ind ))))
		(snd-display ";selection position past eof: ~A ~A" (selection-position ind 0) (1- (frames ind ))))
	    (revert-sound ind)
	    (src-sound '(0 .5 1 1.75665))
	    ;; trying to hit previous dur on the nose "by accident..."
	    (close-sound ind)
	    )
	  
	  (let ((ind (open-sound "oboe.snd"))
		(mtests 100))
	    (do ((i 0 (1+ i)))
		((= i mtests))
	      (let* ((current-marks (marks ind 0))
		     (current-samples (map mark-sample current-marks)))
		
		(if (not (null? current-marks))
		    (let ((id (list-ref current-marks (irandom (- (length current-marks) 1)))))
		      (if (not (= id (find-mark (mark-sample id)))) 
			  (snd-display ";two marks at ~A? ~A" (mark-sample id) (map mark-sample current-marks)))
		      (if (find-mark "not-a-name") (snd-display ";find-bogus-mark: ~A" (find-mark "not-a-name")))))
		
		(case (irandom 15)
		  ((0) (let* ((beg (irandom (frames)))
			      (dur (max 1 (irandom 100)))
			      (end (+ beg dur)))
			 (insert-silence beg dur)
			 (if (not (null? current-marks))
			     (for-each
			      (lambda (id old-loc)
				(if (> old-loc beg)
				    (begin
				      (if (not (mark? id))
					  (snd-display ";insert clobbered mark: ~A" id)
					  (if (not (= (mark-sample id) (+ old-loc dur)))
					      (snd-display ";insert, mark ~D ~D -> ~D (~D)" id old-loc (mark-sample id) dur))))))
			      current-marks
			      current-samples))))
		  ((1) (if (> (car (edits ind 0)) 0) (undo)))
		  ((2) (if (> (cadr (edits ind 0)) 0) (redo)))
		  ((3) (if (> (maxamp ind 0) .1) (scale-channel .5) (scale-channel 2.0))
		   (if (not (equal? (marks ind 0) current-marks))
		       (snd-display ";scaling changed marks: ~A ~A" (marks ind 0) current-marks))
		   (if (not (equal? (map mark-sample (marks ind 0)) current-samples))
		       (snd-display ";scaling changed mark locations: ~A ~A" (map mark-sample (marks ind 0)) current-samples)))
		  ((4) (set! (sample (irandom (1- (frames)))) .5)
		   (if (not (equal? (marks ind 0) current-marks))
		       (snd-display ";set-sample changed marks: ~A ~A" (marks ind 0) current-marks))
		   (if (not (equal? (map mark-sample (marks ind 0)) current-samples))
		       (snd-display ";set-sample changed mark locations: ~A ~A" (map mark-sample (marks ind 0)) current-samples)))
		  ((5) (let* ((beg (irandom (frames)))
			      (dur (max 1 (irandom 100)))
			      (end (+ beg dur)))
			 (delete-samples beg dur)
			 (if (not (null? current-marks))
			     (for-each
			      (lambda (id old-loc)
				(if (and (> old-loc beg)
					 (< old-loc end)
					 (mark? id))
				    (snd-display ";delete did not clobber mark: ~A ~A [~A ~A]" id old-loc beg end)
				    (if (and (> old-loc end)
					     (not (= (mark-sample id) (- old-loc dur))))
					(snd-display ";delete ripple mark ~D ~D -> ~D (~D)" id old-loc (mark-sample id) dur)
					(if (and (< old-loc beg)
						 (not (= (mark-sample id) old-loc)))
					    (snd-display ";delete but mark before: ~A ~A ~A ~A" id old-loc (mark-sample id) beg)))))
			      current-marks
			      current-samples))))
		  ((6) (revert-sound))
		  ((7) (if (and (not (null? current-marks))
				(> (length current-marks) 1))
			   (let ((id (list-ref current-marks (irandom (- (length current-marks) 1)))))
			     (delete-mark id)
			     (if (mark? id)
				 (snd-display ";delete-mark failed? ~A" id))
			     (if (not (= (length (marks ind 0)) (1- (length current-marks))))
				 (snd-display ";delete-mark list trouble: ~A ~A ~A" id current-marks (marks ind 0))))))
		  ((8) (let ((rate (if (> (frames) 200000) 2.0 0.5)))
			 (src-channel rate)
			 (if (not (null? current-marks))
			     (for-each
			      (lambda (id old-loc)
				(if (not (mark? id))
				    (snd-display ";src-channel clobbered mark: ~A" id)
				    (if (> (abs (- (/ old-loc rate) (mark-sample id))) 2)
					(snd-display ";src moved mark: ~A ~A ~A (~A -> ~A)" 
						     id old-loc (mark-sample id) rate (- (/ old-loc rate) (mark-sample id))))))
			      current-marks
			      current-samples))))
		  ((9) (reverse-channel)
		   (if (not (null? current-marks))
		       (for-each
			(lambda (id old-loc)
			  (if (not (mark? id))
			      (snd-display ";reverse-channel clobbered mark: ~A" id)
			      (if (> (abs (- (- (frames) old-loc) (mark-sample id))) 2)
				  (snd-display ";reverse moved mark: ~A ~A ~A (~A)" 
					       id old-loc (- (frames) old-loc) (mark-sample id)))))
			current-marks
			current-samples)))
		  (else (add-mark (irandom (1- (frames))))))))
	    (close-sound ind))
	  
	  (if (and (provided? 'snd-motif) (provided? 'xm)) (mark-sync-color "blue"))
	  (let ((ind (open-sound "oboe.snd")))
	    (let ((m0 (add-mark 4321)))
	      (delete-sample 100)
	      (let ((m1 (add-mark 1234)))
		(let ((val0 (describe-mark m0))
		      (val1 (describe-mark m1)))
		  (if (or (not (= (list-ref (car val0) 1) m0))
			  (not (= (list-ref (car val0) 3) ind))
			  (not (= (list-ref (car val0) 6) 0))
			  (not (= (list-ref val0 1) 4321))
			  (not (= (list-ref val0 2) 4320)))
		      (snd-display ";describe-mark m0: ~A" val0))
		  (if (or (not (= (list-ref (car val1) 1) m1))
			  (not (= (list-ref (car val1) 3) ind))
			  (not (= (list-ref (car val1) 6) 0))
			  (not (eq? (list-ref val1 1) #f))
			  (not (= (list-ref val1 2) 1234)))
		      (snd-display ";describe-mark m1: ~A" val1))
		  (delete-mark m0)
		  (delete-sample 5000)
		  (set! val0 (describe-mark m0))
		  (set! val1 (describe-mark m1))
		  (if (or (not (= (list-ref (car val0) 1) m0))
			  (not (= (list-ref (car val0) 3) ind))
			  (not (= (list-ref (car val0) 6) 0))
			  (not (= (list-ref val0 1) 4321))
			  (not (eq? (list-ref val0 2) #f))
			  (not (eq? (list-ref val0 3) #f)))
		      (snd-display ";describe-mark m0 [1]: ~A" val0))
		  (if (or (not (= (list-ref (car val1) 1) m1))
			  (not (= (list-ref (car val1) 3) ind))
			  (not (= (list-ref (car val1) 6) 0))
			  (not (eq? (list-ref val1 1) #f))
			  (not (= (list-ref val1 2) 1234))
			  (not (= (list-ref val1 3) 1234)))
		      (snd-display ";describe-mark m1 [1]: ~A" val1)))))
	    (revert-sound ind)
	    (add-hook! draw-mark-hook (lambda (id) #t))
	    (let ((m0 (add-mark 4321))
		  (m1 (add-mark 1234))
		  (dur (/ (frames ind) (srate ind))))
	      (pad-marks (list m0 m1) .01)
	      (if (fneq (/ (frames ind) (srate ind)) (+ dur .02))
		  (snd-display ";pad-marks: ~A ~A" dur (/ (frames ind) (srate ind))))
	      (if (and (not (= (mark-sample m0) 4763)) 
		       (not (= (mark-sample m0) 4761)))
		  (snd-display ";pad-marks m0 pos: ~A" (mark-sample m0)))
	      (if (fneq (sample 1235) 0.0) (snd-display ";pad-marks 1235: ~A" (sample 1235))))
	    (close-sound ind))
	  (reset-hook! draw-mark-hook)
	  (let ((ind (open-sound "oboe.snd")))
	    (if (forward-mark) (snd-display ";forward-mark when no marks: ~A" (forward-mark)))
	    (if (backward-mark) (snd-display ";backward-mark when no marks: ~A" (backward-mark)))
	    (if (find-mark 12345) (snd-display ";find-mark when no marks: ~A" (find-mark 12345)))
	    (let ((m0 (add-mark 123 ind 0)))
	      (delete-sample 0)
	      (let ((m1 (add-mark 23 ind 0)))
		(set! (mark-name m1) "23")
		(delete-sample 0)
		(let ((m00 (find-mark 123 ind 0 0))
		      (m01 (find-mark "23"))
		      (m02 (find-mark 121)))
		  (if (not m00) (snd-display ";can't find 00th mark"))
		  (if (not m01) (snd-display ";can't find 01th mark"))
		  (if (not m02) (snd-display ";can't find 02th mark"))
		  (delete-mark (find-mark "23"))
		  (scale-by 2.0)
		  (set! m1 (add-mark 1234))
		  (set! (mark-name m1) "23")
		  (let ((m10 (find-mark "23"))
			(m11 (find-mark "23" ind 0 1))
			(m12 (find-mark "23" ind 0 2)))
		    (if (not m10) (snd-display ";can't find 10th mark")
			(if (not (= (mark-sample m10) 1234)) (snd-display ";mark 10th: ~A" (mark-sample m10))))
		    (if (not m11) (snd-display ";can't find 11th mark")
			(if (not (= (mark-sample m11 1) 23)) (snd-display ";mark 11th: ~A" (mark-sample m11 1))))
		    (if (mark? m12) (snd-display ";found 12th mark: ~A ~A ~A" m12 (mark-sample m12 2) (mark-name m12 2)))))))
	    (close-sound ind))
	  (if (string? sf-dir)
	      (let ((ind (open-sound (string-append sf-dir "forest.aiff"))))
		(mark-loops)
		(let ((pos (map mark-sample (marks ind 0))))
		  (if (not (equal? pos (list 24981 144332)))
		      (snd-display ";forest marked loops: ~A ~A" (marks ind 0) pos)))
		(close-sound ind)))
	  
	  ))
      (run-hook after-test-hook 10)
      ))


;;; ---------------- test 11: dialogs ----------------

(define (string-equal-ignoring-white-space s1 s2)
  (let ((len1 (string-length s2)))
    (define (white-space? str pos)
      (or (char=? (string-ref str pos) #\space)
	  (char=? (string-ref str pos) #\newline)))
    (call-with-current-continuation
     (lambda (return)
       (do ((i 0)
	    (j 0))
	   ((= i len1) (begin (while (and (< j len1) (white-space? s2 j)) (set! j (+ j 1))) (= j len1)))
	 (if (char=? (string-ref s1 i) (string-ref s2 j))
	     (begin
	       (set! i (+ i 1))
	       (set! j (+ j 1)))
	     (begin
	       (while (and (< i len1) (white-space? s1 i)) (set! i (+ i 1)))
	       (while (and (< j len1) (white-space? s2 j)) (set! j (+ j 1)))
	       (if (not (char=? (string-ref s1 i) (string-ref s2 j)))
		   (return #f)))))))))


(defvar env1 '(0 0 1 0))
(defvar env2 '(0 0 1 1))
(defvar ramp-up-env '(0 0 1 1))
(define-envelope env4 '(0 1 1 0))

(if (and with-gui
	 (or full-test (= snd-test 11) (and keep-going (<= snd-test 11))))
    (begin
      (run-hook before-test-hook 11)
      (without-errors (peaks))
      (mus-audio-describe) 
      (let ((envd (enved-dialog) ))
	(let ((cold (color-dialog))
	      (ord (orientation-dialog))
	      (trd (transform-dialog))
	      (fild (view-files-dialog))
	      (regd (view-regions-dialog))
	      (pd (print-dialog))
	      (ehd (without-errors (edit-header-dialog))))
	  (if (not (equal? cold (list-ref (dialog-widgets) 0)))
	      (snd-display ";color-dialog -> ~A ~A" cold (list-ref (dialog-widgets) 0)))
	  (if (not (equal? pd (list-ref (dialog-widgets) 17)))
	      (snd-display ";print-dialog -> ~A ~A" pd (list-ref (dialog-widgets) 17)))
	  (if (not (equal? trd (list-ref (dialog-widgets) 5)))
	      (snd-display ";transform-dialog -> ~A ~A" trd (list-ref (dialog-widgets) 5)))
	  (if (not (equal? regd (list-ref (dialog-widgets) 19)))
	      (snd-display ";view-regions-dialog -> ~A ~A" trd (list-ref (dialog-widgets) 19)))
	  (let ((wid (open-file-dialog #f)))
	    (if (not (equal? wid (list-ref (dialog-widgets) 6)))
		(snd-display ";open-file-dialog-> ~A ~A" wid (list-ref (dialog-widgets) 6))))
	  (let ((wid (mix-file-dialog #f)))
	    (if (not (equal? wid (list-ref (dialog-widgets) 11)))
		(snd-display ";mix-file-dialog-> ~A ~A" wid (list-ref (dialog-widgets) 11))))
	  (if (not (provided? 'snd-gtk))
	      (begin
					;(recorder-dialog) 
		(set! (recorder-file) "hiho.snd")
		(if (not (string=? (recorder-file) "hiho.snd")) (snd-display ";set-recorder-file: ~A?" (recorder-file)))
		(set! (recorder-in-format) mus-mulaw)
		(if (not (= (recorder-in-format) mus-mulaw)) (snd-display ";set-recorder-in-format: ~A?" (recorder-in-format)))
		(set! (recorder-in-device) mus-audio-line-in)
		(if (not (= (recorder-in-device) mus-audio-line-in)) (snd-display ";set-recorder-in-device: ~A?" (recorder-in-device)))
		(set! (recorder-out-format) mus-mulaw)
		(if (not (= (recorder-out-format) mus-mulaw)) (snd-display ";set-recorder-out-format: ~A?" (recorder-out-format)))
		(set! (recorder-out-type) mus-aifc)
		(if (not (= (recorder-out-type) mus-aifc)) (snd-display ";set-recorder-out-type: ~A?" (recorder-out-type)))
		(set! (recorder-srate) 44100)
		(if (not (= (recorder-srate) 44100)) (snd-display ";set-recorder-srate: ~A?" (recorder-srate)))
		(set! (recorder-gain 0) 0.5)
		(if (fneq (recorder-gain 0) 0.5) (snd-display ";set-recorder-gain: ~A?" (recorder-gain 0)))
		(set! (recorder-out-amp 0) 0.5)
		(if (> (abs (- (recorder-out-amp 0) 0.5)) .01) (snd-display ";set-recorder-out-amp: ~A?" (recorder-out-amp 0)))
		(set! (recorder-in-amp 0 0) 0.5)
		(if (> (abs (- (recorder-in-amp 0 0) 0.5)) .01) (snd-display ";set-recorder-in-amp: ~A?" (recorder-in-amp 0 0)))))
	  (let ((held (help-dialog "Test" "snd-test here")))
	    (if (not (= (length (menu-widgets)) 6)) (snd-display ";menu-widgets: ~A?" (menu-widgets)))
	    (if (not (equal? (widget-position (car (menu-widgets))) (list 0 0)))
		(snd-display ";position main menubar: ~A?" (widget-position (car (menu-widgets)))))
	    (if (not (equal? held (list-ref (dialog-widgets) 14)))
		(snd-display ";help-dialog -> ~A ~A" held (list-ref (dialog-widgets) 14)))
	    (save-envelopes "hiho.env")
	    (load "hiho.env")
	    (if (not (equal? env4 (list 0.0 1.0 1.0 0.0))) (snd-display ";save-envelopes: ~A?" env4))
	    (delete-file "hiho.env")
	    (help-dialog "test2" "this is the next test" 
			 (list "string 1{open-sound}" "{env-sound}string2" "string{close-sound}3")
			 (list "extsnd.html#sndopen" "extsnd.html#sndenv" "extsnd.html#sndclose"))
	    (dismiss-all-dialogs)
	    )))
      
      (let ((ind (open-sound "oboe.snd")))
	(edit-header-dialog ind)
	(dismiss-all-dialogs)
	(close-sound ind))
      (if (not (string=? (snd-url 'open-sound) "extsnd.html#opensound"))
	  (snd-display ";snd-url 'open-sound: ~A" (snd-url 'open-sound)))
      (if (not (string=? (snd-url "open-sound") "extsnd.html#opensound"))
	  (snd-display ";snd-url \"open-sound\": ~A" (snd-url "open-sound")))
      (if (not (list? (snd-urls))) (snd-display ";snd-urls: ~A" (snd-urls)))
      (let ((str1 (snd-help open-sound))
	    (str2 (snd-help 'open-sound))
	    (str3 (snd-help "open-sound")))
	(if (or (not (string? str1)) ; can happen if we're running -DTIMING
		(not (string-equal-ignoring-white-space str2 str3)))
	    (snd-display ";snd-help open-sound: ~A ~A ~A" str1 str2 str3)))
      (if (not (string? (snd-help 'open-soud)))
	  (snd-display ";snd-help open-soud (misspelled on purpose) failed"))
      (if (not (string-equal-ignoring-white-space (snd-help enved-base) "(enved-base): envelope editor exponential base value (1.0)"))
	  (snd-display ";snd-help enved-base: ~A?" (snd-help enved-base)))
      (if (not (string-equal-ignoring-white-space (snd-help vu-font) "(vu-font): name of font used to make VU meter labels (courier)"))
	  (snd-display ";snd-help vu-font: ~A?" (snd-help vu-font)))
      (if (not (string-equal-ignoring-white-space (snd-help 'enved-base) "(enved-base): envelope editor exponential base value (1.0)"))
	  (snd-display ";snd-help 'enved-base: ~A?" (snd-help 'enved-base)))
      (if (not (string-equal-ignoring-white-space (snd-help 'vu-font) "(vu-font): name of font used to make VU meter labels (courier)"))
	  (snd-display ";snd-help 'vu-font: ~A?" (snd-help 'vu-font)))
      (if (not (string-equal-ignoring-white-space (snd-help "enved-base") "(enved-base): envelope editor exponential base value (1.0)"))
	  (snd-display ";snd-help \"enved-base\": ~A?" (snd-help "enved-base")))
      (if (not (string-equal-ignoring-white-space (snd-help "vu-font") "(vu-font): name of font used to make VU meter labels (courier)"))
	  (snd-display ";snd-help \"vu-font\": ~A?" (snd-help "vu-font")))
      (let ((old-val hamming-window))
	(let ((str1 (snd-help 'hamming-window))
	      (str2 (snd-help "hamming-window")))
	  (if (or (not (string-equal-ignoring-white-space str1 str2))
		  (not (string-equal-ignoring-white-space str1 "A raised cosine")))
	      (snd-display ";snd-help hamming-window: ~A ~A" str1 str2)))
	(if (or (not (number? hamming-window))
		(not (= hamming-window old-val)))
	    (snd-display ";snd-help clobbered out-of-module variable: ~A ~A" old-value hamming-window)))
      
      (set! (show-indices) #t)
      (let ((ind (open-sound "oboe.snd")))
	(if (< (length (sound-widgets ind)) 4)
	    (snd-display ";sound-widgets: ~A?" (sound-widgets ind)))
	(report-in-minibuffer "hi there" ind)
	(if with-gui
	    (let ((str (widget-text (list-ref (sound-widgets ind) 3))))
	      (if (not (string=? str "hi there"))
		  (snd-display ";report-in-minibuffer: ~A?" str))))
	(append-to-minibuffer "away!" ind)
	(if with-gui
	    (let ((str (widget-text (list-ref (sound-widgets ind) 3))))
	      (if (not (string=? str "hi thereaway!"))
		  (snd-display ";report-in-minibuffer 1: ~A?" str))
	      (if (widget-text (cadr (main-widgets))) 
		  (snd-display ";widget text should be #f: ~A" (widget-text (cadr (main-widgets)))))
	      (let ((str (format #f "~D: ~A" ind (short-file-name ind)))
		    (txt (widget-text (cadr (sound-widgets ind)))))
		(if (or (not (string? txt))
			(not (string=? str txt)))
		    (snd-display ";name text: ~A ~A" str txt)))))
	(close-sound ind))
      (if (file-exists? "link-oboe.snd")
	  (let* ((ind (open-sound "link-oboe.snd"))
		 (linked-str (format #f "~D: (~A)" ind (short-file-name ind))))
	    (if with-gui
		(if (not (string=? linked-str (widget-text (cadr (sound-widgets ind)))))
		    (snd-display ";linked name text: ~A ~A" linked-str (widget-text (cadr (sound-widgets ind))))))
	    (if (and (provided? 'xm) (provided? 'snd-debug))
		(XtCallCallbacks (cadr (sound-widgets ind)) XmNactivateCallback (snd-sound-pointer ind)))
	    (close-sound ind)))
      (set! (show-indices) #f)
      (if (file-exists? "link-oboe.snd")
	  (let* ((ind (open-sound "link-oboe.snd"))
		 (linked-str (format #f "(~A)" (short-file-name ind))))
	    (if with-gui
		(if (not (string=? linked-str (widget-text (cadr (sound-widgets ind)))))
		    (snd-display ";linked name text (no index): ~A ~A" linked-str (widget-text (cadr (sound-widgets ind))))))
	    (close-sound ind)))
      
      (if (and (provided? 'xm)
	       (provided? 'snd-motif))
	  (begin
	    (snd-error "a test")
	    (let ((errwid (list-ref (dialog-widgets) 3)))
	      (if (not (Widget? errwid))
		  (snd-display ";snd-error no dialog?")
		  (let ((OK (find-child errwid "OK")))
		    (if (Widget? OK)
			(XtCallCallbacks OK XmNactivateCallback #f)))))))

      (define-envelope test-ramp '(0 0 1 1))
      (if (not (equal? test-ramp '(0 0 1 1))) (snd-display ";define-envelope test-ramp: ~A" test-ramp))
      (define-envelope test-ramp '(0 1 1 0))
      (if (not (equal? test-ramp '(0 1 1 0))) (snd-display ";re-define-envelope test-ramp: ~A" test-ramp))
      
      (define-envelope ramp32 '(0 0 1 1) 32.0)
      (define-envelope ramp032 '(0 0 1 1) 0.032)
      (define-envelope ramp12 '(0 0 1 1 2 0) 3.0)
      (define-envelope ramp012 '(0 0 1 1 2 0) .3)

      (for-each
       (lambda (vals)
	 (let* ((e (car vals))
		(name (cadr vals))
		(base (caddr vals))
		(ebase (object-property e 'envelope-base)))
	   (if (fneq ebase base) (snd-display ";define-envelope ~A base: ~A ~A" name base ebase))))
       (list (list ramp32 "ramp32" 32.0)
	     (list ramp032 "ramp032" .032)
	     (list ramp12 "ramp12" 3.0)
	     (list ramp012 "ramp012" .3)
	     (list test-ramp "test-ramp" 1.0)))

	  (let ((ind (new-sound "fmv.snd" mus-aifc mus-bshort 22050 1 "define-envelope tests" 10)))
	    (map-channel (lambda (y) 1.0))
	    (env-sound ramp32)
	    (let ((data (channel->vct)))
	      (if (not (vequal data (vct 0.000 0.015 0.037 0.070 0.118 0.189 0.293 0.446 0.670 1.000)))
		  (snd-display ";define envelope ramp32 env-sound: ~A" data))
	      (undo)
	      (env-channel ramp032)
	      (set! data (channel->vct))
	      (if (not (vequal data (vct 0.000 0.328 0.552 0.705 0.809 0.880 0.929 0.962 0.985 1.000)))
		  (snd-display ";define envelope ramp032 env-sound: ~A" data))
	      (undo)
	      (if (provided? 'snd-debug)
		  (begin
		    (let ((we (window-env ramp32 0 100 0 200)))
		      (if (not (feql we (list 0.0 0.0 1.0 0.150)))
			  (snd-display ";window ramp32: ~A" we)))
		    (let ((we (window-env ramp32 100 100 0 200)))
		      (if (not (feql we (list 0.0 0.150 1.0 1.0)))
			  (snd-display ";window (100 100) ramp32: ~A" we)))
		    (let ((we (window-env ramp032 0 100 0 200)))
		      (if (not (feql we (list 0.0 0.0 1.0 0.848)))
			  (snd-display ";window ramp032: ~A" we)))
		    (let ((we (window-env ramp032 50 50 0 200)))
		      (if (not (feql we (list 0.0 0.596 1.0 0.848)))
			  (snd-display ";window (50 50) ramp032: ~A" we)))
		    (let ((we (window-env ramp12 50 100 0 200)))
		      (if (not (feql we (list 0.0 0.366 0.5 1.0 1.0 0.366)))
			  (snd-display ";window ramp12: ~A" we)))
		    (let ((me (multiply-envs '(0 1 1 1) ramp32 .1)))
		      (if (not (feql me (list 0.0 0.0 0.1 0.0134 0.2 0.032 0.3 0.059 0.4 0.097 0.5 0.150 0.6 0.226 0.7 0.333 0.8 0.484 0.9 0.697 1.0 1.0)))
			  (snd-display ";multiply ramp32+flat: ~A" me)))
		    (let ((me (multiply-envs '(0 0 1 1) ramp32 .1)))
		      (if (not (feql me (list 0.0 0.0 0.1 0.001 0.2 0.006 0.3 0.017 0.4 0.0388 0.5 0.075 0.6 0.135 0.7 0.233 0.8 0.387 0.9 0.628 1.0 1.0)))
			  (snd-display ";multiply ramp32+ramp: ~A" me)))
		    (let ((me (multiply-envs ramp32 ramp032 .1)))
		      (if (not (feql me (list 0.0 0.0 0.1 0.004 0.2 0.016 0.3 0.039 0.4 0.075 0.5 0.127 0.6 0.204 0.7 0.313 0.8 0.468 0.9 0.688 1.0 1.0)))
			  (snd-display ";multiply ramp32+ramp032: ~A" me)))
		    (let ((me (invert-env ramp32)))
		      (if (or (not (feql me (list 0.0 1.0 1.0 0.0)))
			      (not (number? (object-property me 'envelope-base)))
			      (fneq (object-property me 'envelope-base) (/ 1.0 32.0)))
			  (snd-display ";invert env (ramp32): ~A ~A" me (object-properties me))))
		    (let ((me (invert-env ramp012)))
		      (if (or (not (feql me (list 0.0 1.0 1.0 0.0 2.0 1.0)))
			      (not (number? (object-property me 'envelope-base)))
			      (fneq (object-property me 'envelope-base) (/ 1.0 0.3)))
			  (snd-display ";invert env (ramp012): ~A ~A" me (object-properties me))))
		    (let ((me (invert-env test-ramp)))
		      (if (or (not (feql me (list 0.0 0.0 1.0 1.0)))
			      (and (number? (object-property me 'envelope-base))
				   (fneq (object-property me 'envelope-base) 1.0)))
			  (snd-display ";invert env (test-ramp): ~A ~A" me (object-properties me))))
		    (let ((we (window-env ramp32 0 100 0 200 .1)))
		      (if (not (feql we (list 0.0 0.0 0.200 0.0133 0.400 0.0322 0.600 0.0589 0.800 0.0967 1.0 0.150)))
			  (snd-display ";window ramp32 i: ~A" we)))
		    (let ((we (window-env ramp32 100 100 0 200 .2)))
		      (if (not (feql we (list 0.0 0.150 0.4 0.333 0.8 0.697 1.0 1.0)))
			  (snd-display ";window (100 100) ramp32 i: ~A" we)))
		    (let ((we (window-env ramp032 0 100 0 200 .1)))
		      (if (not (feql we (list 0.0 0.0 0.200 0.300 0.400 0.514 0.600 0.665 0.800 0.772 1.0 0.848)))
			  (snd-display ";window ramp032 i: ~A" we)))
		    (let ((we (window-env ramp032 50 50 0 200 .2)))
		      (if (not (feql we (list 0.0 0.596 0.8 0.813 1.0 0.848)))
			  (snd-display ";window (50 50) ramp032 i: ~A" we)))
		    (let ((we (window-env ramp12 50 100 0 200 .1)))
		      (if (not (feql we (list 0.0 0.366 0.100 0.466 0.200 0.578 0.300 0.704 0.400 0.843 
					      0.5 1.0 0.60 0.843 0.70 0.704 0.80 0.578 0.90 0.466 1.0 0.366)))
			  (snd-display ";window ramp12 i: ~A" we)))))

	      (close-sound ind)))


      (run-hook after-test-hook 11)
      ))

(define sf-dir-files
  (if (string? sf-dir) 
      (let ((good-files '()))
	(for-each ; omit bad headers (test cases) 
	 (lambda (file)
	   (catch 'mus-error
		  (lambda () 
		    (if (and (< (mus-sound-chans (string-append sf-dir file)) 256)
			     (> (mus-sound-chans (string-append sf-dir file)) 0)
			     (>= (mus-sound-data-format (string-append sf-dir file)) 0)
			     (> (mus-sound-srate (string-append sf-dir file)) 0)
			     (>= (mus-sound-frames (string-append sf-dir file)) 0))
			(set! good-files (cons file good-files))))
		  (lambda args 
		    (car args))))
	 (sound-files-in-directory sf-dir))
	good-files)
      #f))

(define sf-dir-len (if sf-dir-files (length sf-dir-files) 0))
(define buffer-menu #f)

(define (remove-if p l)
  (cond ((null? l) '())
	((p (car l)) (remove-if p (cdr l)))
	(else (cons (car l) (remove-if p (cdr l))))))


;;; ---------------- test 12:  extensions etc ----------------

(define (spectral-difference snd1 snd2)
  (let* ((size (max (frames snd1) (frames snd2)))
	 (pow2 (ceiling (/ (log size) (log 2))))
	 (fftlen (inexact->exact (expt 2 pow2)))
	 (fdr1 (make-vct fftlen))
	 (fdr2 (make-vct fftlen)) )
    (samples->vct 0 fftlen snd1 0 fdr1)
    (samples->vct 0 fftlen snd2 0 fdr2)
    (let* ((spectr1 (snd-spectrum fdr1 blackman2-window fftlen #t))
	   (spectr2 (snd-spectrum fdr2 blackman2-window fftlen #t))
	   (diff 0.0)
	   (diffs (vct-subtract! spectr1 spectr2))
	   (len (vct-length diffs)))
      (do ((i 0 (1+ i)))
	  ((= i len) diff)
	(set! diff (+ diff (abs (vct-ref diffs i))))))))

(define (test-spectral-difference snd1 snd2 maxok)
  (let ((s1 (open-sound snd1))
	(s2 (open-sound snd2)))
    (if (or (not (sound? s1))
	    (not (sound? s2)))
	(snd-display ";open-sound ~A or ~A failed?" snd1 snd2))
    (let ((diff (spectral-difference s1 s2)))
      (close-sound s1)
      (close-sound s2)
      (if (> diff maxok)
	  (snd-display ";translate spectral difference ~A ~A: ~A > ~A?" snd1 snd2 diff maxok)))))

(if (or full-test (= snd-test 12) (and keep-going (<= snd-test 12)))
    (if sf-dir-files
	(let ((open-files '())
	      (open-ctr 0))
	  (run-hook before-test-hook 12)
	  (add-sound-file-extension "wave")

      (do ((clmtest 0 (1+ clmtest))) ((= clmtest tests)) 
	(log-mem clmtest)

	  (do ()
	      ((= open-ctr 32))
	    (let* ((len (length open-files))
		   (open-chance (* (- 8 len) .125))
		   (close-chance (* len .125)))
	      (if (or (= len 0) (> (random 1.0) .5))
		  (let* ((choice (inexact->exact (floor (my-random sf-dir-len))))
			 (name (string-append sf-dir (list-ref sf-dir-files choice)))
			 (ht (catch #t (lambda () (mus-sound-header-type name)) (lambda args 0)))
			 (df (catch #t (lambda () (mus-sound-data-format name)) (lambda args 0)))
			 (fd (if (or (= ht mus-raw)
				     (= ht mus-unsupported)
				     (= df mus-unknown))
				 -1 
				 (or (catch #t
					    (lambda () (view-sound name))
					    (lambda args
					      (snd-display ";~A ~A ~A" name ht df)
					      -1))
				     -1))))
		    (if (not (= fd -1))
			(begin
			  (set! open-ctr (+ open-ctr 1))
			  (set! open-files (cons fd open-files)))))
		  (if (and (> len 0) (> (random 1.0) 0.3))
		      (let* ((choice (inexact->exact (floor (my-random (exact->inexact (length open-files))))))
			     (fd (list-ref open-files choice)))
			(close-sound fd)
			(set! open-files (remove-if (lambda (a) (= a fd)) open-files)))))))
	  (if open-files (for-each close-sound open-files))
          (set! open-files '())

	  (if (not (= (length (sounds)) 0)) (snd-display ";active-sounds: ~A ~A?" (sounds) (map short-file-name (sounds))))
	  (let ((fd (open-raw-sound :file (string-append sf-dir "addf8.nh") :channels 1 :srate 8012 :data-format mus-mulaw)))
	    (if (not (= (data-format fd) mus-mulaw)) (snd-display ";open-raw-sound: ~A?" (mus-data-format-name (data-format fd))))
	    (close-sound fd))
	  
	  (reset-hook! bad-header-hook)
	  (time (test-spectral-difference "oboe.snd" (string-append sf-dir "oboe.g723_24") 20.0))
	  (test-spectral-difference "oboe.snd" (string-append sf-dir "oboe.g723_40") 3.0)
	  (test-spectral-difference "oboe.snd" (string-append sf-dir "oboe.g721") 6.0)
	  (test-spectral-difference (string-append sf-dir "o2.wave") (string-append sf-dir "o2_dvi.wave") 10.0)
	  (test-spectral-difference (string-append sf-dir "wood.riff") (string-append sf-dir "wood.sds") 4.0)
	  (test-spectral-difference (string-append sf-dir "nist-10.wav") (string-append sf-dir "nist-shortpack.wav") 1.0)
	  (add-hook! bad-header-hook (lambda (n) #t))
	  
	  ;;; dangling readers (overall)
	  (let ((ind (open-sound "oboe.snd")))
	    (let ((hi (make-sample-reader 0 ind 0)))
	      (close-sound ind)
	      (if (not (sample-reader? hi)) (snd-display ";dangling reader? ~A" hi))
	      (let ((name (format #f "~A" hi)))
		(if (not (string? name)) (snd-display ";dangling reader format: ~A" name)))
	      (let* ((val (hi))
		     (val1 (next-sample hi))
		     (val2 (previous-sample hi))
		     (val3 (read-sample hi)))
		(if (or (fneq val 0.0) (fneq val1 0.0) (fneq val2 0.0) (fneq val3 0.0))
		    (snd-display ";dangling read: ~A ~A ~A ~A" val val1 val2 val3))
		(if (sample-reader-home hi) (snd-display ";dangling reader home: ~A" (sample-reader-home hi)))
		(if (not (= (sample-reader-position hi) 0)) (snd-display ";dangling sample-reader-position: ~A" (sample-reader-position hi)))
		(if (not (sample-reader-at-end? hi)) (snd-display ";dangling reader eof: ~A" (sample-reader-at-end? hi)))
		(free-sample-reader hi))))
	  ;;; same (pruned edit)
	  (let ((ind (open-sound "oboe.