/* SNAC
 * Copyright (C) 1999 the Free Software Foundation
 * Authors : Matias Mutchinick, Jan Struyf          
 *         
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "ag_defs.h"

#include "errorstring.h"
#include "evalstring.h"
#include "formatstring.h"


extern AgError errState;
extern AgModePref mode_prefs;



void 
str_result_of_expr(gchar      *exprStr, 
		   gchar      *resultStr)
{
	gdouble  res;
	gint     angle = 0, size = 0, format = 0;
	
	
	/*
	 * Get the angle messure code
	 */
	if(!strcmp(mode_prefs.angle,"Rad "))
	{
		angle = RAD;
	}
	else if(!strcmp(mode_prefs.angle,"Grad"))
	{
		angle = GRAD;
	}	
	else if(!strcmp(mode_prefs.angle,"Deg "))
	{
		angle = DEG;
	}
	

	/*
	 * Get the format code
	 */
	if(!strcmp(mode_prefs.format,"Std"))
	{
		format = STD;
	}	
	else if(!strcmp(mode_prefs.format,"Fix"))
	{
		format = FIX;
	}	
	else if(!strcmp(mode_prefs.format,"Eng"))
	{
		format = ENG;
	}	
	else if(!strcmp(mode_prefs.format,"Sci"))
	{
		format = SCI;
	}
	else if(!strcmp(mode_prefs.format,"Hex"))
	{
		format = HEX;
	}
	else if(!strcmp(mode_prefs.format,"Bin"))
	{
		format = BIN;
	}
	else if(!strcmp(mode_prefs.format,"Oct"))
	{
		format = OCT;
	}
	
	
	/*
	 * Get the size code
	 */
	size = atoi(mode_prefs.size);
	
	
	/*
	 * Clear error
	 */
	errState.state = 0;
	
	res = eval_string(exprStr,format,size,angle);
	
	if(errState.state)
	{
		str_error(resultStr);
		return;
	}
	
	switch(format){
	case FIX: /* case Fix */
	{
		str_fix_format(res,size,resultStr);
		break;    
	}
	case ENG: /* Case Eng */
	{
		str_eng_format(res,size,resultStr);
		break;
	}
	case SCI: /* Case S.. */
	{
		str_sci_format(res,size,resultStr);
		break;
	}
	case STD: /* Case Std */
	{
		gcvt(res,size,resultStr);
		break;
	}
	case HEX: /* Case Hex */
	{
		str_base_format(res,size,resultStr,'h',16);
		break;
	}
	case BIN: /* Case Bin */
	{
		str_base_format(res,size,resultStr,'b',2);
		break;
	}
	case OCT: /* Case Oct */
	{
		str_base_format(res,size,resultStr,'o',8);
		break;
	}	
	} /* siwtch */
	
	return;
}



/************************************************************************/
/* Funcion: str_fix_format                                              */
/* Entrada: cadena, double, numero de digitos                           */
/* Salida : convierte un double a una cadena formato FIX                */
/************************************************************************/ 

void 
str_fix_format(gdouble       res, 
	       gint          formatSize,
	       gchar        *resultStr)
{
  gint    dec, sign, expSize = 0, expSign = 0;
  gchar  *tmpStr;
  



  tmpStr = ecvt(res,MAX_FORMAT_SIZE,&dec,&sign);

  
  if( dec > MAX_FORMAT_SIZE ) { 
    expSize = dec - 1;
    dec = 1;
  } 
  else 
    if ( dec <= -formatSize ) {
      expSize = 1 - dec;
      dec = 1;
      expSign = 1;
    }
  
  if( dec + formatSize < MAX_FORMAT_SIZE )
    *(tmpStr + dec + formatSize) = '\0';
  
  if(sign) 
    *resultStr++ = '-';
  
  if(dec <= 0 ) {
    *resultStr++ = '0'; 
    *resultStr++ = '.';
    while(dec < 0) {
      *resultStr++ = '0'; 
      dec++;
    }
    while( *tmpStr != '\0' )
      *resultStr++ = *tmpStr++;
  }
  else {
    while( *tmpStr != '\0' ) {
      if( dec == 0 ) 
	*resultStr++ = '.';
      *resultStr++ = *tmpStr++;
      dec--;
    }
  }
  
  if(expSize != 0) { 
    *resultStr++ = 'e';
    if(expSign)
      *resultStr++ = '-';
    else
      *resultStr++ = '+';
    gcvt(expSize,4,resultStr);
  } 
  else 
    *resultStr = '\0'; 
  
  return;
}


/************************************************************************/
/* Funcion: str_sci_format                                              */
/* Entrada: cadena, double, numero de digitos                           */
/* Salida : convierte un double a una cadena formato SCI                */
/************************************************************************/ 

void 
str_sci_format(gdouble       res,
	       gint          formatSize,
	       gchar        *resultStr)
{
  gint    dec,sign, expSize = 0, expSign = 0;
  gchar  *tmpStr;
  
  if (formatSize >= MAX_FORMAT_SIZE)
    tmpStr = ecvt(res,MAX_FORMAT_SIZE,&dec,&sign);
  else
    tmpStr = ecvt(res,formatSize+1,&dec,&sign);
  
  if(dec > 0 ) 
    expSize = dec - 1;
  else {
    expSize = 1 - dec;
    expSign = 1;
  }
  dec = 1;  
  
  if(sign) 
    *resultStr++ = '-';
  
  *resultStr++ = *tmpStr++ ;
  *resultStr++ = '.';
  while( *tmpStr != '\0' )
    *resultStr++ = *tmpStr++ ;
  
  *resultStr++ = 'e';
  if(expSign)
    *resultStr++ = '-';
  else
    *resultStr++ = '+';
  gcvt(expSize,4,resultStr);
  
  return;
}


/************************************************************************/
/* Funcion: str_eng_format                                              */
/* Entrada: cadena, double, numero de digitos                           */
/* Salida : convierte un double a una cadena formato ENG                */
/************************************************************************/ 

void 
str_eng_format(gdouble       res, 
	       gint          formatSize, 
	       gchar        *resultStr)
{
  gint    dec,sign, expSize = 0, expSign = 0;
  gchar  *tmpStr;
  
  if (formatSize >= MAX_FORMAT_SIZE)
    tmpStr = ecvt(res,MAX_FORMAT_SIZE,&dec,&sign);
  else
    tmpStr = ecvt(res,formatSize + 1,&dec,&sign);
  
  expSize = dec - 1;
  dec = 1;
  while( expSize % 3 != 0 ) {
    expSize--;
    dec++;
  }
  if(expSize < 0) {
    expSize = -expSize;
    expSign = 1; 
  }
  
  if(formatSize + 1 < dec)
    *(tmpStr + ++formatSize) = '0';
  
  if(sign) 
    *resultStr++ = '-';
  
  while( *tmpStr != '\0' ) {
    if( dec == 0 ) 
      *resultStr++ = '.';
    *resultStr++ = *tmpStr++;
    dec--;
  }
  
  *resultStr++ = 'e';
  if(expSign)
    *resultStr++ = '-';
  else
    *resultStr++ = '+';
  
  gcvt(expSize,4,resultStr);  
  
  return;
}

/************************************************************************/
/* Function:     str_base_format                                        */
/* Description : convert a [0,15] int to a character 0123456789ABCDEF   */
/* Author :      Jan Struyf                                             */
/************************************************************************/ 
gchar
get_base_char(gint value)
{
	if (value < 10) return (gchar)('0' + value);
	else return (gchar)('A' + value - 10);
}

/************************************************************************/
/* Function:     str_base_format                                        */
/* Description : convert a double to a base-i string                    */
/* Examples :    1Ah, 1001b, -1.Ah*16^26,...                            */
/* Author :      Jan Struyf                                             */
/************************************************************************/ 
void 
str_base_format(gdouble res,
                gint formatSize,
		gchar *resultStr, 
		gchar baseid, 
		gint base) 
{
	int numpos = 0, idx = 0, pos = 0, sign = 0, scale;
	char expstr[30];
	gchar oldchar;
	gdouble mybase, mod, power;
	/* Init vars */
	mybase = (gdouble) base;
	/* Test sign */
	if (res < 0.0) {
		sign = 1;
		res = fabs(res);
	}	
	/* Scale? */
	scale = formatSize*2/3;
	if (scale < 4) scale = 4;
	if ((res >= 4.0e15 || res <= pow(mybase, -scale)) && res != 0.0) {
		power = floor(log(res) / log(mybase));
		res *= pow(mybase, -power);
		scale = (int)power;			
	} else {
		scale = 0;
	}
	/* Build the string */
	if (res >= 1.0) {		
		while (pos < MAX_FORMAT_SIZE && res >= 1.0) {
			mod = floor(fmod(res, mybase));
			resultStr[pos++] = get_base_char((gint)mod);
			res = (res-mod) / mybase;
			numpos++;
			if ((numpos % 4) == 0 && pos < MAX_FORMAT_SIZE && res >= 1.0)
				resultStr[pos++] = (gchar)39;
		} 
	}
	/* Include zero, sign? */
	if (pos == 0 && pos < MAX_FORMAT_SIZE) resultStr[pos++] = '0';		
	if (sign == 1 && pos < MAX_FORMAT_SIZE) resultStr[pos++] = '-';	
	/* Revert the string */
	if (pos > 0) {
		for (idx = 0; idx < pos/2; idx++) {
			oldchar = resultStr[idx];
			resultStr[idx] = resultStr[pos-idx-1];
			resultStr[pos-idx-1] = oldchar;
		}
	}
	/* Build fractional part */
	if (formatSize != 0 && pos < MAX_FORMAT_SIZE && res != 0.0) {
		resultStr[pos++] = '.';	
		while (pos < MAX_FORMAT_SIZE && formatSize > 0) {	
			mod = floor(res*mybase);
			res = res*mybase - mod;
			resultStr[pos++] = get_base_char((gint)mod);			
			formatSize--;						
		}	
	}	
	/* Add base id */
	if (pos < MAX_FORMAT_SIZE)
		resultStr[pos++] = baseid;
	/* Add scale factor */
	if (scale != 0 && pos < MAX_FORMAT_SIZE) {
		sprintf(expstr,"*%d^%d",base,scale);
		idx = strlen(expstr);
		if (idx > MAX_FORMAT_SIZE-pos) idx = MAX_FORMAT_SIZE-pos;
		strncpy(resultStr+pos, expstr, idx);
		pos += idx;	
	}
	/* Terminate string */	
	resultStr[pos] = 0;
}


