/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* $Id: sg_log.c,v 1.1.1.1 2005/11/04 07:19:35 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <limits.h>

#include "sg_log.h"

typedef enum {
  SG_LOG_HEAD = 1,
  SG_LOG_TAIL = 2
} sg_log_control_t;

unsigned int sg_log_pri = LOG_MAIL | LOG_INFO;
char logmsg[USHRT_MAX];
char *plogmsg = logmsg;

static FILE *sg_logfp;
static int sg_info_level;
static int sg_debug_level;
static char *(*sg_strerror_func)(int) = strerror;

int syslog_output = 0;

void sg_set_logfp(FILE *fp)
{
  sg_logfp = fp;
}

void sg_set_info_level(int level)
{
  sg_info_level = level;
}

void sg_set_debug_level(int level)
{
  sg_debug_level = level;
}

void sg_set_strerror(char *(*func)(int))
{
  sg_strerror_func = func;
}

char *sg_strerror(int error)
{
  return sg_strerror_func(error);
}

static void sg_log_init()
{
  static int sg_log_init_done;
  if (!sg_log_init_done) {
    if (sg_logfp == NULL) {
      sg_set_logfp(stderr);
      sg_log_init_done = 1;
    }
  }
}

static void sg_output_func()
{
  plogmsg = &logmsg[0];
  if (syslog_output) {
	  (void)syslog(sg_log_pri, "%s", logmsg);
  }
  else {
	  (void)fprintf(sg_logfp, "%s",logmsg);
    (void)fprintf(sg_logfp, "\n");
    (void)fflush(sg_logfp);
  }
}

static void sg_log(sg_log_control_t x, char *pfx, int error,
		   char *fmt, va_list ap)
{
  sg_log_init();
  if (sg_logfp == NULL) return;
  if (x & SG_LOG_HEAD) {
    (void)sprintf(plogmsg, "%s: ", pfx);
    while(plogmsg[0]) plogmsg++;
  }
  if (fmt != NULL) {
    (void)vsprintf(plogmsg, fmt, ap);
    while(plogmsg[0]) plogmsg++;
    if (error) {
      (void)sprintf(plogmsg, ": ");
      while(plogmsg[0]) plogmsg++;
    }
  }
  if (error) {
    (void)sprintf(plogmsg, "%s", sg_strerror_func(error));
    while(plogmsg[0]) plogmsg++;
  }
  if (x & SG_LOG_TAIL) {
    (void)sg_output_func();
  }
}

void sg_err(int error, char *fmt, ...)
{
  va_list ap;
  va_start(ap, fmt);
  sg_log(SG_LOG_HEAD|SG_LOG_TAIL, "error", error, fmt, ap);
  va_end(ap);
}

void sg_info(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_info_level) return;
  va_start(ap, fmt);
  sg_log(SG_LOG_HEAD|SG_LOG_TAIL, "info", 0, fmt, ap);
  va_end(ap);
}

void sg_info_head(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_info_level) return;
  va_start(ap, fmt);
  sg_log(SG_LOG_HEAD, "info", 0, fmt, ap);
  va_end(ap);
}

void sg_info_body(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_info_level) return;
  va_start(ap, fmt);
  sg_log(0, "info", 0, fmt, ap);
  va_end(ap);
}

void sg_info_tail(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_info_level) return;
  va_start(ap, fmt);
  sg_log(SG_LOG_TAIL, "info", 0, fmt, ap);
  va_end(ap);
}

void sg_debug(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_debug_level) return;
  va_start(ap, fmt);
  sg_log(SG_LOG_HEAD|SG_LOG_TAIL, "debug", 0, fmt, ap);
  va_end(ap);
}

void sg_debug_head(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_debug_level) return;
  va_start(ap, fmt);
  sg_log(SG_LOG_HEAD, "debug", 0, fmt, ap);
  va_end(ap);
}
void sg_debug_body(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_debug_level) return;
  va_start(ap, fmt);
  sg_log(0, "debug", 0, fmt, ap);
  va_end(ap);
}
void sg_debug_tail(int level, char *fmt, ...)
{
  va_list ap;
  if (level > sg_debug_level) return;
  va_start(ap, fmt);
  sg_log(SG_LOG_TAIL, "debug", 0, fmt, ap);
  va_end(ap);
}
