/*
 * AudioWav.cc -- Audio IO via Wav
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "AudioWav.h"

/** Constructor for the WavData class - note that this is the base class for
	both readers and writers of wav files.
*/
	
WavData::WavData( ) :
	BufferReader( 20480 ),
	BufferWriter( 20480 )
{
	memset( &header, 0, sizeof( WAVHeader ) );
}

/** Used when using WavData to construct a new wav file - this method defines
	the paramters of the wav.
*/

void WavData::SetInfo( int16_t channels, int rate, int bytespersample ) 
{
	memcpy( header.riff, "RIFF", 4 );
	header.riff_length = sizeof(header) - sizeof(header.riff) - sizeof(header.riff_length);
	header.data_length = 0;
	memcpy( header.type, "WAVE", 4 );
	memcpy( header.format, "fmt ", 4 );
	header.format_length = 0x10;
	header.filler = 0x01;
	header.channels = channels;
	header.rate = rate;
	header.bytespersecond = rate * channels * bytespersample;
	header.bytespersample = bytespersample * channels;
	header.bitspersample = bytespersample * 8;
	memcpy( header.data, "data", 4 );
}

/** Returns the number of channels.
*/

int WavData::GetChannels( ) 
{
	return header.channels;
}

/** Returns the frequency.
*/

int WavData::GetFrequency( ) 
{
	return header.rate;
}

/** Returns the number of bytes per sample.
*/

int WavData::GetBytesPerSample( ) 
{
	return header.bytespersample / GetChannels( );
}

/** Returns the length of the wav.
*/

int WavData::GetLength( ) 
{
	return header.data_length;
}

/** Writes a header.
*/

int WavData::WriteHeader( ) 
{
	int bytes;
	
	bytes = PutBuffer( (uint8_t*)header.riff, 4 );
	bytes += PutBuffer( header.riff_length );
	bytes += PutBuffer( (uint8_t*)header.type, 4 );

	bytes += PutBuffer( (uint8_t*)header.format, 4 );
	bytes += PutBuffer( header.format_length );
	bytes += PutBuffer( header.filler );
	bytes += PutBuffer( header.channels );
	bytes += PutBuffer( header.rate );
	bytes += PutBuffer( header.bytespersecond );
	bytes += PutBuffer( header.bytespersample );
	bytes += PutBuffer( header.bitspersample );
	
	bytes += PutBuffer( (uint8_t*)header.data, 4 );
	bytes += PutBuffer( header.data_length );	

	FlushBuffer( );
	
	return bytes;
}

/** Reads a header.
*/

int WavData::ReadHeader( )
{
	int bytes;
	
	bytes = GetBuffer( (uint8_t*)header.riff, 4 );
	bytes += GetBuffer( header.riff_length );
	bytes += GetBuffer( (uint8_t*)header.type, 4 );
	
	bytes += GetBuffer( (uint8_t*)header.format, 4 );
	bytes += GetBuffer( header.format_length );
	bytes += GetBuffer( header.filler );
	bytes += GetBuffer( header.channels );
	bytes += GetBuffer( header.rate );
	bytes += GetBuffer( header.bytespersecond );
	bytes += GetBuffer( header.bytespersample );
	bytes += GetBuffer( header.bitspersample );
		
	bytes += GetBuffer( (uint8_t*)header.data, 4 );
	bytes += GetBuffer( header.data_length );	

	return bytes;
}

/** Get a number of samples - data is 2 dimensional array of (at least) the
	number of channels with each element being as long as length.
*/

bool WavData::Get( int16_t **data, int length )
{
	bool ret = GetBuffer( temp, length * GetChannels( ) ) == length * 4;
	
	for ( int i = 0; i < length; i ++ )
		for ( int c = 0; c < GetChannels( ); c ++ )
			data[ c ][ i ] = temp[ i * 2 + c ];

	return ret;
}

/** Add samples to the wav.
*/

bool WavData::Set( int16_t *data, int length )
{
	header.data_length += length;
	header.riff_length += length;
	return PutBuffer( data, length / 2 ) == length;
}

/** Determines if the current header describes a valid wav or not.
*/

bool WavData::IsWav( )
{
	//fprintf( stderr, "%-4.4s %-4.4s %-4.4s\n", header.riff, header.type, header.data );
	return !strncmp( header.riff, "RIFF", 4 ) && 
		   !strncmp( header.type, "WAVE", 4 );
}

//
// WAV Importer
//

/** This class provides a means to read a wav file.
*/

bool WavImporter::Open( string input )
{
	// Open the file as read only
	fd = open( input.c_str( ), O_RDONLY | O_NONBLOCK );
	
	// Read the header
	ReadHeader( );
	
	// Check the header info
	return IsWav( );
}

int WavImporter::Read( uint8_t *data, int size )
{
	if ( fd != -1 )
	{
		int bytes = read( fd, data, size );
		while ( bytes > 0 && bytes != size )
		{
			int obtained = read( fd, data + bytes, size - bytes );
			if ( obtained > 0 )
				bytes += obtained;
			else
				break;
		}
		return bytes;
	}
	else
		return 0;
}

bool WavImporter::Seek( long offset )
{
	return lseek( fd, offset, SEEK_SET ) == 0;
}

bool WavImporter::Close( )
{
	close( fd );
	return true;
}

bool WavThreadedReader::Open( string input )
{
	// Open the file as read only
	fd = open( input.c_str( ), O_RDONLY | O_NONBLOCK );
	
	// Read the header
	ReadHeader( );

	// Start the reader thread
	if ( IsWav( ) )
		ThreadStart( );
	
	// Check the header info
	return IsWav( );
}

int WavThreadedReader::Read( uint8_t *data, int size )
{
	if ( fd != -1 )
	{
		int bytes = read( fd, data, size );
		while ( bytes > 0 && bytes != size )
		{
			int obtained = read( fd, data + bytes, size - bytes );
			if ( obtained > 0 )
				bytes += obtained;
			else
				break;
		}
		return bytes;
	}
	else
		return 0;
}

bool WavThreadedReader::Seek( long offset )
{
	return false;
}

bool WavThreadedReader::Close( )
{
	ThreadStop( );
	close( fd );
	return true;
}

void WavThreadedReader::Thread( )
{
	while( ThreadIsRunning( ) )
		GetBuffer( (uint8_t *)NULL, 0 );
}

//
// WAV Exporter
//

/** This class provides the means to create a wav file.
*/

WavExporter::WavExporter( string file )
{
	if ( file != "-" )
		fd = open( file.c_str( ), O_RDWR | O_CREAT | O_TRUNC, 0644 );
	else
		fd = fileno( stdout );
}

int WavExporter::Write( uint8_t *data, int size )
{
	return write( fd, data, size );
}

bool WavExporter::Initialise( Frame &frame )
{
	if ( fd != -1 )
	{
		AudioInfo info;
		frame.GetAudioInfo( info );
		SetInfo( ( int16_t )frame.decoder->audio->num_channels, info.frequency, 2 );
		resampler = new FastAudioResample( info.frequency );
		return WriteHeader( ) != 0;
	}
	else
	{
		return false;
	}
}

bool WavExporter::Output( Frame &frame )
{
	resampler->Resample( frame );
	return Set( resampler->output, resampler->size );
}

bool WavExporter::Flush( )
{
	FlushBuffer( );
	if ( lseek( fd, 0, SEEK_SET ) == 0 )
		WriteHeader( );
	if ( fd != fileno( stdout ) )
		close( fd );
	return true;
}

//
// MP2 Exporter
//

/** This class provides a means to create an mp2 file.
*/

Mp2Exporter::Mp2Exporter( string file ) :
	filename( file )
{
}

int Mp2Exporter::Write( uint8_t *data, int size )
{
	return write( fd, data, size );
}

bool Mp2Exporter::Initialise( Frame &frame )
{
	AudioInfo info;
	frame.GetAudioInfo( info );
	char command[ 2048 ];
	sprintf( command, "mp2enc -v 0 -r %d -o \"%s\"", info.frequency, filename.c_str( ) );
	file = popen( command, "w" );
	if ( file != NULL )
	{
		fd = fileno( file );
		SetInfo( ( int16_t )frame.decoder->audio->num_channels, info.frequency, 2 );
		resampler = new FastAudioResample( info.frequency );
		return WriteHeader( ) != 0;
	}
	else
	{
		return false;
	}
}

bool Mp2Exporter::Output( Frame &frame )
{
	resampler->Resample( frame );
	return Set( resampler->output, resampler->size );
}

bool Mp2Exporter::Flush( )
{
	FlushBuffer( );
	pclose( file );
	return true;
}
