/***************************************************************************
    smb4kpreviewdialog.cpp  -  description
                             -------------------
    begin                : Fre Jul 4 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qtooltip.h>
#include <qiconset.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <ktoolbarbutton.h>

// application specific includes
#include "smb4kpreviewdialog.h"
#include "../smb4k.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kglobal.h"
using namespace Smb4KGlobal;




Smb4KPreviewDialog::Smb4KPreviewDialog( Smb4KShareItem *item, QWidget *parent, const char *name )
: KDialogBase( Plain, i18n( "Preview" ), Close, Close, parent, name, false, true ), m_item( item ),
m_ip( QString::null ), m_share( QString::null )  /* will be determined in an instant */
{
  setWFlags( Qt::WDestructiveClose );

  if ( !m_item )
  {
    close();
  }

  if ( Smb4KHostItem *host = smb4k_core->scanner()->getHost( m_item->host(), m_item->workgroup() ) )
  {
    m_ip = host->ip();
  }

  m_current_item = m_history.end();
  m_move_through = false;

  setupView();

  readOptions();

  setMinimumSize( sizeHint() );

  connect( this,      SIGNAL( finished() ),
           this,      SLOT( slotFinished() ) );
  connect( m_view,    SIGNAL( executed( QIconViewItem * ) ),
           this,      SLOT( slotItemExecuted( QIconViewItem * ) ) );
  connect( m_toolbar, SIGNAL( clicked( int ) ),
           this,      SLOT( slotButtonClicked( int ) ) );
  connect( m_combo,   SIGNAL( activated( const QString & ) ),
           this,      SLOT( slotItemActivated( const QString & ) ) );

  connect( smb4k_core->scanner(), SIGNAL( previewResult( const QValueList<Smb4KPreviewItem *> & ) ),
           this,                  SLOT( slotReceivedData( const QValueList<Smb4KPreviewItem *> & ) ) );

  connect( kapp, SIGNAL( iconChanged( int ) ),
           this, SLOT( slotIconChanged( int ) ) );

  m_share = smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_item->name(), QString::null );

  m_address = QString( "//%1/%2/" ).arg( m_item->host() ).arg( m_share );
}


Smb4KPreviewDialog::~Smb4KPreviewDialog()
{
  // Do not delete m_item here!
}


void Smb4KPreviewDialog::setupView()
{
  QFrame *frame = plainPage();
  QGridLayout *layout = new QGridLayout( frame );

  m_view = new KIconView( frame, 0, 0 );
  m_view->setItemTextPos( KIconView::Right );
  m_view->setResizeMode( KIconView::Adjust );
  m_view->setArrangement( KIconView::TopToBottom );
  m_view->setSpacing( 1 );
  m_view->setGridX( 200 );
  m_view->setWordWrapIconText( false );
  m_view->setShowToolTips( true );
  m_view->setAutoArrange( true );
  m_view->setSorting( true, true );

  m_toolbar = new KToolBar( frame, 0, true, false );
  m_toolbar->insertButton( "reload", Reload, true, i18n( "Reload" ), 0 );
  m_toolbar->insertButton( "back", Back, false, i18n( "Back" ), 1 );
  m_toolbar->insertButton( "forward", Forward, false, i18n( "Forward" ), 2 );
  m_toolbar->insertButton( "up", Up, false, i18n( "Up" ), 3 );

  m_combo = new KComboBox( false, m_toolbar, 0 );
  m_combo->listBox()->setHScrollBarMode( QScrollView::Auto );
  m_combo->listBox()->setVScrollBarMode( QScrollView::Auto );
  m_toolbar->insertWidget( Combo, 10, m_combo, 4 );
  m_toolbar->setItemAutoSized( Combo, true );

  layout->addWidget( m_view, 0, 0, 0 );
  layout->addWidget( m_toolbar, 1, 0, 0 );

  // Resize the dialog. Will later be overwritten by readOptions():
  setInitialSize( QSize( 400, 300 ) );

  setTooltips();
}


void Smb4KPreviewDialog::saveOptions()
{
  config()->setGroup( "Dialogs" );
  config()->writeEntry( "Preview Dialog Geometry", size() );

  config()->sync();
}


void Smb4KPreviewDialog::readOptions()
{
  config()->setGroup( "Dialogs" );

  QSize dialog_size = config()->readSizeEntry( "Preview Dialog Geometry", 0L );

  if ( !dialog_size.isEmpty() )
  {
    setInitialSize( dialog_size );
  }
}


void Smb4KPreviewDialog::setTooltips()
{
  QToolTip::add( m_view, i18n( "Contents" ) );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KPreviewDialog::slotFinished()
{
  saveOptions();
}


void Smb4KPreviewDialog::slotReceivedData( const QValueList<Smb4KPreviewItem *> &list )
{
  config()->setGroup( "User Interface" );
  bool show_hidden = config()->readBoolEntry( "Preview Hidden", true );

  Smb4KPreviewItem *preview_item = (Smb4KPreviewItem *)list.first();

  if ( !list.isEmpty() &&
       QString::compare( preview_item->host(), m_item->host() ) == 0 &&
       QString::compare( preview_item->share(), m_item->name() ) == 0 &&
       QString::compare( preview_item->path(), m_path ) == 0 )
  {
    m_view->clear();

    QString item = QString( "%1%2" ).arg( m_address ).arg( m_path );

    if ( m_current_item == m_history.end() /* in case we just started the dialog */ ||
         m_current_item == m_history.at( m_history.count() - 1 ) /* default: last entry in the list */ )
    {
      m_history.append( item );

      // New last item:
      m_current_item = m_history.at( m_history.count() - 1 );
    }
    else
    {
      // Maybe this is a bit rude, but we'll do it anyway:
      // Insert the new item at the position where we are and
      // remove the rest behind it:

      if ( !m_move_through )
      {
        m_current_item = m_history.insert( m_current_item, item );
        m_current_item = m_history.erase( m_current_item++, m_history.end() );
        m_current_item--;

        // Now m_current_item points to the last item in the list.
      }
      else
      {
        // Do nothing.
      }
    }

    m_combo->clear();

    for ( QStringList::Iterator it = m_history.begin(); it != m_history.end(); it++ )
    {
      if ( !m_combo->listBox()->findItem( *it, Qt::CaseSensitive|Qt::ExactMatch ) )
      {
        m_combo->insertItem( *it, -1 );
      }
    }

    m_combo->setCurrentText( *m_current_item );

    for ( QValueList<Smb4KPreviewItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      // Do not show hidden items if the user does not want
      // to see them.
      if ( (*it)->isHidden() && !show_hidden )
      {
        continue;
      }

      if ( (*it)->isFile() )
      {
        KIconViewItem *view_item = new KIconViewItem( m_view, (*it)->name(), SmallIcon( "file" ) );
        view_item->setKey( QString( "[file]_%1" ).arg( (*it)->name() ) );

        continue;
      }
      else
      {
        if ( QString::compare( (*it)->name(), "." ) != 0 && QString::compare( (*it)->name(), ".." ) != 0 )
        {
          KIconViewItem *view_item = new KIconViewItem( m_view, (*it)->name(), SmallIcon( "folder" ) );
          view_item->setKey( QString( "[directory]_%1" ).arg( (*it)->name() ) );

          continue;
        }
      }
    }

    // Now activate or deactivate the buttons:

    // Activate the 'Up' button if the current address is
    // not equal to the base address.
    m_toolbar->setItemEnabled( Up, QString::compare( m_address, item ) != 0 );

    // Activate/Deactivate 'Back' and 'Forward' buttons.
    m_toolbar->setItemEnabled( Back, m_current_item != m_history.at( 0 ) );
    m_toolbar->setItemEnabled( Forward, m_current_item != m_history.at( m_history.count() - 1 ) );
  }
  else
  {
    if ( m_path.contains( "/" ) == 1 )
    {
      m_path = QString::null;
    }
    else
    {
      m_path = m_path.section( "/", 0, -2 ).stripWhiteSpace();
    }
  }
}


void Smb4KPreviewDialog::slotItemExecuted( QIconViewItem *item )
{
  if ( !item->key().startsWith( "[file]_" ) )
  {
    m_move_through = false;

    QString tmp = item->text();
    m_path.append( tmp ).append( "/" );

    smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_share, m_path );
  }
}


void Smb4KPreviewDialog::slotButtonClicked( int id )
{
  switch ( id )
  {
    case Reload:
    {
      smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_share, m_path );

      break;
    }
    case Up:
    {
      m_move_through = false;

      // Do nothing if the path is empty because
      // we are already in the root directory.
      if ( m_path.isEmpty() )
      {
        return;
      }

      if ( m_path.contains( "/" ) > 1 )
      {
        m_path = m_path.section( "/", 0, -3 ).append( "/" );
      }
      else if ( m_path.contains( "/", true ) == 1 )
      {
        m_path = QString::null;
      }

      smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_share, m_path );

      break;
    }
    case Back:
    {
      // Move one item back in the list:
      if ( m_current_item != m_history.at( 0 ) )
      {
        m_current_item--;
        m_move_through = true;
      }
      else
      {
        m_move_through = false;
        return;
      }

      // Get the path:
      if ( (*m_current_item).contains( "/", true ) == 3 )
      {
        m_path = QString::null;
      }
      else
      {
        m_path = (*m_current_item).section( "/", 4, -1 );

        if ( !m_path.isEmpty() )
        {
          m_path.append( "/" );
        }
        else
        {
          // Do nothing.
        }
      }

      smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_share, m_path );

      break;
    }
    case Forward:
    {
      // Move one item forward in the list:
      if ( m_current_item != m_history.at( m_history.count() - 1 ) )
      {
        m_current_item++;
        m_move_through = true;
      }
      else
      {
        m_move_through = false;
        return;
      }

      // Get the path:
      if ( (*m_current_item).contains( "/", true ) == 3 )
      {
        m_path = QString::null;
      }
      else
      {
        m_path = (*m_current_item).section( "/", 4, -1 );

        if ( !m_path.isEmpty() )
        {
          m_path.append( "/" );
        }
        else
        {
          // Do nothing.
        }
      }

      smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_share, m_path );

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KPreviewDialog::slotItemActivated( const QString &item )
{
  m_move_through = false;

  // First we have to strip the address:
  QString path = item.section( m_address, 1, 1 ).stripWhiteSpace();

  smb4k_core->scanner()->getPreview( m_item->workgroup(), m_item->host(), m_ip, m_share, path );

  m_path = path;
}


void Smb4KPreviewDialog::slotIconChanged( int )
{
  QIconViewItem *item = m_view->firstItem();
  QIconViewItem *next_item = NULL;

  while ( item )
  {
    next_item = item->nextItem();

    // Reload icons:
    if ( item->key().startsWith( "[directory]_" ) )
    {
      item->setPixmap( SmallIcon( "folder" ) );
    }
    else if ( item->key().startsWith( "[file]_" ) )
    {
      item->setPixmap( SmallIcon( "file" ) );
    }

    item = next_item;
  }

  m_view->updateContents();
}


#include "smb4kpreviewdialog.moc"

