/***************************************************************************
    smb4kmounter.cpp  -  The core class that mounts the shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qapplication.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qmap.h>
#include <qthread.h>
#include <qfile.h>
#include <qtextstream.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>

// system includes
#include <sys/statvfs.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>

#ifdef __FreeBSD__
#include <pwd.h>
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#include <qfileinfo.h>
#endif

// Application specific includes
#include "smb4kmounter.h"
#include "smb4kauthinfo.h"
#include "smb4ksambaoptionsinfo.h"
#include "smb4kerror.h"
#include "smb4kglobal.h"
using namespace Smb4KGlobal;


class Smb4KMounterPrivate
{
  public:
    Smb4KMounterPrivate()
    {
      timerTicks = 0;
      clearData();
    }
    ~Smb4KMounterPrivate()
    {
    }
    int timerTicks;
    class Thread : public QThread
    {
      public:
        Thread() : QThread(), m_mountpoint( QString::null ), m_broken( true ) {}
        ~Thread() {}

        void setMountpoint( const QString &mp ) { m_mountpoint = mp; }

        virtual void run()
        {
          if ( m_mountpoint.isEmpty() )
          {
            kdFatal() << "Smb4KMounterPrivate::Thread: No mountpoint specified" << endl;
          }

          struct statvfs fs;

          if ( statvfs( m_mountpoint.local8Bit(), &fs ) == -1 )
          {
            m_broken = true;
            m_total = -1;
            m_free = -1;
          }
          else
          {
            m_broken = false;

            double kB_block = (double)(fs.f_bsize / 1024);
            double total = (double)(fs.f_blocks*kB_block);
            double free = (double)(fs.f_bfree*kB_block);

            m_total = total;
            m_free = free;
          }

          m_mountpoint = QString::null;
        }

        bool isBroken() { return m_broken; }

        double totalDiskSpace() { return m_total; }

        double freeDiskSpace() { return m_free; }

      private:
        QString m_mountpoint;
        bool m_broken;
        double m_total;
        double m_free;
    };

    Thread thread;
    void clearData();
    const QString &workgroup() { return m_workgroup; };
    const QString &host() { return m_host; }
    const QString &share() { return m_share; }
    const QString &ip() { return m_ip; }
    const QString &path() { return m_path; }
    const QString &filesystem() { return m_filesystem; }
    const QString &cifsLogin() { return m_cifsLogin; }
    void setWorkgroup ( const QString &wg ) { m_workgroup = wg; }
    void setHost( const QString &h ) { m_host = h; }
    void setShare( const QString &s ) { m_share = s; }
    void setIP( const QString &i ) { m_ip = i; }
    void setPath( const QString &p ) { m_path = p; }
    void setFilesystem( const QString &f ) { m_filesystem = f; }
    void setCIFSLogin( const QString &l ) { m_cifsLogin = l; }

  private:
    QString m_workgroup;
    QString m_host;
    QString m_share;
    QString m_ip;
    QString m_path;
    QString m_filesystem;
    QString m_cifsLogin;
};

void Smb4KMounterPrivate::clearData()
{
  m_workgroup = QString::null;
  m_host = QString::null;
  m_share = QString::null;
  m_ip = QString::null;
  m_path = QString::null;
  m_filesystem = QString::null;
  m_cifsLogin = QString::null;
}

static Smb4KMounterPrivate p;




Smb4KMounter::Smb4KMounter( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "MounterProcess" );
  m_proc->setUseShell( true );

  m_working = false;

  // Do some initial actions:
  // - Import all mounted SMB shares,
  // - Mount all shares that have to be mounted.
  m_queue.setAutoDelete( true );

  m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
  m_queue.enqueue( new QString( QString( "%1:" ).arg( MountRecent ) ) );

  connect( m_proc,  SIGNAL( processExited( KProcess * ) ),
           this,    SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc,  SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,    SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc,  SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,    SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( timer(), SIGNAL( timeout() ),
           this,    SLOT( init() ) );
  connect( kapp,    SIGNAL( shutDown() ),
           this,    SLOT( slotShutdown() ) );
}


Smb4KMounter::~Smb4KMounter()
{
  abort();

  for ( QValueList<Smb4KShare *>::Iterator it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    delete *it;
  }

  m_mounted_shares.clear();
}


/***************************************************************************
   Aborts any action of the mounter.
***************************************************************************/


void Smb4KMounter::abort()
{
  config()->setGroup( "Super User Privileges" );
  bool run_suid = config()->readBoolEntry( "Run SUID", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  m_queue.clear();

  if ( run_suid && !suid_program.isEmpty() )
  {
    if ( m_proc->isRunning() )
    {
      KProcess p;
      p.setUseShell( true );
      p << QString( "%1 smb4k_kill -15 %2" ).arg( suid_program ).arg( m_proc->pid() );
      p.start( KProcess::DontCare, KProcess::NoCommunication );
    }
  }
  else
  {
    if ( m_proc->isRunning() )
    {
      m_proc->kill();
    }
  }
}


/***************************************************************************
   Mounts recently used shares.
***************************************************************************/


void Smb4KMounter::mountRecent()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  if ( config()->readBoolEntry( "Mount Recent", false ) )
  {
    if ( config()->hasGroup( "Recently Mounted Shares" ) )
    {
      // FIXME: This is for backward compatibility only.
      // Remove with version 0.9.0 or later.

      config()->setGroup( "Recently Mounted Shares" );

      QString temp;

      for ( int index = 0; ; index++ )
      {
        temp = config()->readEntry( QString( "%1" ).arg( index ), QString::null );

        if ( !temp.isEmpty() )
        {
          Smb4KShare *share = findShareByName( temp );

          if ( !share || share->isForeign() )
          {
#ifndef __FreeBSD__
            mountShare( QString::null, temp.section( "/", 2, 2 ), QString::null, temp.section( "/", 3, 3 ) );
#else
            mountShare( QString::null, temp.section( "/", 2, 2 ).section( "@", 1, 1 ), QString::null, temp.section( "/", 3, 3 ) );
#endif
          }
          else
          {
            continue;
          }
        }
        else
        {
          break;
        }
      }

      config()->deleteGroup( "Recently Mounted Shares" );

      m_working = false;
      emit state( MOUNTER_STOP );
    }
    else
    {
      const QValueList<Smb4KSambaOptionsInfo *> *list = &(optionsHandler()->customOptionsList());

      for ( QValueList<Smb4KSambaOptionsInfo *>::ConstIterator it = list->begin();
            it != list->end(); ++it )
      {
        if ( (*it)->remount() )
        {
          Smb4KShare *share = findShareByName( (*it)->itemName() );

          if ( !share || share->isForeign() )
          {
#ifndef __FreeBSD__
            mountShare( QString::null, (*it)->itemName().section( "/", 2, 2 ), QString::null, (*it)->itemName().section( "/", 3, 3 ) );
#else
            mountShare( QString::null, (*it)->itemName().section( "/", 2, 2 ).section( "@", 1, 1 ), QString::null, (*it)->itemName().section( "/", 3, 3 ) );
#endif
          }

          // If the share is to be remounted the next time,
          // slotShutdown() will tell the options handler.
          (*it)->setRemount( false );

          continue;
        }
        else
        {
          continue;
        }
      }

      m_working = false;
      emit state( MOUNTER_STOP );
    }
  }
  else
  {
    m_working = false;
    emit state( MOUNTER_STOP );
  }
}


/***************************************************************************
   Imports all shares, that are mounted externally.
***************************************************************************/

void Smb4KMounter::import()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  QValueList<Smb4KShare *> shares;

#ifndef __FreeBSD__

  if ( m_proc_mounts.name().isEmpty() )
  {
    m_proc_mounts.setName( "/proc/mounts" );
  }

  if ( !QFile::exists( m_proc_mounts.name() ) )
  {
    if ( !m_proc_error )
    {
      m_proc_error = true;
      Smb4KError::error( ERROR_FILE_NOT_FOUND, m_proc_mounts.name() );
    }
  }
  else
  {
    QStringList contents, list;

    // Read /proc/mounts:

    if ( m_proc_mounts.open( IO_ReadOnly ) )
    {
      QTextStream ts( &m_proc_mounts );
      ts.setEncoding( QTextStream::Locale );

      contents = QStringList::split( "\n", ts.read(), false );

      m_proc_mounts.close();
    }
    else
    {
      Smb4KError::error( ERROR_OPENING_FILE, m_proc_mounts.name() );

      return;
    }

    // Process the SMBFS and CIFS entries:

    list += contents.grep( " smbfs ", true );
    list += contents.grep( " cifs ", true );

    if ( !list.isEmpty() )
    {
      for ( QStringList::Iterator it = list.begin(); it != list.end(); it++ )
      {
        Smb4KShare *new_share = NULL;

        if ( (*it).contains( " smbfs ", false ) != 0 )
        {
          QString share_and_path = (*it).section( "smbfs", 0, 0 ).stripWhiteSpace();
          QString name = share_and_path.section( " ", 0, 0 ).stripWhiteSpace().replace( "\\040", "\040" );
          QString path = share_and_path.section( " ", 1, 1 ).stripWhiteSpace();

          if ( path.contains( "\\040" ) != 0 || path.contains( "\040" ) != 0 )
          {
            name.replace( "_", "\040" );
            path.replace( "\\040", "\040" );
          }

          int uid = (*it).section( "uid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace().toInt();
          int gid = (*it).section( "gid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace().toInt();

          new_share = new Smb4KShare( name, path, "smbfs", uid, gid );
        }
        else if ( (*it).contains( " cifs ", false ) != 0 )
        {
          QString share_and_path = (*it).section( "cifs", 0, 0 ).stripWhiteSpace();
          QString name = share_and_path.section( " ", 0, 0 ).stripWhiteSpace().replace( "\\040", "\040" );
          QString path = share_and_path.section( " ", 1, 1 ).stripWhiteSpace();

          if ( path.contains( "\\040" ) != 0 || path.contains( "\040" ) != 0 )
          {
            name.replace( "_", "\040" );
            path.replace( "\\040", "\040" );
          }

          QString login = (*it).section( "username=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();

          new_share = new Smb4KShare( name, path, "cifs", login );
        }
        else
        {
          continue;
        }

        if ( new_share )
        {
          // If the a representative of the new share is already in the list of
          // mounted shares, replace the new with the old one.

          Smb4KShare *existing_share = findShareByPath( new_share->path() );

          if ( existing_share )
          {
            delete new_share;
            new_share = new Smb4KShare( *existing_share );
          }

          // Check if the share is broken and/or foreign.

          if ( (existing_share && !existing_share->isBroken()) || !existing_share )
          {
            p.thread.setMountpoint( new_share->path() );
            p.thread.start();
            p.thread.wait( THREAD_WAITING_TIME );
            p.thread.terminate();
            p.thread.wait();

            new_share->setBroken( p.thread.isBroken() );
            new_share->setTotalDiskSpace( p.thread.totalDiskSpace() );
            new_share->setFreeDiskSpace( p.thread.freeDiskSpace() );
          }
          else
          {
            // Since new_share is a copy of existing_share, we do not need to do
            // anything here.
          }

          if ( !existing_share && QString::compare( new_share->filesystem(), "cifs" ) == 0 )
          {
            bool foreign = true;

            if ( (!new_share->isBroken() &&
                 (qstrncmp( new_share->canonicalPath(), QDir( mount_prefix ).canonicalPath(), QDir( mount_prefix ).canonicalPath().length() ) == 0 ||
                  qstrncmp( new_share->canonicalPath(), QDir::home().canonicalPath(), QDir::home().canonicalPath().length() ) == 0)) ||
                 (new_share->isBroken() &&
                 (qstrncmp( new_share->path(), QDir::homeDirPath(), QDir::homeDirPath().length() ) == 0 ||
                  qstrncmp( new_share->path(), mount_prefix, mount_prefix.length() ) == 0)) )
            {
              foreign = false;
            }

            new_share->setForeign( foreign );
          }

          shares.append( new_share );
        }
      }
    }
  }

#else

  struct statfs *buf;
  int count = getmntinfo( &buf, 0 );

  if ( count == 0 )
  {
    int err_code = errno;

    Smb4KError::error( ERROR_IMPORTING_SHARES, QString::null, strerror( err_code ) );

    m_working = false;
    return;
  }

  for ( int i = 0; i < count; ++i )
  {
    if ( !strcmp( buf[i].f_fstypename, "smbfs" ) )
    {
      QString share_name( buf[i].f_mntfromname );
      QString path( buf[i].f_mntonname );
      QString fs( buf[i].f_fstypename );

      QFileInfo info( QString( buf[i].f_mntonname )+"/." );

      int uid = (int)info.ownerId();
      int gid = (int)info.groupId();

      Smb4KShare *existing_share = findShareByPath( path );
      Smb4KShare *new_share = NULL;

      if ( existing_share )
      {
        new_share = new Smb4KShare( *existing_share );
      }
      else
      {
        new_share = new Smb4KShare( share_name, path, fs, uid, gid );
      }

      // Test if share is broken
      if ( (existing_share && !existing_share->isBroken()) || !existing_share )
      {
        p.thread.setMountpoint( new_share->path() );
        p.thread.start();
        p.thread.wait( THREAD_WAITING_TIME );
        p.thread.terminate();
        p.thread.wait();

        new_share->setBroken( p.thread.isBroken() );
        new_share->setTotalDiskSpace( p.thread.totalDiskSpace() );
        new_share->setFreeDiskSpace( p.thread.freeDiskSpace() );
      }
      else
      {
        // Since new_share is a copy of existing_share, we do not need to do
        // anything here.
      }

      shares.append( new_share );
    }
  }

  // Apparently, under FreeBSD we do not need to delete
  // the pointer (see manual page).

#endif

  // Delete all entries of m_mounted_shares.
  for ( QValueList<Smb4KShare *>::Iterator it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    delete *it;
  }

  m_mounted_shares.clear();

  m_mounted_shares = shares;

  emit updated();

  m_working = false;
}


/***************************************************************************
   Mounts a share. (Public part)
***************************************************************************/

void Smb4KMounter::mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  QString share_name = QString::null;

  if ( QString::compare( share, "homes" ) == 0 )
  {
    share_name = specifyUser( host, kapp->mainWidget() ? kapp->mainWidget() : 0, "SpecifyUser" );
  }
  else
  {
    share_name = share;
  }

  if ( !share_name.stripWhiteSpace().isEmpty() )
  {
    m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5" ).arg( Mount ).arg( workgroup ).arg( host ).arg( ip ).arg( share_name ) ) );
  }
}



/***************************************************************************
   Mounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  Smb4KSambaOptionsInfo *info = optionsHandler()->findItem( "//"+host+"/"+share );

  p.setWorkgroup( workgroup );
  p.setHost( host );
  p.setShare( share );
  p.setIP( ip );

  config()->setGroup( "Super User Privileges" );
  bool run_suid = config()->readBoolEntry( "Run SUID", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  config()->setGroup( "Samba" );
#ifndef __FreeBSD__
  QString default_fs = config()->readEntry( "Mount Filesystem", "smbfs" );
  p.setFilesystem( info && !info->filesystem().isEmpty() ? info->filesystem() : default_fs );
#else
  int default_port = config()->readNumEntry( "Port", 139 );
#endif

  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readPathEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  bool lowercase = config()->readBoolEntry( "Force Lowercase", false );

  QString share_dir = share.stripWhiteSpace();

  Smb4KShare *test_share;

  if ( (test_share = findShareByPath( QString( "%1%2/%3" ).arg( mount_prefix, p.host(), share_dir ) )) != NULL )
  {
    if ( !test_share->isForeign() )
    {
      emit alreadyMountedShare( test_share->canonicalPath() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }
  else
  {
    if ( (test_share = findShareByName( QString( "//%1/%2" ).arg( p.host(), p.share() ) )) != NULL )
    {
      if ( !test_share->isForeign() )
      {
        emit alreadyMountedShare( test_share->canonicalPath() );
        m_working = false;
        emit state( MOUNTER_STOP );
        return;
      }
    }
  }

  delete test_share;

  // Create the mount point.
  QDir *dir = new QDir( mount_prefix );

  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      Smb4KError::error( ERROR_MKDIR_FAILED, dir->path() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  QString host_dir = lowercase ? host.lower() : host;

  dir->setPath( dir->path()+"/"+host_dir );

  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      Smb4KError::error( ERROR_MKDIR_FAILED, dir->path() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  share_dir = lowercase ? share_dir.lower() : share_dir;

  dir->setPath( dir->path()+"/"+share_dir );

  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      Smb4KError::error( ERROR_MKDIR_FAILED, dir->path() );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  p.setPath( QDir::cleanDirPath( dir->path() ) );

  delete dir;

  // Mount the share.
  Smb4KAuthInfo *auth = passwordHandler()->readAuth( workgroup, host, share );

  QString command;

#ifndef __FreeBSD__

  if ( !run_suid || suid_program.isEmpty() )
  {
    command = QString( "smb4k_mount --no-suid -t %1 " ).arg( p.filesystem() );
  }
  else if ( run_suid && !suid_program.isEmpty() )
  {
    command = QString( "%1 smb4k_mount --suid -t %2 " ).arg( suid_program ).arg( p.filesystem() );
  }

  command.append( QString( "//%1/%2 %3 " ).arg( KProcess::quote( host ) ).arg( KProcess::quote( share ) ).arg( KProcess::quote( p.path() ) ) );

  command.append( "-o " );

  if ( !workgroup.isEmpty() )
  {
    if ( QString::compare( p.filesystem(), "smbfs" ) == 0 )
    {
      command.append( QString( "workgroup=%1," ).arg( KProcess::quote( workgroup ) ) );
    }
    else
    {
      command.append( QString( "domain=%1," ).arg( KProcess::quote( workgroup ) ) );
    }
  }

  if ( !ip.isEmpty() )
  {
    command.append( QString( "ip=%1," ).arg( KProcess::quote( ip ) ) );
  }

  command.append( optionsHandler()->mountOptions( QString( "//%1/%2" ).arg( p.host(), p.share() ) ) );

  // Setting the USER environment won't give you the right
  // user when using super/sudo.
  if ( !auth->user().isEmpty() )
  {
    command.append( QString( "username=%1" ).arg( KProcess::quote( auth->user() ) ) );
    p.setCIFSLogin( auth->user() );
  }
  else
  {
    command.append( "guest" );
    p.setCIFSLogin( "guest" );
  }

  m_proc->setEnvironment( "PASSWD", !auth->password().isEmpty() ? auth->password() : "" );

#else

  if ( !run_suid || suid_program.isEmpty() )
  {
    command = QString( "smb4k_mount --no-suid " );
  }
  else if ( run_suid && !suid_program.isEmpty() )
  {
    command = QString( "%1 smb4k_mount --suid " ).arg( suid_program );
  }

  command.append( optionsHandler()->mountOptions( QString( "//%1/%2" ).arg( p.host(), p.share() ) ) );

  if ( !p.ip().isEmpty() )
  {
    command.append( " -I "+KProcess::quote( p.ip() ) );
  }

  if ( !p.workgroup().isEmpty() )
  {
    command.append( " -W "+KProcess::quote( p.workgroup() ) );
  }

  command.append( " -N" );

  QString port = QString( "%1" ).arg( info && info->port() != -1 ? info->port() : default_port );

  if ( !auth->user().isEmpty() )
  {
    command.append( " //"+KProcess::quote( auth->user() )+"@"+KProcess::quote( host )+":"+port+"/"+KProcess::quote( share )+" "+KProcess::quote( p.path() ) );
  }
  else
  {
    command.append( " //guest@"+KProcess::quote( host )+":"+port+"/"+KProcess::quote( share )+" "+KProcess::quote( p.path() ) );
  }

#endif

  delete auth;

  // Start the mount process:
  *m_proc << command;

  startProcess( Mount );
}


/****************************************************************************
   Unmount a share. (Public part)
****************************************************************************/

void Smb4KMounter::unmountShare( Smb4KShare *share, bool force, bool noMessage )
{
  // Do *not* change share->canonicalPath(). It is necessary for the
  // checks below to work.
  QString *input = new QString( QString( "%1:%2:%3:%4" ).arg( Unmount ).arg( share->canonicalPath() ).arg( force, noMessage ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmount a share. (Private part)
***************************************************************************/

void Smb4KMounter::unmount( const QString &mountpoint, bool force, bool noMessage )
{
  config()->setGroup( "Super User Privileges" );
  bool run_suid = config()->readBoolEntry( "Run SUID", false );
  bool force_unmount = config()->readBoolEntry( "Force Unmount", false );
  QString suid_program = config()->readEntry( "SUID Program", QString::null );

  config()->setGroup( "Mount Options" );
  bool allow_unmount_foreign = config()->readBoolEntry( "Unmount Foreign", false );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  QString kernel_version = kernelVersion();

  // Do some checks if the user wants to force the unmounting of the share.
  // Abort, if one of the conditions is not met.
  if ( force )
  {
    // Abort, if the setting is not enabled.
    if ( !force_unmount )
    {
      Smb4KError::error( ERROR_FEATURE_NOT_ENABLED );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }

    // This will most likely never be the case, but we have to be prepared:
    // Abort if neither 'sudo' nor 'super' was found.
    if ( suid_program.isEmpty() )
    {
      Smb4KError::error( ERROR_PROGRAM_MISSING, "super/sudo" );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }

    // Check if the kernel supports the lazy unmount:
    int kern_maj = kernel_version.section( ".", 0, 0 ).toInt();
    int kern_min = kernel_version.section( ".", 1, 1 ).toInt();
    int kern_rel = kernel_version.section( ".", 2, 2 ).toInt();

    if ( kern_maj < 2 ||
        (kern_maj <= 2 && kern_min < 4) ||
        (kern_maj <= 2 && kern_min <= 4 && kern_rel < 11) )
    {
      Smb4KError::error( ERROR_WRONG_KERNEL, kernel_version );
      m_working = false;
      emit state( MOUNTER_STOP );
      return;
    }
  }

  // Compose the unmount command:
  if ( !mountpoint.isEmpty() )
  {
    QString command;
    bool execute = false;
    QString path = mountpoint;
    p.setPath( path.replace( '\044', "\044" ) );

    Smb4KShare *share = findShareByPath( mountpoint );

    if ( share && !share->isForeign() )
    {
      if ( force )
      {
        if ( KMessageBox::questionYesNo( 0, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
        {
          command = QString( "%1 smb4k_umount --suid --%2 -l " ).arg( suid_program ).arg( share->filesystem() );
          execute = true;
        }
        else
        {
          m_working = false;
          emit state( MOUNTER_STOP );
          return;
        }
      }
      else
      {
        if ( !run_suid || suid_program.isEmpty() )
        {
          command = QString( "smb4k_umount --no-suid --%1 " ).arg( share->filesystem() );
        }
        else if ( run_suid && !suid_program.isEmpty() )
        {
          command = QString( "%1 smb4k_umount --suid --%2 " ).arg( suid_program ).arg( share->filesystem() );
        }
      }
    }
    else if ( share && share->isForeign() )
    {
      if ( allow_unmount_foreign )
      {
        if ( force )
        {
          if ( KMessageBox::questionYesNo( 0, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
          {
            command = QString( "%1 smb4k_umount --suid --%2 -l " ).arg( suid_program ).arg( share->filesystem() );
            execute = true;
          }
          else
          {
            m_working = false;
            emit state( MOUNTER_STOP );
            return;
          }
        }
        else
        {
          if ( !run_suid || suid_program.isEmpty() )
          {
            command = QString( "smb4k_umount --no-suid --%1 " ).arg( share->filesystem() );
          }
          else if ( run_suid && !suid_program.isEmpty() )
          {
            command = QString( "%1 smb4k_umount --suid --%2 " ).arg( suid_program ).arg( share->filesystem() );
          }
        }
      }
      else
      {
        if ( !noMessage )
        {
          Smb4KError::error( ERROR_UNMOUNTING_NOT_ALLOWED );
        }

        m_working = false;
        emit state( MOUNTER_STOP );

        return;
      }
    }

    command.append( KProcess::quote( p.path() ) );

    if ( force && !execute )
    {
      return;
    }

    *m_proc << command;
    startProcess( Unmount );
  }
  else
  {
    Smb4KError::error( ERROR_MOUNTPOINT_EMPTY );
    m_working = false;
    emit state( MOUNTER_STOP );
  }
}


/***************************************************************************
   Unmounts all shares at once. (Public part)
***************************************************************************/

void Smb4KMounter::unmountAllShares()
{
  QString *input = new QString( QString( "%1" ).arg( UnmountAll ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts all shares at once.
***************************************************************************/

void Smb4KMounter::unmountAll()
{
  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    unmountShare( *it, false, true );
  }

  m_working = false;
}


/***************************************************************************
   Starts any process.
***************************************************************************/

void Smb4KMounter::startProcess( int state )
{
  m_buffer = QString::null;
  m_state = state;

  if ( m_state != Import )
  {
    QApplication::setOverrideCursor( waitCursor );
  }

  m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/***************************************************************************
   Ends any process. This functions tells the mounter what to do
   afterwards.
***************************************************************************/

void Smb4KMounter::endProcess()
{
  switch ( m_state )
  {
    case Mount:
      processMount();
      break;
    case Unmount:
      processUnmount();
      break;
    default:
      break;
  }

  m_state = Idle;

  p.clearData();

  QApplication::restoreOverrideCursor();
  m_proc->clearArguments();

  m_working = false;
  emit state( MOUNTER_STOP );
}


/***************************************************************************
   Process mounts.
***************************************************************************/

void Smb4KMounter::processMount()
{
  Smb4KShare *share = NULL;

#ifndef __FreeBSD__

  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "failed", true ) == 0 &&
         m_buffer.contains( "ERR", true ) == 0 &&
         m_buffer.contains( "/bin/sh:" ) == 0 &&
         m_buffer.contains( "mount:", true ) == 0 &&
         m_buffer.contains( "smbmnt" ) == 0 &&
         m_buffer.contains( p.path() ) == 0 &&
         m_buffer.contains( "mount error" ) == 0 )
    {
      QString name = QString( "//%1/%2" ).arg( p.host() ).arg( p.share() );

      bool broken = true;

      p.thread.setMountpoint( p.path() );
      p.thread.start();
      p.thread.wait( 1000 );
      p.thread.terminate();
      p.thread.wait();

      broken = p.thread.isBroken();

      if ( QString::compare( p.filesystem(), "smbfs" ) == 0 )
      {
        share =  new Smb4KShare( name, p.path(), p.filesystem(), (int)getuid(), (int)getgid(), broken );
        m_mounted_shares.append( share );
      }
      else if ( QString::compare( p.filesystem(), "cifs" ) == 0 )
      {
        share = new Smb4KShare( name, p.path(), p.filesystem(), p.cifsLogin(), false, broken );
        m_mounted_shares.append( share );
      }

      // If you change something here, please check the code in import()
      // as well.
      if ( share && !share->isBroken() )
      {
        struct statvfs fs;

        if ( statvfs( share->path(), &fs ) == -1 )
        {
          int err_code = errno;

          share->setTotalDiskSpace( -1 );
          share->setFreeDiskSpace( -1 );

          Smb4KError::error( ERROR_GETTING_USAGE, QString::null, strerror( err_code ) );
        }
        else
        {
          double kB_block = (double)(fs.f_bsize / 1024);
          double total = (double)(fs.f_blocks*kB_block);
          double free = (double)(fs.f_bfree*kB_block);

          share->setTotalDiskSpace( total );
          share->setFreeDiskSpace( free );
        }

        // You don't want to use ANY Qt function here, or Smb4K will
        // slow down dramatically.
      }
      else
      {
        share->setTotalDiskSpace( -1 );
        share->setFreeDiskSpace( -1 );
      }

      emit mountedShare( p.path() );
    }
    else
    {
      if ( m_buffer.contains( "ERRbadpw" ) != 0 ||
           m_buffer.contains( "ERRnoaccess" ) != 0 ||
           m_buffer.contains( "mount error 13 = Permission denied" ) != 0 )
      {
        int state = Smb4KPasswordHandler::None;

        if ( m_buffer.contains( "ERRbadpw" ) != 0 )
        {
          state = Smb4KPasswordHandler::BadPassword;
        }
        else if ( m_buffer.contains( "ERRnoaccess" ) != 0 )
        {
          state = Smb4KPasswordHandler::AccessDenied;
        }
        else if ( m_buffer.contains( "mount error 13 = Permission denied" ) != 0 )
        {
          state = Smb4KPasswordHandler::PermDenied;
        }

        // If the user supplied auth information, we will retry mounting.
        if ( passwordHandler()->askpass( p.workgroup(), p.host(), p.share(), state ) )
        {
          mountShare( p.workgroup(), p.host(), p.ip(), p.share() );
        }
      }
      else if ( m_buffer.contains( "ERRnosuchshare" ) != 0 && p.share().contains( "_" ) != 0 )
      {
        QString share_string = ((QString)p.share()).replace( "_", " " );
        mountShare( p.workgroup(), p.host(), p.ip(), share_string );
      }
      else
      {
        QString name = QString( "//%1/%2" ).arg( p.host() ).arg( p.share() );

        Smb4KError::error( ERROR_MOUNTING_SHARE, name, m_buffer );
      }
    }
  }

#else

  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "syserr =", true ) == 0 &&
         /* To make sure we catch all errors, also check for the following
            strings. Maybe we can remove them?? */
         m_buffer.contains( "Authentication error", true ) == 0 &&
         m_buffer.contains( "Connection refused", true ) == 0 &&
         m_buffer.contains( "Operation not permitted", true ) == 0 )
    {
      import();  // FIXME: *cough* What is this for???

      Smb4KAuthInfo *auth = passwordHandler()->readAuth( p.workgroup(), p.host(), p.share() );

      QString name = QString( "//%1@%2/%3" ).arg( auth->user().upper(), p.host().upper(), p.share().upper() );

      delete auth;

      bool broken = true;

      p.thread.setMountpoint( p.path() );
      p.thread.start();
      p.thread.wait( 1000 );
      p.thread.terminate();
      p.thread.wait();

      broken = p.thread.isBroken();

      share = new Smb4KShare( name, p.path(), p.filesystem(), (int)getuid(), (int)getgid(), broken );
      m_mounted_shares.append( share );

      // If you change something here, please check the code in import()
      // as well.
      if ( share && !share->isBroken() )
      {
        struct statvfs fs;

        if ( statvfs( share->path(), &fs ) == -1 )
        {
          int err_code = errno;

          share->setTotalDiskSpace( -1 );
          share->setFreeDiskSpace( -1 );

          Smb4KError::error( ERROR_GETTING_USAGE, QString::null, strerror( err_code ) );
        }
        else
        {
          double kB_block = (double)(fs.f_bsize / 1024);
          double total = (double)(fs.f_blocks*kB_block);
          double free = (double)(fs.f_bfree*kB_block);

          share->setTotalDiskSpace( total );
          share->setFreeDiskSpace( free );
        }

        // You don't want to use ANY Qt function here, or Smb4K will
        // slow down dramatically.
      }
      else
      {
        share->setTotalDiskSpace( -1 );
        share->setFreeDiskSpace( -1 );
      }

      emit mountedShare( p.path() );
    }
    else
    {
      if ( m_buffer.contains( "Authentication error" ) != 0 )
      {
        // If the user supplied auth information, we will retry mounting.
        if ( passwordHandler()->askpass( p.workgroup(), p.host(), p.share(), Smb4KPasswordHandler::AuthError ) )
        {
          mountShare( p.workgroup(), p.host(), p.ip() , p.share() );
        }
      }
      else
      {
        Smb4KAuthInfo *auth = passwordHandler()->readAuth( p.workgroup(), p.host(), p.share() );

        QString name = QString( "//%1@%2/%3" ).arg( auth->user().upper(), p.host().upper(), p.share().upper() );

        delete auth;

        Smb4KError::error( ERROR_MOUNTING_SHARE, name, m_buffer );
      }
    }
  }

#endif

  emit updated();
}


/***************************************************************************
   Process unmounts.
***************************************************************************/

void Smb4KMounter::processUnmount()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  // Get the share:
  Smb4KShare *share = findShareByPath( p.path() );

  if ( m_proc->normalExit() )
  {
    // FIXME: This is a potential problem. What if the unmount
    // process returns some debugging info like the mount process
    // under SuSE? Need more info.
    if ( m_buffer.isEmpty() )
    {
      if ( qstrncmp( share->canonicalPath(),
                     QDir( mount_prefix ).canonicalPath().local8Bit(),
                     QDir( mount_prefix ).canonicalPath().local8Bit().length() ) == 0 )
      {
        QDir dir( share->canonicalPath() );

        if ( dir.rmdir( dir.canonicalPath(), true ) )
        {
          dir.cdUp();
          dir.rmdir( dir.canonicalPath(), true );
        }
      }

      m_mounted_shares.remove(share);
    }
    else
    {
      Smb4KError::error( ERROR_UNMOUNTING_SHARE, share->name(), m_buffer );
    }
  }

  emit updated();
}


/***************************************************************************
   Check if a share is already mounted
***************************************************************************/

bool Smb4KMounter::isMounted( const QString &name )
{
  return findShareByName( name ) != NULL ? true : false;
}


/***************************************************************************
   Find a share in the list with its path
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByPath( const QString &path )
{
  if ( path.isEmpty() || m_mounted_shares.isEmpty() )
  {
    return NULL;
  }

  Smb4KShare *share = NULL;

  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( QString::compare( (*it)->path().upper(), path.upper() ) == 0 ||
        QString::compare( (*it)->canonicalPath().upper(), path.upper() ) == 0 )
    {
      share = *it;

      break;
    }
  }

  return share;
}


/***************************************************************************
   Find a share in the list with its name
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByName( const QString &name )
{
  if ( name.isEmpty() || m_mounted_shares.isEmpty() )
  {
    return NULL;
  }

  QString n = name;

  Smb4KShare *share = NULL;

  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( QString::compare( (*it)->name().upper(), name.upper() ) == 0 )
    {
      share = *it;

      break;
    }
    else if ( QString::compare( (*it)->name().upper(), n.replace( " ", "_" ).upper() ) == 0 )
    {
      share = *it;

      break;
    }
    else
    {
      continue;
    }
  }

  return share;
}


/***************************************************************************
   Returns a list of mount points that belong to broken shares
***************************************************************************/

const QValueList<Smb4KShare *> Smb4KMounter::getBrokenShares()
{
  QValueList<Smb4KShare *> broken_shares;

  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if ( (*it)->isBroken() )
    {
      broken_shares.append( *it );

      continue;
    }
    else
    {
      continue;
    }
  }

  return broken_shares;
}


void Smb4KMounter::prepareForShutdown()
{
  slotShutdown();
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KMounter::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KMounter::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::init()
{
  if ( !m_working && !m_queue.isEmpty() )
  {
    // Tell the mounter, that it is busy.
    m_working = true;

    QString *item = m_queue.dequeue();
    int todo = item->section( ":", 0, 0 ).toInt();

    switch ( todo )
    {
      case MountRecent:
        mountRecent();
        break;
      case Import:
        import();
        break;
      case Mount:
        emit state( MOUNTER_MOUNTING );
        mount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ) );
        break;
      case Unmount:
        emit state( MOUNTER_UNMOUNTING );
        unmount( item->section( ":", 1, 1 ), (bool)item->section( ":", 2, 2 ).toInt() /* force */, (bool)item->section( ":", 3, 3 ).toInt() /* noMessage */);
        break;
      case UnmountAll:
        unmountAll();
        break;
      default:
        break;
    }

    delete item;
  }

  p.timerTicks++;

  config()->setGroup( "Mount Options" );
  int millisec = config()->readNumEntry( "Check Interval", 2500 );

  if ( p.timerTicks * timerInterval() >= millisec /* msec */ && (!m_working || m_queue.isEmpty()) )
  {
    m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
    p.timerTicks = 0;
  }
}


void Smb4KMounter::slotShutdown()
{
  config()->setGroup( "Mount Options" );
  bool remount = config()->readBoolEntry( "Mount Recent", false );
  bool unmount_on_exit = config()->readBoolEntry( "Unmount All", false );
  QString mount_prefix = config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  // Abort any action:
  abort();

  // Prepare for shutdown:
  if ( remount && !m_mounted_shares.isEmpty() )
  {
    for ( QValueList<Smb4KShare *>::ConstIterator it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      optionsHandler()->remount( *it, true );
    }
  }

  optionsHandler()->sync();

  QDir dir;

  dir.cd( mount_prefix );

  QStringList dirs = dir.entryList( QDir::Dirs, QDir::DefaultSort );

  QValueList<Smb4KShare *> broken_shares = getBrokenShares();

  for ( QStringList::ConstIterator it = dirs.begin(); it != dirs.end(); ++it )
  {
    if ( QString::compare( *it, "." ) != 0 && QString::compare( *it, ".." ) != 0 )
    {
      bool broken = false;

      for ( QValueListIterator<Smb4KShare *> bs = broken_shares.begin(); bs != broken_shares.end(); ++bs )
      {
        if ( qstrncmp( (*bs)->path(), mount_prefix+*it, (mount_prefix+*it).length() ) == 0 ||
             qstrncmp( (*bs)->canonicalPath(), mount_prefix+*it, (mount_prefix+*it).length() ) == 0 )
        {
          broken = true;

          break;
        }
        else
        {
          continue;
        }
      }

      if ( !broken )
      {
        dir.cd( *it );

        QStringList subdirs = dir.entryList( QDir::Dirs, QDir::DefaultSort );

        for ( QStringList::ConstIterator i = subdirs.begin(); i != subdirs.end(); ++i )
        {
          if ( QString::compare( *i, "." ) != 0 && QString::compare( *i, ".." ) != 0 )
          {
            dir.rmdir( *i );
          }
        }

        dir.cdUp();
        dir.rmdir( *it );
      }
    }
  }

  broken_shares.clear();

  if ( unmount_on_exit )
  {
    config()->setGroup( "Super User Privileges" );
    bool run_suid = config()->readBoolEntry( "Run SUID", false );
    QString suid_program = config()->readEntry( "SUID Program", QString::null );

    KProcess p;
    p.setUseShell( true );
    p.detach();

    QString command;

    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      if ( !(*it)->isForeign() )
      {
        if ( run_suid )
        {
          command.append( QString( "%1 smb4k_umount --suid --%2 " ).arg( suid_program ).arg( (*it)->filesystem() ) );
          command.append( KProcess::quote( (*it)->path() ) );
          command.append( " ; " );
        }
        else
        {
          command.append( QString( "smb4k_umount --no-suid --%1 " ).arg( (*it)->filesystem() ) );
          command.append( KProcess::quote( (*it)->path() ) );
          command.append( " ; " );
        }

        dir.setPath( (*it)->canonicalPath() );

#ifndef __FreeBSD__
        command.append( "rmdir --ignore-fail-on-non-empty " );
        command.append( KProcess::quote( dir.canonicalPath() ) );
        command.append( " ; " );
        command.append( "rmdir --ignore-fail-on-non-empty " );
        dir.cdUp();
        command.append( KProcess::quote( dir.canonicalPath() ) );
        command.append( " ; " );
#else
        command.append( "rmdir " );
        command.append( KProcess::quote( dir.canonicalPath() ) );
        command.append( " ; " );
        command.append( "rmdir " );
        dir.cdUp();
        command.append( KProcess::quote( dir.canonicalPath() ) );
        command.append( " ; " );
#endif
      }
      else
      {
        continue;
      }
    }

    command.truncate( command.length() - 2 );

    p << command;
    p.start( KProcess::DontCare, KProcess::NoCommunication );
  }
}


#include "smb4kmounter.moc"
