/***************************************************************************
    smb4koptionsdlg.cpp  -  description
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <qmap.h>

#include <qscrollview.h>

// KDE includes
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kstandarddirs.h>

// system includes
#include <sys/types.h>
#include <sys/stat.h>

// application specific includes
#include "smb4koptionsdlg.h"
#include "../core/smb4kauthinfo.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kglobal.h"

using namespace Smb4K_Global;


/**
 * This class handles the configuration dialog. It initializes the pages and saves
 * and reads the options.
 */

Smb4KOptionsDlg::Smb4KOptionsDlg( QWidget *parent, const char *name, bool modal )
: KDialogBase( IconList, i18n( "Configuration" ), Ok|Apply|Cancel, Ok, parent, name, modal, true )
{
  setWFlags( Qt::WDestructiveClose );

  setupAppearanceOptions();
  setupNetworkOptions();
  setupShareOptions();
  setupAuthOptions();
  setupSambaOptions();
  setupSuperUserOptions();

  readOptions();

  // Connections this the core
  connect( smb4k_core->fileIO(),           SIGNAL( no_suid_program() ),            this, SLOT( slotNoSUIDProgram() ) );
  connect( smb4k_core->fileIO(),           SIGNAL( finished_suid_writing() ),      this, SLOT( slotFinishedSUIDWriting() ) );

  // Others
  connect( m_sambaPage->m_filesystem,      SIGNAL( activated( const QString & ) ), this, SLOT( slotSambaFileSystem( const QString & ) ) );
  connect( m_appearancePage->m_showHidden, SIGNAL( stateChanged( int ) ),          this, SLOT( slotShowHiddenShares( int ) ) );
  connect( m_networkPage->m_useNMB,        SIGNAL( stateChanged( int ) ),          this, SLOT( slotBrowseListMethod( int ) ) );
  connect( m_superPage->m_remove,          SIGNAL( clicked() ),                    this, SLOT( slotRemoveSuperUserEntries() ) );
  connect( m_authPage->m_use_wallet,       SIGNAL( stateChanged( int ) ),          this, SLOT( slotKWalletButtonState( int ) ) );
  connect( m_authPage->m_defaultAuth,      SIGNAL( stateChanged( int ) ),          this, SLOT( slotDefaultAuthButtonState( int ) ) );

}


Smb4KOptionsDlg::~Smb4KOptionsDlg()
{
}


/****************************************************************************
   Set up the network options page.
****************************************************************************/

void Smb4KOptionsDlg::setupNetworkOptions()
{
  QFrame *frame = addPage( i18n( "Network" ), i18n( "Network" ), DesktopIcon( "network" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_networkPage = new Smb4KNetworkOptions( container->viewport() );
  container->addChild( m_networkPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the share options page.
****************************************************************************/

void Smb4KOptionsDlg::setupShareOptions()
{
  QFrame *frame = addPage( i18n( "Shares" ), i18n( "Shares" ), DesktopIcon( "hdd_mount" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_sharePage = new Smb4KShareOptions( container->viewport() );
  container->addChild( m_sharePage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the authentication management page.
****************************************************************************/

void Smb4KOptionsDlg::setupAuthOptions()
{
  QFrame *frame = addPage( i18n( "Authentication" ), i18n( "Authentication" ), DesktopIcon( "identity" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_authPage = new Smb4KAuthOptions( container->viewport() );
  container->addChild( m_authPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the apprearance options page.
****************************************************************************/

void Smb4KOptionsDlg::setupAppearanceOptions()
{
  QFrame *frame = addPage( i18n( "Appearance" ), i18n( "Appearance" ), DesktopIcon( "colorize" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_appearancePage = new Smb4KAppearanceOptions( container->viewport() );
  container->addChild( m_appearancePage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the advanced Samba options page.
****************************************************************************/

void Smb4KOptionsDlg::setupSambaOptions()
{
  QFrame *frame = addPage( i18n( "Samba" ), i18n( "Samba Options" ), DesktopIcon( "misc" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_sambaPage = new Smb4KSambaOptions( container->viewport() );
  container->addChild( m_sambaPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0 );
}


/****************************************************************************
   Set up the misc options page.
****************************************************************************/

void Smb4KOptionsDlg::setupSuperUserOptions()
{
  QFrame *frame = addPage( i18n( "Super User" ), i18n( "Super User Privileges" ), DesktopIcon( "misc" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_superPage = new Smb4KSuperUserOptions( container->viewport() );
  container->addChild( m_superPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0 );
}


/****************************************************************************
   Saves the options.
****************************************************************************/


bool Smb4KOptionsDlg::saveOptions()
{
  //
  // General options
  //
  config()->setGroup( "General Options" );
  config()->writeEntry( "Configuration Dialog Geometry", size() );

  //
  // Mount options
  //
  config()->setGroup( "Mount Options" );

  // Check, if the default path ends with a slash.
  if ( m_sharePage->m_defaultPath->lineEdit()->text().endsWith( "/" ) )
    config()->writeEntry( "Default Path", m_sharePage->m_defaultPath->lineEdit()->text() );
  else
    config()->writeEntry( "Default Path", m_sharePage->m_defaultPath->lineEdit()->text().append( "/" ) );

  config()->writeEntry( "Unmount All", m_sharePage->m_unmountAll->isChecked() );
  config()->writeEntry( "Mount Recent", m_sharePage->m_mountRecent->isChecked() );
  config()->writeEntry( "Unmount Foreign", m_sharePage->m_allowUnmountForeign->isChecked() );
  config()->writeEntry( "Force Lowercase", m_sharePage->m_lowercase->isChecked() );

  //
  // Browse options
  //
  config()->setGroup( "Browse Options" );

  if ( m_networkPage->m_useNMB->isChecked() )
  {
    config()->writeEntry( "Browse List", "nmblookup" );

    if ( !m_networkPage->m_hostName->text().stripWhiteSpace().isEmpty() )
    {
      config()->writeEntry( "Query Host", m_networkPage->m_hostName->text().stripWhiteSpace().upper() );
    }
    else
    {
      config()->deleteEntry( "Query Host" );
    }
    config()->writeEntry( "Auth For Browse List", m_networkPage->m_hostNeedsAuth->isChecked() );
  }
  else if ( m_networkPage->m_useHost->isChecked() )
  {
    config()->writeEntry( "Browse List", "host" );
    config()->writeEntry( "Query Host", m_networkPage->m_hostName->text().stripWhiteSpace().upper() );
    config()->writeEntry( "Auth For Browse List", m_networkPage->m_hostNeedsAuth->isChecked() );
  }

  config()->writeEntry( "Master Browser Auth", m_networkPage->m_mastersAuth->isChecked() );

  if ( m_networkPage->m_nmbSearch->isChecked() )
    config()->writeEntry( "Network Search", "nmblookup" );
  else
    config()->writeEntry( "Network Search", "smbclient" );


  //
  // Appearance
  //
  config()->setGroup( "Appearance" );
  config()->writeEntry( "Show Hidden", m_appearancePage->m_showHidden->isChecked() );
  config()->writeEntry( "Show IPC", m_appearancePage->m_showIPC->isChecked() );
  config()->writeEntry( "Show ADMIN", m_appearancePage->m_showAdmin->isChecked() );
  config()->writeEntry( "Show Printer", m_appearancePage->m_showPrinter->isChecked() );
  config()->writeEntry( "Show IP", m_appearancePage->m_showIP->isChecked() );
  config()->writeEntry( "Show Type", m_appearancePage->m_showType->isChecked() );
  config()->writeEntry( "Show Comment", m_appearancePage->m_showComment->isChecked() );
  if ( m_appearancePage->m_shareview->isChecked() )
    config()->writeEntry( "Show Shares", "list" );
  else
    config()->writeEntry( "Show Shares", "icons" );
  config()->writeEntry( "Show Mount Point", m_appearancePage->m_description->isChecked() );
  config()->writeEntry( "Show All Shares", m_appearancePage->m_external->isChecked() );

  //
  // Authentication
  //
  config()->setGroup( "Authentication" );
  config()->writeEntry( "Use Wallet", m_authPage->m_use_wallet->isChecked() );
  config()->writeEntry( "Remember Passwords", m_authPage->m_remember_passwords->isChecked() );
  config()->writeEntry( "Default Authentication", m_authPage->m_defaultAuth->isChecked() );

  if ( m_authPage->m_defaultAuth->isChecked() )
  {
    Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null, m_authPage->m_defaultLogin->text().stripWhiteSpace(), m_authPage->m_defaultPasswd->text().stripWhiteSpace() );
    smb4k_core->passwordHandler()->writeDefaultAuth( auth );
  }

  //
  // Samba options
  //
  config()->setGroup( "Samba" );

  // General
  if ( !m_sambaPage->m_netbios->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(readGlobalSMBOptions().find( "netbios name" )), m_sambaPage->m_netbios->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "NetBIOS Name", m_sambaPage->m_netbios->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "NetBIOS Name" );
  }

  if ( m_sambaPage->m_domain->isEnabled() )
  {
    if ( !m_sambaPage->m_domain->text().stripWhiteSpace().isEmpty() &&
         QString::compare( m_sambaPage->m_domain->text().stripWhiteSpace().upper(), *(readGlobalSMBOptions().find( "workgroup" )) ) != 0 )
    {
      config()->writeEntry( "Domain", m_sambaPage->m_domain->text().stripWhiteSpace().upper() );
    }
    else
    {
      config()->deleteEntry( "Domain" );
    }
  }

  if ( !m_sambaPage->m_sockopt->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(readGlobalSMBOptions().find( "socket options" )), m_sambaPage->m_sockopt->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "Socket Options", m_sambaPage->m_sockopt->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "Socket Options" );
  }

  if ( !m_sambaPage->m_scope->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(readGlobalSMBOptions().find( "netbios scope")), m_sambaPage->m_scope->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "Scope", m_sambaPage->m_scope->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "NetBIOS Scope" );
  }

  config()->writeEntry( "Port", m_sambaPage->m_port->value() );

  if ( m_sambaPage->m_krb->isEnabled() )
    config()->writeEntry( "Use Kerberos", m_sambaPage->m_krb->isChecked() );
  else
    config()->deleteEntry( "Use Kerberos" );

  // mount
  if ( QString::compare( m_sambaPage->m_filesystem->currentText(), "SMBFS" ) == 0 )
  {
    config()->writeEntry( "Mount Filesystem", "smbfs" );
  }
  else
  {
    config()->writeEntry( "Mount Filesystem", "cifs" );
  }

  if ( QString::compare( m_sambaPage->m_uid->currentText(), i18n( "default" ) ) != 0 )
  {
    config()->writeEntry( "Mount UID", m_sambaPage->m_uid->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount UID" );
  }

  if ( QString::compare( m_sambaPage->m_gid->currentText(), i18n( "default" ) ) != 0 )
  {
    config()->writeEntry( "Mount GID", m_sambaPage->m_gid->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount GID" );
  }

  if ( QString::compare( m_sambaPage->m_charset->currentText(), i18n( "default" ) ) != 0 &&
       m_sambaPage->m_charset->isEnabled() )
  {
    config()->writeEntry( "Mount Charset", m_sambaPage->m_charset->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount Charset" );
  }

  if ( QString::compare( m_sambaPage->m_codepage->currentText(), i18n( "default" ) ) != 0 &&
       m_sambaPage->m_codepage->isEnabled() )
  {
    config()->writeEntry( "Mount Codepage", m_sambaPage->m_codepage->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount Codepage" );
  }

  QString fmask = m_sambaPage->m_fmask->text().stripWhiteSpace();
  QString dmask = m_sambaPage->m_dmask->text().stripWhiteSpace();

  if ( !fmask.startsWith( "0" ) )
  {
    fmask.prepend( "0" );
  }

  if ( !dmask.startsWith( "0" ) )
  {
    dmask.prepend( "0" );
  }

  config()->writeEntry( "Mount FMASK", fmask );
  config()->writeEntry( "Mount DMASK", dmask );

  if ( QString::compare( m_sambaPage->m_read_mode->currentText(), i18n( "read-write" ) ) == 0 )
  {
    config()->writeEntry( "Mount ReadWrite", true );
  }
  else
  {
    config()->writeEntry( "Mount ReadWrite", false );
  }

  if ( m_sambaPage->m_ttl->value() != 1000 && m_sambaPage->m_ttl->isEnabled() )
  {
    config()->writeEntry( "Mount Cache", m_sambaPage->m_ttl->value() );
  }
  else
  {
    config()->deleteEntry( "Mount Cache" );
  }

  if ( m_sambaPage->m_unicode->isEnabled() )
  {
    config()->writeEntry( "Mount Unicode", m_sambaPage->m_unicode->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Mount Unicode" );
  }

  if ( m_sambaPage->m_lfs->isEnabled() )
  {
    config()->writeEntry( "Mount LFS", m_sambaPage->m_lfs->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Mount LFS" );
  }

  config()->writeEntry( "Mount RSize", m_sambaPage->m_rsize->isChecked() );
  config()->writeEntry( "Mount WSize", m_sambaPage->m_wsize->isChecked() );

  // net
  if ( m_sambaPage->m_net_auto->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "auto" );
  }
  else if ( m_sambaPage->m_net_ads->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "ads" );
  }
  else if ( m_sambaPage->m_net_rpc->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "rpc" );
  }
  else if ( m_sambaPage->m_net_rap->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "rap" );
  }

  if ( m_sambaPage->m_net_machine_account->isChecked() )
  {
    config()->writeEntry( "Net Machine Account", true );
  }
  else
  {
    config()->writeEntry( "Net Machine Account", false );
  }

  // smbclient
  if ( !m_sambaPage->m_resolve->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(readGlobalSMBOptions().find( "name resolve order" )), m_sambaPage->m_resolve->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "Client Resolve Order", m_sambaPage->m_resolve->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "Client Resolve Order" );
  }

  if ( m_sambaPage->m_buffer->value() == 65520 )
  {
    config()->deleteEntry( "Client Buffer Size" );
  }
  else
  {
    config()->writeEntry( "Client Buffer Size", m_sambaPage->m_buffer->value() );
  }

  if ( !m_sambaPage->m_broadcast->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(readGlobalSMBOptions().find( "interfaces" )), m_sambaPage->m_broadcast->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "NMB Broadcast", m_sambaPage->m_broadcast->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "NMB Broadcast" );
  }

  config()->writeEntry( "NMB Port 137", m_sambaPage->m_udp->isChecked() );


  //
  // The super user tab
  //
  config()->setGroup( "Super User Privileges" );

  if ( m_superPage->isEnabled() )
  {
    if ( m_superPage->m_super->isChecked() )
    {
      config()->writeEntry( "SUID Program", "super" );
    }
    else if ( m_superPage->m_sudo->isChecked() )
    {
      config()->writeEntry( "SUID Program", "sudo" );
    }
    config()->writeEntry( "Force Unmount", m_superPage->m_unmount_dead->isChecked() );
    config()->writeEntry( "Run SUID", m_superPage->m_use_suid->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Force Unmount" );
    config()->deleteEntry( "Run SUID" );
    config()->deleteEntry( "SUID Program" );
  }

  config()->sync();
  return true;
}


/****************************************************************************
   Reads the options.
****************************************************************************/

void Smb4KOptionsDlg::readOptions()
{
  int mask = getUMASK();
  QString system = getSystemName();
  QString kernel = getKernelVersion();

  int k_major = kernel.section( ".", 0, 0 ).toInt();
  int k_minor = kernel.section( ".", 1, 1 ).toInt();
  int k_rev = kernel.section( ".", 2, 2 ).toInt();

  int s_major = smb4k_core->getSambaVersion().section( ".", 0, 0 ).toInt();

  config()->setGroup( "Programs" );
  QString super = config()->readPathEntry( "super" );
  QString sudo = config()->readPathEntry( "sudo" );
  QString mount_cifs = config()->readPathEntry( "mount.cifs" );
  QString net = config()->readPathEntry( "net" );

  //
  // Mount options
  //
  config()->setGroup( "Mount Options" );
  m_sharePage->m_defaultPath->lineEdit()->setText( config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) ) );
  m_sharePage->m_unmountAll->setChecked( config()->readBoolEntry( "Unmount All", false ) );
  m_sharePage->m_mountRecent->setChecked( config()->readBoolEntry( "Mount Recent", false ) );
  m_sharePage->m_allowUnmountForeign->setChecked( config()->readBoolEntry( "Unmount Foreign", false ) );
  m_sharePage->m_lowercase->setChecked( config()->readBoolEntry( "Force Lowercase", false ) );


  //
  // Browse options
  //
  config()->setGroup( "Browse Options" );
  // Toggle the browse list buttons.
  QString browselist = config()->readEntry( "Browse List", "nmblookup" );
  QString host = config()->readEntry( "Query Host", QString::null );

  QString wins = getWINSServer();

  if ( QString::compare( browselist.stripWhiteSpace(), "nmblookup" ) == 0 )
  {
    m_networkPage->m_useNMB->setChecked( true );
    m_networkPage->m_hostLabel->setEnabled( false );
    m_networkPage->m_hostName->setEnabled( false );
    m_networkPage->m_hostNeedsAuth->setEnabled( false );
  }
  else if ( QString::compare( browselist.stripWhiteSpace(), "host" ) == 0 )
  {
    m_networkPage->m_useHost->setChecked( true );
  }

  if ( !host.isEmpty() )
    m_networkPage->m_hostName->setText( host );

  if ( !wins.isEmpty() )
    m_networkPage->m_winsYesNo->setText( wins );

  m_networkPage->m_hostNeedsAuth->setChecked( config()->readBoolEntry( "Auth For Browse List", false ) );
  m_networkPage->m_mastersAuth->setChecked( config()->readBoolEntry( "Master Browser Auth", false ) );

  if ( QString::compare( config()->readEntry( "Network Search", "nmblookup" ), "nmblookup" ) == 0 )
  {
    m_networkPage->m_nmbSearch->setChecked( true );
  }
  else
  {
    m_networkPage->m_clientSearch->setChecked( true );
  }


  //
  // Appearance
  //
  config()->setGroup( "Appearance" );
  bool show_hidden = config()->readBoolEntry( "Show Hidden", true );
  m_appearancePage->m_showHidden->setChecked( show_hidden );

  if ( !show_hidden )
  {
    m_appearancePage->m_showIPC->setEnabled( false );
    m_appearancePage->m_showAdmin->setEnabled( false );
  }

  m_appearancePage->m_showIPC->setChecked( config()->readBoolEntry( "Show IPC", false ) );
  m_appearancePage->m_showAdmin->setChecked( config()->readBoolEntry( "Show ADMIN", false ) );
  m_appearancePage->m_showPrinter->setChecked( config()->readBoolEntry( "Show Printer", true ) );
  m_appearancePage->m_showIP->setChecked( config()->readBoolEntry( "Show IP", true ) );
  m_appearancePage->m_showType->setChecked( config()->readBoolEntry( "Show Type", true ) );
  m_appearancePage->m_showComment->setChecked( config()->readBoolEntry( "Show Comment", true ) );

  if ( QString::compare( config()->readEntry( "Show Shares", "icons" ), "icons" ) == 0 )
  {
    m_appearancePage->m_shareview->setChecked( false );
  }
  else if ( QString::compare( config()->readEntry( "Show Shares", "icons" ), "list" ) == 0 )
  {
    m_appearancePage->m_shareview->setChecked( true );
  }

  if ( !config()->readBoolEntry( "Show Mount Point", false ) )
  {
    m_appearancePage->m_description->setChecked( false );
  }
  else
  {
    m_appearancePage->m_description->setChecked( true );
  }

  m_appearancePage->m_external->setChecked( config()->readBoolEntry( "Show All Shares", false ) );


  //
  // General options
  //
  config()->setGroup( "General Options" );
  setInitialSize( config()->readSizeEntry( "Configuration Dialog Geometry", 0L ), false );


  //
  // Authentication
  //
  config()->setGroup( "Authentication" );

  bool uses_wallet = config()->readBoolEntry( "Use Wallet", true );
  m_authPage->m_login_box->setEnabled( uses_wallet );
  m_authPage->m_use_wallet->setChecked( uses_wallet );
  m_authPage->m_remember_passwords->setChecked( config()->readBoolEntry( "Remember Passwords", true ) );

  if ( config()->readBoolEntry( "Default Authentication", false ) )
  {
    m_authPage->m_defaultAuth->setChecked( true );
    m_authPage->m_authWidget->setEnabled( true );
  }
  else
  {
    m_authPage->m_defaultAuth->setChecked( false );
    m_authPage->m_authWidget->setEnabled( false );
  }

  if ( uses_wallet )
  {
    Smb4KAuthInfo *defaultAuth = smb4k_core->passwordHandler()->readDefaultAuth();

    if ( defaultAuth )
    {
      m_authPage->m_defaultLogin->setText( defaultAuth->user() );
      m_authPage->m_defaultPasswd->setText( defaultAuth->password() );
    }
  }


  //
  // Samba options
  //
  config()->setGroup( "Samba" );
  m_sambaPage->m_netbios->setText( config()->readEntry( "NetBIOS Name", *(readGlobalSMBOptions().find( "netbios name" )) ) );
  m_sambaPage->m_domain->setText( config()->readEntry( "Domain", *(readGlobalSMBOptions().find( "workgroup" )) ) );
  m_sambaPage->m_sockopt->setText( config()->readEntry( "Socket Options", *(readGlobalSMBOptions().find( "socket options" )) ) );
  m_sambaPage->m_scope->setText( config()->readEntry( "NetBIOS Scope", *(readGlobalSMBOptions().find( "netbios scope" )) ) );
  m_sambaPage->m_port->setValue( config()->readNumEntry( "Port", 139 ) );

  if ( s_major >= 3 )
  {
    m_sambaPage->m_krb->setChecked( config()->readBoolEntry( "Use Kerberos", false ) );
    m_sambaPage->m_unicode->setChecked( config()->readBoolEntry( "Mount Unicode", false ) );
    m_sambaPage->m_lfs->setChecked( config()->readBoolEntry( "Mount LFS", false ) );
  }
  else
  {
    m_sambaPage->m_krb->setEnabled( false );
    m_sambaPage->m_lfs->setEnabled( false );
    m_sambaPage->m_unicode->setEnabled( false );
  }

  QString filesystem = config()->readEntry( "Mount Filesystem", "smbfs" );

  if ( QString::compare( filesystem, "smbfs" ) == 0 || mount_cifs.isEmpty() )
  {
    m_sambaPage->m_filesystem->setCurrentText( "SMBFS" );
    slotSambaFileSystem( "SMBFS" );
  }
  else
  {
    m_sambaPage->m_filesystem->setCurrentText( "CIFS" );
    slotSambaFileSystem( "CIFS" );
  }

  // TODO: Check this with a system that has no mount.cifs installed!
  if ( mount_cifs.isEmpty() )
  {
    m_sambaPage->m_filesystem->removeItem( 1 );
  }

  QValueList<Smb4KUser *> list = smb4k_core->fileIO()->getUsers();
  QStringList uid, gid;

  for ( QValueList<Smb4KUser *>::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    // Put the UIDs in the respective combo box.
    if ( uid.find( QString( "%1" ).arg( (*it)->uid() ) ) == uid.end() )
    {
      uid.append( QString( "%1" ).arg( (*it)->uid() ) );
    }

    // Put the GIDs in the respective combo box.
    if ( gid.find( QString( "%1" ).arg( (*it)->gid() ) ) == gid.end() )
    {
      gid.append( QString( "%1" ).arg( (*it)->gid() ) );
    }
  }

  uid.sort();
  gid.sort();

  m_sambaPage->m_uid->insertStringList( uid, -1 );
  m_sambaPage->m_uid->setCurrentText( config()->readEntry( "Mount UID", i18n( "default" ) ) );

  m_sambaPage->m_gid->insertStringList( gid, -1 );
  m_sambaPage->m_gid->setCurrentText( config()->readEntry( "Mount GID", i18n( "default" ) ) );

  if ( k_major >= 2 && k_minor >= 4 && k_rev >= 0 )
  {
    m_sambaPage->m_codepage->setCurrentText( config()->readEntry( "Mount Codepage", i18n( "default" ) ) );
    m_sambaPage->m_charset->setCurrentText( config()->readEntry( "Mount Charset", i18n( "default" ) ) );
  }
  else
  {
    m_sambaPage->m_codepage->setEnabled( false );
    m_sambaPage->m_charset->setEnabled( false );
  }

  QString fmask = config()->readEntry( "Mount FMASK", QString( "%1" ).arg( 0777 - mask, 0, 8 ) );
  QString dmask = config()->readEntry( "Mount DMASK", QString( "%1" ).arg( 0777 - mask, 0, 8 ) );

  if ( !fmask.startsWith( "0" ) )
  {
    fmask.prepend( "0" );
  }

  if ( !dmask.startsWith( "0" ) )
  {
    dmask.prepend( "0" );
  }

  m_sambaPage->m_fmask->setText( fmask );
  m_sambaPage->m_dmask->setText( dmask );

  if ( config()->readBoolEntry( "Mount ReadWrite", true ) )
  {
    m_sambaPage->m_read_mode->setCurrentText( i18n( "read-write" ) );
  }
  else
  {
    m_sambaPage->m_read_mode->setCurrentText( i18n( "read-only" ) );
  }

  if ( k_major >= 2 && k_minor >= 4 && k_rev >= 2 )
    m_sambaPage->m_ttl->setValue( config()->readNumEntry( "Mount Cache", 1000 ) );
  else
    m_sambaPage->m_ttl->setEnabled( false );

  m_sambaPage->m_rsize->setChecked( config()->readBoolEntry( "Mount RSize", false ) );
  m_sambaPage->m_wsize->setChecked( config()->readBoolEntry( "Mount WSize", false ) );

  m_sambaPage->m_buffer->setValue( config()->readNumEntry( "Client Buffer Size", 65520 ) );
  m_sambaPage->m_resolve->setText( config()->readEntry( "Client Resolve Order", *(readGlobalSMBOptions().find( "name resolve order" )) ) );

  if ( s_major >= 3 )
  {
    m_sambaPage->m_domain->setEnabled( true );
  }
  else
  {
    m_sambaPage->m_domain->setEnabled( false );
  }

  m_sambaPage->m_broadcast->setText( config()->readEntry( "NMB Broadcast", *(readGlobalSMBOptions().find( "interfaces" )) ) );
  m_sambaPage->m_udp->setChecked( config()->readBoolEntry( "NMB Port 137", false ) );

  QString protocol = config()->readEntry( "Net Protocol", "auto" );

  if ( net.isEmpty() )
  {
    ((QWidget *)m_sambaPage->child( "SambaNetTab", "QWidget" ))->setEnabled( false );
  }

  if ( QString::compare( protocol, "auto" ) == 0 )
  {
    m_sambaPage->m_net_auto->setChecked( true );
  }
  else if ( QString::compare( protocol, "ads" ) == 0 )
  {
    m_sambaPage->m_net_ads->setChecked( true );
  }
  else if ( QString::compare( protocol, "rpc" ) == 0 )
  {
    m_sambaPage->m_net_rpc->setChecked( true );
  }
  else if ( QString::compare( protocol, "rap" ) == 0 )
  {
    m_sambaPage->m_net_rap->setChecked( true );
  }

  m_sambaPage->m_net_machine_account->setChecked( config()->readBoolEntry( "Net Machine Account", false ) );


  //
  // Super user options
  //
  config()->setGroup( "Super User Privileges" );

  if ( super.isEmpty() && sudo.isEmpty() )
  {
    m_superPage->setEnabled( false );
  }
  else if ( !super.isEmpty() && sudo.isEmpty() )
  {
    m_superPage->m_super->setChecked( true );
    m_superPage->m_sudo->setEnabled( false );
  }
  else if ( super.isEmpty() && !sudo.isEmpty() )
  {
    m_superPage->m_sudo->setChecked( true );
    m_superPage->m_super->setEnabled( false );
  }
  else
  {
    if ( QString::compare( config()->readEntry( "SUID Program", "super" ), "super" ) == 0 )
    {
      m_superPage->m_super->setChecked( true );
    }
    else if ( QString::compare( config()->readEntry( "SUID Program", "super" ), "sudo" ) == 0 )
    {
      m_superPage->m_sudo->setChecked( true );
    }
  }

  if ( QString::compare( system, "Linux" ) == 0 && k_major >= 2 && ( ( k_minor == 4 && k_rev >= 11 ) || k_minor > 4 ) )
  {
    m_superPage->m_unmount_dead->setChecked( config()->readBoolEntry( "Force Unmount", false ) );
  }
  else
  {
    m_superPage->m_unmount_dead->setChecked( false );
    m_superPage->m_unmount_dead->setEnabled( false );
  }

  m_superPage->m_use_suid->setChecked( config()->readBoolEntry( "Run SUID", false ) );
}


/****************************************************************************
   Checks, whether everything has been done
****************************************************************************/


bool Smb4KOptionsDlg::check()
{
  bool ok = true;

  //
  // Network page
  //
  if ( m_networkPage->m_useHost->isChecked() && m_networkPage->m_hostName->text().stripWhiteSpace().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the host you want to query to retrieve the browse list.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Network );
      m_networkPage->m_hostName->setFocus();
    }
  }

  //
  // Shares page
  //
  if ( m_sharePage->m_defaultPath->url().stripWhiteSpace().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the path where Smb4K will mount the shares to.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Shares );
      m_sharePage->m_defaultPath->setFocus();
    }
  }

  //
  // Authentication page
  //
  if ( m_authPage->m_defaultAuth->isChecked() )
  {
    if ( m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default login and password.\nDo you want to enter them now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultLogin->setFocus();
      }
    }
    else if ( m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && !m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default login.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultLogin->setFocus();
      }
    }
    else if ( !m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default password.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultPasswd->setFocus();
      }
    }
  }

  return ok;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KOptionsDlg::slotOk()
{
  if ( check() )
  {
    if ( m_superPage->m_unmount_dead->isChecked() || m_superPage->m_use_suid->isChecked() )
    {
      m_button = "OK";

      QString prog;

      if ( m_superPage->m_super->isChecked() )
      {
        prog = "super";
      }
      else if ( m_superPage->m_sudo->isChecked() )
      {
        prog = "sudo";
      }

      if ( smb4k_core->fileIO()->writeSuperUserEntries( prog ) )
      {
        if ( saveOptions() )
        {
          emit read();
        }
      }
    }
    else
    {
      if ( saveOptions() )
      {
        emit read();
        accept();
      }
    }
  }
}


void Smb4KOptionsDlg::slotApply()
{
  if ( check() )
  {
    if ( m_superPage->m_unmount_dead->isChecked() || m_superPage->m_use_suid->isChecked() )
    {
      m_button = "Apply";

      QString prog;

      if ( m_superPage->m_super->isChecked() )
      {
        prog = "super";
      }
      else if ( m_superPage->m_sudo->isChecked() )
      {
        prog = "sudo";
      }

      if ( smb4k_core->fileIO()->writeSuperUserEntries( prog ) )
      {
        if ( saveOptions() )
        {
          emit read();
        }
      }
    }
    else
    {
      if ( saveOptions() )
      {
        emit read();
      }
    }
  }
}


void Smb4KOptionsDlg::slotNoSUIDProgram()
{
  m_superPage->m_unmount_dead->setChecked( false );
  m_superPage->m_use_suid->setChecked( false );
}


void Smb4KOptionsDlg::slotFinishedSUIDWriting()
{
  if ( QString::compare( m_button, "OK" ) == 0 )
  {
    accept();
  }
}

/****************************************************************************
   SLOTS FOR WIDGETS
****************************************************************************/

void Smb4KOptionsDlg::slotSambaFileSystem( const QString &fs )
{
  if ( QString::compare( fs, "SMBFS" ) == 0 )
  {
    ((QLabel *)m_sambaPage->child( "SambaCachingTime", "QLabel" ))->setEnabled( true );
    ((QLabel *)m_sambaPage->child( "MountCodepage", "QLabel" ))->setEnabled( true );
    m_sambaPage->m_codepage->setEnabled( true );
    m_sambaPage->m_ttl->setEnabled( true );
    m_sambaPage->m_unicode->setEnabled( true );
    m_sambaPage->m_lfs->setEnabled( true );
    m_sambaPage->m_rsize->setEnabled( false );
    m_sambaPage->m_wsize->setEnabled( false );
  }
  else
  {
    ((QLabel *)m_sambaPage->child( "SambaCachingTime", "QLabel" ))->setEnabled( false );
    ((QLabel *)m_sambaPage->child( "MountCodepage", "QLabel" ))->setEnabled( false );
    m_sambaPage->m_codepage->setEnabled( false );
    m_sambaPage->m_ttl->setEnabled( false );
    m_sambaPage->m_unicode->setEnabled( false );
    m_sambaPage->m_lfs->setEnabled( false );
    m_sambaPage->m_rsize->setEnabled( true );
    m_sambaPage->m_wsize->setEnabled( true );
  }
}


void Smb4KOptionsDlg::slotShowHiddenShares( int state )
{
  if ( state == QCheckBox::On )
  {
    m_appearancePage->m_showIPC->setEnabled( true );
    m_appearancePage->m_showAdmin->setEnabled( true );
  }
  else if ( state == QCheckBox::Off )
  {
    m_appearancePage->m_showIPC->setEnabled( false );
    m_appearancePage->m_showAdmin->setEnabled( false );
  }
}


void Smb4KOptionsDlg::slotBrowseListMethod( int state )
{
  if ( state == QRadioButton::On )
  {
    m_networkPage->m_hostLabel->setEnabled( false );
    m_networkPage->m_hostName->setEnabled( false );
    m_networkPage->m_hostNeedsAuth->setEnabled( false );
  }
  else if ( state == QRadioButton::Off )
  {
    m_networkPage->m_hostLabel->setEnabled( true );
    m_networkPage->m_hostName->setEnabled( true );
    m_networkPage->m_hostNeedsAuth->setEnabled( true );
  }
}


void Smb4KOptionsDlg::slotRemoveSuperUserEntries()
{
  if ( smb4k_core->fileIO()->removeSuperUserEntries() )
  {
    m_superPage->m_unmount_dead->setChecked( false );
    m_superPage->m_use_suid->setChecked( false );
  }
}


void Smb4KOptionsDlg::slotKWalletButtonState( int state )
{
  if ( state == QCheckBox::On )
  {
    m_authPage->m_login_box->setEnabled( true );
  }
  else if ( state == QCheckBox::Off )
  {
    m_authPage->m_login_box->setEnabled( false );
  }
}


void Smb4KOptionsDlg::slotDefaultAuthButtonState( int state )
{
  if ( state == QCheckBox::On )
  {
    m_authPage->m_authWidget->setEnabled( true );
  }
  else if ( state == QCheckBox::Off )
  {
    m_authPage->m_authWidget->setEnabled( false );
  }
}

#include "smb4koptionsdlg.moc"
