/***************************************************************************
                          smb4kcore  -  description
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdir.h>

// KDE includes
#include <kdeversion.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <krun.h>
#include <kapplication.h>

// system includes
#include <stdlib.h>
#include <unistd.h>
#include <sys/statvfs.h>
#include <errno.h>

// application specific includes
#include "smb4kscanner.h"
#include "smb4kmounter.h"
#include "smb4kpasswordhandler.h"
#include "smb4kfileio.h"
#include "smb4kipfinder.h"
#include "smb4kbookmarkhandler.h"
#include "smb4kprint.h"
#include "smb4kshellio.h"

#include "smb4kcore.h"
#include "smb4kdefs.h"

Smb4KCore *Smb4KCore::m_this_class = 0L;



Smb4KCore::Smb4KCore( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_this_class = this;

  searchPrograms();

  m_shellIO = new Smb4KShellIO( this, "Core_ShellIO" );
  m_fileIO = new Smb4KFileIO( this, "Core_FileIO" );
  m_fileIO->convertSuperTab();

  m_scanner = new Smb4KScanner( this, "Core_Scanner" );
  m_mounter = new Smb4KMounter( this, "Core_Mounter" );
  m_bookmarkHandler = new Smb4KBookmarkHandler( this, "Core_BookmarkHandler" );
  m_finder = new Smb4KIPFinder( this, "Core_IPFinder" );
  m_print = new Smb4KPrint( this, "Core_PrinterHandler" );
  m_password_handler = new Smb4KPasswordHandler( this, "Core_PasswordHandler" );

  m_scanner_state = SCANNER_STOP;
  m_scanner_running = false;

  m_mounter_state = MOUNTER_STOP;
  m_mounter_running = false;

  m_ipfinder_state = IPFINDER_STOP;
  m_ipfinder_running = false;

  m_print_state = PRINT_STOP;
  m_print_running = false;

  m_samba_version = QString::null;

  m_path = QString::null;

  m_timer = new QTimer( this, "Core_Timer" );

  // Connections:
  connect( m_scanner,         SIGNAL( running( int, bool ) ),            this, SLOT( slotScannerIsRunning( int, bool ) ) );
  connect( m_mounter,         SIGNAL( running( int, bool ) ),            this, SLOT( slotMounterIsRunning( int, bool ) ) );
  connect( m_finder,          SIGNAL( running( int, bool ) ),            this, SLOT( slotIPFinderIsRunning( int, bool ) ) );
  connect( m_print,           SIGNAL( running( int, bool ) ),            this, SLOT( slotPrinterHandlerIsRunning( int, bool ) ) );
  connect( m_scanner,         SIGNAL( error( int, const QString & ) ),   this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_mounter,         SIGNAL( error( int, const QString & ) ),   this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_fileIO,          SIGNAL( error( int, const QString & ) ),   this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_print,           SIGNAL( error( int, const QString & ) ),   this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_bookmarkHandler, SIGNAL( error( int, const QString & ) ),   this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_shellIO,         SIGNAL( sambaVersion( const QString & ) ), this, SLOT( slotReceiveSambaVersion( const QString & ) ) );
  connect( kapp,              SIGNAL( shutDown() ),                      this, SLOT( slotShutdown() ) );
  connect( m_timer,           SIGNAL( timeout() ),                       this, SLOT( slotGetUsage() ) );

  connect( m_scanner,         SIGNAL( members( const QValueList<Smb4KHostItem *> & ) ),
           m_finder,          SLOT( slotInput( const QValueList<Smb4KHostItem *> & ) ) );

  QTimer::singleShot( 10, this, SLOT( slotGetSambaVersion() ) );
}


Smb4KCore::~Smb4KCore()
{
  // Do not call abort() here. This will lead
  // to crashes.
}


/****************************************************************************
   Returns a bool that tells the program whether a core process is running.
****************************************************************************/

bool Smb4KCore::isRunning()
{
  if ( m_scanner_running || m_mounter_running || m_ipfinder_running || m_print_running )
  {
    return true;
  }
  else
  {
    return false;
  }
}


/****************************************************************************
   Returns the current state of the core.
****************************************************************************/

void Smb4KCore::setCurrentState( int state )
{
  if ( state != SCANNER_STOP && state != MOUNTER_STOP && state != IPFINDER_STOP && state != PRINT_STOP )
  {
    m_current_state = state;
  }
  else
  {
    if ( !m_scanner_running && !m_mounter_running && !m_ipfinder_running && !m_print_running )
    {
      m_current_state = CORE_STOP;
    }
    else
    {
      if ( m_ipfinder_running )
      {
        m_current_state = m_ipfinder_state;
      }
      else if ( m_scanner_running )
      {
        m_current_state = m_scanner_state;
      }
      else if ( m_print_running )
      {
        m_current_state = m_print_state;
      }
      else if ( m_mounter_running )
      {
        m_current_state = m_mounter_state;
      }
    }
  }
}


/****************************************************************************
   Aborts any process of the core.
****************************************************************************/

void Smb4KCore::abort()
{
  m_scanner->abort();
  m_mounter->abort();
  m_finder->abort();
  m_print->abort();
}


/****************************************************************************
   Opens the given URL.
****************************************************************************/

void Smb4KCore::open( Smb4KShare *share )
{
  if ( !share || share->isBroken() )
  {
    return;
  }

#if KDE_VERSION_MAJOR == 3 && KDE_VERSION_MINOR <= 3 && KDE_VERSION_RELEASE <= 92

  if ( QString::compare( share->getFilesystem(), "cifs" ) == 0 )
  {
    if( KMessageBox::warningContinueCancel( (QWidget *)this, i18n( "Up to KDE 3.3.2, KIO and Konqueror cannot handle CIFS shares. Konqueror will hang if you try to access it.\nDo you want to continue?" ) ) == KMessageBox::Cancel )
    {
      return;
    }
  }

#endif

  KURL url;
  url.setPath( share->getCanonicalPath() );

  new KRun( url, 0, true, true );
}


/****************************************************************************
   Searches for the programs needed by Smb4K
****************************************************************************/

void Smb4KCore::searchPrograms()
{
  config()->deleteGroup( "Programs" );
  config()->setGroup( "Programs" );

  QStringList pathList = QStringList::split( ":", QString( "%1" ).arg( getenv( "PATH" ) ), false );
  pathList << "/sbin";
  pathList << "/usr/sbin";
  pathList << "/usr/local/sbin";

  // all mandatory programs that are needed by Smb4K
  QStringList fileList;
  fileList << "grep";
  fileList << "awk";
  fileList << "sed";
  fileList << "xargs";
  fileList << "rmdir";
  fileList << "ls";
  fileList << "nmblookup";
  fileList << "smbclient";
#ifndef __FreeBSD__
  fileList << "smbmount";
  fileList << "smbumount";
  fileList << "mount";
#else
  fileList << "mount_smbfs";
#endif
  fileList << "umount";
  fileList << "smb4k_mount";
  fileList << "smb4k_umount";
  fileList << "smb4k_kill";
  fileList << "smbspool";

  // optional programs
  QStringList optional;
  optional << "super";
  optional << "sudo";
  optional << "dvips";
  optional << "enscript";
  optional << "net";
#ifndef __FreeBSD__
  optional << "mount.cifs";
#endif

  fileList += optional;

  config()->setGroup( "Programs" );

  for ( QStringList::ConstIterator it = pathList.begin(); it != pathList.end(); ++it )
  {
    QDir::setCurrent( *it );

    for ( QStringList::Iterator bin = fileList.begin(); bin != fileList.end(); ++bin )
    {
      if ( QFile::exists( *bin ) )
      {
        config()->writeEntry( *bin, QString( "%1/%2" ).arg( QDir::currentDirPath(), *bin ) );
        *bin = QString::null;
        continue;
      }
      else
        continue;
    }
  }

  fileList.remove( QString::null );

  if ( fileList.count() != 0 )
  {
    for ( QStringList::Iterator it = optional.begin(); it != optional.end(); ++it )
    {
      if ( fileList.find( *it ) == fileList.end() )
      {
        *it = QString::null;
        continue;
      }
      else
      {
        fileList.remove( fileList.find( *it ) );
        continue;
      }
    }

    optional.remove( QString::null );

    if ( fileList.count() != 0 )
    {
      fileList.sort();
      slotShowErrorMessage( ERROR_MISSING_PROGRAMS, fileList.join( ", " ) );
      exit( EXIT_SUCCESS );
    }

    if ( optional.count() != 0 )
    {
      config()->setGroup( "Mount Options" );

      bool disable_suid = false;
      QString suid_program = config()->readEntry( "SUID Program", QString::null );

      if ( !suid_program.isEmpty() )
      {
        if ( QString::compare( suid_program, "super" ) == 0 && optional.find( "super" ) != optional.end() )
        {
          disable_suid = true;
          slotShowErrorMessage( ERROR_SUID_PROGRAM_MISSING, suid_program );
        }
        else if ( QString::compare( suid_program, "sudo" ) == 0 && optional.find( "sudo" ) != optional.end() )
        {
          disable_suid = true;
          slotShowErrorMessage( ERROR_SUID_PROGRAM_MISSING, suid_program );
        }
      }

      if ( disable_suid )
      {
        config()->setGroup( "Super User Privileges" );
        config()->writeEntry( "Run SUID", false );
        config()->writeEntry( "Force Unmount", false );
      }
    }
  }
}


/****************************************************************************
   Set the share for which the usage should be returned
****************************************************************************/

void Smb4KCore::setShareForUsageInfo( Smb4KShare *share )
{
  if ( share && !share->isBroken() )
  {
    m_path = QString::fromLocal8Bit( share->getCanonicalPath() );
    m_timer->start( 1000, false );
  }
  else
  {
    m_timer->stop();
    m_path = QString::null;
  }

  slotGetUsage();
}


/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KCore::slotScannerIsRunning( int state, bool on )
{
  m_scanner_state = state;
  m_scanner_running = on;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotMounterIsRunning( int state, bool on )
{
  m_mounter_state = state;
  m_mounter_running = on;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotIPFinderIsRunning( int state, bool on )
{
  m_ipfinder_state = state;
  m_ipfinder_running = on;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotPrinterHandlerIsRunning( int state, bool on )
{
  m_print_state = state;
  m_print_running = on;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotShowErrorMessage( int error_code, const QString &error_message )
{
  switch( error_code )
  {
    case ERROR_GETTING_BROWSELIST:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the browse list." ), error_message );
      break;
    case ERROR_GETTING_MEMBERS:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the list of workgroup members." ), error_message );
      break;
    case ERROR_GETTING_SHARES:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the list of shares." ), error_message );
      break;
    case ERROR_GETTING_PREVIEW:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the preview." ), error_message );
      break;
    case ERROR_ALREADY_MOUNTED:
      KMessageBox::sorry( 0, i18n( "This share has already been mounted." ) );
      break;
    case ERROR_UNMOUNTING_NOT_ALLOWED:
      KMessageBox::error( 0, i18n( "You are not allowed to unmount this share." ) );
      break;
    case ERROR_MOUNTING_SHARE:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to mount the share." ), error_message );
      break;
    case ERROR_UNMOUNTING_SHARE:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to unmount the share." ), error_message );
      break;
    case ERROR_UNMOUNTING_ALL:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to unmount all shares." ), error_message );
      break;
    case ERROR_FILE_NOT_FOUND:
        KMessageBox::error( 0, i18n( "The file %1 could not be found." ).arg( error_message ) );
      break;
    case ERROR_READING_FILE:
      KMessageBox::error( 0, i18n( "The file %1 could not be read." ).arg( error_message ) );
      break;
    case ERROR_UNKNOWN:
      KMessageBox::detailedError( 0, i18n( "An error occurred." ), error_message );
      break;
    case ERROR_GETTING_HOSTNAME:
      KMessageBox::error( 0, i18n( "The hostname could not be determined." ) );
      break;
    case ERROR_MIMETYPE_NOT_SUPPORTED:
      KMessageBox::error( 0, i18n( "The mimetype \"%1\" is not supported. Please convert the file to PostScript or PDF." ).arg( error_message ) );
      break;
    case ERROR_LOCKED:
      {
        QString file = error_message.section( ":", 1, 1 );
        QString user = error_message.section( ":", 0, 0 );
        KMessageBox::error( 0, i18n( "The file %1 is currently edited by the user %2. Please try again later." ).arg( file, user ) );
        break;
      }
    case ERROR_MKDIR_FAILED:
      KMessageBox::error( 0, i18n( "The directory %1 could not be created." ).arg( error_message ) );
      break;
    case ERROR_MISSING_PROGRAMS:
      KMessageBox::error( 0, i18n( "Either your PATH variable is not set correctly or there are the following programs missing on your system:\n%1\nPlease correct this and start Smb4K again." ).arg( error_message ) );
      break;
    case ERROR_SUID_PROGRAM_MISSING:
      KMessageBox::information( 0, i18n( "You previously chose to use %1, but now it is missing on your system. Smb4K will disable this feature." ).arg( error_message ) );
      break;
    case ERROR_WRITING_FILE:
      KMessageBox::error( 0, i18n( "The file %1 could not be written successfully." ).arg( error_message ) );
      break;
    case ERROR_GETTING_USAGE:
      KMessageBox::detailedError( 0, i18n( "The disk usage could not be determined." ), error_message );
      break;
    case ERROR_MOUNTPOINT_EMPTY:
      KMessageBox::error( 0, i18n( "This share could not be unmounted because the mountpoint string was empty." ) );
      break;
    case ERROR_WRONG_KERNEL:
      KMessageBox::error( 0, i18n( "Your kernel (%1) does not support this operation." ).arg( error_message ) );
      break;
    case ERROR_FEATURE_NOT_ENABLED:
      KMessageBox::error( 0, i18n( "This feature has not been enabled." ) );
      break;
    case ERROR_WRONG_BOOKMARK_TYPE:
      KMessageBox::error( 0, i18n( "Printers cannot be bookmarked." ) );
      break;
    case ERROR_IP_CANNOT_BE_USED:
      KMessageBox::sorry( 0, i18n( "The search method you chose (smbclient) can't handle IP addresses correctly. Please choose \"Use nmblookup\" in the configuration dialog and try again." ) );
      break;
    case ERROR_COMMAND_NOT_FOUND:
      KMessageBox::error( 0, i18n( "The command %1 could not be found." ).arg( error_message ) );
      break;
    case ERROR_OPENING_WALLET:
      KMessageBox::error( 0, i18n( "The wallet \"%1\" could not be opened." ).arg( error_message ) );
      break;
    case ERROR_PRINTING:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to print." ), error_message );
      break;
   case ERROR_CREATING_TEMP_DIR:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to create a temporary directory." ), error_message );
      break;
    case ERROR_CREATING_TEMP_FILE:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to create a temporary file." ), error_message );
      break;
    case ERROR_DIRECTORY_NOT_FOUND:
      KMessageBox::error( 0, i18n( "The directory %1 could not be found." ).arg( error_message ) );
      break;
    case ERROR_LOCK_FILE_IS_SYMLINK:
      KMessageBox::error( 0, i18n( "The lock file (%1) is a symlink. Since this might indicate that someone trys to exploit your system, please inform your system administrator. No action has been performed." ).arg( error_message ) );
      break;
    case ERROR_SHARE_WENT_OFFLINE:
      // FIXME.
      break;
    case ERROR_GETTING_PERMISSIONS:
      KMessageBox::detailedError( 0, i18n( "An error occurred while determining file permissions." ), error_message );
      break;
    default:
      break;
  }
}


void Smb4KCore::slotShutdown()
{
  m_mounter->exit();
//   m_fileIO->exit();

  config()->sync();
}


void Smb4KCore::slotGetSambaVersion()
{

  m_shellIO->getSambaVersion();
}


void Smb4KCore::slotReceiveSambaVersion( const QString &version )
{
  config()->setGroup( "Samba" );
  config()->writeEntry( "Version", version );

  m_samba_version = version;
}


void Smb4KCore::slotGetUsage()
{
  if ( !m_path.isEmpty() )
  {
    Smb4KShare *s = mounter()->findShareByPath( m_path );

    if ( s && !s->isBroken() )
    {
      QDir *dir = new QDir( m_path );

      QStringList dir_list = dir->entryList( QDir::Dirs, QDir::DefaultSort );
      int dirs = dir_list.count() > 0 ? dir_list.count()-2 : 0;

      QStringList file_list = dir->entryList( QDir::Files, QDir::DefaultSort );
      int files = file_list.count() > 0 ? file_list.count()-2 : 0;

      delete dir;

      struct statvfs fs;

      if ( statvfs( m_path, &fs ) == -1 )
      {
        int err_code = errno;

        m_path = QString::null;

        emit usage( m_path, -1, -1, -1, -1, -1, -1 );

        slotShowErrorMessage( ERROR_GETTING_USAGE, strerror( err_code ) );

        return;
      }

      double kB_block = (double)(fs.f_bsize / 1024);
      double total = (double)(fs.f_blocks*kB_block);
      double used = (double)((fs.f_blocks - fs.f_bfree)*kB_block);
      double free = (double)(fs.f_bfree*kB_block);
      double percent = used/total*100;

      emit usage( m_path, total, used, free, percent, dirs, files );
    }
    else
    {
      m_path = QString::null;

      emit usage( m_path, -1, -1, -1, -1, -1, -1 );
    }
  }
  else
  {
    m_path = QString::null; // This might be redundant ...

    emit usage( m_path, -1, -1, -1, -1, -1, -1 );
  }
}

#include "smb4kcore.moc"
