/***************************************************************************
    smb4kstarter.cpp  -  Stuff that's carried out on start-up.
                             -------------------
    begin                : Mit Apr 30 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>

// KDE includes.
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kmessagebox.h>

// system includes
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/stat.h>

// application specific includes
#include "smb4kstarter.h"
#include "../smb4k.h"
#include "smb4kdefs.h"

Smb4KStarter::Smb4KStarter( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_config = kapp->config();

  m_proc = new KProcess( this, "StarterProcess" );
  m_proc->setUseShell( true );
  
  modifyConfig();

  connect( m_proc, SIGNAL( processExited( KProcess * ) ),               this, SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
}


Smb4KStarter::~Smb4KStarter()
{
}


/****************************************************************************
   Takes care of everything that has to be done before the main window
   comes up and shows it afterwards.
****************************************************************************/


void Smb4KStarter::init()
{
  // Modify the configuration file in case some entries 
  // changed:
  modifyConfig();
  
  // Get the OS name:
  detectOS();

  // Get the UID and GID:
  getID();

  // Get the UMASK.
  getUMASK();

  // Does the host use a WINS server? Let's find out.
  checkForWINS();

  // Get all programs, that are installed on the system and
  // also Samba's version. Smb4KApp will be started after the
  // process ended.
  getProgramInfos();
}


/****************************************************************************
   Runs the program.
****************************************************************************/


void Smb4KStarter::runProgram()
{
  if ( kapp->isRestored() )
  {
    //RESTORE( Smb4KApp );
    (new Smb4KApp)->restore(1, false);
  }
  else
  {
    //Create a new application but without displaying it is
    //to be started docked in the system tray
    new Smb4KApp();

    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

    args->clear();
  }
}


/****************************************************************************
   Detect the operating system.
****************************************************************************/


void Smb4KStarter::detectOS()
{
  m_config->setGroup( "System" );

  // Initialize system info stuff.
  struct utsname system_info;
  uname( &system_info );

  m_config->writeEntry( "Name", QString( "%1" ).arg( system_info.sysname ) );

  // Make sure we only get the official version of the kernel:
  m_config->writeEntry( "Version", QString( "%1" ).arg( system_info.release ).section( "-", 0, 0 ) );
}


/****************************************************************************
   Determines whether there is a WINS server configured.
****************************************************************************/


void Smb4KStarter::checkForWINS()
{
  QString wins;

  QFile file( "smb.conf" );
  QDir *dir = new QDir();

  if ( QFile::exists( "/etc/smb.conf" ) )
  {
    dir->setPath( "/etc" );
    QDir::setCurrent( dir->path() );
  }
  else if ( QFile::exists( "/etc/samba/smb.conf" ) )
  {
    dir->setPath( "/etc/samba" );
    QDir::setCurrent( dir->path() );
  }
  else if ( QFile::exists( "/usr/local/etc/smb.conf" ) )
  {
    dir->setPath( "/usr/local/etc" );
    QDir::setCurrent( dir->path() );
  }
  else if ( QFile::exists( "/usr/local/etc/samba/smb.conf" ) )
  {
    dir->setPath( "/usr/local/etc/samba" );
    QDir::setCurrent( dir->path() );    
  }
  else
    error( ERROR_FILE_NOT_FOUND, file.name() );

  if ( !dir->path().isEmpty() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );
      while( !ts.atEnd() )
      {
        QString line = ts.readLine().stripWhiteSpace().upper();
        if ( line.startsWith( "WINS SERVER" ) )
        {
          wins = line.section( "=", 1, 1 ).stripWhiteSpace();
          break;
        }
        else if ( line.startsWith( "WINS SUPPORT" ) )
        {
          if ( line.section( "=", 1, 1 ).stripWhiteSpace() == "YES" || line.section( "=", 1, 1 ).stripWhiteSpace() == "TRUE" )
            wins = "127.0.0.1";
          break;
        }
      }
      file.close();
    }
    else
      error( ERROR_READING_FILE,  file.name() );  
   }

  // If there is a WINS server defined, put it into the config file
  // and enable the WINS support.
  if ( !wins.isEmpty() )
  {
    m_config->setGroup( "Browse Options" );
    m_config->writeEntry( "WINS Support", true );
    m_config->writeEntry( "WINS Server", wins );

    m_config->sync();
  }
  else
  {
    m_config->setGroup( "Browse Options" );
    m_config->writeEntry( "WINS Support", false );
    if ( m_config->hasKey( "WINS Server" ) );
      m_config->deleteEntry( "WINS Server" );
    m_config->sync();
  }
}


/****************************************************************************
   Determines the UID and GID of the user and writes them into the
   config file.
****************************************************************************/

void Smb4KStarter::getID()
{
  m_uid = getuid();
  m_gid = getgid();

  m_config->setGroup( "System" );
  m_config->writeEntry( "UID", m_uid );
  m_config->writeEntry( "GID", m_gid );
}


/****************************************************************************
   Starts the checking for mandatory programs and the Samba version.
****************************************************************************/

void Smb4KStarter::getProgramInfos()
{
  QStringList pathList = QStringList::split( ":", QString( "%1" ).arg( getenv( "PATH" ) ), false );

  // Detect all mandatory programs that are needed by Smb4K.  
  QStringList fileList;
  fileList << "grep";
  fileList << "awk";
  fileList << "sed";
  fileList << "xargs";
  fileList << "rm";
  fileList << "ls";
  fileList << "nmblookup";
  fileList << "smbclient";
#ifndef __FreeBSD__
  fileList << "smbmount";
  fileList << "smbumount";
#else
  fileList << "mount_smbfs";
#endif  
  // Detect the optional programs.
  fileList << "super";
  fileList << "dvips";
  fileList << "enscript";
  
  for ( QStringList::ConstIterator it = pathList.begin(); it != pathList.end(); ++it )
  {
    QDir::setCurrent( *it );
    
    for ( QStringList::Iterator bin = fileList.begin(); bin != fileList.end(); ++bin )
    {
      if ( QFile::exists( *bin ) )
        *bin = QString::null;
      else
        continue;
    }
  }
  
  fileList.remove( QString::null );

  if ( fileList.count() != 0 )
  {
    if ( fileList.find( "super" ) != fileList.end() )
    {
      fileList.remove( fileList.find( "super" ) );
      
      m_config->setGroup( "Mount Options" );
      m_config->writeEntry( "Super", false );
      
      if ( m_config->readBoolEntry( "Use SU Privileges", false ) || m_config->readBoolEntry( "Force Unmount", false ) )
      {
        error( ERROR_SUPER_MISSING, QString::null );
        m_config->writeEntry( "Use SU Privileges", false );
        m_config->writeEntry( "Force Unmount", false );
      }
    }
    else
    {
      m_config->setGroup( "Mount Options" );
      m_config->writeEntry( "Super", true );    
    }
    
    if ( fileList.find( "dvips" ) != fileList.end() )
    {
      fileList.remove( fileList.find( "dvips" ) );
      
      m_config->setGroup( "Printing" );
      m_config->writeEntry( "DVIPS", false );
    }
    else
    {
      m_config->setGroup( "Printing" );
      m_config->writeEntry( "DVIPS", true );
    }
    
    if ( fileList.find( "enscript" ) != fileList.end() )
    {
      fileList.remove( fileList.find( "enscript" ) );
      
      m_config->setGroup( "Printing" );
      m_config->writeEntry( "Enscript", false );
    }
    else
    {
      m_config->setGroup( "Printing" );
      m_config->writeEntry( "Enscript", true );
    }
    
    if ( fileList.count() != 0 )
    {    
      fileList.sort();
      error( ERROR_MISSING_PROGRAMS, fileList.join( ", " ) );
      return;
    }
  }
  else
  {
    m_config->setGroup( "Mount Options" );
    m_config->writeEntry( "Super", true );
    
    m_config->setGroup( "Printing" );
    m_config->writeEntry( "DVIPS", true );
    m_config->writeEntry( "Enscript", true );
  }
  
  // Get program versions  
  *m_proc << "smbclient -V";
  m_proc->start( KProcess::NotifyOnExit, KProcess::Stdout );    
}


/****************************************************************************
  Get the umask that is defined for this user.
****************************************************************************/

void Smb4KStarter::getUMASK()
{
  int mask = umask( 0 );

  m_config->setGroup( "System" );
  m_config->writeEntry( "UMASK", QString( "%1" ).arg( mask ) );
}


/****************************************************************************
  Modify the config file, if something changed in it.
****************************************************************************/

void Smb4KStarter::modifyConfig()
{
  // The following entries can be removed with 
  // version 0.5.0 or later.
  m_config->setGroup( "System" );
  if ( m_config->hasKey( "Super" ) )
    m_config->deleteEntry( "Super" );
    
  m_config->setGroup( "Appearance" );
  if ( m_config->hasKey( "Display Shares" ) )
  {
    m_config->writeEntry( "Show Shares", m_config->readEntry( "Display Shares" ) );
    m_config->deleteEntry( "Display Shares" );
  }
}


/****************************************************************************
   Displays an error message.
****************************************************************************/

void Smb4KStarter::error( int error_code, const QString &error_message )
{
  // In case something is to be changed here, check whether something 
  // has to be changed in Smb4KCore::slotShowErrorMessage(), too.
  switch( error_code )
  {
    case ERROR_FILE_NOT_FOUND:
      KMessageBox::error( (QWidget *)this, QString( i18n( "Could not find file %1!" ) ).arg( error_message ) );
      break;
    case ERROR_READING_FILE:
      KMessageBox::error( (QWidget *)this, QString( i18n( "Could not read from file %1!" ) ).arg( error_message ) );
      break;
    case ERROR_MISSING_PROGRAMS:
      KMessageBox::error( (QWidget *)this, QString( i18n( "Either your PATH variable is not set correctly or there are the following programs missing on your system:\n%1\nPlease correct this and start Smb4K again." ) ).arg( error_message ) );
      break;
    case ERROR_SUPER_MISSING:
      KMessageBox::information( (QWidget *)this, i18n( "You previously chose to use the program super, but now it is missing on your system. Smb4K will disable this feature!" ) );
      break;
    default:
      break;
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KStarter::slotProcessExited( KProcess * )
{
  m_proc->clearArguments();

  QStringList output = QStringList::split( "\n", m_buffer, false );

  m_config->setGroup( "System" );
  m_config->writeEntry( "Samba", output.grep( "Version" ).first().section( " ", 1, -1 ) );

  runProgram();
}


void Smb4KStarter::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}

#include "smb4kstarter.moc"

