/***************************************************************************
                          smb4kcore  -  description
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 
// KDE includes
#include <klocale.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <krun.h>

// application specific includes
#include "smb4kcore.h"
#include "smb4kdefs.h"


Smb4KCore::Smb4KCore( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_reader = new Smb4KPasswdReader( (QWidget *)parent, "Core_PasswdReader" );
  m_scanner = new Smb4KScanner( this, "Core_Scanner" );
  m_mounter = new Smb4KMounter( this, "Core_Mounter" );
  m_bookmarkHandler = new Smb4KBookmarkHandler( this, "Core_BookmarkHandler" );
  m_fileIO = new Smb4KFileIO( this, "Core_FileIO" );
  m_finder = new Smb4KIPFinder( this, "Core_IPFinder" );
  m_print = new Smb4KPrint( this, "Core_PrinterHandler" );
  m_shellIO = new Smb4KShellIO( this, "Core_ShellIO" );
  
  m_scanner_state = SCANNER_STOP;
  m_scanner_running = false;
  
  m_mounter_state = MOUNTER_STOP;
  m_mounter_running = false;
  
  m_ipfinder_state = IPFINDER_STOP;
  m_ipfinder_running = false;
  
  m_print_state = PRINT_STOP;
  m_print_running = false;
  
  // Connections:
  connect( m_scanner, SIGNAL( running( int, bool ) ), this, SLOT( slotScannerIsRunning( int, bool ) ) );
  connect( m_mounter, SIGNAL( running( int, bool ) ), this, SLOT( slotMounterIsRunning( int, bool ) ) );
  connect( m_finder,  SIGNAL( running( int, bool ) ), this, SLOT( slotIPFinderIsRunning( int, bool ) ) );
  connect( m_print,   SIGNAL( running( int, bool ) ), this, SLOT( slotPrinterHandlerIsRunning( int, bool ) ) );
  
  connect( m_scanner, SIGNAL( error( int, const QString & ) ), this, SLOT( slotShowErrorMessage( int, const QString & ) ) );  
  connect( m_mounter, SIGNAL( error( int, const QString & ) ), this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_fileIO,  SIGNAL( error( int, const QString & ) ), this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
  connect( m_print,   SIGNAL( error( int, const QString & ) ), this, SLOT( slotShowErrorMessage( int, const QString & ) ) );
}


Smb4KCore::~Smb4KCore()
{
  // Do not call abort() here. This will lead
  // to crashes.
}


/****************************************************************************
   Returns a bool that tells the program whether a core process is running.
****************************************************************************/

bool Smb4KCore::isRunning()
{
  if ( m_scanner_running || m_mounter_running || m_ipfinder_running || m_print_running )
    return true;
  else
    return false;
}


/****************************************************************************
   Returns the current state of the core.
****************************************************************************/

void Smb4KCore::setCurrentState( int state )
{
  if ( state != SCANNER_STOP && state != MOUNTER_STOP && state != IPFINDER_STOP && state != PRINT_STOP )
  {
    m_current_state = state;
  }
  else
  {
    if ( !m_scanner_running && !m_mounter_running && !m_ipfinder_running && !m_print_running )
    {
      m_current_state = CORE_STOP;
    }
    else
    {
      if ( m_ipfinder_running )
        m_current_state = m_ipfinder_state;
      else if ( m_scanner_running )
        m_current_state = m_scanner_state;
      else if ( m_print_running )
        m_current_state = m_print_state;
      else if ( m_mounter_running )
        m_current_state = m_mounter_state;
    }
  }
}


/****************************************************************************
   Reads the options for all core classes.
****************************************************************************/

void Smb4KCore::readOptions()
{
  m_scanner->readOptions();
  m_mounter->readOptions();
  m_finder->readOptions();
  m_print->readOptions();
}


/****************************************************************************
   Aborts any process of the core.
****************************************************************************/

void Smb4KCore::abort()
{
  m_scanner->abort();
  m_mounter->abort();
  m_finder->abort();
  m_print->abort();
}


/****************************************************************************
   Opens the given URL.
****************************************************************************/

void Smb4KCore::open( const QString &url )
{
  new KRun( KURL( url ), 0, true, true );
}

/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KCore::slotScannerIsRunning( int state, bool on )
{
  m_scanner_state = state;
  m_scanner_running = on; 
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotMounterIsRunning( int state, bool on )
{
  m_mounter_state = state;
  m_mounter_running = on;
  setCurrentState( state );  
  emit runStateChanged();
}


void Smb4KCore::slotIPFinderIsRunning( int state, bool on )
{
  m_ipfinder_state = state;
  m_ipfinder_running = on;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotPrinterHandlerIsRunning( int state, bool on )
{
  m_print_state = state;
  m_print_running = on;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotShowErrorMessage( int error_code, const QString &error_message )
{
  switch( error_code )
  {
    case ERROR_GETTING_BROWSELIST:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to get the browse list!" ), error_message );
      break;
    case ERROR_GETTING_MEMBERS:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to get the list of workgroup members!" ), error_message );
      break;
    case ERROR_GETTING_SHARES:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to get the list of shares!" ), error_message );
      break;
    case ERROR_GETTING_PREVIEW:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to get the preview!" ), error_message );
      break;
    case ERROR_GETTING_INFO:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to get additional information from the host!" ), error_message );
      break;
    case ERROR_ALREADY_MOUNTED:
      KMessageBox::sorry( (QWidget *)this, i18n( "This share has already been mounted!" ) );
      break;
    case ERROR_UNMOUNTING_NOT_ALLOWED:
      KMessageBox::error( (QWidget *)this, i18n( "You are not allowed to unmount this share!" ) );
      break;
    case ERROR_MOUNTING_SHARE:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to mount the share!" ), error_message );
      break;
    case ERROR_UNMOUNTING_SHARE:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to unmount the share!" ), error_message );
      break;
    case ERROR_UNMOUNTING_ALL:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred while trying to unmount all shares!" ), error_message );
      break;
    case ERROR_FILE_NOT_FOUND:
        KMessageBox::error( (QWidget *)this, QString( i18n( "Could not find file %1!" ) ).arg( error_message ) );
      break;
    case ERROR_READING_FILE:
      KMessageBox::error( (QWidget *)this, QString( i18n( "Could not read from file %1!" ) ).arg( error_message ) );
      break;
    case ERROR_UNKNOWN:
      KMessageBox::detailedError( (QWidget *)this, i18n( "An error occurred!" ), error_message );
      break;
    case ERROR_GETTING_HOSTNAME:
      KMessageBox::error( (QWidget *)this, i18n( "Could not get hostname!" ) );
      break;
    case ERROR_MIMETYPE_NOT_SUPPORTED:
      KMessageBox::error( (QWidget *)this, QString( i18n( "The mimetype \"%1\" is not supported! Please convert the file to PostScript or PDF." ) ).arg( error_message ) );
      break;
    case ERROR_LOCKED:
      KMessageBox::error( (QWidget *)this, i18n( "Another user is currently editing one of the system files, that Smb4K wants to write to. Please try again later." ) );
      break;
    default:
      break;
  }
}


#include "smb4kcore.moc"
