/***************************************************************************
    smb4khost  -  Smb4K's container class for information about a host.
                             -------------------
    begin                : Sa Jan 26 2008
    copyright            : (C) 2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KHOST_H
#define SMB4KHOST_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <QString>

// KDE includes
#include <kdemacros.h>

/**
 * This class is a container that carries information about a host found in
 * the network neighborhood. It is part of the core classes of Smb4K.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class KDE_EXPORT Smb4KHost
{
  public:
    /**
     * The default constructor. It takes the name of the host as only argument.
     * You have to set all other information with the other functions provided
     * by this class.
     *
     * @param name                The name of the host
     */
    Smb4KHost( const QString &name );

    /**
     * The copy constructor. This constructor takes a Smb4KHost item as argument
     * and copies its values.
     *
     * @param host                The Smb4KHost object that is to be copied.
     */
    Smb4KHost( const Smb4KHost &host );

    /**
     * The empty constructor. It does not take any argument and you have to set
     * all information by the other functions provided by this class.
     */
    Smb4KHost();

    /**
     * The destructor.
     */
    ~Smb4KHost();

    /**
     * Set the name of the host.
     *
     * @param name                The name of the host
     */
    void setName( const QString &name );

    /**
     * Returns the name of the host.
     *
     * @returns the host's name.
     */
    const QString &name() const { return m_name; }

    /**
     * Set the workgroup where this host is located.
     *
     * @param workgroup           The workgroup name
     */
    void setWorkgroup( const QString &workgroup );

    /**
     * Returns the name of the workgroup where this host is located.
     *
     * @returns the workgroup name.
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Set the IP address of this host. @p ip will only be accepted
     * if it is compatible with either IPv4 or IPv6.
     *
     * When this function is invoked, it sets the m_ip_checked property
     * to TRUE so that ipChecked() retruns TRUE.
     *
     * @param ip                  The IP address of this host.
     */
    void setIP( const QString &ip );

    /**
     * Returns the IP address of the host. If the IP address was not
     * compatible with IPv4 and IPv6 or if no IP address was supplied,
     * an empty string is returned.
     *
     * @returns the host's IP address or an empty string.
     */
    const QString &ip() const { return m_ip; }

    /**
     * Returns TRUE if the IP address has already been checked (i.e. set)
     * and false otherwise. It does not matter if actually there is an
     * IP address present. The only thing that is important is that
     * setIP() was called.
     *
     * @returns TRUE if the IP address has been checked.
     */
    bool ipChecked() const { return m_ip_checked; }

    /**
     * Set the comment that was defined for the host.
     *
     * @param comment             The comment string
     */
    void setComment( const QString &comment );

    /**
     * Returns the comment that was defined or an empty string if there
     * was no comment.
     *
     * @returns the comment or an empty string.
     */
    const QString &comment() const { return m_comment; }

    /**
     * Set the "Server" and the "OS" (operating system) strings as
     * they are reported by the host.
     *
     * When this function is invoked, it will set the m_info_checked
     * property to TRUE, so that infoChecked() returns TRUE.
     *
     * @param serverString        The "Server" string
     *
     * @param osString            The "OS" string
     */
    void setInfo( const QString &serverString = QString(),
                  const QString &osString = QString() );

    /**
     * With this function you can manually reset the "info is checked" flag,
     * i.e. it will be set to FALSE. In addition the server string and the
     * operating system string will be cleared.
     */
    void resetInfo();

    /**
     * Returns TRUE if the infomation has already been checked (i.e. set)
     * and FALSE otherwise. It does not matter if actually there is some
     * information present. The only thing that is important is that
     * setInfo() was called.
     *
     * @returns TRUE if the infomation has been checked.
     */
    bool infoChecked() const { return m_info_checked; }

    /**
     * Returns the "Server" string as reported by the host.
     *
     * @returns the "Server" string.
     */
    const QString &serverString() const { return m_server_string; }

    /**
     * Returns the "OS" (operating system) string as reported by the
     * host.
     *
     * @returns the OS string.
     */
    const QString &osString() const { return m_os_string; }

    /**
     * Set this host to be a master browser.
     *
     * @param master              Set this to TRUE if the host is a master
     *                            browser.
     */
    void setIsMasterBrowser( bool master );

    /**
     * Returns TRUE if the host is a master browser and FALSE otherwise.
     *
     * @returns TRUE if the host is a master browser.
     */
    bool isMasterBrowser() const { return m_is_master; }

    /**
     * Enumeration for the protocol that should be used with this host.
     */
    enum Protocol{ Automatic,
                   RPC,
                   RAP,
                   ADS };

    /**
     * Set the protocol that should be used with this host. The default
     * is "Automatic".
     *
     * @param protocol            The protocol that should be used.
     */
    void setProtocol( Smb4KHost::Protocol protocol );

    /**
     * Returns the protocol that should be used with this host. The default
     * is "Automatic".
     *
     * @returns the protocol that should be used with this host.
     */
    Smb4KHost::Protocol protocol() const { return m_protocol; }

    /**
     * Returns TRUE if the item is empty and FALSE otherwise. An item is not
     * empty if at least one string (workgroup name, master name, etc.) has been
     * set. A modified boolean will not be considered.
     *
     * @returns TRUE if the item is empty.
     */
    bool isEmpty() const;

  private:
    /**
     * The host's name
     */
    QString m_name;

    /**
     * The workgroup the host is in
     */
    QString m_workgroup;

    /**
     * The host's IP address
     */
    QString m_ip;

    /**
     * The comment
     */
    QString m_comment;

    /**
     * The server string
     */
    QString m_server_string;

    /**
     * The OS string
     */
    QString m_os_string;

    /**
     * Have we checked for info yet?
     */
    bool m_info_checked;

    /**
     * Have we already checked for the IP address?
     */
    bool m_ip_checked;

    /**
     * Determines if the host is a master browser
     */
    bool m_is_master;

    /**
     * The protocol that should be used
     */
    Protocol m_protocol;

    /**
     * This function checks if the given IP address is either
     * compatible with IPv4 or IPv6. If it is not, an empty string
     * is returned.
     *
     * @param ip              The IP address that needs to be checked.
     *
     * @returns the IP address or an empty string if the IP address
     * is not compatible with either IPv4 or IPv6.
     */
    const QString &ipIsValid( const QString &ip );
};

#endif
