/***************************************************************************
    smb4kauthinfo.h  -  This class provides a container for the
    authentication data.
                             -------------------
    begin                : Sa Feb 28 2004
    copyright            : (C) 2004-2009 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KAUTHINFO_H
#define SMB4KAUTHINFO_H

// Qt includes
#include <QString>
#include <QByteArray>

// KDE includes
#include <kdemacros.h>

// application specific includes
#include <smb4khost.h>
#include <smb4kshare.h>

/**
 * This class provides a container for the authentication data.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */


class KDE_EXPORT Smb4KAuthInfo
{
  public:
    /**
     * Enumeration that determines the type of network item the
     * authentication information is for.
     */
    enum Type { Host,
                Share,
                Default,
                Unknown };

    /**
     * Constructor for a host item.
     *
     * @param host      The Smb4KHost item.
     */
    Smb4KAuthInfo( const Smb4KHost *host );


    /**
     * Constructor for a share item.
     *
     * @param share     The Smb4KShare item.
     */
    Smb4KAuthInfo( const Smb4KShare *share );

    /**
     * The empty constructor.
     */
    Smb4KAuthInfo();

    /**
     * The copy constructor.
     *
     * @param info      The Smb4KAuthInfo object that will be copied.
     */
    Smb4KAuthInfo( Smb4KAuthInfo &info );

    /**
     * The destructor
     */
    ~Smb4KAuthInfo();

    /**
     * Set the host item. This overwrites all previous data that this object
     * might have carried including the password.
     *
     * @param host      The Smb4KHost item
     */
    void setHost( Smb4KHost *host );

    /**
     * Set the share item. This overwrites all previous data that this object
     * might have carried including the password.
     *
     * @param share     The Smb4KShare item
     */
    void setShare( Smb4KShare *share );

    /**
     * Sets the workgroup name. This function should only be used if you neither can
     * use setHost() nor setShare().
     *
     * @param workgroup The name of the workgroup
     */
    void setWorkgroupName( const QString &workgroup );

    /**
     * Returns the name of the workgroup.
     *
     * @returns         The workgroup of the server/share for which this
     *                  authentication data is for.
     */
    const QString &workgroupName() const { return m_workgroup; }

    /**
     * This enumeration is used to determine how a UNC should be formatted. It is
     * used by several core classes.
     *
     * @param UNCMinimal      Always gives //HOST/SHARE
     *
     * @param UNCWithLogin    Inserts the user into the UNC (e.g. //USER@HOST/SHARE)
     *
     * @param UNCHostOnly     Removes the share name from the UNC if present
     *
     * @param UNCWithProtocol Prepends the protocol to the UNC, e.g. (smb://USER@HOST/SHARE)
     */
    enum UNCFormat { UNCMinimal = 0x00001,
                     UNCWithLogin = 0x00010,
                     UNCHostOnly = 0x00100,
                     UNCWithProtocol = 0x01000 };

    /**
     * Sets the UNC of this item. This function should only be used if you neither
     * can use setHost() nor setShare().
     *
     * The UNC may contain the protocol, i.e. "smb://". If a wrong protocol is passed,
     * this function will return immediately without doing anything.
     *
     * @param unc       The UNC of the item
     */
    void setUNC( const QString &unc );

    /**
     * Returns the UNC in the form //[USER@]HOST/SHARE depending on
     * the format specified by @p format. In case one of the components is
     * missing something that is needed to assemble the requested UNC format,
     * it is omitted.
     *
     * You can OR several of the format parameters together. If you use the
     * HostOnly parameter, the UNC will never contain a share name, and using
     * UNCMinimal will result in the shortest UNC possible.
     *
     * @returns the UNC.
     */
    QString unc( int format = UNCMinimal ) const;

    /**
     * Returns the host name.
     *
     * @returns the host name.
     */
    const QString &hostName() const { return m_host_name; }

    /**
     * Returns the share name.
     *
     * @returns the share name.
     */
    const QString &shareName() const { return m_share_name; }

    /**
     * Sets the login.
     *
     * @param login     The login for the server/share
     */
    void setLogin( const QString &login );

    /**
     * Returns the login name.
     *
     * @returns         The lo,gin
     */
    const QByteArray &login() const { return m_login; }

    /**
     * Sets the password.
     *
     * @param passwd    The password for the server/share
     */
    void setPassword( const QString &passwd );

    /**
     * Returns the (unescaped) password.
     */
    const QByteArray &password() const { return m_password; }

    /**
     * Returns the type.
     *
     * @returns the type.
     */
    Type type() const { return m_type; }

    /**
     * If the item is a homes share, this function returns TRUE. In
     * all other cases, this function returns FALSE.
     *
     * @returns TRUE if the item is a homes share.
     */
    bool isHomesShare() const { return m_homes_share; }

    /**
     * This function is needed to change a 'homes' share to a 'user' share.
     * It does only something if this item is of type 'Share' and is a
     * 'homes' share.
     *
     * @param user      The user name that is replacing the 'homes' string
     */
    void setHomesUser( const QString &user );

    /**
     * This function sets the type of this authentication information to
     * "Default", i.e. it carries the default authentication information.
     */
    void setDefaultAuthInfo();

    /**
     * Set the list of defined users in case this is a homes share.
     *
     * Note that this function will only set the list if this indead is a
     * homes share. It will just return otherwise.
     *
     * @param users           The list of defined 'homes' share users.
     */
    void setHomesUsers( const QStringList &users );

    /**
     * Return the list of users that were defined for a homes share. If this
     * authentication information represents a host or a share, that is not a
     * 'homes' share, an empty list will be returned.
     *
     * @returns a list of users for a homes share.
     */
    const QStringList &homesUsers() const { return m_homes_users; }

  private:
    /**
     * The type
     */
    Type m_type;

    /**
     * The workgroup
     */
    QString m_workgroup;

    /**
     * The host name
     */
    QString m_host_name;

    /**
     * The share name
     */
    QString m_share_name;

    /**
     * Is this a homes share
     */
    bool m_homes_share;

    /**
     * The user name for this share.
     */
    QByteArray m_login;

    /**
     * The password object for this share.
     */
    QByteArray m_password;

    /**
     * List of users defined for a homes share.
     */
    QStringList m_homes_users;
};

#endif
