-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
deferred class PRECURSOR_CALL
	--
	-- Handling of the `Precursor' construct. Common ancestor of
	-- PRECURSOR_CALL_EXPRESSION (when `Precursor' is inside a function call) and
	-- PRECURSOR_CALL_INSTRUCTION (when `Precursor' is inside a procedure call).
	--

insert
	GLOBALS

feature {ANY}
	start_position: POSITION -- Of the "Precursor" keyword.

	arguments: EFFECTIVE_ARG_LIST

	specialized_anonymous_feature: ANONYMOUS_FEATURE
			-- The specialized version of the precursor feature

	specialized_parent: CLASS_TEXT
			-- The parent the feature effectively comes from

	parent: TYPE_MARK
			-- Used when only when the call looks like: {TYPE_MARK} Precursor...
			-- The goal of this extra {TYPE_MARK} is to remove a possible ambiguity in case of multiple 
			-- inheritance.

	frozen use_current (type: TYPE): BOOLEAN is
		do
			-- Could be actually better.
			Result := True
		end

	frozen safety_check (type: TYPE) is
		do
			if arguments /= Void then
				arguments.safety_check(type)
			end
		end

	frozen compile_to_c (type: TYPE) is
		do
			cpp.push_precursor(type, run_feature, arguments)
			run_feature.mapping_c
			cpp.pop
		end

	frozen compile_to_jvm (type: TYPE) is
		do
			jvm.push_precursor(type, run_feature, arguments)
			run_feature.mapping_jvm
			jvm.pop
		end

	frozen specialize_in (new_type: TYPE): like Current is
		local
			saf: like specialized_anonymous_feature; arg: like arguments
		do
			--|*** error_handler.append("Precursor routine not found.")
			--|*** error_handler.append("Precursor construct not implemented for external features (sorry).")
			--|*** PH(14/04/04): assignment handler should be warned!
			saf := feature_accumulator.current_mixer.build_precursor(Current, new_type)
			if arguments /= Void then
				arg := arguments.specialize_in(new_type)
			end
			smart_eiffel.argument_count_check(start_position, saf, arg)
			Result := current_or_twin_init(saf, arg)
		end

	frozen specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			saf: like specialized_anonymous_feature; arg: like arguments
		do
			check
				specialized_anonymous_feature /= Void
			end
			saf := feature_accumulator.current_mixer.specialize_precursor(specialized_anonymous_feature, parent_type, parent_edge, new_type)
			if arguments /= Void then
				arg := arguments.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(saf, arg)
		end

	frozen has_been_specialized: BOOLEAN is
		do
			Result := arguments /= Void implies arguments.has_been_specialized
			Result := Result and then specialized_anonymous_feature /= Void
			Result := Result and then specialized_parent /= Void
		end

	adapt_for (t: TYPE): like Current is
		local
			arg: like arguments; rf: RUN_FEATURE
		do
			if arguments /= Void then
				arg := arguments.adapt_for(t)
			end
			rf := t.live_type.precursor_run_feature(specialized_anonymous_feature)
			check
				rf.type_of_current = t
			end
			if arg /= arguments or else run_feature /= Void and then run_feature /= rf then
				Result := twin
				Result.set_arguments(arg)
			else
				Result := Current
			end
			Result.set_run_feature(rf)
		end

feature {ANONYMOUS_FEATURE_MIXER}
	set_specialized_parent (sp: like specialized_parent) is
		require
			sp /= Void
			parent /= Void implies parent.class_text = sp
			specialized_parent /= Void implies specialized_parent = sp
		do
			specialized_parent := sp
		ensure
			specialized_parent = sp
		end

feature {PRECURSOR_CALL}
	set_arguments (a: like arguments) is
		do
			arguments := a
		end

	set_run_feature (rf: like run_feature) is
		do
			run_feature := rf
		end

	init (af: like specialized_anonymous_feature; arg: like arguments) is
		do
			specialized_anonymous_feature := af
			arguments := arg
		end

feature {}
	run_feature: RUN_FEATURE
			-- Corresponding super feature.

	make (sp: like start_position; pc: like parent; pal: like arguments) is
		require
			not sp.is_unknown
		do
			start_position := sp
			parent := pc
			arguments := pal
		ensure
			start_position = sp
			parent = pc
			arguments = pal
		end

	current_or_twin_init (af: like specialized_anonymous_feature; a: like arguments): like Current is
		require
			af /= Void
			a /= Void = (arguments /= Void)
		do
			if af = specialized_anonymous_feature and then a = arguments then
				Result := Current
			else
				Result := twin
				Result.init(af, a)
			end
		ensure
			Result.specialized_anonymous_feature = af
			Result.arguments = a
		end

	frozen pretty_ (indent_level: INTEGER) is
		local
			buffer: STRING
		do
			pretty_printer.put_string(once "Precursor")
			if parent /= Void then
				pretty_printer.put_character(' ')
				pretty_printer.put_character('{')
				buffer := once "......................."
				buffer.clear_count
				parent.pretty_in(buffer)
				pretty_printer.put_string(buffer)
				pretty_printer.put_character('}')
				pretty_printer.put_character(' ')
			end
			if arguments /= Void then
				arguments.pretty(indent_level + 1)
			end
		end

invariant
	parent /= Void and specialized_parent /= Void implies parent.class_text = specialized_parent

end -- class PRECURSOR_CALL
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
