-- See the Copyright notice at the end of this file.
--
expanded class XML_DTD_PUBLIC_REPOSITORY
	--
	-- This class allows you to register public DTD files as local files (useful when the network is not
	-- available, or for performance reasons, or for any other reason).
	--

feature {ANY}
	register (public_id, local_path: STRING) is
			-- Register the public DTD `public_id' as having a local couterpart file in the given `local_path'.
		require
			valid_public_id: not public_id.is_empty
			valid_local_path: not local_path.is_empty
			not_registered: not is_registered(public_id)
			file_exists: (create {FILE_TOOLS}).file_exists(local_path)
		do
			-- We twin the strings to be sure the client cannot modify the registered version (the hashed
			-- dictionary would not like that)
			dtd.put(local_path.twin, public_id.twin)
		ensure
			registered: is_registered(public_id)
		end

	is_registered (public_id: STRING): BOOLEAN is
			-- Does the given `public_id' have a local counterpart file?
		require
			valid_public_id: not public_id.is_empty
		do
			Result := dtd.has(public_id)
		end

feature {XML_DTD_PARSER}
	public_dtd (public_id, url: STRING): INPUT_STREAM is
			-- Opens the given `public_id' either by network connection to the `url' or, if the `public_id'
			-- `is_registered', by reading the local file.
		require
			not public_id.is_empty
			not is_registered(public_id) implies not url.is_empty
		do
			last_error_memory.set_item(Void)
			if is_registered(public_id) then
				Result := local_dtd(public_id)
			else
				Result := remote_dtd(url)
			end
		ensure
			connected_or_void_stream: Result /= Void implies Result.is_connected
			not_connected_is_error: Result = Void implies last_error /= Void
		end

	free (public_id: STRING; stream: INPUT_STREAM) is
			-- Disconnect from the stream, which must have been opend by `public_dtd'.
		local
			tfr: TEXT_FILE_READ
		do
			if stream.is_connected then
				stream.disconnect
			end
			if is_registered(public_id) then
				tfr ::= stream
				free_file(tfr)
			end
		end

	last_error: STRING is
		do
			Result := last_error_memory.item
		end

feature {}
	dtd: DICTIONARY[STRING, STRING] is
			-- The registered URLs
		once
			create {HASHED_DICTIONARY[STRING, STRING]} Result.make
		end

	last_error_memory: REFERENCE[STRING] is
		once
			create Result
		end

feature {} -- Local DTD
	local_dtd (public_id: STRING): TEXT_FILE_READ is
		require
			is_registered(public_id)
		local
			path: STRING
		do
			path := dtd.at(public_id)
			Result := connect_to_file(path)
		end

	free_files: WEAK_ARRAY[TEXT_FILE_READ] is
		once
			create Result.make
		end

	connect_to_file (path: STRING): TEXT_FILE_READ is
		do
			Result := free_files.item
			if Result = Void then
				create Result.connect_to(path)
			else
				check
					not Result.is_connected
				end
				Result.connect_to(path)
			end
			if not Result.is_connected then
				free_file(Result)
				Result := Void
			end
		ensure
			Result /= Void implies Result.is_connected
		end

	free_file (file: TEXT_FILE_READ) is
		require
			not file.is_connected
		do
			free_files.recycle(file)
		end

feature {} -- Network DTD
	remote_dtd (a_url: STRING): INPUT_STREAM is
		require
			not is_registered(a_url)
		local
			url: URL
		do
			create url.set_url(a_url) --|*** memory leak
			url.set_error_handler(agent_error)
			url.connect
			if url.is_connected then
				Result := url.input
			end
		ensure
			Result /= Void implies Result.is_connected
		end

	agent_error: PROCEDURE[TUPLE[STRING]] is
		once
			Result := agent set_error
		end

	set_error (a_error: STRING) is
		do
			last_error_memory.set_item(a_error)
		end

end -- class XML_DTD_PUBLIC_REPOSITORY
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
