/*
 * Perform an ALPS inventory and print out node attributes, divided into NUMA/total
 * sections. Nodes are printed in the currend ALPS_NIDORDER, which corresponds to
 * 'apstat -no' and 'apbasil -Q'.
 */
#include "basil_alps.h"
#include "basil_torus.h"

/* Print recursively to revert the LIFO order */
static void print_numa_node(struct basil_node *node, bool is_gemini)
{
	struct basil_segment *seg;
	uint8_t numa_segments = 0,
		numa_cores = 0;
	uint32_t numa_mem = 0,
		 cmhz = 0;
	enum basil_proc_type arch = BPT_NONE;

	if (!node)
		return;
	print_numa_node(node->next, is_gemini);

	/* FIXME: assuming uniform system for now */
	for (seg = node->seg_head; seg; seg = seg->next, numa_segments++) {
		struct basil_node_processor *proc;
		struct basil_node_memory *mem;
		uint8_t procs = 0;

		for (proc = seg->proc_head; proc; proc = proc->next, procs++) {
			if (arch != BPT_NONE && arch != proc->arch)
				errx(1, "nid%05u processor %d architecture '%d'"
					"does not match '%d'", node->node_id,
					proc->ordinal, proc->arch, arch);
			arch = proc->arch;

			if (cmhz && cmhz != proc->clock_mhz)
				errx(1, "nid%05u processor %d frequency '%d'"
					"does not match '%d'", node->node_id,
					proc->ordinal, proc->clock_mhz, cmhz);
			cmhz = proc->clock_mhz;
		}

		if (numa_cores && numa_cores != procs)
			warnx("nid%05u segment %d has %d cores instead of %d",
				node->node_id, seg->ordinal, procs, numa_cores);
		numa_cores = procs;

		for (mem = seg->mem_head; mem; mem = mem->next) {
			uint32_t seg_mem = mem->page_size_kb * mem->page_count;

			if (numa_mem && numa_mem != seg_mem)
				warnx("nid%05u segment %d has %d Mb instead of %d",
				      node->node_id, seg->ordinal, seg_mem >> 10,
				      numa_mem >> 10);
			numa_mem = seg_mem;
		}
	}

	printf("%5u %12.12s ", node->node_id, node->name);

	if (is_gemini) {
		const char *gemini = node_string2gemini_string(node->name);

		printf("%12.12s", gemini ? : "???");
	}
	printf("  %2.2s  %1.1s   %2u  %5u %6uM   %5u %6uM %9d\n",
		node->state == BNS_UP ? "UP" : "DN", nam_noderole[node->role],
		numa_segments, numa_cores, numa_mem >> 10,
		numa_segments * numa_cores,
		numa_segments * (numa_mem >> 10), cmhz);
}

int main(int ac, char **av)
{
	enum basil_version version  = get_basil_version();

	struct basil_inventory *inv = get_full_inventory(version);
	if (inv == NULL)
		err(1, "allocation of inventory data failed");
	else if (!inv->f->node_head)
		errx(0, "no node data");

	printf("                   %s         |---- NUMA ------|   |-- TOTAL --|\n",
		inv->is_gemini ? "            " : "");
	printf("  NID     NODENAME %s  STATE  Seg  Cores  Memory   Cores  Memory  ClockMhz\n",
		inv->is_gemini ? "      GEMINI" : "");
	print_numa_node(inv->f->node_head, inv->is_gemini);

	return EXIT_SUCCESS;
}
