/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slid_remote_info.h"
#include "slid_log.h"
#include <stralloc.h>
#include <byte.h>
#include <str.h>
#include <scan.h>

static stralloc remoteip = { 0 };	/* dotted quad */
static stralloc remoteport = { 0 };	/* integer port number */

stralloc *
slid_get_remote_ip(void)
{
	if (!remoteip.s) {
		slid_setup_remoteinfo();
	}
	return (&remoteip);
}

stralloc *
slid_get_remote_port(void)
{
	if (!remoteport.s) {
		slid_setup_remoteinfo();
	}
	return (&remoteport);
}

void
slid_setup_remoteinfo(void)
{
	if (slid_running_tcpserver) {
		unsigned short test;
		char *e = getenv("TCPREMOTEPORT");
		unsigned int done, this;
		if (e && scan_ushort(e, &test) == str_len(e)
		    && test < SLIDENTD_MAX_PORTNUM) {
			/* untainted: its just a valid port number */
			stralloc_copys(&remoteport, e);
		} else {
			buffer *log_buffer = slid_get_log_buffer();
			slid_put_datestamp(log_buffer);
			buffer_puts(log_buffer,
				    "TCPREMOTEPORT seems tainted. Contains: ");
			buffer_puts(log_buffer, e);
			buffer_putsflush(log_buffer, "\n");
			slid_dies();
		}
		e = getenv("TCPREMOTEIP");
		if (e &&
		    (this = scan_ushort(e, &test)) <= 3 && test < 256
		    && (done = this + 1) && e[done - 1] == '.'
		    && (this = scan_ushort(e + done, &test)) <= 3 && test < 256
		    && (done += this + 1) && e[done - 1] == '.'
		    && (this = scan_ushort(e + done, &test)) <= 3 && test < 256
		    && (done += this + 1) && e[done - 1] == '.'
		    && scan_ushort(e + done, &test) <= 3 && test < 256) {
			/* untainted: it looks like a dotted-quad ip addr */
			stralloc_copys(&remoteip, e);
		} else {
			buffer *log_buffer = slid_get_log_buffer();
			slid_put_datestamp(log_buffer);
			buffer_puts(log_buffer,
				    "TCPREMOTEIP seems tainted. Contains: ");
			buffer_puts(log_buffer, e);
			buffer_putsflush(log_buffer, "\n");
			slid_dies();
		}

	} else {
		struct sockaddr_in name;
		char remote_ip_buff[16];

		socklen_t len = sizeof (name);
		byte_zero(&name, sizeof (name));
		getpeername(0, (struct sockaddr *) &name, &len);
		stralloc_catint(&remoteport, (int) name.sin_port);
		inet_ntop(AF_INET, &name.sin_addr, remote_ip_buff,
			  sizeof (remote_ip_buff));
		stralloc_copys(&remoteip, remote_ip_buff);
	}
	stralloc_0(&remoteport);
	stralloc_0(&remoteip);
}
