/*
 * The Sleuth Kit
 *
 * $Date: 2006/08/30 21:09:11 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2006 Brian Carrier, Basis Technology.  All Rights reserved
 * Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
 * 
 * Copyright (c) 1997,1998,1999, International Business Machines          
 * Corporation and others. All Rights Reserved.
 *
 *
 * LICENSE
 *	This software is distributed under the IBM Public License.
 * AUTHOR(S)
 *	Wietse Venema
 *	IBM T.J. Watson Research
 *	P.O. Box 704
 *	Yorktown Heights, NY 10598, USA
--*/

#include <errno.h>
#include "fs_tools_i.h"


/* fs_read_block - read a block given the address - calls the read_random at the img layer */

SSIZE_T
fs_read_block(FS_INFO * fs, DATA_BUF * buf, OFF_T len, DADDR_T addr)
{
    OFF_T offs;
    SSIZE_T cnt;

    if (len % fs->dev_bsize) {
	tsk_errno = TSK_ERR_FS_READ;
	snprintf(tsk_errstr, TSK_ERRSTR_L,
	    "fs_read_block: length %" PRIuOFF " not a multiple of %d",
	    len, fs->dev_bsize);
	tsk_errstr2[0] = '\0';
	return -1;
    }


    if (len > buf->size) {
	tsk_errno = TSK_ERR_FS_READ;
	snprintf(tsk_errstr, TSK_ERRSTR_L,
	    "fs_read_block: Buffer too small - %"
	    PRIuOFF " > %Zd", len, buf->size);
	tsk_errstr2[0] = '\0';
	return -1;
    }

    if (addr > fs->last_block) {
	tsk_errno = TSK_ERR_FS_READ;
	snprintf(tsk_errstr, TSK_ERRSTR_L,
	    "fs_read_block: Address is too large: %" PRIuDADDR ")", addr);
	tsk_errstr2[0] = '\0';
	return -1;
    }

    buf->addr = addr;
    offs = (OFF_T) addr *fs->block_size;

    cnt =
	fs->img_info->read_random(fs->img_info, fs->offset, buf->data, len,
	offs);
    buf->used = cnt;
    return cnt;
}

SSIZE_T
fs_read_block_nobuf(FS_INFO * fs, char *buf, OFF_T len, DADDR_T addr)
{
    if (len % fs->dev_bsize) {
	tsk_errno = TSK_ERR_FS_READ;
	snprintf(tsk_errstr, TSK_ERRSTR_L,
	    "fs_read_block_nobuf: length %" PRIuOFF
	    " not a multiple of %d", len, fs->dev_bsize);
	tsk_errstr2[0] = '\0';
	return -1;
    }

    if (addr > fs->last_block) {
	tsk_errno = TSK_ERR_FS_READ;
	snprintf(tsk_errstr, TSK_ERRSTR_L,
	    "fs_read_block: Address is too large: %" PRIuDADDR ")", addr);
	tsk_errstr2[0] = '\0';
	return -1;
    }

    return fs->img_info->read_random(fs->img_info, fs->offset, buf, len,
	(OFF_T) addr * fs->block_size);
}


typedef struct {
    char *base;
    char *cur;
    size_t total;
    size_t left;
} FS_READ_FILE;

static uint8_t
fs_load_file_act(FS_INFO * fs, DADDR_T addr, char *buf, size_t size,
    int flags, void *ptr)
{
    FS_READ_FILE *buf1 = (FS_READ_FILE *) ptr;
    size_t cp_size;

    if (size > buf1->left)
	cp_size = buf1->left;
    else
	cp_size = size;

    memcpy(buf1->cur, buf, cp_size);
    buf1->left -= cp_size;
    buf1->cur = (char *) ((uintptr_t) buf1->cur + cp_size);

    if (buf1->left > 0)
	return WALK_CONT;
    else
	return WALK_STOP;
}

/* Read the contents of the file pointed to by fsi.  Flags are the same
 * as used by FILE_WALK...
 * */
char *
fs_load_file(FS_INFO * fs, FS_INODE * fsi, uint32_t type, uint16_t id,
    int flags)
{
    FS_READ_FILE lf;

    if (NULL == (lf.base = (char *) mymalloc((size_t) fsi->size))) {
	return NULL;
    }
    lf.left = lf.total = (size_t) fsi->size;
    lf.cur = lf.base;

    if (fs->file_walk(fs, fsi, type, id, flags, fs_load_file_act,
	    (void *) &lf)) {
	free(lf.base);
	strncat(tsk_errstr2, " - fs_read_file",
	    TSK_ERRSTR_L - strlen(tsk_errstr2));
	return NULL;
    }

    /* Not all of the file was copied */
    if (lf.left > 0) {
	tsk_errno = TSK_ERR_FS_FWALK;
	snprintf(tsk_errstr, TSK_ERRSTR_L,
	    "fs_read_file: Error reading file %" PRIuINUM, fsi->addr);
	tsk_errstr2[0] = '\0';
	free(lf.base);
	return NULL;
    }

    return lf.base;
}
