/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGNAMEDOBJECT_H
#define SKGNAMEDOBJECT_H
/** @file
 * This file defines classes SKGNamedObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgobjectbase.h"
#include "skgerror.h"
#include "skgdefine.h"

/**
 * This class is a named object base.
 * This is a generic way to manipule objects with name.
 */
class SKGBASEMODELER_EXPORT SKGNamedObject : public SKGObjectBase
{
    Q_OBJECT
    /**
     * Name of the object
     */
    Q_PROPERTY(QString name READ getName WRITE setName USER true)

public:
    /**
    * Return the object (@p oObject) corresponding to a name (@p iName ).
    * If more than one objects are returned by the query, then an error is generated
    * If 0 object is returned by the query, then an error is generated
    * @param iDocument the document where to search
    * @param iTable the table where to search
    * @param iName name of the object in @p iTable
    * @param oObject the result
    * @return an object managing the error
    *   @see SKGError
    */
    static SKGError getObjectByName(SKGDocument* iDocument, const QString& iTable,
                                    const QString& iName, SKGObjectBase& oObject);

    /**
     * Constructor
     * @param iDocument the document containing the object
     * @param iTable the table of the object
     * @param iID the identifier in @p iTable of the object
     */
    explicit SKGNamedObject(SKGDocument* iDocument = NULL, const QString& iTable = "", int iID = 0);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    SKGNamedObject(const SKGNamedObject& iObject);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    explicit SKGNamedObject(const SKGObjectBase& iObject);

    /**
     * Operator affectation
     * @param iObject the object to copy
     */
    virtual const SKGNamedObject& operator= (const SKGObjectBase& iObject);

    /**
     * Destructor
     */
    virtual ~SKGNamedObject();

    /**
     * Set the name of this object
     * @param iName the name
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setName(const QString& iName);

    /**
     * Get the name of this object
     * @return the name
     */
    virtual QString getName() const;

protected:
    /**
     * Get where clause needed to identify objects.
     * For this class, the whereclause is based on name
     * @return the where clause
     */
    virtual QString getWhereclauseId() const;
};
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGNamedObject, Q_MOVABLE_TYPE);
#endif
