/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A tree view with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtreeview.h"
#include "skgtraces.h"
#include "skgobjectmodelbase.h"
#include "skgservices.h"
#include "skgdocument.h"
#include "skgerror.h"
#include "skgtransactionmng.h"
#include "skgobjectbase.h"
#include "skgmainpanel.h"

#include <ksavefile.h>
#include <kicon.h>
#include <kmenu.h>
#include <klocale.h>
#include <kfiledialog.h>

#include <QDomDocument>
#include <QHeaderView>
#include <QSortFilterProxyModel>
#include <QBasicTimer>
#include <QEvent>
#include <QMouseEvent>
#include <QScrollBar>
#include <QApplication>
#include <QPrinter>
#include <QPainter>
#include <QDesktopServices>
#include <QTimer>

SKGTreeView::SKGTreeView ( QWidget *parent )
        : QTreeView ( parent ),
        autoResize ( true ), actAutoResize ( NULL ),
        document ( NULL ), textResizable(true)
{
    //Delayed resize
    timerDelayedResize = new QTimer ( this );
    timerDelayedResize->setSingleShot ( true );
    connect ( timerDelayedResize, SIGNAL ( timeout() ), this, SLOT ( resizeColumnsToContents() ), Qt::QueuedConnection );

    timerSelectionChanged = new QTimer ( this );
    timerSelectionChanged->setSingleShot ( true );
    connect ( timerSelectionChanged, SIGNAL ( timeout() ), this, SIGNAL ( selectionChangedDelayed() ), Qt::QueuedConnection );

    timerScrollSelection = new QTimer ( this );
    timerScrollSelection->setSingleShot ( true );
    connect ( timerScrollSelection, SIGNAL ( timeout() ), this, SLOT ( scroolOnSelection() ), Qt::QueuedConnection );

    //Menu
    QHeaderView* hori=header();
    hori->setContextMenuPolicy ( Qt::CustomContextMenu );
    header_menu = new KMenu ( this );

    setContextMenuPolicy ( Qt::ActionsContextMenu );
    connect ( hori,SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showHeaderMenu ( const QPoint& ) ) );
    connect ( hori,SIGNAL ( sortIndicatorChanged ( int , Qt::SortOrder ) ),this,SLOT ( onSortChanged () ) );

    //Headers
    hori->setMovable ( true );

    header()->setResizeMode ( QHeaderView::Fixed );
    setWordWrap ( false );

    connect ( header(), SIGNAL ( sectionMoved ( int, int, int ) ), this, SLOT ( moveSection() ), Qt::QueuedConnection );

    connect ( this, SIGNAL ( collapsed ( QModelIndex ) ), this, SLOT ( onCollapse ( QModelIndex ) ), Qt::QueuedConnection );
    connect ( this, SIGNAL ( expanded ( QModelIndex ) ), this, SLOT ( onExpand ( QModelIndex ) ), Qt::QueuedConnection );

    QAbstractScrollArea *scrollArea = dynamic_cast<QAbstractScrollArea*> ( this );
    if ( scrollArea )
    {
        QWidget *viewport = scrollArea->viewport();
        if ( viewport )
        {
            viewport->installEventFilter ( this );
            scrollArea->installEventFilter ( this );
        }
    }

    //Save original size
    fontOriginalPointSize=this->font().pointSize();
    iconOriginalSize=this->iconSize().height();
    if (iconOriginalSize<=0) iconOriginalSize=16;
}

SKGTreeView::~SKGTreeView()
{
    document=NULL;
    header_menu=NULL;
    timerDelayedResize=NULL;
    timerSelectionChanged=NULL;
    timerScrollSelection=NULL;
}

bool SKGTreeView::isAutoResized()
{
    return autoResize;
}

QString SKGTreeView::getState()
{
    SKGTRACEIN ( 10, "SKGTreeView::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );


    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();

    QHeaderView* hHeader=header();
    if ( hHeader && model )
    {
        if ( isSortingEnabled() )
        {
            root.setAttribute ( "sortOrder", SKGServices::intToString ( ( int ) hHeader->sortIndicatorOrder() ) );
            root.setAttribute ( "sortColumn", model->getAttribute ( hHeader->sortIndicatorSection() ) );
        }

        //Memorize order
        int nb=hHeader->count();
        if ( nb )
        {
            QString columns;
            QString columnsSize;
            QString columnsVisibility;
            for ( int i=0; i<nb; ++i )
            {
                int idx=hHeader->logicalIndex ( i );
                if ( i ) columns+=';';
                columns+=model->getAttribute ( idx );

                if ( i ) columnsSize+=';';
                columnsSize+=SKGServices::intToString ( hHeader->sectionSize ( idx ) );

                if ( i ) columnsVisibility+=';';
                columnsVisibility+= ( hHeader->isSectionHidden ( idx ) ? "N" : "Y" );
            }
            root.setAttribute ( "columns", columns );
            if ( !autoResize ) root.setAttribute ( "columnsSize", columnsSize );
            root.setAttribute ( "columnsVisibility", columnsVisibility );
            root.setAttribute ( "columnsAutoResize", autoResize ? "Y" : "N" );
        }
    }
    root.setAttribute ( "alternatingRowColors", alternatingRowColors() ? "Y" : "N" );
    root.setAttribute ( "zoomPosition", SKGServices::intToString(zoomPosition()) );
    return doc.toString();
}

void SKGTreeView::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGTreeView::setState" );
    resetColumnsOrder();

    QDomDocument doc ( "SKGML" );

    QString viewState=iState;
    if ( viewState.isEmpty() && document )
    {
        //Get default state
        viewState=document->getParameter ( parameterName );
    }

    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();

    Qt::SortOrder qtsortorder=Qt::AscendingOrder;
    if ( doc.setContent ( viewState ) )
    {
        QDomElement root = doc.documentElement();

        QString sortOrder=root.attribute ( "sortOrder" );
        QString sortColumn=root.attribute ( "sortColumn" );
        QString columns=root.attribute ( "columns" );
        QString columnsSize=root.attribute ( "columnsSize" );
        QString columnsVisibility=root.attribute ( "columnsVisibility" );
        QString columnsAutoResize=root.attribute ( "columnsAutoResize" );
        QString alternatingRowColors=root.attribute ( "alternatingRowColors" );
        QString zoomPosition=root.attribute ( "zoomPosition" );

        //Set column order
        QStringList listAtt;
        if ( !columns.isEmpty() )
        {
            listAtt=SKGServices::splitCSVLine ( columns, ';' );
            QStringList sizes=SKGServices::splitCSVLine ( columnsSize, ';' );
            QStringList visibilities=SKGServices::splitCSVLine ( columnsVisibility, ';' );

            int nb=listAtt.count();
            int nbvisibilities=visibilities.count();
            int nbsizes=sizes.count();
            for ( int i=0; i<nb; ++i )
            {
                if ( nbvisibilities==nb )
                {
                    listAtt[i]=listAtt[i]+'|'+visibilities[i];
                    if ( nbsizes==nb )
                    {
                        listAtt[i]=listAtt[i]+'|'+sizes[i];
                    }
                }

            }
        }
        if ( model )
        {
            model->setSupportedAttributes ( listAtt );
            model->refresh();
        }

        //Set autoResize
        if ( !columnsAutoResize.isEmpty() )
        {
            autoResize= ( columnsAutoResize=="Y" );
            header()->setResizeMode ( autoResize ? QHeaderView::Fixed : QHeaderView::Interactive );
        }

        //Set sort
        if ( model && isSortingEnabled() && !sortOrder.isEmpty() && !sortColumn.isEmpty() )
        {
            int index=SKGServices::splitCSVLine ( columns, ';' ).indexOf ( sortColumn );
            if ( index==-1 ) index=model->getIndexAttribute ( sortColumn );
            if ( index==-1 ) index=0;
            qtsortorder=( Qt::SortOrder ) SKGServices::stringToInt ( sortOrder );
            this->sortByColumn ( index, qtsortorder );
        }

        //Set alternatingRowColors
        if ( !alternatingRowColors.isEmpty() ) setAlternatingRowColors ( alternatingRowColors=="Y" );
        if ( !zoomPosition.isEmpty() ) setZoomPosition(SKGServices::stringToInt(zoomPosition));
    }
    else
    {
        if ( model )
        {
            model->setSupportedAttributes ( QStringList() );
            model->refresh();
        }

        this->sortByColumn ( 0, qtsortorder);
    }

    setupHeaderMenu();

    if (qtsortorder==Qt::AscendingOrder) scrollToBottom();
    else scrollToTop();
}

void SKGTreeView::insertGlobalAction(const QString& iRegisteredAction)
{
    if (iRegisteredAction.isEmpty())
    {
        QAction* sep=new QAction ( this );
        sep->setSeparator ( true );
        this->insertAction ( 0, sep );
    }
    else
    {
        KAction* act=SKGMainPanel::getMainPanel()->getGlobalAction ( iRegisteredAction );
        this->insertAction ( 0, act );
    }
}

void SKGTreeView::resizeColumnsToContentsDelayed()
{
    SKGTRACEIN ( 10, "SKGTreeView::resizeColumnsToContentsDelayed" );
    if ( timerDelayedResize ) timerDelayedResize->start ( 300 );
}

void SKGTreeView::resizeColumnsToContents()
{
    SKGTRACEIN ( 10, "SKGTreeView::resizeColumnsToContents" );
    int nb=header()->count();
    for ( int i=nb-1; i>=0; --i )
        resizeColumnToContents ( i );
}

void SKGTreeView::showHeaderMenu ( const QPoint& pos )
{
    if ( header_menu ) header_menu->popup ( header()->mapToGlobal ( pos ) );
}

void SKGTreeView::setDefaultSaveParameters ( SKGDocument* iDocument, const QString& iParameterName )
{
    document= iDocument;
    parameterName=iParameterName;
}

void SKGTreeView::moveSection()
{
    setupHeaderMenu ( false );
}

void SKGTreeView::setupHeaderMenu ( bool iRefreshColumnSize )
{
    SKGTRACEIN ( 10, "SKGTreeView::setupHeaderMenu" );
    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();

    if ( model && model->isRefreshBlocked() ) return;

    if ( header_menu )
    {
        header_menu->clear();

        //Processing
        QMenu* columns=header_menu->addMenu ( i18nc ("Noun, Menu name", "Columns" ) );

        // Set right click menu
        if ( model )
        {
            QList<SKGDocument::SKGModelTemplate> schemas=model->getSchemas();
            int nbSchemas=schemas.count();
            if ( nbSchemas )
            {
                QMenu* viewAppearanceMenu=columns->addMenu ( KIcon ( "view-file-columns" ), i18nc ("Noun, user action", "View appearance" ) );

                for ( int i=0; i<nbSchemas; ++i )
                {
                    SKGDocument::SKGModelTemplate schema=schemas.at ( i );
                    QAction* act = viewAppearanceMenu->addAction ( schema.name );
                    if ( !schema.icon.isEmpty() ) act->setIcon ( KIcon ( schema.icon ) );
                    act->setData ( schema.schema );

                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( changeSchema() ) );
                }
            }
        }

        QAction* actResize = columns->addAction ( KIcon ( "zoom-fit-width" ), i18nc ("Noun, user action", "Resize to content" ) );
        connect ( actResize, SIGNAL ( triggered ( bool ) ), this, SLOT ( resizeColumnsToContents() ) );

        actAutoResize = columns->addAction ( i18nc ("Noun, user action", "Auto resize" ) );
        actAutoResize->setCheckable ( true );
        actAutoResize->setChecked ( autoResize );
        connect ( actAutoResize, SIGNAL ( triggered ( bool ) ), this, SLOT ( switchAutoResize() ) );

        QAction* actAlternatingRowColors = header_menu->addAction ( i18nc ("Noun, user action", "Alternate row colors" ) );
        if ( actAlternatingRowColors )
        {
            actAlternatingRowColors->setCheckable ( true );
            actAlternatingRowColors->setChecked ( alternatingRowColors() );
            connect ( actAlternatingRowColors, SIGNAL ( triggered ( bool ) ), this, SLOT ( setAlternatingRowColors ( bool ) ) );
        }

        if ( document )
        {
            QAction* actDefault = header_menu->addAction ( KIcon ( "document-save" ), i18nc ("Noun, user action", "Save parameters" ) );
            connect ( actDefault, SIGNAL ( triggered ( bool ) ), this, SLOT ( saveDefaultClicked() ) );
        }

        columns->addSeparator();

        if ( model )
        {
            QHeaderView* hHeader=header();
            int nbcol=hHeader->count();
            for ( int i=0; i<nbcol; ++i )
            {
                int idx=hHeader->logicalIndex ( i );
                QString col = model->headerData ( idx,Qt::Horizontal,Qt::UserRole ).toString();
                QStringList values=col.split ( '|' );

                if ( iRefreshColumnSize )
                {
                    if ( values.count() >1 ) hHeader->setSectionHidden ( idx, values.at ( 1 ) =="N" );
                    if ( values.count() >2 )
                    {
                        int s=SKGServices::stringToInt ( values.at ( 2 ) );
                        if ( s>0 ) hHeader->resizeSection ( idx, s );
                    }
                }
                QAction* act = columns->addAction ( values.at ( 0 ) );
                if ( act )
                {
                    act->setCheckable ( true );
                    act->setChecked ( !hHeader->isSectionHidden ( idx ) );
                    act->setIcon ( model->headerData ( idx,Qt::Horizontal,Qt::DecorationRole ).value<QIcon>() );
                    act->setData ( idx );

                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( showHideColumn() ) );
                }
            }
        }
        header_menu->addSeparator();

        QMenu* exp=header_menu->addMenu ( i18nc ("Noun, Menu name", "Export" ) );

        QAction* actPDF = exp->addAction ( KIcon ( "application-pdf" ), i18nc ("Noun, user action", "Export PDF..." ) );
        connect ( actPDF, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportPDF() ) );

        QAction* actCSV = exp->addAction ( KIcon ( "text-csv" ), i18nc ("Noun, user action", "Export CSV..." ) );
        connect ( actCSV, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportCSV() ) );

        QAction* actTXT = exp->addAction ( KIcon ( "text-plain" ), i18nc ("Noun, user action", "Export TXT..." ) );
        connect ( actTXT, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportTXT() ) );

        if ( autoResize ) resizeColumnsToContentsDelayed();
    }
}

void SKGTreeView::setSelectionModel ( QItemSelectionModel * selectionModel )
{
    if ( this->selectionModel() ) disconnect ( this->selectionModel(), SIGNAL ( selectionChanged ( QItemSelection,QItemSelection ) ), this, SLOT ( onSelectionChanged() ) );
    QTreeView::setSelectionModel ( selectionModel );
    if ( selectionModel ) connect ( selectionModel, SIGNAL ( selectionChanged ( QItemSelection,QItemSelection ) ), this, SLOT ( onSelectionChanged() ) );
}

void SKGTreeView::onSelectionChanged()
{
    if ( timerSelectionChanged ) timerSelectionChanged->start ( 300 );
}

void SKGTreeView::saveDefaultClicked()
{
    if ( document )
    {
        SKGError err;
        SKGBEGINTRANSACTION ( *document, i18nc("Noun, name of the user action", "Save default parameters" ), err );
        err=document->setParameter ( parameterName, getState() );
    }
}

void SKGTreeView::switchAutoResize()
{
    autoResize=actAutoResize->isChecked();
    header()->setResizeMode ( autoResize ? QHeaderView::Fixed : QHeaderView::Interactive );
    if ( autoResize ) resizeColumnsToContentsDelayed();
}

void SKGTreeView::showHideColumn()
{
    QAction* sender=static_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        QHeaderView* hHeader=header();

        int idx=sender->data().toInt();
        hHeader->setSectionHidden ( idx, !hHeader->isSectionHidden ( idx ) );
        resizeColumnToContents ( idx );
    }
}

void SKGTreeView::resetColumnsOrder()
{
    QHeaderView* hHeader=header();
    int nbcol=hHeader->count();
    for ( int i=0; i<nbcol; ++i )
    {
        int idx=hHeader->visualIndex ( i );
        if ( idx!=i ) hHeader->moveSection ( idx,i );
    }
}

void SKGTreeView::changeSchema()
{
    QStringList list;

    QAction* sender=static_cast<QAction*> ( this->sender() );
    if ( sender )  list=SKGServices::splitCSVLine ( sender->data().toString(), ';' );

    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();
    if ( model )
    {
        //Reset column oder
        resetColumnsOrder();

        model->setSupportedAttributes ( list );
        model->refresh();
        header()->setSortIndicator ( 0, Qt::AscendingOrder );

        setupHeaderMenu();
    }

}

void SKGTreeView::setAlternatingRowColors ( bool enable )
{
    QTreeView::setAlternatingRowColors ( enable );
}

SKGObjectBase::SKGListSKGObjectBase SKGTreeView::getSelectedObjects()
{
    SKGObjectBase::SKGListSKGObjectBase selection;
    QItemSelectionModel *selModel=selectionModel();
    if ( selModel )
    {
        SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
        if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();

        if ( model )
        {
            QModelIndexList indexes=selModel->selectedRows();
            foreach ( const QModelIndex& index, indexes )
            {
                QModelIndex idxs= ( proxyModel ? proxyModel->mapToSource ( index ) : index );
                SKGObjectBase obj=model->getObject ( idxs );
                selection.push_back ( obj );
            }
        }
    }

    return selection;
}

int SKGTreeView::getNbSelectedObjects()
{
    QItemSelectionModel *selModel=selectionModel();
    return ( selModel ? selModel->selectedRows().count() : 0 );
}

void SKGTreeView::saveSelection()
{
    SKGTRACEIN ( 10, "SKGTreeView::saveSelection" );

    selection.clear();

    SKGObjectBase::SKGListSKGObjectBase objs=getSelectedObjects();
    int nb=objs.count();
    //We save the selection only if not too big
    if ( nb<=100 )
    {
        for ( int i=0; i<nb; ++i )
        {
            QString id=objs.at ( i ).getUniqueID();
            selection.push_back ( id );
        }
    }
    SKGTRACEL(10) << selection.count() << " objects saved" << endl;
}

void SKGTreeView::selectObject ( const QString& iUniqueID )
{
    SKGTRACEIN ( 10, "SKGTreeView::selectObject" );
    QStringList tmp;
    tmp.push_back ( iUniqueID );
    selectObjects ( tmp, true );
}

void SKGTreeView::selectObjects ( const QStringList& iUniqueIDs, bool iFocusOnFirstOne )
{
    SKGTRACEIN ( 10, "SKGTreeView::selectObjects" );
    int nbset=0;
    QItemSelectionModel *selModel=selectionModel();
    if ( selModel )
    {
        bool previous=selModel->blockSignals ( true );

        selModel->clearSelection();

        SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
        if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();

        if ( model )
        {
            //Get all indexes
            QList<QModelIndex> items;
            items.push_back(QModelIndex());
            for (int i=0; i<items.count(); ++i) //Dynamic size because the list is modified
            {
                QModelIndex mi=items.at(i);
                int nbRows=model->rowCount (mi  );
                for ( int j=0; j<nbRows; ++j )
                {
                    items.push_back(model->index ( j, 0, mi ));
                }
            }
            items.removeAt(0);

            int nbRows=items.count();
            if ( nbRows )
            {
                //Expand nodes
                foreach ( const QString& exp, expandedNodes )
                {
                    for ( int i=0; i<nbRows; ++i )
                    {
                        QModelIndex index=items.at(i);
                        SKGObjectBase obj=model->getObject ( index );
                        if ( obj.getUniqueID() ==exp )
                        {
                            QModelIndex idxs= ( proxyModel ? proxyModel->mapFromSource ( index ) : index );
                            setExpanded ( idxs, true );
                            break;
                        }
                    }
                }

                //Set selection
                bool focusDone=false;
                foreach ( const QString& sel, iUniqueIDs )
                {
                    for ( int i=0; i<nbRows; ++i )
                    {
                        QModelIndex index=items.at(i);
                        SKGObjectBase obj=model->getObject ( index );
                        if ( obj.getUniqueID() ==sel )
                        {
                            QModelIndex idxs= ( proxyModel ? proxyModel->mapFromSource ( index ) : index );
                            selModel->select ( idxs, QItemSelectionModel::Select|QItemSelectionModel::Rows );
                            ++nbset;
                            if ( iFocusOnFirstOne && !focusDone )
                            {
                                scrollTo ( idxs );
                                focusDone=true;
                            }
                            break;
                        }
                    }
                }
            }
        }
        selModel->blockSignals ( previous );
    }

    SKGTRACEL(10) << iUniqueIDs.count() << " objects to select" << endl;
    SKGTRACEL(10) << nbset << " objects selected" << endl;
}

void SKGTreeView::resetSelection()
{
    SKGTRACEIN ( 10, "SKGTreeView::resetSelection" );

    selectObjects ( selection );
}

void SKGTreeView::onSortChanged ()
{
    if ( timerScrollSelection ) timerScrollSelection->start ( 300 );
}

void SKGTreeView::scroolOnSelection ()
{
    SKGObjectBase::SKGListSKGObjectBase selection;
    QItemSelectionModel *selModel=selectionModel();
    if ( selModel )
    {
        SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
        if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();

        if ( model )
        {
            QModelIndexList indexes=selModel->selectedRows();
            if (indexes.count()) scrollTo ( indexes.at(0) );
        }
    }
}

void SKGTreeView::onExpand ( const QModelIndex & index )
{
    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();
    if (model)
    {
        QModelIndex idxs= ( proxyModel ? proxyModel->mapToSource ( index ) : index );

        SKGObjectBase obj=model->getObject ( idxs );
        QString id=obj.getUniqueID();
        expandedNodes.push_back ( id );
    }
}

void SKGTreeView::onCollapse ( const QModelIndex & index )
{
    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();
    if (model)
    {
        QModelIndex idxs= ( proxyModel ? proxyModel->mapToSource ( index ) : index );

        SKGObjectBase obj=model->getObject ( idxs );

        QString id=obj.getUniqueID();
        expandedNodes.removeOne ( id );
    }
}

void SKGTreeView::setZoomPosition(int iZoomPosition)
{
    int newZoomPos=qMax(qMin(iZoomPosition, 10), -10);
    if (newZoomPos!=zoomPosition() && fontOriginalPointSize+newZoomPos>1)
    {
        QFont font = this->font();
        font.setPointSize( fontOriginalPointSize+newZoomPos);
        int iconSize=qMax(iconOriginalSize + newZoomPos, 1);

        this->setFont( font );
        this->setIconSize(QSize(iconSize, iconSize));
        this->header()->setIconSize(QSize(iconSize, iconSize));

        if ( autoResize ) resizeColumnsToContentsDelayed();

        Q_EMIT zoomChanged(newZoomPos);
    }
}

int SKGTreeView::zoomPosition() const
{
    return this->font().pointSize()-fontOriginalPointSize;
}

/**
 * Event filter
 * @param object object
 * @param event event
 * @return traited or not
 */
bool SKGTreeView::eventFilter ( QObject *object, QEvent *event )
{
    QWheelEvent* e=dynamic_cast<QWheelEvent*> ( event );
    if ( textResizable && e && e->orientation() == Qt::Vertical && QApplication::keyboardModifiers() &Qt::ControlModifier )
    {
        int numDegrees = e->delta() / 8;
        int numTicks = numDegrees / 15;

        setZoomPosition(zoomPosition()+( numTicks >0 ? 1 : -1));
        e->setAccepted ( true );
        return true;
    }

    return QTreeView::eventFilter(object, event);
}

void SKGTreeView::mousePressEvent ( QMouseEvent * event )
{
    if ( event && event->button() ==Qt::LeftButton && ! ( this->indexAt ( event->pos() ).isValid() ) )
    {
        Q_EMIT clickEmptyArea();
        clearSelection();
    }
    QTreeView::mousePressEvent ( event );
}

bool SKGTreeView::isTextResizable() const
{
    return textResizable;
}

void SKGTreeView::setTextResizable(bool resizable)
{
    textResizable=resizable;
}

SKGStringListList SKGTreeView::getTable()
{
    //Build table
    SKGStringListList table;

    //Get header names
    SKGObjectModelBase* model= ( SKGObjectModelBase* ) this->model();
    QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *> ( model );
    if ( proxyModel ) model= ( SKGObjectModelBase* ) proxyModel->sourceModel();
    if (model)
    {
        int nb=model->columnCount();
        QStringList cols;
        for ( int i=0; i<nb; i++ )
        {
            cols.append ( model->headerData ( i,Qt::Horizontal, Qt::UserRole ).toString().split ( '|' ).at ( 0 ) );
        }
        table.append ( cols );

        //Get content
        int nb2=model->rowCount();
        for ( int i=0; i<nb2; i++ )
        {
            QStringList row;
            for ( int j=0; j<nb; j++ )
            {
                //We have to check the type for 214849
                SKGServices::AttributeType type=model->getAttributeType (j);
                QString display=model->data ( model->index ( i,j ), type==SKGServices::FLOAT ?  Qt::DisplayRole : Qt::UserRole ).toString();
                row.append ( display );
            }
            table.append ( row );
        }
    }
    return table;
}

void SKGTreeView::onExportCSV()
{
    _SKGTRACEIN ( 10, "SKGTreeView::onExportCSV" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "*.csv|"+i18nc ("File format", "CSV Files" ) , this );
    if ( fileName.isEmpty() ) return;
    {
        SKGError err;

        //Write file
        KSaveFile file ( fileName );
        if ( !file.open () )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc("Error message",  "Save file '%1' failed" , fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            QStringList dump=SKGServices::tableToDump ( getTable(), SKGServices::DUMP_CSV );
            int nbl=dump.count();
            for ( int i=0; i<nbl; ++i )
            {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.finalize();
        file.close();

        SKGMainPanel::getMainPanel()->displayErrorMessage(err);
    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGTreeView::onExportTXT()
{
    _SKGTRACEIN ( 10, "SKGTreeView::onExportTXT" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "*.txt|"+i18nc ("File format", "Text document" ) , this );
    if ( fileName.isEmpty() ) return;
    {
        SKGError err;

        //Write file
        KSaveFile file ( fileName );
        if ( !file.open () )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc("Error message",  "Save file '%1' failed" , fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            QStringList dump=SKGServices::tableToDump ( getTable(), SKGServices::DUMP_TEXT );
            int nbl=dump.count();
            for ( int i=0; i<nbl; ++i )
            {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.finalize();
        file.close();

        SKGMainPanel::getMainPanel()->displayErrorMessage(err);
    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGTreeView::onExportPDF()
{
    _SKGTRACEIN ( 10, "SKGTreeView::onExportPDF" );
#ifndef QT_NO_PRINTER
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "application/pdf" , this );
    if ( fileName.isEmpty() ) return;

    {
        QImage image ( this->size(), QImage::Format_ARGB32 );
        QPainter painter ( &image );
        this->render ( &painter );
        painter.end();

        {
            QPrinter printer ( QPrinter::HighResolution );
            printer.setOutputFileName ( fileName );
            QPainter painter ( &printer );

            QRect rect = painter.viewport();
            QSize size = image.size();
            size.scale ( rect.size(), Qt::KeepAspectRatio );
            painter.setViewport ( rect.x(), rect.y(), size.width(), size.height() );
            painter.setWindow ( image.rect() );
            painter.drawImage ( 0, 0, image );
        }
    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
#endif
}
#include "skgtreeview.moc"
