/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A Graphic view with more functionalities.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skggraphicsview.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"

#include <kmenu.h>
#include <kfiledialog.h>

#include <QDesktopServices>
#include <QPrinter>
#include <QPrintDialog>
#include <QClipboard>
#include <QApplication>
#include <QDomDocument>
#include <QSvgGenerator>
#include <QGraphicsSceneWheelEvent>
#include <qmath.h>

SKGGraphicsView::SKGGraphicsView ( QWidget *parent )
        : QWidget ( parent ), oscale(1), toolBarVisible ( true )
{
    ui.setupUi ( this );
    graphicsView()->installEventFilter ( this );

    //Set icons
    ui.kZoomIn->setIcon ( KIcon ( "zoom-in" ) );
    ui.kZoomOut->setIcon ( KIcon ( "zoom-out" ) );
    ui.kZoomOriginal->setIcon ( KIcon ( "zoom-original" ) );
    ui.kPrint->setIcon ( KIcon ( "printer" ) );
    ui.kCopy->setIcon ( KIcon ( "edit-copy" ) );

    //Build contextual menu
    graphicsView()->setContextMenuPolicy ( Qt::CustomContextMenu );

    mainMenu = new KMenu ( graphicsView() );

    actZoomOriginal = mainMenu->addAction ( ui.kZoomOriginal->icon(), ui.kZoomOriginal->toolTip() );
    if ( actZoomOriginal )
    {
        actZoomOriginal->setCheckable ( true );
        actZoomOriginal->setChecked ( true );
        connect ( actZoomOriginal, SIGNAL ( triggered ( bool ) ), this, SLOT ( onZoomOriginal() ) );
    }

    //Timer to refresh
    timer = new QTimer( this );
    timer->setSingleShot( true );
    connect( timer, SIGNAL( timeout() ), this, SLOT( onZoomOriginal() ) );

    QAction* actZoomIn = mainMenu->addAction ( ui.kZoomIn->icon(), ui.kZoomIn->toolTip() );
    connect ( actZoomIn, SIGNAL ( triggered ( bool ) ), this, SLOT ( onZoomIn() ) );

    QAction* actZoomOut = mainMenu->addAction ( ui.kZoomOut->icon(), ui.kZoomOut->toolTip() );
    connect ( actZoomOut, SIGNAL ( triggered ( bool ) ), this, SLOT ( onZoomOut() ) );

    mainMenu->addSeparator();

    QAction* actCopy = mainMenu->addAction ( ui.kCopy->icon(), ui.kCopy->toolTip() );
    connect ( actCopy, SIGNAL ( triggered ( bool ) ), this, SLOT ( onCopy() ) );

    QAction* actPrint = mainMenu->addAction ( ui.kPrint->icon(), ui.kPrint->toolTip() );
    connect ( actPrint, SIGNAL ( triggered ( bool ) ), this, SLOT ( onPrint() ) );

    QMenu* exp=mainMenu->addMenu ( i18nc ("Noun, Menu name", "Export" ) );

    QAction* actPDF = exp->addAction ( KIcon ( "application-pdf" ), i18nc ("Noun, user action", "Export PDF..." ) );
    connect ( actPDF, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSavePDF() ) );

    QAction* actSvg = exp->addAction ( KIcon ( "image-svg+xml" ), i18nc ("Noun, user action", "Export SVG..." ) );
    connect ( actSvg, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSaveSvg() ) );

    QAction* actImage = exp->addAction ( KIcon ( "image-png" ), i18nc ("Noun, user action", "Export image..." ) );
    connect ( actImage, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSaveImage() ) );

    mainMenu->addSeparator();

    actAntialiasing = mainMenu->addAction ( i18nc ("Noun, user action", "Antialiasing" ) );
    if ( actAntialiasing )
    {
        actAntialiasing->setCheckable ( true );
        actAntialiasing->setChecked ( true );
        onSwitchAntialiasing();
        connect ( actAntialiasing, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSwitchAntialiasing() ) );
    }

    connect ( graphicsView(), SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showMenu ( const QPoint& ) ) );

    actShowToolBar = mainMenu->addAction ( i18nc ("Noun, user action", "Show tool bar" ) );
    if ( actShowToolBar )
    {
        actShowToolBar->setCheckable ( true );
        actShowToolBar->setChecked ( true );
        connect ( actShowToolBar, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSwitchToolBarVisibility() ) );
    }
}

SKGGraphicsView::~SKGGraphicsView()
{
    actAntialiasing=NULL;
    actShowToolBar=NULL;
    actZoomOriginal=NULL;
    mainMenu=NULL;
    timer=NULL;
}

void SKGGraphicsView::setScene ( QGraphicsScene * iScene )
{
    graphicsView()->setScene ( iScene );
    if (iScene) iScene->installEventFilter ( this );
    onZoomOriginal();
}

QGraphicsView* SKGGraphicsView::graphicsView()
{
    return ui.kGraphicsView;
}

QString SKGGraphicsView::getState() const
{
    SKGTRACEIN ( 10, "SKGGraphicsView::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "isToolBarVisible", isToolBarVisible() ? "Y" : "N" );
    if ( actAntialiasing ) root.setAttribute ( "antialiasing", actAntialiasing->isChecked() ? "Y" : "N" );

    return doc.toString();
}

void SKGGraphicsView::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGGraphicsView::setState" );
    QDomDocument doc ( "SKGML" );
    if ( doc.setContent ( iState ) )
    {
        QDomElement root = doc.documentElement();

        QString isToolBarVisible=root.attribute ( "isToolBarVisible" );
        if ( !isToolBarVisible.isEmpty() ) setToolBarVisible ( isToolBarVisible=="Y" );

        QString antialiasing=root.attribute ( "antialiasing" );
        if ( !antialiasing.isEmpty() && actAntialiasing ) actAntialiasing->setChecked ( antialiasing=="Y" );
    }
}

KMenu* SKGGraphicsView::getContextualMenu() const
{
    return mainMenu;
}

void SKGGraphicsView::showMenu ( const QPoint& pos )
{
    if ( mainMenu ) mainMenu->popup ( graphicsView()->mapToGlobal ( pos ) );
}

void SKGGraphicsView::setToolBarVisible ( bool iVisibility )
{
    toolBarVisible=iVisibility;
    ui.toolBar->setVisible ( toolBarVisible );
    if ( actShowToolBar ) actShowToolBar->setChecked ( toolBarVisible );
}

bool SKGGraphicsView::isToolBarVisible() const
{
    return toolBarVisible;
}

void SKGGraphicsView::onSwitchToolBarVisibility()
{
    setToolBarVisible ( !isToolBarVisible() );
}

void SKGGraphicsView::onSwitchAntialiasing()
{
    if ( actAntialiasing ) graphicsView()->setRenderHint ( QPainter::Antialiasing, actAntialiasing->isChecked() );
}

bool SKGGraphicsView::eventFilter ( QObject *object, QEvent *event )
{
    if ( object==graphicsView()->scene() )
    {
        QGraphicsSceneWheelEvent* e=dynamic_cast<QGraphicsSceneWheelEvent*> ( event );
        if ( e && e->orientation() == Qt::Vertical && QApplication::keyboardModifiers() &Qt::ControlModifier )
        {
            int numDegrees = e->delta() / 8;
            int numTicks = numDegrees / 15;

            if ( numTicks >0 )
            {
                onZoomIn();
            }
            else
            {
                onZoomOut();
            }
            e->setAccepted ( true );
            return true;
        }
    }
    else if ( object==graphicsView() &&  event && event->type()==QEvent::Resize )
    {
        Q_EMIT resized();
        if (ui.kZoomOriginal->isChecked() && timer) timer->start(300 );
    }
    return QWidget::eventFilter ( object, event );
}

void SKGGraphicsView::onZoom()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onZoom" );
    int sliderValue=ui.kZoomSlider->value();
    if (graphicsView()->scene())
    {
        if ( sliderValue==0 )
        {
            graphicsView()->fitInView ( graphicsView()->scene()->sceneRect(), Qt::KeepAspectRatio );
            oscale=1;
        }
        else
        {
            qreal scale = qPow ( qreal ( 1.5 ), ( sliderValue ) / qreal ( 50 ) );
            graphicsView()->scale(scale/oscale, scale/oscale);
            oscale=scale;
        }
    }
    if (ui.kZoomOriginal) ui.kZoomOriginal->setChecked(sliderValue==0);
    if (actZoomOriginal) actZoomOriginal->setChecked(sliderValue==0);
}

void SKGGraphicsView::onZoomIn()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onZoomIn" );
    ui.kZoomSlider->setValue ( ui.kZoomSlider->value() + 50 );
}

void SKGGraphicsView::onZoomOut()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onZoomOut" );
    ui.kZoomSlider->setValue ( ui.kZoomSlider->value() - 50 );
}

void SKGGraphicsView::onZoomOriginal()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onZoomOriginal" );
    ui.kZoomSlider->setValue ( 0 );
    onZoom(); //To force
}

void SKGGraphicsView::onPrint()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onPrint" );
    QPrinter printer;
    QPointer<QPrintDialog> dialog=new QPrintDialog ( &printer, this );
    if ( dialog->exec() == QDialog::Accepted )
    {
        QPainter painter ( &printer );
        graphicsView()->render ( &painter );
        painter.end();
    }
    delete dialog;
}

void SKGGraphicsView::onSavePDF()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onSavePDF" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "application/pdf", this );
    if ( fileName.isEmpty() ) return;

    {
        QPrinter printer;
        printer.setOutputFileName ( fileName );
        QPainter painter ( &printer );
        graphicsView()->render ( &painter );
        painter.end();
    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGGraphicsView::onSaveSvg()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onSaveSvg" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "image/svg+xml", this );
    if ( fileName.isEmpty() ) return;

    {
        QSvgGenerator generator;
        generator.setFileName ( fileName );
        generator.setSize(QSize(200, 200));
        generator.setViewBox(QRect(0, 0, 200, 200));
        generator.setTitle(i18nc("Title of the content SVG export", "Skrooge SVG export"));
        generator.setDescription(i18nc("Description of the content SVG export", "A SVG drawing created by the Skrooge."));

        QPainter painter ( &generator );
        graphicsView()->render ( &painter );
        painter.end();
    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGGraphicsView::onSaveImage()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onCopy" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "image/png image/jpeg image/gif image/tiff" , this );
    if ( fileName.isEmpty() ) return;

    {
        QImage image ( graphicsView()->size(), QImage::Format_ARGB32 );
        QPainter painter ( &image );
        graphicsView()->render ( &painter );
        painter.end();
        image.save ( fileName );
    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGGraphicsView::onCopy()
{
    _SKGTRACEIN ( 10, "SKGGraphicsView::onCopy" );
    QClipboard *clipboard = QApplication::clipboard();
    if ( clipboard )
    {

        QImage image ( graphicsView()->size(), QImage::Format_ARGB32 );
        QPainter painter ( &image );
        graphicsView()->render ( &painter );
        painter.end();
        clipboard->setImage ( image );
    }
}
#include "skggraphicsview.moc"

