/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a plugin for undo and redo operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgundoredoplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgmainpanel.h"
#include "skgundoredoplugindockwidget.h"
#include "skgservices.h"
#include "skgundoredo_settings.h"

#include <QtGui/QWidget>

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <ktoolbarpopupaction.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGUndoRedoPluginFactory, registerPlugin<SKGUndoRedoPlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGUndoRedoPluginFactory ( "skg_undoredo", "skg_undoredo" ) )

SKGUndoRedoPlugin::SKGUndoRedoPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent ), dockWidget ( NULL )
{
    SKGTRACEIN ( 10, "SKGUndoRedoPlugin::SKGUndoRedoPlugin" );
}

SKGUndoRedoPlugin::~SKGUndoRedoPlugin()
{
    SKGTRACEIN ( 10, "SKGUndoRedoPlugin::~SKGUndoRedoPlugin" );
    currentDocument=NULL;
    dockWidget=NULL;
    undoSaveAction=NULL;
    undoAction=NULL;
    redoAction=NULL;
    undoMenu=NULL;
    redoMenu=NULL;
}

bool SKGUndoRedoPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
    SKGTRACEIN ( 10, "SKGUndoRedoPlugin::setupActions" );
    Q_UNUSED ( iArgument );

    currentDocument=iDocument;


    setComponentData ( SKGUndoRedoPluginFactory::componentData() );
    setXMLFile ( "skg_undoredo.rc" );

    dockWidget = new QDockWidget (SKGMainPanel::getMainPanel());
    dockWidget->setObjectName ( QString::fromUtf8 ( "skg_undoredo_docwidget" ) );
    dockWidget->setAllowedAreas ( Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea );
    dockWidget->setWindowTitle ( title() );
    dockWidget->setWidget ( new SKGUndoRedoPluginDockWidget ( currentDocument ) );

    //Menu
    undoSaveAction = new KAction ( KIcon ( "document-revert" ), i18nc ( "Verb, action to cancel previous action", "Undo document" ), this );
    connect ( undoSaveAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionUndoSave() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_undolastsave" ), undoSaveAction );
    undoSaveAction->setShortcut ( Qt::CTRL+Qt::ALT+Qt::Key_Z );
    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_undolastsave", undoSaveAction );

    undoAction = new KToolBarPopupAction ( KIcon ( "edit-undo" ), i18nc ( "Verb, action to cancel previous action", "Undo" ), this );
    connect ( undoAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionUndo() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_undo" ), undoAction );
    undoAction->setShortcut ( Qt::CTRL+Qt::Key_Z );

    undoMenu= static_cast<KMenu*>(undoAction->menu());
    connect ( undoMenu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowUndoMenu() ) );

    undoAction->setStickyMenu ( false );
    undoAction->setData ( 1 );

    //undoAction=KStandardAction::undo(this, SLOT(actionUndo()), actionCollection());
    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit-undo", undoAction );

    redoAction = new KToolBarPopupAction ( KIcon ( "edit-redo" ), i18nc ( "Verb, action to redo previous cancelled action", "Redo" ), this );
    connect ( redoAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionRedo() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_redo" ), redoAction );
    redoAction->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_Z );
    redoMenu= static_cast<KMenu*>(redoAction->menu());
    connect ( redoMenu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowRedoMenu() ) );

    redoAction->setStickyMenu ( false );
    redoAction->setData ( 1 );

    //redoAction=KStandardAction::redo(this, SLOT(actionRedo()), actionCollection());
    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit-redo", redoAction );

    // add action to control hide / display of history
    dockWidget->toggleViewAction()->setShortcut ( Qt::SHIFT+Qt::Key_F11 );
    actionCollection()->addAction ( "view_transactions", dockWidget->toggleViewAction() );

    return true;
}

void SKGUndoRedoPlugin::refresh()
{
    SKGTRACEIN ( 10, "SKGUndoRedoPlugin::refresh" );

    if ( currentDocument ) {
        bool undoPossible= ( currentDocument->getNbTransaction ( SKGDocument::UNDO ) >0 );
        if ( undoSaveAction ) undoSaveAction->setEnabled ( undoPossible );
        if ( undoAction ) undoAction->setEnabled ( undoPossible );
        if ( redoAction ) redoAction->setEnabled ( currentDocument->getNbTransaction ( SKGDocument::REDO ) >0 );

        //Refresh undo redo
        QString name;
        currentDocument->getTransactionToProcess ( SKGDocument::UNDO, &name );
        QString message = i18nc ("Verb",  "Undo operation '%1'.", name );
        if ( !name.length() ) message="";
        if ( undoAction ) undoAction->setStatusTip ( message );

        currentDocument->getTransactionToProcess ( SKGDocument::REDO, &name );
        message = i18nc ("Verb",  "Redo operation '%1'.", name );
        if ( !name.length() ) message="";
        if ( redoAction ) redoAction->setStatusTip ( message );
    }
}


QWidget* SKGUndoRedoPlugin::getPreferenceWidget()
{
    SKGTRACEIN ( 10, "SKGUndoRedoPlugin::getPreferenceWidget" );
    //Read Setting
    if ( currentDocument ) {
        KSharedConfigPtr config=KSharedConfig::openConfig ();
        KConfigGroup pref=config->group ( "skg_undoredo" );
        pref.writeEntry ( "maxNumberOfUndo",SKGServices::stringToInt ( currentDocument->getParameter ( "SKG_UNDO_MAX_DEPTH" ) ) );
        pref.writeEntry ( "cleanHistoryOnSave",(currentDocument->getParameter ( "SKG_UNDO_CLEAN_AFTER_SAVE" )=="Y") );
    }

    //Create widget
    QWidget* widget=new QWidget();
    ui.setupUi ( widget );
    return widget;
}

KConfigSkeleton* SKGUndoRedoPlugin::getPreferenceSkeleton()
{
    return skgundoredo_settings::self();
}

SKGError SKGUndoRedoPlugin::savePreferences() const
{
    SKGError err;
    if ( currentDocument ) {
        //Read Setting
        QString max = SKGServices::intToString (skgundoredo_settings::maxNumberOfUndo());
        QString clean = (skgundoredo_settings::cleanHistoryOnSave() ? "Y" : "N");

        //Save setting in document
        if ( max!=currentDocument->getParameter ( "SKG_UNDO_MAX_DEPTH" ) ) err=currentDocument->setParameter ( "SKG_UNDO_MAX_DEPTH", max );
        if ( clean!=currentDocument->getParameter ( "SKG_UNDO_CLEAN_AFTER_SAVE" ) ) err=currentDocument->setParameter ( "SKG_UNDO_CLEAN_AFTER_SAVE",clean );
    }
    return err;
}

QString SKGUndoRedoPlugin::title() const
{
    return i18nc("Noun", "History" );
}

QString SKGUndoRedoPlugin::icon() const
{
    return "edit-undo";
}

QString SKGUndoRedoPlugin::toolTip () const
{
    return i18nc("Noun", "History" );
}

QStringList SKGUndoRedoPlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... you can undo and redo your modifications.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can modify the maximum size of the undo/redo stack in the settings.</p>" ) );
    return output;
}

int SKGUndoRedoPlugin::getOrder() const
{
    return 4;
}

void SKGUndoRedoPlugin::close()
{
    SKGTRACEIN ( 10, "SKGUndoRedoPlugin::close" );
}

QList< SKGInterfacePlugin::SKGAdvice > SKGUndoRedoPlugin::advices() const
{
    QList< SKGInterfacePlugin::SKGAdvice > output;
    //Get nb transaction
    int nbObject=currentDocument->getNbTransaction();
    int priority=qMin(10, nbObject/50);
    if (priority>0)
    {
        SKGInterfacePlugin::SKGAdvice ad;
        ad.uuid="skgundoredoplugin_too_big";
        ad.priority=priority;
        ad.shortMessage=i18nc("Advice on making the best (short)", "History is too large");
        ad.longMessage=i18nc("Advice on making the best (long)", "You can improve performances by reducing your history size in settings.");
        ad.autoCorrection.push_back(i18nc("Advice on making the best (action)", "Clear history"));
        ad.autoCorrection.push_back(i18nc("Advice on making the best (action)", "Open settings panel"));
        output.push_back(ad);
    }

    return output;
}

SKGError SKGUndoRedoPlugin::executeAdviceCorrection(const QString& iAdviceIdentifier, int iSolution) const
{
    if (currentDocument && iAdviceIdentifier=="skgundoredoplugin_too_big")
    {
        if (iSolution==0)
        {
            QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
            SKGError err=currentDocument->removeAllTransactions();
            QApplication::restoreOverrideCursor();

            //status bar
            if (err.isSucceeded()) err=SKGError(0, i18nc("Message for successful user action","Clear history successfully done."));
            else err.addError(ERR_FAIL, i18nc("Error message","Clear history failed"));

            //Display error
            SKGMainPanel::getMainPanel()->displayErrorMessage(err);
        }
        else SKGMainPanel::getMainPanel()->optionsPreferences(this->objectName());
        return SKGError();
    }
    return SKGInterfacePlugin::executeAdviceCorrection(iAdviceIdentifier, iSolution);
}

void SKGUndoRedoPlugin::actionUndoSave()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGUndoRedoPlugin::actionUndoSave",err );
    if ( currentDocument && SKGMainPanel::getMainPanel() ) {
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        err=currentDocument->undoRedoTransaction ( SKGDocument::UNDOLASTSAVE );
        QApplication::restoreOverrideCursor();

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Undo successfully done." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Undo failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGUndoRedoPlugin::actionUndo()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGUndoRedoPlugin::actionUndo",err );
    if ( currentDocument && SKGMainPanel::getMainPanel() ) {
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        int pos= ( ( QAction* ) sender() )->data().toInt();
        for ( int i = 1 ; err.isSucceeded() && i <=pos; ++i )
            err=currentDocument->undoRedoTransaction ( SKGDocument::UNDO );
        QApplication::restoreOverrideCursor();

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Undo successfully done." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Undo failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGUndoRedoPlugin::onShowUndoMenu()
{
    if ( undoMenu ) {
        undoMenu->clear();
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder ( currentDocument,
                                                "SELECT t_name, t_savestep FROM doctransaction WHERE t_mode='U' ORDER BY d_date DESC LIMIT 7",
                                                listTmp );
        int nb=listTmp.count();
        for ( int i=1; i<nb; ++i ) {
            QAction* act=undoMenu->addAction ( listTmp.at ( i ).at ( 1 )=="Y" ?  KIcon("document-revert") : KIcon("edit-undo"), listTmp.at ( i ).at ( 0 ));
            if ( act ) {
                act->setData ( i );
                connect ( act, SIGNAL ( triggered() ), this, SLOT ( actionUndo() ) );
            }
        }
    }
}
void SKGUndoRedoPlugin::actionRedo()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGUndoRedoPlugin::actionRedo",err );
    if ( currentDocument && SKGMainPanel::getMainPanel() ) {
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        int pos= ( ( QAction* ) sender() )->data().toInt();
        for ( int i = 1 ; err.isSucceeded() && i <=pos; ++i )
            err=currentDocument->undoRedoTransaction ( SKGDocument::REDO );
        QApplication::restoreOverrideCursor();

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Redo successfully done." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Redo failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGUndoRedoPlugin::onShowRedoMenu()
{
    if ( redoMenu ) {
        redoMenu->clear();
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder ( currentDocument,
                                                "SELECT t_name FROM doctransaction WHERE t_mode='R' ORDER BY d_date ASC LIMIT 7",
                                                listTmp );
        int nb=listTmp.count();
        for ( int i=1; i<nb; ++i ) {
            QAction* act=redoMenu->addAction ( KIcon("edit-redo"), listTmp.at ( i ).at ( 0 ) );
            if ( act ) {
                act->setData ( i );
                connect ( act, SIGNAL ( triggered() ), this, SLOT ( actionRedo() ) );
            }
        }
    }
}

QDockWidget* SKGUndoRedoPlugin::getDockWidget()
{
    return dockWidget;
}
#include "skgundoredoplugin.moc"
