/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for unit management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgunitplugin.h"
#include "skgunitpluginwidget.h"
#include "skgunitboardwidget.h"
#include "skgunit_settings.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"

#include <kinputdialog.h>
#include <kactioncollection.h>
#include <kaction.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGUnitPluginFactory, registerPlugin<SKGUnitPlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGUnitPluginFactory("skrooge_unit", "skrooge_unit"))

SKGUnitPlugin::SKGUnitPlugin(QObject* iParent, const QVariantList& /*iArg*/)
                :SKGInterfacePlugin(iParent), currentBankDocument(NULL),
                splitShareAction(NULL)
{
        SKGTRACEIN(10, "SKGUnitPlugin::SKGUnitPlugin");
}

SKGUnitPlugin::~SKGUnitPlugin()
{
        SKGTRACEIN(10, "SKGUnitPlugin::~SKGUnitPlugin");
        currentBankDocument=NULL;
        splitShareAction=NULL;
}

void SKGUnitPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGUnitPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;


        setComponentData( SKGUnitPluginFactory::componentData() );

        setXMLFile("skrooge_unit.rc");

        //Menu
        splitShareAction = new KAction(KIcon("skrooge_split_stock"), i18nc("Verb","Split share..."), this);
        connect(splitShareAction, SIGNAL(triggered(bool)), this, SLOT(actionSplitShare()));
        actionCollection()->addAction( QLatin1String("edit_split_stock"), splitShareAction );
        splitShareAction->setShortcut(Qt::ALT+Qt::Key_Slash);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("edit_split_stock", splitShareAction);
}

void SKGUnitPlugin::close()
{
        SKGTRACEIN(10, "SKGUnitPlugin::close");
}

int SKGUnitPlugin::getNbDashboardWidgets()
{
        return 1;
}

QString SKGUnitPlugin::getDashboardWidgetTitle(int iIndex)
{
        if (iIndex==0) return i18nc("Noun, the title of a section", "Shares && Indexes");
        return "";
}

SKGWidget* SKGUnitPlugin::getDashboardWidget(int iIndex)
{
        if (iIndex==0) return new SKGUnitBoardWidget(currentBankDocument);
        return NULL;
}

void SKGUnitPlugin::refresh()
{
        SKGTRACEIN(10, "SKGUnitPlugin::refresh");
        if (SKGMainPanel::getMainPanel() && currentBankDocument) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (splitShareAction) {
                        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                        if (selection.count()==1 && selection.at(0).getRealTable()=="unit") {
                                SKGUnitObject unit=selection.at(0);
                                splitShareAction->setEnabled(unit.getType()==SKGUnitObject::SHARE);
                        } else {
                                splitShareAction->setEnabled(false);
                        }
                }

                //Automatic download
                if (test) {
                        QString doc_id=currentBankDocument->getUniqueIdentifier();
                        if (docUniqueIdentifier!=doc_id) {
                                docUniqueIdentifier=doc_id;

                                //Read Setting
                                if (skgunit_settings::download_on_open()) {
                                        SKGError err;
                                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

                                        //download all units
                                        SKGObjectBase::SKGListSKGObjectBase selection;
                                        err=SKGObjectBase::getObjects(currentBankDocument, "unit","", selection);
                                        int nb=selection.count();
                                        for (int i=0; err.isSucceeded() && i<nb; ++i) {
                                                SKGUnitObject unit=selection.at(i);
                                                err=SKGUnitPluginWidget::downloadUnitValue(unit);
                                        }
                                        QApplication::restoreOverrideCursor();

                                        //Display error
                                        SKGMainPanel::getMainPanel()->displayErrorMessage(err);
                                }
                        }
                }
        }
}

SKGTabPage* SKGUnitPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGUnitPlugin::getWidget");
        return new SKGUnitPluginWidget((SKGDocumentBank*) currentBankDocument);
}

QWidget* SKGUnitPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGUnitPlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);

        return widget;
}

KConfigSkeleton* SKGUnitPlugin::getPreferenceSkeleton()
{
        return skgunit_settings::self();
}

QString SKGUnitPlugin::title() const
{
        return i18nc("Noun, units for operations, usually currencies or a shares","Units");
}

QString SKGUnitPlugin::icon() const
{
        return "skrooge_unit";
}

QString SKGUnitPlugin::toolTip () const
{
        return i18nc("A tool tip", "Unit management");
}

QStringList SKGUnitPlugin::tips() const
{
        QStringList output;
        output.push_back(i18nc("Description of a tips", "<p>... you can download units.</p>"));
        output.push_back(i18nc("Description of a tips", "<p>... units can be downloaded automatically when a document is opened.</p>"));
        output.push_back(i18nc("Description of a tips", "<p>... you can split a share.</p>"));
        return output;
}

int SKGUnitPlugin::getOrder() const
{
        return 60;
}

bool SKGUnitPlugin::isInContext() const
{
        return true;
}

void SKGUnitPlugin::actionSplitShare()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGUnitPlugin::actionSplitShare",err);

        //Get Selection
        if (SKGMainPanel::getMainPanel()) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                if (nb==1) {
                        bool ok=false;
                        double ratio=KInputDialog::getDouble  (i18nc("Question", "Split share"),
                                                               i18nc("Question", "Ratio (2 means 2-for-1, 0.5 means 1-for-2) :"), 2.0,
                                                               0, DBL_MAX, 8, &ok, SKGMainPanel::getMainPanel());
                        if (ok) {
                                SKGUnitObject unit=selection.at(0);
                                SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Split stock [%1] by [%2]", unit.getName(), ratio), err);
                                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                                err=unit.split(ratio);
                                QApplication::restoreOverrideCursor();
                        }
                }

                //status
                if (err.isSucceeded())  err=SKGError(0, i18nc("Successful message after an user action", "Stock split."));
                else err.addError(ERR_FAIL, i18nc("Error message", "Splitting stock failed."));

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage(err);
        }
}

#include "skgunitplugin.moc"
