/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A dashboard for skrooge.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgdashboardpluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skginterfaceplugin.h"
#include "skgservices.h"

#include <kmenu.h>

#include <QDomDocument>
#include <QSpacerItem>

SKGDashboardPluginWidget::SKGDashboardPluginWidget ( SKGDocument* iDocument )
        : SKGTabPage ( iDocument ), gridLayout ( NULL ), hSpacer ( NULL ), vSpacer ( NULL ), menu ( NULL ), addMenu ( NULL ), removeMenu ( NULL ),
        col ( 0 ), row ( 0 ), nbMaxCol ( 3 )
{
    SKGTRACEIN ( 1, "SKGDashboardPluginWidget::SKGDashboardPluginWidget" );

    ui.setupUi ( this );

    ui.kTitle->setPixmap ( KIcon ( "skrooge" ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
    ui.kTitle->setComment ( "<html><body><b>"+i18nc ( "Message", "Welcome to Skrooge" ) +"</b></body></html>" );

    //Build menu
    setContextMenuPolicy ( Qt::CustomContextMenu );
    menu = new KMenu ( this );
    connect ( this ,SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showHeaderMenu ( const QPoint& ) ) );

    addMenu=menu->addMenu ( KIcon ( "list-add" ),  i18nc ( "Verb", "Add" ) );
    //Build menu
    if ( addMenu )
    {
        addMenu->clear();

        int index=1;
        while ( index>0 )
        {
            SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByIndex ( index );
            if ( plugin )
            {
                int nbdbw=plugin->getNbDashboardWidgets();
                for ( int j=0; j<nbdbw; ++j )
                {
                    //Create menu
                    QAction* act = addMenu->addAction ( plugin->getDashboardWidgetTitle ( j ) );
                    act->setIcon ( KIcon ( plugin->icon() ) );
                    act->setData ( plugin->objectName() +'-'+SKGServices::intToString ( j ) );

                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onAddWidget() ) );
                }
            }
            else index=-1;
            ++index;
        }
    }

    removeMenu=menu->addMenu ( KIcon ( "list-remove" ),  i18nc ( "Verb", "Remove" ) );
    previousMenu=menu->addMenu ( KIcon ( "arrow-left" ),  i18nc ( "Verb", "Move before" ) );
    nextMenu=menu->addMenu ( KIcon ( "arrow-right" ),  i18nc ( "Verb", "Move after" ) );
    QMenu* colMenu=menu->addMenu ( KIcon ( "view-split-left-right" ), i18nc ( "Verb", "Number of columns" ) );
    for ( int i=1; i<=5; ++i )
    {
        //Create menu
        QAction* act = colMenu->addAction ( SKGServices::intToString ( i ) );
        if ( act )
        {
            act->setData ( i );
            connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onColumnNbChanged() ) );
        }
    }

    //Plug buttons with menus
    ui.kAddWidget->setIcon ( addMenu->icon() );
    ui.kAddWidget->setMenu ( addMenu );
    ui.kAddWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kRemoveWidget->setIcon ( removeMenu->icon() );
    ui.kRemoveWidget->setMenu ( removeMenu );
    ui.kRemoveWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kPreviousWidget->setIcon ( previousMenu->icon() );
    ui.kPreviousWidget->setMenu ( previousMenu );
    ui.kPreviousWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kNextWidget->setIcon ( nextMenu->icon() );
    ui.kNextWidget->setMenu ( nextMenu );
    ui.kNextWidget->setPopupMode ( QToolButton::InstantPopup );

    ui.kNbColumns->setIcon ( colMenu->icon() );
    ui.kNbColumns->setMenu ( colMenu );
    ui.kNbColumns->setPopupMode ( QToolButton::InstantPopup );

    //Build layout
    gridLayout = new QGridLayout ( ui.kContent );
}

SKGDashboardPluginWidget::~SKGDashboardPluginWidget()
{
    SKGTRACEIN ( 1, "SKGDashboardPluginWidget::~SKGDashboardPluginWidget" );
    menu=NULL;
    addMenu=NULL;
    removeMenu=NULL;
    nextMenu=NULL;
    previousMenu=NULL;
    gridLayout=NULL;
    hSpacer=NULL;
    vSpacer=NULL;
}

QString SKGDashboardPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGDashboardPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "nbMaxCol", SKGServices::intToString ( nbMaxCol ) );

    int nb=items.count();
    for ( int i=0; i<nb; ++i )
    {
        QDomElement element = doc.createElement ( "ITEM-"+SKGServices::intToString ( i+1 ) );
        root.appendChild ( element );

        QStringList param=SKGServices::splitCSVLine ( items.at ( i ),'-' );
        SKGWidget* item=itemsPointers.at ( i );
        if ( item )
        {
            element.setAttribute ( "name", param.at ( 0 ) );
            element.setAttribute ( "index", param.at ( 1 ) );
            element.setAttribute ( "state", item->getState() );
        }
    }

    return doc.toString();
}

void SKGDashboardPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGDashboardPluginWidget::setState" );

    //Initialisation
    int nb=items.count();
    for ( int i=0; gridLayout && i<nb; ++i )
    {
        SKGWidget* item=itemsPointers.at ( i );
        if ( item )
        {
            gridLayout->removeWidget ( item );
            item->hide();
            delete item;
        }
    }
    if ( hSpacer )
    {
        gridLayout->removeItem ( hSpacer );
        delete hSpacer;
        hSpacer=NULL;
    }
    if ( vSpacer )
    {
        gridLayout->removeItem ( vSpacer );
        delete vSpacer;
        vSpacer=NULL;
    }
    removeMenu->clear ();
    nextMenu->clear();
    previousMenu->clear();
    items.clear();
    itemsPointers.clear();
    row=0;
    col=0;

    if ( iState.isEmpty() )
    {
        int index=1;
        while ( index>0 )
        {
            SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByIndex ( index );
            if ( plugin )
            {
                int nb=plugin->getNbDashboardWidgets();
                for ( int j=0; j<nb; ++j )
                {
                    addItem ( plugin, j );
                }
            }
            else index=-1;
            ++index;
        }
    }
    else
    {
        QDomDocument doc ( "SKGML" );
        doc.setContent ( iState );
        QDomElement root = doc.documentElement();

        QString nbMaxColS=root.attribute ( "nbMaxCol" );
        if ( !nbMaxColS.isEmpty() ) nbMaxCol=SKGServices::stringToInt ( nbMaxColS );

        int index=1;
        while ( index>0 )
        {
            QDomElement element=root.firstChildElement ( "ITEM-"+SKGServices::intToString ( index ) );
            if ( !element.isNull() )
            {
                SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByName ( element.attribute ( "name" ) );
                QString index=element.attribute ( "index" );
                if ( index.isEmpty() ) index='0';
                if ( plugin ) addItem ( plugin, SKGServices::stringToInt ( index ), element.attribute ( "state" ) );
            }
            else index=-1;
            ++index;
        }
    }


    //Add horizontal spacer
    if ( items.count() <nbMaxCol )
    {
        hSpacer=new QSpacerItem ( 0, 0, QSizePolicy::Expanding );
        gridLayout->addItem ( hSpacer, row, col, QSizePolicy::Expanding );
    }

    //Add vertical spacer
    vSpacer=new QSpacerItem ( 0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding );
    gridLayout->addItem ( vSpacer, row+1, 0, QSizePolicy::Minimum, QSizePolicy::Expanding );
}

QString SKGDashboardPluginWidget::getDefaultStateAttribute()
{
    return "SKGDASHBOARD_DEFAULT_PARAMETERS";
}

QWidget* SKGDashboardPluginWidget::getWidgetForPrint()
{
    return this;
}

void SKGDashboardPluginWidget::refresh()
{
    SKGTRACEIN ( 1, "SKGDashboardPluginWidget::refresh" );

}

void SKGDashboardPluginWidget::showHeaderMenu ( const QPoint& pos )
{
    //Display menu
    if ( menu ) menu->popup ( mapToGlobal ( pos ) );
}

void SKGDashboardPluginWidget::onAddWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        QString id=sender->data().toString();
        QStringList param=SKGServices::splitCSVLine ( id,'-' );

        SKGInterfacePlugin* db=SKGMainPanel::getMainPanel()->getPluginByName ( param.at ( 0 ) );
        if ( db )
        {
            SKGWidget* dbw=db->getDashboardWidget ( SKGServices::stringToInt ( param.at ( 1 ) ) );
            if ( dbw )
            {
                //Add
                items.push_back ( id );
                itemsPointers.push_back ( dbw );

                //Get new state
                QString state=getState();

                //Remove added item
                items.pop_back();
                itemsPointers.pop_back();

                //Reset state
                setState ( state );
            }
        }
    }
}

void SKGDashboardPluginWidget::onColumnNbChanged()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        nbMaxCol=sender->data().toInt();

        setState ( getState() );
    }
}

void SKGDashboardPluginWidget::onMoveAfterWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>=0 && pos<items.count()-1 ) //last one can not be moved after
        {
            //Get item
            QString id=items.at ( pos );
            SKGWidget* wgt=itemsPointers.at ( pos );

            //Remove item
            items.removeAt ( pos );
            itemsPointers.removeAt ( pos );

            //Add it after
            items.insert ( pos+1, id );
            itemsPointers.insert ( pos+1, wgt );

            //Get new state
            QString state=getState();

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::onMoveBeforeWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>0 && pos<=items.count()-1 ) //first one can not be moved before
        {
            //Get item
            QString id=items.at ( pos );
            SKGWidget* wgt=itemsPointers.at ( pos );

            //Remove item
            items.removeAt ( pos );
            itemsPointers.removeAt ( pos );

            //Add it after
            items.insert ( pos-1, id );
            itemsPointers.insert ( pos-1, wgt );

            //Get new state
            QString state=getState();

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::onRemoveWidget()
{
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender )
    {
        int pos=sender->data().toInt();
        if ( pos>=0 )
        {
            //Get item
            QString id=items.at ( pos );
            SKGWidget* wgt=itemsPointers.at ( pos );

            //Remove item
            items.removeAt ( pos );
            itemsPointers.removeAt ( pos );

            //Get new state
            QString state=getState();

            //We add removed object to be sure that it will be removed
            items.push_back ( id );
            itemsPointers.push_back ( wgt );

            //Reset state
            setState ( state );
        }
    }
}

void SKGDashboardPluginWidget::addItem ( SKGInterfacePlugin* iDashboard, int iIndex, const QString& iState )
{
    if ( iDashboard )
    {
        SKGWidget* dbw=iDashboard->getDashboardWidget ( iIndex );
        if ( dbw )
        {
            //Remove Spacer
            if ( hSpacer )
            {
                delete hSpacer;
                hSpacer=NULL;
            }
            if ( vSpacer )
            {
                delete vSpacer;
                vSpacer=NULL;
            }

            //Add widget
            dbw->setParent ( ui.kContent );
            dbw->setState ( iState );
            gridLayout->addWidget ( dbw, row, col );

            ++col;
            if ( col==nbMaxCol )
            {
                ++row;
                col=0;
            }

            //Add remove item in menu
            QString id=iDashboard->objectName() +'-'+SKGServices::intToString ( iIndex );
            {
                QAction* act=removeMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onRemoveWidget() ) );
                }
            }

            //Add move next item in menu
            {
                QAction* act=nextMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onMoveAfterWidget() ) );
                }
            }

            //Add remove item in menu
            {
                QAction* act=previousMenu->addAction ( iDashboard->getDashboardWidgetTitle ( iIndex ) );
                if ( act )
                {
                    act->setIcon ( KIcon ( iDashboard->icon() ) );
                    act->setData ( items.count() );
                    connect ( act, SIGNAL ( triggered ( bool ) ), this, SLOT ( onMoveBeforeWidget() ) );
                }
            }

            items.push_back ( id );
            itemsPointers.push_back ( dbw );
        }
    }
}

#include "skgdashboardpluginwidget.moc"
