/////////////////////////////////////////////////////////////////////////////
// File:        basetypeimpl.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2011 Cesar Mauri Loba - CREA Software Systems
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_BASETYPEIMPL_H
#define SPCORE_BASETYPEIMPL_H

#include "spcore/basetype.h"
#include "spcore/coreruntime.h"
#include "spcore/pinimpl.h"
#include <boost/type_traits/is_pointer.hpp>

namespace spcore {

/**
 Template for basic types factories
*/
template<class T> class SimpleTypeFactory : public ITypeFactory
{
public:
	virtual const char * GetName() const {
		return T::getTypeName();
	}
	virtual SmartPtr<CTypeAny> CreateInstance(int id) {
		return SmartPtr<CTypeAny>(new T(id), false);
	}
};

/**
	Adapter to implement composite types. For non-composite type inherit directly from CTypeAny
*/
class CompositeTypeAdapter : public CTypeAny
{
public:
	virtual int AddChild(SmartPtr<CTypeAny> component) {
		std::vector<CTypeAny*>::iterator it=
			find(m_children.begin(), m_children.end(), component.get());
		if (it!= m_children.end()) return -1;	// Already registered
		component->AddRef();
		m_children.push_back(component.get());
		return 0;
	}
	virtual SmartPtr<IIterator<CTypeAny*> > QueryChildren() const {
		return SmartPtr<IIterator<CTypeAny*> > (new CIteratorVector<CTypeAny*>(m_children), false);		
	}
protected:
	CompositeTypeAdapter(int id) : CTypeAny(id) {}
	virtual ~CompositeTypeAdapter() {
		std::vector<CTypeAny*>::iterator it= m_children.begin();
		for (; it!= m_children.end(); ++it) (*it)->Release();
	}

	virtual bool CopyTo ( CTypeAny& dst, bool recurse) const {
		assert (dst.GetTypeID()== this->GetTypeID());
		assert (this!= &dst);

		CompositeTypeAdapter* dst_cast= sptype_static_cast<CompositeTypeAdapter>(&dst);
				
		// Contents are copyed by the derived class that should call this method

		if (recurse) {
			// 
			// composite copy of a composite type. update structure and contents recursively 
			//
			std::vector<CTypeAny*>::const_iterator it_src= this->m_children.begin();
			std::vector<CTypeAny*>::iterator it_dst= dst_cast->m_children.begin();	

			// While both have children just copy
			while (it_src!= this->m_children.end() && it_dst!= dst_cast->m_children.end()) {
				SmartPtr<CTypeAny> newInstance= (*it_src)->Clone (*it_dst, recurse);
				if (newInstance.get()== NULL) {
					// Should not happen
					assert (false);
					return false;
				}
				if (newInstance.get()!= *it_dst) {
					// TODO: remove the following assert which is only intended to
					// detect when this portion of code is used. Note that this means
					// that the old dst had a different type form the src which is
					// strange but not, in principle, wrong
					assert (false);

					// Copy process created a new instance. Substitute in vector and
					// and destroy the old one
					(*it_dst)->Release();
					*it_dst= newInstance.get();
					(*it_dst)->AddRef();
				}

				++it_src;
				++it_dst;		
			}

			if (it_src!= this->m_children.end()) {
				// src has more children than dst, create and clone
				while (it_src!= this->m_children.end()) {
					SmartPtr<CTypeAny> new_child= (*it_src)->Clone (NULL, recurse);
					if (new_child.get()== NULL) {
						// Should not happen
						assert (false);
						return false;
					}
					new_child.get()->AddRef();	// Goes to non smart ptr'ed vector
					dst_cast->m_children.push_back (new_child.get());
					
					++it_src;
				}
			}
			else {
				// dst has remaining children that need to be removed
				while (it_dst!= dst_cast->m_children.end()) {
					(*it_dst)->Release();
					it_dst= dst_cast->m_children.erase(it_dst);
				}
			}
		}
		else {
			// shallow copy, remove destination children if any
			std::vector<CTypeAny*>::iterator it_dst= dst_cast->m_children.begin();
			while (it_dst!= dst_cast->m_children.end()) {
				(*it_dst)->Release();
				it_dst= dst_cast->m_children.erase(it_dst);
			}
		}	

		return true;
	}

private:
	std::vector<CTypeAny*> m_children;
};


/**
	Template to define basic type common static operations
*/
template <class BASE, class RESULT>
class SimpleTypeBasicOperations
{
public:
	static int getTypeID() { 
		static int typeID= TYPE_INVALID;
		if (typeID== TYPE_INVALID) typeID= getSpCoreRuntime()->ResolveTypeID(BASE::getTypeName());
		return typeID;
	}
	static SmartPtr<RESULT> CreateInstance() {
		int typeID= SimpleTypeBasicOperations::getTypeID();
		if (typeID== TYPE_INVALID) return SmartPtr<RESULT>(NULL, false);    			
		return SmartPtr<RESULT>(static_cast<RESULT *>(getSpCoreRuntime()->CreateTypeInstance(typeID).get()));		
	}
	// Note that the two methods below can throw although 
	// shouldn't unless a fatal error ocurrs
	static SmartPtr<IOutputPin> CreateOutputPin(const char* name) {
		return SmartPtr<IOutputPin>(new COutputPin (name, BASE::getTypeName()), false);
	}
	static SmartPtr<IOutputPin> CreateOutputPinLock(const char* name) {
		return SmartPtr<IOutputPin>(new COutputPinLock (name, BASE::getTypeName()), false);
	}
};

/**
	Template to define basic types
*/

template<class BASE>
class SimpleType : public BASE, public SimpleTypeBasicOperations<BASE, SimpleType<BASE> >
{
	friend class SimpleTypeFactory<SimpleType>;
protected:
	SimpleType(int typeID) : BASE(typeID) {}
		
private:
	// Disable default copy constructor and operator
	SimpleType( const SimpleType& );      // not implemented
	SimpleType& operator=( const SimpleType& );     // not implemented
};

//
// Contents template class for scalar types
//
template<class T>
class ScalarTypeContents : public CTypeAny {
	BOOST_STATIC_ASSERT (!boost::is_pointer<T>::value);
public:	
	virtual const T getValue() const { return m_value; }
	virtual void setValue(T value) { m_value= value; }
protected:
	ScalarTypeContents(int id) : CTypeAny(id) {
		m_value= 0;
	}
	
	virtual bool CopyTo ( CTypeAny& dst, bool ) const {
		assert (dst.GetTypeID()== this->GetTypeID());
		sptype_static_cast<ScalarTypeContents>(&dst)->m_value= m_value;
		return true;
	}

	// Destructor not needed
    T m_value;
};


} // namespace spcore
#endif
