/* 
   sitecopy, for managing remote web sites. Generic(ish) FTP routines.
   Copyright (C) 1998-2002, Joe Orton <joe@manyfish.co.uk> (except
   where otherwise indicated).
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  

*/

/* This contains an FTP client implementation.
 * It performs transparent connection management - it it dies,
 * it will be reconnected automagically.
 */

#include <config.h>

#include <sys/types.h>

#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <time.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifdef HAVE_SNPRINTF_H
#include "snprintf.h"
#endif

#include <ne_alloc.h>
#include <ne_string.h>
#include <ne_socket.h>

#include "common.h"
#include "frontend.h"
#include "ftp.h"
#include "protocol.h"
#include "i18n.h"

struct ftp_session_s {
    /* User options */
    unsigned int use_passive:1;
    unsigned int echo_quit:1;

    unsigned int connection:1; /* true when open */

    /* Userdata passed to fe_authenticate call */
    void *feauth_userdata;
    
     /* The address of the server for the DTP connection - 
      * this goes here because it's got from ftp_read() rather than in a decent
      * manner */
    const char *hostname; /* for fe_login */
    unsigned short dtp_port;
    struct in_addr dtp_addr;
    nsocket *dtp_socket;
    
    struct in_addr pi_addr;
    unsigned short pi_port;
    nsocket *pi_socket;

    /* Stores the current transfer type is:
     *  -1: Unknown
     *   0: Binary
     *   1: ASCII
     */
    int using_ascii;

    /* time from MDTM response... bit crap having this here. */
    time_t get_modtime;
    
    /* remember these... we may have to log in more than once. */
    char username[FE_LBUFSIZ], password[FE_LBUFSIZ];

    unsigned int echo_response:1;

    /* Receive buffer */
    char rbuf[BUFSIZ];

    /* Error string */
    char error[BUFSIZ];
};

static int ftp_close(ftp_session *sess);

/* Sets error string */
static void ftp_seterror(ftp_session *sess, const char *error);
/* Sets error string, appending strerror(saved_errno) */
static void ftp_seterror_err(ftp_session *sess, const char *error,
			     int saved_errno);

static void handle_socket_error(ftp_session *sess, nsocket *sock, 
				const char *doing, int err);

static int ftp_login(ftp_session *sess); /* Performs the login procedure */

static int ftp_settype(ftp_session *sess, int ascii);
static int ftp_active_open(ftp_session *sess, const char *cmd);
/* Opens the data connection */
static int ftp_data_open(ftp_session *sess, const char *parm, ...) 
#ifdef __GNUC__
                __attribute__ ((format (printf, 2, 3)))
#endif /* __GNUC__ */
;

static int ftp_data_close(ftp_session *sess); /* Closes the data connection */
static int ftp_connect_pasv(ftp_session *sess);

/* Parses a PASV response into the DTP address + port */
static int ftp_read_pasv(const char *, ftp_session *sess);
/* Parses a MDTM response into a time_t */
static int ftp_read_mdtm(char *response, time_t *modtime);

/* Gets the return code */
static int ftp_response(ftp_session *sess, char *, const int);

/* Used by ftp_response to get the actual code */
static int get_reply_code(const char *); 

/* Execute a command, reads response */

static int ftp_exec(ftp_session *sess, const char *, ...)
#ifdef __GNUC__
                __attribute__ ((format (printf, 2, 3)))
#endif /* __GNUC__ */
;

static int ftp_read(ftp_session *sess);
static int get_modtime(ftp_session *sess, const char *root,
		       const char *filename);
static mode_t ftp_decode_perms(const char *perms);

/* Dump the given filename down the given socket, in ASCII translation
 * mode.  Performs LF -> CRLF conversion.
 * Returns zero on success or non-zero on error. */
static int send_file_ascii(int fd, nsocket *sock, int size) 
{
    int ret;
    char buffer[BUFSIZ], *pnt;
    ssize_t total = 0, lasttotal = 0;
    FILE *f;
    
    f = fdopen(fd, "r");
    if (f == NULL) return -1;

    /* Init to success */
    ret = 0;

    while (1) {
	if (fgets(buffer, BUFSIZ - 1, f) == NULL) {
	    if (ferror(f)) {
		ret = -1;
		break;
	    }
	    /* Finished upload */
	    ret = 0;
	    break;
	}
	/* To send in ASCII mode, we need to send CRLF as the EOL.
	 * We might or might not already have CRLF-delimited lines.
	 * So we mess about a bit to ensure that we do.
	 */
	pnt = strchr(buffer, '\r');
	if (pnt == NULL) {
	    /* We need to add the CR in */
	    pnt = strchr(buffer, '\n');
	    if (pnt == NULL) {
		/* No CRLF found at all */
		pnt = strchr(buffer, '\0');
		if (pnt == NULL) /* crud in buffer */
		    pnt = buffer;
	    }
	    /* Now, pnt points to the first character after the 
	     * end of the line, i.e., where we want to put the CR.
	     */
	    *pnt++ = '\r';
	    /* And lob in an LF afterwards */
	    *pnt-- = '\n';
	}
	/* At this point, pnt points to the CR.
	 * We send everything between pnt and the beginning of the buffer,
	 * +2 for the CRLF
	 */
	if (sock_fullwrite(sock, buffer, (pnt - buffer) +2) != 0) {
	    ret = -1;
	    break;
	}
	total += (pnt - buffer) + 2;
	/* Only call progress every 4K otherwise it is way too 
	 * chatty. */
	if (total > lasttotal + 4096) {
	    lasttotal = total;
	    sock_call_progress(sock, total, -1);
	}
    }
    fclose(f); /* any point in checking that one? */
    /* Return true */
    return ret;
}

/* Dump from given socket into given file. 
 * Returns number of bytes written on success, or -1 on error */
static int recv_file(nsocket *sock, const char *filename)
{
    int fd, rd, wr = 0;
    off_t count = 0;
    char buffer[BUFSIZ];
    fd = open(filename, O_WRONLY|O_TRUNC|O_CREAT|OPEN_BINARY_FLAGS, 0644);
    if (fd < 0) {
	return -1;
    }
    do {
	rd = sock_read(sock, buffer, BUFSIZ);
	if (rd == SOCK_CLOSED) {
	    /* The EOF condition, this is good. */
	    break;
	} else if (rd < 0) {
	    (void) close(fd);
	    return rd;
	}
	/* count up */
	count += rd;
	sock_call_progress(sock, count, -1);
	/* now write it out */
	if (rd > 0) {
	    char *pnt = buffer;
	    int left = rd;
	    do {
		wr = write(fd, pnt, left);
		pnt += wr;
		left -= wr;
	    } while (wr > 0 && left > 0);
	}
    } while (rd > 0 && wr > 0);
    
    if (close(fd) == -1 || wr < 0) {
	/* Close failed - file was not written correctly */
	return -1;
    }

    return 0;
}

/* Initializes the driver + connection.
 * Returns FTP_OK on success, or FTP_LOOKUP if the hostname
 * could not be resolved.
 */
ftp_session *ftp_init(void)
{
    ftp_session *sess = ne_calloc(sizeof(ftp_session));
    sess->connection = 0;
    sess->using_ascii = -1; /* unknown */
    return sess;
}

void ftp_set_passive(ftp_session *sess, int use_passive) 
{
    sess->use_passive = use_passive;
}

int ftp_set_server(ftp_session *sess, struct site_host *server)
{
    if (server->username) {
	strcpy(sess->username, server->username);
    }
    if (server->password) {
	strcpy(sess->password, server->password);
    }
    sess->hostname = server->hostname;
    sess->pi_port = server->port;
    fe_connection(fe_namelookup, server->hostname);
    if (sock_name_lookup(server->hostname, &sess->pi_addr))
	return FTP_LOOKUP;
    return FTP_OK;
}

/* Cleans up and closes the control connection.
 * Returns FTP_OK if the connection is closed, else FTP_ERROR;
 */
int ftp_finish(ftp_session *sess)
{
    int ret = FTP_OK;
    int old_er = sess->echo_response;
    if (sess->connection) {
	sess->echo_response = sess->echo_quit;
	if (ftp_close(sess) != FTP_CLOSED) {
	    ret = FTP_ERROR;
	}
	sess->echo_response = old_er;
    }
    return ret;
}

/* Closes the PI connection. */
int ftp_close(ftp_session *sess)
{
    int ret;
    ret = ftp_exec(sess, "QUIT");
    sock_close(sess->pi_socket);
    sess->connection = 0; /* although it should have been done already */
    return ret;
}

/* Creates the given directory
 * FTP state response */
int ftp_mkdir(ftp_session *sess, const char *dir)
{
    return ftp_exec(sess, "MKD %s", dir);
}
 
/* Renames or moves a file */
int ftp_move(ftp_session *sess, const char *from, const char *to)
{
    if (ftp_exec(sess, "RNFR %s", from) == FTP_FILEMORE) {
	return ftp_exec(sess, "RNTO %s", to);
    }
    return FTP_ERROR;
}

int ftp_delete(ftp_session *sess, const char *filename)
{
    return ftp_exec(sess, "DELE %s", filename);
}

int ftp_rmdir(ftp_session *sess, const char *filename)
{
    return ftp_exec(sess, "RMD %s", filename);
}

/* FTP non-PASV mode open */
int ftp_active_open(ftp_session *sess, const char *command)
{
    char *a, *p;
    int ret;
    ksize_t slen;
    int listener, realerr;
    struct sockaddr_in gotaddr;
    struct sockaddr_in localaddr;

    ret = ftp_open(sess);
    if (ret != FTP_OK) return ret;
    
    slen = sizeof(localaddr);
    if (getsockname(sock_get_fd(sess->pi_socket), 
		    (struct sockaddr *)&localaddr, &slen) < 0) {
	realerr = errno;
	ftp_seterror_err(sess, 
			 _("Could not create active data socket (getsockname failed)"), 
			 realerr);
    }

    /* But let bind pick a random port for us to use */
    localaddr.sin_port = 0;

    /* Create a local socket to accept the connection on */

    listener = socket(AF_INET, SOCK_STREAM, 0);
    if (listener < 0) {
	realerr = errno;
	ftp_seterror_err(sess, 
			 _("Could create active data socket (socket failed)"),
			 realerr);
	return FTP_ERROR;
    }

    /* Bind it to a port. */
    if (bind(listener, (struct sockaddr *)&localaddr, sizeof(localaddr)) < 0) {
	realerr = errno;
	ftp_seterror_err(sess, 
			 _("Could not create active data socket (bind failed)"),
			 realerr);
	(void) close(listener);
	return FTP_ERROR;
    }

    slen = sizeof(struct sockaddr_in);
    if (getsockname(listener, (struct sockaddr *)&gotaddr, &slen) < 0) {
	realerr = errno;
	ftp_seterror_err(sess, 
			 _("Could not create active data socket (getsockname failed)"),
			 realerr);
	(void) close(listener);
	return FTP_ERROR;
    }

    if (listen(listener, 1) < 0) {
	realerr = errno;
	ftp_seterror_err(sess,
			 ("Could not create active data socket (listen failed)"),
			 realerr);
	(void) close(listener);
	return FTP_ERROR;
    }

    /* Let the remote end know the address of our port.
     * Q(joe): Will this actually work on differently-endian machines? 
     * A(joe): Yes, because it is defined to be in network byte order?
     */
#define	UC(b)	(((int)b)&0xff)
    a = (char *)&gotaddr.sin_addr.s_addr;
    p = (char *)&gotaddr.sin_port;

    /* Execute the PORT command */
    ret = ftp_exec(sess, "PORT %d,%d,%d,%d,%d,%d",
		    UC(a[0]), UC(a[1]), UC(a[2]), UC(a[3]),
		    UC(p[0]), UC(p[1]));
    if (ret != FTP_OK) {
	/* Failed to execute the PORT command - close the socket */
	NE_DEBUG(DEBUG_FTP, "PORT command failed.\n");
	close(listener);
	return ret;
    }

    /* Send the command.  This will make the remote end
     * initiate the connection.
     */
    ret = ftp_exec(sess, "%s", command);
    
    /* Do they want it? */
    if (ret != FTP_READY) {
	NE_DEBUG(DEBUG_FTP, "Command failed.\n");
    } else {
	/* Now wait for a connection from the remote end. */
	sess->dtp_socket = sock_accept(listener);
	if (sess->dtp_socket == NULL) {
	    realerr = errno;
	    ftp_seterror_err(sess, 
			     _("Could not create active data socket (accept failed)"),
			     realerr);
	    ret = FTP_ERROR;
	} else {
	    sock_register_progress(sess->dtp_socket, 
				   site_sock_progress_cb,
				   NULL);
	}
    }

    (void) close(listener);
    
    return ret;
}

int ftp_chmod(ftp_session *sess, const char *filename, const mode_t mode)
{
    return ftp_exec(sess, "SITE CHMOD %03o %s", mode & 0777, filename);
}

static int ftp_data_open(ftp_session *sess, const char *template, ...) 
{
    int ret;
    va_list params;
    char buf[BUFSIZ];

    va_start(params, template);
#ifdef HAVE_VSNPRINTF
    vfsnprintf(buf, BUFSIZ, template, params);
#else
    vsprintf(buf, template, params);
#endif
    va_end(params);

    if (sess->use_passive) {
	ret = ftp_exec(sess, "PASV");
	if (ret == FTP_PASSIVE) {
	    if (ftp_connect_pasv(sess)) {
		return ftp_exec(sess, "%s", buf);
	    } else {
		return FTP_ERROR;
	    }
	} else {
	    return FTP_NOPASSIVE;
	}
    } else {
	/* we are not using passive mode. */
	return ftp_active_open(sess, buf);
    }
}

/* Closes the data connection */
static int ftp_data_close(ftp_session *sess)
{
    /* Read the response line */
    int ret;
    
    ret = sock_close(sess->dtp_socket);
    if (ret < 0) {
	ret = errno;
	ftp_seterror_err(sess, _("Error closing data socket"), ret);
	return FTP_ERROR;
    } else {
	ret = ftp_read(sess);
	if (ret == FTP_OK || ret == FTP_SENT)
	    return FTP_SENT;
	else
	    return FTP_ERROR;
    }
}

/* Set the transfer type appropriately.
 * Only set it if it *needs* setting, though.
 * Returns FTP_OK on success, or something else otherwise. */
int ftp_settype(ftp_session *sess, int ascii) 
{
    int newascii, ret;
    newascii = ascii?1:0;
    if ((sess->using_ascii == -1) || (newascii != sess->using_ascii)) {
	ret = ftp_exec(sess, ascii?"TYPE A":"TYPE I");
	if (ret == FTP_OK) {
	    sess->using_ascii = newascii;
	} else {
	    sess->using_ascii = -1; /* unknown */
	}
    } else {
	ret = FTP_OK;
    }
    return ret;
}

/* upload the given file */
int ftp_put(ftp_session *sess, 
	    const char *localfile, const char *remotefile, int ascii) 
{
    int fd, ret, realerr;
    struct stat st;

    /* Set the transfer type correctly */
    if (ftp_settype(sess, ascii) != FTP_OK)
	return FTP_ERROR;
    
    if (ascii) {
	fd = open(localfile, O_RDONLY);
    } else {
	fd = open(localfile, O_RDONLY | OPEN_BINARY_FLAGS);
    }
    
    if (fd < 0) {
	realerr = errno;
	ftp_seterror_err(sess, _("Could not open file"), realerr);
	return FTP_ERROR;
    }

    if (fstat(fd, &st) < 0) {
	realerr = errno;
	ftp_seterror_err(sess, _("Could not determine length of file"),
			 realerr);
	(void) close(fd);
	return FTP_ERROR;
    }

    ret = ftp_data_open(sess, "STOR %s", remotefile);
    if (ret == FTP_READY) {

	if (ascii) {
	    ret = send_file_ascii(fd, sess->dtp_socket, st.st_size);
	} else {
	    ret = sock_transfer(fd, sess->dtp_socket, st.st_size);
	    /* FIXME */
	    if (ret > 0) {
		ret = 0;
	    }
	}
	
	/* Get the error out before we destroy the socket */
	if (ret) {
	    handle_socket_error(sess, sess->dtp_socket, 
				_("Error sending file"), ret);
	}

	if (ftp_data_close(sess) == FTP_SENT) {
	    if (ret == 0) { 
		/* Ignore return value of close: if anything went wrong whilst 
		 * reading from the fd, we'd have picked it up already. */
		(void) close(fd);
		return FTP_OK;
	    }
	}

    }

    /* As above */
    (void) close(fd);
    return FTP_ERROR;
}

/* Conditionally upload the given file */
int ftp_put_cond(ftp_session *sess, const char *localfile, 
		 const char *remotefile, int ascii, const time_t mtime)
{
    int ret;
    
    ret = get_modtime(sess, remotefile, "");
    if (ret != FTP_OK) {
	return ret;
    }
    
    /* Is the retrieved time different from the given time? */
    if (sess->get_modtime != mtime) {
	return FTP_FAILED;
    }

    /* Do a normal upload */
    return ftp_put(sess, localfile, remotefile, ascii);
}

/* Slightly dodgy ftp_get in that you need to know the remote file
 * size. Works, but not very generic. */
int ftp_get(ftp_session *sess, const char *localfile, const char *remotefile, 
	    int ascii) 
{
    int ret;

    if (ftp_settype(sess, ascii) != FTP_OK)
	return FTP_ERROR;

    if (ftp_data_open(sess, "RETR %s", remotefile) == FTP_READY) {

	/* Receive the file */
	ret = recv_file(sess->dtp_socket, localfile);

	/* Get any error out before destroying the DTP socket */
	if (ret) {
	    handle_socket_error(sess, sess->dtp_socket,
				_("Error while downloading file"), ret);
	}

	if (ftp_data_close(sess) == FTP_SENT && ret == 0) {
	    /* Success! */
	    return FTP_OK;
	}
    }

    return FTP_ERROR;
}

int 
ftp_read_file(ftp_session *sess, const char *remotefile,
	      sock_block_reader reader, void *userdata) 
{
    int ret;

    /* Always use binary mode */
    if (ftp_settype(sess, false) != FTP_OK)
	return FTP_ERROR;

    if (ftp_data_open(sess, "RETR %s", remotefile) == FTP_READY) {
	ret = sock_readfile_blocked(sess->dtp_socket, -1, reader, userdata);
	if ((ftp_data_close(sess) == FTP_SENT) && (ret == 0)) {
	    return FTP_OK;
	}
    }
    return FTP_ERROR;
}

/* Connect to the given host 
 * Returns non-zero value if successfull */
int ftp_connect_pasv(ftp_session *sess) 
{
    sess->dtp_socket = sock_connect(sess->dtp_addr, sess->dtp_port);
    if (sess->dtp_socket == NULL) {
	int realerr = errno;
	ftp_seterror_err(sess, _("Could not connect passive data socket"),
			 realerr);
	return 0;
    } else {
	sock_register_progress(sess->dtp_socket, site_sock_progress_cb, NULL);
	return 1;
    }
}

int ftp_open(ftp_session *sess) 
{
    int ret;

    if (sess->connection) return FTP_OK;
    NE_DEBUG(DEBUG_FTP, "Opening socket to port %d\n", sess->pi_port);
    /* Open the socket. */
    
    fe_connection(fe_connecting, NULL);

    sess->pi_socket = sock_connect(sess->pi_addr, sess->pi_port);
    if (sess->pi_socket == NULL) {
	return FTP_CONNECT;
    }

    fe_connection(fe_connected, NULL);

    /* Read the hello message */
    ret = ftp_read(sess);
    if (ret != FTP_OK) {
	return FTP_HELLO;
    }
    sess->connection = 1;
    if (ftp_login(sess) != FTP_OK) {
/*	ftp_seterror(sess, "Could not log in to server."); */
	sess->connection = 0;
	sock_close(sess->pi_socket);
	return FTP_LOGIN;
    }

    /* <tr@oxlug.org>: Don't know whether in ASCII or binary mode so try 
     * to fix it.  I don't like this one bit as ftp_settype can call 
     * ftp_open. 
     * joe: Note, ftp_using_ascii is set to -1 to avoid infinite 
     * recursion. Alter with care.
     */
    if (sess->using_ascii != -1) {
           int ascii = sess->using_ascii;
           sess->using_ascii = -1;
           return ftp_settype(sess, ascii);
    }

    return FTP_OK;
}

int ftp_login(ftp_session *sess) 
{
    int ret;
    if (strlen(sess->username) == 0 || strlen(sess->password) == 0) {
	if (fe_login(fe_login_server, NULL, sess->hostname,
		     sess->username, sess->password))
	    return FTP_ERROR;
    }
    NE_DEBUG(DEBUG_FTP,  "FTP: Logging in as %s...\n", sess->username);
    ret = ftp_exec(sess, "USER %s", sess->username);
    if (ret == FTP_NEEDPASSWORD) {
	NE_DEBUG(DEBUG_FTP,  "FTP: Supplying password...\n");
	ret = ftp_exec(sess, "PASS %s", sess->password);
    }
    return ret;
}

/* Returns anything which ftp_read() does */
int ftp_exec(ftp_session *sess, const char *template, ...) 
{
    va_list params;
    int tries = 0, ret = FTP_ERROR;
    char buf[BUFSIZ];

    va_start(params, template);
#ifdef HAVE_VSNPRINTF
    vfsnprintf(buf, BUFSIZ, template, params);
#else
    vsprintf(buf, template, params);
#endif
    va_end(params);

    while (++tries < 3) {
	if (ftp_open(sess) != FTP_OK) break;
	if (strncmp(buf, "PASS ", 5) == 0) {
	    NE_DEBUG(DEBUG_FTP, "> PASS xxxxxxxx\n");
	} else {
	    NE_DEBUG(DEBUG_FTP, "> %s\n", buf);
	}
	/* Send the line */
	if (sock_sendline(sess->pi_socket, buf) == 0) {
	    /* Sent the line... try and read the response */
	    ret = ftp_read(sess);
	    if (ret != FTP_BROKEN) {
		/* Read the response okay */
		break;
	    }
	}
    }
    /* Don't let FTP_BROKEN get out */
    if (ret == FTP_BROKEN) 
	ret = FTP_ERROR;
    return ret;
}

/* Returns anything ftp_read() does, or FTP_BROKEN when the
 * socket read fails (indicating a broken socket).  
 */

int ftp_read(ftp_session *sess) 
{
    int multiline = 0, len, reply_code;
    char *buffer = sess->rbuf;
    
    for (;;) {
	len = sock_readline(sess->pi_socket, buffer, BUFSIZ);
	if (len < 0) {
	    /* It broke. */
	    handle_socket_error(sess, sess->pi_socket,
				_("Could not read response line"), len);
	    sess->connection = 0;
	    break;
	}
	
	NE_DEBUG(DEBUG_FTP, "< %s", buffer);
	if (sess->echo_response)
	    printf(buffer);
	if (len<5) /* this is a multi-liner, ignore it and carry on */
	    continue; 
	/* parse the reply code from the line */
	reply_code = get_reply_code(buffer);
	
	/* If we have a VALID reply code and we are currently
	 * in a multi line response then this is the end of the
	 * multi line response */
	if (multiline && reply_code)
	    multiline=0;
	
	/* Now, if we aren't in a multi line response... */
	if (!multiline) { 
	    if (buffer[3]=='-') {
		/* A dash in char four denotes beginning of multi
		 * line response  'xxx-' */
		multiline=1;
	    } else {
		/* Looks like we've got a real response line */
		return ftp_response(sess, buffer, reply_code);
	    }
	}
    }
    return FTP_BROKEN;
}

void ftp_seterror(ftp_session *sess, const char *error)
{
    memset(sess->error, 0, BUFSIZ);
    strncpy(sess->error, error, BUFSIZ);
}

const char *ftp_get_error(ftp_session *sess)
{
    return sess->error;
}    

/* Sets the error string and appends ": strerror(errno)" */
void ftp_seterror_err(ftp_session *sess, const char *error, 
		      int saved_errno)
{
    snprintf(sess->error, BUFSIZ, "%s: %s", error, strerror(saved_errno));
    NE_DEBUG(DEBUG_FTP, "FTP Error set: %s\n", error);
}

/* Sets the error string using the error from the PI socket.
 * Code dupe from neon's http_request:set_sockerr() */
static void handle_socket_error(ftp_session *sess, nsocket *sock, 
				const char *doing, int sockerr)
{
    const char *str;

    switch(sockerr) {
    case SOCK_CLOSED:
	snprintf(sess->error, BUFSIZ, 
		 _("%s: connection was closed by server."), doing);
	break;
    case SOCK_TIMEOUT:
	snprintf(sess->error, BUFSIZ, _("%s: connection timed out."), doing);
	break;
    default:
	/* Can get an error string from the socket? */
	str = sock_get_error(sock);
	if (str != NULL) {
	    snprintf(sess->error, BUFSIZ,
		     "%s: %s", doing, str);
	} else {
	    snprintf(sess->error, BUFSIZ, _("%s: unknown error."),
		     doing);
	}
	break;
    }
}

int ftp_response(ftp_session *sess, char *response, int code) 
{
    char *newline;
    /* Set the error string up. */
    ftp_seterror(sess, response);
    /* Chop the newline */
    newline = strrchr(sess->error, '\r');
    if (newline) *newline = '\0';	
    switch (code) {
    case 200: /* misc OK codes */
    case 220:
    case 230:
    case 250: /* completed file action */
    case 253: /* delete successful */
    case 257: /* mkdir success */
	return FTP_OK;
    case 226: /* received file okay */
	return FTP_SENT;
    case 150: /* file transfer... ready for data */
    case 125:
	return FTP_READY;
    case 550: /* couldn't complete file action */
	return FTP_FILEBAD;
    case 331: /* got username, want password */
	return FTP_NEEDPASSWORD;
    case 350: /* file action pending further info - RNFR */
	return FTP_FILEMORE;
    case 221:
	/* They've closed the connection, the swine. */
	sess->connection = false;
	return FTP_CLOSED;
    case 421: /* service denied */
	return FTP_DENIED;
    case 213: /* MDTM response, hopefully */
	return ftp_read_mdtm(response, &sess->get_modtime);
    case 227: /* PASV response, hopefully */
	return ftp_read_pasv(response, sess); 
    case 553: /* couldn't create directory */
	return FTP_ERROR;
    default:
	return FTP_ERROR;
    }
}

/* Parses the 213 response to a MDTM command... on success,
 * returns FTP_MODTIME and sets modtime to the time in the response.
 * On failute, returns FTP_ERROR. */
static int ftp_read_mdtm(char *response, time_t *modtime) 
{
    struct tm ptime;
    char year[5], month[3], mday[3], hour[3], min[3], sec[3];
    char *pnt;

    if ((pnt = strrchr(response, '\n'))!=NULL) *pnt='\0';
    if ((pnt = strrchr(response, '\r'))!=NULL) *pnt='\0';
    NE_DEBUG(DEBUG_FTP, "Reading modtime: %s\n", response);
    if (strlen(response) != 18) {
	NE_DEBUG(DEBUG_FTP, "Incorrect length response.");
	return FTP_ERROR;
    }
    if (sscanf(response, "213 %4s%2s%2s" "%2s%2s%2s",
		year, month, mday,   hour, min, sec) < 6) {
	NE_DEBUG(DEBUG_FTP, "sscanf couldn't parse it.\n");
	return FTP_ERROR;
    }
    NE_DEBUG(DEBUG_FTP, "Parsed: %d/%d/%d %s:%s:%s\n",
	   atoi(year), atoi(month), atoi(mday), hour, min, sec);
    
    memset(&ptime, 0, sizeof(struct tm));
    
    ptime.tm_year = atoi(year) - 1900; /* years since 1900 */
    ptime.tm_mon = atoi(month) - 1; /* months since jan */
    ptime.tm_mday = atoi(mday);
    
    ptime.tm_hour = atoi(hour);
    ptime.tm_min = atoi(min);
    ptime.tm_sec = atoi(sec);

    ptime.tm_isdst = -1;
    *modtime = mktime(&ptime);

    NE_DEBUG(DEBUG_FTP, "Converted to: %s", ctime(modtime));

    return FTP_MODTIME;
}

/* Parses the response to a PASV command.
 * Sets ftp_dtp_port to the port and ftp_dtp_addr to the address given
 * in the response and returns FTP_PASSIVE on success.
 * On failure, returns FTP_ERROR;
 */
int ftp_read_pasv(const char *response, ftp_session *sess)
{
    int h1, h2, h3, h4, p1, p2;
    char *start;
    start = strchr(response, ' ');
    if (start == NULL)
	return FTP_ERROR;
    while (! isdigit(*start) && (*start != '\0'))
	start++;
    /* get the host + port */
    if (sscanf(start, "%d,%d,%d,%d,%d,%d", &h1, &h2, &h3, &h4, &p1, &p2) < 6)
	/* didn't match, give up */
	return FTP_ERROR;
    /* Record this for future reference */
    sess->dtp_port = (p1<<8) | p2;
    sess->dtp_addr.s_addr = htonl((h1<<24) | (h2<<16) | (h3<<8) | h4);
    return FTP_PASSIVE;
}

/* Takes the response line from an FTP command and returns the value
 * of the response code, or 0 if none is found.
 */
int get_reply_code(const char *buffer)
{
    if (strlen(buffer) > 3)
	if (isdigit((unsigned)buffer[0]) &&
	    isdigit((unsigned)buffer[1]) && 
	    isdigit((unsigned)buffer[2]))
	    /* looks good */
	    return atoi(buffer);
    return 0;
}

/* Decode a ls -l permissions string. */
static mode_t ftp_decode_perms(const char *perms)
{
    mode_t ret = 0;
    const char *p = perms;
    while (*p) {
	ret <<= 1;
	if (*p != '-') {
	    ret |= 1;
        }
        p++;
    }
    ret &= 0777; /* throw away any "extra" bits (like from leading "d") */
    NE_DEBUG(DEBUG_FTP, "Decoded [%s] is %o\n", perms, ret);
    return ret;
}

/* This does the ls -laR, and tries it's best to parse the resulting
 * list. Currently implemented only for Unix-style servers, which go:
 * dirlist
 * new/directory/name:
 * dirlist
 * another/directory/name:
 * dirlist
 * etc. where dirlist is a straight ls -al listing
 */
int 
ftp_fetch(ftp_session *sess, const char *startdir, struct proto_file **files) 
{
    struct proto_file *this_file, *last_file;
    char *curdir,   /* Holds the path of the current directory */
	*buffer = sess->rbuf,
	perms[16], filename[BUFSIZ], tmp[BUFSIZ], *topdir;
    int ret, filesize, buflen, success = FTP_OK;
    int afterblank;

    if ((ret = ftp_data_open(sess, "LIST -laR %s", startdir)) != FTP_READY) {
	return FTP_ERROR;
    }

    memset(buffer, 0, BUFSIZ);
    
    /* The current directory is a 0-length string. */
    curdir = ne_strdup("");
    last_file = NULL;

    topdir = strdup(startdir);
    
    /* Get rid of the trailing slash. */
    if (topdir[strlen(topdir)-1] == '/') {
	topdir[strlen(topdir)-1] = '\0';
    }

    NE_DEBUG(DEBUG_FTP, "Top dir is [%s]\n", topdir);

    afterblank = false;
    for (;;) {
	ret = sock_readline(sess->dtp_socket, buffer, BUFSIZ);
	if (ret <= 0) {
	    if (ret < 0) {
		handle_socket_error(sess, sess->dtp_socket,
				    _("Could not read 'LIST' response."),
				    ret);
	    }
	    break;
	}
	STRIP_EOL(sess->rbuf);
	NE_DEBUG(DEBUG_FTP, "[ls] < %s\n", buffer);
	buflen = strlen(buffer);
	if (buflen == 0) {
	    NE_DEBUG(DEBUG_FTP, "Blank line.\n");
	    afterblank = true;
	    continue;
	} else if (strncmp(buffer, "total ", 6) == 0) {
	    /* ignore the line */
	    NE_DEBUG(DEBUG_FTP, "Line ignored.\n");
	} else if (buffer[buflen-1] == ':' && 
		   (afterblank || (strchr(buffer, ' ') == NULL))) {
	    /* A new directory name indicator, which goes:
	     *    `directory/name/here:'
	     * We want directory names as:
	     *    `directory/name/here/' 
	     * Hence a bit of messing about. */
	    free(curdir);
	    curdir = buffer;
	    /* Skip a leading Windows drive specification */
	    if (strlen(curdir) > 3 &&
		isalpha(curdir[0]) && curdir[1] == ':' && curdir[2] == '/') {
		curdir += 2;
	    }
	    if (strncmp(curdir, topdir, strlen(topdir)) == 0) {
		curdir += strlen(topdir);
	    }
	    /* Skip a single . if .:  */
	    if (strcmp(curdir,".:") == 0) {
		curdir++;
	    }
	    /* Skip a leading "./" */
	    if (strncmp(curdir, "./", 2) == 0) {
		curdir += 2;
	    }
	    /* Skip repeated '/' characters */
	    while (*curdir == '/') curdir++;
	    curdir = ne_strdup(curdir);
	    if (strlen(curdir) > 1) {
		curdir[strlen(curdir)-1] = '/';  /* change ':' to '/' */
	    } else {
		/* this is just the top-level directory... */
		curdir[0] = '\0';
	    }
	    NE_DEBUG(DEBUG_FTP, "Now in directory: [%s]\n", curdir);
	} else {
	    /* Weird bit at the end should pick up everything
	     * to the EOL... filenames could have whitespace in.
	     */
	    sscanf(buffer, "%15s %s %s %s %d %s %s %s %[^*]",  
		   perms, tmp, tmp, tmp, &filesize,
		   tmp, tmp, tmp, filename);
	    if (perms!=NULL && filename!=NULL && strlen(filename) > 0) {
		if (*perms == '-') {
		    /* Normal file. */
		    NE_DEBUG(DEBUG_FTP, "File: %s, size %d\n",
			   filename, filesize);
		    this_file = ne_calloc(sizeof(struct proto_file));
		    this_file->next = *files;
		    this_file->mode = ftp_decode_perms(perms);
		    *files = this_file;
		    if (last_file==NULL) last_file = this_file;
		    CONCAT2(this_file->filename, curdir, filename);
		    this_file->type = proto_file;
		    this_file->size = filesize;
		} else if (*perms == 'd') {
		    /* Subdirectory */
		    if (strcmp(filename, ".") == 0 ||
			strcmp(filename, "..") == 0) {
			NE_DEBUG(DEBUG_FTP, "Ignoring: %s\n", filename);
		    } else {
			NE_DEBUG(DEBUG_FTP, "Subdir: %s\n", filename);
			this_file = ne_calloc(sizeof(struct proto_file));
			if (last_file==NULL) {
			    *files = this_file;
			} else {
			    last_file->next = this_file;
			}
			last_file = this_file;
			CONCAT2(this_file->filename, curdir, filename);
			this_file->type = proto_dir;
		    }
		} else { 
		    /* Summant else... ignore */
		    NE_DEBUG(DEBUG_FTP, "Ignoring: %s\n", filename);
		}
	    } else {
		NE_DEBUG(DEBUG_FTP, "Could not parse line.\n");
		success = FTP_ERROR;
		break;
	    }
	}
    }

    free(topdir);

    NE_DEBUG(DEBUG_FTP, "Fetch finished successfully.\n");
    if (ftp_data_close(sess) == FTP_SENT) {
	return success;
    } else {
	return success;
    }
}

static int 
get_modtime(ftp_session *sess, const char *root, const char *filename) 
{
    NE_DEBUG(DEBUG_FTP, "Getting modtime.\n");
    if (ftp_exec(sess, "MDTM %s%s", root, filename) == FTP_MODTIME) {
	NE_DEBUG(DEBUG_FTP, "Got modtime.\n");
	return FTP_OK;
    } else {
	return FTP_ERROR;
    }
}

int ftp_get_modtime(ftp_session *sess, const char *filename, time_t *modtime) 
{
    if (get_modtime(sess, filename, "") == FTP_OK) {
	*modtime = sess->get_modtime;
	return FTP_OK;
    } else {
	*modtime = -1;
	return FTP_ERROR;
    }
}

/* Sorts out the modtimes for all the files in the list.
 * Returns FTP_OK on success, else FTP_ERROR. */
int 
ftp_fetch_modtimes(ftp_session *sess, const char *rootdir, 
		   struct proto_file *files) 
{
    struct proto_file *this_file;
 
    for (this_file=files; this_file!=NULL; this_file=this_file->next) {
	if (this_file->type != proto_file) {
	    continue;
	}
	NE_DEBUG(DEBUG_FTP, "File: %s%s\n", rootdir, this_file->filename);
	if (get_modtime(sess, rootdir, this_file->filename) == FTP_OK) {
	    this_file->modtime = sess->get_modtime;
	} else {
	    NE_DEBUG(DEBUG_FTP, "Didn't get modtime.\n");
	    return FTP_ERROR;
	}
    }
    NE_DEBUG(DEBUG_FTP, "Walk finished ok.\n");

    return FTP_OK;
}
