/* 
   sitecopy WebDAV protocol driver module
   Copyright (C) 2000, Joe Orton <joe@manyfish.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <config.h>

#include <sys/types.h>

#include <sys/stat.h> /* For S_IXUSR */

#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <errno.h>

#include "common.h" /* for strerror */

#include <ne_request.h>
#include <ne_basic.h>
#include <ne_basic.h>
#include <ne_props.h>
#include <ne_alloc.h>
#include <ne_uri.h>
#include <ne_auth.h>
#include <ne_dates.h>

#include "protocol.h"
#include "frontend.h"
#include "i18n.h"

struct fetch_context {
    struct proto_file **files;
    const char *root;
};

#define ENABLE_PROGRESS do { ne_set_progress(sess, site_sock_progress_cb, NULL); } while (0)

#define DISABLE_PROGRESS do { ne_set_progress(sess, NULL, NULL); } while (0)

/* TODO:
 * not really sure whether we should be using an enum here... what
 * should the client do with resourcetypes it doesn't understand?
 * ignore them, or presume they have the same semantics as "normal"
 * resources. I really don't know.
 */
struct private {
    int iscollection;
};

#define ELM_resourcetype (NE_ELM_PROPS_UNUSED)
#define ELM_collection (NE_ELM_PROPS_UNUSED + 1)

/* The element definitinos for the complex prop handler. */
static const struct ne_xml_elm fetch_elms[] = {
    { "DAV:", "resourcetype", ELM_resourcetype, 0 },
    { "DAV:", "collection", ELM_collection, 0 },
    { NULL }
};

static const ne_propname props[] = {
    { "DAV:", "getcontentlength" },
    { "DAV:", "getlastmodified" },
    { "http://apache.org/dav/props/", "executable" },
    { "DAV:", "resourcetype" },
    { NULL }
};

static inline int get_depth(const char *href) {
    const char *pnt;
    int count = 0;
    for (pnt=href; *pnt != '\0'; pnt++) /* oneliner */
	if (*pnt == '/') count++;
    return count;
}

/* TODO: get rid of this? */
static void set_err(ne_session *sess, const char *msg)
{
    char *err;
    CONCAT2(err, msg, strerror(errno));
    ne_set_error(sess, err);
    free(err);
}

static int get_server_port(struct site *site)
{
    const char *svc;
    int port, defport;
    if (site->http_secure) {
	svc = "https";
        defport = 443;
    } else {
	svc = "http";
	defport = 80;
    }
    port = sock_service_lookup(svc);
    if (port == 0) {
	port = defport;
	NE_DEBUG(NE_DBG_HTTP, "Using default port for %s: %d\n", svc, port);
    }
    return port;
}

static int get_proxy_port(struct site *site)
{
    return 8080;
}

static int auth_common(void *userdata, fe_login_context ctx,
		       const char *realm, int attempt,
		       char *username, char *password)
{
    struct site_host *host = userdata;
    if (host->username && host->password) {
	strcpy(username, host->username);
	strcpy(password, host->password);
	return attempt;
    } else {
	return fe_login(ctx, realm, host->hostname, username, password);
    }
}

static int 
server_auth_cb(void *userdata, const char *realm, int attempt,
	       char *username, char *password)
{
    return auth_common(userdata, fe_login_server, realm, attempt, 
		       username, password);
}

static int 
proxy_auth_cb(void *userdata, const char *realm, int attempt,
	      char *username, char *password)
{
    return auth_common(userdata, fe_login_proxy, realm, attempt, 
		       username, password);
}

static void notify_cb(void *userdata, ne_conn_status status, const char *info)
{

#define MAP(a) case ne_conn_##a: fe_connection(fe_##a, info); break

    switch (status) {
	MAP(namelookup);
	MAP(connecting);
	MAP(connected);
    default:
	break;
    }

#undef MAP
}

static int h2s(ne_session *sess, int errcode)
{
    switch (errcode) {
    case NE_OK:
	return SITE_OK;
    case NE_AUTH:
	return SITE_AUTH;
    case NE_AUTHPROXY:
	return SITE_PROXYAUTH;
    case NE_FAILED:
	return SITE_FAILED;
    case NE_CONNECT:
	return SITE_CONNECT;
    case NE_LOOKUP:
	return SITE_LOOKUP;
    case NE_TIMEOUT:
	ne_set_error(sess, _("The connection timed out."));
	return SITE_ERRORS;
    case NE_SERVERAUTH:
	ne_set_error(sess, 
		       _("The server did not authenticate itself correctly.\n"
			 "Report this error to your server administrator."));
	return SITE_ERRORS;
    case NE_PROXYAUTH:
	ne_set_error(sess, 
		 _("The proxy server did not authenticate itself correctly.\n"
		   "Report this error to your proxy server administrator."));
	return SITE_ERRORS;
    case NE_ERROR:
    default:
	return SITE_ERRORS;
    }
}

/* Special h2s() wrapper for init() functions. */
static int h2s_init(ne_session *sess, int errcode)
{
    int ret = h2s(sess, errcode);
    if (ret == SITE_ERRORS)
	ret = SITE_FAILED;
    return ret;
}

static int 
init(void **session, struct site *site)
{
    ne_session *sess = ne_session_create();
    ne_server_capabilities caps = {0};
    int ret;

    *session = sess;

    ne_set_status(sess, notify_cb, NULL);

    if (site->http_use_expect)
 	ne_set_expect100(sess, 1);
    else
	ne_set_expect100(sess, 0);

    if (site->http_limit)
	ne_set_persist(sess, 0);

    if (site->http_secure) {
	if (ne_set_secure(sess, 1)) {
	    ne_set_error(sess, _("SSL support has not be compiled in."));
	    return SITE_FAILED;
	}
    }

    /* Note, this won't differentiate between xsitecopy and
     * sitecopy... maybe we should put a comment in as well. */
    ne_set_useragent(sess, PACKAGE "/" VERSION);

    if (site->proxy.hostname) {
	ne_set_proxy_auth(sess, proxy_auth_cb, &site->proxy);
	ret = ne_session_proxy(sess, site->proxy.hostname, site->proxy.port);
	if (ret == NE_LOOKUP)
	    return SITE_PROXYLOOKUP;
	else if (ret != NE_OK) {
	    return h2s_init(sess, ret);
	}
    }

    ne_set_server_auth(sess, server_auth_cb, &site->server);
    ret = ne_session_server(sess, site->server.hostname, site->server.port);
    if (ret != NE_OK) {
	return h2s_init(sess, ret);
    }
    
    if (site->http_tolerant) {
	/* Skip the OPTIONS, since we ignore failure anyway. */
	return SITE_OK;
    }

    ret = ne_options(sess, site->remote_root, &caps);
    if (ret == NE_OK) {
	if (!caps.dav_class1) {
	    ne_set_error(sess, 
			    _("The server does not appear to be a WebDAV server."));
	    return SITE_FAILED;
	} else if (site->perms != sitep_ignore && !caps.dav_executable) {
	    /* Need to set permissions, but the server can't do that */
	    ne_set_error(sess, 
			    _("The server does not support the executable live property."));
	    return SITE_FAILED;
	}
    } else {
	return h2s_init(sess, ret);
    }


    return SITE_OK;
}

static void finish(void *session) 
{
    ne_session *sess = session;
    ne_session_destroy(sess);
}

static int file_move(void *session, const char *from, const char *to) 
{
    ne_session *sess = session;
    char *efrom, *eto;
    int ret;

    efrom = uri_abspath_escape(from);
    eto = uri_abspath_escape(to);

    /* Always overwrite destination. */
    ret = ne_move(sess, 1, efrom, eto);

    free(efrom);
    free(eto);
    
    return h2s(sess, ret);
}

static int file_upload(void *session, const char *local, const char *remote, 
		       int ascii)
{
    int ret, fd = open(local, O_RDONLY | OPEN_BINARY_FLAGS);
    ne_session *sess = session;
    char *eremote;

    if (fd < 0) {
	set_err(sess, _("Could not open file: "));
	return SITE_ERRORS;
    }
    
    eremote = uri_abspath_escape(remote);
    ENABLE_PROGRESS;
    ret = ne_put(sess, eremote, fd);
    DISABLE_PROGRESS;
    free(eremote);

    (void) close(fd);

    return h2s(sess, ret);
}

static int 
file_upload_cond(void *session, const char *local, const char *remote,
		 int ascii, time_t t)
{
    ne_session *sess = session;
    int ret, fd = open(local, O_RDONLY | OPEN_BINARY_FLAGS);
    char *eremote;

    if (fd < 0) {
	set_err(sess, _("Could not open file: "));
	return SITE_ERRORS;
    }
    
    eremote = uri_abspath_escape(remote);
    ENABLE_PROGRESS;
    ret = h2s(sess, ne_put_if_unmodified(sess, eremote, fd, t));
    DISABLE_PROGRESS;
    free(eremote);
    
    (void) close(fd);

    return ret;
}

static int file_get_modtime(void *session, const char *remote, time_t *modtime)
{
    ne_session *sess = session;
    int ret;
    char *eremote;
 
    eremote = uri_abspath_escape(remote);
    ret = ne_getmodtime(sess,eremote,modtime);
    free(eremote);

    return h2s(sess, ret);
}
    
static int file_download(void *session, const char *local, const char *remote,
			 int ascii) 
{
    ne_session *sess = session;
    int ret, fd = open(local, O_CREAT | O_WRONLY | OPEN_BINARY_FLAGS, 0644);
    char *eremote;

    if (fd < 0) {
	set_err(sess, _("Could not open file: "));
	return SITE_ERRORS;
    }

    eremote = uri_abspath_escape(remote);
    ENABLE_PROGRESS;
    ret = h2s(sess, ne_get(sess, eremote, fd));
    DISABLE_PROGRESS;
    free(eremote);
    
    if (close(fd)) {
	ret = SITE_ERRORS;
    }
    
    return ret;
}

static int file_read(void *session, const char *remote, 
		     sock_block_reader reader, void *userdata) 
{
    ne_session *sess = session;
    int ret;
    char *eremote;

    eremote = uri_abspath_escape(remote);    
    ENABLE_PROGRESS;
    ret = ne_read_file(sess, eremote, reader, userdata);
    DISABLE_PROGRESS;
    free(eremote);

    return h2s(sess, ret);
}

static int file_delete(void *session, const char *remote) 
{
    ne_session *sess = session;
    char *eremote = uri_abspath_escape(remote);
    int ret = ne_delete(sess, eremote);

    free(eremote);
    return h2s(sess, ret);
}

static int file_chmod(void *session, const char *remote, mode_t mode) 
{
    ne_session *sess = session;
    static const ne_propname execprop = 
    { "http://apache.org/dav/props/", "executable" };
    /* Use a single operation; set the executable property to... */
    ne_proppatch_operation ops[] = { 
	{ &execprop, ne_propset, NULL }, { NULL } 
    };
    char *eremote = uri_abspath_escape(remote);
    int ret;
    
    /* True or false, depending... */
    if (mode & S_IXUSR) {
	ops[0].value = "T";
    } else {
	ops[0].value = "F";
    }

    ret = ne_proppatch(sess, eremote, ops);
    free(eremote);

    return h2s(sess, ret);
}

static int dir_create(void *session, const char *dirname)
{
    ne_session *sess = session;
    char *edirname = uri_abspath_escape(dirname);
    int ret = ne_mkcol(sess, edirname);
    free(edirname);
    return h2s(sess, ret);
}

/* TODO: check whether it is empty first */
static int dir_remove(void *session, const char *dirname)
{
    ne_session *sess = session;
    char *edirname = uri_abspath_escape(dirname);
    int ret = ne_delete(sess, edirname);
    free(edirname);
    return h2s(sess, ret);
}

/* Insert the file in the list in the appropriate position (keeping it
 * sorted). */
static void insert_file(struct proto_file **list, struct proto_file *file)
{
    struct proto_file *previous, *current;
    previous = NULL;
    current = *list;
    while (current != NULL && current->depth < file->depth) {
	previous = current;
	current = current->next;
    }
    if (previous == NULL) {
	*list = file;
    } else {
	previous->next = file;
    }
    file->next = current;
}

static void pfind_results(void *userdata, const char *href,
			  const ne_prop_result_set *set)
{
    struct fetch_context *ctx = userdata;
    struct private *private = ne_propset_private(set);
    const char *clength = NULL, *modtime = NULL, *isexec = NULL;
    struct proto_file *file;
    int iscoll;
    char *uhref;

    /* Find out whether this is a collection or not, then
     * free the private structure so we don't have to worry 
     * about it any more. */
    iscoll = private->iscollection;
    free(private);

    NE_DEBUG(NE_DBG_HTTP, "URI: [%s]: ", href);
    
    /* Strip down to the abspath segment */
    if (strncmp(href, "http://", 7) == 0)
	href = strchr(href+7, '/');
    
    if (strncmp(href, "https://", 8) == 0)
	href = strchr(href+8, '/');

    if (href == NULL) {
	NE_DEBUG(NE_DBG_HTTP, "invalid!\n");
	return;
    }

    uhref = uri_unescape(href);

    if (!uri_childof(ctx->root, uhref)) {
	/* URI not a child of the root collection...  ignore this
	 * resource */
	NE_DEBUG(NE_DBG_HTTP, "outside root collection!\n");
	return;
    }
   
    NE_DEBUG(NE_DBG_HTTP, "okay.\n");

    if (!iscoll) {
	const ne_status *status = NULL;

	clength = ne_propset_value(set, &props[0]);    
	modtime = ne_propset_value(set, &props[1]);
	isexec = ne_propset_value(set, &props[2]);
	
	if (clength == NULL)
	    status = ne_propset_status(set, &props[0]);
	if (modtime == NULL)
	    status = ne_propset_status(set, &props[1]);

	if (clength == NULL || modtime == NULL) {
	    fe_warning(_("Could not access resource"), uhref, 
		       status?status->reason_phrase:NULL);
	    return;
	}

    }

    file = ne_calloc(sizeof(struct proto_file));
    file->filename = ne_strdup(uhref+strlen(ctx->root));
    file->depth = get_depth(file->filename);
    
    if (iscoll) {
	file->type = proto_dir;

	/* Strip the trailing slash if it has one. */
	if (uri_has_trailing_slash(file->filename)) {
	    file->filename[strlen(file->filename) - 1] = '\0';
	}

    } else {
	file->type = proto_file;
	file->size = atoi(clength);
	file->modtime = modtime?ne_httpdate_parse(modtime):0;
	if (isexec && strcasecmp(isexec, "T") == 0) {
	    file->mode = 0755;
	} else {
	    file->mode = 0644;
	}
    }

    /* Insert the file into the files list. */
    insert_file(ctx->files, file);

}

static int check_context(void *ud, ne_xml_elmid parent, ne_xml_elmid child)
{
    if (parent == NE_ELM_prop && child == ELM_resourcetype)
	return NE_XML_VALID;

    if (parent == ELM_resourcetype && child == ELM_collection)
	return NE_XML_VALID;

    return NE_XML_DECLINE;
}

static int end_element(void *userdata, 
		       const struct ne_xml_elm *elm, const char *cdata)
{
    ne_propfind_handler *handler = userdata;
    struct private *private = ne_propfind_current_private(handler);

    if (elm->id == ELM_collection)
	private->iscollection = 1;
    
    return 0;
}

/* Creates the private structure. */
static void *create_private(void *userdata, const char *uri)
{
    struct private *private = ne_calloc(sizeof *private);
    return private;
}

/* TODO: optimize: only ask for lastmod + executable when we really
 * need them: it does waste bandwidth and time to ask for executable
 * when we don't want it, since it forces a 404 propstat for each
 * non-collection resource if it is not defined.  */
static int fetch_list(void *session, const char *dirname, int need_modtimes,
		       struct proto_file **files) 
{
    ne_session *sess = session;
    int ret;
    struct fetch_context ctx;
    ne_propfind_handler *ph;
    char *edirname = uri_abspath_escape(dirname);

    ctx.root = dirname;
    ctx.files = files;
    ph = ne_propfind_create(sess, edirname, NE_DEPTH_INFINITE);

    /* The complex props. */
    ne_propfind_set_private(ph, create_private, NULL);

    /* Register the handler for the complex props. */
    ne_xml_push_handler(ne_propfind_get_parser(ph), fetch_elms,
			 check_context, NULL, end_element, ph);

    ret = ne_propfind_named(ph, props, pfind_results, &ctx);

    free(edirname);

    return h2s(sess,ret);
}

static int unimp_link2(void *session, const char *l, const char *target)
{
    ne_session *sess = session;
    ne_set_error(sess, "Operation not supported");
    return SITE_UNSUPPORTED;
}
 
static int unimp_link1(void *session, const char *l)
{
    ne_session *sess = session;
    ne_set_error(sess, "Operation not supported");
    return SITE_UNSUPPORTED;
}


static const char *error(void *session) 
{
    ne_session *sess = session;
    return ne_get_error(sess);
}

/* The WebDAV protocol driver */
const struct proto_driver dav_driver = {
    init,
    finish,
    file_move,
    file_upload,
    file_upload_cond,
    file_get_modtime,
    file_download,
    file_read,
    file_delete,
    file_chmod,
    dir_create,
    dir_remove,
    unimp_link2, /* create link */
    unimp_link2, /* change link target */
    unimp_link1, /* delete link */
    fetch_list,
    error,
    get_server_port,
    get_proxy_port,
    "WebDAV"
};
