/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <sys/stat.h>
#include <assert.h>
#include <math.h>
#include <gsl/gsl_fit.h>

#include "siril.h"
#include "proto.h"
#include "callbacks.h"
#include "single_image.h"
#include "registration.h"
#include "stacking.h"
#include "PSF.h"
#include "PSF_list.h"
#include "histogram.h"	// update_gfit_histogram_if_needed();

static struct stacking_args stackparam;	// parameters passed to stacking

static stack_method stacking_methods[] = {
	stack_summing, stack_mean_with_rejection, stack_median, addmax_stacking
};

static gboolean end_stacking(gpointer p);

/** STACK method **
 * This method takes several images and create a new being the sum of all
 * others (normalized to the maximum value of unsigned SHORT).
 */
int stack_summing(struct stacking_args *args) {
	int x, y, nx, ny, i, ii, j, shiftx, shifty, layer, reglayer;
	unsigned long *somme[3], *from, maxim = 0;
	WORD *to;
	double ratio;
	double exposure=0.0;
	unsigned int nbdata = 0;
	char filename[256];
	int retval = 0;
	int nb_frames, cur_nb = 0;
	fits *fit = &wfit[0];
	char *tmpmsg;
	memset(fit, 0, sizeof(fits));

	/* should be pre-computed to display it in the stacking tab */
	nb_frames = args->nb_images_to_stack;
	reglayer = get_registration_layer(args->seq);

	if (nb_frames <= 1) {
		siril_log_message("No frame selected for stacking (select at least 2). Aborting.\n");
		return -1;
	}

	somme[0] = NULL;
	assert(nb_frames <= args->seq->number);

	for (j=0; j<args->seq->number; ++j){
		if (!get_thread_run()) {
			retval = -1;
			goto free_and_reset_progress_bar;
		}
		if (!args->filtering_criterion(args->seq, j, args->filtering_parameter)) {
			fprintf(stdout, "image %d is excluded from stacking\n", j);
			continue;
		}
		if (!seq_get_image_filename(args->seq, j, filename)) {
			retval = -1;
			goto free_and_reset_progress_bar;
		}
		tmpmsg = siril_log_message("Processing image %s\n", filename);
		tmpmsg[strlen(tmpmsg)-1] = '\0';
		set_progress_bar_data(tmpmsg, (double)cur_nb/((double)nb_frames+1.));
		
		cur_nb++;	// only used for progress bar

		if (seq_read_frame(args->seq, j, fit)) {
			siril_log_message("Stacking: could not read frame, aborting\n");
			retval = -3;
			goto free_and_reset_progress_bar;
		}
		
		assert(args->seq->nb_layers == 1 || args->seq->nb_layers == 3);
		assert(fit->naxes[2] == args->seq->nb_layers);
	
		/* first loaded image: init data structures for stacking */
		if (!nbdata) {
			nbdata = fit->ry * fit->rx;
			somme[0] = calloc(nbdata, sizeof(unsigned long)*fit->naxes[2]);
			if (somme[0] == NULL){
				siril_log_message("Stacking: memory allocation failure\n");
				retval = -2;
				goto free_and_reset_progress_bar;
			}
			if(args->seq->nb_layers == 3){
				somme[1] = somme[0] + nbdata;	// index of green layer in somme[0]
				somme[2] = somme[0] + nbdata*2;	// index of blue layer in somme[0]
			}
			//~ siril_log_message("Stacking: successfully allocated memory for "
			//~ "stacking operation\n");
		} else if (fit->ry * fit->rx != nbdata) {
			siril_log_message("Stacking: image in sequence doesn't has the same dimensions\n");
			retval = -3;
			goto free_and_reset_progress_bar;
		}

		update_used_memory();

		/* load registration data for current image */
		if(reglayer != -1 && args->seq->regparam[reglayer]) {
			shiftx = args->seq->regparam[reglayer][j].shiftx;
			shifty = args->seq->regparam[reglayer][j].shifty;
		} else {
			shiftx = 0;
			shifty = 0;
		}
		siril_log_message("Stack image %d with shift x=%d y=%d\n", j, shiftx, shifty);

		/* Summing the exposure */
		exposure += fit->exposure;

		/* stack current image */
		i=0;	// index in somme[0]
		for (y=0; y < fit->ry; ++y){
			for (x=0; x < fit->rx; ++x){
				nx = x - shiftx;
				ny = y - shifty;
				//printf("x=%d y=%d sx=%d sy=%d i=%d ii=%d\n",x,y,shiftx,shifty,i,ii);
				if (nx >= 0 && nx < fit->rx && ny >= 0 && ny < fit->ry) {
					ii = ny * fit->rx + nx;		// index in somme[0] too
					//printf("shiftx=%d shifty=%d i=%d ii=%d\n",shiftx,shifty,i,ii);
					if (ii > 0 && ii < fit->rx * fit->ry){
						for(layer=0; layer<args->seq->nb_layers; ++layer){
							WORD current_pixel = fit->pdata[layer][ii];
							somme[layer][i] += current_pixel;
							if (somme[layer][i] > maxim){
								maxim = somme[layer][i];
							}
						}
					}
				}
				++i;
			}
		}
	}
	set_progress_bar_data("Finalizing stacking...", (double)nb_frames/((double)nb_frames + 1.));

	copyfits(fit, &gfit, CP_ALLOC|CP_FORMAT, 0);
	gfit.hi = maxim > USHRT_MAX ? USHRT_MAX : maxim;
	gfit.bitpix = USHORT_IMG;
	gfit.exposure = exposure;

	if (maxim > USHRT_MAX)
		ratio = USHRT_MAX_DOUBLE / (double)maxim;
	else	ratio = 1.0;

	if (somme[0]) {
		assert(args->seq->nb_layers == 1 || args->seq->nb_layers == 3);
		for(layer=0; layer<args->seq->nb_layers; ++layer){
			from = somme[layer];
			to = gfit.pdata[layer];
			for (y=0; y < fit->ry * fit->rx; ++y) {
				if (ratio == 1.0)
					*to++ = round_to_WORD(*from++);
				else	*to++ = round_to_WORD((double)(*from++) * ratio);
			}
		}
	}

free_and_reset_progress_bar:
	if (somme[0]) free(somme[0]);
	if (retval) {
		set_progress_bar_data("Stacking failed. Check the log.", PROGRESS_RESET);
		siril_log_message("Stacking failed.\n");
	} else {
		set_progress_bar_data("Stacking complete.", PROGRESS_DONE);
		char *msg = siril_log_message("Stacking is complete, %d images have been stacked.\n", nb_frames);
		show_dialog(msg, "Stacking", "gtk-dialog-info");
	}
	update_used_memory();
	return retval;
}

/******************************* MEDIAN STACKING ******************************
 * This is a bit special as median stacking requires all images to be in memory
 * So we dont use the generic readfits but directly the cfitsio routines, and
 * allocates as many pix tables as needed.
 * ****************************************************************************/
int stack_median(struct stacking_args *args) {
	fitsfile **fptr;	/* FITS intput files */
	int nb_frames;		/* number of frames actually used */
	int status = 0;		/* CFITSIO status value MUST be initialized to zero! */
	int bitpix;
	int ii, naxis, oldnaxis = -1, cur_nb = 0;
	long npixels_per_row, nbdata, firstpix[3] = {1,1,1};
	long naxes[3], oldnaxes[3];
	WORD **pix, *stack;
	int i, j;
	double *offset = NULL, *mul = NULL, *scale = NULL;
	double exposure=0.0;
	char filename[256], msg[256];
	int retval;
	fits *fit = &wfit[0];
	int normalize = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget("combonormalize")));

	nb_frames = args->nb_images_to_stack;

	if (args->seq->type != SEQ_REGULAR) {
		char *msg = siril_log_message("Median stacking is only supported for FITS image sequences.\n");
		show_dialog(msg, "Error", "gtk-dialog-error");
		return -1;
	}
	if (nb_frames < 2) {
		siril_log_message("Select at least two frames for stacking. Aborting.\n");
		return -1;
	}

	assert(nb_frames <= args->seq->number);

	/* allocate data structures */
	pix = calloc(nb_frames, sizeof(WORD *));
	stack = calloc(nb_frames, sizeof(WORD));
	fptr = calloc(nb_frames, sizeof(fitsfile *));
	if (!fptr || !pix) {
		siril_log_message("allocation issue for FITS files in median stacking (OOM)\n");
		if (fptr) free(fptr);
		if (pix) free(pix);
		if (stack) free(stack);
		return -1;
	}
	oldnaxes[0] = oldnaxes[1] = oldnaxes[2] = 0;	// fix compiler warning
	naxes[0] = naxes[1] = 0; naxes[2] = 1;

	/* first loop: open all fits files and check they are of same size */
	for (j=0, i=0; j<args->seq->number; ++j) {
		if (!get_thread_run()) {
			retval = -1;
			goto free_and_close;
		}
		if (!args->filtering_criterion(args->seq, j, args->filtering_parameter))
			continue;
		if (!fit_sequence_get_image_filename(args->seq, j, filename, TRUE))
			continue;
		if (i == nb_frames) {
			siril_log_message("Median stack error: wrong number of frames\n");
			retval = -2;
			goto free_and_close;
		}

		snprintf(msg, 255, "Median stack: opening image %s", filename);
		msg[255] = '\0';
		set_progress_bar_data(msg, PROGRESS_NONE);

		/* open input images */
		fits_open_file(&fptr[i], filename, READONLY, &status);
		if (status) {
			fits_report_error(stderr, status);
			fptr[i] = NULL;
			retval = status;
			goto free_and_close;
		}

		fits_get_img_param(fptr[i], 3, &bitpix, &naxis, naxes, &status);
		if (status) {
			fits_report_error(stderr, status); /* print error message */
			retval = status;
			goto free_and_close;
		}
		if (naxis > 3) {
			siril_log_message("Median stack error: images with > 3 dimensions "
					"are not supported\n");
			retval = -1;
			goto free_and_close;
		}

		if(oldnaxis > 0) {
			if(naxis != oldnaxis ||
					oldnaxes[0] != naxes[0] ||
					oldnaxes[1] != naxes[1] ||
					oldnaxes[2] != naxes[2]) {
				siril_log_message("Median stack error: input images have "
						"different sizes\n");
				retval = -2;
				goto free_and_close;
			}
		} else {
			oldnaxis = naxis;
			oldnaxes[0] = naxes[0];
			oldnaxes[1] = naxes[1];
			oldnaxes[2] = naxes[2];
		}
		
		/* exposure summing */
		double tmp;
		status=0;
		fits_read_key (fptr[i], TDOUBLE, "EXPTIME", &tmp, NULL, &status);
		if (status || tmp == 0.0) {
			status=0;
			fits_read_key (fptr[i], TDOUBLE, "EXPOSURE", &tmp, NULL, &status);
		}
		exposure+=tmp;
		i++;
	}
	update_used_memory();

	offset	= calloc(nb_frames, sizeof(double));
	mul		= calloc(nb_frames, sizeof(double));
	scale	= calloc(nb_frames, sizeof(double));
	if (!offset || !mul || !scale) {
		siril_log_message("allocation issue for FITS files in Rejection stacking\n");
		retval = -1;
		goto free_and_close;
	}

	if (compute_normalisation(args->seq, args, nb_frames, offset, mul, scale, normalize)) {
		retval = -1;
		goto free_and_close;
	}

	npixels_per_row = naxes[0]; /* number of pixels to read in each row */
	if (npixels_per_row == 0) {
		// no image has been loaded
		siril_log_message("Median stack error: no input images\n");
		retval = -2;
		goto free_and_close;
	}
	if (naxes[2] == 0)
		naxes[2] = 1;
	fprintf(stdout, "image size: %ldx%ld, %ld layers\n", naxes[0], naxes[1], naxes[2]);

	for (i=0; i<nb_frames; ++i){
		pix[i] = calloc(npixels_per_row, sizeof(WORD)); /* mem for 1 row */
		if (pix[i] == NULL) {
			fprintf(stderr, "Memory allocation error at pix index %d\n", i);
			retval = -1;
			goto free_and_close;
		}
	}

	/* initialize result image */
	nbdata = naxes[0] * naxes[1];
	memset(fit, 0, sizeof(fits));
	fit->data = malloc(nbdata * naxes[2] * sizeof(WORD));
	if (!fit->data) {
		fprintf(stderr, "Memory allocation error for result\n");
		retval = -1;
		goto free_and_close;
	}
	fit->bitpix = USHORT_IMG;
	fit->naxes[0] = naxes[0];
	fit->naxes[1] = naxes[1];
	fit->naxes[2] = naxes[2];
	fit->rx = naxes[0];
	fit->ry = naxes[1];
	fit->naxis = naxis;
	if(fit->naxis == 3) {
		fit->pdata[RLAYER]=fit->data;
		fit->pdata[GLAYER]=fit->data + nbdata;
		fit->pdata[BLAYER]=fit->data + nbdata * 2;
	} else {
		fit->pdata[RLAYER]=fit->data;
		fit->pdata[GLAYER]=fit->data;
		fit->pdata[BLAYER]=fit->data;
	}
	update_used_memory();

	set_progress_bar_data("Median stacking in progress...", PROGRESS_NONE);

	int total = naxes[2] * naxes[1];	// only used for progress bar
	/* loop over all planes of the cube (2D images have 1 plane) */
	for (firstpix[2] = 1; firstpix[2] <= naxes[2]; firstpix[2]++)
	{
		/* loop over all rows of the plane */
		for (firstpix[1] = 1; firstpix[1] <= naxes[1]; firstpix[1]++)
		{
			if (!get_thread_run()) {
				retval = -1;
				goto free_and_close;
			}
			set_progress_bar_data(NULL, (double)cur_nb/((double)total+1.));
			cur_nb++;	// only used for progress bar

			int pixel_idx = (firstpix[1] - 1) * npixels_per_row;
			if ((firstpix[1] & 127) == 0)
				set_progress_bar_data(NULL, ((double)firstpix[1] - 128) / (double)naxes[1]);

			/* Give starting pixel coordinate and no. of pixels to read. */
			/* This version does not support undefined pixels in the image. */
			for (i=0; i<nb_frames; ++i){
				status = 0;
				if (fits_read_pix(fptr[i], TUSHORT, firstpix, npixels_per_row,
							NULL, pix[i], NULL, &status)) {
					fprintf(stderr, "error reading row %ld in image %d\n", firstpix[1], i);
					retval = -1;
					goto free_and_close;
				}
			}
			for (i=0; i<npixels_per_row; ++i){
				/* copy all images pixel value in the same row array, to
				 * optimize caching and improve readability */
				for (ii=0; ii<nb_frames; ++ii) {
					double tmp = (double)pix[ii][i] * scale[ii];
					switch(normalize) {
						default:
						case 0:			//	no normalisation (scale[ii] = 1, offset[ii] = 0, mul[ii] = 1)
						case 1:			//  additive (scale[ii] = 1, mul[ii] = 1)
						case 3:			//  additive + scale (mul[ii] = 1)
							stack[ii] = tmp - offset[ii];
							break;
						case 2:			// multiplicative  (scale[ii] = 1, offset[ii] = 0)
						case 4:			// multiplicative + scale (offset[ii] = 0)
							stack[ii] = tmp * mul[ii];
							break;
					}
				}
				quicksort_s(stack, nb_frames);
				fit->pdata[firstpix[2]-1][pixel_idx] = get_median_value_from_sorted_word_data(stack, nb_frames);
				pixel_idx++;
			}
		}
	} /* end of loop over planes */

	if (status) {
		fits_report_error(stderr, status); /* print any error message */
	} else {
		set_progress_bar_data("Finalizing stacking...", PROGRESS_NONE);
		/* copy result to gfit if success */
		copyfits(fit, &gfit, CP_FORMAT, 0);
		if (gfit.data) free(gfit.data);
		gfit.data = fit->data;
		fit->data = NULL;

		memcpy(gfit.pdata, fit->pdata, 3*sizeof(WORD *));
		memset(fit->pdata, 0, 3*sizeof(WORD *));
	}

	retval = status;

free_and_close:
	for (i=0; i<nb_frames; ++i) {
		if (pix[i]) free(pix[i]);
		if (fptr[i]) fits_close_file(fptr[i], &status);
	}
	free(pix);
	free(stack);
	free(fptr);
	free(offset);
	free(mul);
	free(scale);
	if (retval) {
		/* if retval is set, gfit has not been modified */
		if (fit->data) free(fit->data);
		set_progress_bar_data("Median stacking failed. Check the log.", PROGRESS_RESET);
		siril_log_message("Stacking failed.\n");
	} else {
		set_progress_bar_data("Median stacking complete.", PROGRESS_DONE);
		char *msg = siril_log_message("Median stacking complete. %d have been stacked.\n", nb_frames);
		show_dialog(msg, "Stacking", "gtk-dialog-info");
	}
	update_used_memory();
	return retval;
}

/** Addmax STACK method **
 * This method is very close to the summing one instead that the result
 * takes only the pixel if it is brighter than the previous one at the 
 * same coordinates.
 */
int addmax_stacking(struct stacking_args *args) {
	int x, y, nx, ny, i, ii, j, shiftx, shifty, layer, reglayer;
	unsigned long *final_pixel[3], *from, maxim = 0;
	WORD *to;
	double exposure=0.0;
	unsigned int nbdata = 0;
	char filename[256];
	int retval = 0;
	int nb_frames, cur_nb = 0;
	fits *fit = &wfit[0];
	char *tmpmsg;
	memset(fit, 0, sizeof(fits));

	/* should be pre-computed to display it in the stacking tab */
	nb_frames = args->nb_images_to_stack;
	reglayer = get_registration_layer(args->seq);

	if (nb_frames <= 1) {
		siril_log_message("No frame selected for stacking (select at least 2). Aborting.\n");
		return -1;
	}

	final_pixel[0] = NULL;
	assert(args->seq->nb_layers == 1 || args->seq->nb_layers == 3);
	assert(nb_frames <= args->seq->number);

	for (j=0; j<args->seq->number; ++j){
		if (!get_thread_run()) {
			retval = -1;
			goto free_and_reset_progress_bar;
		}
		if (!args->filtering_criterion(args->seq, j, args->filtering_parameter)) {
			fprintf(stdout, "image %d is excluded from stacking\n", j);
			continue;
		}
		if (!seq_get_image_filename(args->seq, j, filename)) {
			retval = -1;
			goto free_and_reset_progress_bar;
		}
		tmpmsg = siril_log_message("Processing image %s\n", filename);
		tmpmsg[strlen(tmpmsg)-1] = '\0';
		set_progress_bar_data(tmpmsg, (double)cur_nb/((double)nb_frames+1.));
		cur_nb++;	// only used for progress bar

		if (seq_read_frame(args->seq, j, fit)) {
			siril_log_message("Stacking: could not read frame, aborting\n");
			retval = -3;
			goto free_and_reset_progress_bar;
		}
			
		assert(args->seq->nb_layers == 1 || args->seq->nb_layers == 3);
		assert(fit->naxes[2] == args->seq->nb_layers);			
		
		/* first loaded image: init data structures for stacking */
		if (!nbdata) {		
			nbdata = fit->ry * fit->rx;
			final_pixel[0] = calloc(nbdata, sizeof(unsigned long)*fit->naxes[2]);
			if (final_pixel[0] == NULL){
				siril_log_message("Stacking: memory allocation failure\n");
				retval = -2;
				goto free_and_reset_progress_bar;
			}
			if(args->seq->nb_layers == 3){
				final_pixel[1] = final_pixel[0] + nbdata;	// imendex of green layer in final_pixel[0]
				final_pixel[2] = final_pixel[0] + nbdata*2;	// index of blue layer in final_pixel[0]
			}
			//~ siril_log_message("Stacking: successfully allocated memory for "
					//~ "stacking operation\n");
		} else if (fit->ry * fit->rx != nbdata) {
			siril_log_message("Stacking: image in sequence doesn't has the same dimensions\n");
			retval = -3;
			goto free_and_reset_progress_bar;
		}

		update_used_memory();

		/* load registration data for current image */
		if(reglayer != -1 && args->seq->regparam[reglayer]) {
			shiftx = args->seq->regparam[reglayer][j].shiftx;
			shifty = args->seq->regparam[reglayer][j].shifty;
		} else {
			shiftx = 0;
			shifty = 0;
		}
		siril_log_message("stack image %d with shift x=%d y=%d\n", j, shiftx, shifty);

		/* Summing the exposure */
		exposure += fit->exposure;

		/* stack current image */
		i=0;	// index in final_pixel[0]
		for (y=0; y < fit->ry; ++y){
			for (x=0; x < fit->rx; ++x){
				nx = x - shiftx;
				ny = y - shifty;
				//printf("x=%d y=%d sx=%d sy=%d i=%d ii=%d\n",x,y,shiftx,shifty,i,ii);
				if (nx >= 0 && nx < fit->rx && ny >= 0 && ny < fit->ry) {
					ii = ny * fit->rx + nx;		// index in final_pixel[0] too
					//printf("shiftx=%d shifty=%d i=%d ii=%d\n",shiftx,shifty,i,ii);
					if (ii > 0 && ii < fit->rx * fit->ry){
						for(layer=0; layer<args->seq->nb_layers; ++layer){
							WORD current_pixel = fit->pdata[layer][ii];
							if (current_pixel > final_pixel[layer][i])	// we take the brighter pixel
								final_pixel[layer][i] = current_pixel;
							if (final_pixel[layer][i] > maxim){
								maxim = final_pixel[layer][i];
							}
						}
					}
				}
				++i;
			}
		}
	}
	if (!get_thread_run()) {
		retval = -1;
		goto free_and_reset_progress_bar;
	}
	set_progress_bar_data("Finalizing stacking...", (double)nb_frames/((double)nb_frames+1.));

	copyfits(fit, &gfit, CP_ALLOC|CP_FORMAT, 0);
	gfit.hi = maxim > USHRT_MAX ? USHRT_MAX : maxim;	// Should not be greater than USHRT_MAX with this method
	gfit.bitpix = USHORT_IMG;
	gfit.exposure = exposure;						// TODO : think if exposure has a sense here

	if (final_pixel[0]) {
		assert(args->seq->nb_layers == 1 || args->seq->nb_layers == 3);
		for (layer=0; layer<args->seq->nb_layers; ++layer){
			from = final_pixel[layer];
			to = gfit.pdata[layer];
			for (y=0; y < fit->ry * fit->rx; ++y) {
				*to++ = round_to_WORD(*from++);
			}
		}
	}

free_and_reset_progress_bar:
	if (final_pixel[0]) free(final_pixel[0]);
	if (retval) {
		set_progress_bar_data("Stacking failed. Check the log.", PROGRESS_RESET);
		siril_log_message("Stacking failed.\n");
	} else {
		set_progress_bar_data("Stacking complete.", PROGRESS_DONE);
		char *msg = siril_log_message("Stacking is complete, %d images have been stacked.\n", nb_frames);
		show_dialog(msg, "Stacking", "gtk-dialog-info");
	}
	update_used_memory();
	return retval;
}

int percentile_clipping(WORD pixel, double sig[], double median, uint64_t rej[]) {
	double plow = sig[0];
	double phigh = sig[1];	

	if ((median - (double)pixel) / median > plow) {
		rej[0]++;
		return -1;
	}
	else if (((double)pixel - median) / median > phigh) {
		rej[1]++;
		return 1;
	}
	else return 0;
}

/* Rejection of pixels, following sigma_(high/low) * sigma.
 * The function returns 0 if no rejections are required, 1 if it's a high
 * rejection and -1 for a low-rejection */
int sigma_clipping(WORD pixel, double sig[], double sigma, double median, uint64_t rej[]) {
	double sigmalow = sig[0];
	double sigmahigh = sig[1];

	if (median - (double)pixel > sigmalow * sigma) {
		rej[0]++;
		return -1;
	}
	else if ((double)pixel - median > sigmahigh * sigma) {
		rej[1]++;
		return 1;
	}
	else return 0;
}

int Winsorized(WORD *pixel, double m0, double m1) {
	if (*pixel < m0) *pixel = round_to_WORD(m0);
	else if (*pixel > m1) *pixel = round_to_WORD(m1);

	return 0;
}

int line_clipping(WORD pixel, double sig[], double sigma, int i, double a, double b, uint64_t rej[]) {
	double sigmalow = sig[0];
	double sigmahigh = sig[1];

	if (((a * (double)i + b - (double)pixel) / sigma) > sigmalow) {
		rej[0]++;
		return -1;
	}
	else if ((((double)pixel - a * (double)i - b) / sigma) > sigmahigh) {
		rej[1]++;
		return 1;
	}
	else return 0;
}

int compute_normalisation(sequence *seq, struct stacking_args *args,
		int nb_frames, double *offset, double *mul, double *scale, int mode) {

	stats *stat = NULL;
	int i, j, retval = 0, cur_nb = 0;
	double scale0, mul0, offset0;
	char *tmpmsg;
	fits fit;
	memset(&fit, 0, sizeof(fits));

	tmpmsg = siril_log_message("Computing normalisation...\n");
	tmpmsg[strlen(tmpmsg)-1] = '\0';
	set_progress_bar_data(tmpmsg, PROGRESS_NONE);

	for (i=0; i<nb_frames; i++) {
		offset[i] = 0.0;
		mul[i] = 1.0;
		scale[i] = 1.0;
	}
	scale0 = mul0 = offset0 = 0.0;
	if (mode == 0) return 0;

	for (j=0, i=0; j<seq->number; ++j) {
		if (!get_thread_run()) {
			retval = 1;
			break;
		}
		if (!args->filtering_criterion(seq, j, args->filtering_parameter))
			continue;
		set_progress_bar_data(NULL, (double)cur_nb/((double)nb_frames+1.));
		cur_nb++;	// only used for progress bar
		seq_read_frame(seq, j, &fit);
		stat = statistics(&fit, 0, NULL);
		switch(mode) {
			default:
			case 3:
				scale[i] = stat->avgdev;
				if (i==0) scale0=scale[0];
				scale[i] = scale0 / scale[i];
			case 1:
				offset[i] = stat->median;
				if (i==0) offset0=offset[0];
				offset[i] = scale[i] * offset[i] - offset0;
				break;
			case 4:
				scale[i] = stat->avgdev;
				if (i==0) scale0=scale[0];
				scale[i] = scale0 / scale[i];
			case 2:
				mul[i] = stat->median;
				if (i==0) mul0=mul[0];
				mul[i] = mul0 / mul[i];			
				break;
		}
		if (stat) free(stat);
		i++;
		clearfits(&fit);
	}
	set_progress_bar_data(NULL, PROGRESS_DONE);
	return retval;
}

int stack_mean_with_rejection(struct stacking_args *args) {
	fitsfile **fptr;	/* FITS intput files */
	int nb_frames;		/* number of frames actually used */
	int reglayer;
	double *offset = NULL, *mul = NULL, *scale = NULL;
	double exposure=0.0;
	uint64_t irej[3][2] = {{0,0}, {0,0}, {0,0}};
	int status = 0;		/* CFITSIO status value MUST be initialized to zero! */
	int bitpix;
	int /*iter,*/ naxis, oldnaxis = -1, cur_nb = 0;
	long npixels_per_row, nbdata, fpix[3] = {1,1,1};
	long naxes[3], oldnaxes[3];
	WORD **pix, *stack;
	int i, j, ii, jj, shiftx, shifty;
	char filename[256];
	char *tmpmsg;
	int retval;
	fits *fit = &wfit[0];
	rejection type_of_rejection = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget("comborejection")));
	double sig[2];
	sig[0]	= gtk_spin_button_get_value(GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")));
	sig[1]	= gtk_spin_button_get_value(GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")));
	int normalize = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget("combonormalize")));

	nb_frames = args->nb_images_to_stack;
	reglayer = get_registration_layer(args->seq);

	if (args->seq->type != SEQ_REGULAR) {
		char *msg = siril_log_message("Rejection stacking is only supported for FITS image sequences.\n");
		show_dialog(msg, "Error", "gtk-dialog-error");
		return -1;
	}
	if (nb_frames < 2) {
		siril_log_message("Select at least two frames for stacking. Aborting.\n");
		return -1;
	}
	assert(nb_frames <= args->seq->number);

	/* allocate data structures */
	pix = calloc(nb_frames, sizeof(WORD *));
	stack = calloc(nb_frames, sizeof(WORD));
	fptr = calloc(nb_frames, sizeof(fitsfile *));
	if (!fptr || !pix) {
		siril_log_message("allocation issue for FITS files in Rejection stacking (OOM)\n");
		if (fptr) free(fptr);
		if (pix) free(pix);
		if (stack) free(stack);
		return -1;
	}
	oldnaxes[0] = oldnaxes[1] = oldnaxes[2] = 0;	// fix compiler warning
	naxes[0] = naxes[1] = 0; naxes[2] = 1;

	/* first loop: open all fits files and check they are of same size */
	for (j=0, i=0; j<args->seq->number; ++j) {
		if (!get_thread_run()) {
			retval = -1;
			goto free_and_close;
		}
		if (!args->filtering_criterion(args->seq, j, args->filtering_parameter))
			continue;
		if (!fit_sequence_get_image_filename(args->seq, j, filename, TRUE))
			continue;
		if (i == nb_frames) {
			siril_log_message("Rejection stack error: wrong number of frames\n");
			retval = -2;
			goto free_and_close;
		}

		/* open input images */
		fits_open_file(&fptr[i], filename, READONLY, &status);
		if (status) {
			fits_report_error(stderr, status);
			fptr[i] = NULL;
			retval = status;
			goto free_and_close;
		}

		fits_get_img_param(fptr[i], 3, &bitpix, &naxis, naxes, &status);
		if (status) {
			fits_report_error(stderr, status); /* print error message */
			retval = status;
			goto free_and_close;
		}
		if (naxis > 3) {
			siril_log_message("Rejection stack error: images with > 3 dimensions "
					"are not supported\n");
			retval = -1;
			goto free_and_close;
		}

		if(oldnaxis > 0) {
			if(naxis != oldnaxis ||
					oldnaxes[0] != naxes[0] ||
					oldnaxes[1] != naxes[1] ||
					oldnaxes[2] != naxes[2]) {
				siril_log_message("Rejection stack error: input images have "
						"different sizes\n");
				retval = -2;
				goto free_and_close;
			}
		} else {
			oldnaxis = naxis;
			oldnaxes[0] = naxes[0];
			oldnaxes[1] = naxes[1];
			oldnaxes[2] = naxes[2];
		}
		
		/* exposure summing */
		double tmp;
		status=0;
		fits_read_key (fptr[i], TDOUBLE, "EXPTIME", &tmp, NULL, &status);
		if (status || tmp == 0.0) {
			status=0;
			fits_read_key (fptr[i], TDOUBLE, "EXPOSURE", &tmp, NULL, &status);
		}
		exposure+=tmp;
		i++;
	}
	update_used_memory();

	offset	= calloc(nb_frames, sizeof(double));
	mul		= calloc(nb_frames, sizeof(double));
	scale	= calloc(nb_frames, sizeof(double));
	if (!offset || !mul || !scale) {
		siril_log_message("allocation issue for FITS files in Rejection stacking\n");
		retval = -1;
		goto free_and_close;
	}

	if (compute_normalisation(args->seq, args, nb_frames, offset, mul, scale, normalize)) {
		retval = -1;
		goto free_and_close;
	}	

	npixels_per_row = naxes[0]; /* number of pixels to read in each row */
	if (npixels_per_row == 0) {
		// no image has been loaded
		siril_log_message("Rejection stack error: no input images\n");
		retval = -2;
		goto free_and_close;
	}
	if (naxes[2] == 0)
		naxes[2] = 1;
	fprintf(stdout, "image size: %ldx%ld, %ld layers\n", naxes[0], naxes[1], naxes[2]);

	for (i=0; i<nb_frames; ++i){
		pix[i] = calloc(npixels_per_row, sizeof(WORD)); /* mem for 1 row */
		if (pix[i] == NULL) {
			fprintf(stderr, "Memory allocation error at pix index %d\n", i);
			retval = -1;
			goto free_and_close;
		}
	}

	/* initialize result image */
	nbdata = naxes[0] * naxes[1];

	siril_log_message("Stacking: successfully allocated memory for "
			"stacking operation\n");

	memset(fit, 0, sizeof(fits));
	fit->data = calloc(1, nbdata * naxes[2] * sizeof(WORD));
	if (!fit->data) {
		fprintf(stderr, "Memory allocation error for result\n");
		retval = -1;
		goto free_and_close;
	}
	fit->bitpix = USHORT_IMG;
	fit->naxes[0] = naxes[0];
	fit->naxes[1] = naxes[1];
	fit->naxes[2] = naxes[2];
	fit->rx = naxes[0];
	fit->ry = naxes[1];
	fit->naxis = naxis;
	if(fit->naxis == 3) {
		fit->pdata[RLAYER]=fit->data;
		fit->pdata[GLAYER]=fit->data + nbdata;
		fit->pdata[BLAYER]=fit->data + nbdata * 2;
	} else {
		fit->pdata[RLAYER]=fit->data;
		fit->pdata[GLAYER]=fit->data;
		fit->pdata[BLAYER]=fit->data;
	}
	update_used_memory();

	int total = naxes[2] * naxes[1];	// only used for progress bar
	if (type_of_rejection != NONE) {
		tmpmsg = siril_log_message("Stacking in progress "
			"(sigma low=%.3lf, sigma high=%.3lf)...\n", sig[0], sig[1]);
	}
	else {
		tmpmsg = siril_log_message("Stacking in progress...\n");
		printf("%d\n\n", type_of_rejection);
	}
	tmpmsg[strlen(tmpmsg)-1] = '\0';
	set_progress_bar_data(tmpmsg, PROGRESS_NONE);

	/* loop over all planes of the cube (2D images have 1 plane) */
	for (fpix[2] = 1; fpix[2] <= naxes[2]; fpix[2]++) {
		/* loop over all rows of the plane */
		for (fpix[1] = 1; fpix[1] <= naxes[1]; fpix[1]++) {
			if (!get_thread_run()) {
				retval = -1;
				goto free_and_close;
			}
			set_progress_bar_data(NULL, (double)cur_nb/((double)total+1.));
			cur_nb++;	// only used for progress bar
			int pixel_idx = (fpix[1] - 1) * npixels_per_row;
			if ((fpix[1] & 127) == 0)
				set_progress_bar_data(NULL, ((double)fpix[1] - 128) / (double)naxes[1]);

			/* Give starting pixel coordinate and no. of pixels to read. */
			/* This version does not support undefined pixels in the image. */
			for (j=0, i=0; j<args->seq->number; ++j){
				if (!args->filtering_criterion(args->seq, j, args->filtering_parameter)) {
					continue;
				}
				status = 0;
				/* load registration data for current image */
				if(reglayer != -1 && args->seq->regparam[reglayer]) {
					shifty = args->seq->regparam[reglayer][j].shifty;
				} else {
					shifty = 0;
				}
				if ((shifty < fpix[1]) && fpix[1]-shifty <= naxes[1]) {
					long fpix_shifted[3] = {fpix[0], fpix[1]-shifty, fpix[2]};
					if (fits_read_pix(fptr[i], TUSHORT, fpix_shifted, npixels_per_row,
								NULL, pix[i], NULL, &status)) {
						fprintf(stderr, "Error reading row %ld in image %d\n", fpix_shifted[1], i);
						retval = -1;
						goto free_and_close;
					}
				}
				++i;
			}

			for (i=0; i<npixels_per_row; ++i){
				/* copy all images pixel value in the same row array, to
				 * optimize caching and improve readability */
				double sum = 0.0;
				for (j=0, ii=0; j<args->seq->number; ++j) {
					assert(ii < nb_frames);
					if (!args->filtering_criterion(args->seq, j, args->filtering_parameter)) {
						continue;
					}
					if(reglayer != -1 && args->seq->regparam[reglayer]) {
						shiftx = args->seq->regparam[reglayer][j].shiftx;
					}
					else {
						shiftx = 0;
					}
					if (i-shiftx >= npixels_per_row || i-shiftx < 0) {
						stack[ii] = 0;
					}
					else {
						double tmp = (double)pix[ii][i-shiftx] * scale[ii];
						switch(normalize) {
							default:
							case 0:			//	no normalisation (scale[ii] = 1, offset[ii] = 0, mul[ii] = 1)
							case 1:			//  additive (scale[ii] = 1, mul[ii] = 1)
							case 3:			//  additive + scale (mul[ii] = 1)
								tmp -= offset[ii];
								break;
							case 2:			// multiplicative  (scale[ii] = 1, offset[ii] = 0)
							case 4:			// multiplicative + scale (offset[ii] = 0)
								tmp *= mul[ii];
								break;
						}
						stack[ii] = round_to_WORD(tmp);
					}
					++ii;
				}
				int N = nb_frames;
				double sigma, sigma0, m;
				int n;
				switch(type_of_rejection){
					case PERCENTILE:	
						quicksort_s(stack, N);
						m = get_median_value_from_sorted_word_data(stack, N);
						for (ii = 0; ii < N; ii++) {
							if (percentile_clipping(stack[ii], sig, m, irej[fpix[2]-1])) {
								for (jj=ii; jj < N; jj++) 
									stack[jj] = (jj+1 == N) ? 0 : stack[jj+1];		// reject
								N--;
							}
						}
						break;
					case NORMAL:
						do {
							sigma = get_standard_deviation(stack, N);
							quicksort_s(stack, N);
							m = get_median_value_from_sorted_word_data(stack, N);
							n = 0;
							for (ii = 0; ii < N; ii++) {
								if (sigma_clipping(stack[ii], sig, sigma, m, irej[fpix[2]-1])) {
									for (jj=ii; jj < N - n; jj++)
										stack[jj] = (jj+1 == N) ? 0 : stack[jj+1];		// reject
									n++;
									if (N - n < 4) break;
								}
							}
							N = N - n;
						} while (n > 0 && N > 3);					
						break;
					case MEDIAN:
						do {
							sigma = get_standard_deviation(stack, N);
							quicksort_s(stack, N);
							m = get_median_value_from_sorted_word_data(stack, N);
							n = 0;
							for (ii = 0; ii < N; ii++) {
								if (sigma_clipping(stack[ii], sig, sigma, m, irej[fpix[2]-1])) {
									stack[ii] = round_to_WORD(m);
									n++;
								}
							}
						} while (n > 0 && N > 3);
						break;
					case WINSORIZED:
						do {
							sigma = get_standard_deviation(stack, N);
							quicksort_s(stack, N);
							m = get_median_value_from_sorted_word_data(stack, N);
							WORD *w_stack = calloc(N, sizeof(WORD));
							memcpy(w_stack, stack, N * sizeof(WORD));
							do {
								int jj;
								double m0 = m - 1.5*sigma;
								double m1 = m + 1.5*sigma;
								for (jj=0; jj < N; jj++)
									Winsorized(&w_stack[jj], m0, m1);
								quicksort_s(w_stack, N);
								m = get_median_value_from_sorted_word_data(w_stack, N);
								sigma0 = sigma;
								sigma = 1.134 * get_standard_deviation(w_stack, N);
							} while ((fabs(sigma - sigma0) / sigma0) > 0.0005);
							free(w_stack);
							n = 0;
							for (ii=0; ii < N; ii++) {
								if (sigma_clipping(stack[ii], sig, sigma, m, irej[fpix[2]-1])) {
									for (jj=ii; jj < N - n; jj++)
										stack[jj] = (jj+1 == N) ? 0 : stack[jj+1];		// reject
									n++;
									if (N - n < 4) break;
								}
							}
							N = N - n;
						} while (n > 0 && N > 3);
						break;
					case LINEARFIT:
						do {
							double *x = malloc(N*sizeof(double));
							double *y = malloc(N*sizeof(double));
							double a, b, cov00, cov01, cov11, sumsq;
							quicksort_s(stack, N);
							for (ii=0; ii < N; ii++) {
								x[ii] = (double)ii;
								y[ii] = (double)stack[ii];
							}
							gsl_fit_linear (x, 1, y, 1, N, &b, &a, &cov00, &cov01, &cov11, &sumsq);
							sigma = 0.0;
							for (ii=0; ii < N; ii++) sigma += (fabs((double)stack[ii] - (a*(double)ii + b)));
							sigma /= (double)N;
							n=0;
							for (ii=0; ii < N; ii++) {
								if (line_clipping(stack[ii], sig, sigma, ii, a, b, irej[fpix[2]-1])) {
									for (jj=ii; jj < N - n; jj++)
										stack[jj] = (jj+1 == N) ? 0 : stack[jj+1];		// reject
									n++;
									if (N - n < 4) break;
								}
							}
							N = N - n;
							free(x);
							free(y);
						} while (n > 0 && N > 3);
						break;
					default:
					case NONE: ;		// Nothing to do, no rejection
				}

				for (ii=0; ii<N; ++ii) {
					sum += stack[ii];
				}
				fit->pdata[fpix[2]-1][pixel_idx] = round_to_WORD(sum/(double)N);
				pixel_idx++;
			}
		}
	} /* end of loop over planes */

	if (status) {
		fits_report_error(stderr, status); /* print any error message */
	} else {
		set_progress_bar_data("Finalizing stacking...", PROGRESS_NONE);
		double nb_tot = (double)naxes[0]*naxes[1]*nb_frames;
		for (fpix[2] = 1; fpix[2] <= naxes[2]; fpix[2]++){
			siril_log_message("Pixel rejection in channel #%d: %.3lf%% - %.3lf%%\n", 
					fpix[2]-1, irej[fpix[2]-1][0]/(nb_tot)*100.0, irej[fpix[2]-1][1]/(nb_tot)*100.0);
		}
		
		/* copy result to gfit if success */
		copyfits(fit, &gfit, CP_FORMAT, 0);
		if (gfit.data) free(gfit.data);
		gfit.data = fit->data;
		gfit.bitpix = USHORT_IMG;
		gfit.exposure = exposure;
		memcpy(gfit.pdata, fit->pdata, 3*sizeof(WORD *));

		fit->data = NULL;
		memset(fit->pdata, 0, 3*sizeof(WORD *));		
	}

	retval = status;

free_and_close:
	for (i=0; i<nb_frames; ++i) {
		if (pix[i]) free(pix[i]);
		if (fptr[i]) fits_close_file(fptr[i], &status);
	}
	if (pix)	free(pix);
	if (stack)	free(stack);
	if (fptr)	free(fptr);
	if (offset)	free(offset);
	if (mul)	free(mul);
	if (scale)	free(scale);
	if (retval) {
		/* if retval is set, gfit has not been modified */
		if (fit->data) free(fit->data);
		set_progress_bar_data("Rejection stacking failed. Check the log.", PROGRESS_RESET);
		siril_log_message("Stacking failed.\n");
	} else {
		set_progress_bar_data("Rejection stacking complete.", PROGRESS_DONE);
		char *msg = siril_log_message("Rejection stacking complete. %d have been stacked.\n", nb_frames);
		show_dialog(msg, "Stacking", "gtk-dialog-info");
	}
	update_used_memory();
	return retval;
}

/* the function that runs the thread. Easier to do the simple indirection than
 * changing all return values and adding the idle everywhere. */
gpointer stack_function_handler(gpointer p) {
	struct stacking_args *args = (struct stacking_args *)p;
	args->retval = args->method(p);
	gdk_threads_add_idle(end_stacking, args);
	return GINT_TO_POINTER(args->retval);	// not used anyway
}

/* starts a summing operation using data stored in the stackparam structure
 * function is not reentrant but can be called again after it has returned and the thread is running */
void start_stacking() {
	static GtkComboBox *method_combo = NULL;
	static GtkEntry *output_file = NULL;
	static GtkToggleButton *overwrite = NULL;

	if (method_combo == NULL) {
		method_combo = GTK_COMBO_BOX(gtk_builder_get_object(builder, "comboboxstack_methods"));
		output_file = GTK_ENTRY(gtk_builder_get_object(builder, "entryresultfile"));
		overwrite = GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "checkbutoverwrite"));
	}

	if (get_thread_run()) {
		siril_log_message("Another task is already in progress, ignoring new request.\n");
		return;
	}

	stackparam.method = stacking_methods[gtk_combo_box_get_active(method_combo)];
	stackparam.seq = &com.seq;

	siril_log_color_message("Stacking: processing...\n", "red");	
	gettimeofday (&stackparam.t_start, NULL);
	set_cursor_waiting(TRUE);
	siril_log_message(stackparam.description);

	stackparam.output_overwrite = gtk_toggle_button_get_active(overwrite);
	stackparam.output_filename = gtk_entry_get_text(output_file);

	/* Stacking. Result is in gfit if success */
	start_in_new_thread(stack_function_handler, &stackparam);
}

static gboolean end_stacking(gpointer p) {
	struct timeval t_end;
	struct stacking_args *args = (struct stacking_args *)p;
	fprintf(stdout, "Ending stacking idle function, retval=%d\n", args->retval);
	stop_processing_thread();	// can it be done here in case there is no thread?
	if (!args->retval) {
		clear_stars_list();
		com.seq.current = RESULT_IMAGE;
		/* Warning: the previous com.uniq is not freed, but calling
		 * close_single_image() will close everything before reopening it,
		 * which is quite slow */
		com.uniq = calloc(1, sizeof(single));
		com.uniq->comment = strdup("Stacking result image");
		com.uniq->nb_layers = gfit.naxes[2];
		com.uniq->layers = calloc(com.uniq->nb_layers, sizeof(layer_info));
		com.uniq->fit = &gfit;
		/* Giving noise estimation */
		int i;
		double bgnoise[3] = {0.0, 0.0, 0.0};
		double norm = (double)get_normalized_value(com.uniq->fit);
		if (!backgroundnoise(com.uniq->fit, bgnoise)) {
			for (i = 0; i < com.uniq->nb_layers; i++) {
				siril_log_message("Noise estimation (channel: #%d): %.3e\n", i, bgnoise[i]/norm);
			}
		}

		/* save result */
		if (args->output_filename != NULL && args->output_filename[0] != '\0') {
			struct stat st;
			if (!stat(args->output_filename, &st)) {
				int failed = !args->output_overwrite;
				if (!failed) {
					if (unlink(args->output_filename) == -1)
						failed = 1;
					if (!failed && savefits(args->output_filename, &gfit))
						failed = 1;
					if (!failed)
						com.uniq->filename = strdup(args->output_filename);
				}
				if (failed)
					com.uniq->filename = strdup("Unsaved stacking result");
			}
			else {
				if (!savefits(args->output_filename, &gfit))
					com.uniq->filename = strdup(args->output_filename);
				else com.uniq->filename = strdup("Unsaved stacking result");
			}
			display_filename();
		}

		/* equivalent to adjust_cutoff_from_updated_gfit() but with the
		 * sliders' max value reset */
		compute_histo_for_gfit(1);
		initialize_display_mode();
		image_find_minmax(&gfit, 1);
		init_layers_hi_and_lo_values(MIPSLOHI);
		sliders_mode_set_state(com.sliders);	
		set_cutoff_sliders_max_values();
		set_cutoff_sliders_values();
		redraw(com.cvport, REMAP_ALL);
		redraw_previews();
		sequence_list_change_current();
	}
	set_cursor_waiting(FALSE);
	gettimeofday (&t_end, NULL);
	show_time(args->t_start, t_end);
	return FALSE;
}

void on_seqstack_button_clicked (GtkButton *button, gpointer user_data){
	control_window_switch_to_tab(OUTPUT_LOGS);
	start_stacking();
}

void on_comboboxstack_methods_changed (GtkComboBox *box, gpointer user_data) {
	static GtkNotebook* notebook = NULL;
	if (!notebook)
		notebook = GTK_NOTEBOOK(gtk_builder_get_object(builder, "notebook4"));
	int method = gtk_combo_box_get_active(box);

	gtk_notebook_set_current_page(notebook, method);
	update_stack_interface();
}	

void on_comborejection_changed (GtkComboBox *box, gpointer user_data) {
	rejection type_of_rejection = gtk_combo_box_get_active(box);
	GtkLabel *label_rejection[2] = {NULL, NULL};

	if (!label_rejection[0]) {
		label_rejection[0] = GTK_LABEL(lookup_widget("label120"));
		label_rejection[1] = GTK_LABEL(lookup_widget("label122"));
	}
	/* set default values */
	switch (type_of_rejection) {
		case NONE:
			gtk_widget_set_sensitive(lookup_widget("stack_siglow_button"), FALSE);
			gtk_widget_set_sensitive(lookup_widget("stack_sighigh_button"), FALSE);
			break;
		case PERCENTILE :
			gtk_widget_set_sensitive(lookup_widget("stack_siglow_button"), TRUE);
			gtk_widget_set_sensitive(lookup_widget("stack_sighigh_button"), TRUE);
			gtk_spin_button_set_range (GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")), 0.0, 1.0);
			gtk_spin_button_set_range (GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")), 0.0, 1.0);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")), 0.2);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")), 0.1);
			gtk_label_set_text (label_rejection[0], "Percentile low: ");
			gtk_label_set_text (label_rejection[1], "Percentile high: ");
			break;
		case LINEARFIT:
			gtk_widget_set_sensitive(lookup_widget("stack_siglow_button"), TRUE);
			gtk_widget_set_sensitive(lookup_widget("stack_sighigh_button"), TRUE);
			gtk_spin_button_set_range (GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")), 0.0, 10.0);
			gtk_spin_button_set_range (GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")), 0.0, 10.0);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")), 5.0);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")), 2.5);
			gtk_label_set_text (label_rejection[0], "Linear low: ");
			gtk_label_set_text (label_rejection[1], "Linear high: ");
			break;
		default:
		case NORMAL:
		case WINSORIZED:
			gtk_widget_set_sensitive(lookup_widget("stack_siglow_button"), TRUE);
			gtk_widget_set_sensitive(lookup_widget("stack_sighigh_button"), TRUE);
			gtk_spin_button_set_range (GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")), 0.0, 10.0);
			gtk_spin_button_set_range (GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")), 0.0, 10.0);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget("stack_siglow_button")), 4.0);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget("stack_sighigh_button")), 2.0);	
			gtk_label_set_text (label_rejection[0], "Sigma low: ");
			gtk_label_set_text (label_rejection[1], "Sigma high: ");
	}
}


/******************* IMAGE FILTERING CRITERIA *******************/

/* a criterion exists for each image filtering method, and is called in a
 * processing to verify if an image should be included or not.
 * These functions have the same signture, defined in stacking.h as
 * stack_filter, and return 1 if the image is included and 0 if not.
 * The functions are also called in a loop to determine the number of images to
 * be processed.
 */
int stack_filter_all(sequence *seq, int nb_img, double any) {
	return 1;
}

int stack_filter_included(sequence *seq, int nb_img, double any) {
	return (seq->imgparam[nb_img].incl);
}

/* filter for deep-sky */
int stack_filter_fwhm(sequence *seq, int nb_img, double max_fwhm) {
	int layer;
	if (!seq->regparam) return 0;
	layer = get_registration_layer(seq);
	if (layer == -1) return 0;
	if (!seq->regparam[layer]) return 0;
	if (seq->imgparam[nb_img].incl && seq->regparam[layer][nb_img].fwhm) {
		double max;
		if (seq->regparam[layer][nb_img].fwhm->FWHMX > seq->regparam[layer][nb_img].fwhm->FWHMY)
			max = seq->regparam[layer][nb_img].fwhm->FWHMX;
		else max = seq->regparam[layer][nb_img].fwhm->FWHMY;
		return (max <= max_fwhm);
	}
	else return 0;
}

/* filter for planetary */
int stack_filter_quality(sequence *seq, int nb_img, double max_quality) {
	int layer;
	if (!seq->regparam) return 0;
	layer = get_registration_layer(seq);
	if (layer == -1) return 0;
	if (!seq->regparam[layer]) return 0;
	if (seq->imgparam[nb_img].incl && seq->regparam[layer][nb_img].entropy > 0)
		return (seq->regparam[layer][nb_img].entropy <= max_quality);
	else return 0;
}

/* browse the images to konw how many fit the criterion, from global data */
int compute_nb_filtered_images() {
	int i, count = 0;
	if (!sequence_is_loaded()) return 0;
	for (i=0; i<com.seq.number; i++) {
		if (stackparam.filtering_criterion(
					&com.seq, i,
					stackparam.filtering_parameter))
			count++;
	}
	return count;
}

/****************************************************************/

/* For a sequence of images with PSF registration data and a percentage of
 * images to include in a processing, computes the highest FWHM value accepted.
 */
double compute_highest_accepted_fwhm(double percent) {
	int i, layer;
	double *val = malloc(com.seq.number * sizeof(double));
	double highest_accepted;
	layer = get_registration_layer(&com.seq);
	if (layer == -1 || !com.seq.regparam || !com.seq.regparam[layer]) {
		free(val);
		return 0.0;
	}
	// copy values
	for (i=0; i<com.seq.number; i++) {
		if (!com.seq.regparam[layer][i].fwhm) {
			siril_log_message("Error in highest FWHM accepted for sequence processing: some images don't have this kind of information available\n");
			free(val);
			return 0.0;
		}
		if (com.seq.regparam[layer][i].fwhm->FWHMX > com.seq.regparam[layer][i].fwhm->FWHMY)
			val[i] = com.seq.regparam[layer][i].fwhm->FWHMX;
		else val[i] = com.seq.regparam[layer][i].fwhm->FWHMY;
	}

	//sort values
	quicksort_d(val, com.seq.number);
	/*fprintf(stdout, "sorted values:\n");
	  for (i=0; i<com.seq.number; i++)
	  fprintf(stdout, "%g ", val[i]);
	  fputc('\n', stdout);*/

	// get highest accepted
	highest_accepted = val[(int)(percent*(double)com.seq.number/100.0)-1];
	free(val);
	return highest_accepted;
}

/* For a sequence of images with entropy registration data and a percentage of
 * images to include in a processing, computes the highest entropy value accepted.
 */
double compute_highest_accepted_entropy(double percent) {
	int i, layer;
	double *val = malloc(com.seq.number * sizeof(double));
	double highest_accepted;
	layer = get_registration_layer(&com.seq);
	if (layer == -1 || !com.seq.regparam || !com.seq.regparam[layer]) {
		free(val);
		return 0.0;
	}
	// copy values
	for (i=0; i<com.seq.number; i++) {
		if (com.seq.imgparam[i].incl && com.seq.regparam[layer][i].entropy <= 0.0) {
			siril_log_message("Error in highest entropy accepted for sequence processing: some images don't have this kind of information available for channel #%d.\n", layer);
			free(val);
			return 0.0;
		}
		else val[i] = com.seq.regparam[layer][i].entropy;
	}

	//sort values
	quicksort_d(val, com.seq.number);

	// get highest accepted
	highest_accepted = val[(int)(percent*(double)com.seq.number/100.0)-1];
	free(val);
	return highest_accepted;
}


/* Activates or not the stack button if there are 2 or more selected images,
 * all data related to stacking is set in stackparam, except the method itself,
 * determined at stacking start.
 */
void update_stack_interface() {	// was adjuststackspin
	static GtkAdjustment *stackadj = NULL;
	static GtkWidget *go_stack = NULL, *stack[] = {NULL, NULL}, *widgetnormalize=NULL;
	static GtkComboBox *stack_type = NULL, *method_combo = NULL;
	double percent;
	char labelbuffer[256];

	if(!stackadj) {
		go_stack = lookup_widget("gostack_button");
		stack[0] = lookup_widget("stackspin");
		stackadj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(stack[0]));
		stack[1] = lookup_widget("label27");
		stack_type = GTK_COMBO_BOX(lookup_widget("comboboxstacksel"));
		method_combo = GTK_COMBO_BOX(lookup_widget("comboboxstack_methods"));
		widgetnormalize = lookup_widget("combonormalize");

	}

	switch (gtk_combo_box_get_active(method_combo)) {
		default:
		case 0:
		case 3:
			gtk_widget_set_sensitive(widgetnormalize, FALSE);
			break;
		case 1:
		case 2:
			gtk_widget_set_sensitive(widgetnormalize, TRUE);
	}

	switch (gtk_combo_box_get_active(stack_type)) {
		case 0:
			stackparam.filtering_criterion = stack_filter_all;
			stackparam.nb_images_to_stack = com.seq.number;
			sprintf(stackparam.description, "Stacking all images in the sequence (%d)\n", com.seq.number);
			gtk_widget_set_sensitive(stack[0], FALSE);
			gtk_widget_set_sensitive(stack[1], FALSE);
			break;
		case 1:
			stackparam.filtering_criterion = stack_filter_included;
			stackparam.nb_images_to_stack = com.seq.selnum;
			sprintf(stackparam.description, "Stacking all selected images in the sequence (%d)\n", com.seq.selnum);
			gtk_widget_set_sensitive(stack[0], FALSE);
			gtk_widget_set_sensitive(stack[1], FALSE);
			break;
		case 2:
			/* we should check if the sequence has this kind of data
			 * available before allowing the option to be selected. */
			percent = gtk_adjustment_get_value(stackadj);
			stackparam.filtering_criterion = stack_filter_fwhm;
			stackparam.filtering_parameter = compute_highest_accepted_fwhm(percent);
			stackparam.nb_images_to_stack = compute_nb_filtered_images();
			sprintf(stackparam.description, "Stacking images of the sequence with a FWHM lower than %g (%d)\n",
					stackparam.filtering_parameter,
					stackparam.nb_images_to_stack);
			gtk_widget_set_sensitive(stack[0], TRUE);
			gtk_widget_set_sensitive(stack[1], TRUE);
			if (stackparam.filtering_parameter > 0.0)
				sprintf(labelbuffer, "Based on FWHM < %.2f (%d images)", stackparam.filtering_parameter, stackparam.nb_images_to_stack);
			else
				sprintf(labelbuffer, "Based on FWHM");
			gtk_label_set_text(GTK_LABEL(stack[1]), labelbuffer);
			break;

		case 3:
			percent = gtk_adjustment_get_value(stackadj);
			stackparam.filtering_criterion = stack_filter_quality;
			stackparam.filtering_parameter = compute_highest_accepted_entropy(percent);
			stackparam.nb_images_to_stack = compute_nb_filtered_images();
			sprintf(stackparam.description, "Stacking images of the sequence with a quality lower than %g (%d)\n",
					stackparam.filtering_parameter,
					stackparam.nb_images_to_stack);
			gtk_widget_set_sensitive(stack[0], TRUE);
			gtk_widget_set_sensitive(stack[1], TRUE);
			if (stackparam.filtering_parameter > 0.0)
				sprintf(labelbuffer, "Based on quality < %.2f (%d images)", stackparam.filtering_parameter, stackparam.nb_images_to_stack);
			else
				sprintf(labelbuffer, "Based on quality");
			gtk_label_set_text(GTK_LABEL(stack[1]), labelbuffer);
			break;

		default:	// could it be -1?
			fprintf(stderr, "unexpected value from the stack type combo box\n");
			stackparam.nb_images_to_stack = 0;
	}

	if (stackparam.nb_images_to_stack >= 2) {
		gtk_widget_set_sensitive(go_stack, TRUE);
	} else {
		gtk_widget_set_sensitive(go_stack, FALSE);
	}
}

void on_stacksel_changed(GtkComboBox *widget, gpointer user_data) {
	update_stack_interface();
}	

void on_spinbut_percent_change(GtkSpinButton *spinbutton, gpointer user_data) {
	update_stack_interface();
}	
