############################################################################
##      Copyright (C) 2005 Subredu Manuel  <diablo@iasi.roedu.net>.        #
##                                                                         #
## This program is free software; you can redistribute it and/or modify    #
## it under the terms of the GNU General Public License v2 as published by #
## the Free Software Foundation.                                           #
##                                                                         #
## This program is distributed in the hope that it will be useful,         #
## but WITHOUT ANY WARRANTY; without even the implied warranty of          #
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
## GNU General Public License for more details.                            #
##                                                                         #
## You should have received a copy of the GNU General Public License       #
## along with this program; if not, write to the Free Software             #
## Foundation, Inc., 59 Temple Place - Suite 330, Boston,                  #
## MA 02111-1307,USA.                                                      #
############################################################################

package RoPkg::Simba::Command;

use strict;
use warnings;

use Scalar::Util qw(blessed);

use RoPkg::DBObject;
use RoPkg::Exceptions;

use vars qw($table $VERSION);

$VERSION='0.1.2';

my $table='Commands';
use base qw(RoPkg::DBObject);

my $pf = {
  id   => q{-},
  Path => q{-},
  Args => q{-},
};

sub new {
  my ($class, %opt) = @_;
  my $self;

  $opt{methods} = $pf;
  $self = $class->SUPER::new(%opt);
  $self->{_sql_table} = $table;

  return $self;
}

sub table {
  my ($self) = @_;

  return $table;
}

##################################
### Database functions - BEGIN ###
##################################

sub Add {
  my ($self) = @_;

  if (!blessed($self)) {
    OutsideClass->throw('Called from outside class');
  }
  
  return $self->SQL_Insert();
}

sub Delete {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(id));

  return $self->SQL_Delete(qw(id));
}

sub Update {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(id));

  return $self->SQL_Update(qw(id));
}

sub Load {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(id));
  
  return $self->SQL_Select(qw(id));
}

##################################
###  Database functions - END  ###
##################################

1;

__END__

=head1 NAME

RoPkg::Simba::Command

=head1 VERSION

0.1.2

=head1 DESCRIPTION

RoPkg::Simba::Command is the class used by simba to manipulate a
command. It has the basic sql methods (inherited from RoPkg::DBObject).

=head1 SYNOPSIS

 !#/usr/bin/perl
 
 use RoPkg::DB;
 use RoPkg::Simba::Command;
 
 sub main {
   my $dbp = new RoPkg::DB();
   $dbp->Add('dbi:mysql:database=mysql;host=localhost',
             'root',
             '',
             'local');

   my $c = new RoPkg::Simba::Command(db => $dbp, db_method => 'db_local');
   $c->id(1);
   $c->Load();
 }
 
 main();

=head1 SUBROUTINES/METHODS

All methods (besides new) raise OutsideClass exception when
called outside class instance. Also, some methods may rise
diferent exceptions. Please read the section in which the 
method is described to find out more information about exceptions.

=head2 new()

The class constructor. At this moment, it just calls
RoPkg::DBObject->new() . Please read the RoPkg::DBObject
manual page for more information about the new() parameters.

=head2 table()

Returns the name of the commands database table.

The following methods are get/set methods for
all fields of a mirror.

=over 3

=item *) id

=item *) Path

=item *) Args

=back

=head2 Add()

Adds the mirror to the database. This method is a
wrapper for RoPkg::DBObject::SQL_Insert . On success
0 is returned. On error, DBI exception is raised.

=head2 Delete()

Deletes the current command from the database. Before
calling this method, you should set the B<id> of the
command . If you don't set the B<id> Param::Missing 
exception is raised. On database operation success,
0 is returned. On database error, DBI exception is
raised.

=head2 Update()

Update the current command object with the database. Before
calling this method, you should set the B<id> of the
command . If you don't set the B<id> Param::Missing 
exception is raised. On database operation success,
0 is returned. On database error, DBI exception is
raised.

=head2 Load()

Load the command from the database, into
the current object. Before calling this method
you should have set B<id>. If id or
is not set, then Param::Missing is raised.
On database operation success 0 is returned. On
database error, DBI exception is raised.

=head1 DIAGNOSTICS

This module comes with his own tests. To run the tests, unpack
the source, and use 'make test' commmand

=head1 CONFIGURATION AND ENVIRONMENT

This module doesn't use configuration files and environment
variables.

=head1 DEPENDENCIES

RoPkg::DBObject and RoPkg::Exceptions

=head1 INCOMPATIBILITIES

None known to the author

=head1 BUGS AND LIMITATIONS

None known to the author

=head1 PERL CRITIC

The code is perl critic level 2 compliant (with one exception)

=head1 SEE ALSO

L<RoPkg::Simba::Commands> L<RoPkg::Exceptions> L<RoPkg::Object>

=head1 AUTHOR

Subredu Manuel <diablo@iasi.roedu.net>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2005 Subredu Manuel.  All Rights Reserved.
This module is free software; you can redistribute it 
and/or modify it under the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut
