/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of Qt Creator.
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
****************************************************************************/

#include "multitextcursor.h"
#include <QKeyEvent>
#include <QTextBlock>

MultiTextCursor::MultiTextCursor() {}

MultiTextCursor::MultiTextCursor(const QList<QTextCursor> &cursors) : m_cursors(cursors) {
    mergeCursors();
}

MultiTextCursor::MultiTextCursor(const QTextCursor &cursor) : m_cursors(){
    m_cursors.append(cursor);
}

void MultiTextCursor::addCursor(const QTextCursor &cursor) {
    //QTC_ASSERT(!cursor.isNull(), return);
    m_cursors.append(cursor);
    mergeCursors();
}

void MultiTextCursor::setCursors(const QList<QTextCursor> &cursors) {
    m_cursors = cursors;
    mergeCursors();
}

bool MultiTextCursor::aCursorInThisBlock(int blocknumber){
    for (int i=0; i<m_cursors.size(); i++){
        if (m_cursors.at(i).blockNumber()==blocknumber) return true;
    }
    return false;
}

void MultiTextCursor::emplace(MultiTextCursor mc){
  m_cursors.clear();
  m_cursors=mc.m_cursors;
  mergeCursors();
}

const QList<QTextCursor> MultiTextCursor::cursors() const {
    return m_cursors;
}

void MultiTextCursor::replaceMainCursor(const QTextCursor &cursor) {
    //QTC_ASSERT(!cursor.isNull(), return);
    takeMainCursor();
    addCursor(cursor);
}

QTextCursor MultiTextCursor::mainCursor() const {
    //if (m_cursors.isEmpty()) return {};
    return m_cursors.last();
}

QTextCursor MultiTextCursor::takeMainCursor() {
    //if (m_cursors.isEmpty()) return {};
    return m_cursors.takeLast();
}

void MultiTextCursor::beginEditBlock() {
    //QTC_ASSERT(!m_cursors.empty(), return);
    m_cursors.last().beginEditBlock();
}

void MultiTextCursor::endEditBlock() {
    //QTC_ASSERT(!m_cursors.empty(), return);
    m_cursors.last().endEditBlock();
}

bool MultiTextCursor::isNull() const {
    return m_cursors.isEmpty();
}

bool MultiTextCursor::hasMultipleCursors() const {
    return m_cursors.size() > 1;
}

int MultiTextCursor::cursorCount() const {
    return m_cursors.size();
}

void MultiTextCursor::movePosition(QTextCursor::MoveOperation operation, QTextCursor::MoveMode mode, int n) {
    //for (QTextCursor &cursor : m_cursors) cursor.movePosition(operation, mode, n);
    for (int i=0; i<m_cursors.size(); i++) m_cursors[i].movePosition(operation, mode, n);
    mergeCursors();
}

bool MultiTextCursor::hasSelection() const {
    for (int i=0; i<m_cursors.size(); i++)
        if (m_cursors.at(i).hasSelection()) return true;
    return false;
}

void MultiTextCursor::clearSelection(){
    for (int i=0; i<m_cursors.size(); i++) m_cursors[i].clearSelection();
}

QString MultiTextCursor::selectedText() const {
    QString text;
    QList<QTextCursor> cursors = m_cursors;
#if (QT_VERSION >= 0x050000)
    std::sort(cursors.begin(),cursors.end());
#else
    qSort(cursors);
#endif
    //for (const QTextCursor &cursor : cursors) {
    for (int i=0; i<cursors.size(); i++) {
        const QString &cursorText = cursors[i].selectedText();
        if (cursorText.isEmpty())
            continue;
        if (!text.isEmpty()) {
            if (text.endsWith(QChar::ParagraphSeparator))
                text.chop(1);
            text.append('\n');
        }
        text.append(cursorText);
    }
    return text;
}

void MultiTextCursor::removeSelectedText() {
    beginEditBlock();
    //for (QTextCursor &c : m_cursors)
    for (int i=0; i<m_cursors.size(); i++)
        m_cursors[i].removeSelectedText();
    endEditBlock();
    mergeCursors();
}

static void insertAndSelect(QTextCursor &cursor, const QString &text, bool selectNewText) {
    if (selectNewText) {
        const int anchor = cursor.position();
        cursor.insertText(text);
        const int pos = cursor.position();
        cursor.setPosition(anchor);
        cursor.setPosition(pos, QTextCursor::KeepAnchor);
    } else {
        cursor.insertText(text);
    }
}

void MultiTextCursor::insertText(const QString &text, bool selectNewText) {
    if (m_cursors.isEmpty())
        return;
    m_cursors.last().beginEditBlock();
    if (hasMultipleCursors()) {
        QStringList lines = text.split('\n');
        if (!lines.isEmpty() && lines.last().isEmpty())
            lines.pop_back();
        int index = 0;
        if (lines.count() == m_cursors.count()) {
            QList<QTextCursor> cursors = m_cursors;
#if (QT_VERSION >= 0x050000)
            std::sort(cursors.begin(),cursors.end());
#else
            qSort(cursors);
#endif
            //Util::sort(cursors);
            //for (QTextCursor &cursor : cursors)
            for (int i=0; i<cursors.size(); i++)
                insertAndSelect(cursors[i], lines.at(index++), selectNewText);
            m_cursors.last().endEditBlock();
            return;
        }
    }
    //for (QTextCursor &cursor : m_cursors)
    for(int i=0; i<m_cursors.size();i++)
        insertAndSelect(m_cursors[i], text, selectNewText);
    m_cursors.last().endEditBlock();
}

bool equalCursors(const QTextCursor &lhs, const QTextCursor &rhs) {
    return lhs == rhs && lhs.anchor() == rhs.anchor();
}

bool MultiTextCursor::operator==(const MultiTextCursor &other) const {
    if (m_cursors.size() != other.m_cursors.size())
        return false;
    if (m_cursors.isEmpty())
        return true;
    QList<QTextCursor> thisCursors = m_cursors;
    QList<QTextCursor> otherCursors = other.m_cursors;
    if (!equalCursors(thisCursors.takeLast(), otherCursors.takeLast()))
        return false;
    //for (const QTextCursor &oc : otherCursors) {
    for (int i=0; i<otherCursors.size();i++) {
        //auto compare = [oc](const QTextCursor &c) { return equalCursors(oc, c); };
        if (!thisCursors.contains(otherCursors[i]))
            return false;
    }
    return true;
}

bool MultiTextCursor::operator!=(const MultiTextCursor &other) const {
    return !operator==(other);
}

static bool cursorsOverlap(const QTextCursor &c1, const QTextCursor &c2) {
    if (c1.hasSelection()) {
        if (c2.hasSelection()) {
            return c2.selectionEnd() > c1.selectionStart()
                   && c2.selectionStart() < c1.selectionEnd();
        }
        const int c2Pos = c2.position();
        return c2Pos > c1.selectionStart() && c2Pos < c1.selectionEnd();
    }
    if (c2.hasSelection()) {
        const int c1Pos = c1.position();
        return c1Pos > c2.selectionStart() && c1Pos < c2.selectionEnd();
    }
    return c1 == c2;
}

static void _mergeCursors(QTextCursor &c1, const QTextCursor &c2) {
    if (c1.position() == c2.position() && c1.anchor() == c2.anchor())
        return;
    if (c1.hasSelection()) {
        if (!c2.hasSelection())
            return;
        int pos = c1.position();
        int anchor = c1.anchor();
        if (c1.selectionStart() > c2.selectionStart()) {
            if (pos < anchor)
                pos = c2.selectionStart();
            else
                anchor = c2.selectionStart();
        }
        if (c1.selectionEnd() < c2.selectionEnd()) {
            if (pos < anchor)
                anchor = c2.selectionEnd();
            else
                pos = c2.selectionEnd();
        }
        c1.setPosition(anchor);
        c1.setPosition(pos, QTextCursor::KeepAnchor);
    } else {
        c1 = c2;
    }
}

void MultiTextCursor::mergeCursors() {
    //printf("Merge Cursors\n");
    QList<QTextCursor> gcursors;
    //QTextCursor c;c.setVerticalMovementX(1);
    int mustcol=-1;
    for (int i=0; i<m_cursors.size(); i++){
        if ((mustcol!=-1)&&(m_cursors.at(i).columnNumber()!=mustcol))
            m_cursors[i] = QTextCursor();//.setVerticalMovementX(m_cursors.at(i).columnNumber()-mustcol);//
        mustcol=m_cursors.at(i).columnNumber();
    }

    //*/
    for (int i=0; i<m_cursors.size(); i++) if (!m_cursors.at(i).isNull()) gcursors.append(m_cursors[i]);

    for (int i=0; i<gcursors.size(); i++){
        gcursors[i].setVisualNavigation(true);
        for (int j=i+1; j<gcursors.size(); ){
            if (cursorsOverlap(gcursors[i], gcursors[j])) {
                _mergeCursors(gcursors[i], gcursors[j]);
                //printf("I merge cursors\n");
                gcursors.removeOne(gcursors[j]);
                continue;
            }
            j++;
        }
    }
    m_cursors = gcursors;
}

// could go into QTextCursor...
static QTextLine currentTextLine(const QTextCursor &cursor) {
    const QTextBlock block = cursor.block();
    //if (!block.isValid())      return {};

    const QTextLayout *layout = block.layout();
    //if (!layout) return {};

    const int relativePos = cursor.position() - block.position();
    return layout->lineForTextPosition(relativePos);
}

bool MultiTextCursor::multiCursorAddEvent(QKeyEvent *e, QKeySequence::StandardKey matchKey) {
    uint searchkey = (e->modifiers() | e->key())
                     & ~(Qt::KeypadModifier
                         | Qt::GroupSwitchModifier
                         | Qt::AltModifier
                         | Qt::ShiftModifier);

    const QList<QKeySequence> bindings = QKeySequence::keyBindings(matchKey);
    return bindings.contains(QKeySequence(searchkey));
}

bool MultiTextCursor::handleMoveKeyEvent(QKeyEvent *e){
    //,QPlainTextEdit *edit,bool camelCaseNavigationEnabled) {
    //printf("%d %d \n",e->modifiers(),(e->modifiers() & Qt::AltModifier ));
    if (e->modifiers() & Qt::AltModifier ) {
        QTextCursor::MoveOperation op = QTextCursor::NoMove;
        if (multiCursorAddEvent(e, QKeySequence::MoveToNextWord)) {
            op = QTextCursor::WordRight;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToPreviousWord)) {
            op = QTextCursor::WordLeft;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToEndOfBlock)) {
            op = QTextCursor::EndOfBlock;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToStartOfBlock)) {
            op = QTextCursor::StartOfBlock;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToNextLine)) {
            op = QTextCursor::Down;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToPreviousLine)) {
            op = QTextCursor::Up;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToStartOfLine)) {
            op = QTextCursor::StartOfLine;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToEndOfLine)) {
            op = QTextCursor::EndOfLine;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToStartOfDocument)) {
            op = QTextCursor::Start;
        } else if (multiCursorAddEvent(e, QKeySequence::MoveToEndOfDocument)) {
            op = QTextCursor::End;
        } else {
            return false;
        }
        printf("handleMoveKeyEvent op=%d\n",op);
        //printf("?-> %d\n",m_cursors.size())  ;        
        QList<QTextCursor> cursors = m_cursors;
        for (int i=0; i<cursors.size(); i++){
            //int pos=m_cursors[i].position();
            //bool moved=cursors[i].movePosition(op, QTextCursor::MoveAnchor);
            m_cursors << cursors[i];
            //printf("handleMoveKeyEvent op=%d %d %d [%d]=%d=!=%d %d\n", op, m_cursors.size(), cursorCount(),i,cursors[i].position(),pos,moved);
        }
        mergeCursors();
        return true;
    }

    //for (QTextCursor &cursor : m_cursors) {
    for (int i=0; i<m_cursors.size(); i++){
        QTextCursor::MoveMode mode = QTextCursor::MoveAnchor;
        QTextCursor::MoveOperation op = QTextCursor::NoMove;

        if (e == QKeySequence::MoveToNextChar) {
            op = QTextCursor::Right;
        } else if (e == QKeySequence::MoveToPreviousChar) {
            op = QTextCursor::Left;
        } else if (e == QKeySequence::SelectNextChar) {
            op = QTextCursor::Right;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectPreviousChar) {
            op = QTextCursor::Left;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectNextWord) {
            op = QTextCursor::WordRight;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectPreviousWord) {
            op = QTextCursor::WordLeft;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectStartOfLine) {
            op = QTextCursor::StartOfLine;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectEndOfLine) {
            op = QTextCursor::EndOfLine;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectStartOfBlock) {
            op = QTextCursor::StartOfBlock;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectEndOfBlock) {
            op = QTextCursor::EndOfBlock;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectStartOfDocument) {
            op = QTextCursor::Start;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectEndOfDocument) {
            op = QTextCursor::End;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectPreviousLine) {
            op = QTextCursor::Up;
            mode = QTextCursor::KeepAnchor;
        } else if (e == QKeySequence::SelectNextLine) {
            op = QTextCursor::Down;
            mode = QTextCursor::KeepAnchor;
            {
                QTextBlock block = m_cursors[i].block();
                QTextLine line = currentTextLine(m_cursors[i]);
                if (!block.next().isValid() && line.isValid()
                    && line.lineNumber() == block.layout()->lineCount() - 1)
                    op = QTextCursor::End;
            }
        } else if (e == QKeySequence::MoveToNextWord) {
            op = QTextCursor::WordRight;
        } else if (e == QKeySequence::MoveToPreviousWord) {
            op = QTextCursor::WordLeft;
        } else if (e == QKeySequence::MoveToEndOfBlock) {
            op = QTextCursor::EndOfBlock;
        } else if (e == QKeySequence::MoveToStartOfBlock) {
            op = QTextCursor::StartOfBlock;
        } else if (e == QKeySequence::MoveToNextLine) {
            op = QTextCursor::Down;
        } else if (e == QKeySequence::MoveToPreviousLine) {
            op = QTextCursor::Up;
        } else if (e == QKeySequence::MoveToStartOfLine) {
            op = QTextCursor::StartOfLine;
        } else if (e == QKeySequence::MoveToEndOfLine) {
            op = QTextCursor::EndOfLine;
        } else if (e == QKeySequence::MoveToStartOfDocument) {
            op = QTextCursor::Start;
        } else if (e == QKeySequence::MoveToEndOfDocument) {
            op = QTextCursor::End;
        } else {
            //printf("handle fail\n");
            return false;
        }

        // Except for pageup and pagedown, macOS has very different behavior, we don't do it all, but
        // here's the breakdown:
        // Shift still works as an anchor, but only one of the other keys can be down Ctrl (Command),
        // Alt (Option), or Meta (Control).
        // Command/Control + Left/Right -- Move to left or right of the line
        //                 + Up/Down -- Move to top bottom of the file. (Control doesn't move the cursor)
        // Option + Left/Right -- Move one word Left/right.
        //        + Up/Down  -- Begin/End of Paragraph.
        // Home/End Top/Bottom of file. (usually don't move the cursor, but will select)

        bool visualNavigation = m_cursors[i].visualNavigation();
        m_cursors[i].setVisualNavigation(true);
        bool moved = m_cursors[i].movePosition(op, mode);
        //printf("[%d]moved %d\n",i,moved);

        if (!moved && mode == QTextCursor::MoveAnchor)
            m_cursors[i].clearSelection();
        m_cursors[i].setVisualNavigation(visualNavigation);
    }
    mergeCursors();
    return true;
}

