/* @(#)walk.c	1.3 04/10/10 Copyright 2004 J. Schilling */
#ifndef lint
static	char sccsid[] =
	"@(#)walk.c	1.3 04/10/10 Copyright 2004 J. Schilling";
#endif
/*
 *	Walk a directory tree
 *
 *	Copyright (c) 2004 J. Schilling
 */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <mconfig.h>
#include <stdio.h>
#include <unixstd.h>
#include <stdxlib.h>
#ifdef	HAVE_FCHDIR
#include <fctldefs.h>
#else
#include <maxpath.h>
#endif
#include <statdefs.h>
#include <errno.h>
#include <strdefs.h>
#include <standard.h>
#include <getcwd.h>
#include <schily.h>
#include <nlsdefs.h>
#include "walk.h"
#include "fetchdir.h"
#include "mem.h"

#ifndef	HAVE_LSTAT
#define	lstat	stat
#endif

#ifdef	HAVE_FCHDIR
LOCAL	int	WHome = -1;
#else
LOCAL	char	WHome[MAXPATHNAME+1];
#endif

#define	DIR_INCR	1024
LOCAL	char	*Curdir;
LOCAL	int	Curdtail = 0;
LOCAL	int	Curdlen = 0;
LOCAL	struct stat Ts;

struct pdirs {
	struct pdirs	*p_last;
	dev_t		p_dev;
	ino_t		p_ino;
};


typedef	int	(*statfun)	__PR((const char *_nm, struct stat *_fs));

EXPORT	int	treewalk	__PR((char *nm, walkfun fn, int flags));
LOCAL	int	walk		__PR((char *nm, statfun sf, walkfun fn, int flags, struct WALK *state, struct pdirs *last));
LOCAL	int	incr_dspace	__PR((int amt));
LOCAL	int	xchdotdot	__PR((struct pdirs *last, int tail));
LOCAL	int	xchdir		__PR((char *p));

EXPORT int
treewalk(nm, fn, flags)
	char		*nm;	/* The name to start walking		*/
	walkfun		fn;	/* The function to call for each node	*/
	int		flags;	/* Walk flags				*/
{
	struct WALK	w;
	statfun		statf = &stat;
	int		nlen;

	if (Curdir == NULL) {
		Curdir = __malloc(DIR_INCR, "path buffer");
		Curdlen = DIR_INCR;
	}
	Curdir[0] = 0;
	Curdtail = 0;
#ifdef	HAVE_FCHDIR
	WHome = open(".", 0);
#ifdef	F_SETFD
	fcntl(WHome, F_SETFD, 1);
#endif
#else
	if (getcwd(WHome, sizeof (WHome)) == NULL)
		comerr(gettext("Cannot get working directory.\n"));
#endif

	if (nm == NULL || nm[0] == '\0')
		nm = ".";

	/*
	 * If initial Curdir space is not sufficient, expand it.
	 */
	nlen = strlen(nm);
	if ((Curdlen - 2) < nlen)
		incr_dspace(nlen + 2);

	lstat(nm, &Ts);

	w.flags = 0;
	w.base = 0;
	w.level = 0;

	if (flags & WALK_PHYS)
		statf = &lstat;

	if (flags & (WALK_ARGFOLLOW|WALK_ALLFOLLOW))
		statf = &stat;

	nlen = walk(nm, statf, fn, flags, &w, (struct pdirs *)0);
#ifdef	HAVE_FCHDIR
	close(WHome);
#endif
	return (nlen);
}

LOCAL int
walk(nm, sf, fn, flags, state, last)
	char		*nm;	/* The current name for the walk	*/
	statfun		sf;	/* stat() or lstat()			*/
	walkfun		fn;	/* The function to call for each node	*/
	int		flags;	/* Flags from treewalk()		*/
	struct WALK	*state;	/* For communication with (*fn)()	*/
	struct pdirs	*last;	/* This helps to avoid loops		*/
{
	struct pdirs	thisd;
	struct stat	fs;
	int		type;
	int		ret = 0;
	int		otail;
	char		*p;

	otail = Curdtail;
	state->base = otail;
	state->flags = 0;
	if (Curdtail == 0 || Curdir[Curdtail-1] == '/') {
		p = strcatl(&Curdir[Curdtail], nm, 0);
		Curdtail = p-Curdir;
	} else {
		p = strcatl(&Curdir[Curdtail], "/", nm, 0);
		Curdtail = p-Curdir;
		state->base++;
	}

	if ((*sf)(nm, &fs) >= 0) {
		if (S_ISDIR(fs.st_mode))
			type = WALK_D;
		else if (S_ISLNK(fs.st_mode))
			type = WALK_SL;
		else
			type = WALK_F;
	} else {
		int	err = geterrno();
		if (lstat(nm, &fs) >= 0 &&
		    S_ISLNK(fs.st_mode)) {
			seterrno(err);
			ret = (*fn)(Curdir, &fs, WALK_SLN, state);
			goto out;
		} else {
			ret = (*fn)(Curdir, &fs, WALK_NS, state);
			goto out;
		}
	}
	if ((flags & WALK_MOUNT) != 0 && Ts.st_dev != fs.st_dev)
		goto out;

	if (type == WALK_D) {
		BOOL		isdot = (nm[0] == '.' && nm[1] == '\0');
		struct pdirs	*pd = last;

		if ((flags & (WALK_PHYS|WALK_ALLFOLLOW)) == WALK_PHYS)
			sf = &lstat;

		/*
		 * Search parent dir structure for possible loops.
		 */
		thisd.p_last = last;
		thisd.p_dev  = fs.st_dev;
		thisd.p_ino  = fs.st_ino;

		while (pd) {
			if (pd->p_dev == fs.st_dev &&
			    pd->p_ino == fs.st_ino) {
				ret = (*fn)(Curdir, &fs, WALK_DP, state);
				goto out;
			}
			pd = pd->p_last;
		}

		if ((flags & WALK_DEPTH) == 0)
			ret = (*fn)(Curdir, &fs, type, state);

		if (state->flags & WALK_PRUNE)
			goto out;

		if (!isdot && chdir(nm) < 0) {
			state->flags |= WALK_NOCHDIR;
			ret = (*fn)(Curdir, &fs, WALK_DNR, state);
			state->flags &= ~WALK_NOCHDIR;
			goto out;
		} else {
			char	*dp;
			char	*odp;
			int	nents;
			int	Dspace;

			/*
			 * Add space for '/' & '\0'
			 */
			Dspace = Curdlen - Curdtail - 2;

			if ((dp = fetchdir(".", &nents, NULL, NULL)) == NULL) {
				ret = (*fn)(Curdir, &fs, WALK_DNR, state);
				goto skip;
			}

			odp = dp;
			while (nents > 0 && ret == 0) {
				register char	*name;
				register int	nlen;

				name = &dp[1];
				nlen = strlen(name);

				if (Dspace < nlen)
					Dspace += incr_dspace(nlen + 2);

				state->level++;
				ret = walk(name, sf, fn, flags, state, &thisd);
				state->level--;

				nents--;
				dp += nlen +2;
			}
			free(odp);
		skip:
			if (!isdot && state->level > 0 && xchdotdot(last, otail) < 0)
				comerr(
				gettext("Cannot chdir to '..' from '%s/'.\n"),
						Curdir);
		}
		if ((flags & WALK_DEPTH) != 0)
			ret = (*fn)(Curdir, &fs, type, state);
	} else {
		ret = (*fn)(Curdir, &fs, type, state);
	}
out:
	Curdir[otail] = '\0';
	Curdtail = otail;
	return (ret);
}

LOCAL int
incr_dspace(amt)
	int	amt;
{
	int	incr = DIR_INCR;

	if (amt < 0)
		amt = 0;
	while (incr < amt)
		incr += DIR_INCR;
	Curdir = __realloc(Curdir, Curdlen + incr, "path buffer");
	Curdlen += incr;
	return (incr);
}

LOCAL int
xchdotdot(last, tail)
	struct pdirs	*last;
	int		tail;
{
	char	c;
	struct stat sb;

	if (chdir("..") >= 0) {
		seterrno(0);
		if (stat(".", &sb) >= 0) {
			if (sb.st_dev == last->p_dev &&
			    sb.st_ino == last->p_ino)
				return (0);
		}
	}
	c = Curdir[tail];
	Curdir[tail] = '\0';
#ifdef	HAVE_FCHDIR
	if (fchdir(WHome) < 0)
		return (-1);
#else
	if (chdir(WHome) < 0)
		return (-1);
#endif
	if (chdir(Curdir) < 0) {
		if (geterrno() != ENAMETOOLONG)
			return (-1);
		if (xchdir(Curdir) < 0)
			return (-1);
	}
	Curdir[tail] = c;
	if (stat(".", &sb) >= 0) {
		if (sb.st_dev == last->p_dev &&
		    sb.st_ino == last->p_ino)
			return (0);
	}
	return (-1);
}

LOCAL int
xchdir(p)
	char	*p;
{
	char	*p2;

	while (*p) {
		if ((p2 = strchr(p, '/')) != NULL)
			*p2 = '\0';
		if (chdir(p) < 0)
			return (-1);
		if (p2 == NULL)
			break;
		*p2 = '/';
		p = &p2[1];
	}
	return (0);
}
