#
# Copyright (C) 2006 Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

import gettext
_ = gettext.translation('setroubleshoot-plugins', '/usr/share/locale', fallback=True).lgettext

from setroubleshoot.util import *
from setroubleshoot.Plugin import Plugin

class plugin(Plugin):
    summary =_('''
    SELinux is preventing $SOURCE from changing a writable memory segment executable.
    ''')
    
    problem_description = _('''
    The $SOURCE application attempted to change the access protection
    of memory (e.g., allocated using malloc).  This is a potential
    security problem.  Applications should not be doing this. Applications
    are sometimes coded incorrectly and request this permission.  The
    <a href="http://people.redhat.com/drepper/selinux-mem.html">SELinux Memory Protection Tests</a>
    web page explains how to remove this requirement.  If $SOURCE does not work and you
    need it to work, you can configure SELinux temporarily to allow this
    access until the application is fixed. Please file a <a
    href="http://bugzilla.redhat.com/bugzilla/enter_bug.cgi">bug
    report</a> against this package.
    ''')
    
    fix_description = _('''
    If you trust $SOURCE to run correctly, you can change the context
    of the executable to unconfined_execmem_exec_t.
    "chcon -t unconfined_execmem_exec_t '$SOURCE_PATH'".
    You must also change the default file context files on the system in order to preserve them even on a full relabel.  "semanage fcontext -a -t unconfined_execmem_exec_t '$SOURCE_PATH'"
    ''')
    
    fix_cmd = "chcon -t unconfined_execmem_exec_t '$SOURCE_PATH'"

    def __init__(self):
        Plugin.__init__(self, __name__)
        
    def analyze(self, avc):
        if avc.matches_source_types(['unconfined_t']) and \
           avc.has_any_access_in(['execmem']):
            # MATCH
            return self.report(avc, _("Memory"),
                               self.summary, self.problem_description,
                               self.fix_description, self.fix_cmd)
        else:
            return None

