//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "sequence.h"
#include "seqedit.h"
#include <stdlib.h>

sequence::sequence( )
{
    m_list_event     = new list<event>();
    m_list_clipboard = new list<event>();
    m_list_trigger   = new list<trigger>();

    m_editing       = false;
    m_playing       = false;
    m_was_playing   = false;
    m_recording     = false;
    m_thru          = false;
	m_queued        = false;

    m_time_beats_per_measure = 4;
    m_time_beat_width = 4;

    //m_tag           = 0;

    m_name          = c_dummy;
    m_bus           = 0;
    m_length        = 4 * c_ppqn;
    m_midi_channel  = 0;
  
    /* no notes are playing */
    for (int i=0; i< c_midi_notes; i++ )
		m_playing_notes[i] = 0;

    m_last_tick = 0;

	m_masterbus = NULL;
	m_dirty_main = true;
	m_dirty_edit = true;
}



void 
sequence::set_master_midi_bus( mastermidibus *a_mmb )
{
	lock();

	m_masterbus = a_mmb;

	unlock();
}



void 
sequence::set_bpm( long a_beats_per_measure )
{
	lock();
	m_time_beats_per_measure = a_beats_per_measure;
	unlock();
}

long 
sequence::get_bpm( void )
{
  return m_time_beats_per_measure;
}

void 
sequence::set_bw( long a_beat_width )
{
	lock();
	m_time_beat_width = a_beat_width;
	m_dirty_main = m_dirty_edit = true;
	unlock();
}

long 
sequence::get_bw( void )
{
  return m_time_beat_width;
}


sequence::~sequence()
{
    delete m_list_event;
    delete m_list_clipboard;
}

/* adds event in sorted manner */
void 
sequence::add_event( const event *a_e )
{
    lock();

    m_list_event->push_front( *a_e );
    m_list_event->sort( );

    reset_draw_marker();

    unlock();
}

void 
sequence::set_orig_tick( long a_tick )
{
	lock();
    m_last_tick = a_tick;
	unlock();
}


void 
sequence::toggle_queued( void )
{
	lock();

	m_dirty_main = m_dirty_edit = true;

	m_queued = !m_queued;
	m_queued_tick = m_last_tick - (m_last_tick % m_length) + m_length;

	unlock();
}

void
sequence::off_queued( void )
{

	lock();

	m_dirty_main = m_dirty_edit = true;

	m_queued = false;

	unlock();
}

bool 
sequence::get_queued( void )
{
    return m_queued;
}

long 
sequence::get_queued_tick( void )
{
	return m_queued_tick;
}


/* tick comes in as global tick */
void 
sequence::play( long a_tick, bool a_playback_mode )
{

    lock();

    //printf( "a_tick[%ld] a_playback[%d]\n", a_tick, a_playback_mode );

    /* turns sequence off after we play in this frame */
    bool trigger_turning_off = false;

    long times_played  = m_last_tick / m_length;
    long offset_base   = times_played * m_length; 
    
    long start_tick = m_last_tick;
    long end_tick = a_tick;
    
     /* if we are using our in sequence on/off triggers */
    if ( a_playback_mode ){

  
	
  	bool trigger_state = m_playing;
	long trigger_tick = 0;
	bool triggers = false;

	list<trigger>::iterator i = m_list_trigger->begin();

	//printf( "start[%ld] => <= end[%ld]\n", start_tick, end_tick );

   	while ( i != m_list_trigger->end()){
	 
	    //printf( "    tick[%ld]\n", (*i).m_tick );
   
  	    if ( (*i).m_tick <= end_tick ){
		
  		trigger_state = (*i).m_state;
  		trigger_tick = (*i).m_tick;
		
		/* we have a change */
		triggers = true;
	    } 
	    else if ( (*i).m_tick >  end_tick ){
		
		break;
	    }

	    i++;
	}

	/* we had triggers in our slice and its not equal to current state */
	if ( triggers &&
	     trigger_state != m_playing ){

	    //printf( "trigger %d\n", trigger_state );

  	    /* we are turning on */
  	    if ( trigger_state ){

		if ( trigger_tick < m_last_tick )
		    start_tick = m_last_tick;
		else
		    start_tick = trigger_tick;

		set_playing( true );

 	    } else {

		/* we are on and turning off */
 		end_tick = trigger_tick - 1;
  		trigger_turning_off = true;
  	    }

	}

	if( m_list_trigger->size() == 0 && 
	    m_playing ){
	    
	    set_playing(false);
	    
	}
    }

    /* play the notes in our frame */
    if ( m_playing ){
	
	list<event>::iterator e = m_list_event->begin();
	
	while ( e != m_list_event->end()){

	    //printf ( "s[%ld] -> t[%ld] ", start_tick, end_tick  ); (*e).print();
 	    if ( ((*e).get_timestamp() + offset_base ) >= start_tick  &&
		 ((*e).get_timestamp() + offset_base ) <= end_tick ){
		
  		put_event_on_bus( &(*e) );
		//printf( "bus: ");(*e).print();
	    }
	    
	    else if ( ((*e).get_timestamp() + offset_base) >  end_tick ){
		
		//printf( "break\n");
		break;
	    }
	    
	    /* advance */
	    e++;
	    
	    /* did we hit the end ? */
	    if ( e == m_list_event->end() ){
		
		e = m_list_event->begin();
		offset_base += m_length;
	    }
	}
    }

    /* if our triggers said we should turn off */
    if ( trigger_turning_off ){
	
 	set_playing( false );
    }

    /* update for next frame */
    m_last_tick = end_tick + 1;
    m_was_playing = m_playing;

    unlock();
}












// /* tick comes in as global tick */
// void 
// sequence::play( long a_tick, bool a_playback_mode )
// {
//     lock();
    
//     long times_played  = m_last_tick / m_length;
//     long offset_base   = times_played * m_length;  
    
//     long start_tick;

//     /* turns sequence off after we play in this frame */
//     bool trigger_turning_off = false;

//     /* if we were playing, we buffered up past a_tick,
//        if we didnt play, we should just start from where
//        we are now */
    
//     if ( m_was_playing )
// 	start_tick = m_last_tick;
//     else
// 	start_tick = a_tick;

//     /* if we are using our in sequence on/off triggers */
//     if ( a_playback_mode ){

//  	list<trigger>::iterator i = m_list_trigger->begin();
	
//  	bool trigger_state = m_playing;
//  	long trigger_tick = 0;

//  	if ( m_list_trigger->size() == 0 )
//  	    trigger_state = false;

//  	while ( i != m_list_trigger->end()){

//  	    if ( (*i).m_tick < a_tick ||
// 		 (*i).m_tick == start_tick ){

//  		trigger_state = (*i).m_state;
//  		trigger_tick = (*i).m_tick;
//  	    }
//  	    else
//  		break;

//  	    i++;
//  	}

// 	if ( trigger_state != m_playing ){
	    

// 	    //  playing between    start_tick        a_tick
// 	    //                       |                 |
// 	    //  turning off       ***|********|  <---  |    move a_tick to trigger_tick
// 	    //  turning on           |     |***********|**  move start_tick to a_tick
// 	    //                                              but only if its greater to avoid
// 	    //                                              when trigger_tick is way back and
// 	    //                                              start_tick is where we really 
// 	    //                                              start playback
// 	    //          *****|       |                 |    

//  	    /* we are turning on */
//  	    if ( trigger_state ){


// 		/* only bump up start_tick if trigger_tick is greater */
// 		if ( trigger_tick > start_tick )
// 		    start_tick = trigger_tick;

		

// 		set_playing( true );

// 	    } else {

// 		/* we are on and turning off */
// 		a_tick = trigger_tick;
//  		trigger_turning_off = true;
//  	    }

// 	    /*  */
// 	    if ( ! m_was_playing )
// 		start_tick = a_tick;
//  	}
//     }

//     /* play the notes in our frame */
//     if ( m_playing ){
	
// 	list<event>::iterator e = m_list_event->begin();
	
// 	while ( e != m_list_event->end()){

// 	    //printf ( "s[%ld] -> t[%ld] ", start_tick, a_tick  ); (*e).print();
//  	    if ( ((*e).get_timestamp() + offset_base) >= start_tick  &&
// 		 ((*e).get_timestamp() + offset_base) <  a_tick ){
		
//  		long delta_tick = ( (*e).get_timestamp() 
//  				    + offset_base     
//  				    - a_tick );
		
//  		if ( delta_tick < 0 )
//  		    delta_tick = 0;
		
//  		put_event_on_bus( &(*e), delta_tick  );
// 		//printf( "bus: ");(*e).print();
// 	    }
	    
// 	    else if ( ((*e).get_timestamp() + offset_base) >  a_tick ){
		
// 		//printf( "break\n");
// 		break;
// 	    }
	    
// 	    /* advance */
// 	    e++;
	    
// 	    /* did we hit the end ? */
// 	    if ( e == m_list_event->end() ){
		
// 		e = m_list_event->begin();
// 		offset_base += m_length;
// 	    }
// 	}
//     }
    
//     /* if our triggers said we should turn off */
//     if ( trigger_turning_off ){
	
// 	set_playing( false );
//     }
    
//     /* update for next frame */
//     m_last_tick = a_tick;
//     m_was_playing = m_playing;
    
//     unlock();
// }

void 
sequence::zero_markers( void )
{
    lock();

    m_last_tick = 0;

    //m_masterbus->flush( );

    unlock();
}


/* verfies state, all noteons have an off,
   links noteoffs with their ons */
void 
sequence::verify_and_link()
{

    list<event>::iterator i;
    list<event>::iterator on;
    list<event>::iterator off;

    lock();

    /* unselect all */
    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	(*i).unselect();
	(*i).clear_link();
    }

    on = m_list_event->begin();
	    
    /* pair ons and offs */
    while ( on != m_list_event->end() ){

	/* check for a note on, then look for its
	   note off */
	if ( (*on).is_note_on() ){

	    /* get next possible off node */
	    off = on; off++;

	    while ( off != m_list_event->end() ){
		
		/* is a off event, == notes, and isnt
		   selected  */
		if ( (*off).is_note_off()                  &&
		     (*off).get_note() == (*on).get_note() && 
		     ! (*off).is_selected()                  ){

		    /* link + select */
		    (*on).link( &(*off) );
		    (*off).link( &(*on) );
		    (*on).select(  );
		    (*off).select( );

		    break;
		}
		off++;
	    }
	}    
	on++;
    }

    /* unselect all */
    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	(*i).unselect();
    }

    /* kill those not in range */
    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	
	/* if our current time stamp is greater then the length */
	if ( (*i).get_timestamp() >= m_length ||
	     (*i).get_timestamp() < 0            ){
	    
	    /* we have to prune it */
	    (*i).select();
	    if ( (*i).is_linked() )
		(*i).get_linked()->select();
	}
    }

    remove_selected( );
    unlock();
}
    




void 
sequence::link_new( )
{
    list<event>::iterator on;
    list<event>::iterator off;

    lock();

    on = m_list_event->begin();

    /* pair ons and offs */
    while ( on != m_list_event->end()){

	/* check for a note on, then look for its
	   note off */
	if ( (*on).is_note_on() &&
	     ! (*on).is_linked() ){
	    
	    /* get next element */
	    off = on; off++;
	    
	    while ( off != m_list_event->end()){

		/* is a off event, == notes, and isnt
		   selected  */
		if ( (*off).is_note_off()                    &&
		     (*off).get_note() == (*on).get_note() && 
		     ! (*off).is_linked()                    ){
		    
		    /* link */
		    (*on).link( &(*off) );
		    (*off).link( &(*on) );
		    
		    break;
		}
		off++;
	    }
	}    
	on++;
    }
    unlock();
}




void 
sequence::remove_selected( )
{
    list<event>::iterator i, t;

    lock();

    i = m_list_event->begin();
    while( i != m_list_event->end() ){
	
	if ((*i).is_selected()){
	    
	    /* if its a note off, and that note is currently
	       playing, send a note off */
	    if ( (*i).is_note_off()  &&
		 m_playing_notes[ (*i).get_note()] > 0 ){
		
			m_masterbus->play( m_bus, &(*i), m_midi_channel );
			m_playing_notes[(*i).get_note()]--;
	    }
	   
	    t = i; t++;
	    m_list_event->erase(i);
	    i = t;
	}
	else {

	    i++;
	}
    }

    reset_draw_marker();

    unlock();
}

/* returns the 'box' of the selected items */
void 
sequence::get_selected_box( long *a_tick_s, int *a_note_h, 
			    long *a_tick_f, int *a_note_l )
{

    list<event>::iterator i;

    *a_tick_s = c_maxbeats * c_ppqn;
    *a_tick_f = 0;

    *a_note_h = 0;
    *a_note_l = 128;

    long time;
    int note;

    lock();

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	if( (*i).is_selected() ){
	    
	    time = (*i).get_timestamp();
	    
	    if ( time < *a_tick_s ) *a_tick_s = time;
	    if ( time > *a_tick_f ) *a_tick_f = time;
	    
	    note = (*i).get_note();

	    if ( note < *a_note_l ) *a_note_l = note;
	    if ( note > *a_note_h ) *a_note_h = note;
	}
    }
    
    unlock();
}

void 
sequence::get_clipboard_box( long *a_tick_s, int *a_note_h, 
			     long *a_tick_f, int *a_note_l )
{

    list<event>::iterator i;
    
    *a_tick_s = c_maxbeats * c_ppqn;
    *a_tick_f = 0;
    
    *a_note_h = 0;
    *a_note_l = 128;
    
    long time;
    int note;
    
    lock();

    if ( m_list_clipboard->size() == 0 ) {
	*a_tick_s = *a_tick_f = *a_note_h = *a_note_l = 0; 
    }

    for ( i = m_list_clipboard->begin(); i != m_list_clipboard->end(); i++ ){
	
	time = (*i).get_timestamp();
	
	if ( time < *a_tick_s ) *a_tick_s = time;
	if ( time > *a_tick_f ) *a_tick_f = time;
	
	note = (*i).get_note();
	
	if ( note < *a_note_l ) *a_note_l = note;
	if ( note > *a_note_h ) *a_note_h = note;
    }
   
    unlock();
}



int 
sequence::get_num_selected_notes( )
{
	int ret = 0;

	list<event>::iterator i;

    lock();

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

        if( (*i).is_note_on()                &&
			(*i).is_selected() ){
			ret++;
		}
    }

    unlock();

	return ret;

}


int 
sequence::get_num_selected_events( unsigned char a_status, 
								   unsigned char a_cc )
{
	int ret = 0;
	list<event>::iterator i;

	lock();

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

		if( (*i).get_status()    == a_status ){

			unsigned char d0,d1;
			(*i).get_data( &d0, &d1 );

			if ( (a_status == EVENT_CONTROL_CHANGE && d0 == a_cc )
				 || (a_status != EVENT_CONTROL_CHANGE) ){

				if ( (*i).is_selected( ))
					ret++;
			}
	    }
	}
    
	unlock();
    
    return ret;
}


/* selects events in range..  tick start, note high, tick end
   note low */
int
sequence::select_note_events( long a_tick_s, int a_note_h,
			      long a_tick_f, int a_note_l )
{
    int ret=0;
    list<event>::iterator i;

    lock();

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

        if( (*i).is_note_off()                &&
            (*i).get_timestamp() >= a_tick_s &&
            (*i).get_note()      <= a_note_h &&
            (*i).get_note()      >= a_note_l ){

	    if ( (*i).is_linked() ){

		event *ev = (*i).get_linked();

		if ( ev->get_timestamp() <= a_tick_f ){
		    
		    (*i).select( );
		    ev->select( );
		    ret++;
		}
	    }
        }
	
	if ( ! (*i).is_linked() &&
	     ( (*i).is_note_on() ||
	       (*i).is_note_off() ) &&
	     (*i).get_timestamp()  >= a_tick_s - 16 &&
	     (*i).get_timestamp()  <= a_tick_f && 
	     (*i).get_note()       <= a_note_h &&
	     (*i).get_note()       >= a_note_l ) {

	    (*i).select( );
	    ret++;
	}
    }

    unlock();

    return ret;
}




/* select events in range, returns number
   selected */
int 
sequence::select_events( long a_tick_s, long a_tick_f, 
			 unsigned char a_status, 
			 unsigned char a_cc )
{
    int ret=0;
    list<event>::iterator i;

    lock();

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	if( (*i).get_status()    == a_status && 
	    (*i).get_timestamp() >= a_tick_s &&
	    (*i).get_timestamp() <= a_tick_f ){

	    unsigned char d0,d1;
	    (*i).get_data( &d0, &d1 );

	    if ( (a_status == EVENT_CONTROL_CHANGE &&
		  d0 == a_cc )
		 || (a_status != EVENT_CONTROL_CHANGE) ){

		(*i).select( );
		ret++;
	    }
	}
    }
    
    unlock();
    
    return ret;
}




void
sequence::select_all( void )
{
    lock();

    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ )
	(*i).select( );

    unlock();
}


/* unselects every event */
void 
sequence::unselect( void )
{
    lock();

    list<event>::iterator i;
    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ )
	(*i).unselect();

    unlock();
}


/* removes and adds readds selected in position */
void 
sequence::move_selected_notes( long a_delta_tick, int a_delta_note )
{
    event e;

    lock();

    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	
	/* is it being moved ? */
	if ( (*i).is_selected() ){

	    /* copy event */
	    e  = (*i);
	    e.unselect();

	    if ( (e.get_timestamp() + a_delta_tick) >= 0   &&
		 (e.get_note() + a_delta_note)      >= 0   &&
		 (e.get_note() + a_delta_note)      <  c_num_keys ){
		
		e.set_timestamp( e.get_timestamp() + a_delta_tick );
		e.set_note( e.get_note() + a_delta_note );
		
		add_event( &e );
	    }
	}
    }

    remove_selected();
    verify_and_link();

    unlock();
}




/* moves note off event */
void 
sequence::grow_selected( long a_delta_tick )
{
    event *on, *off, e;

    lock();

    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	
	if ( (*i).is_selected() &&
	     (*i).is_note_on() &&
	     (*i).is_linked() ){
	    
	    on = &(*i);
	    off = (*i).get_linked();

	    long length = 
		off->get_timestamp() - 
		on->get_timestamp() +
		a_delta_tick;

	    if ( length < 1 )
		length = 1;
 
	    /* unselect it */
	    on->unselect();

	    /* copy event */
	    e  = *off;
	    e.unselect();
	    
	    e.set_timestamp( on->get_timestamp() + length );
	    add_event( &e );
	}
    }
    
    remove_selected();
    verify_and_link();
    
    unlock();
}


void 
sequence::increment_selected( unsigned char a_status, unsigned char a_control )
{
    lock();

    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	
		if ( (*i).is_selected() &&
			 (*i).get_status() == a_status ){
			
			if ( a_status == EVENT_NOTE_ON || 
				 a_status == EVENT_NOTE_OFF || 
				 a_status == EVENT_AFTERTOUCH ||
				 a_status == EVENT_CONTROL_CHANGE || 
				 a_status == EVENT_PITCH_WHEEL ){
				
				(*i).increment_data2();
			}
			
			if ( a_status == EVENT_PROGRAM_CHANGE ||
				 a_status == EVENT_CHANNEL_PRESSURE ){
				
				(*i).increment_data1();
			}
		}
    }
        
    unlock();
}


void 
sequence::decrement_selected(unsigned char a_status, unsigned char a_control )
{
    lock();

    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){
	
	   	if ( (*i).is_selected() &&
			 (*i).get_status() == a_status ){
			
			if ( a_status == EVENT_NOTE_ON || 
				 a_status == EVENT_NOTE_OFF || 
				 a_status == EVENT_AFTERTOUCH ||
				 a_status == EVENT_CONTROL_CHANGE || 
				 a_status == EVENT_PITCH_WHEEL ){
				
				(*i).decrement_data2();
			}
			
			if ( a_status == EVENT_PROGRAM_CHANGE ||
				 a_status == EVENT_CHANNEL_PRESSURE ){
				
				(*i).decrement_data1();
			}
			
		}
	}
        
	unlock();
}



void sequence::clear_clipboard( void )
{
    m_list_clipboard->clear( );
}

void
sequence::copy_selected( void )
{
    list<event>::iterator i;

	lock();

    m_list_clipboard->clear( );

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	if ( (*i).is_selected() ){
	    m_list_clipboard->push_back( (*i) );
	}
    }

    long first_tick = (*m_list_clipboard->begin()).get_timestamp();
    
    for ( i = m_list_clipboard->begin(); i != m_list_clipboard->end(); i++ ){

	(*i).set_timestamp((*i).get_timestamp() - first_tick );
    }

	unlock();
}

void 
sequence::paste_selected( long a_tick, int a_note )
{



    list<event>::iterator i;
    int highest_note = 0;

	lock();

    for ( i = m_list_clipboard->begin(); i != m_list_clipboard->end(); i++ ){
	(*i).set_timestamp((*i).get_timestamp() + a_tick );
    }

    if ((*m_list_clipboard->begin()).is_note_on() ||
	(*m_list_clipboard->begin()).is_note_off() ){
	
	for ( i = m_list_clipboard->begin(); i != m_list_clipboard->end(); i++ )
	    if ( (*i).get_note( ) > highest_note ) highest_note = (*i).get_note();

       

	for ( i = m_list_clipboard->begin(); i != m_list_clipboard->end(); i++ ){

	    (*i).set_note( (*i).get_note( ) - (highest_note - a_note) );
	}
    }

    m_list_event->merge( *m_list_clipboard );
    m_list_event->sort();

    verify_and_link();

    reset_draw_marker();
    
	unlock();

}


/* change */
void 
sequence::change_event_data_range( long a_tick_s, long a_tick_f,
				   unsigned char a_status,
				   unsigned char a_cc,
				   unsigned char a_data_s,
				   unsigned char a_data_f )
{
    lock();

    unsigned char d0, d1;
    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	/* initially false */
	bool set = false;	
	(*i).get_data( &d0, &d1 );
	
	/* correct status and not CC */
	if ( a_status != EVENT_CONTROL_CHANGE &&
	     (*i).get_status() == a_status )
	    set = true;
	
	/* correct status and correct cc */
	if ( a_status == EVENT_CONTROL_CHANGE &&
	     (*i).get_status() == a_status &&
	     d0 == a_cc )
	    set = true;
	
	/* in range? */
	if ( !((*i).get_timestamp() >= a_tick_s &&
	       (*i).get_timestamp() <= a_tick_f ))
	    set = false;
	
	if ( set ){
	    
	    float weight;
	    
	    /* no divide by 0 */
	    if( a_tick_f == a_tick_s )
		a_tick_f = a_tick_s + 1;
	    
	    /* ratio of v1 to v2 */
	    weight = 
		(float)( (*i).get_timestamp() - a_tick_s ) /
		(float)( a_tick_f - a_tick_s ); 
	    
	    int newdata = (int) 
		((weight         * (float) a_data_f ) +
		 ((1.0f - weight) * (float) a_data_s ));
	    
	    if ( a_status == EVENT_NOTE_ON )
		d1 = newdata;

	    if ( a_status == EVENT_NOTE_OFF )
		d1 = newdata;

	    if ( a_status == EVENT_AFTERTOUCH )
		d1 = newdata;

	    if ( a_status == EVENT_CONTROL_CHANGE )
		d1 = newdata;

	    if ( a_status == EVENT_PROGRAM_CHANGE )
		d0 = newdata; /* d0 == new patch */

	    if ( a_status == EVENT_CHANNEL_PRESSURE )
		d0 = newdata; /* d0 == pressure */

	    if ( a_status == EVENT_PITCH_WHEEL )
		d1 = newdata;

	    (*i).set_data( d0, d1 );
	}	    
    }

    unlock();
}




void 
sequence::add_note( long a_tick, long a_length, int a_note )
{

    lock();

    if ( a_tick >= 0 && 
	 a_note >= 0 &&
	 a_note < c_num_keys ){

	event e;

	e.set_status( EVENT_NOTE_ON );
	e.set_data( a_note, 127 );
	e.set_timestamp( a_tick );

	add_event( &e );

	e.set_status( EVENT_NOTE_OFF );
	e.set_data( a_note, 127 );
	e.set_timestamp( a_tick + a_length );

	add_event( &e );
    }

    verify_and_link();
    unlock();
}


void 
sequence::add_event( long a_tick, 
		     unsigned char a_status,
		     unsigned char a_d0,
		     unsigned char a_d1 )
{
    lock();

    if ( a_tick >= 0 ){
	
	event e;

	e.set_status( a_status );
	e.set_data( a_d0, a_d1 );
	e.set_timestamp( a_tick );

	add_event( &e );
    }
    verify_and_link();

    unlock();
}


void 
sequence::stream_event(  event *a_ev  )
{
    lock();

    /* adjust tick */

    a_ev->mod_timestamp( m_length );

    if ( m_recording ){
	
	add_event( a_ev );
	m_dirty_main = m_dirty_edit = true;
    }

    if ( m_thru ){

	put_event_on_bus( a_ev );
    }

    link_new();
    /* update view */

    unlock();
} 

bool
sequence::is_dirty_main( )
{
    lock();

    bool ret = m_dirty_main;
    m_dirty_main = false;

    unlock();

    return ret;
}


bool
sequence::is_dirty_edit( )
{
    lock();

    bool ret = m_dirty_edit;
    m_dirty_edit = false;

    unlock();

    return ret;
}


/* plays a note from the paino roll */
void 
sequence::play_note_on( int a_note )
{
    lock();

    event e;

    e.set_status( EVENT_NOTE_ON );
    e.set_data( a_note, 127 );
    m_masterbus->play( m_bus, &e, m_midi_channel ); 

    m_masterbus->flush();

    unlock();
}


/* plays a note from the paino roll */
void 
sequence::play_note_off( int a_note )
{
    lock();

    event e;

    e.set_status( EVENT_NOTE_OFF );
    e.set_data( a_note, 127 );
    m_masterbus->play( m_bus, &e, m_midi_channel ); 

    m_masterbus->flush();

    unlock();
}


void 
sequence::clear_triggers( void )
{
	lock();
	m_list_trigger->clear();
	unlock();

}


/* adds trigger, a_state = true, range is on.
                 a_state = false, range is off */ 
void 
sequence::add_trigger( long a_tick, long a_length, bool a_state )
{
    lock();

    trigger e1,e2;

    e1.m_tick  = a_tick;
    e2.m_tick  = a_tick + a_length;
 
    e1.m_state = a_state;
    e2.m_state = ! a_state;

    m_list_trigger->push_front( e1 );
    m_list_trigger->push_front( e2 );

    m_list_trigger->sort();

    list<trigger>::iterator i = m_list_trigger->begin();
    list<trigger>::iterator j = m_list_trigger->begin();	 
    j++;

    list<trigger>::iterator first = m_list_trigger->begin();
    list<trigger>::iterator last = m_list_trigger->begin();

    bool contain = false;

    /* range on */
    if ( a_state ){

	while(  j != m_list_trigger->end() ){
	
	    /* both on */
	    if ( (*i).m_state == true && (*j).m_state == true ){
		first = j;
		contain = true;
	    }
	    
	    /* both off */
	    if ( (*i).m_state == false && (*j).m_state == false ){
		last = j;
	    }

	    i++; j++;
	}
	
    }
    /* range off */
    else {

	if ( m_list_trigger->size() == 2 ){
	    
	    m_list_trigger->clear();
	    unlock();
	    return;
	}

	while(  j != m_list_trigger->end() ){
	    
	    /* both off */
	    if ( (*i).m_state == false && (*j).m_state == false ){
		first = j; last = m_list_trigger->end();
		contain = true;
	    }
	    
	    
	    /* both on */
	    if ( (*i).m_state == true && (*j).m_state == true ){
		last = j;
	    }

	    i++; j++;
	}
    }

    /* we might get an OFF at the begining */
    if ( (*m_list_trigger->begin()).m_state == false ){
	first = m_list_trigger->begin(); contain = true;
    }

    if ( contain )
	m_list_trigger->erase(first,last);

    if ( (*m_list_trigger->begin()).m_state == false )
	m_list_trigger->pop_front();

    validate_triggers( );

    /* elements should always be in pairs,
       if not clear, or the drawing routines will hang,
       this should really never be called */
    if ( m_list_trigger->size() % 2 != 0 ){
	
	m_list_trigger->clear();
    }

    unlock();
}

void 
sequence::validate_triggers( void  )
{

    list<trigger>::iterator i = m_list_trigger->begin();
    list<trigger>::iterator j = m_list_trigger->begin();

	lock();

    j++;
    
    while(  j != m_list_trigger->end() ){
	
 	if ( ((*i).m_state != (*j).m_state  &&
	      (*i).m_tick + 1 >= (*j).m_tick )){
	    
 	    m_list_trigger->erase(i);
 	    m_list_trigger->erase(j);
	    
	    i = j = m_list_trigger->begin(); j++;
	    
 	} else {
	    
 	    i++; j++;
 	}
    }

	unlock();
}

void 
sequence::move_triggers( long a_start_tick, 
			 long a_distance, 
			 bool a_direction )
{

	lock();

    long first_tick = a_start_tick;

    if ( ! a_direction ){

	add_trigger( a_start_tick, a_distance, false );
	first_tick += a_distance;
    }

    list<trigger>::iterator i = m_list_trigger->begin(); 

    while(  i != m_list_trigger->end() ){

	if ( (*i).m_tick >= first_tick ){

	    if ( a_direction )
		(*i).m_tick += a_distance;
	    else
		(*i).m_tick -= a_distance;
	}
	i++;
    }

    validate_triggers( );

    if ( a_direction ){
	add_trigger( a_start_tick, a_distance, false );
    }

	unlock();
}


void
sequence::copy_triggers( long a_start_tick, 
			 long a_distance  )
{

    long first_tick = a_start_tick;

	lock();

    move_triggers( a_start_tick, 
		   a_distance, 
		   true );
    
	
    list<trigger> m_list_trigger_copy = *m_list_trigger;
 
    list<trigger>::iterator i = m_list_trigger_copy.begin(); 
    trigger e1;

    while(  i != m_list_trigger_copy.end() ){

 	if ( (*i).m_tick >= first_tick + a_distance &&
 	     (*i).m_tick < first_tick + (a_distance * 2) ){

	    e1.m_tick  = (*i).m_tick - a_distance;
	    e1.m_state = (*i).m_state;

	    m_list_trigger->push_front( e1 );
	}
	
	i++;
    }

    /* last one turned on */
    if ( e1.m_state == true ){
	
	e1.m_state = false;
	e1.m_tick = first_tick + a_distance;

	m_list_trigger->push_front( e1 );
    }
	

    m_list_trigger->sort();
    validate_triggers( );

	unlock();
}



long
sequence::get_max_trigger( void )
{
    lock();

    long ret;

    if ( m_list_trigger->size() > 0 )
	ret = m_list_trigger->back().m_tick;
    else
	ret = 0;

    unlock();

    return ret;
}

bool 
sequence::get_trigger_state( long a_tick )
{
    lock();

    bool ret = false;
    list<trigger>::iterator i;

    for ( i = m_list_trigger->begin(); i != m_list_trigger->end(); i++ ){

	if ( (*i).m_tick <= a_tick )
	    ret = (*i).m_state;
	else 
	    break;
    }

    unlock();

    return ret;
}




/* this refreshes the play marker to the LastTick */
void 
sequence::reset_draw_marker( void )
{
	lock();

    m_iterator_draw = m_list_event->begin();

	unlock();
}

void
sequence::reset_draw_trigger_marker( void )
{
	lock();

    m_iterator_draw_trigger = m_list_trigger->begin();

	unlock();
}


int
sequence::get_lowest_note_event( void )
{
    lock();

    int ret = 127;
    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	if ( (*i).is_note_on() || (*i).is_note_off() )
	    if ( (*i).get_note() < ret )
		ret = (*i).get_note();
    }

    unlock();

    return ret;
}



int
sequence::get_highest_note_event( void )
{
    lock();

    int ret = 0;
    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	if ( (*i).is_note_on() || (*i).is_note_off() )
	    if ( (*i).get_note() > ret )
		ret = (*i).get_note();
    }

    unlock();

    return ret;
}

draw_type 
sequence::get_next_note_event( long *a_tick_s,
			       long *a_tick_f,
			       int  *a_note,
			       bool *a_selected,
			       int  *a_velocity  )
{

    draw_type ret = DRAW_FIN;

    while (  m_iterator_draw  != m_list_event->end() )
    {
	*a_tick_s   = (*m_iterator_draw).get_timestamp();
	*a_note     = (*m_iterator_draw).get_note();
	*a_selected = (*m_iterator_draw).is_selected();
	*a_velocity = (*m_iterator_draw).get_note_velocity();

	/* note on, so its linked */
	if( (*m_iterator_draw).is_note_on() &&
	    (*m_iterator_draw).is_linked() ){

	    *a_tick_f   = (*m_iterator_draw).get_linked()->get_timestamp();
	    
	    ret = DRAW_NORMAL_LINKED;
	    m_iterator_draw++;
	    return ret;
	}
	
	else if( (*m_iterator_draw).is_note_on() &&
		 (! (*m_iterator_draw).is_linked()) ){
	    
	    ret = DRAW_NOTE_ON;
	    m_iterator_draw++;
	    return ret;
	}
	
	else if( (*m_iterator_draw).is_note_off() &&
		 (! (*m_iterator_draw).is_linked()) ){
	    
	    ret = DRAW_NOTE_OFF;
	    m_iterator_draw++;
	    return ret;
	}
	
	/* keep going until we hit null or find a NoteOn */
	m_iterator_draw++;
    }
    return DRAW_FIN;
}


bool 
sequence::get_next_event( unsigned char a_status,
			  unsigned char a_cc,
			  long *a_tick,
			  unsigned char *a_D0,
			  unsigned char *a_D1,
			  bool *a_selected )
{
    while (  m_iterator_draw  != m_list_event->end() ){ 
	
	/* note on, so its linked */
	if( (*m_iterator_draw).get_status() == a_status ){
	    
	    (*m_iterator_draw).get_data( a_D0, a_D1 );
	    *a_tick   = (*m_iterator_draw).get_timestamp();
	    *a_selected = (*m_iterator_draw).is_selected();
	    
	    /* either we have a control chage with the right CC
	       or its a different type of event */
	    if ( (a_status == EVENT_CONTROL_CHANGE &&
		  *a_D0 == a_cc )
		 || (a_status != EVENT_CONTROL_CHANGE) ){
		
		/* we have a good one */
		/* update and return */
		m_iterator_draw++;
		return true;
	    }
	}
	/* keep going until we hit null or find a NoteOn */
	m_iterator_draw++;
    }
    return false;
}

bool 
sequence::get_next_trigger( long *a_tick_on, long *a_tick_off )
{
    while (  m_iterator_draw_trigger  != m_list_trigger->end() ){ 

	*a_tick_on  = (*m_iterator_draw_trigger).m_tick;
	m_iterator_draw_trigger++;
	*a_tick_off = (*m_iterator_draw_trigger).m_tick;
	m_iterator_draw_trigger++;

	return true;
    }
    return false;
}


void 
sequence::remove_all( void )
{
    lock();

    m_list_event->clear();

    unlock();

}

sequence& 
sequence::operator= (const sequence& a_rhs)
{
    lock();

    /* dont copy to self */
    if (this != &a_rhs){
	
	(*m_list_event)   = (*a_rhs.m_list_event);
	(*m_list_trigger)   = (*a_rhs.m_list_trigger);

	m_midi_channel = a_rhs.m_midi_channel;
	m_masterbus    = a_rhs.m_masterbus;
	m_bus          = a_rhs.m_bus;
	m_name         = a_rhs.m_name;
	m_length       = a_rhs.m_length;

	m_time_beats_per_measure = a_rhs.m_time_beats_per_measure;
	m_time_beat_width = a_rhs.m_time_beat_width;

	m_playing      = false;

	/* no notes are playing */
	for (int i=0; i< c_midi_notes; i++ )
	    m_playing_notes[i] = 0;

	/* reset */
	zero_markers( );

    }

    verify_and_link();

    unlock();

    return *this;

}



void 
sequence::lock( )
{
    m_mutex.lock();
}


void 
sequence::unlock( )
{   
    m_mutex.unlock();
}



const char* 
sequence::get_name()
{
    return m_name.c_str();
}

long 
sequence::get_last_tick( )
{
    return m_last_tick % m_length;
}

void
sequence::set_midi_bus( char  a_mb )
{
    lock();

    /* off notes except initial */
	off_playing_notes( );

    this->m_bus = a_mb;
	m_dirty_main = m_dirty_edit = true;

    unlock();
}

char
sequence::get_midi_bus(  )
{
    return this->m_bus;
}



void 
sequence::set_length( long a_len )
{
    lock();

    if ( a_len > c_ppqn * 64 )
      a_len = c_ppqn * 64;
 
    bool was_playing = get_playing();

    /* turn everything off */
    set_playing( false );

    m_length = a_len;

    verify_and_link();

    reset_draw_marker();
    
    /* start up and refresh */
    if ( was_playing )
	set_playing( true );
    
    unlock();
}


long 
sequence::get_length( )
{
    return m_length;
}



void 
sequence::set_playing( bool a_p )
{
    lock();

    if (a_p){
	
	/* turn on */
	m_playing = true;

    } else {

	/* turn off */
	m_playing = false;
	off_playing_notes();

    }  

	m_dirty_main = m_dirty_edit = true;
	m_queued = false;

    unlock();
}


void 
sequence::toggle_playing()
{
    set_playing( ! get_playing() );
}

bool 
sequence::get_playing( )
{
    return m_playing;
}



void 
sequence::set_recording( bool a_r )
{
    lock();
    m_recording = a_r;
    unlock();
}


bool 
sequence::get_recording( )
{
    return m_recording;
}



void 
sequence::set_thru( bool a_r )
{
    lock();
    m_thru = a_r;
    unlock();
}


bool 
sequence::get_thru( )
{
    return m_thru;
}


/* sets sequence name */
void 
sequence::set_name( char *a_name )
{
    m_name = a_name;
}

void
sequence::set_name( string a_name )
{
    m_name = a_name;
}

void 
sequence::set_midi_channel( unsigned char a_ch )
{
    lock();
    off_playing_notes( );
    m_midi_channel = a_ch;
	m_dirty_main = m_dirty_edit = true;
    unlock();
}

unsigned char 
sequence::get_midi_channel( )
{
    return m_midi_channel;
}


void 
sequence::print()
{
    printf("[%s]\n", m_name.c_str()  );

    for( list<event>::iterator i = m_list_event->begin(); i != m_list_event->end(); i++ )
	(*i).print();
    printf("events[%d]\n\n",m_list_event->size());

}



void 
sequence::put_event_on_bus( event *a_e )
{		
    lock();

    char note = a_e->get_note();
	bool skip = false;
	
    if ( a_e->is_note_on() ){
		
		m_playing_notes[note]++;
    }
    if ( a_e->is_note_off() ){

		if (  m_playing_notes[note] <= 0 ){
			skip = true;
		}
		else {
			m_playing_notes[note]--;
		}		
    }

	if ( !skip ){
		m_masterbus->play( m_bus, a_e,  m_midi_channel );   
	}

    m_masterbus->flush();
    
    unlock();
}


void 
sequence::off_playing_notes()
{
    lock();


    event e;
    
    for ( int x=0; x< c_midi_notes; x++ ){
	
		while( m_playing_notes[x] > 0 ){
			
			e.set_status( EVENT_NOTE_OFF );
			e.set_data( x, 0 );
			
			m_masterbus->play( m_bus, &e, m_midi_channel ); 
			
			m_playing_notes[x]--;
		}
    }
    
    m_masterbus->flush();
    
 
    unlock();
}






void 
addListVar( list<char> *a_list, long a_var )
{
    long buffer;
    buffer = a_var & 0x7F;

    /* we shift it right 7, if there is 
       still set bits, encode into buffer
       in reverse order */
    while ( ( a_var >>= 7) ){
	buffer <<= 8;
	buffer |= ((a_var & 0x7F) | 0x80);
    }

    while (true){
	
	a_list->push_front( (char) buffer & 0xFF );

	if (buffer & 0x80)
	    buffer >>= 8;
	else
	    break;
    }
}

void
addLongList( list<char> *a_list, long a_x )
{
    a_list->push_front(  (a_x & 0xFF000000) >> 24 );
    a_list->push_front(  (a_x & 0x00FF0000) >> 16 );
    a_list->push_front(  (a_x & 0x0000FF00) >> 8  );
    a_list->push_front(  (a_x & 0x000000FF)       );
}
 

void 
sequence::fill_list( list<char> *a_list, int a_pos )
{

    lock();

    /* clear list */
    *a_list = list<char>();
    
    /* sequence number */
    addListVar( a_list, 0 );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x00 );
    a_list->push_front( 0x02 );
    a_list->push_front( (a_pos & 0xFF00) >> 8 );
    a_list->push_front( (a_pos & 0x00FF)      );
            
    /* name */
    addListVar( a_list, 0 );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x03 );

    int length =  m_name.length();
    if ( length > 0x7F ) length = 0x7f;
    a_list->push_front( length );

    for ( int i=0; i< length; i++ )
	a_list->push_front( m_name.c_str()[i] );	
 
    long timestamp = 0, delta_time = 0, prev_timestamp = 0;
    list<event>::iterator i;

    for ( i = m_list_event->begin(); i != m_list_event->end(); i++ ){

	event e = (*i);
	timestamp = e.get_timestamp();
	delta_time = timestamp - prev_timestamp;
	prev_timestamp = timestamp;

	/* encode delta_time */
	addListVar( a_list, delta_time );

	/* now that the timestamp is encoded, do the status and
	   data */

	a_list->push_front( e.m_status | m_midi_channel );

	switch( e.m_status & 0xF0 ){

	case 0x80:	  
	case 0x90:
	case 0xA0:
	case 0xB0:
	case 0xE0: 
	    
	    a_list->push_front(  e.m_data[0] );
	    a_list->push_front(  e.m_data[1] );

	    //printf ( "- d[%2X %2X]\n" , e.m_data[0], e.m_data[1] ); 

	    break;
	    
	case 0xC0:
	case 0xD0:
	    
	    a_list->push_front(  e.m_data[0] );

	    //printf ( "- d[%2X]\n" , e.m_data[0] ); 

	    break;
	    
	default: 
	    break;
	}
    }

    int num_triggers = m_list_trigger->size();
    list<trigger>::iterator t = m_list_trigger->begin();

    addListVar( a_list, 0 );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x7F );
    addListVar( a_list, num_triggers * 4 + 4);
    addLongList( a_list, c_triggers );

    for ( int i=0; i<num_triggers; i++ ){

	addLongList( a_list, (*t).m_tick );
	t++;
    }

    /* bus */
    addListVar( a_list, 0 );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x7F );
    a_list->push_front( 0x05 );
    addLongList( a_list, c_midibus );
    a_list->push_front( m_bus  );

    /* timesig */
    addListVar( a_list, 0 );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x7F );
    a_list->push_front( 0x06 );
    addLongList( a_list, c_timesig );
    a_list->push_front( m_time_beats_per_measure  );
    a_list->push_front( m_time_beat_width  );

    /* channel */
    addListVar( a_list, 0 );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x7F );
    a_list->push_front( 0x05 );
    addLongList( a_list, c_midich );
    a_list->push_front( m_midi_channel );

    delta_time = m_length - prev_timestamp;
 
    /* meta track end */
    addListVar( a_list, delta_time );
    a_list->push_front( 0xFF );
    a_list->push_front( 0x2F );
    a_list->push_front( 0x00 );

    unlock();
}





//     list<char> triggers;

//     /* triggers */

//     list<trigger>::iterator t;
//     for ( t = m_list_trigger->begin(); t != m_list_trigger->end(); t++ ){

// 	addLongList( &triggers, (*t).m_tick );
// 	printf ( "[%ld]\n", (*t).m_tick );
//     }    

//     addListVar( a_list, 0 );
//     a_list->push_front( 0xFF );
//     a_list->push_front( 0x7F );

//     a_list->push_front( 0x05 );
//     addLongList( a_list, c_triggersmidibus );


   
