//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free So%ftware Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "event.h"
#include "seqroll.h"

seqroll::seqroll(sequence *a_seq, 
		 int a_zoom, 
		 int a_snap, 
		 seqdata *a_seqdata_wid, 
		 seqevent *a_seqevent_wid,
		 seqkeys *a_seqkeys_wid,
		 mainwid *a_mainwid,
		 int a_pos )
: DrawingArea() 
{    
    using namespace Menu_Helpers;

    Gdk_Colormap colormap = get_default_colormap();

    m_black = Gdk_Color( "black" );
    m_white = Gdk_Color( "white" );
    m_grey  = Gdk_Color( "grey" );

    colormap.alloc( m_black );
    colormap.alloc( m_white );
    colormap.alloc( m_grey );

    m_seq =   a_seq;
    m_zoom = a_zoom;
    m_snap =  a_snap;
    m_seqdata_wid = a_seqdata_wid;
    m_seqevent_wid = a_seqevent_wid;
    m_mainwid = a_mainwid;
    m_seqkeys_wid = a_seqkeys_wid;
    m_pos = a_pos;

    m_clipboard = new sequence( );

    add_events( GDK_BUTTON_PRESS_MASK | 
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_KEY_PRESS_MASK |
		GDK_KEY_RELEASE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_ENTER_NOTIFY_MASK |
		GDK_LEAVE_NOTIFY_MASK );

    m_selecting = false;
    m_moving    = false;
    m_growing   = false;
    m_adding    = false;
    m_paste     = false;

    m_old_progress_x = 0;

    m_scale = 0;
    m_key = 0;

} 

seqroll::~seqroll( )
{

}

/* popup menu calls this */
void 
seqroll::set_adding( bool a_adding )
{
    if ( a_adding ){

	get_window().set_cursor(  Gdk_Cursor( GDK_PENCIL ));
	m_adding = true;
    
    } else {

	get_window().set_cursor( Gdk_Cursor( GDK_LEFT_PTR ));
	m_adding = false;
    }
}




void 
seqroll::realize_impl()
{
    // we need to do the default realize
    Gtk::DrawingArea::realize_impl();


    set_flags( GTK_CAN_FOCUS );

    // Now we can allocate any additional resources we need
    m_window = get_window();
    m_gc.create( m_window );
    m_window.clear();

}


void 
seqroll::update_sizes()
{
    /* set default size */
    size( m_seq->get_length() / m_zoom , c_rollarea_y );

    /* window size */
    m_window_x = m_seq->get_length() / m_zoom;
    m_window_y = c_rollarea_y;

    /* create pixmaps with window dimentions */
    m_pixmap = Gdk_Pixmap( m_window,
			   m_window_x,
			   m_window_y );

    m_background = Gdk_Pixmap( m_window,
			       m_window_x,
			       m_window_y );

    /* and fill the background ( dotted lines n' such ) */
    fill_background_pixmap();

    m_mainwid->update_sequence_on_window( m_pos );
}

/* basically resets the whole widget as if it was realized again */
void 
seqroll::reset()
{
    update_sizes();
    update_pixmap();
    draw_pixmap_on_window();
}

/* updates background */
void 
seqroll::fill_background_pixmap()
{
    /* clear background */
    m_gc.set_foreground(m_white);
    m_background.draw_rectangle(m_gc,true,
				0,
				0, 
				m_window_x, 
				m_window_y );

    /* draw horz grey lines */
    m_gc.set_foreground(m_grey);
    m_gc.set_line_style( GDK_LINE_ON_OFF_DASH );
    m_gc.set_dashes( 1 );

    for ( int i=0; i<c_num_keys; i++ )
    {
	m_background.draw_line(m_gc,
			       0,
			       i * c_key_y,
			       m_window_x,
			       i * c_key_y );
    }

    if ( m_scale != c_scale_off ){

      for ( int i=0; i<c_num_keys; i++ ){
	
	if ( !c_scales_policy[m_scale][ (c_num_keys - i - 1 + ( 12 - m_key )) % 12] )

	  m_background.draw_rectangle(m_gc,true,
			       0,
			       i * c_key_y + 1,
			       m_window_x,
			       c_key_y - 1 );
      }
    }

    int numberLines = 128 / m_seq->get_bw() / m_zoom; 
    int distance = c_ppqn / 32;
    
    /* draw vert lines */
    for ( int i=0; i< c_maxbeats; i++ ){
	
	int base_line = i * (c_ppqn * 4) / m_seq->get_bw() / m_zoom;
	
	/* bars are darker */
	if ( i % m_seq->get_bpm() == 0 )
	    m_gc.set_foreground(m_black);
	else
	    m_gc.set_foreground(m_grey);

	/* solid line on every beat */
	m_gc.set_line_style( GDK_LINE_SOLID );
	m_background.draw_line(m_gc,
			       base_line,
			       0,
			       base_line,
			       m_window_y);
	
	/* in betweens */
 	for ( int j=1; j < numberLines; j++ ){
	    
 	    m_gc.set_foreground(m_grey);
 	    m_gc.set_line_style( GDK_LINE_ON_OFF_DASH );
 	    m_gc.set_dashes( 1 );
 	    m_background.draw_line(m_gc,
 				   base_line + j * distance,
 				   0,
 				   base_line + j * distance,
 				   m_window_y);
 	}
    }

    /* reset line style */
    m_gc.set_line_style( GDK_LINE_SOLID );
}

/* sets zoom, resets */
void 
seqroll::set_zoom( int a_zoom )
{
    if ( m_zoom != a_zoom ){

	m_zoom = a_zoom;
	reset();
    }
}

/* simply sets the snap member */
void 
seqroll::set_snap( int a_snap )
{
    m_snap = a_snap;
}

/* sets the music scale */
void 
seqroll::set_scale( int a_scale )
{
  if ( m_scale != a_scale ){
    m_scale = a_scale;
    reset();
  }

}

/* sets the key */
void 
seqroll::set_key( int a_key )
{
  if ( m_key != a_key ){
    m_key = a_key;
    reset();
  }
}



/* draws background pixmap on main pixmap,
   then puts the events on */
void 
seqroll::update_pixmap()
{
    m_pixmap.draw_pixmap(m_gc, 
			 m_background, 
			 0,
			 0,
			 0,
			 0,
			 m_window_x,
			 m_window_y);
    draw_events_on_pixmap();

    m_mainwid->update_sequence_on_window( m_pos );
}




void 
seqroll::draw_progress_on_window()
{

	
	m_window.draw_pixmap(m_gc, 
			     m_pixmap, 
			     m_old_progress_x,
			     0,
			     m_old_progress_x,
			     0,
			     1,
			     m_window_y );
	
	m_old_progress_x = m_seq->get_last_tick() / m_zoom;

	if ( m_old_progress_x != 0 ){	
	    
	    m_gc.set_foreground(m_black);
	    m_window.draw_line(m_gc,
			       m_old_progress_x,
			       0,
			       m_old_progress_x, 
			       m_window_y);
	}
}



void seqroll::draw_events_on( Gdk_Drawable *a_draw )
{
    long tick_s;
    long tick_f;
    int note;
    
    int note_x;
    int note_width;
    int note_y;
    int note_height;

    bool selected;

    int velocity;

    draw_type dt;

    m_seq->reset_draw_marker();

    /* draw boxes from sequence */
    m_gc.set_foreground( m_black );

    while ( (dt = m_seq->get_next_note_event( &tick_s, &tick_f, &note, 
					      &selected, &velocity )) != DRAW_FIN ){
	
	/* turn into screen corrids */
	note_x = tick_s / m_zoom;
	note_y = c_rollarea_y -(note * c_key_y) - c_key_y - 1 + 2;
	note_height = c_key_y - 3;

	//printf( "drawing note[%d] tick_start[%d] tick_end[%d]\n",
	//	    note, tick_start, tick_end );

	int in_shift = 0;
	int length_add = 0;

	if ( dt == DRAW_NORMAL_LINKED ){

	    note_width = (tick_f - tick_s) / m_zoom;
	    if ( note_width < 1 ) note_width = 1;

	}
	else {
	    note_width = 16 / m_zoom;
	}
	    
	if ( dt == DRAW_NOTE_ON ){

	    in_shift = 0;
	    length_add = 2;
	}       

	if ( dt == DRAW_NOTE_OFF ){

	    in_shift = -1;
	    length_add = 1;
	}     
	 
	m_gc.set_foreground(m_black);
	a_draw->draw_rectangle(m_gc,true,
			       note_x,
			       note_y, 
			       note_width, 
			       note_height);

	/* draw inside box if there is room */
	if ( note_width > 3 ){
	    
	    if ( selected )
		m_gc.set_foreground(m_grey);
	    else
		m_gc.set_foreground(m_white);
	    
	    a_draw->draw_rectangle(m_gc,true,
				   note_x + 1 + in_shift,
				   note_y + 1, 
				   note_width - 3 + length_add, 
				   note_height - 3);	   
	}
    }
} 



/* fills main pixmap with events */
void 
seqroll::draw_events_on_pixmap()
{
    draw_events_on( &m_pixmap ); 
}

/* draws pixmap, tells event to do the same */
void 
seqroll::draw_pixmap_on_window()
{
    /* we changed something on this window, and chances are we
       need to update the event widget as well */
   
    queue_draw(); 
}

int 
seqroll::idle_redraw()
{
    draw_events_on( &m_window );
    draw_events_on( &m_pixmap );
  
    return true;
}



void 
seqroll::draw_selection_on_window()
{
    int x,y,w,h;

    
    if ( m_selecting  ||  m_moving || m_paste ||  m_growing ){
	 
	m_gc.set_line_style( GDK_LINE_SOLID );

	/* replace old */
	m_window.draw_pixmap(m_gc, 
			     m_pixmap, 
			     m_old.x,
			     m_old.y,
			     m_old.x,
			     m_old.y,
			     m_old.width + 1,
			     m_old.height + 1 );
    }

    if ( m_selecting ){
	
	xy_to_rect ( m_drop_x,
		     m_drop_y,
		     m_current_x,
		     m_current_y,
		     &x, &y,
		     &w, &h );
	
	m_old.x = x;
	m_old.y = y;
	m_old.width = w;
	m_old.height = h;

	m_gc.set_foreground(m_black);
	m_window.draw_rectangle(m_gc,false,
				x,
				y, 
				w, 
				h );
    }
    
    if ( m_moving || m_paste ){

	int delta_x = m_current_x - m_drop_x;
	int delta_y = m_current_y - m_drop_y;

	x = m_selected.x + delta_x;
	y = m_selected.y + delta_y;

	m_gc.set_foreground(m_black);
	m_window.draw_rectangle(m_gc,false,
				x,
				y, 
				m_selected.width, 
				m_selected.height );
	m_old.x = x;
	m_old.y = y;
	m_old.width = m_selected.width;
	m_old.height = m_selected.height;
    }

    if ( m_growing ){

	int delta_x = m_current_x - m_drop_x;
	int width = delta_x + m_selected.width;

	if ( width < 1 ) 
	    width = 1;

	m_gc.set_foreground(m_black);
	m_window.draw_rectangle(m_gc,false,
				m_selected.x,
				m_selected.y, 
				width, 
				m_selected.height );

	m_old.x = m_selected.x;
	m_old.y = m_selected.y;
	m_old.width = width;
	m_old.height = m_selected.height;

    }

}

int 
seqroll::expose_event_impl(GdkEventExpose* e)
{
    m_window.draw_pixmap(m_gc, 
    			 m_pixmap, 
    			 e->area.x,
			 e->area.y,
			 e->area.x,
			 e->area.y,
			 e->area.width,
			 e->area.height );

    draw_selection_on_window();
    return true;
}





/* takes screen corrdinates, give us notes and ticks */
void 
seqroll::convert_xy( int a_x, int a_y, long *a_tick, int *a_note)
{
    *a_tick = a_x * m_zoom; 
    *a_note = (c_rollarea_y - a_y - 2) / c_key_y; 
}


/* notes and ticks to screen corridinates */
void 
seqroll::convert_tn( long a_ticks, int a_note, int *a_x, int *a_y)
{
    *a_x = a_ticks /  m_zoom;
    *a_y = c_rollarea_y - ((a_note + 1) * c_key_y);
}



/* checks mins / maxes..  the fills in x,y
   and width and height */
void 
seqroll::xy_to_rect( int a_x1,  int a_y1,
		     int a_x2,  int a_y2,
		     int *a_x,  int *a_y,
		     int *a_w,  int *a_h )
{
    if ( a_x1 < a_x2 ){
	*a_x = a_x1; 
	*a_w = a_x2 - a_x1;
    } else {
	*a_x = a_x2; 
	*a_w = a_x1 - a_x2;
    }
    
    if ( a_y1 < a_y2 ){
	*a_y = a_y1; 
	*a_h = a_y2 - a_y1;
    } else {
	*a_y = a_y2; 
	*a_h = a_y1 - a_y2;
    }
}

void
seqroll::convert_tn_box_to_rect( long a_tick_s, long a_tick_f,
				 int a_note_h, int a_note_l,
				 int *a_x, int *a_y, 
				 int *a_w, int *a_h )
{
    int x1, y1, x2, y2;

    /* convert box to X,Y values */
    convert_tn( a_tick_s, a_note_h, &x1, &y1 );
    convert_tn( a_tick_f, a_note_l, &x2, &y2 );

    xy_to_rect( x1, y1, x2, y2, a_x, a_y, a_w, a_h );

    *a_h += c_key_y;
}



void
seqroll::start_paste( )
{
     long tick_s;
     long tick_f;
     int note_h;
     int note_l;

     snap_x( &m_current_x );
     snap_y( &m_current_x );

     m_drop_x = m_current_x;
     m_drop_y = m_current_y;

     m_paste = true;

     /* get the box that selected elements are in */
     m_seq->get_clipboard_box( &tick_s, &note_h, 
			       &tick_f, &note_l );

     convert_tn_box_to_rect( tick_s, tick_f, note_h, note_l,
			     &m_selected.x,
			     &m_selected.y,
			     &m_selected.width,
			     &m_selected.height );

     /* adjust for clipboard being shifted to tick 0 */
     m_selected.x += m_drop_x;
     m_selected.y += (m_drop_y - m_selected.y);
}
	

int 
seqroll::button_press_event_impl(GdkEventButton* a_ev)
{
    int numsel;
	
    long tick_s;
    long tick_f;
    int note_h;
    int note_l;
	
    int norm_x, norm_y, snapped_x, snapped_y;
	
    grab_focus( ); 
	
    snapped_x = norm_x = (int) a_ev->x;
    snapped_y = norm_y = (int) a_ev->y;
	
    snap_x( &snapped_x );
    snap_y( &snapped_y );
	
    /* y is always snapped */
    m_current_y = m_drop_y = snapped_y;
	
    /* reset box that holds dirty redraw spot */
    m_old.x = 0;
    m_old.y = 0;
    m_old.width = 0;
    m_old.height = 0;
	
    if ( m_paste ){
		
		convert_xy( snapped_x, snapped_y, &tick_s, &note_h );
		m_paste = false;
		m_seq->paste_selected( tick_s, note_h );
    }
	

	
    /*      left mouse button     */
    if ( a_ev->button == 1 ){ 

		/* selection, normal x */
		m_current_x = m_drop_x = norm_x;
		
		/* turn x,y in to tick/note */
		convert_xy( m_drop_x, m_drop_y, &tick_s, &note_h );
		
		/* set to playing */
		numsel = m_seq->select_note_events( tick_s, note_h, 
											tick_s, note_h );
		
		/* none selected, start selection box */
		if ( numsel == 0 && !m_adding ){ 
			
			m_seq->unselect();	    
			m_selecting = true;
		}
		
		/* add a new note if we didnt select anything */
		else if ( numsel == 0 && m_adding ){
			
			/* adding, snapped x */
			m_current_x = m_drop_x = snapped_x;
			convert_xy( m_drop_x, m_drop_y, &tick_s, &note_h );
			
			/* add note, length = little less than snap */
			m_seq->add_note( tick_s, m_snap - 2, note_h );
		}
		
		/* if we clicked on an event, we can start moving all selected
		   notes */
		else if ( numsel > 0 ){
			
			m_moving = true;
			
			/* moving, snapped x */
			m_current_x = m_drop_x = snapped_x;
			convert_xy( m_drop_x, m_drop_y, &tick_s, &note_h );
			
			/* get the box that selected elements are in */
			m_seq->get_selected_box( &tick_s, &note_h, 
									 &tick_f, &note_l );
			
			convert_tn_box_to_rect( tick_s, tick_f, note_h, note_l,
									&m_selected.x,
									&m_selected.y,
									&m_selected.width,
									&m_selected.height );
			
			/* aligns to snap */
			m_drop_x = m_selected.x;
			m_drop_y = m_selected.y;
		} 
    }
    
    /*     right mouse button      */
    if ( a_ev->button == 3 ){
		set_adding( true );
    }
    
    /*     middle mouse button      */
    if ( a_ev->button == 2 ){	
		
		/* moving, normal x */
		m_current_x = m_drop_x = norm_x;
		convert_xy( m_drop_x, m_drop_y, &tick_s, &note_h );
		
		/* set to playing */
		numsel = m_seq->select_note_events( tick_s,   note_h, 
											tick_s+1, note_h );
		if ( numsel > 0 ){
			
			m_growing = true;
			
			/* get the box that selected elements are in */
			m_seq->get_selected_box( &tick_s, &note_h, 
									 &tick_f, &note_l );
			
			convert_tn_box_to_rect( tick_s, tick_f, note_h, note_l,
									&m_selected.x,
									&m_selected.y,
									&m_selected.width,
									&m_selected.height );	
		}
    }
    
    /* if they clicked, something changed */
    update_pixmap();
    draw_pixmap_on_window();  
    
    m_seqevent_wid->reset(); 
    m_seqdata_wid->reset();
    
    return true;
    
}


int 
seqroll::button_release_event_impl(GdkEventButton* a_ev)
{
    long tick_s;
    long tick_f;
    int note_h;
    int note_l;
    int x,y,w,h;
    int numsel;

    m_current_x = (int) a_ev->x;
    m_current_y = (int) a_ev->y;

    snap_y ( &m_current_y );

    if ( m_moving )
	snap_x( &m_current_x );

    int delta_x = m_current_x - m_drop_x;
    int delta_y = m_current_y - m_drop_y;

    long delta_tick;
    int delta_note;

    if ( a_ev->button == 1 ){

	if ( m_selecting ){
	    
	    xy_to_rect ( m_drop_x,
			 m_drop_y,
			 m_current_x,
			 m_current_y,
			 &x, &y,
			 &w, &h );

	    convert_xy( x,     y, &tick_s, &note_h );
	    convert_xy( x+w, y+h, &tick_f, &note_l );

	    numsel = m_seq->select_note_events( tick_s, note_h, tick_f, note_l +1 );
	}

	if ( m_moving ){
	    
	    /* convert deltas into screen corridinates */
	    convert_xy( delta_x, delta_y, &delta_tick, &delta_note );
	    
	    /* since delta_note was from delta_y, it will be filpped
	       ( delta_y[0] = note[127], etc.,so we have to adjust */
	    delta_note = delta_note - c_num_keys;
    
	    m_seq->move_selected_notes( delta_tick, delta_note );
	}
    }

    if ( a_ev->button == 2 ){	
	
	if ( m_growing ){
	    
	    /* convert deltas into screen corridinates */
	    convert_xy( delta_x, delta_y, &delta_tick, &delta_note );
	    m_seq->grow_selected( delta_tick );
	}
    }

    if ( a_ev->button == 3 ){	
	set_adding( false );
    }

    /* turn off */
    m_selecting = false;
    m_moving = false;
    m_growing = false;
    m_paste = false;

    /* if they clicked, something changed */

    update_pixmap();
    draw_pixmap_on_window(); 

    m_seqevent_wid->reset(); 
    m_seqdata_wid->reset();

    return true;
}

 

int 
seqroll::motion_notify_event_impl(GdkEventMotion* a_ev)
{
    m_current_x = (int) a_ev->x;
    m_current_y = (int) a_ev->y;

    int note;
    long tick;

    
    snap_y( &m_current_y );
    convert_xy( 0, m_current_y, &tick, &note );

    m_seqkeys_wid->set_hint_key( note );

    if ( m_selecting || m_moving || m_growing || m_paste ){

	if ( m_moving || m_paste )
	    snap_x( &m_current_x );
 
	draw_selection_on_window();
	return true;

    }
    return false;
}



/* performs a 'snap' on y */
void 
seqroll::snap_y( int *a_y )
{
    *a_y = *a_y - (*a_y % c_key_y);
}

/* performs a 'snap' on x */
void 
seqroll::snap_x( int *a_x )
{
    //snap = number pulses to snap to
    //m_zoom = number of pulses per pixel
    //so snap / m_zoom  = number pixels to snap to
    int mod = (m_snap / m_zoom);
    if ( mod <= 0 )
	mod = 1;
    
    *a_x = *a_x - (*a_x % mod );
}



int
seqroll::enter_notify_event_impl(GdkEventCrossing* a_p0)
{
  m_seqkeys_wid->set_hint_state( true );
  return false;
}



int 
seqroll::leave_notify_event_impl(GdkEventCrossing* a_p0)
{
  m_seqkeys_wid->set_hint_state( false );
  return false;
}



int 
seqroll::focus_in_event_impl(GdkEventFocus*)
{
    set_flags(GTK_HAS_FOCUS);

    m_seq->clear_clipboard();

    draw_focus( );
    return false;
}


int 
seqroll::focus_out_event_impl(GdkEventFocus*)
{
    unset_flags(GTK_HAS_FOCUS);
    draw_focus();

    return false;
}

int 
seqroll::key_press_event_impl(GdkEventKey* a_p0)
{
    bool ret = false;

    if ( a_p0->type == GDK_KEY_PRESS ){

	if ( a_p0->keyval ==  GDK_Delete ){

	    m_seq->remove_selected();
	    ret = true;
	}

	if ( a_p0->state & GDK_CONTROL_MASK ){

	    /* cut */
	    if ( a_p0->keyval == GDK_x || a_p0->keyval == GDK_X ){
		
		m_seq->copy_selected();
		m_seq->remove_selected();

		ret = true;
	    }
	    /* copy */
	    if ( a_p0->keyval == GDK_c || a_p0->keyval == GDK_C ){
		
		m_seq->copy_selected();
		m_seq->unselect();
		ret = true;
	    }
	    /* paste */
	    if ( a_p0->keyval == GDK_v || a_p0->keyval == GDK_V ){
		
		start_paste();
		ret = true;
	    }
	}
    }

    if ( ret == true ){
	
	reset();
	m_seqevent_wid->reset(); 
	m_seqdata_wid->reset();
	return true;
    }
    else

	return false;
}

