//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "perform.h"
#include "midibus.h"
#include "event.h"
#include <stdio.h>
#include <time.h>
#include <sched.h>

perform::perform()
{
    for (int i=0; i< c_max_sequence; i++ ){

		m_seqs[i] = NULL;
		m_seqs_active[i] = false;
		
		
    }

    m_running = false;
    m_looping = false;
    m_inputing = true;
    m_tick = 0;

    thread_trigger_width_ms = c_thread_trigger_width_ms;

    m_left_tick = 0;
    m_right_tick = c_ppqn * 4;

	midi_control zero = {false,false,0,0,0};

    for ( int i=0; i<c_seqs_in_set; i++ ){

		m_midi_cc_toggle[i] = zero;
		m_midi_cc_on[i] = zero;
		m_midi_cc_off[i] = zero;
    }

	m_offset = 0;
}

perform::~perform()
{
    m_inputing = false;
    m_running = false;

    pthread_join( m_out_thread, NULL );
    pthread_join( m_in_thread, NULL );

    for (int i=0; i< c_max_sequence; i++ ){
	if ( m_seqs_active[i] == true ){
	    delete m_seqs[i];
	}
    }
}

void 
perform::set_left_tick( long a_tick )
{
    m_left_tick = a_tick;
 
    if ( m_left_tick >= m_right_tick )
	m_right_tick = m_left_tick + c_ppqn * 4;
 
}

long 
perform::get_left_tick( void )
{ 
    return m_left_tick; 
}

void 
perform::set_right_tick( long a_tick ) 
{
    if ( a_tick >= c_ppqn * 4 ){
	
	m_right_tick = a_tick; 
	
	if ( m_right_tick <= m_left_tick )
	    m_left_tick = m_right_tick - c_ppqn * 4;
    }
}

long 
perform::get_right_tick( void )
{ 
    return m_right_tick; 
}


void 
perform::add_sequence( sequence *a_seq, int a_perf )
{
    /* check for perferred */
    if ( a_perf < c_max_sequence &&  
	 m_seqs_active[a_perf] == false &&
	 a_perf >= 0 ){
		    
      m_seqs[a_perf] = a_seq;
      m_seqs_active[a_perf] = true;
	    //a_seq->set_tag( a_perf );

    } else {

	for (int i=0; i< c_max_sequence; i++ ){
	    
	    if ( m_seqs_active[i] == false ){

	      m_seqs[i] = a_seq;
	      m_seqs_active[i] = true;

		//a_seq->set_tag( i  );
		break;
	    }
	}
    }
}


bool 
perform::is_active( int a_sequence )
{
    if ( a_sequence < 0 || a_sequence >= c_max_sequence )
	return false;
    
    return m_seqs_active[ a_sequence ];
}

sequence* 
perform::get_sequence( int a_sequence )
{
    return m_seqs[a_sequence];
}

mastermidibus* 
perform::get_master_midi_bus( )
{
    return &m_master_bus; 
}



void
perform::sequence_queued_toggle( int a_sequence )
{
    if ( m_seqs_active[a_sequence] == true ){
		assert( m_seqs[a_sequence] );
		m_seqs[a_sequence]->toggle_queued( );
    } 
}


void
perform::sequence_queued_off( int a_sequence )
{
    if ( m_seqs_active[a_sequence] == true ){
		assert( m_seqs[a_sequence] );
		m_seqs[a_sequence]->off_queued( );
    } 
}


void
perform::sequence_playing_toggle( int a_sequence )
{
    if ( m_seqs_active[a_sequence] == true ){
	assert( m_seqs[a_sequence] );
	m_seqs[a_sequence]->toggle_playing();
    } 
}


void
perform::sequence_playing_on( int a_sequence )
{
    if ( m_seqs_active[a_sequence] == true ){
		assert( m_seqs[a_sequence] );
		m_seqs[a_sequence]->set_playing(true);
    } 
}


void
perform::sequence_playing_off( int a_sequence )
{
    if ( m_seqs_active[a_sequence] == true ){
		assert( m_seqs[a_sequence] );
		m_seqs[a_sequence]->set_playing(false);
    } 
}




void 
perform::set_running( bool a_running )
{
    m_running = a_running;
}

bool 
perform::is_running( void )
{
    return m_running;
}

void 
perform::set_bpm(int a_bpm)
{
    if ( a_bpm < 20 )  a_bpm = 20;
    if ( a_bpm > 500 ) a_bpm = 500;

    m_master_bus.set_bpm( a_bpm );
}

int  
perform::get_bpm( )
{
    return  m_master_bus.get_bpm( );
}

void
perform::delete_sequence( int a_num )
{
  m_seqs_active[a_num] = false;

    if ( m_seqs[a_num] != NULL ){

	m_seqs[a_num]->set_playing( false );
	delete m_seqs[a_num];

    }   

}


void 
perform::new_sequence( int a_sequence )
{

    m_seqs[ a_sequence ] = new sequence();
    m_seqs[ a_sequence ]->set_master_midi_bus( &m_master_bus );
    m_seqs_active[ a_sequence ] = true;

}

midi_control *
perform::get_midi_control_toggle( unsigned int a_seq )
{
	if ( a_seq >= c_seqs_in_set )
		return NULL;
	return &m_midi_cc_toggle[a_seq];

}

midi_control *
perform::get_midi_control_on( unsigned int a_seq )
{
	if ( a_seq >= c_seqs_in_set )
		return NULL;
	return &m_midi_cc_on[a_seq];
}

midi_control *
perform::get_midi_control_off( unsigned int a_seq )
{
	if ( a_seq >= c_seqs_in_set )
		return NULL;
	return &m_midi_cc_off[a_seq];
}





void 
perform::print()
{
    //   for( int i=0; i<m_numSeq; i++ ){
	
	//printf("Sequence %d\n", i);
	//m_seqs[i]->print();
    // }

    //  m_master_bus.print();
}

void 
perform::set_screen_set_notepad( int a_screen_set, string *a_notepad )
{
    if ( a_screen_set < c_max_sets )
	m_screen_set_notepad[a_screen_set] = *a_notepad;
}


string *
perform::get_screen_set_notepad( int a_screen_set )
{
     return &m_screen_set_notepad[a_screen_set];
}




void 
perform::set_offset( int a_offset ) 
{ 
	m_offset = a_offset  * c_mainwnd_rows * c_mainwnd_cols; 
}


void 
perform::play( long a_tick )
{

    /* just run down the list of sequences and have them dump */

    m_tick = a_tick;
	
    for (int i=0; i< c_max_sequence; i++ ){
		
		if ( m_seqs_active[i] == true ){
			assert( m_seqs[i] );
			
			
			if ( m_seqs[i]->get_queued() &&
				 m_seqs[i]->get_queued_tick() <= a_tick ){
				
				m_seqs[i]->play( m_seqs[i]->get_queued_tick() - 1, m_playback_mode );
				m_seqs[i]->toggle_playing();
				m_seqs[i]->toggle_queued();
				
			}
			
			m_seqs[i]->play( a_tick, m_playback_mode );
		} 
    }
	
    /* flush the bus */
    m_master_bus.flush();
}

void 
perform::set_orig_ticks( long a_tick  )
{
    for (int i=0; i< c_max_sequence; i++ ){
	
	if ( m_seqs_active[i] == true ){
	    assert( m_seqs[i] );
	    m_seqs[i]->set_orig_tick( a_tick );
	} 
    }
}

void
perform::move_triggers( bool a_direction )
{
    if ( m_left_tick < m_right_tick ){

	long distance = m_right_tick - m_left_tick;

	for (int i=0; i< c_max_sequence; i++ ){

	    if ( m_seqs_active[i] == true ){
		assert( m_seqs[i] );
		m_seqs[i]->move_triggers( m_left_tick, distance, a_direction );
	    } 
	}
    }
}


/* copies between L and R -> R */
void
perform::copy_triggers( )
{
    if ( m_left_tick < m_right_tick ){

	long distance = m_right_tick - m_left_tick;

	for (int i=0; i< c_max_sequence; i++ ){

	    if ( m_seqs_active[i] == true ){
		assert( m_seqs[i] );
		m_seqs[i]->copy_triggers( m_left_tick, distance );
	    } 
	}
    }
}




void 
perform::start( bool a_state )
{
    if ( ! is_running() ){
	set_running( true );

	if ( a_state )
	    off_sequences( );

	launch_output_thread( a_state );
    }
}

void 
perform::off_sequences( void )
{
    for (int i=0; i< c_max_sequence; i++ ){
	
	if ( m_seqs_active[i] == true ){
	    assert( m_seqs[i] );
	    m_seqs[i]->set_playing( false );

	} 
    }
}


void 
perform::stop( )
{
    set_running( false );
    pthread_join( m_out_thread, NULL );
    reset_sequences(  );
}


void 
perform::reset_sequences( void )
{
    for (int i=0; i< c_max_sequence; i++ ){
	
	if ( m_seqs_active[i] == true ){
	    assert( m_seqs[i] );

	    m_seqs[i]->off_playing_notes( );
	    m_seqs[i]->set_playing( false );
	    m_seqs[i]->zero_markers( ); 
	} 
    }
    /* flush the bus */
    m_master_bus.flush();
}

void 
perform::launch_output_thread( bool a_playback_mode )
{
    m_playback_mode = a_playback_mode;

    int err;
    err = pthread_create( &m_out_thread, 
			  NULL, 
			  output_thread_func,
			  this );
}


void 
perform::launch_input_thread()
{
    int err;
    err = pthread_create( &m_in_thread, 
			  NULL, 
			  input_thread_func,
			  this );

}


long 
perform::get_max_trigger( void )
{
    long ret = 0, t;

    for (int i=0; i< c_max_sequence; i++ ){
	
	if ( m_seqs_active[i] == true ){
	    assert( m_seqs[i] );
	    
	    t = m_seqs[i]->get_max_trigger( );  
	    if ( t > ret )
		ret = t;
	} 
    }

    return ret;
}



void*
output_thread_func(void *a_pef )
{
    /* set our performance */
    perform *p = (perform *) a_pef;
    assert(p);

    /* begning time */
    struct timespec last;
    /* current time */
    struct timespec current;

    struct timespec stats_loop_start;
    struct timespec stats_loop_finish;
    

    /* difference between last and current */
    struct timespec delta;

    /* tick and tick fraction */
    long elapsed_tick   = 0;
    long elapsed_tick_f = 0;

    long clock_tick   = 0;
    long clock_tick_f = 0;

    long stats_clock_tick = 0;

    long stats_loop_index = 0;
    long stats_min = 0x7FFFFFFF;
    long stats_max = 0;
    long stats_avg = 0;
    long stats_last_clock_us = 0;
    long stats_clock_width_us = 0;

    long stats_all[100];
    long stats_clock[100];
   

    for( int i=0; i<100; i++ ){
      stats_all[i] = 0;
      stats_clock[i] = 0;
    }

    /* if we are in the performance view, we care 
       about starting from the offset */
    if ( p->m_playback_mode ){

	elapsed_tick = p->m_left_tick;
	p->set_orig_ticks( p->m_left_tick ); 

    }
    int ppqn = p->m_master_bus.get_ppqn();

    p->m_master_bus.start();

    /* get start time position */
    clock_gettime(CLOCK_REALTIME, &last);

    if ( global_stats )
      stats_last_clock_us= (last.tv_sec * 1000000) + (last.tv_nsec / 1000);


    struct sched_param *schp = new sched_param;
    /*
     * set the process to realtime privs
     */
    
    if ( global_priority ){

	memset(schp, 0, sizeof(sched_param));
	schp->sched_priority = 1;

	if (sched_setscheduler(0, SCHED_FIFO, schp) != 0) 	{

	    printf("output_thread_func: couldnt sched_setscheduler(FIFO), you need to be root.\n");
	    pthread_exit(0);
	}
    }

    while( p->m_running ){

	/************************************

	  Get delta time ( current - last )
	  Get delta ticks from time
	  Add to elapsed_ticks
	  Compute prebuffer ticks
	  play from current tick to prebuffer

	**************************************/

        if ( global_stats ){
	    clock_gettime(CLOCK_REALTIME, &stats_loop_start);
        }


	/* delta */
	clock_gettime(CLOCK_REALTIME, &current);
	delta.tv_sec  =  (current.tv_sec  - last.tv_sec  );
	delta.tv_nsec =  (current.tv_nsec - last.tv_nsec );
	long delta_us = (delta.tv_sec * 1000000) + (delta.tv_nsec / 1000);


	/* bpm */
	int bpm  = p->m_master_bus.get_bpm();

	/* get delta ticks, delta_ticks_f is in 1000th of a tick */
	long delta_tick   =  (long) (bpm * ppqn * (delta_us/60000000.0f) ); 
	long delta_tick_f = ((long) (bpm * ppqn * (delta_us/60000.0f) )) % 1000; 

	//printf ( "delta_tick[%ld.%03ld]\n", delta_tick, delta_tick_f  ); 

	/* current ticks */
	elapsed_tick   += delta_tick;
	elapsed_tick_f += delta_tick_f;
	elapsed_tick   += ( elapsed_tick_f / 1000 );
	elapsed_tick_f %= 1000;

	clock_tick   += delta_tick;
	clock_tick_f += delta_tick_f;
	clock_tick   += (clock_tick_f / 1000 );
	clock_tick_f %= 1000;

	if ( p->m_looping ){

	    if ( elapsed_tick >= p->get_right_tick() ){
		
		long leftover_tick = elapsed_tick - (p->get_right_tick());
		
		p->play( p->get_right_tick() - 1 );
		p->reset_sequences();
		
		p->set_orig_ticks( p->get_left_tick() );
		elapsed_tick = p->get_left_tick() + leftover_tick;
	    }
	}
	
	/* play */
	p->play( elapsed_tick ); 

	/* midi clock */
	p->m_master_bus.clock( clock_tick );













	if ( global_stats ){	  

	  while ( stats_clock_tick <= clock_tick ){

	    /* was there a tick ? */
	    if ( stats_clock_tick % (c_ppqn / 24) == 0 ){
	          
	      long current_us = (current.tv_sec * 1000000) + (current.tv_nsec / 1000);
	      stats_clock_width_us = current_us - stats_last_clock_us;
	      stats_last_clock_us = current_us;

	      int index = stats_clock_width_us / 300;
	      if ( index >= 100 ) index = 99; 
	      stats_clock[index]++;

	    }
	    stats_clock_tick++;
	  }

	}




	
	/***********************************
         
          Figure out how much time 
          we need to sleep, and do it

	************************************/

	/* set last */
	last = current;

	clock_gettime(CLOCK_REALTIME, &current);
	delta.tv_sec  =  (current.tv_sec  - last.tv_sec  );
	delta.tv_nsec =  (current.tv_nsec - last.tv_nsec );
	long elapsed_us = (delta.tv_sec * 1000000) + (delta.tv_nsec / 1000);
	//printf( "elapsed_us[%ld]\n", elapsed_us );

	/* now, we want to trigger every c_thread_trigger_width_ms,
	   and it took us delta_us to play() */ 

	delta_us = (c_thread_trigger_width_ms * 1000) - elapsed_us;
	//printf( "sleeping_us[%ld]\n", delta_us );


	/* check midi clock adjustment */

	long next_clock_tick = (clock_tick + (c_ppqn / 24)) - (clock_tick % (c_ppqn / 24)); 

	long next_clock_delta   = (next_clock_tick - clock_tick - 1); 
	long next_clock_delta_f = (1000 - clock_tick_f);

	long next_clock_delta_us =  (( next_clock_delta ) * 60000000 / c_ppqn  / bpm );
	next_clock_delta_us += (( next_clock_delta_f ) * 60000 / c_ppqn  / bpm );
 
	if ( next_clock_delta_us < (c_thread_trigger_width_ms * 1000 * 2) ){
	  delta_us = next_clock_delta_us;
	} 

 
	if ( delta_us > 0 ){

	    delta.tv_sec =  (delta_us / 1000000);
	    delta.tv_nsec = (delta_us % 1000000) * 1000;

	    //printf("sleeping() ");
	    nanosleep( &delta, NULL );
	
	} 

	else {

	  printf ("underrun\n" );
	  //pthread_exit(0);  

	}

	if ( global_stats ){	  
	  clock_gettime(CLOCK_REALTIME, &stats_loop_finish);
	}

	if ( global_stats ){

	  delta.tv_sec  =  (stats_loop_finish.tv_sec  - stats_loop_start.tv_sec  );
	  delta.tv_nsec =  (stats_loop_finish.tv_nsec - stats_loop_start.tv_nsec );
	  long delta_us = (delta.tv_sec * 1000000) + (delta.tv_nsec / 1000);

	  int index = delta_us / 100;
	  if ( index >= 100  ) index = 99;

	  stats_all[index]++;

	  if ( delta_us > stats_max )
	    stats_max = delta_us;
	  if ( delta_us < stats_min )
	    stats_min = delta_us;

	  stats_avg += delta_us;

	  stats_loop_index++;	  

	  if ( stats_loop_index > 200 ){

	    stats_loop_index = 0;
	    stats_avg /= 200;

	    printf( "stats_avg[%d]us stats_min[%d]us stats_max[%d]us\n", stats_avg, stats_min, stats_max );

	    stats_min = 0x7FFFFFFF;
	    stats_max = 0;
	    stats_avg = 0;

	  }
	  
        }
    }

   
    if ( global_stats ){
      
      printf ( "\n\n-- trigger width --\n" );
      for ( int i=0; i<100; i++ ){
	printf( "[%3d][%8d]\n", i * 100, stats_all[i] );
      }
      printf ( "\n\n-- clock width --\n" );
      int bpm  = p->m_master_bus.get_bpm();

      printf ( "optimal : [%ld]us\n", ((c_ppqn / 24)* 60000000 / c_ppqn  / bpm )); 
      
      
      for ( int i=0; i<100; i++ ){
	printf( "[%3d][%8d]\n", i * 300, stats_clock[i] );
      }
      

    }


    p->m_elapsed_tick_copy = elapsed_tick;
    p->m_tick = 0;
    p->m_master_bus.stop();
    pthread_exit(0);
}

void*
input_thread_func(void *a_pef )
{

    struct sched_param *schp = new sched_param;
    /*
     * set the process to realtime privs
     */
	
    if ( global_priority ){
		
		memset(schp, 0, sizeof(sched_param));
		schp->sched_priority = 1;
		
		if (sched_setscheduler(0, SCHED_FIFO, schp) != 0) 	{
			
			printf("input_thread_func: couldnt sched_setscheduler(FIFO), you need to be root.\n");
			pthread_exit(0);
		}
    }
	
    /* set our performance */
    perform *p = (perform *) a_pef;
    assert(p);
	
    event ev;
	
    while( p->m_inputing ){
		
		if ( p->m_master_bus.poll_for_midi() > 0 ){
			
			do {
				
				if ( p->m_master_bus.get_midi_event( &ev ) ){
				
					/* filter system wide messages */
					if ( ev.get_status() <= EVENT_SYSEX ){  
						
						if( global_showmidi) 
							ev.print();
						
						/* is there a sequence set ? */
						if ( p->m_master_bus.is_dumping( )) {
							
							/* dump to it */
							(p->m_master_bus.get_sequence( ))->stream_event( &ev );
							
						}
						
						/* use it to control our sequencer */
						else {
							
							for ( int i=0; i<c_seqs_in_set; i++ ){
								
								unsigned char data[2] = {0,0};
								unsigned char status = ev.get_status();
								
								ev.get_data( &data[0], &data[1] );
								
								if(  p->get_midi_control_toggle(i)->m_active &&
									 status  == p->get_midi_control_toggle(i)->m_status &&
									 data[0] == p->get_midi_control_toggle(i)->m_data ){

									if ( data[1] >= p->get_midi_control_toggle(i)->m_min_value &&
										 data[1] <= p->get_midi_control_toggle(i)->m_max_value ){
										p->sequence_playing_toggle( i + p->m_offset );
									}
								}
								
								if ( p->get_midi_control_on(i)->m_active && 
									 status  == p->get_midi_control_on(i)->m_status &&
									 data[0] == p->get_midi_control_on(i)->m_data ){
									
									if ( data[1] >= p->get_midi_control_on(i)->m_min_value &&
										 data[1] <= p->get_midi_control_on(i)->m_max_value ){
										p->sequence_playing_on( i  + p->m_offset);
									} else if (  p->get_midi_control_on(i)->m_inverse_active ){
										p->sequence_playing_off(  i + p->m_offset );
									}
									
								}
								
								if ( p->get_midi_control_off(i)->m_active &&
									 status  == p->get_midi_control_off(i)->m_status &&
									 data[0] == p->get_midi_control_off(i)->m_data ){
									
									if ( data[1] >= p->get_midi_control_off(i)->m_min_value &&
										 data[1] <= p->get_midi_control_off(i)->m_max_value ){
										p->sequence_playing_off(  i + p->m_offset );
									} else if ( p->get_midi_control_off(i)->m_inverse_active ){
										p->sequence_playing_on(  i + p->m_offset );
									}
									
								}
								
							}
						}
						
					}
				
					if ( ev.get_status() == EVENT_SYSEX ){  
						
						if( global_showmidi ) 
							ev.print();
						
						p->m_master_bus.sysex( &ev );   
					}
				}
				
			} while ( p->m_master_bus.is_more_input( ));
		}
    }
    pthread_exit(0);
	
}






