/*
 * Sentinella
 * http://sourceforge.net/projects/sentinella/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "MainWindow.h"
#include <errno.h>
#include <libsysactivity/libsysactivity.h>
#include <QFile>
#include <QDBusInterface>
#include <QDBusReply>
#include <QNetworkInterface>
#include <QSetIterator>
#include <QTreeView>
#include <kmenu.h>
#include <kaction.h>
#include <KUrl>
#include <kfiledialog.h>
#include <kstandardaction.h>
#include <kactioncollection.h>
#include <KConfig>
#include <KSharedConfig>
#include <kpassivepopup.h>
#include <kdialog.h>
#include <kaboutapplicationdialog.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ksysguard/ksysguardprocesslist.h>
#include <kiconloader.h>
#include "Conditions/CPU.h"
#include "Conditions/DateTime.h"
#include "Conditions/Memory.h"
#include "Conditions/Network.h"
#include "Conditions/ProcessDies.h"
#include "Actions/ExecuteCommand.h"
#include "Actions/KillProcess.h"
#include "Actions/PlayAlarm.h"
#include "Actions/Shutdown.h"
#include "Actions/Sleep.h"

// Ids of the condition radiobuttons
#define RADIOBUTTONCPU 0
#define RADIOBUTTONMEMORY 1
#define RADIOBUTTONNETWORK 2
#define RADIOBUTTONDATETIME 3
#define RADIOBUTTONPROCESSDIES 4

// Ids of the action radiobuttons (shutdown and reboot are the same because they use the same QStackedWiget page)
#define RADIOBUTTONSHUTDOWN 0
#define RADIOBUTTONREBOOT 0
#define RADIOBUTTONSLEEP 1
#define RADIOBUTTONEXECUTECOMMAND 2
#define RADIOBUTTONPLAYALARM 3
#define RADIOBUTTONKILLPROCESS 4

MainWindow::MainWindow(const KAboutData& aboutData, KCmdLineArgs* cmdLineArgs,
		const short cmdSelectedCondition, const short cmdSelectedAction) :
	KMainWindow(NULL, Qt::WindowContextHelpButtonHint|Qt::WindowCloseButtonHint), aboutData(aboutData), cmdLineArgs(cmdLineArgs) {
	condition = NULL;
	action = NULL;
	supportedShutdown = false;
	pidProcessDies = pidKillProcess = 0;
	availableSounds = NULL;

	checkShutdownSleepStates();
	config = KGlobal::config();
	iconLoader = KIconLoader::global();

	gui.setupUi(this);
	gui.pushButtonStartStop->setIcon(QIcon(iconLoader->loadIcon("arrow-right", KIconLoader::Small)));

	trayIcon.setAssociatedWidget(this);
	trayIcon.setCategory(KStatusNotifierItem::ApplicationStatus);
	trayIcon.setStatus(KStatusNotifierItem::Active);
	trayIcon.contextMenu()->addAction(KStandardAction::aboutApp(this, SLOT(showAboutApp()),
			trayIcon.contextMenu()));
	trayIcon.setIconByName("sentinella");

	conditionsGroup.addButton(gui.radioButtonCPU, RADIOBUTTONCPU);
	conditionsGroup.addButton(gui.radioButtonMemory, RADIOBUTTONMEMORY);
	conditionsGroup.addButton(gui.radioButtonNetwork, RADIOBUTTONNETWORK);
	conditionsGroup.addButton(gui.radioButtonDateTime, RADIOBUTTONDATETIME);
	conditionsGroup.addButton(gui.radioButtonProcessDies, RADIOBUTTONPROCESSDIES);
	connect(gui.radioButtonNetwork, SIGNAL(toggled(bool)), this, SLOT(radioButtonNetworkToggled(bool)));
	connect(&conditionsGroup, SIGNAL(buttonClicked(int)), gui.stackedWidgetConditions, SLOT(setCurrentIndex(int)));
	connect(gui.pushButtonProcessDies, SIGNAL(clicked(bool)), this, SLOT(programDiesClicked()));

	if (supportedShutdown) {
		actionsGroup.addButton(gui.radioButtonShutdown, RADIOBUTTONSHUTDOWN);
		actionsGroup.addButton(gui.radioButtonReboot, RADIOBUTTONREBOOT);
	} else {
		gui.radioButtonShutdown->setEnabled(false);
		gui.radioButtonReboot->setEnabled(false);
	}
	if (supportedSleepStates.empty())
		gui.radioButtonSleep->setEnabled(false);
	else
		actionsGroup.addButton(gui.radioButtonSleep, RADIOBUTTONSLEEP);
	actionsGroup.addButton(gui.radioButtonExecuteCommand, RADIOBUTTONEXECUTECOMMAND);
	actionsGroup.addButton(gui.radioButtonPlayAlarm, RADIOBUTTONPLAYALARM);
	actionsGroup.addButton(gui.radioButtonKillProcess, RADIOBUTTONKILLPROCESS);
	connect(gui.radioButtonPlayAlarm, SIGNAL(toggled(bool)), this, SLOT(radioButtonPlayAlarmToggled(bool)));
	connect(&actionsGroup, SIGNAL(buttonClicked(int)), gui.stackedWidgetActions, SLOT(setCurrentIndex(int)));
	connect(gui.pushButtonKillProcess, SIGNAL(clicked(bool)), this, SLOT(killProcessClicked()));

	connect(gui.pushButtonStartStop, SIGNAL(clicked(bool)), this, SLOT(startStopClicked()));

	initConditionWidgets(cmdSelectedCondition);
	initActionWidgets(cmdSelectedAction);
	if (cmdLineArgs->isSet("start"))
		startClicked();

	cmdLineArgs->clear(); // the given cmd args are not needed anymore
}

MainWindow::~MainWindow() {
	delete availableSounds;
}

void MainWindow::checkShutdownSleepStates() {
	//see http://majewsky.wordpress.com/2009/07/11/shutdown-your-machine-automatically-or-from-remote/
	QDBusInterface* interface = new QDBusInterface("org.kde.ksmserver", "/KSMServer",
			"org.kde.KSMServerInterface");
	QDBusReply<bool> boolReply = interface->call("canShutdown");
	if (boolReply.isValid()) {
		supportedShutdown = boolReply;
		isKDE = true;
		delete interface;

		interface = new QDBusInterface("org.kde.kded", "/modules/powerdevil", "org.kde.PowerDevil");
		QDBusReply<QVariantMap> mapReply = interface->call("getSupportedSuspendMethods");
		if (mapReply.isValid()) {
			QList<QVariant> values = mapReply.value().values();
			for (int i = 0; i < values.size(); i++)
				supportedSleepStates.insert(
						(Solid::PowerManagement::SleepState) values.at(i).toInt());
		}
	} else {
		//see http://upower.freedesktop.org/docs/UPower.html
		delete interface;
		interface = new QDBusInterface("org.freedesktop.ConsoleKit",
				"/org/freedesktop/ConsoleKit/Manager", "org.freedesktop.ConsoleKit.Manager",
				QDBusConnection::systemBus());
		boolReply = interface->call("CanStop");
		if (boolReply.isValid()) {
			supportedShutdown = boolReply;
			isKDE = false;
			delete interface;

			interface = new QDBusInterface("org.freedesktop.UPower", "/org/freedesktop/UPower",
					"org.freedesktop.UPower", QDBusConnection::systemBus());
			boolReply = interface->call("SuspendAllowed");
			if (boolReply.isValid() && boolReply)
				supportedSleepStates.insert(Solid::PowerManagement::SuspendState);
			boolReply = interface->call("HibernateAllowed");
			if (boolReply.isValid() && boolReply)
				supportedSleepStates.insert(Solid::PowerManagement::HibernateState);
		}
	}

	delete interface;
}

void MainWindow::initConditionWidgets(const short cmdSelectedCondition) {
	int aux;
	if (cmdSelectedCondition != -1)
		aux = cmdSelectedCondition;
	else {
		KConfigGroup group = config->group("Conditions");
		aux = group.readEntry("LastExecuted", 0);
		if (aux < 0 || aux > 4)
			aux = 0;
	}

	conditionsGroup.button(aux)->setChecked(true);
	gui.stackedWidgetConditions->setCurrentIndex(aux);

	if (cmdLineArgs->isSet("higher"))
		aux = 0;
	else if (cmdLineArgs->isSet("lower"))
		aux = 1;
	else
		aux = -1;

	KConfigGroup groupCPU = config->group("ConditionCPU");
	gui.comboBoxCPU->setCurrentIndex(aux != -1 ? aux : groupCPU.readEntry("Higher", 1));
	gui.spinBoxCPU->setValue(cmdSelectedCondition == 0 ? cmdLineArgs->getOption("cpu").toInt()
			: groupCPU.readEntry("Percentage", 10));
	gui.timeEditCPU->setTime(QTime::fromString(cmdSelectedCondition == 0 && cmdLineArgs->isSet(
			"during") ? cmdLineArgs->getOption("during") : groupCPU.readEntry("Time", "01:00"),
			"mm:ss"));

	KConfigGroup groupMemory = config->group("ConditionMemory");
	gui.comboBoxMemory->setCurrentIndex(aux != -1 ? aux : groupMemory.readEntry("Higher", 1));
	gui.spinBoxMemory->setValue(
			cmdSelectedCondition == 1 ? cmdLineArgs->getOption("memory").toInt()
					: groupMemory.readEntry("Amount", 512));
	gui.timeEditMemory->setTime(QTime::fromString(cmdSelectedCondition == 1 && cmdLineArgs->isSet(
			"during") ? cmdLineArgs->getOption("during") : groupMemory.readEntry("Time", "01:00"),
			"mm:ss"));

	KConfigGroup groupNetwork = config->group("ConditionNetwork");
	gui.comboBoxNetworkDirection->setCurrentIndex(groupNetwork.readEntry("Direction", 0));
	gui.comboBoxNetworkHigher->setCurrentIndex(aux != -1 ? aux
			: groupNetwork.readEntry("Higher", 0));
	gui.spinBoxNetwork->setValue(
			cmdSelectedCondition == 2 ? cmdLineArgs->getOption("network").toInt()
					: groupNetwork.readEntry("Amount", 10));
	gui.timeEditNetwork->setTime(QTime::fromString(cmdSelectedCondition == 2 && cmdLineArgs->isSet(
			"during") ? cmdLineArgs->getOption("during") : groupNetwork.readEntry("Time", "01:00"),
			"mm:ss"));

	QDateTime dateTime;
	if (cmdSelectedCondition == 3)
		dateTime = QDateTime::fromString(cmdLineArgs->getOption("date-time"), Qt::ISODate);
	if (cmdSelectedCondition != 3 || !dateTime.isValid()) {
		KConfigGroup groupDateTime = config->group("ConditionDateTime");
		dateTime = groupDateTime.readEntry("DateTime", QDateTime::currentDateTime());
	}
	QTime readTime(dateTime.time().hour(), dateTime.time().minute(), 0); // Setting 0 seconds
	dateTime.setTime(readTime);
	gui.kdatetimewidgetDateTime->setDateTime(dateTime);

	if (cmdSelectedCondition == 4) {
#ifdef SA_OPEN_PROCESS
		if (sa_open_process() != 0) {
#ifdef SA_CLOSE_PROCESS
			sa_close_process();
#endif
			return;
		}
#endif
		struct sa_process procInfo;
		if (sa_get_process(cmdLineArgs->getOption("program-dies").toInt(), &procInfo) == 0) {
			pidProcessDies = procInfo.pid;
			gui.labelProcessDies->setText(QString::fromAscii(procInfo.filename));
		}
#ifdef SA_CLOSE_PROCESS
		sa_close_process();
#endif
	}
}

void MainWindow::initActionWidgets(const short cmdSelectedAction) {
	int chosen;
	if (cmdSelectedAction != -1)
		chosen = cmdSelectedAction;
	else {
		KConfigGroup group = config->group("Actions");
		chosen = group.readEntry("LastExecuted", 0);
		if (chosen < 0 || chosen > 5)
			chosen = 0;
	}

	if (((chosen == 0 || chosen == 1) && !supportedShutdown) || (chosen == 2
			&& supportedSleepStates.empty()))
		chosen = 3;
	switch (chosen) {
	case 0:
		gui.radioButtonShutdown->setChecked(true);
		break;
	case 1:
		gui.radioButtonReboot->setChecked(true);
		break;
	default:
		actionsGroup.button(chosen - 1)->setChecked(true);
	}
	gui.stackedWidgetActions->setCurrentIndex(chosen - 1);

	if (!isKDE) {
		gui.checkBoxShutdownForce->setChecked(false);
		gui.checkBoxShutdownForce->setEnabled(false);
	} else if (cmdSelectedAction == 0 || cmdSelectedAction == 1) {
		gui.checkBoxShutdownForce->setChecked(cmdLineArgs->getOption(
				cmdSelectedAction == 0 ? "shutdown" : "reboot") == "yes");
	} else {
		KConfigGroup groupShutdown = config->group("ActionShutdown");
		gui.checkBoxShutdownForce->setChecked(groupShutdown.readEntry("Force", true));
	}

	if (!supportedSleepStates.empty()) {
		gui.radioButtonSleepStandBy->setEnabled(supportedSleepStates.contains(
				Solid::PowerManagement::StandbyState));
		gui.radioButtonSleepSuspend->setEnabled(supportedSleepStates.contains(
				Solid::PowerManagement::SuspendState));
		gui.radioButtonSleepHibernate->setEnabled(supportedSleepStates.contains(
				Solid::PowerManagement::HibernateState));

		Solid::PowerManagement::SleepState state;
		if (cmdSelectedAction == 2) { // cmd arg
			QString cmdSleepArg = cmdLineArgs->getOption("sleep");
			if (cmdSleepArg == "Standby")
				state = Solid::PowerManagement::StandbyState;
			else if (cmdSleepArg == "Suspend")
				state = Solid::PowerManagement::SuspendState;
			else
				state = Solid::PowerManagement::HibernateState;
		} else { // config
			KConfigGroup groupSleep = config->group("ActionSleep");
			state = (Solid::PowerManagement::SleepState) groupSleep.readEntry("Mode",
					(int) Solid::PowerManagement::HibernateState);
		}

		if (!supportedSleepStates.contains(state)) { // Just in case the mode read from the config or cmd arg is unsupported
			QSetIterator<Solid::PowerManagement::SleepState> i(supportedSleepStates);
			state = i.next();
		}

		switch (state) {
		case Solid::PowerManagement::StandbyState:
			gui.radioButtonSleepStandBy->setChecked(true);
			break;
		case Solid::PowerManagement::SuspendState:
			gui.radioButtonSleepSuspend->setChecked(true);
			break;
		default:
			gui.radioButtonSleepHibernate->setChecked(true);
		}
	}

	if (cmdSelectedAction == 5) {
#ifdef SA_OPEN_PROCESS
		if (sa_open_process() != 0) {
#ifdef SA_CLOSE_PROCESS
			sa_close_process();
#endif
			return;
		}
#endif
		struct sa_process procInfo;
		if (sa_get_process(cmdLineArgs->getOption("terminate-program").toInt(), &procInfo) == 0) {
			pidKillProcess = procInfo.pid;
			gui.labelKillProcess->setText(QString::fromAscii(procInfo.filename));
		}
#ifdef SA_CLOSE_PROCESS
		sa_close_process();
#endif
	}

	KConfigGroup groupExecuteCommand = config->group("ActionExecuteCommand");
	gui.texteditExecuteCommand->setPlainText(groupExecuteCommand.readEntry("Command"));
}

void MainWindow::radioButtonNetworkToggled(bool checked) {
	if (!checked)
		return;

	// Populating the combobox
	gui.comboBoxNetworkInterface->clear();
	QList<QNetworkInterface> interfaces = QNetworkInterface::allInterfaces();
	int i;
	for (i = 0; i < interfaces.size(); i++) {
		const QNetworkInterface interface = interfaces[i];
		if (interface.flags() & QNetworkInterface::IsUp)
			gui.comboBoxNetworkInterface->addItem(interface.name());
	}
	// Loading config for the combobox (just the first time)
	static bool doneOnce = false;
	if (!doneOnce) {
		QString name;
		if (cmdLineArgs->isSet("interface"))
			name = cmdLineArgs->getOption("interface");
		else {
			KConfigGroup groupNetwork = config->group("ConditionNetwork");
#ifdef Q_OS_FREEBSD
			name = groupNetwork.readEntry("Interface", "em0");
#else
			name = groupNetwork.readEntry("Interface", "eth0");
#endif
		}
		i = gui.comboBoxNetworkInterface->findText(name);
		if (i != -1)
			gui.comboBoxNetworkInterface->setCurrentIndex(i);

		doneOnce = true;
	}
}

void MainWindow::radioButtonPlayAlarmToggled(bool checked) {
	if (!checked)
		return;

	disconnect(gui.radioButtonPlayAlarm, SIGNAL(toggled(bool)), this, SLOT(radioButtonPlayAlarmToggled(bool)));
	availableSounds = new QStringList(KGlobal::dirs()->findAllResources("sound", "Sentinella/*",
			KStandardDirs::NoDuplicates));
	availableSounds->sort();
	// Loading config for the combobox
	KConfigGroup groupPlayAlarm = config->group("ActionPlayAlarm");
	QString savedFileName = groupPlayAlarm.readEntry("File");
	int savedIndex = -1;
	// Populating the combobox
	for (int i = 0; i < availableSounds->size(); i++) {
		if (savedIndex == -1 && savedFileName.compare(availableSounds->at(i)) == 0)
			savedIndex = i;
		QFileInfo fileInfo = QFileInfo(availableSounds->at(i));
		gui.comboBoxPlayAlarm->addItem(fileInfo.baseName());
	}

	if (savedIndex != -1)
		gui.comboBoxPlayAlarm->setCurrentIndex(savedIndex);
}

void MainWindow::chooseProcess(bool isAction) {
	KDialog* procsDialog = new KDialog(this);
	procsDialog->setCaption(i18nc("@title:window", "Choose a program"));
	procsDialog->setButtons(KDialog::Ok | KDialog::Cancel);

	KSysGuardProcessList* processList = new KSysGuardProcessList(procsDialog);
	processList->setUpdateIntervalMSecs(1500);
	processList->setKillButtonVisible(false);
	if (isAction)
		processList->setState(ProcessFilter::OwnProcesses);
	processList->treeView()->setSelectionMode(QAbstractItemView::SingleSelection);
	procsDialog->setMainWidget(processList);

	if (procsDialog->exec() == QDialog::Accepted) {
		QList<KSysGuard::Process*> selectedList = processList->selectedProcesses();
		if (!selectedList.isEmpty()) {
			KSysGuard::Process* selected = selectedList.at(0);
			int index = selected->name.indexOf(' ');
			if (isAction) {
				pidKillProcess = selected->pid;
				gui.labelKillProcess->setText(index == -1 ? selected->name : selected->name.left(
						index));
			} else {
				pidProcessDies = selected->pid;
				gui.labelProcessDies->setText(index == -1 ? selected->name : selected->name.left(
						index));
			}
		}
	}

	delete processList;
	delete procsDialog;
}

void MainWindow::programDiesClicked() {
	chooseProcess(false);
}

void MainWindow::killProcessClicked() {
	chooseProcess(true);
}

void MainWindow::startStopClicked() {
	if (condition == NULL)
		startClicked();
	else
		stopClicked();
}

void MainWindow::startClicked() {
	// validating actions
	int actionCheckedId = actionsGroup.checkedId();
	switch (actionCheckedId) {
	case RADIOBUTTONSLEEP:
		if (!gui.radioButtonSleepStandBy->isChecked() && !gui.radioButtonSleepSuspend->isChecked()
				&& !gui.radioButtonSleepHibernate->isChecked()) {
			KMessageBox::sorry(this, i18nc(
					"@info The user has to choose between hibernation, standby and Suspend modes",
					"You must select sleep mode."));
			return;
		}
		break;
	case RADIOBUTTONEXECUTECOMMAND:
		if (gui.texteditExecuteCommand->toPlainText().isEmpty()) {
			KMessageBox::sorry(this, i18n("A command must be specified."));
			return;
		}
		break;
	case RADIOBUTTONKILLPROCESS:
		if (pidKillProcess == 0) {
			KMessageBox::sorry(this, i18n("You must select a program to kill."));
			return;
		}
	}

	// validating conditions
	int seconds = 0;
	int conditionCheckedId = conditionsGroup.checkedId();
	if (conditionCheckedId == RADIOBUTTONCPU || conditionCheckedId == RADIOBUTTONMEMORY
			|| conditionCheckedId == RADIOBUTTONNETWORK) {
		QTime time;
		switch (conditionCheckedId) {
		case RADIOBUTTONCPU:
			time = gui.timeEditCPU->time();
			break;
		case RADIOBUTTONMEMORY:
			time = gui.timeEditMemory->time();
			break;
		case RADIOBUTTONNETWORK:
			time = gui.timeEditNetwork->time();
			break;
		}

		seconds = time.minute() * 60 + time.second();
		if (seconds == 0) {
			KMessageBox::sorry(this, i18nc("@info The user has chosen a wrong time: 00:00",
					"The specified time must be greater than zero."));
			return;
		}
	}
	// TODO Check if the chosen processes to watch for still exists

	switch (conditionCheckedId) {
	case RADIOBUTTONCPU:
		condition = new CPU(seconds, 2, gui.spinBoxCPU->value(), gui.comboBoxCPU->currentIndex()
				== 0);
		break;
	case RADIOBUTTONMEMORY:
		condition = new Memory(seconds, 4, gui.spinBoxMemory->value() * 1048576,
				gui.comboBoxMemory->currentIndex() == 0);
		break;
	case RADIOBUTTONNETWORK:
		condition = new Network(seconds, 2, gui.comboBoxNetworkInterface->currentText(),
				gui.spinBoxNetwork->value() * 1024, gui.comboBoxNetworkHigher->currentIndex() == 0,
				gui.comboBoxNetworkDirection->currentIndex() == 0 ? DOWNLOAD : UPLOAD);
		break;
	case RADIOBUTTONDATETIME:
		seconds = QDateTime::currentDateTime().secsTo(gui.kdatetimewidgetDateTime->dateTime());
		if (seconds <= 0) {
			KMessageBox::sorry(this, i18nc("@info",
					"You have selected a date in the past. Please input a future date."));
			return;
		}
		condition = new DateTime(seconds);
		break;
	case RADIOBUTTONPROCESSDIES:
		if (pidProcessDies == 0) {
			KMessageBox::sorry(this, i18nc("@info", "You must select a program to monitor."));
			return;
		}
#ifdef SA_OPEN_PROCESS
		if (sa_open_process() != 0) {
			showError(ENOSYS);
			return;
		}
#endif
#if SA_VERSION_MAJOR > 0 || (SA_VERSION_MAJOR == 0 && SA_VERSION_SMALL >= 6)
		struct sa_process_activity dst;
		int ret = sa_get_process_activity(pidProcessDies, &dst);
		if (ret == 0)
			ret = sa_get_process_activity(pidKillProcess, &dst);
#else
		struct sa_process dst;
		int ret = sa_get_process(pidProcessDies, &dst);
		if (ret == 0)
			ret = sa_get_process(pidKillProcess, &dst);
#endif
#ifdef SA_CLOSE_PROCESS
		if (sa_close_process() != 0) {
			showError(ENOSYS);
			return;
		}
#endif
		if (ret != 0) {
			showError(ret);
			return;
		}
		condition = new ProcessDies(2, 2, pidProcessDies);
	}

	connect(condition, SIGNAL(finished()), this, SLOT(conditionFinished()));
	condition->start(QThread::HighPriority);
	gui.groupBoxConditions->setEnabled(false);
	gui.groupBoxActions->setEnabled(false);
	gui.pushButtonStartStop->setText(i18nc("@action:button Stop the running task", "Stop"));
	gui.pushButtonStartStop->setIcon(
			QIcon(iconLoader->loadIcon("process-stop", KIconLoader::Small)));
	saveConfig(conditionCheckedId, actionCheckedId);
}

inline void MainWindow::saveConfig(int conditionCheckedId, int actionCheckedId) {
	KConfigGroup group = config->group("Conditions");
	group.writeEntry("LastExecuted", conditionCheckedId);
	KConfigGroup configGroup;
	switch (conditionCheckedId) {
	case RADIOBUTTONCPU:
		configGroup = config->group("ConditionCPU");
		configGroup.writeEntry("Higher", gui.comboBoxCPU->currentIndex());
		configGroup.writeEntry("Percentage", gui.spinBoxCPU->value());
		configGroup.writeEntry("Time", gui.timeEditCPU->time().toString("mm:ss"));
		break;
	case RADIOBUTTONMEMORY:
		configGroup = config->group("ConditionMemory");
		configGroup.writeEntry("Higher", gui.comboBoxMemory->currentIndex());
		configGroup.writeEntry("Amount", gui.spinBoxMemory->value());
		configGroup.writeEntry("Time", gui.timeEditMemory->time().toString("mm:ss"));
		break;
	case RADIOBUTTONNETWORK:
		configGroup = config->group("ConditionNetwork");
		configGroup.writeEntry("Direction", gui.comboBoxNetworkDirection->currentIndex());
		configGroup.writeEntry("Higher", gui.comboBoxNetworkHigher->currentIndex());
		configGroup.writeEntry("Amount", gui.spinBoxNetwork->value());
		configGroup.writeEntry("Interface", gui.comboBoxNetworkInterface->currentText());
		configGroup.writeEntry("Time", gui.timeEditNetwork->time().toString("mm:ss"));
		break;
	case RADIOBUTTONDATETIME:
		configGroup = config->group("ConditionDateTime");
		configGroup.writeEntry("DateTime", gui.kdatetimewidgetDateTime->dateTime());
		break;
	}

	group = config->group("Actions");
	group.writeEntry("LastExecuted", gui.radioButtonShutdown->isChecked() ? 0 : actionCheckedId + 1);
	switch (actionCheckedId) {
	case RADIOBUTTONSHUTDOWN:
		configGroup = config->group("ActionShutdown");
		configGroup.writeEntry("Force", gui.checkBoxShutdownForce->isChecked());
		break;
	case RADIOBUTTONSLEEP:
		Solid::PowerManagement::SleepState state;
		configGroup = config->group("ActionSleep");
		if (gui.radioButtonSleepStandBy->isChecked())
			state = Solid::PowerManagement::StandbyState;
		else if (gui.radioButtonSleepSuspend->isChecked())
			state = Solid::PowerManagement::SuspendState;
		else
			state = Solid::PowerManagement::HibernateState;
		configGroup.writeEntry("Mode", (int) state);
		break;
	case RADIOBUTTONEXECUTECOMMAND:
		configGroup = config->group("ActionExecuteCommand");
		configGroup.writeEntry("Command", gui.texteditExecuteCommand->toPlainText());
		break;
	case RADIOBUTTONPLAYALARM:
		configGroup = config->group("ActionPlayAlarm");
		configGroup.writeEntry("File", availableSounds->at(gui.comboBoxPlayAlarm->currentIndex()));
	}
	config->sync();
}

void MainWindow::stopClicked() {
	disconnect(condition, SIGNAL(finished()), this, SLOT(conditionFinished()));
	connect(condition, SIGNAL(finished()), condition, SLOT(deleteLater()));
	condition->aborted = 1;
	condition = NULL;

	gui.groupBoxConditions->setEnabled(true);
	gui.groupBoxActions->setEnabled(true);
	gui.pushButtonStartStop->setText(i18nc("@action:button Verb in imperative form", "Start"));
	gui.pushButtonStartStop->setIcon(QIcon(iconLoader->loadIcon("arrow-right", KIconLoader::Small)));
}

void MainWindow::conditionFinished() {
#define PASSIVE_POPUP_TITLE i18nc("@info The name of the app in the title of a passive popup", "Sentinella")
	int error = condition->error;
	delete condition;
	condition = NULL;

	if (error != 0) {
		showError(error);
	} else {
		int actionCheckedId = actionsGroup.checkedId();
		switch (actionCheckedId) {
		case RADIOBUTTONSHUTDOWN:
			action = new Shutdown(isKDE, gui.radioButtonShutdown->isChecked(),
					gui.checkBoxShutdownForce->isChecked());
			if (gui.radioButtonShutdown->isChecked())
				KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
						"Condition met. Halting the system."), this);
			else
				KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
						"Condition met. Rebooting the system."), this);
			break;
		case RADIOBUTTONSLEEP:
			Solid::PowerManagement::SleepState state;
			if (gui.radioButtonSleepStandBy->isChecked()) {
				state = Solid::PowerManagement::StandbyState;
				KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
						"Condition met. Setting the system's state to standby."), this);
			} else if (gui.radioButtonSleepSuspend->isChecked()) {
				state = Solid::PowerManagement::SuspendState;
				KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
						"Condition met. Suspending the system."), this);
			} else {
				state = Solid::PowerManagement::HibernateState;
				KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
						"Condition met. Hibernating the system."), this);
			}
			action = new Sleep(isKDE, state);
			break;
		case RADIOBUTTONEXECUTECOMMAND:
			action = new ExecuteCommand(gui.texteditExecuteCommand->toPlainText());
			KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
					"Condition met. Executing command."), this);
			break;
		case RADIOBUTTONPLAYALARM:
			action = new PlayAlarm(availableSounds->at(gui.comboBoxPlayAlarm->currentIndex()));
			KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
					"Condition met. Throwing alarm."), this);
			break;
		case RADIOBUTTONKILLPROCESS:
			action = new KillProcess(pidKillProcess);
			KPassivePopup::message(PASSIVE_POPUP_TITLE, i18nc("@info",
					"Condition met. Killing the selected program."), this);
			break;
		}
		action->execute();

		if (actionCheckedId == RADIOBUTTONPLAYALARM) {
			KMessageBox::information(this, i18n("Close this message to turn off the alarm."));
			((PlayAlarm*) action)->stop();
			delete action;
			action = NULL;
		}
	}

	gui.groupBoxConditions->setEnabled(true);
	gui.groupBoxActions->setEnabled(true);
	gui.pushButtonStartStop->setText(i18nc("@action:button Verb in imperative form", "Start"));
	gui.pushButtonStartStop->setIcon(QIcon(iconLoader->loadIcon("arrow-right", KIconLoader::Small)));
}

void MainWindow::showError(const int error) {
	switch (error) {
	case ENOSYS:
		KMessageBox::error(this, i18n("Feature not available."));
		break;
	case ENODEV:
		KMessageBox::error(this, i18n("No such device."));
		break;
	case ENOMEM:
		KMessageBox::error(this, i18n("Not enough memory."));
		break;
	case ESRCH:
		KMessageBox::error(this, i18n("No such process."));
		break;
	case EIO:
		KMessageBox::error(this, i18n("I/O error."));
		break;
	default:
		KMessageBox::error(this, i18n("An internal error occurred."));
	}
}

void MainWindow::showAboutApp() {
	KAboutApplicationDialog* aboutAppDialog = new KAboutApplicationDialog(&aboutData, this);
	aboutAppDialog->exec();
	delete aboutAppDialog;
}

void MainWindow::closeEvent(QCloseEvent *event) {
	if (event->spontaneous()) {
		setVisible(false);
		event->ignore();
	} else {
		event->accept();
	}
}

#include "MainWindow.moc"
