#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h> 
#include <sys/stat.h>

#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/string.h"

#include "texturelistio.h"
#include "sar.h"
#include "config.h"


void SARTextureListDeleteAll(
	sar_texture_name_struct ***tl, int *total
);
int SARTextureListLoadFromFile(
	const char *filename,
	sar_texture_name_struct ***tl, int *total
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Deletes textures list.
 */
void SARTextureListDeleteAll(
	sar_texture_name_struct ***tl, int *total
)
{
	int i;
	sar_texture_name_struct *tn_ptr;

	if((tl == NULL) || (total == NULL))
	    return;

	for(i = 0; i < (*total); i++)
	{
	    tn_ptr = (*tl)[i];
	    if(tn_ptr == NULL)
		continue;

	    free(tn_ptr->name);
	    free(tn_ptr->filename);
	    free(tn_ptr);
	}

	if(*tl != NULL)
	{
	    free(*tl);
	    *tl = NULL;
	}
	*total = 0;
}


/*
 *	Loads textures list from file.
 *
 *	Given pointer array should be set to valid values. It will be
 *	deallocated first before the file is loaded.
 */
int SARTextureListLoadFromFile(
	const char *filename,
	sar_texture_name_struct ***tl, int *total
)
{
	int i;
        FILE *fp;
        char *buf = NULL;
        struct stat stat_buf;
	double value[10];
	sar_texture_name_struct *tn_ptr = NULL;

	if((filename == NULL) || (tl == NULL) || (total == NULL))
	    return(-1);

	/* Deallocate any existing texture items in the list. */
	SARTextureListDeleteAll(tl, total);

        if(stat(filename, &stat_buf))
        {
            fprintf(stderr, "%s: No such file.\n", filename);
            return(-1);
        }
#ifdef S_ISREG
        if(!S_ISREG(stat_buf.st_mode))
        {
            fprintf(stderr,
                "%s: Not a file.\n",
                filename
            );
            return(-1);
        }
#endif	/* S_ISREG */

	/* Open file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
        {
            fprintf(stderr, "%s: Cannot open.\n", filename);
            return(-1);
        }

        do
        {
            buf = FSeekNextParm(
                fp,
                buf,
                SAR_COMMENT_CHAR,
                SAR_CFG_DELIM_CHAR
            );
            if(buf == NULL)
                break;

            if(!strcasecmp(buf, "Version"))
            {
                FGetValuesF(fp, value, 2);

            }

	    else if(!strcasecmp(buf, "TextureAdd"))
            {
                char *strptr = FGetString(fp);

		tn_ptr = NULL;


		i = *total;
		*total = i + 1;
		*tl = (sar_texture_name_struct **)realloc(
		    *tl,
		    (*total) * sizeof(sar_texture_name_struct *)
		);
		if(*tl == NULL)
		{
		    *total = 0;
		}
		else
		{
		    (*tl)[i] = tn_ptr = (sar_texture_name_struct *)calloc(
			1, sizeof(sar_texture_name_struct)
		    );
		}


		/* Set reference name. */
		if(tn_ptr != NULL)
		{
                    StringStripSpaces(strptr);
		    free(tn_ptr->name);
		    tn_ptr->name = strptr;
		    strptr = NULL;
		}
		else
		{
		    free(strptr);
		    strptr = NULL;
		}
	    }
            else if(!strcasecmp(buf, "TextureFileName"))
            {
		char *strptr = FGetString(fp);

                if(tn_ptr != NULL)
		{
		    StringStripSpaces(strptr);
		    free(tn_ptr->filename);
                    tn_ptr->filename = strptr;
		    strptr = NULL;
		}
                else
		{
                    free(strptr);
		    strptr = NULL;
		}
            }
            else if(!strcasecmp(buf, "TextureFlags"))
            {
                FGetValuesF(fp, value, 1);

		/* Reserved. */
	    }
            else if(!strcasecmp(buf, "TexturePriority"))
            {
                FGetValuesF(fp, value, 1);

                if(tn_ptr != NULL)
                {
                    tn_ptr->priority = CLIP(value[0], 0.0, 1.0);
                } 
            }

            else
            {
                fprintf(
		    stderr,
                    "%s: Unsupported parameter `%s'.\n",
                    filename, buf
                );
                FSeekNextLine(fp);
            }

        } while(1);

	/* Close file. */            
        FClose(fp);

	return(0);
}
