#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <sys/stat.h>

#include "../include/fio.h"
#include "../include/string.h"
#include "../include/disk.h"

#include "menu.h"
#include "weather.h"
#include "sar.h"
#include "sarmenuop.h"
#include "sarmenucb.h"
#include "sarmenuoptions.h"
#include "sarmenucodes.h"
#include "config.h"


static char *SARGetFileObjectName(const char *path);

static int SARMenuBuildStandardButton(
	sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        int w, int h,
	Boolean sensitive,
        const char *label,
        int id_code
);
static int SARMenuBuildOptionsButton(
        sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        int w, int h,
	Boolean sensitive,
        const char *label,
        int id_code
);
static int SARMenuBuildButtonImage(
        sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        int w, int h,
        const char *label,
        int id_code,
	sar_image_struct *unarmed_img,
	sar_image_struct *armed_img,
	sar_image_struct *highlighted_img
);
static int SARMenuBuildStandardSwitch(
	sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        int w, int h,
        const char *label,
        Boolean state,
        int id_code,
        void (*change_cb)(void *, int, void *, Boolean)
);
static int SARMenuBuildStandardSpin(
	sar_core_struct *core_ptr, sar_menu_struct *menu,
	float x, float y,
	float w, float h,
	const char *label,
	int id_code,
	void (*change_cb)(void *, int, void *, char *)
);
static int SARMenuBuildStandardSlider(
        sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        float w, float h,
        const char *label,
        int id_code,
        void (*change_cb)(void *, int, void *, float)
);

static char *SARMenuBuildGetFullPath(const char *name);

static void SARBuildMenusAddToList(
	sar_core_struct *core_ptr, sar_menu_struct *menu
);

static void SARBuildMenusAddDirectory(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu, int list_num,
	const char *subdir
);

int SARBuildMenus(sar_core_struct *core_ptr);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? (float)atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Returns a statically allocated string containing the object's
 *	title (if any) or returns just the contents of path on failure.
 */
static char *SARGetFileObjectName(const char *path)
{
	FILE *fp;
	char *strptr;
	const char *cstrptr;
	static char name[SAR_OBJ_NAME_MAX];


	if(path == NULL)
	    return("(null)");

	*name = '\0';	/* Reset name. */

#define DO_RETURN_JUST_PATH				\
{							\
 strptr = strrchr(path, '/');				\
 cstrptr = (strptr != NULL) ? (strptr + 1) : path;	\
 strncpy(name, cstrptr, SAR_OBJ_NAME_MAX);		\
 name[SAR_OBJ_NAME_MAX - 1] = '\0';			\
}

	/* Open file for reading. */
	fp = FOpen(path, "rb");
	if(fp == NULL)
	{
	    DO_RETURN_JUST_PATH
	    return(name);
	}

        /* Name. */
        if(!FSeekToParm(fp, "name", SAR_COMMENT_CHAR, '\0'))
        {
	    strptr = FGetString(fp);
	    if(strptr != NULL)
	    {
		strncpy(name, strptr, SAR_OBJ_NAME_MAX);
		name[SAR_OBJ_NAME_MAX - 1] = '\0';
		free(strptr);
	    }
	}

	FClose(fp);

	/* Did we get a name? */
	if(*name == '\0')
	    DO_RETURN_JUST_PATH

#undef DO_RETURN_JUST_PATH

	return(name);
}


/*
 *	Procedure to create a standard button.
 *
 *	Returns whatever SARMenuCreateButton() returns.
 */
static int SARMenuBuildStandardButton(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu,
	float x, float y,
	int w, int h,		/* Entire size of button + label. */
	Boolean sensitive,
	const char *label,
	int id_code
)
{
	int n, bw, w2, h2;
	sar_menu_color_struct c;
	sar_menu_label_struct *label_ptr;
	sar_menu_button_struct *button_ptr;
	const sar_option_struct *opt;


        /* Button text color. */
        c.a = 1.0f;
        c.r = 0.9f;
        c.g = 0.9f;
        c.b = 0.9f;

	if(core_ptr == NULL)
	    return(-1);

	opt = &core_ptr->option;

	/* Calculate button width (normal size of button is 42x50). */
	bw = (int)((float)h * 42.0 / 50.0);

	/* Calculate label size. */
	w2 = MAX(w - bw, 0);
	h2 = h;
	/* Create label. */ 
	n = SARMenuCreateLabel(
	    menu, x, y,
	    w2, h2,
	    label,
	    &c, opt->menu_font,
	    core_ptr->menu_button_label_img
	);
	if(n > -1)
	{
	    label_ptr = SAR_MENU_LABEL(menu->object[n]);
	    label_ptr->sensitive = sensitive;
	    label_ptr->align = SAR_MENU_LABEL_ALIGN_CENTER;
	    label_ptr->offset_x = bw / 2;
	}

	/* Create button. */
	n = SARMenuCreateButton(
	    menu, x, y,
	    bw, h,
	    NULL,		/* No label. */
	    &c, opt->menu_font,
	    core_ptr->menu_button_unarmed_img,
	    core_ptr->menu_button_armed_img,
	    core_ptr->menu_button_highlighted_img,
	    core_ptr, id_code,
	    SARMenuButtonCB
	);
        if(n > -1)
        {
            button_ptr = (sar_menu_button_struct *)menu->object[n];
	    button_ptr->sensitive = sensitive;
            button_ptr->offset_x = (int)((bw / 2.0) - (w / 2.0));
        }

	return(n);
}

/*
 *      Procedure to create a standard button.
 *
 *      Returns whatever SARMenuCreateButton() returns.
 *
 *	This is the same as SARMenuBuildStandardButton() except that
 *	the callback is set to SARMenuOptionsButtonCB().
 */
static int SARMenuBuildOptionsButton(
        sar_core_struct *core_ptr,
        sar_menu_struct *menu,
        float x, float y,
        int w, int h,           /* Entire size of button + label. */
	Boolean sensitive,
        const char *label,
        int id_code
)
{
        int n, bw, w2, h2;
        sar_menu_color_struct c;
        sar_menu_label_struct *label_ptr;
        sar_menu_button_struct *button_ptr;
        const sar_option_struct *opt;


        /* Button text color */
        c.a = 1.0f;
        c.r = 0.9f;
        c.g = 0.9f;
        c.b = 0.9f;

        if(core_ptr == NULL)
            return(-1);

	opt = &core_ptr->option;

        /* Calculate button width (normal size of button is 42x50). */
        bw = (int)((float)h * 42.0 / 50.0);

        /* Calculate label size. */
        w2 = MAX(w - bw, 0);
        h2 = h;
        /* Create label. */
        n = SARMenuCreateLabel(
            menu, x, y,
            w2, h2,
            label,
            &c, opt->menu_font,
            core_ptr->menu_button_label_img
        );
        if(n > -1)
        {
            label_ptr = SAR_MENU_LABEL(menu->object[n]);
	    label_ptr->sensitive = sensitive;
            label_ptr->align = SAR_MENU_LABEL_ALIGN_CENTER;
            label_ptr->offset_x = bw / 2;
        }

        /* Create button. */
        n = SARMenuCreateButton(
            menu, x, y,
            bw, h,
            NULL,               /* No label. */
            &c, opt->menu_font,
            core_ptr->menu_button_unarmed_img,
            core_ptr->menu_button_armed_img,
            core_ptr->menu_button_highlighted_img,
            core_ptr, id_code,
            SARMenuOptionsButtonCB
        );
        if(n > -1)
        {
            button_ptr = (sar_menu_button_struct *)menu->object[n];
	    button_ptr->sensitive = sensitive;
            button_ptr->offset_x = (int)((bw / 2.0) - (w / 2.0));
        }

        return(n);
}

/*
 *	Similar to SARMenuBuildStandardButton() but you get to specify
 *	the images.
 */
static int SARMenuBuildButtonImage(
        sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        int w, int h,
        const char *label,
        int id_code,
        sar_image_struct *unarmed_img,
        sar_image_struct *armed_img,
        sar_image_struct *highlighted_img
)
{
        sar_menu_color_struct button_color;
        const sar_option_struct *opt;

	/* Button text color */
        button_color.a = 1.0f;
        button_color.r = 0.9f;
        button_color.g = 0.9f;
        button_color.b = 0.9f;

        if(core_ptr == NULL)
            return(-1);

	opt = &core_ptr->option;

        return(
            SARMenuCreateButton(
                menu, x, y, w, h,
                label,
                &button_color, opt->menu_font,
                unarmed_img,
                armed_img,
                highlighted_img,
                core_ptr, id_code,
                SARMenuButtonCB
            )
        );
}

/*
 *	Procedure to create a standard switch.
 *
 *      Returns whatever SARMenuCreateSwitch() returns.
 */
static int SARMenuBuildStandardSwitch(
        sar_core_struct *core_ptr,
        sar_menu_struct *menu,
        float x, float y,
        int w, int h,
	const char *label,
	Boolean state,
        int id_code,
        void (*change_cb)(void *, int, void *, Boolean)
)
{
        sar_menu_color_struct switch_color;
        const sar_option_struct *opt;

        if(core_ptr == NULL)
            return(-1);

	opt = &core_ptr->option;

        switch_color.a = 1.0f;
        switch_color.r = 0.9f;
        switch_color.g = 0.9f;
        switch_color.b = 0.9f;

	return(
	    SARMenuCreateSwitch(
		menu, x, y, w, h, 
		&switch_color, opt->menu_font,
		label,
                core_ptr->menu_switch_bg_img,
                core_ptr->menu_switch_off_img,
                core_ptr->menu_switch_on_img,
		state,
		core_ptr, id_code,
		change_cb
            )
        );
}


/*
 *      Procedure to create a standard spin.
 *
 *	Returns whatever SARMenuCreateSpin() returns.
 */
static int SARMenuBuildStandardSpin(
	sar_core_struct *core_ptr,
        sar_menu_struct *menu,
        float x, float y,
        float w, float h,
	const char *label,
        int id_code,
	void (*change_cb)(void *, int, void *, char *)
)
{
        sar_menu_color_struct label_color, value_color;
        const sar_option_struct *opt;

        label_color.a = 1.0f;
        label_color.r = 0.9f;
        label_color.g = 0.9f;
        label_color.b = 0.9f;

	value_color.a = 1.0f;
	value_color.r = 0.0f;
	value_color.g = 0.9f;
	value_color.b = 0.0f;


        if(core_ptr == NULL)
            return(-1);

	opt = &core_ptr->option;

	return(
	    SARMenuCreateSpin(
                menu, x, y, w, h,
		&label_color, &value_color,
		opt->menu_font, opt->message_font, label,
                core_ptr->menu_spin_label_img,
                core_ptr->menu_spin_value_img,
                core_ptr->menu_spin_dec_armed_img,
                core_ptr->menu_spin_dec_unarmed_img,
                core_ptr->menu_spin_inc_armed_img,
                core_ptr->menu_spin_inc_unarmed_img,
                core_ptr, id_code,
		change_cb
	    )
	);
}

/*
 *      Procedure to create a standard slider.
 *
 *      Returns whatever SARMenuCreateSlider() returns.
 */
static int SARMenuBuildStandardSlider(
        sar_core_struct *core_ptr, sar_menu_struct *menu,
        float x, float y,
        float w, float h,
        const char *label,
        int id_code,
        void (*change_cb)(void *, int, void *, float)
)
{
        sar_menu_color_struct label_color;
        const sar_option_struct *opt;

        label_color.a = 1.0f;
        label_color.r = 0.9f;
        label_color.g = 0.9f;
        label_color.b = 0.9f;

        if(core_ptr == NULL)
            return(-1);

	opt = &core_ptr->option;

        return(
            SARMenuCreateSlider(
                menu, x, y, w, h,
                &label_color,
                opt->menu_font, label,
                core_ptr->menu_slider_label_img,
                core_ptr->menu_slider_trough_img,
                core_ptr->menu_slider_handle_img,
                core_ptr, id_code,
                change_cb
            )
        );
}


/*
 *	Prefixes local data dir to name and checks if file exists,
 *	if not then prefixes global data dir regardless if it exists
 *	or not.
 *
 *	Return is statically allocated.
 */
static char *SARMenuBuildGetFullPath(const char *name)
{
	const char *s;
	struct stat stat_buf;
	static char path[PATH_MAX + NAME_MAX];

	if(name == NULL)
	    return("");

	s = PrefixPaths(dname.local_data, name);
	if(s == NULL)
	    return("");

	strncpy(path, s, PATH_MAX + NAME_MAX);
	path[PATH_MAX + NAME_MAX - 1] = '\0';

	if(stat(path, &stat_buf))
	{
	    /* Does not exist locally, prefix global. */
	    s = PrefixPaths(dname.global_data, name);
	    if(s == NULL)
		return("");

	    strncpy(path, s, PATH_MAX + NAME_MAX);
            path[PATH_MAX + NAME_MAX - 1] = '\0';
	}

	return(path);
}

/*
 *	Adds the specified menu to the list of menus on the core
 *	structure.
 */
static void SARBuildMenusAddToList(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu
)
{
	int n;

	if(core_ptr->total_menus < 0)
	    core_ptr->total_menus = 0;

	n = core_ptr->total_menus;

	core_ptr->total_menus++;
	core_ptr->menu = (sar_menu_struct **)realloc(
	    core_ptr->menu,
	    core_ptr->total_menus * sizeof(sar_menu_struct *)
	);
	if(core_ptr->menu == NULL)
	{
	    core_ptr->total_menus = 0;
	    return;
	}

	core_ptr->menu[n] = menu;
}

/*
 *	Adds directory listing to the specified list.
 *
 *	Local data directory listing will be added first, then global.
 */
static void SARBuildMenusAddDirectory(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu, int list_num,
	const char *subdir	/* Subdir to local and global data dirs. */
)
{
	int strc;
	const char *cstrptr, *parent;
	char **strv;
        char tmp_path[PATH_MAX + NAME_MAX];
	struct stat stat_buf;


	if((core_ptr == NULL) || (menu == NULL) || (subdir == NULL))
	    return;

	/* Complete path to local directory. */
	parent = dname.local_data;

	cstrptr = PrefixPaths(parent, subdir);
        if(cstrptr != NULL)
        {
            strncpy(tmp_path, cstrptr, PATH_MAX + NAME_MAX);
            tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

	    /* Get directory listing. */
            strv = GetDirEntNames(tmp_path);
            if(strv != NULL)
            {
		int i;
		char *name;
		const char *full_path, *item_name;
		sar_menu_list_item_data_struct *data;

		/* Get total number of strings. */
		for(strc = 0; strv[strc] != NULL; strc++);

		/* Sort array. */
		strv = StringQSort(strv, strc);

		/* Add each file name to the listing. */
		for(i = 0; i < strc; i++)
                {
                    name = strv[i];

		    /* Skip directories. */
                    if(!strcasecmp(name, ".") ||
                       !strcasecmp(name, "..")
                    )
		    {
			free(name);
			continue;
		    }

		    /* Get complete file name. */
		    full_path = PrefixPaths(tmp_path, name);
		    if(stat(full_path, &stat_buf))
		    {
                        free(name);
                        continue;
                    }
#ifdef S_ISDIR
		    if(S_ISDIR(stat_buf.st_mode))
		    {
                        free(name);  
                        continue;
		    }
#endif	/* S_ISDIR */

		    /* Allocate list item data. */
                    data = SAR_MENU_LIST_ITEM_DATA(calloc(
                        1, sizeof(sar_menu_list_item_data_struct)
                    ));
                    if(data != NULL)
                    {
                        data->core_ptr = core_ptr;
                        data->filename = STRDUP(full_path);
			item_name = SARGetFileObjectName(data->filename);
                    }
		    else
		    {
			item_name = name;
		    }
		    /* Add item to list. */
                    SARMenuListAppendItem(
                        menu, list_num,
                        item_name,	/* Item name. */
                        data,		/* Item data. */
                        0		/* Flags. */
                    );

                    free(name);
                }

		/* Free pointer array (each pointer storage already
		 * free'ed.
		 */
                free(strv);
            }
	}

/* No global data for __MSW__. */
#ifndef __MSW__
	/* Complete path to global data directory. */
	parent = dname.global_data;

	cstrptr = PrefixPaths(parent, subdir);
	if(cstrptr != NULL)
        {
            strncpy(tmp_path, cstrptr, PATH_MAX + NAME_MAX);
            tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';
        
            /* Get directory listing. */
            strv = GetDirEntNames(tmp_path);
            if(strv != NULL)
            {
                int i;
                char *name;
                const char *full_path, *item_name;
                sar_menu_list_item_data_struct *data;

                /* Get total number of strings. */
                for(strc = 0; strv[strc] != NULL; strc++);

                /* Sort array. */
                strv = StringQSort(strv, strc);

                /* Add each listing. */
                for(i = 0; i < strc; i++)
                {
                    name = strv[i];

                    /* Skip directories. */
                    if(!strcasecmp(name, ".") ||
                       !strcasecmp(name, "..")
                    )
                    {
                        free(name);
                        continue;
		    }

                    /* Get complete file name. */
                    full_path = PrefixPaths(tmp_path, name);
                    if(stat(full_path, &stat_buf))
                    {
                        free(name);
                        continue;
                    }
#ifdef S_ISDIR
                    if(S_ISDIR(stat_buf.st_mode))
                    {   
                        free(name);
                        continue;
                    }
#endif	/* S_ISDIR */

                    /* Allocate list item data. */
                    data = SAR_MENU_LIST_ITEM_DATA(calloc(
                        1, sizeof(sar_menu_list_item_data_struct)
                    ));
                    if(data != NULL)
                    {
                        data->core_ptr = core_ptr;
                        data->filename = STRDUP(full_path);
                        item_name = SARGetFileObjectName(data->filename);
                    }
                    else
                    {
                        item_name = name;
                    }
                    /* Add item to list. */
                    SARMenuListAppendItem(
                        menu, list_num,
                        item_name,	/* Item name. */
                        data,		/* Item data. */
                        0		/* Flags. */
                    );

                    free(name);
                }

                /* Free pointer array (each pointer storage already
                 * free'ed.
                 */
                free(strv);
            }
        }
#endif	/* Not __MSW__ */
}


/*
 *	Builds menus.
 */
int SARBuildMenus(sar_core_struct *core_ptr)
{
	int btn_num, list_num, switch_num, spin_num, slider_num, map_num;
	const char *img_path;
	gw_display_struct *display;
	sar_menu_struct *menu;
	sar_menu_list_struct *list;
	sar_menu_spin_struct *spin;
	sar_menu_color_struct	list_color, mesgbox_color,
				switch_color, progress_color;
	GWFont *font, *value_font;
        const sar_option_struct *opt;

/* Standard button size is 42x50 and button label is 208x50 */
int btn_width_std = 180, btn_height_std = 40;
int btn_width_main = 180, btn_height_main = 40;
int btn_width_opt = 250, btn_height_opt = 40;
int btn_width_map = 32, btn_height_map = 32;

	if(core_ptr == NULL)
	    return(-1);

        display = core_ptr->display;
        opt = &core_ptr->option;

	list_color.a = 1.0f;
	list_color.r = 0.0f;
	list_color.g = 0.9f;
	list_color.b = 0.0f;

	mesgbox_color.a = 1.0f;
	mesgbox_color.r = 0.0f;
	mesgbox_color.g = 0.9f;
	mesgbox_color.b = 0.0f;

	switch_color.a = 1.0f;
	switch_color.r = 0.9f;
	switch_color.g = 0.9f;
	switch_color.b = 0.9f;

	progress_color.a = 1.0f;
	progress_color.r = 0.9f;
	progress_color.g = 0.9f;
	progress_color.b = 0.9f;

	font = opt->menu_font;
	value_font = opt->message_font;


	/* Load images for menus (these will be shared amoung the
	 * menu object/widgets.
	 */
#define DO_LOAD_IMG(i,p)		\
{ if(*(i) != NULL)			\
   SARImageDestroy(*(i));		\
 *(i) = SARImageLoadFromFile(p);	\
}
	/* Allocate pointers for 9 images for the list menu object
	 * background and load each image.
	 */
	core_ptr->menu_list_bg_img = (sar_image_struct **)realloc(
	    core_ptr->menu_list_bg_img,
	    9 * sizeof(sar_image_struct *)
	);
	if(core_ptr->menu_list_bg_img != NULL)
	{
	    int i;
	    char tmp_path[PATH_MAX + NAME_MAX];

	    for(i = 0; i < 9; i++)
	    {
		sprintf(tmp_path, "images/list%i.tga", i);

		core_ptr->menu_list_bg_img[i] = NULL;
		DO_LOAD_IMG(
		    &core_ptr->menu_list_bg_img[i],
		    tmp_path
		);
	    }
	}

	/* Button images. */
	DO_LOAD_IMG(
	    &core_ptr->menu_button_armed_img,
	    "images/button_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_unarmed_img,
	    "images/button_unarmed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_highlighted_img,
	    "images/button_highlighted.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_label_img,
	    "images/button_label.tga"
	);

	/* Label images. */
	DO_LOAD_IMG(
	    &core_ptr->menu_label_bg_img,
	    "images/label_bg.tga"
	);

	/* Switch image. */
	DO_LOAD_IMG(
	    &core_ptr->menu_switch_bg_img,
	    "images/label_bg.tga"	/* Use label for now. */
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_switch_off_img,
	    "images/switch_off.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_switch_on_img,
	    "images/switch_on.tga"
	);

	/* Progress bar images. */
	DO_LOAD_IMG(
	    &core_ptr->menu_progress_bg_img,
	    "images/progress_bar_bg.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_progress_fg_img,
	    "images/progress_bar_fg.tga"
	);

	/* Map panning buttons. */
	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_up_armed_img,
	    "images/button_pan_up_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_up_unarmed_img,
	    "images/button_pan_up_unarmed.tga"
	);

	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_down_armed_img,
	    "images/button_pan_down_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_down_unarmed_img,
	    "images/button_pan_down_unarmed.tga"
	);

	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_left_armed_img,
	    "images/button_pan_left_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_left_unarmed_img,
	    "images/button_pan_left_unarmed.tga"
	);

	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_right_armed_img,
	    "images/button_pan_right_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_pan_right_unarmed_img,
	    "images/button_pan_right_unarmed.tga"
	);

	/* Map zoom buttons. */
	DO_LOAD_IMG(
	    &core_ptr->menu_button_zoom_in_armed_img,
	    "images/button_zoom_in_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_zoom_in_unarmed_img,
	    "images/button_zoom_in_unarmed.tga"
	);

	DO_LOAD_IMG(
	    &core_ptr->menu_button_zoom_out_armed_img,
	    "images/button_zoom_out_armed.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_button_zoom_out_unarmed_img,
	    "images/button_zoom_out_unarmed.tga"
	);

	/* Spin images. */
	DO_LOAD_IMG(
	    &core_ptr->menu_spin_label_img,
	    "images/spin_label.tga"
	);
	DO_LOAD_IMG(
	    &core_ptr->menu_spin_value_img,
	    "images/spin_value.tga"
	);
        DO_LOAD_IMG(
            &core_ptr->menu_spin_dec_armed_img,
            "images/spin_dec_btn_armed.tga"
        );
        DO_LOAD_IMG(
            &core_ptr->menu_spin_dec_unarmed_img,
            "images/spin_dec_btn_unarmed.tga"
        );
        DO_LOAD_IMG(
            &core_ptr->menu_spin_inc_armed_img,
            "images/spin_inc_btn_armed.tga"
        );
        DO_LOAD_IMG(
            &core_ptr->menu_spin_inc_unarmed_img,
            "images/spin_inc_btn_unarmed.tga"
        );

	/* Slider images. */
	DO_LOAD_IMG(
	    &core_ptr->menu_slider_label_img,
	    "images/slider_label.tga"
	);
        DO_LOAD_IMG(
            &core_ptr->menu_slider_trough_img,
            "images/slider_trough.tga"
        );
        DO_LOAD_IMG(
            &core_ptr->menu_slider_handle_img,
            "images/slider_handle.tga"
        );

	/* Menu's map icon images. */
	DO_LOAD_IMG(
	    &core_ptr->menumap_helipad_img,
	    SAR_DEF_MISSION_MAPICON_HELIPAD_FILE
	);
	DO_LOAD_IMG(
	    &core_ptr->menumap_intercept_img,
	    SAR_DEF_MISSION_MAPICON_INTERCEPT_FILE
	);
	DO_LOAD_IMG(
	    &core_ptr->menumap_helicopter_img,
	    SAR_DEF_MISSION_MAPICON_HELICOPTER_FILE
	);
	DO_LOAD_IMG(
	    &core_ptr->menumap_victim_img,
	    SAR_DEF_MISSION_MAPICON_VICTIM_FILE
	);
	DO_LOAD_IMG(
	    &core_ptr->menumap_vessel_img,
	    SAR_DEF_MISSION_MAPICON_VESSEL_FILE
	);
	DO_LOAD_IMG(
	    &core_ptr->menumap_crash_img,
	    SAR_DEF_MISSION_MAPICON_CRASH_FILE
	);

#undef DO_LOAD_IMG

	/* ****************************************************** */
	/* Menu: Main */
	img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_MAIN_FILE);
	menu = SARMenuCreate(
	    SAR_MENU_TYPE_STANDARD,
	    SAR_MENU_NAME_MAIN,
	    img_path
	);
	if(menu == NULL)
	    return(-1);
	else
	    SARBuildMenusAddToList(core_ptr, menu);

        /* Mission button. */
	SARMenuBuildStandardButton(
	    core_ptr, menu, 0.18f, 0.23f, btn_width_main, btn_height_main,
	    True,
#ifdef PROG_LANGUAGE_ENGLISH
"Mission"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Misin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Mission"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Mission"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Missione"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Missie"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Misso"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Misjon"
#endif
            ,
	    SAR_OP_MENU_TO_MISSION
        );

	/* Campaign button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.18f, 0.37f, btn_width_main, btn_height_main,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Campaign"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Campaa"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Campagne"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Wahlkampf"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Campagna"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Campagne"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Campanha"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Campaign"
#endif
	    ,
	    SAR_OP_MENU_TO_CAMPAIGN
        );

	/* Free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.18f, 0.51f, btn_width_main, btn_height_main,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Free Flight"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Liberta Vuelo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Librer Vol"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Flug"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Libera Volo"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Vlucht"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Libertam Vo"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Free Flight"
#endif
            ,
	    SAR_OP_MENU_TO_FREE_FLIGHT
        );

        /* Options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.18f, 0.65f, btn_width_main, btn_height_main,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Options"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Opciones"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Options"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Option"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Opzioni"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Opties"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Opes"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Options"
#endif
            ,
	    SAR_OP_MENU_TO_OPTIONS
        );

        /* Exit button. */
	SARMenuBuildStandardButton(
            core_ptr, menu, 0.18f, 0.79f, btn_width_main, btn_height_main,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Exit"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Salida"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Sortie"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Ausgang"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"L'Uscita"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Uitgang"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Sada"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Exit"
#endif
            ,
	    SAR_OP_MENU_EXIT
        );


        /* ****************************************************** */
        /* Menu: Mission */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION,
            img_path
        );
        if(menu == NULL)   
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Missions list. */
        list_num = SARMenuCreateList(
            menu, 0.50f, 0.38f, 0.95f, 0.60f,
            &list_color,
            value_font,
            "Select Mission:",
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_MISSION_LIST_SELECT,
            SARMenuListSelectCB,
	    SARMenuListActivateCB
        );
        SARBuildMenusAddDirectory(
            core_ptr, menu, list_num,
            SAR_DEF_MISSIONS_DIR
        );
        if(list_num > -1)
        {
            /* Restore previously selected mission. */
            list = SAR_MENU_LIST(menu->object[list_num]);
	    SARMenuListSelect(
		display, menu, list_num,
		opt->last_selected_mission,
		True,		/* Scroll to. */
		False		/* No redraw. */
	    );
        }

	/* Last mission results button. */
	SARMenuBuildStandardButton(
	    core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
	    True, "Last Log...", SAR_OP_MENU_TO_MISSION_LOG_MAP
	);


        /* Back to main button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    ,
	    SAR_OP_MENU_TO_MAIN
        );

        /* Mission briefing button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
            True, "Briefing", SAR_OP_MENU_TO_MISSION_BRIEF
        );


        /* ****************************************************** */
        /* Menu: Mission->Briefing */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION_BRIEF,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Mission briefing message box. */
        SARMenuCreateMessageBox(
            menu, 0.50f, 0.38f, 0.95f, 0.60f,
            &mesgbox_color,
            value_font,
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_MISSION_BRIEF_MESG,
            NULL        /* No message */
        );

        /* Back to missions list. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_MISSION
        );

        /* Mission map button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
            True, "Map", SAR_OP_MENU_TO_MISSION_MAP
        );

        /* Mission begin button. */
        SARMenuBuildStandardButton(   
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
            True, "Begin", SAR_OP_MENU_TO_MISSION_BEGIN
        );


        /* ****************************************************** */   
        /* Menu: Mission->Briefing->Map */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION_MAP,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

	/* Map. */
	map_num = SARMenuCreateMap(
	    menu, 0.60f, 0.42f, 0.74f, 0.7f,
            value_font, &mesgbox_color,
	    (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    0.0f, 0.0f,
	    (float)SAR_MAP_DEF_MTOP_COEFF,
	    SAR_MENU_MAP_SHOW_MARKING_ALL
	);

	/* Map pan buttons. */
	SARMenuBuildButtonImage(
	    core_ptr, menu, 0.12f, 0.34f, btn_width_map, btn_height_map,
	    NULL,
	    SAR_OP_MISSION_MAP_UP,
	    core_ptr->menu_button_pan_up_unarmed_img,
	    core_ptr->menu_button_pan_up_armed_img,
	    core_ptr->menu_button_pan_up_unarmed_img
	);
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.12f, 0.46f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_DOWN,
            core_ptr->menu_button_pan_down_unarmed_img,
            core_ptr->menu_button_pan_down_armed_img,  
            core_ptr->menu_button_pan_down_unarmed_img 
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.06f, 0.40f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_LEFT,
            core_ptr->menu_button_pan_left_unarmed_img,
            core_ptr->menu_button_pan_left_armed_img,  
            core_ptr->menu_button_pan_left_unarmed_img 
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.18f, 0.40f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_RIGHT,
            core_ptr->menu_button_pan_right_unarmed_img,
            core_ptr->menu_button_pan_right_armed_img,
            core_ptr->menu_button_pan_right_unarmed_img
        );

        SARMenuBuildButtonImage(
            core_ptr, menu, 0.07f, 0.18f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_ZOOM_OUT,
            core_ptr->menu_button_zoom_out_unarmed_img,
            core_ptr->menu_button_zoom_out_armed_img,
            core_ptr->menu_button_zoom_out_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.17f, 0.18f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_ZOOM_IN,
            core_ptr->menu_button_zoom_in_unarmed_img,
            core_ptr->menu_button_zoom_in_armed_img,
            core_ptr->menu_button_zoom_in_unarmed_img
        );

        /* Back to mission briefing button. */ 
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_MISSION_BRIEF
        );


        /* ****************************************************** */
        /* Menu: Mission->Log Map */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION_LOG_MAP,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Map. */
        map_num = SARMenuCreateMap(
            menu, 0.60f, 0.42f, 0.74f, 0.7f,
            value_font, &mesgbox_color,
	    (const sar_image_struct **)core_ptr->menu_list_bg_img,
            0.0f, 0.0f,
            (float)SAR_MAP_DEF_MTOP_COEFF,
            SAR_MENU_MAP_SHOW_MARKING_SELECTED
        );

        /* Map pan buttons. */
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.12f, 0.34f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_UP,
            core_ptr->menu_button_pan_up_unarmed_img,  
            core_ptr->menu_button_pan_up_armed_img,
            core_ptr->menu_button_pan_up_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.12f, 0.46f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_DOWN,
            core_ptr->menu_button_pan_down_unarmed_img,
            core_ptr->menu_button_pan_down_armed_img,
            core_ptr->menu_button_pan_down_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.06f, 0.40f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_LEFT,
            core_ptr->menu_button_pan_left_unarmed_img,
            core_ptr->menu_button_pan_left_armed_img,
            core_ptr->menu_button_pan_left_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.18f, 0.40f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_RIGHT,
            core_ptr->menu_button_pan_right_unarmed_img,
            core_ptr->menu_button_pan_right_armed_img,
            core_ptr->menu_button_pan_right_unarmed_img
        );

        SARMenuBuildButtonImage(
            core_ptr, menu, 0.07f, 0.18f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_ZOOM_OUT,
            core_ptr->menu_button_zoom_out_unarmed_img,
            core_ptr->menu_button_zoom_out_armed_img,
            core_ptr->menu_button_zoom_out_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu, 0.17f, 0.18f, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_ZOOM_IN,
            core_ptr->menu_button_zoom_in_unarmed_img,
            core_ptr->menu_button_zoom_in_armed_img,
            core_ptr->menu_button_zoom_in_unarmed_img
        );

	/* Log event prev/next buttons. */
	SARMenuBuildStandardButton(
            core_ptr, menu, 0.12f, 0.57f, 100, btn_height_std,
            True, "Next", SAR_OP_MISSION_LOG_MAP_EVENT_NEXT
        );
	SARMenuBuildStandardButton(
            core_ptr, menu, 0.12f, 0.67f, 100, btn_height_std,
	    True, "Prev", SAR_OP_MISSION_LOG_MAP_EVENT_PREV
	);

        /* Back to missions button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True, "Continue", SAR_OP_MENU_TO_MISSION
        );

        /* ****************************************************** */
        /* Menu: Free Flight */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

	/* Scenery list. */
	list_num = SARMenuCreateList(
            menu, 0.50f, 0.20f, 0.95f, 0.35f,
            &list_color,
            value_font,
	    "Select Theater:",
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_FREE_FLIGHT_SCENERY_LIST,
	    SARMenuListSelectCB,
	    SARMenuListActivateCB
	);
	SARBuildMenusAddDirectory(
	    core_ptr, menu, list_num,
	    SAR_DEF_SCENERY_DIR
	);

        /* Locations list. */
        list_num = SARMenuCreateList(
            menu, 0.50f, 0.60f, 0.95f, 0.30f,
            &list_color,
            value_font,
            "Select Location:",
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_FREE_FLIGHT_LOCATIONS_LIST,
            SARMenuListSelectCB,
	    SARMenuListActivateCB
        );

        /* Back to main button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_MAIN
        );

        /* Aircraft button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
            True, "Aircraft", SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT
        );

	/* Restore previously selected scene. */
	list = SAR_MENU_LIST(SARMenuGetObjectByID(
	    menu, SAR_OP_FREE_FLIGHT_SCENERY_LIST, &list_num
	));
	if(list != NULL)
        {
	    int i = opt->last_selected_ffscene;
	    sar_menu_list_struct *loc_list;
	    int loc_list_num;
	    sar_menu_list_item_struct *item;


	    /* Get scene item pointer on the scene list object. */
	    item = SARMenuListGetItemByNumber(list, i);

	    /* Get locations list object. */
	    loc_list = SAR_MENU_LIST(SARMenuGetObjectByID(
		menu, SAR_OP_FREE_FLIGHT_LOCATIONS_LIST, &loc_list_num
	    ));
	    if((item != NULL) && (loc_list != NULL))
	    {
		sar_menu_list_item_data_struct *data = SAR_MENU_LIST_ITEM_DATA(
		    item->client_data
		);

		/* Restore previously selected free flight scenery on
		 * the scene list object.
		 */
		SARMenuListSelect(
		    display, menu, list_num,
		    i,
		    True,	/* Scroll to. */
		    False	/* No redraw. */
		);

		/* Fetch locations list from scene to the locations
		 * list object.
		 */
		if(data != NULL)
		    SARSceneLoadLocationsToList(
			core_ptr,
			menu, loc_list, loc_list_num,
			data->filename		/* Scene file. */
		    );

		/* Select first item on locations list. */
		if(loc_list->total_items > 0)
		    loc_list->selected_item = 0;
	    }
        }


        /* ****************************************************** */
        /* Menu: Free Flight->Aircraft */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Aircrafts list. */
        list_num = SARMenuCreateList(
            menu, 0.50f, 0.38f, 0.95f, 0.60f,
            &list_color,
            value_font,
            "Select Aircraft:",
	    (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_FREE_FLIGHT_AIRCRAFTS_LIST,
            SARMenuListSelectCB,
	    SARMenuListActivateCB
        );
        SARBuildMenusAddDirectory(
            core_ptr, menu, list_num,   
            SAR_DEF_AIRCRAFTS_DIR
        );
        if(list_num > -1)
        {
            /* Restore previously selected aircraft. */
            list = menu->object[list_num];
            SARMenuListSelect(
		display, menu, list_num,
		opt->last_selected_ffaircraft,
		True,		/* Scroll to. */
		False		/* No redraw. */
            );
        }

        /* Aircraft info. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
            True, "Details", SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT_INFO
        );

        /* Weather. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.74f, btn_width_std, btn_height_std,
            True, "Weather", SAR_OP_MENU_TO_FREE_FLIGHT_WEATHER
        );

        /* Back to Free Flight: Select Scenery button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_FREE_FLIGHT
        );

        /* Begin free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
            True, "Begin", SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN
        );


        /* ****************************************************** */
        /* Menu: Free Flight->Aircraft->Weather */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT_WEATHER,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

	/* Weather presets spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.2f, 0.9f, 0.0f,
            "Condition",
	    SAR_OP_MENU_FREE_FLIGHT_WEATHER_CONDITION,
	    SARMenuSpinCB
        );
	if(spin_num > -1)
	{
	    int i;
	    sar_weather_data_struct *weather_data = core_ptr->weather_data;
	    sar_weather_data_entry_struct *wdp_ptr;

	    spin = SAR_MENU_SPIN(menu->object[spin_num]);
	    spin->allow_warp = False;
	    if(weather_data != NULL)
	    {
		for(i = 0; i < weather_data->total_presets; i++)
		{
		    wdp_ptr = weather_data->preset[i];
		    if(wdp_ptr == NULL)
			continue;

		    if(wdp_ptr->name == NULL)
			continue;

		    SARMenuSpinAddValue(
			menu, spin_num,
			wdp_ptr->name
		    );
		}
	    }

	    /* Restore previously selected item. */
	    spin->cur_value = opt->last_selected_ffweather;
	}

        /* Free flight start up using system time. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (0.0 * 1.0 / 4.0)), 0.4f,
            0, 0,
            "System Time",
            opt->system_time_free_flight,
	    SAR_OP_MENU_FREE_FLIGHT_SYSTEM_TIME,
	    SARMenuSwitchCB
        );

        /* Back to Free Flight: Select Aircraft button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT
        );

        /* Begin free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
            True, "Begin", SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN
        );


        /* ****************************************************** */
        /* Menu: Free Flight->Aircraft->Aircraft Info */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT_INFO,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);


        /* Aircraft information 3d object viewer. */
	if(True)
	{
	    sar_menu_color_struct color;

	    color.a = 1.0f;
	    color.r = 0.0f;
	    color.g = 0.9f;
	    color.b = 0.0f;

	    SARMenuCreateObjView(
		menu,
		0.5f, 0.41f,
	        0.95f, 0.75f,
		opt->message_font,
		&color,
		(const sar_image_struct **)core_ptr->menu_list_bg_img,
		"Untitled"
	    );
	}


        /* Back to Free Flight: Select Aircraft button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT
        );

        /* Begin free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
            True, "Begin", SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN
        );


        /* ****************************************************** */
        /* Menu: Options */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to main button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_MAIN
        );

        SARMenuBuildStandardButton(
            core_ptr, menu, 0.28f, 0.13f, btn_width_opt, btn_height_opt,
            True, "Simulation...", SAR_OP_MENU_TO_OPTIONS_SIMULATION
        );
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.28f, 0.32f, btn_width_opt, btn_height_opt,
            True, "Controller...", SAR_OP_MENU_TO_OPTIONS_CONTROLLER
        );
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.28f, 0.51f, btn_width_opt, btn_height_opt,
            True, "Graphics...", SAR_OP_MENU_TO_OPTIONS_GRAPHICS
        );
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.28f, 0.70f, btn_width_opt, btn_height_opt,
            True, "Sound...", SAR_OP_MENU_TO_OPTIONS_SOUND
        );


        /* ****************************************************** */
        /* Menu: Options->Simulation */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(  
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_SIMULATION,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS
        );

        /* Hoist contact. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.20f, 0.9f, 0.0f,
            "Hoist Contact", SAR_OPT_SELECT_HOIST_CONTACT,
            SARMenuOptionsSpinCB
        );
	if(spin_num > -1)
	{
	    spin = SAR_MENU_SPIN(menu->object[spin_num]);
	    spin->allow_warp = False;
            SARMenuSpinAddValue(menu, spin_num, "Easy");
            SARMenuSpinAddValue(menu, spin_num, "Moderate");
            SARMenuSpinAddValue(menu, spin_num, "Authentic");
	}

        /* Damage resistance. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.40f, 0.9f, 0.0f,
            "Damage Resistance", SAR_OPT_SELECT_DAMAGE_RESISTANCE,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
            SARMenuSpinAddValue(menu, spin_num, "Strong");
            SARMenuSpinAddValue(menu, spin_num, "Nominal");  
            SARMenuSpinAddValue(menu, spin_num, "Authentic");
	}

        /* Flight physics. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.60f, 0.9f, 0.0f,
            "Flight Physics", SAR_OPT_SELECT_FLIGHT_PHYSICS,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
	    spin = SAR_MENU_SPIN(menu->object[spin_num]);
	    spin->allow_warp = False;
	    SARMenuSpinAddValue(menu, spin_num, "Easy");
	    SARMenuSpinAddValue(menu, spin_num, "Moderate");
	    SARMenuSpinAddValue(menu, spin_num, "Realistic");
	}

        /* ****************************************************** */
        /* Menu: Options->Controller */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_CONTROLLER,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS
        );

	/* Joystick button mapping. */
        btn_num = SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.92f, btn_width_std, btn_height_std,
#if defined(JS_SUPPORT)
            True,
#else
	    False,
#endif
	    "Buttons", SAR_OP_MENU_TO_OPTIONS_CONTROLLER_JS_BTN
        );

        /* First joystick (js0) connection type. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.50f, 0.14f, 0.9f, 0.0f,
            "Joystick #1", SAR_OPT_SELECT_JS0_CONNECTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
#ifdef JS_SUPPORT
            SARMenuSpinAddValue(menu, spin_num, "Standard");
            SARMenuSpinAddValue(menu, spin_num, "USB");
#else
            SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif  /* JS_SUPPORT */
	}

        /* First joystick (js0) axis roles. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.50f, 0.26f, 0.9f, 0.0f,
            "Axises", SAR_OPT_SELECT_JS0_AXISES,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
#ifdef JS_SUPPORT
            SARMenuSpinAddValue(menu, spin_num, "Off");
            SARMenuSpinAddValue(menu, spin_num, "2D");
            SARMenuSpinAddValue(menu, spin_num, "2D with throttle");
            SARMenuSpinAddValue(menu, spin_num, "2D with hat");
            SARMenuSpinAddValue(menu, spin_num, "2D with throttle & hat");
            SARMenuSpinAddValue(menu, spin_num, "3D");
            SARMenuSpinAddValue(menu, spin_num, "3D with throttle");
            SARMenuSpinAddValue(menu, spin_num, "3D with hat");
            SARMenuSpinAddValue(menu, spin_num, "3D with throttle & hat");
            SARMenuSpinAddValue(menu, spin_num, "As throttle & rudder");
#else
	    SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif	/* JS_SUPPORT */
	}

        /* Second joystick (js1) connection type. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.50f, 0.43f, 0.9f, 0.0f,
            "Joystick #2", SAR_OPT_SELECT_JS1_CONNECTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
#ifdef JS_SUPPORT
            SARMenuSpinAddValue(menu, spin_num, "Standard");
            SARMenuSpinAddValue(menu, spin_num, "USB");
#else
            SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif  /* JS_SUPPORT */
	}

        /* Second joystick (js1) axis roles. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.50f, 0.55f, 0.9f, 0.0f,
            "Axises", SAR_OPT_SELECT_JS1_AXISES,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
#ifdef JS_SUPPORT
            SARMenuSpinAddValue(menu, spin_num, "Off");
            SARMenuSpinAddValue(menu, spin_num, "2D");
            SARMenuSpinAddValue(menu, spin_num, "2D with throttle");
            SARMenuSpinAddValue(menu, spin_num, "2D with hat");
            SARMenuSpinAddValue(menu, spin_num, "2D with throttle & hat");
            SARMenuSpinAddValue(menu, spin_num, "3D");
            SARMenuSpinAddValue(menu, spin_num, "3D with throttle");
            SARMenuSpinAddValue(menu, spin_num, "3D with hat");
            SARMenuSpinAddValue(menu, spin_num, "3D with throttle & hat");
            SARMenuSpinAddValue(menu, spin_num, "As throttle & rudder");
#else
	    SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif	/* JS_SUPPORT */
	}

        /* Priority spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.27f, 0.69f, 0.45f, 0.0f,
            "Priority", SAR_OPT_SELECT_JS_PRIORITY,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
#ifdef JS_SUPPORT
            SARMenuSpinAddValue(menu, spin_num, "Background");
            SARMenuSpinAddValue(menu, spin_num, "Foreground");
            SARMenuSpinAddValue(menu, spin_num, "Preempt");
#else
            SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif	/* JS_SUPPORT */
        }

        /* Test joystick button. */
	btn_num = SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.74f, btn_width_std, btn_height_std,
#if defined(JS_SUPPORT)
	    True,
#else
	    False,
#endif
            "Test...", SAR_OP_MENU_TO_OPTIONS_CONTROLLER_TEST
        );

        /* ****************************************************** */
        /* Menu: Options->Controller->Buttons */
#ifdef JS_SUPPORT
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_CONTROLLER_JS_BTN,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to controller button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS_CONTROLLER
        );

        /* Joystick 1 (js0) button action spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.14f, 0.9f, 0.0f,
            "Joystick #1 Action", SAR_OPT_SELECT_JS0_BUTTON_ACTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
            SARMenuSpinAddValue(menu, spin_num, "Rotate Modifier");
            SARMenuSpinAddValue(menu, spin_num, "Air Brakes");
            SARMenuSpinAddValue(menu, spin_num, "Wheel Brakes");
            SARMenuSpinAddValue(menu, spin_num, "Zoom In");
            SARMenuSpinAddValue(menu, spin_num, "Zoom Out");
            SARMenuSpinAddValue(menu, spin_num, "Hoist Up");
            SARMenuSpinAddValue(menu, spin_num, "Hoist Down");
	}

        /* Joystick 1 (js0) button number spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.26f, 0.9f, 0.0f,
            "Button Number", SAR_OPT_SELECT_JS0_BUTTON_NUMBER,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
	    /* Button values start from spin value - 1, since spin
	     * value 0 really means button -1 (none/invalid).
	     */
	    SARMenuSpinAddValue(menu, spin_num, "None");
            SARMenuSpinAddValue(menu, spin_num, "1");
            SARMenuSpinAddValue(menu, spin_num, "2");
            SARMenuSpinAddValue(menu, spin_num, "3");
            SARMenuSpinAddValue(menu, spin_num, "4");   
            SARMenuSpinAddValue(menu, spin_num, "5");
            SARMenuSpinAddValue(menu, spin_num, "6");
            SARMenuSpinAddValue(menu, spin_num, "7");
            SARMenuSpinAddValue(menu, spin_num, "8");
            SARMenuSpinAddValue(menu, spin_num, "9");
            SARMenuSpinAddValue(menu, spin_num, "10");
            SARMenuSpinAddValue(menu, spin_num, "11");
            SARMenuSpinAddValue(menu, spin_num, "12");
            SARMenuSpinAddValue(menu, spin_num, "13");
            SARMenuSpinAddValue(menu, spin_num, "14");
            SARMenuSpinAddValue(menu, spin_num, "15");
            SARMenuSpinAddValue(menu, spin_num, "16");
	}


        /* Joystick 2 (js1) button action spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.43f, 0.9f, 0.0f,
            "Joystick #2 Action", SAR_OPT_SELECT_JS1_BUTTON_ACTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
            SARMenuSpinAddValue(menu, spin_num, "Rotate Modifier");
            SARMenuSpinAddValue(menu, spin_num, "Air Brakes");
            SARMenuSpinAddValue(menu, spin_num, "Wheel Brakes");
            SARMenuSpinAddValue(menu, spin_num, "Zoom In");
            SARMenuSpinAddValue(menu, spin_num, "Zoom Out");
            SARMenuSpinAddValue(menu, spin_num, "Hoist Up");
            SARMenuSpinAddValue(menu, spin_num, "Hoist Down");
	}

        /* Joystick 2 (js1) button number spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.5f, 0.55f, 0.9f, 0.0f,
            "Button Number", SAR_OPT_SELECT_JS1_BUTTON_NUMBER,
            SARMenuOptionsSpinCB
        );
        /* Button values start from spin value - 1, since spin
         * value 0 really means button -1 (none/invalid).
         */
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
            SARMenuSpinAddValue(menu, spin_num, "None");
            SARMenuSpinAddValue(menu, spin_num, "1");
            SARMenuSpinAddValue(menu, spin_num, "2");
            SARMenuSpinAddValue(menu, spin_num, "3");
            SARMenuSpinAddValue(menu, spin_num, "4");
            SARMenuSpinAddValue(menu, spin_num, "5");
            SARMenuSpinAddValue(menu, spin_num, "6");
            SARMenuSpinAddValue(menu, spin_num, "7");
            SARMenuSpinAddValue(menu, spin_num, "8");
            SARMenuSpinAddValue(menu, spin_num, "9");
            SARMenuSpinAddValue(menu, spin_num, "10");
            SARMenuSpinAddValue(menu, spin_num, "11");
            SARMenuSpinAddValue(menu, spin_num, "12");
            SARMenuSpinAddValue(menu, spin_num, "13");
            SARMenuSpinAddValue(menu, spin_num, "14");
            SARMenuSpinAddValue(menu, spin_num, "15");
            SARMenuSpinAddValue(menu, spin_num, "16");
	}

        /* Test joystick button. */
        btn_num = SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.74f, btn_width_std, btn_height_std,
            True, "Test...", SAR_OP_MENU_TO_OPTIONS_CONTROLLER_TEST
        );
#endif	/* JS_SUPPORT */


        /* ****************************************************** */
        /* Menu: Options->Controller->Test */
#ifdef JS_SUPPORT
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_CONTROLLER_TEST,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to controller button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS_CONTROLLER
        );

	/* Joystick output multi-purpose display. */
	SARMenuCreateMDisplay(
	    menu, 0.50f, 0.36f, 0.95f, 0.56f,
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OPT_UPDATE_JS_TEST,
	    SARMenuOptionsJoystickTestDrawCB
	);

        /* Refresh controller button. */
        SARMenuBuildOptionsButton(
            core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
	    True, "Refresh", SAR_OPT_CONTROLLER_REFRESH
        );

#endif  /* JS_SUPPORT */


        /* ****************************************************** */
        /* Menu: Options->Graphics */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_GRAPHICS,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS
        );

        /* Ground texture. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
	    (float)((1.0 / 8.0) + (0 * 1.0 / 4.0)), 0.13f,
	    0, 0,
            "Ground Texture",
            False, SAR_OPT_SELECT_GROUND_TEXTURE,
            SARMenuOptionsSwitchCB 
        );
        /* Object texture. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
	    (float)((1.0 / 8.0) + (1 * 1.0 / 4.0)), 0.13f,
	    0, 0,
            "Object Texture",
            False, SAR_OPT_SELECT_OBJECT_TEXTURE,
            SARMenuOptionsSwitchCB
        );
        /* Clouds. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (2 * 1.0 / 4.0)), 0.13f,
	    0, 0,
            "Clouds",
            False, SAR_OPT_SELECT_CLOUDS,
            SARMenuOptionsSwitchCB
        );
        /* Atmosphere. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (3 * 1.0 / 4.0)), 0.13f,
	    0, 0,
            "Atmosphere",
            False, SAR_OPT_SELECT_ATMOSPHERE,
            SARMenuOptionsSwitchCB
        );
        /* Dual pass depth. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (0 * 1.0 / 4.0)), 0.35f,
	    0, 0,
            "Dual Pass Depth",
            False, SAR_OPT_SELECT_DUAL_PASS_DEPTH,
            SARMenuOptionsSwitchCB
        );
        /* Prop wash. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (1 * 1.0 / 4.0)), 0.35f,
            0, 0,
            "Prop Wash",
            False, SAR_OPT_SELECT_PROP_WASH,
            SARMenuOptionsSwitchCB
        );
        /* Smoke Trails. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (2 * 1.0 / 4.0)), 0.35f,
            0, 0,
            "Smoke Trails",
            False, SAR_OPT_SELECT_SMOKE_TRAILS,
            SARMenuOptionsSwitchCB
        );
        /* Celestial objects. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
            (float)((1.0 / 8.0) + (3 * 1.0 / 4.0)), 0.35f,
            0, 0,
            "Celestial",
            False, SAR_OPT_SELECT_CELESTIAL_OBJECTS,
            SARMenuOptionsSwitchCB
        );

	/* Visibility maximum. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.27f, 0.5f, 0.44f, 0.0f,
            "Visibility", SAR_OPT_SELECT_MAX_VISIBILITY,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
        }
        SARMenuSpinAddValue(menu, spin_num, "3 miles");
        SARMenuSpinAddValue(menu, spin_num, "6 miles");
        SARMenuSpinAddValue(menu, spin_num, "9 miles");
        SARMenuSpinAddValue(menu, spin_num, "12 miles");
        SARMenuSpinAddValue(menu, spin_num, "15 miles");
        SARMenuSpinAddValue(menu, spin_num, "18 miles");
        SARMenuSpinAddValue(menu, spin_num, "21 miles");

        /* Graphics acceleration. */
        slider_num = SARMenuBuildStandardSlider(
            core_ptr, menu,
            0.27f, 0.6f, 0.44f, 0.0f,
            "Graphics Accel",
            SAR_OPT_ADJUST_GRAPHICS_ACCELERATION,
            SARMenuOptionsSliderCB
        );
        if(slider_num > -1)
        {
            SARMenuSliderSetValueBounds(
                display, menu, slider_num,
                0.0f, 1.0f, False
            );
            SARMenuSliderSetValue(
                display, menu, slider_num,
                0.0f, False
            );
        }

        /* Resolution. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.73f, 0.5f, 0.44f, 0.0f,
            "Resolution", SAR_OPT_SELECT_RESOLUTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
        }
        SARMenuSpinAddValue(menu, spin_num, "100x70");
        SARMenuSpinAddValue(menu, spin_num, "320x240");
        SARMenuSpinAddValue(menu, spin_num, "640x480");
        SARMenuSpinAddValue(menu, spin_num, "800x600");
        SARMenuSpinAddValue(menu, spin_num, "1024x768");

        /* Fullscreen. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu,
	    0.73f, 0.62f,
            0, 0,
            "Fullscreen",
            False, SAR_OPT_SELECT_FULLSCREEN,
            SARMenuOptionsSwitchCB
        );

        /* Graphics info button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.74f, btn_width_std, btn_height_std,
            True, "Info...", SAR_OP_MENU_TO_OPTIONS_GRAPHICS_INFO
        );


        /* ****************************************************** */
        /* Menu: Options->Graphics->Info */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_GRAPHICS_INFO,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to graphics button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS_GRAPHICS
        );

        /* Graphics info message box. */
        SARMenuCreateMessageBox(
            menu, 0.50f, 0.36f, 0.95f, 0.56f,
            &mesgbox_color,
            value_font,
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_GRAPHICS_INFO_MESG,
            NULL        /* No message */
        );

        /* Save button. */
        SARMenuBuildOptionsButton(
            core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
            True, "Save", SAR_OPT_GRAPHICS_INFO_SAVE
        );


        /* ****************************************************** */
        /* Menu: Options->Sound */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_SOUND,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);
        
        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS
        );

	/* Sound level. */
	spin_num = SARMenuBuildStandardSpin(
	    core_ptr, menu, 0.5f, 0.17f, 0.9f, 0.0f,
	    "Sound Level", SAR_OPT_SELECT_SOUND_LEVEL,
	    SARMenuOptionsSpinCB
	);
        if(spin_num > -1)
        {
	    spin = SAR_MENU_SPIN(menu->object[spin_num]);
	    spin->allow_warp = False;
#if defined(HAVE_Y2)
	    SARMenuSpinAddValue(menu, spin_num, "Off");
	    SARMenuSpinAddValue(menu, spin_num, "Events");
	    SARMenuSpinAddValue(menu, spin_num, "Events and Engine");
	    SARMenuSpinAddValue(menu, spin_num, "Events, Engine, and Voice");
#else
	    SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif
	}

        /* Priority spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu, 0.27f, 0.33f, 0.44f, 0.0f,
            "Priority", SAR_OPT_SELECT_SOUND_PRIORITY,
            SARMenuOptionsSpinCB
        );
        if(spin_num > -1)
        {
            spin = SAR_MENU_SPIN(menu->object[spin_num]);
            spin->allow_warp = False;
#if defined(HAVE_Y2)
            SARMenuSpinAddValue(menu, spin_num, "Background");
            SARMenuSpinAddValue(menu, spin_num, "Foreground");
            SARMenuSpinAddValue(menu, spin_num, "Preempt");
#else
            SARMenuSpinAddValue(menu, spin_num, "No support");
	    SARMenuObjectSetSensitive(display, menu, spin_num, False, False);
#endif
        }

	/* Music. */
	switch_num = SARMenuBuildStandardSwitch(
            core_ptr, menu,
	    0.27f, 0.46f,
	    0, 0,
            "Music",
            False, SAR_OPT_SELECT_MUSIC,
            SARMenuOptionsSwitchCB
	);
	if(switch_num > -1)
	{
#if defined(HAVE_Y2)

#else
	    SARMenuObjectSetSensitive(display, menu, switch_num, False, False);
#endif
	}

	/* Begin creating volume controls. */
#ifdef __MSW__
	/* Windows only has sound and music volume. */
	/* Sound volume. */
	slider_num = SARMenuBuildStandardSlider(
	    core_ptr, menu,
	    0.73f, 0.33f, 0.44f, 0.0f,
	    "Sound Volume ",
	    SAR_OPT_ADJUST_SOUND_VOLUME,
	    SARMenuOptionsSliderCB
	);
	if(slider_num > -1)
	{
#if defined(HAVE_Y2)
	    SARMenuSliderSetValueBounds(
		display, menu, slider_num,
		0.0f, 1.0f, False
	    );
	    SARMenuSliderSetValue(
		display, menu, slider_num,
		0.0f, False
	    );
#else
            SARMenuObjectSetSensitive(
                display, menu, slider_num,
                False, False
            );
#endif
	}

        /* Music volume. */
        slider_num = SARMenuBuildStandardSlider(
            core_ptr, menu,
            0.73f, 0.46f, 0.44f, 0.0f,
            "Music Volume ",
            SAR_OPT_ADJUST_MUSIC_VOLUME,
            SARMenuOptionsSliderCB
        );
        if(slider_num > -1)
        {
#if defined(HAVE_Y2)
            SARMenuSliderSetValueBounds(
                display, menu, slider_num,
                0.0f, 1.0f, False
            );
#else
            SARMenuSliderSetValue(
                display, menu, slider_num,
                0.0f, False
            );
            SARMenuObjectSetSensitive(
                display, menu, slider_num,
                False, False
            );
#endif
        }
#else
	/* UNIX volume controls */
        /* Master volume. */
        slider_num = SARMenuBuildStandardSlider(
            core_ptr, menu,
            0.73f, 0.33f, 0.44f, 0.0f,
            "Master Volume",
            SAR_OPT_ADJUST_MASTER_VOLUME,
            SARMenuOptionsSliderCB
        );
        if(slider_num > -1)
        {
#if defined(HAVE_Y2)
            SARMenuSliderSetValueBounds(
                display, menu, slider_num,
                0.0f, 1.0f, False
            );
            SARMenuSliderSetValue(
                display, menu, slider_num,
                0.0f, False
            );
#else
            SARMenuObjectSetSensitive(
                display, menu, slider_num,
                False, False
            );
#endif
        }

	/* Sound volume. */
	slider_num = SARMenuBuildStandardSlider(
	    core_ptr, menu,
	    0.73f, 0.46f, 0.44f, 0.0f,
	    "Sound Volume ",
	    SAR_OPT_ADJUST_SOUND_VOLUME,
	    SARMenuOptionsSliderCB
	);
	if(slider_num > -1)
	{
#if defined(HAVE_Y2)
	    SARMenuSliderSetValueBounds(
		display, menu, slider_num,
		0.0f, 1.0f, False
	    );
	    SARMenuSliderSetValue(
		display, menu, slider_num,
		0.0f, False
	    );
#else
            SARMenuObjectSetSensitive(
                display, menu, slider_num,
                False, False
            );
#endif
	}

        /* Music volume. */
        slider_num = SARMenuBuildStandardSlider(
            core_ptr, menu,
            0.73f, 0.58f, 0.44f, 0.0f,
            "Music Volume ",
            SAR_OPT_ADJUST_MUSIC_VOLUME,
            SARMenuOptionsSliderCB
        );
        if(slider_num > -1)
        {
#if defined(HAVE_Y2)
            SARMenuSliderSetValueBounds(
                display, menu, slider_num,
                0.0f, 1.0f, False
            );
#else
            SARMenuSliderSetValue(
                display, menu, slider_num,
                0.0f, False
            );
            SARMenuObjectSetSensitive(
                display, menu, slider_num,
                False, False
            );
#endif
        }
#endif	/* __MSW__ */

        /* Sound test button. */
        SARMenuBuildOptionsButton(
            core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
            True, "Test", SAR_OPT_SOUND_TEST
        );

        /* Sound info button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.75f, 0.74f, btn_width_std, btn_height_std,
            True, "Info...", SAR_OP_MENU_TO_OPTIONS_SOUND_INFO
        );


        /* ****************************************************** */
        /* Menu: Options->Sound->Info */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_SOUND_INFO,
            img_path
        );
        if(menu == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Back to sound button. */
        SARMenuBuildStandardButton(
            core_ptr, menu, 0.25f, 0.92f, btn_width_std, btn_height_std,
            True,
#ifdef PROG_LANGUAGE_ENGLISH
"Back"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Espalda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Dos"
#endif
#ifdef PROG_LANGUAGE_GERMAN
"Zurck"
#endif
#ifdef PROG_LANGUAGE_ITALIAN
"Dorso"
#endif
#ifdef PROG_LANGUAGE_DUTCH
"Rug"
#endif
#ifdef PROG_LANGUAGE_PORTUGUESE
"Costas"
#endif
#ifdef PROG_LANGUAGE_NORWEGIAN
"Frem"
#endif
	    , SAR_OP_MENU_TO_OPTIONS_SOUND
        );

        /* Sound info message box. */
        SARMenuCreateMessageBox(
            menu, 0.50f, 0.36f, 0.95f, 0.56f,
            &mesgbox_color,
            value_font,
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    core_ptr, SAR_OP_SOUND_INFO_MESG,
            NULL        /* No message */
        );

        /* Save button. */
        SARMenuBuildOptionsButton(
            core_ptr, menu, 0.25f, 0.74f, btn_width_std, btn_height_std,
            True, "Save", SAR_OPT_SOUND_INFO_SAVE
        );


	/* ****************************************************** */
        /* Menu: Loading Simulation */
        img_path = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_PROGRESS_FILE);
        menu = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_LOADING_SIMULATION,
            img_path
        );
        if(menu == NULL)
            return(-1); 
        else
            SARBuildMenusAddToList(core_ptr, menu);

        /* Progress bar. */
	SARMenuCreateProgress(
	    menu,
	    0.5, 1.0,
	    0, 0,
	    NULL,
            &progress_color,
            value_font,
	    core_ptr->menu_progress_bg_img,
	    core_ptr->menu_progress_fg_img,
	    0
	);


	return(0);
}
