#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#include "../include/string.h"
#include "../include/strexp.h"

#include "gw.h"
#include "messages.h"
#include "cmd.h"
#include "sar.h"
#include "scenesound.h"
#include "sartime.h"


void SARCmdOption(SAR_CMD_PROTOTYPE);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? (float)atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? ((int)strlen(s)) : 0)

#define RADTODEG(r)     ((r) * 180.0 / PI)
#define DEGTORAD(d)     ((d) * PI / 180.0)

#define STR_IS_YES(s)	(StringIsYes(s))

#define NOTIFY(s)			\
{ if(SAR_CMD_IS_VERBOSE(flags) &&	\
     (scene != NULL) && ((s) != NULL)	\
  ) { SARMessageAdd(scene, (s)); }	\
}


/*
 *	Global option modification.
 */
void SARCmdOption(SAR_CMD_PROTOTYPE)
{
	int argc;
	char **argv;
	char *parm, *val;
	Boolean got_match = False;
	sar_core_struct *core_ptr = SAR_CORE(data);
	sar_scene_struct *scene = core_ptr->scene;
	sar_option_struct *opt = &core_ptr->option;

        /* No argument? */
        if(*arg == '\0')
        {
	    NOTIFY(
		"Usage: option <parameter>=<value>"
	    );
            return;
        }

	/* Parse arguments, format:
	 *
	 * <parameter>=<value>
	 */
	argv = strchrexp(arg, '=', &argc);
	if(argv == NULL)
	    return;

	/* First argument is option/parameter. */
	if(argc < 2)
	{
	    NOTIFY(
		"Usage: option <parameter>=<value>"
	    );
	    StringFreeArray(argv, argc);
	    return;
	}

	/* Get links to parm and val in the strv array. */
	if(argc > 0)
	    parm = argv[0];
	else
	    parm = NULL;

	if(argc > 1)
	    val = argv[1];
        else
            val = NULL;

	if((parm == NULL) || (val == NULL))
	{
	    NOTIFY(
		"Usage: option <parameter>=<value>"
	    );
            StringFreeArray(argv, argc);
            return;
        } 

	StringStripSpaces(parm);
	StringStripSpaces(val);


	/* Begin handling parameter. */

	/* menu_backgrounds */
	if(!strcasecmp(parm, "menu_backgrounds"))
	{
	    opt->menu_backgrounds = STR_IS_YES(val);
	    got_match = True;
	}
        /* menu_change_affects */
        else if(!strcasecmp(parm, "menu_change_affects"))
        {
            opt->menu_change_affects = STR_IS_YES(val);
            got_match = True;
        }
        /* console_quiet */
        else if(!strcasecmp(parm, "console_quiet"))
        {
            opt->console_quiet = STR_IS_YES(val);
            got_match = True;
        }
        /* internal_debug */
        else if(!strcasecmp(parm, "internal_debug"))
        {
            opt->internal_debug = STR_IS_YES(val);
            got_match = True;
        }
        /* runtime_debug */
        else if(!strcasecmp(parm, "runtime_debug"))
        {
            opt->runtime_debug = STR_IS_YES(val);
            got_match = True;
        }
        /* prioritize_memory */
        else if(!strcasecmp(parm, "prioritize_memory"))
        {
            opt->prioritize_memory = STR_IS_YES(val);
            got_match = True;
        }

        /* textured_ground */
        else if(!strcasecmp(parm, "textured_ground"))
        {
            opt->textured_ground = STR_IS_YES(val);
            got_match = True;
        }
        /* textured_clouds */
        else if(!strcasecmp(parm, "textured_clouds"))
        {
            opt->textured_clouds = STR_IS_YES(val);
            got_match = True;
        }
        /* textured_objects */
        else if(!strcasecmp(parm, "textured_objects"))
        {
            opt->textured_objects = STR_IS_YES(val);
            got_match = True;
        }
        /* atmosphere */
        else if(!strcasecmp(parm, "atmosphere"))
        {
            opt->atmosphere = STR_IS_YES(val);
            got_match = True;
        }
        /* dual_pass_depth */
        else if(!strcasecmp(parm, "dual_pass_depth"))
        {
            opt->dual_pass_depth = STR_IS_YES(val);
            got_match = True;
        }
        /* prop_wash */
        else if(!strcasecmp(parm, "prop_wash"))
        {
            opt->prop_wash = STR_IS_YES(val);
            got_match = True;
        }
        /* smoke_trails */
        else if(!strcasecmp(parm, "smoke_trails"))
        {
            opt->smoke_trails = STR_IS_YES(val);
            got_match = True;
        }
        /* celestial_objects */
        else if(!strcasecmp(parm, "celestial_objects"))
        {
            opt->celestial_objects = STR_IS_YES(val);
            got_match = True;
        }
        /* gl_polygon_offset_factor */
        else if(!strcasecmp(parm, "gl_polygon_offset_factor") ||
                !strcasecmp(parm, "gl_polygon_offset")
        )
        {
            opt->gl_polygon_offset_factor = ATOF(val);
            got_match = True;
        }
        /* gl_shade_model */
        else if(!strcasecmp(parm, "gl_shade_model"))
        {
	    if(!strcasecmp(val, "GL_SMOOTH"))
		opt->gl_shade_model = GL_SMOOTH;
	    else
		opt->gl_shade_model = GL_FLAT;
            got_match = True;
        }
        /* visibility_max */
        else if(!strcasecmp(parm, "visibility_max"))
        {
	    /* Visibility in miles = (n * 3) + 3 */
	    opt->visibility_max = (int)CLIP(
		(ATOF(val) - 3) / 3,
		0, 6
	    );
            got_match = True;
        }
        /* graphics_acceleration */
        else if(!strcasecmp(parm, "graphics_acceleration"))
        {
            opt->graphics_acceleration = (float)CLIP(
		ATOF(val), 0.0, 1.0
	    );
            got_match = True;
        }

#define UPDATE_SCENE_SOUND	\
{ if(scene != NULL) {		\
 SARSceneSoundUpdate(		\
  core_ptr,			\
  opt->engine_sounds,		\
  opt->event_sounds,		\
  opt->voice_sounds,		\
  opt->music			\
 );				\
} }
        /* engine_sounds */
        else if(!strcasecmp(parm, "engine_sounds"))
        {
            opt->engine_sounds = STR_IS_YES(val);
	    UPDATE_SCENE_SOUND
            got_match = True;
        }
        /* event_sounds */
        else if(!strcasecmp(parm, "event_sounds"))
        {
            opt->event_sounds = STR_IS_YES(val);
	    UPDATE_SCENE_SOUND
            got_match = True;
        }
        /* voice_sounds */
        else if(!strcasecmp(parm, "voice_sounds"))
        {
            opt->voice_sounds = STR_IS_YES(val);
	    UPDATE_SCENE_SOUND
            got_match = True;
        }
        /* music */
        else if(!strcasecmp(parm, "music"))
        {
            opt->music = STR_IS_YES(val);
	    UPDATE_SCENE_SOUND
            got_match = True;
        }
#undef UPDATE_SCENE_SOUND

        /* system_time_free_flight */
        else if(!strcasecmp(parm, "system_time_free_flight"))
        {
            opt->system_time_free_flight = STR_IS_YES(val);
            got_match = True;
        }

        /* explosion_frame_int */
        else if(!strcasecmp(parm, "explosion_frame_int"))
        {
            opt->explosion_frame_int = (time_t)MAX(ATOL(val), 0);
            got_match = True;
        }
        /* splash_frame_int */
        else if(!strcasecmp(parm, "splash_frame_int"))
        {
            opt->splash_frame_int = (time_t)MAX(ATOL(val), 0);
            got_match = True;
        }
        /* crash_explosion_life_span */
        else if(!strcasecmp(parm, "crash_explosion_life_span"))
        {
            opt->crash_explosion_life_span = (time_t)MAX(ATOL(val), 0);
            got_match = True;
        }
        /* fuel_tank_life_span */
        else if(!strcasecmp(parm, "fuel_tank_life_span"))
        {
            opt->fuel_tank_life_span = (time_t)MAX(ATOL(val), 0);
            got_match = True;
        }

        /* rotor_wash_vis_coeff */
        else if(!strcasecmp(parm, "rotor_wash_vis_coeff"))
        {
            opt->rotor_wash_vis_coeff = (float)CLIP(
		ATOF(val), 0.0, 1.0
	    );
            got_match = True;
        }

        /* hoist_contact_expansion_coeff */
        else if(!strcasecmp(parm, "hoist_contact_expansion_coeff"))
        {
            opt->hoist_contact_expansion_coeff = ATOF(val);
            got_match = True;
        }
        /* damage_resistance_coeff */
        else if(!strcasecmp(parm, "damage_resistance_coeff"))
        {
            opt->damage_resistance_coeff = ATOF(val);
            got_match = True;
        }
        /* flight_physics */
        else if(!strcasecmp(parm, "flight_physics"))
        {
	    if(!strcasecmp(val, "realistic"))
		opt->flight_physics = FLIGHT_PHYSICS_REALISTIC;
            else if(!strcasecmp(val, "moderate"))
                opt->flight_physics = FLIGHT_PHYSICS_MODERATE;
	    else
                opt->flight_physics = FLIGHT_PHYSICS_EASY;
            got_match = True;
        }

/* Add other option parameters here */

	if(got_match)
	{
	    char *s = (char *)malloc(
		(80 + STRLEN(parm) + STRLEN(val)) * sizeof(char)
	    );
	    sprintf(
		s,
"Option \"%s\" set to \"%s\".",
		parm, val
	    );
	    NOTIFY(s);
	    free(s);
	}
	else
	{
            char *s = (char *)malloc(
                (80 + STRLEN(parm)) * sizeof(char)
            );
            sprintf(
                s,
"%s: No such option.",
                parm
            );
            NOTIFY(s);
            free(s);
	}

	/* Deallocate exploded argument strings. */
	StringFreeArray(argv, argc);

	/* Links to parm and val strings are now invalid. */
	parm = NULL;
	val = NULL;
}
